from docx.document import Document
from docx.table import Table
from docx.text.paragraph import Paragraph
from docx.oxml.xmlchemy import OxmlElement, qn
from docx.oxml.table import CT_Tbl
from docx.oxml.text.paragraph import CT_P
from collections import OrderedDict
from docx.enum.dml import MSO_COLOR_TYPE
from docx.shared import RGBColor
import logging
from copy import deepcopy

logger = logging.getLogger(__name__)


class DocumentHandler(object):
    """
    Extends the docx API to overcome its limitations
    """
    @staticmethod
    def delete_paragraph(paragraph):
        p = paragraph._element
        if p.getparent() is not None:
            p.getparent().remove(p)
        p._p = p._element = None

    @staticmethod
    def delete_table(table):
        tbl = table._element
        if tbl.getparent() is not None:
            tbl.getparent().remove(tbl)
        tbl._tbl = tbl._element = None

    @staticmethod
    def get_paragraph(document, paragraph_pointer):
        for paragraph in document.paragraphs:
            if paragraph._p == paragraph_pointer:
                return paragraph
        return None

    @staticmethod
    def delete_element(element):
        if isinstance(element, Table):
            DocumentHandler.delete_table(element)
        elif isinstance(element, Paragraph):
            DocumentHandler.delete_paragraph(element)
        else:
            logger.warn("Unable to delete element : unsupported type %s", element.__class__.__name__)

    @staticmethod
    def between(document, block):
        logger.debug("Extracting elements between %s and %s", block.start, block.end)
        elements = DocumentHandler.elements(document)
        next = DocumentHandler.next(elements, block.start.paragraph._p)
        extracted = []
        while next is not None and next != block.end.paragraph._p:
            extracted.append(next)
            next = DocumentHandler.next(elements, next)
        otw = DocumentHandler.ooxml_to_wrapper(document)
        wrappers = []
        for e in extracted:
            if e in otw:
                wrappers.append(otw[e])
        logger.debug("Extracted elements are %s", wrappers)
        return wrappers

    @staticmethod
    def next(elements, key):
        if hasattr(elements, "_OrderedDict__map"):
            # Python2
            if key in elements._OrderedDict__map:
                link_prev, link_next, key = elements._OrderedDict__map[key]
                return link_next[2]
            else:
                return None
        else:
            # Python3
            if key in elements:
                keys = list(elements)
                key_index = keys.index(key)
                if key_index < len(keys) - 1:
                    return keys[key_index + 1]
            return None

    @staticmethod
    def ooxml_to_wrapper(document):
        elt_ids = {}
        for p in document.paragraphs:
            elt_ids[p._p] = p
        for tbl in document.tables:
            elt_ids[tbl._tbl] = tbl
        return elt_ids

    @staticmethod
    def elements(document):
        """
        Returns the dict of the elements.
        This dict is ordered. We use a dict to easily access next and previous elements of an item.
        We must use it because a `docx.Document` does not expose a combined list of paragraphs and tables but maintains two separated lists.
        Internally, he maintains a ordered list of ooxml element mixing paragraph and tables
        Or when dealing with block placeholders (aka {{placeholder}}TABLE{{placeholder}}), 
        we need to find the proper table between the two placeholders.
        """
        elts = OrderedDict()
        if not isinstance(document, Document):
            for elt in document._element:
                try:
                    elts[elt] = elt
                except:
                    logger.error("%s not found", elt)
        else:
            for elt in document._body._element:
                try:
                    elts[elt] = elt
                except:
                    logger.error("%s not found", elt)
        return elts

    @staticmethod
    def insert_paragraph_after(paragraph, text=None, style=None):
        """Insert a new paragraph after the given paragraph."""
        new_p = OxmlElement("w:p")
        paragraph._p.addnext(new_p)
        new_para = Paragraph(new_p, paragraph._parent)
        if text:
            new_para.add_run(text)
        if style is not None:
            new_para.style = style
        DocumentHandler.copy_paragraph_format(paragraph.paragraph_format, new_para.paragraph_format)
        return new_para
    

    @staticmethod
    def copy_element_after_paragraph(paragraph, to_copy):
        """
        Insert a copy of the paragraph or table to_copy after the given paragraph.
        """
        # We use deepcopy, which is a bit 'low-level' because not all props of the paragraph/table are visible/copyable through the API
        if isinstance(to_copy, Paragraph):
            new_p = deepcopy(to_copy._p)
            paragraph._p.addnext(new_p)
            return Paragraph(new_p, paragraph._parent)
        elif isinstance(to_copy, Table):
            new_tbl = deepcopy(to_copy._tbl)
            paragraph._p.addnext(new_tbl)
            return Table(new_tbl, paragraph._parent)

    @staticmethod
    def copy_paragraph_format(src, dest):
        """
        Copy paragraph format attributes to an other one.
        Warning: it is impossible to copy tab_stops as the docx paragraph_format api doesn't offers any setter
        """
        dest.alignment = src.alignment
        dest.first_line_indent = src.first_line_indent
        dest.keep_together = src.keep_together
        dest.keep_with_next = src.keep_with_next
        dest.left_indent = src.left_indent
        dest.line_spacing = src.line_spacing
        dest.line_spacing_rule = src.line_spacing_rule
        dest.page_break_before = src.page_break_before
        dest.right_indent = src.right_indent
        dest.space_after = src.space_after
        dest.space_before = src.space_before
        dest.widow_control = src.widow_control

    @staticmethod
    def copy_font(from_run, to_run):
        to_run.font.name = from_run.font.name
        to_run.font.size = from_run.font.size
        to_run.font.italic = from_run.font.italic
        to_run.font.cs_italic = from_run.font.cs_italic
        to_run.font.bold = from_run.font.bold
        to_run.font.cs_bold = from_run.font.cs_bold
        to_run.font.underline = from_run.font.underline
        to_run.font.strike = from_run.font.strike
        if from_run.font.color:
            if from_run.font.color.rgb:
                to_run.font.color.rgb = from_run.font.color.rgb
            if from_run.font.color.theme_color:
                to_run.font.color.theme_color = from_run.font.color.theme_color
                w_theme_shade = from_run.font.color._color.xpath("@w:themeShade")
                if len(w_theme_shade) > 0:
                    to_run.font.color._color.set(qn("w:themeShade"), w_theme_shade[0])
                w_theme_tint = from_run.font.color._color.xpath("@w:themeTint")
                if len(w_theme_tint) > 0:
                    to_run.font.color._color.set(qn("w:themeTint"), w_theme_tint[0])
            if from_run.font.color.type == MSO_COLOR_TYPE.AUTO:
                # This refers to the global auto color (usually black), which overloads the current style value
                # this is different to keep the color at None, because it color = None inherit from the current style value (and paragraph style if adequate)

                # Unfortunately, the python docx wrapper allows to read this 'auto' color, but not to wrap it, as it only allows RGB values to be written there.
                # Here we use an awful hack, to circumvent the attribute value validation.
                to_run.font.color.rgb = RGBColor(0,0,0) # we leverage this just for the wrapper to create the adequate elements
                to_run.font.color._color.set(qn("w:val"), "auto") # and then we brutally override the rgb value with the special string "auto".


    @staticmethod
    def iter_block_items(document):
        """
        Given a document returns an iterator that will traverse paragraphs and tables in the order they appear in the document.
        
        :param [Document or Cell] document: document object
        :rtype: [Paragraph or Table] iterator
        """
        
        container = document._element.body if isinstance(document, Document) else document._element

        for child in container.iterchildren():
            if isinstance(child, CT_P):
                yield Paragraph(child, document)
            elif isinstance(child, CT_Tbl):
                yield Table(child, document)


class TableHandler(object):
    @staticmethod
    def copy_table_style(table, source):
        """
        Duplicate the style from table_style into table
        :param table: a empty table that need a style
        :param table_style: a reference tab with a style
        """

        # Start by copying the global style of the table
        TableHandler.__copy_table_metadata__(table, source)

        # Report existing columns/rows
        row_id = 0

        while row_id < len(source.rows) and row_id < len(table.rows):
            col_id = 0
            source_cells = source.rows[row_id].cells
            table_cells = table.rows[row_id].cells
            while col_id < len(source_cells) and col_id < len(table_cells):
                TableHandler.__copy_cell_style__(source_cells[col_id], table_cells[col_id])
                col_id += 1
            row_id += 1

        # new rows
        row_id = len(source.rows)
        if row_id > 0:
            while row_id < len(table.rows):
                col_id = 0
                previous_table_cells = table.rows[row_id - 1].cells
                table_cells = table.rows[row_id].cells
                while col_id < len(table_cells):
                    TableHandler.__copy_cell_style__(previous_table_cells[col_id], table_cells[col_id])
                    col_id += 1
                row_id += 1

        # new columns
        row_id = 0
        while row_id < len(table.rows):
            col_id = len(source.rows[0].cells)
            table_cells = table.rows[row_id].cells
            if col_id > 0:
                while col_id < len(table_cells):
                    TableHandler.__copy_cell_style__(table_cells[col_id - 1], table_cells[col_id])
                    col_id += 1
            row_id += 1

    @staticmethod
    def __extract_attribute__(reference_tc_pr, output_element, path, attribute_name, default=None):
        """
        Use Xpath to extract an element from reference_tc_pr and duplicate it on output_element
        :param reference_tc_pr: input element ad a tc_pr with attributes
        :param output_element: empty output element
        :param path: XML path
        :param attribute_name: name of the attribute
        :param default: default value if no value were found
        """
        attribute = reference_tc_pr.xpath(path + "/@" + attribute_name)
        if len(attribute) > 0:
            output_element.set(qn(attribute_name), attribute[0])
        elif default:
            output_element.set(qn(attribute_name), default)

    @staticmethod
    def __copy_table_metadata__(table, source):
        """
        Copy the table metadata, like "is the first row a header"
        :param table:
        :param source:
        :return:
        """
        # useful for debug, display the input xml: logging.debug("XML from _tbl %s", source.rows._tbl.xml)
        from_tbl_pr = source.rows._tbl.tblPr
        if len(from_tbl_pr.xpath("w:tblLook")) > 0:
            to_tbl_pr = table.rows._tbl.tblPr
            if len(to_tbl_pr.xpath("w:tblLook")) > 0:
                to_tbl_pr.remove_all("w:tblLook")
            w_tbl_look = OxmlElement("w:tblLook")

            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:val")
            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:firstRow")
            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:lastRow")
            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:firstColumn")
            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:lastColumn")
            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:noHBand")
            TableHandler.__extract_attribute__(from_tbl_pr, w_tbl_look, "w:tblLook", "w:noVBand")
            to_tbl_pr.append(w_tbl_look)

    @staticmethod
    def __copy_border__(reference_tc_pr, element_name):
        """
        Copy the different element of a border into a new OxmlElement
        :param reference_tc_pr: element we want to get border information from
        :param element_name: the type of border we want to get (w:top, w:bottom, w:left or w:right)
        :return: a new OxmlElement that copy the style of the input
        """
        element = OxmlElement(element_name)

        tc_borders_path = "w:tcBorders/"
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_borders_path + element_name, "w:val")
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_borders_path + element_name, "w:color")
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_borders_path + element_name, "w:sz")
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_borders_path + element_name, "w:space")
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_borders_path + element_name, "w:shadow")
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_borders_path + element_name, "w:frame")
        return element

    @staticmethod
    def __copy_margin__(reference_tc_pr, element_name):
        """
        Copy the different element of a margin into a new OxmlElement
        :param reference_tc_pr: element we want to get border information from
        :param element_name: the type of border we want to get (w:top, w:bottom, w:left or w:right)
        :return: a new OxmlElement that copy the style of the input
        """
        element = OxmlElement(element_name)

        tc_mar_path = "w:tcMar/"
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_mar_path + element_name, "w:fill")
        TableHandler.__extract_attribute__(reference_tc_pr, element, tc_mar_path + element_name, "w:w")
        return element

    @staticmethod
    def __copy_cell_style__(from_cell, to_cell):
        """
        Copy the style of "from_cell" into the "to_cell"
        :param from_cell: a cell containing a pretty style
        :param to_cell: a cell without style
        """
        from_tc_pr = from_cell._tc.get_or_add_tcPr()
        to_tc_pr = to_cell._tc.get_or_add_tcPr()
        # useful for debug, display the input xml: logging.debug("XML tc_pr %s", from_tc_pr.xml)

        # Copy the configuration (has header/footer/...)
        if len(from_tc_pr.xpath("w:cnfStyle")) > 0:
            w_cnf_style = OxmlElement("w:cnfStyle")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:val")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:firstRow")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:lastRow")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:firstColumn")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:lastColumn")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:oddVBand")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:evenVBand")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:oddHBand")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:evenHBand")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:firstRowFirstColumn")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:firstRowLastColumn")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:lastRowFirstColumn")
            TableHandler.__extract_attribute__(from_tc_pr, w_cnf_style, "w:cnfStyle", "w:lastRowLastColumn")
            to_tc_pr.append(w_cnf_style)


        # Copy the color of the text and the background
        if len(from_tc_pr.xpath("w:shd")) > 0:
            w_shd = OxmlElement("w:shd")
            TableHandler.__extract_attribute__(from_tc_pr, w_shd, "w:shd", "w:fill", "auto")
            TableHandler.__extract_attribute__(from_tc_pr, w_shd, "w:shd", "w:color", "auto")
            TableHandler.__extract_attribute__(from_tc_pr, w_shd, "w:shd", "w:themeColor")
            TableHandler.__extract_attribute__(from_tc_pr, w_shd, "w:shd", "w:themeFill")
            TableHandler.__extract_attribute__(from_tc_pr, w_shd, "w:shd", "w:themeShade")
            TableHandler.__extract_attribute__(from_tc_pr, w_shd, "w:shd", "w:themeFillShade")
            to_tc_pr.append(w_shd)

        # Copy the borders
        if len(from_tc_pr.xpath("w:tcBorders")) > 0:
            w_tc_borders = OxmlElement("w:tcBorders")
            w_tc_borders.append(TableHandler.__copy_border__(from_tc_pr, "w:top"))
            w_tc_borders.append(TableHandler.__copy_border__(from_tc_pr, "w:bottom"))
            w_tc_borders.append(TableHandler.__copy_border__(from_tc_pr, "w:left"))
            w_tc_borders.append(TableHandler.__copy_border__(from_tc_pr, "w:right"))
            to_tc_pr.append(w_tc_borders)

        if len(from_tc_pr.xpath("w:tcMar")) > 0:
            w_tc_mars = OxmlElement("w:tcMar")
            w_tc_mars.append(TableHandler.__copy_margin__(from_tc_pr, "w:top"))
            w_tc_mars.append(TableHandler.__copy_margin__(from_tc_pr, "w:bottom"))
            w_tc_mars.append(TableHandler.__copy_margin__(from_tc_pr, "w:left"))
            w_tc_mars.append(TableHandler.__copy_margin__(from_tc_pr, "w:right"))
            to_tc_pr.append(w_tc_mars)


        if len(from_tc_pr.xpath("w:tcW")) > 0:
            w_tc_w = OxmlElement("w:tcW")
            TableHandler.__extract_attribute__(from_tc_pr, w_tc_w, "w:tcW", "w:type")
            TableHandler.__extract_attribute__(from_tc_pr, w_tc_w, "w:tcW", "w:w")
            to_tc_pr.append(w_tc_w)
