import sys
import logging
import argparse
import socket
import json
from tornado.web import RequestHandler
from dataiku.base.utils import watch_stdin, get_json_friendly_error

logging.basicConfig(level=logging.INFO, format="%(asctime)s %(levelname)s %(message)s")

STREAMLIT_DEFAULT_PORT = 8501
STREAMLIT_STATUS_PATH = "/__ping"  # defined in StreamlitWebAppMeta.LIVENESS_CHECK_PATH

def write_error_json(error_details):
    with open("error.json", "w") as f:
        json.dump(error_details, f)

try:
    from streamlit import config as st_config
except ImportError as e:
    logging.error(f"Unable to import streamlit. Use a code env containing the streamlit package: {str(e)}")
    write_error_json(get_json_friendly_error())
    raise e

try:
    from streamlit.web import bootstrap as st_bootstrap
except:
    import streamlit.bootstrap as st_bootstrap

try:
    from streamlit.web.server import Server
except:
    from streamlit.server.server import Server

class StatusHandler(RequestHandler):
    """
    liveness check/probe handler
    """
    async def get(self):
        self.write("ok")
        self.set_status(200)

def patch_create_app():
    """
    Patch Streamlit's Tornado app creation with additional handler(s)
    :return:
    """
    orig_create_app = Server._create_app

    def patched_create_app(self, *a, **kw):
        app = orig_create_app(self, *a, **kw)
        app.add_handlers(r".*", [(rf"{STREAMLIT_STATUS_PATH}", StatusHandler)])
        return app

    Server._create_app = patched_create_app
    logging.info("patched streamlit.web.server.Server._create_app")

def patch_on_server_start(port):
    """
    Monkey patch bootstrap._on_server_start to log "started" message after streamlit init
    :param port:
    :return:
    """
    orig_on_server_start = st_bootstrap._on_server_start

    def patched_on_server_start(server: Server):
        orig_on_server_start(server)
        logging.info(f"Started Streamlit on port {port}")

    st_bootstrap._on_server_start = patched_on_server_start
    logging.info("Patched streamlit.web.bootstrap._on_server_start")


def setup_streamlit_config(host, port) -> dict:
    options = {
        "server.address": host,
        "server.port": port,
        "server.headless": True,
        "server.runOnSave": False,
        "server.fileWatcherType": "none",
        "server.baseUrlPath": "",
        "server.folderWatchList": [],
        "browser.gatherUsageStats": False,
        "browser.serverPort": port,
    }
    for key, value in options.items():
        st_config.set_option(key, value)
    return options

def find_next_available_port(
    start_port: int, host: str = "127.0.0.1", attempts: int = 10
) -> int:
    """
    todo maybe this already exists or there is a better place for it
    Find the next available port starting from start_port.

    Args:
        start_port (int): The port number to start checking from.
        host (str): The host interface to bind to (default: 127.0.0.1).
        attempts (int): how many time to try before giving up

    Returns:
        int: The next available port starting with start_port.
    """
    port = start_port
    while port <= 65535 and attempts > 0:
        attempts = attempts - 1
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
            try:
                s.bind((host, port))
                return port
            except OSError:
                port += 1
    raise RuntimeError(f"No available port found in range {start_port}–{port}")

def parse_command_line():
    logging.info(f"run_streamlit invoked with command {sys.argv}")
    parser = argparse.ArgumentParser(
        prog="run_streamlit",
        description="Starts and monitors streamlit and handles dss integration",
        epilog="Text at the bottom of help",
    )
    parser.add_argument(
        "-H", "--host", default="127.0.0.1", help="local hostname or IP to bind to"
    )
    parser.add_argument(
        "-p",
        "--port",
        default=0,
        help=f"streamlit port. Defaults to next available port from {STREAMLIT_DEFAULT_PORT}",
    )
    parser.add_argument(
        "-w", "--workers", default="1", help="number of monitor workers. defaults to 1"
    )
    parser.add_argument(
        "-b", "--external-base-url", default=None, help="external base url"
    )
    parser.add_argument("script", help="streamlit application script")
    args = parser.parse_args()

    candidate_port = int(args.port)
    if candidate_port == 0:
        candidate_port = find_next_available_port(
            STREAMLIT_DEFAULT_PORT, host=args.host
        )

    return args.script, candidate_port, args.host, args.workers, args.external_base_url

def main():
    try:
        watch_stdin()
        (main_script_path, port, host, workers, external_base_url) = parse_command_line()

        logging.info(f"Starting Streamlit app for {main_script_path} on {host}:{port}")

        try:
            patch_create_app()
            patch_on_server_start(port)
        except Exception as e:
            logging.warning("Code could not be instrumented to boostrap your application. Please use a supported streamlit version in your code environment", e)
            raise

        setup_streamlit_config(host, port)

        # start streamlit in-process
        # todo find out why we cannot use args or flag_options to pass options
        try:
            st_bootstrap.run(main_script_path, is_hello=False, args=[], flag_options={})
        except:
            st_bootstrap.run(main_script_path, command_line=None, args=[], flag_options={})

    except Exception as e:
        logging.exception(f"Backend main loop failed: {e}")
        write_error_json(get_json_friendly_error())

if __name__ == "__main__":
    main()
