from datetime import datetime

class ComputedMetrics(object):
    """
    Handle to the metrics of a DSS object and their last computed value

    .. important::

        Do not create this class directly, instead use :meth:`dataiku.Dataset.get_last_metric_values`, 
        :meth:`dataiku.Folder.get_last_metric_values`, :meth:`dataiku.ModelEvaluationStore.get_last_metric_values` or
        :meth:`dataiku.Project.get_last_metric_values`

    """

    def __init__(self, raw):
        self.raw = raw

    def get_metric_by_id(self, metric_id):
        """
        Retrieve the info for a given metric
        
        Usage example

        .. code-block:: python

            dataset = dataiku.Dataset("my_dataset")
            metrics = dataset.get_last_metric_values()
            count_files_metric = metrics.get_metric_by_id("basic:COUNT_FILES")
            for value in count_files_metric['lastValues']:
                print("partition=%s -> count of files=%s" % (value['partition'], value['value']))        

        :param string metric_id: identifier of the metric

        :return: information about the metric and its values. Top-level fields are

                    * **metric** : definition of the metric
                    * **meta** : display metadata, as a dict of **name** and **fullName**
                    * **computingProbe** : name of the probe that computes the metric
                    * **displayedAsMetric** : whether the metric is among the metrics displayed on the "Status" tab of the object
                    * **notExistingViews** : list of the possible types of metrics datasets not yet created on the object
                    * **partitionsWithValue** : list of the partition identifiers for which some value of the metric exists
                    * **lastValues** : list of the last computed value, per partition. Each list element has

                            * **partition** : the partition identifier, as a string.
                            * **value** : the metric value, as a string
                            * **dataType** : expected type of **value** (one of BIGINT, DOUBLE, STRING, BOOLEAN)
                            * **computed** : timestamp of computation, in milliseconds since epoch


        :rtype: dict
        """
        all_ids = []
        for metric in self.raw["metrics"]:
            all_ids.append(metric["metric"]["id"])
            if metric["metric"]["id"] == metric_id:
                return metric
        raise Exception("Metric %s not found among: %s" % (metric_id, all_ids))

    def get_global_data(self, metric_id):
        """
        Get the global value point of a given metric, or throws.
        
        For a partitioned dataset, the global value is the value of the metric computed
        on the whole dataset (coded as partition 'ALL').
        
        :param string metric_id: identifier of the metric

        :returns: the metric data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **value** : the metric value, as a string
                    * **dataType** : expected type of **value** (one of BIGINT, DOUBLE, STRING, BOOLEAN)
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_metric_by_id(metric_id)["lastValues"]:
            # for folders, partition is not defined
            if "partition" not in partition_data or partition_data["partition"] == "" or partition_data["partition"] == "NP" or partition_data["partition"] == "ALL":
                return partition_data
        raise Exception("No data found for global partition for metric %s" % metric_id)

    def get_global_value(self, metric_id):
        """
        Get the global value of a given metric, or throws.
        
        For a partitioned dataset, the global value is the value of the metric computed
        on the whole dataset (coded as partition 'ALL').

        Usage example

        .. code-block:: python

            dataset = dataiku.Dataset("my_dataset")
            metrics = dataset.get_last_metric_values()
            print("record count = %s" % metrics.get_global_value('records:COUNT_RECORDS'))
        
        :param string metric_id: identifier of the metric

        :returns: the value of the metric for the partition
        :rtype: str, int or float
        """
        return ComputedMetrics.get_value_from_data(self.get_global_data(metric_id))

    def get_partition_data(self, metric_id, partition):
        """
        Get the value point of a given metric for a given partition, or throws.
        
        :param string metric_id: identifier of the metric
        :param string partition: partition identifier

        :returns: the metric data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **value** : the metric value, as a string
                    * **dataType** : expected type of **value** (one of BIGINT, DOUBLE, STRING, BOOLEAN)
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_metric_by_id(metric_id)["lastValues"]:
            if partition_data["partition"] == partition:
                return partition_data

    def get_partition_value(self, metric_id, partition):
        """
        Get the value of a given metric for a given partition, or throws.
        
        :param string metric_id: identifier of the metric
        :param string partition: partition identifier

        :returns: the value of the metric for the partition
        :rtype: str, int or float
        """
        return ComputedMetrics.get_value_from_data(self.get_partition_data(metric_id, partition))

    def get_first_partition_data(self, metric_id):
        """
        Get a value point of a given metric, or throws. The first value encountered is returned.
        
        :param string metric_id: identifier of the metric

        :returns: the metric data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **value** : the metric value, as a string
                    * **dataType** : expected type of **value** (one of BIGINT, DOUBLE, STRING, BOOLEAN)
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_metric_by_id(metric_id)["lastValues"]:
            return partition_data
        raise Exception("No partition data for metric %s" % metric_id)

    def get_partition_data_for_version(self, metric_id, version_id):
        """
        Get the metric of the first partition matching version_id, for saved models

        :param string metric_id: identifier of the metric
        :param string version_id: identifier of the version

        :returns: the metric data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **value** : the metric value, as a string
                    * **dataType** : expected type of **value** (one of BIGINT, DOUBLE, STRING, BOOLEAN)
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_metric_by_id(metric_id)["lastValues"]:
            if partition_data["partition"] == version_id:
                return partition_data
        raise Exception("No partition with value data for metric %s" % metric_id)

    def get_all_ids(self):
        """
        Get the identifiers of all metrics defined in this object

        :returns: list of metric identifiers
        :rtype: list[string]
        """
        all_ids = []
        for metric in self.raw["metrics"]:
            all_ids.append(metric["metric"]["id"])
        return all_ids


    @staticmethod
    def get_value_from_data(data):
        """
        Retrieves the value from a metric point, cast in the appropriate type (str, int or float).
        
        For other types, the value is not cast and left as a string.
        
        :param dict data: a value point for a metric, retrieved with :meth:`~ComputedMetrics.get_global_data` or  :meth:`~ComputedMetrics.get_partition_data`
        
        :returns: the value, cast to the appropriate Python type
        :rtype: str, int or float
        """
        dtype = data.get("dataType", "STRING")
        if dtype in ["BIGINT", "INT"]:
            return int(data["value"])
        elif dtype in ["FLOAT", "DOUBLE"]:
            return float(data["value"])
        else:
            return data["value"]
            
# technical class, used in Python checks. Mirros the MetricDataPoint Java class
class MetricDataPoint(object):
    """
    A value of a metric, on a partition

    .. note:: 

        Instances of this class are only created by Python checks

    """
    def __init__(self, raw):
        self.raw = raw

    def get_metric(self):
        """
        Get the definition of the metric

        :returns: a dict defining the metric. Fields are

                    * **id** : the metric full identifier 
                    * **type** : type of the probe computing the metric
                    * **metricType** : type of the metric for the probe
                    * **dataType** : type of the value computed (of BIGINT, DOUBLE, STRING, BOOLEAN)
                    * **column** : (optional) name of the column the metric is computed for

        :rtype: dict
        """
        return self.raw['metric']

    def get_metric_id(self):
        """
        Get the metric's full identifier.

        :rtype: string
        """
        return self.raw['metric'].get('id', None)

    def get_partition(self):
        """
        Get the identifier of the partition on which the value was computed.

        :rtype: string
        """
        return self.raw['partition']

    def get_value(self):
        """
        Get the raw value of the metric.

        Usage example:

        .. code-block:: python

            # the code for a Python check that errors if there are more 
            # than 10k records in the dataset.
            # the parameters of process() are filled by DSS:
            # - last_values is a dict of metric name to MetricDataPoint
            # - dataset is a handle on the dataset
            # - partition_id is the partition for which the check is run
            def process(last_values, dataset, partition_id):
                # get the MetricDataPoint
                last_known_record_count = last_values.get('records:COUNT_RECORDS')
                if last_known_record_count is None:
                    return 'EMPTY', 'Record count not yet computed'
                record_count = int(last_known_record_count.get_value())
                if record_count < 10000:    
                    return 'OK'
                else:
                    return 'ERROR', 'Too many records'        

        :rtype: string
        """
        return self.raw['value']

    def get_compute_time(self):
        """
        Get the time at which the value was computed.

        :rtype: :class:`datetime.datetime`
        """
        return datetime.utcfromtimestamp((int)(self.raw['time'] / 1000))

    def get_type(self):
        """
        Get the type of the value.

        :returns: a type, of BIGINT, DOUBLE, BOOLEAN, STRING
        :rtype: string
        """
        return self.raw['type']

    def __repr__(self,):
        return self.get_metric_id() + ' on ' + self.get_partition() + ' / ' + self.get_compute_time().strftime('%Y-%m-%d %H:%M:%S') + ' = ' + self.get_value() + ' (' + self.get_type() + ')'
        
# this class isn't very useful, because you can't get the checks out of datasets, folders or models
class ComputedChecks(object):
    """
    Handle to the checks of a DSS object and their last computed value

    .. important::

        Do not create this class directly, instead use :meth:`dataiku.Project.get_last_check_values`

    """
    def __init__(self, raw):
        self.raw = raw

    def get_check_by_name(self, check_name):
        """
        Retrive the info for a given check
        
        :param string check_name: identifier of the check
        """
        all_names = []
        for check in self.raw["checks"]:
            all_names.append(check["name"])
            if check["name"] == check_name:
                return check
        raise Exception("Check %s not found among: %s" % (check_name, all_names))

    def get_global_data(self, check_name):
        """
        Get the global value point of a given check, or throws.
        
        For a partitioned dataset, the global value is the value of the check computed
        on the whole dataset (coded as partition 'ALL').
        
        :param string check_name: identifier of the check

        :returns: the check data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **outcome** : one of OK, ERROR, WARNING, EMPTY
                    * **message** : (optional) message of the check
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_check_by_name(check_name)["lastValues"]:
            # for folders, partition is not defined
            if "partition" not in partition_data or partition_data["partition"] == "" or partition_data["partition"] == "NP" or partition_data["partition"] == "ALL":
                return partition_data
        raise Exception("No data found for global partition for check %s" % check_name)

    def get_global_value(self, check_name):
        """
        Get the global value of a given check, or throws.
        
        For a partitioned dataset, the global value is the value of the check computed
        on the whole dataset (coded as partition 'ALL').
        
        :param string check_name: identifier of the check

        :returns: outcome of the check (OK, ERROR, WARNING or EMPTY)
        :rtype: string
        """
        return ComputedChecks.get_outcome_from_data(self.get_global_data(check_name))

    def get_partition_data(self, check_name, partition):
        """
        Get the value point of a given check for a given partition, or throws.
        
        :param string check_name: identifier of the check
        :param string partition: partition identifier

        :returns: the check data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **outcome** : one of OK, ERROR, WARNING, EMPTY
                    * **message** : (optional) message of the check
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_check_by_name(check_name)["lastValues"]:
            if partition_data["partition"] == partition:
                return partition_data

    def get_partition_value(self, check_name, partition):
        """
        Get the value of a given check for a given partition, or throws.
        
        :param string check_name: identifier of the check
        :param string partition: partition identifier

        :returns: outcome of the check for this partition (OK, ERROR, WARNING or EMPTY)
        :rtype: string
        """
        return ComputedChecks.get_outcome_from_data(self.get_partition_data(check_name, partition))

    def get_first_partition_data(self, check_name):
        """
        Get a value point of a given check, or throws. The first value encountered is returned.
        
        :param string check_name: identifier of the check

        :returns: the check data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **outcome** : one of OK, ERROR, WARNING, EMPTY
                    * **message** : (optional) message of the check
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_check_by_name(check_name)["lastValues"]:
            return partition_data
        raise Exception("No partition data for check %s" % check_name)

    def get_partition_data_for_version(self, check_name, version_id):
        """
        Get the check of the first partition matching version_id, for saved models

        :param string check_name: identifier of the check
        :param string version_id: identifier of the version

        :returns: the check data, as a dict. Fields are

                    * **partition** : the partition identifier, as a string.
                    * **outcome** : one of OK, ERROR, WARNING, EMPTY
                    * **message** : (optional) message of the check
                    * **computed** : timestamp of computation, in milliseconds since epoch
        :rtype: dict        
        """
        for partition_data in self.get_check_by_name(check_name)["lastValues"]:
            if partition_data["partition"] == version_id:
                return partition_data
        raise Exception("No partition with value data for check %s" % check_name)

    def get_all_names(self):
        """
        Get the identifiers of all checks defined in this object

        :returns: list of check identifiers
        :rtype: list[string]
        """
        all_names = []
        for check in self.raw["checks"]:
            all_names.append(check["name"])
        return all_names


    @staticmethod
    def get_outcome_from_data(data):
        """
        Retrieves the value from a check data point
        
        :param dict data: a value point for a check, retrieved with :meth:`~ComputedChecks.get_global_data` or  :meth:`~ComputedChecks.get_partition_data`

        :returns: a check result (OK, ERROR, WARNING or EMPTY)
        :rtype: string
        """
        return data["outcome"]
            
            
# doesn't seem used anymore, keeping for fear of breaking something
class CheckDataPoint(object):
    """
    A value of a check, on a partition

    .. note:: 

        Instances of this class are only created by Python checks

    """
    def __init__(self, raw):
        self.raw = raw

    def get_check(self):
        """
        Returns the definition of the check

        :returns: a dict of the check definition. Notable fields are

                    * **type** : the type of check
                    * **meta** : the display metadata, as a dict of **name** and **label**

        :rtype: dict
        """
        return self.raw['check']

    def get_partition(self):
        """
        Returns the partition on which the value was computed

        :returns: a partition identifier
        :rtype: string
        """
        return self.raw['partition']

    def get_value(self):
        """
        Returns the value of the check, as a string

        :returns: one of OK, ERROR, WARNING, EMPTY (means "no data, check can't be computed")
        :rtype: string
        """
        return self.raw['outcome']

    def get_compute_time(self):
        """
        Returns the time at which the value was computed

        :rtype: :class:`datetime.datetime`
        """
        return datetime.utcfromtimestamp((int)(self.raw['time'] / 1000))

    def __repr__(self,):
        return self.get_check_name() + ' on ' + self.get_partition() + ' / ' + self.get_compute_time().strftime('%Y-%m-%d %H:%M:%S') + ' = ' + self.get_value()
                