(function(){
'use strict';

var app = angular.module('dataiku.admin.security', []);

app.controller("AdminSecurityController", function(){
});


app.directive("projectGrantItem", function(){
    return {
        template : `
            <ul class="grant-matrix">
                <li><abbr ng-show="grant.item.readProjectContent" class="grant grant--readProjectContent" title="Read project content">RPC</abbr></li>
                <li><abbr ng-show="grant.item.writeProjectContent" class="grant grant--writeProjectContent" title="Write project content">WPC</abbr></li>
                <li><abbr ng-show="grant.item.shareToWorkspaces" class="grant grant--shareToWorkspaces" title="Publish to workspaces">PW</abbr></li>
                <li><abbr ng-show="grant.item.publishToDataCollections" class="grant grant--publishToDataCollections" title="Publish to Data Collections">PDC</abbr></li>
                <li><abbr ng-show="grant.item.readDashboards" class="grant grant--readDashboards" title="Read dashboard">RD</abbr></li>
                <li><abbr ng-show="grant.item.writeDashboards" class="grant grant--writeDashboards" title="Write dashboards">WD</abbr></li>
                <li ng-if="showLegacyPermissions"><abbr ng-show="grant.item.moderateDashboards" class="grant grant--moderateDashboards" title="Moderate dashboards">MD</abbr></li>
                <li ng-if="showLegacyPermissions"><abbr ng-show="grant.item.manageAdditionalDashboardUsers" class="grant grant--manageAdditionalDashboardUsers" title="Manage dashboard users">MDU</abbr></li>
                <li><abbr ng-show="grant.item.manageDashboardAuthorizations" class="grant grant--manageDashboardAuthorizations" title="Manage authorized objects">MAO</abbr></li>
                <li><abbr ng-show="grant.item.manageExposedElements" class="grant grant--manageExposedElements" title="Manage shared objects">MSO</abbr></li>
                <li><abbr ng-show="grant.item.runScenarios" class="grant grant--runScenarios" title="Run scenarios">RS</abbr></li>
                <li><abbr ng-show="grant.item.executeApp" class="grant grant--executeApp" title="Execute app">EA</abbr></li>
                <li><abbr ng-show="grant.item.editPermissions" class="grant grant--editPermissions" title="Edit permissions">EP</abbr></li>
                <li><abbr ng-show="grant.item.admin" class="grant grant--admin" title="Admin">A</abbr></li>
            </ul>
        `,
        scope : {
            grant : '=',
            showLegacyPermissions : '='
        }
    }
});


app.directive("authorizationMatrixTable", function(){
    return {
        scope : true,
        link : function($scope) {
            $scope.hover = {
                col : null
            }
        }
    }
});

/**
 * users selection modal for the authorization matrix
 */
app.controller("AdminSecurityAuthorizationMatrixUserController", function($scope) {
    // create a map because we need to save the index of the selected users in the authorization matrix not the whole object
    const loginIndexMap = {};
    $scope.authorizationMatrix.perUser.users.forEach((user, userIdx) => {
        loginIndexMap[user.login] = userIdx;
    });
    $scope.localSelectedUsers = $scope.selectedUsers.map((idx) => $scope.authorizationMatrix.perUser.users[idx]);

    $scope.deselectUser = function(userIndex) {
        // we use filter instead of splice so changes gets detected by dku-bs-select
        $scope.localSelectedUsers = $scope.localSelectedUsers.filter((_, idx) => idx !== userIndex);
    }

    $scope.save = function() {
        const localSelectedIndexes = $scope.localSelectedUsers.map((user) => loginIndexMap[user.login]);
        $scope.saveUsers(localSelectedIndexes, $scope.unselectedUsers);
        $scope.dismiss();
    }
});

/**
 * groups selection modal for the authorization matrix
 */
app.controller("AdminSecurityAuthorizationMatrixGroupController", function($scope) {
    // create a map because we need to save the index of the selected groups in the authorization matrix not the whole object
    const groupIndexMap = {};
    $scope.authorizationMatrix.perGroup.groups.forEach((group, groupIdx) => {
        groupIndexMap[group] = groupIdx;
    });
    $scope.localSelectedGroups = $scope.selectedGroups.map((idx) => $scope.authorizationMatrix.perGroup.groups[idx]);

    $scope.deselectGroup = function(groupIndex) {
        // we use filter instead of splice so changes gets detected by dku-bs-select
        $scope.localSelectedGroups = $scope.localSelectedGroups.filter((_, idx) => idx !== groupIndex);
    }

    $scope.save = function(group) {
        const localSelectedIndexes = $scope.localSelectedGroups.map((group) => groupIndexMap[group]);
        $scope.saveGroups(localSelectedIndexes, $scope.unselectedGroups);
        $scope.dismiss();
    }
});

app.factory("AdminSecurityAuthorizationMatrixExportService", () => {
    function isUserContext(context) {
        return context === "USERS";
    }

    function globalAuthorizationTitleKeyPair(context) {
        return [
            ["Global admin", "admin"],
            ["Manage user-defined meanings", "mayManageUDM"],
            ["Create projects", "mayCreateProjects"],
            ["Create workspaces", "mayCreateWorkspaces"],
            ["Share to workspaces", "mayShareToWorkspaces"],
            ["Create Data Collections", "mayCreateDataCollections"],
            ["Publish to Data Collections", "mayPublishToDataCollections"],
            [isUserContext(context) ? "Manage own code envs" : "Create code envs", "mayCreateCodeEnvs"],
            ["Manage all code envs", "mayManageCodeEnvs"],
            [isUserContext(context) ? "Manage own cluster" : "Create clusters", "mayCreateClusters"],
            ["Manage all clusters", "mayManageClusters"],
            [isUserContext(context) ? "Manage own Code Studio template" : "Create Code Studio templates", "mayCreateCodeStudioTemplates"],
            ["Manage all Code Studio templates", "mayManageCodeStudioTemplates"],
            ["Develop plugins", "mayDevelopPlugins"],
            ["Edit lib folders", "mayEditLibFolders"],
            ["Create user connections", "mayCreateAuthenticatedConnections"],
            ["Write unisolated code", "mayWriteUnsafeCode"],
            ["Write isolated code", "mayWriteSafeCode"],
            ["Create published API services", "mayCreatePublishedAPIServices"],
            ["Create published projects", "mayCreatePublishedProjects"],
            ["May write in root project folder", "mayWriteInRootProjectFolder"],
            ["May create active web content", "mayCreateActiveWebContent"],
            ["Create Data Collections", "mayCreateDataCollections"],
            ["Publish to Data Collections", "mayPublishToDataCollections"],
        ];
    }

    function projectAuthorizationTitleKeyPair() {
        return [
            ["Read project content", "readProjectContent"],
            ["Write project content", "writeProjectContent"],
            ["Run scenarios", "runScenarios"],
            ["Share to workspaces", "shareToWorkspaces"],
            ["Read dashboard", "readDashboards"],
            ["Write dashboards", "writeDashboards"],
            ["Moderate dashboards", "moderateDashboards"],
            ["Manage authorized objects", "manageDashboardAuthorizations"],
            ["Manage shared objects", "manageExposedElements"],
            ["Manage dashboard users", "manageAdditionalDashboardUsers"],
            ["Share to Data Collections", "publishToDataCollections"],
            ["Execute app", "executeApp"],
            ["Admin", "admin"],
        ];
    }

    function createHeader(context) {
        let headers = [
            { name: "Permission", type: "string" },
            { name: "Type", type: "string" },
            { name: "Name", type: "string" }
        ];
        if(isUserContext(context)) {
            headers.push(
                { name: "User", type: "string" },
                { name: "Login", type: "string" }
            );
        } else {
            headers.push({ name: "Group", type: "string" });
        }
        headers.push({ name: "Value", type: "string" });
        return headers;
    }

    function createInstanceAuthorization(element, index, rawAuthorizationMatrix, context) {
        const selector = isUserContext(context) ? "perUser" : "perGroup";
        return globalAuthorizationTitleKeyPair(context).map((authorization) => {
            let result = [
                authorization[0],
                "GLOBAL",
                null
            ];
            if(isUserContext(context)) {
                result.push(element.displayName, element.login);
            } else {
                result.push(element);
            }
            result.push(`${rawAuthorizationMatrix[selector][authorization[1]][index] ? 1 : 0}`);
            return result;
        });
    }

    function createProjectAuthorization(element, elementIndex, rawAuthorizationMatrix, context) {
        const selector = isUserContext(context) ? "perUser" : "perGroup";
        return rawAuthorizationMatrix[selector].projectsGrants.map((projectGrants) => {
            return projectAuthorizationTitleKeyPair().map((authorization) => {
                let result = [
                    authorization[0],
                    "PROJECT",
                    projectGrants.projectKey
                ];
                if(isUserContext(context)) {
                    result.push(element.displayName, element.login);
                } else {
                    result.push(element);
                }
                result.push(projectGrants.grants[elementIndex]
                    ? `${projectGrants.grants[elementIndex].item[authorization[1]] ? 1 : 0}`
                    : "0");
                return result;
            });
        });
    }

    function createElement(rawAuthorizationMatrix, context) {
        return function (element, index) {
            return [
                ...createInstanceAuthorization(element, index, rawAuthorizationMatrix, context),
                ...createProjectAuthorization(element, index, rawAuthorizationMatrix, context).flat(),
            ];
        };
    }

    function createData(rawAuthorizationMatrix, context) {
        return rawAuthorizationMatrix[isUserContext(context) ? "perUser" : "perGroup"][
            isUserContext(context) ? "users" : "groups"
        ].map(createElement(rawAuthorizationMatrix, context)).flat();
    }

    return function (rawAuthorizationMatrix, context) {
        return {
            name: `${isUserContext(context) ? 'users' : 'groups'}-matrix`,
            description: "The matrix",
            columns: createHeader(context),
            data: createData(rawAuthorizationMatrix, context),
        };
    };
});

app.controller("AdminSecurityAuthorizationMatrixController", function($scope, DataikuAPI, TopNav, CreateModalFromTemplate, ExportUtils, AdminSecurityAuthorizationMatrixExportService, $filter) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.formData = {};
    $scope.selectedUsers = [];
    $scope.selectedGroups = [];
    $scope.userPagination = {
        selectedPage: 0,
        pageSize: 50,
    }
    $scope.groupPagination = {
        selectedPage: 0,
        pageSize: 50,
    }
    $scope.uiState = {
        showPermissionsBy: 'USERS',
        query: '',
        showLegacyPermissions: false,
    }
    $scope.authorizationMatrix = {
        perUser: {
            projectsGrants: []
        },
        perGroup: {
            projectsGrants: [],
        }
    }

    function createRangeOfTenOrLessFromList(list){
        return Array.from({length:Math.min(list.length, 10)}, (_,k)=>k);
    }

    function isUserSelected(index){
        return $scope.selectedUsers.includes(index);
    }

    function isGroupSelected(index){
        return $scope.selectedGroups.includes(index);
    }

    $scope.saveUsers = function(selectedUsers, unselectedUsers){
        $scope.selectedUsers = selectedUsers;
        $scope.unselectedUsers = unselectedUsers;
    }

    $scope.showUserModal = function(){
        CreateModalFromTemplate("/templates/admin/security/authorization-matrix-user.html", $scope, "AdminSecurityAuthorizationMatrixUserController")
    }

    $scope.saveGroups = function(selectedGroups, unselectedGroups){
        $scope.selectedGroups = selectedGroups;
        $scope.unselectedGroups = unselectedGroups;
    }

    $scope.showGroupModal = function(){
        CreateModalFromTemplate("/templates/admin/security/authorization-matrix-group.html", $scope, "AdminSecurityAuthorizationMatrixGroupController")
    }

    DataikuAPI.security.getAuthorizationMatrix().success(function(data){
        $scope.authorizationMatrix = data;
        $scope.selectedUsers = createRangeOfTenOrLessFromList(data.perUser.users);
        $scope.selectedGroups = createRangeOfTenOrLessFromList(data.perGroup.groups);
        $scope.unselectedUsers = data.perUser.users.filter((_, index) => !isUserSelected(index));
        $scope.unselectedGroups = data.perGroup.groups.filter((_, index) => !isGroupSelected(index));
    }).error(setErrorInScope.bind($scope));

    $scope.onGroupPageSelect = function(event){
        $scope.groupPagination.selectedPage = event.pageIndex;
    }

    $scope.onUserPageSelect = function(event){
        $scope.userPagination.selectedPage = event.pageIndex;
    }

    function resetPaginationSelectedPageIfNeeded(filteredOutput, paginationSettings) {
        if (Math.ceil(filteredOutput.length / paginationSettings.pageSize) <= paginationSettings.selectedPage) {
            paginationSettings.selectedPage = 0;
        }
    }

    $scope.$watchCollection(() => [$scope.authorizationMatrix.perGroup.projectsGrants, $scope.uiState.query], ([projectsGrantsGroup, filterQuery]) => {
        $scope.filteredGroup = $filter('filter')(projectsGrantsGroup, filterQuery);
        resetPaginationSelectedPageIfNeeded($scope.filteredGroup, $scope.groupPagination);
    })

    $scope.$watchCollection(() => [$scope.authorizationMatrix.perUser.projectsGrants, $scope.uiState.query], ([projectsGrantsUser, filterQuery]) => {
        $scope.filteredUser = $filter('filter')(projectsGrantsUser, filterQuery);
        resetPaginationSelectedPageIfNeeded($scope.filteredUser, $scope.userPagination);
    })

    $scope.uiState = $scope.uiState || {};
    $scope.uiState.showPermissionsBy = $scope.uiState.showPermissionsBy || "USERS";

    $scope.export = function() {
        ExportUtils.exportUIData(
            $scope,
            AdminSecurityAuthorizationMatrixExportService($scope.authorizationMatrix, $scope.uiState.showPermissionsBy),
            `Export ${$scope.uiState.showPermissionsBy === 'USERS' ? 'users' : 'groups'} matrix`,
            { downloadOnly: true, hideAdvancedParameters:true }
        );
    };
});

// we can't create a component here due to the fact that <user-authorization> is not
// a valid html element
app.directive("authorizations", function () {
  return {
    restrict: "A",
    scope: { title: "@", selectedElements: "<", allElements: "<", key: "@" },
    template: `
            <th class="table-matrix__title table-matrix__title--grant">{{title}}</th>
            <td ng-repeat="elementIndex in selectedElements"
                ng-class="{'global-grant': allElements[key][elementIndex]}">
                <i class="icon-ok-sign" ng-if="allElements[key][elementIndex]" />
                <span class="sr-only" ng-if="allElements[key][elementIndex]">Enabled</span>
            </td>
    `,
  };
});

app.component('paginationInfo', {
    bindings: {
        selectedPage: '<',
        pageSize: '<',
        listSize: '<',
    },
    controller: function(){
        this.$onChanges = () => {
            this.firstVisibleElement = this.selectedPage * this.pageSize + 1;
            this.lastVisibleElement = Math.min((this.selectedPage + 1) * this.pageSize, this.listSize);
        }
    },
    template: `
        <span class="pagination__label">Projects &nbsp;{{$ctrl.firstVisibleElement}}-{{$ctrl.lastVisibleElement}} of {{$ctrl.listSize}}</span>
    `
});

app.component('discoverabilitySettings', {
    bindings: {
        objectType: '<',
        requestType: '<',
        requestDocumentation: '<',
        visibilitySettings: '<'
    },
    templateUrl: '/templates/admin/security/discoverability-settings-dropdowns.html'
});

app.controller("UsersController", function($scope, $filter, DataikuAPI, Dialogs, TopNav, Logger, CreateModalFromTemplate, ListFilter, WT1, FutureProgressModal) {
	TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.canFetchExternalUsers = false;
    DataikuAPI.admin.users.getFetchableSourceTypes().success(function(data) {
        $scope.canFetchExternalUsers = (data || []).length > 0;
    }).error(setErrorInScope.bind($scope));

    // filter and pagination of the user table
    $scope.selection = { orderQuery: "login", orderReversed: false };
    $scope.pagination = new ListFilter.Pagination([], 50);
    $scope.isOutOfTrials = false;

    function arrayToHtmlList(a) {
        return a.map(e => `<li>${sanitize(e)}</li>`).join('');
    }

    function formatUsers(users) {
        return users.map(u => `${u.login}: ${u.displayName}`);
    }

	function refreshList() {
	    const lastActivity = true;
        DataikuAPI.admin.users.list(lastActivity).success(function(data) {
            $scope.users = data;
            refreshPage();
        }).error(setErrorInScope.bind($scope));
    };

    function refreshPage() {
        $scope.pagination.updateAndGetSlice($scope.selection.filteredObjects);
    };

    $scope.switchToTrial = function (login) {
        const targetLoginH = md5(login.toLowerCase());

        function startUserTrial(selectedItem) {
            WT1.event("admin-user-trial-start", {
                "forLoginh": targetLoginH,
                "chosenUserProfile": selectedItem["title"]
            });

            DataikuAPI.admin.users.switchToTrial(login, selectedItem["title"]).success(function(data) {
                WT1.event("admin-user-trial-start-success", {
                    "forLoginh": targetLoginH,
                    "chosenUserProfile": selectedItem["title"]
                });
                refreshList();
            }).error(function(a, b, c) {
                setErrorInScope.bind($scope)(a, b, c);
                WT1.event("admin-user-trial-start-failed", {
                    "forLoginh": targetLoginH,
                    "chosenUserProfile": selectedItem["title"]
                });
            });
        }

        DataikuAPI.admin.getTrialStatus().success(function(data) {
            const trialLicenses = window.dkuAppConfig.licensing.userProfiles
                .filter(profile => profile !== "NONE")
                .map(profile => {
                    const profileTrialStatus = data.allocations.filter(elt => elt.userProfile === profile);
                    const remainingTokensForProfile = profileTrialStatus.length > 0 && profileTrialStatus[0].mode === "TOTAL_CREDIT" ?
                        profileTrialStatus[0].remainingCredit
                        : 0;
                    const selectable = remainingTokensForProfile > 0 || profileTrialStatus.length && profileTrialStatus[0].mode !== "TOTAL_CREDIT";
                    return ({
                        "title": profile,
                        "desc": "Start a trial as profile " + profile,
                        selectable,
                        unselectableReason: !selectable ? `No more trial tokens available for the "${profile}" profile. Please select another profile or reach out to Dataiku to request additional trial tokens.` : ''
                    });
                });

            const selectableTrialLicences = trialLicenses.filter(item => item.selectable);

            if (selectableTrialLicences.length === 0) {
                $scope.isOutOfTrials = true;
                return;
            }
            $scope.isOutOfTrials = false;

            Dialogs.select($scope, "Start trial for " + login,
                "Please select the user profile for which a trial will be started",
                trialLicenses,
                selectableTrialLicences[0] || null,
                {
                    alert: "Dataiku needs to store the user's login in order to grant the personal trial license. Additionally, Dataiku may contact the user during the trial, for onboarding purposes.",
                    alertLevel: "info"
                }
            )
                .then(startUserTrial);
        }).error(setErrorInScope.bind($scope));
    }


    function getConversionOptions(userProfile) {
        let ret = [
            {"title": "Remove access", "desc": "User profile will be changed to NONE"},
            {"title": "Convert to normal user", "desc": "User will become a normal user, with the user profile " + userProfile}
        ]
        if (window.dkuAppConfig && window.dkuAppConfig.licensing && window.dkuAppConfig.licensing.userProfiles &&
            window.dkuAppConfig.licensing.userProfiles.indexOf("AI_CONSUMER") >= 0) {
            ret.push({"title": "Convert to AI Consumer", "desc": "User will become an AI Consumer user"})
        } else {
            ret.push({"title": "Convert to Reader", "desc": "User will become a Reader user"})
        }
        return ret;
    }

    $scope.convertExpiredTrial = function(user) {
        const login = user.login;
        const targetLoginH = md5(login.toLowerCase());
        const items = getConversionOptions(user.userProfile)
        Dialogs.select($scope, "Convert expired trial for " + login,
            login + " had a trial, but it expired. What do you want to do next?",
            items, items[0]).then(function(selectedItem){

            let action = null;
            if (selectedItem === items[0]) {
                action = "SWITCH_TO_NONE";
            } else if (selectedItem === items[1]) {
                action = "SWITCH_TO_REGULAR";
            } else if (selectedItem === items[2]) {
                action = "SWITCH_TO_READER";
            }
            WT1.event("admin-user-trial-convert-expired", {"forLoginh": targetLoginH, "action": action});

            DataikuAPI.admin.users.convertFromTrial(login, action).success(function(data) {
                refreshList();
            }).error(setErrorInScope.bind($scope));
        });
    }

    $scope.convertActiveTrial = function(user) {
        const login = user.login;
        const targetLoginH = md5(login.toLowerCase());
        const items = getConversionOptions(user.userProfile);
        Dialogs.select($scope, "Convert trial for " + login,
            login + " is currently running a trial. What do you want to do next?",
            items, items[0]).then(function(selectedItem){

            let action = null;
            if (selectedItem === items[0]) {
                action = "SWITCH_TO_NONE";
            } else if (selectedItem === items[1]) {
                action = "SWITCH_TO_REGULAR";
            } else if (selectedItem === items[2]) {
                action = "SWITCH_TO_READER";
            }

            WT1.event("admin-user-trial-convert-not-expired", {"forLoginh": targetLoginH, "action": action});

            DataikuAPI.admin.users.convertFromTrial(login, action).success(function(data) {
                refreshList();
            }).error(setErrorInScope.bind($scope));
        });
    }

    $scope.canDoDisableEnableMassAction = function (selectedUsers, activate) {
        return selectedUsers.some(u => u.enabled !== activate);
    };

    $scope.activateDeactivateUsers = function (users, activate) {
        event.preventDefault();
        if (!$scope.canDoDisableEnableMassAction(users, activate)) {
            return;
        }
        users = users.filter(u => u.enabled !== activate);

        const title = `Confirm user${users.length > 1 ? 's' : ''} ${activate ? 'activation' : 'deactivation'}`;
        const loginsText = arrayToHtmlList(formatUsers(users));
        const text = `Are you sure you want to ${activate ? 'enable' : 'disable'} the following user${users.length > 1 ? 's' : ''}<ul>${loginsText}</ul>`;
        const logins = users.map(u => u.login);

        if (activate) {
            Dialogs.confirmPositive($scope, title, text).then(() => {
                DataikuAPI.admin.users.enableOrDisable(logins, true).success(() => {
                    refreshList();
                    logins.forEach((login) => {
                        WT1.event('user-enable', {"forLoginh": md5(login.toLowerCase())});
                    });
                }).error(setErrorInScope.bind($scope));
            });
        } else {
            DataikuAPI.admin.users.prepareDisable(logins).success(data => {
                Dialogs.confirmInfoMessages($scope, title, data, text, false).then(() => {
                    DataikuAPI.admin.users.enableOrDisable(logins, false).success(() => {
                        refreshList();
                        logins.forEach((login) => {
                            WT1.event('user-disable', {"forLoginh": md5(login.toLowerCase())});
                        });
                    }).error(setErrorInScope.bind($scope));
                });
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.deleteUsers = function(selectedUsers) {
        const loginsText = arrayToHtmlList(formatUsers(selectedUsers));
        const text = `Are you sure you want to delete the following users<ul>${loginsText}</ul>`;

        const logins = selectedUsers.map(u => u.login);
        DataikuAPI.admin.users.prepareDelete(logins).success(function(data) {
            Dialogs.confirmInfoMessages($scope, 'Confirm users deletion', data, text, false).then(function() {
                DataikuAPI.admin.users.delete(logins).success(function(data) {
                    refreshList();
                    logins.forEach((login) => {
                        WT1.event('user-delete', {"forLoginh": md5(login.toLowerCase())});
                    });
                }).error(setErrorInScope.bind($scope));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteUser = function(user) {
        DataikuAPI.admin.users.prepareDelete([user.login]).success(function(data) {
            Dialogs.confirmInfoMessages($scope, 'Confirm user deletion', data, 'Are you sure you want to delete user "'+sanitize(user.login) + '"?', false).then(function() {
                DataikuAPI.admin.users.delete([user.login]).success(function(data) {
                    refreshList();
                    WT1.event('user-delete', {"forLoginh": md5(user.login.toLowerCase())});
                }).error(setErrorInScope.bind($scope));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.openAssignUsersToGroupModal = function(users, groups) {
        CreateModalFromTemplate("/templates/admin/security/assign-users-groups-modal.html", $scope, null, function(newScope) {
            newScope.users = users;
            const newGroups = {};

            groups.forEach(group => {
                const empty = ! users.some(u => u.groups.includes(group));
                const full = users.every(u => u.groups.includes(group));
                newGroups[group] = {
                    name: group,
                    selected: full,
                    originallyAssigned: !empty,
                    indeterminate: !empty && !full
                };
            });

            newScope.groups = newGroups;

            let newAssignedGroups = {};
            newScope.assignGroup = function(group) {
                if (!group.selected && group.originallyAssigned) {
                    newAssignedGroups[group.name] = false;
                } else if (group.selected && !group.originallyAssigned) {
                    newAssignedGroups[group.name] = true;
                } else if (group.selected && group.indeterminate) {
                    newAssignedGroups[group.name] = true;
                } else {
                    delete newAssignedGroups[group.name];
                }
            };

            newScope.wereGroupsChanged = function() {
                return Object.keys(newAssignedGroups).length > 0;
            };


            newScope.assignGroups = function(users) {
                const groupsToAdd = Object.keys(newAssignedGroups).filter(k => newAssignedGroups[k]);
                const groupsToRemove = Object.keys(newAssignedGroups).filter(k => ! newAssignedGroups[k]);


                const loginsText = arrayToHtmlList(formatUsers(users));
                let text = `The following users <ul>${loginsText}</ul>`;
                if (groupsToAdd.length > 0) {
                    const groupsToAddText = arrayToHtmlList(groupsToAdd);
                    text += `will be added to groups <ul>${groupsToAddText}</ul>`;
                }
                if (groupsToRemove.length > 0) {
                    const groupsToRemoveText = arrayToHtmlList(groupsToRemove);
                    text += `will be removed from groups <ul>${groupsToRemoveText}</ul>`;
                }

                const logins = users.map(u => u.login);
                DataikuAPI.admin.users.prepareAssignUsersGroups(logins, groupsToAdd, groupsToRemove).success(function(data) {
                    Dialogs.confirmInfoMessages($scope, 'Confirm reassigning users to groups', data, text, false).then(function() {
                        newScope.dismiss();  // close first modal
                        Logger.info("Adding users", logins, "to group", groupsToAdd, " and removing from groups", groupsToRemove);
                        DataikuAPI.admin.users.assignUsersGroups(logins, groupsToAdd, groupsToRemove).success(function(data) {
                            refreshList();
                        }).error(setErrorInScope.bind($scope));
                    });
                }).error(setErrorInScope.bind($scope));
            };
        });
    };

    $scope.openAssignUsersToProfileModal = function(users) {
        CreateModalFromTemplate("/templates/admin/security/assign-users-profile-modal.html", $scope, null, function(newScope) {
            newScope.users = users;
            newScope.newProfile = null;
            newScope.assignProfile = function(users, newProfile) {
                const usersByProfile = new Map();
                const usersToUpdate = users.filter(u => u.userProfile !== newProfile);
                usersToUpdate.forEach((user) => {
                    if (user.userProfile !== newProfile) { // Skip users that already have the correct profile
                        const usersForProfile = usersByProfile.get(user.userProfile);
                        if (usersForProfile) {
                            usersForProfile.push(user);
                        } else {
                            usersByProfile.set(user.userProfile, [user]);
                        }
                    }
                });
                let text = "";
                usersByProfile.forEach((affectedUsers, profile) => {
                    const loginsText = arrayToHtmlList(formatUsers(affectedUsers));
                    const niceProfile = $filter('niceProfileName')(profile);
                    const niceNewProfile = $filter('niceProfileName')(newProfile);
                    text += `The following users will have their profile changed from <strong>${niceProfile}</strong> to <strong>${niceNewProfile}</strong>: <ul class="mbot8">${loginsText}</ul></p>`;
                });

                Dialogs.confirmInfoMessages($scope, 'Confirm changing profile for users', null, text, false).then(() => {
                    newScope.dismiss();  // close first modal
                    const logins = usersToUpdate.map(u => u.login);
                    DataikuAPI.admin.users.assignUsersProfile(logins, newProfile).success(() => {
                        refreshList();
                    }).error(setErrorInScope.bind($scope));
                });
            };
        });
    };

    // Used to change the page displayed when clicking on number or arrows
    $scope.$watch('pagination.page', () => {
        $scope.pagination.update();
    });

    // Filter/sorting is done by filtered-multi-select-rows, just refresh the pagination
    $scope.$watch('selection.filteredObjects', () => {
        refreshPage();
    });

    refreshList();
    DataikuAPI.security.listGroups(true).success(function(data) {
        if (data) {
            data.sort();
        }
        $scope.groups = data;
    }).error(setErrorInScope.bind($scope));

    // Sync all users that comes from external sources (LDAP, custom user supplier)
    $scope.syncAll = function() {
        DataikuAPI.admin.users.syncAll().success(function(data) {
            FutureProgressModal.show($scope, data, "Syncing users...", null, 'static', false, true).then(function(result) {
                Dialogs.infoMessagesDisplayOnly($scope, "Sync result", result, result.futureLog);
                refreshList();
            })
        }).error(setErrorInScope.bind($scope));
    }

    // The "sync all" action is possible only if at least one user source type is syncable
    $scope.canSyncAll = false;
    DataikuAPI.admin.users.getSyncableSourceTypes().success(function(data) {
        $scope.canSyncAll = data.length > 0;
    }).error(setErrorInScope.bind($scope));
});


app.controller("UserController", function($scope, $state, $stateParams, DataikuAPI, $route, TopNav, Dialogs, ActivityIndicator, WT1, FutureProgressModal) {
	TopNav.setLocation(TopNav.DSS_HOME, "administration");
    let savedUser;
    $scope.user = {
            groups: [],
            login:'',
            sourceType: 'LOCAL',
            displayName:'',
            userProfile : 'DATA_SCIENTIST',
            //codeAllowed : true,
            password:'',
            trialTokenStrategy: 'NO_ATTEMPT'
    };
    if ($scope.appConfig && $scope.appConfig.licensing && $scope.appConfig.licensing.userProfiles) {
        $scope.user.userProfile = $scope.appConfig.licensing.userProfiles[0];
    }

    if ($stateParams.login) {
        $scope.creation = false;
        fetchUser();
    } else {
        $scope.creation = true;
        DataikuAPI.security.listGroups(true).success(function(data) {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
        }).error(setErrorInScope.bind($scope));
    }

    function fetchUser() {
        DataikuAPI.security.listGroups(true).success(function(data) {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
            DataikuAPI.admin.users.get($stateParams.login).success(function(data) {
                $scope.user = data;
                savedUser = angular.copy(data);
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    }

    $scope.prepareSaveUser = function() {
        if ($scope.creation) {
            $scope.saveUser();
        } else {
            DataikuAPI.admin.users.prepareUpdate($scope.user).success(function(data) {
                Dialogs.confirmInfoMessages($scope,
                    'Confirm user edition', data, 'Are you sure you want to edit user "'+sanitize($scope.user.login) + '"?', true
                ).then($scope.saveUser);
            }).error(setErrorInScope.bind($scope));
        }
    };

    const buildUserWT1Params = (user) => {
        const params = {
            "forLoginh": md5(user.login.toLowerCase()),
            "type": user.sourceType,
            "chosenUserProfile": user.userProfile
        };
        if (user.email && user.email != "") {
            params.forEmailh = md5(user.email.toLowerCase());
        }
        return params;
    };

    $scope.saveUser = function() {
        if ($scope.creation) {
            DataikuAPI.admin.users.create($scope.user).success(function(data) {
                WT1.event('user-create', buildUserWT1Params($scope.user));
                Dialogs.infoMessagesDisplayOnly($scope, "Warnings during user creation", data).then(function(){
                    $state.go("admin.security.users.list");
                });
            }).error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.admin.users.update($scope.user).success(function(data) {
                WT1.event('user-edit', buildUserWT1Params($scope.user));
                $state.go("admin.security.users.list");
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.userIsDirty = function() {
        return !angular.equals(savedUser, $scope.user);
    };

    var getGeneralSettings = function() {
    	DataikuAPI.admin.getGeneralSettings().success(function(gs) {
            $scope.generalSettings = gs;
    	}).error(setErrorInScope.bind($scope));
    }

    $scope.$watch('user', function() {
        if (! $scope.user.adminProperties) {
            $scope.user.adminProperties = {};
        }
        if (! $scope.user.userProperties) {
            $scope.user.userProperties = {};
        }
    });
    // Init
    getGeneralSettings();

    // Sync the user with its source (LDAP, custom user supplier)
    $scope.syncUser = function() {
        DataikuAPI.admin.users.sync($stateParams.login).success(function(data) {
            FutureProgressModal.show($scope, data, "Syncing user...", null, 'static', false, true).then(function(result) {
                Dialogs.infoMessagesDisplayOnly($scope, "Sync result", result, result.futureLog);
                fetchUser();
            })
        }).error(setErrorInScope.bind($scope));
    };

    // Syncing is possible only for syncable source types and not in creation mode
    if (!$scope.creation) {
        DataikuAPI.admin.users.getSyncableSourceTypes().success(function(data) {
            $scope.syncableSourceTypes = data;
        }).error(setErrorInScope.bind($scope));
    }
});


app.controller("GroupsController", function($scope, $state, $stateParams, DataikuAPI, $route, $modal, $q, Dialogs, TopNav, CreateModalFromTemplate) {
	TopNav.setLocation(TopNav.DSS_HOME, "administration");
	$scope.usersByGroup = {};
	$scope.apiKeysByGroup = {};
    // Populate UI
    var loadGroups = function() {
        const promises = [
            DataikuAPI.security.listGroupsFull(),
            DataikuAPI.security.listUsers(),
            DataikuAPI.admin.publicApi.listGlobalKeys()
        ]

        $q.all(promises).then(function(responses) {
            const groups = responses[0].data;
            const users = responses[1].data;
            const apiKeys = responses[2].data;

            for (const group of groups) {
                group.userCount = 0;
                $scope.usersByGroup[group.name] = [];
                for (const userDesc of users) {
                    if (userDesc.groups.includes(group.name)) {
                        group.userCount++;
                        $scope.usersByGroup[group.name].push(userDesc);
                    }
                }
                group.apiKeyCount = 0;
                $scope.apiKeysByGroup[group.name] = [];
                for (const apiKeyDesc of apiKeys) {
                    if (apiKeyDesc.groups && apiKeyDesc.groups.includes(group.name)) {
                        group.apiKeyCount++;
                        $scope.apiKeysByGroup[group.name].push(apiKeyDesc);
                    }
                }
            }
            $scope.groups = groups;
        }).catch(setErrorInScope.bind($scope));
    };

    // Delete a group
    $scope.deleteGroup = function(group) {
        DataikuAPI.security.prepareDeleteGroup(group.name).success(function(data) {
            Dialogs.confirmInfoMessages($scope, 'Delete group', data, 'Are you sure you want to delete group "' + sanitize(group.name) + '" ?', false).then(function () {
                DataikuAPI.security.deleteGroup(group.name).success(function (data) {
                    loadGroups();
                }).error(setErrorInScope.bind($scope));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.showGroupUsers = (groupName) => {
        CreateModalFromTemplate("/templates/admin/security/group-members-modal.html", $scope, null, function (newScope) {
            newScope.groupName = groupName;
            newScope.usersInGroup = $scope.usersByGroup[groupName];

            newScope.matchesQuery = (object) => {
                if (!object) return false;
                if (!newScope.filterQuery) return true;
                if (object.login && object.login.toLowerCase().includes(newScope.filterQuery.toLowerCase())) {
                    return true;
                }
                if (object.displayName && object.displayName.toLowerCase().includes(newScope.filterQuery.toLowerCase())) {
                    return true;
                }
                return false;
            }
        });
    };

    $scope.showGroupApiKeys = (groupName) => {
        CreateModalFromTemplate("/templates/admin/security/group-apikeys-modal.html", $scope, null, function (newScope) {
            newScope.groupName = groupName;
            newScope.apiKeysInGroup = $scope.apiKeysByGroup[groupName];

            newScope.matchesQuery = (object) => {
                if (!object) return false;
                if (!newScope.filterQuery) return true;
                if (object.id && object.id.toLowerCase().includes(newScope.filterQuery.toLowerCase())) {
                    return true;
                }
                if (object.label && object.label.toLowerCase().includes(newScope.filterQuery.toLowerCase())) {
                    return true;
                }
                return false;
            }
        });
    };

    // Init
    loadGroups();
});


app.controller("GroupController",function($scope, $state, $stateParams, DataikuAPI, $route, TopNav, Dialogs) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    let savedGroup;
    if ($stateParams.name) {
        $scope.creation = false;
        DataikuAPI.security.getGroup($stateParams.name).success(function(data) {
            $scope.group = data;
            savedGroup = angular.copy(data);
        }).error(setErrorInScope.bind($scope));
    } else {
        $scope.creation = true;
        $scope.group = {
            sourceType: 'LOCAL',
            mayWriteSafeCode : true,
            mayWriteInRootProjectFolder: true,
            mayCreateProjects: true,
            mayCreateActiveWebContent: true,
            mayShareToWorkspaces: true,
            mayCreateDataCollections: true,
            mayPublishToDataCollections: true,
        };
    }

    $scope.prepareSaveGroup = function() {
        if ($scope.creation) {
            $scope.saveGroup();
        } else {
            DataikuAPI.security.prepareUpdateGroup($scope.group).success(function(data) {
                Dialogs.confirmInfoMessages($scope, 'Confirm user edition', data, 'Are you sure you want to edit group "'+sanitize($scope.group.name) + '"?', true)
                    .then($scope.saveGroup);
            }).error(setErrorInScope.bind($scope));
        }
    };

    // Create or update a group
    $scope.saveGroup = function() {
        if ($scope.creation) {
            DataikuAPI.security.createGroup($scope.group).success(function(data) {
                $state.go("admin.security.groups.list");
            }).error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.security.updateGroup($scope.group).success(function(data) {
                $state.go("admin.security.groups.list");
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.groupIsDirty = function() {
        return !angular.equals(savedGroup, $scope.group);
    };

    var getGeneralSettings = function() {
        DataikuAPI.admin.getGeneralSettings().success(function(gs) {
            $scope.generalSettings = gs;
        }).error(setErrorInScope.bind($scope));
    }

    // Force the required state of external group mappings
    $scope.$watch('[group.sourceType, group.ldapGroupNames, group.azureADGroupNames, group.ssoGroupNames, group.customGroupNames]', () => {
        const invalidGroupMappingRequiredState =
            !$scope.group ||
            ($scope.group.sourceType === 'LDAP' && ($scope.group.ldapGroupNames || []).length === 0) ||
            ($scope.group.sourceType === 'AZURE_AD' && ($scope.group.azureADGroupNames || []).length === 0) ||
            ($scope.group.sourceType === 'LOCAL_NO_AUTH' && ($scope.group.ssoGroupNames || []).length === 0) ||
            ($scope.group.sourceType === 'CUSTOM' && ($scope.group.customGroupNames || []).length === 0);
        $scope.groupDescriptionForm.$setValidity('ldapGroupMappingRequired', !invalidGroupMappingRequiredState);
    });

    // Init
    getGeneralSettings();
});

app.controller("ExternalUsersController", function($scope, DataikuAPI, Dialogs, ListFilter, FutureProgressModal, FutureWatcher) {
    // filter and pagination of the user table
    $scope.selection = { orderQuery: "userAttributes.login", orderReversed: false };
    $scope.pagination = new ListFilter.Pagination([], 50);

    $scope.fetchUsers = function() {
        let groupName = $scope.groupName ? $scope.groupName : $scope.groupNameSelect;
        DataikuAPI.admin.users.fetchUsers($scope.userSourceType, {login: $scope.login, email: $scope.email, groupName: groupName}).success(function(initialResponse) {
            FutureProgressModal.show($scope, initialResponse, "Fetching users...", null, 'static', false, true).then(function(result) {
                $scope.users = result.users;
                refreshPage();
                if(result.unusedFilters && result.unusedFilters.length > 0) {
                    Dialogs.ack($scope, "⚠️ Warning: Incompatible source type", "This source type did not use the following filters: " + result.unusedFilters.join(", "));
                }
            });
        }).error(setErrorInScope.bind($scope));
    };

    function fetchUsersNoModal() {
        let groupName = $scope.groupName ? $scope.groupName : $scope.groupNameSelect;
        DataikuAPI.admin.users.fetchUsers($scope.userSourceType, {login: $scope.login, groupName: groupName}).success(function(initialResponse) {
            FutureWatcher.watchJobId(initialResponse.jobId).success(function(data) {
                $scope.users = data.result.users;
                refreshPage();
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    }

    $scope.provisionUsers = function(selectedUsers) {
        let usersToSync = selectedUsers.filter(u => u.status === 'NOT_PROVISIONED');
        if (usersToSync.length > 0) {
            DataikuAPI.admin.users.provisionUsers($scope.userSourceType, usersToSync).success(function(data) {
                FutureProgressModal.show($scope, data, "Provision users...", null, 'static', false, true).then(function(result) {
                    let title = "Provisioning result";
                    if (result.maxSeverity !== 'INFO') {
                        let summaryMessage = result.messages.find(m => m.code === 'INFO_SECURITY_SUPPLIER_PROVISIONING_SUMMARY');
                        if(summaryMessage) {
                            const index = result.messages.indexOf(summaryMessage);
                            result.messages.splice(index, 1);
                            title = summaryMessage.details;
                        }
                    }
                    Dialogs.infoMessagesDisplayOnly($scope, title, result, result.futureLog);
                    fetchUsersNoModal();
                })
            }).error(setErrorInScope.bind($scope));
        } else {
            let msg = `<div>All selected users are already provisioned</div>`;
            Dialogs.error($scope, `Failed to provision selected users.`, msg);
        }
    }

    $scope.syncUsers = function(selectedUsers) {
        let usersToSync = selectedUsers.filter(u => u.status === 'UNSYNCED').map(u => { return u.userAttributes.login});
        if (usersToSync.length > 0) {
            DataikuAPI.admin.users.syncUsers(usersToSync).success(function(data) {
                FutureProgressModal.show($scope, data, "Syncing users...", null, 'static', false, true).then(function(result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Sync result", result, result.futureLog);
                    fetchUsersNoModal();
                })
            }).error(setErrorInScope.bind($scope));
        } else {
            let msg = `<div>All selected users cannot be synced (not yet provisionned or already synced)</div>`;
            Dialogs.error($scope, `Failed to sync selected users.`, msg);
        }
    }

    $scope.fetchGroups = function() {
        $scope.enableGroups = true;
        $scope.enableEmail = $scope.userSourceType == "AZURE_AD";
        if ($scope.userSourceType == "LDAP") {
            $scope.enableGroups = $scope.generalSettings.ldapSettings.enableGroups;
        }
        $scope.users = [];
        refreshPage();
        if ($scope.enableGroups) {
            DataikuAPI.admin.users.fetchGroups($scope.userSourceType).success(function(data) {
                FutureProgressModal.show($scope, data, "Fetching groups for filtering...", null, 'static', false, true).then(function(result) {
                    $scope.groups = result;
                })
            }).error(setErrorInScope.bind($scope));
        }
    }

    // Used to change the page displayed when clicking on number or arrows
    $scope.$watch('pagination.page', () => {
        $scope.pagination.update();
    });

    // Filter/sorting is done by filtered-multi-select-rows, just refresh the pagination
    $scope.$watch('selection.filteredObjects', () => {
        refreshPage();
    });

    function refreshPage() {
        $scope.pagination.updateAndGetSlice($scope.selection.filteredObjects);
    };

    var getGeneralSettings = function() {
        DataikuAPI.admin.getGeneralSettings().success(function(gs) {
            $scope.generalSettings = gs;
        }).error(setErrorInScope.bind($scope));
    }

    // Init
    getGeneralSettings();

    $scope.fetchableSourceTypes = [];
    DataikuAPI.admin.users.getFetchableSourceTypes().success(function(data) {
        $scope.fetchableSourceTypes = data;
    }).error(setErrorInScope.bind($scope));
});

app.directive("globalPermissionsEditor", function() {
    return {
        scope: {
            'permissions': '='
        },
        templateUrl: '/templates/admin/security/global-permissions-editor.html',
        link: function($scope) {
            $scope.$watch("permissions", function(nv, ov) {
                if (!nv) return;
                /* Handle implied permissions */
                nv.$mayCreateProjectsFromMacrosDisabled = false;
                nv.$mayCreateProjectsFromTemplatesDisabled = false;

                if (nv.mayCreateProjects || nv.admin) {
                    nv.$mayCreateProjectsFromMacrosDisabled = true;
                    nv.$mayCreateProjectsFromTemplatesDisabled = true;
                }
            }, true);
        }
    }
});

app.controller("AdminSecurityAuditBufferController",function($scope, $state, $stateParams, DataikuAPI, $route, TopNav, Dialogs) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {
        includeAllCalls: false
    }

    $scope.refreshList = function(){
        DataikuAPI.security.getAuditBuffer($scope.uiState.includeAllCalls).success(function(data) {
            $scope.auditBuffer = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.refreshList();
});

app.filter('auditBufferEventDetails', function(Logger) {
    return function(obj) {
        var sa = []
        Object.keys(obj).forEach(function(x) {
            if (x != "callPath" && x != "msgType" && x != "authSource" && x != "authUser"
                && x != "clientIP" && x != "originalIP") {
                    let v = obj[x];
                    if (typeof v === 'object'){
                        try {
                            v = JSON.stringify(v);
                        } catch (e) {
                            Logger.debug("could not stringify key: " + x);
                        }
                    }
                    sa.push(x + ": " + v);
            }
        });
        return sa.join(", ");
    };
});


app.controller("GlobalPublicAPIKeysController", function ($scope, $state, DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.refreshApiKeysList = function () {
        DataikuAPI.admin.publicApi.listGlobalKeys().success(function (data) {
            $scope.apiKeys = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshApiKeysList();

    $scope.deleteGlobalKey = function (keyId) {
        Dialogs.confirm($scope, "Remove API key", "Are you sure you want to remove this API key?").then(function () {
            DataikuAPI.admin.publicApi.deleteGlobalKey(keyId).success(function (data) {
                $scope.refreshApiKeysList();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.viewQRCode = function (key) {
        CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
            newScope.apiKeyQRCode = JSON.stringify({
                k : key.key,
                u : $scope.appConfig.dssExternalURL
            });
        });
    };

    $scope.isExpired = function (key) {
        return key.expiresOn != 0 && key.expiresOn < new Date().getTime();
    };

});


app.controller("EditGlobalPublicAPIKeyController", function ($scope, $state, DataikuAPI, TopNav, $stateParams, CreateModalFromTemplate, ClipboardUtils, Dialogs) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    let oldKey;

    // Unlike for users, we include non-local groups here
    DataikuAPI.security.listGroups(false).success(function(data) {
        if (data) {
            data.sort();
        }
        $scope.allGroups = data;
    }).error(setErrorInScope.bind($scope));

    if ($stateParams.id) {
        $scope.creation = false;
        DataikuAPI.admin.publicApi.getGlobalKey($stateParams.id).success(function(data) {
            $scope.apiKey = data;
            oldKey = Object.assign({}, data)
        }).error(setErrorInScope.bind($scope));
    } else {
        $scope.creation = true;
        $scope.apiKey = {
            label : "New key",
            groups : []
        };
    }

    $scope.migrateToGroups = function() {
        if ($scope.apiKey) {
            $scope.apiKey.groups = [];
        }
    }

    $scope.create = function () {
        DataikuAPI.admin.publicApi.createGlobalKey($scope.apiKey).success(function (data) {
            CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                newScope.hashedApiKeysEnabled = $scope.appConfig.hashedApiKeysEnabled;
                newScope.key = data;
                newScope.uiSref = "admin.security.globalapi.list";

                newScope.copyKeyToClipboard = function() {
                    ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                };

                newScope.viewQRCode = function() {
                    CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                        newScope.apiKeyQRCode = JSON.stringify({
                            k : data.key,
                            u : $scope.appConfig.dssExternalURL
                        });
                    });
                };

                newScope.$on("$destroy",function() {
                    $state.go(newScope.uiSref);
                });
            });
        }).error(setErrorInScope.bind($scope));
    };

    const doSave = function() {
        DataikuAPI.admin.publicApi.saveGlobalKey($scope.apiKey).success(function (data) {
            $state.go("admin.security.globalapi.list");
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function () {
        if ((oldKey.groups === null || typeof (oldKey.groups) === 'undefined') && ($scope.apiKey.groups || $scope.apiKey.groups === [])) {
            Dialogs.confirmAlert($scope, 'Switch to group-based permissions', 'API key permissions will now solely be determined by group membership.', "Any previously defined permissions will be deleted.", 'WARNING').then(function() {
                doSave();
            });
        } else {
            doSave();
        }
    };
});

app.controller("AdminPersonalPublicAPIKeysController", function ($scope, $state, translate, DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.refreshApiKeysList = function () {
        DataikuAPI.admin.publicApi.listPersonalKeys().success(function (data) {
            $scope.apiKeys = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.isExpired = function (key) {
        return key.expiresOn != 0 && key.expiresOn < new Date().getTime();
    };

    $scope.refreshApiKeysList();

    $scope.editPersonalAPIKey = function (apiKey) {
        CreateModalFromTemplate("/templates/admin/security/personal-api-key-modal.html", $scope, null, function (newScope) {
            newScope.apiKey = {
                ...apiKey
            };
            newScope.creation = false;
        });
    };

    $scope.deletePersonalAPIKey = function (keyId) {
        Dialogs.confirm($scope,  translate("PROFILE.API_KEYS.REMOVE_DIALOG.TITLE", "Remove API key"), translate("PROFILE.API_KEYS.REMOVE_DIALOG.MESSAGE", "Are you sure you want to remove this API key?")).then(function () {
            DataikuAPI.admin.publicApi.deletePersonalKey(keyId).success(function (data) {
                $scope.refreshApiKeysList();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.viewQRCode = function (key) {
        CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
            newScope.apiKeyQRCode = JSON.stringify({
                k : key.key,
                u : $scope.appConfig.dssExternalURL
            });
        });
    };
});

app.controller("EditPersonalAPIKeyModalController", function($scope, DataikuAPI, CreateModalFromTemplate, ClipboardUtils, WT1) {
    $scope.create = function(){
        WT1.event("user-profile-create-API-key", {});
        DataikuAPI.profile.createPersonalAPIKey($scope.apiKey).success(function(data) {
            $scope.refreshApiKeysList();
            CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                newScope.hashedApiKeysEnabled = $scope.appConfig.hashedApiKeysEnabled;
                newScope.key = data;

                newScope.copyKeyToClipboard = function() {
                    ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                    $scope.dismiss();
                };

                newScope.viewQRCode = function() {
                    CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                        newScope.apiKeyQRCode = JSON.stringify({
                            k : data.key,
                            u : $scope.appConfig.dssExternalURL
                        });
                    });
                };

                newScope.$on("$destroy", function(){
                    $scope.dismiss();
                });
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function(apiKey) {
        WT1.event("user-profile-edit-API-key", {});
        DataikuAPI.profile.editPersonalAPIKey(apiKey).success(function(data) {
            $scope.dismiss();
            $scope.refreshApiKeysList();
        }).error(setErrorInScope.bind($scope));
    };
});

})();
