(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardsCommonController', function($scope, TileUtils, $rootScope, TopNav, DataikuAPI, ActivityIndicator, CreateModalFromTemplate, $state, WT1, DashboardUtils) {
        function makeDashboardListed(dashboard, noNotification) {
            return DataikuAPI.dashboards.makeListed(dashboard.projectKey, [dashboard.id], !dashboard.listed)
                .success(function(data) {
                    if (!noNotification) {
                        ActivityIndicator.success('Saved!');
                    }
                    dashboard.listed = !dashboard.listed;
                    if ($scope.origDashboard) {
                        $scope.origDashboard.listed = dashboard.listed;
                    }
                    $scope.$broadcast('objectTimelineChanged');
                }).error(setErrorInScope.bind($scope));
        }

        function findFilterTiles(pages) {
            const filterTilesMap = new Object();
            pages && pages.forEach(page => {
                const filterTile = page.grid.tiles.find(tile => TileUtils.isFilterTile(tile));
                if (filterTile) {
                    filterTilesMap[page.id] = filterTile;
                }
            });
            return filterTilesMap;
        }

        $scope.DashboardUtils = DashboardUtils;

        $scope.canEditDashboard = function(dashboard) {
            return dashboard && $scope.canWriteDashboards() && ($scope.canModerateDashboards() || dashboard.owner == $scope.appConfig.login);
        };

        $scope.toggleDashboardListed = function(dashboard, closeToggle) {
            if (!dashboard.listed && dashboard.hasMissingReaderAuthorizations) {
                CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                    newScope.initForDashboards([dashboard], true);
                }).then(function() {
                    makeDashboardListed(dashboard, true).success($scope.list);
                });
            } else {
                if (closeToggle) {
                    $('.tooltip').remove();
                }
                makeDashboardListed(dashboard, true);
            }
        };

        $scope.makeDashboardListed = makeDashboardListed;

        $scope.openInsightAccessModal = function(dashboard) {
            CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                newScope.initForDashboards([dashboard], false);
            }).then($scope.list);
        };


        $scope.copy = function(dashboard, callBackFunc) {
            CreateModalFromTemplate('/templates/dashboards/copy-dashboard-modal.html', $scope, 'CopyDashboardModalController', function(newScope) {
                newScope.init(dashboard);
            })
                .then(function() {
                    if (typeof callBackFunc === 'function') {
                        callBackFunc();
                    }
                });
        };


        $scope.exportDashboard = function(massExport, fromDashboardViewOrEdit = false, dashboardNotSaved = false) {
            CreateModalFromTemplate('/templates/dashboards/export-dashboard-modal.html', $scope, 'ExportDashboardModalController', function(newScope) {
                if (fromDashboardViewOrEdit) {
                    if ($scope.dashboard.pages.length !== 1) {
                        newScope.showCheckbox = true;
                        newScope.pageIdx = newScope.uiState.currentPageIdx;
                    }
                    newScope.dashboardNotSaved = dashboardNotSaved;
                }

                const fromByStateName = {
                    'projects.project.dashboards.dashboard.edit': 'EDIT',
                    'projects.project.dashboards.dashboard.view': 'VIEW',
                    'projects.project.dashboards.list': 'LIST'
                };
                const options = {
                    from: fromByStateName[$state.current.name],
                    massExport
                };
                newScope.init($scope.dashboard, options);
            });
        };

        $scope.share = function() {
            CreateModalFromTemplate('/templates/dashboards/share-dashboard-modal.html', $scope, 'ShareDashboardModalController', function(newScope) {
                newScope.init($scope.dashboard);
            });
        };

        $scope.saveAndCopy = function(dashboard, callBackFund) {
            $scope.saveDashboard().then(function() {
                $scope.copy($scope.dashboard);
            });
        };

        $scope.saveCustomFields = function(newCustomFields) {
            WT1.event('custom-fields-save', { objectType: 'DASHBOARD' });
            const oldCustomFields = angular.copy($scope.dashboard.customFields);
            $scope.dashboard.customFields = newCustomFields;
            return $scope.saveDashboard().then(function() {
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.dashboard.customFields);
            }, function() {
                $scope.dashboard.customFields = oldCustomFields;
            });
        };

        $scope.editCustomFields = function() {
            if (!$scope.dashboard) {
                return;
            }
            const modalScope = angular.extend($scope, { objectType: 'DASHBOARD', objectName: $scope.dashboard.name, objectCustomFields: $scope.dashboard.customFields });
            CreateModalFromTemplate('/templates/taggable-objects/custom-fields-edit-modal.html', modalScope).then(function(customFields) {
                $scope.saveCustomFields(customFields);
            });
        };

        $scope.deleteUnusedFiltersInsights = function() {
            /*
             * Delete previous dashboard filters when needed :
             * -> if a new filter insight has replaced a previous one from slide; or if it was on a slide that has been deleted.
             */
            const oldFilterTiles = findFilterTiles($scope.origDashboard.pages);
            const newFilterTiles = $scope.currentFilterTiles || findFilterTiles($scope.dashboard.pages);
            for (const [key, oldFilterTile] of Object.entries(oldFilterTiles)) {
                const isDeleted = !newFilterTiles[key];
                const isReplaced = newFilterTiles[key] && newFilterTiles[key].insightId !== oldFilterTile.insightId;
                if (isDeleted || isReplaced) {
                    const deletionRequests = [{
                        type: 'INSIGHT',
                        projectKey: $scope.dashboard.projectKey,
                        id: oldFilterTile.insightId,
                        displayName: oldFilterTile.name
                    }];
                    DataikuAPI.taggableObjects.delete(deletionRequests, $scope.dashboard.projectKey).error(setErrorInScope.bind($scope));
                }
            };
        };

        $scope.saveDashboard = function(commitMessage) {
            $scope.currentFilterTiles = findFilterTiles($scope.dashboard.pages);
            if (Object.keys($scope.currentFilterTiles).length) {
                WT1.event('dashboard-filters-save');
            }

            $scope.$broadcast('dashboardSaved');

            return DataikuAPI.dashboards.save($scope.dashboard, commitMessage)
                .success(function(data) {
                    data.pages.forEach(function(page, i) {
                        $scope.dashboard.pages[i].id = page.id;
                    });
                    $scope.dashboard.versionTag = data.versionTag;

                    if ($scope.origDashboard) {
                        $scope.deleteUnusedFiltersInsights();
                    }

                    $scope.origDashboard = angular.copy($scope.dashboard);
                }).error(setErrorInScope.bind($scope));
        };

        $scope.$on('saveDashboard', () => $scope.saveDashboard());

        const unregisterDashboardEditActionRequestEvent = $rootScope.$on('dashboard-edit-action-request', (_, options) => {
            $scope.editActionRequested = options;
        });

        $scope.$on('$destroy', () => {
            unregisterDashboardEditActionRequestEvent();
        });
    });
})();
