from typing import Any, Optional
from collections.abc import Callable

import pinecone

import dataiku
from dataiku.base.utils import package_is_at_least_no_import
from dataiku.core.vector_stores.dku_vector_store import DkuRemoteVectorStore, logger
from dataiku.core.vector_stores.vector_store_document_filter import MongoDBLikeVectorStoreDocumentFilter
from dataiku.llm.types import RetrievableKnowledge
from dataikuapi.dss.admin import DSSConnectionInfo
from dataikuapi.dss.langchain import DKUEmbeddings

try:
    from langchain_pinecone import PineconeVectorStore as LangchainPinecone  # type: ignore
except ImportError:
    from langchain_community.vectorstores import Pinecone as LangchainPinecone  # type: ignore


class PineconeV2VectorStore(DkuRemoteVectorStore):

    def __init__(self, kb: RetrievableKnowledge, exec_folder: str, connection_info_retriever: Callable[[str], DSSConnectionInfo]):
        kb["resolvedIndexName"] = kb["pineconeIndexName"]  # TODO @rag Standardise pinecone to work the same as ElasticSearch with index name resolution
        super(PineconeV2VectorStore, self).__init__(kb, exec_folder, connection_info_retriever)
        self.pinecone_index_name = kb["pineconeIndexName"]
        self.pinecone_connection_info: Optional[DSSConnectionInfo] = None
        self.document_filter = MongoDBLikeVectorStoreDocumentFilter(self.metadata_column_type)

    def init_connection(self) -> None:
        self.pinecone_connection_info = self.connection_info_retriever(self.connection_name)

    def get_db(self, embeddings: DKUEmbeddings, allow_creation:bool = False, **kwargs: Any) -> LangchainPinecone:
        index = self.get_pinecone_index()
        if package_is_at_least_no_import("langchain-pinecone", "0.1"):
            embed = embeddings
        else:
            embed = embeddings.embed_query
        db = LangchainPinecone(index, embed, "text")

        # allow_creation param unused: We always require the Pinecone index to have been created already
        db.get_pinecone_index(self.pinecone_index_name)  # This raises an exception with a meaningful message if the index does not exist

        return db

    def get_pinecone_index(self) -> pinecone.Index:
        assert self.pinecone_connection_info is not None, "Pinecone connection is not initialized"
        environment = self.pinecone_connection_info.get_params().get("env", None)

        if not environment and self.pinecone_connection_info.get_params().get("version", "") == "POST_APRIL_2024":
            raise ValueError("Selected pinecone connection version is incompatible with the selected code env\n"
                             "We recommend updating your code env to include a version of pinecone > 6.0.0 or pinecone-client > 3.0.0 (deprecated)")

        pinecone.init(api_key=self.pinecone_connection_info["resolvedAPIKey"], environment=self.pinecone_connection_info.get_params()["env"])
        return pinecone.Index(self.pinecone_index_name)

    def clear_index(self) -> None:
        index = self.get_pinecone_index()
        index.delete(delete_all=True)
        logger.info("Cleared Pinecone index {}".format(self.pinecone_index_name))

    def check_connection(self):
        # TODO @rag Overriding this method as a temporary workaround because we don't have connection.get_settings().allow_knowledge_banks for old pinecone connections
        #           Really need a migration to add this
        if self.connection_name is None:
            raise ValueError("You must provide a connection to be used with the knowledge bank.")
