# -*- coding: utf-8 -*-
from dataiku.doctor.docgen.renderer.renderer_helpers import RendererHelpers
from dataiku.doctor.docgen.extractor.docx_parser import DocxParser
from dataiku.doctor.docgen.renderer.widget import Table, Text, PuppeteerContent
import logging

logger = logging.getLogger(__name__)


class Renderer(object):
    """
    This is the renderer for the MDG workflow.
    It handles a two-step machanism :
    - takes as input resolved conditional placeholders, and keep / removes them based on their value
    - takes all other placeholders, and replaces them by their adequate values.
    """

    def __init__(self):
        self.docx_parser = DocxParser()
        self.helpers = RendererHelpers()

    def __replace_simple_placeholders__(self, document, placeholders, context, font_reference):
        """
        Replace placeholders inside the document by their content defined in the variable placeholders
        :param Union[Document, _Cell] document: the current document (or sub-document) that contains paragraph with placeholders
        :param Placeholder placeholders: data representing the location of a placeholder
        :param Dict[Any, Union[Table, Text, PuppeteerContent]] context: location and initial data of a placeholder
        :param str font_reference: the input docx it will be use as a reference for font
        """
        # keep the reference to a paragraph when it was sliced
        for ph in placeholders:
            if ph.tagname in context:
                widget = context[ph.tagname]
                self.helpers.replace_simple_placeholder(document, ph, widget, font_reference)
            else:
                logger.warn("No corresponding value found for %s", ph.tagname)

    def __replace_placeholders_in_table__(self, document, context):
        """
        Replace placeholders inside the tables present in a document by their content
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param Dict[Any, Union[Table, Text, PuppeteerContent]] context: location and initial data of a placeholder
        """
        for table_placeholder in self.docx_parser.parse_table(document):
            cell = table_placeholder.table.cell(table_placeholder.row_idx, table_placeholder.col_idx)
            (blocks, singles) = self.docx_parser.extract_blocks(table_placeholder.placeholders)
            self.__replace_simple_placeholders__(cell, list(reversed(singles)), context, document)
            self.__replace_block_placeholders__(cell, blocks, context)

    def __replace_placeholders_in_header__(self, document,  context):
        """
        Replace placeholders inside the header of the different sections by their content
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param Dict[Any, Union[Table, Text, PuppeteerContent]] context: location and initial data of a placeholder
        """
        header_placeholder = self.docx_parser.parse_headers(document)
        for i in range(len(document.sections)):
            header_placeholder_filtered = [p for p in header_placeholder if p.section_number == i]
            self.__replace_simple_placeholders__(document.sections[i].header,
                                                 list(reversed(header_placeholder_filtered)), context, document)

    def __replace_placeholders_in_footer__(self, document, context):
        """
        Replace placeholders inside the footer of the different sections by their content
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param Dict[Any, Union[Table, Text, PuppeteerContent]] context: location and initial data of a placeholder
        """
        footer_placeholder = self.docx_parser.parse_footers(document)
        for i in range(len(document.sections)):
            footer_placeholder_filtered = [p for p in footer_placeholder if p.section_number == i]
            self.__replace_simple_placeholders__(document.sections[i].footer,
                                                 list(reversed(footer_placeholder_filtered)), context, document)

    def __replace_block_placeholders__(self, document, blocks, context):
        """
        Replace pair of placeholders inside the document by their content
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param [BlockPlaceholder] blocks: a list of BlockPlaceholder
        :param Dict[Any, Union[Table, Text, PuppeteerContent]] context: location and initial data of a placeholder
        """
        for block in blocks:
            if block.extract_name() in context:
                widget = context[block.extract_name()]
                self.helpers.replace_block_placeholder(document, block, widget)

    @staticmethod
    def __is_valid_condition__(condition, widget_value):
        if "==" in condition:
            expected_value = condition.split('==')[-1].strip()
            return widget_value == expected_value
        else: # !=
            expected_value = condition.split('!=')[-1].strip()
            return widget_value != expected_value

    def __replace_conditional_placeholders__(self, document, placeholders, context):
        """
        Replace pair of placeholders inside the document by their content
        :param Document document: the current document that contains tables which will contains paragraph with placeholders
        :param [BlockPlaceholder] placeholders: a list of Conditional placeholders
        """
        for placeholder in placeholders:
            tagname = placeholder.start.extract_name()
            if tagname in context:
                widget = context[tagname]
            else:
                widget = None
            if widget is None or not isinstance(widget, Text) or \
                    not self.__is_valid_condition__(placeholder.start.tagname, widget.inner_text):
                # Not a valid condition, remove everything
                self.helpers.remove_block_placeholder(document, placeholder)
            else:
                self.helpers.strip_block_placeholder(placeholder)

    def render(self, puppeteer_extractor, document, resolved_placeholders):
        self.docx_parser.debug(document)

        placeholder_context = self.__get_placeholder_context__(resolved_placeholders, puppeteer_extractor)

        # parse updated document to locate the remaining placeholders
        placeholders = self.docx_parser.parse_text(document)
        (blocks, singles) = self.docx_parser.extract_blocks(placeholders)
        logger.debug("BLOCKS:")
        for b in blocks:
            logger.debug("\t%s", b)
        logger.debug("SINGLES:")
        for s in singles:
            logger.debug("\t%s", s)

        # render placeholders inside tables first, to avoid parsing every cell of tables generated from placeholders
        self.__replace_placeholders_in_table__(document, placeholder_context)

        # reverse the order of the singles list, in order to always reach the expecting element,
        # even if a previous element remove some characters of the document
        self.__replace_simple_placeholders__(document, list(reversed(singles)), placeholder_context, document)
        self.__replace_block_placeholders__(document, blocks, placeholder_context)

        self.__replace_placeholders_in_header__(document, placeholder_context)
        self.__replace_placeholders_in_footer__(document, placeholder_context)

    def resolve_conditional_placeholder(self, document, resolved_placeholders):
        """
        Resolve any conditional placeholder. If the conditional placeholder is valid, keep its content, otherwise remove everything.
        :param Document document: the input document
        :param resolved_placeholders: the value of the different conditional placeholders
        :return: the input document with the text updated to remove conditional placeholder
        """
        self.docx_parser.debug(document)

        placeholder_context = self.__get_placeholder_context__(resolved_placeholders)

        # parse raw document and remove conditional placeholders
        placeholders = self.docx_parser.parse_text(document)
        conditionals = self.docx_parser.extract_conditionals_placeholders(placeholders)
        logger.debug("CONDITIONALS:")
        for c in conditionals:
            logger.debug("\t%s", c)
        self.__replace_conditional_placeholders__(document, conditionals, placeholder_context)


        for table_placeholder in self.docx_parser.parse_table(document):
            cell = table_placeholder.table.cell(table_placeholder.row_idx, table_placeholder.col_idx)
            table_conditionals = self.docx_parser.extract_conditionals_placeholders(table_placeholder.placeholders)
            self.__replace_conditional_placeholders__(cell, table_conditionals, placeholder_context)
        return document

    @staticmethod
    def __get_placeholder_context__(resolved_placeholders, puppeteer_extractor = None):
        placeholder_context = {}
        for resolved_placeholder in resolved_placeholders["resolved"]:
            placeholder_id = resolved_placeholder["name"]
            placeholder_type = resolved_placeholder["type"]
            value = resolved_placeholder["value"]

            if placeholder_type in ["JSON_TABLE", "MLFLOW_MODEL_VERSION_INFO_TABLE"]:
                placeholder_context[placeholder_id] = Table(value)
            elif placeholder_type in ["JSON_TEXT", "MLFLOW_MODEL_VERSION_INFO_FIELD"]:
                placeholder_context[placeholder_id] = Text(value)
            elif placeholder_type == "PUPPETEER":
                steps = value["stepList"]
                puppeteer_config_name = value["puppeteerConfigName"]

                for step in steps:
                    step_type = step["type"]
                    if step_type == "TEXT_EXTRACTION" or step_type == "SCREENSHOT" or step_type == "JSON_EXTRACTION":
                        placeholder_context[placeholder_id] = PuppeteerContent(puppeteer_extractor, puppeteer_config_name)
            else:
                logger.warn("Unsupported resolved placeholder type: %s for %s" % (placeholder_type, placeholder_id))

        logger.debug("Placeholder names: %s", placeholder_context.keys())
        return placeholder_context
