import os
import datetime, time
import subprocess
from abc import ABC
from pathlib import Path
from itertools import chain

from cleaning import delete_and_report


class LockedLogsDeletionMacro(ABC):
    OBJECT_ROOT_FOLDER_NAME = None
    OBJECT_TYPE_NAME = None
    ONLY_KERNEL_TYPE = None

    def __init__(self, project_key, config, plugin_config):
        self.project_key = project_key
        self.config = config

    def get_progress_target(self):
        return 100, 'NONE'

    def run(self, progress_callback):
        dip_home = os.environ['DIP_HOME']
        perform_deletion = self.config.get("performDeletion", False)
        maximum_age = int(self.config.get("age", 15))
        maximum_timestamp = int(time.mktime((datetime.datetime.now() - datetime.timedelta(days=maximum_age)).timetuple()))

        objects_root_dir = os.path.join(dip_home, self.OBJECT_ROOT_FOLDER_NAME)

        if self.config.get('allProjects', False):
            project_keys = [project_key for project_key in os.listdir(objects_root_dir)]
        else:
            project_keys = [self.project_key]

        to_delete = []
        for project_key in project_keys:
            # Step 1: find logs directories
            project_objects_folder = Path(os.path.join(objects_root_dir, project_key))
            runs_dirs = project_objects_folder.rglob('**/runs')
            dev_runs_dirs = project_objects_folder.rglob('**/dev-runs')
            parent_run_dirs = chain(runs_dirs, dev_runs_dirs)
            candidate_folders = []
            for parent_dir in parent_run_dirs:
                if parent_dir.is_dir():
                    candidate_folders.extend([
                        run_folder for run_folder in parent_dir.iterdir() if run_folder.is_dir()
                    ])
            if not candidate_folders:
                continue

            # Step 2: for each candidate folder, check if all its contents are old and not locked
            for folder in sorted(candidate_folders):
                # check that the folder is not excluded
                if self.ONLY_KERNEL_TYPE and not folder.name.startswith(self.ONLY_KERNEL_TYPE + '-') and not f"__{self.ONLY_KERNEL_TYPE}-" in folder.name:
                    continue

                is_folder_deletable = True

                for item in folder.rglob('*'):
                    if not item.is_file() or not str(item).endswith(".log"):
                        continue

                    # check that the file is old enough
                    if item.stat().st_mtime >= maximum_timestamp:
                        is_folder_deletable = False
                        break

                    # check that the file is not being used by a kernel
                    lock_file_path = None
                    try:
                        lock_file_path = str(item) + ".lock"
                        with open(lock_file_path, 'r') as f:
                            pid = int(f.read().strip())
                    except FileNotFoundError:
                        continue
                    except (IOError, ValueError):
                        lock_file_is_stale = True
                        print("failed to read pid from lock file")
                    else:
                        lock_file_is_stale = not is_process_alive(pid)

                    # ignore lock files for dead processes
                    if lock_file_is_stale:
                        if perform_deletion and lock_file_path:
                            os.remove(lock_file_path)
                    else:
                        is_folder_deletable = False
                        break

                if is_folder_deletable:
                    to_delete.append([project_key, folder.relative_to(project_objects_folder)])

        html = delete_and_report(to_delete, objects_root_dir, progress_callback, perform_deletion, self.OBJECT_TYPE_NAME, ['Project', 'Logs directory'])
        return html


def is_process_alive(pid):
    if os.name == 'posix':
        try:
            os.kill(pid, 0)  # sending signal 0 to a PID checks for its existence without harming it
            print("the process referenced by a lock file is alive")
            return True
        except OSError:
            return False
    elif os.name == 'nt':
        try:
            command = ['tasklist', '/fi', f'PID eq {pid}']
            output = subprocess.check_output(command, stderr=subprocess.DEVNULL).decode()
            if str(pid) not in output:
                return False
            else:
                return True
        except (subprocess.CalledProcessError, FileNotFoundError):
            return False
    else:
        print("unsupported OS, ignoring is_process_alive check")
        return False
