
$repo="https://cloud.r-project.org"

if (-not $env:DIP_HOME) {
    Write-Host "DIP_HOME is not defined, can't continue."
    Exit
}

$DIP_HOME=$env:DIP_HOME
$BINDIR = "$DIP_HOME\bin"
& $BINDIR/env-default.ps1

if ($null -eq $env:DKUINSTALLDIR) {
    Write-Host "FATAL: DKUINSTALLDIR is not defined. Please check $BINDIR\env-default.ps1"
    Exit
}

# . = call in same scope, to access functions
. $env:DKUINSTALLDIR/scripts/_startup.inc.ps1

bkdl_set_py_env
bkdl_load_env_files

if ($env:DKURBIN) {
    $DKURBIN=$env:DKURBIN
}
elseif (Get-Command "R.exe" -errorAction SilentlyContinue) {
    $DKURBIN=(Get-Command "R.exe").Source
} else {
    Write-Host "R is not installed, or not in PATH. Please install it, or add it to Path, and relaunch this command."
    Exit
}

if (-not $env:DKUPYTHONBIN) {
    $env:DKUPYTHONBIN="$env:DIP_HOME/pyenv/Scripts/python.exe"
}

$DKURLIB = "$env:DIP_HOME\R.lib"
Write-Host "$DKURLIB"
New-Item -Path "$DKURLIB" -Type Directory -ErrorAction Ignore

if ($env:R_LIBS -eq "") {
    $env:R_LIBS = "$DKURLIB"
} else {
    $env:R_LIBS = "$DKURLIB;$env:R_LIBS"
}

$env:R_LIBS_USER = "$DKURLIB"

Write-Host "[+] Installing required R packages into $DKURLIB"
$DKURLIB = $DKURLIB.Replace("\", "/")
@"
dependencies <- read.table(text="
    pkg             ver
    httr            1.2
    RJSONIO         1.3
    dplyr           0.5
    curl            2.4
    IRkernel        0.7.1
    sparklyr        0.5.1
    ggplot2         2.2.1
    gtools          3.5.0
    tidyr           0.6.1
    base64enc       0.1
    filelock        1.0.2
", header=TRUE, stringsAsFactors=FALSE)

checkPackages <- function() {
	message("Checking installed packages ...")
	installedVersions <- installed.packages(noCache=TRUE)[,'Version']
	l <- apply(dependencies, 1, function(x) {
		p <- x['pkg']
		v <- x['ver']
		if (is.na(installedVersions[p])) {
			message("Package not installed: ", p)
			p
		} else if (package_version(installedVersions[p]) < package_version(v)) {
			message("Package too old: ", p, " installed=", installedVersions[p] , " required=", v)
			p
		} else {
			NA
		}
	})
	na.omit(l)
}

toInstall <- checkPackages()
if (length(toInstall) > 0) {
	message("Installing packages: ", paste(toInstall, collapse=" "))
	install.packages(toInstall, "$DKURLIB", repos="$repo")
	if (length(checkPackages()) > 0) {
		stop("at least one package failed to install required version")
	}
}
"@ | & $DKURBIN --slave --no-restore

Write-Host "[+] Installing R kernel for Jupyter"
$KERNELSPEC_LOCATION=. $DKURBIN --slave --no-restore -e "cat(system.file('kernelspec', package='IRkernel'))"

$TMP_DIR = "$env:DIP_HOME\R.tmp"
New-Item -Path "$TMP_DIR" -Type Directory -ErrorAction Ignore *> $null

$env:JUPYTER_DATA_DIR = "$DIP_HOME/jupyter-run/jupyter"
$env:PYTHONPATH="$env:DKUINSTALLDIR/dku-jupyter/packages"

@"
import json, os, shutil, sys
from jupyter_client.kernelspecapp import KernelSpecApp

srcDir = sys.argv[1]
dstDir = os.path.join(sys.argv[2], 'ir')
rBin = sys.argv[3]

# Copy IRkernel kernelspec to tmp dir and patch DKURBIN into command line
shutil.copytree(srcDir, dstDir)
kernFile = os.path.join(dstDir, 'kernel.json')
with open(kernFile) as f:
	kernelDef = json.load(f)
kernelDef['argv'][0] = rBin
with open(kernFile, 'w') as f:
	json.dump(kernelDef, f, indent=2)

# Install kernel spec into Jupyter
# jupyter kernelspec install --user --replace --name ir DIR
sys.argv = [ '-', 'install', '--user', '--replace', '--name', 'ir', dstDir ]
sys.exit(KernelSpecApp.launch_instance())
"@ | & $env:DKUPYTHONBIN - "$KERNELSPEC_LOCATION" "$TMP_DIR" "$DKURBIN"

Remove-Item -Recurse -Force $TMP_DIR

Write-Host "[+] Done"
