import os
import io
import nbformat

def get_notebook_with_outputs_path(os_path):
    """Return path to notebook with cells outputs

    Given the path to a notebook (located in the config directory), return the path to
    its counterpart - containing the cells outputs - (located in the notebook_result directory).

    Parameters
    ----------
    os_path : string
        The OS path (absolute or relative) of the notebook without the cells outputs.

    Returns
    -------
    path : string
        The OS path (absolute or relative) of the notebook with the cells outputs
        or None is os_path does not reference a valid notebook file
    """
    parent_dir, notebook_filename = os.path.split(os.path.normpath(os_path))
    if parent_dir != '':
        grandparent_dir, parent_dirname = os.path.split(parent_dir)
        if parent_dirname == 'ipython_notebooks':
            projects_dir, project_key = os.path.split(grandparent_dir)
            if projects_dir != '':
                config_dir = os.path.split(projects_dir)[0]
                if config_dir != '':
                    dip_home_dir, config_dirname = os.path.split(config_dir)
                    if config_dirname == 'config':
                        cells_outputs_dir = os.path.join(dip_home_dir, "notebook_results", "jupyter", project_key)
                        return os.path.join(cells_outputs_dir, notebook_filename)
    return None

def read_notebook_file(notebook_path, as_version):
    """Read a notebook from a file

    Read a notebook from a file as a NotebookNode of the given version.

    Parameters
    ----------
    notebook_path : string
        The path (absolute or relative) of the notebook file, or None.

    as_version : int
        The version of the notebook format to return. The notebook will be converted, if necessary. 
        Pass nbformat.NO_CONVERT to prevent conversion.

    Returns
    -------
    nb : NotebookNode
        The notebook that was read or None if the specified path does not designate a file
    """
    if notebook_path is not None and os.path.isfile(notebook_path):
        with io.open(notebook_path, 'r', encoding='utf-8') as notebook_file:
            return nbformat.read(notebook_file, as_version=as_version)
    return None

def compare_notebooks(notebook, notebook_with_outputs):
    """Check whether two notebooks are identical, outputs set aside.

    Read a notebook from a file as a NotebookNode of the given version.

    Parameters
    ----------
    notebook : NotebookNode
        Original notebook, usually without cells outputs.

    notebook_with_outputs : NotebookNode
        Second notebook with cells outputs.

    Returns
    -------
    res : bool
        True if notebook is identical to notebook_with_outputs when its output cells are cleared 
    """
    # Compare the metadata
    if not notebook.metadata == notebook_with_outputs.metadata:
        return False

    # Compare the source cells
    if len(notebook.cells) != len(notebook_with_outputs.cells):
        return False
    for c1, c2 in zip(notebook.cells,notebook_with_outputs.cells):
        if c1.cell_type == c2.cell_type and c1.cell_type == 'code':
            # backup execution_count & outputs
            execution_count, outputs = c2.execution_count, c2.outputs
            # compare cells after clearing execution_count & outputs
            c2.execution_count, c2.outputs = 0, []
            cells_are_equals = c1 == c2
            # restore execution_count & outputs
            c2.execution_count, c2.outputs = execution_count, outputs
            if not cells_are_equals:
                return False
        else:
            if not c1 == c2:
                return False
    return True