(function() {
'use strict';

/**
 * This file groups functionalities for selecting and highlighting items in the flow graphs
 * (flow, inter project graph and job preview subgraph)
*/


const app = angular.module('dataiku.flow.project');


app.service('FlowGraphSelection', function($rootScope, FlowGraphFolding, TaggableObjectsUtils, FlowGraph, FlowGraphHighlighting) {
    /*
    * In all this service, "items" are flow items corresponding to graph nodes
    *
    * Several selection strategies (single items selectec at a time, etc)
    * Selection strategies are also responsible for adding highlighting
    * (not for removing it as for now as we always do before calling then)
    *
    * A selection strategy implements:
    * - onItemClick(nodeId, event)
    * - clearSelection()
    */
    const svc = this;

    // contains all selected nodes, including all duplicated objects node if they are in multiple flow-zones
    // however only one node of it has the field 'selected' set to true (later referred as flag-selected)
    // => calling FlowGraphSelection.getSelectedNodes will only get flag-selected nodes, so only one instance of each real item
    // => when applicable, the flag-selected node is the last selected by the user, so that reading its zone gives the zone from which it was selected (not always its source flow-zone)
    let selectedItems = [];

    function _isSelected(item) {
        return selectedItems.includes(item);
    }

    function _hasOnlyZonesOrNone(item) {
        return (_isZone(item) && selectedItems.find(elem => !_isZone(elem)) === undefined) || (!_isZone(item) && selectedItems.find(elem => _isZone(elem)) === undefined);
    }

    /*
    * Return true if all the currently selected objects are zones
    */
    function _hasOnlyZones() {
        return selectedItems.every(_isZone);
    }

    /*
    * Return true if all the currently selected objects are not zones
    */
    function _hasOnlyNonZones() {
        return !selectedItems.some(_isZone);
    }

    function _clearSelection() {
        const element = FlowGraph.getDOMElement();
        if (!element) return; // too early
        selectedItems.forEach(it => it.selected = false);
        FlowGraphHighlighting.removeAllZoneClusterHighlights();
        d3.select('#flow-graph').classed('has-selection', false);
        d3.selectAll('#flow-graph .zone_cluster.selected').attr('style', null);
        d3.selectAll('#flow-graph .selected').classed('selected', false);
        selectedItems = [];
    }

    function _addToSelection(items) {

        items = items.filter(it => !_isSelected(it));
        // For objects used in multiple zones, we mark as flag-selected the nodes just clicked
        // but in case of lasso-selection, it's possible multiple nodes for the same real object have been captured.
        // In this case, we flag-select one node. which one is semi-random but stable (depends on DOM order)
        _.uniqBy(items, it => it.realId).forEach(it => it.selected = true);

        var realIdsSelected = items.map(it => it.realId);
        selectedItems = selectedItems.filter(it => !it.realId || !realIdsSelected.includes(it.realId));
        if (!items.length) {
            return;
        }

        const selector = items.map(it => _isZoned(it) ? `svg [data-node-id="${it.realId}"]` : _isZone(it) ? `svg [id="cluster_${it.id}"]` : `svg [data-id="${it.id}"]`).join(', ');
        d3.selectAll(selector).each(function() {
            const type = this.getAttribute('data-type');
            let key = this.getAttribute('data-id');
            if (type === "ZONE") {
                key = key.replace("cluster_", "");
                FlowGraphHighlighting.highlightZoneCluster(this)
            }
            selectedItems.push(FlowGraph.node(key));
        }).classed("selected", true);

        if (selectedItems.length) {
            $('#flow-graph').addClass('has-selection');
        }
        FlowGraphHighlighting.removeHighlights();
    }

    function _isZoned(item) {
        return item.id && item.realId && item.id !== item.realId;
    }

    function _isZone(item) {
        return item && item.nodeType === 'ZONE';
    }

    function _removeFromSelection(item) {
        const items = _isZoned(item) ? selectedItems.filter(it => it.realId === item.realId) : [item];

        items.forEach(item => {
            item.selected = false;
            const index = selectedItems.indexOf(item);
            if (index > -1) {
                if (_isZone(item)) {
                    d3.select(`g[data-id="${item.id}"]`)[0][0].style = null;
                    FlowGraph.d3ZoneNodeWithId(item.id).classed("selected", false);
                } else {
                    FlowGraph.d3NodeWithId(item.id).classed("selected", false);
                }
                selectedItems.splice(index, 1);
            }
        });

        if (!selectedItems.length) {
            $('#flow-graph').removeClass('has-selection');
        }
    }

    /*
    * Return true if the newly selected object is of a type (zone or non-zoned) different that all currently selected objects
    */
    function _selectedObjectMixTypes(item) {
        if (selectedItems == 0) {
            return false;
        }
        const itemIsZone = _isZone(item);
        return (!itemIsZone && _hasOnlyZones()) || (itemIsZone && _hasOnlyNonZones())
    }

    /**
     * Selects one item.
     * Highlights predecessors and successors.
     */
    const defaultSingleSelectionStrategy = {
        onItemClick: function (item) {
            _clearSelection();
            _addToSelection([item]);
            if (item.filterRemove) {
                return;
            }
            FlowGraphHighlighting.highlightPredecessors(item);
            FlowGraphHighlighting.highlightSuccessors(item);
            if (!_isZone(item)) {
                FlowGraphHighlighting.highlightUsedZonesForSingleItem(item);
            }
            if (_hasOnlyZonesOrNone(item)) {
                FlowGraphHighlighting.highlightZoneElements(item.name);
            }
        }
    }

    /**
     * Selects multiple items at once.
     * Highlights predecessors and successors.
     */
    const defaultMultiSelectionStrategy = {
        onItemClick: function (item, evt) {
            if (_selectedObjectMixTypes(item)) {
                return;
            }
            if (!_isSelected(item)) {
                _addToSelection([item]);
            } else {
                _removeFromSelection(item);
            }
            if (new Set(selectedItems.map(item => item.realId)).size == 1) {
                // only one real element is selected, redirect to single click selection strategy
                return selectionStrategies.SINGLE.onItemClick(selectedItems[0], evt);
            } else {
                svc.recomputeHighlight();
            }
        }
    }

    function recomputeHighlight() {
        FlowGraphHighlighting.removeAllZoneClusterHighlights();
        FlowGraphHighlighting.removeAllEdgesBetweenZonesHighlights();

        const zoneList = selectedItems.filter(_isZone).map(it => it.name);
        zoneList.forEach(it => {
            FlowGraphHighlighting.highlightZoneById(it);
            FlowGraphHighlighting.highlightZoneElements(it);
        });

        const links = d3.selectAll('#flow-graph g');
        links.filter(function() {
            const from = d3.select(this).attr('data-from');
            const to = d3.select(this).attr('data-to');
            if(from === to) return false;  // Early return if they are the same
          
            // Remove the "zone_" prefix
            const fromId = from.replace('zone_', '');
            const toId = to.replace('zone_', '');
          
            // Check that both zone IDs are in the zoneList
            return zoneList.includes(fromId) && zoneList.includes(toId);
        }).classed('highlight', true); // highlight links between selected zones (the links between selected zone gets bolder)

        // Highlight the zone of all selected elements (even when it exists in multiple zones)
        const zoneIdList = selectedItems.filter(_isZoned)
            .map(it => it.id.toString().split('__')[1]);
        const zoneIdSet = new Set(zoneIdList);

        zoneIdSet.forEach(it => {
            FlowGraphHighlighting.highlightZoneById(it);
        });
    }

    svc.recomputeHighlight = recomputeHighlight;

    function withAllPredecessorsOrSuccessorsMultiItemSelectionStrategy(mode) {
        if (mode != 'predecessors' && mode != 'successors') {
            throw new Error("mode should be either 'predecessors' or 'successors'")
        }
        return {
            onItemClick: function(item, evt) {
                // For performance we list and then select them all at once:
                const itemsToSelect = this._listPredecessorsOrSuccessors(item).map(FlowGraph.node);
                _addToSelection(itemsToSelect);
            },

            _listPredecessorsOrSuccessors: function(item, list) {
                list = list || [];

                if (list.includes(item.id)) {
                    // Avoid loops
                    return list;
                }
                list.push(item.id);

                const that = this;

                if (_isZoned(item) && !(mode == "predecessors" && item.usedByZones.length == 0)) {
                    let zoneNode = FlowGraph.node("zone_" + (item.usedByZones[0] || item.ownerZone));
                    if (zoneNode) {
                        $.each(zoneNode[mode], function (index, otherZoneNodeId) {
                            const otherNodeId = graphVizEscape((mode == "predecessors" ? `zone_${item.ownerZone}` : otherZoneNodeId));
                            const otherNode = FlowGraph.node(otherNodeId + "__" + item.realId);
                            if (otherNode) {
                                list = that._listPredecessorsOrSuccessors(otherNode, list);
                            }
                        });
                    }
                }

                $.each(item[mode], function (index, otherNodeId) {
                    const otherNode = FlowGraph.node(otherNodeId);
                    list = that._listPredecessorsOrSuccessors(otherNode, list);
                });
                return list;
            }
        }
    }

    const selectionStrategies = {
        SINGLE: defaultSingleSelectionStrategy,
        MULTI: defaultMultiSelectionStrategy,
        MULTI_WITH_SUCCESSORS: withAllPredecessorsOrSuccessorsMultiItemSelectionStrategy('successors'),
        MULTI_WITH_PREDECESSORS: withAllPredecessorsOrSuccessorsMultiItemSelectionStrategy('predecessors')
    };

    let activeSingleSelectionStrategy = selectionStrategies.SINGLE;
    let activeMultiSelectionStrategy = selectionStrategies.MULTI;

    this.setSingleSelectionStrategy = strategy => {
        activeSingleSelectionStrategy = strategy;
    };

    this.setMultiSelectionStrategy = strategy => {
        activeMultiSelectionStrategy = strategy;
    };

    this.resetSelectionStrategies = () => {
        activeSingleSelectionStrategy = selectionStrategies.SINGLE;
        activeMultiSelectionStrategy = selectionStrategies.MULTI;
    };

    this.onRectangularSelection = function (nodeIds) {
        const nodes = FlowGraph.nodes(nodeIds).filter(node => !node.filterRemove);
        if (new Set(nodes.map(item => item.realId)).size === 1) {
            // only one real element is selected, redirect to single click selection strategy
            activeSingleSelectionStrategy.onItemClick(nodes[0]);
        } else {
            _clearSelection();
            _addToSelection(nodes);
            svc.recomputeHighlight();
        }
        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
    }

    this.onItemClick = function (item, evt) {
        if (evt && (evt.shiftKey || evt.metaKey || evt.ctrlKey)) {
            activeMultiSelectionStrategy.onItemClick(item, evt);
        } else {
            activeSingleSelectionStrategy.onItemClick(item, evt);
        }
        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
        $rootScope.$emit('flowItemClicked', evt, item);
    };

    this.selectNodesFromIds = function (nodeIds) {
        if(!nodeIds || nodeIds.length === 0) {
            return
        }
        _clearSelection();
        const nodes = FlowGraph.nodes(nodeIds).filter(node => angular.isDefined(node));
        if(nodes.length === 1) {
            activeSingleSelectionStrategy.onItemClick(nodes[0]);
        } else {
            nodes.forEach(item => {
                activeMultiSelectionStrategy.onItemClick(item);
            })
        }
        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
        nodes.forEach(item => {
            $rootScope.$emit('flowItemClicked', null, item);
        })
    }

    this.clearSelection = function() {
        if (selectedItems.length) {
            FlowGraphHighlighting.removeHighlights();
            _clearSelection();

            $rootScope.$emit('flowSelectionUpdated');
            $rootScope.$emit('flowDisplayUpdated');
        }
    };

    // We need to call this after a new serialized graph has been fetched and rendered
    this.refreshStyle = function(redoSelection = false) {
        const element = FlowGraph.getDOMElement();
        if (!element) {
            return; //Too early
        }
        d3.selectAll(element.find('svg .selected')).classed("selected", false);
        if (selectedItems.length) {
            // Selected items are nodes, if they are part of the old graph they should be replaced by their new version
            selectedItems = selectedItems.map(it => Object.assign({}, FlowGraph.node(it.id), {selected: it.selected})).filter(x => !!x && x.id);
            const selector = selectedItems.map(it => _isZoned(it) ? `svg [data-node-id="${it.realId}"]` : _isZone(it) ? `svg [id="cluster_${it.id}"]` : `svg [data-id="${it.id}"]`).join(', ');
            if (selector.length) {
                d3.selectAll(selector).each(function() {
                    const type = this.getAttribute('data-type');
                    if (type === "ZONE") {
                        FlowGraphHighlighting.highlightZoneCluster(this)
                    }
                }).classed("selected", true);
            }
            if (redoSelection) {
                const oldSelection = [...selectedItems.filter(it => it.selected)];
                const strategy = oldSelection.length > 1 ? selectionStrategies.MULTI : selectionStrategies.SINGLE;
                oldSelection.forEach(item => {
                    strategy.onItemClick(item); // Deselect in multi
                    if (oldSelection.length > 1) {
                        strategy.onItemClick(item); // Select in multi
                    }
                });
            }
        }
        $rootScope.$emit('flowSelectionUpdated');
    };

    function select(predicate1) {
        return function (predicate) {
            let toSelect = Object.values(FlowGraph.get().nodes);
            if (predicate1) {
                toSelect = toSelect.filter(predicate1);
            }
            if (predicate) {
                toSelect = toSelect.filter(predicate);
            }
            toSelect = toSelect.filter((node) => !FlowGraphFolding.isNodeFolded(node.id));
            _clearSelection();
            _addToSelection(toSelect);
            selectedItems.filter(_isZone).forEach(it => FlowGraphHighlighting.highlightZoneElements(it.name));
            $rootScope.$emit('flowSelectionUpdated');
        };
    }

    /* These functions allow to select items based on a predicate
    * (User code does NOT provide items, they are read from FlowGraph)
     */
    this.select = select();
    this.selectAllByType = taggableType => {
        select(it => !it.filterRemove && TaggableObjectsUtils.fromNodeType(it.nodeType) == taggableType)();
    };

    this.filterByTaggableType = function(taggableType) {
        const selectedBefore = selectedItems.length;

        const toRemove = selectedItems.filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) != taggableType);
        toRemove.forEach(_removeFromSelection);

        const selectedAfter = selectedItems.length;
        if (selectedAfter != selectedBefore) {
            $rootScope.$emit('flowSelectionUpdated');
        }
    };

    this.getSelectedNodes = function() {
        return selectedItems.filter(it => it.selected);
    };

    this.getSelectedTaggableObjectRefs = function() {
        return svc.getSelectedNodes().map(TaggableObjectsUtils.fromNode);
    };

    this.selectSuccessors = function (item, evt) {
        $('.select-preview').removeClass('select-preview'); // remove preview styling
        selectionStrategies.MULTI_WITH_SUCCESSORS.onItemClick(item, evt);
        $rootScope.$emit('flowSelectionUpdated');
    };

    this.selectPredecessors = function (item, evt) {
        $('.select-preview').removeClass('select-preview'); // remove preview styling
        selectionStrategies.MULTI_WITH_PREDECESSORS.onItemClick(item, evt);
        $rootScope.$emit('flowSelectionUpdated');
    };

    this.hasPredecessorsInOtherZone = function(item) {
        return item.usedByZones.length > 0;
    };
    this.hasSuccessorsInOtherZone = function(item) {
        let hasSuccessors = false;
        if (_isZoned(item)) {
            let zoneNode = FlowGraph.node("zone_" + item.ownerZone);
            if (zoneNode) {
                $.each(zoneNode["successors"], function (index, otherZoneNodeId) {
                    const otherNode = FlowGraph.node(graphVizEscape(otherZoneNodeId) + "__" + item.realId);
                    if (otherNode) {
                        hasSuccessors = true;
                        return false;
                    }
                });
            }
        }
        return hasSuccessors;
    }

    this.canInsertRecipeBetweenSelection = (selectedNodes, selectedDatasets, selectedRecipes) => {
        if (selectedDatasets.length !== 1   // Only accept one dataset as origin
        || !selectedRecipes.length   // At least one recipe must be selected
        || selectedNodes.length !== selectedDatasets.length + selectedRecipes.length    // Disable insert if any type other than recipe and dataset is selected
        ) return false;        
        const selectedDataset = selectedDatasets[0];
        return selectedRecipes.every(recipe => recipe.predecessors.includes(selectedDataset.id));
    };

    this.selectMulti = function(nodes) {
        _clearSelection();
        _addToSelection(nodes);
        svc.recomputeHighlight();

        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
    }
});


app.service('FlowGraphHighlighting', function(FlowGraph) {

    function removeHighlights() {
        d3.selectAll('#flow-graph .highlight, #flow-graph .fade-out').classed('highlight', false).classed('fade-out', false);
    }

    function highlightPredecessors(item) {
        let element;
        let nodeElt;
        function _highlightPredecessorsRecursive(nodeType, nodeId) {
            nodeElt = FlowGraph.d3NodeWithIdFromType(nodeId, nodeType);
            if (!nodeElt || !nodeElt.node()) {
                // eslint-disable-next-line no-console
                console.debug('Graph node not found', nodeId)
                return;
            }
            if (nodeElt.classed('filter-remove')) {
                return;
            }

            if (!nodeElt.classed('highlight')) {
                // prevents cycles and dreadful infinite loops
                nodeElt.classed('highlight', true).classed('fade-out', false);
                // highlight nodes
                FlowGraph.rawEdgesWithToId(nodeId).forEach(function (elt) {
                    d3.select(elt).classed('highlight', true).classed('fade-out', false);
                });
                const node = FlowGraph.node(nodeId);
                // highlight original node in other zone and its predecessors
                if (node.usedByZones && node.usedByZones.length) {
                    const originalNodeId = `zone__${node.ownerZone}__${node.realId}`
                    _highlightPredecessorsRecursive(nodeType, originalNodeId);
                }
                // highlight former nodes
                $.each(node.predecessors, function (index, id) {
                    _highlightPredecessorsRecursive(nodeType, id);
                });
            }
        }
        try {
            element = FlowGraph.getDOMElement();
            _highlightPredecessorsRecursive(item.nodeType, item.id);
            d3.selectAll('svg .node:not(.highlight), svg .edge:not(.highlight)').classed('fade-out', true);
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error("Failed to highlight items", nodeElt, e); // NOSONAR: OK to use console.
        }
    }

    function highlightSuccessors(item) {
        let element;
        let nodeElt;
        function _highlightSuccessorsRecursive(nodeType, nodeId, force) {
            nodeElt = FlowGraph.d3NodeWithIdFromType(nodeId, nodeType);
            if (!nodeElt || !nodeElt.node()) {
                // eslint-disable-next-line no-console
                console.debug('Graph node not found', nodeId)
                return;
            }
            if (nodeElt.classed('filter-remove')) {
                return;
            }

            if (force || !nodeElt.classed('highlight')) {
                // prevents cycles and dreadful infinite loops
                nodeElt.classed('highlight', true).classed('fade-out', false);
                // highlight nodes
                FlowGraph.rawEdgesWithFromId(nodeId).forEach(function (elt) {
                    d3.select(elt).classed('highlight', true).classed('fade-out', false);
                });
                
                const node = FlowGraph.node(nodeId);
                // highlight usages in other zones and their successors
                const zonesUsingDataset = FlowGraph.nodeSharedBetweenZones(node);
                if (zonesUsingDataset != null) {
                    zonesUsingDataset.forEach((index, zoneId) => {
                        const nodeIdForZone = `zone__${zoneId}__${node.realId}`
                        _highlightSuccessorsRecursive(nodeType, nodeIdForZone, false);
                    });
                }
                // highlight following nodes
                $.each(node.successors, function (index, successorNodeId) {
                    _highlightSuccessorsRecursive(nodeType, successorNodeId, false);
                });
            }
        }

        try {
            element = FlowGraph.getDOMElement();
            _highlightSuccessorsRecursive(item.nodeType, item.id, true);
            d3.selectAll('svg .node:not(.highlight), svg .edge:not(.highlight)').classed('fade-out', true);
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error("Failed to highlight items", nodeElt, e); // NOSONAR: OK to use console.
        }
    }

    function highlightZoneElements(zoneId) {
        d3.selectAll(`#flow-graph .node:not(.filter-remove)[data-zone-id="${zoneId}"]`)
            .classed('highlight', true).classed('fade-out', false)
            .each(function () { d3.selectAll(`#flow-graph .edge:not(.filter-remove)[data-to="${this.id}"]`)
                    .classed('highlight', true).classed('fade-out', false); });
    }

    function addDropFeedbackToZoneElement(zoneId) {
        d3.selectAll(".zone_cluster.clusterDropFeedback").classed("clusterDropFeedback", false);

        let cluster = d3.select(`g[id="cluster_${zoneId}"]`)[0][0];
        $(cluster).toggleClass('clusterDropFeedback',true);
    }
    function removeDropFeedbackFromZoneElements() {
        d3.selectAll(".zone_cluster.clusterDropFeedback").classed("clusterDropFeedback", false);
    }

    function highlightUsedZonesForSingleItem(item) {
        let zoneIdList = [...FlowGraph.nodeSharedBetweenZones(item) || [], item.ownerZone];
        zoneIdList.forEach(highlightZoneById);
        zoneIdList.filter(id => id !== item.ownerZone).forEach(id => d3.select(`#flow-graph g[data-to="zone_${id}"][data-from="zone_${item.ownerZone}"]`).classed('highlight', true));
    }

    function highlightZoneById(zoneId) {
        const zone = 'zone_' + zoneId;
        highlightZoneCluster(d3.select(`#flow-graph g[id="cluster_${zone}"]`).node());
    }

    function removeAllZoneClusterHighlights() {
        d3.selectAll('#flow-graph .zone_cluster.clusterHighlight').classed("clusterHighlight", false).attr('style', null);
    }

    function removeAllEdgesBetweenZonesHighlights() {
        d3.selectAll(`#flow-graph g.edge[data-from^="zone_"]:not([data-from^="zone__"])`).classed('highlight', false);
    }

    function highlightZoneCluster(cluster, forcedColor) {
        if (!cluster) return;
        let node = FlowGraph.node(cluster.getAttribute("data-id"));
        const color = d3.rgb(forcedColor || node.customData.color);
        let zoneTitleColor = (color.r*0.299 + color.g*0.587 + color.b*0.114) >= 128 ? "#000" : "#FFF"; //black or white depending on the zone color
        $(cluster).toggleClass('clusterHighlight',true);
        cluster.style = `color:${zoneTitleColor};background-color:${color.toString()};stroke:${color.toString()}`
    }

    return {
        removeHighlights,
        highlightPredecessors,
        highlightSuccessors,
        highlightZoneElements,
        highlightUsedZonesForSingleItem,
        highlightZoneById,
        highlightZoneCluster,
        removeAllZoneClusterHighlights,
        removeAllEdgesBetweenZonesHighlights,
        addDropFeedbackToZoneElement,
        removeDropFeedbackFromZoneElements
    };
});


app.directive('highlightDependenciesOnHover', function($rootScope, $timeout, FlowGraphSelection, FlowGraphHighlighting, FlowGraph) {
    return {
        link: function(scope, element) {
            let cur = null;

            const shouldSkipHighlight = (event) => {
                // when the user is moving (zoom, panning) OR some items are selected
                // then we don't highlight dependencies to avoid slowdowns
                const movingFlow = event.currentTarget.closest('#flow-graph > svg.moving');
                return movingFlow || FlowGraphSelection.getSelectedNodes().length;
            }

            element.on('mouseenter', 'svg [class~=node]', function (e) {
                if (shouldSkipHighlight(e)) {
                    return;
                }
                let node = $(this);
                if (cur) $timeout.cancel(cur);
                cur = $timeout(function() {
                    const nodeId = node.attr('data-id');
                    const item = FlowGraph.node(nodeId);
                    if (!item || item.filterRemove) {
                        return;
                    }
                    FlowGraphHighlighting.highlightPredecessors(item);
                    FlowGraphHighlighting.highlightSuccessors(item);
                    cur = null;
                }, 100);
                e.stopPropagation();
            });

            element.on('mouseleave', 'svg [class~=node]', function (e) {
                if (shouldSkipHighlight(e)) {
                    return;
                }
                if (cur) $timeout.cancel(cur);
                FlowGraphHighlighting.removeHighlights();
            });
        }
    };
});

})();
