(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsDrawer', GroupedColumnsDrawer);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsDrawer(ChartDimension, Fn, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartColorUtils, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, CHART_TYPES, HierarchicalChartsUtils, ChartDrilldown, ChartHierarchyDimension, ValuesInChartPlacementMode, CHART_VARIANTS) {
        const backgroundXMargin = 4;

        return function(g, chartDef, chartHandler, chartData, chartBase, groupsData, f, labelCollisionDetectionHandler, drawReferenceLines = true) {

            const xDimension = ChartDimension.getGenericDimension(chartDef),
                xLabels = chartData.getAxisLabels('x'),
                xAxis = chartBase.xAxis,
                rightAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes),
                leftAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes),
                getScale = function(d) {
                    return chartDef.genericMeasures[d.measure].displayAxis == 'axis1' ? leftAxis.scale() : rightAxis.scale();
                },
                getRectValue = function(d) {
                    return chartData.aggr(d.measure).get(d);
                };

            const nonEmptyBinsCounts = {};
            // do a first pass to compute nonEmptyBinsIdx, we have to do it here, and not in the "prepareData" method, because it would not have the current animation frame or subchart
            groupsData.forEach(group => {
                group.columns.forEach(c => {
                    const columnIndex = c.x;
                    if (_.isNil(nonEmptyBinsCounts[columnIndex])) {
                        nonEmptyBinsCounts[columnIndex] = 0;
                    }
                    const value = getRectValue(c);

                    let nonEmptyIdx = undefined;
                    if (value !== 0) {
                        nonEmptyIdx = nonEmptyBinsCounts[columnIndex];
                        nonEmptyBinsCounts[columnIndex] += 1;
                    }
                    c.nonEmptyIdx = nonEmptyIdx;
                });
            });
            // second pass to set nbNonEmptyBins
            groupsData.forEach(function(group) {
                group.columns.forEach(c => c.nbNonEmptyBins = nonEmptyBinsCounts[c.x]);
            });

            const nbMaxColumns = Math.max(...groupsData.map(group => group.columns.filter(c => !_.isNil(c.nonEmptyIdx)).length));

            let groupWidth = ChartDimension.isUngroupedNumerical(xDimension) ? 10 : Math.max(1, xAxis.ordinalScale.rangeBand());
            // Adjust barWidth if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
            groupWidth = Math.max(groupWidth * ChartAxesUtils.getCustomExtentRatio(chartDef.xAxisFormatting.customExtent), 1);

            let barWidth = groupWidth;
            if (groupsData.length) {
                if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                    barWidth = nbMaxColumns ? groupWidth / nbMaxColumns : 0;
                } else {
                    barWidth = groupWidth / groupsData[0].columns.length;
                }
            }

            // Wrapper to contain all rectangles (used to apply clip-path)
            let wrapper = g.selectAll('g.group-wrapper');
            if (wrapper.empty()) {
                g.append('g').attr('class', 'group-wrapper');
                wrapper = g.selectAll('g.group-wrapper');
            }
            const groups = wrapper.selectAll('g.group-bars').data(groupsData);
            groups.enter().append('g')
                .attr('class', 'group-bars');
            groups.exit().remove();
            groups.attr('transform', BarChartUtils.translate('x', xAxis, xLabels, groupWidth));

            const positionBarWrappers = function(barWrappers) {
                return barWrappers.attr('transform', function(d, i) {
                    let xTranslate = barWidth * i;
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        xTranslate = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * d.nonEmptyIdx + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2);
                    }
                    return 'translate(' + xTranslate + ')';
                });
            };

            const positionRects = function(rects) {
                return rects.attr('height', function(d) {
                    const yScale = getScale(d);
                    return Math.max(chartDef.variant === CHART_VARIANTS.waterfall ? 1 : 0, Math.abs(yScale(d.base) - yScale(d.top)));
                }).attr('y', function(d) {
                    const yScale = getScale(d);
                    return yScale(Math.max(d.base, d.top));
                });
            };

            const hasColorDim = ChartColorUtils.getColorDimensionOrMeasure(chartDef) !== undefined;

            const labelCollisionDetectionHandlersByColorScaleIndex = {};

            const barWrappers = groups.selectAll('.bar-wrapper').data(Fn.prop('columns'));
            barWrappers.enter().append('g')
                .attr('class', 'bar-wrapper')
                .call(positionBarWrappers);
            barWrappers.exit().remove();

            const rects = barWrappers.selectAll('rect').data(Fn.prop('data'));
            rects.enter().append('rect')
                .attr('fill', function(d) {
                    return chartBase.colorScale(ChartColorUtils.getColorIndex(d, hasColorDim));
                })
                .attr('opacity', function(d) {
                    return d.transparent ? 0 : chartDef.colorOptions.transparency;
                })
                .each(function(d) {
                    chartBase.tooltips.registerEl(this, angular.extend({}, d, { facet: f }), 'fill', undefined, undefined, hasColorDim);
                    chartBase.contextualMenu.addContextualMenuHandler(this, angular.extend({}, d, { facet: f }));
                });
            rects.exit().remove();
            rects
                .attr('width', barWidth)
                .attr('stroke-width', function(d) {
                    return d.isUnaggregated ? 0.5 : 0;
                })
                .attr('stroke', function(d) {
                    return chartHandler.legendsWrapper.getLegend(0).items[ChartColorUtils.getColorIndex(d, hasColorDim)].desaturatedColor;
                })
                .attr('x', 0)
                .interrupt('updateColumns').transition('updateColumns').ease('easeOutQuad')
                .call(positionRects);

            // mix chart uses the same drawer, hence the condition on the chartType
            if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                const getLabelYPosition = (d) => {
                    if (!d.textsElements.length) {
                        // for animation purpose: place empty labels on the X axis, to have a smoother transition if they re-appear at the next animation frame
                        return getScale(d)(0);
                    }

                    const rectValue = getRectValue(d);
                    const defaultSpacing = chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES ? 5 : 2;
                    const spacing = d.valuesInChartDisplayOptions.spacing ?? defaultSpacing;

                    const yScale = getScale(d);
                    const scaleValue = yScale(rectValue);
                    if (chartDef.variant === CHART_VARIANTS.waterfall) {
                        const placementMode = d.valuesInChartDisplayOptions.placementMode;
                        // for now only one value in waterfal, so we just use the first data element
                        const data = d.data[0];
                        switch (placementMode) {
                            case ValuesInChartPlacementMode.BELOW:
                                return rectValue < 0 ? (yScale(data.top) + spacing) : (yScale(data.base) + spacing);
                            case ValuesInChartPlacementMode.ABOVE:
                                return rectValue < 0 ? (yScale(data.base) - spacing - d.height) : (yScale(data.top) - spacing - d.height);
                            default:
                                return rectValue < 0 ? (yScale(data.top) + spacing) : (yScale(data.top) - spacing - d.height);
                        }
                    }

                    const endBarY = isNaN(scaleValue) ? 0 : scaleValue - (rectValue >= 0 ? spacing : -spacing);
                    if (rectValue < 0) {
                        // for negative values, the label is below the bar
                        return endBarY;
                    }
                    return endBarY - d.height;
                };

                const getLabelXPosition = (d, i) => {
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        // nonEmptyIdx undefined means that the bin is empty
                        const xPos = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * (d.nonEmptyIdx + 0.5) + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2);
                        return xPos;
                    }
                    return barWidth * (i + 0.5);
                };

                const getBackgroundXPosition = (d, i) => {
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        // nonEmptyIdx undefined means that the bin is empty
                        const xPos = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * (d.nonEmptyIdx + 0.5) + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2) - d.width / 2 - backgroundXMargin;
                        return xPos;
                    }
                    return barWidth * (i + 0.5) - d.width / 2 - backgroundXMargin;
                };

                const getBackgroundYPosition = (d) => {
                    const y = getLabelYPosition(d);
                    return SVGUtils.getSubTextYPosition(d, y);
                };

                const getLabelText = (d) => {
                    const value = chartData.aggr(d.aggregationIndex).get(d);
                    return BarChartUtils.shouldDisplayBarLabel(chartData.getCount(d), getRectValue(d)) ? chartBase.measureFormatters[d.aggregationIndex](value) : '';
                };

                const labelsDrawContext = {
                    node: wrapper,
                    data: groupsData.map(d => d.columns),
                    axisName: 'x',
                    axis: xAxis,
                    labels: xLabels,
                    thickness: groupWidth,
                    opacity: chartDef.colorOptions.transparency,
                    overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                    colorScale: chartBase.colorScale,
                    getExtraData: function(d, i, n, onHoverMode) {
                        const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, ChartColorUtils.getColorIndex(d, hasColorDim), getLabelXPosition, getLabelYPosition, chartDef.valuesInChartDisplayOptions.overlappingStrategy, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);

                        return { labelPosition: extraData.labelPosition };
                    },
                    getLabelXPosition: (d) => {
                        return d.labelPosition.x;
                    },
                    getLabelYPosition: (d) => {
                        return d.labelPosition.y;
                    },
                    getLabelText,
                    getBackgroundXPosition,
                    getBackgroundYPosition,
                    backgroundXMargin,
                    theme: chartHandler.getChartTheme()
                };

                BarChartUtils.drawLabels(labelsDrawContext);
            }

            // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
            SVGUtils.clipPaths(chartBase, g, wrapper);

            if (drawReferenceLines) {
                const isPercentScaleOnLeftYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis === 'axis1' && ChartDimension.isPercentScale([measure]));
                const isPercentScaleOnRightYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis !== 'axis1' && ChartDimension.isPercentScale([measure]));

                const d3RightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
                const d3LeftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
                const leftYFormattingOptions = d3LeftYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, d3LeftYAxis.id);
                const rightYFormattingOptions = d3RightYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, d3RightYAxis.id);

                const refLinesXAxis = { ...chartBase.xAxis, isPercentScale: false, formattingOptions: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting };
                const refLinesYAxes = [];

                if (d3LeftYAxis) {
                    refLinesYAxes.push({ ...d3LeftYAxis, isPercentScale: isPercentScaleOnLeftYAxis, formattingOptions: leftYFormattingOptions });
                }
                if (d3RightYAxis) {
                    refLinesYAxes.push({ ...d3RightYAxis, isPercentScale: isPercentScaleOnRightYAxis, formattingOptions: rightYFormattingOptions });
                }

                const dataSpec = chartHandler.getDataSpec();
                const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
                const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
                ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartDef.$axisSpecs && chartDef.$axisSpecs.x, undefined),
                    referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures, chartDef.variant !== 'waterfall');

                ReferenceLines.drawReferenceLines(
                    wrapper,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    refLinesXAxis,
                    refLinesYAxes,
                    displayedReferenceLines,
                    referenceLinesValues
                );
            }

            if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
            }

            const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrapper, d3, g);
            ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
        };
    };
})();
