(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedChartUtils', StackedChartUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/stacked-columns.js
     */
    function StackedChartUtils(ChartAxesUtils, ChartsStaticData, SVGUtils, ChartMeasure, UnaggregatedMeasureComputeModes, CHART_VARIANTS) {
        const computeBaseAndTop = (value, positiveTotal, negativeTotal) => {
            const base = value >= 0 ? positiveTotal : negativeTotal;
            const top = base + value;

            return {
                base,
                top
            };
        };

        const computeUAValueData = function(genericData, chartData, measure, measureIndex, positiveTotal, negativeTotal) {
            const result = {
                ...genericData,
                isUnaggregated: true,
                uaComputeMode: measure.uaComputeMode
            };

            let newPositiveTotal = positiveTotal;
            let newNegativeTotal = negativeTotal;
            const uaValues = chartData.aggr(measureIndex).get();
            if (measure.uaComputeMode === UnaggregatedMeasureComputeModes.STACK) {
                result.data = uaValues.map(function(uaValue, idx) {
                    const { base, top } = computeBaseAndTop(uaValue, newPositiveTotal, newNegativeTotal);
                    if (uaValue >= 0) {
                        newPositiveTotal = top;
                    } else {
                        newNegativeTotal = top;
                    }
                    return {
                        isUnaggregated: true,
                        isStacked: true,
                        ...genericData,
                        base,
                        top,
                        uaValueIdx: idx
                    };
                });
            } else {
                // overlay mode
                const positiveData = [];
                const negativeData = [];
                uaValues.forEach(function(uaValue, idx) {
                    const { base, top } = computeBaseAndTop(uaValue, positiveTotal, negativeTotal);
                    const commonData = {
                        isUnaggregated: true,
                        base,
                        top,
                        ...genericData,
                        uaValueIdx: idx
                    };
                    if (uaValue >= 0) {
                        positiveData.push(commonData);
                        newPositiveTotal = Math.max(newPositiveTotal, commonData.top);
                    } else {
                        negativeData.push(commonData);
                        newNegativeTotal = Math.min(newNegativeTotal, commonData.top);
                    }
                });
                result.data = [
                    /*
                     * in Overlay mode, we display one bar per ua value, to handle tooltip and hover effect
                     * but we color only the biggest bar for positive values and for negative values, other bars are transparent by default
                     */
                    ...positiveData.map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    }),
                    ...negativeData.reverse().map(function(data, idx) {
                        return {
                            ...data,
                            transparent: idx > 0
                        };
                    })
                ];
            }

            // return also the new value for totals
            return { ...result, positiveTotal: newPositiveTotal, negativeTotal: newNegativeTotal };
        };


        return {
            prepareData: function(chartDef, chartData, axis = 'x') {
                const colorLabels = chartData.getAxisLabels('color') || [null],
                    facetLabels = chartData.getAxisLabels('facet') || [null],
                    animationLabels = chartData.getAxisLabels('animation') || [null],
                    mainAxisLabels = chartData.getAxisLabels(axis),
                    // only one axis at a time can display in logscale so that logic is enough
                    hasLogScale = chartDef.xAxisFormatting.isLogScale || ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID);

                const getBarExtremity = (total) => {
                    if (total === 0 && hasLogScale) {
                        return 1;
                    }
                    return total;
                };
                const animationData = { frames: [], maxTotal: 0, minTotal: 0 };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [], maxTotal: 0, minTotal: 0 };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { stacks: [], maxTotal: 0, minTotal: 0 };
                        mainAxisLabels.forEach(function(_, val) {
                            chartData.fixAxis(axis, val);

                            let positiveTotal = 0;
                            let negativeTotal = 0;

                            let count = 0;
                            const stackData = [];

                            const totalSubTextElementsData = SVGUtils.getLabelSubTexts(chartDef, undefined, chartDef.stackedColumnsOptions ? chartDef.stackedColumnsOptions.totalsInChartDisplayOptions : chartDef.valuesInChartDisplayOptions, true).map(function(subTextElementData) {
                                const res = {
                                    ...subTextElementData,
                                    aggregationTotal: positiveTotal + negativeTotal
                                };
                                res[axis] = val;
                                return res;
                            });

                            colorLabels.forEach(function(colorLabel, c) {
                                chartData.fixAxis('color', c);
                                totalSubTextElementsData.forEach(function(subTextElementData) {
                                    if (subTextElementData.aggregationIndex !== undefined) {
                                        // compute total for custom aggregations
                                        const d = chartData.aggr(subTextElementData.aggregationIndex).get();
                                        subTextElementData.aggregationTotal += d;
                                    }
                                });

                                chartDef.genericMeasures.forEach(function(measure, m) {
                                    const d = chartData.aggr(m).get();

                                    if (chartDef.variant == CHART_VARIANTS.stacked100 && d < 0) {
                                        throw new ChartIAE('Cannot represent negative values on a 100% Stacked chart. Please use another chart.');
                                    }

                                    let subTextElementsData = [];
                                    let data = [];
                                    const commonData = {
                                        color: c,
                                        measure: m,
                                        facet: f,
                                        animation: a,
                                        count: chartData.getNonNullCount({}, m),
                                        value: d,
                                        [axis]: val
                                    };
                                    if (ChartMeasure.isRealUnaggregatedMeasure(measure)) {
                                        // in that case the value is a double array instead of a single value
                                        if (chartDef.variant == CHART_VARIANTS.stacked100) {
                                            throw new ChartIAE('Cannot represent unaggregated values on a 100% Stacked chart. Please use another chart.');
                                        }

                                        const {
                                            data: uaData,
                                            positiveTotal: newPositiveTotal,
                                            negativeTotal: newNegativeTotal
                                        } = computeUAValueData(commonData, chartData, measure, m, positiveTotal, negativeTotal);
                                        data = uaData;
                                        positiveTotal = newPositiveTotal;
                                        negativeTotal = newNegativeTotal;
                                    } else {
                                        commonData.base = (d >= 0) ? positiveTotal : negativeTotal;
                                        commonData.top = commonData.base + d;

                                        if (d >= 0) {
                                            commonData.base = getBarExtremity(commonData.base);
                                            commonData.top = getBarExtremity(commonData.top);
                                            positiveTotal = commonData.top;
                                        } else {
                                            negativeTotal = commonData.top;
                                        }

                                        subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, m, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                            return {
                                                ...subTextElementData,
                                                // add measure data at sub text level also, to retrieve it easily
                                                ...commonData
                                            };
                                        });

                                        data = [commonData];
                                    }

                                    const point = {
                                        ...commonData,
                                        data,
                                        textsElements: subTextElementsData,
                                        valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                    };
                                    stackData.push(point);

                                    count += chartData.getNonNullCount({}, m);
                                });
                            });

                            if (chartDef.variant == CHART_VARIANTS.stacked100 && positiveTotal > 0) {
                                // Do a second pass and divide by total
                                let totalPercent = 0;
                                stackData.forEach(function(point, p) {
                                    const update = {
                                        value: point.value / positiveTotal,
                                        base: totalPercent,
                                        top: point.value / positiveTotal + totalPercent
                                    };
                                    Object.assign(point, update);
                                    // update data on sub texts as well
                                    point.textsElements.forEach(function(subTextElementData) {
                                        Object.assign(subTextElementData, update);
                                    });
                                    point.data.forEach(function(dataElement) {
                                        Object.assign(dataElement, update);
                                    });

                                    totalPercent += point.value;
                                });

                                positiveTotal = 1;
                            }

                            const totalData = { data: stackData, total: positiveTotal + negativeTotal, positiveTotal, negativeTotal, count,
                                spacing: chartDef.stackedColumnsOptions?.totalsInChartDisplayOptions?.spacing
                            };
                            facetData.stacks.push({ ...totalData, textsElements: totalSubTextElementsData.map(function(subTextElementData) {
                                return { ...subTextElementData, ...totalData };
                            }) });
                            facetData.maxTotal = Math.max(facetData.maxTotal, positiveTotal);
                            facetData.minTotal = Math.min(facetData.minTotal, negativeTotal);
                        });

                        frameData.maxTotal = Math.max(frameData.maxTotal, facetData.maxTotal);
                        frameData.minTotal = Math.min(frameData.minTotal, facetData.minTotal);
                        frameData.facets.push(facetData);
                    });

                    animationData.maxTotal = Math.max(animationData.maxTotal, frameData.maxTotal);
                    animationData.minTotal = Math.min(animationData.minTotal, frameData.minTotal);
                    animationData.frames.push(frameData);
                });

                return animationData;
            }
        };
    }

})();
