
$me=(Split-Path -Path $PSCommandPath -Leaf) # Get script name

$BINDIR=$env:BINDIR

function ExitWith($code) {
    $Host.SetShouldExit($code)
    #Exit $code
}

function Usage() {
    Write-Host "Usage: $me (start | stop | status | restart)"
    # $me (start | stop | status | restart) PROCESS
    # $me run PROCESS
    # $me run
    # PROCESS : (backend | ipython | nginx | collectd)
    ExitWith 1
}

function FindDistrib() {
    return (Get-ComputerInfo).WindowsProductName
}

switch ($args[0]) {
    { "start", "stop", "status", "restart", "run" -contains "$_" } {
        $action = $args[0]
    }
    Default {
        Usage
    }
}

$env:DIP_HOME=Split-Path -Path $BINDIR # dirname $BINDIR
$DIP_HOME=$env:DIP_HOME

function NormalizePath {
    param (
        [Parameter(Mandatory=$true)] [string] $Path
    )

    return $Path.Replace("\", "/").TrimEnd("/")
}

if ("start", "restart" -contains "$action") {
    if (Test-Path("$DIP_HOME/install-support/expected-dip-home.txt")) {
        $expected = (Get-Content "$DIP_HOME/install-support/expected-dip-home.txt" -Raw) -replace "`t|`n|`r",""
        if ((NormalizePath $expected) -ne (NormalizePath $DIP_HOME)) {
            Write-Host "[!] $DIP_HOME appears to be a copy from $expected"
            Write-Host "[!] Was the install properly migrated? (with installer.sh -u ...)"
            Write-Host "[!] Aborting."
            ExitWith 1
        }
    } else {
        Write-Host "[!] $DIP_HOME does not appear to be a validated DSS data directory (missing installation flag file)"
        Write-Host "[!] Initial installation may have failed before completion"
        ExitWith 1
    }
}

& $BINDIR/env-default.ps1

if ($null -eq $env:DKUINSTALLDIR) {
    Write-Host "FATAL: DKUINSTALLDIR is not defined. Please check $BINDIR\env-default.ps1"
    ExitWith 1
}

# . = call in same scope, to access functions
. $env:DKUINSTALLDIR/scripts/_startup.inc.ps1

bkdl_set_py_env
bkdl_set_R_env
bkdl_load_env_files

$env:DKUBIN="$BINDIR/dku.cmd"
$env:DKUJEKBIN="$BINDIR/jek.cmd"
$env:DKUFEKBIN="$BINDIR/fek.cmd"
$env:DKUCAKBIN="$BINDIR/cak.cmd"
$env:DKUHPROXYBIN="$BINDIR/hproxy.cmd"
$env:DKUDISTRIB= FindDistrib

# Setup Graphviz to have our shipped binary
if (-not ($env:PATH -match "graphviz\bin")) {
    $env:PATH="$env:DKUINSTALLDIR\third-party\graphviz\bin;$env:PATH"
}

$CP = bkdl_get_cp

$DKURUNDIR = "$DIP_HOME/run"

function setup_jupyter_dirs() {
    $env:JUPYTER_DATA_DIR="$DIP_HOME/jupyter-run/jupyter"
    $env:JUPYTER_RUNTIME_DIR="$env:JUPYTER_DATA_DIR/runtime"
    $env:JUPYTER_CONFIG_DIR="$env:JUPYTER_DATA_DIR/config"
    $env:IPYTHONDIR="$DIP_HOME/jupyter-run/ipython"
    New-Item -Path "$env:JUPYTER_DATA_DIR" -Type Directory -ErrorAction Ignore *>$null
    New-Item -Path "$env:JUPYTER_RUNTIME_DIR" -Type Directory -ErrorAction Ignore *>$null
    New-Item -Path "$env:IPYTHONDIR" -Type Directory -ErrorAction Ignore *>$null
}

########################################
# Child processes definition: DSS
########################################

function run_backend() {
    $env:PYTHONPATH="$DIP_HOME\lib\python;$env:DKUINSTALLDIR\python;$env:DKUINSTALLDIR\dku-jupyter\packages"
    bkdl_set_R_libs

    # Make sure tmp folder exists

    setup_jupyter_dirs

    $params = "$env:DKU_JAVA_OPTS".Split(" ")
    $params+= "$env:DKU_DKU_JAVA_OPTS".Split(" ")
    $params+="-D dku.backend",
      "-XX:+CrashOnOutOfMemoryError",
      "-XX:ErrorFile=`"$env:DIP_HOME/run/hs_err_pid%p.log`"",

      "-Djava.library.path=`"$env:DKU_DKU_JAVA_LIBRARY_PATH:$env:DKU_HADOOP_JAVA_LIBRARY_PATH`"",
      "-cp",
      $CP,
      "com.dataiku.dip.DSSBackendMain"
    & $env:DKUJAVABIN $params
}

function run_ipython() {
    $env:PYTHONPATH="$DIP_HOME\lib\python;$env:DKUINSTALLDIR\python;$env:DKUINSTALLDIR\dku-jupyter\packages"
    bkdl_set_R_libs

    setup_jupyter_dirs

    $params="$env:DKUINSTALLDIR/dku-jupyter/dkulauncher.py",
        "--config=`"$env:DKUINSTALLDIR/dku-jupyter/dataiku_jupyter_config.py`"",
        "--notebook-dir=`"$DIP_HOME/config/ipython_notebooks`"",
        "--ip=127.0.0.1",
        "--port=`"$env:DKU_IPYTHON_PORT`"",
        "--no-browser",
        "--port-retries=0"
    & $env:DKUPYTHONBIN $params
}

function run_nginx() {
    & "$env:DKUINSTALLDIR/third-party/nginx/nginx.exe" -c "$DIP_HOME/install-support/nginx.conf"
}

########################################
# Child processes definition: API Node
########################################

function run_apimain() {
  $env:PYTHONPATH="$DIP_HOME\lib\python;$env:DKUINSTALLDIR\python"
  bkdl_set_R_libs

  $env:DKU_LAMBDA_HOME="$DIP_HOME"

  $apinodeCP=apinode_get_cp

  $params = "$env:DKU_JAVA_OPTS".Split(" ")
  $params+= "$env:DKU_APIMAIN_JAVA_OPTS".Split(" ")
  $params+="-D dku.apimain",
    "-XX:+CrashOnOutOfMemoryError",
    "-XX:ErrorFile=`"$env:DIP_HOME/run/hs_err_pid%p.log`"",

    "-Djava.library.path=`"$env:DKU_APIMAIN_JAVA_LIBRARY_PATH`"",
    "-cp",
    $apinodeCP,
    "com.dataiku.lambda.LambdaServerMain"
  & $env:DKUJAVABIN $params
}

function run_apimain_dev() {
  $env:PYTHONPATH="$DIP_HOME\lib\python;$env:DKUINSTALLDIR\python"
  bkdl_set_R_libs

  $apinodeCP=apinode_get_cp

  $params = "$env:DKU_JAVA_OPTS".Split(" ")
  $params+= "$env:DKU_APIMAIN_JAVA_OPTS".Split(" ")
  $params+="-D dku.apimain",
    "-XX:+CrashOnOutOfMemoryError",
    "-XX:ErrorFile=`"$env:DIP_HOME/run/hs_err_pid%p.log`"",

    "-Djava.library.path=`"$env:DKU_APIMAIN_JAVA_LIBRARY_PATH`"",
    "-cp",
    $apinodeCP,
    "com.dataiku.lambda.LambdaServerMain"
  & $env:DKUJAVABIN $params
}

########################################
# Supervisord process manager
########################################

function start_supervisord() {
    $env:DIP_HOME=$env:DIP_HOME.Replace("\", "/")
    # Here, we start a process in the background, without any window.
    # This process will persist even after we close the terminal
    # Daemon mode is not supported on Windows
    Start-Process -WindowStyle hidden -FilePath $env:DKUPYTHONBIN -ArgumentList ("-m", "supervisor.supervisord", "-c", "$env:DIP_HOME/install-support/supervisord.conf")
    $env:DIP_HOME=$env:DIP_HOME.Replace("/", "\")
}

function run_supervisord() {
    # This is required as Supervisor can't seem to work with backslashes
    $env:DIP_HOME=$env:DIP_HOME.Replace("\", "/")
    & "$env:DKUPYTHONBIN" -m supervisor.supervisord -c "$env:DIP_HOME/install-support/supervisord.conf" -n
    $env:DIP_HOME=$env:DIP_HOME.Replace("/", "\")
}

function supervisorctl() {
    $res = & "$env:DKUPYTHONBIN" -m supervisor.supervisorctl -c "$DIP_HOME/install-support/supervisord.conf" "pid"
    if ($res | Select-String -Pattern "refused connection" -Quiet) {
        Write-Host "$($me): DSS supervisor is not running."
        return 1
    }

    Write-Host ((& "$env:DKUPYTHONBIN" -m supervisor.supervisorctl -c "$DIP_HOME/install-support/supervisord.conf" $args) -Join "`n")
    return 0
}

########################################
# Wait for process transitions
########################################

function wait_supervisor_start() {
    Write-Host "Waiting for DSS supervisor to start..."
    for ($i = 0; $i -lt 10; $i++) {
        Start-Sleep 2
        $r = supervisorctl status 6>$null # Not working in PS < 5, but it's ok
        if ($r -eq 0) {
            Write-Host "DSS started, pid=$(Get-Content "$DKURUNDIR\supervisord.pid")"
            return 0
        }
    }

    Write-Host "[] $me : DSS supervisor failed to start.
You can look for startup error messages in log file:
$DKURUNDIR/supervisord.log
"
    return 1
}

function wait_supervisor_stop() {
    Write-Host "Waiting for DSS to stop..."
    for ($i = 0; $i -lt 10; $i++) {
        Start-Sleep 2
        $p = Get-Content -Path "$DKURUNDIR\supervisord.pid" -ErrorAction ignore
        if ($null -eq $p) {
            Write-Host "DSS stopped"
            return 0
        }
    }
    Write-Host "[] $me : DSS failed to stop"
    return 1
}

function wait_backend_start() {
    Write-Host -NoNewline "Waiting for DSS backend to start"
    for ($i = 0; $i -lt 60; $i++) {
        Start-Sleep 1
        try {
            Invoke-WebRequest -URI "http://localhost:$env:DKU_BACKEND_PORT/dip/api/ping" -ErrorAction Stop *> $null
            Write-Host ""
            return 0
        } catch {
            Write-Host -NoNewline "."
        }
    }

    Write-Host "DSS failed to start. Check $DKURUNDIR\backend.log for details."
    return 1
}

#######################################################
# Check the runtime environment
#######################################################

function check_server_ports() {
    $check_ports = & "$env:DKUPYTHONBIN" "$env:DKUINSTALLDIR/scripts/dkuinstall/install_config.py" -get server check_ports

    if ($check_ports -contains "false") {
        return
    }

    $ports = $env:DKU_NGINX_PORT, $env:DKU_BACKEND_PORT, $env:DKU_IPYTHON_PORT

    & "$env:DKUPYTHONBIN" "$env:DKUINSTALLDIR/scripts/dkuinstall/check_server_port.py" $ports
}

#######################################################
# Global actions
#######################################################

function start_all() {
    check_server_ports
    if ($LASTEXITCODE -eq 0) {
        start_supervisord | Out-Null
        wait_supervisor_start | Out-Null
        wait_backend_start | Out-Null
    }
}

function stop_all() {
    supervisorctl shutdown | Out-Null
    if ($env:SkipWait) {
        Write-Host "SkipWait set to '$env:SkipWait'. Not waiting for dss to stop."
        return;
    }

    if ($LASTEXITCODE -eq 0) {
        wait_supervisor_stop | Out-Null
    }
}

########################################
#
# Main entry point
#
########################################

if ($args.length -eq 1 -and $action -ne "run") {
    $action="${action}_all"
} elseif ($args.length -eq 1) {
    $process="supervisord"
} elseif ($args.length -eq 2) {
    switch ($args[1]) {
        { "backend", "ipython", "nginx", "apimain", "apimain_dev" -contains "$_" } {
            $process = $args[1]
        }
        Default { Usage }
    }
} else {
    Usage
}

switch ($action) {
    "start_all" { start_all }
    "stop_all" { stop_all }
    "status_all" { ExitWith $(supervisorctl status) }
    "restart_all" {
        stop_all
        start_all
    }
    { "start", "stop", "restart", "status" -contains "$_" } { ExitWith $(supervisorctl "$action" "$process") }
    "run" { . "run_$process" }
    Default {}
}
