(function() {
'use strict';

const app = angular.module('dataiku.services');

// To simplify the migration to ng1.6 which removes the .success and .error from $http
// we monkeypatch $http in this module with replacement versions of the missing functions
// Call addSuccessErrorToPromise to directly add these methods to any promise.
//
app.run (function initMonkeyPatchPromiseWithSuccessError($http) {
    app.addSuccessErrorToPromise = function (promise) {
        promise.success = function(callback) {
            promise.then( (resp) => {return callback(resp.data, resp.status, resp.headers, resp.config, resp.statusText, resp.xhrStatus)});
            return promise;
        };

        promise.error = function(callback) {
            promise.then(null, resp => callback(resp.data, resp.status, resp.headers, resp.config, resp.statusText, resp.xhrStatus));
            return promise;
        };
        return promise;
    }

    function monkeyPatchHttpGetForSuccessError() {
        const get = $http.get;
        $http.get = function () {
            let promise =  get.apply(this, arguments);
            const decoratedPromise =  app.addSuccessErrorToPromise(promise);
            return decoratedPromise;
        }
    }
    monkeyPatchHttpGetForSuccessError();
});

app.factory("DataikuCloudAPI", function(APIXHRService, $q, $rootScope) {
return {
    getWebConfig: function() {
        var deferred = $q.defer();
        $.ajax({
            url: "https://usage.dataiku.com/public/globalId",
            jsonp: "fun",
            dataType: "jsonp",
            success: function(response) {
                deferred.resolve(response)
            }
        })
        return deferred.promise;
    },
    getNewWebConfig: function() {
        var deferred = $q.defer();
        $.ajax({
            url: "https://www.dataiku.com/api/get-config.php",
            jsonp: "callback",
            dataType: "jsonp",
            success: function(response) {
                deferred.resolve(response)
            }
        })
        return deferred.promise;
    },
    community: {
        register: function(firstName, lastName, company, persona, email,
                                     newsletter, wantEETrial, dssVersion, webVisitorId, webVisitorLocalId, webVisitorHSId, registrationChannel) {
            return APIXHRService("POST",
                $rootScope.appConfig.saasManagerURL + "/community/register-v5", {
                    firstName: firstName,
                    lastName: lastName,
                    company: company,
                    email: email,
                    persona: persona,
                    newsletter: newsletter,
                    wantEETrial: wantEETrial,
                    dssVersion: dssVersion,
                    webVisitorId: webVisitorId,
                    webVisitorLocalId: webVisitorLocalId,
                    webVisitorHSId: webVisitorHSId,
                    registrationChannel: registrationChannel
                }
            );
        },
        requestEETrial: function(instanceId, updatedEmailAddress) {
            return APIXHRService("POST",
                $rootScope.appConfig.saasManagerURL + "/community/request-ee-trial", {
                    instanceId: instanceId, updatedEmailAddress: updatedEmailAddress
                }
            );
        }
    }
}
});


app.factory("DataikuAPI", ["APIXHRService", '$q', '$rootScope', 'Logger', '$$cookieReader', function(APIXHRService, $q, $rootScope, Logger, $$cookieReader) {
    var API_PATH = '/dip/api/';
    var FIXTURES_PATH = '/fixtures/';
    var JUPYTER_API_PATH = '/jupyter/';

    var uploadFileRequest = function(requestUrl, formdataCustomizer, callback) {
        var url = API_PATH + requestUrl;

        // angular doesn't provide a way to get the progress event yet, we explicitly redo it
        var deferred = $q.defer();

        var xhr = new XMLHttpRequest();

        if (callback != null) {
            xhr.upload.addEventListener("progress", callback, false);
        }
        xhr.addEventListener("load", function(e) {
            var payload = e.target||e.srcElement;
            if(payload.status == 200) {
                deferred.resolve(payload.response);
            } else {
                deferred.reject(payload);
            }
            $rootScope.$apply();
        }, false);
        xhr.addEventListener("error", function(e) {
            var payload = e.target||e.srcElement;
            deferred.reject(payload);
            $rootScope.$apply();
        }, false);

        var start = new Date().getTime();
        Logger.debug("[S] POST_FILE " + requestUrl);

        var logDone = function(result) {
            var end = new Date().getTime();
            Logger.debug("[D] POST_FILE " + requestUrl + " (" + (end-start) +"ms)");
        };

        xhr.open("POST", url);
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

        const xsrfToken = $$cookieReader()[$rootScope.appConfig.xsrfCookieName];
        xhr.setRequestHeader('X-XSRF-TOKEN', xsrfToken);

        var formdata = new FormData();
        formdataCustomizer(formdata);
        xhr.send(formdata);

        deferred.promise.then(logDone, logDone);
        return app.addSuccessErrorToPromise(deferred.promise);
    };

    var stripDollarKeys = function(key, value) {
        return key.startsWith("$") ? undefined : value;
    };

    // Unlike the regular HTTP client, this request will not automatically trigger digest cycles
    // Digest cycles will need to be handled at the applicative layer.
    const streamRequest = (method, url, data, readerFn, abortController) => {
        const xsrfToken = $$cookieReader()[$rootScope.appConfig.xsrfCookieName];
        const params = new URLSearchParams(data);
        const options = {
            method,
            signal: abortController.signal,
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8',
                'x-xsrf-token': xsrfToken,
            }
        };
        const deferred = $q.defer();

        if (method === 'GET') {
            url += '?' + params.toString();
        } else {
            options.body = params;
        }
        let httpResponse;

        fetch(url, options).then(response => {
            // mimic $http response object
            httpResponse = {
                status: response.status,
                statusText: response.statusText,
                headers: (header) => response.headers.get(header),
                config: options,
                data: null
            };

            if (!response.ok) {
                return response.json().then(error => {
                    const err = new Error();
                    err.info = error;
                    throw err;
                });
            }

            readerFn(response, () => deferred.resolve({ ...httpResponse }), (error) => {
                Logger.error('Error reading stream', error);
                deferred.reject(error)
            });
        }).catch(error => {
            if (error.name !== 'AbortError') { // request was not aborted, so really handle the error
                deferred.reject({
                    ...httpResponse,
                    data: error ? error.info : {}
                });
            }
        });

        return deferred.promise;
    };

    const readStreamEventData = (chunkCallback) => {
        return (response, onDone, onError) => {
            const reader = response.body.getReader();
            const decoder = new TextDecoder('utf-8');
            let buffer = '';
            let event = null;

            function readChunk() {
                reader.read().then(({ done, value }) => {
                    if (done) {
                        onDone();
                        return;
                    }

                    buffer += decoder.decode(value, { stream: true });
                    let lines = buffer.split('\n');
                    buffer = lines.pop(); // if last element is not '' it means we have an incomplete chunk

                    lines.forEach(line => {
                        line = line.trim();
                        if (line) {
                            if (line.startsWith('event: ')) {
                                event = line.slice(7);
                            } else if (line.startsWith('data: ')) {
                                try {
                                    let data = JSON.parse(line.slice(6));
                                    chunkCallback({ event, data });
                                } catch(e) {
                                    Logger.error('Invalid JSON', e);
                                }
                                event = null;
                            }
                        }
                    });

                    readChunk();
                }).catch(error => {
                    if (error.name !== 'AbortError') { // request was not aborted, so really handle the error
                        onError(error);
                    }
                });
            }

            // Start reading the first chunk
            readChunk();
        }
    }

return {
debug: {

},
usage: {
    popNextReport: function() {
        return APIXHRService("POST", API_PATH + "pop-next-report", null, 'nospinner');
    },
    popReflectedEvents: function() {
        return APIXHRService("POST", API_PATH + "pop-reflected-events", null, 'nospinner');
    }

},
registration: {
    initialRegisterCommunity: function(firstName, lastName, userEmail, instanceId, license) {
        return APIXHRService("POST", API_PATH + "registration/initial-register-community", {
            userFirstName: firstName, userLastName: lastName,
            userEmail: userEmail,
            instanceId: instanceId, license: JSON.stringify(license)
        });
    },
    initialRegisterLicensed: function(license) {
        return APIXHRService("POST", API_PATH + "registration/post-initial-register-licensed", {
            license: license
        });
    },
    setOfflineLicense: function(license) {
        return APIXHRService("POST", API_PATH + "registration/set-offline-license", {
            license: license
        });
    },
    renewExpiredLicense: function(license) {
        return APIXHRService("POST", API_PATH + "registration/renew-expired-license", {
            license: license
        });
    }
},
projects: {
    list: function(includeLimitedVisibility = false) {
        return APIXHRService("GET", API_PATH + "projects/list", {includeLimitedVisibility});
    },
    listHeads: function(requiredPrivilege = null) {
        return APIXHRService("GET", API_PATH + "projects/list-heads", {requiredPrivilege});
    },
    listExtended: function(lightMode, withTagsFile, withGitInfo, nospinner) {
        return APIXHRService("GET", API_PATH + "projects/list-extended", {lightMode, withTagsFile, withGitInfo}, nospinner ? "nospinner" : undefined);
    },
    getExtended: function(projectKey, lightMode, withTagsFile, withGitInfo, nospinner) {
        return APIXHRService("GET", API_PATH + "projects/get-extended", {projectKey, lightMode, withTagsFile, withGitInfo}, nospinner ? "nospinner" : undefined);
    },
    listPromotedWikis: function(withHomeArticle, nospinner){
      return APIXHRService("GET", API_PATH + "projects/wikis/list-promoted", {withHomeArticle:withHomeArticle}, nospinner ? "nospinner" : undefined);
    },
    getGraph: function(layoutEngine, projectFolderId, recursive) {
        return APIXHRService("GET", API_PATH + "flow/projects/get-graph-serialized", {layoutEngine: layoutEngine, projectFolderId: projectFolderId, recursive: recursive});
    },
    checkDeletability: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/check-deletability", {projectKey: projectKey});
    },
    delete: function(projectKey, clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs) {
        return APIXHRService("POST", API_PATH + "projects/delete", {projectKey, clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs});
    },
    listAllKeys: function() {
        return APIXHRService("GET", API_PATH + "projects/list-all-keys");
    },
    listReadableKeys: function() {
        return APIXHRService("GET", API_PATH + "projects/list-readable-keys");
    },
    listAllTags: function() {
        return APIXHRService("GET", API_PATH + "projects/list-all-tags");
    },
    getSummary: function(projectKey, options = {}) {
        return APIXHRService("GET", API_PATH + "projects/get-summary", {projectKey: projectKey, ...options});
    },
    getProjectGovernanceStatus: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-governance-status", {projectKey: projectKey}, "nospinner");
    },
    getProjectAccessInfo: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-project-access-info", { projectKey: projectKey });
    },
    getVisibleSummary: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-visible-summary", { projectKey: projectKey });
    },
    create: function(projectKey, name, projectFolderId) {
         return APIXHRService("POST", API_PATH + "projects/create", {projectKey: projectKey, name: name, projectFolderId: projectFolderId});
    },
    getSettings: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-settings", {projectKey: projectKey});
    },
    getPermissions: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-permissions", {projectKey: projectKey});
    },
    getDashboardAuthorizations: function(projectKey, withLicensingCheck = true) {
        return APIXHRService("GET", API_PATH + "projects/get-dashboard-authorizations", {projectKey, withLicensingCheck});
    },
    getExposedObjects: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-exposed-objects", {projectKey: projectKey});
    },
    getEnrichedExposedObjects: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-enriched-exposed-objects", {projectKey: projectKey});
    },
    saveExposedObjects: function(projectKey, exposedObjects) {
        return APIXHRService("POST", API_PATH + "projects/save-exposed-objects", {
            projectKey: projectKey,
            exposedObjects: JSON.stringify(exposedObjects, stripDollarKeys)
        });
    },
    addExposedObject: function(projectKey, type, objectId, targetProjectKey, targetZoneId) {
        return APIXHRService("POST", API_PATH + "projects/add-exposed-object", {
            projectKey,
            type,
            objectId,
            targetProjectKey,
            targetZoneId
        });
    },
    getObjectAuthorizations: function(projectKey, objectType, objectId) {
        return APIXHRService("GET", API_PATH + "projects/get-object-authorizations", {projectKey: projectKey, objectType: objectType, objectId: objectId});
    },
    getObjectExposition: function(projectKey, objectType, objectId) {
        return APIXHRService("GET", API_PATH + "projects/get-object-exposition", {projectKey: projectKey, objectType: objectType, objectId: objectId});
    },
    getObjectsExpositions: function(items) {
        //POST because request might be big
        return APIXHRService("POST", API_PATH + "projects/get-objects-expositions", {
            items: JSON.stringify(items)
        });
    },
    saveObjectExposition: function(projectKey, objectType, objectId, objectExposition) {
        return APIXHRService("POST", API_PATH + "projects/save-object-exposition", {projectKey: projectKey, objectType: objectType, objectId: objectId, objectExposition: JSON.stringify(objectExposition) });
    },
    appendObjectExposition: function(projectKey, objectType, objectId, newProjectsSharedTo, quickShareEnabled=undefined) { // quickShareEnabled == undef or null means 'do not attempt to change it'
        return APIXHRService("POST", API_PATH + "projects/append-object-expositions", {projectKey: projectKey, objectType: objectType, objectId: objectId, newProjectsSharedTo: JSON.stringify(newProjectsSharedTo), quickShareEnabled });
    },
    addObjectsExpositions: function(items, settings) {
        return APIXHRService("POST", API_PATH + "projects/add-objects-expositions", {
            items: JSON.stringify(items),
            settings: JSON.stringify(settings)
        });
    },
    enableQuickSharing: function(projectKey, objectType, objectId) {
        return APIXHRService("POST", API_PATH + "projects/enable-quick-sharing", {projectKey: projectKey, objectType: objectType, objectId: objectId });
    },
    unshare: function(items, targetProjectKey) {
        return APIXHRService("POST", API_PATH + "projects/unshare", {
            items: JSON.stringify(items),
            targetProjectKey: targetProjectKey
        });
    },
    addReaderAuthorizations: function(projectKey, readerAuthorizations) {
        return APIXHRService("POST", API_PATH + "projects/add-reader-authorizations", {projectKey: projectKey, readerAuthorizations: JSON.stringify(readerAuthorizations)});
    },
    saveDashboardAuthorizations: function(projectKey, dashboardAuthorizations) {
        return APIXHRService("POST", API_PATH + "projects/save-dashboard-authorizations", {projectKey: projectKey, dashboardAuthorizations: JSON.stringify(dashboardAuthorizations)});
    },
    getAdditionalDashboardUsers : function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-additional-dashboard-users", {projectKey: projectKey})
    },
    saveAdditionalDashboardUsers: function(projectKey, additionalDashboardUsers) {
        return APIXHRService("POST", API_PATH + "projects/save-additional-dashboard-users", {projectKey: projectKey, data: JSON.stringify(additionalDashboardUsers)});
    },
    upgradePermissionsVersion: function(projectKey) {
        return APIXHRService("POST", API_PATH + "projects/upgrade-permissions-version", { projectKey });
    },
    saveSummary: function(projectKey, data) {
        return APIXHRService("POST", API_PATH + "projects/save-summary", {projectKey: projectKey, data: JSON.stringify(data)});
    },
    saveSettings: function(projectKey, data) {
        return APIXHRService("POST", API_PATH + "projects/save-settings", {projectKey: projectKey, data: JSON.stringify(data)});
    },
    savePermissions: function(projectKey, project){
      return APIXHRService("POST", API_PATH + "projects/save-permissions", {projectKey: projectKey, project: JSON.stringify(project)});
    },
    savePermissionsNonAdmin: function(projectKey, permissions){
      return APIXHRService("POST", API_PATH + "projects/save-permissions-non-admin", {projectKey: projectKey, permissions: JSON.stringify(permissions)});
    },
    resendProjectInvitationEmail: function(projectKey, email){
      return APIXHRService("POST", API_PATH + "projects/resend-project-invitation-email", {projectKey: projectKey, email: email});
    },
    createTutorial: function(tutorialId, tutorialType, projectFolderId, nospinner=false) {
        return APIXHRService("POST", API_PATH + "projects/create-tutorial", {id: tutorialId, type: tutorialType, projectFolderId: projectFolderId}, nospinner ? "nospinner" : undefined);
    },
    downloadTutorial: function(tutorialId, tutorialType) {
        return APIXHRService("POST", API_PATH + "projects/download-tutorial", {id: tutorialId, type: tutorialType})
    },
    prepareImportTutorial: function(importId) {
        return APIXHRService("POST", API_PATH + "projects/prepare-tutorial", {importId: importId})
    },
    startImportTutorial: function(importId, tutorialId, targetProjectKey, targetProjectFolderId) {
        return APIXHRService("POST", API_PATH + "projects/start-import-tutorial", {
            importId: importId,
            tutorialId: tutorialId,
            targetProjectKey: targetProjectKey,
            targetProjectFolderId: targetProjectFolderId,
        })
    },
    listTutorials: function() {
        return APIXHRService("GET", API_PATH + "tutorials/list");
    },
    startProjectExport: function(projectKey, exportOptions) {
        return APIXHRService("POST", API_PATH + "projects/start-export", {
            projectKey: projectKey,
            exportOptions: JSON.stringify(exportOptions)
        })
    },
    startProjectDuplication: function(projectKey, duplicateOptions) {
        return APIXHRService("POST", API_PATH + "projects/duplicate", {
            projectKey: projectKey,
            duplicateOptions: JSON.stringify(duplicateOptions)
        })
    },
    getProjectDatasets: function(projectKey){
        return APIXHRService("POST", API_PATH + "projects/export/prepare", {
            projectKey: projectKey
        })
    },
    getProjectExportURL: function(projectKey, exportId) {
        return API_PATH + "projects/download-export?projectKey=" + encodeURIComponent(projectKey)
        + '&exportId='+encodeURIComponent(exportId);
    },
    uploadForImport: function(file, callback) {
        return uploadFileRequest("projects/import/upload", function(formdata) {
            formdata.append("file", file);
        }, callback)
    },
    startImport: function(importId, importSettings) {
        return APIXHRService("POST", API_PATH + "projects/import/start", {
            importId: importId,
            importSettings: JSON.stringify(importSettings)
        })
    },
    prepareImport: function(importId, importSettings) {
        return APIXHRService("POST", API_PATH + "projects/import/prepare", {
            importId: importId,
            importSettings: JSON.stringify(importSettings)
        })
    },
    resyncHDFSDatasetPermissions : function(projectKey) {
        return APIXHRService("POST", API_PATH + "projects/admin/resync-hdfs-permissions", {
            projectKey: projectKey
        })
    },
    governIntegrationSync: function(projectKey) {
        return APIXHRService("POST", API_PATH + "projects/admin/govern-integration-sync", {
            projectKey: projectKey
        });
    },
    getAppManifest: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-app-manifest", {
            projectKey: projectKey
        })
    },
    saveAppManifest: function(projectKey, appManifest) {
        return APIXHRService("POST", API_PATH + "projects/save-app-manifest", {
            projectKey: projectKey,
            appManifest: JSON.stringify(appManifest)
        })
    },
    getAppRemapping: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-app-remapping", {
            projectKey: projectKey
        })
    },
    saveAppRemapping: function(projectKey, settings) {
        return APIXHRService("POST", API_PATH + "projects/save-app-remapping", {
            projectKey: projectKey,
            settings: JSON.stringify(settings)
        })
    },
    getProjectFeaturesUsage: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/get-project-features-usage", {
            projectKey: projectKey
        });
    },
    publicApi: {
        listProjectApiKeys: function(projectKey) {
            return APIXHRService("GET", API_PATH + "admin/publicapi/get-project-api-keys", {
                projectKey: projectKey
            });
        },
        createProjectApiKey: function(key) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/create-project-api-key", {
                key: JSON.stringify(key)
            });
        },
        saveProjectApiKey: function(key) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/save-project-api-key", {
                key: JSON.stringify(key)
            });
        },
        deleteProjectApiKey: function(projectKey, keyId) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/delete-project-api-key", {
                projectKey: projectKey,
                keyId: keyId
            });
        },
        getApiKeyDetails: function(projectKey, secret) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/get-api-key-details", {
                projectKey: projectKey,
                secret: secret
            });
        }
    },
    variables: {
        get: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/variables/get", {
                projectKey: projectKey
            })
        },
        save: function(projectKey, data) {
            return APIXHRService("POST", API_PATH + "projects/variables/save", {
                projectKey: projectKey,
                data: JSON.stringify(data)
            })
        }
    },
    design: {
        prepareBundleCreation: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/design/prepare-bundle-creation", {
                projectKey: projectKey
            })
        },
        createBundle: function(projectKey, bundleId, preparationResult, evaluateProjectStandardsChecks) {
            return APIXHRService("POST", API_PATH + "projects/design/create-bundle", {
                projectKey: projectKey,
                bundleId: bundleId,
                preparationResult: JSON.stringify(preparationResult),
                evaluateProjectStandardsChecks: evaluateProjectStandardsChecks
            });
        },
        deleteBundle: function(projectKey, bundleId) {
            return APIXHRService("POST", API_PATH + "projects/design/delete-bundle", {
                projectKey: projectKey,
                bundleId: bundleId
            })
        },
        deleteFailedBundle: function(projectKey, bundleId) {
            return APIXHRService("POST", API_PATH + "projects/design/delete-failed-bundle", {
                projectKey: projectKey,
                bundleId: bundleId
            })
        },
        listBundles: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/design/list-bundles", {
                projectKey: projectKey
            })
        },
        getBundleDetails: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "projects/design/get-bundle-details", {
                projectKey: projectKey,
                bundleId: bundleId
            })
        },
        getBundleDownloadURL: function(projectKey, bundleId) {
            return API_PATH
                    +"projects/design/download-bundle?projectKey=" + encodeURIComponent(projectKey)
                    + '&bundleId='+ encodeURIComponent(bundleId);
        },
        getBundleExporterSettings: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/design/get-bundle-exporter-settings", {
                projectKey: projectKey
            })
        },
        saveBundleExporterSettings: function(projectKey, settings) {
            return APIXHRService("POST", API_PATH + "projects/design/save-bundle-exporter-settings", {
                projectKey: projectKey, settings: JSON.stringify(settings)
            })
        },
        checkBundleReversion: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "projects/design/check-bundle-reversion", {
                projectKey: projectKey, bundleId: bundleId
            })
        },
        revertBundle: function(projectKey, bundleId, importOptions) {
            return APIXHRService("POST", API_PATH + "projects/design/revert-bundle", {
                projectKey: projectKey, bundleId: bundleId, importOptions: JSON.stringify(importOptions)
            })
        },
        publishToDeployer: function(projectKey, bundleId, publishedProjectKey) {
            return APIXHRService("POST", API_PATH + "projects/design/publish-to-project-deployer", {
                projectKey, bundleId, publishedProjectKey
            });
        },
        getBundleGovernanceStatus: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "projects/design/get-governance-status", {
                projectKey: projectKey,
                bundleId: bundleId,
            }, "nospinner");
        }
    },
    automation: {
        createWithInitialBundle: function(file, projectFolderId, projectKey, callback) {
            return uploadFileRequest("projects/automation/create-with-initial-bundle", function(formdata) {
                formdata.append("file", file);
                formdata.append("projectFolderId", projectFolderId || "");
                formdata.append("projectKey", projectKey || "");
            }, callback);
        },
        importBundle: function(projectKey, file) {
            return uploadFileRequest("projects/automation/import-bundle", function(formdata) {
                formdata.append("projectKey", projectKey);
                formdata.append("file", file);
            }, null);
        },
        listBundles: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/automation/list-bundles", {
                projectKey: projectKey
            })
        },
        getBundleDetails: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "projects/automation/get-bundle-details", {
                projectKey: projectKey, bundleId: bundleId
            })
        },
        checkBundleActivation: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "projects/automation/check-bundle-activation", {
                projectKey: projectKey, bundleId: bundleId
            })
        },
        preloadBundle: function(projectKey, bundleId) {
            return APIXHRService("POST", API_PATH + "projects/automation/preload-bundle", {
                projectKey: projectKey, bundleId: bundleId
            })
        },
        activateBundle: function(projectKey, bundleId) {
            return APIXHRService("POST", API_PATH + "projects/automation/activate-bundle", {
                projectKey: projectKey, bundleId: bundleId
            })
        },
        getBundleActivationSettingsExt: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/automation/get-activation-settings-ext", {
                projectKey: projectKey
            })
        },
        saveBundleActivationSettings: function(projectKey, settings) {
            return APIXHRService("POST", API_PATH + "projects/automation/save-activation-settings", {
                projectKey: projectKey, settings: JSON.stringify(settings)
            });
        },
        deleteBundle: function(projectKey, bundleId) {
            return APIXHRService("POST", API_PATH + "projects/automation/delete-bundle", {
                projectKey: projectKey, bundleId: bundleId
            })
        },
    },
    activity: {
        getActivitySummary: function(projectKey, timeSpan) {
            return APIXHRService("GET", API_PATH + "projects/activity/get-summary", {
                projectKey: projectKey, timeSpan: timeSpan
            })
        }
    },
    folderEdit: {
        listContents: function(projectKey, type) {
            return APIXHRService("GET", API_PATH + "projects/folder-edition/list-contents", {
                projectKey: projectKey, type: type
            });
        },
        getContent: function(projectKey, type, path, sendAnyway) {
            return APIXHRService("GET", API_PATH + "projects/folder-edition/get-content", {
                projectKey: projectKey, type: type, path: path, sendAnyway: sendAnyway
            });
        },
        previewImageURL: function(projectKey, type, path, mimeType) {
            return '/dip/api/projects/folder-edition/preview-image?projectKey=' + projectKey + '&type=' + type + '&path=' + encodeURIComponent(path) + '&contentType=' + encodeURIComponent(mimeType);
        },
        setContent: function(projectKey, type, path, data) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/set-content", {
                projectKey: projectKey, type: type, path: path, data: data
            });
        },
        setContentMultiple: function(projectKey, type, contentMap) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/set-content-multiple", {
                projectKey: projectKey, type: type, contentMap: JSON.stringify(contentMap)
            });
        },
        createContent: function(projectKey, type, path, isFolder) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/create-content", {
                projectKey: projectKey, type: type, path: path, isFolder: isFolder
            });
        },
        deleteContent: function(projectKey, type, path) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/delete-content", {
                projectKey: projectKey, type: type, path: path
            });
        },
        decompressContent: function(projectKey, type, path) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/decompress-content", {
                projectKey: projectKey, type: type, path: path
            });
        },
        renameContent: function(projectKey, type, path, newName) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/rename-content", {
                projectKey: projectKey, type: type, path: path, newName: newName
            });
        },
        moveContent: function(projectKey, type, path, toPath) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/move-content", {
                projectKey: projectKey, type: type, path: path, toPath: toPath
            });
        },
        copyContent: function(projectKey, type, path) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/copy-content", {
                projectKey: projectKey, type: type, path: path
            });
        },
        uploadContent: function(projectKey, type, path, file, callback) {
            return uploadFileRequest("projects/folder-edition/upload-content", function(formdata) {
                formdata.append("projectKey", projectKey);
                formdata.append("type", type);
                formdata.append("path", path);
                formdata.append("file", file);
            }, callback);
        },
        checkUploadContent: function(projectKey, type, path, filePaths) {
            return APIXHRService("POST", API_PATH + "projects/folder-edition/check-upload-content", {
                projectKey: projectKey, type: type, path: path, filePaths: JSON.stringify(filePaths)
            });
        },
        downloadURL: function(projectKey, type, path) {
            return '/dip/api/projects/folder-edition/download-content?projectKey=' + projectKey + '&type=' + type + '&path=' + encodeURIComponent(path);
        }
    },
    resourcesFolderEdit: {
        listContents: function(projectKey, type) {
            return APIXHRService("GET", API_PATH + "projects/resources-folder-edition/list-contents", {
                projectKey: projectKey, type: type
            });
        },
        getContent: function(projectKey, type, path, sendAnyway) {
            return APIXHRService("GET", API_PATH + "projects/resources-folder-edition/get-content", {
                projectKey: projectKey, type: type, path: path, sendAnyway: sendAnyway
            });
        },
        previewImageURL: function(projectKey, type, path, mimeType) {
            return '/dip/api/projects/resources-folder-edition/preview-image?projectKey=' + projectKey + '&type=' + type + '&path=' + encodeURIComponent(path) + '&contentType=' + encodeURIComponent(mimeType);
        },
        setContent: function(projectKey, type, path, data) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/set-content", {
                projectKey: projectKey, type: type, path: path, data: data
            });
        },
        setContentMultiple: function(projectKey, type, contentMap) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/set-content-multiple", {
                projectKey: projectKey, type: type, contentMap: JSON.stringify(contentMap)
            });
        },
        createContent: function(projectKey, type, path, isFolder) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/create-content", {
                projectKey: projectKey, type: type, path: path, isFolder: isFolder
            });
        },
        deleteContent: function(projectKey, type, path) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/delete-content", {
                projectKey: projectKey, type: type, path: path
            });
        },
        decompressContent: function(projectKey, type, path) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/decompress-content", {
                projectKey: projectKey, type: type, path: path
            });
        },
        renameContent: function(projectKey, type, path, newName) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/rename-content", {
                projectKey: projectKey, type: type, path: path, newName: newName
            });
        },
        moveContent: function(projectKey, type, path, toPath) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/move-content", {
                projectKey: projectKey, type: type, path: path, toPath: toPath
            });
        },
        copyContent: function(projectKey, type, path) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/copy-content", {
                projectKey: projectKey, type: type, path: path
            });
        },
        uploadContent: function(projectKey, type, path, file, callback) {
            return uploadFileRequest("projects/resources-folder-edition/upload-content", function(formdata) {
                formdata.append("projectKey", projectKey);
                formdata.append("type", type);
                formdata.append("path", path);
                formdata.append("file", file);
            }, callback);
        },
        checkUploadContent: function(projectKey, type, path, filePaths) {
            return APIXHRService("POST", API_PATH + "projects/resources-folder-edition/check-upload-content", {
                projectKey: projectKey, type: type, path: path, filePaths: JSON.stringify(filePaths)
            });
        },
        downloadURL: function(projectKey, type, path) {
            return '/dip/api/projects/resources-folder-edition/download-content?projectKey=' + projectKey + '&type=' + type + '&path=' + encodeURIComponent(path);
        }
    },
    checkReaderAuthorizations: function(projectKey, readerAuthorizations) {
        return APIXHRService("POST", API_PATH + "projects/check-reader-authorizations", {
            projectKey: projectKey,
            readerAuthorizations: JSON.stringify(readerAuthorizations)
        });
    },
    listComputedMetrics: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/list-computed-metrics", {
            projectKey: projectKey
        });
    },
    saveMetrics: function(projectKey, metrics, checks) {
        return APIXHRService("POST", API_PATH + "projects/save-metrics", {
            projectKey: projectKey,
            metricsData: JSON.stringify(metrics),
            checksData: JSON.stringify(checks)
        });
    },
    listAvailableMetrics: function(projectKey, folderId) {
        return APIXHRService("GET", API_PATH + "projects/list-available-metrics", {
            projectKey: projectKey,
            folderId: folderId
        });
    },
    getPreparedMetricHistory: function(projectKey, partitionId, metric, metricId) {
        return APIXHRService("GET", API_PATH + "projects/get-prepared-metric-history", {
            projectKey: projectKey,
            data: JSON.stringify(metric),
            metricId: metricId,
            partitionId: partitionId
        });
    },
    getPreparedMetricHistories: function(projectKey, displayedState) {
        return APIXHRService("POST", API_PATH + "projects/get-prepared-metric-histories", {
            projectKey: projectKey,
            data: JSON.stringify(displayedState || {})
        });
    },
    getCheckHistories: function(projectKey, displayedState) {
        return APIXHRService("POST", API_PATH + "projects/get-prepared-check-histories", {
            projectKey: projectKey,
            data: JSON.stringify(displayedState || {})
        });
    },
    listComputedChecks: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/list-computed-checks", {
            projectKey: projectKey
        });
    },
    createMetricsDataset: function(projectKey, view, partition, filter) {
        return APIXHRService("GET", API_PATH + "datasets/create-metrics-dataset", {
            projectKey: projectKey,
            objectId: '',
            view: view,
            partition: partition,
            filter: filter,
            taggableType: "PROJECT"
        });
    },
    clearMetrics: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/clear-metrics", {
            projectKey: projectKey
        });
    },
    saveExternalMetricsValues: function(projectKey, metrics, types) {
        return APIXHRService("POST", API_PATH + "projects/save-external-metrics-values", {
            projectKey: projectKey,
            data: JSON.stringify(metrics),
            typesData: JSON.stringify(types || {})
        });
    },
    saveExternalChecksValues: function(projectKey, checks) {
        return APIXHRService("POST", API_PATH + "projects/save-external-checks-values", {
            projectKey: projectKey,
            data: JSON.stringify(checks)
        });
    },
    switchAppType: function(projectKey, appType, settings, manifest) {
        return APIXHRService("POST", API_PATH + "projects/switch-app-type", {
            projectKey: projectKey,
            appType: appType,
            settings: settings ? JSON.stringify(settings) : null,
            manifest: manifest ? JSON.stringify(manifest) : null
        });
    },
    setSetupSection: function(projectKey, enabled) {
        return APIXHRService("POST", API_PATH + "projects/set-setup-section", {projectKey, enabled})
    },
    createOrUpdatePlugin: function(projectKey, pluginId, appName) {
        return APIXHRService("POST", API_PATH + "projects/create-or-update-plugin", {
            projectKey: projectKey,
            pluginId: pluginId,
            appName: appName
        });
    },
    git: {
        pull: function(projectKey, remoteName, branchName) {
            return APIXHRService("GET", API_PATH + "projects/git/pull", {
                projectKey: projectKey,
                remoteName: remoteName,
                branchName: branchName
            });
        },
        fetch: function(projectKey, remoteName) {
            return APIXHRService("GET", API_PATH + "projects/git/fetch", {
                projectKey: projectKey,
                remoteName: remoteName
            });
        },
        push: function(projectKey, remoteName, branchName) {
            return APIXHRService("GET", API_PATH + "projects/git/push", {
                projectKey: projectKey,
                remoteName: remoteName,
                branchName: branchName
            });
        },
        getDSSVersionForASpecificCommit: function(projectKey, hash) {
            return APIXHRService("GET", API_PATH + "git/dss-version-commit", {
                projectKey: projectKey,
                hash: hash,
            });
        },
        resetToUpstream: function(projectKey, remoteName, branchName) {
            return APIXHRService("POST", API_PATH + "projects/git/reset-to-upstream", {
                projectKey: projectKey,
                remoteName: remoteName,
                branchName: branchName
            });
        },
        resetToHead: function(projectKey) {
            return APIXHRService("POST", API_PATH + "projects/git/reset-to-head", {
                projectKey: projectKey
            });
        },
        setDirty: function(projectKey, gitLib) {
            return APIXHRService("POST", API_PATH + "projects/git/set-dirty", {
                projectKey: projectKey, gitLibs: JSON.stringify(Array(gitLib))
            });
        },
        setAllDirty: function(projectKey, gitLibs) {
            return APIXHRService("POST", API_PATH + "projects/git/set-dirty", {
                projectKey: projectKey, gitLibs: JSON.stringify(Array.from(gitLibs))
            });
        },
        getDirty: function(projectKey, gitLib) {
            return APIXHRService("POST", API_PATH + "projects/git/get-dirty", {
                projectKey: projectKey, gitLib
            });
        },
        getAllDirty: function(projectKey) {
            return APIXHRService("POST", API_PATH + "projects/git/get-dirty", {
                projectKey: projectKey
            });
        },
        getFullStatus: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/git/get-full-status", {
                projectKey: projectKey
            });
        },
        listBranches: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/git/list-branches", {
                projectKey
            });
        },
        listBranchesByType: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/git/list-branches-by-type", {
                projectKey
            });
        },
        deleteBranches: function(projectKey, /*String[]*/branchNames, deleteOptions) {
            return APIXHRService("GET", API_PATH + "projects/git/delete-branches", {
                projectKey: projectKey,
                branchNames: JSON.stringify(branchNames),
                remoteDelete: deleteOptions.remoteDelete,
                forceDelete: deleteOptions.forceDelete
            });
        },
        commit: function(projectKey, commitMessage) {
            return APIXHRService("GET", API_PATH + "projects/git/commit", {
                projectKey: projectKey,
                commitMessage: commitMessage
            });
        },
        prepareCommit: function(projectKey) {
            return APIXHRService("GET", API_PATH + "projects/git/prepare-commit", {
                projectKey: projectKey
            });
        },
        createBranch: function(projectKey, branchName, commitId) {
            return APIXHRService("GET", API_PATH + "projects/git/create-branch", {
                projectKey: projectKey,
                branchName: branchName,
                commitId: commitId
            });
        },
        switchBranch: function(projectKey, branchName, clearOutputDatasets) {
            return APIXHRService("GET", API_PATH + "projects/git/switch-branch", {
                projectKey: projectKey,
                branchName: branchName,
                clearOutputDatasets: clearOutputDatasets
            });
        },
        setRemote: function(projectKey, remoteName, remoteUrl) {
            return APIXHRService("POST", API_PATH + "projects/git/set-remote", {
                projectKey: projectKey,
                remoteName: remoteName,
                remoteUrl: remoteUrl
            });
        },
        removeRemote: function(projectKey, remoteName) {
            return APIXHRService("GET", API_PATH + "projects/git/rm-remote", {
                projectKey: projectKey,
                remoteName: remoteName
            });
        },
        listProjectsMatchingRemoteRepository: function(projectKey, branchName) {
            return APIXHRService("GET", API_PATH + "projects/git/list-projects-matching-remote-repository", {
                projectKey: projectKey,
                branchName: branchName
            });
        }
    }
},
projectFolders: {
    listRootContents: (lightMode, withTagsFile, includeLimitedVisibility, withGitInfo = false) =>
        APIXHRService("GET", `${API_PATH}project-folders/list-contents`, { lightMode, withTagsFile, includeLimitedVisibility, withGitInfo }),
    listContents: (folderId, lightMode, withTagsFile, includeLimitedVisibility, maxLevel = -1, silent = false, withGitInfo = false) =>
        APIXHRService("GET",  `${API_PATH}project-folders/${folderId}/list-contents`, { lightMode, withTagsFile, includeLimitedVisibility, maxLevel, withGitInfo }, silent === true ? 'nospinner' : undefined),
    getSummary: (folderId, maxLevel, includeAncestry, silent, includeProjectKeys = false) =>
        APIXHRService("GET", `${API_PATH}project-folders/${folderId}/summary`,{maxLevel, includeAncestry, includeProjectKeys},  silent === true ? 'nospinner' : undefined),
    getDefaultFolderSummary: (contextFolderId, silent) =>
        APIXHRService("GET", `${API_PATH}project-folders/${contextFolderId}/default-folder-summary`,{}, silent === true ? 'nospinner' : undefined),
    create: (parentId, name) =>
        APIXHRService("POST", `${API_PATH}project-folders/create`, { folderId: parentId, name }),
    moveItems: (destination, folderIds, projectKeys = null) =>
        APIXHRService("POST", `${API_PATH}project-folders/move-items`, { folderIds: JSON.stringify(folderIds), projectKeys: JSON.stringify(projectKeys), destination }),
    delete: (folders, destination) =>
        APIXHRService("POST", `${API_PATH}project-folders/delete`, { folderIds: JSON.stringify(folders), destination }),
    getSettings: (folderId) =>
        APIXHRService("GET", `${API_PATH}project-folders/${folderId}/settings`),
    setSettings: (folderId, settings) =>
        APIXHRService("PUT", `${API_PATH}project-folders/${folderId}/settings`, { projectFolderSettings: JSON.stringify(settings) }),
    getEffectiveReaders: (folderId) =>
        APIXHRService("GET", `${API_PATH}project-folders/${folderId}/list-readers`, {}, 'nospinner'),
    listExtended: silent =>
        APIXHRService("GET", `${API_PATH}project-folders/`, {}, silent === true ? 'nospinner' : undefined),
},
globalfinder: {
    search: (query, limit = 10, contextualProjectKey) => APIXHRService("POST", `${API_PATH}search/`, { query, limit, contextualProjectKey}, 'nospinner'),
},
apps: {
    listTemplates: function(includeLimitedVisibility, noSpinner) {
        return APIXHRService("GET", API_PATH + "apps/list-templates", { includeLimitedVisibility }, noSpinner ? "nospinner" : undefined);
    },
    listInstances: function() {
        return APIXHRService("GET", API_PATH + "apps/list-instances");
    },
    getAccessInfo: function(appId) {
        return APIXHRService("GET", API_PATH + "apps/get-access-info", {appId: appId});
    },
    getVisibleTemplateSummary: function(appId) {
        return APIXHRService("GET", API_PATH + "apps/get-visible-template-summary", {appId: appId});
    },
    getTemplateSummary: function(appId) {
        return APIXHRService("GET", API_PATH + "apps/get-template-summary", {appId: appId});
    },
    getInstanceSummary: function(projectKey) {
        return APIXHRService("GET", API_PATH + "apps/get-instance-summary", {projectKey: projectKey});
    },
    instantiate: function(appId, targetProjectKey, targetProjectLabel) {
        return APIXHRService("POST", API_PATH + "apps/instantiate", {
            appId: appId,
            targetProjectKey:targetProjectKey,
            targetProjectLabel: targetProjectLabel
        })
    },
    updateInstance: function(appId, projectKey, keepProjectDescription, keepVariables) {
        return APIXHRService("POST", API_PATH + "apps/update-instance", {
            appId: appId,
            projectKey: projectKey,
            keepProjectDescription: keepProjectDescription,
            keepVariables: keepVariables
        })
    },
    skipInstanceUpdate: function(appId, projectKey, versionToSkip) {
        return APIXHRService("POST", API_PATH + "apps/skip-instance-update", {
            appId: appId,
            projectKey: projectKey,
            versionToSkip: versionToSkip
        })
    },
    createOrUpdateTestInstance: function(appId, fullUpdate) {
        return APIXHRService("POST", API_PATH + "apps/create-or-update-test-instance", {appId: appId, fullUpdate: fullUpdate});
    },
    getTestInstance: function(appId) {
        return APIXHRService("GET", API_PATH + "apps/get-test-instance", {appId: appId});
    },
    getAppRecipeUsability: function(recipeType) {
        return APIXHRService("GET", API_PATH + "apps/get-app-recipe-usability", {recipeType: recipeType});
    },
    checkInstancesDeletability: function(appId, projectKeys) {
        return APIXHRService("GET", API_PATH + "apps/check-instances-deletability", {appId:appId, projectKeys: JSON.stringify(projectKeys)});
    },
    deleteInstances: function(appId, projectKeys, clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs) {
        return APIXHRService("POST", API_PATH + "apps/delete-instances", {
            appId:appId,
            projectKeys: JSON.stringify(projectKeys),
            clearManagedDatasets: clearManagedDatasets,
            clearOutputManagedFolders: clearOutputManagedFolders,
            clearJobAndScenarioLogs: clearJobAndScenarioLogs
        });
    }
},
git: {
    getObjectLog: function(projectKey, objectType, objectId, since, count, projectZone) {
        return APIXHRService("GET", API_PATH + "git/get-object-log", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId,
            since: since,
            count: count,
            projectZone: projectZone
        });
    },
    getObjectLogSince: function(projectKey, objectType, objectId, branch, since, count) {
        return APIXHRService("GET", API_PATH + "git/get-object-log-since", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId,
            branch: branch,
            since: since,
            count: count
        });
    },
    getObjectLogUntil: function(projectKey, objectType, objectId, branch, until, count) {
        return APIXHRService("GET", API_PATH + "git/get-object-log-until", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId,
            branch: branch,
            until: until,
            count: count
        });
    },
    revertObjectToRevision: function(projectKey, objectType, objectId, hash) {
        return APIXHRService("POST", API_PATH + "git/revert-object-to-revision", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId,
            hash:hash
        });
    },
    revertProjectToRevision: function(projectKey, hash, projectZone) {
        return APIXHRService("POST", API_PATH + "git/revert-project-to-revision", {
            projectKey,
            hash,
            projectZone
        });
    },

    revertSingleCommit : function(projectKey, objectRef, hash) {
        return APIXHRService("POST", API_PATH + "git/revert-single-commit", {
            projectKey: projectKey,
            objectRef: objectRef,
            hash:hash
        });
    },

    prepareObjectCommit: function(projectKey, objectType, objectId) {
        return APIXHRService("GET", API_PATH + "git/prepare-object-commit", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId
        });
    },
    commitObject: function(projectKey, objectType, objectId, message) {
        return APIXHRService("POST", API_PATH + "git/commit-object", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId, message: message
        });
    },
    addTag: function(projectKey, objectType, objectId, reference, name, message=null) {
        return APIXHRService("POST", API_PATH + "git/tag", {
            projectKey, objectType, objectId, reference, name, message
        });
    },
    removeTag: function(projectKey, objectType, objectId, name) {
        return APIXHRService("POST", API_PATH + "git/remove-tag", {
            projectKey, objectType, objectId, name
        });
    },
    getCommitDiff: function(projectKey, objectRef, commitId) {
        return APIXHRService("GET", API_PATH + "git/get-commit-diff", {
            projectKey: projectKey,
            objectRef: objectRef,
            commitId: commitId
        });
    },
    getRevisionsDiff: function(projectKey, commitFrom, commitTo, objectRef) {
        return APIXHRService("GET", API_PATH + "git/get-revisions-diff", {
            projectKey: projectKey,
            commitFrom: commitFrom,
            commitTo: commitTo,
            objectRef: JSON.stringify(objectRef)
        });
    },
    listRemoteRefs: function(repository, login, password) {
        return APIXHRService("POST", API_PATH + "git/list-remote-references", {
            repository, login, password
        });
    },
    listRemotes : function(projectKey) {
        return APIXHRService("GET", API_PATH + "git/list-remotes", {
            projectKey: projectKey
        })
    },
    addRemote : function(projectKey, name, url) {
        return APIXHRService("POST", API_PATH + "git/add-remote", {
            projectKey: projectKey,
            name: name,
            url: url
        })
    },
    removeRemote : function(projectKey, name) {
        return APIXHRService("POST", API_PATH + "git/remove-remote", {
            projectKey: projectKey,
            name: name
        })
    },
    setProjectGitRef: function(projectKey, gitRef, gitRefPath, addPythonPath) {
        return APIXHRService("POST", API_PATH + "git/set-project-git-ref", {
            projectKey: projectKey,
            gitReference: JSON.stringify(gitRef),
            gitReferencePath: gitRefPath,
            addPythonPath: addPythonPath
        });
    },
    rmProjectGitRef: function(projectKey, gitRefPath, deleteDirectory) {
        return APIXHRService("POST", API_PATH + "git/rm-project-git-ref", {
            projectKey: projectKey,
            gitReferencePath: gitRefPath,
            deleteDirectory: deleteDirectory
        });
    },
    pullProjectGitRef: function(projectKey, gitRefPath) {
        return APIXHRService("GET", API_PATH + "git/pull-project-git-ref", {
            projectKey: projectKey,
            gitReferencePath: gitRefPath
        });
    },
    getProjectExternalLibs: function(projectKey) {
        return APIXHRService("GET", API_PATH + "git/get-project-external-libraries", {
            projectKey: projectKey
        });
    },
    pullProjectGitRefs: function(projectKey) {
        return APIXHRService("GET", API_PATH + "git/pull-project-git-refs", {
            projectKey: projectKey
        });
    },
    pushProjectGitRefs: function (projectKey, commitMessage, gitReferencePath=null) {
        return APIXHRService("GET", API_PATH + "git/push-project-git-refs", {projectKey, commitMessage, gitReferencePath});
    },
    revertAllFiles: function (projectKey, gitLib) {
        return APIXHRService("GET", API_PATH + "git/revert-all-files", {projectKey, gitLib});
    },
    markAsResolved: function (projectKey, fileName, gitLibPath) {
        return APIXHRService("GET", API_PATH + "git/mark-as-resolved", {projectKey, fileName, gitLibPath});
    },
    mergeRequests: {
        create: function(projectKey, mergeRequest){
            return APIXHRService("POST", API_PATH + "git/merge-requests", {
                projectKey: projectKey,
                request: JSON.stringify(mergeRequest)
            });
        },
        list: function(projectKey){
            return APIXHRService("GET", API_PATH + "git/merge-requests", {projectKey});
        },
        get: function(projectKey, mergeRequestId){
            return APIXHRService("GET", API_PATH + "git/merge-requests/" + mergeRequestId, {projectKey});
        },
        update: function(projectKey, mergeRequest){
            return APIXHRService("PUT", API_PATH + "git/merge-requests/" + mergeRequest.id, {
                projectKey,
                request: JSON.stringify(mergeRequest)
            });
        },
        delete: function(projectKey, mergeRequestId){
            return APIXHRService("DELETE", API_PATH + "git/merge-requests/" + mergeRequestId + "?" +
                "projectKey=" + encodeURIComponent(projectKey)
            );
        },
        merge: function(projectKey, mergeRequestId){
            return APIXHRService("POST", API_PATH + "git/merge-requests/" + mergeRequestId + "/action/merge", {projectKey});
        },
        getFile: function(projectKey, mergeRequestId, file){
            return APIXHRService("GET", API_PATH + "git/merge-requests/" + mergeRequestId + "/file", {
                projectKey,
                file
            });
        },
        saveFile: function(projectKey, mergeRequestId, file, content){
            return APIXHRService("POST", API_PATH + "git/merge-requests/" + mergeRequestId + "/file", {
                projectKey,
                file,
                content
            });
        },
        markFileAsResolved: function(projectKey, mergeRequestId, file, resolutionStrategy){
            return APIXHRService("POST", API_PATH + "git/merge-requests/" + mergeRequestId + "/action/markFileAsResolved", {
                projectKey,
                file,
                resolutionStrategy
            });
        },
        markAllFilesAsResolved: function(projectKey, mergeRequestId, resolutionStrategy){
            return APIXHRService("POST", API_PATH + "git/merge-requests/" + mergeRequestId + "/action/markAllFilesAsResolved", {
                projectKey,
                resolutionStrategy
            });
        },
        getCommitDiff: function(projectKey, mergeRequestId, commitId){
            return APIXHRService("GET", API_PATH + "git/merge-requests/" + mergeRequestId + "/get-commit-diff", {
                projectKey,
                commitId
            });
        },
        getRevisionsDiff: function(projectKey, mergeRequestId, commitFrom, commitTo){
            return APIXHRService("GET", API_PATH + "git/merge-requests/" + mergeRequestId + "/get-revisions-diff", {
                projectKey,
                commitFrom,
                commitTo
            });
        },
        listBranches: function(projectKey) {
            return APIXHRService("GET", API_PATH + "git/merge-requests/list-branches", {
                projectKey
            });
        },
        refreshMergedBranch: function(projectKey, mergeRequestId) {
            return APIXHRService("POST", API_PATH + "git/merge-requests/" + mergeRequestId + "/action/refresh-merged-branch", {
                projectKey
            });
        }
    }
},
wikis : {
    getWiki: function(projectKey) {
        return APIXHRService("GET", API_PATH + "projects/wikis/get-wiki", {
            projectKey: projectKey
        });
    },
    getArticleSummary: function(projectKey, articleIdOrName) {
        return APIXHRService("GET", API_PATH + "projects/wikis/get-article-summary", {
            projectKey: projectKey,
            articleIdOrName: articleIdOrName
        });
    },
    getArticlePayload: function(projectKey, articleId) {
        return APIXHRService("GET", API_PATH + "projects/wikis/get-article-payload",  {
            projectKey: projectKey,
            articleId: articleId
        });
    },
    createArticle: function(projectKey, articleName, parent, templateDesc) {
        return APIXHRService("POST", API_PATH + "projects/wikis/create-article", {
            projectKey: projectKey,
            articleName: articleName,
            parent: parent,
            templateDesc: JSON.stringify(templateDesc)
        });
    },
    checkSaveConflict: function(article) {
        return APIXHRService("POST", API_PATH + "projects/wikis/check-save-article-conflict", {
            article: JSON.stringify(article)
        });
    },
    addAttachment: function(article, articleAttachment){
        let articleRequest = {
            article: JSON.stringify(article),
            articleAttachment: JSON.stringify(articleAttachment),
        };
        return APIXHRService("POST", API_PATH + "projects/wikis/add-attachment", articleRequest);
    },
    deleteAttachment: function(article, articleAttachment){
        let articleRequest = {
            article: JSON.stringify(article),
            articleAttachment: JSON.stringify(articleAttachment),
        };
        return APIXHRService("POST", API_PATH + "projects/wikis/delete-attachment", articleRequest);
    },
    saveArticle: function(article, payload, commitMessage) {
        let articleRequest = {
            article: JSON.stringify(article),
            commitMessage: commitMessage
        };
        if (angular.isString(payload)) {
            articleRequest['payload'] = payload;
        }
        return APIXHRService("POST", API_PATH + "projects/wikis/save-article", articleRequest);
    },
    changeArticleParent: function(projectKey, id, parentId) {
        return APIXHRService("POST", API_PATH + "projects/wikis/change-article-parent", {
            projectKey: projectKey,
            id: id,
            parentId: parentId
        });
    },
    editTaxonomy: function(projectKey, wiki) {
        return APIXHRService("POST", API_PATH + "projects/wikis/edit-taxonomy", {
            projectKey: projectKey,
            wiki: JSON.stringify(wiki)
        });
    },
    setHomeArticle: function(projectKey, homeArticleId) {
        return APIXHRService("POST", API_PATH + "projects/wikis/set-home-article", {
            projectKey: projectKey,
            homeArticleId: homeArticleId
        });
    },
    renameArticle: function(projectKey, oldId, newId) {
        return APIXHRService("POST", API_PATH + "projects/wikis/rename-article", {
            projectKey: projectKey,
            oldId: oldId,
            newId: newId
        });
    },
    deleteArticle: function(projectKey, articleId, deleteChildren) {
        return APIXHRService("POST", API_PATH + "projects/wikis/delete-article", {
            projectKey: projectKey,
            articleId: articleId,
            deleteChildren: deleteChildren
        });
    },
    listTemplates: function() {
        return APIXHRService("GET", API_PATH + "projects/wikis/list-templates", {});
    },
    upload: function(projectKey, articleId, file, callback) {
        return uploadFileRequest("projects/wikis/upload", function(formdata) {
            formdata.append("projectKey", projectKey);
            formdata.append("articleId", articleId);
            formdata.append("file", file);
        }, callback);
    },
    copyArticle: function(projectKey, articleName, parent, originalArticleId, withAttachments) {
        return APIXHRService("POST", API_PATH + "projects/wikis/copy-article", {
            projectKey: projectKey,
            articleName: articleName,
            parent: parent,
            originalArticleId: originalArticleId,
            withAttachments: withAttachments
        });
    },
    exportArticle: function(projectKey, articleId, exportFormat, exportChildren, exportAttachments) {
        return APIXHRService("POST", API_PATH + "projects/wikis/export", {
            projectKey: projectKey,
            articleId: articleId,
            exportFormat: JSON.stringify(exportFormat),
            exportChildren: exportChildren,
            exportAttachments: exportAttachments
        });
    },
    getExportURL: function(projectKey, exportId) {
        return API_PATH + "projects/wikis/download-export?"
        + "projectKey=" + encodeURIComponent(projectKey)
        + "&exportId=" + encodeURIComponent(exportId);
    },
    getAuthorizedExtensions() {
        return APIXHRService("GET", API_PATH + "projects/wikis/list-authorized-extensions");
    }
},
analysis: {
    listHeads: function(projectKey, withMLTasks) {
        return APIXHRService("GET", API_PATH + "analysis/list-heads", {
            projectKey: projectKey,
            withMLTasks: !!withMLTasks
        });
    },
    listOnDataset: function(projectKey, datasetSmartName, withMLTasks) {
        return APIXHRService("GET", API_PATH + "analysis/list-on-dataset", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName,
            withMLTasks: !!withMLTasks
        });
    },
    create: function(projectKey, inputDatasetSmartName, name) {
         return APIXHRService("POST", API_PATH + "analysis/create", {
            projectKey: projectKey,
            inputDatasetSmartName: inputDatasetSmartName,
            name: name
        });
    },
    createPredictionTemplate: function(projectKey, inputDatasetSmartName, analysisName, mlBackendType, mlBackendName,
        targetVariable, guessPolicy, managedFolderSmartId, predictionType, timeVariable, timeseriesIdentifiers, treatmentVariable) {
        return APIXHRService("POST", API_PATH + "analysis/create-prediction-template", {
            projectKey,
            inputDatasetSmartName,
            analysisName,
            mlBackendType,
            mlBackendName,
            targetVariable,
            guessPolicy,
            managedFolderSmartId,
            predictionType,
            timeVariable,
            timeseriesIdentifiers: JSON.stringify(timeseriesIdentifiers),
            treatmentVariable
        });
    },
    createClusteringTemplate: function(projectKey, inputDatasetSmartName, analysisName, mlBackendType, mlBackendName, guessPolicy) {
         return APIXHRService("POST", API_PATH + "analysis/create-clustering-template", {
            projectKey: projectKey,
            inputDatasetSmartName: inputDatasetSmartName,
            analysisName: analysisName,
            mlBackendType: mlBackendType,
            mlBackendName: mlBackendName,
            guessPolicy: guessPolicy
        });
    },
    duplicate: function(projectKey, analysisId) {
        return APIXHRService("POST", API_PATH + "analysis/duplicate", {projectKey: projectKey, analysisId: analysisId });
    },
    getCore: function(projectKey, analysisId) {
        return APIXHRService("GET", API_PATH + "analysis/get-core", {projectKey: projectKey, analysisId: analysisId });
    },
    getPostScriptSchema: function(projectKey, analysisId) {
        return APIXHRService("GET", API_PATH + "analysis/get-post-script-schema", {projectKey: projectKey, analysisId: analysisId});
    },
    getSummary: function(projectKey, analysisId, withMLTasksAndSavedModels) {
        return APIXHRService("GET", API_PATH + "analysis/get-summary", {
            projectKey: projectKey,
            analysisId: analysisId,
            withMLTasksAndSavedModels: !!withMLTasksAndSavedModels
        });
    },
    saveCore: function(data, saveInfo) {
        return APIXHRService("POST", API_PATH + "analysis/save-core", {data: JSON.stringify(data), saveInfo: JSON.stringify(saveInfo || {})});
    },
    listMLTasks: function(projectKey, analysisId) {
        return APIXHRService("GET", API_PATH + "analysis/list-mltasks", {projectKey: projectKey, analysisId: analysisId});
    },
    listSavedModels: function(projectKey, analysisId) {
        return APIXHRService("GET", API_PATH + "analysis/list-saved-models", { projectKey: projectKey, analysisId: analysisId });
    },
    addToFlow: function(projectKey, analysisId, createOutput, outputDatasetName, outputDatasetSettings, options) {
        return APIXHRService("POST", API_PATH + "analysis/add-to-flow/", {
                projectKey: projectKey,
                analysisId: analysisId,
                createOutput: createOutput,
                outputDatasetName: outputDatasetName,
                outputDatasetSettings: JSON.stringify(outputDatasetSettings),
                options: JSON.stringify(options)
            }
        );
    },
    exportProcessedData: function(projectKey, analysisId, params) {
        return APIXHRService("POST", API_PATH + "analysis/export-processed-data/", {
                projectKey: projectKey,
                analysisId: analysisId,
                params: JSON.stringify(params)
            }
        );
    },
    mlcommon: {
        stopGridSearch: function(fullModelIds) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/stop-grid-search", {fullModelIds: JSON.stringify(fullModelIds) });
        },
        stopGridSearchSession: function(projectKey, analysisId, mlTaskId, sessionId) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/stop-grid-search-session", {projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, sessionId: sessionId });
        },
        listBackends: function(projectKey, datasetSmartName, taskType) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/list-backends", {projectKey: projectKey, datasetSmartName: datasetSmartName, taskType: taskType });
        },
        setModelMeta: function(fullModelId, data) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/set-model-meta", {fullModelId: fullModelId, data: JSON.stringify(data)});
        },
        getCurrentSettings: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/get-current-settings", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        forgetFeatureSelection: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/forget-feature-selection", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        trainAbort: function(projectKey, analysisId, mlTaskId, pauseQueue = false) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/train-abort", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, pauseQueue
            });
        },
        trainAbortPartial: function(projectKey, analysisId, mlTaskId, fullModelIds, pauseQueue = false) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/train-abort-partial", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, fullModelIds: JSON.stringify(fullModelIds), pauseQueue
            });
        },
        deleteMLTask: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/delete-mltask", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        listQueuedSessions: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/mlcommon/list-queued-sessions", {
                projectKey, analysisId, mlTaskId
            });
        },
        pauseQueue: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/pause-queue", {
                projectKey, analysisId, mlTaskId
            });
        },
        deleteQueuedSessions: function(projectKey, analysisId, mlTaskId, sessionIds) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/delete-queued-sessions", {
                projectKey, analysisId, mlTaskId, sessionIds: JSON.stringify(sessionIds)
            });
        },
        copyFeatureSettings: function(projectKeyFrom, analysisIdFrom, mlTaskIdFrom, projectKeyTo, analysisIdTo, mlTaskIdTo) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/copy-features-handling", {
               projectKeyFrom: projectKeyFrom, analysisIdFrom: analysisIdFrom, mlTaskIdFrom: mlTaskIdFrom,
               projectKeyTo: projectKeyTo, analysisIdTo: analysisIdTo, mlTaskIdTo: mlTaskIdTo
           });
        },
        revertScriptToSession: function(projectKey, analysisId, mlTaskId, sessionId) {
            return APIXHRService("POST", API_PATH + "analysis/mlcommon/revert-script-to-session", {
                projectKey:projectKey,
                analysisId:analysisId,
                mlTaskId:mlTaskId,
                sessionId:sessionId
            });
        },
        getTrainDiagnosisURL: function(fullModelId, includeTrainingData) {
            return API_PATH + "analysis/mlcommon/download-train-diagnosis?"
                + "fullModelId=" + encodeURIComponent(fullModelId)
                + "&includeTrainingData=" + encodeURIComponent(includeTrainingData);
        },
    },
    /* Prediction ML Task */
    pml: {
        listPredictableColumns: function(projectKey, analysisId) {
            return APIXHRService("GET", API_PATH + "analysis/pml/list-predictable-columns", {projectKey: projectKey, analysisId: analysisId});
        },
        listCustomPythonAlgos: function(projectKey) {
            return APIXHRService("GET", API_PATH + "analysis/pml/list-custom-python-algos", {projectKey: projectKey});
        },
        createAndGuess: function(projectKey, analysisId, targetVariable, mlBackendType, mlBackendName, guessPolicy,
            predictionType, managedFolderSmartId, timeVariable, timeseriesIdentifiers, treatmentVariable) {
            return APIXHRService("POST", API_PATH + "analysis/pml/create-and-guess", {
                projectKey,
                analysisId,
                targetVariable,
                mlBackendType,
                predictionType,
                managedFolderSmartId,
                timeVariable,
                treatmentVariable,
                timeseriesIdentifiers: JSON.stringify(timeseriesIdentifiers),
                guessPolicy: guessPolicy || 'DEFAULT',
                mlBackendName: mlBackendName || '',
            });
        },
        duplicate: function(projectKeyFrom, analysisIdFrom, mlTaskIdFrom, projectKeyTo, analysisIdTo, newTarget) {
            return APIXHRService("POST", API_PATH + "analysis/pml/duplicate", {
                projectKeyFrom,
                analysisIdFrom,
                mlTaskIdFrom,
                projectKeyTo,
                analysisIdTo,
                newTarget
            });
        },
        getModelSnippets: function(projectKey, analysisId, mlTaskId, fullModelIds, spinner) {
            return APIXHRService("GET", API_PATH + "analysis/pml/get-model-snippets", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, fullModelIds: JSON.stringify(fullModelIds)
            }, spinner ? undefined : "nospinner");
        },
        getPartitionedModelSnippets: function(fullModelId, spinner) {
            return APIXHRService("GET", API_PATH + "analysis/pml/get-partitioned-model-snippets", {
                fullModelId: fullModelId
            }, spinner ? undefined : "nospinner");
        },
        getTaskStatus: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/pml/get-mltask-status", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            }, "nospinner");
        },
        getLightTaskStatus: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/pml/get-light-mltask-status", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            }, "nospinner");
        },
        getUpdatedSettings: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/pml/get-updated-settings", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        getSessionTask: function(projectKey, analysisId, mlTaskId, sessionId) {
            return APIXHRService("POST", API_PATH + "analysis/pml/get-session-task", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, sessionId: sessionId
            });
        },
        getPretrainEquivalentMLTask: function(fullModelId, usePostTrain) {
            return APIXHRService("POST", API_PATH + "analysis/pml/get-pretrain-equivalent-mltask", {
                fullModelId:fullModelId,
                usePostTrain: usePostTrain
            });
        },
        getPreTrainStatus: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/pml/get-pretrain-status", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        retrainStart: function(projectKey, analysisId, mlTaskId, sessionId, fullModelIds) {
            return APIXHRService("POST", API_PATH + "analysis/pml/retrain-start", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, sessionId: sessionId, fullModelIds: JSON.stringify(fullModelIds)
            });
        },
        createNewModelWithDifferentOverrides: function(fullModelId, newModelName, overridesParams) {
            return APIXHRService("POST", API_PATH + "analysis/pml/create-model-with-different-overrides", {
                fullModelId: fullModelId, newModelName: newModelName, overridesParams: JSON.stringify(overridesParams)
            });
        },
        trainStart: function(projectKey, analysisId, mlTaskId, userSessionName, userSessionDescription, forceRefresh, runQueue) {
            return APIXHRService("POST", API_PATH + "analysis/pml/train-start", {
                projectKey, analysisId, mlTaskId,
                userSessionName, userSessionDescription,
                forceRefresh,
                runQueue
            });
        },
        enqueueSession: function(projectKey, analysisId, mlTaskId, pauseQueue = false, userSessionName, userSessionDescription, forceRefresh) {
            return APIXHRService("POST", API_PATH + "analysis/pml/enqueue-session", {
                projectKey, analysisId, mlTaskId,
                pauseQueue, userSessionName, userSessionDescription, forceRefresh
            });
        },
        trainQueue: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/pml/train-queue", {
                projectKey, analysisId, mlTaskId
            });
        },
        checkCanEnsemble: function(modelIds) {
            return APIXHRService("POST", API_PATH + "analysis/pml/check-can-ensemble", {
                modelIds: JSON.stringify(modelIds),
            });
        },
        createEnsemble: function(projectKey, analysisId, mlTaskId, modelIds, method) {
            return APIXHRService("POST", API_PATH + "analysis/pml/create-ensemble", {
                projectKey: projectKey,
                analysisId: analysisId,
                mlTaskId: mlTaskId,
                modelIds: JSON.stringify(modelIds),
                method: method
            });
        },
        saveSettings: function(projectKey, analysisId, mlTask) {
            return APIXHRService("POST", API_PATH + "analysis/pml/save-settings", {
                projectKey: projectKey, analysisId: analysisId, mlTask: JSON.stringify(mlTask)
            });
        },
        reguess: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess", { projectKey, analysisId, mlTaskId });
        },
        reguessWithTarget: function(projectKey, analysisId, mlTaskId, targetVariable, redetect) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess-with-target", {
                projectKey, analysisId, mlTaskId, targetVariable, redetect
            });
        },
        reguessWithType: function(projectKey, analysisId, mlTaskId, newType, redetect) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess-with-type", {
                projectKey, analysisId, mlTaskId, newType, redetect
            });
        },
        reguessWithTimeVariable: function(projectKey, analysisId, mlTaskId, timeVariable, redetect) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess-with-time-variable", {
                projectKey, analysisId, mlTaskId, timeVariable, redetect
            });
        },
        reguessWithTimeseriesIdentifiers: function(projectKey, analysisId, mlTaskId, timeseriesIdentifiers, redetect) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess-with-timeseries-identifiers", {
                projectKey,
                analysisId,
                mlTaskId,
                redetect,
                timeseriesIdentifiers: JSON.stringify(timeseriesIdentifiers)
            });
        },
        reguessWithTimestepParams: function(projectKey, analysisId, mlTaskId, timestepParams, predictionLength, redetect) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess-with-timestep-params", {
                projectKey,
                analysisId,
                mlTaskId,
                predictionLength,
                redetect,
                timestepParams: JSON.stringify(timestepParams)
            });
        },
        reguessWithTreatmentVariable: function(projectKey, analysisId, mlTaskId, treatmentVariable, redetect) {
            return APIXHRService("POST", API_PATH + "analysis/pml/reguess-with-treatment-variable", {
                projectKey,
                analysisId,
                mlTaskId,
                treatmentVariable,
                redetect
            });
        },
        changeGuessPolicy: function(projectKey, analysisId, mlTaskId, policyId) {
            return APIXHRService("POST", API_PATH + "analysis/pml/change-guess-policy", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId,
                newPolicyId: policyId
            });
        },
        saveCostMatrixWeights: function(fullModelId, data) {
          return APIXHRService("POST", API_PATH + "analysis/pml/save-costmatrix-weights", {
                fullModelId: fullModelId, data: JSON.stringify(data)
            });
        },
        deployTrain: function(fullModelId, trainDatasetSmartName, testDatasetSmartName, modelName, managedFolderSmartId, options) {
            return APIXHRService("POST", API_PATH + "analysis/pml/flow/deploy-train", {
                fullModelId: fullModelId,
                trainDatasetSmartName: trainDatasetSmartName,
                testDatasetSmartName: testDatasetSmartName,
                modelName: modelName,
                managedFolderSmartId: managedFolderSmartId,
                options: JSON.stringify(options)
            });
        },
        listRedeployableTrain: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "analysis/pml/flow/list-redeployable-train", {fullModelId: fullModelId });
        },
        redeployTrain: function(fullModelId, recipeName, activate, options) {
            return APIXHRService("POST", API_PATH + "analysis/pml/flow/redeploy-train", {
                fullModelId: fullModelId,
                recipeName: recipeName,
                activate: !!activate,
                options: JSON.stringify(options)
            });
        },
        createNotebook: function(fullModelId, notebookTitle) {
            return APIXHRService("POST", API_PATH + "analysis/pml/create-notebook", {fullModelId: fullModelId, notebookTitle: notebookTitle});
        },
        validateArchitecture: function(payload, envSelection, projectKey) {
            return APIXHRService("POST", API_PATH + "analysis/pml/validate-keras-architecture", {
                payload : payload,
                envSelection : JSON.stringify(envSelection),
                projectKey : projectKey
            });
        },
        copyAlgorithmSettings: function(projectKeyFrom, analysisIdFrom, mlTaskIdFrom, projectKeyTo, analysisIdTo, mlTaskIdTo) {
            return APIXHRService("POST", API_PATH + "analysis/pml/copy-algorithm-settings", {
               projectKeyFrom: projectKeyFrom, analysisIdFrom: analysisIdFrom, mlTaskIdFrom: mlTaskIdFrom,
               projectKeyTo: projectKeyTo, analysisIdTo: analysisIdTo, mlTaskIdTo: mlTaskIdTo
           });
        },
        isHPSearchNeeded: function(projectKey, task) {
           return APIXHRService("POST", API_PATH + "/analysis/pml/is-hp-search-needed", {
               projectKey,
               taskString: JSON.stringify(task)
           });
        },
        listAlgosIncompatibleWithSearchStrategy: function(projectKey, task) {
            return APIXHRService("POST", API_PATH + "analysis/pml/list-algos-incompatible-with-search-strategy", {
                projectKey,
                taskString: JSON.stringify(task)
            });
        },
        getOutcomeStatsPerTreatmentState: function(projectKey, analysisId, mlTaskId, controlValue, dropMissingValues) {
            return APIXHRService("GET", API_PATH + "analysis/pml/treated-control-stats", {
                projectKey, analysisId, mlTaskId, controlValue, dropMissingValues,
            });
        },
        getTreatmentColValues: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/pml/treated-col-values", {
                projectKey, analysisId, mlTaskId,
            });
        },
    },
    /* Clustering MLTask */
    cml: {
        listGuessPolicies: function() {
            return APIXHRService("GET", API_PATH + "analysis/cml/list-guess-policies");
        },
        createAndGuess: function(projectKey, analysisId, mlBackendType, mlBackendName, guessPolicy) {
            return APIXHRService("POST", API_PATH + "analysis/cml/create-and-guess", {
                projectKey: projectKey,
                analysisId: analysisId,
                mlBackendType: mlBackendType,
                mlBackendName: mlBackendName || '',
                guessPolicy: guessPolicy || 'DEFAULT'
            });
        },
        duplicate: function(projectKeyFrom, analysisIdFrom, mlTaskIdFrom, projectKeyTo, analysisIdTo) {
            return APIXHRService("POST", API_PATH + "analysis/cml/duplicate", {
                projectKeyFrom,
                analysisIdFrom,
                mlTaskIdFrom,
                projectKeyTo,
                analysisIdTo
            });
        },
        getModelSnippets: function(projectKey, analysisId, mlTaskId, fullModelIds) {
            return APIXHRService("GET", API_PATH + "analysis/cml/get-model-snippets", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, fullModelIds: JSON.stringify(fullModelIds)
            }, "nospinner");
        },
        getTaskStatus: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/cml/get-mltask-status", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            }, "nospinner");
        },
        getUpdatedSettings: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/get-updated-settings", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        getSessionTask: function(projectKey, analysisId, mlTaskId, sessionId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/get-session-task", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId, sessionId: sessionId
            });
        },
        getPreTrainStatus: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("GET", API_PATH + "analysis/cml/get-pretrain-status", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        trainStart: function(projectKey, analysisId, mlTaskId, userSessionName, userSessionDescription, forceRefresh, runQueue) {
            return APIXHRService("POST", API_PATH + "analysis/cml/train-start", {
                projectKey, analysisId, mlTaskId,
                userSessionName, userSessionDescription,
                forceRefresh,
                runQueue
            });
        },
        enqueueSession: function(projectKey, analysisId, mlTaskId, pauseQueue = false, userSessionName, userSessionDescription, forceRefresh) {
            return APIXHRService("POST", API_PATH + "analysis/cml/enqueue-session", {
                projectKey, analysisId, mlTaskId,
                pauseQueue, userSessionName, userSessionDescription, forceRefresh
            });
        },
        trainQueue: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/train-queue", {
                projectKey, analysisId, mlTaskId
            });
        },
        saveSettings: function(projectKey, analysisId, mlTask) {
            return APIXHRService("POST", API_PATH + "analysis/cml/save-settings", {
                projectKey: projectKey, analysisId: analysisId, mlTask: JSON.stringify(mlTask)
            });
        },
        reguess: function(projectKey, analysisId, mlTaskId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/reguess", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId
            });
        },
        deployTrain: function(fullModelId, inputDatasetSmartName, modelName) {
            return APIXHRService("POST", API_PATH + "analysis/cml/flow/deploy-train", {
                fullModelId: fullModelId,
                inputDatasetSmartName: inputDatasetSmartName,
                modelName: modelName
            });
        },
        listRedeployableTrain: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/flow/list-redeployable-train", {fullModelId: fullModelId });
        },
        redeployTrain: function(fullModelId, recipeName, activate) {
            return APIXHRService("POST", API_PATH + "analysis/cml/flow/redeploy-train", {
                fullModelId: fullModelId, recipeName: recipeName, activate: !!activate
            });
        },
        deployCluster: function(fullModelId, inputDatasetSmartName, outputDatasetName, outputDatasetSettings) {
            return APIXHRService("POST", API_PATH + "analysis/cml/flow/deploy-cluster", {
                fullModelId: fullModelId,
                inputDatasetSmartName:inputDatasetSmartName,
                outputDatasetName:outputDatasetName,
                outputDatasetSettings: JSON.stringify(outputDatasetSettings)
            });
        },
        listRedeployableCluster: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/flow/list-redeployable-cluster", {fullModelId: fullModelId });
        },
        redeployCluster: function(fullModelId, recipeName) {
            return APIXHRService("POST", API_PATH + "analysis/cml/flow/redeploy-cluster", {
                fullModelId: fullModelId, recipeName: recipeName
            });
        },
        createNotebook: function(fullModelId, notebookTitle) {
            return APIXHRService("POST", API_PATH + "analysis/cml/create-notebook", {
                fullModelId: fullModelId, notebookTitle: notebookTitle
            })
        },
        copyAlgorithmSettings: function(projectKeyFrom, analysisIdFrom, mlTaskIdFrom, projectKeyTo, analysisIdTo, mlTaskIdTo) {
            return APIXHRService("POST", API_PATH + "analysis/cml/copy-algorithm-settings", {
               projectKeyFrom: projectKeyFrom, analysisIdFrom: analysisIdFrom, mlTaskIdFrom: mlTaskIdFrom,
               projectKeyTo: projectKeyTo, analysisIdTo: analysisIdTo, mlTaskIdTo: mlTaskIdTo
           });
        },
        changeGuessPolicy: function(projectKey, analysisId, mlTaskId, policyId) {
            return APIXHRService("POST", API_PATH + "analysis/cml/change-guess-policy", {
                projectKey: projectKey, analysisId: analysisId, mlTaskId: mlTaskId,
                newPolicyId: policyId
            });
        },
        getDiagnosticsDefinition: function(backendType) {
            return APIXHRService("GET", API_PATH + "analysis/cml/diagnostics-definition", {
                backendType: backendType
            });
        },
    },
    /* Predicted data (common to ML tasks) */
    predicted: {
        predictedRefreshTable: function(fullModelId, displayScript, allowCache, filters) {
            return APIXHRService("POST", API_PATH + "analysis/predicted/refresh-table", {
                fullModelId: fullModelId,
                displayScript: JSON.stringify(displayScript),
                allowCache: allowCache,
                filters: JSON.stringify(filters)
            })
        },
        predictedGetTableChunk: function(fullModelId, displayScript, firstRow, nbRows, firstCol, nbCols, filters) {
            return APIXHRService("POST", API_PATH + "analysis/predicted/get-table-chunk", {
                fullModelId: fullModelId,
                displayScript: JSON.stringify(displayScript),
                filters: JSON.stringify(filters),
                firstRow: firstRow,
                nbRows: nbRows,
                firstCol: firstCol,
                nbCols: nbCols
            })
        },
        chartsGetColumnsSummary: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "analysis/predicted/charts-get-columns-summary", {
                fullModelId: fullModelId
            });
        },
        chartsGetPivotResponse: function(fullModelId, request, requestedSampleId) {
            return APIXHRService("POST", API_PATH + "analysis/predicted/charts-get-pivot-response", {
                fullModelId: fullModelId,
                request: JSON.stringify(request),
                requestedSampleId: requestedSampleId
            });
        },
        detailedColumnAnalysis: function(fullModelId, data, column, alphanumMaxResults) {
            return APIXHRService("POST", API_PATH + "analysis/predicted/detailed-column-analysis/", {
                fullModelId: fullModelId,
                data: JSON.stringify(data),
                column: column,
                alphanumMaxResults: alphanumMaxResults
            });
        },
        fixExpression: function(fullModelId, expression, fixName) {
            return APIXHRService("POST", API_PATH + "analysis/predicted/fix-expression", {
                fullModelId,
                expression,
                fixName
            });
        },
    }
},
ml: {
    prediction: {
        getModelDetails: function(fullModelId, treatment) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-model-details", {
                fullModelId: fullModelId,
                treatment: treatment,
            });
        },
        getKfoldPerfs: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-kfold-perf-metrics", {
                fullModelId: fullModelId,
            });
        },
        getForecasts: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-model-forecasts", { fullModelId });
        },
        getPerTimeseriesMetrics: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-per-timeseries-metrics", { fullModelId });
        },
        getInformationCriteria: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/timeseries-information-criteria-start", { fullModelId });
        },
        computePerTimeseriesResiduals: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/compute-per-timeseries-residuals", { fullModelId });
        },
        getPerTimeseriesResiduals: function(fullModelId, timeseriesIdentifiers) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-per-timeseries-residuals", { fullModelId, timeseriesIdentifiers: JSON.stringify(timeseriesIdentifiers) });
        },
        getTreeSummary: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-tree-summary", {
                 fullModelId: fullModelId
             });
        },
        getEnsembleSummary: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-ensemble-summary", {
                 fullModelId: fullModelId
             });
        },
        getCoefPath: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-coef-path", {
                 fullModelId: fullModelId
             });
        },
        getPreparedInputSchema: function(recipe) {
            return APIXHRService("POST", API_PATH + "ml/prediction/get-prepared-input-schema", {
                 recipeData: JSON.stringify(recipe)
             });
        },
        getSql: function(recipe) {
            return APIXHRService("GET", API_PATH + "ml/prediction/export-sql?recipeData="+ JSON.stringify(recipe));
        },
        getScoringModelDownloadURL: function(format, exportId) {
            format = format.toLowerCase();
            return `${API_PATH}ml/prediction/get-export?format=${format}&exportId=${encodeURIComponent(exportId)}`;
        },
        createScoringModelFile: function(format, fullModelId, params) {
            fullModelId = encodeURIComponent(fullModelId);
            format = format.toLowerCase();
            if (!params) {
                params = "";
            }
            switch (format) {
                case "jar-fat":
                    format = "jar";
                    params += "&includeLibs=true";
                    break;
                case "jar-thin":
                    format = "jar";
                    params += "&includeLibs=false";
                    break;
            }
            const url = `${API_PATH}ml/prediction/export-${format}?fullModelId=${fullModelId}${params}`;
            return APIXHRService("GET", url);
        },
        docGenCustom: function(file, fullModelId, callback) {
            return uploadFileRequest("ml/prediction/render-custom", function(formdata) {
                formdata.append("file", file);
                formdata.append("fullModelId", fullModelId);
            }, callback);
        },
        docGenDefault: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "ml/prediction/render-default" , {
                fullModelId: fullModelId
            });
        },
        getPreDocGenInfoMessages: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "ml/prediction/get-pre-docgen-info-messages" , {
                fullModelId: fullModelId
            });
        },
        exportToSnowflakeFunction: function(connectionName, fullModelId, functionName) {
            return APIXHRService("POST", API_PATH + "ml/prediction/export-to-snowflake-function", {
                connectionName, fullModelId, functionName
            });
        },
        exportToDatabricksRegistry: function(connectionName, fullModelId, useUnityCatalog, modelName, experimentName) {
            return APIXHRService("POST", API_PATH + "ml/prediction/export-to-databricks-registry", {
                connectionName, fullModelId, useUnityCatalog, modelName, experimentName
            });
        },
        subpopulationComputationStart: function(fullModelId, features, computationParams) {
            return APIXHRService("POST", API_PATH + "ml/prediction/subpopulation-computation-start" , {
                fullModelId: fullModelId,
                features: JSON.stringify(features),
                computationParams: JSON.stringify(computationParams),
            });
        },
        pdpComputationStart: function(fullModelId, features, computationParams) {
            return APIXHRService("POST", API_PATH + "ml/prediction/pdp-computation-start" , {
                fullModelId: fullModelId,
                features: JSON.stringify(features),
                computationParams: JSON.stringify(computationParams),
            });
        },
        globalExplanationsComputationStart: function(fullModelId, computationParams) {
            return APIXHRService("POST", API_PATH + "ml/prediction/global-explanations-computation-start" , {
                fullModelId: fullModelId,
                computationParams: JSON.stringify(computationParams),
            });
        },
        getGlobalExplanations: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-global-explanations" , {
                fullModelId: fullModelId,
            });
        },
        getGlobalExplanationsFacts: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-global-explanations-facts" , {
                fullModelId: fullModelId,
            });
        },
        individualExplanationsComputationStart: function(fullModelId, computationParams) {
            return APIXHRService("POST", API_PATH + "ml/prediction/individual-explanations-computation-start" , {
                fullModelId: fullModelId,
                computationParams: JSON.stringify(computationParams),
            });
        },
        getIndividualExplanations: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-individual-explanations" , {
                fullModelId: fullModelId,
            });
        },
        getSubpopulation: function(fullModelId, features) {
            return APIXHRService("POST", API_PATH + "ml/prediction/get-subpopulation" , {
                fullModelId: fullModelId,
                features: JSON.stringify(features)
            });
        },
        getSubpopulationsInfo: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-subpopulations-info", {
                fullModelId: fullModelId
            });
        },
        getPartitionsPerf: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-partitions-perf" , {
                fullModelId: fullModelId
            });
        },
        getCollectorData: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-collector-data" , {
                fullModelId: fullModelId
            });
        },
        getFeaturesDistribution: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-features-distribution" , {
                fullModelId: fullModelId
            });
        },
        getColumnImportance: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-column-importance" , {
                fullModelId: fullModelId
            });
        },
        getSplitDesc: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-split-desc" , {
                fullModelId: fullModelId
            });
        },
        getInputDatasetSchema: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-input-dataset-schema" , { fullModelId });
        },
        getPreparationScript: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-preparation-script" , {
                fullModelId: fullModelId
            });
        },
        learningCurvesStart: function(fullModelId, numberOfPoints) {
            return APIXHRService("POST", API_PATH + "ml/prediction/learning-curves-start" , {
                fullModelId,
                numberOfPoints,
            });
        },
        getLearningCurves: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "ml/prediction/get-learning-curves" , {
                fullModelId: fullModelId,
            });
        },
        createInteractiveScoringScenario: function(fullModelId, timeseriesIdentifier, scenarioId, startDate, scenarioName) {
            return APIXHRService("POST", API_PATH + "ml/prediction/create-timeseries-interactive-scoring-scenario" , {
                fullModelId: fullModelId,
                timeseriesIdentifier,
                scenarioId,
                startDate,
                scenarioName
            });
        },
        renameInteractiveScoringScenario: function(fullModelId, timeseriesIdentifier, scenarioId, newName) {
            return APIXHRService("POST", API_PATH + "ml/prediction/rename-timeseries-interactive-scoring-scenario" , {
                fullModelId,
                timeseriesIdentifier,
                scenarioId,
                newName
            });
        },
        deleteInteractiveScoringScenario: function(fullModelId, timeseriesIdentifier, scenarioId) {
            return APIXHRService("POST", API_PATH + "ml/prediction/delete-timeseries-interactive-scoring-scenario" , {
                fullModelId,
                timeseriesIdentifier,
                scenarioId
            });
        },
        getInteractiveScoringScenario: function(fullModelId, timeseriesIdentifier) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-timeseries-interactive-scoring-scenario" , {
                fullModelId,
                timeseriesIdentifier
            });
        },
        computeInteractiveScoringScenario: function(fullModelId, timeseriesIdentifier) {
            return APIXHRService("POST", API_PATH + "ml/prediction/compute-timeseries-interactive-scoring-scenarios", {
                fullModelId: fullModelId,
                timeseriesIdentifier
            });
        },
        getInteractiveScoringForecasts: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-timeseries-interactive-scoring-forecasts", {
                fullModelId: fullModelId,
            });
        },
        getInteractiveScoringForecastsByIdentifier: function(fullModelId, timeseriesIdentifier) {
            return APIXHRService("GET", API_PATH + "ml/prediction/get-timeseries-interactive-scoring-forecasts-by-identifier", {
                fullModelId,
                timeseriesIdentifier
            });
        }
    },
    clustering: {
        getModelDetails: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/clustering/get-model-details", {
                fullModelId: fullModelId
            });
        },
        getScatterPlot: function(projectKey, fullModelId, variable1, variable2) {
            return APIXHRService("GET", API_PATH + "ml/clustering/get-scatter-plot" , {
                projectKey: projectKey, fullModelId: fullModelId,
                variable1: variable1, variable2: variable2
            });
        },
        getClusterHierarchy: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/clustering/get-cluster-hierarchy" , {
                fullModelId: fullModelId
            });
        },
        getAnomalies: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/clustering/get-anomalies" , {
                fullModelId: fullModelId
            });
        },
        getPreparedInputSchema: function(recipe) {
            return APIXHRService("POST", API_PATH + "ml/clustering/get-prepared-input-schema", {
                 recipeData: JSON.stringify(recipe)
             });
        },
        rescore: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "ml/clustering/rescore" , {
                fullModelId: fullModelId
            });
        }
    },
    llm: {  // TODO @llm: perhaps rename to llmGeneric
        getModelDetails: function(fullModelId) {
            return APIXHRService("GET", API_PATH + "ml/llm/get-model-details", { fullModelId });
        },
    },
    saveModelUserMeta: function(fullModelId, data) {
        return APIXHRService("POST", API_PATH + "ml/save-model-user-meta", {
            fullModelId: fullModelId, data: JSON.stringify(data)
        });
    },
    deleteModels: function(list) {
        return APIXHRService("POST", API_PATH + "ml/delete-models", {
            list: JSON.stringify(list)
        });
    },
    exportModelDataset: function(fullModelId, exportParams) {
        return APIXHRService("POST", API_PATH + "ml/export-model-dataset", {
            fullModelId: fullModelId,
            exportParams: JSON.stringify(exportParams),
        });
    },
    exportModelPredictedData: function(fullModelId, exportParams) {
        return APIXHRService("POST", API_PATH + "ml/export-model-predicted-data", {
            fullModelId: fullModelId,
            exportParams: JSON.stringify(exportParams),
        });
    },
    exportTimeseriesResiduals: function(fullModelId, exportParams) {
        return APIXHRService("POST", API_PATH + "ml/export-timeseries-residuals", {
            fullModelId: fullModelId,
            exportParams: JSON.stringify(exportParams),
        });
    }
},
interactiveModel: {
    compute: function(fullModelId, computationParams, records, computeEvenIfCached=false) {
        return APIXHRService("POST", API_PATH + "ml/interactive-model/compute", {
            fullModelId: fullModelId,
            computationParams: JSON.stringify(computationParams),
            records: JSON.stringify(records),
            computeEvenIfCached: computeEvenIfCached
        });
    }
},
savedmodels: {
    get: function(projectKey, smartId) {
        return APIXHRService("GET", API_PATH + "savedmodels/get", {projectKey: projectKey, smartId: smartId});
    },
    getSummary: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "savedmodels/get-summary", {projectKey: projectKey, id: id});
    },
    getFullInfo: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "savedmodels/get-full-info", {projectKey: projectKey, id: id});
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "savedmodels/list", {projectKey: projectKey});
    },
    listWithAccessible: function(projectKey) {
        return APIXHRService("GET", API_PATH + "savedmodels/list-with-accessible/", {projectKey: projectKey});
    },
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "savedmodels/list-heads", {projectKey: projectKey});
    },
    save: function(data, saveInfo) {
        return APIXHRService("POST", API_PATH + "savedmodels/save", {
            data: JSON.stringify(data),
            saveInfo: JSON.stringify(saveInfo || {})
        });
    },
    rename: function(projectKey, savedModelId, newName) {
        return APIXHRService("POST", API_PATH + "savedmodels/rename", { projectKey, savedModelId, newName })
    },
    /* Predicted data */
    predicted: {
        getPredictionDisplayScript: function(fullModelId) {
            return APIXHRService("POST", API_PATH + "savedmodels/predicted/get-prediction-display-script", {fullModelId});
        },
        predictedRefreshTable: function(fullModelId, displayScript, allowCache, filters) {
            return APIXHRService("POST", API_PATH + "savedmodels/predicted/refresh-table", {
                fullModelId: fullModelId,
                displayScript: JSON.stringify(displayScript),
                allowCache: allowCache,
                filters: JSON.stringify(filters)
            })
        },
        predictedGetTableChunk: function(fullModelId, displayScript, firstRow, nbRows, firstCol, nbCols, filters) {
            return APIXHRService("POST", API_PATH + "savedmodels/predicted/get-table-chunk", {
                fullModelId: fullModelId,
                displayScript: JSON.stringify(displayScript),
                filters: JSON.stringify(filters),
                firstRow: firstRow,
                nbRows: nbRows,
                firstCol: firstCol,
                nbCols: nbCols
            })
        },
        detailedColumnAnalysis: function(fullModelId, displayScript, column, alphanumMaxResults) {
            return APIXHRService("POST", API_PATH + "savedmodels/predicted/detailed-column-analysis/", {
                fullModelId: fullModelId,
                displayScript: JSON.stringify(displayScript),
                column: column,
                alphanumMaxResults: alphanumMaxResults
            });
        }
    },
    prediction: {
        getStatus: function(projectKey, savedModelId) {
            return APIXHRService("GET", API_PATH + "savedmodels/prediction/get-status", {projectKey: projectKey, savedModelId: savedModelId });
        },
        deployScoring: function(projectKey, options) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/deploy-scoring", {
                projectKey: projectKey, options: JSON.stringify(options)
            });
        },
        deployEvaluation: function(projectKey, options) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/deploy-evaluation",{
                    projectKey:projectKey, options : JSON.stringify(options)
            });
        },
        deployStandaloneEvaluation: function(projectKey, options) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/deploy-standalone-evaluation",{
                    projectKey:projectKey, options : JSON.stringify(options)
            });
        },
        createModelVersionWithDifferentOverrides: function(fullModelId, newModelName, overridesParams) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/create-model-version-with-different-overrides", {
                fullModelId: fullModelId, newModelName: newModelName, overridesParams: JSON.stringify(overridesParams)
            });
        },
        setActive: function(projectKey, savedModelId, newActiveVersion) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/set-active", {
                projectKey: projectKey,
                savedModelId: savedModelId,
                newActiveVersion: newActiveVersion
            });
        },
        deleteVersions: function(projectKey, savedModelId, versions) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/delete-versions", {
                projectKey: projectKey,
                savedModelId: savedModelId,
                versions: JSON.stringify(versions)
            });
        },
        createExternal: function(projectKey, savedModelType, predictionType, name, proxyModelConfiguration) {
            return APIXHRService("POST", API_PATH + "savedmodels/prediction/create-external", {
                projectKey: projectKey,
                savedModelType: savedModelType,
                predictionType: predictionType,
                name: name,
                proxyModelConfiguration: JSON.stringify(proxyModelConfiguration)
            });
        },
        importFromDatabricks: function(projectKey, smId, connectionName, useUnityCatalog, modelName, modelVersion,
                                       versionId, modelVersionInfo, samplingParam, activate,
                                       binaryClassificationThreshold, useOptimalThreshold, skipExpensiveReports) {
            return APIXHRService("POST", API_PATH + "savedmodels/import-from-databricks", {
                projectKey,
                smId,
                connectionName,
                useUnityCatalog,
                modelName,
                modelVersion,
                versionId,
                modelVersionInfo: JSON.stringify(modelVersionInfo),
                samplingParam: samplingParam ? JSON.stringify(samplingParam) : undefined,
                activate,
                binaryClassificationThreshold,
                useOptimalThreshold,
                skipExpensiveReports
            });
        },
        createProxySavedModelVersion: function(projectKey, savedModelId, versionId, modelVersionInfo, samplingParam, binaryClassificationThreshold,
            useOptimalThreshold, activate, skipExpensiveReports, containerSelection) {
            return APIXHRService("POST", API_PATH + "savedmodels/deploy-proxy-model", {
                projectKey: projectKey,
                smId: savedModelId,
                versionId: versionId,
                modelVersionInfo: JSON.stringify(modelVersionInfo),
                samplingParam: JSON.stringify(samplingParam),
                binaryClassificationThreshold: binaryClassificationThreshold,
                useOptimalThreshold: useOptimalThreshold,
                activate: activate,
                skipExpensiveReports: skipExpensiveReports,
                containerExecSelection: JSON.stringify(containerSelection)
            });
        },
        checkProxyModelVersionEndpoint: function(projectKey, savedModelId, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/check-proxy-model-endpoint", {
                projectKey: projectKey,
                smId: savedModelId,
                versionId: versionId
            });
        },
        setupProxyModelMonitoring: function(projectKey, smId, versionId, connection, params) {
            return APIXHRService("POST", API_PATH + "savedmodels/setup-proxy-model-monitoring", {
                projectKey,
                smId,
                versionId,
                connection,
                params: JSON.stringify(params)
            });
        },
        importMlflowModelArchive: function(projectKey, file, callback) {
            return uploadFileRequest("/savedmodels/import-archive", function(formdata) {
                formdata.append("projectKey", projectKey);
                formdata.append("file", file);
            }, callback);
        }
    },
    clustering: {
        getStatus: function(projectKey, savedModelId) {
            return APIXHRService("GET", API_PATH + "savedmodels/clustering/get-status", {projectKey: projectKey, savedModelId: savedModelId});
        },
        deployScoring: function(projectKey, savedModelSmartName, inputDatasetSmartName, createOutput, outputDatasetSmartName, outputDatasetSettings) {
            return APIXHRService("POST", API_PATH + "savedmodels/clustering/deploy-scoring", {
                projectKey: projectKey,
                savedModelSmartName: savedModelSmartName,
                inputDatasetSmartName: inputDatasetSmartName,
                createOutput: createOutput,
                outputDatasetSmartName: outputDatasetSmartName,
                outputDatasetSettings: JSON.stringify(outputDatasetSettings)
            });
        },
        setActive: function(projectKey, savedModelId, newActiveVersion) {
            return APIXHRService("POST", API_PATH + "savedmodels/clustering/set-active", {
                projectKey: projectKey, savedModelId: savedModelId,
                newActiveVersion: newActiveVersion
            });
        },
        deleteVersions: function(projectKey, savedModelId, versions) {
            return APIXHRService("POST", API_PATH + "savedmodels/clustering/delete-versions", {
                projectKey: projectKey,
                savedModelId: savedModelId,
                versions: JSON.stringify(versions)
            });
        }
    },
    llmCommon: {
        getStatus: function(projectKey, savedModelId) {
            return APIXHRService("GET", API_PATH + "savedmodels/llm/get-status", {projectKey: projectKey, savedModelId: savedModelId});
        },
    },
    llmGeneric: {  //  For now, means fine tuned models; perhaps would be a better wording
        setActive: function(projectKey, savedModelId, newActiveVersion, deployNewActiveModel, deleteInactiveDeployments) {
            return APIXHRService("POST", API_PATH + "savedmodels/llm-generic/set-active", {
                projectKey, savedModelId, newActiveVersion, deployNewActiveModel, deleteInactiveDeployments
            });
        },
        deleteVersions: function(projectKey, savedModelId, versions) {
            return APIXHRService("POST", API_PATH + "savedmodels/llm-generic/delete-versions", {
                versions: JSON.stringify(versions),
                projectKey,
                savedModelId,
            });
        },
        deployments: {
            list: function(projectKey, savedModelId) {
                return APIXHRService("GET", API_PATH + "savedmodels/llm-generic/deployments/list", {projectKey: projectKey, savedModelId: savedModelId});
            },
            get: function(projectKey, fullModelId) {
                return APIXHRService("GET", API_PATH + "savedmodels/llm-generic/deployments/get", {projectKey: projectKey, fullModelId: fullModelId});
            },
            detach: function(projectKey, fullModelId) {
                return APIXHRService("DELETE", API_PATH + "savedmodels/llm-generic/deployments/detach?projectKey=" + encodeURIComponent(projectKey) + "&fullModelId=" + encodeURIComponent(fullModelId));
            },
            attach: function(projectKey, fullModelId, deploymentId) {
                return APIXHRService("PUT", API_PATH + "savedmodels/llm-generic/deployments/attach", {projectKey: projectKey, fullModelId: fullModelId, deploymentId: deploymentId});
            },
            delete: function(projectKey, fullModelId) {
                return APIXHRService("DELETE", API_PATH + "savedmodels/llm-generic/deployments/delete?projectKey=" + encodeURIComponent(projectKey) + "&fullModelId=" + encodeURIComponent(fullModelId));
            },
            create: function(projectKey, fullModelId, deploymentId) {
                return APIXHRService("POST", API_PATH + "savedmodels/llm-generic/deployments/create" , {projectKey: projectKey, fullModelId: fullModelId, deploymentId: deploymentId});
            },
        }
    },
    agents: {
        save: function(savedModel, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/save", {
                savedModel: JSON.stringify(savedModel),
                versionId : versionId
            });
        },
        checkSaveConflict: function(savedModel) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/check-save-conflict", {
                savedModel: JSON.stringify(savedModel)
            });
        },
        createPython: function(projectKey, name, zoneId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/create-python", {
                projectKey, name, zoneId
            });
        },
        createAgentVersion: function(projectKey, smId, versionId, initialData) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/create-version", {
                projectKey, smId, versionId, initialData: JSON.stringify(initialData)
            });
        },
        createPlugin: function(projectKey, type, name, zoneId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/create-plugin", {
                projectKey, type, name, zoneId
            });
        },
        createToolsUsing: function(projectKey,  name, zoneId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/create-tools-using", {
                projectKey, name, zoneId
            });
        },
        setActive: function(projectKey, savedModelId, newActiveVersion) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/set-active", {
                projectKey, savedModelId, newActiveVersion,
            });
        },
        deleteVersions: function(projectKey, savedModelId, versions) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/delete-versions", {
                versions: JSON.stringify(versions),
                projectKey,
                savedModelId,
            });
        },
        duplicateVersion: function(projectKey, savedModelId, versionIdToCopy, newVersionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/duplicate-version", {
                versionIdToCopy,
                newVersionId,
                projectKey,
                savedModelId,
            });
        },
        listLogs: function(projectKey, smId, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/list-logs", {
                projectKey, smId, versionId
            });
        },
        getLog: function(projectKey, smId, versionId, logName){
            return APIXHRService("POST", API_PATH + "savedmodels/agents/get-log", {
                projectKey, smId, versionId, logName
            });
        },
        deleteLog: function(projectKey, smId, versionId, logName) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/delete-log", {
                projectKey, smId, versionId, logName
            });
        },
        clearLogs: function(projectKey, smId, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/clear-logs", {
                projectKey, smId, versionId
            });
        },
        test: function(projectKey, smId, versionId, query, timeoutPromise){
            return APIXHRService("POST", API_PATH + "savedmodels/agents/test", {
                projectKey, smId, versionId, query: query
            }, null, null, timeoutPromise);
        },
        chat: function(projectKey, smId, versionId, chatInput, chunkCallback, abortController) {
            return streamRequest('POST', API_PATH + 'savedmodels/agents/chat', {
                projectKey, smId, versionId, chatInput: JSON.stringify(chatInput)
            }, readStreamEventData(chunkCallback), abortController);
        },
        stopDevKernel: function(projectKey, smId, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/agents/stop-dev-kernel", {
                projectKey, smId, versionId
            })
        }
    },
    retrievalAugmentedLLMs:{
        save: function(savedModel) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/save", {
                savedModel: JSON.stringify(savedModel),
            });
        },
        checkSaveConflict: function(savedModel) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/check-save-conflict", {
                savedModel: JSON.stringify(savedModel)
            });
        },
        create: function(projectKey, name, knowledgeBankRef, llmId) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/create", {
                projectKey, name, knowledgeBankRef, llmId
            });
        },
        list: function(projectKey) {
            return APIXHRService("GET", API_PATH + "savedmodels/retrieval-augmented-llm/list", {
                projectKey,
            });
        },
        test: function(projectKey, smId, versionId, query, timeoutPromise) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/test", {
                projectKey, smId, versionId, query: query
            }, null, null, timeoutPromise);
        },
        chat: function(projectKey, smId, versionId, chatInput, chunkCallback, abortController) {
            return streamRequest('POST', API_PATH + 'savedmodels/retrieval-augmented-llm/chat', {
                projectKey, smId, versionId, chatInput: JSON.stringify(chatInput)
            }, readStreamEventData(chunkCallback), abortController);
        },
        stopDevKernel: function(projectKey, smId, versionId) {  // versionId is kept to be the same as agents.stopDevKernel()
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/stop-dev-kernel", {
                projectKey, smId
            })
        },
        listLogs: function(projectKey, smId, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/list-logs", {
                projectKey, smId, versionId
            });
        },
        getLog: function(projectKey, smId, versionId, logName){
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/get-log", {
                projectKey, smId, versionId, logName
            });
        },
        deleteLog: function(projectKey, smId, versionId, logName) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/delete-log", {
                projectKey, smId, versionId, logName
            });
        },
        clearLogs: function(projectKey, smId, versionId) {
            return APIXHRService("POST", API_PATH + "savedmodels/retrieval-augmented-llm/clear-logs", {
                projectKey, smId, versionId
            });
        },
    },
    getPreparedMetricHistory: function(projectKey, modelId, metric, metricId) {
        return APIXHRService("GET", API_PATH + "savedmodels/get-prepared-metric-history", {
            projectKey: projectKey,
            modelId: modelId,
            metricId: metricId
        });
    },
    getPreparedMetricHistories: function(projectKey, modelId, displayedState) {
        return APIXHRService("POST", API_PATH + "savedmodels/get-prepared-metric-histories", {
            projectKey: projectKey,
            modelId: modelId,
            data: JSON.stringify(displayedState || {})
        });
    },
    listComputedMetrics: function(projectKey, modelId) {
        return APIXHRService("GET", API_PATH + "savedmodels/list-computed-metrics", {
            projectKey: projectKey,
            modelId: modelId
        });
    },
    createMetricsDataset: function(projectKey, modelId, view, partition, filter) {
        return APIXHRService("GET", API_PATH + "datasets/create-metrics-dataset", {
            projectKey: projectKey,
            objectId: modelId,
            view: view,
            partition: partition,
            filter: filter,
            taggableType: "SAVED_MODEL"
        });
    },
    getPreparedMetricPartitions: function(projectKey, modelId, displayedState) {
        return APIXHRService("POST", API_PATH + "savedmodels/get-prepared-metric-partitions", {
            projectKey: projectKey,
            modelId: modelId,
            data: JSON.stringify(displayedState || {})
        });
    },
    runChecks: function(projectKey, modelId) {
        return APIXHRService("GET", API_PATH + "savedmodels/run-checks", {
            projectKey: projectKey,
            modelId: modelId
        });
    },
    runCheck: function(projectKey, modelId, metricsChecks) {
        return APIXHRService("POST", API_PATH + "savedmodels/run-check", {
            projectKey: projectKey,
            modelId: modelId,
            data: JSON.stringify(metricsChecks)
        });
    },
    getCheckHistories: function(projectKey, modelId, displayedState) {
        return APIXHRService("POST", API_PATH + "savedmodels/get-prepared-check-histories", {
            projectKey: projectKey,
            modelId: modelId,
            data: JSON.stringify(displayedState || {})
        });
    },
    listComputedChecks: function(projectKey, modelId) {
        return APIXHRService("GET", API_PATH + "savedmodels/list-computed-checks", {
            projectKey: projectKey,
            modelId: modelId
        });
    },
    getHint: function(projectKey, modelId, probe) {
        return APIXHRService("POST", API_PATH + "savedmodels/get-hint", {
            projectKey: projectKey,
            modelId: modelId,
            data: JSON.stringify(probe)
        });
    },
    clearMetrics: function(projectKey, modelId, partition) {
        return APIXHRService("GET", API_PATH + "savedmodels/clear-metrics", {
            projectKey: projectKey,
            modelId: modelId,
            partition: partition
        });
    },
    getModelDocumentationExportURL: function(exportId) {
        return API_PATH + "savedmodels/model-documentation-export?exportId=" + encodeURIComponent(exportId);
    },
    guessTrainDeploy: function(projectKey, modelId) {
        return APIXHRService("GET", API_PATH + "savedmodels/guess-train-deploy", {
            projectKey: projectKey,
            modelId: modelId
        });
    },
    getModelVersionGovernanceStatus: function(projectKey, id, fullModelId) {
        return APIXHRService("GET", API_PATH + "savedmodels/get-governance-status", {
            projectKey: projectKey,
            id: id,
            fullModelId: fullModelId
        }, "nospinner");
    }
},
experimenttracking: {
    compareWithExperimentTracking: function(projectKey, modelIds, experimentId) {
        return APIXHRService("POST", API_PATH + "experiment-tracking/compare-with-experiment-tracking", {
            projectKey: projectKey,
            modelIds: JSON.stringify(modelIds),
            experimentId
        });
    },
    createExperimentForAnalysisModels: function(projectKey, experimentName) {
        return APIXHRService("POST", API_PATH + "experiment-tracking/create-experiment-tracking-for-analysis-models", {
            projectKey: projectKey,
            experimentName : experimentName
        });
    },
    getExperiments: function(projectKey) {
        return APIXHRService("GET", API_PATH + "experiment-tracking/experiments/list", {
            projectKey: projectKey
        });
    },
},
retrievableknowledge: {
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/list", {projectKey});
    },
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/list-heads", {projectKey});
    },
    get: function(projectKey, knowledgeBankRef) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/get", {projectKey, knowledgeBankRef});
    },
    getFullInfo: function(projectKey, knowledgeBankRef) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/get-full-info", {projectKey, knowledgeBankRef});
    },
    save: function(data, saveInfo) {
        return APIXHRService("POST", API_PATH + "knowledge-bank/save", {
            data: JSON.stringify(data),
            saveInfo: JSON.stringify(saveInfo || {})
        });
    },
    getCurrentVersionInfo: function(projectKey, knowledgeBankRef) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/get-current-version-info", { projectKey, knowledgeBankRef });
    },
    getVectorStoreConnections: function() {
        return APIXHRService("GET", API_PATH + "knowledge-bank/list-vector-store-connections", {});
    },
    checkPineconeCompatibility: function(projectKey, pineconeConnection, envSelection) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/check-pinecone-compatibility", {projectKey, pineconeConnection, envSelection});
    },
    clear: function(projectKey, knowledgeBankRef) {
        return APIXHRService("POST", API_PATH + "knowledge-bank/clear", { projectKey, knowledgeBankRef });
    },
    getRetrievalAugmentedLLMList: function(projectKey, knowledgeBankRef) {
        return APIXHRService("GET", API_PATH + "knowledge-bank/get-retrieval-augmented-llm-list", { projectKey, knowledgeBankRef });
    },
    search: function(projectKey, knowledgeBankRef, input, params) {
        return APIXHRService("POST", API_PATH + "knowledge-bank/search", {
            projectKey, knowledgeBankRef, input, params: JSON.stringify(params)
        })
    },
},
modelevaluationstores: {
    get: function(projectKey, smartId) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/get", {projectKey: projectKey, smartId: smartId});
    },
    getSummary: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/get-summary", {projectKey: projectKey, id: id});
    },
    getFullInfo: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/get-full-info", {projectKey: projectKey, smartId: id});
    },
    list: function(projectKey, mesFlavors) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/list", {projectKey: projectKey, mesFlavors: mesFlavors});
    },
    listHeads: function(projectKey, mesFlavors) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/list-heads", {projectKey: projectKey, mesFlavors: mesFlavors});
    },
    listEvaluations: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/list-evaluations", {projectKey: projectKey, smartId: id});
    },
    deleteEvaluations: function(projectKey, id, evaluations) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/delete-evaluations", {projectKey: projectKey, smartId: id, evaluations:JSON.stringify(evaluations)});
    },
    getEvaluationDetails: function(projectKey, id, evaluationId) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/get-evaluation-details", {projectKey: projectKey, smartId: id, evaluationId: evaluationId});
    },
    getLatestEvaluationDetails: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/get-latest-evaluation-details", {projectKey: projectKey, smartId: id});
    },
    listWithAccessible: function(projectKey, mesFlavors) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/list-with-accessible/", {projectKey: projectKey, mesFlavors: mesFlavors});
    },
    save: function(data, saveInfo) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/save", {
            data: JSON.stringify(data),
            saveInfo: JSON.stringify(saveInfo || {})
        });
    },
    rename: function(projectKey, mesId, newName) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/rename", { projectKey, mesId, newName })
    },
    getPreparedMetricHistory: function(projectKey, mesId, metric, metricId) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/get-prepared-metric-history", {
            projectKey: projectKey,
            mesId: mesId,
            data: JSON.stringify(metric),
            metricId: metricId
        });
    },
    getPreparedMetricHistories: function(projectKey, mesId, displayedState) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/get-prepared-metric-histories", {
            projectKey: projectKey,
            mesId: mesId,
            data: JSON.stringify(displayedState || {})
        });
    },
    listComputedMetrics: function(projectKey, mesId) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/list-computed-metrics", {
            projectKey: projectKey,
            mesId: mesId
        });
    },
    createMetricsDataset: function(projectKey, mesId, view, partition, filter) {
        return APIXHRService("GET", API_PATH + "datasets/create-metrics-dataset", {
            projectKey: projectKey,
            objectId: mesId,
            view: view,
            partition: partition,
            filter: filter,
            taggableType: "MODEL_EVALUATION_STORE"
        });
    },
    getPreparedMetricPartitions: function(projectKey, mesId, displayedState) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/get-prepared-metric-partitions", {
            projectKey: projectKey,
            mesId: mesId,
            data: JSON.stringify(displayedState || {})
        });
    },
    runChecks: function(projectKey, mesId, evaluationId, allRuns) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/run-checks", {
            projectKey: projectKey,
            mesId: mesId,
            evaluationId: evaluationId,
            allRuns: allRuns
        });
    },
    runCheck: function(projectKey, mesId, evaluationId, allRuns, metricsChecks) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/run-check", {
            projectKey: projectKey,
            mesId: mesId,
            evaluationId: evaluationId,
            allRuns: allRuns,
            data: JSON.stringify(metricsChecks)
        });
    },
    listComputedChecks: function(projectKey, mesId) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/list-computed-checks", {
            projectKey: projectKey,
            mesId: mesId
        });
    },
    getCheckHistories: function(projectKey, mesId, displayedState) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/get-prepared-check-histories", {
            projectKey: projectKey,
            mesId: mesId,
            data: JSON.stringify(displayedState || {})
        });
    },
    getHint: function(projectKey, mesId, probe) {
        return APIXHRService("POST", API_PATH + "modelevaluationstores/get-hint", {
            projectKey: projectKey,
            mesId: mesId,
            data: JSON.stringify(probe)
        });
    },
    clearMetrics: function(projectKey, mesId, partition) {
        return APIXHRService("GET", API_PATH + "modelevaluationstores/clear-metrics", {
            projectKey: projectKey,
            mesId: mesId,
            partition: partition
        });
    },
},
modelevaluations: {
    get: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get", {
            fme: fme
        });
    },
    saveEvaluationModelUserMeta: function(fme, data) {
        return APIXHRService("POST", API_PATH + "modelevaluations/save-evaluation-model-user-meta", {
            fme: fme, meta: JSON.stringify(data)
        });
    },
    saveEvaluationUserMeta: function(fme, data) {
        return APIXHRService("POST", API_PATH + "modelevaluations/save-evaluation-user-meta", {
            fme: fme, meta: JSON.stringify(data)
        });
    },
    listCompatibleReferencesForDrift: function(projectKey, currentId, matchLabels) {
        return APIXHRService("GET", API_PATH + "modelevaluations/list-compatible-references-for-drift", {
            projectKey, currentId, matchLabels
        });
    },
    getPerTimeseriesMetrics: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get-per-timeseries-metrics", { fme });
    },
    getForecasts: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get-model-forecasts", { fme });
    },
    getSubpopulationsInfo: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get-subpopulations-info", {
            fme: fme
        });
    },
    subpopulationComputationStart: function(fme, features, computationParams) {
        return APIXHRService("POST", API_PATH + "modelevaluations/subpopulation-computation-start" , {
            fme: fme,
            features: JSON.stringify(features),
            computationParams: JSON.stringify(computationParams),
        });
    },
    getSubpopulation: function(fme, features) {
        return APIXHRService("POST", API_PATH + "modelevaluations/get-subpopulation" , {
            fme: fme,
            features: JSON.stringify(features)
        });
    },
    pdpComputationStart: function(fme, features, computationParams) {
        return APIXHRService("POST", API_PATH + "modelevaluations/pdp-computation-start" , {
            fme: fme,
            features: JSON.stringify(features),
            computationParams: JSON.stringify(computationParams),
        });
    },
    getGlobalExplanations: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get-global-explanations" , {
            fme: fme,
        });
    },
    getGlobalExplanationsFacts: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get-global-explanations-facts" , {
            fme: fme,
        });
    },
    individualExplanationsComputationStart: function(fme, computationParams) {
        return APIXHRService("POST", API_PATH + "modelevaluations/individual-explanations-computation-start" , {
            fme: fme,
            computationParams: JSON.stringify(computationParams),
        });
    },
    getIndividualExplanations: function(fme) {
        return APIXHRService("GET", API_PATH + "modelevaluations/get-individual-explanations" , {
            fme: fme
        });
    },
    computeDrift: function(projectKey, referenceId, currentId, params) {
        return APIXHRService("POST", API_PATH + "modelevaluations/compute-drift" , {
            referenceId: referenceId,
            currentId: currentId,
            projectKey: projectKey,
            params: JSON.stringify(params)
        });
    },
    sampleRefreshTable: function(fme, displayScript, filters) {
        return APIXHRService("POST", API_PATH + "modelevaluations/sample/refresh-table", {
            fme: fme,
            displayScript: JSON.stringify(displayScript),
            filters: JSON.stringify(filters),
        })
    },
    sampleGetTableChunk: function(fme, displayScript, firstRow, nbRows, firstCol, nbCols, filters) {
        return APIXHRService("POST", API_PATH + "modelevaluations/sample/get-table-chunk", {
            fme: fme,
            displayScript: JSON.stringify(displayScript),
            filters: JSON.stringify(filters),
            firstRow: firstRow,
            nbRows: nbRows,
            firstCol: firstCol,
            nbCols: nbCols,
        })
    },
    sampleDetailedColumnAnalysis: function(fme, data, column, alphanumMaxResults) {
        return APIXHRService("POST", API_PATH + "modelevaluations/sample/detailed-column-analysis", {
            fme: fme,
            data: JSON.stringify(data),
            column: column,
            alphanumMaxResults: alphanumMaxResults
        });
    },
    exportEvaluationSample: function(fme, exportParams) {
        return APIXHRService("POST", API_PATH + "modelevaluations/export-evaluation-sample", {
            fme: fme,
            exportParams: JSON.stringify(exportParams),
        });
    },
},
modelcomparisons: {
    get: function(projectKey, modelComparisonId) {
        return APIXHRService("GET", API_PATH + "modelcomparisons/get", {
            projectKey: projectKey,
            modelComparisonId: modelComparisonId
        });
    },
    getSummary: function(projectKey, modelComparisonId) {
        return APIXHRService("GET", API_PATH + "modelcomparisons/get-summary", {projectKey: projectKey, modelComparisonId: modelComparisonId});
    },
    browseComparables: function(projectKey, source, modelTaskType, smvs) {
        const params = {
            projectKey: projectKey,
            modelTaskType: modelTaskType,
            source: source,
            versions: JSON.stringify(smvs)
        };
        return APIXHRService("GET", API_PATH + "modelcomparisons/browse-comparables", params);
    },
    getFullInfo: function(projectKey, modelComparisonId) {
        return APIXHRService("GET", API_PATH + "modelcomparisons/get-full-info", {projectKey: projectKey, modelComparisonId: modelComparisonId});
    },
    delete: function(projectKey, modelComparisonId) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/delete", {projectKey: projectKey, modelComparisonId: modelComparisonId});
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "modelcomparisons/list", {projectKey: projectKey});
    },
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "modelcomparisons/list-heads", {projectKey: projectKey});
    },
    save: function(data, saveInfo) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/save", {
            data: JSON.stringify(data),
            saveInfo: JSON.stringify(saveInfo || {})
        });
    },
    create: function(projectKey, displayName, modelTaskType) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/create", {
            projectKey: projectKey,
            displayName: displayName,
            modelTaskType: modelTaskType
        });
    },
    getModelsDetails: function(fullIds) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/get-details", {
            fullIds: JSON.stringify(fullIds)
        });
    },
    getSampleSchema: function(projectKey, fmeIds) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/llm/sample/get-schema", {
            projectKey: projectKey,
            fmeIds: JSON.stringify(fmeIds)
        });
    },
    sampleRefreshTable: function(projectKey, modelComparisonId, fmeIds, displayScript, filters) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/llm/sample/refresh-table", {
            projectKey: projectKey,
            modelComparisonId: modelComparisonId,
            fmeIds: JSON.stringify(fmeIds),
            displayScript: JSON.stringify(displayScript),
            filters: JSON.stringify(filters)
        });
    },
    sampleGetTableChunk: function(projectKey, modelComparisonId, fmeIds, displayScript, firstRow, nbRows, firstCol, nbCols, filters) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/llm/sample/get-table-chunk", {
            projectKey: projectKey,
            modelComparisonId: modelComparisonId,
            fmeIds: JSON.stringify(fmeIds),
            displayScript: JSON.stringify(displayScript),
            filters: JSON.stringify(filters),
            firstRow: firstRow,
            nbRows: nbRows,
            firstCol: firstCol,
            nbCols: nbCols,
        })
    },
    sampleDetailedColumnAnalysis: function(projectKey, modelComparisonId, fmeIds, data, column, alphanumMaxResults) {
        return APIXHRService("POST", API_PATH + "modelcomparisons/llm/sample/detailed-column-analysis", {
            projectKey: projectKey,
            modelComparisonId: modelComparisonId,
            fmeIds: JSON.stringify(fmeIds),
            data: JSON.stringify(data),
            column: column,
            alphanumMaxResults: alphanumMaxResults
        });
    }
},
labelingtasks: {
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "labelingtasks/list", {projectKey});
    },
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "labelingtasks/list-heads", {projectKey});
    },
    get: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "labelingtasks/get", {projectKey, id});
    },
    getFullInfo: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "labelingtasks/get-full-info", {projectKey, id});
    },
    rename: function(projectKey, id, newName) {
        return APIXHRService("GET", API_PATH + "labelingtasks/rename", {projectKey, id, newName});
    },
    save: function(labelingTask, zoneId, saveInfo, commitMessage) {
        return APIXHRService("POST", API_PATH + "labelingtasks/save", {
            labelingTask: JSON.stringify(labelingTask),
            zoneId: zoneId,
            saveInfo: JSON.stringify(saveInfo),
            commitMessage,
        });
    },
    invalidateMetadataTableCache(projectKey, labelingTaskId) {
        return APIXHRService("GET", API_PATH + "labelingtasks/invalidate-metadata-table-cache", {projectKey, labelingTaskId});
    },
    getLabelingTaskPrivileges: function(projectKey, labelingTaskId) {
        return APIXHRService("GET", API_PATH + "labelingtasks/get-labeling-task-privileges", {projectKey, labelingTaskId});
    },
    checkSaveConflict: function(labelingTask) {
        return APIXHRService("POST", API_PATH + "labelingtasks/check-save-conflict", {
            newLabelingTask: JSON.stringify(labelingTask)
        });
    }
},
images: {
    uploadImage: function(projectKey, type, id, dataUrl) {
        return uploadFileRequest("image/set-image", function(formdata) {
            formdata.append("projectKey", projectKey);
            formdata.append("type", type);
            formdata.append("id", id);
            formdata.append("dataUrl", dataUrl);
        }, null);
    },
    removeImage: function(projectKey, type, id) {
        return APIXHRService("GET", API_PATH + "image/remove-image", {projectKey: projectKey, type: type, id: id}, "nospinner");
    },
    getObjectThumbnailInfo: function(projectKey, type, id) {
        return APIXHRService("GET", API_PATH + "image/get-object-thumbnail-info", {projectKey: projectKey, type: type, id: id}, "nospinner");
    },
    uploadCapturedObjectThumbnail: function(projectKey, type, id, dataUrl) {
        return uploadFileRequest("image/set-captured-object-thumbnail", function(formdata) {
            formdata.append("projectKey", projectKey);
            formdata.append("type", type);
            formdata.append("id", id);
            formdata.append("dataUrl", dataUrl);
        }, null);
    }
},
notifications: {
    count: function() {
        return APIXHRService("GET", API_PATH + "notifications/count", {}, "nospinner");
    },
    get: function() {
        return APIXHRService("GET", API_PATH + "notifications/get", {});
    },
    ack: function(timestamp) {
        return APIXHRService("POST", API_PATH + "notifications/acknowledge", {timestamp: timestamp}, "nospinner" );
    }
},
timelines: {
    getForObject: function(projectKey, objectType, objectId, from, limit) {
        return APIXHRService("GET", API_PATH + "timelines/get-for-object", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId,
            from: from,
            limit: limit
        });
    },
    getForProject: function(projectKey, from, limit) {
        return APIXHRService("GET", API_PATH + "timelines/get-for-project", {projectKey: projectKey, from: from, limit: limit});
    }
},
requests: {
    listRequests: function(facets, offset, limit) {
        return APIXHRService("GET", API_PATH + "requests/", { facets: JSON.stringify(facets), offset: offset, limit: limit });
    },
    getRequest: function(requestId) {
        return APIXHRService("GET", API_PATH + "requests/get-request", { requestId: requestId });
    },
    createAccessRequest: function(objectType, objectProjectKey, objectId, requestMessage) {
        return APIXHRService("POST", API_PATH + "requests/create-access-request", { objectType, objectProjectKey, objectId, requestMessage });
    },
    createSharingRequest: function(objectType, objectProjectKey, objectId, sharingTargetProject, requestMessage) {
        return APIXHRService("POST", API_PATH + "requests/create-sharing-request", { objectType, objectProjectKey, objectId, sharingTargetProject, requestMessage });
    },
    createPluginRequest: function(requestType, objectId, requestMessage) {
        return APIXHRService("POST", API_PATH + "requests/create-plugin-request", { requestType, objectId, requestMessage });
    },
    createCodeEnvRequest: function(requestType, objectId, envLang, envDesc, specPackagesList, specCondaEnvironment, requestMessage, source) {
        return APIXHRService("POST", API_PATH + "requests/create-code-env-request", { requestType, objectId, envLang, envDesc, specPackagesList, specCondaEnvironment, requestMessage, source });
    },
    createInternalCodeEnvRequest: function(requestType, objectId, requestMessage, source) {
        return APIXHRService("POST", API_PATH + "requests/create-code-env-request", { requestType, objectId, envLang:"PYTHON", envDesc: JSON.stringify({"deploymentMode": "DSS_INTERNAL"}), specPackagesList: "", specCondaEnvironment:"", requestMessage, source });
    },
    createInstanceAccessRequest: function(requestMessage) {
        return APIXHRService("POST", API_PATH + "requests/create-instance-access-request", { requestMessage });
    },
    createProfileUpgradeRequest: function(requestMessage) {
        return APIXHRService("POST", API_PATH + "requests/create-profile-upgrade-request", { requestMessage });
    },
    approve: function(requestId, requestResponse, closingMessage) {
        return APIXHRService("POST", API_PATH + "requests/approve", {
            requestId: requestId,
            requestResponse: JSON.stringify(requestResponse),
            closingMessage: closingMessage
        });
    },
    reject: function(requestId, closingMessage) {
        return APIXHRService("POST", API_PATH + "requests/reject", { requestId: requestId, closingMessage: closingMessage });
    },
    getLatestRequestForCurrentUser: function(objectId, objectType, objectProjectKey){
        return APIXHRService("GET", API_PATH + "requests/get-latest-object-request-for-user", {objectId, objectType, objectProjectKey});
    }
},
discussions: {
    getCounts: function(projectKey, objectType, objectId, workspaceKey) {
        return APIXHRService("GET", API_PATH + "discussions/get-discussion-counts", {
            projectKey,
            objectType,
            objectId,
            workspaceKey,
        }, "nospinner");
    },
    getForObject: function(projectKey, objectType, objectId, workspaceKey) {
        return APIXHRService("GET", API_PATH + "discussions/get-for-object", {
            projectKey,
            objectType,
            objectId,
            workspaceKey,
        });
    },
    create: function(projectKey, objectType, objectId, topic, reply, workspaceKey) {
        return APIXHRService("POST", API_PATH + "discussions/create", {
            projectKey,
            objectType,
            objectId,
            topic,
            reply,
            workspaceKey,
        });
    },
    save: function(projectKey, objectType, objectId, discussionId, topic, workspaceKey) {
        return APIXHRService("POST", API_PATH + "discussions/save", {
            projectKey,
            objectType,
            objectId,
            discussionId,
            topic,
            workspaceKey,
        });
    },
    reply: function(projectKey, objectType, objectId, discussionId, content, replyId, workspaceKey) {
        return APIXHRService("POST", API_PATH + "discussions/reply", {
            projectKey,
            objectType,
            objectId,
            discussionId,
            replyId,
            content,
            workspaceKey,
        });
    },
    ack: function(projectKey, objectType, objectId, discussionId, workspaceKey) {
        return APIXHRService("POST", API_PATH + "discussions/ack", {
            projectKey,
            objectType,
            objectId,
            discussionId,
            workspaceKey,
        });
    },
    close: function(projectKey, objectType, objectId, discussionId, closed, workspaceKey) {
        return APIXHRService("POST", API_PATH + "discussions/close", {
            projectKey,
            objectType,
            objectId,
            discussionId,
            closed,
            workspaceKey,
        });
    },
    delete: function(projectKey, objectType, objectId, discussionId, workspaceKey) {
        return APIXHRService("POST", API_PATH + "discussions/delete-discussion", {
            projectKey,
            objectType,
            objectId,
            discussionId,
            workspaceKey,
        });
    },
    inbox: {
        search: function (query, facets) {
            return APIXHRService("POST", API_PATH + "inbox/search", {
                query: query,
                facets: JSON.stringify(facets)
            });
        },
    }
},
interests: {
    getForObject: function(user, objectType, projectKey, objectId, workspaceKey) {
        return APIXHRService("GET", API_PATH + "interests/get-interest-for-object", {
            user: user,
            objectType,
            projectKey,
            objectId,
            workspaceKey,
        });
    },
    getUserInterests: function(user, offset, maxItems, filters, starsOnly, nospinner) {
        return APIXHRService("GET", API_PATH + "interests/get-user-interests", {
            user: user,
            offset: offset,
            maxItems: maxItems,
            filters: filters,
            starsOnly: starsOnly
        }, nospinner ? "nospinner" : undefined);
    },
    watch: function(items, watch) {
        return APIXHRService("POST", API_PATH + "interests/watch", {
            items: JSON.stringify(items),
            watch: watch
        });
    },
    star: function(items, star) {
        return APIXHRService("POST", API_PATH + "interests/star", {
            items: JSON.stringify(items),
            star: star
        });
    },
    listWatchingUsers: function(objectType, projectKey, objectId, workspaceKey) {
        return APIXHRService("GET", API_PATH + "interests/list-watching-users", {
            objectType,
            projectKey,
            objectId,
            workspaceKey,
        });
    },
    listUsersWithStar: function(objectType, projectKey, objectId, workspaceKey) {
        return APIXHRService("GET", API_PATH + "interests/list-users-with-star", {
            objectType,
            projectKey,
            objectId,
            workspaceKey,
        });
    }
},
magic: function(filter) {
    return APIXHRService("GET", API_PATH + "home/magic", {filter: filter});
},
translations: {
    get: function(location, language) {
        return APIXHRService("GET", API_PATH + "get-translations", { location: location, language: language });
    },
    save: function(location, language, translations, raw=false) {
        return APIXHRService("POST", API_PATH + "save-translations", { location: location, language: language, translations: raw ? translations : JSON.stringify(translations) });
    }
},
getConfiguration: function() {
    return APIXHRService("GET", API_PATH + "get-configuration", {});
},
getHomeArticles: function(nospinner) {
    return APIXHRService("GET", API_PATH + "get-home-articles", undefined, nospinner ? "nospinner" : undefined);
},
running: {
    listPersonal: function() {
        return APIXHRService("GET", API_PATH + "running/list-personal", {});
    },
    listAll: function() {
        return APIXHRService("GET", API_PATH + "running/list-all", undefined, "nospinner");
    }
},
futures: {
    getUpdate: function(futureId) {
        return APIXHRService("GET", API_PATH + "futures/get-update", {futureId: futureId}, "nospinner");
    },
    peekUpdate: function(futureId) {
        return APIXHRService("GET", API_PATH + "futures/peek-update", {futureId: futureId}, "nospinner");
    },
    abort: function(futureId, spinner=true) {
        return APIXHRService("POST", API_PATH + "futures/abort", {futureId: futureId}, spinner? undefined : "nospinner");
    },
    list: function() {
        return APIXHRService("GET", API_PATH + "futures/list");
    },
    listAll: function() {
        return APIXHRService("GET", API_PATH + "futures/list-all");
    },
    listScenarios: function() {
        return APIXHRService("GET", API_PATH + "futures/list-scenarios");
    },
    listAllScenarios: function() {
        return APIXHRService("GET", API_PATH + "futures/list-all-scenarios");
    }
},
login: function(login, password) {
    return APIXHRService("POST", API_PATH + "login", {
        login: login, password: password
    });
},
noLoginLogin: function() {
    return APIXHRService("POST", API_PATH + "no-login-login");
},
getSAMLRedirectURL : function(redirectTo){
    return APIXHRService("POST", API_PATH + "get-saml-redirect-url", {redirectTo});
},

getOpenIDRedirectURL : function(){
    return APIXHRService("GET", API_PATH + "get-openid-redirect-url", {dssUrl: urlWithProtocolAndHost()});
},

exchangeAuthorizationCode : function(code, state) {
    return APIXHRService("POST", API_PATH + "openid-callback", {code: code, state: state, dssUrl: urlWithProtocolAndHost()});
},

logout: function() {
    return APIXHRService("POST", API_PATH + "logout");
},
security: {
    listUsers: function(projectKey) {
        return APIXHRService("GET", API_PATH + "security/list-users", { projectKey });
    },
    listWorkspaceUsers: function(workspaceKey) {
        return APIXHRService("GET", API_PATH + "security/list-users", { workspaceKey });
    },
    listConnectedUsers: function(projectKey) {
        return APIXHRService("GET", API_PATH + "security/list-connected-users", {projectKey});
    },
    listGroups: function(localOnly) {
        return APIXHRService("GET", API_PATH + "security/list-groups", {localOnly});
    },
    listGroupsWithSecurityOption: function(localOnly) {
        return APIXHRService("GET", API_PATH + "security/list-groups-with-security-options", {localOnly});
    },
    listGroupsFull: function() {
        return APIXHRService("GET", API_PATH + "security/list-groups-full");
    },
    listUsersMatchingEmail: function(projectKey, workspaceKey, email) {
        return APIXHRService("GET", API_PATH + "security/list-users-matching-email", { projectKey, workspaceKey, email });
    },
    updateGroup: function(groupData) {
        return APIXHRService("POST", API_PATH + "security/update-group", {groupData: JSON.stringify(groupData)});
    },
    prepareUpdateGroup: function(groupData) {
        return APIXHRService("POST", API_PATH + "security/prepare-update-group", {groupData: JSON.stringify(groupData)});
    },
    createGroup: function(groupData) {
        return APIXHRService("POST", API_PATH + "security/create-group", {groupData: JSON.stringify(groupData)});
    },
    getGroup: function(groupName) {
        return APIXHRService("GET", API_PATH + "security/get-group", {groupName: groupName});
    },
    deleteGroup: function(groupName) {
        return APIXHRService("POST", API_PATH + "security/delete-group", {groupName: groupName});
    },
    prepareDeleteGroup: function(groupName) {
        return APIXHRService("POST", API_PATH + "security/prepare-delete-group", {groupName: groupName});
    },
    getAuthorizationMatrix: function() {
        return APIXHRService("GET", API_PATH + "security/get-authorization-matrix");
    },
    getAuditBuffer: function(includeAllCalls) {
        return APIXHRService("GET", API_PATH + "security/get-audit-buffer", {
            includeAllCalls: includeAllCalls
        });
    },
    startTrialFromNoneUser: function() {
        return APIXHRService("POST", API_PATH + "security/start-trial-from-none-user")
    }
},
zones : {
    getFullInfo: function(projectKey, zoneId) {
        return APIXHRService("GET", API_PATH + "zones/get-full-info", { projectKey: projectKey, zoneId: zoneId });
    },
    get: function(projectKey, zoneId) {
        return APIXHRService("GET", API_PATH + "zones/get", { projectKey: projectKey, zoneId: zoneId });
    }
},
dashboards : {
    save: function(dashboardData, commitMessage) {
        return APIXHRService("POST", API_PATH + "dashboards/save", { dashboardData: JSON.stringify(dashboardData), commitMessage: commitMessage });
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "dashboards/list", { projectKey: projectKey });
    },
    listEditable: function(projectKey) {
        return APIXHRService("GET", API_PATH + "dashboards/list-editable", {projectKey: projectKey});
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "dashboards/list-heads", {projectKey: projectKey, tagFilter: tagFilter});
    },
    listSummaries: function(projectKey) {
        return APIXHRService("GET", API_PATH + "dashboards/list-summaries", {projectKey: projectKey});
    },
    get: function(projectKey, dashboardId, exportId) {
        return APIXHRService("GET", API_PATH + "dashboards/get", { projectKey, dashboardId, exportId });
    },
    getFullInfo: function(projectKey, dashboardId) {
        return APIXHRService("GET", API_PATH + "dashboards/get-full-info", { projectKey: projectKey, dashboardId: dashboardId });
    },
    getSummary: function(projectKey, dashboardId, exportId) {
        return APIXHRService("GET", API_PATH + "dashboards/get-summary", { projectKey, dashboardId, exportId });
    },
    copy: function(projectKey, dashboardId, name, deepCopy) {
        return APIXHRService("POST", API_PATH + "dashboards/copy", { projectKey: projectKey, dashboardId: dashboardId, name: name, deepCopy: deepCopy });
    },
    getEnrichedPage: function(projectKey, dashboardId, pageIdx, exportId) {
        return APIXHRService("GET", API_PATH + "dashboards/get-enriched-page", { projectKey, dashboardId, pageIdx, exportId });
    },
    copyPage: function(projectKey, sourceDashboardId, page, targetedDashboardId, copyPageName, pointerMode) {
        return APIXHRService("POST", API_PATH + "dashboards/copy-page", {projectKey: projectKey, sourceDashboardId: sourceDashboardId, pageData: JSON.stringify(page), targetedDashboardId: targetedDashboardId, copyPageName: copyPageName, pointerMode: pointerMode});
    },
    makeListed: function(projectKey, dashboardIds, listed) {
        return APIXHRService("POST", API_PATH + "dashboards/make-listed", { projectKey: projectKey, dashboardIds: JSON.stringify(dashboardIds), listed: listed });
    },
    getMissingReaderAuthorizations: function(projectKey, dashboardIds) {
        return APIXHRService("GET", API_PATH + "dashboards/get-missing-reader-authorizations", { projectKey: projectKey, dashboardIds: JSON.stringify(dashboardIds) });
    },
    export: function(projectKey, exportFormat, dashboards) {
        return APIXHRService("POST", API_PATH + "dashboards/export", { projectKey: projectKey, exportFormat: JSON.stringify(exportFormat), dashboards: JSON.stringify(dashboards) });
    },
    getExportURL: function(projectKey, exportId) {
        return API_PATH + "dashboards/download-export?"
        + "projectKey=" + encodeURIComponent(projectKey)
        + "&exportId=" + encodeURIComponent(exportId);
    },
    share: function(projectKey, dashboardId, invitedEmails, invitedUsers, personalMessage) {
        return APIXHRService("POST", API_PATH + "dashboards/share", {projectKey: projectKey, dashboardId: dashboardId, invitedEmails: JSON.stringify(invitedEmails), invitedUsers: JSON.stringify(invitedUsers), personalMessage: personalMessage});
    },
    multiPin: function(projectKey, insightId, tilesData, pinningOrdersData, pointerMode) {
        return APIXHRService("POST", API_PATH + "dashboards/multi-pin", {
            projectKey: projectKey,
            insightId: insightId,
            tilesData: JSON.stringify(tilesData),
            pinningOrdersData: JSON.stringify(pinningOrdersData),
            pointerMode: pointerMode
        });
    },
    insights: {
        save: function(insightData, commitMessage, payload) {
            return APIXHRService("POST", API_PATH + "dashboards/insights/save", { insightData: JSON.stringify(insightData), commitMessage: commitMessage, payload: payload });
        },
        createAndPin: function(projectKey, insightData, tileData, pinningOrdersData, payloads) {
            return APIXHRService("POST", API_PATH + "dashboards/insights/create-and-pin",
            {
                projectKey: projectKey,
                insightData: JSON.stringify(insightData),
                tileData: JSON.stringify(tileData),
                pinningOrdersData: JSON.stringify(pinningOrdersData),
                payloadsData: payloads?JSON.stringify(payloads):null
            });
        },
        list: function(projectKey) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/list", { projectKey: projectKey });
        },
        listWithAccessState: function(projectKey) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/list-with-access-state", { projectKey: projectKey });
        },
        listHeads: function(projectKey, tagFilter) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/list-heads", {projectKey: projectKey, tagFilter: tagFilter});
        },
        get: function(projectKey, insightId) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/get", { projectKey: projectKey, insightId: insightId });
        },
        getFullInfo: function(projectKey, insightId) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/get-full-info", { projectKey: projectKey, insightId: insightId });
        },
        getWithPayload: function(projectKey, insightId) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/get-with-payload", { projectKey: projectKey, insightId: insightId });
        },
        copy: function(projectKey, insightIds, names, dashboardId) {
            var requestParams = { projectKey: projectKey, insightIds: JSON.stringify(insightIds), names: JSON.stringify(names)};
            if (dashboardId) {
                requestParams.dashboardId = dashboardId
            }
            return APIXHRService("POST", API_PATH + "dashboards/insights/copy", requestParams);
        },
        makeListed: function(projectKey, insightIds, listed) {
            return APIXHRService("POST", API_PATH + "dashboards/insights/make-listed", { projectKey: projectKey, insightIds: JSON.stringify(insightIds), listed: listed });
        },
        getMissingReaderAuthorizations: function(projectKey, insightIds) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/get-missing-reader-authorizations", { projectKey: projectKey, insightIds: JSON.stringify(insightIds) });
        },
        viewStaticFile: function(projectKey, insightId) {
            return APIXHRService("GET", API_PATH + "dashboards/insights/view-static-file", {
                projectKey: projectKey,
                insightId: insightId
            })
        }
    }
},
webapps : {
    listTypes: function() {
        return APIXHRService("GET", API_PATH + "webapps/list-types", {});
    },
    save: function(webAppData, commitMessage, forceRestartBackend) {
        return APIXHRService("POST", API_PATH + "webapps/save", {
            webAppData: JSON.stringify(webAppData),
            commitMessage: commitMessage,
            forceRestartBackend: forceRestartBackend
        });
    },
    trust: function(projectKey, webAppId, trustForEverybody, fingerprint) {
        return APIXHRService("POST", API_PATH + "webapps/trust", {
            projectKey: projectKey,
            webAppId: webAppId,
            trustForEverybody: trustForEverybody,
            fingerprint: fingerprint
        });
    },
    saveMetadata: function(webAppData) {
        return APIXHRService("POST", API_PATH + "webapps/save-metadata", {
            webAppData: JSON.stringify(webAppData)
        });
    },
    create: function(projectKey, name, type, templateDesc, config, params) {
        return APIXHRService("POST", API_PATH + "webapps/create", { projectKey: projectKey, name: name, type: type, templateDesc: JSON.stringify(templateDesc), config: JSON.stringify(config), initialParams: (params == null ? null : JSON.stringify(params))});
    },
    copy: function(projectKey, webAppId, newWebAppName) {
        return APIXHRService("POST", API_PATH + "webapps/copy", {
            projectKey: projectKey,
            webAppId: webAppId,
            newWebAppName: newWebAppName
        });
    },
    listTemplates: function(type, language) {
        return APIXHRService("GET", API_PATH + "webapps/list-templates", { type: type });
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "webapps/list", { projectKey: projectKey });
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "webapps/list-heads", {projectKey: projectKey, tagFilter: tagFilter});
    },
    getFullInfo: function(projectKey, webAppId) {
        return APIXHRService("GET", API_PATH + "webapps/get-full-info", { projectKey: projectKey, webAppId: webAppId });
    },
    getPublicInfo: function(projectKey, webAppId) {
        return APIXHRService("GET", API_PATH + "webapps/get-public-info", { projectKey: projectKey, webAppId: webAppId });
    },
    getSummary: function(projectKey, webAppId) {
        return APIXHRService("GET", API_PATH + "webapps/get-summary", { projectKey: projectKey, webAppId: webAppId });
    },
    getBackendUrl: function(projectKey, webAppId, apiKey) {
        return APIXHRService("GET", API_PATH + "webapps/get-backend-url", {projectKey: projectKey, webAppId: webAppId, apiKey: apiKey});
    },
    getTrustedCodeReport: function(projectKey, webAppId) {
        return APIXHRService("GET", API_PATH + "webapps/get-trusted-code-report", {projectKey: projectKey, webAppId: webAppId});
    },
    getBackendState: function(webApp) {
        return APIXHRService("GET", API_PATH + "webapps/get-backend-state", {projectKey: webApp.projectKey, webAppId: webApp.id});
    },
    restartBackend: function(webApp) {
        return APIXHRService("POST", API_PATH + "webapps/restart-backend", {projectKey: webApp.projectKey, webAppId: webApp.id});
    },
    stopBackend: function(webApp) {
        return APIXHRService("POST", API_PATH + "webapps/stop-backend", {projectKey: webApp.projectKey, webAppId: webApp.id});
    },
    startTensorboard: function(projectKey, analysisId, taskId, sessionId ) {
        return APIXHRService("POST", API_PATH + "webapps/webapp-start-tensorboard", {projectKey: projectKey, analysisId: analysisId, taskId:taskId, sessionId: sessionId});
    },
    setDatasetPrivileges: function(projectKey, apiKeyId, data) {
        return APIXHRService("POST", API_PATH + "webapps/set-apikey-dataset-privileges", {
            projectKey: projectKey, keyId: apiKeyId,
            data: JSON.stringify(data)
        })
    },
    getDatasetPrivileges: function(projectKey, apiKeyId) {
        return APIXHRService("POST", API_PATH + "webapps/get-apikey-dataset-privileges", {projectKey: projectKey, keyId: apiKeyId})
    },
    listAllBackendsStates : function(){
        return APIXHRService("GET", API_PATH + "webapps/list-all-backends-states")
    },
    convertToCustom: function(projectKey, webappId, targetPluginId, newWebAppType, targetPluginMode) {
        return APIXHRService("GET", API_PATH + "webapps/convert-to-custom", {projectKey, webappId, targetPluginId, newWebAppType, targetPluginMode});
    },
    getOrCreatePluginSkin: function(projectKey, objectType, objectId, webAppType, webAppConfig, contextualCodeEnv, webAppId) {
        return APIXHRService("POST", API_PATH + "webapps/get-or-create-plugin-skin", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId,
            webAppType: webAppType,
            webAppConfig: JSON.stringify(webAppConfig),
            webAppId: webAppId,
            contextualCodeEnv: contextualCodeEnv,
        });
    },
    getBackendLogURL: function(projectKey, webAppId) {
        return API_PATH + "webapps/backend-log?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&webAppId=" + encodeURIComponent(webAppId);
    },
    getBackendDiagnosisURL: function(projectKey, webAppId) {
        return API_PATH + "webapps/backend-diagnosis?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&webAppId=" + encodeURIComponent(webAppId);
    },
},
webappsBackends : {
    get: function(projectKey, webAppId, path, spinner = true) {
         return  APIXHRService("GET", "/web-apps-backends"
                                         + "/" + encodeURIComponent(projectKey)
                                         + "/" + encodeURIComponent(webAppId)
                                         + "/" + path, null, spinner ? '' : "nospinner");
    },
},

codeStudioBlocks : {
    listTypes: function(codeStudioTemplateId) {
        return APIXHRService("GET", API_PATH + "code-studio-blocks/list-types", { codeStudioTemplateId: codeStudioTemplateId});
    }
},
codeStudioTemplates : {
    listTypes: function(all) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/list-types", all ? {all:true} : {});
    },
    save: function(codeStudioData, commitMessage) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/save", {
            codeStudioData: JSON.stringify(codeStudioData),
            commitMessage
        });
    },
    create: function(label, type) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/create", { label: label, type: type});
    },
    list: function() {
        return APIXHRService("GET", API_PATH + "code-studio-templates/list", { });
    },
    listHeads: function(tagFilter) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/list-heads", {tagFilter: tagFilter});
    },
    getFullInfo: function(codeStudioTemplateId) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/get-full-info", { codeStudioTemplateId: codeStudioTemplateId });
    },
    delete: function(codeStudioTemplateId, deleteRuntimes, deleteImages) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/delete", { codeStudioTemplateId: codeStudioTemplateId, deleteRuntimes: deleteRuntimes, deleteImages: deleteImages });
    },
    massDelete: function(codeStudioTemplateIds, deleteRuntimes, deleteImages) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/mass-delete", { codeStudioTemplateIds: JSON.stringify(codeStudioTemplateIds), deleteRuntimes: deleteRuntimes, deleteImages: deleteImages });
    },
    build: function(codeStudioTemplateId, withNoCache) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/build", { codeStudioTemplateId, withNoCache });
    },
    massBuild: function(codeStudioTemplateIds) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/mass-build", { codeStudioTemplateIds: JSON.stringify(codeStudioTemplateIds) });
    },
    massSetContainerConf: function(codeStudioTemplateIds, codeStudioTemplate) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/mass-set-container-conf", { codeStudioTemplateIds: JSON.stringify(codeStudioTemplateIds), codeStudioTemplate: JSON.stringify(codeStudioTemplate) });
    },
    massGetUsage: function(codeStudioTemplateIds, withCurrentUses) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/mass-get-usage", { codeStudioTemplateIds: JSON.stringify(codeStudioTemplateIds), withCurrentUses: !!withCurrentUses});
    },
    listLogs: function(codeStudioTemplateId) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/list-logs", { codeStudioTemplateId: codeStudioTemplateId });
    },
    getLogInfo: function(codeStudioTemplateId, buildId) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/get-log-info", { codeStudioTemplateId: codeStudioTemplateId, buildId: buildId });
    },
    getBuildDiagnosisURL: function(codeStudioTemplateId, buildId) {
        return API_PATH + "code-studio-templates/download-build-diagnosis?"
                 + "codeStudioTemplateId=" + encodeURIComponent(codeStudioTemplateId)
                 + "&buildId=" + encodeURIComponent(buildId);
    },
    getBuildLogURL: function(codeStudioTemplateId, buildId) {
        return API_PATH + "code-studio-templates/build-log?"
                 + "codeStudioTemplateId=" + encodeURIComponent(codeStudioTemplateId)
                 + "&buildId=" + encodeURIComponent(buildId);
    },
    getUsage: function(codeStudioTemplateId, withCurrentUses) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/get-usage", { codeStudioTemplateId: codeStudioTemplateId, withCurrentUses: !!withCurrentUses });
    },
    getExportURL: function(codeStudioTemplateId) {
        return API_PATH + "code-studio-templates/export?"
                 + "codeStudioTemplateId=" + encodeURIComponent(codeStudioTemplateId);
    },
    import: function(file, label, importSpec) {
        return uploadFileRequest("code-studio-templates/import", function(formdata) {
            formdata.append("file", file);
            formdata.append("label", label);
            formdata.append("importSpec", JSON.stringify(importSpec));
        }, null);
    },
    exportToPlugin: function(codeStudioTemplateId, pluginId, label, overwrite) {
        return APIXHRService("POST", API_PATH + "code-studio-templates/export-to-plugin", { codeStudioTemplateId: codeStudioTemplateId, pluginId: pluginId, label: label, overwrite: overwrite });
    },
    canEditInCodeStudio: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-studio-templates/can-edit-in-code-studio", { projectKey: projectKey });
    },
    resources: {
        listContents: function(codeStudioTemplateId) {
            return APIXHRService("GET", API_PATH + "code-studio-templates/resources/list-contents", {
                codeStudioTemplateId: codeStudioTemplateId
            });
        },
        getContent: function(codeStudioTemplateId, path, sendAnyway) {
            return APIXHRService("GET", API_PATH + "code-studio-templates/resources/get-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path, sendAnyway: sendAnyway
            });
        },
        setContent: function(codeStudioTemplateId, path, data) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/set-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path, data: data
            });
        },
        setContentMultiple: function(codeStudioTemplateId, contentMap) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/set-content-multiple", {
                codeStudioTemplateId: codeStudioTemplateId, contentMap: JSON.stringify(contentMap)
            });
        },
        createContent: function(codeStudioTemplateId, path, isFolder) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/create-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path, isFolder: isFolder
            });
        },
        deleteContent: function(codeStudioTemplateId, path) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/delete-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path
            });
        },
        decompressContent: function(codeStudioTemplateId, path) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/decompress-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path
            });
        },
        renameContent: function(codeStudioTemplateId, path, newName) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/rename-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path, newName: newName
            });
        },
        moveContent: function(codeStudioTemplateId, path, toPath) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/move-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path, toPath: toPath
            });
        },
        copyContent: function(codeStudioTemplateId, path) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/copy-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path
            });
        },
        uploadContent: function(codeStudioTemplateId, path, file, callback) {
            return uploadFileRequest("code-studio-templates/resources/upload-content", function(formdata) {
                formdata.append("codeStudioTemplateId", codeStudioTemplateId);
                formdata.append("path", path);
                formdata.append("file", file);
            }, callback);
        },
        checkUploadContent: function(codeStudioTemplateId, path, filePaths) {
            return APIXHRService("POST", API_PATH + "code-studio-templates/resources/check-upload-content", {
                codeStudioTemplateId: codeStudioTemplateId, path: path, filePaths: JSON.stringify(filePaths)
            });
        },
        downloadURL: function(codeStudioTemplateId, path) {
            return '/dip/api/code-studio-templates/resources/download-content?codeStudioTemplateId=' + codeStudioTemplateId + '&path=' + encodeURIComponent(path);
        }
    }
},

codeStudioObjects : {
    listTypes: function() {
        return APIXHRService("GET", API_PATH + "code-studio-objects/list-types", { });
    },
    save: function(codeStudioData, commitMessage) {
        return APIXHRService("POST", API_PATH + "code-studio-objects/save", {
            codeStudioData: JSON.stringify(codeStudioData),
            commitMessage: commitMessage
        });
    },
    saveMetadata: function(codeStudioData) {
        return APIXHRService("POST", API_PATH + "code-studio-objects/save-metadata", {
            codeStudioData: JSON.stringify(codeStudioData)
        });
    },
    create: function(projectKey, name, templateId) {
        return APIXHRService("POST", API_PATH + "code-studio-objects/create", { projectKey: projectKey, name: name, templateId: templateId});
    },
    copy: function(projectKey, codeStudioObjectId, newCodeStudioName) {
        return APIXHRService("POST", API_PATH + "code-studio-objects/copy", {
            projectKey: projectKey,
            codeStudioObjectId: codeStudioObjectId,
            newCodeStudioName: newCodeStudioName
        });
    },
    changeOwner: function(projectKey, codeStudioObjectId, newOwner) {
        return APIXHRService("POST", API_PATH + "code-studio-objects/change-owner", {
            projectKey,
            codeStudioObjectId,
            newOwner
        });
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/list", { projectKey: projectKey });
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/list-heads", {projectKey: projectKey, tagFilter: tagFilter});
    },
    getFullInfo: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/get-full-info", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    getSummary: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/get-summary", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    getState: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/get-state", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId});
    },
    openFile: function(projectKey, codeStudioObjectId, zone, fileToOpen) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/open-file", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId,
            zone: zone, fileToOpen: fileToOpen });
    },
    tryRestartServer: function(projectKey, codeStudioObjectId, exposedLabel) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/try-restart-server", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId,
            exposedLabel: exposedLabel});
    },
    stop: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/stop", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    restart: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/restart", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    issueCommand: function(projectKey, codeStudioObjectId, commandType, commandParams, behavior) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/issue-command", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, commandType:commandType, commandParams:JSON.stringify(commandParams), behavior:behavior });
    },
    getRunInfo: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/get-run-info", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    getLogURL: function(projectKey, codeStudioObjectId) {
        return API_PATH + "code-studio-objects/get-log?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&codeStudioObjectId=" + encodeURIComponent(codeStudioObjectId);
    },
    listContainerFiles: function(projectKey, codeStudioObjectId, path) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/list-container-files", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, path: path }, "nospinner");
    },
    previewContainerFile: function(projectKey, codeStudioObjectId, path) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/preview-container-file", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, path: path });
    },
    getCurrentUsage: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/get-current-usage", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    pullBundle: function(projectKey, codeStudioObjectId, zone, force) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/pull/" + zone, { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, force: force, async: true });
    },
    checkConflicts: function(projectKey, codeStudioObjectId, zones) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/check-conflicts", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, zones: JSON.stringify(zones) });
    },
    getChanges: function(projectKey, codeStudioObjectId) {
        return APIXHRService("GET", API_PATH + "code-studio-objects/get-changes", { projectKey: projectKey, codeStudioObjectId: codeStudioObjectId });
    },
    getDiagnosisURL: function(projectKey, codeStudioObjectId) {
        return API_PATH + "code-studio-objects/download-diagnosis?"
                 + "codeStudioObjectId=" + encodeURIComponent(codeStudioObjectId)
                 + "&projectKey=" + encodeURIComponent(projectKey);
    },
    folderEdit: {
        listContents: function(projectKey, codeStudioObjectId, type) {
            return APIXHRService("GET", API_PATH + "code-studio-objects/folder-edition/list-contents", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type
            });
        },
        getContent: function(projectKey, codeStudioObjectId, type, path, sendAnyway) {
            return APIXHRService("GET", API_PATH + "code-studio-objects/folder-edition/get-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path, sendAnyway: sendAnyway
            });
        },
        setContent: function(projectKey, codeStudioObjectId, type, path, data) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/set-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path, data: data
            });
        },
        setContentMultiple: function(projectKey, codeStudioObjectId, type, contentMap) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/set-content-multiple", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, contentMap: JSON.stringify(contentMap)
            });
        },
        createContent: function(projectKey, codeStudioObjectId, type, path, isFolder) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/create-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path, isFolder: isFolder
            });
        },
        deleteContent: function(projectKey, codeStudioObjectId, type, path) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/delete-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path
            });
        },
        decompressContent: function(projectKey, codeStudioObjectId, type, path) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/decompress-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path
            });
        },
        renameContent: function(projectKey, codeStudioObjectId, type, path, newName) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/rename-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path, newName: newName
            });
        },
        moveContent: function(projectKey, codeStudioObjectId, type, path, toPath) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/move-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path, toPath: toPath
            });
        },
        copyContent: function(projectKey, codeStudioObjectId, type, path) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/copy-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path
            });
        },
        uploadContent: function(projectKey, codeStudioObjectId, type, path, file, callback) {
            return uploadFileRequest("code-studio-objects/folder-edition/upload-content", function(formdata) {
                formdata.append("projectKey", projectKey);
                formdata.append("codeStudioObjectId", codeStudioObjectId);
                formdata.append("type", type);
                formdata.append("path", path);
                formdata.append("file", file);
            }, callback);
        },
        checkUploadContent: function(projectKey, codeStudioObjectId, type, path, filePaths) {
            return APIXHRService("POST", API_PATH + "code-studio-objects/folder-edition/check-upload-content", {
                projectKey: projectKey, codeStudioObjectId: codeStudioObjectId, type: type, path: path, filePaths: JSON.stringify(filePaths)
            });
        },
        downloadURL: function(projectKey, codeStudioObjectId, type, path) {
            return '/dip/api/code-studio-objects/folder-edition/download-content?projectKey=' + projectKey + '&codeStudioObjectId=' + codeStudioObjectId + '&type=' + type + '&path=' + encodeURIComponent(path);
        }
    }
},

reports : {
    create: function(projectKey, name, templateDesc) {
        return APIXHRService("POST", API_PATH + "reports/create", {
            projectKey: projectKey,
            name: name,
            templateDesc: JSON.stringify(templateDesc)
        });
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "reports/list", { projectKey: projectKey });
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "reports/list-heads", {
            projectKey: projectKey,
            tagFilter: tagFilter
        });
    },
    listTemplates: function(type) {
        return APIXHRService("GET", API_PATH + "reports/list-templates", { type: type });
    },
    getSummary: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "reports/get-summary", {
            projectKey: projectKey,
            id: id
        });
    },
    getFullInfo: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "reports/get-full-info", {
            projectKey: projectKey,
            id: id
        });
    },
    save: function(report, script, commitMessage) {
        return APIXHRService("POST", API_PATH + "reports/save", {
            reportData: JSON.stringify(report),
            script: script,
            commitMessage: commitMessage
        });
    },
    saveMetadata: function(report) {
        return APIXHRService("POST", API_PATH + "reports/save-metadata", {
            reportData: JSON.stringify(report)
        });
    },
    copy: function(projectKey, id, newName) {
        return APIXHRService("POST", API_PATH + "reports/copy", {
            projectKey: projectKey,
            id: id,
            newName: newName
        });
    },
    build: function(projectKey, id) {
        return APIXHRService("POST", API_PATH + "reports/build", {
            projectKey: projectKey,
            id: id
        });
    },
    prepareDownload: function(projectKey, id, format) {
        return APIXHRService("POST", API_PATH + "reports/prepare-download", {
            projectKey: projectKey,
            id: id,
            format: format
        });
    },
    getDownloadReportURL: function(projectKey, id, format) {
        return API_PATH + "reports/download?projectKey=" + encodeURIComponent(projectKey) + "&id=" + encodeURIComponent(id) + "&format=" + encodeURIComponent(format) ;
    },
    view: function(projectKey, reportId, inEditMode) {
        return APIXHRService("GET", API_PATH + "reports/view", {
            projectKey: projectKey,
            id: reportId,
            inEditMode: inEditMode
        });
    },
    snapshots: { // Should mostly stay in sync with jupyterNotebooks.export
        create: function(projectKey, reportId) {
            return APIXHRService("POST", API_PATH + "reports/snapshots/create", {
                projectKey: projectKey,
                reportId: reportId
            });
        },
        list: function(projectKey, reportId) {
            return APIXHRService("GET", API_PATH + "reports/snapshots/list", {
                projectKey: projectKey,
                reportId: reportId
            });
        },
        view: function(projectKey, reportId, timestamp, format) {
            return APIXHRService("GET", API_PATH + "reports/snapshots/view", {
                projectKey: projectKey,
                id: reportId,
                format: format,
                timestamp: timestamp
            });
        },
        listForAll: function(projectKey) {
            return APIXHRService("GET", API_PATH + "reports/snapshots/list-for-all", {
                projectKey: projectKey
            });
        },
        get: function(projectKey, id, timestamp) {
            return APIXHRService("GET", API_PATH + "reports/snapshots/get", {
                projectKey: projectKey,
                id: id,
                timestamp: timestamp
            });
        }
    }
},
runnables: {
    manualRun: function(projectKey, runnableType, params, adminParams) {
        return APIXHRService("POST", API_PATH + "runnables/manual-run", {
            projectKey: projectKey, runnableType: runnableType, params: JSON.stringify(params), adminParams: JSON.stringify(adminParams)
        });
    },
    clusterRun: function(clusterId, runnableType, params, adminParams) {
        return APIXHRService("POST", API_PATH + "runnables/cluster-run", {
            clusterId: clusterId, runnableType: runnableType, params: JSON.stringify(params), adminParams: JSON.stringify(adminParams)
        });
    },
    projectCreationRun: function(runnableType, params, projectFolderId) {
        return APIXHRService("POST", API_PATH + "runnables/project-creation-run", {
            runnableType: runnableType, params: JSON.stringify(params), projectFolderId: projectFolderId
        });
    },
    insightRun: function(projectKey, insightId) {
        return APIXHRService("GET", API_PATH + "runnables/insight-run", {
            projectKey: projectKey, insightId: insightId
        });
    },
    loadKeptFile: function(projectKey, runnableType, item, clusterId) {
        return APIXHRService("GET", API_PATH + "runnables/load-kept-file", {
            projectKey: projectKey,
            clusterId: clusterId,
            runnableType: runnableType,
            item: JSON.stringify(item)
        });
    },
    getDownloadURL: function(projectKey, runnableType, item, clusterId) {
        return API_PATH + "runnables/download-kept-file?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&clusterId=" + encodeURIComponent(clusterId)
                 + "&runnableType=" + encodeURIComponent(runnableType)
                 + "&item=" + encodeURIComponent(JSON.stringify(item));
    },
    listAccessible: function(projectKey) {
        return APIXHRService("GET", API_PATH + "runnables/list-accessible", {
            projectKey: projectKey
        });
    },
    inlinePythonRun: function(projectKey, desc, envSelection, code) {
        return APIXHRService("POST", API_PATH + "runnables/inline-python-run", {
            projectKey, desc : JSON.stringify(desc), envSelection: JSON.stringify(envSelection), code
        })
    }
},
scenarios: {
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "scenarios/list", {
            projectKey: projectKey
        });
    },
    listAccessible: function(projectKey) {
        return APIXHRService("GET", API_PATH + "scenarios/list-accessible");
    },
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "scenarios/list-heads", {
            projectKey: projectKey
        });
    },
    listAllHeads: function() {
        return APIXHRService("GET", API_PATH + "scenarios/list-all-heads");
    },
    listAllReporters: function() {
        return APIXHRService("GET", API_PATH + "scenarios/list-all-reporters");
    },
    get: function(projectKey, scenarioId) {
        return APIXHRService("GET", API_PATH + "scenarios/get", {
            projectKey: projectKey,
            scenarioId: scenarioId
        });
    },
    getSummary: function(projectKey, scenarioId) {
        return APIXHRService("GET", API_PATH + "scenarios/get-summary", {
            projectKey: projectKey,
            scenarioId: scenarioId
        });
    },
    getScript: function(projectKey, scenarioId) {
        return APIXHRService("GET", API_PATH + "scenarios/get-script", {
            projectKey: projectKey,
            scenarioId: scenarioId
        });
    },
    manualRun: function(projectKey, smartScenarioId, params, waitForStart, waitForCompletion) {
        return APIXHRService("POST", API_PATH + "scenarios/manual-run", {
            projectKey: projectKey,
            smartScenarioId: smartScenarioId,
            params: JSON.stringify(params),
            waitForStart: waitForStart,
            waitForCompletion: waitForCompletion
        });
    },
    create: function(projectKey, data) {
        return APIXHRService("GET", API_PATH + "scenarios/create", {
            projectKey: projectKey,
            data: JSON.stringify(data)
        });
    },
    duplicate: function(projectKeyFrom,projectKeyTo,idFrom,idTo, name) {
        return APIXHRService("POST", API_PATH + "scenarios/duplicate", {
            projectKeyFrom: projectKeyFrom,
            projectKeyTo: projectKeyTo,
            idFrom: idFrom,
            idTo: idTo,
            name: name
        });
    },
    rename: function(projectKey, scenarioId, newName) {
        return APIXHRService("POST", API_PATH + "scenarios/rename", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            newName: newName
        });
    },
    save: function(projectKey, data, scriptData, saveInfo) {
        return APIXHRService("POST", API_PATH + "scenarios/save", {
            projectKey: projectKey,
            data: JSON.stringify(data),
            scriptData: scriptData,
            saveInfo: JSON.stringify(saveInfo)
        });
    },
    saveNoParams: function(projectKey, data, saveInfo, fieldsToSave = ["active"]) {
        return APIXHRService("POST", API_PATH + "scenarios/save-no-params", {
            projectKey: projectKey,
            data: JSON.stringify(data),
            fieldsToSave: JSON.stringify(fieldsToSave),
            saveInfo: JSON.stringify(saveInfo)
        });
    },
    saveReporterState: function(projectKey, scenarioId, data, saveInfo) {
        return APIXHRService("POST", API_PATH + "scenarios/save-reporter-state", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            data: JSON.stringify(data),
            saveInfo: JSON.stringify(saveInfo)
        });
    },
    deleteReporter: function(projectKey, scenarioId, data, saveInfo) {
        return APIXHRService("POST", API_PATH + "scenarios/delete-reporter", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            data: JSON.stringify(data),
            saveInfo: JSON.stringify(saveInfo)
        });
    },
    getLastScenarioRuns: function(projectKey, smartScenarioId, withFullScenario, limit) {
        return APIXHRService("GET", API_PATH + "scenarios/get-last-scenario-runs", {
            projectKey: projectKey,
            smartScenarioId: smartScenarioId,
            withFullScenario: withFullScenario,
            limit: limit
        });
    },
    getScenarioRunDetails: function(projectKey, scenarioId, runId) {
        return APIXHRService("GET", API_PATH + "scenarios/get-scenario-run-details", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            runId: runId
        });
    },
    getLastTriggerRuns: function(projectKey, scenarioId) {
        return APIXHRService("GET", API_PATH + "scenarios/get-last-trigger-runs", {
            projectKey: projectKey,
            scenarioId: scenarioId
        });
    },
    getScenarioReport: function(projectKey, scenarioId, fromDate, toDate) {
        return APIXHRService("GET", API_PATH + "scenarios/get-scenario-report", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            fromDate: fromDate,
            toDate: toDate
        });
    },
    getProjetReport: function(projectKey, fromDate, toDate) {
        return APIXHRService("GET", API_PATH + "scenarios/get-project-report", {
            projectKey: projectKey,
            fromDate: fromDate,
            toDate: toDate
        });
    },
    getInstanceReport: function(fromDate, toDate) {
        return APIXHRService("GET", API_PATH + "scenarios/get-instance-report", {
            fromDate: fromDate,
            toDate: toDate
        });
    },
    getProjetActivities: function(projectKey, fromDate, toDate) {
        return APIXHRService("GET", API_PATH + "scenarios/get-project-activities", {
            projectKey: projectKey,
            fromDate: fromDate,
            toDate: toDate
        });
    },
    getOutcomes: function(fromDate, toDate, projectKey, scenarioId) {
        return APIXHRService("GET", API_PATH + "scenarios/get-outcomes", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            fromDate: fromDate,
            toDate: toDate
        }, 'nospinner');
    },
    getOutcomesSummary: function(projectKey, fromDate, toDate) {
        return APIXHRService("GET", API_PATH + "scenarios/get-outcomes-summary", {
            projectKey: projectKey,
            fromDate: fromDate,
            toDate: toDate
        }, 'nospinner');
    },
    getOutcomesDetails: function(projectKey, scenarioId, date) {
        return APIXHRService("GET", API_PATH + "scenarios/get-outcomes-details", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            date: date
        });
    },
    getRunDiagnosisURL: function(projectKey, scenarioId, runId) {
        return API_PATH + "scenarios/download-run-diagnosis?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&scenarioId=" + encodeURIComponent(scenarioId)
                 + "&runId=" + encodeURIComponent(runId);
    },
    getRunLogURL: function(projectKey, scenarioId, runId) {
        return API_PATH + "scenarios/run-log?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&scenarioId=" + encodeURIComponent(scenarioId)
                 + "&runId=" + encodeURIComponent(runId);
    },
    getStepRunLogURL: function(projectKey, scenarioId, runId, stepRunId) {
        return API_PATH + "scenarios/step-run-log?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&scenarioId=" + encodeURIComponent(scenarioId)
                 + "&runId=" + encodeURIComponent(runId)
                 + "&stepRunId=" + encodeURIComponent(stepRunId);
    },
    listReportTemplates: function() {
        return APIXHRService("GET", API_PATH + "scenarios/list-report-templates", {
        });
    },
    loadKeptFile: function(projectKey, scenarioId, runId, stepName, item) {
        return APIXHRService("GET", API_PATH + "scenarios/load-kept-file", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            runId: runId,
            stepName: stepName,
            item: JSON.stringify(item)
        });
    },
    getDownloadURL: function(projectKey, scenarioId, runId, stepName, item) {
        return API_PATH + "scenarios/download-kept-file?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&scenarioId=" + encodeURIComponent(scenarioId)
                 + "&runId=" + encodeURIComponent(runId)
                 + "&stepName=" + encodeURIComponent(stepName)
                 + "&item=" + encodeURIComponent(JSON.stringify(item));
    },
    addToScenario: function(items, options) {
        return APIXHRService("POST", API_PATH + "scenarios/add-to-scenario", {
            items: JSON.stringify(items),
            options: JSON.stringify(options)
        });
    },
    schedule: function(projectKey, data) {
        return APIXHRService("POST", API_PATH + "scenarios/schedule", {
            projectKey,
            data: JSON.stringify(data)
        });
    },
},
lambda: {
    services: {
        list: function(projectKey) {
            return APIXHRService("GET", API_PATH + "lambda-services/list", {
                projectKey: projectKey
            });
        },
        listHeads: function(projectKey) {
            return APIXHRService("GET", API_PATH + "lambda-services/list-heads", {
              projectKey
            });
        },
        getSummary: function(projectKey, id) {
            return APIXHRService("GET", API_PATH + "lambda-services/get-summary", {
                projectKey: projectKey,
                id: id
            });
        },
        deleteMulti: function(requests, contextProjectKey) {
            return APIXHRService("POST", API_PATH + "lambda-services/delete-multi/", {
                requests: JSON.stringify(requests),
                contextProjectKey: contextProjectKey
            });
        },
        create: function(projectKey, id) {
            return APIXHRService("POST", API_PATH + "lambda-services/create", {
                projectKey: projectKey,
                id: id
            });
        },
        save: function(projectKey, service) {
            const sanitized_service = {
                ...service,
                endpoints: service.endpoints?.map(endpoint => ({
                ...endpoint,
                testQueries: endpoint.testQueries?.filter(testQuery => testQuery.q !== null)
                })) ?? []
            };
            return APIXHRService("POST", API_PATH + "lambda-services/save", {
                projectKey: projectKey,
                service: JSON.stringify(sanitized_service)
            });
        },
        addEndpoint: function(projectKey, serviceId, createService, endpoint) {
            return APIXHRService("POST", API_PATH + "lambda-services/add-endpoint", {
                projectKey: projectKey,
                serviceId: serviceId,
                createService: createService,
                endpoint: angular.toJson(endpoint)
            });
        },
        startPreparePackage: function(projectKey, serviceId, packageId, releaseNotes) {
            return APIXHRService("POST", API_PATH + "lambda-services/packages/start-prepare", {
                projectKey: projectKey,
                serviceId: serviceId,
                packageId: packageId,
                releaseNotes: releaseNotes
            });
        },
        deployDev: function(projectKey, id) {
            return APIXHRService("POST", API_PATH + "lambda-services/deploy-to-dev", {
                projectKey: projectKey,
                id: id
            });
        },
        playTestQueries: function(projectKey, serviceId, endpointId, testType, queries) {
            return APIXHRService("POST", API_PATH + "lambda-services/play-test-queries", {
                projectKey: projectKey,
                serviceId: serviceId,
                endpointId: endpointId,
                testType: testType,
                queries: JSON.stringify(queries)
            });
        },
        getSampleQueriesFromDataset: function(projectKey, queriesName, datasetName, modelRef, batchSize=1, method="HEAD_SEQUENTIAL", shouldIncludeNulls) {
            return APIXHRService("GET", API_PATH + "lambda-services/get-sample-queries", {
                projectKey: projectKey,
                queriesName: queriesName,
                datasetName: datasetName,
                modelRef: modelRef,
                batchSize: batchSize,
                method: method,
                shouldIncludeNulls: shouldIncludeNulls
            });
        },
        generateEndpointOpenAPI: function(projectKey, endpoint, enabled, isManual) {
            return APIXHRService("POST", API_PATH + "lambda-services/generate-endpoint-open-api", {
                projectKey: projectKey,
                endpoint: angular.toJson(endpoint),
                enabled: enabled,
                isManual: isManual
            });
        },
        monitoring:{
            create: function(projectKey, deploymentId, endpointId) {
                return APIXHRService("POST", API_PATH + "lambda-services/monitoring/create", {
                    projectKey: projectKey,
                    deploymentId: deploymentId,
                    endpointId: endpointId,
                });
            },
            delete: function(projectKey, deploymentId, endpointId, deleteObjects) {
                return APIXHRService("POST", API_PATH + "lambda-services/monitoring/delete", {
                    projectKey: projectKey,
                    deploymentId: deploymentId,
                    endpointId: endpointId,
                    deleteObjects: deleteObjects
                });
            },
            list: function(projectKey, deploymentIds, endpointId) {
                return APIXHRService("GET", API_PATH + "lambda-services/monitoring/list", {
                    projectKey: projectKey,
                    deploymentIds: deploymentIds,
                    endpointId: endpointId
                });
            }
        }
    },
    packages: {
        list: function(projectKey, serviceId) {
            return APIXHRService("GET", API_PATH + "lambda-services/packages", {
                projectKey: projectKey,
                serviceId: serviceId
            });
        },
        delete: function(projectKey, serviceId, packageId) {
            return APIXHRService("POST", API_PATH + "lambda-services/package/delete", {
                projectKey: projectKey,
                serviceId: serviceId,
                packageId: packageId
            });
        },
        publishToAPIDeployer: function(projectKey, serviceId, packageId, publishedServiceId) {
            return APIXHRService("POST", API_PATH + "lambda-services/package/publish-to-api-deployer", {
                projectKey: projectKey,
                serviceId: serviceId,
                packageId: packageId,
                publishedServiceId: publishedServiceId
            });
        },
        getSummary: function(projectKey, serviceId, packageId,) {
            return APIXHRService("GET", API_PATH + "lambda-services/package/get-summary", {
                projectKey: projectKey,
                serviceId: serviceId,
                packageId: packageId
            });
        }
    },
    devServer: {
        getStatus: function(projectKey) {
            return APIXHRService("GET", API_PATH + "lambda-devserver", {
                projectKey: projectKey,
            });
        },
        stopDevServer: function(projectKey) {
            return APIXHRService("DELETE", API_PATH + "lambda-devserver?projectKey=" + encodeURIComponent(projectKey));
        }
    }
},
exports: {
    list: function() {
        return APIXHRService("GET", API_PATH + "exports/list", {},"nospinner");
    },
    remove: function(exportId) {
        return APIXHRService("POST", API_PATH + "exports/remove", {exportId: exportId});
    },
    getDownloadURL: function(exportId) {
        return API_PATH + "exports/download/?exportId=" + encodeURIComponent(exportId);
    },
    clear: function() {
        return APIXHRService("POST", API_PATH + "exports/clear");
    },
    create: function(name, params) {
        return APIXHRService("POST", API_PATH + "exports/create", {name: name, params: JSON.stringify(params)});
    },
    getOptions: function() {
        return APIXHRService("GET", API_PATH + "exports/get-export-options");
    },
    exportUIData: function(data, params) {
        return APIXHRService("POST", API_PATH + "exports/export-ui-data", {
            data: JSON.stringify(data),
            params: JSON.stringify(params)
        });
    }
},
promptStudios: {
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "prompt-studios/list-heads", { projectKey })
    },
    get: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "prompt-studios/get", {
            projectKey, id
        })
    },
    getFullInfo: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "prompt-studios/get-full-info", { projectKey, id });
    },
    save: function(promptStudio) {
        return APIXHRService("POST", API_PATH + "prompt-studios/save", {
            promptStudio: JSON.stringify(promptStudio)
        })
    },
    create: function(projectKey, name) {
        return APIXHRService("POST", API_PATH + "prompt-studios/create", {
            projectKey, name
        })
    },
    getPromptLastResponse: function(projectKey, promptStudioId, promptId){
        return APIXHRService("GET", API_PATH + "prompt-studios/get-prompt-last-response",{projectKey, promptStudioId, promptId});
    },
    copy: function(projectKey, promptStudioId) {
        return APIXHRService("POST", API_PATH + "prompt-studios/copy", {
            projectKey, promptStudioId
        })
    },
    getPromptHistory: function(projectKey, promptStudioId, promptId){
        return APIXHRService("GET", API_PATH + "prompt-studios/get-prompt-history",{projectKey, promptStudioId, promptId})
    },
    getPromptHistoryResponse: function(projectKey, promptStudioId, promptId, promptRunId){
        return APIXHRService("GET", API_PATH + "prompt-studios/get-prompt-history-response",{projectKey, promptStudioId, promptId, promptRunId})
    },
    clearLastResponse: function(projectKey, promptStudioId, promptId) {
        return APIXHRService("POST", API_PATH + "prompt-studios/clear-last-response", { projectKey, promptStudioId, promptId })
    },
    revertPrompt: function(projectKey, promptStudio, promptId, promptRunId){
        return APIXHRService("POST", API_PATH + "prompt-studios/revert-prompt", {
            promptStudio: JSON.stringify(promptStudio), projectKey, promptId, promptRunId
        });
    },
    startExecutePrompt: function(promptStudio, prompt) {
        return APIXHRService("POST", API_PATH + "prompt-studios/start-execute-prompt", {
            promptStudio: JSON.stringify(promptStudio),
            prompt: JSON.stringify(prompt)
        })
    },
    startExecuteComparison: function(promptStudio, promptIds) {
        return APIXHRService("POST", API_PATH + "prompt-studios/start-execute-comparison", {
            promptStudio: JSON.stringify(promptStudio),
            promptIds: JSON.stringify(promptIds)
        })
    },
    editFromRecipe: function(projectKey, recipeName, promptCreationSettings) {
        return APIXHRService("POST", API_PATH + "prompt-studios/edit-from-recipe", {
            promptCreationSettings: JSON.stringify(promptCreationSettings),
            recipeName,
            projectKey,
        })
    },
    createFromSM: function(projectKey, smRef, promptCreationSettings) {
        return APIXHRService('POST', API_PATH + 'prompt-studios/create-from-sm', {
            projectKey,
            smRef,
            promptCreationSettings: JSON.stringify(promptCreationSettings),
        });
    },
    getRecipePayloadFromPrompt: function(projectKey, promptStudioId, promptId) {
        return APIXHRService("GET", API_PATH + "prompt-studios/get-recipe-payload-from-prompt", {
            promptStudioPromptId: promptId, promptStudioId, projectKey,
        });
    },
    overwriteRecipe: function(projectKey, recipeName, promptStudioId, promptId) {
        return APIXHRService("POST", API_PATH + "prompt-studios/overwrite-recipe", {
            promptStudioPromptId: promptId, promptStudioId, recipeName, projectKey,
        });
    },
    getPromptDatasetPreview: function(promptStudio, prompt) {
        return APIXHRService("POST", API_PATH + "prompt-studios/get-prompt-dataset-preview", {
            promptStudio: JSON.stringify(promptStudio),
            prompt: JSON.stringify(prompt)
        })
    },
    streamedCompletion: function(promptStudio, prompt, chunkCallback, abortController) {
        return streamRequest('POST', API_PATH + 'prompt-studios/streamed-completion', {
            promptStudio: JSON.stringify(promptStudio),
            prompt: JSON.stringify(prompt)
        }, readStreamEventData(chunkCallback), abortController);
    },
    forkSession: function(promptStudio, prompt, sourcePromptId, sourceUserMessageId) {
        return APIXHRService("POST", API_PATH + "prompt-studios/fork-chat-session", {
            promptStudio: JSON.stringify(promptStudio),
            prompt: JSON.stringify(prompt),
            sourcePromptId,
            sourceUserMessageId
        });
    }
},

agentTools: {
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "agent-tools/list", { projectKey })
    },
    listHeads: function(projectKey, withForeign) {
        return APIXHRService("GET", API_PATH + "agent-tools/list-heads", { projectKey: projectKey, withForeign: withForeign })
    },
    listAvailable: function(projectKey) {
        return APIXHRService("GET", API_PATH + "agent-tools/list-available", { projectKey })
    },
    get: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "agent-tools/get", {
            projectKey, id
        })
    },
    getDescriptor: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "agent-tools/get-descriptor", {
            projectKey, id
        })
    },
    getFullInfo: function(projectKey, agentToolRef) {
        return APIXHRService("GET", API_PATH + "agent-tools/get-full-info", { projectKey, agentToolRef });
    },
    getUsage: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "agent-tools/get-usage", { projectKey, id });
    },
    test: function(projectKey, id, query, timeoutPromise) {
        return APIXHRService("POST", API_PATH + "agent-tools/test", {
            projectKey, id, query: query
        }, null, null, timeoutPromise);
    },
    stopDevKernel: function(projectKey, id) {
        return APIXHRService("POST", API_PATH + "agent-tools/stop-dev-kernel", {
            projectKey, id
        })
    },
    hasSampleQuery: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "agent-tools/has-sample-query", {
            projectKey, id
        })
    },
    loadSampleQuery: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "agent-tools/load-sample-query", {
            projectKey, id
        })
    },
    save: function(agentTool) {
        return APIXHRService("POST", API_PATH + "agent-tools/save", {
            agentTool: JSON.stringify(agentTool)
        })
    },
    create: function(projectKey, proto) {
        return APIXHRService("POST", API_PATH + "agent-tools/create", {
            projectKey: projectKey, proto: JSON.stringify(proto)
        })
    },
    createFromKB: function(projectKey, proto, knowledgeBankRef) {
        return APIXHRService("POST", API_PATH + "agent-tools/create-from-kb", {
            projectKey: projectKey, proto: JSON.stringify(proto), knowledgeBankRef: knowledgeBankRef
        })
    },
    checkSaveConflict: function(agentTool) {
        return APIXHRService("POST", API_PATH + "agent-tools/check-save-conflict", {
            agentTool: JSON.stringify(agentTool)
        })
    },
    copy: function(projectKey, agentToolId) {
        return APIXHRService("POST", API_PATH + "agent-tools/copy", {
            projectKey, agentToolId
        })
    },
    listLogs: function(projectKey, id) {
        return APIXHRService("POST", API_PATH + "agent-tools/list-logs", {
            projectKey, id
        });
    },
    getLog: function(projectKey, id, logName){
        return APIXHRService("POST", API_PATH + "agent-tools/get-log", {
            projectKey, id, logName
        });
    },
    deleteLog: function(projectKey, id, logName) {
        return APIXHRService("POST", API_PATH + "agent-tools/delete-log", {
            projectKey, id, logName
        });
    },
    clearLogs: function(projectKey, id) {
        return APIXHRService("POST", API_PATH + "agent-tools/clear-logs", {
            projectKey, id
        });
    },
    listEmailTemplates: function() {
        return APIXHRService("GET", API_PATH + "agent-tools/list-email-templates", {
        });
    },
    getEmailTemplateContent: function(templateName) {
        return APIXHRService("GET", API_PATH + "agent-tools/get-email-template-content", {
            templateName
        });
    },
    canCreateLocalMCPTool: function() {
        return APIXHRService("GET", API_PATH + "agent-tools/can-create-local-mcp-tool");
    },
},

typeBadges: {
    syncProject: function(projectKey) {
        return APIXHRService("POST", API_PATH + "type-badges/projects/sync", {
            projectKey: projectKey
        });
    },
    listApiServices: function(projectKey) {
        return APIXHRService("GET", API_PATH + "type-badges/api-services/list", {
            projectKey: projectKey
        });
    },
    syncApiServices: function(projectKey) {
        return APIXHRService("POST", API_PATH + "type-badges/api-services/sync", {
            projectKey: projectKey
        });
    }
},

codeAssistant: {
    listAvailableLLMs: function () {
        return APIXHRService("GET", API_PATH + "code-assistant/list-available-llms", {})
    }
},
notebooks: {
    listTemplates: function(type, language) {
        return APIXHRService("GET", API_PATH + "notebooks/list-templates", {
            type: type, language: language
        });
    }
},
searchNotebooks: {
    create: function(projectKey, connection, name) {
        return APIXHRService("POST", API_PATH + "search-notebooks/create/", {projectKey: projectKey, connection: connection, name: name});
    },
    copy: function(projectKey, notebookId, newNotebookName) {
        return APIXHRService("POST", API_PATH + "search-notebooks/copy/", {projectKey: projectKey, notebookId: notebookId, newNotebookName: newNotebookName});
    },
    listConnections: function(projectKey) {
        return APIXHRService("GET", API_PATH + "search-notebooks/list-connections/");
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "search-notebooks/list-heads/", {projectKey: projectKey, tagFilter: tagFilter});
    },
    getSummary: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "search-notebooks/get-summary/", {projectKey: projectKey, id: id});
    },
    save: function(notebook) {
        return APIXHRService("POST", API_PATH + "search-notebooks/save/", {notebook: angular.toJson(notebook)}, "nospinner");
    },
    inferSchema: function(projectKey, connection, index) {
        return APIXHRService("POST", API_PATH + "search-notebooks/infer-schema/", {projectKey: projectKey, connection: connection, index: index});
    },
    search: function(projectKey, connection, index, schema, query, from, maxRows, batchSize) {
        const size = Math.min(batchSize, maxRows - from);
        query = angular.copy(query);
        query.size = size;
        query.from = from;

        return APIXHRService("POST", API_PATH + "search-notebooks/interactive-es-search/", {
            projectKey, connection, index,
            schema: JSON.stringify(schema),
            searchQuery: JSON.stringify(query)
        });
    },
    listUsableDatasets: function(projectKey, preSelectedDatasets, connection, loadAll) {
        return APIXHRService("POST", API_PATH + "search-notebooks/list-usable-datasets",  {projectKey:projectKey, preSelectedDatasets: JSON.stringify(preSelectedDatasets), connection: connection, loadAll: loadAll});
    },
    resolveDatasets: function(projectKey, datasetNames) {
        return APIXHRService("POST", API_PATH + "search-notebooks/resolve-datasets",  {projectKey:projectKey, datasetNames: JSON.stringify(datasetNames)});
    },
},
sqlNotebooks: {
    abort: function(projectKey, notebookId, cellId, qid) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/abort", {projectKey: projectKey, notebookId: notebookId, cellId: cellId, qid: qid});
    },
    clearHistory: function(projectKey, notebookId, cellId) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/clear-history/", {
            projectKey: projectKey,
            notebookId: notebookId,
            cellId: cellId
        }, "nospinner");
    },
    computeFullCount: function(projectKey, notebookId, cellId, qid) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/full-count/", {
            projectKey: projectKey,
            notebookId: notebookId,
            cellId: cellId,
            qid: qid
        }, "nospinner");
    },
    copy: function(projectKey, notebookId, newNotebookName) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/copy/", {projectKey: projectKey, notebookId: notebookId, newNotebookName: newNotebookName});
    },
    create: function(projectKey, connection, name) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/create/", {projectKey: projectKey, connection: connection, name: name});
    },
    exportResults: function(projectKey, notebookId, cellId, queryId, params) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/export-results/", {
            projectKey: projectKey,
            notebookId: notebookId,
            cellId: cellId,
            qid: queryId,
            params: JSON.stringify(params)
        });
    },
    get: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get/", {projectKey: projectKey, id: id});
    },
    getPivotResponse: function(request, projectKey, notebookId, qid) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-pivot-response/", {
            projectKey: projectKey,
            notebookId : notebookId,
            qid: qid,
            request: JSON.stringify(request)
        });
    },
    getColumnSummary: function(projectKey, notebookId, qid) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-columns-summary/", {projectKey: projectKey, notebookId: notebookId, qid: qid});
    },
    fixExpression: function(projectKey, notebookId, qid, expression, fixName) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/fix-expression", {
            projectKey, notebookId, qid, expression, fixName});
    },
    sampleRefreshTable: function(projectKey, notebookId, qid, displayScript, filters) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/refresh-table", {
            projectKey: projectKey,
            notebookId: notebookId,
            qid: qid,
            displayScript: JSON.stringify(displayScript),
            filters: JSON.stringify(filters),
        })
    },
    sampleGetTableChunk: function(projectKey, notebookId, qid, displayScript, firstRow, nbRows, firstCol, nbCols, filters) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/get-table-chunk", {
            projectKey: projectKey,
            notebookId: notebookId,
            qid: qid,
            displayScript: JSON.stringify(displayScript),
            filters: JSON.stringify(filters),
            firstRow: firstRow,
            nbRows: nbRows,
            firstCol: firstCol,
            nbCols: nbCols,
        })
    },
    sampleDetailedColumnAnalysis: function(projectKey, notebookId, qid, data, column, alphanumMaxResults) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/detailed-column-analysis", {
            projectKey: projectKey,
            notebookId: notebookId,
            qid: qid,
            data: JSON.stringify(data),
            column: column,
            alphanumMaxResults: alphanumMaxResults
        });
    },
    getExploreScript: function(projectKey, notebookId, cellId) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-explore-script", {projectKey, notebookId, cellId});
    },
    saveExploreScript: function(projectKey, notebookId, cellId, exploreScript) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/save-explore-script",
            {projectKey, notebookId, cellId, exploreScript: JSON.stringify(exploreScript)});
    },
    getExploreChart: function(projectKey, notebookId, cellId) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-explore-chart", {projectKey, notebookId, cellId});
    },
    saveExploreChart: function(projectKey, notebookId, cellId, exploreChart) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/save-explore-chart",
            {projectKey, notebookId, cellId, exploreChart: JSON.stringify(exploreChart)});
    },
    getSummary: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-summary/", {projectKey: projectKey, id: id});
    },
    getHistory: function(projectKey, notebookId) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-history/", {projectKey: projectKey, notebookId: notebookId});
    },
    getCellHistory: function(projectKey, notebookId, cellId) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/get-cell-history/", {projectKey: projectKey, notebookId: notebookId, cellId: cellId});
    },
    getHistoryResult: function(projectKey, notebookId, qid) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/get-history-result/", {
            projectKey: projectKey,
            notebookId: notebookId,
            qid: qid
        }, "nospinner");
    },
    startSQLQueryGeneration: function(projectKey, query, name) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/generate-query",  {
            projectKey: projectKey,
            query: query,
            name: name
        })
    },
    createForDataset: function(projectKey, datasetSmartName, type, name) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/create-for-dataset/", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName,
            type: type,
            name: name
        });
    },
    getProgress: function(projectKey, notebookId, cellId, qid) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/get-progress", {
            projectKey: projectKey,
            notebookId: notebookId,
            cellId: cellId,
            qid: qid
        }, "nospinner");
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/list-heads/", {projectKey: projectKey, tagFilter: tagFilter});
    },
    listConnections: function(projectKey) {
        return APIXHRService("GET", API_PATH + "sql-notebooks/list-connections/", {projectKey:projectKey});
    },
    removeQuery: function(projectKey, notebookId, cellId, qid) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/remove-query/", {projectKey: projectKey, notebookId: notebookId, cellId: cellId, qid: qid});
    },
    run: function(projectKey, notebookId, cellId, query, fullCount) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/run/", {
            projectKey: projectKey,
            notebookId: notebookId,
            cellId: cellId,
            queryData: JSON.stringify(query),
            fullCount: fullCount
        }, "nospinner");
    },
    getExecutionPlan: function(projectKey, query) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/get-execution-plan", {
            projectKey: projectKey,
            queryData: JSON.stringify(query)
        });
    },
    save: function(notebook) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/save/", {notebook: angular.toJson(notebook)}, "nospinner");
    },
    testStreamedExport: function(projectKey, notebookId, cellId, qid) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/test-streamed-export/", {
            projectKey: projectKey,
            notebookId: notebookId,
            cellId: cellId,
            qid: qid
        });
    },
    saveBackToRecipe: function(projectKey, notebookId, sql) {
        return APIXHRService("POST", API_PATH + "sql-notebooks/save-back-to-recipe", {
            projectKey: projectKey,
            notebookId: notebookId,
            sql: sql
        });
    }
},
explores: {
    getScript: function(projectKey, datasetSmartName) {
        return APIXHRService("GET", API_PATH + "explores/get-script", {projectKey: projectKey, datasetSmartName: datasetSmartName});
    },
    saveScript: function(projectKey, datasetSmartName, script) {
        return APIXHRService("POST", API_PATH + "explores/save-script", {
            projectKey: projectKey, datasetSmartName: datasetSmartName,
            script: angular.toJson(script)
        });
    },
    getCaptureScript: function(projectKey, streamingEndpointId) {
        return APIXHRService("GET", API_PATH + "explores/get-capture-script", {projectKey: projectKey, streamingEndpointId: streamingEndpointId});
    },
    saveCaptureScript: function(projectKey, streamingEndpointId, script) {
        return APIXHRService("POST", API_PATH + "explores/save-capture-script", {
            projectKey: projectKey, streamingEndpointId: streamingEndpointId,
            script: angular.toJson(script)
        });
    },
    get: function(projectKey, datasetSmartName) {
        return APIXHRService("GET", API_PATH + "explores/get", {projectKey: projectKey, datasetSmartName: datasetSmartName});
    },
    save: function(projectKey, datasetSmartName, data) {
        return APIXHRService("POST", API_PATH + "explores/save", {
            projectKey: projectKey, datasetSmartName: datasetSmartName,
            data: angular.toJson(data)
        });
    },
    setExploreOnSinglePartition: function(projectKey, datasetName, partitionId) {
         return APIXHRService("POST", API_PATH + "explores/set-explore-on-single-partition/", {
            projectKey: projectKey,
            datasetName: datasetName,
            partitionId: partitionId
        });
    },
    setColumnMeaning: function(projectKey, datasetName, columnName, meaning) {
        return APIXHRService("POST", API_PATH + "explores/set-column-meaning/", {
            projectKey: projectKey,
            datasetName: datasetName,
            columnName: columnName,
            meaning: meaning
        });
    },
    setColumnsMeanings: function(data) {
        return APIXHRService("POST", API_PATH + "explores/set-columns-meanings/", {data: JSON.stringify(data)});
    },
    setColumnStorageType: function(projectKey, datasetName, columnName, storageType, actionIds) {
        return APIXHRService("POST", API_PATH + "explores/set-column-storage-type/", {
            projectKey: projectKey, datasetName: datasetName,
            columnName: columnName, storageType: storageType, actionIds: JSON.stringify(actionIds)
        });
    },
    getSetColumnStorageTypeImpact: function(projectKey, datasetName, columnName, storageType) {
        return APIXHRService("POST", API_PATH + "explores/get-set-column-storage-type-impact/", {
            projectKey: projectKey, datasetName: datasetName,
            columnName: columnName, storageType: storageType
        });
    },
    updateColumn: function(projectKey, datasetName, column) {
        return APIXHRService("POST", API_PATH + "explores/update-column", {
            projectKey: projectKey, datasetName: datasetName,
            column: JSON.stringify(column)
        });
    },
    listPluginChartDescs: function(projectKey) {
        return APIXHRService("GET", API_PATH + "explores/list-plugin-chart-descs", {
            projectKey: projectKey
        });
    },
    getOrCreatePluginChart: function(projectKey, datasetSmartName, chartDef, webAppId) {
        return APIXHRService("POST", API_PATH + "explores/get-or-create-plugin-chart", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName,
            webAppId: webAppId,
            chartDef: JSON.stringify(chartDef)
        });
    },
    getChartEngines: function(dataset, script, chartDef) {
        return APIXHRService("GET", API_PATH + "explores/get-chart-engines", {projectKey: dataset.projectKey, datasetName: dataset.name, script: JSON.stringify(script), chartDef: JSON.stringify(chartDef)});
    },
    getInteractiveSearchSavedQueries: function(projectKey, datasetSmartName) {
        return APIXHRService("GET", API_PATH + "explores/get-es-search-saved-queries/", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName
        });
    },
    saveInteractiveSearchQuery: function(projectKey, datasetSmartName, query, oldSavedQueryName) {
        return APIXHRService("POST", API_PATH + "explores/save-es-search-query/", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName,
            savedQuery: JSON.stringify(query),
            oldSavedQueryName: oldSavedQueryName
        });
    },
    deleteInteractiveSearchSavedQuery: function(projectKey, datasetSmartName, savedQueryName) {
        return APIXHRService("POST", API_PATH + "explores/delete-es-search-saved-query/", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName,
            savedQueryName: savedQueryName
        });
    },
},
shakers: {
    multiColumnAnalysis: function(contextProjectKey, projectKey, datasetName, streamingEndpointId, data, requestedSampleId, columns, compute, histogram) {
        return APIXHRService("POST", API_PATH + "shaker/multi-column-analysis/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            streamingEndpointId: streamingEndpointId,
            data: JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            columns: columns,
            compute: compute,
            histogram: histogram || 1
        });
    },
    multiColumnFullAnalysis: function(contextProjectKey, projectKey, datasetName, data, fullSamplePartitionId, columns) {
        return APIXHRService("POST", API_PATH + "shaker/multi-column-full-analysis/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(data),
            fullSamplePartitionId: fullSamplePartitionId,
            columns: columns
        });
    },
    convertCellValueToGeoJSON: function(cellGeometry) {
        return APIXHRService("POST", API_PATH + "shaker/convert-to-geojson/", {cellGeometry : cellGeometry})
    },
    detailedColumnAnalysis: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column,
            alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics, forceTimePeriodAnalysis) {
        return APIXHRService("POST", API_PATH + "shaker/detailed-column-analysis/", {
            contextProjectKey,
            projectKey,
            datasetName,
            data: JSON.stringify(data),
            requestedSampleId,
            column,
            alphanumMaxResults,
            fullSamplePartitionId,
            withFullSampleStatistics,
            forceTimePeriodAnalysis
        });
    },
    detailedStreamingColumnAnalysis: function(contextProjectKey, projectKey, streamingEndpointId, data, requestedSampleId, column, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
        // same call, different parameters
        return APIXHRService("POST", API_PATH + "shaker/detailed-column-analysis/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            streamingEndpointId: streamingEndpointId,
            data: JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column,
            alphanumMaxResults : alphanumMaxResults,
            fullSamplePartitionId : fullSamplePartitionId,
            withFullSampleStatistics: withFullSampleStatistics
        });
    },
    textAnalysis: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column, simplificationParameters) {
        return APIXHRService("POST", API_PATH + "shaker/text-analysis/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column,
            simplificationParameters: JSON.stringify(simplificationParameters)
        });
    },
    autoDetectCategoriesForImageView: function(projectKey, datasetName, imageViewSettings) {
        return APIXHRService("POST", API_PATH + "shaker/auto-detect-categories-for-image-view", {
            projectKey: projectKey,
            datasetName: datasetName,
            imageViewSettings: JSON.stringify(imageViewSettings)
        });
    },
    generateSteps: function(projectKey, datasetName, data, requestedSampleId, query, previousRequestId) {
        return APIXHRService("POST", API_PATH + "shaker/ai-completion", {
            projectKey, datasetName, data: JSON.stringify(data), requestedSampleId, query, previousRequestId
        });
    },
    sendGenerateStepsRating: function (isSuccess, feedback, requestId) {
        return APIXHRService("POST", API_PATH + "/flow/ai-tools/log-rating", {
            isSuccess, feedback, requestId, featureRated: "generateSteps"
        });
    },
    smartExtractor: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column, selections, excluded, customRegex, onColumnNames, firstSentence, filters) {
        return APIXHRService("POST", API_PATH + "shaker/smart-extractor/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column,
            selections: JSON.stringify(selections),
            excluded: JSON.stringify(excluded),
            customRegex: customRegex || "",
            onColumnNames: JSON.stringify(onColumnNames),
            firstString: firstSentence,
            filters: JSON.stringify(filters)
        });
    },
    smartDateGuess: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column) {
        return APIXHRService("POST", API_PATH + "shaker/smart-date-guess/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column
        });
    },
    smartDateValidate: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column, format) {
        return APIXHRService("POST", API_PATH + "shaker/smart-date-validate/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column,
            format: format
        });
    },
    suggestionPreview: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column, regex) {
        return APIXHRService("POST", API_PATH + "shaker/suggestion-preview/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column,
            regex: regex
        });
    },
    searchElasticDataset: function(contextProjectKey, projectKey, datasetName, data, query, from, maxRows, batchSize) {
        const size = Math.min(batchSize, maxRows - from);
        query = angular.copy(query);
        query.size = size;
        query.from = from;

        return APIXHRService("POST", API_PATH + "shaker/interactive-es-search/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            searchQuery: JSON.stringify(query)
        });
    },
    refreshTable: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, allowCache, filters, isForPreview) {
        return APIXHRService("POST", API_PATH + "shaker/refresh-table/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            allowCache: allowCache,
            filters: JSON.stringify(filters),
            isForPreview: isForPreview
        });
    },
    refreshCapture: function(contextProjectKey, projectKey, streamingEndpointId, data, requestedSampleId, allowCache, filters, recipeSchema) {
        return APIXHRService("POST", API_PATH + "shaker/refresh-capture/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            streamingEndpointId: streamingEndpointId,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            allowCache: allowCache,
            filters: JSON.stringify(filters),
            recipeSchema: recipeSchema == null ? null : JSON.stringify(recipeSchema)
        });
    },
    getClusters: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, column, setBased, radius, timeOut, blockSize) {
        return APIXHRService("POST", API_PATH + "shaker/get-clusters/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            column: column,
            setBased: setBased,
            radius: radius,
            timeOut: timeOut,
            blockSize: blockSize
        });
    },
    getTableChunk: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, firstRow, nbRows, firstCol, nbCols, filters) {
        return APIXHRService("POST", API_PATH + "shaker/get-table-chunk/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            firstRow: firstRow,
            nbRows: nbRows,
            firstCol: firstCol,
            nbCols: nbCols,
            filters: JSON.stringify(filters)
        });
    },
    getGrokExpressionSample: function(contextProjectKey, projectKey, datasetName, data, requestedSampleId, firstRow, nbRows, firstCol, nbCols, grokExpression){
        return APIXHRService("POST", API_PATH + "shaker/get-grok-sample", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            datasetName: datasetName,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            firstRow: firstRow,
            nbRows: nbRows,
            firstCol: firstCol,
            nbCols: nbCols,
            grokExpression,
        })
    },
    getCaptureChunk: function(contextProjectKey, projectKey, streamingEndpointId, data, requestedSampleId, firstRow, nbRows, firstCol, nbCols, filters) {
        return APIXHRService("POST", API_PATH + "shaker/get-capture-chunk/", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            streamingEndpointId: streamingEndpointId,
            data : JSON.stringify(data),
            requestedSampleId: requestedSampleId,
            firstRow: firstRow,
            nbRows: nbRows,
            firstCol: firstCol,
            nbCols: nbCols,
            filters: JSON.stringify(filters)
        });
    },
    // returns a Future. Typically debounced and pending futures canceled using MonoFuture.
    previewColumn: function(projectKey, datasetName, columnName, disableSpinner) {
        return APIXHRService("POST", API_PATH + "shaker/preview-column", {
            projectKey:projectKey,
            datasetName:datasetName,
            columnName: columnName
        }, disableSpinner ? "nospinner" : undefined);
    },
    // returns a Future. Typically debounced and pending futures canceled using MonoFuture.
    validateExpression: function(projectKey, datasetProjectKey, datasetName, data,
                                 requestedSampleId, expression, stepId, subStepId, stepDepth, avoidRepeatingFormulaInError, spinner=true) {
        return APIXHRService("POST", API_PATH + "shaker/validate-expression", {
            projectKey:projectKey,
            datasetProjectKey:datasetProjectKey,
            datasetName:datasetName,
            data: JSON.stringify(data),
            requestedSampleId:requestedSampleId,
            expression:expression,
            stepId:stepId,
            subStepId:subStepId,
            stepDepth: stepDepth,
            avoidRepeatingFormulaInError
        }, spinner?'':'nospinner');
    },
    fixExpression: function(projectKey, datasetProjectKey, datasetName, data,
        requestedSampleId, expression, fixName = "plus", stepId = 1, subStepId = 1, stepDepth = -1, copyScriptFromExplore = false, useUDAFRegistry = false) {
        return APIXHRService("POST", API_PATH + "shaker/fix-expression", {
            projectKey,
            datasetProjectKey,
            datasetName,
            data: JSON.stringify(data),
            requestedSampleId,
            expression,
            fixName,
            stepId,
            subStepId,
            stepDepth,
            copyScriptFromExplore,
            useUDAFRegistry
        });
    },
    validateUdf: function(projectKey, datasetProjectKey, datasetName, data, requestedSampleId, expression, stepId, subStepId, stepDepth) {
        return APIXHRService("POST", API_PATH + "shaker/validate-udf", {
            projectKey: projectKey,
            datasetProjectKey: datasetProjectKey,
            datasetName: datasetName,
            data: JSON.stringify(data),
            requestedSampleId:requestedSampleId,
            udf:expression,
            stepId: stepId,
            subStepId: subStepId,
            stepDepth: stepDepth
        });
    },
    randomizeColors: function() {
        return APIXHRService("POST", API_PATH + "shaker/randomize-colors");
    },
    convertToASCII: function(inputs, lowercase, replaceNonAlphanumByUnderscore) {
        return APIXHRService("POST", API_PATH + "shaker/convert-to-ascii", {
            inputs:  JSON.stringify(inputs), lowercase, replaceNonAlphanumByUnderscore});
    },
    computeRelativeDateInterval: function(params) {
        return APIXHRService("POST", API_PATH + "shaker/compute-relative-date-interval", {
            params: JSON.stringify(params)
        });
    },
    /* Static data for the Shaker */
    getProcessorsLibrary: function(lang) {
        return APIXHRService("GET", API_PATH + "shaker/get-processors-library/", {lang});
    },
    /**
     *
     * @param {string} projectKey
     * @param {object} visualIfDesc visual if description
     * @param {list<string>} columns columns in dataset schema
     * @returns
     */
    convertVisualIf: function(projectKey, visualIfDesc, columns) {
        return APIXHRService("GET", API_PATH + "visualif/convert", { projectKey, processorDesc: JSON.stringify(visualIfDesc), columns });
    },
    charts: {
        validateExpression: function(projectKey, datasetProjectKey, datasetName, data, requestedSampleId, expression, spinner=false) {
            return APIXHRService("POST", API_PATH + "shaker/charts/validate-expression",{
                projectKey,
                datasetProjectKey,
                datasetName,
                dataSpec: JSON.stringify(data),
                requestedSampleId,
                expression
                },
                spinner?'':'nospinner');
            },
        getColumnsSummary: function(projectKey, dataSpec) {
            return APIXHRService("POST", API_PATH + "shaker/charts/get-columns-summary", {
                projectKey:projectKey,
                dataSpec: JSON.stringify(dataSpec)
            });
        },
        exportToExcel: function(chartDef, pivotResponse, animationFrameIdx, colorMaps) {
            const params = {
                chartDef:JSON.stringify(chartDef),
                pivotResponse: JSON.stringify(pivotResponse)
            };

            if (animationFrameIdx !== undefined) {
                params.animationFrameIdx = animationFrameIdx;
            }
            if (colorMaps) {
                params.colorMaps = JSON.stringify(colorMaps);
            }

            return APIXHRService("POST", API_PATH + "shaker/charts/excel-export", params);
        },
        downloadExcelUrl: function(id) {
            return API_PATH + "shaker/charts/excel-download/?id=" + encodeURIComponent(id);
        },
        getPivotResponse: function(projectKey, dataSpec, request, requestedSampleId) {
            return APIXHRService("POST", API_PATH + "shaker/charts/get-pivot-response", {
                projectKey: projectKey,
                dataSpec : JSON.stringify(dataSpec),
                request: JSON.stringify(request),
                requestedSampleId: requestedSampleId
            });
        },
        getSamplesMetaData: function(projectKey, chartsData) {
            return APIXHRService("POST", API_PATH + "shaker/charts/metadata", {
                projectKey: projectKey,
                chartsData: JSON.stringify(chartsData),
            }, 'nospinner');
        }
    },
    getCorrelationResponse: function(shakerId, data, requestedSampleId) {
         return APIXHRService("POST", API_PATH + "shaker/charts/get-correlation-response", {
            shakerId: shakerId,
            data: JSON.stringify(data),
            requestedSampleId: requestedSampleId
        });
    },
    getLastKnownCurrencyRateDate: function() {
        return APIXHRService("GET", API_PATH + "shaker/get-last-known-currency-rate-date");
    }
},
connections: {
   getNames: function(type, checkAllowManagedFolders = false) {
        return APIXHRService("GET", API_PATH + "connections/get-names/", {type: type, checkAllowManagedFolders});
   },
   getTypeAndNames: function (connectionType, checkAllowManagedFolders = false) {
        return APIXHRService("GET", API_PATH + "connections/get-type-and-names", { connectionType: connectionType, checkAllowManagedFolders });
   },
   listCloudConnectionsHDFSRoot: function(type, checkAllowManagedFolders = false) {
        return APIXHRService("GET", API_PATH + "connections/list-connections-for-thirdparty-logging", { type: type, checkAllowManagedFolders });
   },
   getHiveNames: function(projectKey) {
       return APIXHRService("GET", API_PATH + "connections/get-hive-names/", {projectKey:projectKey});
   },
   list: function(type="all", checkAllowManagedFolders=false) {
       return APIXHRService("GET", API_PATH + "connections/list", {type, checkAllowManagedFolders});
   },
   listUsages: function(projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-usages", {projectKey: projectKey});
   },
   listSQLTables: function(connectionName, projectKey, catalog, schema) {
        return APIXHRService("GET", API_PATH + "connections/list-sql-tables-future", {connectionName:connectionName, projectKey:projectKey, catalog: catalog, schema: schema});
   },
   listSQLTablesFromProject: function(connectionName, projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-sql-tables-from-project", {connectionName:connectionName, projectKey:projectKey});
   },
   listSQLFields: function(name, tables, projectKey) {
       var x = {tables: tables};
        return APIXHRService("POST", API_PATH + "connections/list-sql-fields", {name:name, tables: JSON.stringify(x), projectKey:projectKey});
   },
   getSQLTableMapping: function(connection) {
       return APIXHRService("GET", API_PATH + "connections/get-sql-table-mapping", {connection:connection});
   },
   getTwitterConfig: function() {
       return APIXHRService("GET", API_PATH + "connections/get-twitter-config");
   },
   getDefaultTimezone: function(connectionName) {
       return APIXHRService("GET", API_PATH + "connections/get-default-assumed-timezone", {connectionName: connectionName});
   },
   getFSLike: function() {
        // TODO TODO FIXME TODO TODO
        return APIXHRService("GET", API_PATH + "connections/get-names/", {type: "Filesystem"});
   },
    countIndexedAndUnindexed: function (data) {
        return APIXHRService("GET", API_PATH + "/connections/count-indexed-and-unindexed");
    },
    listMassImportSources: function(projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-mass-import-sources", {projectKey: projectKey});
    },
    listSQLMassImportSchemas: function(connectionName, projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-sql-mass-import-schemas", {connectionName: connectionName, projectKey:projectKey});
    },
    listSQLMassImportSchemasWithCatalogs: function(connectionName, projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-sql-mass-import-schemas-with-catalogs", {connectionName: connectionName, projectKey:projectKey});
    },
    listSQLSchemas: function(connectionName, projectKey, catalog) {
        if (catalog) {
            return APIXHRService("GET", API_PATH + "connections/list-sql-schemas", {connectionName: connectionName, projectKey: projectKey, catalog: catalog});
        } else {
            return APIXHRService("GET", API_PATH + "connections/list-sql-schemas", {connectionName: connectionName, projectKey: projectKey});
        }
    },
    listSQLCatalogs: function(connectionName, projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-sql-catalogs", {connectionName: connectionName, projectKey: projectKey});
    },
    listSQLMassImportTables: function (connectionName, sourceCatalog, sourceSchema, projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-sql-mass-import-tables", {
            connectionName: connectionName, sourceCatalog: sourceCatalog, sourceSchema: sourceSchema, projectKey:projectKey
        });
    },
    listElasticSearchMassImportIndices: function (connectionName) {
        return APIXHRService("GET", API_PATH + "connections/list-elasticsearch-mass-import-indices", {
            connectionName: connectionName
        });
    },
    listHiveMassImportTables: function (connectionName, projectKey) {
        return APIXHRService("GET", API_PATH + "connections/list-hive-mass-import-tables", {
            connectionName: connectionName, projectKey: projectKey
        });
    },
    getTableImportCandidatesFromExplorer: function (workflowType, tables, projectKey, targetHDFSConnection) {
        return APIXHRService("POST", API_PATH + "connections/get-table-import-candidates-from-explorer", {
            workflowType: workflowType,
            tables: JSON.stringify(tables),
            projectKey: projectKey,
            targetHDFSConnection: targetHDFSConnection
        });
    },
    getTableImportCandidatesFromKeys: function (tables, remarks, projectKey, targetConnection) {
        return APIXHRService("POST", API_PATH + "connections/get-table-import-candidates-from-keys", {
            tables: JSON.stringify(tables),
            projectKey: projectKey,
            remarks: JSON.stringify(remarks),
            targetConnection:targetConnection
        });
    },
    massImportTableCandidates: function (projectKey, sqlTableCandidates, hiveTableCandidates, elasticSearchIndexCandidates, zoneId) {
        return APIXHRService("POST", API_PATH + "connections/mass-import-candidates", {
            projectKey: projectKey,
            sqlTableCandidates: JSON.stringify(sqlTableCandidates),
            hiveTableCandidates: JSON.stringify(hiveTableCandidates),
            elasticSearchIndexCandidates: JSON.stringify(elasticSearchIndexCandidates),
            zoneId
        });
    },

   /* Alation "embedded catalog chooser" integration */
   getTableImportCandidatesFromAlationMCC: function(projectKey, alationSelection) {
        return APIXHRService("POST", API_PATH + "connections/get-table-import-candidates-from-alation-mcc", {
            projectKey: projectKey,
            alationSelection: JSON.stringify(alationSelection)
        })
   },

   /* Alation "open in" integration" */
   registerAlationOpener : function(alationAPIToken) {
        return APIXHRService("POST", API_PATH + "connections/register-alation-opener", {
            alationAPIToken : alationAPIToken
        })
   },
   getAlationOpenInfo: function(alationOpenId) {
        return APIXHRService("POST", API_PATH + "connections/get-alation-open-info", {
            alationOpenId: alationOpenId
        })
    }
},

managedfolder: {
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "managedfolder/list/", {projectKey: projectKey})
    },
    listWithAccessible: function(projectKey) {
        return APIXHRService("GET", API_PATH + "managedfolder/list-with-accessible/", {projectKey: projectKey});
    },
    get: function(contextProjectKey, projectKey, id) {
        return APIXHRService("GET", API_PATH + "managedfolder/get/", {contextProjectKey: contextProjectKey, projectKey: projectKey, id: id})
    },
    getSummary: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-summary/", {projectKey: projectKey, id: id})
    },
    getForInsight: function(contextProjectKey, projectKey, smartId) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-for-insight/", {contextProjectKey: contextProjectKey, projectKey: projectKey, smartId: smartId})
    },
    getWithStatus: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-with-status/", {projectKey: projectKey, id: id})
    },
    getFullInfo: function(contextProjectKey, projectKey, id) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-full-info/", {contextProjectKey: contextProjectKey, projectKey: projectKey, id: id})
    },
    testPartitioning: function(odb) {
        return APIXHRService("POST", API_PATH + "managedfolder/test-partitioning", {data: JSON.stringify(odb)});
    },
    detectPartitioning: function(odb) {
        return APIXHRService("POST", API_PATH + "managedfolder/detect-partitioning", {data: JSON.stringify(odb)});
    },
    browse: function(projectKey, folderId, path, contextProjectKey) {
        return APIXHRService("GET", API_PATH + "managedfolder/browse", {projectKey: projectKey, folderId: folderId, path: path, contextProjectKey: contextProjectKey})
    },
    listFS: function(projectKey, smartId) {
        return APIXHRService("GET", API_PATH + "managedfolder/list-fs", {projectKey: projectKey, smartId: smartId})
    },
    listPartitionFS: function(projectKey, smartId, partition) {
        return APIXHRService("GET", API_PATH + "managedfolder/list-partition-fs", {projectKey: projectKey, smartId: smartId, partition: partition})
    },
    listPartitionsWithName: function(projectKey, folderId) {
        return APIXHRService("POST", API_PATH + "managedfolder/list-partitions-with-name/", {
            projectKey: projectKey, folderId: folderId
        });
    },
    save: function(data, saveInfo) {
        return APIXHRService("POST", API_PATH + "managedfolder/save", {data: JSON.stringify(data), saveInfo: JSON.stringify(saveInfo || {})})
    },
    rename: function(projectKey, folderId, newName) {
        return APIXHRService("POST", API_PATH + "managedfolder/rename", { projectKey, folderId, newName })
    },
    deleteItems: function(projectKey, odbId, itemPaths) {
        return APIXHRService("POST", API_PATH + "managedfolder/delete-items", {
            projectKey: projectKey, odbId: odbId, itemPaths: JSON.stringify(itemPaths)
        })
    },
    deleteItem: function(projectKey, odbId, itemPath, isDirectory) {
        return APIXHRService("GET", API_PATH + "managedfolder/delete-item", {
            projectKey: projectKey, odbId: odbId, itemPath: itemPath, isDirectory: isDirectory
        });
    },
    renameItem: function(projectKey, odbId, itemPath, isDirectory, newName) {
        return APIXHRService("GET", API_PATH + "managedfolder/rename-item", {
            projectKey: projectKey, odbId: odbId, itemPath: itemPath, newName: newName, isDirectory: isDirectory
        })
    },
    moveItems: function(projectKey, odbId, items) {
        return APIXHRService("POST", API_PATH + "managedfolder/move-items", {
            projectKey: projectKey, odbId: odbId, items: JSON.stringify(items)
        })
    },
    clearPartitions: function(projectKey, odbId, partitions) {
        return APIXHRService("POST", API_PATH + "managedfolder/clear-partitions/", {
            projectKey:projectKey,
            odbId:odbId,
            partitions:JSON.stringify(partitions)
        });
    },
    setExploreOnSinglePartition: function(projectKey, folderId, partitionId) {
        return APIXHRService("POST", API_PATH + "managedfolder/set-explore-on-single-partition/", {
           projectKey: projectKey,
           folderId: folderId,
           partitionId: partitionId
       });
    },
    saveSampling: function(projectKey, folderId, data) {
        return APIXHRService("GET", API_PATH + "managedfolder/save-sampling", {
            projectKey: projectKey,
            folderId: folderId,
            data: JSON.stringify(data)
        });
    },
    uploadItem: function(projectKey, odbId, path, file, forceUpload, callback) {
        return uploadFileRequest("managedfolder/upload-item", function(formdata) {
            formdata.append("projectKey", projectKey);
            formdata.append("odbId", odbId);
            formdata.append("file", file);
            formdata.append("path", path);
            formdata.append("forceUpload", forceUpload);
        }, callback);
    },
    createSubFolder: function(projectKey, folderId, path) {
        return APIXHRService("POST", API_PATH + "managedfolder/create-sub-folder", {
            projectKey: projectKey,
            folderId: folderId,
            path: path
        });
    },
    getDownloadItemURL: function(contextProjectKey, projectKey, obdId, path) {
        return API_PATH + "managedfolder/download-item/?contextProjectKey=" + encodeURIComponent(contextProjectKey)
            + "&projectKey=" + encodeURIComponent(projectKey)
            + "&obdId=" + encodeURIComponent(obdId)
            + "&path=" + encodeURIComponent(path) ;
    },
    getDownloadFolderURL: function(contextProjectKey, projectKey, obdId, path) {
        return API_PATH + "managedfolder/download-folder/?contextProjectKey=" + encodeURIComponent(contextProjectKey) + "&projectKey=" + encodeURIComponent(projectKey) + "&obdId=" + encodeURIComponent(obdId) + "&path=" + encodeURIComponent(path) ;
    },
    getDownloadFolderExportURL: function(contextProjectKey, projectKey, obdId, exportId) {
        return API_PATH + "managedfolder/download-export/?contextProjectKey=" + encodeURIComponent(contextProjectKey) + "&projectKey=" + encodeURIComponent(projectKey) + "&obdId=" + encodeURIComponent(obdId) + "&exportId=" + encodeURIComponent(exportId) ;
    },
    startDownloadFolder: function (contextProjectKey, projectKey, obdId, path) {
        return APIXHRService("POST", API_PATH + "managedfolder/start-export", {
            contextProjectKey: contextProjectKey,
            projectKey: projectKey,
            obdId: obdId,
            path: path
        })
    },
    getItemInfo: function(projectKey, smartId, itemPath) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-item-info", {
            projectKey: projectKey, smartId: smartId, itemPath: itemPath
        })
    },
    previewItem: function(contextProjectKey, projectKey, smartId, itemPath) {
        return APIXHRService("GET", API_PATH + "managedfolder/preview-item", {
            contextProjectKey: contextProjectKey, projectKey: projectKey, smartId: smartId, itemPath: itemPath
        })
    },
    decompressItem: function(projectKey, odbId, itemPath) {
        return APIXHRService("GET", API_PATH + "managedfolder/decompress-item", {
            projectKey: projectKey, odbId: odbId, itemPath: itemPath
        })
    },
    listAvailableMetrics: function(projectKey, folderId) {
        return APIXHRService("GET", API_PATH + "managedfolder/list-available-metrics", {
            projectKey: projectKey,
            folderId: folderId
        });
    },
    saveMetrics: function(projectKey, folderId, metrics, checks) {
        return APIXHRService("POST", API_PATH + "managedfolder/save-metrics", {
            projectKey: projectKey,
            folderId: folderId,
            metricsData: JSON.stringify(metrics),
            checksData: JSON.stringify(checks)
        });
    },
    computeMetrics: function(projectKey, folderId, partitionId, allPartitions) {
        return APIXHRService("GET", API_PATH + "managedfolder/compute-metrics", {
            projectKey: projectKey,
            folderId: folderId,
            partitionId: partitionId,
            allPartitions: allPartitions
        });
    },
    computePlan: function(projectKey, folderId, metrics) {
        return APIXHRService("POST", API_PATH + "managedfolder/compute-plan", {
            projectKey: projectKey,
            folderId: folderId,
            metricsData: JSON.stringify(metrics)
        });
    },
    getPreparedMetricHistory: function(projectKey, folderId, partitionId, metric, metricId) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-prepared-metric-history", {
            projectKey: projectKey,
            folderId: folderId,
            data: JSON.stringify(metric),
            metricId: metricId,
            partitionId: partitionId
        });
    },
    getPreparedMetricHistories: function(projectKey, folderId, displayedState) {
        return APIXHRService("POST", API_PATH + "managedfolder/get-prepared-metric-histories", {
            projectKey: projectKey,
            folderId: folderId,
            data: JSON.stringify(displayedState || {})
        });
    },
    getPreparedMetricPartitions: function(projectKey, folderId, displayedState) {
        return APIXHRService("POST", API_PATH + "managedfolder/get-prepared-metric-partitions", {
            projectKey: projectKey,
            folderId: folderId,
            data: JSON.stringify(displayedState || {})
        });
    },
    getPartitionListMetric: function(projectKey, folderId ) {
        return APIXHRService("GET", API_PATH + "managedfolder/get-partition-list-metric", {
            projectKey: projectKey,
            folderId: folderId
        });
    },
    refreshPartitionListMetric: function(projectKey, folderId ) {
        return APIXHRService("GET", API_PATH + "managedfolder/refresh-partition-list-metric", {
            projectKey: projectKey,
            folderId: folderId
        });
    },
    listComputedMetrics: function(projectKey, folderId) {
        return APIXHRService("GET", API_PATH + "managedfolder/list-computed-metrics", {
            projectKey: projectKey,
            folderId: folderId
        });
    },
    computeProbe: function(projectKey, folderId, partitionId, allPartitions, metrics) {
        return APIXHRService("POST", API_PATH + "managedfolder/compute-probe", {
            projectKey: projectKey,
            folderId: folderId,
            partitionId: partitionId,
            allPartitions: allPartitions,
            data: JSON.stringify(metrics)
        });
    },
    runChecks: function(projectKey, folderId, partitionId, allPartitions) {
        return APIXHRService("GET", API_PATH + "managedfolder/run-checks", {
            projectKey: projectKey,
            folderId: folderId,
            partitionId: partitionId,
            allPartitions: allPartitions
        });
    },
    runCheck: function(projectKey, folderId, partitionId, allPartitions, metricsChecks) {
        return APIXHRService("POST", API_PATH + "managedfolder/run-check", {
            projectKey: projectKey,
            folderId: folderId,
            partitionId: partitionId,
            allPartitions: allPartitions,
            data: JSON.stringify(metricsChecks)
        });
    },
    createMetricsDataset: function(projectKey, folderId, view, partition, filter) {
        return APIXHRService("GET", API_PATH + "datasets/create-metrics-dataset", {
            projectKey: projectKey,
            objectId: folderId,
            view: view,
            partition: partition,
            filter: filter,
            taggableType: "MANAGED_FOLDER"
        });
    },
    getCheckHistories: function(projectKey, folderId, displayedState) {
        return APIXHRService("POST", API_PATH + "managedfolder/get-prepared-check-histories", {
            projectKey: projectKey,
            folderId: folderId,
            data: JSON.stringify(displayedState || {})
        });
    },
    listComputedChecks: function(projectKey, folderId) {
        return APIXHRService("GET", API_PATH + "managedfolder/list-computed-checks", {
            projectKey: projectKey,
            folderId: folderId
        });
    },
    getHint: function(projectKey, folderId, probe) {
        return APIXHRService("POST", API_PATH + "managedfolder/get-hint", {
            projectKey: projectKey,
            folderId: folderId,
            data: JSON.stringify(probe)
        });
    },
    clearMetrics: function(projectKey, folderId) {
        return APIXHRService("GET", API_PATH + "managedfolder/clear-metrics", {
            projectKey: projectKey,
            folderId: folderId
        });
    }
},
samples: {
    list: function() {
        return APIXHRService("POST", API_PATH + "samples/list/");
    }
},
fsproviders: {
    testConnection: function(type, params, projectKey, contextVars, listBuckets) {
        return APIXHRService("POST", API_PATH + "fsproviders/test-connection/", {
            type:type,
            params:JSON.stringify(params),
            projectKey:projectKey,
            contextVars:JSON.stringify(contextVars),
            listBuckets:listBuckets
        });
    },
    browse: function(type, params, projectKey, contextVars, path, isDirectory) {
        return APIXHRService("POST", API_PATH + "fsproviders/fs-browse/", {
            type: type,
            params: JSON.stringify(params),
            projectKey: projectKey,
            contextVars: JSON.stringify(contextVars),
            path: path,
            isDirectory: isDirectory
        });
    },
    listFiles: function(type, params, projectKey, contextVars, selectionRules, selectedOnly) {
        return APIXHRService("POST", API_PATH + "fsproviders/list-files/", {
            type:type,
            params:JSON.stringify(params),
            projectKey:projectKey,
            contextVars:JSON.stringify(contextVars),
            selectionRules:JSON.stringify(selectionRules),
            selectedOnly:selectedOnly
        });
    },
    expandSelectionRules: function(type, params, projectKey) {
        return APIXHRService("POST", API_PATH + "fsproviders/expand-selection-rules/", {
            type,
            params: JSON.stringify(params),
            projectKey,
        });
    },
},
taggableObjects: {
    countAccessibleObjects: function(projectKey) {
        return APIXHRService("GET", API_PATH + "taggable-objects/count-accessible-objects", {
            projectKey: projectKey
        });
    },
    listAccessibleObjects: function(projectKey, type, mode) {
        return APIXHRService("GET", API_PATH + "taggable-objects/list-accessible-objects", {
            projectKey: projectKey,
            type: type,
            mode: mode
        });
    },
    listTags: function(projectKey) {
        return APIXHRService("GET", API_PATH + "taggable-objects/list-tags", {projectKey: projectKey});
    },
    listAllTags: function(includeLimitedVisibility = false) {
        return APIXHRService("GET", API_PATH + "taggable-objects/list-all-tags", {
            includeLimitedVisibility: includeLimitedVisibility,
        });
    },
    listTagsUsage: function(projectKey, options, spinnerMode) {
        return APIXHRService("GET", API_PATH + "taggable-objects/list-tags-usage", {
            projectKey: projectKey,
            options: JSON.stringify(options)
        }, spinnerMode);
    },
    setTags: function(projectKey, tags) {
        return APIXHRService("POST", API_PATH + "taggable-objects/set-tags", {
            projectKey: projectKey,
            data: JSON.stringify({tags: tags})
        });
    },
    applyTagging: function(projectKey, request)  {
        return APIXHRService("POST", API_PATH + "taggable-objects/apply-tagging/", {
            projectKey: projectKey,
            request: JSON.stringify(request)
        });
    },
    clear: function(requests) {
        return APIXHRService("POST", API_PATH + "taggable-objects/clear/", {
            requests: JSON.stringify(requests)
        });
    },
    checkDeletedObjects: function (requests) {
        return APIXHRService("POST", API_PATH + "taggable-objects/check-deleted-objects/", {
            requests: JSON.stringify(requests)
        });
    },
    computeDeletionImpact: function(request, contextProjectKey) {
        //POST because requests might be large
        return APIXHRService("POST", API_PATH + "taggable-objects/compute-deletion-impact/", {
            request: JSON.stringify(request),
            contextProjectKey: contextProjectKey
        });
    },
    delete: function(request, contextProjectKey) {
        return APIXHRService("POST", API_PATH + "taggable-objects/delete/", {
            request: JSON.stringify(request),
            contextProjectKey: contextProjectKey
        });
    },
    setShortDesc: function(taggableObject, shortDesc) {
        return APIXHRService("POST", API_PATH + "taggable-objects/set-short-desc/", {
            taggableObject: JSON.stringify(taggableObject),
            shortDesc: shortDesc
        });
    },
    getMetadata: function(taggableObject) {
        return APIXHRService("GET", API_PATH + "taggable-objects/get-metadata/", {
            taggableObject: JSON.stringify(taggableObject)
        });
    },
    setMetaData: function(taggableObject, request) {
        return APIXHRService("POST", API_PATH + "taggable-objects/set-metadata/", {
            taggableObject: JSON.stringify(taggableObject),
            request: JSON.stringify(request)
        });
    },

},
datasets: {
    /* CRUD stuff */
    rename: function(projectKey, oldName, newName) {
        return APIXHRService("POST", API_PATH + "datasets/rename/", {projectKey: projectKey, oldName: oldName, newName: newName});
    },
    list: function(projectKey) {
        return APIXHRService("GET", API_PATH + "datasets/list/", {projectKey: projectKey});
    },
    listWithAccessible: function(projectKey) {
        return APIXHRService("GET", API_PATH + "datasets/list-with-accessible/", {projectKey: projectKey});
    },
    listNames: function(projectKey) {
        return APIXHRService("GET", API_PATH + "datasets/list-names/", {projectKey: projectKey});
    },
    listHeaders: function(projectKey) {
        return APIXHRService("GET", API_PATH + "datasets/list-headers/", {projectKey: projectKey});
    },
    // Takes into account Tags Filter, withStatus & withForeign are false when omitted
    listHeads: function(projectKey, tagFilter, withStatus, withForeign) {
        return APIXHRService("GET", API_PATH + "datasets/list-heads/", {projectKey: projectKey, tagFilter: tagFilter, withStatus: withStatus, withForeign});
    },
    listCreatableDatasetTypes: function(projectKey) {
        return APIXHRService("GET", API_PATH + "datasets/list-creatable-dataset-types/", { projectKey });
    },
    create: function(projectKey, data, zoneId) {
        return APIXHRService("POST", API_PATH + "datasets/create/", {projectKey: projectKey, data:JSON.stringify(data), zoneId});
    },
    checkNameSafety: function(projectKey, datasetName, data) {
        return APIXHRService("POST", API_PATH + "datasets/check-name-safety/", {
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(data)
        });
    },
    checkSaveConflict: function(projectKey, data) {
        return APIXHRService("POST", API_PATH + "datasets/check-save-conflict/", {projectKey: projectKey, data:JSON.stringify(data)});
    },
    save: function(projectKey, data, saveInfo) {
        return APIXHRService("POST", API_PATH + "datasets/save/", {projectKey: projectKey, data:JSON.stringify(data), saveInfo:JSON.stringify(saveInfo)});
    },
    saveWithRecipesFixup: function(projectKey, data, saveInfo, forceRecipesFixup, shouldFixAllAvailablePartitioning) {
        return APIXHRService("POST", API_PATH + "datasets/save-with-recipes-fixup", {
            projectKey: projectKey,
            data:JSON.stringify(data),
            saveInfo:JSON.stringify(saveInfo),
            forceRecipesFixup:forceRecipesFixup,
            shouldFixAllAvailablePartitioning:shouldFixAllAvailablePartitioning
        });
    },
    exportDS: function(contextProjectKey, projectKey, name, exportParams) {
        return APIXHRService("POST", API_PATH + "datasets/export/", {contextProjectKey:contextProjectKey, projectKey: projectKey, name: name, params: JSON.stringify(exportParams)});
    },
    get: function(projectKey, name, contextProjectKey) {
        return APIXHRService("GET", API_PATH + "datasets/get/", {projectKey: projectKey, name: name, contextProjectKey: contextProjectKey});
    },
    getForExport: function(projectKey, name, contextProjectKey) {
        return APIXHRService("GET", API_PATH + "datasets/get-for-export/", {projectKey: projectKey, name: name, contextProjectKey: contextProjectKey});
    },
    getWithMetricsStatus: function(projectKey, name) {
        return APIXHRService("GET", API_PATH + "datasets/get-with-metrics-status/", {projectKey: projectKey, name: name});
    },
    getSummary: function(projectKey, name) {
        return APIXHRService("GET", API_PATH + "datasets/get-summary/", {projectKey: projectKey, name: name});
    },
    computeRenamingImpact: function(projectKey, name, nospinner) {
        return APIXHRService("GET", API_PATH + "datasets/compute-renaming-impact/", {projectKey: projectKey, name: name}, nospinner && "nospinner");
    },
    controlTwitterStreaming: function(projectKey, name, start) {
        return APIXHRService("POST", API_PATH + "datasets/control-twitter-streaming/", {projectKey: projectKey, name: name, start: start});
    },
    getTwitterStatus: function(projectKey, name) {
        return APIXHRService("GET", API_PATH + "datasets/get-twitter-status/", {projectKey: projectKey, name: name});
    },
    clearPartitions: function(projectKey, datasetName, partitions) {
        return APIXHRService("POST", API_PATH + "datasets/clear-partitions/", {
            projectKey:projectKey,
            datasetName:datasetName,
            partitions:JSON.stringify(partitions)
        });
    },
    /* Autodetection stuff */
    testAndDetectFormat: function(projectKey, data, detectPossibleFormats, inferStorageTypes) {
        return APIXHRService("POST", API_PATH + "datasets/test-and-detect-format/", {
            projectKey: projectKey,
            data: JSON.stringify(data),
            detectPossibleFormats: detectPossibleFormats,
            inferStorageTypes: inferStorageTypes
        });
    },
    detect_format: function(projectKey, data, inferStorageTypes) {
        return APIXHRService("POST", API_PATH + "datasets/test-and-detect-format/", {
            projectKey: projectKey,
            data: JSON.stringify(data),
            detectPossibleFormats: true,
            inferStorageTypes: inferStorageTypes
        });
    },
    preview: function(projectKey, data, inferStorageTypes) {
        return APIXHRService("POST", API_PATH + "datasets/test-and-detect-format/", {
            projectKey: projectKey,
            data:JSON.stringify(data),
            detectPossibleFormats:false,
            inferStorageTypes: inferStorageTypes
        });
    },
    detectOneFormat: function(projectKey, data, format) {
        return APIXHRService("POST", API_PATH + "datasets/detect-one-format/", {
            projectKey: projectKey,
            data:JSON.stringify(data), format:format
        });
    },
    testFilePartitioning: function(dataset) {
        return APIXHRService("POST", API_PATH + "datasets/test-file-partitioning", {data: JSON.stringify(dataset)});
    },
    testGeneralPartitioning: function(dataset) {
        return APIXHRService("POST", API_PATH + "datasets/test-general-partitioning", {data: JSON.stringify(dataset)});
    },
    detectFilePartitioning: function(dataset) {
        return APIXHRService("POST", API_PATH + "datasets/detect-file-partitioning", {data: JSON.stringify(dataset)});
    },
    listPartitions: function(data) {
        return APIXHRService("POST", API_PATH + "datasets/list-partitions/", {data: JSON.stringify(data)});
    },
    listPartitionsWithName: function(projectKey, datasetName) {
        return APIXHRService("POST", API_PATH + "datasets/list-partitions-with-name/", {
            projectKey: projectKey, datasetName: datasetName
        });
    },
    synchronizeHiveMetastore: function(datasets) {
        return APIXHRService("POST", API_PATH + "datasets/synchronize-hive-metastore", {datasets: JSON.stringify(datasets)});
    },
    synchronizeOneHiveMetastore: function(datasetRef, datasetParams) {
        return APIXHRService("POST", API_PATH + "datasets/synchronize-one-hive-metastore", {ref: JSON.stringify(datasetRef), params: JSON.stringify(datasetParams)});
    },
    updateFromHive : function(projectKey, name) {
        return APIXHRService("POST", API_PATH + "datasets/update-from-hive", {
            projectKey : projectKey, name:name
        });
    },
    checkHiveSync : function(projectKey, name) {
        return APIXHRService("POST", API_PATH + "datasets/check-hive-sync", {
            projectKey : projectKey, name:name
        }, "nospinner");
    },
    /* auto stuff (done backend-side) */
    autoUpdateFormat: function(projectKey, data, detectPossibleFormats, inferStorageTypes) {
        return APIXHRService("POST", API_PATH + "datasets/auto-update-format/", {
            projectKey: projectKey,
            data: JSON.stringify(data),
            detectPossibleFormats: detectPossibleFormats,
            inferStorageTypes: inferStorageTypes
        });
    },
    autoUpdateSQLSchema: function(projectKey, data) {
        return APIXHRService("POST", API_PATH + "datasets/auto-update-sql-schema/", {
            projectKey: projectKey,
            data: JSON.stringify(data)
        });
    },
    /* TODO: Move to a proper API path */
    newManagedDataset: function(projectKey, name, settings) {
        return APIXHRService("POST", API_PATH + "flow/recipes/new-managed-dataset", {
            projectKey: projectKey, name: name,
            settings: JSON.stringify(settings)
        });
    },
    newManagedFolder: function(projectKey, name, settings) {
        return APIXHRService("POST", API_PATH + "flow/recipes/new-managed-folder", {
            projectKey: projectKey, name: name,
            settings: JSON.stringify(settings)
        });
    },
    newModelEvaluationStore: function(projectKey, name, settings) {
        return APIXHRService("POST", API_PATH + "flow/recipes/new-model-evaluation-store", {
            projectKey: projectKey, name: name,
            settings: JSON.stringify(settings)
        });
    },
    newStreamingEndpoint: function(projectKey, name, settings) {
        return APIXHRService("POST", API_PATH + "flow/recipes/new-streaming-endpoint", {
            projectKey: projectKey, name: name,
            settings: JSON.stringify(settings)
        });
    },
    listManagedDatasetConnections: function() {
        return APIXHRService("GET", API_PATH + "flow/recipes/list-managed-dataset-connections");
    },
    listManagedUploadableConnections: function(projectKey) {
        return APIXHRService("GET", API_PATH + "flow/recipes/list-managed-uploadable-connections",{
            projectKey:projectKey
        });
    },
    getManagedDatasetOptions: function(recipeData, role) {
        return APIXHRService("POST", API_PATH + "flow/recipes/get-managed-dataset-options", {
            recipeData: JSON.stringify(recipeData), role: role
        });
    },
    getManagedDatasetOptionsNoContext: function(projectKey) {
        return APIXHRService("GET", API_PATH + "flow/recipes/get-managed-dataset-options-no-context", { projectKey: projectKey });
    },
    getManagedFolderOptions: function(recipeData, role) {
        return APIXHRService("POST", API_PATH + "flow/recipes/get-managed-folder-options", {
            recipeData: JSON.stringify(recipeData), role: role
        });
    },
    getManagedFolderOptionsNoContext: function(projectKey) {
        return APIXHRService("GET", API_PATH + "flow/recipes/get-managed-folder-options-no-context", { projectKey: projectKey });
    },
    getModelEvaluationStoreOptions: function(recipeData, role) {
        return APIXHRService("POST", API_PATH + "flow/recipes/get-model-evaluation-store-options", {
            recipeData: JSON.stringify(recipeData), role: role
        });
    },
    getModelEvaluationStoreOptionsNoContext: function(projectKey) {
        return APIXHRService("GET", API_PATH + "flow/recipes/get-model-evaluation-store-options-no-context", { projectKey: projectKey });
    },
    // getRetrievableKnowledgeOptions: function(recipeData, role) {
    //     return APIXHRService("POST", API_PATH + "flow/recipes/get-retrievable-knowledge-options", {
    //         recipeData: JSON.stringify(recipeData), role: role
    //     });
    // },
    // getRetrievableKnowledgeOptionsNoContext: function(projectKey) {
    //     return APIXHRService("GET", API_PATH + "flow/recipes/get-retrievable-knowledge-options-no-context", { projectKey: projectKey });
    // },
    getStreamingEndpointOptions: function(recipeData, role) {
        return APIXHRService("POST", API_PATH + "flow/recipes/get-streaming-endpoint-options", {
            recipeData: JSON.stringify(recipeData), role: role
        });
    },
    getStreamingEndpointOptionsNoContext: function(projectKey) {
        return APIXHRService("GET", API_PATH + "flow/recipes/get-streaming-endpoint-options-no-context", { projectKey: projectKey });
    },
    listRemoteDatasetConnections: function() {
        return APIXHRService("GET", API_PATH + "flow/recipes/list-remote-dataset-connections");
    },
    listFSProviderTypes: function(withNonWritable) {
        return APIXHRService("GET", API_PATH + "flow/recipes/list-fs-providers", {withNonWritable : withNonWritable});
    },
    getForTimeRange: function(projectKey, name, computeRecords, runChecks, forceRefresh, rangeStart, rangeEnd) {
        return APIXHRService("GET", API_PATH + "datasets/get-status-for-time-range/", {
            projectKey: projectKey,
            name: name,
            computeRecords: computeRecords,
            runChecks: runChecks,
            forceRefresh: forceRefresh,
            rangeStart: rangeStart,
            rangeEnd: rangeEnd
        });
    },
    getFullStatus: function(projectKey, name, computeRecords, runChecks, forceRefresh) {
        return APIXHRService("GET", API_PATH + "datasets/get-full-status/", {
            projectKey: projectKey,
            name: name,
            computeRecords: computeRecords,
            runChecks: runChecks,
            forceRefresh: forceRefresh
        });
    },
    getHead: function(contextProjectKey, projectKey, name) {
        return APIXHRService("GET", API_PATH + "datasets/get-head", {contextProjectKey: contextProjectKey, projectKey: projectKey, name: name});
    },
    getFullInfo: function(contextProjectKey, projectKey, name) {
        return APIXHRService("GET", API_PATH + "datasets/get-full-info", {contextProjectKey: contextProjectKey, projectKey: projectKey, name: name});
    },
    getPublicInfo: function(contextProjectKey, projectKey, name) {
        return APIXHRService("GET", API_PATH + "datasets/get-public-info", {contextProjectKey: contextProjectKey, projectKey: projectKey, name: name});
    },
    setDataSteward: function(projectKey, name, dataSteward) {
        return APIXHRService("POST", API_PATH + "datasets/set-data-steward/", {projectKey: projectKey, name: name, dataSteward: dataSteward});
    },
    /** Futures */
    //TODO @flow deprecated
    getRefreshedSummaryStatus: function(projectKey, datasetName, computeRecords, forceRecompute) {
        return APIXHRService("POST", API_PATH + "datasets/get-refreshed-summary-status", {
            projectKey: projectKey,
            datasetName:datasetName,
            computeRecords:computeRecords,
            forceRecompute: forceRecompute
        });
    },
    refreshSummaries: function(datasets, computeRecords, forceRecompute) {
        return APIXHRService("POST", API_PATH + "datasets/refresh-summaries", {
            datasets: JSON.stringify(datasets),
            computeRecords: computeRecords,
            forceRecompute: forceRecompute
        });
    },
    markAsBuilt: function(datasets) {
        return APIXHRService("POST", API_PATH + "datasets/mark-as-built", {
            datasets: JSON.stringify(datasets)
        });
    },
    testSchemaConsistency: function(data) {
        return APIXHRService("POST", API_PATH + "datasets/test-schema-consistency", {
            data: JSON.stringify(data)
        });
    },
    testSchemaConsistencyOnAllFiles: function(data) {
        return APIXHRService("POST", API_PATH + "datasets/test-schema-consistency-on-all-files", {
            data: JSON.stringify(data)
        });
    },
    pushColumnDescriptionsToTable: function(dataset, newSchema, detectedSchema) {
        return APIXHRService("POST", API_PATH + "datasets/push-column-descriptions-to-table", {
            data: JSON.stringify(dataset),
            newSchema: JSON.stringify(newSchema),
            detectedSchema: JSON.stringify(detectedSchema)
        });
    },
    reloadSchema: function(projectKey, datasetName) {
        return APIXHRService("POST", API_PATH + "datasets/reload-schema", {
            projectKey, datasetName
        });
    },
    listNotebooks: function(projectKey, datasetSmartName) {
        return APIXHRService("POST", API_PATH + "datasets/list-notebooks", {
            projectKey: projectKey, datasetSmartName: datasetSmartName
        });
    },
    setVirtualizable: function(items, virtualizable) {
        return APIXHRService("POST", API_PATH + "datasets/set-virtualizable", {
            items: JSON.stringify(items),
            virtualizable: virtualizable
        });
    },
    setAutoCountOfRecords: function(items, autoCountOfRecords) {
        return APIXHRService("POST", API_PATH + "datasets/set-auto-count-of-records", {
            items: JSON.stringify(items),
            autoCountOfRecords: autoCountOfRecords
        });
    },
    managedSQL: {
        test: function(projectKey, dataset, maxSamples, connectionOnly) {
            return APIXHRService("POST", API_PATH + "datasets/managed-sql/test/", {projectKey: projectKey, data: JSON.stringify(dataset),
                    maxSamples: maxSamples, connectionOnly: connectionOnly });
        },
        testDatasetStorageConflicts: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/managed-sql/test-dataset-storage-conflicts/", {projectKey, data: JSON.stringify(dataset)});
        },
        createTable: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/managed-sql/create-table/", {projectKey: projectKey, data: JSON.stringify(dataset)});
        },
        dropTable: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/managed-sql/drop-table/", {projectKey: projectKey, data: JSON.stringify(dataset)});
        }
    },
    externalSQL: {
        test: function(projectKey, dataset, maxSamples, listTables, testTableOrQuery, getTableMetadata) {
            return APIXHRService("POST", API_PATH + "datasets/external-sql/test/", {
                projectKey: projectKey,
                data: JSON.stringify(dataset),
                maxSamples: maxSamples,
                listTables: listTables,
                testTableOrQuery: testTableOrQuery,
                getTableMetadata: getTableMetadata
            });
        },
        listPartitions: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/external-sql/list-partitions/", {projectKey: projectKey, data: JSON.stringify(dataset)});
        },
        // returns a Future. Typically debounced and pending futures canceled using MonoFuture.
        preview: function(dataset) {
            return APIXHRService("POST", API_PATH + "datasets/external-sql/preview/", {
                data: JSON.stringify(dataset)
            });
        }
    },
    SQL: {
        checkSparkIntegration: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/sql/check-spark-integration/", {
                projectKey: projectKey,
                data: JSON.stringify(dataset)
            });
        }
    },
    elasticsearch: {
        test: function(projectKey, dataset, connectionOnly) {
            return APIXHRService("POST", API_PATH + "datasets/elasticsearch/test", {
                projectKey: projectKey,
                data: JSON.stringify(dataset),
                connectionOnly: connectionOnly
            });
        },
        listIndices: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/elasticsearch/list-indices", {
                projectKey: projectKey,
                data: JSON.stringify(dataset),
            });
        },
        // returns a Future. Typically debounced and pending futures canceled using MonoFuture.
        preview: function(dataset) {
            return APIXHRService("POST", API_PATH + "datasets/elasticsearch/preview/", {
                data: JSON.stringify(dataset)
            });
        }
    },
    mongoDB: {
        test: function(projectKey, dataset, connectionOnly, inferStorageTypes) {
            return APIXHRService("POST", API_PATH + "datasets/mongodb/test", {projectKey: projectKey, data: JSON.stringify(dataset),
                connectionOnly: connectionOnly,
                inferStorageTypes: inferStorageTypes == null ? false : inferStorageTypes });
        },
        createCollection: function(dataset) {
            return APIXHRService("POST", API_PATH + "datasets/mongodb/create-collection", {data: JSON.stringify(dataset) });
        },
        deleteCollection: function(dataset) {
            return APIXHRService("POST", API_PATH + "datasets/mongodb/delete-collection", {data: JSON.stringify(dataset) });
        }
    },
    sharepointOnlineList: {
        test: function(projectKey, dataset, connectionOnly, inferStorageTypes) {
            return APIXHRService("POST", API_PATH + "datasets/sharepointonlinelist/test", {projectKey: projectKey, data: JSON.stringify(dataset),
                            connectionOnly: connectionOnly,
                            inferStorageTypes: inferStorageTypes == null ? false : inferStorageTypes });
        },
    },
    dynamoDB: {
            test: function(projectKey, dataset, connectionOnly, inferStorageTypes, listTables) {
                return APIXHRService("POST", API_PATH + "datasets/dynamodb/test", {projectKey: projectKey, data: JSON.stringify(dataset),
                    connectionOnly: connectionOnly,
                    listTables: listTables,
                    inferStorageTypes: inferStorageTypes == null ? false : inferStorageTypes });
            },
            createTable: function(dataset) {
                return APIXHRService("POST", API_PATH + "datasets/dynamodb/create-table", {data: JSON.stringify(dataset) });
            },
            deleteTable: function(dataset) {
                return APIXHRService("POST", API_PATH + "datasets/dynamodb/delete-table", {data: JSON.stringify(dataset) });
            },
            updateIndex: function(dataset) {
                            return APIXHRService("POST", API_PATH + "datasets/dynamodb/update-index", {data: JSON.stringify(dataset) });
            }
    },
    cassandra: {
        test: function(projectKey, dataset, connectionOnly) {
            return APIXHRService("POST", API_PATH + "datasets/cassandra/test", {projectKey: projectKey, data: JSON.stringify(dataset),
                connectionOnly: connectionOnly });
        },
        createTable: function(dataset) {
            return APIXHRService("POST", API_PATH + "datasets/cassandra/create-table", {data: JSON.stringify(dataset) });
        },
        dropTable: function(dataset) {
            return APIXHRService("POST", API_PATH + "datasets/cassandra/drop-table", {data: JSON.stringify(dataset) });
        }
    },
    editable: {
        getData: function(projectKey, dataset) {
            return APIXHRService("GET", API_PATH + "datasets/editable/get", {
                projectKey: projectKey,
                dataset: dataset
            });
        },
        save: function(projectKey, dataset, data) {
            return APIXHRService("POST", API_PATH + "datasets/editable/save", {
                projectKey: projectKey,
                dataset: dataset,
                data: data
            });
        },
        import: function(projectKey, dataset, data) {
            return APIXHRService("POST", API_PATH + "datasets/editable/import", {
                projectKey: projectKey,
                dataset: dataset, data: data
            });
        },
        test: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/editable/test", {
                projectKey: projectKey,
                dataset: dataset
            });
        }
    },
    customDataset: {
        test: function(projectKey, dataset, showPreview) {
            return APIXHRService("POST", API_PATH + "datasets/custom-dataset/test", {
                projectKey: projectKey,
                dataset: JSON.stringify(dataset),
                showPreview: showPreview
            })
        }
    },
    jobsdb: {
        test: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/jobsdb/test/", {projectKey: projectKey, data: JSON.stringify(dataset)});
        }
    },
    labels: {
        test: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/labels/test", {projectKey: projectKey, data: JSON.stringify(dataset)});
        }
    },
    statsdb: {
        test: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/statsdb/test/", {projectKey: projectKey, data: JSON.stringify(dataset)});
        }
    },
    experimentsdb: {
        test: function(projectKey, dataset) {
            return APIXHRService("POST", API_PATH + "datasets/experimentsdb/test/", {projectKey: projectKey, data: JSON.stringify(dataset)});
        }
    },
    /* OpenStreetMap specific */
    osm: {
        get_data_files: function() {
            return APIXHRService("GET", API_PATH + "datasets/osm/get-data-files", {});
        }
    },
    upload: {
        /* Returns {id: uploadBoxId } */
        createUploadBox: function() {
            return APIXHRService("POST", API_PATH + "datasets/upload/create-upload-box", {});
        },
        /* This method should only be called in case of premature abort.
         * In case of normal dataset creation, the upload box is automatically purged */
        deleteUploadBox: function(uploadBoxId) {
            return APIXHRService("POST", API_PATH + "datasets/upload/delete-upload-box", {
                uploadBoxId: uploadBoxId
            });
        },
        createDataset: function(projectKey, datasetName, uploadBoxId) {
            return APIXHRService("POST", API_PATH + "datasets/upload/create-dataset", {
                projectKey: projectKey,
                datasetName:datasetName,
                uploadBoxId: uploadBoxId
            });
        },
        /* Returns [ {path, length}] */
        listFiles: function(projectKey, datasetName) {
            return APIXHRService("GET", API_PATH + "datasets/upload/list-files", {projectKey: projectKey, datasetName:datasetName});
        },
        /* Returns [ {path, length}] */
        addFileToDataset: function(projectKey, file, dataset, callback) {
            return uploadFileRequest("datasets/upload/add-file", function(formdata) {
                formdata.append("projectKey", projectKey);
                formdata.append("file", file);
                formdata.append('dataset', JSON.stringify(dataset));
            }, callback);
        },
        removeFile: function(projectKey, dataset, path) {
            return APIXHRService("POST", API_PATH + "datasets/upload/remove-file", {
                projectKey: projectKey,
                dataset:JSON.stringify(dataset),
                fileName:path
            });
        }
    },
    getRequirements: function(projectKey, datasetType) {
        return APIXHRService("GET", API_PATH + "datasets/get-requirements", {
            projectKey:projectKey,
            datasetType:datasetType
        });
    },
    listAvailableMetrics: function(projectKey, datasetName) {
        return APIXHRService("GET", API_PATH + "datasets/list-available-metrics", {
            projectKey: projectKey,
            datasetName: datasetName
        });
    },
    saveMetrics: function(projectKey, datasetName, metrics, checks) {
        return APIXHRService("POST", API_PATH + "datasets/save-metrics", {
            projectKey: projectKey,
            datasetName: datasetName,
            metricsData: JSON.stringify(metrics),
            checksData: JSON.stringify(checks)
        });
    },
    computeMetrics: function(projectKey, datasetName, partitionId, allPartitions) {
        return APIXHRService("GET", API_PATH + "datasets/compute-metrics", {
            projectKey: projectKey,
            datasetName: datasetName,
            partitionId: partitionId,
            allPartitions: allPartitions
        });
    },
    computeColumnMetrics : function(projectKey, datasetName, columnName, partitionId, allPartitions) {
        return APIXHRService("GET", API_PATH + "datasets/compute-column-metrics",{
            projectKey : projectKey,
            datasetName : datasetName,
            columnName : columnName,
            partitionId : partitionId,
            allPartitions: allPartitions
        });
    },
    computeDetailedColumnMetrics : function(projectKey, datasetName, columnName, statisticsConfig, partitionId, forceRefresh) {
        return APIXHRService("GET", API_PATH + "datasets/compute-detailed-column-metrics",{
            projectKey : projectKey,
            datasetName : datasetName,
            columnName : columnName,
            statisticsConfig : JSON.stringify(statisticsConfig),
            partitionId : partitionId,
            forceRefresh : forceRefresh
        });
    },
    computePlan: function(projectKey, datasetName, metrics) {
        return APIXHRService("POST", API_PATH + "datasets/compute-plan", {
            projectKey: projectKey,
            datasetName: datasetName,
            metricsData: JSON.stringify(metrics)
        });
    },
    computeProbe: function(projectKey, datasetName, partitionId, allPartitions, metrics) {
        return APIXHRService("POST", API_PATH + "datasets/compute-probe", {
            projectKey: projectKey,
            datasetName: datasetName,
            partitionId: partitionId,
            allPartitions: allPartitions,
            data: JSON.stringify(metrics)
        });
    },
    runChecks: function(projectKey, datasetName, partitionId, allPartitions) {
    // TODO the backend route has been deleted, to clean up
        return APIXHRService("GET", API_PATH + "datasets/run-checks", {
            projectKey: projectKey,
            datasetName: datasetName,
            partitionId: partitionId,
            allPartitions: allPartitions
        });
    },
    runCheck: function(projectKey, datasetName, partitionId, allPartitions, metricsChecks) {
        // TODO the backend route has been deleted, to clean up
        return APIXHRService("POST", API_PATH + "datasets/run-check", {
            projectKey: projectKey,
            datasetName: datasetName,
            partitionId: partitionId,
            allPartitions: allPartitions,
            data: JSON.stringify(metricsChecks)
        });
    },
    getPreparedMetricHistory: function(projectKey, datasetName, partitionId, metric, metricId) {
        return APIXHRService("GET", API_PATH + "datasets/get-prepared-metric-history", {
            projectKey: projectKey,
            datasetName: datasetName,
            partitionId: partitionId,
            data: JSON.stringify(metric),
            metricId: metricId
        });
    },
    getPreparedMetricHistories: function(projectKey, datasetName, displayedState) {
        return APIXHRService("POST", API_PATH + "datasets/get-prepared-metric-histories", {
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(displayedState || {})
        });
    },
    getPreparedMetricPartitions: function(projectKey, datasetName, displayedState) {
        return APIXHRService("POST", API_PATH + "datasets/get-prepared-metric-partitions", {
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(displayedState || {})
        });
    },
    getPreparedMetricColumns : function(projectKey, datasetName, displayedState) {
        return APIXHRService("POST", API_PATH + "datasets/get-prepared-metric-columns",{
            projectKey : projectKey,
            datasetName : datasetName,
            data : JSON.stringify(displayedState || {})
        });
    },
    listComputedMetrics: function(projectKey, datasetName) {
        return APIXHRService("GET", API_PATH + "datasets/list-computed-metrics", {
            projectKey: projectKey,
            datasetName: datasetName
        });
    },
    getPartitionListMetric: function(projectKey, datasetName ) {
        return APIXHRService("GET", API_PATH + "datasets/get-partition-list-metric", {
            projectKey: projectKey,
            datasetName: datasetName
        });
    },
    refreshPartitionListMetric: function(projectKey, datasetName ) {
        return APIXHRService("GET", API_PATH + "datasets/refresh-partition-list-metric", {
            projectKey: projectKey,
            datasetName: datasetName
        });
    },
    createMetricsDataset: function(projectKey, datasetName, view, partition, filter) {
        return APIXHRService("GET", API_PATH + "datasets/create-metrics-dataset", {
            projectKey: projectKey,
            objectId: datasetName,
            view: view,
            partition: partition,
            filter: filter,
            taggableType: "DATASET"
        });
    },
    getCachedNbRecords: function(projectKey, datasetName) {
        return APIXHRService("GET", API_PATH + "datasets/get-cached-nb-records", {
            projectKey: projectKey,
            datasetName: datasetName
        });
    },
    updateNbRecords: function(projectKey, datasetName, recomputeAll) {
        return APIXHRService("GET", API_PATH + "datasets/update-nb-records", {
            projectKey: projectKey,
            datasetName: datasetName,
            recomputeAll: recomputeAll
        });
    },
    getCheckHistories: function(projectKey, datasetName, displayedState) {
        return APIXHRService("POST", API_PATH + "datasets/get-prepared-check-histories", {
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(displayedState || {})
        });
    },
    listComputedChecks: function(projectKey, datasetName) {
        return APIXHRService("GET", API_PATH + "datasets/list-computed-checks", {
            projectKey: projectKey,
            datasetName: datasetName
        });
    },
    getHint: function(projectKey, datasetName, probe) {
        return APIXHRService("POST", API_PATH + "datasets/get-hint", {
            projectKey: projectKey,
            datasetName: datasetName,
            data: JSON.stringify(probe)
        });
    },
    clearMetrics: function(projectKey, datasetName, partition) {
        return APIXHRService("GET", API_PATH + "datasets/clear-metrics", {
            projectKey: projectKey,
            datasetName: datasetName,
            partition: partition
        });
    },
    getFullSampleStatisticsConfig : function(contextProjectKey, projectKey, datasetName) {
        return APIXHRService("GET", API_PATH + "datasets/get-full-sample-statistics-config",{
            contextProjectKey: contextProjectKey,
            projectKey : projectKey,
            datasetName : datasetName
        });
    },
    setDatasetDescriptions: function(projectKey, datasetName, shortDesc, longDesc, colDesc) {
        return APIXHRService("POST", API_PATH + "datasets/set-dataset-descriptions", {
            projectKey: projectKey,
            datasetName: datasetName,
            shortDesc: shortDesc,
            longDesc: longDesc,
            colDesc: JSON.stringify(colDesc)
        });
    },
},
statistics: {
    get: function(projectKey, worksheetId) {
        return APIXHRService("GET", API_PATH + '/eda/worksheets/get-worksheet', {
            projectKey: projectKey,
            worksheetId: worksheetId
        });
    }
},
featureStore: {
    setFeatureGroup: function(projectKey, name, isFeatureGroup) {
        return APIXHRService("POST", API_PATH + "feature-store/feature-groups/set", {
            projectKey: projectKey,
            name: name,
            isFeatureGroup: isFeatureGroup
        });
    }
},
jupyterNotebooks: {
    get: function(projectKey, notebookName, kernelName) {
        if (kernelName) {
            return APIXHRService("GET", "/jupyter/api/contents/" + projectKey + "/" + notebookName + ".ipynb?kernel_name=" + kernelName);
        } else {
            return APIXHRService("GET", "/jupyter/api/contents/" + projectKey + "/" + notebookName + ".ipynb");
        }
    },
    getNotebook: function(projectKey, notebookName, kernelName) {
        return APIXHRService("GET", API_PATH + "jupyter/get-notebook", {projectKey:projectKey, name:notebookName, kernelName:kernelName});
    },
    createOnDataset: function(projectKey, datasetSmartName, language) {
        return APIXHRService("POST", API_PATH + "jupyter/new-dataset-notebook", {
            projectKey: projectKey,
            datasetSmartName: datasetSmartName,
            language: language
        });
    },
    create: function(projectKey, language) {
        return APIXHRService("POST", API_PATH + "jupyter/new-notebook", {
            projectKey: projectKey,
            language: language
        });
    },
    unload: function(session_id) {
        return APIXHRService("DELETE", JUPYTER_API_PATH + "api/sessions/" + session_id, {
        });
    },
    clearOutputs: function(projectKey, notebookName) {
        return APIXHRService("POST", API_PATH + "jupyter/clear-notebook-outputs/", { projectKey: projectKey, notebookName: notebookName});
    },
    listHeads: function(projectKey, tagFilter) {
        return APIXHRService("GET", API_PATH + "jupyter/list-heads", {projectKey: projectKey, tagFilter: tagFilter});
    },
    listAccessible: function() {
        return APIXHRService("GET", API_PATH + "jupyter/list-accessible-notebooks");
    },
    listAll: function() {
        return APIXHRService("GET", API_PATH + "jupyter/list-all-notebooks");
    },
    mapNotebooksToExports: function(projectKey) {
        return APIXHRService("GET", API_PATH + "jupyter/map-notebook-exports", {projectKey: projectKey});
    },
    newNotebookWithTemplate: function(projectKey, baseName, templateDesc, codeEnv, containerConf) {
        return APIXHRService("POST", API_PATH + "jupyter/new-notebook-with-template", {
            projectKey: projectKey,
            baseName: baseName,
            templateDesc: JSON.stringify(templateDesc),
            codeEnv:codeEnv,
            containerConf:containerConf
        });
    },
    newNotebookFromFile: function(projectKey, baseName, language, datasetSmartName, file) {
        return uploadFileRequest('jupyter/new-notebook-from-file', formdata => {
            formdata.append('projectKey', projectKey);
            formdata.append('baseName', baseName);
            formdata.append('language', language);
            if (datasetSmartName) {
                formdata.append('datasetSmartName', datasetSmartName);
            }
            formdata.append('file', file);
        }, null);
    },
    newNotebookForDataset: function(projectKey, baseName, datasetSmartName, templateDesc, codeEnv, containerConf) {
        return APIXHRService("POST", API_PATH + "jupyter/new-notebook-for-dataset", {
            projectKey: projectKey,
            baseName: baseName,
            datasetSmartName: datasetSmartName,
            templateDesc: JSON.stringify(templateDesc),
            codeEnv:codeEnv,
            containerConf:containerConf
        });
    },
    newNotebookForKB: function(projectKey, baseName, knowledgeBankRef, templateDesc) {
        return APIXHRService("POST", API_PATH + "jupyter/new-notebook-for-kb", {
            projectKey: projectKey,
            baseName: baseName,
            knowledgeBankRef: knowledgeBankRef,
            templateDesc: JSON.stringify(templateDesc),
        });
    },
    createRecipeFromNotebook: function(projectKey, notebook, recipe) {
        return APIXHRService("POST", API_PATH + "jupyter/create-recipe-from-notebook", {
            projectKey: projectKey,
            notebook: notebook,
            recipe: JSON.stringify(recipe)
        });
    },
    saveBackToRecipe: function(projectKey, notebook) {
        return APIXHRService("POST", API_PATH + "jupyter/save-back-to-recipe", {
            projectKey: projectKey,
            notebook: notebook
        });
    },
    export: { // Should mostly stay in sync with reports.snapshots
        create: function(projectKey, notebookSmartName, execute) {
            return APIXHRService("POST", API_PATH + "jupyter/export/create", {
                projectKey: projectKey,
                notebookSmartName: notebookSmartName,
                execute: !!execute
            });
        },
        get: function(projectKey, notebookSmartName, timestamp) {
            return APIXHRService("GET", API_PATH + "jupyter/export/get", {
                projectKey: projectKey,
                notebookSmartName: notebookSmartName,
                timestamp: timestamp
            });
        },
        getLast: function(projectKey, notebookSmartName) {
            return APIXHRService("GET", API_PATH + "jupyter/export/get-last", {
                projectKey: projectKey,
                notebookSmartName: notebookSmartName
            });
        },
        list: function(projectKey, notebookSmartName, withContent) {
            return APIXHRService("GET", API_PATH + "jupyter/export/list", {
                projectKey: projectKey,
                notebookSmartName: notebookSmartName,
                withContent: withContent
            });
        }
    },
    git: {
        listRemoteNotebooks: function(repository, login, password, ref) {
            return APIXHRService("POST", API_PATH + "jupyter/git/list-remote-notebooks", { repository, login, password, ref });
        },
        importNotebooks: function(projectKey, repository, login, password, ref, notebooks) {
            return APIXHRService("POST", API_PATH + "jupyter/git/import-remote-notebooks", {
                projectKey,
                repository,
                login,
                password,
                ref,
                notebooks: JSON.stringify(notebooks)
            });
        },
        getConflictingNotebooks: function(projectKey, notebooks, checkPull = false) {
            return APIXHRService("GET", API_PATH + "jupyter/git/get-conflicting-notebooks", {projectKey, checkPull, notebooks: JSON.stringify(notebooks)});
        },
        pushNotebooksToGit: function(projectKey, notebooks, message) {
            return APIXHRService("POST", API_PATH + "jupyter/git/push-remote-notebooks", {projectKey, message, notebooksWithHash: JSON.stringify(notebooks.map(n => ({notebookName: n.notebookName, remoteHashFileDuringConflict: n.remoteHashFileDuringConflict})))});
        },
        pullNotebooks: function(projectKey, notebooks) {
            return APIXHRService("POST", API_PATH + "jupyter/git/pull-remote-notebooks", {projectKey, notebooks: JSON.stringify(notebooks)});
        },
        editReference: function(projectKey, notebook, gitReference) {
            return APIXHRService("POST", API_PATH + "jupyter/git/edit-reference", {projectKey, notebook, reference: JSON.stringify(gitReference)});
        },
        unlinkReference: function(projectKey, notebook) {
            return APIXHRService("POST", API_PATH + "jupyter/git/unlink-reference", {projectKey, notebook});
        },
    }
},
continuousActivities: {
    start : function(projectKey, recipeId, loopParams) {
        return APIXHRService("POST", API_PATH + "continuous-activities/start", {projectKey: projectKey, recipeId:recipeId, loopParams:JSON.stringify(loopParams)});
    },
    stop : function(projectKey, recipeId) {
        return APIXHRService("POST", API_PATH + "continuous-activities/stop", {projectKey: projectKey, recipeId:recipeId});
    },
    listProjectStates: function(projectKey) {
        return APIXHRService("GET", API_PATH + "continuous-activities/list-project-states", {projectKey: projectKey});
    },
    getState: function(projectKey, continuousActivityId) {
        return APIXHRService("GET", API_PATH + "continuous-activities/get-state", {
            projectKey: projectKey, continuousActivityId: continuousActivityId
        }, 'nospinner');
    },
    getStates: function(projectKey) {
        return APIXHRService("GET", API_PATH + "continuous-activities/get-states", {
            projectKey: projectKey
        });
    },
    getFullInfo: function(projectKey, continuousActivityId) {
        return APIXHRService("GET", API_PATH + "continuous-activities/get-full-info", {
            projectKey: projectKey, continuousActivityId: continuousActivityId
        });
    },
    listLastRuns : function(projectKey, continuousActivityId) {
        return APIXHRService("GET", API_PATH + "continuous-activities/list-last-runs", {
            projectKey: projectKey, continuousActivityId: continuousActivityId
        });
    },
    listRunLastAttempts: function(projectKey, continuousActivityId, runId) {
        return APIXHRService("GET", API_PATH + "continuous-activities/list-run-last-attempts", {
            projectKey: projectKey, continuousActivityId: continuousActivityId,
            runId: runId,
        });
    },
    smartTailAttemptLog: function(projectKey, continuousActivityId, runId, attemptId) {
        return APIXHRService("GET", API_PATH + "continuous-activities/smart-tail-attempt-log", {
            projectKey: projectKey, continuousActivityId: continuousActivityId,
            runId: runId, attemptId: attemptId
        });
    },
    getGlobalStatus : function(){
        return APIXHRService("GET", API_PATH + "continuous-activities/get-global-status")
    },
    getDownloadURL: function(projectKey, continuousActivityId, runId, attemptId) {
        return API_PATH + "continuous-activities/cat-attempt-log?"
                 + "projectKey=" + encodeURIComponent(projectKey)
                 + "&continuousActivityId=" + encodeURIComponent(continuousActivityId)
                 + "&runId=" + encodeURIComponent(runId)
                 + "&attemptId=" + encodeURIComponent(attemptId);
    },
},
streamingEndpoints:  {
    listHeads: function(projectKey) {
        return APIXHRService("GET", API_PATH + "streaming-endpoints/list", {projectKey:projectKey})
    },
    listNames: function(projectKey) {
        return APIXHRService("GET", API_PATH + "streaming-endpoints/list-names/", {projectKey: projectKey});
    },
    create:function(projectKey, streamingEndpoint) {
      return APIXHRService("POST", API_PATH + "streaming-endpoints/create", {
            projectKey:projectKey,
            streamingEndpoint: JSON.stringify(streamingEndpoint
        )})
    },
    get: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "streaming-endpoints/get", {projectKey:projectKey, id:id})
    },
    save: function(projectKey, streamingEndpoint, saveInfo) {
        return APIXHRService("POST", API_PATH + "streaming-endpoints/save", {
            projectKey:projectKey,
            streamingEndpoint: JSON.stringify(streamingEndpoint),
            saveInfo:JSON.stringify(saveInfo)
        })
    },
    getFullInfo: function(projectKey, id) {
        return APIXHRService("GET", API_PATH + "streaming-endpoints/get-full-info", {projectKey:projectKey, id:id})
    },
    collectSample: function(projectKey, streamingEndpoint, limit, timeout, inferStorageTypes) {
        return APIXHRService("GET", API_PATH + "streaming-endpoints/collect-sample", {projectKey:projectKey, streamingEndpoint:JSON.stringify(streamingEndpoint), limit:limit, timeout:timeout, inferStorageTypes:inferStorageTypes})
    },
    testKafka: function(projectKey, streamingEndpoint) {
        return APIXHRService("POST", API_PATH + "streaming-endpoints/test-kafka", {projectKey:projectKey, streamingEndpoint:JSON.stringify(streamingEndpoint)})
    },
    testSQS: function(projectKey, streamingEndpoint) {
        return APIXHRService("POST", API_PATH + "streaming-endpoints/test-sqs", {projectKey:projectKey, streamingEndpoint:JSON.stringify(streamingEndpoint)})
    },
    testHttpSSE: function(projectKey, streamingEndpoint) {
        return APIXHRService("POST", API_PATH + "streaming-endpoints/test-httpsse", {projectKey:projectKey, streamingEndpoint:JSON.stringify(streamingEndpoint)})
    },
    syncKsql: function(projectKey, streamingEndpointId, terminateQueries) {
        return APIXHRService("POST", API_PATH + "streaming-endpoints/sync-ksql", {projectKey:projectKey, streamingEndpointId:streamingEndpointId, terminateQueries:terminateQueries || false})
    },
    fetchKafkaSchema: function(projectKey, streamingEndpoint) {
        return APIXHRService("POST", API_PATH + "streaming-endpoints/fetch-kafka-schema", {projectKey:projectKey, streamingEndpoint:JSON.stringify(streamingEndpoint)})
    }
},
flow: {
    zones: {
        create: (projectKey, name, color) => {
            return APIXHRService("POST", API_PATH + "flow/create-zone", { projectKey, name, color});
        },
        moveItems: (projectKey, zoneId, movingItems) => {
            return APIXHRService("POST", API_PATH + "flow/move-to-zone", { projectKey, zoneId, movingItems: JSON.stringify(movingItems)});
        },
        shareItems: (projectKey, zoneId, sharingItems) => {
            return APIXHRService("POST", API_PATH + "flow/share-to-zone", { projectKey, zoneId, sharingItems: JSON.stringify(sharingItems)});
        },
        unshareItems: (projectKey, zoneIds, sharingItems) => {
            return APIXHRService("POST", API_PATH + "flow/unshare-to-zone", { projectKey, zoneIds: JSON.stringify(zoneIds), sharingItems: JSON.stringify(sharingItems)});
        },
        list: projectKey => {
            return APIXHRService("GET", API_PATH + "flow/zones", { projectKey });
        },
        edit: (projectKey, id, name, color) => {
            return APIXHRService("POST", API_PATH + "flow/edit-zone", { projectKey, zoneId: id, newName: name, newColor: color});
        },
        delete: (projectKey, id) => {
            return APIXHRService("POST", `${API_PATH}flow/zones/${id}/delete`, { projectKey });
        },
        getZoneId: (projectKey, item) => {
            return APIXHRService("POST", `${API_PATH}flow/retrieve-zone-item`, { projectKey, item: JSON.stringify(item) });
        },
        savePosition: (projectKey, id, x, y) => {
            return APIXHRService("POST", API_PATH + "flow/save-zone-position", { projectKey, zoneId: id, x, y}, "nospinner");
        },
        setManualPositioningSetting: (projectKey, zonesManualPositioning) => {
            return APIXHRService("POST", API_PATH + "flow/set-zones-manual-positioning", { projectKey, zonesManualPositioning}, "nospinner");
        },
        resetPositions: (projectKey) => {
            return APIXHRService("POST", API_PATH + "flow/reset-zone-positions", { projectKey });
        },
    },
    jobs: {
        /* ********** Status ************ */
        listLastJobs: function(projectKey, nb) {
            return APIXHRService("GET", API_PATH + "flow/jobs/list-last-jobs", {projectKey: projectKey, nb:nb});
        },
        listRunningJobs: function() {
            return APIXHRService("GET", API_PATH + "flow/jobs/list-running-jobs");
        },
        listAllRunningJobs: function() {
            return APIXHRService("GET", API_PATH + "flow/jobs/list-all-running-jobs");
        },
        getJobStatus: function(projectKey, jobId) {
            return APIXHRService("GET", API_PATH + "flow/jobs/get-job-status", {
                projectKey: projectKey,
                jobId: jobId,
            });
        },
        fetchYarnLogs: function(projectKey, jobId) {
            return APIXHRService("GET", API_PATH + "flow/jobs/fetch-yarn-logs", {
                projectKey: projectKey,
                jobId: jobId,
            });
        },
        getJobGraphURL: function(projectKey, jobId, type) {
            return API_PATH + "flow/jobs/get-job-graph?jobId=" + jobId + "&projectKey=" + projectKey+ "&type=" + type;
        },
        getJobGraphData: function(projectKey, jobId, type) {
            return APIXHRService("GET", API_PATH +  "flow/jobs/get-job-graph?jobId=" + jobId + "&projectKey=" + projectKey+ "&type=" + type);
        },
        tailJobLog: function(projectKey, jobId) {
            return APIXHRService("GET", API_PATH + "flow/jobs/tail-job-log", {
                projectKey: projectKey,
                jobId: jobId,
            });
        },
        getJobDiagnosisURL: function(projectKey, jobId) {
            return API_PATH + "flow/jobs/download-job-diagnosis?"
                     + "projectKey=" + encodeURIComponent(projectKey)
                     + "&jobId=" + encodeURIComponent(jobId);
        },
        smartTailActivityLog: function(projectKey, jobId, activityId, nbLines) {
            return APIXHRService("GET", API_PATH + "flow/jobs/smart-tail-activity-log", {
                projectKey: projectKey,
                jobId: jobId, activityId: activityId, nbLines: nbLines
            });
        },
        smartTailActivityAdditionalLog: function(projectKey, jobId, activityId, path, nbLines) {
            return APIXHRService("GET", API_PATH + "flow/jobs/smart-tail-activity-additional-log",
                {projectKey, jobId, activityId, path, nbLines});
        },
        startSparkHistoryServer: function(projectKey, jobId, activityId) {
            return APIXHRService("GET", API_PATH + "flow/jobs/start-spark-history-server",
                {projectKey, jobId, activityId});
        },

        /* ************* Execution *********** */
        startPreview: function(data) {
            return APIXHRService("POST", API_PATH + "flow/jobs/start-preview/", {
                data: JSON.stringify(data)
            });
        },
        getPreviewResult: function(jobProjectKey, jobId) {
            return APIXHRService("GET", API_PATH + "flow/jobs/get-preview-result", {
                jobProjectKey: jobProjectKey,
                jobId: jobId
            });
        },
        validateRunFully: function(jobProjectKey, jobId, skippedActivityIds) {
            return APIXHRService("POST", API_PATH + "flow/jobs/validate-run-fully", {
                jobProjectKey: jobProjectKey,
                jobId: jobId,
                skippedActivityIds: JSON.stringify(skippedActivityIds)
            });
        },
        start: function(data) {
            return APIXHRService("POST", API_PATH + "flow/jobs/start/", {
                data: JSON.stringify(data)
            });
        },
        abort: function(jobProjectKey, jobId) {
            return APIXHRService("POST", API_PATH + "flow/jobs/abort/", {
                jobProjectKey: jobProjectKey,
                jobId: jobId
            });
        },
        retry: function(jobProjectKey, jobId) {
            return APIXHRService("POST", API_PATH + "flow/jobs/retry/", {
                jobProjectKey: jobProjectKey,
                jobId: jobId
            });
        },
        /* ***************** List handling *************** */
        clearLogsWithFilter: function(jobProjectKey, filter) {
            return APIXHRService("POST", API_PATH + "flow/jobs/clear-logs-with-filter/", {
                jobProjectKey: jobProjectKey,
                filter:filter
            });
        },
        clearLogs: function(jobProjectKey, jobId) {
            return APIXHRService("POST", API_PATH + "flow/jobs/clear-logs/", {
                jobProjectKey: jobProjectKey,
                jobId:jobId
            });
        }
    },
    status: {
        getDailyPartitionedDatasets: function() {
            return APIXHRService("GET", API_PATH + "flow/status/get-daily-partitioned-datasets");
        },
        initMultiStatusOperation: function() {
            return APIXHRService("GET", API_PATH + "flow/status/init-multi-status-operation");
        },
        getDatasetPartitionStatusMulti: function(operationId, dataset, partition) {
            return APIXHRService("GET", API_PATH + "flow/status/get-dataset-partition-status-multi", {
                operationId: operationId,
                dataset:dataset,
                partition:partition
            });
        }
    },
    snippets: {
        getSnippets: function(snippetType, categories) {
            return APIXHRService("GET", API_PATH + "flow/snippets/get", {snippetType: snippetType, categories: categories.join(",")});
        },
        saveSnippet: function(snippetAsString, snippetType, category) {
            return APIXHRService("POST", API_PATH + "flow/snippets/save", {data: snippetAsString, snippetType: snippetType, category: category});
        },
        deleteSnippet: function(snippetId, snippetType) {
            return APIXHRService("POST", API_PATH + "flow/snippets/delete", {snippetId: snippetId, snippetType: snippetType});
        },
        getUserSnippetPermission: function(snippetId, snippetType) {
            return APIXHRService("GET", API_PATH + "flow/snippets/get-user-snippet-permission", { snippetId: snippetId, snippetType: snippetType });
        },
    },
    refactoring: {
        startChangeConnections: function(projectKey, items) {
            // POST because request might be big
            return APIXHRService("POST", API_PATH + "flow/refactoring/start-change-connections", {
                projectKey: projectKey,
                items: JSON.stringify(items)
            });
        },
        testChangeConnections: function(projectKey, items, options) {
            return APIXHRService("POST", API_PATH + "flow/refactoring/test-change-connections", {
                projectKey: projectKey,
                items: JSON.stringify(items),
                options: JSON.stringify(options)
            });
        },
        changeConnections: function(projectKey, items, options) {
            return APIXHRService("POST", API_PATH + "flow/refactoring/change-connections", {
                projectKey: projectKey,
                items: JSON.stringify(items),
                options: JSON.stringify(options)
            });
        },
        startCopySubFlow: function(items) {
            // POST because request might be big
            return APIXHRService("POST", API_PATH + "flow/refactoring/start-copy-subflow", {
                items: JSON.stringify(items)
            });
        },
        testCopySubFlow: function(items, options, contextProjectKey) {
            // POST because request might be big
            return APIXHRService("POST", API_PATH + "flow/refactoring/test-copy-subflow", {
                items: JSON.stringify(items),
                options: JSON.stringify(options),
                contextProjectKey: contextProjectKey
            });
        },
        copySubFlow: function(items, options, contextProjectKey) {
            return APIXHRService("POST", API_PATH + "flow/refactoring/copy-subflow", {
                items: JSON.stringify(items),
                options: JSON.stringify(options),
                contextProjectKey: contextProjectKey
            });
        }
    },
    tools: {
        start: function(projectKey, type, data) {
            return APIXHRService("POST", API_PATH + "flow/tools/start", {
                projectKey: projectKey,
                type: type,
                data: JSON.stringify(data)
            });
        },
        getStateNoSession: function(projectKey, tool, options, spinnerMode) {
            return APIXHRService("GET", API_PATH + "flow/tools/get-state-no-session", {
                projectKey: projectKey,
                tool: tool,
                options: JSON.stringify(options)
            }, spinnerMode);
        },
        stop: function(projectKey, toolId) {
            return APIXHRService("POST", API_PATH + "flow/tools/stop", {
                projectKey: projectKey,
                toolId: toolId
            });
        },
        getSessions: function(projectKey) {
            return APIXHRService("GET", API_PATH + "flow/tools/get-sessions", {
                projectKey: projectKey
            });
        },
        setActive: function(projectKey, toolId) {
            return APIXHRService("POST", API_PATH + "flow/tools/set-active", {
                projectKey: projectKey,
                toolId: toolId
            });
        },
        setDefaultActive: function(projectKey) {
            return APIXHRService("POST", API_PATH + "flow/tools/set-default-active", {
                projectKey: projectKey
            });
        },
        getState: function(projectKey, tool, options, spinnerMode) {
            return APIXHRService("GET", API_PATH + "flow/tools/get-state", {
                projectKey: projectKey,
                tool: tool,
                options: JSON.stringify(options)
            }, spinnerMode);
        },
        setFocused: function(projectKey, focused, mode) {
            return APIXHRService("POST", API_PATH + "flow/tools/set-focused", {
                projectKey: projectKey,
                focused: JSON.stringify(focused),
                mode : mode
            }, 'nospinner');
        },
        startUpdate: function(projectKey, tool, updateOptions) {
            return APIXHRService("POST", API_PATH + "flow/tools/start-update", {
                projectKey: projectKey,
                tool: tool,
                updateOptions: JSON.stringify(updateOptions)
            });
        },
        propagateSchema: {
            markRecipeAsOKForced: function(projectKey, recipeName) {
                return APIXHRService("POST", API_PATH + "flow/tools/propagate-schema/mark-recipe-as-ok-forced", {
                    projectKey: projectKey,
                    recipeName: recipeName
                });
            },
            markRecipeAsOKAfterUpdate: function(projectKey, recipeName) {
                return APIXHRService("POST", API_PATH + "flow/tools/propagate-schema/mark-recipe-as-ok-after-update", {
                    projectKey: projectKey,
                    recipeName: recipeName
                });
            },
            markDatasetAsBeingRebuilt: function(projectKey, datasetName) {
                return APIXHRService("POST", API_PATH + "flow/tools/propagate-schema/mark-dataset-as-being-rebuilt", {
                    projectKey: projectKey,
                    datasetName: datasetName
                });
            },
            runAutomatically: function(projectKey, datasetName, rebuild, recipeUpdateOptions, excludedRecipes, partitionByDim, partitionByComputable, markAsOkRecipes) {
                return APIXHRService("POST", API_PATH + "flow/tools/propagate-schema/run-automatically", {
                    projectKey: projectKey,
                    datasetName: datasetName,
                    rebuild: rebuild,
                    recipeUpdateOptions: JSON.stringify(recipeUpdateOptions),
                    partitionByDim: JSON.stringify(partitionByDim),
                    partitionByComputable: JSON.stringify(partitionByComputable),
                    excludedRecipes: JSON.stringify(excludedRecipes),
                    markAsOkRecipes: JSON.stringify(markAsOkRecipes)
                });
            }
        },
        checkConsistency: {
            markAsOK: function(projectKey, nodeIds) {
                return APIXHRService("POST", API_PATH + "flow/tools/check-consistency/mark-as-ok", {
                    projectKey: projectKey,
                    nodeIds: JSON.stringify(nodeIds)
                });
            },
            recheck: function(projectKey, nodeIds) {
                return APIXHRService("POST", API_PATH + "flow/tools/check-consistency/recheck", {
                    projectKey: projectKey,
                    nodeIds: JSON.stringify(nodeIds)
                });
            }
        }
    },
    ai: {
        startExplainProject: function(projectKey, explanationOptions) {
            return APIXHRService("POST", API_PATH + "flow/ai-tools/start-project-explanation", {
                projectKey, explanationOptions: JSON.stringify(explanationOptions)
            }, "nospinner");
        },
        startExplainFlowZone: function(projectKey, zoneId, explanationOptions) {
            return APIXHRService("POST", API_PATH + "flow/ai-tools/start-flow-zone-explanation", {
                projectKey, zoneId, explanationOptions: JSON.stringify(explanationOptions)
            }, "nospinner");
        },
        startExplainDataset: function(projectKey, datasetName, explanationOptions) {
            return APIXHRService("POST", API_PATH + "flow/ai-tools/start-dataset-explanation", {
                projectKey, datasetName, explanationOptions: JSON.stringify(explanationOptions)
            }, "nospinner");
        },
    },
    recipes: {
        getTypesDescriptors: function(lang) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-types-descriptors", {lang: lang});
        },
        list: function(projectKey) {
            return APIXHRService("GET", API_PATH + "flow/recipes/list", {projectKey: projectKey});
        },
        listHeads: function(projectKey, tagFilter) {
            return APIXHRService("GET", API_PATH + "flow/recipes/list-heads", {projectKey: projectKey, tagFilter: tagFilter});
        },
        getCreatingRecipes: function(computables) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-creating-recipes", { computables });
        },
        getWithInlineScript: function(projectKey, name) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-with-inline-script", {
                projectKey: projectKey, name:name
            });
        },
        getFullInfo: function(projectKey, name) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-full-info", {
                projectKey: projectKey, name:name
            });
        },
        save: function(projectKey, recipe, scriptData, commitMessage) {
            return APIXHRService("POST", API_PATH + "flow/recipes/save", {
                projectKey: projectKey,
                recipe:JSON.stringify(recipe),
                scriptData: scriptData,
                commitMessage: commitMessage
            });
        },
        rename: function(projectKey, origName, newName) {
            return APIXHRService("POST", API_PATH + "flow/recipes/rename", {
                projectKey: projectKey, origName: origName, newName: newName
            });
        },
        getShakerSaveImpact: function(projectKey, recipe, shaker, outputSchema) {
            return APIXHRService("POST", API_PATH + "flow/recipes/get-shaker-save-impact", {
                projectKey: projectKey,
                recipe:JSON.stringify(recipe),
                shaker: JSON.stringify(shaker),
                outputSchema: JSON.stringify(outputSchema)
            });
        },
        checkSaveConflict: function(projectKey, name, recipe) {
            return APIXHRService("POST", API_PATH + "flow/recipes/check-save-conflict", {
                projectKey: projectKey,
                name:name,
                recipe:JSON.stringify(recipe)
            });
        },
        getComputableSaveImpact: function(projectKey, recipeData, payloadData) {
            return APIXHRService("POST", API_PATH + "flow/recipes/get-computable-save-impact", {
                projectKey: projectKey, recipe: JSON.stringify(recipeData),
                payload: payloadData
            })
        },
        getComputableSaveImpacts: function(projectKey, recipesData, payloadsData) {
            return APIXHRService("POST", API_PATH + "flow/recipes/get-computable-save-impacts", {
                projectKey: projectKey,
                recipes: JSON.stringify(recipesData),
                payloads: JSON.stringify(payloadsData)
            })
        },
        updateUsabilityOfSMInputs: function(projectKey, inputModelNeedsDataFolder, inputModelPredictionTypeAsString, computableList) {
            return APIXHRService("POST", API_PATH + "flow/recipes/update-sm-usabilities", {
                projectKey, inputModelNeedsDataFolder, inputModelPredictionTypeAsString, computables: JSON.stringify(computableList),
             });
        },
        saveOutputSchema: function(projectKey, computableType, computableId, newSchema, dropAndRecreate, synchronizeMetastore, extraOptions) {
            return APIXHRService("POST", API_PATH + "flow/recipes/save-output-schema", {
                projectKey: projectKey, computableType: computableType, computableId: computableId,
                newSchema: JSON.stringify(newSchema),
                dropAndRecreate: dropAndRecreate,
                synchronizeMetastore: synchronizeMetastore,
                extraOptions: extraOptions == null ? null : JSON.stringify(extraOptions)
            })
        },
        get: function(projectKey, name) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get", {projectKey: projectKey, name:name});
        },
        checkNotebookEdition: function(projectKey, name) {
            return APIXHRService("GET", API_PATH + "flow/recipes/check-notebook-edition", {projectKey:projectKey, name:name});
        },
        editInNotebook: function(projectKey, name, codeEnvSelection, containerSelection) {
            return APIXHRService("POST", API_PATH + "flow/recipes/edit-in-notebook", {projectKey:projectKey, name:name, codeEnvSelection:(codeEnvSelection ? JSON.stringify(codeEnvSelection) : null), containerSelection:(containerSelection ? JSON.stringify(containerSelection) : null)});
        },
        editInSQLNotebook: function(projectKey, recipeName) {
            return APIXHRService("POST", API_PATH + "flow/recipes/edit-in-sql-notebook", {projectKey:projectKey, recipeName:recipeName});
        },
        getGraph: function (projectKey, withSvg, drawZones, zoneId, collapsedZones, nospinner=false) {
            return APIXHRService(
                "GET", API_PATH + "flow/recipes/get-graph-serialized",
                { projectKey, withSvg, drawZones, zoneId, collapsedZones: JSON.stringify(collapsedZones) },
                nospinner ? "nospinner" : null
            );
        },
        getHiveCompatibilityStatus: function() {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-hive-compatibility-status");
        },
        basicResyncSchema: function(projectKey, recipe) {
            return APIXHRService("POST", API_PATH + "flow/recipes/basic-resync-schema", {
                projectKey: projectKey, recipeData: JSON.stringify(recipe)
            });
        },
        basicDropSchema: function(projectKey, recipe) {
            return APIXHRService("POST", API_PATH + "flow/recipes/basic-drop-schema", {
                projectKey: projectKey, recipeData: JSON.stringify(recipe)
            });
        },
        /* Get the schema update result for a single recipe.
         * This is the call used when you are NOT editing the recipe.
         */
        getSchemaUpdateResult: function(projectKey, recipeName) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-schema-update-result", {
                projectKey:projectKey,
                recipeName:recipeName
            });
        },
        getRequirements: function(projectKey, recipeType) {
            return APIXHRService("GET", API_PATH + "flow/recipes/get-requirements", {
                projectKey:projectKey,
                recipeType:recipeType
            });
        },
        getIOChangeResult: function(projectKey, recipeAndPayloadBefore, recipeAndPayloadAfter) {
            return APIXHRService("POST", API_PATH + "flow/recipes/get-io-change-result", {
                projectKey: projectKey,
                recipeAndPayloadBefore : JSON.stringify(recipeAndPayloadBefore),
                recipeAndPayloadAfter : JSON.stringify(recipeAndPayloadAfter)
            });
        },
        getInsertableFragments: function(projectKey, recipe) {
            return APIXHRService("POST", API_PATH + "flow/recipes/get-insertable-fragments", {
                projectKey: projectKey,
                recipe : JSON.stringify(recipe)
            });
        },
        runFromRecipeUI: function(projectKey, recipeName, mode, buildPartitions) {
            return APIXHRService("GET", API_PATH + "flow/recipes/run-from-recipe-ui", {
                projectKey, recipeName, mode, buildPartitions: JSON.stringify(buildPartitions)
            });
        },
        testCustomMetric: function(projectKey, recipe, recipeDesc, metricIndex, serializedInputDataset) {
            return APIXHRService("POST", API_PATH + "flow/recipes/test-custom-metric", {
                projectKey: projectKey,
                recipe: JSON.stringify(recipe),
                recipeDesc: JSON.stringify(recipeDesc),
                metricIndex: metricIndex,
                serializedInputDataset: JSON.stringify(serializedInputDataset)
            });
        },

        massActions: {
            startSetImpalaWriteMode(recipes, runInStreamMode) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/start-set-impala-write-mode", {
                    recipes: JSON.stringify(recipes)
                });
            },
            setImpalaWriteMode(recipes, runInStreamMode) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/set-impala-write-mode", {
                    recipes: JSON.stringify(recipes),
                    runInStreamMode: runInStreamMode
                });
            },
            startSetHiveEngine(recipes) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/start-set-hive-engine", {
                    recipes: JSON.stringify(recipes)
                });
            },
            setHiveEngine(recipes, executionEngine) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/set-hive-engine", {
                    recipes: JSON.stringify(recipes),
                    executionEngine: executionEngine
                });
            },
            startSetSparkEngine(recipes) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/start-set-spark-engine", {
                    recipes: JSON.stringify(recipes)
                });
            },
            setSparkEngine(recipes, executionEngine) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/set-spark-engine", {
                    recipes: JSON.stringify(recipes),
                    executionEngine: executionEngine
                });
            },
            startSetSparkConfig(recipes, sparkConfig) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/start-set-spark-config", {
                    recipes: JSON.stringify(recipes),
                    sparkConfig: JSON.stringify(sparkConfig)
                });
            },
            setSparkConfig(recipes, sparkConfig) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/set-spark-config", {
                    recipes: JSON.stringify(recipes),
                    sparkConfig: JSON.stringify(sparkConfig)
                });
            },
            startSetPipelineability(recipes, pipelineType) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/start-set-pipelineability", {
                    recipes: JSON.stringify(recipes),
                    pipelineType: pipelineType
                });
            },
            setPipelineability(recipes, pipelineType, allowStart, allowMerge) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/set-pipelineability", {
                    recipes: JSON.stringify(recipes),
                    pipelineType: pipelineType,
                    allowStart: allowStart,
                    allowMerge: allowMerge
                });
            },
            testConvertToImpala(recipes, checkRecipesRunOnImpala) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/test-convert-to-impala", {
                    recipes: JSON.stringify(recipes),
                    checkRecipesRunOnImpala: checkRecipesRunOnImpala
                });
            },
            convertToImpala(recipes, checkRecipesRunOnImpala) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/convert-to-impala", {
                    recipes: JSON.stringify(recipes),
                    checkRecipesRunOnImpala: checkRecipesRunOnImpala
                });
            },
            testConvertToHive(recipes, checkRecipesRunOnHive) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/test-convert-to-hive", {
                    recipes: JSON.stringify(recipes),
                    checkRecipesRunOnHive: checkRecipesRunOnHive
                });
            },
            convertToHive(recipes, checkRecipesRunOnHive) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/convert-to-hive", {
                    recipes: JSON.stringify(recipes),
                    checkRecipesRunOnHive: checkRecipesRunOnHive
                });
            },
            startChangeEngines(recipes) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/start-change-engines", {
                    recipes: JSON.stringify(recipes)
                });
            },
            testChangeEngines(recipes, engine) {
                // POST because query might be big
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/test-change-engines", {
                    recipes: JSON.stringify(recipes),
                    engine: engine
                });
            },
            changeEngines(recipes, engine) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/change-engines", {
                    recipes: JSON.stringify(recipes),
                    engine: engine
                });
            },
            changeCodeEnv(recipes, envSelection) {
                return APIXHRService("POST", API_PATH + "flow/recipes/mass-actions/change-code-env", {
                    recipes: JSON.stringify(recipes),
                    envSelection: JSON.stringify(envSelection)
                });
            }
        },
        visual: {
            convert: function(projectKey, recipe, payload, target) {
                return APIXHRService("POST", API_PATH + "flow/recipes/visual/convert", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    payload: payload, target: target
                });
            },
            convertSamplingRecipeToSplitRecipe: function(projectKey, recipe, secondOutputDataset) {
                return APIXHRService("POST", API_PATH + "flow/recipes/visual/convert-sampling-to-split", {
                    projectKey: projectKey,
                    recipeName: recipe.name,
                    secondOutputDataset: secondOutputDataset
                });
            },
            testInputReplacement: function (projectKey, recipe, payload, inputIndex, newInputName) {
            return APIXHRService("POST", API_PATH + "flow/recipes/visual/test-input-replacement", {
                    projectKey: projectKey,
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload,
                    inputIndex: inputIndex,
                    newInputName: newInputName
                });
            }
        },
        nlp: {
            convertToPromptRecipe: function(projectKey, recipeName) {
                return APIXHRService("POST", API_PATH + "flow/recipes/nlp/convert-to-prompt-recipe", { projectKey, recipeName });
            },
            getPromptForNLPRecipe: function(projectKey, recipeName) {
                return APIXHRService("GET", API_PATH + "flow/recipes/nlp/get-prompt-for-nlp-recipe", { projectKey, recipeName });
            },
            getDefaultEmbedDocumentRecipeSettings: function(projectKey) {
                return APIXHRService("GET", API_PATH + "flow/recipes/nlp/get-default-embed-documents-settings", { projectKey: projectKey });
            },
            getDefaultEmbedDocumentRules: function(extractionMode, embeddingLLMId, defaultVlmId, projectKey, defaultImageHandlingMode) {
                return APIXHRService("GET", API_PATH + "flow/recipes/nlp/get-default-embed-documents-rules", { extractionMode: extractionMode, embeddingLLMId: embeddingLLMId, defaultVlmId: defaultVlmId, projectKey: projectKey, defaultImageHandlingMode: defaultImageHandlingMode });
            },
        },

        download: {
            checkDownloadURL: function(projectKey, url) {
                return APIXHRService("POST", API_PATH + "recipes/download/check-url", { url: url, projectKey: projectKey });
            },
            checkDownloadSource: function(projectKey, recipe, source, partitionId) {
            return APIXHRService("POST", API_PATH + "recipes/download/check-source", {
                    recipe: JSON.stringify(recipe),
                    source: JSON.stringify(source),
                    projectKey: projectKey,
                    partitionId: partitionId
                });
            }
        },
        filter: {
            validateExpression: function(expression, schemaData, recipe) {
                return APIXHRService("POST", API_PATH + "recipes/validate-expression", {
                    schemaData,
                    expression,
                    recipeData: recipe,
                }, "nospinner");
            },
            validateExpressionWithProjectKey: function(expression, schemaData, projectKey) {
                return APIXHRService("POST", API_PATH + "recipes/validate-expression", {
                    schemaData,
                    expression,
                    projectKey,
                }, "nospinner");
            },
            validateAst: function(ast, projectKey) {
            return APIXHRService("POST", API_PATH + "recipes/validate-ast", {
                    astData: JSON.stringify(ast),
                    projectKey: projectKey
                }, "nospinner");
            }
        },
        join: {
            getSuggestions: function(projectKey, recipe, payload) {
            return APIXHRService("POST", API_PATH + "flow/recipes/join/get-suggestions", {
                    projectKey: projectKey,
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload
                });
            }
        },
        fuzzyjoin: {
            getSuggestions: function(projectKey, recipe, payload) {
            return APIXHRService("POST", API_PATH + "flow/recipes/fuzzyjoin/get-suggestions", {
                    projectKey: projectKey,
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload
                });
            }
        },
        geojoin: {
            getSuggestions: function(projectKey, recipe, payload) {
            return APIXHRService("POST", API_PATH + "flow/recipes/geojoin/get-suggestions", {
                    projectKey: projectKey,
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload
                });
            }
        },
        autofeaturegeneration: {
            getDefaultColumns: function(projectKey, newInputName, recipe, payload) {
            return APIXHRService("POST", API_PATH + "flow/recipes/autofeaturegeneration/get-default-columns", {
                    projectKey: projectKey,
                    newInputName: newInputName,
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload
                });
            },
            getSuggestions: function(projectKey, recipe, payload) {
            return APIXHRService("POST", API_PATH + "flow/recipes/autofeaturegeneration/get-suggestions", {
                    projectKey: projectKey,
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload
                });
            }
        },
        pig: {
            check: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/pig/check", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            }
        },
        shell: {
            check: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/shell/check", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            }
        },
        scala: {
            checkSyntax: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/scala/check-syntax", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            },
            convertToCustom: function(targetPluginId, targetPluginMode, scriptData, recipeFolder, recipeSerialized) {
            return APIXHRService("POST", API_PATH + "flow/recipes/scala/convert-to-custom", {
                    targetPluginId: targetPluginId,
                    targetPluginMode: targetPluginMode,
                    scriptData: scriptData,
                    recipeFolder: recipeFolder,
                    codeMode: recipeSerialized.params.codeMode
                });
            }
        },
        python: {
            checkSyntax: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/python/check-syntax", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            },
            convertToCustom: function(targetPluginId, targetPluginMode, scriptData, recipeFolder, recipe) {
            return APIXHRService("POST", API_PATH + "flow/recipes/python/convert-to-custom", {
                    targetPluginId: targetPluginId,
                    targetPluginMode: targetPluginMode,
                    recipeData: JSON.stringify(recipe),
                    scriptData: scriptData,
                    recipeFolder: recipeFolder
                });
            }
        },
        hive: {
            checkImpalaConvertibility: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/hive/check-impala-convertibility", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            }
        },
        impala: {
            validate: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/impala/validate", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            },
            getExecutionPlan: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/impala/get-execution-plan", {
                    projectKey: projectKey,
                    recipe: JSON.stringify(recipe),
                    script: scriptData,
                    targetPartition: targetPartition
                });
            },
            run: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/impala/run", {
                    projectKey: projectKey, recipe: JSON.stringify(recipe),
                    script: scriptData, targetPartition: targetPartition
                });
            },
            checkFullSqlAvailability: function(projectKey, recipe) {
            return APIXHRService("POST", API_PATH + "flow/recipes/impala/check-full-sql-availability", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe)
                });
            }
        },
        sqlScript: {
            validate: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/sql-script/validate", {
                    projectKey: projectKey, recipe: JSON.stringify(recipe),
                    script: scriptData, targetPartition: targetPartition
                });
            }
        },
        sqlQuery: {
            //TODO merge validate/get execution plan?
            validate: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/sql-query/validate", {
                    projectKey: projectKey, recipe: JSON.stringify(recipe),
                    script: scriptData, targetPartition: targetPartition
                });
            },
            getExecutionPlan: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/sql-query/get-execution-plan", {
                    projectKey: projectKey,
                    recipe: JSON.stringify(recipe),
                    script: scriptData,
                    targetPartition: targetPartition
                });
            },
            run: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/sql-query/run", {
                    projectKey: projectKey, recipe: JSON.stringify(recipe),
                    script: scriptData, targetPartition: targetPartition
                });
            }
        },
        sql: {
            startSQLQueryGeneration: function(projectKey, query, recipe, targetPartition) {
                return APIXHRService("POST", API_PATH + "flow/recipes/sql/generate-query",  {
                    projectKey: projectKey,
                    query: query,
                    recipe: JSON.stringify(recipe),
                    targetPartition: targetPartition
                })
            }
        },
        sparkSql: {
            //TODO merge validate/get execution plan?
            validate: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/sparksql-query/validate", {
                    projectKey: projectKey, recipe: JSON.stringify(recipe),
                    query: scriptData, targetPartition: targetPartition
                });
            },
        },
        r: {
             validate: function(projectKey, recipe, scriptData, targetPartition) {
            return APIXHRService("POST", API_PATH + "flow/recipes/r/validate", {
                    projectKey: projectKey, recipeData: JSON.stringify(recipe),
                    scriptData: scriptData, targetPartition: targetPartition
                });
            },
            convertToCustom: function(targetPluginId, targetPluginMode, scriptData, recipeFolder) {
            return APIXHRService("POST", API_PATH + "flow/recipes/r/convert-to-custom", {
                    targetPluginId: targetPluginId,
                    targetPluginMode: targetPluginMode,
                    scriptData: scriptData,
                    recipeFolder: recipeFolder
                });
            }
        },
        pyspark: {
            convertToCustom: function(targetPluginId, targetPluginMode, scriptData, recipeFolder) {
            return APIXHRService("POST", API_PATH + "flow/recipes/pyspark/convert-to-custom", {
                    targetPluginId: targetPluginId,
                    targetPluginMode: targetPluginMode,
                    scriptData: scriptData,
                    recipeFolder: recipeFolder
                });
            }
        },
        pivot: {
            getDatasetModalities: function(projectKey, datasetSmartName, pivot) {
                return APIXHRService("POST", API_PATH + "flow/recipes/pivot/get-dataset-modalities", {
                    projectKey : projectKey,
                    datasetSmartName : datasetSmartName,
                    pivot : JSON.stringify(pivot)
                });
            }
        },
        generic: {
            create: function(recipe, creationSettings) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/create", {
                    recipeData : JSON.stringify(recipe),
                    creationSettingsData: JSON.stringify(creationSettings)
                });
            },
            copy: function(projectKey, sourceProjectKey, sourceRecipeName, copySettings) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/copy", {
                    projectKey: projectKey,
                    sourceProjectKey: sourceProjectKey,
                    sourceRecipeName : sourceRecipeName,
                    copySettingsData: JSON.stringify(copySettings)
                });
            },
            getStatus : function(recipe, payload, sequenceId, request) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/get-status", {
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload,
                    sequenceId: sequenceId,
                    requestData : JSON.stringify(request)
                }, "nospinner");
            },
            getStatusWithSpinner: function(recipe, payload, sequenceId, request) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/get-status", {
                    recipeData: JSON.stringify(recipe),
                    payloadData: payload,
                    sequenceId: sequenceId,
                    requestData : JSON.stringify(request)
                });
            },
            validate: function(projectKey, recipe) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/validate", {
                    projectKey: projectKey,
                    recipe: JSON.stringify(recipe),
                });
            },
            pdepTest: function(recipe, pdepInputRef, pdep) {
                return APIXHRService("POST", API_PATH + "flow/recipes/pdep-test", {
                    recipeData: JSON.stringify(recipe),
                    pdepInputRef : pdepInputRef,
                    pdepData: JSON.stringify(pdep)
                });
            },
            getVariables: function(recipe) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/get-variables", {
                    recipeData: JSON.stringify(recipe),
                });
            },
            getVariablesWithProjectKey: function(projectKey) {
                return APIXHRService("POST", API_PATH + "flow/recipes/generic/get-variables", { projectKey });
            },
        }
    },
    listUsableComputables: function(projectKey, filter) {
        return APIXHRService("GET", API_PATH + "flow/list-usable-computable", {
            projectKey: projectKey,
            filter: JSON.stringify(filter)
        });
    },
    listDownstreamComputables: function(projectKey, from) {
        return APIXHRService("GET", API_PATH + "flow/list-downstream-computable",
            angular.extend({projectKey: projectKey}, from)
        );
    },
    checkDownstreamBuildable: function(projectKey, from) {
        return APIXHRService("GET", API_PATH + "flow/check-downstream-buildable/",
            angular.extend({projectKey: projectKey}, from)
        );
    },
    getComputables: function(taggableObjectRefs) {
        // POST because query might be big
        return APIXHRService("POST", API_PATH + "flow/get-computables", {
            items: JSON.stringify(taggableObjectRefs)
        });
    },
    getObjectContext: function(projectKey, objectType, objectId) {
        return APIXHRService("GET", API_PATH + "flow/get-object-context", {
            projectKey: projectKey,
            objectType: objectType,
            objectId: objectId
        }, "nospinner");
    },
    export: function(projectKey, exportFormat) {
        return APIXHRService("POST", API_PATH + "flow/export", { projectKey: projectKey, exportFormat: JSON.stringify(exportFormat) });
    },
    getExportURL: function(projectKey, exportId) {
        return API_PATH + "flow/download-export?"
            + "projectKey=" + encodeURIComponent(projectKey)
            + "&exportId=" + encodeURIComponent(exportId);
    },
    docGenDefault: function(projectKey) {
        return APIXHRService("POST", API_PATH + "flow/generate-default-document" , { projectKey });
    },
    docGenCustom: function(file, projectKey, callback) {
        return uploadFileRequest("flow/generate-custom-document", function(formData) {
            formData.append("file", file);
            formData.append("projectKey", projectKey);
        }, callback);
    },
    getFlowDocumentationExportURL: function(exportId) {
        return API_PATH + "flow/export-flow-document?exportId=" + encodeURIComponent(exportId);
    },
    testDatasetStorageConflicts: function(projectKey) {
        return APIXHRService("POST", API_PATH + "flow/test-dataset-storage-conflicts/", {projectKey});
    },
},
datalineage: {
    export: function(contextProjectKey, smartName, columnName, exportFormat) {
        return APIXHRService("POST", API_PATH + "data-lineage/export", { contextProjectKey, smartName, columnName, exportFormat: JSON.stringify(exportFormat) });
    },
    getExportURL: function(projectKey, exportId) {
        return API_PATH + "data-lineage/download-export?"
            + "projectKey=" + encodeURIComponent(projectKey)
            + "&exportId=" + encodeURIComponent(exportId);    }
},
profile: {
    get: function(login, includeProfileAndTrialComputation) {
        if(login) {
            return APIXHRService("GET", API_PATH + "/profile/get", {login:login, includeProfileAndTrialComputation: includeProfileAndTrialComputation});
        } else {
            return APIXHRService("GET", API_PATH + "/profile/get");
        }
    },
    achievements: function(login) {
        return APIXHRService("GET", API_PATH + "/profile/achievements", {login:login});
    },
    updateATSurveySettings: function (event, action) {
        return APIXHRService("POST", API_PATH + "myprofile/update-ats-settings", {event: event, action: action});
    },
    setNPSSettings: function(action) {
        return APIXHRService("POST", API_PATH + "/myprofile/update-nps-settings" , { action: action });
    },
    edit: function(user) {
        return APIXHRService("POST", API_PATH + "/myprofile/edit" , {user: JSON.stringify(user)});
    },
    uploadPicture: function(file) {
        var url = API_PATH + "myprofile/upload-picture";

        // angular doesn't provide a way to get the progress event yet, we explicitly redo it
        var deferred = $q.defer();

        var xhr = new XMLHttpRequest();

        xhr.addEventListener("load", function(e) {
            deferred.resolve((e.target||e.srcElement).response);
            $rootScope.$apply();
        }, false);
        xhr.addEventListener("error", function(e) {deferred.reject(e);$rootScope.$apply();}, false);

        xhr.open("POST", url);
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

        var formdata = new FormData();
        formdata.append("file", file);
        xhr.send(formdata);

        return deferred.promise;
    },
    setUserSettings: function(userSettings) {
        return APIXHRService("POST", API_PATH + "/myprofile/set-user-settings" , {userSettingsAsString: JSON.stringify(userSettings)});
    },
    setQuestionnaire: function(questionnaireContent) {
        return APIXHRService("POST", API_PATH + "/myprofile/update-questionnaire" , {questionnaire: JSON.stringify(questionnaireContent)});
    },
    getQuestionnaire: function() {
        return APIXHRService("GET", API_PATH + "/myprofile/get-questionnaire");
    },
    updatePageSpecificTourSettings: function(pageSpecificTourSettings) {
        return APIXHRService("POST", API_PATH + "myprofile/update-page-specific-tour-settings", { pageSpecificTourSettings: JSON.stringify(pageSpecificTourSettings) });
    },
    listPersonalAPIKeys: function(){
        return APIXHRService("GET", API_PATH + "/publicapi/list-personal-api-keys");
    },
    createPersonalAPIKey : function(apiKey){
        return APIXHRService("POST", API_PATH + "/publicapi/create-personal-api-key",{
            label: apiKey.label,
            description: apiKey.description
        });
    },
    editPersonalAPIKey : function(apiKey){
        return APIXHRService("POST", API_PATH + "/publicapi/edit-personal-api-key",{
            id: apiKey.id,
            label: apiKey.label,
            description: apiKey.description
        });
    },
    deletePersonalAPIKey: function(keyId) {
        return APIXHRService("POST", API_PATH + "/publicapi/delete-personal-api-key",{
            id:keyId
        });
    },
    listConnectionCredentials : function(){
        return APIXHRService("GET", API_PATH + "/myprofile/list-connection-credentials");
    },
    setBasicConnectionCredential : function(connection, user, password){
        return APIXHRService("POST", API_PATH + "/myprofile/set-basic-connection-credential",{
            connection: connection,
            user: user,
            password: password
        });
    },
    notifyAlertBannerDismissed: function(bannerId) {
        return APIXHRService("POST", API_PATH + "myprofile/notify-alert-banner-dismissed", {bannerId: bannerId});
    },
    exchangeAuthorizationCode : function(code, state, error, errorDescription, errorUri){
        return APIXHRService("POST", API_PATH + "myprofile/oauth2-callback",{
            code: code,
            state: state,
            error: error,
            error_description: errorDescription,
            error_uri: errorUri
        });
    },
    startOAuth2AuthorizationFlow: function(dssUrl, userCurrentState, credential) {
        return APIXHRService("POST", API_PATH + "myprofile/start-oauth2-authorization-flow", {
            dssProtocolHost: dssUrl,
            userCurrentState: userCurrentState,
            credential: credential
        }, null, "json");
    },
    connectionCredentials: {
        azureOAuthDeviceCodeDanceStep1: function(dssUrl, userCurrentState, credential) {
            return APIXHRService("POST", API_PATH + "myprofile/connection-credentials/azure-oauth-devicecode-dance-step1",{
                dssProtocolHost: dssUrl,
                userCurrentState: userCurrentState,
                credential: credential
            }, null, "json");
        },
        azureOAuthDeviceCodeDanceStep2: function(dssUrl, userCurrentState, credential) {
            return APIXHRService("POST", API_PATH + "myprofile/connection-credentials/azure-oauth-devicecode-dance-step2",{
                dssProtocolHost: dssUrl,
                userCurrentState: userCurrentState,
                credential: credential
            }, null, "json");
        }
    },
    pluginCredentials: {
        setBasicCredential : function(pluginId, paramSetId, presetId, paramName, user, password){
            return APIXHRService("POST", API_PATH + "/myprofile/plugin-credentials/set-basic-credential", {
                pluginId: pluginId,
                paramSetId: paramSetId,
                presetId: presetId,
                paramName: paramName,
                user: user,
                password: password
            });
        }
    },
    folderEdit: {
        listContents: function(type) {
            return APIXHRService("GET", API_PATH + "myprofile/folder-edition/list-contents", {
                type: type
            });
        },
        getContent: function(type, path, sendAnyway) {
            return APIXHRService("GET", API_PATH + "myprofile/folder-edition/get-content", {
                type: type, path: path, sendAnyway: sendAnyway
            });
        },
        setContent: function(type, path, data) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/set-content", {
                type: type, path: path, data: data
            });
        },
        setContentMultiple: function(type, contentMap) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/set-content-multiple", {
                type: type, contentMap: JSON.stringify(contentMap)
            });
        },
        createContent: function(type, path, isFolder) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/create-content", {
                type: type, path: path, isFolder: isFolder
            });
        },
        deleteContent: function(type, path) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/delete-content", {
                type: type, path: path
            });
        },
        decompressContent: function(type, path) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/decompress-content", {
                type: type, path: path
            });
        },
        renameContent: function(type, path, newName) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/rename-content", {
                type: type, path: path, newName: newName
            });
        },
        moveContent: function(type, path, toPath) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/move-content", {
                type: type, path: path, toPath: toPath
            });
        },
        copyContent: function(type, path) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/copy-content", {
                type: type, path: path
            });
        },
        uploadContent: function(type, path, file, callback) {
            return uploadFileRequest("myprofile/folder-edition/upload-content", function(formdata) {
                formdata.append("type", type);
                formdata.append("path", path);
                formdata.append("file", file);
            }, callback);
        },
        checkUploadContent: function(type, path, filePaths) {
            return APIXHRService("POST", API_PATH + "myprofile/folder-edition/check-upload-content", {
                type: type, path: path, filePaths: JSON.stringify(filePaths)
            });
        },
        previewImageURL: function(type, path, mimeType) {
            return '/dip/api/myprofile/folder-edition/preview-image?type=' + type + '&path=' + encodeURIComponent(path) + '&contentType=' + encodeURIComponent(mimeType);
        },
        downloadURL: function(type, path) {
            return '/dip/api/myprofile/folder-edition/download-content?type=' + type + '&path=' + encodeURIComponent(path);
        }
    },
    togglePinnedConnectionType: function(type) {
        return APIXHRService("POST", API_PATH + "myprofile/toggle-pinned-connection-type", {
            datasetType: type
        });
    },
    getPinnedConnectionTypes: function(type) {
        return APIXHRService("GET", API_PATH + "myprofile/get-pinned-connection-types");
    }
},
globalTags: {
    getGlobalTagsInfo: function() {
        return APIXHRService("GET", API_PATH + "global-tags/get-global-tags-info");
    }
},
variables: {
    expandExpr: function(projectKey, expr) {
         return APIXHRService("GET", API_PATH + "variables/expand-expr", {
            projectKey: projectKey, expr: expr
         });
    },

    expandExprAllowUnresolved: function(projectKey, expr) {
         return APIXHRService("GET", API_PATH + "variables/expand-expr-unresolved", {
            projectKey: projectKey, expr: expr
         });
    }
},
integrations: {
    listChannelsForIntegrationType: function(integrationType) {
        return APIXHRService("GET", API_PATH + "/integrations/list-available-channels-for-integration-type", {
            integrationType: integrationType
        })
    }
},
codeenvs: {
    list: function(envLang) {
        return APIXHRService("GET", API_PATH + "code-envs/list",{ envLang: envLang })
    },
    listNames: function(envLang) {
        return APIXHRService("GET", API_PATH + "code-envs/list-names",{ envLang: envLang })
    },
    listNamesWithDefault: function(envLang, projectKey) {
        return APIXHRService("GET", API_PATH + "code-envs/list-names-with-default",{ envLang: envLang, projectKey:projectKey })
    },
    listWithVisualMlPackages: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-envs/list-python-with-visual-ml-packages", {projectKey: projectKey})
    },
    listCompatibilityWithMLTask: function(projectKey, mlTask) {
        return APIXHRService("POST", API_PATH + "code-envs/list-python-compatibility-with-mltask", {projectKey, mlTask: JSON.stringify(mlTask)})
    },
    listCompatibilityWithTestPackages: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-envs/list-python-compatibility-with-test", {projectKey: projectKey})
    },
    listCompatibilityMLflowPackages: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-envs/list-python-with-mlflow-packages", {projectKey: projectKey})
    },
    listForPlugins: function(pluginId) {
        return APIXHRService("GET", API_PATH + "code-envs/plugin/list",{ pluginId: pluginId })
    },
    createForPlugin: function(pluginId, desc, associateToPlugin) {
        return APIXHRService("POST", API_PATH + "code-envs/plugin/create",{ pluginId: pluginId, desc: JSON.stringify(desc), associateToPlugin: !!associateToPlugin })
    },
    updateForPlugin: function(pluginId, envName) {
        return APIXHRService("GET", API_PATH + "code-envs/plugin/update",{ pluginId: pluginId, envName: envName })
    },
    listUsages: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-envs/list-usages",{ projectKey: projectKey })
    },
    checkAgentsCodeEnv: function(projectKey) {
        return APIXHRService("GET", API_PATH + "code-envs/agents/check", { projectKey: projectKey });
    },
    createForDSSInternal: function(dssInternalCodeEnvType, pythonInterpreter, codeEnvVersion) {
        return APIXHRService("POST", API_PATH + "code-envs/dss-internal/create", { dssInternalCodeEnvType, pythonInterpreter, codeEnvVersion });
    },
    updateForDSSInternal: function(dssInternalCodeEnvType, codeEnvVersion) {
        return APIXHRService("POST", API_PATH + "code-envs/dss-internal/update", { dssInternalCodeEnvType, codeEnvVersion });
    },
    getSupportedInterpreters: function(dssInternalCodeEnvType, codeEnvVersion) {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/supported-interpreters", { dssInternalCodeEnvType, codeEnvVersion });
    },
    getDSSInternalDefaultAvailableInterpreter: function(dssInternalCodeEnvType, codeEnvVersion) {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/default-available-interpreter", { dssInternalCodeEnvType, codeEnvVersion });
    },
    checkDSSInternalCodeEnv: function(dssInternalCodeEnvType, codeEnvVersion) {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/check", { dssInternalCodeEnvType, codeEnvVersion });
    },
    checkInternalCodeEnvUsedForPII: function() {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/check-use-for-pii");
    },
    checkInternalCodeEnvUsedForRAG: function() {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/check-use-for-rag");
    },
    checkInternalCodeEnvUsedForDocumentExtraction: function() {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/check-use-for-document-extraction");
    },
    checkInternalCodeEnvUsedForHF: function() {
        return APIXHRService("GET", API_PATH + "code-envs/dss-internal/check-use-for-hf");
    },
    getUserAccessibleCodeEnvSettings: function() {
        return APIXHRService("GET", API_PATH + "code-envs/user-accessible-settings");
    },
    getMandatoryPackages: function(envLang, envDesc) {
        return APIXHRService("GET", API_PATH + "code-envs/design/get-mandatory-packages", {
            envLang: envLang, envDesc: JSON.stringify(envDesc)
        });
    },
    listAvailablePythonInterpreters: function() {
        return APIXHRService("GET", API_PATH + "code-envs/list-available-python-interpreters", "nospinner");
    },
    getPluginDefaultAvailableInterpreter: function(pluginId) {
        return APIXHRService("GET", API_PATH + "code-envs/get-plugin-default-available-interpreter", {
            pluginId,
        });
    },
    getDefaultCorePackageSet: function(pythonInterpreter) {
        return APIXHRService("GET", API_PATH + "code-envs/get-default-core-package-set", { pythonInterpreter });
    },
},
containers: {
    listNames: function(type, workloadType) {
        return APIXHRService("GET", API_PATH + "containers/list-names", { type, workloadType });
    },
    listSparkNames: function() {
        return APIXHRService("GET", API_PATH + "containers/list-spark-names");
    },
    listNamesWithDefault: function(projectKey, type, workloadType, containerSpecificContext) {
        return APIXHRService("GET", API_PATH + "containers/list-names-with-default", { projectKey, type, workloadType, containerSpecificContext })
    },
    getConfigInfo: function(projectKey, exposableKind, expositionUsageContext, containerSelection, inlineContainerConfig) {
        return APIXHRService("POST", API_PATH + "containers/get-config-info", {
            projectKey:projectKey,
            exposableKind:exposableKind,
            expositionUsageContext: expositionUsageContext,
            containerSelection:JSON.stringify(containerSelection),
            inlineContainerConfig: JSON.stringify(inlineContainerConfig)
        })
    },
    getExpositions: function(containerType, exposableKind, expositionUsageContext) {
        return APIXHRService("GET", API_PATH + "containers/get-expositions", {
            containerType: containerType,
            exposableKind: exposableKind,
            expositionUsageContext: expositionUsageContext
        })
    },
    listUsages: function(projectKey) {
        return APIXHRService("GET", API_PATH + "containers/list-usages",{ projectKey: projectKey })
    },
},
popularDatasets: {
    compute: function(config) {
        return APIXHRService("POST", API_PATH + "popular-datasets/compute", {config: JSON.stringify(config)});
    },
},
admin: {
    getInstanceInfo: function() {
        return APIXHRService("GET", API_PATH + "/admin/get-instance-info");
    },
    getPromotedContentDisplayNames: function(taggableObjectRefs) {
        return APIXHRService("GET", API_PATH + "admin/get-promoted-content-display-names", {
            taggableObjectRefs: JSON.stringify(taggableObjectRefs)
        });
    },
    apiKeys: {
        updateApiKeysLifetime: function(data) {
            return APIXHRService("POST", API_PATH + "/admin/publicapi/update-api-keys-lifetime")
        },
    },
    profiling: {
        listProfiles: function() {
            return APIXHRService("GET", API_PATH + "/admin/profiling/list");
        },
        downloadProfile: function(profileId) {
            return API_PATH + "/admin/profiling/get?id=" + encodeURIComponent(profileId);
        },
        deleteProfile: function(profileId) {
            return APIXHRService("POST", API_PATH + "/admin/profiling/delete", {id: profileId});
        },
        saveConfig: function(config) {
            return APIXHRService("POST", API_PATH + "/admin/profiling/save-config", {config: JSON.stringify(config)});
        },
        getConfig: function() {
            return APIXHRService("GET", API_PATH + "/admin/profiling/get-config");
        }
    },
    containerExec: {
        pushBaseImages: function(){
            return APIXHRService("POST", API_PATH + "/admin/container-exec/push-base-images")
        },
        installJupyterSupport: function() {
            return APIXHRService("POST", API_PATH + "/admin/container-exec/install-jupyter-support")
        },
        removeJupyterSupport: function() {
            return APIXHRService("POST", API_PATH + "/admin/container-exec/remove-jupyter-support")
        },
        testConf: function(data, clusterDefinedConfig, clusterId, executionConfigsGenericOverrides) {
            return APIXHRService("POST", API_PATH + "/admin/container-exec/test-conf", {data: JSON.stringify(data), clusterDefinedConfig: clusterDefinedConfig || false, clusterId: clusterId, genericOverridesData: JSON.stringify(executionConfigsGenericOverrides)})
        },
        applyK8SPolicies: function(data, clusterDefinedConfig, clusterId, executionConfigsGenericOverrides) {
            return APIXHRService("POST", API_PATH + "/admin/container-exec/apply-kubernetes-policies", {})
        },
        getBaseImageDefaults: function(type) {
            return APIXHRService("GET", API_PATH + "/admin/container-exec/get-base-image-defaults", {type: type})
        },
        buildBaseImage: function(type, options, pushAfterBuild = true) {
            return APIXHRService("POST", API_PATH + "/admin/container-exec/build-base-image", {type:type, options:JSON.stringify(options), pushAfterBuild: !!pushAfterBuild})
        }
    },
    connections: {
        list: function () {
            return APIXHRService("GET", API_PATH + "/admin/connections/list");
        },
        listHiveVirtual: function(){
            return APIXHRService("GET", API_PATH + "/admin/connections/list-hive-virtual");
        },
        listRunningJobs: function () {
            return APIXHRService("GET", API_PATH + "/admin/connections/metadata", null, "nospinner");
        },
        get: function (name) {
            return APIXHRService("GET", API_PATH + "/admin/connections/get", {name: name});
        },
        save: function (data, creation) {
            return APIXHRService("POST", API_PATH + "/admin/connections/save",
            {
                data: JSON.stringify(data),
                creation
            });
        },
        listProcessableConnections: function (type, selectedConnections) {
            return APIXHRService("GET", API_PATH + "/admin/connections/list-indexable-connections", {data: JSON.stringify(selectedConnections)});
        },
        index: function (data) {
            return APIXHRService("POST", API_PATH + "/admin/connections/index", {data: JSON.stringify(data)});
        },
        scan: function (data) {
            return APIXHRService("POST", API_PATH + "/admin/connections/scan", {data: JSON.stringify(data)});
        },
        abortIndexation: function (data) {
            return APIXHRService("GET", API_PATH + "/admin/connections/abort-indexation");
        },
        delete: function (data) {
            return APIXHRService("POST", API_PATH + "/admin/connections/delete", {data: JSON.stringify(data)});
        },
        discoverOAuthConfig: function (data) {
            return APIXHRService("POST", API_PATH + "/admin/connections/discover-oauth-config", {data: JSON.stringify(data)});
        },
        registerDynamicOAuthClient: function (data, dssExternalURL) {
            return APIXHRService("POST", API_PATH + "/admin/connections/register-dynamic-oauth-client", {
                data: JSON.stringify(data),
                dssExternalURL: dssExternalURL
            });
        },
        testSQL: function(data, massImportTargetProjectKey) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-sql", {
                data: JSON.stringify(data),
                massImportTargetProjectKey: massImportTargetProjectKey
            });
        },
        testPostgreSQL: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-postgresql", {data: JSON.stringify(data)});
        },
        testEC2: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-ec2", {data: JSON.stringify(data)});
        },
        testDatabricksVolume: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-databricks-volume", {data: JSON.stringify(data)});
        },
        testSageMaker: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-sagemaker", {data: JSON.stringify(data)});
        },
        testGCS: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-gcs", {data: JSON.stringify(data)});
        },
        testSharePointOnline: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-sharepointonline", {data: JSON.stringify(data)});
        },
        testVertexAIModelDeployment: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-vertexaimodeldeployment", {data: JSON.stringify(data)});
        },
        testDatabricksModelDeployment: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-databricksmodeldeployment", {data: JSON.stringify(data)});
        },
        testAzure: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-azure", {data: JSON.stringify(data)});
        },
        testAzureML: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-azureml", {data: JSON.stringify(data)});
        },
        testMongoDB: function(data, sequenceId) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-mongodb", {data: JSON.stringify(data), sequenceId: sequenceId});
        },
        testDynamoDB: function(data, sequenceId) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-dynamodb", {data: JSON.stringify(data), sequenceId: sequenceId});
        },
        testCassandra: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-cassandra", {data: JSON.stringify(data)});
        },
        testOpenAi: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-openai", {data: JSON.stringify(data)});
        },
        testPinecone: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-pinecone", {data: JSON.stringify(data)});
        },
        testAzureAISearch: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-azureaisearch", {data: JSON.stringify(data)});
        },
        testAzureOpenAi: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-azure-openai", {data: JSON.stringify(data)});
        },
        testAzureAiFoundry: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-azure-ai-foundry", {data: JSON.stringify(data)});
        },
        testAzureMLGenericLLM: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-azure-llm", {data: JSON.stringify(data)});
        },
        testNvidiaNim: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-nvidia-nim", {data: JSON.stringify(data)});
        },
        testCohere: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-cohere", {data: JSON.stringify(data)});
        },
        testMistralAi: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-mistralai", {data: JSON.stringify(data)});
        },
        testAnthropic: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-anthropic", {data: JSON.stringify(data)});
        },
        testBedrock: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-bedrock", {data: JSON.stringify(data)});
        },
        testMosaicML: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-mosaicml", {data: JSON.stringify(data)});
        },
        testVertexAILLM: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-vertexaillm", {data: JSON.stringify(data)});
        },
        testDatabricksLLM: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-databricksllm", {data: JSON.stringify(data)});
        },
        testSnowflakeCortex: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-snowflakecortex", {data: JSON.stringify(data)});
        },
        testCustomLLM: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-customllm", {data: JSON.stringify(data)});
        },
        testHuggingFace: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-hugging-face", {data: JSON.stringify(data)});
        },
        testSageMakerGenericLLM: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-sagemaker-generic-llm", {data: JSON.stringify(data)});
        },
        testTwitter: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-twitter", {data: JSON.stringify(data)});
        },
        testKafka: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-kafka", {data: JSON.stringify(data)});
        },
        testKsql: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-ksql", {data: JSON.stringify(data)});
        },
        testSQS: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-sqs", {data: JSON.stringify(data)});
        },
        testElasticSearch: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-elasticsearch", {data: JSON.stringify(data)});
        },
        testRemoteMCP: function(data) {
            return APIXHRService("POST", API_PATH + "admin/connections/test-remote-mcp", {data: JSON.stringify(data)});
        },
        listHuggingFacePresets: function(overriddenConfig) {
            return APIXHRService("POST", API_PATH + "admin/connections/list-presets-huggingface", {overriddenConfig: JSON.stringify(overriddenConfig)});
        },
        getHfKernelStatus: function(connectionName) {
            return APIXHRService("GET", API_PATH + "admin/connections/huggingface-kernels-status", {connectionName:connectionName}, "nospinner");
        },
        getHfKernelLogs: function(kernelId) {
            return APIXHRService("GET", API_PATH + "admin/connections/huggingface-kernels-logs", {kernelId:kernelId}, "nospinner");
        },
        killHFKernel: function(kernelId) {
            return APIXHRService("GET", API_PATH + "admin/connections/huggingface-kill-kernel", {kernelId:kernelId}, "nospinner");
        },
        setActiveTwitterConnection: function(name) {
           return APIXHRService("POST", API_PATH + "admin/connections/set-active-twitter-connection", {name:name});
        },

        hdfs : {
            resyncPermissions : function(connectionName) {
                return APIXHRService("POST", API_PATH + "admin/connections/hdfs/resync-permissions", {
                    connectionName : connectionName
                });
            },
            resyncRootPermissions : function(connectionName) {
                return APIXHRService("POST", API_PATH + "admin/connections/hdfs/resync-root-permissions", {
                    connectionName : connectionName
                });
            }

        }
    },
    integrationChannels: {
        list: function() {
            return APIXHRService("GET", API_PATH + "admin/integration-channels/list");
        },
        saveAll: function(data) {
            return APIXHRService("POST", API_PATH + "admin/integration-channels/save-all" , {data: JSON.stringify(data)});
        }
    },
    logs: {
        list: function() {
           return APIXHRService("GET", API_PATH + "admin/logs/list");
        },
        get: function(name) {
           return APIXHRService("GET", API_PATH + "admin/logs/get-content", {name: name});
        },
    },
    diagnostics: {
        getLatest: function() {
           return APIXHRService("GET", API_PATH + "admin/diagnostics/get-latest-diagnosis");
        },
        run: function(options) {
           return APIXHRService("POST", API_PATH + "admin/diagnostics/run", {options: JSON.stringify(options)});
        },
        getResults: function(id) {
           return APIXHRService("GET", API_PATH + "admin/diagnostics/get-results", {id: id});
        },
    },
    sanityCheck: {
        run: function(options) {
            return APIXHRService("POST", API_PATH + "admin/sanity-check/run", {exclusionList: JSON.stringify(options)});
        },
        getCodes: function() {
            return APIXHRService("GET", API_PATH + "admin/sanity-check/codes");
        },
        getExclusions: function() {
            return APIXHRService("GET", API_PATH + "admin/sanity-check/exclusions");
        },
        getLatestRun: function() {
            return APIXHRService("GET", API_PATH + "admin/sanity-check/latest-run");
        },
        getRunningJob: function() {
            return APIXHRService("GET", API_PATH + "admin/sanity-check/running-job");
        }
    },
    globalTags: {
        updateGlobalTags: function(data) {
            return APIXHRService("POST", API_PATH + "admin/global-tags/update-global-tags", {data: JSON.stringify(data)});
        }
    },
    users:{
        list: function(lastActivity) {
            return APIXHRService("GET", API_PATH + "admin/users/list", {lastActivity: lastActivity});
        },
        get: function(login) {
            return APIXHRService("GET", API_PATH + "admin/users/get", {login: login});
        },
        update: function(usr) {
            return APIXHRService("POST", API_PATH + "admin/users/edit", {user: JSON.stringify(usr)});
        },
        prepareUpdate: function(usr) {
            return APIXHRService("POST", API_PATH + "admin/users/prepare-edit", {user: JSON.stringify(usr)});
        },
        switchToTrial: function(login, userProfile) {
            return APIXHRService("POST", API_PATH + "admin/users/switch-to-trial", {login, userProfile})
        },
        convertFromTrial: function(login, action) {
            return APIXHRService("POST", API_PATH + "admin/users/convert-from-trial", {login, action})
        },
        create : function(usr) {
            return APIXHRService("POST", API_PATH + "admin/users/create", {user: JSON.stringify(usr)});
        },
        delete: function(logins) {
            return APIXHRService("POST", API_PATH + "admin/users/delete", {logins: JSON.stringify(logins)});
        },
        enableOrDisable: function(logins, enable) {
            return APIXHRService("POST", API_PATH + "admin/users/enable-or-disable", {logins: JSON.stringify(logins), enable: enable});
        },
        prepareDelete: function(logins) {
            return APIXHRService("POST", API_PATH + "admin/users/prepare-delete", {logins: JSON.stringify(logins)});
        },
        prepareDisable: function(logins) {
            return APIXHRService("POST", API_PATH + "admin/users/prepare-disable", {logins: JSON.stringify(logins)});
        },
        assignUsersGroups: function(logins, groupsToAdd, groupsToRemove) {
            return APIXHRService("POST", API_PATH + "admin/users/assign-groups", {groupsToAdd: JSON.stringify(groupsToAdd), groupsToRemove: JSON.stringify(groupsToRemove), logins: JSON.stringify(logins)});
        },
        prepareAssignUsersGroups: function(logins, groupsToAdd, groupsToRemove) {
            return APIXHRService("POST", API_PATH + "admin/users/prepare-assign-groups", {groupsToAdd: JSON.stringify(groupsToAdd), groupsToRemove: JSON.stringify(groupsToRemove), logins: JSON.stringify(logins)});
        },
        assignUsersProfile: function(logins, newProfile) {
            return APIXHRService("POST", API_PATH + "admin/users/assign-profile", {logins: JSON.stringify(logins), newProfile: newProfile});
        },
        getSyncableSourceTypes: function() {
            return APIXHRService("GET", API_PATH + "admin/users/get-syncable-source-types");
        },
        sync: function(login) {
            return APIXHRService("GET", API_PATH + "admin/users/sync", {logins: JSON.stringify([login])});
        },
        syncUsers: function(users) {
            return APIXHRService("GET", API_PATH + "admin/users/sync", {logins: users});
        },
        syncAll: function() {
            return APIXHRService("POST", API_PATH + "admin/users/sync");
        },
        getFetchableSourceTypes: function() {
            return APIXHRService("GET", API_PATH + "admin/users/get-fetchable-source-types");
        },
        fetchUsers: function(userSourceType, userQueryFilter) {
            return APIXHRService("POST", API_PATH + "admin/users/fetch", {userSourceType: userSourceType, userQueryFilter: JSON.stringify(userQueryFilter)});
        },
        fetchGroups: function(userSourceType) {
            return APIXHRService("GET", API_PATH + "admin/groups/fetch", {userSourceType: userSourceType});
        },
        provisionUsers: function(userSourceType, users) {
            return APIXHRService("POST", API_PATH + "admin/users/provision", {userSourceType: userSourceType, users: JSON.stringify(users)});
        }
    },
    publicApi: {
        listGlobalKeys: function() {
            return APIXHRService("GET", API_PATH + "admin/publicapi/get-global-keys");
        },
        createGlobalKey: function(key) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/create-global-key", {
                key: JSON.stringify(key)
            });
        },
        saveGlobalKey: function(key) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/save-global-key", {
                key: JSON.stringify(key)
            });
        },
        getGlobalKey: function(keyId) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/get-global-key", {
                keyId: keyId
            });
        },
        getGlobalKeyDetails: function(key) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/get-global-key-details", {
                key: key
            });
        },
        deleteGlobalKey: function(keyId) {
            return APIXHRService("POST", API_PATH + "admin/publicapi/delete-global-key", {
                keyId: keyId
            });
        },
        listPersonalKeys : function(){
            return APIXHRService("GET", API_PATH + "admin/publicapi/list-personal-api-keys");
        },
        deletePersonalKey : function(keyId){
            return APIXHRService("POST", API_PATH + "admin/publicapi/delete-personal-api-key",{
                id : keyId
            });
        }
    },
    scheduledTasks: {
        getStatus: function() {
            return APIXHRService("GET", API_PATH + "admin/scheduled-tasks/get-status", {
            });
        },
        fire: function(jobGroup, jobName) {
            return APIXHRService("POST", API_PATH + "admin/scheduled-tasks/fire", {
                jobGroup: jobGroup, jobName: jobName
            });
        }
    },
    codeenvs: {
        design: {
            list: function(){
                return APIXHRService("GET", API_PATH + "code-envs/design/list")
            },
            listNames: function(envLang) {
                return APIXHRService("GET", API_PATH + "code-envs/design/list-names",{ envLang: envLang })
            },
            create: function(envLang, envSpec) {
                return APIXHRService("POST", API_PATH + "code-envs/design/create", {
                    envLang: envLang, envSpec: JSON.stringify(envSpec)
                });
            },
            prepareDraftFromRequest: function(requestId) {
                return APIXHRService("POST", API_PATH + "code-envs/design/prepare-from-request", {
                    requestId: requestId
                });
            },
            prepareDraftFromProjectImport: function(envLang, envName, desc, specCondaEnvironment, specPackageList) {
                return APIXHRService("POST", API_PATH + "code-envs/design/prepare-from-project-import", {
                    envLang: envLang, envName: envName, desc: desc, specCondaEnvironment: specCondaEnvironment, specPackageList: specPackageList
                });
            },
            createFromDraft: function(draftId, envLang, envSpec) {
                return APIXHRService("POST", API_PATH + "code-envs/design/draft/" + draftId + "/create", {
                    envLang: envLang, envSpec: JSON.stringify(envSpec)
                });
            },
            get : function(envLang, envName){
                return APIXHRService("GET", API_PATH + "code-envs/design/get", {
                    envLang: envLang, envName: envName
                });
            },
            getDraft : function(draftId){
                return APIXHRService("GET", API_PATH + "code-envs/design/draft/" + draftId, {});
            },
            listPythonPackagePresets : function(interpreter, useConda, corePackageSet) {
                return APIXHRService("GET", API_PATH + "code-envs/design/package-presets/python", {
                    interpreter, useConda, corePackageSet,
                });
            },

            updatePythonPackagePresets : function(specPackageList, interpreter) {
                return APIXHRService("GET", API_PATH + "code-envs/design/update-package-presets/python", {
                    specPackageList: specPackageList, interpreter: interpreter
                });
            },
            listLogs : function(envLang, envName){
                return APIXHRService("POST", API_PATH + "code-envs/design/list-logs", {
                    envLang: envLang, envName: envName
                });
            },
            getLog : function(envLang, envName, logName){
                return APIXHRService("POST", API_PATH + "code-envs/design/get-log", {
                    envLang: envLang, envName: envName, logName: logName
                });
            },
            streamLog : function(envLang, envName, logName){
                return APIXHRService("GET", API_PATH + "code-envs/design/stream-log", {
                    envLang: envLang, envName: envName, logName: logName
                });
            },
            save: function(envLang, envName, data) {
                return APIXHRService("POST", API_PATH + "code-envs/design/save", {
                    envLang: envLang, envName: envName, data: JSON.stringify(data)
                });
            },
            listUsages : function(envLang, envName, globalOnly){
                return APIXHRService("POST", API_PATH + "code-envs/design/list-usages", {
                    envLang: envLang, envName: envName, globalOnly: globalOnly
                });
            },
            delete : function(envLang, envName){
                return APIXHRService("POST", API_PATH + "code-envs/design/delete", {
                    envLang: envLang, envName: envName
                });
            },
            update: function(envLang, envName, updateSettings) {
                return APIXHRService("POST", API_PATH + "code-envs/design/update", {
                    envLang: envLang, envName: envName, updateSettings: JSON.stringify(updateSettings)
                });
            },
            recreate: function(envLang, envName) {
                return APIXHRService("POST", API_PATH + "code-envs/design/recreate", {
                    envLang: envLang, envName: envName
                });
            },
            fetchNonManagedEnvDetails: function(envLang, envName) {
                return APIXHRService("GET", API_PATH + "code-envs/design/fetch-non-managed-env-details", {
                    envLang: envLang, envName: envName
                });
            },
            installJupyterSupport: function(envLang, envName) {
                return APIXHRService("GET", API_PATH + "code-envs/design/install-jupyter-support", {
                    envLang: envLang, envName: envName
                });
            },
            removeJupyterSupport: function(envLang, envName) {
                return APIXHRService("GET", API_PATH + "code-envs/design/remove-jupyter-support", {
                    envLang: envLang, envName: envName
                });
            },
            getExportURL: function(envLang, envName) {
                return API_PATH + "code-envs/design/export?"
                         + "envLang=" + encodeURIComponent(envLang)
                         + "&envName=" + encodeURIComponent(envName);
            },
            getDiagnosticURL: function(envLang, envName) {
                return API_PATH + "code-envs/design/download-diagnostic?"
                         + "envLang=" + encodeURIComponent(envLang)
                         + "&envName=" + encodeURIComponent(envName);
            },
            browseResources : function(envLang, envName, relativePath, withDirectoriesSize){
                return APIXHRService("POST", API_PATH + "code-envs/design/browse-resources", {
                    envLang: envLang,
                    envName: envName,
                    relativePath: relativePath,
                    withDirectoriesSize: withDirectoriesSize
                });
            },
            uploadResources: function(envLang, envName, relativePath, file, overwrite) {
                return uploadFileRequest("code-envs/design/upload-resources", function(formdata) {
                    formdata.append("envLang", envLang);
                    formdata.append("envName", envName);
                    formdata.append("relativePath", relativePath);
                    formdata.append("file", file);
                    formdata.append("overwrite", overwrite);
                }, null);
            },
            clearResources : function(envLang, envName){
                return APIXHRService("POST", API_PATH + "code-envs/design/clear-resources", {
                    envLang: envLang, envName: envName
                });
            },
            getResourcesEnvVars : function(envLang, envName){
                return APIXHRService("POST", API_PATH + "code-envs/design/resources-env-vars", {
                    envLang: envLang, envName: envName
                });
            },
            import: function(file) {
                return uploadFileRequest("code-envs/design/import", function(formdata) {
                    formdata.append("file", file);
                }, null);
            }
        },
        automation: {
            list: function(){
                return APIXHRService("GET", API_PATH + "code-envs/automation/list")
            },
            create: function(envLang, envSpec) {
                 return APIXHRService("POST", API_PATH + "code-envs/automation/create", {
                    envLang: envLang, envSpec:JSON.stringify(envSpec)
                });
            },
            get: function(envLang, envName) {
                return APIXHRService("POST", API_PATH + "code-envs/automation/get", {
                    envLang: envLang, envName: envName
                });
            },
            getVersion: function(envLang, envName, versionId) {
                return APIXHRService("POST", API_PATH + "code-envs/automation/get-version", {
                    envLang: envLang, envName: envName, versionId: versionId
                });
            },
            save: function(envLang, envName, data) {
                return APIXHRService("POST", API_PATH + "code-envs/automation/save", {
                    envLang: envLang, envName: envName, data: JSON.stringify(data)
                });
            },
            saveVersion: function(envLang, envName, versionId, data) {
                return APIXHRService("POST", API_PATH + "code-envs/automation/save-version", {
                    envLang: envLang, envName: envName, versionId: versionId, data: JSON.stringify(data)
                });
            },
            listLogs : function(envLang, envName){
                return APIXHRService("POST", API_PATH + "code-envs/automation/list-logs", {
                    envLang: envLang, envName: envName
                });
            },
            getLog : function(envLang, envName, logName){
                return APIXHRService("POST", API_PATH + "code-envs/automation/get-log", {
                    envLang: envLang, envName: envName, logName: logName
                });
            },
            streamLog : function(envLang, envName, logName){
                return APIXHRService("GET", API_PATH + "code-envs/automation/stream-log", {
                    envLang: envLang, envName: envName, logName: logName
                });
            },
            listUsages : function(envLang, envName, globalOnly){
                return APIXHRService("POST", API_PATH + "code-envs/automation/list-usages", {
                    envLang: envLang, envName: envName, globalOnly: globalOnly
                });
            },
            delete : function(envLang, envName){
                return APIXHRService("POST", API_PATH + "code-envs/automation/delete", {
                    envLang: envLang, envName: envName
                });
            },
            update: function(envLang, envName, updateSettings) {
                return APIXHRService("POST", API_PATH + "code-envs/automation/update", {
                    envLang: envLang, envName: envName, updateSettings: JSON.stringify(updateSettings)
                });
            },
            fetchNonManagedEnvDetails: function(envLang, envName) {
                return APIXHRService("GET", API_PATH + "code-envs/automation/fetch-non-managed-env-details", {
                    envLang: envLang, envName: envName
                });
            },
            installJupyterSupport: function(envLang, envName, versionId) {
                return APIXHRService("GET", API_PATH + "code-envs/automation/install-jupyter-support", {
                    envLang: envLang, envName: envName, versionId: versionId
                });
            },
            removeJupyterSupport: function(envLang, envName, versionId) {
                return APIXHRService("GET", API_PATH + "code-envs/automation/remove-jupyter-support", {
                    envLang: envLang, envName: envName, versionId: versionId
                });
            },
            import: function(file) {
                return uploadFileRequest("code-envs/automation/import", function(formdata) {
                    formdata.append("file", file);
                }, null);
            },
            importVersion: function(file, envLang, envName) {
                return uploadFileRequest("code-envs/automation/import-version", function(formdata) {
                    formdata.append("envName", envName);
                    formdata.append("envLang", envLang);
                    formdata.append("file", file);
                }, null);
            },
            browseResources : function(envLang, envName, versionId, relativePath, withDirectoriesSize){
                return APIXHRService("POST", API_PATH + "code-envs/automation/browse-resources", {
                    envLang: envLang,
                    envName: envName,
                    versionId: versionId,
                    relativePath: relativePath,
                    withDirectoriesSize: withDirectoriesSize
                });
            },
            uploadResources: function(envLang, envName, versionId, relativePath, file, overwrite) {
                return uploadFileRequest("code-envs/automation/upload-resources", function(formdata) {
                    formdata.append("envLang", envLang);
                    formdata.append("envName", envName);
                    if (versionId !== undefined) {
                        formdata.append("versionId", versionId);
                    }
                    formdata.append("relativePath", relativePath);
                    formdata.append("file", file);
                    formdata.append("overwrite", overwrite);
                }, null);
            },
            clearResources : function(envLang, envName, versionId){
                return APIXHRService("POST", API_PATH + "code-envs/automation/clear-resources", {
                    envLang: envLang, envName: envName, versionId: versionId
                });
            },
            getResourcesEnvVars : function(envLang, envName, versionId){
                return APIXHRService("POST", API_PATH + "code-envs/automation/resources-env-vars", {
                    envLang: envLang, envName: envName, versionId: versionId
                });
            },
            getDiagnosticURL: function(envLang, envName) {
                return API_PATH + "code-envs/automation/download-diagnostic?"
                         + "envLang=" + encodeURIComponent(envLang)
                         + "&envName=" + encodeURIComponent(envName);
            }
        }
    },
    costLimiting: {
        getCounters: function(nospinner) {
            return APIXHRService("GET", API_PATH + "admin/get-cost-limit-counters", {}, nospinner ? "nospinner" : undefined);
        },
        getProgress: function(nospinner) {
            return APIXHRService("GET", API_PATH + "admin/get-cost-limit-progress", {}, nospinner ? "nospinner" : undefined);
        },
        getQuotas: function() {
            return APIXHRService("GET", API_PATH + "admin/get-cost-limit-quotas", {});
        }
    },
    rateLimiting: {
        getDefaults: function() {
            return APIXHRService("GET", API_PATH + "admin/llm-rate-limit-defaults", {});
        }
    },
    monitoring: {
        getGlobalUsageSummary: function() {
            return APIXHRService("GET", API_PATH + "admin/monitoring/get-global-usage-summary", {
            }, "nospinner");
        },
        getConnectionTasksHistory: function(connectionId) {
            return APIXHRService("GET", API_PATH + "admin/monitoring/ct/get-history", {
                connectionId: connectionId
            });
        },
        connectionData: {
            get: function(connectionId) {
            return APIXHRService("GET", API_PATH + "admin/monitoring/cd/get", {
                    connectionId: connectionId
                });
            },
            getForProject: function(connectionId, projectKey) {
            return APIXHRService("GET", API_PATH + "admin/monitoring/cd/get-for-project", {
                    connectionId: connectionId, projectKey: projectKey
                });
            },
            updateForProject: function(connectionId, projectKey, computeRecords, forceRecompute) {
            return APIXHRService("POST", API_PATH + "admin/monitoring/cd/update-for-project", {
                    connectionId: connectionId, projectKey: projectKey,
                    computeRecords: computeRecords, forceRecompute: forceRecompute
                });
            },
            updateForDataset: function(projectKey, datasetName, computeRecords, forceRecompute) {
            return APIXHRService("POST", API_PATH + "admin/monitoring/cd/update-for-dataset", {
                    projectKey: projectKey, datasetName: datasetName,
                    computeRecords: computeRecords, forceRecompute: forceRecompute
                });
            }
        },
        getProjectsIntegrations: function() {
            return APIXHRService("GET", API_PATH + "admin/monitoring/get-projects-integrations", {
            });
        },
        saveProjectIntegration: function(projectKey, data) {
            return APIXHRService("POST", API_PATH + "admin/monitoring/save-project-integration", {
                projectKey: projectKey,
                data: JSON.stringify(data)
            });
        },
        deleteProjectIntegration: function(projectKey, data) {
            return APIXHRService("POST", API_PATH + "admin/monitoring/delete-project-integration", {
                projectKey: projectKey,
                data: JSON.stringify(data)
            });
        },
    },
    clusters:{
        list: function() {
            return APIXHRService("GET", API_PATH + "clusters/list");
        },
        listAccessible: function(architecture) {
            return APIXHRService("GET", API_PATH + "clusters/list-accessible", {architecture: architecture});
        },
        get: function(clusterId) {
            return APIXHRService("GET", API_PATH + "clusters/get", {id: clusterId});
        },
        getStatus: function(clusterId) {
            return APIXHRService("GET", API_PATH + "clusters/get-status", {id: clusterId});
        },
        save: function(cluster) {
            return APIXHRService("POST", API_PATH + "clusters/save", {data: JSON.stringify(cluster)});
        },
        create : function(cluster) {
            return APIXHRService("POST", API_PATH + "clusters/create", {data: JSON.stringify(cluster)});
        },
        delete: function(clusterId) {
            return APIXHRService("POST", API_PATH + "clusters/delete", {id: clusterId});
        },
        start: function(clusterId) {
            return APIXHRService("POST", API_PATH + "clusters/start", {id: clusterId});
        },
        stop: function(clusterId, terminate, forceStop) {
            return APIXHRService("POST", API_PATH + "clusters/stop", {id: clusterId, terminate: terminate, forceStop: forceStop});
        },
        markStopped: function(clusterId) {
            return APIXHRService("POST", API_PATH + "clusters/mark-stopped", {id: clusterId});
        },
        listLogs : function(clusterId){
            return APIXHRService("POST", API_PATH + "clusters/list-logs", {
                clusterId: clusterId
            });
        },
        getLog : function(clusterId, logName){
            return APIXHRService("POST", API_PATH + "clusters/get-log", {
                clusterId: clusterId, logName: logName
            });
        },
        streamLog : function(clusterId, logName){
            return APIXHRService("GET", API_PATH + "clusters/stream-log", {
                clusterId: clusterId, logName: logName
            });
        },
        getDiagnosisURL: function(clusterId) {
            return API_PATH + "clusters/download-diagnosis?"
                     + "clusterId=" + encodeURIComponent(clusterId);
        },
        abortKernel : function(prefix, kernelId) {
            return APIXHRService("POST", API_PATH + "clusters/abort-kernel", {
                kernelId: kernelId, prefix: prefix
            });
        },
        preloadYarnClusterFiles : function(yarnClusterSettings) {
            return APIXHRService("POST", API_PATH + "clusters/preload-yarn-cluster-files", {
                yarnClusterSettings: JSON.stringify(yarnClusterSettings)
            });
        },
        runKubectl: function(clusterId, args) {
            return APIXHRService("POST", API_PATH + "clusters/k8s/actions/run-kubectl", {
                clusterId : clusterId,
                args: args
            });
        },
        deleteFinishedJobs: function(clusterId, dryRun, deleteFailed, namespace, labelFilter) {
            return APIXHRService("POST", API_PATH + "clusters/k8s/jobs/actions/delete-finished", {
                clusterId,
                dryRun,
                deleteFailed,
                namespace,
                labelFilter
            });
        },
        deleteFinishedPods: function(clusterId, dryRun, namespace, labelFilter) {
            return APIXHRService("POST", API_PATH + "clusters/k8s/pods/actions/delete-finished", {
                clusterId,
                dryRun,
                namespace,
                labelFilter
            });
        },
        deleteAllPods: function(clusterId, dryRun, namespace, labelFilter) {
            return APIXHRService("POST", API_PATH + "clusters/k8s/pods/actions/delete-all", {
                clusterId,
                dryRun,
                namespace,
                labelFilter
            });
        },
        countNodesAndPods: function(clusterId) {
            return APIXHRService("GET", API_PATH + "clusters/k8s/count-nodes-and-pods", {
                clusterId : clusterId
            });
        },
        hasMetricsServer: function(clusterId) {
            return APIXHRService("POST", API_PATH + "clusters/k8s/has-metrics-server", {
                clusterId : clusterId
            });
        },
        getNodesMetrics: function(clusterId) {
            return APIXHRService("POST", API_PATH + "clusters/k8s/nodes-metrics", {
                clusterId : clusterId
            });
        },
        monitoring: {
            getNodes: function(clusterId, detailedNodeData) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/nodes", {
                    clusterId, detailedNodeData
                });
            },
            getPods: function(clusterId, namespaceFilterType, namespace, detailedPodData) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/pods", {
                    clusterId, namespaceFilterType, namespace, detailedPodData
                });
            },
            getJobs: function(clusterId, namespaceFilterType, namespace) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/jobs", {
                    clusterId : clusterId,
                    namespaceFilterType : namespaceFilterType,
                    namespace : namespace
                });
            },
            getDeployments: function(clusterId, namespaceFilterType, namespace) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/deployments", {
                    clusterId : clusterId,
                    namespaceFilterType : namespaceFilterType,
                    namespace : namespace
                });
            },
            getServices: function(clusterId, namespaceFilterType, namespace) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/services", {
                    clusterId : clusterId,
                    namespaceFilterType : namespaceFilterType,
                    namespace : namespace
                });
            },
            getNamespaces: function(clusterId) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/namespaces", {
                    clusterId : clusterId
                });
            },
            describeObject: function(clusterId, objectType, objectName, namespace) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/describe-object", {
                    clusterId : clusterId,
                    objectType : objectType,
                    objectName : objectName,
                    namespace : namespace
                });
            },
            deleteObject: function(clusterId, objectType, objectName, namespace) {
                return APIXHRService("POST", API_PATH + "clusters/k8s/monitoring/delete-object", {
                    clusterId : clusterId,
                    objectType : objectType,
                    objectName : objectName,
                    namespace : namespace
                });
            },
            tailPodLog: function(clusterId, podName, namespace) {
                return APIXHRService("GET", API_PATH + "clusters/k8s/monitoring/tail-pod-log", {
                    clusterId, podName, namespace
                })
            }
        }
    },
    getGeneralSettings: function() {
        return APIXHRService("GET", API_PATH + "admin/get-general-settings");
    },
    fetchOpenIDConfig: function(wellKnownURL) {
        return APIXHRService("GET", API_PATH + "admin/fetch-openid-config", { wellKnownURL: wellKnownURL });
    },
    getGlobalVariables: function() {
        return APIXHRService("GET", API_PATH + "admin/get-global-variables");
    },
    saveGlobalVariables: function(data) {
        return APIXHRService("POST", API_PATH + "admin/save-global-variables", {data: JSON.stringify(data)});
    },
    saveGeneralSettings: function(data) {
        return APIXHRService("POST", API_PATH + "admin/save-general-settings", {data: JSON.stringify(data)});
    },
    invalidateConfigCache: function(path) {
        return APIXHRService("POST", API_PATH + "admin/invalidate-config-cache", {path});
    },
    getModelCacheDetails: function() {
        return APIXHRService("GET", API_PATH + "admin/model-cache-details");
    },
    testWelcomeEmail: function(email, welcomeEmailSettings, notificationsSettings, studioMailAddress, integrationChannels) {
        return APIXHRService("POST", API_PATH + "admin/test-welcome-email", {
            email,
            studioMailAddress,
            welcomeEmailSettings: JSON.stringify(welcomeEmailSettings),
            notificationsSettings: JSON.stringify(notificationsSettings),
            integrationChannels: JSON.stringify(integrationChannels),
        });
    },
    importModel: function(files) {
        return uploadFileRequest("admin/model-cache", function(formdata) {
            for (const file of files) {
                formdata.append("files", file);
            }
        }, null);
    },
    deleteModelFromCache: function(modelKey) {
        return APIXHRService("GET", API_PATH + "admin/model-cache/delete-model",  {
            "modelKey": modelKey
        });
    },
    clearModelCache: function() {
        return APIXHRService("DELETE", API_PATH + "admin/model-cache");
    },
    testLdapSettings: function(data) {
        return APIXHRService("POST", API_PATH + "admin/test-ldap-settings", {data: JSON.stringify(data)});
    },
    testLdapGetUserDetails: function(data) {
        return APIXHRService("POST", API_PATH + "admin/test-ldap-get-user-details", {data: JSON.stringify(data)});
    },
    testAzureADSettings: function(data) {
        return APIXHRService("POST", API_PATH + "admin/test-azure-ad-settings", {data: JSON.stringify(data)});
    },
    executeVariablesUpdate: function() {
        return APIXHRService("POST", API_PATH + "admin/execute-variables-update");
    },
    getLimitsStatus: function() {
        return APIXHRService("GET", API_PATH + "admin/get-limits-status");
    },
    getTrialStatus: function() {
        return APIXHRService("GET", API_PATH + "admin/get-trial-status");
    },
    getThemes: function() {
        return APIXHRService("GET", API_PATH + "admin/get-themes");
    },
    folderEdit: {
        listContents: function(type) {
            return APIXHRService("GET", API_PATH + "admin/folder-edition/list-contents", {
                type: type
            });
        },
        getContent: function(type, path, sendAnyway) {
            return APIXHRService("GET", API_PATH + "admin/folder-edition/get-content", {
                type: type, path: path, sendAnyway: sendAnyway
            });
        },
        setContent: function(type, path, data) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/set-content", {
                type: type, path: path, data: data
            });
        },
        setContentMultiple: function(type, contentMap) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/set-content-multiple", {
                type: type, contentMap: JSON.stringify(contentMap)
            });
        },
        createContent: function(type, path, isFolder) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/create-content", {
                type: type, path: path, isFolder: isFolder
            });
        },
        deleteContent: function(type, path) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/delete-content", {
                type: type, path: path
            });
        },
        decompressContent: function(type, path) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/decompress-content", {
                type: type, path: path
            });
        },
        renameContent: function(type, path, newName) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/rename-content", {
                type: type, path: path, newName: newName
            });
        },
        moveContent: function(type, path, toPath) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/move-content", {
                type: type, path: path, toPath: toPath
            });
        },
        copyContent: function(type, path) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/copy-content", {
                type: type, path: path
            });
        },
        uploadContent: function(type, path, file, callback) {
            return uploadFileRequest("admin/folder-edition/upload-content", function(formdata) {
                formdata.append("type", type);
                formdata.append("path", path);
                formdata.append("file", file);
            }, callback);
        },
        checkUploadContent: function(type, path, filePaths) {
            return APIXHRService("POST", API_PATH + "admin/folder-edition/check-upload-content", {
                type: type, path: path, filePaths: JSON.stringify(filePaths)
            });
        }
    },
    govern: {
        test: function() {
            return APIXHRService("GET", API_PATH + "admin/govern-integration-test");
        },
        sync: function(projectKey) {
            return APIXHRService("POST", API_PATH + "admin/govern-integration-sync", {
                projectKey: projectKey
            });
        }  ,
        deployerSync: function() {
            return APIXHRService("POST", API_PATH + "admin/govern-integration-deployer-sync", {
            });
        }
    },
    datastory: {
        check: function() {
            return APIXHRService("GET", API_PATH + "admin/datastory-integration-check");
        }
    }
},
plugins: {
    get: function(pluginId, projectKey) {
        return APIXHRService("GET", API_PATH + "plugins/get", {
            pluginId: pluginId,
            projectKey: projectKey
        });
    },
    list: function(forceFetch) {
        return APIXHRService("GET", API_PATH + "plugins/list", {forceFetch: forceFetch});
    },
    listPluginsWithPresets: function() {
        return APIXHRService("GET", API_PATH + "plugins/list-plugins-with-presets");
    },
    install: function(pluginId, update) {
        return APIXHRService("GET", API_PATH + "plugins/install-from-store", {pluginId: pluginId, update: update});
    },
    triggerRestart: function(pluginId, update) {
        return APIXHRService("GET", API_PATH + "plugins/trigger-restart");
    },
    uploadPlugin: function(file, isUpdate) {
        return uploadFileRequest("plugins/upload", function(formdata) {
            formdata.append("file", file);
            formdata.append("isUpdate", isUpdate);
        }, null);
    },
    clonePlugin: function(uri, checkout, path, isUpdate) {
        return APIXHRService("POST", API_PATH + "plugins/clone", {
            repository: uri,
            checkout: checkout,
            path: path,
            isUpdate: isUpdate
        });
    },
    moveToDev: function(pluginId) {
        return APIXHRService("POST", API_PATH + "plugins/move-to-dev", {
            pluginId: pluginId
        });
    },
    useCodeEnv: function(pluginId, envName) {
        return APIXHRService("POST", API_PATH + "plugins/use-code-env", {
            pluginId: pluginId, envName: envName
        });
    },
    saveSettings: function(pluginId, projectKey, settings) {
        return APIXHRService("POST", API_PATH + "plugins/save-settings", {
            pluginId: pluginId,
            projectKey: projectKey,
            data: JSON.stringify(settings)
        });
    },
    installRequirements: function(pluginId, type, envName) {
        return APIXHRService("GET", API_PATH + "plugins/install-requirements", {pluginId: pluginId, type: type, envName: envName});
    },
    callPythonDo: function(sessionId, pluginId, componentId, config, payload, recipeConfig, projectKey, clusterId, part) {
        return APIXHRService("POST", API_PATH + "plugins/call-python-do", {
            sessionId: sessionId,
            pluginId: pluginId,
            componentId: componentId,
            projectKey: projectKey,
            clusterId: clusterId,
            config: JSON.stringify(config),
            payload: (payload != null ? JSON.stringify(payload) : null),
            recipeConfig: (recipeConfig != null ? JSON.stringify(recipeConfig) : null),
            part: part
        });
    },
    listAccessiblePresets: function(pluginId, projectKey, elementId) {
        return APIXHRService("GET", API_PATH + "plugins/list-accessible-presets", {pluginId: pluginId, projectKey: projectKey, elementId: elementId});
    },
    getUsages: function(pluginId, projectKey) {
        return APIXHRService("GET", API_PATH + "plugins/get-usages", {pluginId: pluginId, projectKey: projectKey});
    },
    prepareDelete: function(pluginId) {
        return APIXHRService("GET", API_PATH + "plugins/prepare-delete", {pluginId: pluginId});
    },
    delete: function(pluginId, force = false) {
        return APIXHRService("GET", API_PATH + "plugins/delete", {pluginId: pluginId, force: force});
    },
    canAdminPlugin: function(pluginId) {
        return APIXHRService("GET", API_PATH + "plugins/can-admin-plugin", {pluginId: pluginId});
    }
},
plugindev: {
    git: {
        getLog: function(pluginId, since, count) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-log", {
                pluginId: pluginId,
                since: since,
                count: count
            });
        },
        pull: function(pluginId, remoteName, branchName) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-pull", {
                pluginId: pluginId,
                remoteName: remoteName,
                branchName: branchName
            });
        },
        fetch: function(pluginId, remoteName) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-fetch", {
                pluginId: pluginId,
                remoteName: remoteName
            });
        },
        push: function(pluginId, remoteName, branchName) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-push", {
                pluginId: pluginId,
                remoteName: remoteName,
                branchName: branchName
            });
        },
        resetToUpstream: function(pluginId, remoteName, branchName) {
            return APIXHRService("POST", API_PATH + "plugins-git/git-reset-to-upstream", {
                pluginId: pluginId,
                remoteName: remoteName,
                branchName: branchName
            });
        },
        resetToHead: function(pluginId) {
            return APIXHRService("POST", API_PATH + "plugins-git/git-reset-to-head", {
                pluginId: pluginId
            });
        },
        getFullStatus: function(pluginId) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-full-status", {
                pluginId: pluginId
            });
        },
        listBranches: function(pluginId) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-list-branches", {
                pluginId: pluginId
            });
        },
        commit: function(pluginId, commitMessage) {
            return APIXHRService("GET", API_PATH + "plugins-git/commit", {
                pluginId: pluginId,
                commitMessage: commitMessage
            });
        },
        prepareCommit: function(pluginId) {
            return APIXHRService("GET", API_PATH + "plugins-git/prepare-commit", {
                pluginId: pluginId
            });
        },
        createBranch: function(pluginId, branchName, commitId) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-create-branch", {
                pluginId: pluginId,
                branchName: branchName,
                commitId: commitId
            });
        },
        switchBranch: function(pluginId, branchName, creation) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-switch-branch", {
                pluginId: pluginId,
                branchName: branchName
            });
        },
        deleteBranches: function(pluginId, /*String[]*/branchNames, deleteOptions) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-delete-branches", {
                pluginId: pluginId,
                branchNames: JSON.stringify(branchNames),
                remoteDelete: deleteOptions.remoteDelete,
                forceDelete: deleteOptions.forceDelete
            });
        },
        setRemote: function(pluginId, remoteName, remoteUrl) {
            return APIXHRService("POST", API_PATH + "plugins-git/git-set-remote", {
                pluginId: pluginId,
                remoteName: remoteName,
                remoteUrl: remoteUrl
            });
        },
        removeRemote: function(pluginId, remoteName) {
            return APIXHRService("GET", API_PATH + "plugins-git/git-rm-remote", {
                pluginId: pluginId,
                remoteName: remoteName
            });
        },
        getCommitDiff: function(pluginId, commitId) {
            return APIXHRService("GET", API_PATH + "plugins-git/get-commit-diff", {
                pluginId: pluginId,
                commitId: commitId
            });
        },
        getRevisionsDiff: function(pluginId, commitFrom, commitTo) {
            return APIXHRService("GET", API_PATH + "plugins-git/get-revisions-diff", {
                pluginId: pluginId,
                commitFrom: commitFrom,
                commitTo: commitTo
            });
        },
        revertPluginToRevision: function(pluginId, hash) {
            return APIXHRService("GET", API_PATH + "plugins-git/revert-plugin-to-revision", {
                pluginId: pluginId,
                hash: hash
            });
        },
        revertSingleCommit: function(pluginId, hash) {
            return APIXHRService("GET", API_PATH + "plugins-git/revert-single-commit", {
                pluginId: pluginId,
                hash: hash
            });
        },
        addTag: function(pluginId, reference, name, message=null) {
            return APIXHRService("POST", API_PATH + "plugins-git/tag", {
                pluginId, reference, name, message
            });
        },
        removeTag: function(pluginId, name) {
            return APIXHRService("POST", API_PATH + "plugins-git/remove-tag", {
                pluginId, name
            });
        },
    },
    reloadAll: function() {
        return APIXHRService("GET", API_PATH + "plugins/dev/reload-all");
    },
    list: function() {
       return APIXHRService("GET", API_PATH + "plugins/dev/list");
    },
    get: function(pluginId) {
        return APIXHRService("GET", API_PATH + "plugins/dev/get", {
            pluginId: pluginId
        });
    },
    create: function(pluginId, bootstrapMode, gitRepository, gitCheckout, gitPath) {
        return APIXHRService("POST", API_PATH + "plugins/dev/create", {
            pluginId: pluginId,
            bootstrapMode: bootstrapMode,
            gitRepository: gitRepository,
            gitCheckout: gitCheckout,
            gitPath: gitPath
        });
    },
    delete: function(pluginId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/delete", {
            pluginId: pluginId
        });
    },
    reload: function(pluginId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/reload", {
            pluginId: pluginId
        });
    },
    addPythonDataset: function(pluginId, datasetId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-dataset", {
            pluginId: pluginId,
            datasetId: datasetId
        });
    },
    addJavaDataset: function(pluginId, datasetId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-dataset", {
            pluginId: pluginId,
            datasetId: datasetId,
            className: classNameForPlugin
        });
    },
    addJavaRecipe: function(pluginId, recipeId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-recipe", {
            pluginId: pluginId,
            recipeId: recipeId,
            className: classNameForPlugin
        });
    },
    addJavaDialect: function(pluginId, dialectId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-dialect", {
            pluginId: pluginId,
            dialectId: dialectId,
            className: classNameForPlugin
        });
    },
    addJavaExposition: function(pluginId, expositionId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-exposition", {
            pluginId: pluginId,
            expositionId: expositionId,
            className: classNameForPlugin
        });
    },
    addSampleDataset: function(pluginId, label) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-sample-dataset", {
            pluginId: pluginId,
            label: label
        });
    },
    addJythonProcessor: function(pluginId, stepId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-jython-processor", {
            pluginId: pluginId,
            stepId: stepId
        });
    },
    addPythonFormat: function(pluginId, formatId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-format", {
            pluginId: pluginId,
            formatId: formatId
        });
    },
    addJavaFormat: function(pluginId, formatId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-format", {
            pluginId: pluginId,
            formatId: formatId,
            className: classNameForPlugin
        });
    },
    addPythonProbe: function(pluginId, probeId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-probe", {
            pluginId: pluginId,
            probeId: probeId
        });
    },
    addPythonExporter: function(pluginId, exporterId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-exporter", {
            pluginId: pluginId,
            exporterId: exporterId
        });
    },
    addJavaExporter: function(pluginId, exporterId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-exporter", {
            pluginId: pluginId,
            exporterId: exporterId,
            className: classNameForPlugin
        });
    },
    addPythonCheck: function(pluginId, checkId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-check", {
            pluginId: pluginId,
            checkId: checkId
        });
    },
    addCustomFields: function(pluginId, customFieldsId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-custom-fields", {
            pluginId: pluginId,
            customFieldsId: customFieldsId
        });
    },
    addJavaPolicyHooks: function(pluginId, policyHooksId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-policy-hooks", {
            pluginId: pluginId,
            policyHooksId: policyHooksId,
            className: classNameForPlugin
        });
    },
    addJavaUserSupplier: function(pluginId, userSupplierId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-user-supplier", {
            pluginId: pluginId,
            userSupplierId: userSupplierId,
            className: classNameForPlugin
        });
    },
    addJavaUserAuthenticator: function(pluginId, userSupplierId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-user-authenticator", {
            pluginId: pluginId,
            userSupplierId: userSupplierId,
            className: classNameForPlugin
        });
    },
    addJavaUserAuthenticatorAndSupplier: function(pluginId, userSupplierId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-user-authenticator-and-supplier", {
            pluginId: pluginId,
            userSupplierId: userSupplierId,
            className: classNameForPlugin
        });
    },
    addSqlProbe: function(pluginId, probeId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-sql-probe", {
            pluginId: pluginId,
            probeId: probeId
        });
    },
    addPredictionPythonAlgorithm: function(pluginId, algoId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-prediction-algorithm", {
            pluginId, algoId
        });
    },
    addStandardWebAppTemplate: function(pluginId, webAppId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-standard-webapp-template", {
            pluginId: pluginId,
            webAppId: webAppId
        });
    },
    addBokehWebAppTemplate: function(pluginId, webAppId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-bokeh-webapp-template", {
            pluginId: pluginId,
            webAppId: webAppId
        });
    },
    addDashWebAppTemplate: function(pluginId, webAppId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-dash-webapp-template", {
            pluginId: pluginId,
            webAppId: webAppId
        });
    },
    addShinyWebAppTemplate: function(pluginId, webAppId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-shiny-webapp-template", {
            pluginId: pluginId,
            webAppId: webAppId
        });
    },
    addStreamlitWebAppTemplate: function(pluginId, webAppId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-streamlit-webapp-template", {
            pluginId: pluginId,
            webAppId: webAppId
        });
    },
    addRMarkdownReportTemplate: function(pluginId, reportId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-rmarkdown-report-template", {
            pluginId: pluginId,
            reportId: reportId
        });
    },
    addNotebookTemplate: function(type, language, preBuilt, pluginId, notebookId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-notebook-template", {
            pluginId: pluginId,
            notebookId: notebookId,
            type:type,
            language : language,
            preBuilt : preBuilt
        });
    },
    addPythonCluster: function(pluginId, clusterId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-cluster", {
            pluginId: pluginId,
            clusterId: clusterId
        });
    },
    addPythonCodeStudio: function(pluginId, codeStudioId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-code-studio", {
            pluginId: pluginId,
            codeStudioId: codeStudioId
        });
    },
    addPythonCodeStudioBlock: function(pluginId, codeStudioBlockId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-code-studio-block", {
            pluginId: pluginId,
            codeStudioBlockId: codeStudioBlockId
        });
    },
    addPythonProjectStandardsCheckSpec: function(pluginId, projectStandardsCheckSpecId) {
        return  APIXHRService("POST", API_PATH + "plugins/dev/add-python-project-standards-check-spec", {
            pluginId: pluginId,
            projectStandardsCheckSpecId: projectStandardsCheckSpecId
        });
    },
    addPythonStep: function(pluginId, stepId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-step", {
            pluginId: pluginId,
            stepId: stepId
        });
    },
    addPythonTrigger: function(pluginId, triggerId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-trigger", {
            pluginId: pluginId,
            triggerId: triggerId
        });
    },
    addPythonRunnable: function(pluginId, runnableId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-runnable", {
            pluginId: pluginId,
            runnableId: runnableId
        });
    },
    addJavaRunnable: function(pluginId, runnableId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-runnable", {
            pluginId: pluginId,
            runnableId: runnableId,
            className : classNameForPlugin
        });
    },
    addCustomCodeRecipe: function(pluginId, customCodeRecipeId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-custom-code-recipe", {
            pluginId: pluginId,
            customCodeRecipeId: customCodeRecipeId
        });
    },
    addPythonFSProvider: function(pluginId, fsProviderId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-fs-provider", {
            pluginId: pluginId,
            fsProviderId: fsProviderId
        });
    },
    addJavaFSProvider: function(pluginId, fsProviderId, classNameForPlugin) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-java-fs-provider", {
            pluginId: pluginId,
            fsProviderId: fsProviderId,
            className : classNameForPlugin
        });
    },
    addPythonCodeEnv: function(pluginId, unused, unused2, forceConda) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-python-code-env", {
            pluginId: pluginId,
            forceConda: forceConda
        });
    },
    addRCodeEnv: function(pluginId, unused, unused2, forceConda) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-r-code-env", {
            pluginId: pluginId,
            forceConda: forceConda
        });
    },
    addParameterSet: function(pluginId, parameterSetId) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-parameter-set", {
            pluginId: pluginId,
            parameterSetId: parameterSetId
        });
    },
    addGuardrail: function(pluginId, guardrailId) {
        return  APIXHRService("POST", API_PATH + "plugins/dev/add-guardrail", {
            pluginId: pluginId,
            guardrailId: guardrailId
        });
    },
    addAgent: function(pluginId, agentId, template) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-agent", {
            pluginId: pluginId,
            agentId: agentId,
            template: template ? template.codeSample : null,
        });
    },
    addAgentTool: function(pluginId, toolId) {
        return  APIXHRService("POST", API_PATH + "plugins/dev/add-agent-tool", {
            pluginId: pluginId,
            toolId: toolId
        });
    },
    addCustomLLM: function(pluginId, llmId, codeTemplate) {
        return APIXHRService("POST", API_PATH + "plugins/dev/add-custom-llm", {
            pluginId: pluginId,
            llmId: llmId,
            codeTemplate: codeTemplate,
        });
    },
    listContents: function(pluginId) {
        return APIXHRService("GET", API_PATH + "plugins/dev/list-contents", {
            pluginId: pluginId
        });
    },
    getContent: function(pluginId, path, sendAnyway) {
        return APIXHRService("GET", API_PATH + "plugins/dev/get-content", {
            pluginId: pluginId, path: path, sendAnyway: sendAnyway
        });
    },
    setContent: function(pluginId, path, data) {
        return APIXHRService("POST", API_PATH + "plugins/dev/set-content", {
            pluginId: pluginId, path: path, data: data
        });
    },
    validate: function(pluginId, contentMap) {
        return APIXHRService("POST", API_PATH + "plugins/dev/validate", {
            pluginId: pluginId, contentMap: JSON.stringify(contentMap)
        });
    },
    setContentMultiple: function(pluginId, contentMap) {
        return APIXHRService("POST", API_PATH + "plugins/dev/set-content-multiple", {
            pluginId: pluginId, contentMap: JSON.stringify(contentMap)
        });
    },
    createContent: function(pluginId, path, isFolder) {
        return APIXHRService("POST", API_PATH + "plugins/dev/create-content", {
            pluginId: pluginId, path: path, isFolder: isFolder
        });
    },
    deleteContent: function(pluginId, path) {
        return APIXHRService("POST", API_PATH + "plugins/dev/delete-content", {
            pluginId: pluginId, path: path
        });
    },
    decompressContent: function(pluginId, path) {
        return APIXHRService("POST", API_PATH + "plugins/dev/decompress-content", {
            pluginId: pluginId, path: path
        });
    },
    renameContent: function(pluginId, path, newName) {
        return APIXHRService("POST", API_PATH + "plugins/dev/rename-content", {
            pluginId: pluginId, path: path, newName: newName
        });
    },
    moveContent: function(pluginId, path, toPath) {
        return APIXHRService("POST", API_PATH + "plugins/dev/move-content", {
            pluginId: pluginId, path: path, toPath: toPath
        });
    },
    copyContent: function(pluginId, path) {
        return APIXHRService("POST", API_PATH + "plugins/dev/copy-content", {
            pluginId: pluginId, path: path
        });
    },
    uploadContent: function(pluginId, path, file, callback) {
        return uploadFileRequest("plugins/dev/upload-content", function(formdata) {
            formdata.append("pluginId", pluginId);
            formdata.append("path", path);
            formdata.append("file", file);
        }, callback);
    },
    checkUploadContent: function(pluginId, path, filePaths) {
        return APIXHRService("POST", API_PATH + "plugins/dev/check-upload-content", {
            pluginId: pluginId, path: path, filePaths: JSON.stringify(filePaths)
        });
    },
    useCodeEnv: function(pluginId, envName) {
        return APIXHRService("POST", API_PATH + "plugins/dev/use-code-env", {
            pluginId: pluginId, envName: envName
        });
    },
    setActiveRemote: function(pluginId, remoteName) {
        return APIXHRService("POST", API_PATH + "plugins/dev/set-active-remote", {
            pluginId: pluginId,
            remoteName: remoteName
        });
    },
    removeCodeEnv: function(pluginId) {
        return APIXHRService("GET", API_PATH + "plugins/dev/remove-code-env", {
            pluginId: pluginId
        });
    },
},
metrics: {
    getComputedMetricWithHistory: function(projectKey, objectType, objectSmartName, partitionId, metricId) {
        return APIXHRService("GET", API_PATH + "metrics/get-computed-metric-with-history", {
            projectKey: projectKey,
            objectType: objectType,
            objectSmartName: objectSmartName,
            partitionId: partitionId,
            metricId: metricId
        });
    }
},
catalog: {
    search: function (query, facets, nospinner) {
        return APIXHRService("POST", API_PATH + "catalog/search", {
            query: query,
            facets: JSON.stringify(facets)
        }, nospinner ? "nospinner" : undefined);
    },
    searchColumns: function (query, facets) {
        return APIXHRService("POST", API_PATH + "catalog/search", {
            query: query,
            facets: JSON.stringify($.extend({}, facets, {scope: ['dss-column']}))
        })
    },
    searchMeanings: function (query, facets) {
        return APIXHRService("POST", API_PATH + "catalog/search", {
            query: query,
            facets: JSON.stringify($.extend({}, facets, {scope: ['meanings']})),
            type: "meanings"
        })
    },
    flush: function () {
        return APIXHRService("GET", API_PATH + "catalog/flush");
    },
    listDashboards: function () {
        return APIXHRService("GET", API_PATH + "catalog/list-dashboards");
    }
},
externalTable: {
    summary: function (tableKey) {
        return APIXHRService("POST", API_PATH + "external-table/get-summary", tableKey);
    },
    sample: function (tableKey) {
        return APIXHRService("POST", API_PATH + "external-table/preview", tableKey);
    },
    save: function (tableKey, dssMetadata, updatedColumn, tableRemarks) {
        return APIXHRService("POST", API_PATH + "external-table/save-metadata", $.extend({}, tableKey, {
            dssMetadata: JSON.stringify(dssMetadata),
            updatedColumnData: updatedColumn != null ? JSON.stringify(updatedColumn) : null,
            tableRemarks
        }))
    }
},
meanings: {
    listUDM: function() {
        return APIXHRService("GET", API_PATH + "meanings/udm/list-with-state");
    },
    saveUDM: function(userDefinedMeaning) {
        return APIXHRService("POST", API_PATH + "meanings/udm/save", {
            userDefinedMeaning: JSON.stringify(userDefinedMeaning)
        });
    },
    createUDM: function(userDefinedMeaning) {
        return APIXHRService("POST", API_PATH + "meanings/udm/create", {
            userDefinedMeaning: JSON.stringify(userDefinedMeaning)
        });
    },
    deleteUDM: function(id) {
        return APIXHRService("POST", API_PATH + "meanings/udm/delete", {
            id: id
        });
    },
    prepareDeleteUDM: function(id) {
        return APIXHRService("POST", API_PATH + "meanings/udm/prepare-delete", {
            id: id
        });
    },
    getUDM: function(id) {
        return APIXHRService("GET", API_PATH + "meanings/udm/get", {
            id: id
        });
    }
},
internal: {
    debugKillBackend: function() {
        return APIXHRService("POST", API_PATH + "debugging/kill-backend")
    },
    debugGetBackendStacks: function() {
        return APIXHRService("GET", API_PATH + "debugging/dump-backend-stacks")
    },
    restartAllHTMLBackends: function() {
        return APIXHRService("GET", API_PATH + "debugging/restart-all-html-backends");
    },
    runScenarioTriggers:  function() {
        return APIXHRService("GET", API_PATH + "debugging/run-scenario-triggers");
    },
    fakeScenarioRun: function(projectKey, scenarioId, date, outcome) {
        return APIXHRService("POST", API_PATH + "debugging/run-fake-scenario", {
            projectKey: projectKey,
            scenarioId: scenarioId,
            date: date,
            outcome: outcome
        });
    },
    fakeFuture: function(projectKey, payloadClassName, payloadMethodName, randomUser) {
        return APIXHRService("POST", API_PATH + "debugging/run-fake-future", {
            projectKey: projectKey,
            payloadClassName: payloadClassName,
            payloadMethodName: payloadMethodName,
            randomUser: randomUser
        });
    },
    sendOfflineQueues: function() {
        return APIXHRService("GET", API_PATH + "debugging/send-offline-queues");
    },
    sendDigests: function() {
        return APIXHRService("GET", API_PATH + "debugging/send-digests");
    },
    buildUsageSummaryReports: function() {
        return APIXHRService("GET", API_PATH + "debugging/build-usage-summary-reports");
    },
    getTriggerQueueingInfo: function() {
        return APIXHRService("GET", API_PATH + "debugging/get-trigger-queueing-info");
    },
    slowFail: function(params) {
        return APIXHRService("GET", API_PATH + "debugging/slow-fail", params);
    },
    resyncProjectFolders: () => {
        return APIXHRService("GET", API_PATH + "debugging/resync-project-folders");
    },
    clearScenarioReportsCaches: function() {
        return APIXHRService("POST", API_PATH + "debugging/clear-scenario-report-caches")
    },
    invalidateEDACaches: function() {
        return APIXHRService("POST", API_PATH + "debugging/invalidate-eda-caches")
    },
    invalidateDriftCaches: function() {
        return APIXHRService("POST", API_PATH + "debugging/invalidate-drift-caches")
    },
    invalidateInMemoryAPIKeys: function() {
        return APIXHRService("POST", API_PATH + "debugging/invalidate-in-memory-keys")
    },
    listInMemoryAPIKeys: function() {
        return APIXHRService("POST", API_PATH + "debugging/list-in-memory-keys")
    },
    reloadTutorials: function() {
        return APIXHRService("POST", API_PATH + "debugging/reloadTutorials")
    },
    killLLMKernels: function() {
        return APIXHRService("POST", API_PATH + "debugging/kill-llm-kernels")
    },
    killLLMRequests: function() {
        return APIXHRService("POST", API_PATH + "debugging/kill-llm-requests")
    },
    dumpLLMMeshURL: function() {
        return API_PATH + "debugging/dump-llm-mesh";
    },
    dumpLLMCostLimitCountersURL: function() {
        return API_PATH + "debugging/dump-llm-cost-limit-counters";
    },
    clearLLMCostLimitCounters: function() {
        return APIXHRService("POST", API_PATH + "debugging/clear-llm-cost-limit-counters");
    },
    massGenerateLogs: function() {
        return APIXHRService("POST", API_PATH + "debugging/generate-logs")
    },
},
apideployer: {
    globalLightStatus: function() {
        return APIXHRService("GET", API_PATH + "api-deployer/global-light-status", null);
    },
    listStages: function() {
        return APIXHRService("GET", API_PATH + "api-deployer/stages");
    },
    client: {
        listPublishedServices: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/client/published-api-services/list-light-status");
        },
        listDeployments: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/client/deployments/list-light-status");
        },
        listEndpoints: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/client/endpoints/list");
        },
        getServiceEndpointMonitoring: function(apiServiceId, endpointId) {
            return APIXHRService("GET", API_PATH + "api-deployer/client/api-services/monitoring", {
                apiServiceId: apiServiceId,
                endpointId: endpointId
            });
        },
        getDeploymentHeavyStatus: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "api-deployer/client/deployments/status", {
                deploymentId: deploymentId
            });
        },
    },
    publishedAPIServices: {
        listBasicInfo: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/published-api-services/list-basic-info");
        },
        listLightStatus: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/published-api-services/list-light-status");
        },
        getLightStatus: function(serviceId) {
            return APIXHRService("GET", API_PATH + "api-deployer/published-api-services/get-light-status", {
                serviceId: serviceId
            });
        },
        getSettings: function(serviceId) {
            return APIXHRService("GET", API_PATH + "api-deployer/published-api-services/get-settings", {
                serviceId: serviceId
            });
        },
        create: function(serviceId, label) {
            return APIXHRService("POST", API_PATH + "api-deployer/published-api-services/create", {
                serviceId: serviceId,
                label: label
            });
        },
        save: function(service) {
            return APIXHRService("POST", API_PATH + "api-deployer/published-api-services/save", {
                service: angular.toJson(service)
            });
        },
        delete: function(serviceId) {
            return APIXHRService("POST", API_PATH + "api-deployer/published-api-services/delete", {
                serviceId: serviceId
            });
        },
        publishVersion: function(serviceId, file, callback) {
            return uploadFileRequest("api-deployer/versions/publish", function(formdata) {
                formdata.append("serviceId", serviceId);
                formdata.append("file", file);
            }, callback);
        },
        deletePackage: function(serviceId, versionId) {
            return APIXHRService("POST", API_PATH + "api-deployer/versions/delete", {
                serviceId: serviceId,
                versionId: versionId
            });
        }
    },
    infras: {
        checkStatus: function(infraId) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/check-status", {infraId: infraId}, "nospinner");
        },
        listBasicInfo: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/list-basic-info");
        },
        listLightStatus: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/list-light-status");
        },
        getLightStatus: function(infraId) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-light-status", {
                infraId: infraId
            });
        },
        getSettings: function(infraId) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-settings", {
                infraId: infraId
            });
        },
        create: function(infra) {
            return APIXHRService("POST", API_PATH + "api-deployer/infras/create", {
                infraId: infra.id,
                stage: infra.stage,
                type: infra.type,
                governCheckPolicy: infra.governCheckPolicy
            });
        },
        save: function(infra) {
            return APIXHRService("POST", API_PATH + "api-deployer/infras/save", {
                infra: JSON.stringify(infra)
            });
        },
        delete: function(infraId) {
            return APIXHRService("POST", API_PATH + "api-deployer/infras/delete", {
                infraId: infraId
            });
        },
        getSageMakerInstanceTypes: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-sagemaker-instance-types", {
            }, "nospinner");
        },
        getAzureMLInstanceTypes: function(infraId, authConnection, azWorkspace, azResourceGroup, azSubscription) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-azureml-instance-types", {
                infraId: infraId,
                authConnection: authConnection,
                azWorkspace: azWorkspace,
                azResourceGroup: azResourceGroup,
                azSubscription: azSubscription
            });
        },
        getVertexAIInstanceTypes: function(infraId, authConnection, gcpProject, gcpRegion) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-vertexai-instance-types", {
                infraId: infraId,
                authConnection: authConnection,
                gcpProject: gcpProject,
                gcpRegion: gcpRegion
            });
        },
        getVertexAIAcceleratorTypes: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-vertexai-accelerator-types");
        },
        getDatabricksWorkloadTypes: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-databricks-workload-types");
        },
        getDatabricksWorkloadSizeTypes: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-databricks-workload-size-types");
        },
        getDatabricksExperimentLocationSuggestions: function(connectionName) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-databricks-experiment-location-suggestions", {
                connectionName: connectionName
            });
        },
        getDkuBackendExtUrl: function(forK8s) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-dku-backend-ext-url", {forK8s: forK8s});
        },
        getDkuEventServerUrlAndKeys: function(forK8s) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-dku-event-server-url-keys", {forK8s: forK8s});
        },
        getAuditLogStorageUrl : function(infraId, connectionName, pathWithinConnection, bucket) {
            return APIXHRService("GET", API_PATH + "api-deployer/infras/get-storage-url", {
                infraId: infraId,
                connectionName: connectionName,
                pathWithinConnection: pathWithinConnection,
                bucket: bucket
            });
        },
    },
    deployments: {
        listTags: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/list-tags");
        },
        listBasicInfo: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/list-basic-info");
        },
        listLightStatus: function() {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/list-light-status");
        },
        getLightStatus: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-light-status", {
                deploymentId: deploymentId
            });
        },
        getSettings: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-settings", {
                deploymentId: deploymentId
            });
        },
        create: function(deploymentId, publishedServiceId, infraId, version, endpointId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/create", {
                deploymentId: deploymentId,
                publishedServiceId: publishedServiceId,
                infraId: infraId,
                version: version,
                endpointId: endpointId
            })
        },
        save: function(deployment) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/save", {
                deployment: JSON.stringify(deployment)
            });
        },
        copy: function(deploymentId, newDeploymentId, newInfraId, endpointId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/copy", {
                deploymentId: deploymentId,
                newDeploymentId: newDeploymentId,
                newInfraId: newInfraId,
                endpointId: endpointId
            });
        },
        switchVersion: function(deploymentId, versionId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/switch-version", {
                deploymentId: deploymentId,
                versionId: versionId
            });
        },
        delete: function(deploymentId, ignorePreDeleteErrors) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/delete", {
                deploymentId: deploymentId,
                ignorePreDeleteErrors: ignorePreDeleteErrors
            });
        },
        getGovernanceStatusDeploymentId: function(deploymentId, infraId, version) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/get-governance-status-deployment-id", {
                deploymentId: deploymentId,
                infraId: infraId,
                version: version,
            });
        },
        getGovernanceStatusDeployment: function(deployment) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/get-governance-status-deployment", {
                deployment: JSON.stringify(deployment)
            });
        },
        getGovernanceStatusNewDeployment: function(publishedServiceId, infraId, version) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/get-governance-status-new-deployment", {
                publishedServiceId: publishedServiceId,
                infraId: infraId,
                version: version
            });
        },
        getHeavyStatus : function(deploymentId, withPackageExtraInfo, nospinner = false) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-heavy-status", {
                deploymentId: deploymentId, withPackageExtraInfo: withPackageExtraInfo
            }, nospinner ? "nospinner" : undefined);
        },
        getAuditLogStorageUrl : function(infraId) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-storage-url", {
                infraId: infraId
            });
        },
        listHeavyStatus: function(infraId, withPackageExtraInfo) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/list-heavy-status", {
                infraId: infraId,
                withPackageExtraInfo: withPackageExtraInfo
            }, "nospinner");
        },
        executeSyncDockerBased : function(deploymentId, rebuildCodeEnv)  {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/execute-sync-docker-based", {
                deploymentId: deploymentId,
                ...(rebuildCodeEnv && {rebuildCodeEnv})
            });
        },
        executeSyncStatic : function(deploymentId, forceRefresh, withPrepare)  {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/execute-sync-static", {
                deploymentId: deploymentId,
                forceRefresh: forceRefresh,
                withPrepare: withPrepare
            });
        },
        getChartData : function(deploymentId, endpointId, chartType, timeRange) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/carbonapi-chart", {
                deploymentId:deploymentId,
                endpointId : endpointId,
                chartType : chartType,
                timeRange: timeRange,
                format: "json"
            }, "nospinner");
        },
        chartURL : function(deploymentId, endpointId, chartType, timeRange, format) {
            return API_PATH + `api-deployer/deployments/carbonapi-chart?deploymentId=${deploymentId}&endpointId=${endpointId}&chartType=${chartType}&timeRange=${timeRange}&format=${format}`;
        },
        runTestQuery: function(deploymentId, endpointId, testQueries, unsavedTestQueries) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/run-test-queries", {
                deploymentId: deploymentId,
                endpointId: endpointId,
                testQueries: JSON.stringify(testQueries),
                unsavedTestQueries: JSON.stringify(unsavedTestQueries)
            });
        },
        getDeploymentOpenAPI: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-open-api", {
                deploymentId: deploymentId
            });
        },
        getFullCheckReport: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-full-check-report", {
                deploymentId: deploymentId
            });
        },
        startDiagnosis : function(deploymentId)  {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/start-diagnosis", {
                deploymentId: deploymentId
            });
        },
        getDiagnosisURL: function(deploymentId, diagnosticName) {
            return API_PATH + "api-deployer/deployments/get-diagnosis?"
                     + "deploymentId=" + encodeURIComponent(deploymentId)
                     + "&diagnosticName=" + encodeURIComponent(diagnosticName);
        },
        listLogs : function(deploymentId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/list-logs", {
                deploymentId: deploymentId
            });
        },
        getLog : function(deploymentId, logName) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/get-log", {
                deploymentId: deploymentId,
                logName: logName
            });
        },
        getAllLogsZip : function(deploymentId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/all-logs-zip", {
                deploymentId: deploymentId
            });
        },
        getLastDeploymentAction: function(deploymentId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/get-last-action", {
                deploymentId: deploymentId
            }, 'nospinner');
        },
        listDeploymentUpdateHeads : function(deploymentId){
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/list-update-heads", {
                deploymentId: deploymentId
            });
        },
        getDeploymentUpdate : function(deploymentId, startTimestamp) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-update", {
                deploymentId: deploymentId,
                startTimestamp: startTimestamp
            });
        },
        getDeploymentUpdateSettingsDiff: function(deploymentId, startTimestamp) {
            return APIXHRService("GET", API_PATH + "api-deployer/deployments/get-update-settings-diff", {
                deploymentId: deploymentId,
                startTimestamp: startTimestamp
            });
        },
        peekDeploymentActionProgress: function(jobId, deploymentId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/peek-action-progress", {
                jobId: jobId,
                deploymentId: deploymentId
            }, 'nospinner');
        },
        abortDeploymentAction: function(deploymentId) {
            return APIXHRService("POST", API_PATH + "api-deployer/deployments/abort-action", {
                deploymentId: deploymentId
            }, 'nospinner');
        }
    }
},
deployer: {
    hooks: {
        checkCodeCompiles: function(code, codeEnvName) {
            return APIXHRService("POST", API_PATH + "deployer/hooks/check-code-compiles", {
                code: code,
                codeEnvName: codeEnvName
            });
        }
    }
},
externalinfras: {
    infos: {
        listSagemakerEndpointSummaries : function(project, region, connection) {
            return APIXHRService("GET", API_PATH + "external-infras/list-sagemaker-endpoints", {
                project,
                region,
                connection
            });
        },
        listSagemakerRegions : function() {
            return APIXHRService("GET", API_PATH + "external-infras/list-sagemaker-regions");
        },
        listAzureMLEndpoints : function(project, workspace, resourceGroup, subscription, connection) {
            return APIXHRService("GET", API_PATH + "external-infras/list-azureml-endpoints", {
                project,
                workspace,
                resourceGroup,
                subscription,
                connection
            });
        },
        listVertexAIEndpoints : function(project, region, connection) {
            return APIXHRService("GET", API_PATH + "external-infras/list-vertexai-endpoints", {
                project,
                region,
                connection
            });
        },
        listDatabricksEndpoints : function(project, connectionName) {
            return APIXHRService("GET", API_PATH + "external-infras/list-databricks-endpoints", {
                project,
                connectionName
            });
        },
        listVertexAIProjects : function(connection) {
            return APIXHRService("GET", API_PATH + "external-infras/list-vertexai-projects", {
                connection
            });
        },
        getVertexAIProject : function(projectId, connection) {
            return APIXHRService("GET", API_PATH + "external-infras/get-vertexai-project", {
                projectId,
                connection
            });
        },
        listVertexAIRegions : function() {
            return APIXHRService("GET", API_PATH + "external-infras/list-vertexai-regions");
        },
        listDatabricksRegisteredModels : function(connectionName, useUnityCatalog) {
            return APIXHRService("GET", API_PATH + "external-infras/list-databricks-registered-models", {connectionName, useUnityCatalog});
        },
        listDatabricksModelVersions : function(connectionName, modelName, useUnityCatalog) {
            return APIXHRService("GET", API_PATH + "external-infras/list-databricks-model-versions", {connectionName, modelName, useUnityCatalog});
        },
        listDatabricksExperiments : function(connectionName) {
            return APIXHRService("GET", API_PATH + "external-infras/list-databricks-experiments", {
                connectionName
            });
        },
    }
},
help: {
    search: (params = {}) => {
        return APIXHRService("POST", 'https://discoveryengine.googleapis.com/v1/projects/452762506395/locations/global/collections/default_collection/engines/dataiku-search_1733324706196/servingConfigs/default_search:searchLite?key=AIzaSyCwOlFlMdnp4YzNndK4yTlEk_sW4ywCqGk', Object.assign(params), 'nospinner');
    }
},
projectdeployer: {
    globalLightStatus: function() {
        return APIXHRService("GET", API_PATH + "project-deployer/global-light-status", null);
    },
    client: {
        listPublishedProjects: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/client/published-projects/list-light-status");
        },
        listDeployments: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/client/deployments/list-light-status");
        }
    },
    publishedProjects: {
        listBasicInfo: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/published-projects/list-basic-info");
        },
        listLightStatus: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/published-projects/list-light-status");
        },
        getLightStatus: function(projectKey) {
            return APIXHRService("GET", API_PATH + "project-deployer/published-projects/get-light-status", {
                projectKey: projectKey
            });
        },
        getSettings: function(projectKey) {
            return APIXHRService("GET", API_PATH + "project-deployer/published-projects/get-settings", {
                projectKey: projectKey
            });
        },
        create: function(projectKey, label) {
            return APIXHRService("POST", API_PATH + "project-deployer/published-projects/create", {
                projectKey: projectKey,
                label: label
            });
        },
        save: function(project) {
            return APIXHRService("POST", API_PATH + "project-deployer/published-projects/save", {
                project: angular.toJson(project)
            });
        },
        delete: function(projectKey) {
            return APIXHRService("POST", API_PATH + "project-deployer/published-projects/delete", {
                projectKey: projectKey
            });
        },
        uploadBundle: function(projectKey, file, callback) {
            return uploadFileRequest("project-deployer/bundles/upload", function(formdata) {
                formdata.append("projectKey", projectKey ? projectKey : "");
                formdata.append("file", file);
            }, callback);
        },
        getBundleDetails: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "project-deployer/bundles/get-details", {
                projectKey: projectKey,
                bundleId: bundleId
            });
        },
        getBundleDetailsExtended: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "project-deployer/bundles/get-details-extended", {
                projectKey: projectKey,
                bundleId: bundleId
            });
        },
        deleteBundle: function(projectKey, bundleId) {
            return APIXHRService("POST", API_PATH + "project-deployer/bundles/delete", {
                projectKey: projectKey,
                bundleId: bundleId
            });
        },
        getBundleGovernanceStatus: function(projectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "project-deployer/bundles/get-governance-status", {
                projectKey: projectKey,
                bundleId: bundleId,
            });
        }
    },
    infras: {
        checkStatus: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/check-status", {infraId: infraId}, "nospinner");
        },
        listBasicInfo: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/list-basic-info");
        },
        listLightStatus: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/list-light-status");
        },
        listConnectionsNames: function(infraId, deploymentId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/list-connections-names", {
                infraId: infraId,
                ...(deploymentId && {deploymentId})
            });
        },
        listContainerExecNames: function(infraId, deploymentId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/list-container-exec-names", {
                infraId: infraId,
                ...(deploymentId && {deploymentId})
            });
        },
        getLightStatus: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/get-light-status", {
                infraId: infraId
            });
        },
        getSettings: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/get-settings", {
                infraId: infraId
            });
        },
        create: function(infra) {
            return APIXHRService("POST", API_PATH + "project-deployer/infras/create", {
                infraId: infra.id,
                stage: infra.stage,
                automationNodeUrl: infra.automationNodeUrl,
                apiKey: infra.apiKey,
                governCheckPolicy: infra.governCheckPolicy,
                type: infra.type
            });
        },
        save: function(infra) {
            return APIXHRService("POST", API_PATH + "project-deployer/infras/save", {
                infra: JSON.stringify(infra)
            });
        },
        delete: function(infraId) {
            return APIXHRService("POST", API_PATH + "project-deployer/infras/delete", {
                infraId: infraId
            });
        },
        getProjectKeys: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/get-project-keys", {
                infraId: infraId
            });
        },
        getProjectFolderHierarchy: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/project-folder-hierarchy", {
                infraId: infraId
            });
        },
        listUserLogins: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/list-user-logins", {
                infraId: infraId
            });
        },
        getAutomationNodesFromNodesDirectory: function(infraId) {
            return APIXHRService("GET", API_PATH +   "project-deployer/infras/get-automation-nodes-from-nodes-directory", {
                infraId: infraId
            });
        },
        listOtherAutomationNodeUrls: function(infraId) {
            return APIXHRService("GET", API_PATH +   "project-deployer/infras/list-other-automation-node-urls", {
                infraId: infraId
            });
        },
        generatePersonalAPIKey : function(infraId, label, description, forUser)  {
            return APIXHRService("POST", API_PATH + "project-deployer/infras/generate-personal-api-key", {
                infraId: infraId,
                label: label,
                description: description,
                forUser: forUser,
            });
        },
        runConsistencyChecks: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/infras/run-consistency-checks", {
                infraId: infraId
            });
        }
    },
    deployments: {
        listTags: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/list-tags");
        },
        listBasicInfo: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/list-basic-info");
        },
        listLightStatus: function() {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/list-light-status");
        },
        getLightStatus: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-light-status", {
                deploymentId: deploymentId
            });
        },
        getSettings: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-settings", {
                deploymentId: deploymentId
            });
        },
        create: function(deploymentId, publishedProjectKey, infraId, bundleId, deployedProjectKey, projectFolderId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/create", {
                deploymentId: deploymentId,
                publishedProjectKey: publishedProjectKey,
                infraId: infraId,
                bundleId: bundleId,
                deployedProjectKey: deployedProjectKey,
                projectFolderId: projectFolderId
            })
        },
        save: function(deployment) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/save", {
                deployment: JSON.stringify(deployment)
            });
        },
        copy: function(deploymentId, newDeploymentId, newInfraId, newDeployedProjectKey, newProjectFolderId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/copy", {
                deploymentId: deploymentId,
                newDeploymentId: newDeploymentId,
                newInfraId: newInfraId,
                newDeployedProjectKey: newDeployedProjectKey,
                newProjectFolderId: newProjectFolderId
            });
        },
        switchBundle: function(deploymentId, bundleId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/switch-bundle", {
                deploymentId: deploymentId,
                bundleId: bundleId
            });
        },
        delete: function(deploymentId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/delete", {
                deploymentId: deploymentId
            });
        },
        getGovernanceStatusDeploymentId: function(deploymentId, infraId, bundleId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/get-governance-status-deployment-id", {
                deploymentId: deploymentId,
                infraId: infraId,
                bundleId: bundleId,
            });
        },
        getGovernanceStatusDeployment: function(deployment) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/get-governance-status-deployment", {
                deployment: JSON.stringify(deployment)
            });
        },
        getGovernanceStatusNewDeployment: function(publishedProjectKey, infraId, bundleId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/get-governance-status-new-deployment", {
                publishedProjectKey: publishedProjectKey,
                infraId: infraId,
                bundleId: bundleId
            });
        },
        getProjectStandardsStatusDeploymentId: function(deploymentId) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-project-standards-status-deployment-id", {
                deploymentId: deploymentId
            });
        },
        getProjectStandardsStatusBundleOnInfra: function(infraId, publishedProjectKey, bundleId) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-project-standards-status-bundle-on-infra", {
                infraId: infraId,
                publishedProjectKey: publishedProjectKey,
                bundleId: bundleId
            });
        },
        getHeavyStatus : function(deploymentId, unused, nospinner = false) { // The signature should be the same that the one in apideployer, hence the 'unused' param
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-heavy-status", {
                deploymentId: deploymentId
            }, nospinner ? "nospinner" : undefined);
        },
        listHeavyStatus: function(infraId) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/list-heavy-status", {
                infraId
            }, "nospinner");
        },
        scenarioLastRuns : function(deploymentId, fromDate, toDate)  {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/scenario-runs-in-date-range", {
                deploymentId: deploymentId,
                fromDate: fromDate,
                toDate: toDate
            });
        },
        update : function(deploymentId)  {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/update", {
                deploymentId: deploymentId
            });
        },
        listLogs : function(deploymentId){
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/list-logs", {
                deploymentId: deploymentId
            });
        },
        getLog : function(deploymentId, logName){
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/get-log", {
                deploymentId: deploymentId,
                logName: logName
            });
        },
        getAllLogsZip : function(deploymentId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/all-logs-zip", {
                deploymentId: deploymentId
            });
        },
        getLastDeploymentAction: function(deploymentId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/get-last-action", {
                deploymentId: deploymentId
            }, 'nospinner');
        },
        listDeploymentUpdateHeads : function(deploymentId){
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/list-update-heads", {
                deploymentId: deploymentId
            });
        },
        getDeploymentUpdate : function(deploymentId, startTimestamp) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-update", {
                deploymentId: deploymentId,
                startTimestamp: startTimestamp
            });
        },
        getDeploymentUpdateSettingsDiff: function(deploymentId, startTimestamp) {
            return APIXHRService("GET", API_PATH + "project-deployer/deployments/get-update-settings-diff", {
                deploymentId: deploymentId,
                startTimestamp: startTimestamp
            });
        },
        peekDeploymentActionProgress: function(jobId, deploymentId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/peek-action-progress", {
                jobId: jobId,
                deploymentId: deploymentId
            }, 'nospinner');
        },
        abortDeploymentAction: function(deploymentId) {
            return APIXHRService("POST", API_PATH + "project-deployer/deployments/abort-action", {
                deploymentId: deploymentId
            }, 'nospinner');
        },
    }
},
unifiedMonitoring: {
    deployer: {
        deployedProjects: {
            list : function () {
                return APIXHRService("GET", API_PATH  + "unified-monitoring/deployer/projects/list");
            },
            get: function (deploymentId) {
                return APIXHRService("GET", API_PATH  + "unified-monitoring/deployer/projects/get", {
                    deploymentId
                });
            }
        },
        apiEndpoints: {
            list: function() {
                return APIXHRService("GET", API_PATH + "unified-monitoring/deployer/api-endpoints/list");
            },
            get: function(deploymentId, endpointId) {
                return APIXHRService("GET", API_PATH + "unified-monitoring/deployer/api-endpoints/get", {
                    deploymentId, endpointId
                });
            }
        },
        getSettings: function() {
            return APIXHRService("GET", API_PATH  + "unified-monitoring/deployer/settings/get");
        },
        getRemoteDesignNodesUrls: function() {
            return APIXHRService("GET", API_PATH  + "unified-monitoring/deployer/settings/remote-design-nodes-urls");
        }
    }
},
workspaces: {
    list: function(accessLevel = "READ") {
        return APIXHRService("GET", API_PATH + "workspaces/list", { accessLevel: accessLevel });
    },
    get: function(workspaceKey) {
        return APIXHRService("GET", API_PATH + "workspaces/get", { workspaceKey: workspaceKey });
    },
    create: function(workspace) {
        return APIXHRService("POST", API_PATH + "workspaces/create", { workspace: JSON.stringify(workspace) });
    },
    update: function(workspace) {
        return APIXHRService("PUT", API_PATH + "workspaces/update", { workspace: JSON.stringify(workspace) });
    },
    save: function(workspace) {
        return APIXHRService("POST", API_PATH + "workspaces/save", { workspace: JSON.stringify(workspace) });
    },
    delete: function(workspaceKey) {
        return APIXHRService("POST", API_PATH + "workspaces/delete", { workspaceKey: workspaceKey });
    },
    getObject: function(workspaceKey, objectKey) {
        return APIXHRService("GET", API_PATH + "workspaces/get-object", {
            workspaceKey: workspaceKey,
            objectKey: JSON.stringify(objectKey)
        });
    },
    addObjects: function(workspaceKey, workspaceObjects) {
        return APIXHRService("POST", API_PATH + "workspaces/add-objects", {
            workspaceKey: workspaceKey,
            workspaceObjects: JSON.stringify(workspaceObjects)
        });
    },
    removeObjects: function(workspaceKey, objectsToRemove) {
        return APIXHRService("POST", API_PATH + "workspaces/remove-objects", {
            workspaceKey: workspaceKey,
            objectsToRemove: JSON.stringify(objectsToRemove)
        });
    },
    timeline: function(workspaceKey) {
        return APIXHRService("GET", API_PATH + "workspaces/timeline", { workspaceKey });
    },
},
dataCollections: {
    list: function(privilege = "READ") {
        return APIXHRService("GET", API_PATH + "data-collections/list", { privilege });
    },
    addObjects: function(dataCollectionId, items) {
        return APIXHRService("POST", API_PATH + "data-collections/add-objects", {
            dataCollectionId: dataCollectionId,
            items: JSON.stringify(items)
        })
    },
    getObjectsAuthorizations: function(items) {
        return APIXHRService("POST", API_PATH + "data-collections/get-objects-authorizations", { items: JSON.stringify(items) });
    },
},
dataLineage: {
    saveManualLineage: function(projectKey, recipeName, manualDataLineages, ignoreAutoComputedLineage) {
       return APIXHRService("POST", API_PATH + "data-lineage/save-manual-relations", {
            projectKey: projectKey,
            recipeName: recipeName,
            manualDataLineages: JSON.stringify(manualDataLineages),
            ignoreAutoComputedLineage: ignoreAutoComputedLineage
        });
    },
},
pretrainedModels: {
    getImageEmbeddingModels: function() {
        return APIXHRService("GET", API_PATH + "pretrained-models/image-embedding");
    },
    // The text embedding models need a code env to be specified because of the legacy models from code env resources
    // The project key is used if the backend is an Automation node
    getSentenceEmbeddingModels: function(envSelection, projectKey) {
        return APIXHRService("GET", API_PATH + "pretrained-models/sentence-embedding", {
            envSelection: envSelection, projectKey: projectKey
        });
    },
    listAvailableLLMs: function(projectKey, purpose) {
        return APIXHRService("GET", API_PATH + "llm/list-available-llms", {projectKey, purpose})
    },
    listAvailableConnectionLLMs: function(purpose) {
        return APIXHRService("GET", API_PATH + "llm/list-available-connection-llms", {purpose})
    },
    listAvailableLLMConnections: function() {
        return APIXHRService("GET", API_PATH + "llm/list-available-llm-connections")
    },
    getHuggingFaceConnectionNonSensitiveParams: function(connectionName) {
      return APIXHRService("GET", API_PATH + "llm/get-huggingface-connection-non-sensitive-params", {connectionName})
    }
},
dataQuality : {
    getDatasetCurrentDailyStatus(contextProjectKey, projectKey, datasetName) {
        return APIXHRService("POST", API_PATH + "data-quality/dataset/get-dataset-current-daily-status", {
            contextProjectKey, projectKey, datasetName
        })
    },
    getDatasetCurrentStatusBreakdown(contextProjectKey, projectKey, datasetName) {
        return APIXHRService("POST", API_PATH + "data-quality/dataset/get-current-status-breakdown", {
            contextProjectKey, projectKey, datasetName
        })
    },
    getProjectCurrentDailyStatus(projectKey) {
        return APIXHRService("POST", API_PATH + "data-quality/project/get-current-daily-status", { projectKey })
    },
    getProjectCurrentStatusBreakdown(projectKey) {
        return APIXHRService("POST", API_PATH + "data-quality/project/get-current-status-breakdown", { projectKey })
    },
},
humanId: {
    create: function(alreadyUsedIds) {
        return APIXHRService("GET", API_PATH + "human-readable-id/create", { ids: JSON.stringify(alreadyUsedIds) });
    },
},
code: {
    ai: {
        startExplainCode: function(language, code, explanationOptions) {
            return APIXHRService("POST", API_PATH + "code/ai-tools/start-code-explanation", {
                language, code, explanationOptions: JSON.stringify(explanationOptions)
            }, "nospinner");
        },
    },
},
staticData: {
    getStaticData: function(lang) {
        return APIXHRService("GET", API_PATH + "static-data", { lang });
    }
}
};
}]);

})();
