(function() {
"use strict";

angular.module("dataiku.surveys").factory("ATSurveyService", ATSurveyService);

function ATSurveyService(DataikuAPI, localStorageService, $rootScope, Logger) {
    const atSurveyDisplayService = {
        resetCounter,
        deleteAllCounters,
        updateCounter,
        updateATSurveySettings,
    };

    const ACTION_TRIGGER_LOCAL_STORAGE_PREFIX = "at_";

    // Define all Action Triggers and their content
    const actionsTriggerDefinitions = [
        {event: "WorkspacesOpenEditObject", threshold: 20, title: "workspaces"},
        {event: "AppRun", threshold: 10, title: "apps"},
        {event: "ChartsOpen", threshold: 20, title: "charts"},
        {event: "EmbedDocumentsSave", threshold: 2, title: "the Embed documents recipe"},
        {event: "DashboardEdit", threshold: 20, title: "dashboards"},
        {event: "ConditionalFormattingRuleAdd", threshold: 5, title: "conditional formatting", expiryDate: new Date("2026-06-30T00:00:00")},
        {event: "ApplyAView", threshold: 5, title: "flow views", expiryDate: new Date("2026-06-30T00:00:00")},
        {event: "HomepageOpen", threshold: 25, title: "the homepage"},
    ];

    function getATSurveyStatus() {
        return $rootScope.appConfig.atSurveyStatus;
    }

    function setATSurveyStatus(settings) {
        $rootScope.appConfig.atSurveyStatus = settings;
    }

    function getCounter(event) {
        const counter = localStorageService.get(ACTION_TRIGGER_LOCAL_STORAGE_PREFIX + event);
        if (!counter) {
            return 0;
        }
        return parseInt(counter);
    }

    function setCounter(event, count) {
        localStorageService.set(ACTION_TRIGGER_LOCAL_STORAGE_PREFIX + event, count);
    }

    function deleteCounter(event) {
        localStorageService.remove(ACTION_TRIGGER_LOCAL_STORAGE_PREFIX + event);
    }

    function resetCounter(event) {
        setCounter(event, 0);
    }

    function deleteAllCounters() {
        actionsTriggerDefinitions.forEach((at) => deleteCounter(at.event));
    }

    function getDefinition(event) {
        return actionsTriggerDefinitions.find((at) => at.event === event);
    }

    function updateCounter(event) {
        // Using a try/catch to make sure that calling this function cannot fail (and make dashboards or workspaces unusable)
        try {
            const surveyStatus = getATSurveyStatus();
            if (!surveyStatus) {
                return; // Surveys are disabled
            }
            let counter = getCounter(event);
            counter++;
            setCounter(event, counter);

            checkSurveyTrigger(event, counter);
        } catch (error) {
            Logger.warn(error)
        }
    }

    function checkSurveyTrigger(event, counter) {
        const definition = getDefinition(event);
        if (!definition) {
            Logger.warn("Unknown Action Trigger event: " + event);
            return;
        }
        const threshold = definition.threshold;
        const surveyStatus = getATSurveyStatus();
        const nextAnySurveyDate = new Date(surveyStatus.nextAction);
        const survey = surveyStatus.surveys[event];
        const nextSameSurveyDate = survey ? new Date(survey.nextAction) : new Date(0);
        const now = new Date();
        const isActive = !definition.expiryDate || definition.expiryDate >= now;

        if (surveyStatus.state !== "OPTOUT" && counter >= threshold && now >= nextAnySurveyDate && now >= nextSameSurveyDate && isActive) {
            triggerSurvey(event);
        }
    }

    function triggerSurvey(event) {
        // Send an event "downwards" to be listened by survey's scope
        $rootScope.$broadcast("atSurveyTriggeredEvent", getDefinition(event));
    }

    function updateATSurveySettings(event, action) {
        return DataikuAPI.profile
            .updateATSurveySettings(event, action)
            .success((settings) => {
                setATSurveyStatus(settings);
            })
            .error((error) => Logger.error("Action Trigger error: ", error.errorType + ": " + error.message));
    }

    return atSurveyDisplayService;
}
})();
