/**
 * A somewhat weaker but simple, fast & standalone implementation of the stylelint rule - for the pre-commit hook
 * (weaker as it makes strong assumption on the syntax used, but usual syntaxes & ignore commands will be handled correctly)
 */

import { readdirSync, readFileSync, lstatSync } from 'fs';
import { join } from 'path';
import { argv, env, exit } from 'process';

// retrieve DKUINSTALLDIR by looking at this file position & truncating it (using env variable may not work when committing through a UI like vscode)
const __dirname = import.meta.dirname
const DKUINSTALLDIR = __dirname.slice(0, -'/src/main/platypus/stylelint'.length)


// Code to find all files in relevant directories (for use outside of hook)
function findAllRelevantFiles() {
    const relevantCssPaths = [
        'server/src/frontend/src/app',
        'server/src/frontend/src/static',
        'src/main/platypus/static/dataiku/css',
    ];

    function forEachLessFileDeep(dir, cb) {
        for (const file of readdirSync(dir)) {
            const fileName = join(dir, file);
            if(['bower_components', 'node_modules'].includes(file)) {
                continue
            } else if (lstatSync(fileName).isDirectory()) {
                forEachLessFileDeep(fileName, cb);
            } else if(file.endsWith('.less')) {
                cb(fileName);
            }
        }
    }

    const result = []
    for (const dir of relevantCssPaths) {
        forEachLessFileDeep(join(DKUINSTALLDIR, dir), file => result.push(file.substring(DKUINSTALLDIR.length)));
    }
    return result;
}


// The actual check code
const nonRefImportRegexp = /^@import\s+["']/;
const excludeLine = 'stylelint-disable-line dku/only-reference-imports';
const excludeNextLine = 'stylelint-disable-next-line dku/only-reference-imports';
const disableCommand = 'stylelint-disable dku/only-reference-imports';
const enabledCommand = 'stylelint-enable dku/only-reference-imports';

function checkSingleFile(fileName) {
    const content = readFileSync(join(DKUINSTALLDIR, fileName), 'utf-8').split('\n');
    let disabledUntilLine = -1;

    const errorLines = [];
    content.forEach((line, index) => {
        if(line.includes(disableCommand)) {
            disabledUntilLine = Infinity;
        } else if(line.includes(enabledCommand)) {
            disabledUntilLine = -1;
        } else if(line.includes(excludeNextLine)) {
            disabledUntilLine = Math.max(disabledUntilLine, index + 1);
        } else if(line.includes(excludeLine)) {
            disabledUntilLine = Math.max(disabledUntilLine, index);
        }
        if (index > disabledUntilLine && line.match(nonRefImportRegexp)) {
            errorLines.push({index: index+1, line});
        }
    });

    const errorCnt = errorLines.length;
    if(errorCnt > 0) {
        console.log('In', fileName);
        errorLines.slice(0, 5).forEach(({index, line}) => console.log(`\t${index.toFixed(0).padStart(5)}: ${line}`));
        if(errorCnt > 5) console.log(`\tand ${errorCnt-5} more...`)
    }
    return errorCnt;
}


// the hook core
const filesToCheck = argv.length === 2 //first 2 args are node and this file name
    ? findAllRelevantFiles() // no file parameter => check all files
    : argv.slice(2); // file parameters => use them

const errorCount = filesToCheck.reduce((cnt, fileName) => cnt + checkSingleFile(fileName), 0);

if (errorCount > 0) {
    console.log('Please review imports, add reference where relevant, or add `'+excludeLine+'` on the import line where non-reference import is required');
    console.log('See https://design.analytics.ondku.net/projects/RDWIKI/wiki/2526');
    exit(1);
}
