#! /bin/bash -e

echo "*********************************************"
echo "*           Dataiku DSS installer           *"
echo "*********************************************"

MYDIR=$(cd "$(dirname "$0")" && pwd -P)

# Initial sanity checks
if [ $(id -u) -eq 0 ]; then
  echo >&2 "[-] Installing or running DSS as root is not supported."
  exit 1
fi

dssArch=$(cat "$MYDIR/dss-arch.txt")
case "$(uname)" in
	Linux)
		if [ "$dssArch" != "linux" ]; then
			echo >&2 "[-] This is the installer for the $dssArch version of DSS."
			echo >&2 "[-] Please download the Linux version from www.dataiku.com."
			exit 1
		fi
		;;
	Darwin)
		if [ "$dssArch" != "osx" ]; then
			echo >&2 "[-] This is the installer for the $dssArch version of DSS."
			echo >&2 "[-] Please download the macOS version from www.dataiku.com."
			exit 1
		fi
		;;
	*)
		echo >&2 "[-] Unsupported system: $(uname)"
		exit 1
		;;
esac

# Parse command line options
usage() {
	prog=$(basename "$0")
	echo >&2 "
*** Usage:
New installation:    $prog -d DATADIR -p BASE_PORT [-t NODETYPE] [-s SIZE] [-l LICENSE_FILE] [-n] [-P BASE_PYTHON]
Upgrade:             $prog -u -d DATADIR [-y] [-n] [-P BASE_PYTHON]
Print this help:     $prog -h

  -d DATADIR : specify data directory
  -p BASE_PORT : specify base port
  -l LICENSE_FILE : specify license file
  -t NODETYPE : DSS node type to install ('design', 'automation', 'api', 'deployer' or 'govern' - defaults to 'design')
  -s SIZE: Sizing of the install ('auto', 'big', 'medium', 'small' - defaults to 'auto')
  -n : do not check for required dependencies
  -u : upgrade an existing data directory
  -y : do not prompt, assume answer 'yes'
  -P BASE_PYTHON : Python binary on which to build DSS default virtual environment
  -h : prints this help
  "
	exit 1
}

NODETYPE=design
INSTALL_SIZE=auto
DIP_HOME=
LICENSE=
PORT=
noDeps=
upgrade=
BASE_PYTHON=
yes=
forcePythonRebuild=
rebuildPyenvIfNeeded=
noJupyter=
stories=
while getopts t:s:d:p:l:P:w:hnuyfrj OPT; do
	case "$OPT" in
	t)
		NODETYPE="$OPTARG"
		;;
	s)
		INSTALL_SIZE="$OPTARG"
		;;
	d)
		DIP_HOME="$OPTARG"
		;;
	p)
		PORT="$OPTARG"
		;;
	l)
		LICENSE="$OPTARG"
		;;
	P)
		BASE_PYTHON="$OPTARG"
		;;
	w)
		stories="$OPTARG"
		;;
	h)
		usage
		;;
	n)
		noDeps=1
		;;
	u)
		upgrade=1
		;;
	y)
		yes=1
		export DKU_MIGRATE_YES=1
		;;
	f)
		forcePythonRebuild=1
		;;
	r)
		rebuildPyenvIfNeeded=1
		;;
	j)
		noJupyter=1
		;;
	*)
		usage
		;;
	esac
done

if [ $OPTIND -le $# ]; then
	echo >&2 "[-] Bad usage: invalid argument : ${!OPTIND}"
	usage
fi

#
# Check arguments
#

if [ -z "$NODETYPE" ]; then
	echo >&2 "[-] Bad usage: node type (-t) can't be empty"
fi
if [ "$NODETYPE" = "apideployer" ]; then
	echo >&2 "[!] Warning: node type (-t) 'apideployer' is deprecated, please use 'deployer' instead. Installation will continue using 'deployer' node type."
	NODETYPE=deployer
fi
if [ "$NODETYPE" != "design" -a "$NODETYPE" != "automation" -a "$NODETYPE" != "api" -a "$NODETYPE" != "deployer" -a "$NODETYPE" != "govern" ]; then
	echo >&2 "[-] Bad usage: node type (-t) must be one of 'design', 'automation', 'api', 'deployer' or 'govern'"
	usage
fi
if [ -n "$stories" ]; then
	if [ "$NODETYPE" != "design" ]; then
		echo >&2 "[-] Bad usage: option -w (stories) is only possible with node type 'design'"
		usage
	elif [ "$(uname)" == "Darwin" ]; then
	  echo >&2 "[-] option -w (stories) is not supported on MacOS"
	  exit 1
	fi
fi

if [ -z "$INSTALL_SIZE" ]; then
	echo >&2 "[-] Bad usage: install size (-s) can't be empty"
fi
if [ "$INSTALL_SIZE" != "auto" -a "$INSTALL_SIZE" != "big" -a "$INSTALL_SIZE" != "medium"  -a "$INSTALL_SIZE" != "small" ]; then
	echo >&2 "[-] Bad usage: install size (-s) must be one of 'auto', 'big', 'medium' or 'small'"
	usage
fi

if [ -n "$upgrade" -a \( -n "$PORT" -o -n "$LICENSE" \) ]; then
    echo "[-] Bad usage: cannot specify port or license file while upgrading" >&2
	usage
fi

if [ -z "$DIP_HOME" ]; then
	echo "[-] Bad usage : -d DATADIR argument is mandatory" >&2
	usage
fi
# Sanity check, strip trailing / if any
case "$DIP_HOME" in
	. | ./ | .. | ../ | / )
		echo "[-] Invalid value for DATADIR : $DIP_HOME" >&2
		usage
		;;
	*/ )
		DIP_HOME=$(echo "$DIP_HOME" | sed 's_/$__')
		;;
esac

if [ -n "$upgrade" ]; then
	if [ -d "$DIP_HOME" -a -f "$DIP_HOME"/dss-version.json ]; then
		echo "[+] $DIP_HOME already exists and is a DSS installation"
	else
		echo >&2 "[-] $DIP_HOME does not appear to be a valid DSS installation"
		exit 1
	fi
else
	if [ -z "$PORT" ]; then
		echo "Bad usage : -p BASE_PORT argument is mandatory" >&2
		usage
	fi

	if [ -f "$DIP_HOME"/DSS-INITIAL-INSTALL-IN-PROGRESS.txt ]; then
		echo "[!] *****************************************************" >&2
		echo "[!] $DIP_HOME contains a previously failed install of DSS" >&2
		echo "[!] Moving it out of the way and proceeding" >&2
		echo "[!] *****************************************************" >&2
		mv "$DIP_HOME" "$DIP_HOME.dss_failed_install.`date +%Y%m%d-%H%M%S`"
	fi

	if [ -f "$DIP_HOME"/config/dip.properties ]; then
	    echo "[-] $DIP_HOME appears to already contain a DSS installation" >&2
	    echo "[-] If you want to upgrade it, rerun this installer with the -u flag" >&2
        exit 1
    fi
	if [ ! -e "$DIP_HOME" ]; then
		echo "[+] Creating data directory: $DIP_HOME" >&2
		mkdir -p "$DIP_HOME"
		touch "$DIP_HOME"/DSS-INITIAL-INSTALL-IN-PROGRESS.txt

	elif [ -d "$DIP_HOME" -a -z "$(ls "$DIP_HOME" 2>/dev/null)" ]; then
		echo "[+] Using data directory: $DIP_HOME" >&2
		touch "$DIP_HOME"/DSS-INITIAL-INSTALL-IN-PROGRESS.txt
	else
		echo "[-] Directory $DIP_HOME already exists, but is not empty. Aborting !" >&2
		exit 1
	fi
fi

if [ -n "$stories" ]; then
	enable_stories="true"
fi

if [ -z "$stories" -a ! -f /etc/dataiku-cloud -a -d "$DIP_HOME"/stories ]; then
	echo "[+] $DIP_HOME/stories already exists, will update Stories" >&2
	stories="true"
fi

if [ -z "$stories" -a ! -f /etc/dataiku-cloud -a -d "$DIP_HOME"/story ]; then
	echo "[+] $DIP_HOME/story already exists, will update Stories" >&2
	stories="true"
fi

DIP_HOME=$(cd "$DIP_HOME" && pwd -P)
umask 22

# Start logging script output
mkdir -p "$DIP_HOME"/run
logFile="$DIP_HOME"/run/install.log
echo "[+] Saving installation log to $logFile"
cat >>"$logFile" <<EOF
*********************************************************
Data Science Studio installer: $(date +%Y/%m/%d-%H:%M:%S)
Command line: $0 $@
Version: $(cat "$MYDIR"/dss-version.json)
DIP_HOME: $DIP_HOME

EOF

# Start block | tee -a "$logFile"
set -o pipefail
(
set +o pipefail

#
# Check for running instance
#
if "$DIP_HOME"/bin/dss status >/dev/null 2>/dev/null; then
	echo "[!] *********************************************************"
	echo "[!] There seem to be an already running instance of the Studio"
	echo "[!] using this data directory:"
	"$DIP_HOME"/bin/dss status
	echo "[!]"
	echo "[!] It is STRONGLY RECOMMENDED to stop it before upgrade."
	echo "[!] If you continue, you will not be able to use bin/dss stop"
	echo "[!] and will need to kill the processes manually"
	if [ -z "$yes" ]; then
		echo "[!] Press Enter to continue at your own risk, Ctrl+C to abort"
   		read
   	else
   		echo "[!] Non-interactive mode, continuing anyway"
   	fi
fi

#
# Check for DATADIR inside INSTALLDIR
#
# this check is enough because both DIP_HOME and MYDIR have been cleaned with (cd ... && pwd -P)
if [[ "$DIP_HOME" == "$MYDIR"/* ]]; then
	echo "[!] *********************************************************"
	echo "[!] Warning: the data directory you specified:"
	echo "[!]     $DIP_HOME"
	echo "[!] appears to be a subdirectory of the installation directory:"
	echo "[!]     $MYDIR"
	echo "[!] This is NOT RECOMMENDED for production environment as it complexifies subsequent Studio upgrades."
	if [ -z "$yes" ]; then
		echo "[!] Press Enter to continue, Ctrl+C to abort"
		read
	else
		echo "[!] Non-interactive mode, continuing anyway"
	fi
fi

#
# Check system dependencies
#
deps_flags=
if [ -n "$DKUJAVABIN" ]; then
	echo "[+] Using custom Java environment \"$DKUJAVABIN\""
	javaBin="$DKUJAVABIN"
	deps_flags="$deps_flags -without-java"
elif javaBin=$("$MYDIR"/scripts/_find-java.sh); then
	deps_flags="$deps_flags -without-java"
else
	javaBin=
fi

if [ -n "$DKUPYTHONBIN" ]; then
	echo "[+] Using custom Python environment \"$DKUPYTHONBIN\""
	deps_flags="$deps_flags -without-python"
elif [ -d "$DIP_HOME"/condaenv ]; then
	echo >&2 "[-] Usage of Conda for the base DSS Python environment is not supported anymore"
	echo >&2 "[-] Please rename or remove directory \"$DIP_HOME/condaenv\""
	echo >&2 "[-] This instance will be migrated to a virtualenv-based Python environment"
	exit 1
fi
if [ -n "$stories" ]; then
	deps_flags="$deps_flags -with-docker"
fi

if [ -z "$noDeps" ]; then
	echo "[+] Checking required dependencies"
	"$MYDIR"/scripts/install/install-deps.sh -check $deps_flags || {
		echo >&2 "
[-] Dependency check failed
[-] You can install required dependencies with:
[-]    sudo -i \"$MYDIR/scripts/install/install-deps.sh\"$deps_flags
[-] You can also disable this check with the -n installer flag
"
		exit 1
	}
fi
if [ -z "$javaBin" ]; then
	echo "[-] Could not find suitable Java installation. Aborting!" >&2
	exit 1
fi

if [ -n "$stories" ]; then
	# Check that user is in group docker and that in current shell user is in docker group,
	# else docker commands can't be issued without sudo.
	# If not, ask user to remediate (can't be done programmatically)
	if id -nG "$USER" | grep -qw "docker"; then
		if ! id -nG | grep -qw "docker"; then
			echo "[-] Your user is a member of the group docker but your current shell isn't aware of it"
			echo "[-] Run:"
			echo "[-]    newgrp docker"
			echo "[-] (will start a new shell)"
			exit 1
		fi
	else
		echo "[-] Your user must be a member of the docker group to run docker commands without sudo."
		echo "[-] Run:"
		echo "[-]    sudo usermod -aG docker $USER"
		exit 1
	fi
fi

function precompile_python {
	pythonBin="$1"
	precompilePackages="$2"

	if [ ! -w "$MYDIR" ]; then
		echo "[-] Read-only installation directory $MYDIR, skipping Python precompilation"
		return
	fi

	echo "[+] Precompiling Dataiku Python code"
	"$pythonBin" -m compileall -q "$MYDIR"/python ||
		echo "[-] Error precompiling Dataiku Python code (ignored)"
	echo "[+] Precompiling Jupyter Python code"
	"$pythonBin" -m compileall -q "$MYDIR"/dku-jupyter ||
		echo "[-] Error precompiling Jupyter Python code (ignored)"

	if [ -n "$precompilePackages" ]; then
		pythonVersion=$("$pythonBin" -c "import sysconfig;print(sysconfig.get_python_version())")
		case "$pythonVersion" in
			"3.9") pkgDir="$MYDIR"/python39.packages;;
			"3.10") pkgDir="$MYDIR"/python310.packages;;
			"3.11") pkgDir="$MYDIR"/python311.packages;;
			*) ;;
		esac
		# Ignore errors as there are a few Python-version-specific files in there
		echo "[+] Precompiling third-party Python $pythonVersion code"
		"$pythonBin" -m compileall -q "$pkgDir" >/dev/null || true
	fi
}

function stories_load {
	echo "[+] Ensure Stories folders"
	DKU_STORIES_PATH="$DIP_HOME"/stories
	DKU_STORIES_RUN_PATH="$DIP_HOME"/run/stories
	mkdir -p "$DKU_STORIES_PATH" "$DKU_STORIES_RUN_PATH"

	# Stories flag is an URL
	if [[ "$stories" == http* ]]; then
		echo "[+] Using stories image url from -w"
		DKU_STORIES_IMAGE_URL_WITH_CREDENTIALS="$stories"
	fi

	# Stories flag is a filepath
	if [[ -f "$stories" ]]; then
		echo "[+] Using local stories image from $stories"
		DKU_STORIES_IMAGE_FILE="$stories"
	fi

	DSS_VERSION=$(jq -r ".product_version" "$DIP_HOME"/dss-version.json)
	
	# Stories flag defaults to generate download URL
	if [ -z "$DKU_STORIES_IMAGE_URL_WITH_CREDENTIALS" -a -z "$DKU_STORIES_IMAGE_FILE" ]; then
		echo "[+] Generate stories image url from install.ini flags"
		DKU_STORIES_RELEASE_TYPE=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get stories release_type)
		if [[ -z "$DKU_STORIES_RELEASE_TYPE" || "$DKU_STORIES_RELEASE_TYPE" == "None" ]]; then
			DKU_STORIES_RELEASE_TYPE="public"
		else
			# Only public release type is accessible from CDN
			DKU_STORIES_IMAGE_HOST="downloads.dataiku.com"
		fi

		DKU_STORIES_IMAGE_HOST=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get stories image_host)
		if [[ -z "$DKU_STORIES_IMAGE_HOST" || "$DKU_STORIES_IMAGE_HOST" == "None" ]]; then
			DKU_STORIES_IMAGE_HOST="cdn.downloads.dataiku.com"
		fi

		DKU_STORIES_IMAGE_URL=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get stories image_url)
		if [[ -z "$DKU_STORIES_IMAGE_URL" || "$DKU_STORIES_IMAGE_URL" == "None" ]]; then
			DKU_STORIES_IMAGE_URL="https://${DKU_STORIES_IMAGE_HOST}/${DKU_STORIES_RELEASE_TYPE}/story/${DSS_VERSION}/dataiku-story-images-${DSS_VERSION}.tar.gz"
			echo "[+] Downloading Stories docker image from: $DKU_STORIES_IMAGE_URL"
		fi

		DKU_STORIES_IMAGE_USERNAME=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get stories image_username)
		if [ "$DKU_STORIES_IMAGE_USERNAME" == "None" ]; then
			unset DKU_STORIES_IMAGE_USERNAME
		fi

		DKU_STORIES_IMAGE_PASSWORD=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get stories image_password)
		if [ "$DKU_STORIES_IMAGE_PASSWORD" == "None" ]; then
			unset DKU_STORIES_IMAGE_PASSWORD
		fi
	
		if [ -n "$DKU_STORIES_IMAGE_USERNAME" ]; then
			DKU_STORIES_IMAGE_URL_WITH_CREDENTIALS=$(sed -e "s^//^//$DKU_STORIES_IMAGE_USERNAME:$DKU_STORIES_IMAGE_PASSWORD@^" <<<"$DKU_STORIES_IMAGE_URL")
		else
			DKU_STORIES_IMAGE_URL_WITH_CREDENTIALS="$DKU_STORIES_IMAGE_URL"
		fi

		DKU_STORIES_SKIP_CERTIFICATE_VERIFICATION=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get stories skip_certificate_verification)
		if [[ "$DKU_STORIES_SKIP_CERTIFICATE_VERIFICATION" == "true" ]]; then
			DKU_STORIES_SKIP_CERTIFICATE_VERIFICATION="-k"
			echo "[+] Skipping certificate verification"
		else
			unset DKU_STORIES_SKIP_CERTIFICATE_VERIFICATION
		fi
	fi

	DKU_INSTALL_ID=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get general installid)

	if [ -n "$DKU_STORIES_IMAGE_URL_WITH_CREDENTIALS" ]; then
		DKU_STORIES_IMAGE_FILE="${DIP_HOME}/tmp/stories/dataiku-story-images-${DKU_INSTALL_ID}.tar.gz"
		DKU_STORIES_REMOVE_IMAGE_FILE="true"
		echo "[+] Saving Stories docker image to $DKU_STORIES_IMAGE_FILE"
		curl --fail -LsS $DKU_STORIES_SKIP_CERTIFICATE_VERIFICATION "$DKU_STORIES_IMAGE_URL_WITH_CREDENTIALS" --create-dirs -o "$DKU_STORIES_IMAGE_FILE"
	fi

	echo "[+] Removing existing Stories/Story docker containers"
	docker rm -f "dataiku-stories-$DKU_INSTALL_ID"
	docker rm -f "dataiku-story-$DKU_INSTALL_ID"

	echo "[+] Removing existing Stories/Story docker images"
	docker image prune -a -f --filter "label=app=dku_stories"
	docker image prune -a -f --filter "label=app=dku_story"

	echo "[+] Loading docker image from $DKU_STORIES_IMAGE_FILE"
	DKU_STORIES_IMAGE_TAG="$(docker load -i "$DKU_STORIES_IMAGE_FILE" | sed -n 's/^Loaded image: dataiku\/story://p')"
	if [ -n "$DKU_STORIES_REMOVE_IMAGE_FILE" ]; then
		rm -f "$DKU_STORIES_IMAGE_FILE"
	fi
	if [ -z "$DKU_STORIES_IMAGE_TAG" ]; then
		echo "[-] Invalid Stories image: Stories setup failed"
		exit 1
	fi

  if [ -n "$enable_stories" -a "$enable_stories" == "true" ]; then
    echo "[+] Enable Stories in general settings"
    DKU_GENERAL_SETTINGS_FILEPATH="$DIP_HOME"/config/general-settings.json
    sed -E '/"dataikuStoriesSettings"[[:space:]]*:/,/\}/ s/"enabled":[[:space:]]*false/"enabled": true/' "$DKU_GENERAL_SETTINGS_FILEPATH" > "$DKU_GENERAL_SETTINGS_FILEPATH.next"
    mv "$DKU_GENERAL_SETTINGS_FILEPATH.next" "$DKU_GENERAL_SETTINGS_FILEPATH"
  fi

	# If image_tag has a value, it is because a version of Stories was specified with the -w flag.
	# Otherwise, the version of Stories will be the same as the DSS version.
	if [ "$stories" != "true" ]; then
		echo "[+] Set image_tag flag to $DKU_STORIES_IMAGE_TAG in install.ini"
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -set stories image_tag "$DKU_STORIES_IMAGE_TAG"
	else
		echo "[+] Remove image_tag flag from install.ini"
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -set stories image_tag ""
	fi

	echo "[+] Stories setup completed successfully"
}

if [ -n "$upgrade" ]
then
	######################################################################################
	#                             Upgrade
	######################################################################################

	# Create or upgrade Python environment unless overridden
	if [ -n "$DKUPYTHONBIN" ]; then
		pythonBin="$DKUPYTHONBIN"

	elif [ -d "$DIP_HOME"/pyenv ]; then
		echo "[+] Migrating Python environment"
		installPyenvOptArgs=("-u")
		if [ -n "$rebuildPyenvIfNeeded" ]; then
			installPyenvOptArgs+=("--rebuild-if-needed")
		fi
		if [ -n "$forcePythonRebuild" ]; then
			installPyenvOptArgs+=("--force-rebuild-pyenv")
		fi
		if [ -n "$BASE_PYTHON" ]; then
			installPyenvOptArgs+=("-p" "$BASE_PYTHON")
		fi
		installPyenvOptArgs+=("$DIP_HOME")
		"$MYDIR"/scripts/_install-pyenv.sh "${installPyenvOptArgs[@]}"
		pythonBin="$DIP_HOME/bin/python"
		precompile_python "$pythonBin" "yes"

	else
		if [ -z "$BASE_PYTHON" ]; then
			echo "[+] Initializing Python environment using platform default"
			"$MYDIR"/scripts/_install-pyenv.sh "$DIP_HOME"
		else
			echo "[+] Initializing Python environment using '$BASE_PYTHON'"
			"$MYDIR"/scripts/_install-pyenv.sh -p "$BASE_PYTHON" "$DIP_HOME"
		fi
		pythonBin="$DIP_HOME/bin/python"
		precompile_python "$pythonBin" "yes"
	fi

	# Perform migration
	echo "[+] Migrating data directory"
	PYTHONIOENCODING=UTF-8 PYTHONUNBUFFERED=1 DKUINSTALLDIR="$MYDIR" DKUJAVABIN="$javaBin" \
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/migrate_auto.py "$DIP_HOME"

	NODETYPE=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get nodetype)
	echo "Node type : $NODETYPE"

	#if node type is DESIGN and stories flag
	if [ -n "$stories" ]; then
		if [ "$NODETYPE" = "design" ]; then
			echo "[+] Loading Stories"
			stories_load
		else
			echo "[-] Stories option is only available on nodetype 'design' : not installing Stories"
		fi
	fi
else
	######################################################################################
	#                             Fresh install
	######################################################################################

	echo "[+] Installation starting"
	cp -p "$MYDIR"/dss-version.json "$DIP_HOME"/
	chmod u+w "$DIP_HOME"/dss-version.json   # should not be necessary
	mkdir -p "$DIP_HOME"/bin "$DIP_HOME"/config

	# Create empty env-site.sh
	cat <<EOF >"$DIP_HOME"/bin/env-site.sh
# This file is sourced last by DSS startup scripts
# You can add local customizations to it
EOF

	if [ -n "$LICENSE" ]; then
		echo "[+] Installing license file"
		cp -p "$LICENSE" "$DIP_HOME"/config/license.json
	fi

	# Create Python environment unless overridden
	if [ -n "$DKUPYTHONBIN" ]; then
		pythonBin="$DKUPYTHONBIN"
		precompile_python "$pythonBin" ""
	else
		echo "[+] Initializing Python environment"
		if [ -z "$BASE_PYTHON" ]; then
			echo "[+] Initializing Python environment using platform default"
			"$MYDIR"/scripts/_install-pyenv.sh "$DIP_HOME"
		else
			echo "[+] Initializing Python environment using '$BASE_PYTHON'"
			"$MYDIR"/scripts/_install-pyenv.sh -p "$BASE_PYTHON" "$DIP_HOME"
		fi
		pythonBin="$DIP_HOME"/bin/python
		precompile_python "$pythonBin" "yes"
	fi

	# Perform various installation steps
	if [ "$NODETYPE" = "design" ] || [ "$NODETYPE" = "automation" ] || [ "$NODETYPE" = "deployer" ]
	then
		DKUINSTALLDIR="$MYDIR" DKUJAVABIN="$javaBin" \
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/install_dss.py "$DIP_HOME" $PORT "$NODETYPE" "$INSTALL_SIZE"

		# Create initial stuff in config (FS connections and users/groups)
		echo "[+] Preparing data directory initial data"
		"$DIP_HOME"/bin/dku -s __initial-setup-home "$DIP_HOME" "$NODETYPE"

		if [ -n "$stories" -a "$NODETYPE" = "design" ]; then
			stories_load
		fi
	elif [ "$NODETYPE" = "govern" ]
	then
		DKUINSTALLDIR="$MYDIR" DKUJAVABIN="$javaBin" \
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/install_govern.py "$DIP_HOME" $PORT "$INSTALL_SIZE"

		# Create initial stuff in config (users/groups)
		echo "[+] Preparing data directory initial data"
		"$DIP_HOME"/bin/dkugovern -s __initial-setup-home "$DIP_HOME"

	elif [ "$NODETYPE" = "api" ]
	then
		DKUINSTALLDIR="$MYDIR" DKUJAVABIN="$javaBin" \
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/install_apinode.py "$DIP_HOME" $PORT

		echo "[+] Preparing the encryption key"
		"$DIP_HOME"/bin/dku generate-crypto-key
		# For API node, everything else is created in Python
	else
		echo "[!] Unexpected node type"
		exit 1
	fi
fi

# Install native IPython kernel
if [ "$NODETYPE" != "api" -a "$NODETYPE" != "govern" -a "$noJupyter" != "1" ]
then
	jupyterData="$DIP_HOME"/jupyter-run/jupyter
	JUPYTER_DATA_DIR="$jupyterData" "$pythonBin" -m ipykernel install --user

	jupyterConfig="$jupyterData"

	# Previous to 9.0.5 it was a symlink
	# Remove the link if exists and enable back the extensions
	if [ -L "$jupyterData/nbextensions" ]; then
		rm "$jupyterData/nbextensions"
	fi

	JUPYTER_CONFIG_DIR="$jupyterData" JUPYTER_DATA_DIR="$jupyterData"  PYTHONPATH="$MYDIR/dku-jupyter/packages/" "$pythonBin" -c 'from notebook.nbextensions import main;main()' install --py widgetsnbextension --user

	# Enable extensions by default if we don't already have a file
	if [ ! -f "$jupyterConfig/nbconfig/notebook.json" ]; then
		"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable collapsible_headings/main
		"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable codefolding/main
		"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable toggle_all_line_numbers/main
		"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable hide_input_all/main
		"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable addbefore/main
	else
		# Enable extension if the extension's folder is not present (migration prior to 9.0.5)
		for extension in $(JUPYTER_CONFIG_DIR="$jupyterData/config" "$DIP_HOME/bin/dssadmin" jupyter-nbextensions list | tail -n +2)
		do
			if [ ! -f "$jupyterData/nbextensions/$extension.js" ]; then
				"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable $extension
			fi
		done
		for extension in $("$DIP_HOME/bin/dssadmin" jupyter-nbextensions list | tail -n +2)
		do
			if [ ! -f "$jupyterData/nbextensions/$extension.js" ]; then
				"$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable $extension
			fi
		done
	fi
  "$DIP_HOME/bin/dssadmin" jupyter-nbextensions enable jupyter-js-widgets/extension
fi

# Add log4j config for apinode
if [ "$NODETYPE" = "api" ]
then
	cp -p "$MYDIR/dist/apinode-default-log4j.properties" "$DIP_HOME"/bin/log4j.properties
fi

if [ "$NODETYPE" != "govern" ]
then
	"$DIP_HOME"/bin/dssadmin -noLog regenerate-config
else
	"$DIP_HOME"/bin/govern-admin -noLog regenerate-config
fi

# Final steps that we always do
mkdir -p "$DIP_HOME"/lib/jdbc  "$DIP_HOME"/lib/java "$DIP_HOME"/lib/python

rm -f "$DIP_HOME"/DSS-INITIAL-INSTALL-IN-PROGRESS.txt

echo "$DIP_HOME" > "$DIP_HOME"/install-support/expected-dip-home.txt

echo "***************************************************************"
echo "* Installation complete (DSS node type: $NODETYPE)"

if [ -z "$upgrade" -a "$NODETYPE" = "govern" ]; then
	echo "* First, you will need to setup the PostgreSQL configuration in the following file:"
	echo "         '$DIP_HOME/config/dip.properties'"
	echo "* Secondly, setup the basic structure in the database using:"
	echo "         '$DIP_HOME/bin/govern-admin init-db'"
fi

echo "* Next, start DSS using:"
echo "*         '$DIP_HOME/bin/dss start'"

if [ -z "$upgrade" -a "$NODETYPE" != "api" ]; then
	# Initial installation
	echo "* Dataiku DSS will be accessible on http://<SERVER ADDRESS>:$PORT"
	if [ "$(uname)" != "Darwin" ]; then
		echo "*"
		echo "* You can configure Dataiku DSS to start automatically at server boot with:"
		echo "*    sudo -i \"$MYDIR/scripts/install/install-boot.sh\" \"$DIP_HOME\" $(id -un)"
	fi
fi

echo "***************************************************************"

) 2>&1 | tee -a "$logFile"
