from abc import abstractmethod, ABCMeta
import six


@six.add_metaclass(ABCMeta)
class AbstractDeepHubScoreExplainer(object):
    """ Context manager interface to:
        - attach a hook to the NN model when with_explanations is true and __enter__ is called
        - detach the hook on __exit__
    """
    def __init__(self, deephub_model, nn_model, with_explanations, n_explanations):
        self.enabled = with_explanations
        self.n_explanations = n_explanations

    def __enter__(self):
        if self.enabled:
            self.initialize_model_for_explanations()

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.enabled:
            self.free_model_from_explainer()

    @abstractmethod
    def initialize_model_for_explanations(self):
        raise NotImplementedError

    @abstractmethod
    def free_model_from_explainer(self):
        raise NotImplementedError


class NoOpScoreExplainer(AbstractDeepHubScoreExplainer):
    def initialize_model_for_explanations(self):
        pass

    def free_model_from_explainer(self):
        pass
