# Set a non-impersonated process in a given cgroupv2 cgroup
# To be run as root via sudo

import logging
import os
import os.path as osp
import sys
import securityutils

def handle_main(args):
    # Safety checks
    if os.geteuid() != 0 or not os.environ.get("SUDO_UID"):
        raise Exception("Must be called through sudo")
    dss_uid = int(os.environ["SUDO_UID"])

    proc_uid = os.stat("/proc/%d/stat" % args.pid).st_uid
    if proc_uid != dss_uid:
        raise Exception("Target process %d has uid %d, does not match calling process uid %d" % (args.pid, proc_uid, dss_uid))

    cgroup_procs_file = osp.join(args.cgroup, "cgroup.procs")
    cgroup_procs_file = securityutils.check_within_dip_home(args.security_conf_dir, cgroup_procs_file)
    procs_file_uid = os.stat(cgroup_procs_file).st_uid
    if procs_file_uid != dss_uid:
        raise Exception("Target cgroup %s has uid %d, does not match calling process uid %d" % (cgroup_procs_file, procs_file_uid, dss_uid))

    logging.info("Will set process %d in cgroup %s" % (args.pid, args.cgroup))
    with securityutils.safe_open(cgroup_procs_file, "w") as f:
        f.write("%d" % args.pid)


if __name__ == "__main__":
    import argparse

    logging.basicConfig(level=logging.INFO, format='%(asctime)s %(process)d %(levelname)s %(message)s')

    parser = argparse.ArgumentParser(description='DSS cgroupv2 process classification helper')
    parser.add_argument("security_conf_dir")
    parser.add_argument("--pid", type=int, required=True)
    parser.add_argument("--cgroup", required=True)

    args = parser.parse_args()

    try:
        handle_main(args)
    except Exception as e:
        logging.exception("Unable to set cgroup %s for process %s : %s" % (args.cgroup, args.pid, str(e)))
        sys.exit(1)
