#! /usr/bin/env python

# Gets the stacks of the backend

import os
import os.path as osp
import re
import subprocess
import sys
import requests
import six
import json

session = requests.Session()

TIMEOUT = 20

NODE_TYPE = os.getenv("DKU_NODE_TYPE")
IS_API_NODE = NODE_TYPE == "api"
PORT_VAR = "DKU_APIMAIN_PORT" if IS_API_NODE else "DKU_BACKEND_PORT"
PORT = os.getenv(PORT_VAR)

if not PORT:
    print("%s is not defined" % PORT_VAR)
    sys.exit(1)

SEPARATOR = "\n" + "*" * 32

with open(osp.join(os.getenv("DIP_HOME"), "run", "shared-secret.txt")) as f:
    shared_secret = f.read().strip()

session.headers.update({"X-DKU-IPythonSharedSecret" : shared_secret})


def get_backend_pid():
    pid_re = re.compile("^([A-Za-z]*).*pid ([0-9]+).*")

    (stdout, _) = subprocess.Popen("$DIP_HOME/bin/dss status", shell=True, stdout=subprocess.PIPE).communicate()

    for line in six.ensure_str(stdout).split("\n"):
        match = pid_re.match(line)
        if match and match.group(1) == "backend":
            return match.group(2)
        if match and match.group(1) == "apimain":
            return match.group(2)
    raise Exception("Backend pid not found")


print(SEPARATOR)
print("Stacks")
try:
    r = session.get('http://127.0.0.1:%s/dip/api/debugging/dump-backend-stacks' % PORT, timeout=TIMEOUT)
    print(r.text)
except Exception as e:
    print("Could not get backend stats using HTTP, trying kill -3 (%s)" % e)

    try:
        backend_pid = get_backend_pid()
        print("Killing -3: %s" % backend_pid)
        subprocess.Popen("kill -3 %s" % backend_pid, shell=True)
    except Exception as e2:
        print("Could not get stacks at all: %s" % e2)

print(SEPARATOR)
if IS_API_NODE:
    print("No transactions state dump on API node")
else:
    print("TXN status")
    try:
        r = session.get('http://127.0.0.1:%s/dip/api/debugging/dump-transactions-state' % PORT, timeout=TIMEOUT)
        print(r.text)
    except Exception as e:
        print("Could not get TXN status: %s" % e)

print(SEPARATOR)
if IS_API_NODE:
    print("No named locks dump on API node")
else:
    print("NamedLocks status")
    try:
        r = session.get('http://127.0.0.1:%s/dip/api/debugging/dump-named-locks' % PORT, timeout=TIMEOUT)
        print(r.text)
    except Exception as e:
        print("Could not get locks status: %s" % e)

print(SEPARATOR)
print("Metrics")
try:
    r = session.get('http://127.0.0.1:%s/dip/api/debugging/dump-metrics' % PORT, timeout=TIMEOUT)
    print(r.text)
except Exception as e:
    print("Could not get metrics status: %s" % e)


print(SEPARATOR)
if IS_API_NODE:
    print("No API tickets dump on API node")
else:
    print("Tickets")
    try:
        r = session.get('http://127.0.0.1:%s/dip/api/debugging/dump-api-tickets' % PORT, timeout=TIMEOUT)
        print(r.text)
    except Exception as e:
        print("Could not get API tickets status: %s" % e)


print(SEPARATOR)
print("LLM mesh dump")
try:
    r = session.get('http://127.0.0.1:%s/dip/api/debugging/dump-llm-mesh' % PORT, timeout=TIMEOUT)
    data = r.json()
    print(json.dumps(data, indent=2))

except requests.exceptions.JSONDecodeError:
    print("Could not get LLM mesh dump (Response is not valid JSON):\n", r.text)

except Exception as e:
    print("Could not get LLM mesh dump: %s" % e)