#!/bin/bash -e
# Boostraps or upgrades a Python virtual environment for Dataiku Science Studio

MYDIR=$(cd "$(dirname "$0")" && pwd -P)
INSTALLDIR=$(dirname "$MYDIR")

#
# Parse arguments
#
Usage() {
    echo >&2 "Usage: $0 [-u] [--force-rebuild-pyenv] [-f] [-p PYTHONBIN] DSS_DATADIR"
    echo >&2 "  -u : upgrade existing DSS Python environment (default: create new)"
    echo >&2 "  --force-rebuild-pyenv : rebuild the Python environment even if it already exists"
    echo >&2 "  -f : whether the installation is for Fleet Manager"
    echo >&2 "  -p PYTHONBIN : base Python binary to use when creating new environment"
    exit 1
}

# Version of the python binary
# usage: $(getPythonVersion "$pythonBin")
getPythonVersion() {
    "$1" -c "import sysconfig;print(sysconfig.get_python_version())"
}

# Version and additional info of the python binary
# Can be used to discriminate between different python builds
# output example: 3.7.13 (default, Mar 18 2022, 08:41:12)\n[Clang 12.0.0 (clang-1200.0.32.29)]
# usage: $(getPythonFullVersion "$pythonBin")
getPythonFullVersion() {
    "$1" -c "import sys;print(sys.version)"
}

# Checks if a given python is deprecated
# Should not be used to check if a python is supported
# usage: if isDeprecatedPython $pythonBin; then...
isDeprecatedPython() {
    case "$(getPythonVersion "$1")" in
        "2.7" | "3.6" | "3.7")
            # return 0 is true
            return 0
            ;;
        *)
            # return 1 is false
            return 1
            ;;
    esac
}

upgrade=
forceRebuild=
rebuildIfNeeded=
fleetManagerInstall=
pythonBin=
keepPyenvBackup=
while [[ "$1" == -* ]]; do
    if [[ "$1" == "-u" ]]; then
        upgrade=1
        shift
    elif [[ "$1" == "--force-rebuild-pyenv" ]]; then
        forceRebuild=1
        shift
    elif [[ "$1" == "--rebuild-if-needed" ]]; then
        rebuildIfNeeded=1
        shift
    elif [[ "$1" == "-f" ]]; then
        fleetManagerInstall=1
        shift
    elif [[ $# -ge 2 && "$1" == "-p" ]]; then
        pythonBin="$2"
        shift 2
    else
        Usage
    fi
done
if [[ $# -ne 1 || ! -d "$1" ]]; then
    Usage
fi
DIP_HOME="$1"

if [ -f "$DIP_HOME/pyenv/bin/python" ] && isDeprecatedPython "$DIP_HOME/pyenv/bin/python"; then
    pythonVersion=$(getPythonVersion "$DIP_HOME/pyenv/bin/python")
    echo >&2 "[!] *****************************************"
    echo >&2 "[!] Your builtin python environment ($DIP_HOME/pyenv) is still using Python $pythonVersion. "
    echo >&2 "[!] This version of python is no longer supported."
    echo >&2 "[!] a new python built-in environment will be created."
    echo >&2 "[!] *****************************************"
    forceRebuild=1
    # Keeping pyenv backup in case users need to come back to it later
    keepPyenvBackup=1
fi

# Determine the python binary to use
if [[ -z "$upgrade" || -n "$forceRebuild" ]]; then
    # Fresh install or force rebuild: create a new pyenv
    if [ -z "$pythonBin" ]; then
        pythonBin=$("$MYDIR"/_default_python.sh)
        echo "+ Using default base Python for this platform : $pythonBin"
    else
        echo "+ Using specified base Python binary: $pythonBin"
    fi
else
    # Upgrade existing environment
    if [ -n "$rebuildIfNeeded" ]; then
        # Get current version of Python
        existingPyenvBin="$DIP_HOME"/pyenv/bin/python
        # the following `|| :` are to not crash if the current base env is broken
        existingPyenvFullVersion=$(getPythonFullVersion "$existingPyenvBin") || :
        echo "+ Current DSS Python version: $existingPyenvFullVersion"

        # Get base Python version
        if [ -n "$pythonBin" ]; then
            candidatePythonBin="$pythonBin"
        else
            candidatePythonBin=$("$MYDIR"/_default_python.sh)
        fi
        candidatePythonFullVersion=$(getPythonFullVersion "$candidatePythonBin")
        echo "+ Base Python version: $candidatePythonFullVersion"

        if [ "$existingPyenvFullVersion" != "$candidatePythonFullVersion" ]; then
            echo "! Version mismatch, forcing Python environment rebuild"
            pythonBin="$candidatePythonBin"
            forceRebuild=1
        else
            echo "+ Versions match, no rebuild necessary"
            pythonBin="$existingPyenvBin"
        fi
    elif [ -n "$pythonBin" ]; then
        echo >&2 "[-] Error : cannot specify base Python binary when upgrading environment"
        exit 1
    else
        # Reuse existing virtual environment
        pythonBin="$DIP_HOME"/pyenv/bin/python
    fi
fi

#
# Check Python interpreter
#
if [ "$(uname)" = "Darwin" ]; then
    pythonPlatform="macosx"
else
    pythonPlatform="linux"
fi

pythonVersion=$(getPythonVersion "$pythonBin")

# Binary compat tag should actually be retrieved with : packaging.tags.sys_tags()
# but we may not have this package available, so we hardcode the value for now
case "$pythonVersion" in
    "3.9")
        pythonTag="cp39"
        pythonArch="$pythonTag-$pythonPlatform"
        pkgDir="python39.packages"
        linkNbConvertTemplates="yes"
        ;;
    "3.10")
        pythonTag="cp310"
        pythonArch="$pythonTag-$pythonPlatform"
        pkgDir="python310.packages"
        linkNbConvertTemplates="yes"
        ;;
    "3.11")
        pythonTag="cp311"
        pythonArch="$pythonTag-$pythonPlatform"
        pkgDir="python311.packages"
        linkNbConvertTemplates="yes"
        ;;
    *)
        echo >&2 "Unsupported Python version: $pythonVersion"
        exit 1
        ;;
esac

# Check the arch tag from the python.packages subdirectory
pkgArch=$(awk -F '=' '($1 == "arch"){print $2}' "$INSTALLDIR/$pkgDir/dss-version.txt")

if [ "$pythonArch" != "$pkgArch" ]; then
    echo >&2 "[-] Error: Python interpreter $pythonBin is not compatible with DSS-provided standard packages"
    echo >&2 "[-] Actual architecture $pythonArch differs from expected $pkgArch"
    exit 1
fi

if [[ -n "$forceRebuild" && -d "$DIP_HOME/pyenv" ]]; then
    # Remove the virtualenv, keeping backup
    envBackup="$DIP_HOME/pyenv.backup.$$"
    echo "+ Backing up $DIP_HOME/pyenv to $envBackup"
    rm -rf "$envBackup"
    mv "$DIP_HOME"/pyenv "$envBackup"
else
    envBackup=
fi

if [[ -z "$upgrade" || -n "$forceRebuild" ]]; then
    # Initialize virtualenv
    mkdir "$DIP_HOME"/pyenv
    "$MYDIR"/_create-virtualenv.sh "$pythonBin" --no-download "$DIP_HOME"/pyenv || {
        echo >&2 "[-] Error: could not initialize Python virtualenv in $DIP_HOME/pyenv"
        rm -rf "$DIP_HOME"/pyenv
        if [[ -n "$envBackup" ]]; then
            echo >&2 "[-] Recovering backed up Python virtualenv from $envBackup"
            mv "$envBackup" "$DIP_HOME/pyenv"
        fi
        exit 1
    }
fi

# Link standard packages from kit
rm -f "$DIP_HOME"/pyenv/lib/python"$pythonVersion"/sitecustomize.{py,pyc} \
      "$DIP_HOME"/pyenv/lib/python"$pythonVersion"/__pycache__/sitecustomize.*.pyc  # migration from DSS versions <= 8
mkdir -p "$DIP_HOME"/pyenv/lib/python"$pythonVersion"/site-packages
if [[ -n "$fleetManagerInstall" ]]; then
    cat >"$DIP_HOME"/pyenv/lib/python"$pythonVersion"/site-packages/_dss_packages.pth <<EOF
import site; site.addsitedir('$INSTALLDIR/$pkgDir'); site.addsitedir('$INSTALLDIR/python')
EOF
else
    cat >"$DIP_HOME"/pyenv/lib/python"$pythonVersion"/site-packages/_dss_packages.pth <<EOF
import site; site.addsitedir('$INSTALLDIR/$pkgDir')
EOF
fi

# If DSS and required, link nbconvert packages
if [ "$linkNbConvertTemplates" = "yes" ]; then
    if [ -d "$INSTALLDIR/resources/jupyter/nbconvert-templates" ]; then
        mkdir -p "$DIP_HOME/pyenv/share/jupyter/nbconvert"
        ln -sfn "$INSTALLDIR/resources/jupyter/nbconvert-templates" "$DIP_HOME/pyenv/share/jupyter/nbconvert/templates"
    fi
fi

# Entry point wrapper scripts
mkdir -p "$DIP_HOME"/bin
for x in python pip; do
    rm -f "$DIP_HOME"/bin/$x
    cat <<EOF >"$DIP_HOME"/bin/$x
#!/bin/bash -e
exec "$DIP_HOME/pyenv/bin/$x" "\$@"
EOF
    chmod a+rx "$DIP_HOME"/bin/$x
done

if [[ -n "$envBackup" ]]; then
    if [ -n "$keepPyenvBackup" ]; then
        echo "+ Your builtin python environment was backed up to $envBackup"
        echo "+ You can delete $envBackup after the migration is completed"
    else
        echo "+ Removing backed up environment $envBackup"
        rm -rf "$envBackup"
    fi
fi
