#' Executes SQL query as a recipe fragment
#' @export
dkuSQLExecRecipeFragment <- function(outputDatasetName, query, preQueries = NULL, postQueries = NULL, overwriteOutputSchema=TRUE, dropPartitionedOnSchemaMismatch=FALSE) {
    parsedRSpec <- getDkuFlowSpec()

    currentActivityId <- parsedRSpec$currentActivityId
    getUrl = dku_intercom__get_jek_url("/sql/execute-partial-query-recipe")
	fullDatasetName <- dku__resolve_smart_name(outputDatasetName)
	
    requestBody <- list(outputDataset=fullDatasetName, activityId=currentActivityId, query=query, overwriteOutputSchema=overwriteOutputSchema, dropPartitionedOnSchemaMismatch=dropPartitionedOnSchemaMismatch)
    if ( !(is.null(preQueries)) ) {
    	requestBody[["preQueries"]] <- RJSONIO::toJSON(preQueries)
    }
    if ( !(is.null(postQueries)) ) {
    	requestBody[["postQueries"]] <- RJSONIO::toJSON(postQueries)
    }
    resp = POST(getUrl,body = requestBody,
        encode="form", dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Failed to execute partial recipe")
}

#' Executes a SQL query and returns the results as a data.frame
#'
#' @param connection Name of the SQL connection to use. This can also be a dataset name. In that case, findConnectionFromDataset must be set to TRUE
#' @param query A SQL query that may or may not return results
#' @param preQueries A list of SQL queries to execute before the main query
#' @param postQueries A list of SQL queries to execute after the main query
#' @param findConnectionFromDataset Set this to TRUE if the "connection" you passed was actually a dataset name
#' @param dbType Internal usage, do not modify
#
#' @return A data.frame with the query results, if any. Else, an empty dataframe
#' 
#' @examples
#' \dontrun{
#' # Identify a connection directly
#' dkuSQLQueryToData('my-postgresql-connection', 'SELECT COUNT(*) FROM mytable')
#'
#' # Identify a connection by a dataset name
#' dkuSQLQueryToData('my-dataset', 'SELECT COUNT(*) FROM mytable', findConnectionFromDataset=TRUE)
#'
#' # Insert data and commit
#' dkuSQLQueryToData('my-connection', 'INSERT INTO mytable VALUES (42, 'stuff')', postQueries=c("COMMIT"))
#' }
#' @details
#'
#' In most cases, DSS connections do not automatically commit transactions. When this is the case, in particular, 
#' write operations performed as part of the query will not actually be committed. For example, if you insert or delete records.
#' In that case, you need to add a 'COMMIT' statement as a post query.
#'
#' @export
dkuSQLQueryToData <- function(connection, query, preQueries=NULL, postQueries=NULL, findConnectionFromDataset=FALSE, dbType="sql") {
 	if (findConnectionFromDataset == TRUE) {
		connection <- dku__resolve_smart_name(connection)
	}
    getStartPathURL = dku_intercom__get_backend_url("/sql-queries/start-streaming")
    getPathURL = dku_intercom__get_backend_url("/sql-queries/stream")
    getVerificationPathURL = dku_intercom__get_backend_url("/sql-queries/verify")
    
    requestBody <- list(connection=connection, query=query, dbType=dbType, findConnectionFromDataset=findConnectionFromDataset, projectKey=dku_remoterun__get_env_var("DKU_CURRENT_PROJECT_KEY"))
    if ( !(is.null(preQueries)) ) {
    	requestBody[["preQueries"]] <- RJSONIO::toJSON(preQueries)
    }
    if ( !(is.null(postQueries)) ) {
    	requestBody[["postQueries"]] <- RJSONIO::toJSON(postQueries)
    }
    resp = POST(getStartPathURL,body = requestBody,
        encode="form",  dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Failed to execute query")

	session <- content(resp)
	queryId <- session$queryId
	
	# check for empty result set
	if ( session$hasResults ) {
	    resp = POST(getPathURL,body = list(queryId=queryId),
	        encode="form",  dku__get_auth_headers(), dku_intercom__get_httr_config())

	    dku__check_api_error(resp, "Failed to stream query")

		results <- content(resp, "text")
		# data comes back as csv
		df <- read.delim(textConnection(results) , header=TRUE, stringsAsFactors=F)
	    
	    # verify that the data is not incomplete or that one of the post queries failed
	    resp = POST(getVerificationPathURL,body = list(queryId=queryId),
	        encode="form",  dku__get_auth_headers(), dku_intercom__get_httr_config())

    	dku__check_api_error(resp, "Failed to verify query")
    	df
    }
}

checkHiveImpalaConnectionParameters <- function(database, connection, findConnectionFromDataset) {
    if (findConnectionFromDataset == TRUE) {
        if ( ! is.null(connection) ) {
            stop("The connection cannot be set when using findConnectionFromDataset=True") 
        }
        if ( is.null(database) ) {
            stop("The dataset needs to be specified when using findConnectionFromDataset is True") 
        }
    } else {
        if (! is.null(database) && ! is.null(connection)) {
            stop("Only one of database or connection can be set")
        }
        if (is.null(database) && is.null(connection)) {
            stop("One of database or connection is required") 
        }
    }
}

buildVirtualConnection <- function(database, connection, findConnectionFromDataset, dbType) {
    if (findConnectionFromDataset == TRUE) {
    	return(database)
    }
	virtualConnection <- ""
    if ( is.null(database) ) {
		virtualConnection <- paste0("@virtual(",dbType,"):connection:",connection)
    } else {
		virtualConnection <- paste0("@virtual(",dbType,"):",database)
    }
    return(virtualConnection)
}

#' Executes Hive query as a recipe fragment
#' @export
dkuHiveExecRecipeFragment <- function(outputDatasetName, query, overwriteOutputSchema=TRUE, dropPartitionedOnSchemaMismatch=FALSE) {
    parsedRSpec <- getDkuFlowSpec()

    currentActivityId <- parsedRSpec$currentActivityId
    getUrl = dku_intercom__get_jek_url("/hive/execute-partial-recipe")
	fullDatasetName <- dku__resolve_smart_name(outputDatasetName)
	
    requestBody <- list(outputDataset=fullDatasetName, activityId=currentActivityId, query=query, overwriteOutputSchema=overwriteOutputSchema, dropPartitionedOnSchemaMismatch=dropPartitionedOnSchemaMismatch)
    resp = POST(getUrl,body = requestBody,
        encode="form", dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Failed to execute partial recipe")
}

#' Executes a Hive query and returns the results as a data.frame
#'
#' @param database Name of the Hive database to use. This can also be a dataset name. In that case, findConnectionFromDataset must be set to TRUE
#' @param query A Hive query that may or may not return results
#' @param preQueries A list of Hive queries to execute before the main query
#' @param postQueries A list of Hive queries to execute after the main query
#' @param connection Name of a HDFS connection whose Hive database to use. If this parameter is defined, then database has to be NULL
#' @param findConnectionFromDataset Set this to TRUE if the "database" you passed was actually a dataset name
#
#' @return A data.frame with the query results, if any. Else, an empty dataframe
#' 
#' @examples
#' \dontrun{
#' # Identify a database directly
#' dkuHiveQueryToData('my-database', 'SELECT COUNT(*) FROM mytable')
#'
#' # Identify a connection by a dataset name
#' dkuHiveQueryToData('my-dataset', 'SELECT COUNT(*) FROM mytable', findConnectionFromDataset=TRUE)
#'
#' # Insert data and commit
#' dkuHiveQueryToData('my-database', 'INSERT INTO mytable VALUES (42, 'stuff')', postQueries=c("COMMIT"))
#' }
#'
#' @export
dkuHiveQueryToData <- function(database, query, preQueries=NULL, postQueries=NULL, connection=NULL, findConnectionFromDataset=TRUE) {
    checkHiveImpalaConnectionParameters(database, connection, findConnectionFromDataset)
	virtualConnection <- buildVirtualConnection(database, connection, findConnectionFromDataset, "hive-jdbc")
	df <- dkuSQLQueryToData(virtualConnection, query, preQueries, postQueries, findConnectionFromDataset, "hive-jdbc")
    return(df)
}

#' Executes Impala query as a recipe fragment
#' @export
dkuImpalaExecRecipeFragment <- function(outputDatasetName, query, preQueries = NULL, postQueries = NULL, overwriteOutputSchema=TRUE, useStreamMode=TRUE) {
    parsedRSpec <- getDkuFlowSpec()

    currentActivityId <- parsedRSpec$currentActivityId
    getUrl = dku_intercom__get_jek_url("/impala/execute-partial-query-recipe")
	fullDatasetName <- dku__resolve_smart_name(outputDatasetName)

    requestBody <- list(outputDataset=fullDatasetName, activityId=currentActivityId, query=query, overwriteOutputSchema=overwriteOutputSchema, useStreamMode=useStreamMode)
    if ( !(is.null(preQueries)) ) {
    	requestBody[["preQueries"]] <- RJSONIO::toJSON(preQueries)
    }
    if ( !(is.null(postQueries)) ) {
    	requestBody[["postQueries"]] <- RJSONIO::toJSON(postQueries)
    }
    resp = POST(getUrl,body = requestBody,
        encode="form", dku__get_auth_headers(), dku_intercom__get_httr_config())

    dku__check_api_error(resp, "Failed to execute partial recipe")
}

#' Executes a Impala query and returns the results as a data.frame
#'
#' @param database Name of the Hive database to use. This can also be a dataset name. In that case, findConnectionFromDataset must be set to TRUE
#' @param query A Impala query that may or may not return results
#' @param preQueries A list of Impala queries to execute before the main query
#' @param postQueries A list of Impala queries to execute after the main query
#' @param connection Name of a HDFS connection whose Hive database to use. If this parameter is defined, then database has to be NULL
#' @param findConnectionFromDataset Set this to TRUE if the "database" you passed was actually a dataset name
#
#' @return A data.frame with the query results, if any. Else, an empty dataframe
#' 
#' @examples
#' \dontrun{
#' # Identify a database directly
#' dkuImpalaQueryToData('my-database', 'SELECT COUNT(*) FROM mytable')
#'
#' # Identify a connection by a dataset name
#' dkuImpalaQueryToData('my-dataset', 'SELECT COUNT(*) FROM mytable', findConnectionFromDataset=TRUE)
#'
#' # Insert data and commit
#' dkuImpalaQueryToData('my-database', 'INSERT INTO mytable VALUES (42, 'stuff')', postQueries=c("COMMIT"))
#' }
#'
#' @export
dkuImpalaQueryToData <- function(database, query, preQueries=NULL, postQueries=NULL, connection=NULL, findConnectionFromDataset=TRUE) {
    checkHiveImpalaConnectionParameters(database, connection, findConnectionFromDataset)
	virtualConnection <- buildVirtualConnection(database, connection, findConnectionFromDataset, "impala-jdbc")
	df <- dkuSQLQueryToData(virtualConnection, query, preQueries, postQueries, findConnectionFromDataset, "impala-jdbc")
    return(df)
}
