import sys
import os
import re
from datetime import datetime
import itertools

from dataiku.base.utils import safe_unicode_str
from dataiku.core import flow
from dataiku.base import remoterun


if sys.version_info > (3,0):
    dku_basestring_type = str
    dku_zip_longest = itertools.zip_longest
else:
    dku_basestring_type = basestring
    dku_zip_longest = itertools.izip_longest


def get_shared_secret():
    with open('%s/shared-secret.txt' % get_dip_home(), 'r') as fp:
        secret = fp.read()
        return secret.strip()

def set_dip_home(dip_home):
    remoterun.set_dku_env_var_and_sys_env_var('DIP_HOME', dip_home)

def get_dip_home():
    return remoterun.get_env_var('DIP_HOME')

def is_container_exec():
    return os.environ.get('DKU_CONTAINER_EXEC', '0') == '1'


class Computable(object):

    def _init_data_from_flow(self, obj_type, project_key):
        if self.lookup is None:
            raise Exception("No identifier given for %s" % obj_type)

        self.readable = False
        self.writable = False

        # Flow mode, initialize partitions to read and write and read/write flags
        if flow.FLOW is not None:
            for flow_input in flow.FLOW["in"]:
                if flow_input["smartName"] == self.lookup or flow_input["fullName"] == self.lookup or flow_input.get("boxLabel", None) == self.lookup or flow_input.get("modelLabel", None) == self.lookup:
                    self.readable = True
                    self.spec_item = flow_input
                    self.name = flow_input["fullName"]
                    if "partitions" in flow_input:
                        self.read_partitions = flow_input["partitions"]
            for flow_output in flow.FLOW["out"]:
                if flow_output["smartName"] == self.lookup or flow_output["fullName"] == self.lookup or flow_output.get("boxLabel", None) == self.lookup or flow_output.get("modelLabel", None) == self.lookup:
                    self.name = flow_output["fullName"]
                    self.writable = True
                    self.spec_item = flow_output
                    if "partition" in flow_output:
                        self.writePartition = flow_output["partition"]
            if not self.readable and not self.writable:
                raise Exception("%s %s cannot be used : declare it as input or output of your recipe" % (obj_type, self.lookup))
            (self.project_key, self.short_name) = self.name.split(".", 1)


    @property
    def full_name(self,):
        return self.project_key + "." + self.short_name


    def set_write_partition(self,spec):
        """Sets which partition of the dataset gets written to when
        you create a DatasetWriter. Setting the write partition is
        not allowed in Python recipes, where write is controlled by
        the Flow."""
        if flow.FLOW is not None and self.ignore_flow == False:
            raise Exception("You cannot explicitly set partitions when "
                            "running within Dataiku Flow")
        self.writePartition = spec

    def add_read_partitions(self, spec):
        """Add a partition or range of partitions to read.

        The spec argument must be given in the DSS partition spec format.
        You cannot manually set partitions when running inside
        a Python recipe. They are automatically set using the dependencies.
        """
        if flow.FLOW is not None and self.ignore_flow == False:
            raise Exception("You cannot explicitly set partitions when "
                            "running within Dataiku Flow")
        if self.read_partitions is None:
            self.read_partitions = []
        self.read_partitions.append(spec)


# See com.dataiku.dip.partitioning.PartitioningUtils
class PartitionEscaper:

    ONLY_ALPHANUM = "[^0-9a-zA-Z]"
    ONLY_ALPHANUM_DASHES = "[^0-9a-zA-Z-]"

    PYTHON3 = sys.version_info >= (3,0,0)

    @staticmethod
    def escape(partition_name, keep=ONLY_ALPHANUM):
        clean_partition_name = safe_unicode_str(partition_name)

        def str_to_ord(s):
            if isinstance(s, str):  # for py 2
                return map(ord, s)
            elif isinstance(s, bytes):  # for py 3
                return s
            else:  # should never happen
                return list()

        def escape_char(match):
            match_str = match.group().encode("utf8")
            return safe_unicode_str("".join("_{:02x}".format(c) for c in str_to_ord(match_str)))

        return re.sub(keep, escape_char, clean_partition_name)

    @staticmethod
    def unescape(partition_name):
        DECODE = "(?:_[0-9a-f]{2})+"
        clean_partition_name = safe_unicode_str(partition_name)

        def unescape_char(match):
            match_str = match.group()
            ret = bytearray([int("0x" + hex_str, 0) for hex_str in match_str.split("_")[1:]]).decode("utf8")
            return safe_unicode_str(ret)

        return re.sub(DECODE, unescape_char, clean_partition_name)

    @staticmethod
    def build_partition_id(partition_values, dimension_names, dimension_types):
        """
            Builds partition id using proper encoding and date conversion.

            ex:
            - ("female", "C") => "female|C"
            - ("2020-01", "C") => "2020-01|C"
            - ("2020-01-01T00:00:00.000Z", "C") => "2020-01|C"
            - (pd.Timestamp(2020-01-01 00:00:00+0000), "C") => "2020-01|C"
            - "1577836800000" => "2020-01"  # works only for UNIX timestamp in milliseconds

            :param str/iterable(str) partition_values: single partition value of iterable of partition values
            :param list dimension_names: names of the partition dimensions
            :param list dimension_types: types of the partition dimensions

            Note: partition_values, dimension_names and dimension_types always have the same sizes

            :return: str (py3) or unicode (py2)
        """
        if isinstance(partition_values, (dku_basestring_type, bytes, datetime)):
            assert len(dimension_names) == len(dimension_types) == 1, "dimension_names/dimension_types should only contain 1 element"
            return safe_unicode_str(get_dimension_value(partition_values, dimension_names[0], dimension_types[0]))
        elif hasattr(partition_values, "__iter__"):
            assert len(partition_values) == len(dimension_names) == len(dimension_types), "dimension_names and dimension_types should have the same length as partition_values"
            return safe_unicode_str(
                u"|".join(
                    safe_unicode_str(get_dimension_value(value, dimension_name, dimension_type))
                    for value, dimension_name, dimension_type in zip(partition_values, dimension_names, dimension_types)
                )
            )
        else:  # should not happen
            return safe_unicode_str(partition_values)


def get_dimension_value(value, dimension_name, dimension_type, dimensions_values=None):
    """Extract the exact partition dimension value from the dimension type.

    Duplicated in dataikuscoring.models.partitioned, so make sure to update both.

    Args:
        :param str/int/float/datetime value: value of the partition dimension, can be of many form (str, date, timestamp, ...)
        :param str dimension_name: name of the partition dimension
        :param str dimension_type: type of the partition dimension
        :param dict dimensions_values: dict of list of dimension values per dimension name. Defaults to None.

        :return: str
    """    
    if dimension_type == "DISCRETE":
        return value
    else:
        return _get_time_dimension_value(value, dimension_name, dimension_type, dimensions_values=dimensions_values)


def _get_time_dimension_value(value, dimension_name, dimension_type, dimensions_values=None):
    """
    This is basically a copy of what is done in Java (PartitionedPipeline.getTimeDimensionValue).
    """
    def _get_time_dimension_value_from_timestamp(value, dimension_type, milliseconds=True):
        datetime_value = datetime.fromtimestamp(value / 1000) if milliseconds else datetime.fromtimestamp(value)
        return _get_time_dimension_value_from_datetime(datetime_value, dimension_type)
    
    def _get_time_dimension_value_from_datetime(value, dimension_type):
        dimension_value = ""
        if dimension_type == "HOUR":
            dimension_value = "-%02d" % value.hour
        if dimension_type in ("HOUR", "DAY"):
            dimension_value = "-%02d" % value.day + dimension_value
        if dimension_type in ("HOUR", "DAY", "MONTH"):
            dimension_value = "-%02d" % value.month + dimension_value
        if dimension_type in ("HOUR", "DAY", "MONTH", "YEAR"):
            dimension_value = "%02d" % value.year + dimension_value
        return dimension_value

    if isinstance(value, (str)):
        if dimensions_values and value in dimensions_values[dimension_name]:
            return value
        elif dimension_type == "YEAR" and re.match(r"^\d{4}$", value):
            # a YEAR value of format "yyyy" should not be interpreted as an epoch even though it can be parsed as an int
            return value
        elif value.isdigit() or (value[0] in ("-", "+") and value[1:].isdigit()):
            epoch = int(value)
            return _get_time_dimension_value_from_timestamp(epoch, dimension_type)
        elif re.match(
            r"^\d{4}-\d\d-\d\dT\d\d:\d\d:\d\d(\.\d+)?(([+-]\d\d:\d\d)|Z)?$",
            value,
        ):  # "yyyy-MM-dd'T'HH:mm:ss.SSSXXX"
            return _get_time_dimension_value_from_timestamp(
                (
                    datetime.strptime(value, "%Y-%m-%dT%H:%M:%S.%fZ") - datetime(1970, 1, 1)
                ).total_seconds(), dimension_type, milliseconds=False
            )
        else:
            return value

    elif isinstance(value, (float, int)):
        return _get_time_dimension_value_from_timestamp(value, dimension_type)
    
    elif isinstance(value, datetime):  # works also with pd.Timestamp
        return _get_time_dimension_value_from_datetime(value, dimension_type)

    else:
        raise ValueError("Unknown dimension value type for {} / {} : {}".format(dimension_name, dimension_type, type(value)))
