(function() {
    'use strict';

    const app = angular.module('dataiku.analysis.mlcore');

    app.constant("SINGLE_TIMESERIES_IDENTIFIER", "__single_timeseries_identifier");

    app.controller("TimeseriesPMLTaskBaseController", function($scope, $controller, DataikuAPI, TimeseriesForecastingUtils, VisualMlCodeEnvCompatibility,
                                                               AlgorithmsSettingsService, FeatureFlagsService, CachedAPICalls) {
        $controller("_PMLTrainSessionController", { $scope: $scope });
        $scope.legacyAlgorithms = ["gluonts_mqcnn_timeseries", "gluonts_transformer_timeseries", "gluonts_deepar_timeseries", "gluonts_simple_feed_forward_timeseries"];

        $scope.deferredAfterInitMlTaskDesign.then(() => CachedAPICalls.pmlGuessPolicies)
            .then(pmlGuessPolicies => {
                $scope.guessPolicies = $scope.prepareGuessPolicies(pmlGuessPolicies["timeseries-forecasting"]);
            }).then(() => {
            $scope.setAlgorithms($scope.mlTaskDesign);
            $scope.setSelectedAlgorithm(AlgorithmsSettingsService.getDefaultAlgorithm(
                $scope.mlTaskDesign,
                $scope.base_algorithms[$scope.mlTaskDesign.backendType]
            ));
        })
            .catch(setErrorInScope.bind($scope));

        $scope.hasExternalFeatures = per_feature => !!$scope.mlTaskFeatures(per_feature, ['INPUT', 'INPUT_PAST_ONLY']).length;

        $scope.hasKnownInAdvanceFeatures = per_feature => !!$scope.mlTaskFeatures(per_feature, ['INPUT']).length;

        function getModelingForAlgKey(algKey) {
            // Fun stuff : while the algo key is xgboost_regression, the modeling key is xgboost.
            const safe_alg_key = algKey.startsWith("xgboost") ? "xgboost" : algKey
            return $scope.mlTaskDesign?.modeling[safe_alg_key]
        }

        $scope.listAlgosWithoutExternalFeatures = function(algos) {
            if (!algos || !$scope.mlTaskDesign || !$scope.hasExternalFeatures($scope.mlTaskDesign.preprocessing.per_feature)) return [];

            return algos.filter(algo => getModelingForAlgKey(algo.algKey)?.enabled
                && TimeseriesForecastingUtils.ALGOS_WITHOUT_EXTERNAL_FEATURES.keys.includes(algo.algKey));
        };

        $scope.listAlgosSlowOnMultipleTimeseries = function(algos) {
            if (!$scope.mlTaskDesign || !$scope.mlTaskDesign.timeseriesIdentifiers.length) return [];

            return algos.filter(algo => getModelingForAlgKey(algo.algKey)?.enabled
                && TimeseriesForecastingUtils.ALGOS_SLOW_ON_MULTIPLE_TIMESERIES.includes(algo.algKey));
        };

        $scope.listAlgosIncompatibleWithMs = function(algos) {
            if ($scope.mlTaskDesign?.timestepParams?.timeunit !== "MILLISECOND") return [];
            return algos.filter(algo => getModelingForAlgKey(algo.algKey)?.enabled
                && TimeseriesForecastingUtils.ALGOS_INCOMPATIBLE_WITH_MS.includes(algo.algKey));
        };

        $scope.listAlgosCompatibleWithShiftsWindows = function() {
            if (!$scope.mlTaskDesign || !$scope.mlTaskDesign.modeling || !$scope.algorithms) return [];
            const algos = $scope.algorithms["PY_MEMORY"];
            // templateName because XGBoost (and LightGBM) algKey is not the same as key in modeling...
            const compatList = algos.filter(algo => ((algo.templateName && $scope.mlTaskDesign?.modeling[algo.templateName]?.enabled) || $scope.mlTaskDesign?.modeling[algo.algKey]?.enabled))
                                    .filter(algo => (algo.templateName && TimeseriesForecastingUtils.ALGOS_COMPATIBLE_WITH_SHIFTS_AND_WINDOWS.includes(algo.templateName)) || TimeseriesForecastingUtils.ALGOS_COMPATIBLE_WITH_SHIFTS_AND_WINDOWS.includes(algo.algKey))
                                    .map(x => x.name);
            // Remove duplicates (XGBoost)
            return [...new Set(compatList)];
        };

        $scope.validateHorizonShift = function (shiftFromHorizonValues, fieldName, isPastOnlyFeature) {
            if (shiftFromHorizonValues.some(x => x === null || x === undefined || isNaN(x))) {
                return fieldName + " must be integer.";
            } else if (shiftFromHorizonValues.some(x => !Number.isInteger(x))) {
                return fieldName + " must be integer, float values will be truncated.";
            } else if (isPastOnlyFeature && shiftFromHorizonValues.some(x => x > -$scope.mlTaskDesign.predictionLength)) {
                return fieldName + " must be smaller or equal to -" + $scope.mlTaskDesign.predictionLength + " (forecast horizon timesteps) for target and past-only external features.";
            } else if (shiftFromHorizonValues.some(x => x > 0)) {
                return fieldName + " must not be positive.";
            }
            return "";
        };

        $scope.validateForecastShift = function (shiftFromForecastValues, fieldName) {
            if (shiftFromForecastValues.some(x => x === null || x === undefined || isNaN(x))) {
                return fieldName + " must be integer.";
            } else if (shiftFromForecastValues.some(x => !Number.isInteger(x))) {
                return fieldName + " must be integer, float values will be truncated.";
            } else if (shiftFromForecastValues.some(x => x > 0)) {
                return fieldName + " must not be in the future.";
            }
            return "";
        };

        $scope.validateShiftsAsString = function (shiftFromForecastAsString, shiftFromHorizonAsString, isPastOnlyFeature) {
            const shiftFromForecastValues = shiftFromForecastAsString !== "" ? Array.from(shiftFromForecastAsString.split(",")).map(x => Number(x)) : [];
            const shiftFromHorizonValues = shiftFromHorizonAsString !== "" ? Array.from(shiftFromHorizonAsString.split(",")).map(x => Number(x)) : [];

            let forecastShiftErrorMessage = "";
            if (shiftFromForecastAsString.includes(" ") || shiftFromForecastValues.some(x => isNaN(x))) {
                forecastShiftErrorMessage = "Must contain only comma separated list of integers.";
            } else {
                forecastShiftErrorMessage = $scope.validateForecastShift(shiftFromForecastValues, "Forecast origin shift values");
            }

            let horizonShiftErrorMessage = "";
            if (shiftFromHorizonAsString.includes(" ") || shiftFromHorizonValues.some(x => isNaN(x))) {
                horizonShiftErrorMessage = "Must contain only comma separated list of integers.";
            } else {
                horizonShiftErrorMessage = $scope.validateHorizonShift(shiftFromHorizonValues, "Forecasted point shift values", isPastOnlyFeature);
            }

            return {
                from_forecast: forecastShiftErrorMessage,
                from_horizon: horizonShiftErrorMessage,
            };
        };

        $scope.validatePastOnlyAutoHorizonShiftRange = function (minHorizonShiftPastOnly, maxHorizonShiftPastOnly) {
            let validationError = "";
            if (minHorizonShiftPastOnly >= maxHorizonShiftPastOnly) {
                validationError = "Range length must be positive.";
            }
            if (validationError === "") {
                validationError = $scope.validateHorizonShift([minHorizonShiftPastOnly], "Past only features minimum shift", true);
            }
            if (validationError === "") {
                validationError = $scope.validateHorizonShift([maxHorizonShiftPastOnly], "Past only features maximum shift", true);
            }
            if (validationError === "") {
                const rangeLength = maxHorizonShiftPastOnly - minHorizonShiftPastOnly
                if (rangeLength > 1000) {
                    validationError = "Range length is very large, this can significantly increase preprocessing time.";
                }
            }
            return validationError;
        };

        $scope.validateKnownInAdvanceAutoHorizonShiftRange = function (minHorizonShiftKnownInAdvance, maxHorizonShiftKnownInAdvance) {
            let validationError = "";
            if (minHorizonShiftKnownInAdvance >= maxHorizonShiftKnownInAdvance) {
                validationError = "Range length must be positive.";
            }
            if (validationError === "") {
                validationError = $scope.validateHorizonShift([minHorizonShiftKnownInAdvance], "Known in advance features minimum shift", false);
            }
            if (validationError === "") {
                validationError = $scope.validateHorizonShift([maxHorizonShiftKnownInAdvance], "Known in advance features maximum shift", false);
            }
            if (validationError === "") {
                const rangeLength = maxHorizonShiftKnownInAdvance - minHorizonShiftKnownInAdvance
                if (rangeLength > 1000) {
                    validationError = "Range length is very large, this can significantly increase preprocessing time.";
                }
            }
            return validationError;
        };

        $scope.$watch('mlTaskDesign.preprocessing.feature_generation.shifts', function(nv) {
            $scope.validateShifts();
        }, true);

        $scope.validateShifts = function () {

            $scope.shiftsValidationErrors = {}

            if (!$scope.uiState || !$scope.uiState.laggableFeatures) {
                return;
            }

            for (const feature of $scope.uiState.laggableFeatures) {
                let featureName = feature._name;
                let shiftFromForecastAsString = ($scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_forecast || "").toString();
                let shiftFromHorizonAsString = ($scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_horizon || "").toString();
                let isPastOnlyFeature = $scope.isPastOnlyFeature(featureName);

                $scope.shiftsValidationErrors[featureName] = $scope.validateShiftsAsString(shiftFromForecastAsString, shiftFromHorizonAsString, isPastOnlyFeature);
            }

            $scope.updateSelectedAlgosMissingRequiredExternalFeature();
        };

        $scope.windowHasAnyEnabledOperation = function (window) {
            return window.operations_list.some(
                operations => operations[1].some(
                    operation => operation.enabled
                )
            );
        };

        $scope.isPastOnlyFeature = function (featureName) {
            return ["INPUT_PAST_ONLY", "TARGET"].includes($scope.mlTaskDesign.preprocessing.per_feature[featureName].role);
        };

        $scope.updateWindowLength = function (windowIndex, startShift, shift) {
            $scope.uiState.windows[windowIndex].length = -(startShift - shift);
        };

        $scope.$watch('uiState.windows', function (nv) {
            $scope.validateWindows();
        }, true);

        $scope.validateWindows = function () {

            $scope.windowsValidationErrors = []
            if (!$scope.uiState || !$scope.uiState.windows) {
                return;
            }
            for (let windowIndex = 0; windowIndex < $scope.uiState.windows.length; windowIndex++) {
                let window = $scope.uiState.windows[windowIndex];

                let shiftMessage = "";
                if (window.length <= 0) {
                    shiftMessage = "The window length must be positive.";
                } else if (window.length < 2) {
                    shiftMessage = "Window length must be at least 2 timesteps"
                } else if (window.is_from_forecast) {
                    shiftMessage = $scope.validateForecastShift([window.shift], "Window end");
                } else {
                    let windowHasPastCovariateFeatureEnabled = Object.keys(window.operations_map)
                        .filter(featureName => window.operations_map[featureName].some(x => x.enabled))
                        .some(featureName => $scope.isPastOnlyFeature(featureName));
                    shiftMessage = $scope.validateHorizonShift([window.shift], "Window end", windowHasPastCovariateFeatureEnabled);
                }

                let operationsMessage = "";
                if (!$scope.windowHasAnyEnabledOperation($scope.uiState.windows[windowIndex])) {
                    operationsMessage = "The window doesn't have any aggregation method.";
                }

                $scope.windowsValidationErrors.push({
                    shift: shiftMessage,
                    operations: operationsMessage,
                })
            }

            $scope.updateSelectedAlgosMissingRequiredExternalFeature();
        };

        $scope.updateSelectedAlgosMissingRequiredExternalFeature = function () {
            $scope.selectedAlgosMissingRequiredExternalFeature = [];
            const algosRequiringExternalFeature = $scope.listAlgosCompatibleWithShiftsWindows();
            if (algosRequiringExternalFeature.length > 0) {
                let hasGeneratedFeature = $scope.mlTaskDesign.preprocessing.feature_generation.windows.length > 0;
                for (const feature of $scope.uiState.laggableFeatures) {
                    let featureName = feature._name;
                    const hasForecastShift = $scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_forecast.length > 0;
                    const hasHorizonShift = $scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_horizon_mode == 'FIXED' && $scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_horizon.length > 0;
                    const isAutoHorizonShift = $scope.mlTaskDesign.preprocessing.feature_generation.shifts[featureName].from_horizon_mode == 'AUTO';
                    if (hasForecastShift || hasHorizonShift || isAutoHorizonShift) {
                        hasGeneratedFeature = true;
                        break;
                    }
                }
                if (!hasGeneratedFeature) {
                    $scope.selectedAlgosMissingRequiredExternalFeature = algosRequiringExternalFeature;
                }
            }
        }

        $scope.displayProphetCodeEnvWarning = function(algorithm) {
            if (algorithm.algKey !== "prophet_timeseries" || !getModelingForAlgKey(algorithm.algKey)?.enabled) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvProphetCompatible = envCompat && envCompat.prophet && envCompat.prophet.compatible;
            return !isEnvProphetCompatible;
        };

        $scope.displayGluontsCodeEnvWarning = function(algorithm) {
            const gluonts_algs = ["trivial_identity_timeseries", "seasonal_naive_timeseries", "gluonts_npts_timeseries", "gluonts_simple_feed_forward_timeseries", "gluonts_deepar_timeseries", "gluonts_transformer_timeseries", "gluonts_mqcnn_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, gluonts_algs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvGluontsTorchCompatible = envCompat && envCompat.gluonts && envCompat.gluonts.compatible;
            return !isEnvGluontsTorchCompatible;
        };

        $scope.displayPmdarimaCodeEnvWarning = function(algorithm) {
            if (algorithm.algKey !== "autoarima_timeseries" || !getModelingForAlgKey(algorithm.algKey)?.enabled) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvArimaCompatible = envCompat && envCompat.pmdarima && envCompat.pmdarima.compatible;
            return !isEnvArimaCompatible;
        };

        $scope.displayStatsmodelCodeEnvWarning = function(algorithm) {

            const statsmodelsAlgs = ["arima_timeseries", "ets_timeseries", "seasonal_loess_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, statsmodelsAlgs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvStatsmodelCompatible = envCompat && envCompat.statsmodel && envCompat.statsmodel.compatible;
            return !isEnvStatsmodelCompatible;
        };

        $scope.displayStatsforecastCodeEnvWarning = function(algorithm) {

            const statsforecastAlgs = ["croston_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, statsforecastAlgs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvStatsforecastCompatible = envCompat && envCompat.statsforecast && envCompat.statsforecast.compatible;
            return !isEnvStatsforecastCompatible;
        };

        $scope.displayClassicalMLCodeEnvWarning = function(algorithm) {

            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, TimeseriesForecastingUtils.ALGOS_TIMESERIES_CLASSICAL_ML)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isClassicalMlCompatible = envCompat && envCompat.classicalMLTs && envCompat.classicalMLTs.compatible;
            return !isClassicalMlCompatible;
        };

        $scope.displayMXNetAlgLegacyWarning = function(algorithm) {
            return $scope.isAlgoEnabledAndInList(algorithm.algKey, $scope.legacyAlgorithms);
        };

        $scope.isAlgoEnabledAndInList = function(algKey, algList) {
            return algList.includes(algKey) && getModelingForAlgKey(algKey)?.enabled;
        }

        $scope.displayMXNetAlgCodeEnvWarning = function(algorithm) {
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, $scope.legacyAlgorithms)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvMXNetCompatible = envCompat && envCompat.mxnetTimeseries && envCompat.mxnetTimeseries.compatible;
            return !isEnvMXNetCompatible;
        };

        $scope.displayGluontsTorchAlgCodeEnvWarning = function(algorithm) {
            const torch_algs = ["gluonts_torch_deepar_timeseries", "gluonts_torch_simple_feed_forward_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, torch_algs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvGluontsTorchCompatible = envCompat && envCompat.torchTimeseries && envCompat.torchTimeseries.compatible;
            return !isEnvGluontsTorchCompatible;
        };
    });

    app.controller("TimeseriesPMLTaskDesignController", function($scope, $controller, CreateModalFromTemplate, $timeout, TimeseriesForecastingUtils, TimeseriesForecastingCustomTrainTestFoldsUtils, $filter) {
        $controller("_TabularPMLTaskDesignController", { $scope: $scope });

        $scope.prettyTimeUnit = TimeseriesForecastingUtils.prettyTimeUnit;
        $scope.prettyTimeSteps = TimeseriesForecastingUtils.prettyTimeSteps;
        $scope.plurifiedTimeUnits = TimeseriesForecastingUtils.plurifiedTimeUnits;
        $scope.timeseriesImputeMethods = function(featureType, interpolation, extrapolation) {
            return TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.filter(imputeMethod =>
                (featureType !== undefined ? imputeMethod.featureTypes.includes(featureType) : true) &&
                (interpolation !== undefined ? imputeMethod.interpolation === interpolation : true) &&
                (extrapolation !== undefined ? imputeMethod.extrapolation === extrapolation : true)
            )
        }
        $scope.timeseriesImputeMethodsDescriptions = function(featureType, interpolation, extrapolation) {
            return $scope.timeseriesImputeMethods(featureType, interpolation, extrapolation).map(imputeMethod => imputeMethod.description);
        }
        $scope.duplicateTimestampsHandlingMethods = TimeseriesForecastingUtils.DUPLICATE_TIMESTAMPS_HANDLING_METHODS;

        $scope.getWeekDayName = TimeseriesForecastingUtils.getWeekDayName;
        $scope.getDayNumber = TimeseriesForecastingUtils.getDayNumber;
        $scope.daysInMonth = Array.from({ length: 31 }, (_, index) => index + 1);
        $scope.getMonthName = TimeseriesForecastingUtils.getMonthName;
        $scope.getQuarterName = TimeseriesForecastingUtils.getQuarterName;
        $scope.getHalfYearName = TimeseriesForecastingUtils.getHalfYearName;

        $scope.togglePartitioningSection = function() {
            $scope.partitioningSectionOpened = !$scope.partitioningSectionOpened;
        }

        $scope.addWindow = function() {
            var newWindowParams = {
                length: Math.max($scope.mlTaskDesign.predictionLength, 2),
                shift: 0,
                is_from_forecast: true,
                operations_map: {}};
            for (let feature of $scope.uiState.windowableFeatures || []) {
                if (feature.type === "NUMERIC") {
                    newWindowParams.operations_map[feature._name] = [
                        {operation: "MEAN", enabled: true},
                        {operation: "MEDIAN", enabled: false},
                        {operation: "STD", enabled: true},
                        {operation: "MIN", enabled: false},
                        {operation: "MAX", enabled: false}
                    ];
                } else if (feature.type === "CATEGORY") {
                    newWindowParams.operations_map[feature._name] = [
                        {operation: "FREQUENCY", enabled: true},
                    ];
                }

            }
            $scope.mlTaskDesign.preprocessing.feature_generation.windows.push(newWindowParams);
        }

        $scope.deleteWindow = function(windowIndex) {
            $scope.mlTaskDesign.preprocessing.feature_generation.windows.splice(windowIndex, 1);
        }

        $scope.getValidOperations = function(window, windowableFeature) {
            const unfilteredOperations = window.operations_map[windowableFeature._name];
            if (windowableFeature.type === "NUMERIC") {
                return unfilteredOperations.filter(op => ["MIN", "MAX", "MEAN", "MEDIAN", "SUM", "STD"].includes(op.operation));
            } else if (windowableFeature.type === "CATEGORY") {
                return unfilteredOperations.filter(op => op.operation === "FREQUENCY");
            }
        }

        $scope.$watch('mlTaskDesign', function(nv) {
            if (!nv) return;
            $scope.uiState.timeVariable = nv.timeVariable;
            $scope.uiState.timeseriesIdentifiers = [].concat(nv.timeseriesIdentifiers);
            $scope.uiState.multipleTimeSeries = nv.timeseriesIdentifiers[0] !== undefined;
            $scope.uiState.predictionLength = nv.predictionLength;
            $scope.uiState.timestepParams = Object.assign({}, nv.timestepParams);
            $scope.setDefaultMonthlyAlignment(nv.timestepParams, false);
            $scope.setDefaultUnitAlignment(nv.timestepParams, false);

            $scope.partitioningSectionOpened = $scope.mlTaskDesign.partitionedModel && $scope.mlTaskDesign.partitionedModel.enabled;

            $scope.uiState.numberOfHorizonsInTest = Math.floor($scope.mlTaskDesign.evaluationParams.testSize / $scope.mlTaskDesign.predictionLength);

            $scope.uiState.hyperparamSearchStrategies = [["GRID", "Grid search"],
                ["RANDOM", "Random search"],
                ["BAYESIAN", "Bayesian search"]];

            $scope.uiState.crossValidationModes = [["TIME_SERIES_SINGLE_SPLIT", "Simple split"],
                ["TIME_SERIES_KFOLD", "K-fold cross-validation"]];

            $scope.uiState.preprocessingPerFeature = $scope.mlTaskFeatures(nv.preprocessing.per_feature, ['INPUT', 'INPUT_PAST_ONLY', 'REJECT']);
            $scope.featuresWithDateMeaning = Object.fromEntries(
                Object.entries($scope.mlTaskDesign.preprocessing.per_feature).filter(function(entry) {
                    const feature = entry[1];
                    return ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(feature.state.recordedMeaning) >= 0;
                })
            );
        });

        $scope.$watch('mlTaskDesign.preprocessing.per_feature', function(nv) {
            // make a copy to prevent selection conflict between the feature generation and external features tabs
            let features = $scope.mlTaskFeatures(nv, ['INPUT', 'INPUT_PAST_ONLY', 'TARGET']).map(x => {
                let copy = angular.copy(x);

                // remove the selection coming from external features update
                copy.$selected = false;
                return copy;
            });

            // put the target at the beginning of the list
            let targetIndex = features.findIndex(f => f.role === 'TARGET');
            if (targetIndex > -1) {
                let target = features.splice(targetIndex, 1)[0];
                features.unshift(target);
            }
            updateLaggableFeatures(features);
            updateWindowableFeatures(features);
        }, true);

        $scope.$watchGroup(['algorithms', 'mlTaskDesign.preprocessing.feature_generation.windows.length', 'uiState.windowableFeatures', 'uiState.laggableFeatures'], function(nv, ov) {
            $scope.updateSelectedAlgosMissingRequiredExternalFeature();
        }, true);

        $scope.featureSupportsAutoShifts = function(feature) {
            return feature?.type === 'NUMERIC';
        }

        $scope.setDefaultShifts = function (feature, keepExisting=true) {
            const shifts = $scope.mlTaskDesign.preprocessing.feature_generation.shifts;
            let from_forecast;
            let from_horizon;
            if (feature.role === "INPUT") {
                from_forecast = [];
                from_horizon = [-2, -1, 0]
            } else {
                from_forecast = [-2, -1, 0];
                from_horizon = [-$scope.mlTaskDesign.predictionLength - 2, -$scope.mlTaskDesign.predictionLength - 1, -$scope.mlTaskDesign.predictionLength]
            }
            if (!keepExisting || !(feature._name in shifts)) {
                shifts[feature._name] = {
                    from_forecast: from_forecast,
                    from_horizon: from_horizon,
                    from_horizon_mode: $scope.featureSupportsAutoShifts(feature) ? 'AUTO' : 'FIXED',
                };
            } else {
                if (!$scope.featureSupportsAutoShifts(feature)) {
                    shifts[feature._name].from_horizon_mode = 'FIXED';
                }
            }
        };

        $scope.setDefaultShiftsMultipleFeatures = function (features, keepExisting=false) {
            for (let feature of features) {
                $scope.setDefaultShifts(feature, keepExisting);
            }
        };

        function updateLaggableFeatures(features) {
            const previousNames = ($scope.uiState.laggableFeatures || []).map(x => x._name);
            const newlyAddedFeatures = features.filter(feature => !previousNames.includes(feature._name));

            newlyAddedFeatures.forEach(feature => $scope.setDefaultShifts(feature));

            $scope.uiState.laggableFeatures = features;
        }

        function updateWindowableFeatures(features) {
            const windowableFeatures = features.filter(feature => TimeseriesForecastingUtils.isWindowCompatible(feature))
            const previousNamesAndTypes = ($scope.uiState.windowableFeatures || []).map(x => [x._name, x.type]);
            const newNamesAndTypes = windowableFeatures.map(x => [x._name, x.type]);
            const newlyAddedFeatures = newNamesAndTypes.filter(nameAndType => !previousNamesAndTypes.includes(nameAndType));

            newlyAddedFeatures.forEach(nameAndType => {
                const name = nameAndType[0];
                const type = nameAndType[1];
                $scope.mlTaskDesign.preprocessing.feature_generation.windows.forEach(window => {
                    if (!(name in window.operations_map)) {
                        // If feature is missing from already defined window, add it without any aggregation
                        // otherwise keep previously defined version
                        if (type==="NUMERIC") {
                            window.operations_map[name] = [
                                {operation: "MEAN", enabled: false},
                                {operation: "MEDIAN", enabled: false},
                                {operation: "STD", enabled: false},
                                {operation: "MIN", enabled: false},
                                {operation: "MAX", enabled: false}
                            ];
                        } else if (type === "CATEGORY") {
                            window.operations_map[name] = [
                                {operation: "FREQUENCY", enabled: false}
                            ];
                        }
                    } else {
                        // Type change of feature already occurring in the window
                        // Add potentially missing operations
                        if (type === "NUMERIC") {
                            let operations = window.operations_map[name].map(x => x.operation);
                            ["MEAN", "MEDIAN", "STD", "MIN", "MAX"].filter(op => !operations.includes(op))
                                                                   .forEach(op => window.operations_map[name].push({operation: op, enabled: false}))
                        } else if (type === "CATEGORY") {
                            if(!window.operations_map[name].some(x => x.operation === "FREQUENCY")) {
                                window.operations_map[name] = [{operation: "FREQUENCY", enabled: false}];
                            }
                        }
                    }
                })
            });

            $scope.uiState.windowableFeatures = windowableFeatures;
        }

        $scope.$watchGroup(['mlTaskDesign.preprocessing.feature_generation.windows','mlTaskDesign.preprocessing.feature_generation.windows.length', 'uiState.windowableFeatures'], function(nv, ov) {
            $scope.uiState.windows = $scope.mlTaskDesign.preprocessing.feature_generation.windows.map((x, index) => {
                // make a copy of the original windows variable
                // this enables to use a custom list compatible with mass selection without falling in an unsaved state
                let copy = angular.copy(x);
                copy.id = index;

                // only keep the operations for the windowable features and keep the windowable features themselves to ease the display
                copy.operations_list = nv[2]
                    .filter(x => copy.operations_map.hasOwnProperty(x._name))
                    .map(x => [x._name, copy.operations_map[x._name], x]);

                return copy;
            })
        }, true);

        $scope.$watch('uiState.windows', function(nv, ov) {
            // sync the ui modified values to the actual parameters
            if (!nv || !ov) return;
            nv.forEach(((window, index) => {
                $scope.mlTaskDesign.preprocessing.feature_generation.windows[index].is_from_forecast = window.is_from_forecast;
                $scope.mlTaskDesign.preprocessing.feature_generation.windows[index].shift = window.shift;
                $scope.mlTaskDesign.preprocessing.feature_generation.windows[index].length = window.length;

                window.operations_list.forEach(([key, value]) => {
                    $scope.uiState.windows[index].operations_map[key] = value;
                    $scope.mlTaskDesign.preprocessing.feature_generation.windows[index].operations_map[key] = value;
                }
            );
            }));
        }, true);

        $scope.onChangeTimeVariable = function() {
            if (!$scope.uiState.timeVariable) return;
            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.paramKey = "timeVariable";
            });
        };

        $scope.changeTimeSeriesType = function(multiple) {
            $scope.uiState.multipleTimeSeries = multiple;
            if (multiple || !$scope.mlTaskDesign.timeseriesIdentifiers.length) return;

            // If 'Single time series' is selected, remove time series identifiers
            $scope.uiState.timeseriesIdentifiers = [];
            onChangeTimeseriesIdentifiers();
        };

        function haveTimeseriesIdentifiersChanged() {
            return $scope.mlTaskDesign.timeseriesIdentifiers.length !== $scope.uiState.timeseriesIdentifiers.length
                || $scope.mlTaskDesign.timeseriesIdentifiers.some(function(elem, idx) {
                    return elem !== $scope.uiState.timeseriesIdentifiers[idx];
                });
        }

        function onChangeTimeseriesIdentifiers() {
            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.onCloseCallback = function() {
                    $scope.uiState.preprocessingPerFeature = $scope.mlTaskFeatures($scope.mlTaskDesign.preprocessing.per_feature, ['INPUT', 'INPUT_PAST_ONLY', 'REJECT']);
                    $scope.uiState.multipleTimeSeries = !!$scope.uiState.timeseriesIdentifiers.length;

                };
                newScope.getUIStateParam = param => [].concat(param);

                newScope.paramKey = "timeseriesIdentifiers";
            });
        };

        $scope.newQuantileIsValid = function(newQuantile) {
            const inRange = newQuantile > 0 && newQuantile < 1;
            if (!inRange) return false;

            return parseFloat(newQuantile.toFixed(4)) === newQuantile;
        };

        $scope.$on("$stateChangeStart", function(e) {
            // Prevent to change state if the identifiers have been modified, so that
            // the user first interact with the keep settings/redetect modal
            if (haveTimeseriesIdentifiersChanged() || hasNumberOfTimeunitsChanged() || hasPredictionLengthChanged()) {
                e.preventDefault();
            }
        });

        $scope.onChangeTimeseriesIdentifiers = function() {
            // the timeout prevents the modal to be closed if the dropdown was closed via 'esc'
            $timeout(function() {
                if (haveTimeseriesIdentifiersChanged()) {
                    onChangeTimeseriesIdentifiers();
                }
            }, 100);
        };

        function onChangeTimestepParams() {
            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.getUIStateParam = param => Object.assign({}, param);
                newScope.paramKey = "timestepParams";
            });
        }

        function hasPredictionLengthChanged() {
            if (!$scope.uiState.predictionLength) return;
            return $scope.mlTaskDesign.predictionLength !== $scope.uiState.predictionLength;
        }

        $scope.onChangePredictionLength = function() {
            if (!hasPredictionLengthChanged()) return;

            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }

            const numberOfHorizonsInTest = $scope.uiState.numberOfHorizonsInTest;
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.onCloseCallback = function() {
                    $scope.uiState.numberOfHorizonsInTest = numberOfHorizonsInTest;
                    $scope.updateTimeseriesTestSize();
                };
                newScope.paramKey = "predictionLength";
            });
        };

        $scope.setDefaultMonthlyAlignment = function(params, force) {
            if (!params.monthlyAlignment || force) {
                params.monthlyAlignment = 31;
            }
        }

        $scope.setDefaultUnitAlignment = function(params, force) {
            if (!params.unitAlignment || force) {
                if (params.timeunit === "QUARTER") {
                    params.unitAlignment = 3;
                } else if (params.timeunit === "HALF_YEAR") {
                    params.unitAlignment = 6;
                } else if (params.timeunit === "YEAR") {
                    params.unitAlignment = 12;
                }
            }
        }

        $scope.onChangeTimestepUnit = function() {
            $scope.setDefaultMonthlyAlignment($scope.uiState.timestepParams, true);
            $scope.setDefaultUnitAlignment($scope.uiState.timestepParams, true);
            onChangeTimestepParams();
        }

        $scope.onChangeNumberOfTimeunits = function() {
            if (!hasNumberOfTimeunitsChanged()) return;
            onChangeTimestepParams();
        };

        function hasNumberOfTimeunitsChanged() {
            if (!$scope.uiState.timestepParams.numberOfTimeunits) return;
            return $scope.uiState.timestepParams.numberOfTimeunits !== $scope.mlTaskDesign.timestepParams.numberOfTimeunits;
        };

        $scope.hasRole = function(feature, roles) {
            const role = $scope.mlTaskDesign.preprocessing.per_feature[feature].role;
            return roles.includes(role);
        };

        const randomUUID = function() {
            return window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16);
        }

        $scope.updateTimeseriesTestSize = function() {
            $scope.mlTaskDesign.evaluationParams.testSize = $scope.uiState.predictionLength * $scope.uiState.numberOfHorizonsInTest;
        };

        $scope.formatDateWithoutTimezone = function(uiDate) {
            return moment(uiDate).utc().format("YYYY-MM-DD HH:mm:ss.SSS")
        };

        $scope.addCustomTestSplitInterval = function() {
            const defaultInterval = $scope.mlTaskDesign.customTrainTestIntervals[$scope.mlTaskDesign.customTrainTestIntervals.length - 1];
            let newInterval = {
                _id: randomUUID(),
                train: [new Date(defaultInterval['train'][0]), new Date(defaultInterval['train'][1])],
                test: [new Date(defaultInterval['test'][0]), new Date(defaultInterval['test'][1])]
            }

            $scope.uiState.fixedFoldIntervals = $scope.uiState.fixedFoldIntervals.concat([newInterval])
            $scope.propagateFixedIntervalChangeFn()
        };


        $scope.removeCustomTestSplitInterval = function(intervalIndex) {
            $scope.uiState.fixedFoldIntervals = $scope.uiState.fixedFoldIntervals.filter((_, index) => index != intervalIndex);
            $scope.propagateFixedIntervalChangeFn()
        };

        $scope.initTimeseriesFixedFoldUIState = function() {
            if (!$scope.mlTaskDesign.customTrainTestIntervals.length) {
                $scope.mlTaskDesign.customTrainTestIntervals.push({
                    train: [$scope.formatDateWithoutTimezone(new Date()), $scope.formatDateWithoutTimezone(new Date())],
                    test: [$scope.formatDateWithoutTimezone(new Date()), $scope.formatDateWithoutTimezone(new Date())]
                });
            }
            $scope.uiState.fixedFoldIntervals = $scope.mlTaskDesign.customTrainTestIntervals.map((interval, index) => ({
                    _id: randomUUID(), // ID used in UI to keep track of different intervals and refresh them
                    train: [TimeseriesForecastingCustomTrainTestFoldsUtils.forceConvertToUTCTimezoneDate(interval['train'][0]), TimeseriesForecastingCustomTrainTestFoldsUtils.forceConvertToUTCTimezoneDate(interval['train'][1])],
                    test: [TimeseriesForecastingCustomTrainTestFoldsUtils.forceConvertToUTCTimezoneDate(interval['test'][0]), TimeseriesForecastingCustomTrainTestFoldsUtils.forceConvertToUTCTimezoneDate(interval['test'][1])],
                    arrowDescriptions: {
                        train: $scope.getIntervalArrowDescription(interval['train']),
                        test: $scope.getIntervalArrowDescription(interval['test'])
                    }
            }));
            $scope.validateFixedFolds();
        };

        $scope.propagateFixedIntervalChangeFn = function() {
            $timeout(function() {

                $scope.mlTaskDesign.customTrainTestIntervals = $scope.uiState.fixedFoldIntervals.map((uiInterval) => {
                    uiInterval["arrowDescriptions"] = {
                        train: $scope.getIntervalArrowDescription(uiInterval["train"]),
                        test: $scope.getIntervalArrowDescription(uiInterval["test"]),
                    };
                    return {
                        train: [$scope.formatDateWithoutTimezone(uiInterval['train'][0]), $scope.formatDateWithoutTimezone(uiInterval['train'][1])],
                        test: [$scope.formatDateWithoutTimezone(uiInterval['test'][0]), $scope.formatDateWithoutTimezone(uiInterval['test'][1])]
                    }
                });
                $scope.validateFixedFolds()
            });
        };

        $scope.validateFixedFolds = function() {
            $scope.customTrainTestValidationErrors = []
            for (let intervalIndex = 0 ; intervalIndex < $scope.mlTaskDesign.customTrainTestIntervals.length ; intervalIndex++) {
                const intervalValidationError = TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(
                    $scope.mlTaskDesign.timestepParams,
                    $scope.mlTaskDesign.customTrainTestIntervals[intervalIndex],
                    $scope.mlTaskDesign.predictionLength
                );
                if (intervalValidationError) {
                    $scope.customTrainTestValidationErrors.push("Fold {0} - {1}".format(intervalIndex + 1, intervalValidationError));
                }
            }
        };

        $scope.getIntervalArrowDescription = function(interval) {
            const timestepsCount = TimeseriesForecastingCustomTrainTestFoldsUtils.getIntervalTimestepsCount(interval, $scope.mlTaskDesign.timestepParams);
            return `${timestepsCount} ${$filter('plurify')($scope.mlTaskDesign.timestepParams.timeunit.replace("_", " ").toLowerCase(), timestepsCount)} (${Math.floor(timestepsCount / $scope.mlTaskDesign.timestepParams["numberOfTimeunits"])} timesteps)`;
        }

        $scope.resamplingMethodToGraph = function(method, prefix) {
            switch (method) {
                case "NEAREST":
                    return prefix + "-nearest";
                case "PREVIOUS":
                    return prefix + "-previous";
                case "NEXT":
                    return prefix + "-next";
                case "STAIRCASE":
                    return prefix + "-staircase";
                case "LINEAR":
                    if (prefix === "left") return "left-next";
                    return prefix + "-linear";
                case "QUADRATIC":
                    return prefix + "-quadratic";
                case "CUBIC":
                    return prefix + "-cubic";
                case "CONSTANT":
                    return prefix + "-constant";
                case "PREVIOUS_NEXT":
                    if (prefix === "left") return "left-next";
                    return "right-previous";
                case "NO_EXTRAPOLATION":
                    return prefix + "-dont";
            }
        };

        function isAutoArimaSearchExplicit() {
            if ($scope.mlTaskDesign.modeling.gridSearchParams.strategy == "GRID") {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.gridMode == 'EXPLICIT';
            } else {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.randomMode == 'EXPLICIT';
            }
        };

        $scope.isAutoArimaSeasonal = function() {
            if (isAutoArimaSearchExplicit()) {
                return Math.max(...$scope.mlTaskDesign.modeling.autoarima_timeseries.m.values) > 1;
            } else {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.range.max > 1;
            }
        };

        $scope.isAutoArimaSeasonLengthTooHigh = function() {
            if (isAutoArimaSearchExplicit()) {
                return Math.max(...$scope.mlTaskDesign.modeling.autoarima_timeseries.m.values) > 12;
            } else {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.range.max > 12;
            }
        };

        $scope.isTrendValid = function() {
            // this function should be kept in sync with ArimaMeta's isTrendValid method
            let sum = $scope.mlTaskDesign.modeling.arima_timeseries.d +
                $scope.mlTaskDesign.modeling.arima_timeseries.D;
            switch($scope.mlTaskDesign.modeling.arima_timeseries.trend) {
                case "n":
                    return true;
                case "c":
                case "ct":
                    return sum <= 0;
                case "t":
                    return sum <= 1;
                default:
                    return false;
            }
        };

        $scope.getAutoArimaMaxOrderMinValue = function() {
            let minValue = $scope.mlTaskDesign.modeling.autoarima_timeseries.start_p + $scope.mlTaskDesign.modeling.autoarima_timeseries.start_q + 1;
            if ($scope.isAutoArimaSeasonal()) {
                minValue += $scope.mlTaskDesign.modeling.autoarima_timeseries.start_P + $scope.mlTaskDesign.modeling.autoarima_timeseries.start_Q;
            }
            return minValue;
        }

        $scope.isOdd = number => number % 2 === 1;

        $scope.isPositive = number => number > 0;
    });

    app.directive("forecastExplanation", function(TimeseriesForecastingUtils) {
        return {
            scope: {
                predictionLength: '<',
                gapSize: '<',
                numberOfHorizonsInTest: '<',
                timeUnit: '<',
                numberOfTimeunits: '<',
                loadedStateField: '<'
            },
            restrict: 'A',
            templateUrl: '/templates/analysis/prediction/timeseries/forecasting-schema.html',
            link: function($scope, element) {
                const MAX_NB_OF_HORIZONS_IN_SCHEMA = 5;
                $scope.prettyTimeUnit = TimeseriesForecastingUtils.prettyTimeUnit;

                // Indicate to puppeteer that this content is ready for extraction
                const puppeteerSelectorName = $scope.loadedStateField;
                element.attr(puppeteerSelectorName, true)
                $scope[puppeteerSelectorName] = true;

                $scope.forecastingSchemaHasEllipsis = function() {
                    if (!$scope.predictionLength) return;
                    return $scope.numberOfHorizonsInTest > MAX_NB_OF_HORIZONS_IN_SCHEMA;
                };

                $scope.isEllipsedFromForecastingSchema = function(index) {
                    return $scope.forecastingSchemaHasEllipsis() && index == Math.floor(MAX_NB_OF_HORIZONS_IN_SCHEMA / 2);
                }

                $scope.maxNumberOfHorizonsInSchema = function() {
                    if (!$scope.numberOfHorizonsInTest) return;
                    return Math.min($scope.numberOfHorizonsInTest, MAX_NB_OF_HORIZONS_IN_SCHEMA);
                };

                $scope.getHorizonTickLabel = function(index) {
                    if ($scope.forecastingSchemaHasEllipsis() && index >= Math.floor(MAX_NB_OF_HORIZONS_IN_SCHEMA / 2)) {
                        index = $scope.numberOfHorizonsInTest - (MAX_NB_OF_HORIZONS_IN_SCHEMA - index);
                    }
                    ;
                    const nbHorizons = index + 1;
                    const nbTimeUnitsInHorizon = $scope.predictionLength * $scope.numberOfTimeunits;
                    return nbHorizons * nbTimeUnitsInHorizon;
                };
            }
        };
    })

    app.controller("TimeseriesPMLTaskPreTrainModal", function($scope, $controller, $stateParams, DataikuAPI, Logger, WT1) {
        $controller("PMLTaskPreTrainModal", { $scope });
        $controller("_TabularPMLTaskPreTrainBase", { $scope });

        function redactSensitiveInformation(eventContent) {
            // For now, there are no:
            // - custom metrics
            // - custom code algorithms (keras, or custom python or mllib)
            // - custom feature selection code
            // - feature generation manual interactions

            const redacted = dkuDeepCopy(eventContent, $scope.SettingsService.noDollarKey); // don't want to delete actual values in scope

            if (redacted.timeseriesSamplingParams) {
                delete redacted.timeseriesSamplingParams.numericalInterpolateConstantValue;
                delete redacted.timeseriesSamplingParams.numericalExtrapolateConstantValue;
                delete redacted.timeseriesSamplingParams.categoricalConstantValue;
                redacted.timeseriesSamplingParams = JSON.stringify(redacted.timeseriesSamplingParams)
            }

            return redacted;
        }

        $scope.algosWithoutExternalFeatures = $scope.listAlgosWithoutExternalFeatures($scope.base_algorithms[$scope.mlTaskDesign.backendType]);
        $scope.algosSlowOnMultipleTimeseries = $scope.listAlgosSlowOnMultipleTimeseries($scope.base_algorithms[$scope.mlTaskDesign.backendType]);

        // The settings can be updated from inside the modal (if you activate the GPU training),
        // so we save first before training
        $scope.train = () => $scope.saveSettings().then($scope._doTrainThenResolveModal);

        $scope._doTrain = function() {
            try {
                const algorithms = {};
                $.each($scope.mlTaskDesign.modeling, function(alg, params) {
                    if (params.enabled) {
                        algorithms[alg] = params;
                    }
                });

                let wt1Content = redactSensitiveInformation({
                    backendType: $scope.mlTaskDesign.backendType,
                    taskType: $scope.mlTaskDesign.taskType,
                    predictionType: $scope.mlTaskDesign.predictionType,
                    guessPolicy: $scope.mlTaskDesign.guessPolicy,
                    algorithms: JSON.stringify(algorithms),
                    predictionLength: $scope.mlTaskDesign.predictionLength,
                    gapSize: $scope.mlTaskDesign.evaluationParams.gapSize,
                    testSize: $scope.mlTaskDesign.evaluationParams.testSize,
                    nbTimeseriesIdentifiers: $scope.mlTaskDesign.timeseriesIdentifiers.length,
                    nbExternalFeatures: Object.values($scope.mlTaskDesign.preprocessing.per_feature).filter((feature) => feature.role === "INPUT").length,
                    quantiles: JSON.stringify($scope.mlTaskDesign.quantilesToForecast),
                    timestepParams: JSON.stringify($scope.mlTaskDesign.timestepParams),
                    timeseriesSamplingParams: $scope.mlTaskDesign.preprocessing.timeseriesSampling,
                    isPartitioned: $scope.mlTaskDesign.partitionedModel && $scope.mlTaskDesign.partitionedModel.enabled,
                    nFoldsEvaluation: $scope.mlTaskDesign.splitParams.kfold ? $scope.mlTaskDesign.splitParams.nFolds : null,
                    hasSessionName: !!$scope.uiState.userSessionName,
                    hasSessionDescription: !!$scope.uiState.userSessionDescription,
                    gridSearchParams: JSON.stringify($scope.mlTaskDesign.modeling.gridSearchParams),
                    evaluationMetric: $scope.mlTaskDesign.modeling.metrics.evaluationMetric,
                    runsOnKubernetes: $scope.hasSelectedK8sContainer(),
                    usesCustomTrainTestSplit: $scope.mlTaskDesign.customTrainTestSplit,
                    customTrainTestIntervals: $scope.mlTaskDesign.customTrainTestSplit ? JSON.stringify($scope.mlTaskDesign.customTrainTestIntervals) : null,
                });

                WT1.event("prediction-train", wt1Content);
            } catch (e) {
                Logger.error('Failed to report mltask info', e);
            }
            return DataikuAPI.analysis.pml.trainStart($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId,
                $scope.uiState.userSessionName, $scope.uiState.userSessionDescription, $scope.uiState.forceRefresh, true).error(setErrorInScope.bind($scope));
        };

        $scope.displayMessages = function () {
            return $scope.preTrainStatus.messages.length
                || $scope.algosWithoutExternalFeatures.length
                || $scope.algosSlowOnMultipleTimeseries.length
                || $scope.selectedAlgosMissingRequiredExternalFeature?.length;
        };
    });

    app.controller("TimeseriesPMLTaskResultController", function($scope, $controller, PMLSettings, PMLFilteringService) {
        $scope.algorithmCategories = PMLSettings.algorithmCategories("TIMESERIES_FORECAST");
        $scope.metricMap = PMLFilteringService.metricMap;
        $controller("_TabularPMLTaskResultController", { $scope });

        $scope.uiState.tsSessionDetailView = 'METRICS';

        let metrics;
        let highestMetricValues;
        let default_displayed_metrics;
        $scope.$watch("selection.sessionModels", function(nv, ov) {
            if (!nv || nv.length == 0) return;

            metrics = $scope.timeseriesEvaluationMetrics.map(metric => $scope.metricMap[metric[0]]);
            default_displayed_metrics = ["RMSE", "SMAPE", "MAPE"];
            if (!default_displayed_metrics.includes(nv[0].evaluationMetric)) {
                default_displayed_metrics.push(nv[0].evaluationMetric)
            }

            $scope.displayableMetrics = $scope.timeseriesEvaluationMetrics.map(function(metric) {
                return {
                    name: metric[0],
                    $displayed: default_displayed_metrics.includes(metric[0])
                }
            });

            highestMetricValues = metrics.reduce(function(res, metric) {
                res[metric] = Math.max(...$scope.selection.sessionModels.map(model => model[metric] || 0));
                return res;
            }, {});

            // add all unique custom metrics and get their max values
            let customMetrics = new Set();
            nv.filter(model => model.customMetricsResults).forEach(function(model) {
                model.customMetricsResults.filter(customMetricResult => customMetricResult.didSucceed).forEach(function(customMetricResult) {
                    customMetrics.add(customMetricResult.metric.name)
                    if (!(customMetricResult.metric.name in highestMetricValues) || customMetricResult.value > highestMetricValues[customMetricResult.metric.name]) {
                        highestMetricValues[customMetricResult.metric.name] = customMetricResult.value;
                    }
                });
            });

            customMetrics.forEach(function(customMetric) {
                $scope.displayableMetrics.push({
                    name: customMetric,
                    $displayed: default_displayed_metrics.includes(customMetric)
                });
                metrics.push(customMetric)
            })

            const newSessionId = (nv[0] || {}).sessionId;
            const oldSessionId = (ov && ov[0] || {}).sessionId;

            // Do not change the tab if we are still on the same session
            if (newSessionId === oldSessionId) return;

            if ($scope.isSessionRunning(newSessionId) && $scope.anySessionModelNeedsHyperparameterSearch()) {
                // Switch to HP search chart when starting a training or switching to the session
                // currently being trained
                $scope.uiState.tsSessionDetailView = 'HP_SEARCH';
            } else {
                // Otherwise always show metrics when switching between sessions
                $scope.uiState.tsSessionDetailView = 'METRICS';
            }
        });


        let sessionIdThatWasRunning = null;
        $scope.$on("mlTaskStatusRefresh", function() {
            const currentSessionId = (($scope.selection && $scope.selection.sessionModels && $scope.selection.sessionModels[0]) || {}).sessionId;
            if (!currentSessionId) return;

            if (!sessionIdThatWasRunning && $scope.mlTaskStatus.training && $scope.isSessionRunning(currentSessionId)) {
                sessionIdThatWasRunning = currentSessionId;
            }

            if (currentSessionId === sessionIdThatWasRunning && !$scope.mlTaskStatus.training) {
                // Only switch back to METRICS at the end of the training, if we are currently
                // viewing the session being trained
                $scope.uiState.tsSessionDetailView = 'METRICS';
                sessionIdThatWasRunning = null;
            }
        });

        $scope.anySessionModelHasMetrics = function() {
            return ($scope.selection.sessionModels || []).some(function(model) {
                return metrics.some(metric => model[metric]);
            })
        };

        $scope.getModelMetric = function(model, metricName) {
            if (metricName in $scope.metricMap) {
                return model[$scope.metricMap[metricName]];
            } else {
                const customMetric = model.customMetricsResults.find(customMetric => metricName == customMetric.metric.name);
                if (customMetric) return customMetric.value;
            }
        }

        $scope.getModelMetricBarRatioHeight = function(model, metricName) {
            const metric = $scope.metricMap[metricName] || metricName;
            const metricValueForModel = $scope.getModelMetric(model, metricName);
            if (!metricValueForModel) return 0;
            const highestMetricValue = highestMetricValues && highestMetricValues[metric];
            if (!highestMetricValue) return 0;
            return metricValueForModel / highestMetricValue;
        };

        $scope.selectMetric = function(metricIdx) {
            $scope.displayableMetrics[metricIdx].$displayed = !$scope.displayableMetrics[metricIdx].$displayed;
        };

        $scope.nbDisplayedMetrics = function() {
            return $scope.displayableMetrics.filter(metric => metric.$displayed).length;
        };

        $scope.$watch('uiState.currentMetric', function(nv) {
            if (!nv || !$scope.displayableMetrics) return;
            const currentMetric = $scope.displayableMetrics.find(metric => metric.name === $scope.uiState.currentMetric);
            if (currentMetric) currentMetric.$displayed = true;
        });
    });

    app.controller('ExportTimeseriesDataController', function($scope, ExportUtils) {
        $scope.$on('timeseriesTableEvent', function(event, timeseriesTable) {
            $scope.tableHeaders = timeseriesTable.headers
            $scope.allTableRows = timeseriesTable.rows
        });

        function getSingleTsData(withStats) {
            return $scope.allTableRows.map(row => {
                let newRow = [];
                row.forEach(x => {
                    // Export the displayValue when there is no rawValue
                    newRow.push(x.rawValue || x.displayValue);

                    // add statistics after each coefficient value
                    if (withStats) {
                        if (x.rawStderr !== undefined) {
                            newRow.push(x.rawStderr);
                        }
                        if (x.rawPvalue !== undefined) {
                            newRow.push(x.rawPvalue);
                        }
                        if (x.rawTvalue !== undefined) {
                            newRow.push(x.rawTvalue);
                        }
                    }
                });
                return newRow;
            });
        }

        function getMultipleTsData() {
            return $scope.allTableRows.map(row => {
                let newRow = [];
                for (const header of $scope.tableHeaders) {
                    if (header.rawField) { // handle fields with different display value to actual value
                        newRow.push(row[header.rawField].rawValue);
                    } else {
                        newRow.push(row[header.field]);
                    }
                }
                return newRow;
            });
        }

        $scope.exportTimeseriesData = function(notAgGridRow, label, valueType, exportPerFold, withStats) {
            let nbTimeseriesIdentifiers = $scope.modelData.coreParams.timeseriesIdentifiers.length;
            const columns = [];
            valueType = valueType || "double";  // Use double by default for metrics and coefficients
            exportPerFold = exportPerFold || false;
            withStats = withStats || false;

            // if export per fold, we have 3 extra columns that we want to be considered string type
            nbTimeseriesIdentifiers = exportPerFold ? nbTimeseriesIdentifiers + 3 : nbTimeseriesIdentifiers;
            $scope.tableHeaders.forEach(function(column, i) {
                if (notAgGridRow) {
                    columns.push({ name: column.rawName || column.displayName, type: i < nbTimeseriesIdentifiers ? "string" : valueType });
                } else {
                    columns.push({ name: column.headerName || column.field, type: i < nbTimeseriesIdentifiers ? "string" : valueType });
                }
            });

            let data;
            if (notAgGridRow) {
                data = getSingleTsData(withStats);
            } else {
                data = getMultipleTsData();
            }

            ExportUtils.exportUIData($scope, {
                name: label + " of model " + $scope.modelData.userMeta.name,
                columns: columns,
                data: data
            }, `Export model ${label}`);
        }

        $scope.noModelCoefficientExplanation = function() {
            // with a null order arima, there is no coefficient to show
            let isArimaAndOrderIsZero = false;
            isArimaAndOrderIsZero = $scope.modelData.actualParams.resolved.algorithm === "ARIMA" &&
                ["p", "q", "P", "Q"].every(order => {
                    return $scope.modelData.actualParams.resolved.arima_timeseries_params[order] === 0;
                });
            if (isArimaAndOrderIsZero) {
                return "No coefficients for the selected model since p, q, P and Q are equal to 0"
            }
        }
    });

    app.controller('InformationCriteriaController', function($scope, DataikuAPI, FutureProgressModal) {
        $scope.retrieveInformationCriteria = function(fullModelId) {
            DataikuAPI.ml.prediction.getInformationCriteria(fullModelId)
                .then(function(res) {
                    FutureProgressModal.show($scope, res.data, "Retrieve information criteria").then(function(informationCriteria) {
                        if (informationCriteria) {
                            // will be undefined if computation was aborted
                            $scope.modelData.iperf.informationCriteria = informationCriteria;
                        }
                    });
                }).catch(setErrorInScope.bind($scope));
        };
    });

    app.component("multipleTsInformationCriteria", {
        bindings: {
            informationCriteria: '<',
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            insightId: '<',
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                insight-id="$ctrl.insightId"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
            ></table-manager>
        `,
        controller: function($scope, $stateParams, PerTimeseriesService, InformationCriteriaUtils) {
            const $ctrl = this;

            $ctrl.$onInit = () => {
                $ctrl.featureColumns = $ctrl.informationCriteria.map(ic => {
                    return {
                        field: ic.displayName,
                        rawField: ic.displayName, // used by us for export
                        filter: 'agNumberColumnFilter',
                        valueGetter: p => PerTimeseriesService.getDisplayValue(p, ic)
                    };
                });
                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.informationCriteria[0].values); // if an identifier is lacking the first ic, it won't appear at all
            }

            $ctrl.getHeaders = function($scope) {
                return PerTimeseriesService.initTimeseriesIdentifierTableColumns($scope.timeseriesIdentifierColumns, $ctrl.featureColumns);
            };

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                const row = {...parsedTimeseriesIdentifier};
                $ctrl.informationCriteria.forEach(function(criteria) {
                    const criteriaValue = unparsedTimeseriesIdentifier in criteria.values ? criteria.values[unparsedTimeseriesIdentifier] : "";

                    row[criteria["displayName"]] = {
                        displayValue: InformationCriteriaUtils.formatValue(criteriaValue),
                        rawValue: InformationCriteriaUtils.formatValue(criteriaValue, false)
                    };
                });
                return [row];
            };
        }
    });

    app.component("multipleTsAutoarimaOrders", {
        bindings: {
            postTrain: '<',
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            insightId: '<',
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                insight-id="$ctrl.insightId"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
            ></table-manager>
        `,
        controller: function($scope, $stateParams, AutoArimaOrdersService, PerTimeseriesService) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.resolvedAlgoValues = $ctrl.postTrain.auto_arima_timeseries_params;
                $ctrl.autoArimaRelevantParams = AutoArimaOrdersService.ordersToDisplay($ctrl.resolvedAlgoValues);
                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.resolvedAlgoValues.p)
            };

            $ctrl.getHeaders = function($scope) {
                return PerTimeseriesService.initTimeseriesIdentifierTableColumns($scope.timeseriesIdentifierColumns, $ctrl.autoArimaRelevantParams);
            };

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                const row = {...parsedTimeseriesIdentifier};
                $ctrl.autoArimaRelevantParams.forEach(function(algoParam) {
                    row[algoParam.displayName] = String($ctrl.resolvedAlgoValues[algoParam.fieldName][unparsedTimeseriesIdentifier]);
                });
                return [row];
            };
        },
    });

    app.service('InformationCriteriaUtils', function() {
        this.formatValue = function(informationCriteriaValue, round=true) {
            if (["+∞", "-∞"].includes(informationCriteriaValue)) {
                return informationCriteriaValue
            }
            if (informationCriteriaValue.value !== "") {
                return round ? String(informationCriteriaValue.toFixed(4)) : String(informationCriteriaValue);
            }
            return informationCriteriaValue;
        };
    });
    app.filter('formatFeatureType', function() {
        return function (featureType) {
            if (!featureType) {
                return;
            }
            switch (featureType) {
                case 'NUMERIC':
                    return "Numerical";
                case 'CATEGORY':
                    return "Categorical";
                case 'TEXT':
                    return "Text";
                case 'VECTOR':
                    return "Vector";
                case 'IMAGE':
                    return "Image";
            }
        }
    });

    app.filter('formatFeatureRole', function() {
        return function (featureRole) {
            if (!featureRole) {
                return;
            }
            switch (featureRole) {
                case 'TARGET':
                    return "Target";
                case 'INPUT':
                    return "Known in advance";
                case 'INPUT_PAST_ONLY':
                    return "Past only";
            }
        }
    });

    app.filter('displayTimeseriesWindowOperation', function() {
        return function(windowOperation) {
            if (windowOperation === 'MEAN') {
                return 'Avg';
            } else if (windowOperation === 'MEDIAN') {
                return 'Median';
            } else if (windowOperation === 'STD') {
                return 'Std. dev.';
            } else if (windowOperation === 'MIN') {
                return 'Min';
            } else if (windowOperation === 'MAX') {
                return 'Max';
            } else if (windowOperation === 'FREQUENCY') {
                return 'Frequency';
            } else {
                return windowOperation;
            }
        }
    });

    app.component("singleTsInformationCriteria", {
        bindings: {
            informationCriteria: '<'
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/single-ts-information-criteria.html",
        controller: function($scope, SINGLE_TIMESERIES_IDENTIFIER, InformationCriteriaUtils) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.headers = $ctrl.informationCriteria.map(criteria => {
                    return { displayName: criteria.displayName }
                });
                $ctrl.singleRow = $ctrl.informationCriteria.map(function(criteria) {
                    return {
                        displayValue: InformationCriteriaUtils.formatValue(criteria.values[SINGLE_TIMESERIES_IDENTIFIER]),
                        rawValue: InformationCriteriaUtils.formatValue(criteria.values[SINGLE_TIMESERIES_IDENTIFIER], false)
                    }
                });
                $scope.$emit('timeseriesTableEvent', { headers: $ctrl.headers, rows: [$ctrl.singleRow] });
            };
        },
    });

    app.component("singleTsAutoarimaOrders", {
        bindings: {
            postTrain: '<'
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/single-ts-autoarima-orders.html",
        controller: function($scope, SINGLE_TIMESERIES_IDENTIFIER, AutoArimaOrdersService) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.SINGLE_TIMESERIES_IDENTIFIER = SINGLE_TIMESERIES_IDENTIFIER;
                $ctrl.resolvedAutoArimaValues = $ctrl.postTrain.auto_arima_timeseries_params;
                $ctrl.autoArimaRelevantParams = AutoArimaOrdersService.ordersToDisplay($ctrl.resolvedAutoArimaValues);
                const singleRow = $ctrl.autoArimaRelevantParams.map(function(order) {
                    return { displayValue: $ctrl.resolvedAutoArimaValues[order.displayName][SINGLE_TIMESERIES_IDENTIFIER] }
                });
                $scope.$emit('timeseriesTableEvent', { headers: $ctrl.autoArimaRelevantParams, rows: [singleRow] });
            };
        },
    });

    app.service("AutoArimaOrdersService", function() {
        return {
            ordersToDisplay
        };

        function ordersToDisplay(resolvedAutoArimaValues) {
            const autoArimaRelevantParams = [
                { field: 'p', displayName: 'p', fieldName: 'p', helper: 'Auto-regressive model order' },
                { field: 'q', displayName: 'q', fieldName: 'q', helper: 'Moving-average model order' },
            ];
            if (!resolvedAutoArimaValues.stationary) {
                autoArimaRelevantParams.push({ field: 'd', displayName: 'd', fieldName: 'd', helper: 'Differencing order' });
            }
            const isSeasonalAutoARIMA = resolvedAutoArimaValues.m > 1;
            if (isSeasonalAutoARIMA) {
                autoArimaRelevantParams.push({ field: 'P', displayName: 'P', fieldName: 'P', helper: 'Seasonal auto-regressive model order' });
                autoArimaRelevantParams.push({ field: 'Q', displayName: 'Q', fieldName: 'Q', helper: 'Seasonal moving-average model order' });
                if (!resolvedAutoArimaValues.stationary) {
                    autoArimaRelevantParams.push({ field: 'D', displayName: 'D', fieldName: 'D', helper: 'Seasonal differencing order' });
                }
            }
            return autoArimaRelevantParams;
        }
    });

    app.component("modelCoefficientsExplanation", {
        bindings: {
            algorithm: '<',
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/model-coefficients-explanation.html",
        controller: function() {
            const $ctrl = this;

            const algorithmTitles = {
                "ETS": "ETS coefficients",
                "SEASONAL_LOESS": "Seasonal trend coefficients",
                "AUTO_ARIMA": "Seasonal ARIMA coefficients",
                "ARIMA": "Seasonal ARIMA coefficients",
                "PROPHET": "Prophet coefficients"
            }

            $ctrl.$onInit = () => {
                $ctrl.title = algorithmTitles[$ctrl.algorithm]
            };

            $ctrl.foldableToggle = function() {
                $ctrl.foldableOpen = !$ctrl.foldableOpen;
            };
        },
    });

    app.component("singleTsModelCoefficients", {
        bindings: {
            modelCoefficients: '<',
            algorithm: '<',
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/single-ts-model-coefficients.html",
        controller: function($scope, SINGLE_TIMESERIES_IDENTIFIER, PerTimeseriesService) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.modelCoefficientsHeader = PerTimeseriesService.initModelCoefficientsHeader($ctrl.modelCoefficients, false, false, false);
                $ctrl.canDisplayStatisticalValues = ["AUTO_ARIMA", "ARIMA", "ETS", "SEASONAL_LOESS"].includes($ctrl.algorithm);
                $ctrl.hasStderrs = $ctrl.modelCoefficients.some(coeff => coeff.stderrs !== undefined && Object.keys(coeff.stderrs).length > 0);
                $ctrl.hasPvalues = $ctrl.modelCoefficients.some(coeff => coeff.pvalues !== undefined && Object.keys(coeff.pvalues).length > 0);
                $ctrl.hasTvalues = $ctrl.modelCoefficients.some(coeff => coeff.tvalues !== undefined && Object.keys(coeff.tvalues).length > 0);
                $ctrl.singleRow = $ctrl.modelCoefficients.map(coeff => {
                    const coefficientHasStderr = coeff.stderrs && coeff.stderrs[SINGLE_TIMESERIES_IDENTIFIER] !== undefined;
                    const coefficientHasPvalue = coeff.pvalues && coeff.pvalues[SINGLE_TIMESERIES_IDENTIFIER] !== undefined;
                    const coefficientHasTvalue = coeff.tvalues && coeff.tvalues[SINGLE_TIMESERIES_IDENTIFIER] !== undefined;
                    return {
                        displayValue: coeff.values[SINGLE_TIMESERIES_IDENTIFIER] ? String(coeff.values[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "-",
                        rawValue: coeff.values[SINGLE_TIMESERIES_IDENTIFIER],
                        displayStderr: coefficientHasStderr ? String(coeff.stderrs[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "-",
                        rawStderr: coefficientHasStderr ? coeff.stderrs[SINGLE_TIMESERIES_IDENTIFIER] : ($ctrl.hasStderrs ? "-" : undefined),
                        displayPvalue: coefficientHasPvalue ? String(coeff.pvalues[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "-",
                        rawPvalue: coefficientHasPvalue ? coeff.pvalues[SINGLE_TIMESERIES_IDENTIFIER] : ($ctrl.hasPvalues ? "-" : undefined),
                        displayTvalue: coefficientHasTvalue ? String(coeff.tvalues[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "-",
                        rawTvalue: coefficientHasTvalue ? coeff.tvalues[SINGLE_TIMESERIES_IDENTIFIER] : ($ctrl.hasTvalues ? "-" : undefined),
                    }
                });
                $scope.$emit('timeseriesTableEvent', {
                    headers: PerTimeseriesService.initModelCoefficientsHeader($ctrl.modelCoefficients, $ctrl.hasStderrs, $ctrl.hasPvalues, $ctrl.hasTvalues),
                    rows: [$ctrl.singleRow]
                });
            };
        },
    });

    app.component("multipleTsModelCoefficients", {
        bindings: {
            modelCoefficients: '<',
            postTrain: '<',
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            insightId: '<',
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                insight-id="$ctrl.insightId"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                set-flags="$ctrl.setFlags"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
                hide-columns-with-field="$ctrl.getColumnsToHide"
            ></table-manager>
        `,
        controller: function($scope, $stateParams, PerTimeseriesService) {
            const $ctrl = this;

            $ctrl.$onInit = function() {
                $ctrl.unparsedTimeseriesIdentifiers = PerTimeseriesService.retrieveUnparsedTimeseriesIdentifiers($ctrl.postTrain, $ctrl.modelCoefficients);
            }

            $ctrl.getColumnsToHide = function(uiState) {
                const fieldsToHide = [];

                if (!uiState.timeseriesTableStatisticsDisplay.includes("stderr")) {
                    fieldsToHide.push("stderr")
                }
                if (!uiState.timeseriesTableStatisticsDisplay.includes("p-value")) {
                    fieldsToHide.push("pValue")
                }
                if (!uiState.timeseriesTableStatisticsDisplay.includes("t-stat")) {
                    fieldsToHide.push("tStat")
                }
                return fieldsToHide;
            }

            $ctrl.setFlags = function(uiState) {
                uiState.canDisplayStatisticalValues = ["AUTO_ARIMA", "ARIMA", "ETS", "SEASONAL_LOESS"].includes($ctrl.postTrain.algorithm);
                uiState.hasStderrs = $ctrl.modelCoefficients.some(coeff => coeff.stderrs !== undefined && Object.keys(coeff.stderrs).length > 0);
                uiState.hasPvalues = $ctrl.modelCoefficients.some(coeff => coeff.pvalues !== undefined && Object.keys(coeff.pvalues).length > 0);
                uiState.hasTvalues = $ctrl.modelCoefficients.some(coeff => coeff.tvalues !== undefined && Object.keys(coeff.tvalues).length > 0);
                uiState.isModelCoefficient = true;
                uiState.hasStatistics = uiState.hasStderrs || uiState.hasPvalues || uiState.hasTvalues;
                uiState.timeseriesTableStatisticsDisplay = [];
            }

            $ctrl.getHeaders = function($scope) {
                const modelCoefficientsHeader = PerTimeseriesService.initModelCoefficientsHeader($ctrl.modelCoefficients, true, true, true);
                return PerTimeseriesService.initTimeseriesIdentifierTableColumns($scope.timeseriesIdentifierColumns, modelCoefficientsHeader);
            }

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                const row = {...parsedTimeseriesIdentifier};
                $ctrl.modelCoefficients.forEach(function(coeff) {
                    const coefficientValue = unparsedTimeseriesIdentifier in coeff.values ? coeff.values[unparsedTimeseriesIdentifier] : "";

                    const displayName = coeff["displayName"];

                    row[displayName] = $ctrl.formatValue(coefficientValue);

                    if (uiState.hasStderrs) {
                        row[displayName + " stderr"] = $ctrl.formatValue(coeff.stderrs[unparsedTimeseriesIdentifier]);
                    }

                    if (uiState.hasPvalues) {
                        row[displayName + " p-value"] = $ctrl.formatValue(coeff.pvalues[unparsedTimeseriesIdentifier]);
                    }

                    if (uiState.hasTvalues) {
                        row[displayName + " t-stat"] = $ctrl.formatValue(coeff.tvalues[unparsedTimeseriesIdentifier]);
                    }
                });

                return [row];
            };

            $ctrl.formatValue = function (value) {
                if (value && value !== "") {
                    return {
                        displayValue: String(value.toFixed(4)),
                        rawValue: String(value)
                    }
                }
                return "";
            };
        },
    });

    app.component("granularTimeseriesMetrics", {
        bindings: {
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            fullModelEvaluationId: '<?',
            insightId: '<',
            isKfold: '<',
            isCurrentlyDisplayingKfold: '=' // only used to signal to parent current display status
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                set-flags="$ctrl.setFlags"
                insight-id="$ctrl.insightId"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
                on-ui-change="$ctrl.onUiChange"
                get-additional-filters="$ctrl.getAdditionalFilters"
                get-pinned-rows="$ctrl.calculateAverageRow"
            ></table-manager>
        `,
        controller: function($scope, $filter, DataikuAPI, PMLSettings, PMLFilteringService, PerTimeseriesService, TimeseriesTableService, $stateParams) {
            const $ctrl = this;
            $ctrl.unparsedTimeseriesIdentifiers = [];
            $ctrl.calculateAverageRow = TimeseriesTableService.calculateAverageRow;
            $ctrl.foldHeaders = [
                {
                    headerName: "Fold ID",
                    field: "foldId",
                    filter: 'agNumberColumnFilter',
                    type: 'leftAligned', // ag-grid for content layout
                    valueType: "int", // us to consume when populating rows
                    pinned:"left"
                },
                {
                    headerName: "Start Date",
                    field: "startDate",
                    type: "date"
                },
                {
                    headerName: "End Date",
                    field: "endDate",
                    type: "date"
                }
            ];
            $ctrl.showPerFold = false;
            $ctrl.hasIdentifiers = true;
            $ctrl.isGranularMetrics = true;
            $ctrl.couldNotLocatePerFoldMetrics = false;
            $ctrl.customMetricHeaders = [];

            function getDisplayValueFromMetricNode(params) {
                const rawField = params.colDef.rawField;
                return params.data[rawField].displayValue
            }


            $ctrl.$onInit = function() {
                const PER_TIMESERIES_EVALUATION_METRICS = ["MSE", "MAPE", "MASE", "SMAPE", "MAE", "MSIS"];
                const timeseriesEvaluationMetrics = PMLSettings.taskF().timeseriesEvaluationMetrics;

                $ctrl.metricNameMap = timeseriesEvaluationMetrics
                    .filter(metric => PER_TIMESERIES_EVALUATION_METRICS.includes(metric[0]))
                    .map(function(metric) {
                        return {
                            headerName: PMLSettings.names.evaluationMetrics[metric[0]], // used by ag-grid for the column name
                            field: PMLFilteringService.metricMap[metric[0]] + ".rawValue", // used by ag-grid for the column value
                            rawField: PMLFilteringService.metricMap[metric[0]], // used by us to export
                            rawName: metric[0],
                            filter: 'agNumberColumnFilter',
                            valueFormatter: p => getDisplayValueFromMetricNode(p)
                        };
                    });

                if ($ctrl.timeseriesIdentifierColumns.length === 0) {
                    $ctrl.showPerFold = true;
                    $ctrl.isCurrentlyDisplayingKfold = $ctrl.showPerFold;
                    $ctrl.hasIdentifiers = false;
                }

                if ($ctrl.showPerFold) {
                    $ctrl.getPerFoldData();
                } else {
                    $ctrl.getPerTimeseriesData();
                }
            };

            $ctrl.setFlags = function(uiState) {
                uiState.showPerFold = $ctrl.showPerFold;
                uiState.hasIdentifiers = $ctrl.hasIdentifiers;
                uiState.isGranularMetrics = $ctrl.isGranularMetrics;
                uiState.couldNotLocatePerFoldMetrics = $ctrl.couldNotLocatePerFoldMetrics;
                uiState.isKfold = $ctrl.isKfold;
                uiState.timeseriesTableFoldFilters = [];
                uiState.foldIds = $ctrl.foldIds;
            };

            $ctrl.onUiChange = function(uiState) {
                if ($ctrl.showPerFold !== uiState.showPerFold) {
                    $ctrl.showPerFold = uiState.showPerFold;
                    $ctrl.isCurrentlyDisplayingKfold = $ctrl.showPerFold;

                    if ($ctrl.showPerFold) {
                        $ctrl.getPerFoldData();
                    } else {
                        $ctrl.getPerTimeseriesData();
                    }
                }
            }

            $ctrl.getAdditionalFilters = function(uiState) {
                if (uiState.showPerFold) {
                    return {foldId: uiState.timeseriesTableFoldFilters.map(x => Number(x))} // we tell ag-grid that foldId is a number, so we have to convert here
                }
                return {};
            }

            $ctrl.getPerFoldData = function() {
                if (!$ctrl.perFoldData) {
                    DataikuAPI.ml.prediction.getKfoldPerfs($ctrl.fullModelId).then(function(response) {
                        $ctrl.perFoldData = response.data.perFoldMetrics;
                        $ctrl.foldIds = (Object.values($ctrl.perFoldData)[0].map((x) => x["foldId"]));
                        $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perFoldData);
                    }).catch(response => {
                        if (response.status === 404) {
                            $ctrl.unparsedTimeseriesIdentifiers = [];
                            $ctrl.couldNotLocatePerFoldMetrics = true;
                        } else {
                            setErrorInScope.bind($scope);
                        }
                    });
                } else {
                    $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perFoldData);
                }
            }

            $ctrl.getPerTimeseriesData = function() {
                if (!$ctrl.perTimeseriesMetrics) {
                    if ($ctrl.fullModelEvaluationId) {
                            DataikuAPI.modelevaluations.getPerTimeseriesMetrics($ctrl.fullModelEvaluationId).then(({ data }) => {
                                $ctrl.perTimeseriesMetrics = data.perTimeseries;
                                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perTimeseriesMetrics);
                            }).catch(setErrorInScope.bind($scope));
                        } else {
                            DataikuAPI.ml.prediction.getPerTimeseriesMetrics($ctrl.fullModelId).then(({ data }) => {
                                $ctrl.perTimeseriesMetrics = data.perTimeseries;
                                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perTimeseriesMetrics);
                            }).catch(setErrorInScope.bind($scope));
                        }
                } else {
                    $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perTimeseriesMetrics);
                }
            }

            $ctrl.getCustomMetricHeader = function(foundMetric) {
                const customMetricResults = foundMetric.customMetricsResults || [];
                return customMetricResults.map(x => {
                    return {
                        field: "CUSTOM_METRIC" + sanitize(x.metric.name),
                        rawField: "CUSTOM_METRIC" + sanitize(x.metric.name),
                        rawName: x.metric.name,
                        headerName: sanitize(x.metric.name),
                        valueFormatter: p => getDisplayValueFromMetricNode(p)
                    }
                });
            }

            $ctrl.getHeaders = function($scope) {
                if ($scope.uiState.showPerFold) {
                    $ctrl.customMetricHeaders = $ctrl.getCustomMetricHeader($ctrl.perFoldData[Object.keys($ctrl.perFoldData)[0]][0].metrics)
                    return PerTimeseriesService.initTimeseriesIdentifierTableColumns($ctrl.timeseriesIdentifierColumns, [...$ctrl.foldHeaders, ...$ctrl.metricNameMap, ...$ctrl.customMetricHeaders]);
                } else {
                    $ctrl.customMetricHeaders = $ctrl.getCustomMetricHeader($ctrl.perTimeseriesMetrics[Object.keys($ctrl.perTimeseriesMetrics)[0]])
                    return PerTimeseriesService.initTimeseriesIdentifierTableColumns($ctrl.timeseriesIdentifierColumns, [...$ctrl.metricNameMap, ...$ctrl.customMetricHeaders]);
                }
            };

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                if (uiState.showPerFold) {
                    const rows = [];
                    const perTsMetrics = $ctrl.perFoldData[unparsedTimeseriesIdentifier];
                    for (let foldMetric of perTsMetrics) {
                        const row = {...parsedTimeseriesIdentifier};
                        $ctrl.foldHeaders.forEach(function(header) {
                            if (header.valueType === "int") {
                                row[header.field] = Number(foldMetric[header.field]); // necessary for filtering to work
                            } else {
                                row[header.field] = foldMetric[header.field];
                            }
                        });

                        const metrics = foldMetric.metrics;

                        $ctrl.metricNameMap.filter(metricHeader => !metricHeader.isCustom).forEach(function(metricHeader) {
                            row[metricHeader.rawField] = {
                                rawValue: metrics[metricHeader.rawField],
                                displayValue: PerTimeseriesService.createMetricRow(metrics[metricHeader.rawField], metrics[metricHeader.rawField + 'std'], metricHeader.rawName)
                            }
                        });
                        if ($ctrl.customMetricHeaders.length > 0) {
                            foldMetric["metrics"].customMetricsResults.forEach(function(customMetric) {
                                row["CUSTOM_METRIC" + sanitize(customMetric.metric.name)] = {
                                    rawValue: customMetric.value,
                                    displayValue: PerTimeseriesService.createMetricRow(customMetric.value, customMetric.valuestd, null),
                                    rawStd: customMetric.valuestd
                                }
                            })
                        }
                        rows.push(row);
                    }
                    return rows;
                } else {
                    const row = {...parsedTimeseriesIdentifier};
                    const data = $ctrl.perTimeseriesMetrics[unparsedTimeseriesIdentifier];
                    $ctrl.metricNameMap.filter(metricHeader => !metricHeader.isCustom).forEach(function(metricHeader) {
                        row[metricHeader.rawField] = {
                            rawValue: data[metricHeader.rawField],
                            rawStd: data[metricHeader.rawField + 'std'],
                            displayValue: PerTimeseriesService.createMetricRow(data[metricHeader.rawField], data[metricHeader.rawField + 'std'], metricHeader.rawName)
                        }
                    });

                    if ($ctrl.customMetricHeaders.length > 0) {
                        data.customMetricsResults.forEach(function(customMetric) {
                            row["CUSTOM_METRIC" + sanitize(customMetric.metric.name)] = {
                                rawValue: customMetric.value,
                                displayValue: PerTimeseriesService.createMetricRow(customMetric.value, customMetric.valuestd, null),
                                rawStd: customMetric.valuestd
                            }
                        })
                    }
                    return [row];
                }
            };

        }
    });

    app.service("PerTimeseriesService", function($filter, SINGLE_TIMESERIES_IDENTIFIER) {
        const COLUMN_WIDTH = 160;

        return {
            removeDuplicatesAndSortIdentifierValuesForFilterDropdowns, addIdentifierValues,
            initTimeseriesIdentifierTableColumns, initTimeseriesIdentifiersValues, initModelCoefficientsHeader,
            retrieveUnparsedTimeseriesIdentifiers, createMetricRow, shouldDisplayTimeseries, getDisplayValue
        };

        function initTimeseriesIdentifiersValues(timeseriesIdentifierColumns) {
            const allTimeseriesIdentifierValuesMap = {};
            if (!timeseriesIdentifierColumns || !timeseriesIdentifierColumns.length) return allTimeseriesIdentifierValuesMap;
            timeseriesIdentifierColumns.forEach(function(identifierColumn) {
                allTimeseriesIdentifierValuesMap[identifierColumn] = [];
            });
            return allTimeseriesIdentifierValuesMap;
        }

        function initTimeseriesIdentifierTableColumns(timeseriesIdentifierColumns, specificColHeaders) {
            const tableHeaders = [];
            timeseriesIdentifierColumns.forEach(function(identifierColumn, _) {
                tableHeaders.push({
                    field: identifierColumn,
                    pinned: 'left',
                    menuTabs: ["generalMenuTab"],
                });
            });

            specificColHeaders.forEach(function(header, _) {
                let new_header = { type: 'rightAligned', ...header }
                tableHeaders.push(new_header);
            });

            tableHeaders.forEach(dict => {
                if (!dict.headerName) { // we explicitly set headerName for the metrics columns, therefore we don't want to override
                    dict.headerName = dict.field; // `field` gets auto-capitalised by ag-grid, which causes problems for arima coefficients - headerName is not modified
                }
            });

            return tableHeaders;
        }

        function initModelCoefficientsHeader(coefficients, withStderrs, withPvalues, withTvalues) {
            const modelCoefficientsHeader = [];
            const showStderrs = withStderrs ? coefficients.some((coeff) => coeff.stderrs !== undefined && Object.keys(coeff.stderrs).length > 0) : false;
            const showPvalues = withPvalues ? coefficients.some((coeff) => coeff.pvalues !== undefined && Object.keys(coeff.pvalues).length > 0) : false;
            const showTvalues = withTvalues ? coefficients.some((coeff) => coeff.tvalues !== undefined && Object.keys(coeff.tvalues).length > 0) : false;
            coefficients.forEach(function(coeff) {
                const displayName = coeff.displayName;
                if (coeff.isExternalFeature) {
                    // Replace ":" by "_" when exporting data
                    modelCoefficientsHeader.push({
                        field: sanitize(displayName),
                        rawField: coeff.displayName,
                        displayName: $filter("mlFeature")(displayName, true),
                        rawName: displayName.replace(/:/g, "_"),
                        headerClass:["monospace-column", "ag-right-aligned-header"],
                        filter: 'agNumberColumnFilter',
                        valueGetter: p => getDisplayValue(p, coeff),
                    });
                } else {
                    modelCoefficientsHeader.push({
                        field: displayName,
                        rawField: displayName,
                        displayName: displayName,
                        filter: 'agNumberColumnFilter',
                        valueGetter: p => getDisplayValue(p, coeff),
                    });
                }

                // add columns for statistics after each coefficient value
                // use short names for statistical values to fit table headers, they are still exported with their full name
                const shortNames = {
                    "smoothing_level": "sl",
                    "smoothing_trend": "st",
                    "initial_level": "il",
                    "initial_trend": "it",
                }
                let shortName = displayName;
                if (shortNames[displayName] !== undefined) {
                    shortName = shortNames[displayName];
                }

                // add columns for statistics after each coefficient value
                if (showStderrs) {
                    modelCoefficientsHeader.push({
                        field: displayName + " stderr",
                        rawField: displayName + " stderr",
                        displayName: shortName + " stderr",
                        rawName: displayName + " stderr",
                        stderr:true,
                        filter: 'agNumberColumnFilter'
                    });
                }
                if (showPvalues) {
                    modelCoefficientsHeader.push({
                        field: displayName + " p-value",
                        rawField: displayName + " p-value",
                        displayName: shortName + " p-value",
                        rawName: displayName + " p-value",
                        pValue:true,
                        filter: 'agNumberColumnFilter'
                    });
                }
                if (showTvalues) {
                    modelCoefficientsHeader.push({
                        field: displayName + " t-stat",
                        rawField: displayName + " t-stat",
                        displayName: shortName + " t-stat",
                        rawName: displayName + " t-stat",
                        tStat:true,
                        filter: 'agNumberColumnFilter'
                    });
                }
            });
            return modelCoefficientsHeader;
        }

        function retrieveUnparsedTimeseriesIdentifiers(postTrain, modelCoefficients) {
            // Use model coefficients that are present for all time series to retrieve all time series identifiers.
            // Indeed some coefficients can be defined only for a subset of time series, hence are not good candidates to retrieve all time series identifiers.
            const algorithm = postTrain.algorithm;
            if (algorithm == "AUTO_ARIMA") return Object.keys(postTrain.auto_arima_timeseries_params.p);
            let coeffAlwaysSet;
            if (algorithm == "ARIMA") coeffAlwaysSet = modelCoefficients[0]; // any coefficient can work
            if (["SEASONAL_LOESS", "ETS"].includes(algorithm)) coeffAlwaysSet = modelCoefficients.find(x => x.displayName === "smoothing_level");
            if (algorithm == "PROPHET") coeffAlwaysSet = modelCoefficients.find(x => x.displayName === "k");
            if (coeffAlwaysSet) return Object.keys(coeffAlwaysSet.values);
            return [];
        }

        function addIdentifierValues(parsedTimeseriesIdentifier, timeseriesIdentifierColumns, allTimeseriesIdentifierValuesMap) {
            timeseriesIdentifierColumns.forEach(function(identifierColumn) {
                const identifierValue = parsedTimeseriesIdentifier[identifierColumn];
                allTimeseriesIdentifierValuesMap[identifierColumn].push(identifierValue);
            });
        }

        function removeDuplicatesAndSortIdentifierValuesForFilterDropdowns(allTimeseriesIdentifierValuesMap) {
            Object.keys(allTimeseriesIdentifierValuesMap).forEach(function(identifierColumn) {
                // remove duplicates and sort a copy of the array, to trigger update of dropdown values in basic-select
                const sortedValues = [...new Set(allTimeseriesIdentifierValuesMap[identifierColumn])].sort();
                allTimeseriesIdentifierValuesMap[identifierColumn] = sortedValues;
            });
        }

        function createMetricRow(metricValue, metricValueStd, metricRawName, precision=5) {
            const metricDisplayValue = $filter("mlMetricFormat")(metricValue, metricRawName, precision, metricValueStd, true);
            // this is a copy of what is done in the stripHtml method of the showTooltipOnTextOverflow directive
            // no need to sanitize as the only outside value is the metric value that is always a float or null
            return new DOMParser().parseFromString(metricDisplayValue, 'text/html').body.innerText || "";
        }

        function shouldDisplayTimeseries(unparsedTimeseriesIdentifier, filters) {
            if (unparsedTimeseriesIdentifier === SINGLE_TIMESERIES_IDENTIFIER) return true;
            const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
            for (let filter in filters) {
                if (!filters[filter].length) continue;
                if (!filters[filter].includes(parsedTimeseriesIdentifier[filter])) {
                    return false;
                }
            }
            return true;
        }

        function getDisplayValue(params, value) {
            if (!params.data) {
                return null;
            }
            return params.data[value.displayName].displayValue;
        }
    });

    app.service("TimeseriesTableService", function(PerTimeseriesService) {
        this.calculateAverageRow = function (gridApi, headers) {

            if (!gridApi || !headers) return [];

            const numericColumns = getNumericColumns(headers);

            if (numericColumns.length === 0 || gridApi.getDisplayedRowCount() === 0) return [];

            let columnValues = {}
            numericColumns.forEach(col => (columnValues[col.rawField] = {values: [], rawName: col.rawName}));

            // Retrieve each values displayed in each numerical column
            gridApi.forEachNodeAfterFilter(node => {
                numericColumns.forEach(col => {
                    if (node.data[col.rawField]) {
                        columnValues[col.rawField].values.push(node.data[col.rawField].rawValue)
                    }
                });
            });

            let averageRow = {}

            // Compute average datas for each columns
            for (const [key, columns] of Object.entries(columnValues)) {

                const validValues = columns.values.filter(val => val !== undefined && !Number.isNaN(val));

                if (validValues.length === 0) {
                    averageRow[key] = {
                        rawValue: "-",
                        rawStd: "-",
                        displayValue: "-"
                    }
                }

                const mean = validValues.reduce((sum, val) => sum + val, 0) / validValues.length;
                const variance = validValues.reduce((sum, val) => sum + Math.pow(val - mean, 2), 0) / validValues.length;
                const stdDev = Math.sqrt(variance);

                averageRow[key] = {
                    rawValue: mean,
                    rawStd: stdDev,
                    displayValue: PerTimeseriesService.createMetricRow(mean, stdDev / 2, columns.rawName, 2)
                }
            }
            return [averageRow]
        }

        function getNumericColumns(headers) {
            return headers
                .filter(col => col.type === "rightAligned")
                .map(col => ({
                    rawField: col.rawField,
                    rawName: col.rawName
                }));
        }
    });

    app.controller("TimeseriesPMLReportForecastController", function($scope, PMLSettings, DataikuAPI, SINGLE_TIMESERIES_IDENTIFIER, PerTimeseriesService, TimeseriesForecastingUtils, $stateParams) {
        $scope.algosWithHiddenQuantiles = [
            // We hide quantiles for baseline models, as they have the same value as the prediction
            ...PMLSettings.algorithmCategories("TIMESERIES_FORECAST")["Baseline Models"],
            ...TimeseriesForecastingUtils.ALGOS_WITHOUT_QUANTILES
        ];
        const isDashboardTile = !!$stateParams.dashboardId;
        $scope.objectId = isDashboardTile ? `insightId.${$scope.insight.id}` : `fullModelId.${$scope.modelData.fullModelId}`;
        $scope.timeseriesGraphFilters = {};
        $scope.allDisplayedTimeseries = {};
        $scope.quantiles = {
            lower: null,
            upper: null
        };

        let forecasts;
        const deregister = $scope.$watch('modelData', function() {
            if (!$scope.modelData) return;

            let getForecastPromise;
            if ($scope.fullModelEvaluationId) {
                getForecastPromise = DataikuAPI.modelevaluations.getForecasts($scope.fullModelEvaluationId);
            } else {
                getForecastPromise = DataikuAPI.ml.prediction.getForecasts($scope.modelData.fullModelId);
            }
            getForecastPromise.success(function(data) {
                forecasts = data;
                const algo = $scope.modelData.actualParams?.resolved?.algorithm || $scope.modelData.modeling.algorithm;
                if (!$scope.algosWithHiddenQuantiles.includes(algo) && $scope.modelData.coreParams.quantilesToForecast.length > 1) {
                    $scope.quantiles.lower = Math.min(...$scope.modelData.coreParams.quantilesToForecast);
                    $scope.quantiles.upper = Math.max(...$scope.modelData.coreParams.quantilesToForecast);
                }

                const unparsedTimeseriesArray = Object.keys(forecasts.perTimeseries);
                if (unparsedTimeseriesArray.length > 1) {
                    $scope.allTimeseriesIdentifierValuesMap = PerTimeseriesService.initTimeseriesIdentifiersValues($scope.modelData.coreParams.timeseriesIdentifiers);
                    unparsedTimeseriesArray.forEach(function(unparsedTimeseriesIdentifier) {
                        const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
                        PerTimeseriesService.addIdentifierValues(parsedTimeseriesIdentifier, $scope.modelData.coreParams.timeseriesIdentifiers, $scope.allTimeseriesIdentifierValuesMap);
                    });
                    PerTimeseriesService.removeDuplicatesAndSortIdentifierValuesForFilterDropdowns($scope.allTimeseriesIdentifierValuesMap);
                }

                $scope.updateDisplayedTimeseries();

                deregister();
            }).error(setErrorInScope.bind($scope));
        });

        $scope.updateDisplayedTimeseries = function() {
            if (!forecasts) return;
            for (const unparsedTimeseriesIdentifier in forecasts.perTimeseries) {
                delete $scope.allDisplayedTimeseries[unparsedTimeseriesIdentifier];
                if (PerTimeseriesService.shouldDisplayTimeseries(unparsedTimeseriesIdentifier, $scope.timeseriesGraphFilters)) {
                    $scope.allDisplayedTimeseries[unparsedTimeseriesIdentifier] = forecasts.perTimeseries[unparsedTimeseriesIdentifier];
                }
            }
        };

        $scope.$on('timeseriesIdentifiersFiltersUpdated', function(event, filters) {
            $scope.timeseriesGraphFilters = filters;
            $scope.updateDisplayedTimeseries();
        });

        $scope.displayQuantilesSelector = function() {
            const algo = $scope.modelData.actualParams?.resolved?.algorithm || $scope.modelData.modeling.algorithm;
            return (
                !$scope.algosWithHiddenQuantiles.includes(algo) &&
                $scope.modelData.coreParams.quantilesToForecast.length > 1
            );
        }

        $scope.showNotAllTimeseriesDisplayedWarning = function() {
            if ($scope.modelData
                && $scope.modelData.iperf
                && $scope.modelData.iperf.totalNbOfTimeseries
                && forecasts
                && forecasts.perTimeseries) {
                $scope.nbOfTimeseriesInForecastCharts = Object.keys(forecasts.perTimeseries).length
                return $scope.nbOfTimeseriesInForecastCharts < $scope.modelData.iperf.totalNbOfTimeseries;
            }
            return false;
        }
    });

    app.controller('TimeseriesPMLReportResamplingController', function($scope, Assert, TimeseriesForecastingUtils) {
        Assert.inScope($scope, 'modelData');
        $scope.timestepParams = $scope.modelData.coreParams.timestepParams;
        $scope.timeseriesSampling = $scope.modelData.preprocessing.timeseriesSampling;
        $scope.numericalInterpolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.numericalInterpolateMethod
        );
        $scope.numericalExtrapolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.numericalExtrapolateMethod
        );
        $scope.categoricalImputeMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.categoricalImputeMethod
        );
        $scope.duplicateTimestampsHandlingMethod = TimeseriesForecastingUtils.DUPLICATE_TIMESTAMPS_HANDLING_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.duplicateTimestampsHandlingMethod
        );
        $scope.prettyTimeSteps = TimeseriesForecastingUtils.prettyTimeSteps;
        $scope.prettySelectedDate = TimeseriesForecastingUtils.prettySelectedDate;
        $scope.getWeekDayName = TimeseriesForecastingUtils.getWeekDayName;
    });

    app.service("TimeseriesGraphsService", function($filter) {
        const COLORS = [
            { line: "#1F77B4", area: "#AEC7E8" },
            { line: "#FF7F0E", area: "#FFBB78" },
            { line: "#2CA02C", area: "#98DF8A" },
            { line: "#D62728", area: "#FF9896" },
            { line: "#9467BD", area: "#C5B0D5" },
            { line: "#8C564B", area: "#C49C94" },
            { line: "#E377C2", area: "#F7B6D2" },
            { line: "#7F7F7F", area: "#C7C7C7" },
            { line: "#BCBD22", area: "#DBDB8D" },
            { line: "#17BECF", area: "#9EDAE5" },
            { line: "#393B79", area: "#5254A3" },
            { line: "#637939", area: "#8CA252" },
            { line: "#6B6ECF", area: "#9C9EDE" },
            { line: "#843C39", area: "#AD494A" },
            { line: "#7B4173", area: "#A55194" },
        ];

        function createSeries(id, data, color, name, quantileAsMarkline) {
            return {
                type: 'line',
                symbol: 'circle',
                emphasis: { scale: false },
                // zlevel is 0 by default; if quantiles are displayed as marklines,
                // with zlevel = 1 the data point is drawn in front instead of hidden
                zlevel: quantileAsMarkline ? 1 : 0,
                data,
                id,
                name,
                color,
            };
        }

        function createQuantileSeries(id, lowerQuantileSeriesData, upperQuantileSeriesData, color, name, singleDataPoint) {
            const lowerQuantileSeries = {
                type: 'line',
                data: lowerQuantileSeriesData,
                lineStyle: { opacity: 0 },
                stack: name,
                id: `lower-` + id,
                symbol: 'none',
            };

            const upperQuantileSeries = {
                type: 'line',
                data: upperQuantileSeriesData,
                lineStyle: { opacity: 0 },
                areaStyle: { color },
                stack: name,
                stackStrategy: 'all',
                symbol: 'none',
                emphasis: { disabled: true },
                name,
                id: `upper-` + id,
                color,
            };

            if (singleDataPoint) {
                // Quantile series have one data point, so series-line.stack does not work,
                // we will use a markline instead.

                // When singleDataPoint is true, lower quantile (resp. upper quantile) is an array of
                // one data point, i.e. lowerQuantileSeriesData = [ [ date, y ] ]. So we retrieve the
                // first and only value in the array & then spread it to retrieve date (x) & y.
                const [date, lowerY, _, upperY] = [
                    ...lowerQuantileSeriesData[0],
                    ...upperQuantileSeriesData[0]
                ];
                upperQuantileSeries.markLine = {
                    symbol: 'none',
                    silent: true,
                    lineStyle: { type: 'solid', width: 4, color },
                    data: [[
                        { xAxis: date, yAxis: lowerY },
                        { xAxis: date, yAxis: upperY },
                    ]],
                };
            }
            return [lowerQuantileSeries, upperQuantileSeries];
        }

        function toFixed(number, precision) {
            const fixedNumber = typeof number == 'number' ? number.toFixed(precision) : number;
            const res = parseFloat(fixedNumber);
            return Number.isNaN(res) ? "-" : res;
        }

        function getBaseChartOptions(largeContainer, timeseriesIdentifier, timeVariable, targetVariable, minInterval, firstDate, lastDate, disableInsideZoom) {
            const chartOptions = {
                title: {
                    show: largeContainer,
                    textAlign: "left",
                    text: $filter('displayTimeseriesName')(timeseriesIdentifier),
                    textStyle: { fontWeight: 'bold', fontSize: 16 }
                },
                legend: {
                    show: largeContainer,
                    selectedMode: true,
                    padding: 8,
                    type: "scroll",
                    y: 16,
                    x: "center",
                    width: "75%",
                    textStyle: { lineOverflow: "truncate", width: 184, overflow: "truncate", lineHeight: 32 }
                },
                // TOOLTIP independant now
                textStyle: { fontFamily: 'SourceSansPro' },
                toolbox: {
                    show: largeContainer,
                    top: 16,
                    right: 64,
                    feature: {
                        restore: {
                            title: 'Reset zoom',
                            emphasis: {
                                iconStyle: { textPosition: 'top' }
                            }
                        },
                    }
                },
                xAxis: {
                    axisTick: { show: largeContainer },
                    axisLine: { onZero: false, show: largeContainer },
                    axisLabel: {
                        rotate: 45,
                        show: largeContainer,
                        formatter: {
                            year: '{MMM} {dd}, {yyyy}',
                            month: '{MMM} {dd}, {yyyy}',
                            day: '{MMM} {dd}, {yyyy}',
                            hour: '{MMM} {dd}, {yyyy} \n{HH}:{mm}',
                            minute: '{MMM} {dd}, {yyyy} \n{HH}:{mm}',
                            second: '{MMM} {dd}, {yyyy} \n{HH}:{mm}:{ss}',
                            millisecond: '{MMM} {dd}, {yyyy} \n{HH}:{mm}:{ss} {SSS}'
                        }
                    },
                    name: timeVariable,
                    nameGap: 72,
                    nameLocation: "middle",
                    type: "time",
                    nameTextStyle: { overflow: "truncate", width: 264, fontWeight: 'bold' }
                },
                yAxis: {
                    axisLabel: { formatter: val => toFixed(val, 3) },
                    axisTick: { show: largeContainer },
                    scale: true,
                    name: targetVariable,
                    nameGap: 48,
                    nameLocation: "middle",
                    type: "value",
                    nameTextStyle: { overflow: "truncate", width: 320, fontWeight: 'bold' }
                },
                animation: largeContainer,
            }

            if (largeContainer) {
                const padToLengthTwo = dateElem => dateElem.toString().padStart(2, '0');
                chartOptions.grid = { bottom: 136, right: 64, left: 72 };
                chartOptions.dataZoom = [
                    {
                        type: 'inside',
                        minValueSpan: 3 * minInterval,
                        startValue: new Date(firstDate),
                        endValue: new Date(lastDate),
                        disabled: disableInsideZoom
                    },
                    {
                        type: "slider",
                        startValue: new Date(firstDate),
                        endValue: new Date(lastDate),
                        left: 64,
                        right: 64,
                        textStyle: { fontSize: 11 },
                        labelFormatter: function(timestamp) {
                            const date = new Date(timestamp);
                            const year = date.getFullYear();
                            const month = padToLengthTwo(date.getMonth() + 1);
                            const day = padToLengthTwo(date.getDate());
                            const hour = padToLengthTwo(date.getHours());
                            const minute = padToLengthTwo(date.getMinutes());
                            const second = padToLengthTwo(date.getSeconds());
                            return `${year}-${month}-${day}\n${hour}:${minute}:${second}`;
                        }
                    }
                ];
            } else {
                chartOptions.grid = { top: 8, bottom: 8, right: 8, left: 8 };
            }

            return chartOptions
        }

        function tooltipContentForSeries(seriesName, displayedValue, color) {
            return `<i class='icon-circle mright8' style='color: ${color}'></i>
                        ${seriesName}: ${displayedValue}<br/>`;
        };

        function insertGaps(data, gapDurationThresholdMs) {
            // Insert gaps between discontiguous intervals to avoid interpolation in echart
            const dataWithGaps = [];
            for (let dataPointIndex = 0; dataPointIndex < data.length - 1; dataPointIndex++) {
                const currentDataPoint = data[dataPointIndex];
                const nextDataPoint = data[dataPointIndex + 1];
                dataWithGaps.push(currentDataPoint);
                if (new Date(nextDataPoint[0]) - new Date(currentDataPoint[0]) > gapDurationThresholdMs) {
                    dataWithGaps.push([null, null]); // insert gap
                }
            }
            dataWithGaps.push(data[data.length - 1]);
            return dataWithGaps;
        }


        return {
            COLORS,
            createQuantileSeries,
            createSeries,
            getBaseChartOptions,
            toFixed,
            tooltipContentForSeries,
            insertGaps
        }
    })

    /**
     * Directive for the forecasting charts (used in model reports & snippets)
     *
     * @param {object} forecasts - the different series (actual, backtest, future)
     * @param {string} timeseries - name of the time series
     * @param {object} quantiles - object with properties regarding the quantiles:
     *                             > display: whether to display the quantiles
     *                             > lower & upper: the lower & upper quantiles to display
     *                             (only for reports, where the user can change the default)
     * @param {string} target - (optional) name of the target var. Only used for reports.
     * @param {string} time - (optional) name of the time var. Only used for reports.
     * @param {object} yAxisRange - (optional) object with properties 'min' & 'max' to define
     *                            the range of the y axis. Only used for snippets, to enforce
     *                            a similar range across the different models.
     * @param {boolean} disableInsideZoom - (optional) whether to enable zooming in the chart.
     *                                    Only used for reports (zoom is never available in snippets).
     */
    app.directive("timeseriesForecastingGraphs", function($filter, Debounce, TimeseriesGraphsService) {
        return {
            scope: {
                forecasts: '<',
                timeseriesIdentifier: '<',
                quantiles: '<',
                horizon: '<',
                gapSize: '<',
                testSize: '<',
                targetVariable: '<?',
                timeVariable: '<?',
                yAxisRange: '<?',
                disableInsideZoom: '<?',
                loadedStateField: '<',
            },
            template: `<div class="h100">
                            <div block-api-error />
                            <ng2-lazy-echart [options]="chartOptions" ng-if="chartOptions" (chart-init)="onChartInit($event)"></ng2-lazy-echart>
                        </div>`,
            restrict: 'A',
            link: function(scope, _elem, attrs) {

                // Allow MDG/puppeteer to find this element, and know when it has loaded
                const puppeteerSelectorName = scope.loadedStateField;
                if (puppeteerSelectorName) {
                    _elem.attr(puppeteerSelectorName, true)

                    const setPuppeteerField = function() {
                        scope[puppeteerSelectorName] = true;
                    };

                    // we debounce here as the echart render event actually gets called multiple times during the animation process -
                    // if we mark ready on the first fire, the screenshot will often be not the fully rendered graph
                    var debouncedSetPuppeteerField = Debounce().withDelay(50, 200).wrap(setPuppeteerField);

                    scope.onChartInit = (echart) => {
                        echart.on('rendered', () => {
                            debouncedSetPuppeteerField();
                        });
                    };
                }

                // eslint-disable-next-line no-prototype-builtins
                const largeContainer = attrs.hasOwnProperty("largeContainer");

                // eslint-disable-next-line no-prototype-builtins
                const displayFuture = attrs.hasOwnProperty("displayFuture") && scope.forecasts.futureForecast;
                const toFixed = (number, precision) => parseFloat(number.toFixed(precision));

                function updateGraph() {
                    if (!scope.forecasts) return;

                    if (puppeteerSelectorName) {
                        scope[puppeteerSelectorName] = false;
                    }

                    const folds = new Set(scope.forecasts.foldId);
                    const nrFolds = folds.size;
                    // Nb. folds always divide nb. timestamps (nb.timestamps in backtest folds = test size * nrFolds)
                    const backtestSize = scope.forecasts.forecast.length / nrFolds;

                    let quantilesToDisplay;
                    if (scope.quantiles && scope.quantiles.display) {
                        let lowerQuantileData;
                        let upperQuantileData;
                        if (angular.isNumber(scope.quantiles.lower)) {
                            lowerQuantileData = scope.forecasts.quantiles.find((elem) => elem.quantile === scope.quantiles.lower);
                        }
                        if (angular.isNumber(scope.quantiles.upper)) {
                            upperQuantileData = scope.forecasts.quantiles.find((elem) => elem.quantile === scope.quantiles.upper);
                        }

                        if (!lowerQuantileData || !upperQuantileData) {
                            const sortedQuantiles = scope.forecasts.quantiles.sort((a, b) => a.quantile - b.quantile);
                            lowerQuantileData = lowerQuantileData || sortedQuantiles[0];
                            upperQuantileData = upperQuantileData || sortedQuantiles[sortedQuantiles.length - 1];
                        }

                        if (upperQuantileData.quantile > lowerQuantileData.quantile) {
                            const lowestQuantileValue = scope.yAxisRange && scope.yAxisRange.min || Math.min(
                                ...lowerQuantileData.forecast, ...(displayFuture ? lowerQuantileData.futureForecast : [])
                            );

                            quantilesToDisplay = {
                                legend: `[${toFixed(lowerQuantileData.quantile, 3)}, ${toFixed(upperQuantileData.quantile, 3)}] interval`,
                                lower: {
                                    value: lowerQuantileData.quantile,
                                    backTestData: lowerQuantileData.forecast.map(function(data, idx) {
                                        return [scope.forecasts.forecastTime[idx], data];
                                    })
                                },
                                upper: {
                                    value: upperQuantileData.quantile,
                                    backTestData: upperQuantileData.forecast.map(function(data, idx) {
                                        if (backtestSize === 1) {
                                            // Quantiles are marklines; simply use upper quantile value
                                            return [scope.forecasts.forecastTime[idx], data];
                                        }
                                        // Quantiles are areas (using series-line.stack); use the
                                        // difference between lower & upper quantile values
                                        return [scope.forecasts.forecastTime[idx], data - lowerQuantileData.forecast[idx]];
                                    })
                                }
                            }

                            if (displayFuture) {
                                quantilesToDisplay.lower.forecastData = lowerQuantileData.futureForecast.map(function(data, idx) {
                                    return [scope.forecasts.futureTime[idx], data];
                                });
                                quantilesToDisplay.upper.forecastData = upperQuantileData.futureForecast.map(function(data, idx) {
                                    if (scope.forecasts.futureForecast.length === 1) {
                                        // Quantiles are marklines; simply use upper quantile value
                                        return [scope.forecasts.futureTime[idx], data];
                                    }
                                    // Quantiles are areas (using series-line.stack); use the
                                    // difference between lower & upper quantile values
                                    return [scope.forecasts.futureTime[idx], data - lowerQuantileData.futureForecast[idx]];
                                });
                            }
                        }
                    }

                    let actualData;
                    if (largeContainer) {
                        actualData = scope.forecasts.groundTruth.map(function(data, idx) {
                            return [scope.forecasts.groundTruthTime[idx], data];
                        });
                        if (scope.forecasts.futureForecastContext != undefined) {
                            let forecastContextData = scope.forecasts.futureForecastContext.map(function(data, idx) {
                                return [scope.forecasts.futureForecastContextTime[idx], data];
                            });
                            // Future forecast context contains dataset data whenever the evaluation fold and the forecast
                            // periods are discontiguous. We provide at most 100 context timesteps. If it exceeds that value,
                            // we therefore need to add a line break in the series display.
                            if (scope.forecasts.futureForecastContext.length >= 100) {
                                actualData = [...actualData, [undefined, undefined], ...forecastContextData];
                            } else {
                                actualData = [...actualData, ...forecastContextData];
                            }
                        }
                    } else {
                        const minForecastTime = new Date(Math.min(...scope.forecasts.forecastTime.map(forecastTime => new Date(forecastTime))))
                        let timeWindowLength = scope.forecasts.groundTruthTime.filter(date => new Date(date) >= minForecastTime).length
                        if (timeWindowLength > scope.forecasts.forecastTime.length) {
                            // When there are more ground truth data than forecast data
                            // Do not display all past data in snippets (to make graphs more readable)
                            // Only timestamps starting from the first timestamp with backtest data + two more extra timestamps in case we have too few data points
                            timeWindowLength += 2;
                        }
                        actualData = scope.forecasts.groundTruth.slice(-timeWindowLength).map(function(data, idx) {
                            return [
                                scope.forecasts.groundTruthTime[idx + scope.forecasts.groundTruthTime.length - timeWindowLength],
                                data
                            ];
                        });
                    }


                    if (actualData && actualData.length > 2) {
                        // We assume first 2 datapoints are within the same interval,
                        // and duration between them is expected duration between points within an interval
                        const expectedDurationBetweenDatapoints = new Date(actualData[1][0]) - new Date(actualData[0][0]);
                        // We insert gaps (discontinuity in the line) if 2 points are spaced with more than 2 times
                        // the expected duration between 2 points
                        const gapDurationThresholdMs = expectedDurationBetweenDatapoints * 2;
                        actualData = TimeseriesGraphsService.insertGaps(actualData, gapDurationThresholdMs);
                    }

                    const backtestData = scope.forecasts.forecast.map(function(data, idx) {
                        return [scope.forecasts.forecastTime[idx], data];
                    });

                    const forecastData = (scope.forecasts.futureForecast || []).map(function(data, idx) {
                        return [scope.forecasts.futureTime[idx], data];
                    });

                    const quantilesAsMarklineForBacktest = backtestSize === 1 && !!quantilesToDisplay;
                    const quantilesAsMarklineForForecast = displayFuture && scope.forecasts.futureForecast.length === 1 && !!quantilesToDisplay;
                    const series = [TimeseriesGraphsService.createSeries('actual', actualData, '#666666', 'Actual', quantilesAsMarklineForBacktest)];

                    const legend = [{ name: "Actual", itemStyle: { opacity: 0 } }];

                    for (let fold of folds) {
                        const firstIdx = scope.forecasts.foldId.indexOf(fold);
                        const lastIdx = scope.forecasts.foldId.lastIndexOf(fold);
                        const foldId = fold + 1;
                        const suffix = nrFolds > 1 ? (' (fold ' + (fold + 1) + ')') : '';
                        const color = TimeseriesGraphsService.COLORS[fold % TimeseriesGraphsService.COLORS.length];

                        const foldData = backtestData.slice(firstIdx, lastIdx + 1);
                        series.push(TimeseriesGraphsService.createSeries(
                            `fold-${foldId}`,
                            foldData,
                            color.line,
                            "Backtest" + suffix,
                            quantilesAsMarklineForBacktest
                        ));
                        legend.push({ name: "Backtest" + suffix, itemStyle: { opacity: 0 } });

                        if (quantilesToDisplay) {
                            const quantileAreaName = quantilesToDisplay.legend + suffix;
                            const lowerQuantileFoldData = quantilesToDisplay.lower.backTestData.slice(firstIdx, lastIdx + 1);
                            const upperQuantileFoldData = quantilesToDisplay.upper.backTestData.slice(firstIdx, lastIdx + 1);

                            series.push(...TimeseriesGraphsService.createQuantileSeries(
                                `quantile-fold-${fold}`, lowerQuantileFoldData, upperQuantileFoldData, color.area, quantileAreaName, quantilesAsMarklineForBacktest
                            ));

                            legend.push({ name: quantileAreaName, icon: 'roundRect' });
                        }
                    }

                    if (displayFuture) {
                        const color = TimeseriesGraphsService.COLORS[nrFolds % TimeseriesGraphsService.COLORS.length];
                        series.push(TimeseriesGraphsService.createSeries(
                            'forecast',
                            forecastData,
                            color.line,
                            'Forecast',
                            quantilesAsMarklineForForecast
                        ));
                        legend.push({ name: "Forecast", itemStyle: { opacity: 0 } });

                        if (quantilesToDisplay) {
                            const quantileAreaName = quantilesToDisplay.legend + " (forecast)";
                            series.push(...TimeseriesGraphsService.createQuantileSeries(
                                `quantile-forecast`,
                                quantilesToDisplay.lower.forecastData, quantilesToDisplay.upper.forecastData,
                                color.area, quantileAreaName, quantilesAsMarklineForForecast
                            ));
                            legend.push({ name: quantileAreaName, symbol: 'line' });
                        }
                    }

                    const tooltipContentForSeries = TimeseriesGraphsService.tooltipContentForSeries;

                    function getStepAndHorizonTooltipLine(horizonStep, displayHorizonNb, displayGap = true, prefix = '') {
                        let tooltipLine = "<span class='text-debug'>"
                        if (prefix !== '') {
                            tooltipLine += `${prefix} - `;
                        }
                        if (displayHorizonNb) {
                            const horizonNb = Math.ceil(horizonStep / scope.horizon);
                            tooltipLine += `Horizon ${horizonNb} - `;
                        }
                        const stepNb = (horizonStep - 1) % scope.horizon + 1;
                        const isGapStep = stepNb <= scope.gapSize;
                        tooltipLine += `Step ${stepNb}${isGapStep && displayGap ? ' (ignored for evaluation)' : ''}</span><br/>`;
                        return tooltipLine;
                    }

                    function getQuantileIntervalTooltipLine(lowerQuantilesSeries, upperQuantilesSeries, quantilesAsMarkline) {
                        if (quantilesAsMarkline) {
                            return `[${toFixed(lowerQuantilesSeries.data[1], 3)}, ${toFixed(upperQuantilesSeries.data[1], 3)}]`
                        }
                        return `[${toFixed(lowerQuantilesSeries.data[1], 3)}, ${toFixed(lowerQuantilesSeries.data[1] + upperQuantilesSeries.data[1], 3)}]`;
                    }

                    function getHorizonStep(foldId, backtestSeries) {
                        // Snippet can be clipped, and will contain last forecasted values
                        // forecasts.horizonStep only available in snippet response
                        if (scope.forecasts.horizonStep !== undefined) {
                            const firstIdx = scope.forecasts.foldId.indexOf(foldId - 1); // backend fold id are 0 based
                            const horizonStepIndex = firstIdx + backtestSeries.dataIndex
                            return scope.forecasts.horizonStep[horizonStepIndex]
                        } else {
                            return backtestSeries.dataIndex + 1;
                        }
                    }

                    const nrGroundTruthValues = scope.forecasts.groundTruthTime.length;
                    const pastGroundTruthTime = scope.forecasts.groundTruthTime.filter(date => date < scope.forecasts.forecastTime[0]);

                    const firstDate = pastGroundTruthTime[Math.max(0, pastGroundTruthTime.length - backtestSize)];
                    const lastDate = displayFuture ? scope.forecasts.futureTime[scope.forecasts.futureTime.length - 1] : scope.forecasts.groundTruthTime[nrGroundTruthValues - 1];
                    const minInterval = new Date(scope.forecasts.groundTruthTime[1]) - new Date(scope.forecasts.groundTruthTime[0]);

                    let chartOptions = TimeseriesGraphsService.getBaseChartOptions(largeContainer, scope.timeseriesIdentifier, scope.timeVariable, scope.targetVariable, minInterval, firstDate, lastDate);
                    const tooltip = {
                        trigger: 'axis',
                        textStyle: { fontSize: 13 },
                        appendToBody: true,
                        formatter: function(params) {
                            // In the tooltip there can either be:
                            // - only one series (actual or forecast)
                            // - two series (actual + backtest) - can be multiplied by number of fold
                            // - three series (forecast + lower quantile + upper quantile)
                            // - four series (actual + backtest + lower quantile + upper quantile) - can be multiplied by number of fold

                            let tooltipContent = `<span class="font-weight-bold">${params[0].data[0]}</span><br/>`;

                            let currentIntervalType = ''
                            for (let index = 0; index < params.length; index++) {
                                if (params[index].seriesId.startsWith('actual')) {
                                    const actualSeries = params[index];
                                    tooltipContent += tooltipContentForSeries(actualSeries.seriesName, toFixed(actualSeries.data[1], 3), actualSeries.color);
                                } else if (params[index].seriesId.startsWith('fold-')) {
                                    const backtestSeries = params[index];
                                    const foldId = backtestSeries.seriesId.replace('fold-', '');
                                    const testSizeHasMultipleHorizons = backtestSize / scope.horizon > 1;
                                    let horizonTooltipPrefix = (folds.size > 1 ? `Fold ${foldId}` : '');
                                    const horizonStep = getHorizonStep(foldId, backtestSeries);
                                    tooltipContent += getStepAndHorizonTooltipLine(horizonStep, testSizeHasMultipleHorizons, true, horizonTooltipPrefix)
                                    tooltipContent += tooltipContentForSeries(backtestSeries.seriesName, toFixed(backtestSeries.data[1], 3), backtestSeries.color);
                                    currentIntervalType = 'Backtest'
                                } else if (params[index].seriesId.startsWith('forecast')) {
                                    const forecastSeries = params[index];
                                    if (params.length > 1) { // not only the forecast
                                        tooltipContent += getStepAndHorizonTooltipLine(forecastSeries.dataIndex + 1, false, false, 'Forecast')
                                    }
                                    tooltipContent += tooltipContentForSeries(forecastSeries.seriesName, toFixed(forecastSeries.data[1], 3), forecastSeries.color);
                                    currentIntervalType = 'Forecast'
                                } else if (params[index].seriesId.startsWith('lower-quantile-')
                                    && (index + 1) < params.length && params[index + 1].seriesId.startsWith('upper-quantile-')) {
                                    const lowerQuantilesSeries = params[index];
                                    const upperQuantilesSeries = params[index + 1];
                                    tooltipContent += tooltipContentForSeries(
                                        `${currentIntervalType} interval`, getQuantileIntervalTooltipLine(lowerQuantilesSeries, upperQuantilesSeries, quantilesAsMarklineForBacktest), upperQuantilesSeries.color
                                    );
                                    index++; // skip next which is upper quantile
                                }
                            }
                            return tooltipContent;
                        }
                    };
                    if (folds.size > 1) {
                        // Make it enterable and scrollable for multiple folds tooltip
                        tooltip.enterable = true;
                        tooltip.extraCssText = 'max-height: 160px; overflow:auto';
                    }
                    chartOptions.legend.data = legend;
                    chartOptions = { ...chartOptions, series, tooltip }


                    if (scope.yAxisRange) {
                        chartOptions.yAxis = {
                            min: scope.yAxisRange.min,
                            max: scope.yAxisRange.max,
                            interval: (scope.yAxisRange.max - scope.yAxisRange.min) / 5
                        }
                    }

                    scope.chartOptions = chartOptions;
                }

                scope.$watchGroup(["yAxisRange", "disableInsideZoom", "quantiles"], function() {
                    // For quantiles: We only need a shallow $watch (which is what $watchGroup provides),
                    // not a $watchCollection (~ deep watch), because the quantiles object is defined
                    // in the html -> Creates a new object when properties change.
                    updateGraph();
                });
            }
        };
    });

    app.component("timeseriesInteractiveScenarioGraph", {
            templateUrl: '/templates/ml/prediction-model/timeseries/interactive-scoring/scenarios-forecasts-graph.html',
            bindings: {
                scenariosForecasts: '<',
                timeseriesIdentifier: '<',
                timeVariable: '<',
                targetVariable: '<',
                scenariosMetadata: '<'
            },
            controller: function (TimeseriesGraphsService) {
                const $ctrl = this;
                function prepareHistoricalData(scenariosForecasts) {
                    const historicalValues = {};
                    let timestepDuration;
                    for (const [_, scenarioData] of Object.entries(scenariosForecasts)) {
                        scenarioData.groundTruthTime.forEach((v, idx) => {
                            if (!timestepDuration) { // Ground truth size is always greater than 1
                                timestepDuration = new Date(scenarioData.groundTruthTime[1]) - new Date(scenarioData.groundTruthTime[0]);
                            }
                            if (!(v in historicalValues)) {
                                historicalValues[v] = scenarioData.groundTruth[idx] ? parseFloat(scenarioData.groundTruth[idx]) : "";
                            }
                        });
                    }
                    return TimeseriesGraphsService.insertGaps(Object.entries(historicalValues).sort(), timestepDuration);
                }

                $ctrl.$onChanges = () => {
                    if (!$ctrl.scenariosForecasts || !$ctrl.scenariosMetadata) return;

                    const scenarioIds = Object.keys($ctrl.scenariosForecasts.perScenarios);
                    const metadataIsReady = scenarioIds.every(id => $ctrl.scenariosMetadata.names[id] !== undefined && $ctrl.scenariosMetadata.colors[id] !== undefined);
                    // scenariosForecasts and scenariosMetadata are distinct objects and therefore we don't want to redraw the graphs until both have been eventually updated.
                    if (!metadataIsReady) return;

                    const groundTruthSeriesData = prepareHistoricalData($ctrl.scenariosForecasts.perScenarios);
                    const series = [TimeseriesGraphsService.createSeries('actual', groundTruthSeriesData, '#666666', 'Historical data', false)];
                    const legend = [{ name: "Historical data", itemStyle: { opacity: 0 } }];

                    scenarioIds.sort().forEach(scenarioId => {
                        const scenarioData = $ctrl.scenariosForecasts.perScenarios[scenarioId];
                        const scenarioForecastData = scenarioData.forecast.map(function(v, idx) {
                            return [scenarioData.forecastTime[idx], v];
                        });
                        const scenarioName = $ctrl.scenariosMetadata["names"][scenarioId];
                        const color = $ctrl.scenariosMetadata["colors"][scenarioId];
                        series.push(TimeseriesGraphsService.createSeries(
                            `scenario-${scenarioName}`,
                            scenarioForecastData,
                            TimeseriesGraphsService.COLORS[color % TimeseriesGraphsService.COLORS.length].line,
                            `Scenario ${scenarioName}`,
                            false
                        ));
                        legend.push({ name: `Scenario ${scenarioName}`, itemStyle: { opacity: 0 } });
                    });

                    $ctrl.chartOptions = TimeseriesGraphsService.getBaseChartOptions(
                        true,
                        $ctrl.timeseriesIdentifier,
                        $ctrl.timeVariable,
                        $ctrl.targetVariable,
                        undefined,
                        undefined,
                        undefined,
                        false
                    );

                    const tooltip = {
                        trigger: 'axis',
                        textStyle: { fontSize: 13 },
                        formatter: function(params) {
                            let tooltipContent = `<span class="font-weight-bold">${params[0].data[0]}</span><br/>`;
                            for (let idx = 0; idx < params.length; idx++) {
                                tooltipContent += TimeseriesGraphsService.tooltipContentForSeries(params[idx].seriesName, TimeseriesGraphsService.toFixed(params[idx].data[1], 3), params[idx].color);
                            }

                            return tooltipContent;
                        }
                    };
                    $ctrl.chartOptions.legend.data = legend
                    $ctrl.chartOptions = { ...$ctrl.chartOptions, series, tooltip };
                }
            }
        }
    )

    app.filter('displayTimeseriesName', function(SINGLE_TIMESERIES_IDENTIFIER) {
        return function(unparsedName) {
            if (unparsedName === SINGLE_TIMESERIES_IDENTIFIER) {
                return "Time series";
            }
            return Object.entries(JSON.parse(unparsedName)).map(entry => entry.join(": ")).join(", ");
        }
    });

    app.component("timeseriesSplitSchema", {
        bindings: {
            kfold: '<',
            nFolds: '<',
            predictionLength: '<',
            gapSize: '<',
            timeunit: '<',
            foldOffset: '<',
            equalDurationFolds: '<',
            splitType: '@',
            customTrainTestSplit: '<',
            customTrainTestIntervals: '<',
        },
        templateUrl: '/templates/analysis/prediction/timeseries/split-schema.html',
        controller: function($location, $anchorScroll) {
            const ctrl = this;
            ctrl.scrollToHeader = function(elementId) {
                $location.hash(elementId);
                $anchorScroll();
            };
        }
    });

    app.component("timeseriesSplitSchemaBar", {
        bindings: {
            nFolds: '<',
            predictionLength: '<',
            timeunit: '<',
            foldOffset: '<',
            equalDurationFolds: '<',
            splitType: '<',
            foldId: '<',
            customTrainTestSplit: '<',
            customTrainTestIntervals: '<',
        },
        templateUrl: '/templates/analysis/prediction/timeseries/split-schema-bar.html',
        controller: function ($attrs) {
            const ctrl = this;
            ctrl.kfold = $attrs.hasOwnProperty('kfold');
            ctrl.equalDurationFolds = $attrs.hasOwnProperty('equalDurationFolds');

            function getGlobalCustomIntervalsMinDate() {
                return new Date(Math.min(...ctrl.customTrainTestIntervals.map(i => new Date(i.train[0]))));
            }

            function getGlobalCustomIntervalsMaxDate() {
                return new Date(Math.max(...ctrl.customTrainTestIntervals.map(i => new Date(i.test[1]))));
            }

            function getGlobalCustomIntervalsDurationMs() {
                const minDate = getGlobalCustomIntervalsMinDate();
                const maxDate = getGlobalCustomIntervalsMaxDate();
                return maxDate - minDate;
            }

            function getGapThresholdMs(timeunit) {
                // Return 2 times the timeunit as milliseconds
                switch (timeunit) {
                    case 'YEAR':
                        return 366 * 24 * 60 * 60 * 1000 * 2
                    case 'HALF_YEAR':
                        return 133 * 24 * 60 * 60 * 1000 * 2
                    case 'QUARTER':
                        return 92 * 24 * 60 * 60 * 1000 * 2
                    case 'MONTH':
                        return 31 * 24 * 60 * 60 * 1000 * 2
                    case 'WEEK':
                        return 7 * 24 * 60 * 60 * 1000 * 2
                    case 'BUSINESS_DAY':
                    case 'DAY':
                        return 24 * 60 * 60 * 1000 * 2
                    case 'HOUR':
                        return 60 * 60 * 1000 * 2
                    case 'MINUTE':
                        return 60 * 1000 * 2
                    case 'SECOND':
                        return 1000 * 2
                    case 'MILLISECOND':
                    default:
                        return 2
                }
            }

            function getSpaceBeforeTrainWidth() {
                // Gives the width of evaluation set of the ignored part to the left of the dataset for each bar
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') {
                    let intervalTrainStart = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].train[0]);
                    const intervalBetweenGlobalStartAndTrainStart = intervalTrainStart - getGlobalCustomIntervalsMinDate();
                    const isGap = intervalBetweenGlobalStartAndTrainStart > 0;
                    const widthPercent = isGap ? Math.max(Math.floor(intervalBetweenGlobalStartAndTrainStart * 100 / getGlobalCustomIntervalsDurationMs()), 2) : 0;
                    return 'width:' + widthPercent + '%';
                } else {
                    let offset = 0;
                    if (ctrl.kfold && ctrl.equalDurationFolds && ctrl.foldId > 1) {
                        offset = (ctrl.foldOffset ? 2 : 1) * (ctrl.foldId - 1)
                    }
                    return 'width: calc(' + offset + '* var(--doctor-evaluation-forecast-width))'
                }

            };

            function getTrainWidth() {
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') {
                    const intervalTrainStart = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].train[0]);
                    const intervalTrainEnd = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].train[1]);
                    const trainDurationMs = intervalTrainEnd - intervalTrainStart;
                    const widthPercent = Math.max(Math.floor(trainDurationMs * 100 / getGlobalCustomIntervalsDurationMs()), 2);
                    return 'flex: 0 1 ' + widthPercent + '%';
                } else {
                    return 'min-width: calc(var(--doctor-evaluation-forecast-width))';
                }
            }

            function getSpaceBetweenTrainAndEvaluationWidth() {
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') {
                    const intervalTrainEnd = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].train[1]);
                    const intervalEvaluationStart = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].test[0]);
                    const durationBetweenTrainAndEvaluationMs = intervalEvaluationStart - intervalTrainEnd;
                    const isGap = durationBetweenTrainAndEvaluationMs >= getGapThresholdMs(ctrl.timeunit)
                    const widthPercent = isGap ? Math.max(Math.floor(durationBetweenTrainAndEvaluationMs * 100 / getGlobalCustomIntervalsDurationMs()), 2) : 0;
                    return 'width:' + widthPercent + '%';
                } else {
                    return 'width: calc(var(--doctor-evaluation-forecast-width))';
                }
            }

            function getEvaluationWidth() {
                const minWidth = 'min-width: 16px';
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') {
                    const intervalEvaluationStart = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].test[0]);
                    const intervalEvaluationEnd = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].test[1]);
                    const evaluationDurationMs = intervalEvaluationEnd - intervalEvaluationStart;
                    const widthPercent = Math.max(Math.floor(evaluationDurationMs * 100 / getGlobalCustomIntervalsDurationMs()), 2);

                    const intervalBetweenGlobalEndAndTestEnd = getGlobalCustomIntervalsMaxDate() - intervalEvaluationEnd;
                    const hasGapAfter = intervalBetweenGlobalEndAndTestEnd > 0;
                    if (!hasGapAfter) {
                        // Allow width growth to balance min space when gap on other test intervals
                        return minWidth + ';flex: 1 0 ' + widthPercent + '%';
                    } else {
                        return minWidth + ';width:' + widthPercent + '%';
                    }
                } else {
                    return minWidth + ';width: calc(var(--doctor-evaluation-forecast-width))';
                }
            }

            function getSpaceAfterEvaluationWidth() {
                // Gives the width of evaluation set of the ignored part to the right of the dataset for each bar
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') {
                    const intervalTestEnd = new Date(ctrl.customTrainTestIntervals[ctrl.foldId - 1].test[1]);
                    const intervalBetweenGlobalEndAndTestEnd = getGlobalCustomIntervalsMaxDate() - intervalTestEnd;
                    const isGap = intervalBetweenGlobalEndAndTestEnd > 0;
                    const widthPercent = isGap ? Math.max(Math.floor(intervalBetweenGlobalEndAndTestEnd * 100 / getGlobalCustomIntervalsDurationMs()), 2) : 0;
                    return 'width:' + widthPercent + '%';
                } else {
                    let offset = 0;
                    if (ctrl.kfold && ctrl.foldId !== ctrl.nFolds) {
                        if (ctrl.foldId === 1) {
                            offset = (ctrl.foldOffset ? 2 : 1) * (ctrl.nFolds - 1);
                        } else if (ctrl.foldId === ctrl.nFolds - 1) {
                            offset = ctrl.foldOffset ? 2 : 1;
                        }
                    }
                    return 'flex: 0 0 auto; width: calc(' + offset + '* var(--doctor-evaluation-forecast-width))'
                }
            }

            function updateBar() {
                ctrl.spaceBeforeTrainWidth = getSpaceBeforeTrainWidth();
                ctrl.trainWidth = getTrainWidth();
                ctrl.spaceBetweenTrainAndEvaluationWidth = getSpaceBetweenTrainAndEvaluationWidth();
                ctrl.evaluationWidth = getEvaluationWidth();
                ctrl.spaceAfterEvaluationWidth = getSpaceAfterEvaluationWidth();
            }

            ctrl.$onChanges = function (changes) {
                updateBar();
            };
        }
    });

    app.factory("TimeseriesForecastingUtils", function($filter) {
        const TIME_UNITS = {
            MILLISECOND: "Millisecond",
            SECOND: "Second",
            MINUTE: "Minute",
            HOUR: "Hour",
            DAY: "Day",
            BUSINESS_DAY: "Business day",
            WEEK: "Week",
            MONTH: "Month",
            QUARTER: "Quarter",
            HALF_YEAR: "Half year",
            YEAR: "Year"
        };

        const TIMESERIES_IMPUTE_METHODS = [
            {
                displayName: "Nearest",
                value: "NEAREST",
                description: "Value of the nearest date between previous and next",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Previous",
                value: "PREVIOUS",
                description: "Value of the previous date",
                featureTypes: ['num', 'non-num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Next",
                value: "NEXT",
                description: "Value of the next date",
                featureTypes: ['num', 'non-num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Staircase",
                value: "STAIRCASE",
                description: "Average between previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Linear",
                value: "LINEAR",
                description: "Time-sensitive average between previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: true
            },
            {
                displayName: "Quadratic",
                value: "QUADRATIC",
                description: "Quadratic interpolation of previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: true
            },
            {
                displayName: "Cubic",
                value: "CUBIC",
                description: "Cubic interpolation of previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: true
            },
            {
                displayName: "Constant",
                value: "CONSTANT",
                description: "Constant value",
                featureTypes: ['num', 'non-num'],
                interpolation: true,
                extrapolation: true
            },

            {
                displayName: "Previous or next",
                value: "PREVIOUS_NEXT",
                description: "Previous date (for future extrapolated values), or next date (for past values)",
                featureTypes: ['num', 'non-num'],
                extrapolation: true
            },
            { displayName: "No extrapolation", value: "NO_EXTRAPOLATION", description: "No extrapolation", featureTypes: ['num'], extrapolation: true },

            { displayName: "No value", value: "NULL", description: "No extrapolated value", featureTypes: ['non-num'] },
            { displayName: "Most common", value: "MOST_COMMON", description: "Most common value", featureTypes: ['non-num'] }
        ];

        const DUPLICATE_TIMESTAMPS_HANDLING_METHODS = [
            { displayName: "Fail on conflicting duplicates", value: "FAIL_IF_CONFLICTING" },
            { displayName: "Drop all conflicting duplicates", value: "DROP_IF_CONFLICTING" },
            { displayName: "Use mean (resp. mode) of numerical (resp. categorical) columns", value: "MEAN_MODE" }
        ];

        // To be synced with:
        // java -> com/dataiku/dip/analysis/model/prediction/TimeseriesForecastingModelDetails.java L.43
        // doctor -> dataiku/doctor/timeseries/models/__init__.py L.34
        const ALGOS_WITHOUT_EXTERNAL_FEATURES = {
            keys: [
                "trivial_identity_timeseries",
                "seasonal_naive_timeseries",
                "seasonal_loess_timeseries",
                "croston_timeseries",
                "gluonts_simple_feed_forward_timeseries",
                "gluonts_torch_simple_feed_forward_timeseries",
                "ets_timeseries",
            ],
            names: [
                "TRIVIAL_IDENTITY_TIMESERIES",
                "SEASONAL_NAIVE",
                "CROSTON",
                "SEASONAL_LOESS",
                "GLUONTS_SIMPLE_FEEDFORWARD",
                "GLUONTS_TORCH_SIMPLE_FEEDFORWARD",
                "ETS"
            ]
        };

        const ALGOS_TIMESERIES_CLASSICAL_ML = [
            "random_forest_regression",
            "ridge_regression",
            "xgboost_regression",
            "lightgbm_regression"
        ];

        const ALGOS_WITHOUT_QUANTILES = [
            "CROSTON",
            ...ALGOS_TIMESERIES_CLASSICAL_ML.map(algoKey => algoKey.toUpperCase())
        ];

        const ALGOS_SLOW_ON_MULTIPLE_TIMESERIES = [
            "autoarima_timeseries",
            "arima_timeseries",
            "croston_timeseries",
            "ets_timeseries",
            "seasonal_loess_timeseries",
            "prophet_timeseries",
            ...ALGOS_TIMESERIES_CLASSICAL_ML,
        ];

        const ALGOS_INCOMPATIBLE_WITH_MS = [
            "gluonts_transformer_timeseries",
            "gluonts_deepar_timeseries",
            "gluonts_npts_timeseries"
        ];

        const ALGOS_COMPATIBLE_WITH_SHIFTS_AND_WINDOWS = [
            "random_forest_regression",
            "xgboost",
            "ridge_regression",
            "lightgbm_regression"
        ];

        const service = {
            TIMESERIES_IMPUTE_METHODS,
            DUPLICATE_TIMESTAMPS_HANDLING_METHODS,
            ALGOS_WITHOUT_EXTERNAL_FEATURES,
            ALGOS_SLOW_ON_MULTIPLE_TIMESERIES,
            ALGOS_INCOMPATIBLE_WITH_MS,
            ALGOS_COMPATIBLE_WITH_SHIFTS_AND_WINDOWS,
            ALGOS_WITHOUT_QUANTILES,
            ALGOS_TIMESERIES_CLASSICAL_ML,
            prettyTimeUnit,
            prettyTimeSteps,
            plurifiedTimeUnits,
            getDayNumber,
            getWeekDayName,
            getMonthName,
            getQuarterName,
            getHalfYearName,
            prettySelectedDate,
            isWindowCompatible,
        };

        return service;

        function prettyTimeUnit(timeunit) {
            if (!timeunit) return;
            return TIME_UNITS[timeunit].toLowerCase()
        };

        function prettyTimeSteps(timeSteps, timeunit) {
            return `${timeSteps} ${$filter('plurify')(prettyTimeUnit(timeunit), timeSteps)}`;
        };

        function getDayNumber(index) {
            switch (index) {
                case 0:
                case 31:
                    return "Last";
                case 1:
                    return "First";
                default:
                    return index.toString();
            }
        }

        function getDayName(index) {
            switch (index) {
                case 0:
                case 31:
                    return "Last day";
                case 1:
                    return "First day";
                case 21:
                    return index + "st";
                case 2:
                case 22:
                    return index + "nd";
                case 3:
                case 23:
                    return index + "rd";
                default:
                    return index + "th";
            }
        }

        function getWeekDayName(index) {
            return getDayLabels(index - 1);
        }

        function getMonthName(index) {
            if (index === 0) {
                index = 12;
            }
            const date = new Date(2024, index - 1);
            return date.toLocaleString('en', { month: 'long' });
        }

        function getQuarterName(index) {
            if (index === 0) {
                index = 3;
            }
            return [
                "Jan, Apr, Jul, Oct",
                "Feb, May, Aug, Nov",
                "Mar, Jun, Sep, Dec",
            ][index - 1];
        }

        function getHalfYearName(index) {
            if (index === 0) {
                index = 6;
            }
            return [
                "January and July",
                "February and August",
                "March and September",
                "April and October",
                "May and November",
                "June and December",
            ][index - 1];
        }

        function prettySelectedDate(timeunit, monthlyAlignement, unitAlignment) {
            // this should be the same logic as prettySelectedDate in TimeSeriesUtil.java
            let day = getDayName(monthlyAlignement);
            let period = "the month"
            if (timeunit === "QUARTER") {
                period = getQuarterName(unitAlignment);
            } else if (timeunit === "HALF_YEAR") {
                period = getHalfYearName(unitAlignment);
            } else if (timeunit === "YEAR") {
                period = getMonthName(unitAlignment);
            }
            return day + ' of ' + period;
        }

        function plurifiedTimeUnits(timeSteps) {
            const timeUnits = Object.assign({}, TIME_UNITS);
            angular.forEach(TIME_UNITS, function(displayUnit, rawUnit) {
                timeUnits[rawUnit] = $filter('plurify')(displayUnit, timeSteps);
            });
            return timeUnits;
        };

        function isWindowCompatible(feature) {
            return ['TARGET', 'INPUT', 'INPUT_PAST_ONLY'].includes(feature?.role) && (
                feature?.type === 'NUMERIC' && (feature?.numerical_handling === undefined || feature?.numerical_handling === 'REGULAR') ||
                feature?.type === 'CATEGORY' && (feature?.category_handling === undefined || feature?.category_handling === 'DUMMIFY')
            );
        }

    });

    app.factory("TimeseriesForecastingCustomTrainTestFoldsUtils", function(TimeseriesForecastingUtils) {
        function getIntervalTimestepsCount(interval, timestepParams) {
            let nbTimesteps = 0;
            const intervalDuration = moment.duration(moment(interval[1]).diff(moment(interval[0])));
            switch (timestepParams.timeunit) {
                case "MILLISECOND": {
                    nbTimesteps = intervalDuration.asMilliseconds();
                    break;
                }
                case "SECOND": {
                    nbTimesteps = intervalDuration.asSeconds();
                    break;
                }
                case "MINUTE": {
                    nbTimesteps = intervalDuration.asMinutes();
                    break
                }
                case "HOUR": {
                    nbTimesteps = intervalDuration.asHours();
                    break;
                }
                case "DAY": {
                    nbTimesteps = intervalDuration.asDays();
                    break;
                }
                case "BUSINESS_DAY": {
                    const startDate = new Date(interval[0])
                    const lastDate = new Date(interval[1])
                    let iteratorDate = new Date(Date.UTC(startDate.getUTCFullYear(), startDate.getUTCMonth(), startDate.getUTCDate()));
                    while (iteratorDate < lastDate) {
                        if (iteratorDate.getUTCDay() != 0 && iteratorDate.getUTCDay() != 6) {
                            nbTimesteps++;
                        }
                        iteratorDate.setUTCDate(iteratorDate.getUTCDate() + 1);
                    }
                    break;
                }
                case "WEEK":{
                    const startDate = new Date(interval[0])
                    const lastDate = new Date(interval[1])
                    let iteratorDate = new Date(Date.UTC(startDate.getUTCFullYear(), startDate.getUTCMonth(), startDate.getUTCDate()));
                    while (iteratorDate < lastDate) {
                        let targetWeekDay = timestepParams.endOfWeekDay - 1;
                        const targetDateOfTheWeek = new Date(Date.UTC(iteratorDate.getUTCFullYear(), iteratorDate.getUTCMonth(), iteratorDate.getUTCDate()));
                        if (targetDateOfTheWeek.getUTCDay() != targetWeekDay) {
                            targetDateOfTheWeek.setUTCDate(targetDateOfTheWeek.getUTCDate() + (targetWeekDay > targetDateOfTheWeek.getUTCDay() ? targetWeekDay - targetDateOfTheWeek.getUTCDay() : 7 + targetWeekDay - targetDateOfTheWeek.getUTCDay()));
                        }
                        if (startDate <= targetDateOfTheWeek && targetDateOfTheWeek < lastDate) {
                            nbTimesteps++;
                        }
                        iteratorDate.setUTCDate(iteratorDate.getUTCDate() + 7);
                    }
                    break;
                }
                default: {
                    const startDate = new Date(interval[0])
                    const endDate = new Date(interval[1])
                    let iteratorDate = new Date(Date.UTC(startDate.getUTCFullYear(), startDate.getUTCMonth(), 1));
                    while (iteratorDate < endDate) {
                        const year = iteratorDate.getUTCFullYear();
                        const month = iteratorDate.getUTCMonth();
                        const daysInMonth = new Date(Date.UTC(year, month + 1, 0)).getUTCDate(); // 0 will return the last day of the prior month, hence month + 1
                        const actualTargetDay = timestepParams.monthlyAlignment === 0 ? daysInMonth : Math.min(timestepParams.monthlyAlignment, daysInMonth);
                        const targetDateInMonth = new Date(Date.UTC(year, month, actualTargetDay));

                        if (targetDateInMonth >= startDate && targetDateInMonth < endDate) {
                            nbTimesteps++;
                        }
                        iteratorDate.setUTCMonth(iteratorDate.getUTCMonth() + 1);
                    }
                    switch (timestepParams.timeunit) {
                        case "QUARTER": {
                            nbTimesteps = Math.ceil(nbTimesteps / 3); // Ceil because 4 months = 2 quarters
                            break;
                        }
                        case "HALF_YEAR": {
                            nbTimesteps = Math.ceil(nbTimesteps / 6); // Ceil because 7 months = 2 half years
                            break;
                        }
                        case "YEAR": {
                            nbTimesteps = Math.ceil(nbTimesteps / 12); // Ceil because 1 month = 1 year of data
                            break;
                        }
                        default: break;
                    }
                    break;
                }

            }
            return Math.max(0, Math.floor(nbTimesteps)); // We need to floor as dates can be set through API.
        }

        function isTestIntervalTooSmall(interval, predictionLength, timestepParams) {
            const testIntervalTimesteps = getIntervalTimestepsCount(interval["test"].map(d => forceConvertToUTCTimezoneDate(d)), timestepParams);
            return testIntervalTimesteps < predictionLength;
        }

        function validateDate(dateStr, condition, errorMessage) {
            const dateObj = forceConvertToUTCTimezoneDate(dateStr);
            if (!condition(dateObj)) {
                return errorMessage;
            }
        }

        function validateCustomTrainTestFold(timestepParams, interval, predictionLength) {
            const { timeunit, endOfWeekDay, unitAlignment } = timestepParams;
            if (interval['train'][0] > interval['train'][1]) {
                return "Train interval start date must be before the end date.";
            }
            if (interval['test'][0] > interval['test'][1]) {
                return "Test interval start date must be before the end date.";
            }
            if (interval['train'][1] > interval['test'][0]) {
                return "Test interval start date must be after the end of the train interval.";
            }
            if (["WEEK", "BUSINESS_DAY"].includes(timeunit)) {
                const targetDays = timeunit === "WEEK" ? [endOfWeekDay] : [2, 3, 4, 5, 6];
                const targetDayString = timeunit === "WEEK" ? TimeseriesForecastingUtils.getWeekDayName(endOfWeekDay) : "business day";
                const condition = d => targetDays.includes(d.getUTCDay() + 1);
                const errorMessage =
                    validateDate(interval['train'][0], condition, "Train start is not a " + targetDayString + ".") ||
                    validateDate(interval['train'][1], condition, "Train end is not a " + targetDayString + ".") ||
                    validateDate(interval['test'][0], condition, "Test start is not a " + targetDayString + ".") ||
                    validateDate(interval['test'][1], condition, "Test end is not a " + targetDayString + ".");
                if (errorMessage) return errorMessage;
            }

            if (timeunit === "QUARTER") {
                if (timestepParams.unitAlignment) {
                    const unitAlignmentStartMonth = timestepParams.unitAlignment - 1;
                    const validMonths = [unitAlignmentStartMonth, unitAlignmentStartMonth + 3, unitAlignmentStartMonth + 6, unitAlignmentStartMonth + 9]
                    const condition = d => validMonths.includes(d.getMonth());
                    const quarterName = TimeseriesForecastingUtils.getQuarterName(unitAlignment);
                    const errorMessage =
                        validateDate(interval['train'][0], condition, "Train start is not one of " + quarterName + ".") ||
                        validateDate(interval['train'][1], condition, "Train end is not one of " + quarterName + ".") ||
                        validateDate(interval['test'][0], condition, "Test start is not one of " + quarterName + ".") ||
                        validateDate(interval['test'][1], condition, "Test end is not one of " + quarterName + ".");
                    if (errorMessage) return errorMessage;
                }
            }

            if (timeunit === "HALF_YEAR") {
                if (unitAlignment) {
                    const unitAlignmentStartMonth = unitAlignment - 1;
                    const validMonths = [unitAlignmentStartMonth, unitAlignmentStartMonth + 6]
                    const condition = d => validMonths.includes(d.getMonth());
                    const halfYearName = TimeseriesForecastingUtils.getHalfYearName(unitAlignment);
                    const errorMessage =
                        validateDate(interval['train'][0], condition, "Train start is not one of " + halfYearName + ".") ||
                        validateDate(interval['train'][1], condition, "Train end is not one of " + halfYearName + ".") ||
                        validateDate(interval['test'][0], condition, "Test start is not one of " + halfYearName + ".") ||
                        validateDate(interval['test'][1], condition, "Test end is not one of " + halfYearName + ".");
                    if (errorMessage) return errorMessage;
                }
            }

            if (isTestIntervalTooSmall(interval, predictionLength * timestepParams["numberOfTimeunits"], timestepParams)) {
                return `Test interval is too small: ${TimeseriesForecastingUtils.prettyTimeSteps(predictionLength * timestepParams["numberOfTimeunits"], timeunit)} (${predictionLength} timesteps) required.`
            }
        }

        function forceConvertToUTCTimezoneDate(backendDateAsString) {
            // WARNING: Method duplicated in `interactive-scoring-scenarios.component.ts`. Update there also
            // In the backend, we work with dates without time-zones. ("YYYY-MM-DD HH:mm:ss.SSS")
            // We mark those dates as UTC dates in the front-end for ease of manipulation ("YYYY-MM-DD HH:mm:ss.SSSZ")
            // before returning a Date object.
            if (backendDateAsString.slice(-1) != "Z") {
                backendDateAsString = backendDateAsString + "Z"; // add trailing Z UTC marker if not present
            }
            return new Date(backendDateAsString);
        };

        const service = {
            validateCustomTrainTestFold,
            getIntervalTimestepsCount,
            forceConvertToUTCTimezoneDate: forceConvertToUTCTimezoneDate
        }

        return service;
    });

    app.component("periodDatePicker", {
        bindings: {
            timestepParams: '<',
            updateFn: '&',
            date: '=',
            minDate: '<', // Optional min-date
            maxDate: '<', // Optional max-date
        },
        templateUrl: '/templates/analysis/prediction/timeseries/period-date-picker.html',
        controller: function($timeout, $element, TimeseriesForecastingUtils) {
            const $ctrl = this;

            $ctrl.MONTHS = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December']

            $ctrl.$onInit = () => {
                $ctrl.hoursOptions = getHoursOptions();
                $ctrl.quartersOptions = getQuartersOptions();
                $ctrl.halfYearsOptions = getHalfYearsOptions();
                $ctrl.refreshYearsOptions();
                $ctrl.uiState = {
                    year: $ctrl.date.getUTCFullYear(),
                    month: $ctrl.MONTHS[$ctrl.date.getUTCMonth()],
                    day: $ctrl.date.getUTCDay(),
                    hour: $ctrl.date.getUTCHours(),
                    minute: $ctrl.date.getUTCMinutes(),
                    second: $ctrl.date.getUTCSeconds(),
                    millisecond: $ctrl.date.getUTCMilliseconds(),
                    inputTimeStep: getTimeInputStep(),
                    quarter: $ctrl.MONTHS[$ctrl.date.getUTCMonth()],
                    halfYear: $ctrl.MONTHS[$ctrl.date.getUTCMonth()],
                    dateTime: new Date($ctrl.date)
                };
            }

            $ctrl.initDatePicker = () => {
                // Formatting date as a string for the datepicker to have the correct utc timezone
                const dateString = `${$ctrl.date.getUTCFullYear()}-${($ctrl.date.getUTCMonth() + 1).toString().padStart(2, '0')}-${$ctrl.date.getUTCDate().toString().padStart(2, '0')}`;
                $element.find('.datepicker').datepicker($ctrl.getDatePickerConfig());
                $element.find('.datepicker').datepicker("setDate", dateString);
            }

            $ctrl.showDatePicker = () => {
                $element.find('.datepicker').datepicker("show");
            }

            $ctrl.getDatePickerConfig = function() {
                let isChangingMonthYear = false; // Lock to avoid infinite recursion performing the updates
                const currentYear = $ctrl.date.getUTCFullYear();
                let datePickerSettings = {
                    changeMonth: true,
                    changeYear: true,
                    yearRange: (currentYear - 100) + ":" + (currentYear + 100),
                    showButtonPanel: true,
                    dateFormat: "yy-mm-dd",
                    onClose: function(raw, obj) {
                        if (isChangingMonthYear) return;
                        $ctrl.date.setUTCFullYear(obj.selectedYear);
                        $ctrl.date.setUTCMonth(obj.selectedMonth); // selectedMonth is (0-11) :facepalm:
                        $ctrl.date.setUTCDate(obj.selectedDay);
                        $ctrl.uiState.year = obj.selectedYear;
                        $ctrl.uiState.month = obj.selectedMonth;
                        $ctrl.uiState.day = obj.selectedDay;
                        $ctrl.propagateUpdate();
                    },
                    onChangeMonthYear(newYear, newMonth, inst) {
                        if (isChangingMonthYear) return;
                        $ctrl.date.setUTCFullYear(newYear);
                        $ctrl.date.setUTCMonth(newMonth - 1); // New month is (1-12)
                        $ctrl.uiState.year = newYear;
                        $ctrl.uiState.month = newMonth - 1;
                        if (inst && inst.input) {
                            isChangingMonthYear = true;
                            // Same as in initDatePicker, use a string to prevent timezone issues.
                            const dateString = `${$ctrl.date.getUTCFullYear()}-${($ctrl.date.getUTCMonth() + 1).toString().padStart(2, '0')}-${$ctrl.date.getUTCDate().toString().padStart(2, '0')}`;
                            inst.input.datepicker("setDate", dateString);
                            isChangingMonthYear = false;
                        }
                        $ctrl.propagateUpdate();
                    },
                    onUpdateDatepicker: function() {
                        $('.ui-datepicker-prev').append("<i class='dku-icon-arrow-left-16'></i>");
                        $('.ui-datepicker-next').append("<i class='dku-icon-arrow-right-16'></i>");
                    },
                    minDate: $ctrl.minDate,
                    maxDate: $ctrl.maxDate
                }

                switch ($ctrl.timestepParams.timeunit) {
                    case 'MILLISECOND':
                    case 'SECOND':
                    case 'MINUTE':
                    case 'HOUR':
                    case 'DAY':
                        break;
                    case 'BUSINESS_DAY':
                        datePickerSettings = {
                            beforeShowDay: function (date) {
                                // Disables days
                                // We have to use `getDay` here as jquery loops over calendar dates in the user browser timezone,
                                // and we didn't find a way to configure the component in UTC.
                                const day = date.getDay();
                                return [day != 6 && day != 0]
                            },
                            ...datePickerSettings,
                        }
                        break;
                    case 'WEEK':
                        datePickerSettings = {
                            beforeShowDay: function (date) {
                                // Disables days
                                // We have to use `getDay` here as jquery loops over calendar dates in the user browser timezone,
                                // and we didn't find a way to configure the component in UTC.
                                const day = date.getDay();
                                return [day == $ctrl.timestepParams.endOfWeekDay - 1]
                            },
                            showWeek: true,
                            ...datePickerSettings,
                        }
                        break;
                    default: return undefined;
                }
                return datePickerSettings;
            }

            function getHoursOptions() {
                const hours = [];
                for (let h = 0; h < 24; h++) {
                    hours.push(h);
                }
                return hours;
            }

            function getTimeInputStep() {
                switch ($ctrl.timestepParams.timeunit) {
                    case 'MILLISECOND': return .001;
                    case 'SECOND': return 1;
                    case 'MINUTE': return 60;
                    default: return undefined;
                }
            }

            $ctrl.refreshYearsOptions = function() {
                const currentYear = $ctrl.date.getUTCFullYear();
                const startYear = Math.max(1, currentYear - 500); // Prevent year 0 or negative years
                const endYear = currentYear + 500;
                $ctrl.yearsOptions = Array.from({length: endYear - startYear + 1}, (_, i) => startYear + i);
            }

            function getQuartersOptions() {
                if ($ctrl.timestepParams.timeunit !== "QUARTER") return undefined;
                switch ($ctrl.timestepParams.unitAlignment) {
                    case 1: return ["January", "April", "July", "October"];
                    case 2: return ["February", "May", "August", "November"];
                    case 3: return ["March", "June", "September", "December"];
                }
            }

            function getHalfYearsOptions() {
                if ($ctrl.timestepParams.timeunit !== "HALF_YEAR") return undefined;
                switch ($ctrl.timestepParams.unitAlignment) {
                    case 1: return ["January", "July"];
                    case 2: return ["February", "August"];
                    case 3: return ["March", "September"];
                    case 4: return ["April", "October"];
                    case 5: return ["May", "November"];
                    case 6: return ["June", "December"];
                    default: return undefined;
                }
            }

            function getTimeInputStringValue(date, timeUnit) {
                let stringValue;
                switch (timeUnit) {
                    case 'SECOND': {
                        stringValue = `${date.getUTCHours().toString().padStart(2, '0')}:${date.getUTCMinutes().toString().padStart(2, '0')}:${date.getUTCSeconds().toString().padStart(2, '0')}`;
                        break;
                    }
                    case 'MINUTE': {
                        stringValue = `${date.getUTCHours().toString().padStart(2, '0')}:${date.getUTCMinutes().toString().padStart(2, '0')}`;
                        break;
                    }
                    default:
                        stringValue = `${date.getUTCHours().toString().padStart(2, '0')}:${date.getUTCMinutes().toString().padStart(2, '0')}:${date.getUTCSeconds().toString().padStart(2, '0')}.${date.getUTCMilliseconds().toString().toString().padStart(3, '0')}`;
                        break;
                }
                return stringValue;
            }

            $ctrl.setValue = function() {
                const stringValue = getTimeInputStringValue($ctrl.uiState.dateTime, $ctrl.timestepParams.timeunit);
                $ctrl.date.setUTCMilliseconds($ctrl.uiState.dateTime.getUTCMilliseconds());
                $ctrl.date.setUTCSeconds($ctrl.uiState.dateTime.getUTCSeconds());
                $ctrl.date.setUTCMinutes($ctrl.uiState.dateTime.getUTCMinutes());
                $ctrl.date.setUTCHours($ctrl.uiState.dateTime.getUTCHours());
                $timeout(function() { $element.find('input[type="time"]').val(stringValue);});
                $ctrl.propagateUpdate();
            }

            $ctrl.propagateUpdate = function() {
                $ctrl.setZeros()
                $ctrl.updateFn();
            }

            $ctrl.setZeros = function() {
                switch ($ctrl.timestepParams.timeunit) {
                    case 'SECOND': {
                        $ctrl.date.setUTCMilliseconds(0);
                        break;
                    }
                    case 'MINUTE': {
                        $ctrl.date.setUTCMilliseconds(0);
                        $ctrl.date.setUTCSeconds(0);
                        break;
                    }
                    case 'HOUR': {
                        $ctrl.date.setUTCMilliseconds(0);
                        $ctrl.date.setUTCSeconds(0);
                        $ctrl.date.setUTCMinutes(0);
                        break;
                    }
                    case 'DAY':
                    case 'BUSINESS_DAY':
                    case 'WEEK': {
                        $ctrl.date.setUTCMilliseconds(0);
                        $ctrl.date.setUTCSeconds(0);
                        $ctrl.date.setUTCMinutes(0);
                        $ctrl.date.setUTCHours(0);
                        break;
                    }
                    case 'MONTH':
                    case 'QUARTER':
                    case 'HALF_YEAR': {
                        $ctrl.date.setUTCMilliseconds(0);
                        $ctrl.date.setUTCSeconds(0);
                        $ctrl.date.setUTCMinutes(0);
                        $ctrl.date.setUTCHours(0);
                        $ctrl.date.setUTCDate(1);
                        break;
                    }
                    case 'YEAR': {
                        $ctrl.date.setUTCMilliseconds(0);
                        $ctrl.date.setUTCSeconds(0);
                        $ctrl.date.setUTCMinutes(0);
                        $ctrl.date.setUTCHours(0);
                        $ctrl.date.setUTCDate(1);
                        $ctrl.date.setUTCMonth(0);
                        break;
                    }
                    default: break;
                }
            }
        }
    })

})();
''
