(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Utility functions for dashboards.
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.factory('DashboardUtils', function($rootScope, $injector, $state, Logger, $stateParams, ClipboardUtils, ActivityIndicator, $timeout, Debounce, WT1, translate, DataikuAPI, $q) {

        const COLUMN_NUMBER = 36;

        const svc = {
            canListInsight: (insight) => {
                return insight.type !== 'filters';
            },
            canEditInsight: function(insight, canModerateDashboards) {
                return insight && (canModerateDashboards || insight.owner == $rootScope.appConfig.login);
            },
            hasEditTab: function(insight) {
                return (insight && insight.type && svc.getInsightHandler(insight.type).hasEditTab);
            },

            setLoading() {
                const $scope = this[0];
                $scope.loaded = false;
                $scope.loading = true;
            },

            setError: function(data, status, headers, config, statusText) {
                const $scope = this[0];
                const reject = this[1];

                $scope.loaded = false;
                $scope.loading = false;
                $scope.error = data;
                $scope.unconfigured = false;
                if ($scope.hook && $scope.hook.isErrorMap) {
                    $scope.hook.isErrorMap[$scope.tile.$tileId] = true;
                } else if ($scope.hook && $scope.hook.setErrorInDashboardPageScope) {
                    $scope.hook.setErrorInDashboardPageScope(data, status, headers, config, statusText);
                }
                /*
                 * Mark chart loading process as COMPLETE as it fails initializing
                 * This allow the PDF export to know that the chart is ready to be snapshot
                 */
                if (typeof ($scope.loadedCallback) === 'function') {
                    $scope.loadedCallback();
                }
                if (typeof (reject) === 'function') {
                    reject();
                }
            },
            setLoaded: function(data, status, headers, config, statusText) {
                const $scope = this[0];
                const resolve = this[1];

                $scope.loading = false;
                $scope.loaded = true;
                $scope.error = null;
                $scope.unconfigured = false;
                if ($scope.hook && $scope.hook.isErrorMap) {
                    $scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                }
                if (typeof (resolve) === 'function') {
                    resolve();
                }
            },
            setUnconfigured: function(data, status, headers, config, statusText) {
                const $scope = this[0];
                const reject = this[1];

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;
                $scope.unconfigured = true;
                if ($scope.hook && $scope.hook.isErrorMap) {
                    $scope.hook.isErrorMap[$scope.tile.$tileId] = false;
                }
                /*
                 * Mark chart loading process as COMPLETE as it's not configured
                 * This allow the PDF export to know that the chart is ready to be snapshot
                 */
                if (typeof ($scope.loadedCallback) === 'function') {
                    $scope.loadedCallback();
                }
                if (typeof (reject) === 'function') {
                    reject();
                }
            },
            canGoToInsight: (tile) => {
                return tile.insightType !== 'filters';
            },
            getInsightHandler: function(type) {
                if (!type || type == 'INSIGHT') {
                    return;
                }
                const camelCaseType = type.replace(/-|_/g, ' ').replace(/\b\w/g, function(l) {
                    return l.toUpperCase();
                }).replace(/ /g, '');
                try {
                    return $injector.get(camelCaseType + 'InsightHandler');
                } catch (err) {
                    Logger.error('Failed to inject insight handler for type: ' + type);
                };
            },
            getInsightSourceType: function(insight) {
                const handler = svc.getInsightHandler(insight.type);
                if (!handler) {
                    return null;
                }
                return handler.sourceType || handler.getSourceType(insight);
            },
            getInsightSourceId: function(insight) {
                const handler = svc.getInsightHandler(insight.type);
                return handler.getSourceId(insight);
            },
            getInsightTypeGroup: function(type) {
                const insightTypeGroups = {
                    'scenario_last_runs': 'scenario',
                    'scenario_run_button': 'scenario'
                };

                return insightTypeGroups[type] || type;
            },
            getInsightSourceAccessMode: function(insight) {
                const handler = svc.getInsightHandler(insight.type);
                return handler.accessMode || 'READ';
            },
            getNeededReaderAuthorization: function(insight) {
                const ref = {
                    objectType: svc.getInsightSourceType(insight)
                };

                const resolved = resolveObjectSmartId(svc.getInsightSourceId(insight));
                ref.objectId = resolved.id;
                if (resolved.projectKey) {
                    ref.projectKey = resolved.projectKey;
                }
                const mode = svc.getInsightSourceAccessMode(insight);
                return { objectRef: ref, modes: [mode] };
            },
            hasOptions: function(insight) {
                if (insight && insight.type) {
                    const handler = svc.getInsightHandler(insight.type);
                    if ('hasOptions' in handler) {
                        if (handler.hasOptions instanceof Function) {
                            return handler.hasOptions(insight);
                        }
                        return handler.hasOptions;
                    }
                }
                return true;
            },
            getTooltipPromotionTitleFunction: function(type) {

                return function(item) {
                    if (!item) {
                        return '';
                    }

                    const translation_key_suffix = type.toUpperCase();

                    if (item.listed && this.canModerateDashboards()) {

                        if (this.isAV2ProjectPermissions()) {
                            return translate('DASHBOARD.PERMISSIONS.UNPROMOTE_' + translation_key_suffix, `Unpromote this ${type}`);
                        } else {
                            return translate('DASHBOARD.PERMISSIONS.MAKE_PRIVATE_' + translation_key_suffix, `Make this ${type} private`);
                        }

                    } else if (item.listed) {

                        if (this.isAV2ProjectPermissions()) {
                            return translate('DASHBOARD.PERMISSIONS.PROMOTED_TO_ALL_USERS_' + translation_key_suffix, `This ${type} is promoted to all users`);
                        } else {
                            return translate('DASHBOARD.PERMISSIONS.PUBLIC_TO_ALL_USERS_' + translation_key_suffix, `This ${type} is public to all users`);
                        }

                    } else if (this.canModerateDashboards()) {

                        if (this.isAV2ProjectPermissions()) {
                            return translate('DASHBOARD.PERMISSIONS.PROMOTE_' + translation_key_suffix, `Promote this ${type} to all users`);
                        } else {
                            return translate('DASHBOARD.PERMISSIONS.MAKE_PUBLIC_' + translation_key_suffix, `Make this ${type} public to all users`);
                        }

                    }
                    return translate('DASHBOARD.PERMISSIONS.ONLY_YOURS_' + translation_key_suffix, `This ${type} is only yours to see`);
                };
            },
            getDashboardDefaultName: function() {
                return translate('INSIGHTS.MODAL.NEW_DASHBOARD_PLACEHOLDER', '{{name}}\'s dashboard', { name: $rootScope.appConfig.user.displayName });
            },
            isInDashboard: function() {
                return !!$stateParams.dashboardId || ($stateParams.workspaceKey != null && $stateParams.objectType === 'DASHBOARD' && $stateParams.insightId == null);
            },
            copyToClipboard: (urlToCopy) => {
                try {
                    ClipboardUtils.copyToClipboard(urlToCopy);
                } catch (error) {
                    ActivityIndicator.error('Error while copying URL');
                }
            },
            getCurrentPage: (dashboard, pageId) => {
                if (!dashboard || !dashboard.pages) {
                    return null;
                }
                let currentPage = dashboard.pages.find(({ id }) => id === pageId);
                if (currentPage != null) {
                    return currentPage;
                }

                // When we have just created a new page and before we save it, we look at the page index.
                const pageIndex = parseInt(pageId);
                if (!isNaN(pageIndex) && dashboard.pages.length > pageIndex) {
                    currentPage = dashboard.pages[pageIndex];
                }
                return currentPage;
            },
            toggleFilterPanel: (dashboard, page, toggle) => {
                if (!page) {
                    page = svc.getCurrentPage(dashboard, page.id);
                }
                if (page) {
                    if (_.isBoolean(toggle)) {
                        page.showFilterPanel = toggle;
                    } else {
                        page.showFilterPanel = !page.showFilterPanel;
                    }
                }
            },
            toggleAllFilterPanels: (dashboard, toggle) => {
                dashboard.pages.forEach(page => svc.toggleFilterPanel(dashboard, page, toggle));
            },
            showFiltersPanelForPagesWithFilters: (dashboard) => {
                dashboard.pages.forEach(page => svc.toggleFilterPanel(dashboard, page, Boolean(page.filters && page.filters.length)));
            },
            isDashboardEmbeddedInAWorkspace: () => {
                return $state.current.name === 'workspaces.object';
            },
            getColumnNumber: () => COLUMN_NUMBER,
            resizeDashboard: Debounce().withDelay(200, 200).wrap(() => {
                $timeout(() => {
                    window.dispatchEvent(new Event('resize'));
                });
            }),
            resizeGroupTile: Debounce().withDelay(200, 200).wrap(tile => {
                $timeout(() => {
                    $rootScope.$broadcast('dashboardResizeGroupTile', { tile });
                });
            }),
            sendWT1InsightCreation: function(insightType, options) {
                WT1.event('dashboard-insight-create', {
                    insightType,
                    triggeredFrom: undefined,
                    ...options
                });
            },
            sendWT1TileCreation: function(type, options) {
                WT1.event('dashboard-tile-create', {
                    type: type,
                    insightType: undefined,
                    triggeredFrom: undefined,
                    reuseExistingInsight: false,
                    hasUsedRectangleSelection: false,
                    hasUsedMultiSelectKey: false,
                    childTileCount: 0,
                    ...options
                });
            },

            expandVariables(text) {
                if (text.includes('${')) {
                    return DataikuAPI.variables.expandExprAllowUnresolved($stateParams.projectKey, text).noSpinner();
                }
                return $q.resolve({ data: text });
            }
        };

        return svc;
    });
})();
