(function() {
'use strict';

    /**
     * Directives and functions for the main flow graphs
     * (flow, inter project graph and job preview subgraph)
     */


    const app = angular.module('dataiku.flow.graph', []);

    app.service('GraphZoomTrackerService', function($timeout, $stateParams, localStorageService) {
        // GraphZoomTrackerService tracks the context needed to restore the zoom level, window position and focussed item when returning to the
        // flow view screen.  The zoomCtx (zoom-context) is persisted in local storage and comprises:
        // - focusItem: details of the last focused node, as tracked by the service.
        // - panzoom: $scope.panzoom context managed by the flowGraph directive, an extended version of the viewBox dimensions of the SVG flow graph element
        // - nodeCount: the number of node in the whole flow

        const svc = this;
        const lsKeyPrefix = "zoomCtx-";
        let zoomCtxKeyLoaded = "";
        let currentCtx = {};
        let disabled = false;

        svc.initCtx = function () {
            currentCtx = {};
        }

        svc.disable = function(disable = true) {
            disabled = disable;
        }

        svc.isEnabled = function() {
            return !disabled;
        }

        function zoneId() {
            return $stateParams.zoneId ? $stateParams.zoneId : "";
        }

        function projectKey() {
            return $stateParams.projectKey;
        }

        function zoomCtxKey() { // the key to access the zoomCtx in local storage
            return lsKeyPrefix + projectKey() + zoneId();
        }

        /*
            * currentCtx management: loading it from local storage / cleaning it...
            */

        /**
         * Load in current context (ie: global variables) the zoomContext stored in localstorage.
         * Loading won't occur if we already have a current context matching the current project key and that have a focusItem, unless it is forced.
         */
        function ensureZoomCtxLoaded () {
            let isProjectContextLoaded = function() {
                return zoomCtxKey()==zoomCtxKeyLoaded;
            };

            let isFocusItemCtxSet = function() {
                return (currentCtx.focusItem && currentCtx.focusItem.id && currentCtx.focusItem.projectKey==projectKey())
            };

            let timeToCheckLocalStorageUsage = function() {
                // Date.now returns millseconds.
                // ( date % 5 == 0) is effectively "we check about once in 5 times"
                return Date.now() % 5 == 0;
            };

            if (isProjectContextLoaded() && isFocusItemCtxSet()) {
                // after reverting to an old version the currentCtx can have parts missing so we put them back
                currentCtx = {...blankCtx, ...currentCtx};
                return;
            }

            // reload zoom context from local storage e.g on reload of the flow view
            clearContext();
            const restored = localStorageService.get(zoomCtxKey());
            if (restored && restored.panzoom) {
                zoomCtxKeyLoaded = zoomCtxKey();
                currentCtx.focusItem = restored.focusItem ? restored.focusItem : {};
                currentCtx.nodeCount = restored.nodeCount ? restored.nodeCount : 0;

                if (svc.isValidPanZoom(restored.panzoom)) {
                    angular.copy(restored.panzoom, currentCtx.panzoom);
                } else {
                    currentCtx.panzoom = {};
                }

                currentCtx.foldState = restored.foldState || [];
            }
            if (timeToCheckLocalStorageUsage()) $timeout(() => tidyLocalStorage(), 2000);
        };

        const blankCtx = {
            focusItem: {},
            nodeCount: 0,
            panzoom: {},
            foldState: []
        }

        function clearContext() {
            zoomCtxKeyLoaded = "";
            angular.copy(blankCtx, currentCtx);
        }

        /*
            * Local storage management: saving, cleaning...
            */

        /**
         * Save the currentCtx to localstorage under the key of 'zoomKey'.
         */
        function setLocalStorage(zoomKey) {
            if (projectKey() == undefined || zoomKey !== zoomCtxKey() || currentCtx.isSaving) {
                // Cancel save in case we force a save but a lazy is pending
                // or when the key are not the same
                return;
            }
            currentCtx.modified = Date.now();
            localStorageService.set(zoomKey, currentCtx);
        }

        /**
         * Set current context's panzoom
         * @param pz
         */
        function setPanzoom(pz) {
            const pzToSave = {};
            ['x', 'y', 'height', 'width'].forEach((f) => pzToSave[f] = pz[f]);
            currentCtx.panzoom = angular.copy(pzToSave);
        }

        /**
         * Check if the current context has a panzoom
         */
        svc.hasPanzoom = function() {
            const restored = localStorageService.get(zoomCtxKey());
            if (restored && restored.panzoom) {
                zoomCtxKeyLoaded = zoomCtxKey();
                return svc.isValidPanZoom(restored.panzoom);
            }
            return false;
        };

        /**
         * Save panzoom context immediately to local storage
         * @param pz
         */
        svc.instantSavePanZoomCtx = function(pz) {
            if (disabled === true) {
                return;
            }
            setPanzoom(pz);
            setLocalStorage(zoomCtxKey());
        };

        /**
         * Save the zoom context in local storage, but since this can get called very rapidly when zooming and panning, we make sure we don't write on local storage more than once every second.
         * @param pz
         */
        svc.lazySaveZoomCtx = function(pz) {

            if (disabled === true) {
                return;
            }
            // save the zoom context, but since this can get called v rapidly when zooming and panning,
            setPanzoom(pz);

            if (!currentCtx.isSaving) {
                currentCtx.isSaving = true;
                const zoomKey = zoomCtxKey();
                $timeout (() => {
                    delete currentCtx.isSaving;
                    setLocalStorage(zoomKey);
                }, 1000);
            }
        };

        /**
         * Remove oldest zoomCtxes if we've stored too many of them.
         */
        function tidyLocalStorage() {
            // We allow the number of stored keys to vary between numLsKeysToKeep and maxNumLsKeysAllowed
            // to avoid purging everytime we open a project.  Unclear if this is worthwhile perf saving.
            const numLsKeysToKeep = 10; // number of context entries in localstorage we will keep on purge
            const maxNumLsKeysAllowed = 15; // number of context entries that will trigger a purge.

            let getLsDateModified = function (key) {
                return new Date(localStorageService.get(key).modified);
            }

            const keys = localStorageService.keys().filter((k) => k.startsWith(lsKeyPrefix))

            if (keys.length > maxNumLsKeysAllowed) {
                keys.sort((a,b) => {
                        const aDate = getLsDateModified(a);
                        const bDate = getLsDateModified(b);
                        return aDate > bDate ? -1 : aDate < bDate ? 1 : 0;
                    }
                );

                const keysToDelete = keys.splice(numLsKeysToKeep);
                keysToDelete.forEach((k) => localStorageService.remove(k));
            };

        }

        function updateFoldStateList(foldStateList, foldCmd) {
            const listLen = foldStateList.length;
            let foundFirstFold = false;

            if (foldCmd.action == 'unfold') {
                foldStateList = foldStateList.filter(oldCmd => {
                    //Any unfold commands before a fold are meaningless
                    if (!foundFirstFold) {
                        foundFirstFold = (oldCmd.action == 'fold')
                        if (!foundFirstFold) return false;
                    }
                    // filter out previous fold/unfolds on this node
                    return oldCmd.nodeId != foldCmd.nodeId || oldCmd.direction != foldCmd.direction;
                });
            }

            if (listLen == foldStateList.length) {
                foldStateList.push(foldCmd);
            }

            return foldStateList;
        }

        /*
            * Setters: persisting panzoom ctx elements...
            */

        /**
         * Set current context's focusItem and save it on local storage
         * @param node
         */
        svc.setFocusItemCtx = function(node, nodeChangedByName = false) {

            if (disabled === true) {
                return;
            }

            currentCtx.focusItem = {
                id: node.id,
                nodeChangedByName,
                projectKey: projectKey()
            };
            ensureZoomCtxLoaded();
            setLocalStorage(zoomCtxKey());
        };

        svc.resetFocusItemCtx = function() {
            currentCtx.focusItem = {};
            setLocalStorage(zoomCtxKey());
        };


        /**
         * Set current context's focusItem based on a type and a fullname and save it on local storage
         * (A generalised version of setFocusItemByName which supports foreign datasets, which have a different project key)
         * @param type: node type (recipe, dataset, etc.)
         * @param fullName: node full name (it includes project's key)
         */
        svc.setFocusItemByFullName = function (type, fullName) {

            if (disabled === true) {
                return;
            }
            // A generalised version of setFocusItemByName. It supports foreign datasets,
            // which have a different project key.
            if (!fullName) return;
            ensureZoomCtxLoaded();

            currentCtx.focusItem.id = graphVizEscape(type + "_" + fullName);
            currentCtx.focusItem.nodeChangedByName = true;
            setLocalStorage(zoomCtxKey());
        };

        /**
         * Set current context's focusItem based on a type and a name and save it on local storage
         * Called from various controllers (e.g. dataset/recipe editors) to update the flow item that is will be selected when the flow is next redisplayed.
         * @param type: node type (recipe, dataset, etc.)
         * @param name: node name
         */
        svc.setFocusItemByName = function (type, name) {

            if (disabled === true) {
                return;
            }
            let isIncludeProjRefInNodeId = function(type) {
                return type != "recipe"; //recipes have a slightly different SVG element id format in the flow
            }

            if (!name) return;
            const proj = isIncludeProjRefInNodeId(type) ? projectKey() + "." : "";
            svc.setFocusItemByFullName(type, proj + name);
        };

        svc.setFlowRedrawn = function (newNodeCount) {
            currentCtx.focusItem.nodeChangedByName = false;
            currentCtx.nodeCount = newNodeCount;
        };

        /**
         * Set fold/unfold request
         * Called when the user folds or unfolds a node.  The
         * active fold state is saved so it can be restored when the
         * view is refreshed.
         * On unfold we remove existing 'redundant' commands from the list.
         */
        svc.setFoldCommand = function (foldCmd) {
            currentCtx.foldState = updateFoldStateList(currentCtx.foldState, foldCmd)
            setLocalStorage(zoomCtxKey());
        };

        svc.resetFoldState = function(commands) {
            currentCtx.foldState = commands || [];
            setLocalStorage(zoomCtxKey());
        }

        svc.removeLastFoldCommand = function () {
            currentCtx.foldState.pop();
            setLocalStorage(zoomCtxKey());
        }

        /*
            * Getters: retrieving panzoom ctx elements
            */

        /**
         * Load panzoom ctx that we stored in activePz passed in parameter
         * @param activePz - the current $scope.panZoom.  This is updated not replaced, to avoid any knock on effects
         *                   in the existing graph-handling software
         * @param defaultPz - a panZoom structure with default settings i.e. show the whole flow
         * @returns true if replacement occured, false otherwise (in the case there's no current context's panzoom)
         */
        svc.restoreZoomCtx = function(activePz, defaultPz) {

            if (disabled === true) {
                return;
            }
            ensureZoomCtxLoaded();

            if (currentCtx && currentCtx.panzoom && svc.isValidPanZoom(currentCtx.panzoom) && activePz && defaultPz) {
                angular.copy(defaultPz, activePz); //copy defaultPz in activePz
                angular.extend (activePz, currentCtx.panzoom); //extend activePz with currentCtx.panzoom
                return true;
            }
            return false;
        };

        /**
         * Returns current context focusItem id (or empty string if this item is not contained in the flowGraph passed in parameter)
         * @param flowGraph to validate current context focusItem's id with
         * @returns current context focusItem's id (or empty string if this item is not contained in the flowGraph passed in parameter)
         */

        svc.getSafeFocusItemId = function (flowGraph) {
            if (disabled === true) {
                return;
            }
            if (flowGraph && currentCtx.focusItem && currentCtx.focusItem.id) {
                if (!flowGraph.node(currentCtx.focusItem.id)) {
                    // focus item may have been added by name without taking in account the zones
                    // try to find it
                    if (currentCtx.focusItem.nodeChangedByName) {
                        currentCtx.focusItem.id = svc.getZoomedName(flowGraph, currentCtx.focusItem.id);
                    }
                    if (!flowGraph.node(currentCtx.focusItem.id)) {
                        currentCtx.focusItem.id = "";
                    }
                }
            }
            return currentCtx.focusItem ? currentCtx.focusItem.id : "";
        };

        /**
         * Returns the id with the correct zone if we are not in a zone
         * In case we are in zone, return the id
         */
        svc.getZoomedName = (flowGraph, id) => {
            if (!id || id.startsWith("zone_")) {
                return id;
            }
            const graph = flowGraph.get();
            if (!graph.hasZones && !$stateParams.zoneId) {
                return id;
            }
            const sharedBetweenZones = graph.zonesUsedByRealId[id];
            if (sharedBetweenZones) {
                const node = flowGraph.node(graphVizEscape(`zone_${sharedBetweenZones[0]}_`) + id);
                if (node && flowGraph.node(graphVizEscape(`zone_${node.ownerZone}_`) + id)) {
                    return graphVizEscape(`zone_${node.ownerZone}_`) + id;
                }
            }
            const foundName = Object.keys(graph.nodes).find(it => it.endsWith(id));
            if (foundName) {
                return foundName;
            }
            return id;
        }

        svc.getNodeCount = function() {
            return currentCtx.nodeCount;
        };

        function isValidDimension(d) {
            return angular.isDefined(d) && isFinite(d) && d > 0;
        }
        svc.isValidPanZoom = function(pz) {
            return isValidDimension(pz.width) && isValidDimension(pz.height);
        };

        svc.wasNodeChangedOutsideFlow = function() {
            return !!currentCtx.focusItem.nodeChangedByName;
        };

        svc.getFoldState = function() {
            return currentCtx.foldState;
        }

        svc.getPreviewFoldState = function(foldCmd) {
            const previewFoldState =  angular.copy(currentCtx.foldState);
            return updateFoldStateList(previewFoldState, foldCmd)
        }

    });

    app.factory("FlowZoneMoveService", function() {
        // To keep in sync with ProjectGraphSerializer.java
        const COLLAPSED_ZONE_WIDTH = 300 + 2 * 64;
        const COLLAPSED_ZONE_HEIGHT = 44;

        const service = {
            setZoneEdges,
            updateZonesDimensions,
        }

        return service 

        // Implementations

         /**
         * Update the edge between a specific zone (zoneId) and the other connected zones
         * Updating the edge consists in removing the graphviz-generated curved path, and replacing it by a straight line
         * This methods finds out on which side of the zone it's best to attach the line, and sorts the lines for one side
         * so that they are least likely to intersect.
         * 
         * This method is called on all zones eventually, hence setting both ends of all edges.
        */
        function setZoneEdges(svg, zoneId, nodesGraph) {
            const edgesToZone = svg.find(`g[data-to="${zoneId}"]`);
            const edgesFromZone = svg.find(`g[data-from="${zoneId}"]`);
        
            const zonesFrom = [...new Set(edgesToZone.toArray().map((element) => element.getAttribute("data-from")))];
            const zonesTo = [...new Set(edgesFromZone.toArray().map((element) => element.getAttribute("data-to")))];
        
            const zone = nodesGraph.nodes[zoneId];

            const dispatchedZonesFrom = dispatchZones(zonesFrom, zone, nodesGraph, "TO");
            const dispatchedZonesTo = dispatchZones(zonesTo, zone, nodesGraph, "FROM");
            let rightZones = [...dispatchedZonesFrom.rightZones, ...dispatchedZonesTo.rightZones];
            let leftZones = [...dispatchedZonesFrom.leftZones, ...dispatchedZonesTo.leftZones];
            let topZones = [...dispatchedZonesFrom.topZones, ...dispatchedZonesTo.topZones];
            let bottomZones = [...dispatchedZonesFrom.bottomZones, ...dispatchedZonesTo.bottomZones];

            rightZones.sort(zonesSortingFunction(zoneId, false));
            leftZones.sort(zonesSortingFunction(zoneId, false));
            topZones.sort(zonesSortingFunction(zoneId, true));
            bottomZones.sort(zonesSortingFunction(zoneId, true));
        
            // If the curved path is still there it's removed.
            edgesToZone.find('path').remove();
            edgesFromZone.find('path').remove();
        
            const rightPositions = rightZones.map((otherZoneInfo, idx) => {
                let x = zone.position.x + zone.position.width; 
                if (otherZoneInfo.direction === "TO") {
                    x += 2.8;  // To make ellipse more visible
                }
                const y = computeEdgePositionOnZoneSide(zone, idx, rightZones.length, true);
                return { info: otherZoneInfo, x: x, y: y };
            });
            const leftPositions = leftZones.map((otherZoneInfo, idx) => {
                let x = zone.position.x;
                if (otherZoneInfo.direction === "TO") {
                    x -= 2.8;  // To make ellipse more visible
                }
                const y = computeEdgePositionOnZoneSide(zone, idx, leftZones.length, true);
                return { info: otherZoneInfo, x: x, y: y };
            });
            const topPositions = topZones.map((otherZoneInfo, idx) => {
                const x = computeEdgePositionOnZoneSide(zone, idx, topZones.length, false);
                let y = zone.position.y;
                if (otherZoneInfo.direction === "TO") {
                    y -= 2.8;  // To make ellipse more visible
                }
                return { info: otherZoneInfo, x: x, y: y };
            });
            const bottomPositions = bottomZones.map((otherZoneInfo, idx) => {
                const x = computeEdgePositionOnZoneSide(zone, idx, bottomZones.length, false);
                let y = zone.position.y + zone.position.height;
                if (otherZoneInfo.direction === "TO") {
                    y += 2.8;  // To make ellipse more visible
                }
                return { info: otherZoneInfo, x: x, y: y };
            });
        
            const allPositions = [...rightPositions, ...leftPositions, ...topPositions, ...bottomPositions];
        
            allPositions.forEach((position) => {
                if (position.info.direction === "FROM") {
                    edgesFromZone.filter(`g[data-to="${position.info.id}"]`).each(function() {
                        addOrUpdateLine($(this), position.x, position.y, "FROM");
                    });
                } else {
                    edgesToZone.filter(`g[data-from="${position.info.id}"]`).each(function() {
                        addOrUpdateLine($(this), position.x, position.y, "TO");
                        $(this).find('ellipse').attr("cx", `${position.x}`).attr("cy", `${position.y}`);
                    });
                }
            });
        }

        /**
         * Compute the position of a given edge on the side of a zone/
         * We distribute the edge positions around the middle of the side.
         * First and last edges are spaced by ~half of the side length
         */
        function computeEdgePositionOnZoneSide(zone, edgeIdx, nEdges, isRightOrLeftSide) {
            if (isRightOrLeftSide) {
                return zone.position.y + 0.5 * zone.position.height + (nEdges === 1 ? 0 : (-1/2 + edgeIdx / (nEdges - 1)) * 0.5 * zone.position.height)
            }

            return zone.position.x + 0.5 * zone.position.width + (nEdges === 1 ? 0 : (-1/2 + edgeIdx / (nEdges - 1)) * 0.5 * zone.position.width)
        }

        /**
         * Create or update the line SVG element to the edge G element that connects two zones
         */
        function addOrUpdateLine(edgeGElt, x, y, direction) {
            if (edgeGElt.find('line').length) {
                if (direction === "FROM") {
                    edgeGElt.find('line').attr('x1', `${x}`).attr('y1', `${y}`);
                } else {
                    // direction === "TO"
                    edgeGElt.find('line').attr('x2', `${x}`).attr('y2', `${y}`);
                }
            } else {
                edgeGElt[0].appendChild(makeSVG('line', {
                    x1: `${direction === "FROM" ? x : 0}`,
                    y1: `${direction === "FROM" ? y : 0}`,
                    x2: `${direction === "TO" ? x : 0}`,
                    y2: `${direction === "TO" ? y : 0}`,
                    stroke: "black"
                }));
            }
        }

        function computeZoneHeight(height, zoneId, collapsedZones) {
            if (!collapsedZones.includes(zoneId.replace(/^zone_/, ""))) {
                return height;
            } else {
                return COLLAPSED_ZONE_HEIGHT;
            }
        }

        function computeZoneWidth(width, zoneId, collapsedZones) {
            if (!collapsedZones.includes(zoneId.replace(/^zone_/, ""))) {
                return width;
            } else {
                return Math.min(COLLAPSED_ZONE_WIDTH, width);
            }
        }

        /**
         * Update all the zone dimensions in the nodeGraph taking into account
         * collapsed zones 
         */
        function updateZonesDimensions(nodesGraph, collapsedZones) {
            Object.values(nodesGraph.nodes).filter((node) => node.nodeType === "ZONE").forEach((node) => {
                node.position.width = computeZoneWidth(node.position.width, node.id, collapsedZones);
                node.position.height = computeZoneHeight(node.position.height, node.id, collapsedZones);
            })
        }

        /**
         * Find out for each edge which side it is better to connect to the reference zone
         * (based on both zone sizes & relative positions)
         */
        function dispatchZones(otherZones, referenceZone, nodesGraph, direction) {
            const rightThreshold = referenceZone.position.x + referenceZone.position.width;
            const leftThreshold = referenceZone.position.x;
            const bottomThreshold = referenceZone.position.y + referenceZone.position.height;
        
            let rightZones = [];
            let leftZones = [];
            let topZones = [];
            let bottomZones = [];
        
            otherZones.forEach((otherZoneId) => {
                const otherZone = nodesGraph.nodes[otherZoneId];
        
                if (otherZone.position.x > rightThreshold) {
                    rightZones.push({
                        id: otherZoneId,
                        position: otherZone.position,
                        direction: direction
                    });
                } else if (otherZone.position.y > bottomThreshold) {
                    bottomZones.push({
                        id: otherZoneId,
                        position: otherZone.position,
                        direction: direction
                    });
                } else if (otherZone.position.x + otherZone.position.width < leftThreshold) {
                    leftZones.push({
                        id: otherZoneId,
                        position: otherZone.position,
                        direction: direction
                    });
                } else {
                    topZones.push({
                        id: otherZoneId,
                        position: otherZone.position,
                        direction: direction
                    });
                }
            });
        
            return {rightZones, bottomZones, leftZones, topZones};
        }

        /**
         * Returns a sorting function for zones connected to the referenceZone.
         * Sorting is performed based on zones relative positions, and the fact that
         * the edge is FROM or TO the zone.
         */
        function zonesSortingFunction(referenceZoneId, onXAxis) {
            return function(zoneInfo1, zoneInfo2) {
                if (zoneInfo1.id === zoneInfo2.id) {
                    const fromBeforeTo = zoneInfo1.id < referenceZoneId;
                    
                    if (zoneInfo1.direction === "FROM" && zoneInfo2.direction === "TO") {
                        return fromBeforeTo ? -1 : 1;
                    } else {
                        return fromBeforeTo ? 1 : -1;
                    }
                    
                } else {
                    if (onXAxis) {
                        return zoneInfo1.position.x - zoneInfo2.position.x;
                    } else {
                        return zoneInfo1.position.y - zoneInfo2.position.y;
                    }
                }
            }
        }
    });

    app.service("RecipeFlowIconsService", function() {
        const DICT = {
            'sync': 'recipe_sync_circle_fill',
            'shaker': 'recipe_prepare_circle_fill',
            'update': 'recipe_push_to_editable_circle_fill',
            'sampling': 'recipe_filter_circle_fill',
            'grouping': 'recipe_group_circle_fill',
            'upsert': 'recipe_upsert_circle_fill',
            'distinct': 'recipe_distinct_circle_fill',
            'split': 'recipe_split_circle_fill',
            'topn': 'recipe_top_n_circle_fill',
            'sort': 'recipe_sort_circle_fill',
            'vstack': 'recipe_stack_circle_fill',
            'generate_features': 'recipe_auto_feature_circle_fill',
            'join': 'recipe_join_with_circle_fill',
            'fuzzyjoin': 'recipe_fuzzy_join_circle_fill',
            'geojoin': 'recipe_geo_join_circle_fill',
            'window': 'recipe_window_circle_fill',
            'export': 'recipe_export_circle_fill',
            'extract_failed_rows': 'recipe_extract_failed_rows',
            'pivot' : "recipe_pivot_circle_fill",
            'download': 'recipe_download_circle_fill',
            'merge_folder': 'recipe_merge_folder_circle_fill',
            'list_folder_contents': 'recipe_list_folder_contents_circle_fill',
            'list_access': 'recipe_list_access_circle_fill',
            'embed_documents': 'recipe_embed_documents_circle_fill',
            'extract_content': 'recipe_extract_content_circle_fill',

            'sql_script': 'recipe_sql_circle_fill',
            'sql_query': 'recipe_sql_circle_fill',

            'python': 'recipe_python_circle_fill',
            'julia': 'recipe_julia_circle_fill',
            'r': 'recipe_r_circle_fill',
            'shell': 'recipe_shell_circle_fill',

            'pig': 'recipe_pig_circle_fill',
            'hive': 'recipe_hive_circle_fill',
            'impala': 'recipe_impala_circle_fill',

            'pyspark': 'recipe_sparkpython_circle_fill',
            'sparkr': 'recipe_sparkr_circle_fill',
            'spark_scala': 'recipe_sparkscala_circle_fill',
            'spark_sql_query': 'recipe_sparksql_circle_fill',

            'clustering_cluster': 'recipe_cluster_circle_fill',
            'clustering_training': 'recipe_train_circle_fill',
            'clustering_scoring': 'recipe_cluster_circle_fill',
            'prediction_training': 'recipe_train_circle_fill',
            'prediction_scoring': 'recipe_score_circle_fill',
            'evaluation': 'recipe_evaluation_circle_fill',
            'standalone_evaluation': 'recipe_standalone_evaluation_circle_fill',

            'eda_pca': 'recipe_principal_component_analysis_circle_fill',
            'eda_stats': 'recipe_statistical_test_circle_fill',
            'eda_univariate': 'recipe_univariate_analysis_circle_fill',

            'csync': 'recipe_sync_circle_fill',
            'streaming_spark_scala': 'recipe_sparkscala_circle_fill',
            'cpython': 'recipe_python_circle_fill',
            'ksql': 'recipe_ksql_circle_fill',

            'prompt': 'recipe_prompt_circle_fill',
            'nlp_llm_user_provided_classification': 'recipe_text_classification_circle_fill',
            'nlp_llm_model_provided_classification': 'recipe_text_classification_circle_fill',
            'nlp_llm_summarization': 'recipe_summarization_circle_fill',
            'nlp_llm_finetuning': 'recipe_fine_tuning_circle_fill',
            'nlp_llm_rag_embedding': 'recipe_rag_embedding_circle_fill',
            'nlp_llm_evaluation': 'recipe_evaluation_llm_circle_fill',
            'nlp_agent_evaluation': 'recipe_evaluation_agent_circle_fill',

            'labeling_task': 'recipe_labeling_circle_fill' // Special case for labeling which is not a real recipe
        };

        return {getRecipeFlowIcon};

        function getRecipeFlowIcon(recipeType) {
            if (!!recipeType && (recipeType.startsWith('CustomCode_') || recipeType.startsWith("App_"))) {
                return 'recipe_empty_circle_fill'
            }
            return DICT[recipeType] || 'icon-'+(recipeType||'').toLowerCase();
        };
    });

    app.filter("recipeFlowIcon", function(RecipeFlowIconsService) {
        return function(recipeType) {
            return RecipeFlowIconsService.getRecipeFlowIcon(recipeType);
        };
    });

    app.directive('flowCommon', function($state, $stateParams, $rootScope, FlowGraphSelection, FlowGraph, GraphZoomTrackerService) {
        return {
            restrict: 'EA',
            scope: true,
            link : function(scope, element) {

                function setRightColumnItem() {
                    // This is a quick and dirty hack to keep compatibility with old
                    // right columns that handle only one element at a time
                    scope.rightColumnSelection = FlowGraphSelection.getSelectedNodes();
                    if (scope.rightColumnSelection.length == 0) {
                        scope.rightColumnItem = null;
                        if ($stateParams.zoneId) { 
                            // when zoomed on a flow zone, default to the zone right panel
                            const node = FlowGraph.node(`zone_${$stateParams.zoneId}`);
                            if (node) {
                                scope.rightColumnItem = node;
                                scope.rightColumnSelection = [node];
                                GraphZoomTrackerService.setFocusItemCtx(node);
                            }
                        } else if ($state.current.name.startsWith('projects.project.flow')) { 
                            // when on a project flow, default to the project right panel
                            scope.rightColumnItem = {
                                nodeType: 'PROJECT'
                            };
                        }
                    } else if (scope.rightColumnSelection.length == 1) {
                        scope.rightColumnItem = scope.rightColumnSelection[0];
                    } else if (scope.rightColumnSelection.every((node) => ["DATASET", "LOCAL_DATASET", "FOREIGN_DATASET"].includes(node.nodeType))) {
                        scope.rightColumnItem = {
                            nodeType: 'MULTI_DATASETS',
                            selection: scope.rightColumnSelection
                        };
                    } else {
                        scope.rightColumnItem = {
                            nodeType: 'MULTI',
                            selection: scope.rightColumnSelection
                        };
                    }
                }

                scope.focusLast = function(){
                    if (scope.previousRightColumnItemId) {
                        FlowGraphSelection.clearSelection();
                        scope.zoomGraph(scope.previousRightColumnItemId);
                        FlowGraphSelection.onItemClick(scope.previousRightColumnItemId);
                    }
                };

                const h = $rootScope.$on('flowSelectionUpdated', setRightColumnItem);
                scope.$on('$destroy', h);
            }
        }
    });


    // Main directive for all graphs (project flow, global graph, job graph)
    app.directive('flowGraph', function($rootScope, $stateParams, Logger, FlowGraph, ProjectFlowGraphStyling, InterProjectGraphStyling, ProjectFlowGraphLayout,
                                    InterProjectGraphLayout, FlowGraphFiltering, FlowGraphSelection, JobFlowGraphSelection, FlowGraphFolding, GraphZoomTrackerService, 
                                    Debounce, DataikuAPI, FlowZoneMoveService, DetectUtils) {
                                    
    return {
        restrict: 'EA',
        controller: function ($scope, $element, $attrs) {
            $scope.FlowGraph = FlowGraph;//debug
            // Initialize Zoom tracking context (to make sure the last persisted one will be reloaded in case an empty flow was loaded previously)
            GraphZoomTrackerService.initCtx();
            // Enable Zoom tracking
            GraphZoomTrackerService.disable(false);
            let nextResizeEnabled = true;

            function disableNextResize(disabled = true) {
                nextResizeEnabled = disabled === false;
            }

            $scope.$on('disableNextFlowResize', disableNextResize);

            $scope.setGraphData = function setGraphData (serializedGraph) {
                if (serializedGraph) {
                    $scope.nodesGraph = serializedGraph;
                    if ($scope.zonesManualPositioning) {
                        FlowZoneMoveService.updateZonesDimensions($scope.nodesGraph, $scope.collapsedZones);
                    }
                    FlowGraph.set($scope.nodesGraph);
                    $scope.nodesGraph.nodesOnGraphCount = Object.keys(serializedGraph.nodes).filter(it => $stateParams.zoneId ? it.startsWith(`zone__${$stateParams.zoneId}`) : true).length; //if zoomed on a zone we need the zone node for the color, but not in the filter count

                    $scope.isFlowEmpty = !serializedGraph.hasZoneSharedObjects && $scope.nodesGraph.nodesOnGraphCount === 0;
                    if($scope.FlowGraph.nodeCount() > 100){
                        $element.addClass('performance');
                    }
                }
            };
            function setupZoomBehavior() {
                // The scope.panZoom structure controls:
                // - the position of the flow on the screen, via  SVG coords x,y
                // - How zoomed in it is, via SVG dimensions width, height
                // - How far out you can zoom, via maxWidth, maxHeight, SVG dimensions
                //
                // The size of the HTML container element in HTML coords is held in WIDTH and HEIGHT
                // x, y, width, height are used directly as the SVG viewBox settings.
                $scope.panzoom = {
                    x: 0,
                    y: 0,
                    width: undefined,
                    height: undefined,
                    maxWidth: undefined,
                    maxHeight: undefined,
                    WIDTH: $element.legacyWidth(),
                    HEIGHT: $element.legacyHeight()
                };

                let h;

                const resizeStrategies = {
                    reinit : "reinit", // don't use any saved settings, redisplay whole flow
                    usePanZoom : "zoom", // use the saved zoom context, but don't smart-adjust the positioning at all
                    zoomToFocusItem : "item", // use the saved zoom contrext, but adjust centring for a 'nice fit'
                    highlight : "highlight" // resize around the highlighted nodes
                };

                const smallFlowNodeCount = 10; // any flow less than 10 nodes can be displayed in full everytime.

                /**
                 * Return the set strategy if some, or calculate the best strategy for restoring the saved zoom and focused item. We want as natural
                 * an experience as possible.  If the user hasn't changed anything, we try to restore the flow layout
                 * exactly as it was before - they presumably set it that way!  If they have added/deleted/navigated
                 * then we need to adjust layout so they can see the last item they navigated to.
                 *
                 * @param currentNodeCount - the current number of nodes in the flow
                 * @returns the resize strateegy
                 */
                function getResizeStrategy(currentNodeCount) {
                    let strategy = resizeStrategies.reinit;

                    if ($scope.strategy) {
                        strategy = $scope.strategy;
                    } else {
                        if ($rootScope.appConfig.userSettings.disableFlowZoomTracking || !currentNodeCount) {
                            return strategy;
                        }

                        if (GraphZoomTrackerService.isValidPanZoom($scope.panzoom)) {
                            const focusItemId = GraphZoomTrackerService.getSafeFocusItemId($scope.FlowGraph);
                            if (focusItemId && $scope.FlowGraph.node(focusItemId) && GraphZoomTrackerService.wasNodeChangedOutsideFlow()) {
                                strategy = resizeStrategies.zoomToFocusItem;
                            } else {
                                strategy = resizeStrategies.usePanZoom;
                            }
                        }

                        const previousNodeCount = GraphZoomTrackerService.getNodeCount()
                        if (currentNodeCount <= smallFlowNodeCount && currentNodeCount != previousNodeCount) { //small flows with a flow change
                            strategy = resizeStrategies.reinit; // with small flows, always snap back to the full flow
                        }
                    }
                    return strategy;
                }

                function setResizeStrategy(scope, strategy='reinit') {
                    if (strategy && resizeStrategies[strategy]) {
                        $scope.strategy = resizeStrategies[strategy];
                    }
                }

                $scope.$on('setResizeStrategy', setResizeStrategy);

                /**
                 * Calculate how close the bounding box for an item is to the edge of the viewable area, as
                 * a fraction of the total size of the view.  This function does the calculation for a
                 * specified dimension i.e. the X or Y axis.
                 *
                 * @param bbItem - the bound box of one item in the viewable area (the one we want to set focus to)
                 * @param bbFlow - the bounding box of the whole flow
                 * @param dimension  'x' for the horizontal dimension, 'y' for vertical
                 * @param viewBoxLength - the viewable size of the SVG window in the specified dimension
                 * @returns the fraction of the total size from the edge.  For example 0.5 means we are in the middle
                 */
                function getFractionalDistOfItemToEdge(bbItem, bbFlow, dimension, viewBoxLength) {

                    const length = dimension=='x' ? 'width' : 'height';
                    const start = dimension; // x or y axis

                    let extentOfItem = bbItem[start] + bbItem[length];
                    let extentOfFlow = bbFlow[start] + bbFlow[length];
                    return (extentOfFlow - extentOfItem) /  viewBoxLength;
                }

                /**
                 * We need the .width / .height aspect ratio for the SVG viewbox to match the
                 * aspect ratio of the containing HTML element, otherwise we get inaccuracies when
                 * we try to drag the whole flow around.
                 * @param pz - the pan zoom to be normalised
                 * returns - the same pan zoom, normalised
                 */
                function normaliseAspectRatio(pz) {
                    const viewBoxAR = pz.width / pz.height;
                    const elementAR = pz.WIDTH / pz.HEIGHT;
                    if (viewBoxAR!=elementAR) {
                        pz.width = pz.height * elementAR; // not sure we need to worry about which way round we adjust AR
                    }
                    return pz;
                }

                function resize(forcedReinit) {
                    let zoomTrackingEnabled = GraphZoomTrackerService.isEnabled();
                    if ($scope.isFlowEmpty || !$scope.svg || !$scope.svg.length) {
                        return false;
                    }

                    $scope.panzoom.WIDTH = $element.legacyWidth();
                    $scope.panzoom.HEIGHT = $element.legacyHeight();

                    const bbFlow = $scope.svg.find('g.graph')[0].getBBox(); //get the whole-flow bounding box
                    const defaultPz = buildDefaultPanZoomSettings($scope.panzoom, bbFlow);
                    let pz = $scope.panzoom;
                    if (!GraphZoomTrackerService.isValidPanZoom($scope.panzoom)) {
                        pz = angular.copy(defaultPz);
                    }
                    let isReloadedZoomCtx;
                    if (zoomTrackingEnabled) {
                        isReloadedZoomCtx = (GraphZoomTrackerService.restoreZoomCtx(pz, defaultPz));
                    }

                    $scope.setPanZoom(normaliseAspectRatio(pz));

                    const nodeCount = $scope.FlowGraph.nodeCount();
                    let strategy = getResizeStrategy(nodeCount);
                    let nodeIdToFocus = GraphZoomTrackerService.getSafeFocusItemId($scope.FlowGraph);

                    if (forcedReinit === true) {
                        strategy = resizeStrategies.reinit;
                        nodeIdToFocus = undefined;
                        FlowGraphFolding.clearFoldState();
                        FlowGraph.setGraphBBox(bbFlow);
                    }
                    if (strategy == resizeStrategies.usePanZoom) { // we have saved viewbox setting ie zoom level and positioning.  Reuse these
                        if (isReloadedZoomCtx) $scope.redraw();
                        // SVG will rescale quite nicely by itself with changes in Window size
                    }
                    else if (strategy == resizeStrategies.zoomToFocusItem) { // position this last-used item in the centre of the screen
                        const wPrev = $scope.panzoom.width;
                        const hPrev = $scope.panzoom.height;
                        const node = $scope.nodesGraph.nodes[nodeIdToFocus];
                        const selector = $scope.getSelector(nodeIdToFocus, node);
                        let bb = FlowGraphFiltering.getBBoxFromSelector($scope.svg, selector); //get the focussed-item box

                        let xPosInCell = 0.5; // the middle of the viewable area
                        let yPosInCell = 0.5;

                        const centerOnItemX = bbFlow.width > wPrev; // do we want to try to center the item, or does the whole flow fits in this dimension?
                        const centerOnItemY = bbFlow.height > hPrev;

                        let boxToCentreForX = centerOnItemX ? bb: bbFlow;
                        let boxToCentreForY = centerOnItemY ? bb : bbFlow;

                        if (centerOnItemX) {
                            // we are centring the item in the width of the viewport.
                            // By default we put it in the middle, but if it's near the edges, we adjust it a bit with heuristically developed numbers ;-)
                            const edgeFraction = getFractionalDistOfItemToEdge(bb, bbFlow, "x", wPrev);
                            if (edgeFraction < 0.25) xPosInCell = 0.8 - edgeFraction; // near right hand edge

                            if ((bb.x - bbFlow.x) / wPrev < 0.25)  xPosInCell = 0.25; // near left hand edge
                        }

                        if (centerOnItemY) {
                            // we are centring the item in the height of the item.
                            const edgeFraction = getFractionalDistOfItemToEdge(bb, bbFlow, "y", hPrev) ;
                            if (edgeFraction < 0.2) yPosInCell = 0.8 - edgeFraction;

                            if ((bb.y - bbFlow.y) / hPrev < 0.25) yPosInCell = 0.25;
                        }

                        $scope.panzoom.x = boxToCentreForX.x + boxToCentreForX.width * xPosInCell - wPrev * xPosInCell; //centre the view
                        $scope.panzoom.y = boxToCentreForY.y + boxToCentreForY.height * yPosInCell -hPrev * yPosInCell;
                        $scope.panzoom.height = hPrev; //keep same zoom level as before
                        $scope.panzoom.width = wPrev;

                        $scope.redraw();
                    } else if (strategy === resizeStrategies.highlight) {
                        // Zoom on highlighted nodes
                        let bbox = FlowGraphFiltering.getBBoxFromSelector($scope.svg, '.highlight');
                        $scope.zoomToBbox(bbox);
                    } else { // refit the whole flow.  May have focus item
                        $scope.setPanZoom(defaultPz);

                        let paddingFactor = 1.2;
                        if (nodeCount) paddingFactor += (smallFlowNodeCount-Math.min(nodeCount, smallFlowNodeCount)) * 0.08; // more padding when there are fewer items in the flow.
                        zoomTo(paddingFactor);
                    }

                    // if nodeIdToFocus is set, we want to select this node in the flow and keep the current selection.
                    // We call FlowGraphSelection.selectNodesFromIds with the list of nodes to select.
                    // However we need to force the existing FlowGraph data
                    // structure to load the current set of node ids before this will work.
                    if (nodeIdToFocus) {
                        FlowGraph.indexNodesCoordinates($scope.svg, bbFlow);
                        let selected = FlowGraphSelection.getSelectedNodes() || []
                        const selectedIdsWithNodeIdToFocus = Array.from(new Set([...selected.map(node=>node.id), nodeIdToFocus])) // combine the selected nodes with the node to focus and make sure the ids are unique.
                        FlowGraphSelection.selectNodesFromIds(selectedIdsWithNodeIdToFocus)
                        applyFlowFolding();
                    } else {
                        h && clearTimeout(h);
                        h = setTimeout(function() {
                            FlowGraph.indexNodesCoordinates($scope.svg, bbFlow);
                            applyFlowFolding();
                        }, 500);
                    }

                    zoomTrackingEnabled === true && GraphZoomTrackerService.setFlowRedrawn(nodeCount);

                    return true;
                }
                function applyFlowFolding() {
                    if ($attrs.showFolding) FlowGraphFolding.restoreState(GraphZoomTrackerService.getFoldState());
                }
                function resizeListener() {
                    resize();
                    if (!$scope.$$phase) {
                        $scope.$apply();
                    }
                }
                // Add 20ms debounce on resize because it is costly for big flows
                var debouncedResizeListener = Debounce().withDelay(20,20).wrap(resizeListener);
                $(window).on('resize', debouncedResizeListener);
                $scope.$on('$destroy', param => {
                    if (!param.currentScope.projectFlow || !(param.currentScope.nodesGraph && param.currentScope.nodesGraph.hasZones || param.currentScope.zoneIdLoaded)) {
                        GraphZoomTrackerService.instantSavePanZoomCtx($scope.panzoom);
                    }
                    $(window).off('resize', debouncedResizeListener);
                });
                $scope.$on('resizePane', resize);
                $scope.$watch('svg', function() {
                    // Wait for the svg to be totally rendered before updating the view (we need to make computation based on bbox)
                    // If the resize has been temporarily disabled, re-enable it.
                    if (nextResizeEnabled) {
                        setTimeout(function() {
                            if (resize($scope.isResetZoomNeeded)) $rootScope.$emit('flowDisplayUpdated');
                            $scope.isResetZoomNeeded = false;
                        });
                    } else {
                        nextResizeEnabled = true;
                    }
                });

                function keepPanZoomDimSane(pz, bbflow, start, length, nearFraction) {
                    if (pz[start] + (1-nearFraction)*pz[length] < bbflow[start]) { //rhs
                        pz[start] = (nearFraction-1)*pz[length] + bbflow[start];
                    }
                    if (bbflow[length] + bbflow[start] < pz[start] + nearFraction * pz[length]) { // lhs
                        pz[start] = (bbflow[start] + bbflow[length] - nearFraction*pz[length]);
                    }
                    return pz;
                }

                function keepPanZoomSane(pz) { //prevent flow being pushed out of view
                    const bbFlow = $scope.svg.find('g.graph')[0].getBBox();
                    pz = keepPanZoomDimSane(pz, bbFlow, "x", "width", 0.3);
                    pz = keepPanZoomDimSane(pz, bbFlow, "y", "height", 0.4);
                    return pz;
                }

                $scope.redraw = function () {
                    if ($scope.svg && $scope.svg.length) {
                        $scope.panzoom = keepPanZoomSane($scope.panzoom);
                        GraphZoomTrackerService.lazySaveZoomCtx($scope.panzoom);
                        $scope.svg[0].setAttribute('viewBox', [
                            $scope.panzoom.x,
                            $scope.panzoom.y,
                            $scope.panzoom.width,
                            $scope.panzoom.height
                        ].join(', '));
                    }
                };

                $scope.getPanZoom = function() {
                    return angular.copy($scope.panzoom);
                };

                $scope.setPanZoom = function(panzoom) {
                    $scope.panzoom = angular.copy(panzoom);
                    $scope.redraw();
                };

                $scope.bbox = function(bbox) {
                    $scope.panzoom.x = bbox.x;
                    $scope.panzoom.y = bbox.y;

                    if (bbox.width / bbox.height > $scope.panzoom.WIDTH / $scope.panzoom.HEIGHT) {
                        $scope.panzoom.width = bbox.width;
                        $scope.panzoom.height = bbox.width * ($scope.panzoom.HEIGHT / $scope.panzoom.WIDTH);
                        $scope.panzoom.y = bbox.y - ($scope.panzoom.height - bbox.height) / 2;
                    } else {
                        $scope.panzoom.width = bbox.height * ($scope.panzoom.WIDTH / $scope.panzoom.HEIGHT);
                        $scope.panzoom.height = bbox.height;
                        $scope.panzoom.x = bbox.x - ($scope.panzoom.width - bbox.width) / 2;
                    }

                    $scope.redraw();
                };
                // Safari, Chrome, Opera, IE
                const WHEEL_ZOOM_STEP = 1.1;
                $element.on('mousewheel', function (e) {
                    let scale = 1;
                    if (e.originalEvent.wheelDeltaY != 0) {
                        scale = e.originalEvent.wheelDeltaY < 0 ? WHEEL_ZOOM_STEP : 1 / WHEEL_ZOOM_STEP;
                    }
                    const eOrig = e.originalEvent && angular.isDefined(e.originalEvent.layerX) ? e.originalEvent : undefined;
                    zoomTo(scale, eOrig ? eOrig.layerX : e.offsetX, eOrig ? eOrig.layerY : e.offsetY);
                    e.stopPropagation();
                    e.preventDefault();
                });

                Mousetrap.bind("Z R", () => $scope.resizeToShowAll());
                Mousetrap.bind("Z A", () => $scope.resizeToShowAll());

                $scope.reinitGraph = () => { resize(true); };

                $scope.$on("$destroy", _ => {
                    Mousetrap.unbind("Z R");
                    Mousetrap.unbind("Z A");
                });

                $scope.zoomIn = () => {
                    zoomTo(1 / WHEEL_ZOOM_STEP, $scope.panzoom.WIDTH / 2, $scope.panzoom.HEIGHT / 2);
                }

                $scope.zoomOut = () => {
                    zoomTo(WHEEL_ZOOM_STEP, $scope.panzoom.WIDTH / 2, $scope.panzoom.HEIGHT / 2);
                }

                $scope.resizeToShowAll = () => {
                    $scope.resetPanZoom();
                    $rootScope.$emit('drawGraph', true, true);
                    FlowGraphSelection.refreshStyle(true);
                }

                $scope.resetPanZoom = function () {
                    $scope.isResetZoomNeeded = true;
                }

                // Touchable devices
                if (isTouchDevice()) {
                    let onTouchStart = (function() {
                        let previousPinchDistance;

                        /*
                            * Utils
                            */

                        function computePinchDistance(e) {
                            let t1 = e.originalEvent.touches[0];
                            let t2 = e.originalEvent.touches[1];
                            let distance = Math.sqrt(Math.pow(Math.max(t1.screenX, t2.screenX) - Math.min(t1.screenX, t2.screenX), 2) + Math.pow(Math.max(t1.screenY, t2.screenY) - Math.min(t1.screenY, t2.screenY), 2));
                            return distance;
                        }

                        function computePinchMiddle(e) {
                            let t1 = e.originalEvent.touches[0];
                            let t2 = e.originalEvent.touches[1];
                            let offset = $($element).offset();
                            // point of contact 1
                            let c1 = {
                                x: t1.pageX - offset.left,
                                y: t1.pageY - offset.top
                            };
                            // point of contact 2
                            let c2 = {
                                x: t2.pageX - offset.left,
                                y: t2.pageY - offset.top
                            };
                            // middle
                            let middle = {
                                x: (c1.x + c2.x)/2,
                                y: (c1.y + c2.y)/2
                            };
                            return middle;
                        }

                        /*
                            * Callbacks
                            */

                        function onTouchMove(e) {
                            e.stopPropagation();
                            e.preventDefault();
                            let distance = computePinchDistance(e);
                            if (!isNaN(previousPinchDistance)) {
                                let scale = previousPinchDistance / distance;
                                let middle = computePinchMiddle(e);
                                requestAnimationFrame(_ =>zoomTo(scale, middle.x, middle.y));
                            }
                            previousPinchDistance = distance;
                        }

                        function onTouchEnd(e) {
                            $element.off('touchmove', onTouchMove);
                            $element.off('touchend', onTouchEnd);
                            e.stopPropagation();
                            e.preventDefault();
                        }

                        return function(e){
                            e.stopPropagation();
                            e.preventDefault();
                            if (e.originalEvent.targetTouches.length !== 2) {
                                return;
                            }

                            previousPinchDistance = computePinchDistance(e);

                            $element.on('touchmove', onTouchMove);
                            $element.on('touchend', onTouchEnd);
                        }
                    })();

                    $element.on('touchstart', onTouchStart);
                }

                // Firefox
                $element.on('DOMMouseScroll', function (e) {
                    const scale = e.originalEvent.detail > 0 ? WHEEL_ZOOM_STEP : 1 / WHEEL_ZOOM_STEP;
                    const coordinates = mouseViewportCoordinates(e);
                    zoomTo(scale, coordinates.x, coordinates.y);
                    e.stopPropagation();
                });
                function zoomTo(scale, x, y) {
                    if (scale < 1 && $scope.panzoom.width && $scope.panzoom.width <= 150) {
                        return; // cannot zoom infinitely
                    }
                    if(scale > 1 && $scope.panzoom.width * scale > $scope.panzoom.maxWidth*2) {
                        return; // cannot dezoom infinitely
                    }
                    if (angular.isUndefined(x)) {
                        x = $scope.panzoom.WIDTH / 2;
                    }
                    if (angular.isUndefined(y)) {
                        const menuH = $('#flow-editor-page .menu').legacyHeight();
                        y = ($scope.panzoom.HEIGHT - menuH) / 2 + menuH;
                    }

                    $scope.panzoom.x = $scope.panzoom.x + (x / $scope.panzoom.WIDTH) * $scope.panzoom.width * (1 - scale);
                    $scope.panzoom.y = $scope.panzoom.y + (y / $scope.panzoom.HEIGHT) * $scope.panzoom.height * (1 - scale);

                    $scope.panzoom.width = $scope.panzoom.width * scale;
                    $scope.panzoom.height = $scope.panzoom.height * scale;

                    $scope.redraw();
                }
            }

            let original_coordinates;
            let original_graph_coordinates;
            let original_click;
            
            // Current translation applied to a flow zone that is being moved
            let currentZoneTranslation;

            // Id of the zone that is being moved
            let zoneId;  

            function getEventWithCoordinates(evt) {
                return angular.isUndefined(evt.originalEvent.changedTouches) ? evt.originalEvent : evt.originalEvent.changedTouches[0];
            }

            /* offsetX and offsetY are not really supported in Firefox. They used to be undefined, but May'18 and they are returning 0.
                It's not clear if this was always the case, but this getOffsetXY function implemented the broadly excepted
                substitute calculation.  Clearly userAgent test is not a great solution, but leaving the Chrome solution
                in place seems safer and more efficient.
                */
            const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
            function getOffsetXY(e) {
                let offsets;

                if (isFirefox || angular.isUndefined(e.offsetX)) { // having problem with offsetX as zero, not undefined, on firefox
                    const target = e.target || e.srcElement;
                    const rect = target.getBoundingClientRect();
                    offsets = { x:   e.clientX - rect.left, y:  e.clientY - rect.top};
                }
                else {
                    offsets = { x: e.offsetX, y: e.offsetY};
                }
                return offsets;
            }

            /*
            * Gives the coordinates of a mouse event in a coordinate system fixed relative viewport
            */
            function mouseViewportCoordinates(evt) {
                const formattedEvt = getEventWithCoordinates(evt);
                //mjt I have a feeling that the existing if clause and the new getOffsetXY function are
                // trying to do similar things and might be conflatable.  It's not trivial though.
                if (angular.isUndefined(formattedEvt.offsetX) || !$scope.svg || formattedEvt.target != $scope.svg[0]) {
                    const containerOffset = $element.offset();
                    return {
                        x: formattedEvt.pageX - containerOffset.left,
                        y: formattedEvt.pageY - containerOffset.top
                    };
                } else {
                    return getOffsetXY(formattedEvt)
                }
            }
            /*
            * Gives the coordinates of a mouse event in the graph coordinate systems:
            * Ex: if the graph is resized and moved, clicking on one specific item would give the same coordinates
            */
            let pt;

            function mouseGraphCoordinates(evt) {
                let formattedEvt = getEventWithCoordinates(evt);
                pt = pt || $scope.svg[0].createSVGPoint();
                pt.x = formattedEvt.clientX;
                pt.y = formattedEvt.clientY;
                return pt.matrixTransform($scope.svg[0].getScreenCTM().inverse());
            }

            function setupMoveAndSelectBehavior() {
                $scope.rectangleSelection = false;

                function moveView(evt) {
                    if (!original_coordinates) return;
                    const coordinates = mouseViewportCoordinates(evt);
                    $scope.panzoom.x = original_coordinates.x - $scope.panzoom.width * (coordinates.x / $scope.panzoom.WIDTH);
                    $scope.panzoom.y = original_coordinates.y - $scope.panzoom.height * (coordinates.y / $scope.panzoom.HEIGHT);
                    $scope.redraw();
                }

                function applyMoveZone(evt) {
                    const translation = getTranslationCoordinates(evt);
                    currentZoneTranslation = translation;

                    // Apply translation to svg in scope
                    $scope.svg.find(`g[data-id="${zoneId}"]`).attr('transform', `translate(${translation.x}, ${translation.y})`);

                    // Also apply translation to node graph svg (because it is use to redraw the graph in `drawGraph` event listener)
                    // This is a shame, but probably less costly than refreshing the whole flow
                    const nodeGraphSvgElement = $('<div></div>').append($($scope.nodesGraph.svg)).find('>svg');
                    nodeGraphSvgElement.find(`g[data-id="${zoneId}"], g[data-id="cluster_${zoneId}"]`).attr('transform', `translate(${translation.x}, ${translation.y})`);
                    $scope.nodesGraph.svg = nodeGraphSvgElement[0].outerHTML;
                    let nodeGraphZone = $scope.getNode(zoneId);
                    nodeGraphZone.position.x = translation.x;
                    nodeGraphZone.position.y = translation.y;

                    // Then move the relevant edges
                    FlowZoneMoveService.setZoneEdges($scope.svg, zoneId, $scope.nodesGraph);
                    $scope.svg.find(`g[data-to="${zoneId}"]`).each(function() {
                        FlowZoneMoveService.setZoneEdges($scope.svg, this.getAttribute("data-from"), $scope.nodesGraph);
                    })
                    $scope.svg.find(`g[data-from="${zoneId}"]`).each(function() {
                        FlowZoneMoveService.setZoneEdges($scope.svg, this.getAttribute("data-to"), $scope.nodesGraph);
                    })
                }

                function getTranslationCoordinates(evt) {
                    let oldX = null;
                    let oldY = null;
                    try {
                        const parts = /translate\(\s*([^\s,)]+)[\s,]\s?([^\s,)]+)/.exec(evt.data.transformAttribute);
                        oldX = parseFloat(parts[1]);
                        oldY = parseFloat(parts[2]);
                    } catch (_) {
                        oldX = 0;
                        oldY = 0; 
                    }

                    const coordinates = mouseViewportCoordinates(evt);
                    return {
                        x: oldX + (coordinates.x - original_click.x) * $scope.panzoom.width / $scope.panzoom.WIDTH,
                        y: oldY + (coordinates.y - original_click.y) * $scope.panzoom.height / $scope.panzoom.HEIGHT
                    }
                }

                function getBBoxFromPoints(p1, p2) {
                    return {
                        x: Math.min(p1.x, p2.x),
                        y: Math.min(p1.y, p2.y),
                        width: Math.max(p1.x, p2.x) - Math.min(p1.x, p2.x),
                        height: Math.max(p1.y, p2.y) - Math.min(p1.y, p2.y),
                    };
                }

                function updateSelectionRectangle(evt) {
                    const svg = FlowGraph.getSvg();
                    if (!svg) return;
                    clearSelectionRectangle();

                    const coords = mouseGraphCoordinates(evt);
                    const rect = getBBoxFromPoints(coords, original_graph_coordinates);
                    rect.id = 'flow-selection-rectangle';

                    $(svg).append(makeSVG('rect', rect));
                }

                function clearSelectionRectangle() {
                    const svg = FlowGraph.getSvg();
                    $('#flow-selection-rectangle', svg).remove();
                }

                function commitRectangleSelection(evt) {
                    const coords = mouseGraphCoordinates(evt);
                    const rect = getBBoxFromPoints(coords, original_graph_coordinates);
                    const nodeIds = FlowGraph.getEnclosedNodesIds(rect);
                    if (FlowGraph.isJobGraph()) {
                        JobFlowGraphSelection.rectangularSelectionStrategy($scope)(FlowGraph.nodes(nodeIds));
                    } else {
                        FlowGraphSelection.onRectangularSelection(nodeIds);
                    }
                }

                function clearOriginalCoordinates() {
                    original_coordinates = undefined;
                    clearSelectionRectangle();
                    $('#flow-editor-page .mainPane').removeClass('no-pointer-events');
                }

                $scope.isSavingZonePosition = false;
                const debouncedSaveZonePosition = Debounce().withDelay(400, 400).wrap(saveZonePosition);
                function saveZonePosition() {
                    if (zoneId && currentZoneTranslation) {
                        const projectKey = $stateParams.projectKey;
                        $scope.isSavingZonePosition = true;
                        DataikuAPI.flow.zones.savePosition(projectKey, zoneId.replace(/^zone_/, ""), currentZoneTranslation.x, currentZoneTranslation.y)
                            .catch(setErrorInScope.bind($scope))
                            .finally(() => {
                                $scope.isSavingZonePosition = false;
                            });
                    }
                    currentZoneTranslation = undefined;
                }

                function isLassoKeyPressed(evt) {
                    const isNotMac = !$('html').hasClass('macos')
                    return evt.shiftKey || evt.metaKey || (evt.ctrlKey && isNotMac);
                }

                $('body').on('keydown.rectangleSelection', function (e) {
                    if ((isLassoKeyPressed(e)) && $stateParams.projectKey) {
                        $scope.$apply(() => $scope.rectangleSelection = true);
                    }
                }).on('keyup.rectangleSelection', function (e) {
                    if ((!isLassoKeyPressed(e)) && $stateParams.projectKey) {
                        $scope.$apply(() => $scope.rectangleSelection = false);
                    }
                });

                $scope.$on('$destroy', function() {
                    $('body').off('keydown.rectangleSelection').off('keyup.rectangleSelection');
                });

                /**
                 * Return a function taking an event in input and calling the callback passed on parameter only if this event is a mono touch event
                 */
                function monoTouchCallBack(fn) {
                    return function(evt) {
                        if (evt.originalEvent.touches && evt.originalEvent.touches.length == 1) {
                            fn(evt);
                        }
                    }
                }

                let dragStart = (function() {
                    /* in order to avoid lag on big flows, when dragging we apply the move cursor to the item initially clicked, and on each item entered during the drag
                     * it performs better because the bottleneck is the style recalculation, not the javascript event part
                     */
                    const applyMoveCursorSelector = 'svg, svg [class~=node], svg [class~=zone_cluster], svg [class~=folded-icon]';
                    let $applyMoveCursorInitialItem = $element;
                    function applyMoveCursor(evt) {
                        $(evt.target).addClass('moving');
                        $(evt.target).parentsUntil($element).addClass('moving');
                    }
                    function cleanMoveCursor() {
                        $element.find('.moving').removeClass('moving');
                    }

                    /*
                        *  Utils
                        */
                    // Remove listeners on drag release
                    function removeListeners() {
                        $(document).off('mousemove', drag);
                        $(document).off('mousemove', moveZone);
                        $element.off('mouseenter', applyMoveCursorSelector, applyMoveCursor);
                        $element.off('mouseup', 'svg .node', releaseDragOnNode);
                        $element.off('mouseup', 'svg .zone_cluster', releaseDragOnZone);
                        $element.off('mouseup', 'svg .folded-icon', releaseDragOnUnfoldButton);
                        $(document).off('mouseup', releaseDrag);

                        if (isTouchDevice()) {
                            $(document).off('touchmove', monoTouchCallBack(drag));
                            $element.off('touchend', 'svg .node', releaseDragOnNode);
                            $element.off('touchend', 'svg .zone_cluster', releaseDragOnZone);
                            $element.off('touchend', 'svg .folded-icon', releaseDragOnUnfoldButton);
                            $(document).off('touchend', releaseDrag);
                        }
                    }

                    /*
                    * Callbacks
                    */

                    function _dragCallback(evt, applyDragMethod) {
                        if (!$scope.svg) return;

                        $applyMoveCursorInitialItem.addClass('moving');
                        // we make sure that the top svg element have the class moving applied.
                        $applyMoveCursorInitialItem.closest('#flow-graph > svg').addClass('moving');
                        if (!((isLassoKeyPressed(evt)) && $stateParams.projectKey) && $scope.rectangleSelection) {
                            //This is an old rectangleSelection that was not cleared
                            clearOriginalCoordinates();
                        }
                        if (!(isLassoKeyPressed(evt)) && $scope.rectangleSelection) {
                            $scope.$apply(() => $scope.rectangleSelection = false);
                        }
                        if (original_coordinates) {
                            if ($scope.rectangleSelection) {
                                const coordinates = mouseViewportCoordinates(evt);
                                if (original_click && square_distance(coordinates, original_click) > 16) {
                                    // We don't want to immediately add no-pointer-events because cmd+click will not work (1 pixel moves)
                                    $('#flow-editor-page .mainPane').addClass('no-pointer-events');
                                }
                                updateSelectionRectangle(evt);
                            } else {
                                requestAnimationFrame(_ => applyDragMethod(evt));
                            }
                        }

                        evt.stopPropagation();
                        if (!$rootScope.$$phase) $scope.$apply();
                    }

                    // Ondrag flow zone callback 
                    function moveZone(evt) {
                        _dragCallback(evt, applyMoveZone);
                    }

                    // Ondrag whole flow callback
                    function drag(evt) {
                        _dragCallback(evt, moveView);
                    }

                    // Ondragend on node callback
                    function releaseDragOnNode(e) {

                        if (e.button === 2) {
                            // do nothing on right-click
                            return;
                        }

                        cleanMoveCursor();
                        const coordinates = mouseViewportCoordinates(e);
                        // traveled distance
                        if (original_click) {
                            if (square_distance(coordinates, original_click) < 16) {
                                // if mouse has moved less than 4px
                                if ((e.which == 1) || (e.which == 2) || (e.type == 'touchend')) {
                                    const nodeId = $(this).attr('data-id');
                                    const node = FlowGraph.node(nodeId);
                                    if (node) {
                                        GraphZoomTrackerService.setFocusItemCtx(node);
                                        FlowGraphSelection.onItemClick(node, e);
                                        $scope.$apply();
                                    }
                                }
                            } else if ($scope.rectangleSelection) {
                                commitRectangleSelection(e);
                            }
                        }
                        clearOriginalCoordinates();
                        debouncedSaveZonePosition();
                        removeListeners();
                        e.stopPropagation();
                    }

                    // Ondragend on unfold callback
                    function releaseDragOnUnfoldButton(e) {

                        if (e.button === 2) {
                            // do nothing on right-click
                            return;
                        }

                        cleanMoveCursor();
                        if (e.originalEvent && e.originalEvent.detail == 2) {
                            return; // that's a double click
                        }
                        FlowGraphFolding.unfoldNode(this);
                        clearOriginalCoordinates();
                        debouncedSaveZonePosition();
                        removeListeners();
                        e.stopPropagation();
                    }

                    // Ondragend on unfold callback
                    function releaseDragOnZone(e) {

                        if (e.button === 2) {
                            // do nothing on right-click
                            return;
                        }

                        cleanMoveCursor();
                        if (e.originalEvent && e.originalEvent.detail == 2 && !FlowGraph.isJobGraph()) {
                            //Double click detected, do nothing (previously zooming zone sc-173604)
                            return;
                        }
                        const coordinates = mouseViewportCoordinates(e);
                        // travelled distance
                        if (original_click) {
                            if (square_distance(coordinates, original_click) < 16) {
                                // if mouse has moved less than 4px
                                if ((e.which == 1) || (e.which == 2) || (e.type == 'touchend')) {
                                    const nodeId = $(this).attr('id').split(/_(.+)/)[1];
                                    const node = FlowGraph.node(nodeId);
                                    if (node) {
                                        GraphZoomTrackerService.setFocusItemCtx(node);
                                        FlowGraphSelection.onItemClick(node, e);
                                        $scope.$apply();
                                    }
                                }
                            } else if ($scope.rectangleSelection) {
                                commitRectangleSelection(e);
                            }
                        }
                        clearOriginalCoordinates();
                        debouncedSaveZonePosition();
                        removeListeners();
                        e.stopPropagation();
                    }

                    // Ondragend on something else than a node
                    function releaseDrag(e) {

                        if (e.button === 2) {
                            // do nothing on right-click
                            return;
                        }

                        cleanMoveCursor();
                        try {
                            const coordinates = mouseViewportCoordinates(e);
                            if (original_click) {
                                // travelled distance
                                if (square_distance(coordinates, original_click) < 16) {
                                    // if mouse has moved less than 4px
                                    const isTagEditPopoverOpened = $(".tag-edit-popover__popover")[0];
                                    if ((e.which == 1 || e.which == 2 || e.type == 'touchend') && !(e.metaKey || e.shiftKey) && !isTagEditPopoverOpened && e.originalEvent.detail <= 1 && !(e.ctrlKey && DetectUtils.isMac())) {
                                        // e.originalEvent.detail is detecting double click
                                        if (FlowGraph.isJobGraph())  {
                                            JobFlowGraphSelection.clearSelection($scope);
                                        } else {
                                            // Reset focused item so that we store in local storage the information that no node is selected and the project level right panel should be displayed
                                            GraphZoomTrackerService.resetFocusItemCtx();
                                            FlowGraphSelection.clearSelection();
                                        }
                                    }
                                } else if ($scope.rectangleSelection) {
                                    commitRectangleSelection(e);
                                }
                            }
                        } catch (e) {
                            Logger.error(e);
                        }
                        clearOriginalCoordinates();
                        debouncedSaveZonePosition();
                        removeListeners();
                        $scope.$apply();
                    }

                    return function(evt) {
                        const isZoneMove = (evt.data.isOnZoneHeader || (evt.data.isOnZoneBody && evt.altKey)) && $scope.zonesManualPositioning && $scope.canMoveZones;

                        // The zone element (that contains the data-id and transform attributes is the event's parent element)
                        const zoneElement = this.parentElement;

                        if (isZoneMove) {
                            zoneId = zoneElement.getAttribute("data-id");
                        }
                        if (!$scope.svg) {
                            return;
                        }
                        if (evt.which == 1 || evt.type == 'touchstart') {
                            const coordinates = mouseViewportCoordinates(evt);
                            original_coordinates = {
                                x: $scope.panzoom.x + $scope.panzoom.width * (coordinates.x / $scope.panzoom.WIDTH),
                                y: $scope.panzoom.y + $scope.panzoom.height * (coordinates.y / $scope.panzoom.HEIGHT)
                            };
                            original_graph_coordinates = mouseGraphCoordinates(evt);
                            original_click = {
                                x: coordinates.x,
                                y: coordinates.y
                            };
                        }

                        //store the item under the cusror at the beginning of what could be a drag&drop action
                        $applyMoveCursorInitialItem = $(evt.target);

                        if (isZoneMove) {
                            const transformAttribute = zoneElement.getAttribute("transform")
                            $(document).on('mousemove', null, {transformAttribute: transformAttribute}, moveZone);
                        } else {
                            $(document).on('mousemove', drag);
                        }
                        $element.on('mouseup', 'svg .node', releaseDragOnNode);
                        $element.on('mouseup', 'svg .zone_cluster', releaseDragOnZone);
                        $element.on('mouseup', 'svg .folded-icon', releaseDragOnUnfoldButton);
                        $(document).on('mouseup', releaseDrag);

                        if (isTouchDevice()) {
                            $(document).on('touchmove', monoTouchCallBack(drag));
                            $element.on('touchend', 'svg .node', releaseDragOnNode);
                            $element.on('touchend', 'svg .zone_cluster', releaseDragOnZone);
                            $element.on('touchend', 'svg .folded-icon', releaseDragOnUnfoldButton);
                            $(document).on('touchend', releaseDrag);
                        }

                        evt.stopPropagation();
                        $scope.$apply();
                    };
                })();

                $element.on('mousedown', 'g[data-type="ZONE"] > .zone_header', {isOnZoneHeader: true, isOnZoneBody: false}, dragStart);
                $element.on('mousedown', 'g[data-type="ZONE"] > :not(.zone_header)', {isOnZoneHeader: false, isOnZoneBody: true}, dragStart);
                $element.on('mousedown', ':not(g[data-type="ZONE"], g[data-type="ZONE"] *)', {isOnZoneHeader: false, isOnZoneBody: false}, dragStart);

                $element.on('mousedown', 'g[data-type="ZONE"] > .zone_header', {isOnZoneHeader: true, isOnZoneBody: false}, monoTouchCallBack(dragStart));
                $element.on('mousedown', 'g[data-type="ZONE"] > :not(.zone_header)', {isOnZoneHeader: false, isOnZoneBody: true}, monoTouchCallBack(dragStart));
                $element.on('mousedown', ':not(g[data-type="ZONE"], g[data-type="ZONE"] *)', {isOnZoneHeader: false, isOnZoneBody: false}, monoTouchCallBack(dragStart));

                // do not zoom in on zone using double click : sc-173604
                $element.on('dblclick', 'svg [class~=node]:not(.zone)', function (e) {
                    const nodeId = $(this).attr('data-id');
                    const node = FlowGraph.node(nodeId);
                    clearOriginalCoordinates();
                    $scope.onItemDblClick(node, e);
                    e.stopPropagation();
                });

                const KEYBOARD_FACTOR = 100;

                $scope.moveLeft = () => {
                    $scope.panzoom.x = $scope.panzoom.x + $scope.panzoom.width / 2 - $scope.panzoom.width * ((($scope.panzoom.WIDTH / 2) + KEYBOARD_FACTOR) / $scope.panzoom.WIDTH);
                    $scope.redraw();
                }

                $scope.moveRight = () => {
                    $scope.panzoom.x = $scope.panzoom.x + $scope.panzoom.width / 2 - $scope.panzoom.width * ((($scope.panzoom.WIDTH / 2) - KEYBOARD_FACTOR) / $scope.panzoom.WIDTH);
                    $scope.redraw();
                }

                $scope.moveUp = () => {
                    $scope.panzoom.y = $scope.panzoom.y + $scope.panzoom.height / 2 - $scope.panzoom.height * ((($scope.panzoom.HEIGHT / 2) + KEYBOARD_FACTOR) / $scope.panzoom.HEIGHT);
                    $scope.redraw();
                }

                $scope.moveDown = () => {
                    $scope.panzoom.y = $scope.panzoom.y + $scope.panzoom.height / 2 - $scope.panzoom.height * ((($scope.panzoom.HEIGHT / 2) - KEYBOARD_FACTOR) / $scope.panzoom.HEIGHT);
                    $scope.redraw();
                }
            }

            $scope.getNode = id => {
                if (!id) {
                    return undefined;
                }
                const item = $scope.nodesGraph.nodes[id];
                if (!item) {
                    return $scope.nodesGraph.nodes[id.replace("cluster_", "")];
                }
                return item;
            }

            $scope.getSelector = (id, node) => {
                const item = node ? node : $scope.getNode(id);
                let selector = id ? `g[data-id=${id}]` : 'g .highlight';
                if (item && item.nodeType === 'ZONE') {
                    selector = `#cluster_${id}`;
                }
                return selector;
            }

            // offCenterShift positive : put item a bit below on the screen
            $scope.zoomGraph = (id, paddingFactor = 3, item = null, offCenterShift = 0) => {
                FlowGraphFolding.ensureNodesNotFolded([id]);
                if (!item) {
                    item = $scope.getNode(id);
                }
                const selector = $scope.getSelector(id, item);
                if (item) { 
                    if (item.nodeType === 'ZONE') {
                        paddingFactor = 1.5;
                    } else if(item.nodeType === 'RECIPE') {
                        // in legacy flowSearchPopover directive. In flowSearchSelectIndex padding is set to 5 for recipes
                        paddingFactor = 5;
                    }
                }
                $scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector($scope.svg, selector), paddingFactor, offCenterShift);
            };

            // shift the center of the box from the center of the screen by offCenterShift pixel, positive makes it on top
            $scope.zoomToBbox = function(bbox, paddingFactor = 1.5, offCenterShift = 0) {
                if (!$scope.svg || !bbox) return;
                $scope.panzoom.x = bbox.x;
                $scope.panzoom.y = bbox.y;

                if (bbox.width / bbox.height > $scope.panzoom.WIDTH / $scope.panzoom.HEIGHT) {
                    $scope.panzoom.width = bbox.width;
                    $scope.panzoom.height = bbox.width / ($scope.panzoom.WIDTH / $scope.panzoom.HEIGHT);
                    $scope.panzoom.y = bbox.y - ($scope.panzoom.height - bbox.height) / 2;
                } else {
                    $scope.panzoom.width = bbox.height * ($scope.panzoom.WIDTH / $scope.panzoom.HEIGHT);
                    $scope.panzoom.height = bbox.height;
                    $scope.panzoom.x = bbox.x - ($scope.panzoom.width - bbox.width) / 2;
                }

                const menuHeight = $('#flow-editor-page .menu').legacyHeight();
                const x = $scope.panzoom.WIDTH / 2;
                const y = ($scope.panzoom.HEIGHT - menuHeight) / 2 + menuHeight;
                $scope.panzoom.x = $scope.panzoom.x + $scope.panzoom.width * (x / $scope.panzoom.WIDTH) * (1 - paddingFactor);
                $scope.panzoom.y = $scope.panzoom.y + $scope.panzoom.height * (y / $scope.panzoom.HEIGHT) * (1 - paddingFactor);
                $scope.panzoom.width  = $scope.panzoom.width * paddingFactor;
                $scope.panzoom.height = $scope.panzoom.height * paddingFactor;

                d3.select($scope.svg[0]).transition(300).attr(
                    'viewBox', [
                        $scope.panzoom.x,
                        $scope.panzoom.y + offCenterShift,
                        $scope.panzoom.width,
                        $scope.panzoom.height
                    ].join(', ')
                );
            };

            setupZoomBehavior();
            setupMoveAndSelectBehavior();

            Mousetrap.bind("Z S", _ => $scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector($scope.svg, '.selected')));
            $scope.$on("$destroy", _ => Mousetrap.unbind("Z S"));

            Mousetrap.bind("Z F", _ => $scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector($scope.svg, '.focus')));
            $scope.$on("$destroy", _ => Mousetrap.unbind("Z F"));

            Mousetrap.bind("Z U", _ => $scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector($scope.svg, '.usedDatasets .node')));
            $scope.$on("$destroy", _ => Mousetrap.unbind("Z U"));

            Mousetrap.bind("j e l l y", function(){
                let x = [];
                let y = [];
                let t = c => (c||0)*0.95 + 10 * (Math.random() - 0.5);
                let d = 9000;
                window.setInterval(function(){
                    d3.selectAll("g.node:not(.zone),g.edge").transition().duration(d).ease("elastic").attr("transform", function(d, i) {
                        [x[i], y[i]] = [t(x[i]), t(y[i])];
                        return "translate(" + x[i] + " , " + y[i] + ")";
                    });
                }, d*0.09);
            });
            $scope.$on("$destroy", _ => Mousetrap.unbind("j e l l y"));

            Mousetrap.bind("d i s c o", function(){
                let r = {};
                let g = {};
                let b = {};
                let s = c => (c === undefined ? 255*Math.random() : c) + 100 * (Math.random() - 0.5);
                let bn = c => Math.max(0, Math.min(255, c));
                let sb = c => bn(s(c))
                let d = 2000;
                window.setInterval(function(){
                    d3.selectAll("g.node,g.edge").transition().duration(d).ease("elastic").style("fill", function(d,i) {
                        let id = $(this).attr('id')+'';
                        r[id] = sb(r[id])
                        g[id] = sb(g[id])
                        b[id] = sb(b[id])
                        // [r[id], g[id], b[id]] = [sb(r[id]), sb(g[id]), sb(b[id])];
                        return `rgb(${r[id]}, ${g[id]}, ${b[id]})`;
                    });
                }, d*0.09);
            });
            $scope.$on("$destroy", _ => Mousetrap.unbind("d i s c o"));

            Mousetrap.bind("b e r n a r d", function(){
                d3.select($scope.svg[0])
                    .append("defs")
                    .append('pattern')
                    .attr('id', 'bp')
                    .attr('patternUnits', 'userSpaceOnUse')
                    .attr('width', 120)
                    .attr('height', 120)
                    .append("image")
                    .attr("xlink:href", "https://dev.dataiku.com/egg/bp.jpg")
                    .attr('width', 122)
                    .attr('height', 110)
                    .attr('x', -11);

                d3.select($scope.svg[0]).selectAll("g[data-recipe-type='pivot']").selectAll("path").attr("fill", "url(#bp)");
                d3.select($scope.svg[0]).selectAll("g[data-recipe-type='pivot']").style("opacity", "1")
            })
            $scope.$on("$destroy", _ => Mousetrap.unbind("b e r n a r d"));

            let last_svg_str; //cache used only withing state, going out of the flow discards it
            let last_svg_element;
            let last_$svg;
            let last_nodes;
            let last_unread_ids;
            let deregister = $rootScope.$on('drawGraph', function (tgtScope, ignoreCache = false, indexNodes = false) {
                if (!$scope.nodesGraph.nodes) return; // Too early

                let svgElement;
                let cachedSVG = !!$stateParams.projectKey && !ignoreCache && last_svg_str == $scope.nodesGraph.svg && angular.equals($scope.nodesGraph.nodes, last_nodes) && angular.equals(($rootScope.discussionsUnreadStatus || {}).unreadFullIds || [], last_unread_ids); // TODO and = last filter TODO
                if (cachedSVG) {
                    Logger.debug('use cached svg');
                    svgElement = last_svg_element;
                } else {
                    svgElement = $($scope.nodesGraph.svg);

                    // Manipulate the SVG in a hidden DIV as far as possible before switching to it,
                    // rather than switch early and then apply a sequence of changes. When we switch, don't change the SVG element, but its contents

                    $element.addClass('no-animation'); // Initial rendering should not have animation (especially when a flow view is active)

                    last_svg_str = $scope.nodesGraph.svg;
                    last_svg_element = svgElement;
                    last_$svg = $element.find('svg');
                    last_nodes = $scope.nodesGraph.nodes;
                    last_unread_ids = angular.copy(($rootScope.discussionsUnreadStatus || {}).unreadFullIds || []);
                }

                if (!!$stateParams.zoneId) {
                    svgElement = $(svgElement).find(`.zone#zone_${$stateParams.zoneId}>svg`).removeAttr('x').removeAttr('y')
                }
                if (cachedSVG) {
                    Logger.debug('Graph is ready, reset style');
                    //TODO @flow move to do flow styling + views
                    $('g', $scope.svg).removeAttr('style');
                    $('.newG', $scope.svg).removeAttr('color');
                    $('.tool-simple-zone', $scope.svg).empty();
                    $('.node-label', $scope.svg).remove();
                    $('.node-totem span', $scope.svg).removeAttr('style').removeClass();
                    $('.nodecounter__text div span', $scope.svg).text('');
                    $('.nodecounter__wrapper', $scope.svg).removeClass('nodecounter__wrapper--shown');
                    $('.never-built-computable *', $scope.svg).removeAttr('style');
                    $scope.svg = last_$svg;
                } else {
                    Logger.debug('Graph is not ready, add svg element', svgElement);

                    var isReload = false;

                    if ($element.children().length > 0) {
                        isReload = true;
                        $element.append("<div style='visibility:hidden; width:100%; height:100%' id='hc-svnt-dracones'></div>")
                        var $preloadedGraph = $element.find('#hc-svnt-dracones');
                        $preloadedGraph.append(svgElement);
                        $scope.svg = last_$svg = $preloadedGraph.find('>svg');
                    } else {
                        $element.children().remove();
                        $element.find('svg').remove();
                        $element.append(svgElement);
                        $scope.svg = last_$svg = $element.find('>svg');
                    }

                    const bbFlow = $scope.svg.find('g.graph')[0].getBBox(); //get the whole-flow bounding box
                    const defaultPz = buildDefaultPanZoomSettings($scope.panzoom, bbFlow);
                    if (!GraphZoomTrackerService.isValidPanZoom($scope.panzoom)) {
                        $scope.panzoom = defaultPz;
                    }

                    $scope.svg.attr('height', '100%').attr('width', '100%');

                    // remove background polygon
                    // Firefox use to have an issue but seems resolved with recent version (Check history of the file)
                    $scope.svg[0].setAttribute('viewBox', '-10000 -10000 10 10'); //mjt move offscreen to avoid flicker
                    $scope.svg.find('g').first().attr('transform', '').find('polygon').first().remove();
                    d3.select($scope.svg[0]).selectAll("g.cluster:not(.zone_cluster)").remove()


                    if (!!$stateParams.projectKey) {
                        ProjectFlowGraphStyling.restyleGraph($scope.svg, $scope);
                        ProjectFlowGraphLayout.relayout($scope.svg);
                    } else {
                        InterProjectGraphStyling.restyleGraph($scope.svg, $scope);
                        InterProjectGraphLayout.relayout($scope.svg);
                    }
                    $scope.$emit('graphRendered');
                }
                FlowGraphSelection.refreshStyle();

                $('#flow-graph').attr('style', '');
                $rootScope.$broadcast('reflow');

                //mjt experiment in anti-flicker
                if (isReload) {
                    const $newSvg = $preloadedGraph.children().first();//find('svg');
                    const $origSvg = $element.children().first();
                    $origSvg.children().remove();
                    $origSvg.append($newSvg.children());
                    $scope.svg = last_$svg = $element.find('>svg');
                    $preloadedGraph.remove();
                }

                if (indexNodes === true) {
                    let graphDOM = $scope.svg.find('g.graph')[0];
                    if (graphDOM) {
                        FlowGraph.indexNodesCoordinates($scope.svg, graphDOM.getBBox());
                    }
                }

            }, true);
            $scope.$on('$destroy', deregister);

            $element[0].oncontextmenu = function(evt) {
                const itemElt =  $(evt.target).parents('g[data-type]').first();
                let nodeId = $(itemElt).attr('data-id');
                if (itemElt.attr("data-type") === "ZONE") {
                    nodeId = nodeId.replace("cluster_", "");
                }
                const node = FlowGraph.node(nodeId);
                return $scope.onContextualMenu(node, evt);
            };

            //TODO @flow move?
            $scope.$watchCollection('tool.user.state.focusMap', function(nv,ov) {
                if (!nv ) return;
                $scope.tool.drawHooks.updateFlowToolDisplay();
                $scope.tool.saveFocus();
            })

            $scope.$on('restyleGraph', function() {
                ProjectFlowGraphStyling.restyleGraph($scope.svg, $scope);
            });

            $scope.$on('graphRendered', function() {
                if (FlowGraph.get().nodes.length < 200) { // Too slow for bigger graphs
                    Logger.debug('Reactivate animations')
                    $('#flow-graph').toggleClass('no-animation', false);
                }
            });
            }
        };
    });


    app.directive('flowGraphWithTooltips', function($rootScope, FlowGraph, DatasetUtils, ChartTooltipsUtils, WatchInterestState) {

    return {
        restrict: 'EA',
        link: function (scope, element, attrs) {
            let tooltip, tooltipScope;
            let timeout;
            const DEFAULT_DELAY = 500;
            const SHORT_DELAY = 250; // Just enough so that if you just want to click on an item, you don't see the tooltip

            function show(node, tooltipScope, evt, delay) {
                timeout = setTimeout(function() {
                    ChartTooltipsUtils.handleMouseOverElement(tooltipScope);
                    tooltipScope.node = node;
                    tooltipScope.$apply();
                    ChartTooltipsUtils.appear(tooltip, '#777', evt, element);
                }, delay);
            }

            function hide(digestInProgress) {
                if (tooltipScope == null) return; // might not be ready yet
                clearTimeout(timeout);
                ChartTooltipsUtils.handleMouseOutElement(tooltip, tooltipScope, digestInProgress);
            }
            function addTooltipBehavior(elt, nodeId) {
                const node = FlowGraph.node(nodeId);

                if (node) {
                    elt.on("mouseover", function(d, i) {
                        if (node.filterRemove || tooltipScope == null) return; // might not be ready yet

                        if (scope.tool && scope.tool.drawHooks && scope.tool.drawHooks.setupTooltip) {
                            tooltipScope.tooltip = scope.tool.drawHooks.setupTooltip(node);
                            show(node, tooltipScope, d3.event, SHORT_DELAY);
                        } else if ((node.nodeType.endsWith('DATASET') || node.nodeType.endsWith('ZONE')) &&
                                !node.shortDesc && !(node.tags && node.tags.length) && !node.veLoopDatasetRef) {
                            return; // Let's not display the tooltip just for the name of the dataset...
                        } else {
                            tooltipScope.tooltip = {};
                            show(node, tooltipScope, d3.event, DEFAULT_DELAY);
                        }
                    })
                        .on("mouseout", hide);
                }
            }

            ChartTooltipsUtils.createWithStdAggr1DBehaviour(scope, attrs.tooltipType || 'flow-tooltip', element)
                .then(function(x){
                    tooltip = x[0];
                    tooltipScope = x[1];
                })
                .then(() => {
                    tooltipScope.isWatching = WatchInterestState.isWatching;
                    tooltipScope.DatasetUtils = DatasetUtils;
                });

            scope.setupTooltips = function() {
                
                $('[data-id][data-type]:not([data-type="ZONE"])').each(function(_, g) {
                    const nodeId = d3.select(g).attr('data-id');
                    addTooltipBehavior(d3.select(g), nodeId);
                });
                
                $('.zone_cluster').each(function(_, g) {
                    const headers = $(g).find('.zone_header');
                    const nodeId = d3.select(g).attr('data-id');
                    if (headers.length === 1) {
                        // Add the zone tooltip only on the zone_header
                        addTooltipBehavior(d3.select(headers[0]), nodeId);
                    } else {
                        // All zones should have a zone header, but in case, default to the whole zone_cluster
                        addTooltipBehavior(d3.select(g), nodeId);
                    }
                });
            }

            const h = $rootScope.$on('flowSelectionUpdated', _ => hide(true));
            scope.$on('$destroy', h);

            scope.$on("graphRendered", scope.setupTooltips);
        }
    }
    });

    /**
     * Add local files drag and drop capability to the flow. This allows users to drop Files on the flow to upload them.
     * If files are dropped onto a zone, the resulting datasets are created in this zone.
     */
    app.directive('flowGraphWithDropFile', function($state, $stateParams, FlowGraph, FlowGraphSelection, FlowGraphHighlighting, $timeout) {
        return {
            restrict: 'EA',
            scope: {
                isDroppable: '=?', // parameter used by the directive to expose to its parent a candrop flag
                allowDropFile: '=?' // parameter used by the directive to enable the drag and drop feature only if the user has write access to his project
            },
            link: function (scope, element) {
                const NODE_ID_UNINITIALIZED = "__UNINITIALIZED__";
                const ZONE_ID_UNINITIALIZED = "__UNINITIALIZED__";
                const ZONE_ID_DEFAULT = "default"; // ID of the default zone
                let lastNodeOverId = NODE_ID_UNINITIALIZED; // Starts with uninitialized value so that we compute the zone at least once.
                let lastZoneOverId = ZONE_ID_UNINITIALIZED;

                function hasZones() {
                    let graph = FlowGraph.get();
                    return graph && graph.hasZones
                }
                function getZoneId(nodeId, itemElt) {
                    if (nodeId !== undefined) {
                        if (itemElt.attr("data-type") === "ZONE") {
                            return nodeId.replace("cluster_", "").substring("zone_".length);
                        } else if (nodeId.startsWith("zone__")) {
                            return nodeId.substring("zone__".length, nodeId.indexOf("__", "zone__".length));
                        }
                    }
                    return ZONE_ID_DEFAULT;
                }
                function applyDragEnterLeave(e) {
                    e.stopPropagation();
                    e.preventDefault();
                    scope.candrop = false;
                    lastNodeOverId = NODE_ID_UNINITIALIZED;
                    lastZoneOverId = ZONE_ID_UNINITIALIZED;
                    scope.$apply(function(){
                        FlowGraphHighlighting.removeDropFeedbackFromZoneElements();
                    });
                }
                function cancelEnterLeaveTimeout() {
                    if (scope.enterLeaveTimeout) {
                        $timeout.cancel(scope.enterLeaveTimeout);
                    }
                }
                function dragEnterLeave(e) {
                    cancelEnterLeaveTimeout();
                    scope.enterLeaveTimeout = $timeout(function() {
                        applyDragEnterLeave(e);
                    }, 100);
                }

                // When the directive is instanciated, the attribute 'allowDropFile' can still be undefined.
                // We need to wait for it to be defined before enabling the drag and drop feature.
                const unwatch = scope.$watch('allowDropFile', () => {
                    if (scope.allowDropFile === undefined) {
                        return;
                    }
                    unwatch(); // If 'allowDropFile' is defined (either 'true' or 'false'), stop watching
                    if (scope.allowDropFile === false) {
                        return;
                    }

                    scope.candrop = false;
                    scope.$watch('candrop', () => {
                        scope.isDroppable = scope.candrop;
                    });

                    element.on("dragenter", dragEnterLeave);
                    element.on("dragleave", dragEnterLeave);
                    element.on("dragover", e => {
                        cancelEnterLeaveTimeout();
                        e.stopPropagation();
                        e.preventDefault();
                        scope.$apply(() => {
                            var evt = e.originalEvent;
                            if (evt.dataTransfer && contains_item(evt.dataTransfer.types, 'Files')) {
                                scope.candrop = true;
                                // In case, we have zones and we are NOT zoomed on a zone, let's determine the current zone that is over the cursor
                                if (!$stateParams.zoneId && hasZones()) {
                                    const itemElt = $(e.target).parents('g[data-type]').first();
                                    const nodeId = $(itemElt).attr('data-id');
                                    if (nodeId !== lastNodeOverId) {
                                        lastNodeOverId = nodeId;
                                        const zoneId = getZoneId(nodeId, itemElt);
                                        if (zoneId !== lastZoneOverId) {
                                            lastZoneOverId = zoneId;
                                            if (zoneId !== ZONE_ID_UNINITIALIZED) {
                                                FlowGraphHighlighting.addDropFeedbackToZoneElement("zone_" + zoneId);
                                                let zone = FlowGraph.node("zone_" + zoneId);
                                                if (zone) {
                                                    FlowGraphSelection.onItemClick(zone);
                                                }
                                            }
                                        }
                                    }
                                }
                            } else {
                                scope.candrop = false;
                            }
                        });
                    });
                    element.on("drop", e => {
                        e.stopPropagation();
                        e.preventDefault();
                        lastNodeOverId = NODE_ID_UNINITIALIZED;
                        lastZoneOverId = ZONE_ID_UNINITIALIZED;
                        let zoneId;
                        if (!$stateParams.zoneId) {
                            const itemElt = $(e.target).parents('g[data-type]').first();
                            const nodeId = $(itemElt).attr('data-id');
                            zoneId = getZoneId(nodeId, itemElt);
                        } else {
                            zoneId = $stateParams.zoneId;
                        }

                        scope.$apply(() => {
                            let evt = e.originalEvent;
                            if (evt.dataTransfer && contains_item(evt.dataTransfer.types, 'Files')) {
                                scope.$root.uploadedFiles = evt.dataTransfer.files;
                                let params = { type: "UploadedFiles" };
                                if (zoneId !== ZONE_ID_UNINITIALIZED && zoneId !== ZONE_ID_DEFAULT) {
                                    params["zoneId"] = zoneId;
                                }
                                $state.go('projects.project.datasets.new_with_type.settings', params);
                            }
                            scope.candrop = false;
                        });
                    });
                });
            }
        }
    });

    /**
     * Add local files drag and drop capability to a flow placeholder. This allows users to drop Files on an empty flow to upload them.
     */
    app.directive('flowGraphPlaceholderWithDropFile', function($state, $stateParams, FlowGraph, FlowGraphSelection, FlowGraphHighlighting, $timeout) {
        return {
            restrict: 'EA',
            scope: {
                isDroppable: '=?', // parameter used by the directive to expose to its parent a candrop flag
                allowDropFile: '=?' // parameter used by the directive to enable the drag and drop feature only if the user has write access to his project
            },
            link: function (scope, element) {
                function applyDragEnterLeave(e) {
                    e.stopPropagation();
                    e.preventDefault();
                    scope.candrop = false;
                }
                function cancelEnterLeaveTimeout() {
                    if (scope.enterLeaveTimeout) {
                        $timeout.cancel(scope.enterLeaveTimeout);
                    }
                }
                function dragEnterLeave(e) {
                    cancelEnterLeaveTimeout();
                    scope.enterLeaveTimeout = $timeout(function() {
                        applyDragEnterLeave(e);
                    }, 100);
                }
                // When the directive is instanciated, the attribute 'allowDropFile' can still be undefined.
                // We need to wait for it to be defined before enabling the drag and drop feature.
                const unwatch = scope.$watch('allowDropFile', () => {
                    if (scope.allowDropFile === undefined) {
                        return;
                    }
                    unwatch(); // If 'allowDropFile' is defined (either 'true' or 'false'), stop watching
                    if (scope.allowDropFile === false) {
                        return;
                    }

                    scope.candrop = false;
                    scope.$watch('candrop', () => {
                        scope.isDroppable = scope.candrop;
                    })

                    element.on("dragenter", dragEnterLeave);
                    element.on("dragleave", dragEnterLeave);
                    element.on("dragover", e => {
                        cancelEnterLeaveTimeout();
                        e.stopPropagation();
                        e.preventDefault();
                        scope.$apply(() => {
                            const evt = e.originalEvent;
                            scope.candrop = evt.dataTransfer && contains_item(evt.dataTransfer.types, 'Files');
                        });
                    });
                    element.on("drop", function(e) {
                        e.stopPropagation();
                        e.preventDefault();
                        scope.$apply(() => {
                            let evt = e.originalEvent;
                            if (evt.dataTransfer && contains_item(evt.dataTransfer.types, 'Files')) {
                                scope.$root.uploadedFiles = evt.dataTransfer.files;
                                let params = { type: "UploadedFiles" };
                                if ($stateParams.zoneId) {
                                    params["zoneId"] = $stateParams.zoneId;
                                }
                                $state.go('projects.project.datasets.new_with_type.settings', params);
                            }
                            scope.candrop = false;
                        });
                    });
                });
            }
        }
    });

    // Used to have global variables...
    app.service('FlowGraph', function($rootScope, $state, FlowGraphFiltering) {
        const svc = this;

        let graph;
        let nodesElements;
        let edgesElementsTo;
        let edgesElementsFrom;
        let nodesCoordinates; // simple list: [{nodeId, middlePoint}]
        let zonesElements;
        let graphBBox;

        this.set = function(g) {
            graph = g;
        };

        this.get = function() {
            return graph;
        };

        this.node = function(nodeId) {
            return graph.nodes[nodeId];
        };

        this.nodes = nodesId => nodesId.map(nodeId => this.node(nodeId));

        /**
         * Get all nodes of all zones in the current graph
         * @param {(node: Node) => any} [postProcessNode] - optional post-processing function to be applied to each node
         * @returns {Object.<string, Node[]>} a dictionary of zonesId as key and a list of nodes as value
         */
        this.nodesByZones = (postProcessNode) => {
            const itemsByZones = {};
            for (const nodeId in graph.nodes) {
                const node = graph.nodes[nodeId];
                if (nodeId.startsWith("zone__") && node.nodeType !== 'ZONE' && (!node.isSource || node.isSink || node.nodeType === 'RECIPE')) {
                    // example data
                    // `nodeId`: zone__default__recipe__compute__customers__copy__27
                    // `realId`: recipe__compute__customers__copy__27
                    // `node.id`:  zone__default__recipe__compute__customers__copy__27
                    const zoneId = nodeId.substring("zone__".length, node.id.length - node.realId.length - 2);
                    if (node.ownerZone === zoneId) {
                        if (!itemsByZones[zoneId]) {
                            itemsByZones[zoneId] = [];
                        }
                        const zoneContent = itemsByZones[zoneId];
                        if (postProcessNode) {
                            zoneContent.push(postProcessNode(node));
                        } else {
                            zoneContent.push(node);
                        }
                    }
                }

            }            
            return itemsByZones;
        }

        this.zonesIdFromRealId = nodeRealId => {
            if (!graph.hasZones) return [];
            const zonesId = graph.realNodesZones[nodeRealId];
            return zonesId ? zonesId : []; // handles managed folders
        };

        this.nodesIdFromRealId = nodeRealId => {
            if (!graph.hasZones) return [nodeRealId];
            const zonesId = this.zonesIdFromRealId(nodeRealId);
            return zonesId.map(zoneId => `zone__${zoneId}__${nodeRealId}`);
        };

        this.nodeCount = function() {
            return graph.nodesOnGraphCount;
        };

        this.getDOMElement = function() {
            return $('#flow-graph');
        };

        this.getSvg = function() {
            return $('#flow-graph > svg');
        };

        this.rawNodeWithId = function(nodeId) {
            if (!nodesElements) return; // not ready
            return nodesElements[nodeId];
        };
        this.d3NodeWithId = function(nodeId) {
            if (!nodesElements) return; // not ready
            return d3.select(svc.rawNodeWithId(nodeId));
        };
        this.d3ZoneNodeWithId = function(nodeId) {
            if (!zonesElements) return; // not ready
            return d3.select(zonesElements[nodeId]);
        };

        this.d3NodeWithIdFromType = function(nodeId, nodeType) {
            if(nodeType.toLowerCase() === "zone") {
                return this.d3ZoneNodeWithId(nodeId);
            } else {
                return this.d3NodeWithId(nodeId);
            }
        };

        this.rawZoneNodeWithId = function(nodeId) {
            if (!zonesElements) return; // not ready
            return zonesElements[nodeId];
        };

        this.rawEdgesWithFromId = function(nodeId) {
            if (!edgesElementsFrom) return [];
            return edgesElementsFrom[nodeId] || [];
        };

        this.rawEdgesWithToId = function(nodeId) {
            if (!edgesElementsTo) return [];
            return edgesElementsTo[nodeId] || [];
        };

        // No fancy search (for now?)
        this.getEnclosedNodesIds = function(rect) {
            if (!nodesCoordinates) return []; // not ready
            const bounds = {x1: rect.x, y1: rect.y, x2: rect.x + rect.width, y2: rect.y + rect.height};
            return nodesCoordinates.filter(c => c.middlePoint.x >= bounds.x1 && c.middlePoint.x <= bounds.x2 && c.middlePoint.y >= bounds.y1 && c.middlePoint.y <= bounds.y2).map(c => c.nodeId);
        };

        // Return the graphBBox, filled when indexNodesCoordinates is called
        this.getGraphBBox = function() {
            return graphBBox;
        }

        this.setGraphBBox = function(newBBox) {
            graphBBox = newBBox;
        }

        this.indexNodesCoordinates = function indexNodesCoordinates (globalSvg, svgBBox) {
            nodesElements = {};
            edgesElementsTo = {};
            edgesElementsFrom = {};
            zonesElements = {};
            nodesCoordinates = [];

            function getAbsoluteBBox(element, svg) {
                if (svg) {
                    const bbox = element.getBBox();
                    let topLeft = svg.createSVGPoint();
                    topLeft.x = bbox.x;
                    topLeft.y = bbox.y;
                    topLeft = topLeft.matrixTransform(element.getTransformToElement(svg));
                    bbox.x = topLeft.x;
                    bbox.y = topLeft.y;
                    return bbox;
                }
                return null;
            }

            function pushEltToEdgeMap(nodeId, elt, edgeMap) {
                if (nodeId) {
                    if (!edgeMap[nodeId]) edgeMap[nodeId] = [];
                    edgeMap[nodeId].push(elt);
                }
            }

            this.setGraphBBox(svgBBox);
            const svg = $('#flow-graph > svg')[0];
            $('.usedDatasets .node, .connectedProjects .node', globalSvg).each(function (_, elt) {
                const bbox = getAbsoluteBBox(elt, svg);
                if (bbox) {
                    const middlePoint = { x: bbox.x + bbox.width / 2, y: bbox.y + bbox.height / 2 };
                    const nodeId = $(elt).attr('data-id');
                    nodesElements[nodeId] = elt;
                    nodesCoordinates.push({ nodeId, middlePoint });
                }
            });

            // build a map of edges to accelerate path highlighting hugely on large flows
            $('.edge', globalSvg).each(function (_, elt) {
                pushEltToEdgeMap($(elt).attr('data-from'), elt, edgesElementsFrom);
                pushEltToEdgeMap($(elt).attr('data-to'), elt, edgesElementsTo);
            });
            $('.draftDatasets > .node', globalSvg).each(function (_, elt) {
                let bbox;
                if ($(elt).closest('svg').is(globalSvg)) {
                    bbox = elt.getBBox();
                    bbox.x += svgBBox.x;
                    bbox.y += svgBBox.y;
                } else {
                    bbox = getAbsoluteBBox(elt, svg);
                }
                if (bbox) {
                    const middlePoint = { x: bbox.x + bbox.width / 2, y: bbox.y + bbox.height / 2 };
                    const nodeId = $(elt).attr('data-id');
                    nodesElements[nodeId] = elt;
                    nodesCoordinates.push({ nodeId, middlePoint });
                }
            });

            $('.zone_cluster', globalSvg).each(function (_, elt) {
                const nodeId = $(elt).attr('id').replace('cluster_', '');
                zonesElements[nodeId] = elt;
            });

            $rootScope.$emit('flowDisplayUpdated');
            $rootScope.$broadcast('indexNodesDone');
        };

        this.ready = function() {
            return !!graph && !!nodesCoordinates;
        };

        // Some services make API calls that require to be displayed in API error directive so bound to a scope
        // But the services have no scope and binding errors to rootScope is inconvenient
        // (because the error won't necessarily go away when moving to another state)
        this.setError = function() {
            const flowGraphScope = angular.element('#flow-graph').scope();
            return setErrorInScope.bind(flowGraphScope);
        }

        this.nodeSharedBetweenZones = node => {
            // Handle the case where there is no zone (i.e. project list - graph view)
            if (!graph.zonesUsedByRealId) {
                return null;
            }
            const found = graph.zonesUsedByRealId[node.realId];
            if (found) {
                const set = new Set(found);
                set.delete(node.ownerZone);
                return set;
            }
            return null;
        };

        this.zoomOnSelectedActivities = function(singleSelection, factor = 1) {
            const scope = $('#flow-graph').scope();
            if (scope === undefined) {
                return;
            }
            const paddingFactor = (singleSelection ? 3.0 : 2.0) * factor ;
            scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector(scope.svg, '.selected:not(.zone_cluster)'), paddingFactor);
        }

        this.recipeNodeFromName = recipeName => Object.values(graph.nodes).find(node => node.nodeType === 'RECIPE' && node.name === recipeName);

        this.isJobGraph = () => Boolean($state.$current.includes['projects.project.jobs']);
    });

    function contains_item(collection, item) {
        return collection && collection.includes(item);
    }

    function square_distance(A, B) {
        const dx = A.x - B.x;
        const dy = A.y - B.y;
        return (dx*dx + dy*dy);
    }

    /**
     * Create a panZoom structure that will display the whole flow nicely.
     * This is the classic fall-back display used when there is no saved zoom settings
     * or the flow is too small to justify re-using the saved zoom settings.
     *
     * @param currentPz - the current panZoom settings.
     * @param bbFlow - the SVG bounding box structure for the whole flow
     * @returns a new panZoom structure
     */
    function buildDefaultPanZoomSettings(currentPz, bbFlow) {
        //copy existing settings, in particular the HEIGHT/WIDTH (for the HTML container,
        // and typically the maxHeight / maxWidth, which control how far you can zoom out.

        const pz = angular.copy(currentPz);

        pz.width = bbFlow.width;
        pz.height = bbFlow.height;
        pz.x = bbFlow.x;
        pz.y = bbFlow.y;
        if (bbFlow.width / bbFlow.height > pz.WIDTH / pz.HEIGHT) {
            pz.width = bbFlow.width;
            pz.height = bbFlow.width * (pz.HEIGHT / pz.WIDTH);
            pz.y = bbFlow.y - (pz.height - bbFlow.height) / 2;
        } else {
            pz.width = bbFlow.height * (pz.WIDTH / pz.HEIGHT);
            pz.height = bbFlow.height;
            pz.x = bbFlow.x - (pz.width - bbFlow.width) / 2;
        }

        pz.maxHeight = pz.height;
        pz.maxWidth = pz.width;

        return pz;
    }

})();

;
(function(){
'use strict';

const app = angular.module('dataiku.flow.graph');


/* We get a first SVG flow generated by graphviz in the backend,
 * frontend post-processing is done here
 */


app.service('ProjectFlowGraphStyling', function($filter, FlowGraph, CachedAPICalls, LoggerProvider, TimingService, DataikuAPI, $rootScope, $q, objectTypeFromNodeFlowType, $state, $compile, FlowZoneMoveService) {

const SIZE = 100;

const logger = LoggerProvider.getLogger('flow');

let flowIconset;
 CachedAPICalls.flowIcons.success(function(data) {
    flowIconset = data;
});

const formatters = {
    'RECIPE': restyleRunnableNode("recipe"),
    'LABELING_TASK': restyleRunnableNode("labeling_task"),
    'LOCAL_DATASET': restyleDatasetNode(true),
    'FOREIGN_DATASET': restyleDatasetNode(false),
    'LOCAL_SAVEDMODEL': restyleModelNode(true),
    'FOREIGN_SAVEDMODEL': restyleModelNode(false),
    'LOCAL_MODELEVALUATIONSTORE': restyleEvaluationStoreNode(true),
    'FOREIGN_MODELEVALUATIONSTORE': restyleEvaluationStoreNode(false),
    'LOCAL_GENAIEVALUATIONSTORE': restyleEvaluationStoreNode(true),
    'FOREIGN_GENAIEVALUATIONSTORE': restyleEvaluationStoreNode(false),
    'LOCAL_RETRIEVABLE_KNOWLEDGE': restyleRetrievableKnowledgeNode(true),
    'FOREIGN_RETRIEVABLE_KNOWLEDGE': restyleRetrievableKnowledgeNode(false),
    'LOCAL_MANAGED_FOLDER': restyleFolderNode(true),
    'FOREIGN_MANAGED_FOLDER': restyleFolderNode(false),
    'LOCAL_STREAMING_ENDPOINT': restyleStreamingEndpointNode(true),
    'FOREIGN_STREAMING_ENDPOINT': restyleStreamingEndpointNode(false)
};

this.restyleGraph = TimingService.wrapInTimePrinter("ProjectFlowGraphStyling::restyleGraph", function(svg, $scope) {
    const zones = svg.find('.zone_cluster');
    if (zones) {
        for (let i = 0; i < zones.length; i++) {
            const zone = zones[i];
            try {
                restyleZone(svg, zone, $scope);
            } catch (e) {
                logger.error("Failed to restyle flow zone: ", e);
            }
        }
    }
    svg.find(".zone_label_remove").remove();
    svg.find('text').remove();
    svg.find('title').remove();

    const nodes = $('.node:not(.zone)', svg);
    if (nodes) {
        for (let i = 0; i<nodes.length; i++) {
            const g = nodes[i];
            try {
                restyleNode(g);
            } catch (e) {
                logger.error("Failed to restyle flow node: ", e);
            }
        }
    }
    svg.find('.node.zone>svg g polygon').remove();
});

function drawRepeatedObjectSticker(svg, coords) {
    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-information-sticker__background',
        cx: coords.cx,
        cy: coords.cy,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX,
        y: coords.indicatorY,
        width: 20,
        height: 20,
        class: 'flow-repeated-object__indicator',
    }, $(`<span size="32" class="dku-icon-arrow-repeat-16">`)));
}

function drawInformationSticker(svg, coords, offset = { x: 0, y: 0 }) {
    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-information-sticker__background',
        cx: coords.cx - offset.x,
        cy: coords.cy - offset.y,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX - offset.x,
        y: coords.indicatorY - offset.y,
        width: 20,
        height: 20,
        class: 'flow-information-sticker__indicator',
    }, $(`<span size="32" class="icon-info-sign">`)));
}

function drawFeatureGroupSticker(svg, coords) {
    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-information-sticker__background',
        cx: coords.cx,
        cy: coords.cy,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX,
        y: coords.indicatorY,
        width: 20,
        height: 20,
        class: 'flow-feature-group__indicator',
    }, $(`<span size="32" class="icon-dku-label-feature-store">`)));
}

function drawDiscussionSticker(svg, discussionCount, has_unread_discussions, coords,
        offset = { x: 0, y: 0 }, indicatorOffset = { x: 0, y: 0 }) {

    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-discussions-sticker__background',
        cx: coords.cx - offset.x,
        cy: coords.cy - offset.y,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX - offset.x - indicatorOffset.x,
        y: coords.indicatorY - offset.y - indicatorOffset.y,
        width: 20,
        height: 20,
        class: 'flow-discussions-sticker__indicator' + ((has_unread_discussions) ? ' flow-discussions-sticker__indicator--unread' : ''),
    }, $(`<span size="32" class="icon-dku-discussions">`)));

    svg.appendChild(makeForeignObject({
        x: coords.contentX - offset.x,
        y: coords.contentY - offset.y,
        width: 40,
        height: 20,
        class: 'flow-discussions-sticker__content',
    }, $(`<span>` + (discussionCount > 9 ? '9+' : discussionCount) + `</span>`)));
}

function hasUnreadDiscussion(flowNode) {
    return (($rootScope.discussionsUnreadStatus || {}).unreadFullIds || []).find(discuId => (flowNode.discussionsFullIds || []).includes(discuId));
}

function restyleNode(g) {
    const element = $(g);
    const nodeType = element.attr('data-type');
    if (formatters[nodeType]) {
        formatters[nodeType](element, g);
    }
}

function restyleZone(svg, zone_cluster, $scope) {
    const jZone_cluster = $(zone_cluster);
    const cluster_polygon = jZone_cluster.find('>polygon');
    const text = jZone_cluster.find(">text");
    const id = zone_cluster.id.split("_").splice(2).join('');
    const sanitizedId = sanitize(id);
    const zone = svg.find(`.zone[id='zone_${id}']`);
    const zone_polygon = zone.find('>polygon');
    // Calculate difference between zone top and cluster top (Gives us the height for the text)

    if (zone_polygon && cluster_polygon) {
        const zone_coords = polygonToRectData(zone_polygon);
        const cluster_coords = polygonToRectData(cluster_polygon);
        const zoneNode = FlowGraph.node(`zone_${id}`);
        const collapseIcon = zoneNode.customData.isCollapsed ? "icon-resize-full" : "icon-resize-small";
        let height = (zone_coords.y - cluster_coords.y) * 2;
        if (!height > 0) {
            logger.warn("Calculated height is not a valid number, default to 44");
            height = 44;
        }
        const foreignObject = makeForeignObject({
            x: cluster_coords.x,
            y: cluster_coords.y,
            width: zone_coords.width,
            height,
            class: 'zone_header'
        }, $(`<div><p>${sanitize(text.text())}</p><span ng-if="zonesManualPositioning && canMoveZones" class="move-zone-shortcut">{{altKey}} + Click to drag</span><button ng-click="buildZone('${sanitizedId}')" class="btn btn--secondary" style="position: static" ><i class="icon-play" /> <span translate="PROJECT.FLOW.GRAPH.ZONE.HEADER.BUILD">Build</span></button><i ng-click="toggleZoneCollapse([{id:'${sanitizedId}'}])" class="${collapseIcon} cursor-pointer" id="collapse-button-zone-${sanitizedId}"/><i ng-click="zoomOnZone('${sanitizedId}')" class="icon-DKU_expand cursor-pointer"/></div>`));
        const color = d3.rgb(zoneNode.customData.color);
        const zoneTitleColor = (color.r*0.299 + color.g*0.587 + color.b*0.114) >= 128 ? "#000" : "#FFF";
        foreignObject.style = `background-color: ${color}; color: ${zoneTitleColor}; border-bottom: none;`;
        zone_cluster.appendChild(makeSVG('g', {
            class: 'tool-simple-zone',
            transform: `translate(${cluster_coords.x + cluster_coords.width}, ${cluster_coords.y + cluster_coords.height})`,
            'data-height': 0
        }));
        zone_cluster.appendChild(foreignObject);
        $compile(foreignObject)($scope);
        zone_polygon.remove();
    }

    $(zone_cluster)[0].setAttribute("data-zone-title", sanitize(text[0].textContent));
    $(zone_cluster)[0].setAttribute("data-id", "zone_" + sanitizedId);

    if ($scope.zonesManualPositioning) {
        FlowZoneMoveService.setZoneEdges(svg, `zone_${sanitizedId}`, $scope.nodesGraph);
    }
}

function restyleDatasetNode(local) {
    // Note that differentiation local/foreign is made with CSS
    return function (element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const dataset = FlowGraph.node(nodeId);

        const dotPolygon = element.find("polygon");
        if (dotPolygon.length) {
            const coords = polygonToRectData(dotPolygon);

            let clazz = 'newG';
            if (dataset.neverBuilt) {
                clazz += ' never-built-computable';
            }
            const newG = makeSVG('g', {class: clazz, transform: `translate(${coords.x} ${coords.y})`});
            d3.select(newG).classed("bzicon", true);

            const othersZones = FlowGraph.nodeSharedBetweenZones(dataset);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);
            const margin = isExported && othersZones.size > 0 ? 4 : 0;
            const counterBoxMargin = (isExported || isImported) && othersZones.size > 0 ? 1 : 0;

            if (dataset.partitioned) {
                newG.appendChild(makeSVG('rect', {
                    x: -10,
                    y: -10,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill dataset-rectangle partitioning-indicator' + (isImported ? ' dataset-zone-imported' : '')
                }));

                newG.appendChild(makeSVG('rect', {
                    x: -5,
                    y: -5,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill dataset-rectangle partitioning-indicator' + (isImported ? ' dataset-zone-imported' : '')
                }));
            }


            if (isExported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill dataset-zone-exported'
                }));
            }
            newG.appendChild(makeSVG('rect', {
                x: margin,
                y: margin,
                width: coords.width - (margin*2),
                height: coords.height - (margin*2),
                class: 'fill dataset-rectangle main-dataset-rectangle' + (isImported ? ' dataset-zone-imported' : '')
            }));

            // Grey box containing count records
            newG.appendChild(makeSVG('rect', {
                x: counterBoxMargin,
                y: 61,
                width: coords.width - (counterBoxMargin * 2),
                height: coords.height - counterBoxMargin - 61,
                class: 'nodecounter__wrapper'
            }));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width}, 0)`,
                'data-height': '72'
            }));

            // Display of count records value
            newG.appendChild(makeForeignObject({
                x: -30,
                y: 56,
                width: coords.width + 61,
                height: 48,
                class: 'nodecounter__text'
            }, $('<div><span></span></div>')));

            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.07,
                width: coords.width + 60,
                height: 45,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(dataset.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            const icon = $filter("toModernIcon")($filter("datasetTypeToIcon")(dataset.datasetType, 48), 48);
            const iconElement = $(`<div class="flow-tile faic jcc h100 w100"><i class="${icon}" /></div>`);

            newG.appendChild(makeForeignObject({
                x: 0,
                y: 0,
                width: 72,
                height: 72,
                class: 'nodeicon' + (isImported ? 'dataset-imported' : '')
            }, iconElement));

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: 72,
                height: 72
            }));

            newG.appendChild(makeForeignObject({
                x: 56,
                y: -15,
                width: 34,
                height: 34,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Repeated object sticker
            if (dataset.veLoopDatasetRef) {
                let coords = {
                    cx: 1,
                    cy: 67,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 57,
                };

                drawRepeatedObjectSticker(newG, coords);
                offset.y += 18;
            }

            // Information sticker
            // To display when the dataset has a short description
            if (angular.isDefined(dataset.shortDesc) && dataset.shortDesc.length > 0) {
                let coords = {
                    cx: 1,
                    cy: 67,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 57
                };

                drawInformationSticker(newG, coords, offset);
                offset.y += 18;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && dataset.discussionsFullIds && dataset.discussionsFullIds.length > 0) {
                let unread = hasUnreadDiscussion(dataset);
                let coords = {
                    cx: 0,
                    cy: 67.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: -10,
                    indicatorY: 57.5,
                    contentX: -20,
                    contentY: 57.5,
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, dataset.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});
            }

            // Feature group sticker
            // To display when the dataset is a feature group
            if (angular.isDefined(dataset.featureGroup) && dataset.featureGroup) {
                let coords = {
                    cx: 70,
                    cy: 67,
                    rx: 11,
                    ry: 11,
                    indicatorX: 60,
                    indicatorY: 58
                };

                drawFeatureGroupSticker(newG, coords);
            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleModelNode(local) {
    const isLLMGenericSm = (savedModel) => savedModel.savedModelType && ['LLM_GENERIC', 'PLUGIN_AGENT', 'PYTHON_AGENT', 'TOOLS_USING_AGENT', "RETRIEVAL_AUGMENTED_LLM"].includes(savedModel.savedModelType);
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const sm = FlowGraph.node(nodeId);
        if (isLLMGenericSm(sm)) {
            element.addClass("fine-tuned-sm");
        }


        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);
            const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y-coords.height})`});

            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width * 0.707,
                height: coords.height * 0.707,
                transform: `translate(${coords.width/2}) rotate(45)`,
                opacity: 0,
                class: 'fill'
            }));

            const othersZones = FlowGraph.nodeSharedBetweenZones(sm);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);
            const savedModelClass = isLLMGenericSm(sm) ? ' llm-model' : '';

            if (sm.partitioned) {
                /* Partition boxes */
                for (let offset of [-10, -5, 0]) {
                    newG.appendChild(makeSVG('rect', {
                        x: offset,
                        y: offset,
                        width: coords.width * 1.41421,
                        height: coords.height * 1.41421,
                        transform: `translate(${coords.width}) rotate(45)`,
                        class: 'fill node__rectangle--partitioned partitioning-indicator'
                    }));
                }

                /* White background for the rest of the icon */
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`,
                    class: 'node__rectangle--blank'
                }));
            }

            if (isExported || isImported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`,
                    class: 'fill main-model-rectangle model-zone-' + (isExported ? 'exported' : 'imported') + savedModelClass
                }));
            }

            let iconText;
            if (sm.externalSavedModelType) {
                const externalSmTypeToIcon = {
                    "sagemaker": "model_amazon_sagemaker",
                    "azure-ml": "model_azureml",
                    "vertex-ai": "model_google_vertex",
                    "databricks": "model_databricks",
                    "mlflow": "model_mlflow"
                }
                iconText = flowIconset.icons[externalSmTypeToIcon[sm.externalSavedModelType]];
            }
            if (iconText === undefined) {
                let iconKey = 'clustering';
                if (sm.taskType === 'PREDICTION') {
                    switch (sm.predictionType) {
                        case 'TIMESERIES_FORECAST':
                            iconKey = 'timeseries';
                            break;
                        case 'DEEP_HUB_IMAGE_CLASSIFICATION':
                        case 'DEEP_HUB_IMAGE_OBJECT_DETECTION':
                            iconKey = 'computer_vision';
                            break;
                        case 'CAUSAL_REGRESSION':
                        case 'CAUSAL_BINARY_CLASSIFICATION':
                            iconKey = 'causal';
                            break;
                        default:
                            iconKey = 'regression';
                            if (sm.backendType === 'KERAS') {
                                iconKey = 'deep_learning';
                            }
                    }
                } else if (sm.savedModelType === 'PYTHON_AGENT') {
                    iconKey = 'ai_agent_code';
                } else if (sm.savedModelType === 'PLUGIN_AGENT') {
                    iconKey = 'ai_agent_plugin';
                } else if (sm.savedModelType === 'TOOLS_USING_AGENT') {
                    iconKey = 'ai_agent_visual';
                } else if (sm.savedModelType === 'RETRIEVAL_AUGMENTED_LLM') {
                    iconKey = 'llm_augmented';
                } else if (isLLMGenericSm(sm)) {
                    iconKey = 'fine_tuning';
                }

                iconText = flowIconset.icons['model_' + iconKey];
            }
            const iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("path,rect");
            const newG2 = makeSVG("g");
            iconElt[0].forEach(i => newG2.appendChild(i));
            d3.select(newG2).attr("transform", " scale(0.707, 0.707) translate(1, 1)")
            let klass = "bzicon sm-icon";
            klass += isImported ? ' model-zone-imported': isExported ? ' model-zone-exported' : '';
            klass += savedModelClass;
            d3.select(newG2).classed(klass, true);
            newG.appendChild(newG2);

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width + 26}, 24)`,
                'data-height': coords.height
            }));

            if (!sm.description) {
                sm.description = "Saved model...";
            }

            // Invisible rect to capture mouse events, else
            // the holes in the icon don't capture the mouse
            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width * 1.41421,
                height: coords.height * 1.41421,
                opacity: 0,
                transform: `translate(${coords.width})  rotate(45)`,
            }));

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: coords.width * 1.41421,
                height: coords.height * 1.41421,
                transform: `translate(${coords.width})  rotate(45)`,
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 2.07,
                width: coords.width*2+60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(sm.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 45,
                y: -10,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the sm has a short description
            if (angular.isDefined(sm.shortDesc) && sm.shortDesc.length > 0) {
                let coords = {
                    cx: 27,
                    cy: 63,
                    rx: 10,
                    ry: 10,
                    indicatorX: 17,
                    indicatorY: 53
                };

                drawInformationSticker(newG, coords);
                offset.x += 12;
                offset.y += 12;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && sm.discussionsFullIds && sm.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(sm);
                let coords = {
                    cx: 24,
                    cy: 64.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: 14,
                    indicatorY: 55,
                    contentX: 4,
                    contentY: 55
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, sm.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleEvaluationStoreNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const mes = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);
            const newG = makeSVG('g', {class: 'newG bzicon', transform: `translate(${coords.x} ${coords.y-coords.height})`});

            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width * 0.707,
                height: coords.height * 0.707,
                transform: `translate(${coords.width/2}) rotate(45)`,
                opacity: 0,
                class: 'fill'
            }));

            const othersZones = FlowGraph.nodeSharedBetweenZones(mes);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);

            if (mes.partitioned) {
                /* Partition boxes */
                for (let offset of [-10, -5, 0]) {
                    newG.appendChild(makeSVG('rect', {
                        x: offset,
                        y: offset,
                        width: coords.width * 1.41421,
                        height: coords.height * 1.41421,
                        transform: `translate(${coords.width}) rotate(45)`,
                        class: 'fill node__rectangle--partitioned partitioning-indicator'
                    }));
                }

                /* White background for the rest of the icon */
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`
                }));
            } else {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`
                }));
            }

            if (isExported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`,
                    class: 'fill evaluation-store-zone-exported'
                }));
            }

            const icon = 'dku-icon-model-evaluation-store-48';
            newG.appendChild(makeForeignObject({
                x: 12,
                y: 12,
                width: 48,
                height: 48,
                class: 'nodeicon' + (isImported ? ' evaluation-store-zone-imported': isExported ? ' evaluation-store-zone-exported' : '')
            }, $(`<div class="flow-tile" style="text-align: center"><i class="${icon}" /></div>`)));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width + 26}, 24)`,
                'data-height': coords.height
            }));

            if (!mes.description) {
                mes.description = "Model evaluation store...";
            }

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: coords.width * 1.41421,
                height: coords.height * 1.41421,
                transform: `translate(${coords.width})  rotate(45)`,
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 2.07,
                width: coords.width*2+60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(mes.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 45,
                y: -10,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the mes has a short description
            if (angular.isDefined(mes.shortDesc) && mes.shortDesc.length > 0) {
                let coords = {
                    cx: 27,
                    cy: 63,
                    rx: 10,
                    ry: 10,
                    indicatorX: 17,
                    indicatorY: 53
                };

                drawInformationSticker(newG, coords);
                offset.x += 12;
                offset.y += 12;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && mes.discussionsFullIds && mes.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(mes);
                let coords = {
                    cx: 24,
                    cy: 64.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: 14,
                    indicatorY: 55,
                    contentX: 4,
                    contentY: 55
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, mes.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}


function restyleRetrievableKnowledgeNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const rk = FlowGraph.node(nodeId);

        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);

            let clazz = 'newG bzicon retrievable-knowledge'
            if (rk.neverBuilt) {
                clazz += ' never-built-computable';
            }
            const newG = makeSVG('g', {class: clazz, transform: `translate(${coords.x} ${coords.y})`});

            const othersZones = FlowGraph.nodeSharedBetweenZones(rk);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);
            const margin = isExported && othersZones.size > 0 ? 4 : 0;

            if (isExported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill retrievable-knowledge-zone-exported'
                }));
            }
            newG.appendChild(makeSVG('rect', {
                x: margin,
                y: margin,
                width: coords.width - (margin*2),
                height: coords.height - (margin*2),
                class: 'fill retrievable-knowledge-rectangle main-retrievable-knowledge-rectangle' + (isImported ? ' retrievable-knowledge-zone-imported' : '')
            }));

            const icon = 'dku-icon-cards-stack-48';
            newG.appendChild(makeForeignObject({
                x: 12,
                y: 13,
                width: 48,
                height: 48,
                class: 'nodeicon' + (isImported ? ' retrievable-knowledge-zone-imported': isExported ? ' retrievable-knowledge-zone-exported' : '')
            }, $(`<div class="flow-tile" style="text-align: center"><i class="${icon}" /></div>`)));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width}, 0)`,
                'data-height': coords.height
            }));

            if (!rk.description) {
                rk.description = "Knowledge Bank...";
            }

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.07,
                width: coords.width + 60,
                height: 45,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(rk.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 56,
                y: -15,
                width: 34,
                height: 34,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the rk has a short description
            if (angular.isDefined(rk.shortDesc) && rk.shortDesc.length > 0) {
                let coords = {
                    cx: 1,
                    cy: 67,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 57
                };

                drawInformationSticker(newG, coords);
                offset.y += 20;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && rk.discussionsFullIds && rk.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(rk);
                let coords = {
                    cx: 24,
                    cy: 64.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: 14,
                    indicatorY: 55,
                    contentX: 4,
                    contentY: 55
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, rk.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleFolderNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const folder = FlowGraph.node(nodeId);

        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            const coords = polygonToRectData(dotPolygon);

            const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y})`});

            const othersZones = FlowGraph.nodeSharedBetweenZones(folder);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);

            element.find("text").remove();

            const iconText = flowIconset.icons["folder"];
            let iconElt = d3.select($.parseXML(iconText)).select("svg").select("g")
            iconElt.attr("transform", " scale(0.57, 0.57) ");
            iconElt.classed("bzicon" + (isImported ? " folder-zone-imported" : ''), true);

            newG.appendChild(iconElt[0][0]);

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width}, 0)`,
                'data-height': coords.height
            }));

            if (!folder.description) {
                folder.description = "Managed folder";
            }
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.01,
                width: coords.width + 60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(folder.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            // Invisible rect to capture mouse events, else
            // the holes in the icon don't capture the mouse
            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                opacity: 0
            }));

            newG.appendChild(makeForeignObject({
                x: 48,
                y: -13,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the folder has a short description
            if (angular.isDefined(folder.shortDesc) && folder.shortDesc.length > 0) {
                let coords = {
                    cx: 1,
                    cy: 55,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 45
                };

                drawInformationSticker(newG, coords);
                offset.y += 13;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && folder.discussionsFullIds && folder.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(folder);
                let coords = {
                    cx: 0,
                    cy: 51.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: -10,
                    indicatorY: 42,
                    contentX: -20,
                    contentY: 42
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, folder.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    };
}

function restyleStreamingEndpointNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const sm = FlowGraph.node(nodeId);

        const streamingEndpoint = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);
            const newG = makeSVG('g', {class: 'newG bzicon', transform: `translate(${coords.x} ${coords.y})`});

            newG.appendChild(makeSVG('path', {
                d: "M0,0 L48,0 L64,29 L48,58 L0,58 Z",
                opacity: 1,
                class: 'fill'
            }));

            const icon = $filter("toModernIcon")($filter("datasetTypeToIcon")(streamingEndpoint.streamingEndpointType, 48), 48);
            newG.appendChild(makeForeignObject({
                x: 5,
                y: 5,
                width: 48,
                height: 48,
                class: 'nodeicon'
            }, $(`<div class="flow-tile" style="text-align: center"><i class="${icon}" /></div>`)));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width + 26}, 24)`,
                'data-height': coords.height
            }));

            newG.appendChild(makeSVG('path', {
                class: 'selection-outline',
                d: "M-1,-1 L48.5,-1 L65,29 L48.5,59 L-1,59 Z"
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.2,
                width: coords.width + 60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sm.name.replace(/([-_.])/g, '$1\u200b') + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 45,
                y: -10,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleRunnableNode(runnableType) {
    return function(element, g) {
        if (!(["recipe", "labeling_task"].includes(runnableType))) {return}
        const isRecipe = runnableType === 'recipe';
        const nodeId = element.attr('data-id');
        const runnable = FlowGraph.node(nodeId);

        const iconScale = 0.52

        const dotEllipse = element.find("ellipse");
        if (dotEllipse.length) {
            const coords = circleToRectData(dotEllipse);
            const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y}) scale(${iconScale}, ${iconScale})`});

            const recipeType = isRecipe ? runnable.recipeType : "labeling_task";
            const iconText = flowIconset.icons[$filter("recipeFlowIcon")(recipeType)];
            d3.select(newG).classed("bzicon recipeicon-" + recipeType, true);

            if (isRecipe && (runnable.recipeType.startsWith("CustomCode_") || runnable.recipeType.startsWith("App_"))) {
                const colorClass = $filter("recipeTypeToColorClass")(runnable.recipeType);
                d3.select(newG).classed("universe-fill " + colorClass, true);
            }

            // WARNING: this conflicts with graphviz layout: recipe names and other nodes can overlap.
            // newG.appendChild(makeForeignObject({
            //     x: -30,
            //     y: coords.height * 1.07,
            //     width: coords.width + 60,
            //     height: 45,
            //     class: 'nodelabel-wrapper',
            //     transform: `scale(${1/iconScale}, ${1/iconScale})`
            // }, $('<div><span>' + recipe.description.replace(/([-_])/g, '$1\u200b') + '</span></div>')));

            d3.select(g).attr("data-recipe-type", isRecipe ? runnable.recipeType : "Labeling");

            let iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("g");
            // Some icons don't have a g ...
            if (iconElt.length === 0 || iconElt[0].length === 0) {
                iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("path");
            }
            try {
                $(dotEllipse).replaceWith(newG);
                iconElt[0].forEach(function(x){
                    newG.appendChild(x);
                });
                // Invisible rect to capture mouse events, else
                // the holes in the icon don't capture the mouse
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width / iconScale,
                    height: coords.height / iconScale,
                    opacity: 0
                }));
            } catch (e) {
                logger.error("Failed patch recipe icon", e)
            }

            if (isRecipe && (runnable.recipeType.startsWith("CustomCode_") || runnable.recipeType.startsWith("App_"))){
                const icon = $filter("toModernIcon")($filter("recipeTypeToIcon")(runnable.recipeType, 48), 48);
                newG.appendChild(makeForeignObject({
                    x: 0,
                    y: 0,
                    width: SIZE,
                    height: SIZE,
                    class: 'nodeicon'
                }, $('<div class="recipe-custom-code-object"><i class="' + icon + '"></i></div>')));
            }

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${SIZE}, 0) scale(${1/iconScale}, ${1/iconScale})`,
                'data-height': SIZE * iconScale
            }));

            newG.appendChild(makeSVG('circle', {
                class: 'selection-outline',
                r: 50*iconScale,
                cx: 50*iconScale,
                cy: 50*iconScale,
                transform: `scale(${1/iconScale}, ${1/iconScale})`,
            }));

            newG.appendChild(makeForeignObject({
                x: 30,
                y: -17,
                width: 32,
                height: 32,
                class: 'node-totem',
                transform: `scale(${1/iconScale}, ${1/iconScale})`
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Repeated object sticker
            if (runnable.veLoopDatasetRef) {
                let coords = {
                    cx: 14,
                    cy: 86,
                    rx: 10 / iconScale,
                    ry: 10 / iconScale,
                    indicatorX: -1.5,
                    indicatorY: 76,
                };

                drawRepeatedObjectSticker(newG, coords);
                offset.x += 11;
                offset.y += 34;
            }

            // Information sticker
            // To display when the recipe has a short description
            if (angular.isDefined(runnable.shortDesc) && runnable.shortDesc.length > 0) {
                let coords = {
                    cx: 14,
                    cy: 86,
                    rx: 10 / iconScale,
                    ry: 10 / iconScale,
                    indicatorX: -0.5,
                    indicatorY: 69,
                };

                drawInformationSticker(newG, coords, offset);
                if (offset.x || offset.y) {
                    // We've drawn the repeated object sticker too
                    offset.x -= 10;
                    offset.y += 36.5;
                } else {
                    offset.x += 11;
                    offset.y += 34;
                }
            }

            // Discussion sticker
            // To display when the recipe has an unread discussion
            if (runnable.discussionsFullIds.length > 0) {
                let unread = hasUnreadDiscussion(runnable);
                let coords = {
                    cx: 7.5,
                    cy: 90.5,
                    rx: 10.5 / iconScale,
                    ry: 9 / iconScale,
                    indicatorX: -10,
                    indicatorY: 75,
                    contentX: -12,
                    contentY: 80,
                };
                drawDiscussionSticker(newG, runnable.discussionsFullIds.length, unread, coords, offset);
            }
        }
    }
}
});

app.service('ProjectFlowGraphLayout', function(FlowGraph, LoggerProvider) {

    var logger = LoggerProvider.getLogger('flow');

    // This function takes isolated ("draft") computables
    // and move them to a container to put them next to the flow
    // Note that things are called datasets for historical reasons but they might be other things
    this.relayout = function(globalSvg) {
        const zones = $(globalSvg).find("svg");
        if (zones && zones.length > 0) {
            for (let i = 0; i < zones.length; i++) {
                this.relayoutSVG($(zones[i]), true);
            }
        } else {
            this.relayoutSVG(globalSvg, false);
        }
    };

    this.relayoutSVG = (svg, zones) => {
        const usedDatasets = makeSVG('g', { class: 'usedDatasets' });
        const inputDatasets = makeSVG('g', { class: 'inputDatasets' });
        const draftDatasets = makeSVG('g', { class: 'draftDatasets' });
        svg.find('g[class=graph]').append(usedDatasets);
        $(usedDatasets).append(inputDatasets);
        svg.find('g[class=graph]').append(draftDatasets);

        svg.find('g[data-type]').each(function (index, boxElement) {
            const nodeId = $(boxElement).data("id");
            const node = FlowGraph.node(nodeId);
            if(!node) {
                logger.warn('Graph node does not exist: ', nodeId)
                return;
            }

            if(!node.predecessors.length && !node.successors.length) {
                draftDatasets.appendChild(boxElement);
            } else if(!node.predecessors.length) {
                inputDatasets.appendChild(boxElement);
            } else {
                usedDatasets.appendChild(boxElement);
            }
        });

        // Relayout
        const hasChildNodes = usedDatasets.childNodes.length + inputDatasets.childNodes.length > 1;
        let datasetsPerColumns = Math.max(inputDatasets.childNodes.length, Math.floor(Math.sqrt(draftDatasets.childNodes.length)), 1);
        let gridWidth = 180;
        let gridHeight = 180;
        let columnHeight = gridHeight * datasetsPerColumns;
        if(inputDatasets.childNodes.length && !zones) {
            datasetsPerColumns = inputDatasets.childNodes.length;
            gridWidth = inputDatasets.childNodes[0].getBBox().width + 50;
            columnHeight = inputDatasets.getBBox().height;
            gridHeight = columnHeight / datasetsPerColumns;
        }
        let nbFullRows = Math.floor(draftDatasets.childNodes.length / datasetsPerColumns) || 1;
        for(let index = 0; index < draftDatasets.childNodes.length; index++) {
            // full row
            let offset = 0;
            if (nbFullRows > 0 && index >= nbFullRows * datasetsPerColumns) {
                offset = (columnHeight - gridHeight * (draftDatasets.childNodes.length % datasetsPerColumns))/2;
            }
            let dx = Math.floor(index / datasetsPerColumns) * gridWidth + 20;
            let dy = offset + index % datasetsPerColumns * gridHeight + 50;
            $(draftDatasets).children().eq(index).find('g').first().attr('transform', `translate(${dx}  ${dy})`);
        }

        //  move draftDatasets out of the way
        setTimeout(function() {
            let usedDatasetsBB = usedDatasets.getBBox();
            let draftDatasetsBB = draftDatasets.getBBox();
            let translateX = usedDatasetsBB.x - draftDatasetsBB.x - draftDatasetsBB.width - 200;
            let translateY = usedDatasetsBB.y - draftDatasetsBB.y;
            if (!zones) {
                draftDatasets.setAttribute('transform', 'translate(' + translateX + ' ' + translateY + ')');
            } else {
                let translateGraphY = hasChildNodes && draftDatasets.childNodes.length ? -usedDatasetsBB.y + 32 : hasChildNodes ? -usedDatasetsBB.y : -draftDatasetsBB.y;
                svg.find('g[class=graph]')[0].setAttribute('transform', 'translate(32,' + translateGraphY + ')');
            }
        });
    }
});


app.service('InterProjectGraphLayout', function(FlowGraph) {
    // Similar to project flow, this function takes isolated ("standalone") projects
    // and move them to a container to put them below the graph of connected (by exposed elements) projects
    this.relayout = function(svg) {
        let connectedProjects = makeSVG('g', { class: 'connectedProjects' });
        let inputProjects = makeSVG('g', { class: 'inputProjects' });
        let standAloneProjects = makeSVG('g', { class: 'standAloneProjects' });
        let projectFolders = makeSVG('g', { class: 'projectFolders' });

        svg.find('g[class=graph]').append(projectFolders);
        svg.find('g[class=graph]').append(connectedProjects);
        $(connectedProjects).append(inputProjects);
        svg.find('g[class=graph]').append(standAloneProjects);

        svg.find('g[data-type]').each(function (index, boxElement) {
            let node = FlowGraph.node($(boxElement).data("id"));
            // if its a draft dataset, move it to another elt
            if (node.nodeType == "PROJECT_FOLDER") {
                projectFolders.appendChild(boxElement);
            } else if (!node.predecessors.length && !node.successors.length) {
                standAloneProjects.appendChild(boxElement);
            } else if (!node.predecessors.length) {
                inputProjects.appendChild(boxElement);
            } else {
                connectedProjects.appendChild(boxElement);
            }
        });

        // Relayout
        let minProjectsPerRow = 3;
        let maxProjecsPerRow = 10;

        let projectsPerRows = Math.min(Math.max(minProjectsPerRow, Math.floor(Math.sqrt(standAloneProjects.childNodes.length))), maxProjecsPerRow);
        let cellMargin = 20
        let cellWidth = 150 + cellMargin;
        let cellHeight = 100 + cellMargin;
        let pfCellHeight = 40 + cellMargin;
        let rowWidth = cellWidth * projectsPerRows;
        if (inputProjects.childNodes.length) {
            cellWidth = inputProjects.childNodes[0].getBBox().width + 50;
            cellHeight = inputProjects.childNodes[0].getBBox().height + cellMargin;
            projectsPerRows = Math.min(Math.max(minProjectsPerRow, Math.floor(connectedProjects.getBBox().width / cellWidth)), maxProjecsPerRow);
        }
        if (projectFolders.childNodes.length) {
            pfCellHeight = projectFolders.childNodes[0].getBBox().height + cellMargin;
        }
        for(let index = 0; index < standAloneProjects.childNodes.length; index++) {
            let tr = (index % projectsPerRows) * cellWidth +' '+ Math.floor(index / projectsPerRows) * cellHeight;
            $(standAloneProjects).children().eq(index).find('g').first()
                .attr('transform', 'translate('+ tr +')');
        }
        for (let index = 0; index < projectFolders.childNodes.length; index++) {
            let tr = (index % projectsPerRows) * cellWidth + ' ' + Math.floor(index / projectsPerRows) * pfCellHeight;
            $(projectFolders).children().eq(index).find('g').first()
                .attr('transform', 'translate(' + tr + ')');
        }
        //  move standAloneProjects out of the way
        let connectedProjectsBB = connectedProjects.getBBox();
        let standAloneProjectsBB = standAloneProjects.getBBox();
        let sapTranslateX = connectedProjectsBB.x;
        let sapTranslateY = connectedProjectsBB.y + connectedProjectsBB.height + 100;
        standAloneProjects.setAttribute('transform', 'translate(' + sapTranslateX + ' ' + sapTranslateY + ')');
        // move projectFolders out of the way (upside)
        let projectFoldersBB = projectFolders.getBBox();
        let pfTranslateX = connectedProjectsBB.x;
        let pfTranslateY = connectedProjectsBB.y - (projectFoldersBB.height + 100);
        projectFolders.setAttribute('transform', 'translate(' + pfTranslateX + ' ' + pfTranslateY + ')');
    };
});


app.service('InterProjectGraphStyling', function($filter, ImageUrl, FlowGraph, CachedAPICalls, LoggerProvider) {

    const SIZE = 100;

    const logger = LoggerProvider.getLogger('projectsGraph');

    const formatters = {
        'PROJECT_FOLDER': restyleNodeForProjectFolder,
        'PROJECT': restyleNodeForProject,
        'BUNDLE_EO': restyleNodeForExposedObject,
    };
    let flowIconset;
    CachedAPICalls.flowIcons.success(function(data) {
        flowIconset = data;
    });

    this.restyleGraph = function(svg, graph) {
        svg.find('title').remove();
        svg.find('g[data-type]').each(function (index, g) {
            try {
                restyleNode(g);
            } catch (e) {
                logger.error("Failed to restyle flow node: ", e);
            }
        });
    };

    function restyleNode(g) {
        const element = $(g);
        const nodeType = element.attr('data-type');

        if (formatters[nodeType]) {
            formatters[nodeType](element, g);
        }
    };

    function createExposedObjectSvg(nodeIcons, index, x, y, d) {
        const g = makeSVG('g', {
            class: nodeIcons[index].type,
            transform: 'translate(' + x + ' ' + y + ') scale(' + d/SIZE +',' + d/SIZE +')'
        });
        nodeIcons[index].elt[0].forEach(function(x){
            g.appendChild(x);
        });
        addCountToExposedObjectSvg(g, nodeIcons, index, x, y, d)
        return g
    }

    function addCountToExposedObjectSvg(g, nodeIcons, index, x, y, d) {
        const r = SIZE / 5;
        const cx = SIZE/2 + SIZE/(2 * Math.sqrt(2));
        const cy = SIZE/2 + SIZE/(2 * Math.sqrt(2));

        const circle = makeSVG('circle', {
            r: r,
            cx: cx,
            cy: cy,
            class: 'count-circle'
        });

        const text = makeSVG('text', {
            x: cx,
            y: cy,
            class: "count-text"
        });
        text.textContent = nodeIcons[index].nbElements;

        g.appendChild(circle);
        g.appendChild(text);
    }

    // several exposed objects (of different type) will result in one nodes containing several icons
    function drawExposedObjectNodeIcons(newG, nodeIcons, coords) {
        switch (nodeIcons.length) {
            case 1: {
                const g0 = createExposedObjectSvg(nodeIcons, 0, 1, 1, coords.width - 2);
                newG.appendChild(g0);
                break;
            }
            case 2: {
                const diameter = (coords.width - 2)/2;

                const x0 = 1;
                const y0 = 1 + coords.height/4;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1 + coords.width/2;
                const y1 = 1 + coords.height/4;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                break;
            }
            case 3: {
                const diameter = (coords.width - 2)/2;

                const x0 = 1;
                const y0 = 1 + coords.height/4;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1 + coords.width/2;
                const y1 = 1;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                const x2 = 1 + coords.width/2;
                const y2 = 1 + coords.height/2;
                const g2 = createExposedObjectSvg(nodeIcons, 2, x2, y2, diameter);
                newG.appendChild(g2);

                break;
            }
            case 4: {
                const diameter = (coords.width - 2)/2;

                const x0 = 1;
                const y0 = 1;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1;
                const y1 = 1 + coords.height/2;;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                const x2 = 1 + coords.width/2;
                const y2 = 1;
                const g2 = createExposedObjectSvg(nodeIcons, 2, x2, y2, diameter);
                newG.appendChild(g2);

                const x3 = 1 + coords.width/2;
                const y3 = 1 + coords.height/2;
                const g3 = createExposedObjectSvg(nodeIcons, 3, x3, y3, diameter);
                newG.appendChild(g3);

                break;
            }
            case 5: {
                const diameter = (coords.width - 2)/3;

                const x0 = 1 + coords.width/3;
                const y0 = 1 ;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1;
                const y1 = 1 + coords.height/3;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                const x2 = 1 + coords.width/3;
                const y2 = 1 + coords.height/3;
                const g2 = createExposedObjectSvg(nodeIcons, 2, x2, y2, diameter);
                newG.appendChild(g2);

                const x3 = 1 + coords.width * 2/3;
                const y3 = 1 + coords.height/3;
                const g3 = createExposedObjectSvg(nodeIcons, 3, x3, y3, diameter);
                newG.appendChild(g3);

                const x4 = 1 + coords.width/3;
                const y4 = 1 + coords.height * 2/3;
                const g4 = createExposedObjectSvg(nodeIcons, 4, x4, y4, diameter);
                newG.appendChild(g4);
            }
        }
    }

    function restyleNodeForProjectFolder(element) {
        const nodeId = element.attr('data-id');
        const projectFolder = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');

        //replacing polygon by g of same size
        const coords = polygonToRectData(dotPolygon);
        const newG = makeSVG('g', {class: 'newG', transform: 'translate(' + coords.x + ' ' + coords.y + ')'});
        $(dotPolygon).replaceWith(newG);
        element.find("text").remove();

        // fill
        newG.appendChild(makeSVG('rect', {
            x: 0,
            y: 0,
            width: coords.width,
            height: coords.height,
            class: 'fill'
        }));

        // html-content
        newG.appendChild(makeForeignObject(
            {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                class: 'project-folder-meta nodelabel-wrapper'
            },
            $(
                '<div>' +
                    '<p class="single-line"><i class="icon-folder-close"></i>' + sanitize(projectFolder.description.replace(/([-_.])/g, '$1\u200b')) + '</p>' +
                '</div>'
            )
        ));
    }

    function restyleNodeForProject(element) {
        const nodeId = element.attr('data-id');
        const project = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');

        //replacing polygon by g of same size
        const coords = polygonToRectData(dotPolygon);
        const newG = makeSVG('g', {class: 'newG',
            transform: 'translate(' + coords.x + ' ' + coords.y + ')'
        });
        if (project.isArchived) {
            d3.select(newG).classed("archived", true);
        }
        if (project.isForbidden) {
            d3.select(newG).classed("forbidden", true);
        }
        if (project.isNotInFolder) {
            d3.select(newG).classed("not-in-folder", true);
        }
        $(dotPolygon).replaceWith(newG);
        element.find("text").remove();

        //fill
        newG.appendChild(makeSVG('rect', {
            x: 0,
            y: 0,
            width: coords.width,
            height: coords.height,
            class: 'fill'
        }));

        // html-content
        newG.appendChild(makeForeignObject(
            {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                class: 'project-meta nodelabel-wrapper'
            },
            $(
                '<div>' +
                    '<i class="img-area">' + (project.isForbidden ? '' : ('<img src="' + ImageUrl(project.name, project.projectKey, project.projectKey, "PROJECT", project.objectImgHash, "80x200", project.imgColor, project.imgPattern, project.showInitials) + '" />')) + '</i>' +
                    '<p>' + sanitize(project.description.replace(/([-_.])/g, '$1\u200b')) + '</p>' +
                '</div>'
            )
        ));

        if (project.projectAppType === 'APP_TEMPLATE') {
            const appIconSize = 25;
            newG.appendChild(makeForeignObject(
                {
                    x: 0,
                    y: coords.height - appIconSize,
                    width: appIconSize,
                    height: appIconSize,
                    class: 'project-meta'
                },
                $(`
                    <div class="app-template-overlay app-template-overlay--graph" title="${project.isAppAsRecipe ? 'Application-as-recipe' : 'Visual application'} template">
                        <i class="${project.isAppAsRecipe ? 'icon-dku-application-as-recipe' : 'icon-project-app'}"></i>
                    </div>
                `)
            ));
        }

        // forbidden-project (we can still see it because we can read exposed object from it)
        if (project.isForbidden) {
            newG.appendChild(makeForeignObject({
                x: 113,
                y: 64,
                width: 30,
                height: 30,
                class: 'forbidden-icon'
            }, $('<div><i class="icon-lock" /></div>')));

        }

        // forbidden-project (we can still see it because we can read exposed object from it)
        if (project.isNotInFolder) {
            newG.appendChild(makeForeignObject({
                x: 35,
                y: 63,
                width: 30,
                height: 30,
                class: 'not-in-folder-icon'
            }, $('<div><span class="icon-stack"><i class="icon-folder-close"></i><i class="icon-ban-circle icon-stack-base"></i></span></div>')));

        }
    }

    function restyleNodeForExposedObject(element) {
        const icons = [];
        function addIcon (type, nbElements) {
            const icon = {};
            const iconText = flowIconset.icons[type];
            let iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("circle, path, rect");
            icon.elt = iconElt;
            icon.type = type;
            icon.nbElements = nbElements;
            icons.push(icon);
        }

        function addIcons(bundle) {
            if (bundle.exposedDatasets.length > 0) {
                addIcon("eo_datasets", bundle.exposedDatasets.length);
            }
            if (bundle.exposedFolders.length > 0) {
                addIcon("eo_folders", bundle.exposedFolders.length);
            }
            if (bundle.exposedModels.length > 0) {
                addIcon("eo_models", bundle.exposedModels.length);
            }
            if (bundle.exposedNotebooks.length > 0) {
                addIcon("eo_notebooks", bundle.exposedNotebooks.length);
            }
            if (bundle.exposedWebApps.length > 0) {
                addIcon("eo_webapps", bundle.exposedWebApps.length);
            }
            if (bundle.exposedReports.length > 0) {
                // TODO there is no Flow icon named "eo_reports.svg", so will likely not work. But reports are not shown on Flow, so...
                addIcon("eo_reports", bundle.exposedReports.length);
            }
            if (bundle.errorMessages.length > 0) {
                // TODO there is no Flow icon named "error.svg", so will likely not work. But reports are not shown on Flow, so...
                addIcon("error", bundle.errorMessages.length);
            }
        }

        const nodeId = element.attr('data-id');
        const bundle = FlowGraph.node(nodeId);

        const dotEllipse = element.find("ellipse");

        const coords = circleToRectData(dotEllipse);
        const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y})` });
        d3.select(newG).classed("bzicon", true);

        addIcons(bundle);
        if (icons.length == 0) {
            throw new Error("No icons for type", bundle);
        }
        drawExposedObjectNodeIcons(newG, icons, coords);

        try {
            $(dotEllipse).replaceWith(newG);
            // Invisible rect to capture mouse events, else
            // the holes in the icon don't capture the mouse
            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                opacity: 0
            }));
        } catch (e) { /* Nothing for now */ }

        const text = element.find('text:not(.count-text)');
        text.remove();
    }
});


app.service('FlowGraphFiltering', function() {
    function getBBoxAccountingForTranslation(svgElt) {
        if (!svgElt.hasAttribute("transform")) {
            return svgElt.getBBox();
        }

        const parent = svgElt.parentNode;
        const tmpGElt = makeSVG('g');
        parent.insertBefore(tmpGElt, svgElt);
        tmpGElt.appendChild(svgElt)
        const bbox = tmpGElt.getBBox();
        parent.insertBefore(svgElt, tmpGElt);
        tmpGElt.remove();
        return bbox;
    }

    //TODO @flow move to another service
    this.getBBoxFromSelector = function (globalSvg, selector) {
        const svgs = $(selector).closest('svg'); // Retrieve correct svg (Mostly in case of zones)
        let gTopLeft, gBottomRight;
        svgs.each(function() {
            const svg = $(this);
            let topLeft, bottomRight;
            function addItemToBBox(refBBox) {
                return function() {
                    let bbox = getBBoxAccountingForTranslation(this);
                    if (refBBox && svg.is(globalSvg)) {
                        // "draftDatasets" (unconnected objects) have been translated,
                        // We need to compensate for that (Only when no zones are present)
                        bbox = {
                            x: bbox.x + refBBox.x,
                            y: bbox.y + refBBox.y,
                            width: bbox.width,
                            height: bbox.height,
                        };
                    }

                    if (topLeft === undefined) {
                        topLeft = svg[0].createSVGPoint();
                        topLeft.x = bbox.x;
                        topLeft.y = bbox.y;

                        bottomRight = svg[0].createSVGPoint();
                        bottomRight.x = bbox.x + bbox.width;
                        bottomRight.y = bbox.y + bbox.height;
                    } else {
                        topLeft.x = Math.min(topLeft.x, bbox.x);
                        topLeft.y = Math.min(topLeft.y, bbox.y);

                        bottomRight.x = Math.max(bottomRight.x, bbox.x + bbox.width);
                        bottomRight.y = Math.max(bottomRight.y, bbox.y + bbox.height);
                    }
                }
            }
            const graphBBox = svg.find('g.graph')[0].getBBox();
            const isZone = $(selector).parentsUntil(svg, '.usedDatasets').length === 0 && $(selector).parentsUntil(svg, '.draftDatasets').length === 0;
            const matrix = isZone ? svg[0].getTransformToElement(globalSvg[0]) : $('.usedDatasets', svg)[0].getTransformToElement(globalSvg[0]);

            if (isZone) {
                svg.find(selector).each(addItemToBBox());
            } else {
                $('.usedDatasets', svg).find(selector).each(addItemToBBox());
                $('.draftDatasets', svg).find(selector).each(addItemToBBox(graphBBox));
            }

            if (topLeft === undefined) {
                //console.info("Cannot compute bounding box around empty set of items");
                return undefined;
            }

            topLeft = topLeft.matrixTransform(matrix);
            bottomRight = bottomRight.matrixTransform(matrix);

            if (gTopLeft === undefined) {
                gTopLeft = { x: topLeft.x, y: topLeft.y };
                gBottomRight = { x: bottomRight.x, y: bottomRight.y };
            } else {
                gTopLeft.x = Math.min(gTopLeft.x, topLeft.x);
                gTopLeft.y = Math.min(gTopLeft.y, topLeft.y);

                gBottomRight.x = Math.max(gBottomRight.x, bottomRight.x);
                gBottomRight.y = Math.max(gBottomRight.y, bottomRight.y);
            }
        });

        if (gTopLeft === undefined) {
            return undefined;
        }

        return {
            x: gTopLeft.x,
            y: gTopLeft.y,
            width: gBottomRight.x - gTopLeft.x,
            height: gBottomRight.y - gTopLeft.y
        };
    };

    this.fadeOut = function(svg, filter) {
        // Fade out nodes that need to
        if (filter && filter.doFading) {
            let d3node = d3.select(svg[0]);
            d3node.selectAll('.node').classed('filter-faded', true);
            d3node.selectAll('.edge').classed('filter-faded', true);

            $.each(filter.nonFadedNodes, function(idx, nodeId) {
                let elt = svg.find(' [data-id="' + nodeId + '"]')[0];
                if (elt == null) {
                    // maybe a saved model
                    let savedmodel_nodeId = 'savedmodel' + nodeId.substring('dataset'.length);
                    elt = svg.find(' [data-id="' + savedmodel_nodeId + '"]')[0];
                    if (elt == null) {
                        // maybe a model evaluation store
                        let modelevaluationstore_nodeId = 'modelevaluationstore' + nodeId.substring('dataset'.length);
                        elt = svg.find(' [data-id="' + modelevaluationstore_nodeId + '"]')[0];
                        if (elt == null) {
                            // or managed folder
                            let managedfolder_nodeId = 'managedfolder' + nodeId.substring('dataset'.length);
                            elt = svg.find(' [data-id="' + managedfolder_nodeId + '"]')[0];
                            if (elt == null) {
                                return;
                            }
                        }
                    }
                }
                d3.select(elt).classed('filter-faded', false);
            });
            $.each(filter.nonFadedEdges, function(idx, toNodeId) {
                svg.find(' [data-to="' + toNodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
                let savedmodel_nodeId = 'savedmodel' + toNodeId.substring('dataset'.length);
                svg.find(' [data-to="' + savedmodel_nodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
                let modelevaluationstore_nodeId = 'modelevaluationstore' + toNodeId.substring('dataset'.length);
                svg.find(' [data-to="' + modelevaluationstore_nodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
                let managedfolder_nodeId = 'managedfolder' + toNodeId.substring('dataset'.length);
                svg.find(' [data-to="' + managedfolder_nodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
            });
        }
    };
});


function polygonToRectData(polygon) { // polygon is an svg element
    const points = $(polygon).attr('points').split(' ');
    // points = [top right, top left, bottom left, bottom right, top right]
    return {
        x: parseFloat(points[1].split(',')[0]),
        y: parseFloat(points[1].split(',')[1]),
        width: parseFloat(points[0].split(',')[0], 10) - parseFloat(points[1].split(',')[0], 10),
        height: parseFloat(points[2].split(',')[1], 10) - parseFloat(points[1].split(',')[1], 10)
    };
}

function circleToRectData(ellipse) {// ellipse is an svg element
    const el = $(ellipse);
    return {
        x: el.attr('cx') - el.attr("rx"),
        y: el.attr('cy') - el.attr("ry"),
        width: el.attr('rx') * 2,
        height: el.attr('ry') * 2
    };
}

function makeForeignObject(attrs, jq) {
    const el = makeSVG('foreignObject', attrs)
    $(el).append(jq);
    return el;
}

})();

;
(function() {
'use strict';

/**
* Main flow page functionalities
*/
const app = angular.module('dataiku.flow.project', ['dataiku.flow.graph']);

const FLOW_NODE_STATUS = {
    RUNNING: "RUNNING",
    NOT_STARTED: "NOT_STARTED",
    DONE: "DONE",
}


app.directive('flowRightColumn', function(QuickView, TaggableObjectsUtils, FlowGraphSelection, FlowGraph) {
    return {
        scope: true,
        link: function(scope, element, attrs) {
            scope.QuickView = QuickView;

            scope.$watch("rightColumnItem", function() {
                scope.context = "FLOW";
                scope.selection = {
                    selectedObject: scope.rightColumnItem,
                    confirmedItem: scope.rightColumnItem
                };
            });

            scope.getSelectedNodes = function() {
                return scope.rightColumnSelection || [];
            };

            scope.getSelectedTaggableObjectRefs = function() {
                return scope.getSelectedNodes().map(TaggableObjectsUtils.fromNode);
            };

            scope.computeMovingImpact = function() {
                const computedImpact = [];
                const movingItems = FlowGraphSelection.getSelectedTaggableObjectRefs();

                    function addSuccessors(node, original) {
                        if (!['RECIPE', 'LABELING_TASK'].includes(node.nodeType)) return;
                        node.successors.forEach(function(successor) {
                            let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(successor));
                            if (original && successor == original.id
                                || movingItems.some(it => it.id === newTaggableObjectRef.id)
                                || computedImpact.some(it => it.id === newTaggableObjectRef.id)) return;
                            computedImpact.push(newTaggableObjectRef);
                        });
                    }
                    function computeImpact(node) {
                        let predecessor = node.predecessors[0];
                        if (predecessor && !['RECIPE', 'LABELING_TASK'].includes(node.nodeType)) {
                            let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(predecessor));
                            if (computedImpact.some(it => it.id === newTaggableObjectRef.id)) return;
                            if (!movingItems.some(it => it.id === newTaggableObjectRef.id)) {
                                computedImpact.push(newTaggableObjectRef);
                            }
                            addSuccessors(FlowGraph.node(predecessor), node);
                        }

                    addSuccessors(node);
                }

                FlowGraphSelection.getSelectedNodes().forEach(function(node) {
                    let realNode = node.usedByZones.length ? FlowGraph.node(`zone__${node.ownerZone}__${node.realId}`) : node;
                    computeImpact(realNode);
                });
                return computedImpact;
            }
        }
    };
});



// WARNING Keep the switch in sync with other _XXX_MassActionsCallbacks controllers (flow, taggable objects pages, list pages)
app.controller('FlowMassActionsCallbacks', function($scope, $rootScope, FlowTool, FlowGraphSelection, ToolBridgeService, PIPELINEABILITY_ACTIONS, SummaryService) {

    $scope.onAction = function(action) {
        let affectedViews = [];
        switch (action) {
            case 'action-clear':
            case 'action-build':
            case 'action-change-connection':
            case 'action-share':
            case 'action-set-virtualizable':
                reloadGraph();
                break;
            case 'action-delete':
            case 'action-unshare':
                reloadGraph();
                FlowGraphSelection.clearSelection();
                break;
            case 'action-tag':
                affectedViews = ['TAGS'];
                break;
            case 'action-watch':
            case 'action-star':
                affectedViews = ['WATCH']
                $rootScope.$emit('userInterestsUpdated');
                break;
            case 'action-update-status':
                affectedViews = ['COUNT_OF_RECORDS', 'FILESIZE'];
                break;
            case 'action-set-auto-count-of-records':
                affectedViews = ['COUNT_OF_RECORDS'];
                break;
            case 'action-add-to-scenario':
                affectedViews = ['SCHEDULING'];
                break;
            case 'action-change-spark-config':
                affectedViews = ['SPARK_CONFIG'];
                break;
            case PIPELINEABILITY_ACTIONS.changeSpark:
                affectedViews = ['SPARK_PIPELINES'];
                break;
            case PIPELINEABILITY_ACTIONS.changeSQL:
                affectedViews = ['SQL_PIPELINES'];
                break;
            case 'action-change-impala-write-mode':
                affectedViews = ['IMPALA_WRITE_MODE'];
                break;
            case 'action-change-hive-engine':
                affectedViews = ['HIVE_MODE'];
                break;
            case 'action-convert-to-hive':
            case 'action-convert-to-impala':
                reloadGraph();
                break;
            default:
                break;
        }
        FlowTool.refreshFlowStateWhenViewIsActive(affectedViews);
    }

    /*
    * Fetch the whole flow + the view state if any active
    */
    function reloadGraph() {
        $rootScope.$emit('reloadGraph');
    }
});


app.directive('flowEditor', function($stateParams, $timeout, $rootScope, $controller, $filter, translate, AI_EXPLANATION_MODAL_MODES, Debounce, GraphZoomTrackerService,
            Assert, TopNav, CreateModalFromTemplate, uiCustomizationService, DataikuAPI, ClipboardUtils, ContextualMenu, HistoryService, Logger, StateUtils, TaggableObjectsUtils, localStorageService,
            FlowGraphSelection, FlowToolsRegistry, FlowToolsUtils, FlowGraph, FlowGraphFiltering, FlowGraphFolding, executeWithInstantDigest, PageSpecificTourService, OpalsService, OpalsMessageService,
            Notification, $q, MessengerUtils, DatasetRenameService, FlowBuildService, AnyLoc, WatchInterestState, WT1, ZoneService, RecipeRenameService, ToolBridgeService, Dialogs) {

function drawExposedIndicators(svg, nodesGraph) {
    svg.find('.exposed-indicator').remove();

    svg.find('g[data-type=LOCAL_DATASET], g[data-type=LOCAL_SAVEDMODEL], g[data-type=LOCAL_MODELEVALUATIONSTORE], g[data-type=LOCAL_GENAIEVALUATIONSTORE], g[data-type=LOCAL_MANAGED_FOLDER], g[data-type=LOCAL_RETRIEVABLE_KNOWLEDGE]').each(function (index, boxElement) {
        const nodeId = $(boxElement).attr('data-id');
        const node = nodesGraph.nodes[nodeId];
        if (!node) {
            Logger.warn("Graph node not found:", nodeId);
            return;
        }
        if (node.isExposed) {
            const type = {
                LOCAL_DATASET: 'dataset',
                LOCAL_SAVEDMODEL: 'model',
                LOCAL_MODELEVALUATIONSTORE: 'model evaluation store',
                LOCAL_GENAIEVALUATIONSTORE: 'genai evaluation store',
                LOCAL_MANAGED_FOLDER: 'folder',
                LOCAL_RETRIEVABLE_KNOWLEDGE: "knowledge bank"
            }[$(boxElement).data('type')];

            const exposedSVG = $(makeSVG('foreignObject', {
                    x: 2,
                    y: 2,
                    width: 20,
                    height: 20,
                    class: 'exposed-indicator nodeicon-small'+(type == 'dataset' || type == 'knowledge bank' ? '' : '-dark')
                }))
                .append($(`<div><i class="icon-mail-forward" title="This ${type} is exposed in other projects"></i></div>`));
            if (type == 'folder') {
                $(boxElement).find('>g').first().append(exposedSVG);
            } else {
                $(boxElement).find('>g').append(exposedSVG);
            }
        }
    });
}

function drawForbiddenIndicators(svg, nodesGraph) {
    svg.find('.forbidden-indicator').remove();

    svg.find('g[data-type=FOREIGN_DATASET], g[data-type=FOREIGN_SAVEDMODEL], g[data-type=FOREIGN_RETRIEVABLE_KNOWLEDGE], g[data-type=FOREIGN_MODELEVALUATIONSTORE], g[data-type=FOREIGN_GENAIEVALUATIONSTORE], g[data-type=FOREIGN_MANAGED_FOLDER]').each(function (index, boxElement) {
        const nodeId = $(boxElement).attr('data-id');
        const node = nodesGraph.nodes[nodeId];
        if (!node) {
            Logger.warn("Graph node not found:", nodeId);
            return;
        }
        if (node.isForbiddenObject) {
            const type = {
                FOREIGN_DATASET: 'dataset',
                FOREIGN_SAVEDMODEL: 'model',
                FOREIGN_MODELEVALUATIONSTORE: 'model evaluation store',
                FOREIGN_GENAIEVALUATIONSTORE: 'genai evaluation store',
                FOREIGN_MANAGED_FOLDER: 'folder',
                FOREIGN_RETRIEVABLE_KNOWLEDGE: 'knowledge bank'
            }[$(boxElement).data('type')];

            const svgY = {
                FOREIGN_DATASET: 50,
                FOREIGN_SAVEDMODEL: 50,
                FOREIGN_MODELEVALUATIONSTORE: 50,
                FOREIGN_GENAIEVALUATIONSTORE: 50,
                FOREIGN_RETRIEVABLE_KNOWLEDGE: 50,
                FOREIGN_MANAGED_FOLDER: 35
            }[$(boxElement).data('type')];

            const exposedSVG = $(makeSVG('foreignObject', {
                    x: 2,
                    y: svgY,
                    width: 20,
                    height: 20,
                    class: 'exposed-indicator nodeicon-small'+(type == 'dataset' || type == 'knowledge bank' ? '' : '-dark')
                }))
                .append($(`<div><i class="icon-warning-sign" title="You don't have the rights to access this object"></i></div>`));
            if (type == 'folder') {
                $(boxElement).find('>g').first().append(exposedSVG);
            } else {
                $(boxElement).find('>g').append(exposedSVG);
            }
        }
    });
}

function drawBuildInProgressIndicators(svg, nodesGraph) {
    svg.find('.build-indicator').remove();

    svg.find('g[data-type=LOCAL_DATASET], g[data-type=LOCAL_SAVEDMODEL], g[data-type=LOCAL_MODELEVALUATIONSTORE], g[data-type=LOCAL_GENAIEVALUATIONSTORE], g[data-type=LOCAL_MANAGED_FOLDER], g[data-type=LOCAL_RETRIEVABLE_KNOWLEDGE]').each(function (index, boxElement) {
        let nodeId = $(boxElement).attr('data-id');
        let node = nodesGraph.nodes[nodeId];

        if (!node) {
            Logger.warn("Graph node not found:", nodeId)
            return;
        }

        let iconDom = null;
        if (node.beingBuilt) {
            iconDom = $('<div class="icon-being-built"><i class="icon-play" /></div>');
        } else if (node.aboutToBeBuilt) {
            iconDom = $('<div class="icon-about-to-be-built"><i class="icon-spinner"></i></div>');
        }
        if (iconDom) {
            let $pinSvg = $(makeSVG('foreignObject', {
                    x: 75,
                    y: 55,
                    width: 20,
                    height: 20,
                    'class': 'build-indicator'
            })).append(iconDom);

            if ($(boxElement).data('type') == 'LOCAL_MANAGED_FOLDER') {
                $(boxElement).find('>g').first().append($pinSvg);
            } else {
                $(boxElement).find('>g').append($pinSvg);
            }
        }
    });

    svg.find('g[data-type=RECIPE]').each(function (index, boxElement) {
        let nodeId = $(boxElement).attr('data-id');
        let node = nodesGraph.nodes[nodeId];

        if (!node) {
            Logger.warn("Graph node not found:", nodeId)
            return;
        }

        let iconDom = null;
        if (node.continuousActivityDone) {
            iconDom = $('<div class="icon-continuous-activity-done"><i class="icon-warning-sign" /></div>');
        } else if (node.beingBuilt) {
            iconDom = $('<div class="icon-being-built"><i class="icon-play" /></div>');
        }
        if (iconDom) {
            let $pinSvg = $(makeSVG('foreignObject', {
                    x: 55,
                    y: 40,
                    width: 20,
                    height: 20,
                    'class': 'build-indicator',
                    transform: 'scale(1.92 1.92)'  // scale to conteract the 0.52 iconScale
            })).append(iconDom);

            $(boxElement).find('>g').append($pinSvg);
        }
    });
}

return {
    restrict: 'EA',
    scope: true,
    controller: function($scope, $rootScope, SummaryService, PageSpecificTourService, FlowTool) {
        $controller('FlowMassActionsCallbacks', {$scope: $scope});

        TopNav.setLocation(TopNav.TOP_FLOW, TopNav.LEFT_FLOW, TopNav.TABS_NONE, null);
        TopNav.setNoItem();

        uiCustomizationService.getComputeDatasetTypesStatus($scope, $stateParams.projectKey).then(
            (computeStatus) => {
                $scope.canCreateUploadedFilesDataset = computeStatus("UploadedFiles") === uiCustomizationService.datasetTypeStatus.SHOW;
            });

        $scope.projectFlow = true;
        $scope.nodesGraph = {flowFiltersAndSettings : {}};

        $scope.getZoneColor = zoneId => {
            const nodeFound = $scope.nodesGraph.nodes ? $scope.nodesGraph.nodes[`zone_${zoneId}`] : undefined;
            if (nodeFound && nodeFound.customData) {
                return nodeFound.customData.color
            }
            return "#ffffff";
        };

        function updateUserInterests() {
            DataikuAPI.interests.getUserInterests($rootScope.appConfig.login, 0, 10000, {projectKey: $stateParams.projectKey}).success(function(data) {
                // It would be nice to fetch that with the graph but it is a little dangerous to require the database to be functional to see any flow...
                $scope.userInterests = data.interests.filter(x => ['RECIPE', 'DATASET', 'SAVED_MODEL', 'MODEL_EVALUATION_STORE', 'MANAGED_FOLDER', "STREAMING_ENDPOINT", 'LABELING_TASK', 'RETRIEVABLE_KNOWLEDGE'].includes(x.objectType));

                const indexedInterests = {};
                $scope.userInterests.forEach(function(interest) {
                    //TODO @flow using the node ids as keys would be better but we don't generate them in js for now (I think)
                    indexedInterests[interest.objectType+'___'+interest.objectId] = interest;
                });
                $.each($scope.nodesGraph.nodes, function(nodeId, node) {
                    const taggableType = TaggableObjectsUtils.fromNodeType(node.nodeType);
                    const interest = indexedInterests[taggableType+'___'+node.name]
                    if (interest) {
                        node.interest = interest;
                    } else {
                        node.interest = {
                            starred: false,
                            watching: WatchInterestState.values.ENO
                        };
                    }
                });

            }).error(setErrorInScope.bind($scope));
        }
        $scope.setErrorInScopeCallbackForAngular = (error) => setErrorDetailsInScope.bind($scope)(error); // used to put in scope error already handled in Angular
        $scope.isNotInGraph = function(item) {
            return $scope.nodesGraph && $scope.nodesGraph.nodes &&
                ! $scope.nodesGraph.nodes.hasOwnProperty('dataset_' + item.name);
        };

        $scope.processSerializedFilteredGraphResponse = function processSerializedFilteredGraphResponse(serializedFilteredGraph, zoomTo, errorScope) {
            $scope.setGraphData(serializedFilteredGraph.serializedGraph);
            if (typeof zoomTo === 'string') {
                const deregisterListener = $scope.$root.$on("flowDisplayUpdated", function () {
                    deregisterListener();
                    setTimeout(() => {
                        let id = zoomTo;
                        let node = $scope.nodesGraph.nodes[zoomTo];
                        if (!node) {
                            id = graphVizEscape(zoomTo);
                            node = $scope.nodesGraph.nodes[id];
                        }
                        if (!node && $scope.nodesGraph.hasProjectZones) {
                            id = Object.values($scope.nodesGraph.nodes).filter(it => it.realId == id && !it.usedByZones.length)[0].id;
                        }
                        $scope.zoomGraph(id);
                        GraphZoomTrackerService.instantSavePanZoomCtx($scope.panzoom);
                        GraphZoomTrackerService.setFocusItemCtx($scope.nodesGraph.nodes[id]);
                        FlowGraphSelection.onItemClick($scope.nodesGraph.nodes[id]);
                    });
                })
            }

            $rootScope.$emit('drawGraph');
        };

        $scope.processLoadFlowResponse = function processLoadFlowResponse (resp, zoomTo, graphReloaded, resetZoom, errorScope = $scope) {
            Assert.trueish(resp, "Received empty response");
            if (resp.serializedFilteredGraph) {
                $scope.processSerializedFilteredGraphResponse(resp.serializedFilteredGraph, zoomTo, errorScope);
            }
            if (resetZoom) $scope.resetPanZoom();
            $scope.isFlowLoaded = true;
        };

        $scope.updateGraph = function updateGraph(zoomTo, nospinner=false, shouldUpdateUserInterests=true) {
            DataikuAPI.flow.recipes.getGraph($stateParams.projectKey, true, $scope.drawZones.drawZones, $stateParams.zoneId, $scope.collapsedZones, nospinner)
                .success(function(response) {
                        $scope.zonesManualPositioning = response.zonesManualPositioning.projectSettingsValue;
                        $scope.canMoveZones = response.zonesManualPositioning.canMove;

                        $scope.zoneIdLoaded = $stateParams.zoneId;
                        $scope.processLoadFlowResponse(response, zoomTo, true);
                        if (shouldUpdateUserInterests) {
                            updateUserInterests();
                        }
                        $scope.isMoveFlowZonesToggleLoading = false;
                        $scope.isSavingZonePosition = false;
                    }
                )
                .error(setErrorInScope.bind($scope));

            //TODO @flow move to flow_search
            // DataikuAPI.datasets.list($stateParams.projectKey).success(function(data) {
            //     $scope.datasets = data;
            // }).error(setErrorInScope.bind($scope));
            // DataikuAPI.datasets.listHeads($stateParams.projectKey, {}, false).success(function(data) {
            //     $scope.filteredDatasets = data;
            // }).error(setErrorInScope.bind($scope));
        };

        var storageKey = `dku.flow.drawZones.${$stateParams.projectKey}`;
        $scope.drawZones = {
            drawZones: !!$stateParams.zoneId || JSON.parse(localStorageService.get(storageKey) || true)
        }

        var drawZonesSub = ToolBridgeService.drawZonesToggle$.subscribe(() => {
            ToolBridgeService.emitShouldDrawZones(!$scope.drawZones.drawZones);
        });

        // Cleanup when the scope is destroyed
        $scope.$on('$destroy', function() {
            if (drawZonesSub) {
                drawZonesSub.unsubscribe();
            }
        });

        // init call to restore show/hide zone state on page reload (or project navigation)
        ToolBridgeService.emitShouldDrawZones($scope.drawZones.drawZones);

        $scope.redrawZone = function () {
            if (!$scope.inFlowExport) {
                localStorageService.set(storageKey, $scope.drawZones.drawZones);
                $scope.resetPanZoom();
                $scope.updateGraph();
            }
        };

        var collapsedZonesStorageKey = `dku.flow.collapsedZones.${$stateParams.projectKey}`;

        $scope.cleanupCollapsedZones = (collapsedZones = $scope.collapsedZones) => {
            let changed = false;
            [...collapsedZones].forEach(collapsedZone => {
                const zoneFound = FlowGraph.node(`zone_${collapsedZone}`);
                if (!zoneFound) {
                    const index = collapsedZones.indexOf(collapsedZone);
                    if (index !== -1) {
                        collapsedZones.splice(index, 1);
                        changed = true;
                    }
                }
            });
            if (changed) {
                localStorageService.set(collapsedZonesStorageKey, JSON.stringify(collapsedZones));
            }
            return collapsedZones;
        }
        $scope.collapsedZones = localStorageService.get(collapsedZonesStorageKey) || [];

        $scope.toggleZoneCollapse = (collapseItems, multiItemStrategy) => {
            let zoneIds = collapseItems.map(it => it.id);
            zoneIds.forEach(function(zoneId) {
                let index = $scope.collapsedZones.findIndex(it => it === zoneId);
                if (index > -1 && multiItemStrategy !== 'collapseAll') {
                    $scope.collapsedZones.splice(index, 1);
                } else if (index < 0 && multiItemStrategy !== 'expandAll') {
                    $scope.collapsedZones.push(zoneId);
                }
            });
            localStorageService.set(collapsedZonesStorageKey, JSON.stringify($scope.collapsedZones));
            $scope.updateGraph();
        }

        $scope.updateGraph($stateParams.id);

        $scope.nodeSelectorTooltip = function(type, count) {
            const params = { type: $filter('niceTaggableType')(type, count) };
            if (count > 1) {
                return translate('PROJECT.FLOW.GRAPH.SELECT_ALL_OBJECTS', 'Select all {{type}}', params);
            } else {
                return translate('PROJECT.FLOW.GRAPH.SELECT_AN_OBJECT', 'Select a {{type}}', params);
            }
        };

        $scope.copyNameToClipboard = function(name) {
            ClipboardUtils.copyToClipboard(name);
        }

        $scope.renameDataset = function(datasetNode) {
            DatasetRenameService.startRenaming($scope, datasetNode.projectKey, datasetNode.name, datasetNode.datasetType)
        }

        $scope.renameRecipe = function(recipeNode) {
            RecipeRenameService.startRenamingRecipe($scope, recipeNode.projectKey, recipeNode.name);
        }

        const buildModalParams = (hasPredecessors, hasSuccessors) => {
            // hasPredecessors/hasSuccessors is not *exactly* equal to upstreamBuildable / downstreamBuildable, but is a good approximation
            return {
                upstreamBuildable: hasPredecessors,
                downstreamBuildable: hasSuccessors,
            };
        }

        $scope.buildDataset = function(projectKey, name, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "DATASET", AnyLoc.makeLoc(projectKey, name), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.trainModel = function(projectKey, id,  hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "SAVED_MODEL", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.buildManagedFolder = function(projectKey, id, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "MANAGED_FOLDER", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.buildModelEvaluationStore = function(projectKey, id, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "MODEL_EVALUATION_STORE", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.buildRetrievableKnowledge = function(projectKey, id, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "RETRIEVABLE_KNOWLEDGE", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };

        $scope.startCopy = function() {
            $scope.startTool('COPY', {preselectedNodes: FlowGraphSelection.getSelectedNodes().map(n => n.id)});
        };

        const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);

        const TARGET_TYPES_TO_UPDATE = {
            DATASET: "dataset",
            RETRIEVABLE_KNOWLEDGE: "retrievableknowledge"
        }

        function getNodeNameForItem(itemType, itemName) {
            return itemType + "__" + graphVizEscape(itemName);
        }

        function updateNodeStatus(node, state) {
            if (!node) return false;

            switch (state) {
                case FLOW_NODE_STATUS.RUNNING:
                    node.beingBuilt = true;
                    node.aboutToBeBuilt = false;
                    break;
                case FLOW_NODE_STATUS.DONE:
                    node.beingBuilt = false;
                    node.aboutToBeBuilt = false;
                    break;
                case FLOW_NODE_STATUS.NOT_STARTED:
                    node.aboutToBeBuilt = true;
                    node.beingBuilt = false;
                    break;
            }
            return true;
        }

        function updateNodeStatusFlowWithZones(message, state) {
            // selector matching job output datasets realId (can be in multiple zones)
            let refresh = false;

            if (!message.status) return refresh;

            const selector = message.status.targets.filter(target => target.type in TARGET_TYPES_TO_UPDATE).map(target => {
                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                return `svg [data-node-id="${realId}"]`;
            }).join(', ');
            // update node status for each matching output dataset
            if (selector) {
                d3.selectAll(selector).each(function() {
                    const id = this.getAttribute('data-id');
                    if (updateNodeStatus(FlowGraph.node(id), state)) {
                        refresh = true;
                    }
                });
            }
            return refresh;
        }

        function updateNodeStatusFlowWithoutZones(message, state) {
            let refresh = false;
            if (!message.status) return refresh;


            Object.values(message.status.targets).filter(target => target.type in TARGET_TYPES_TO_UPDATE).forEach(target => {
                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                if (updateNodeStatus($scope.nodesGraph.nodes[realId], state)) {
                    refresh = true;
                }
            });
            return refresh;
        }

        function updateAndDrawActivityNodeStatus(message, state) {
            let refreshGraph;

            if ($scope.nodesGraph.hasProjectZones && $scope.drawZones.drawZones) {
                refreshGraph =  updateNodeStatusFlowWithZones(message, state);
            } else {
                refreshGraph = updateNodeStatusFlowWithoutZones(message, state);
            }

            if (refreshGraph) {
                drawBuildInProgressIndicators(FlowGraph.getSvg(), $scope.nodesGraph);
            }
        }

        const jobActivityStartedListener = Notification.registerEvent("job-activity-started", function(evt, message) {
            updateAndDrawActivityNodeStatus(message, FLOW_NODE_STATUS.RUNNING);
        });

        const jobActivityDoneListener = Notification.registerEvent("job-activity-done", function(evt, message) {
            updateAndDrawActivityNodeStatus(message, FLOW_NODE_STATUS.DONE);
        });

        const jobStatusUpdatedListener = Notification.registerEvent("job-status-updated-light", function(evt, message) {
            let refreshGraph = false;

            function updateAllNodeStatusFlowWithZones() {
                let targets = [];
                Object.values(message.status.activities).forEach(activity => {
                    if (activity.state === "NOT_STARTED" && !activity.skipExplicitOrWriteProtected) {
                        activity.targets.forEach(target => {
                            if (target.type in TARGET_TYPES_TO_UPDATE) {
                                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                                targets.push(`svg [data-node-id="${realId}"]`)
                            }
                        })
                    }
                })

                const selector = targets.join(', ');

                // update node status for each matching output dataset
                if (selector) {
                    d3.selectAll(selector).each(function() {
                        const id = this.getAttribute('data-id');
                        if (updateNodeStatus(FlowGraph.node(id), FLOW_NODE_STATUS.NOT_STARTED)) {
                            refreshGraph = true;
                        }
                    });
                }
            }

            function updateAllNodeStatusFlowWithoutZones() {
                if (!message.status || !message.status.activities) return;
                Object.values(message.status.activities).forEach(activity => {
                    if (activity.state === "NOT_STARTED" && !activity.skipExplicitOrWriteProtected) {
                        activity.targets.forEach(target => {
                            if (target.type in TARGET_TYPES_TO_UPDATE) {
                                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                                if ($scope.nodesGraph.nodes && updateNodeStatus($scope.nodesGraph.nodes[realId], FLOW_NODE_STATUS.NOT_STARTED)) {
                                    refreshGraph = true;
                                }
                            }
                        });
                    }
                });
            }

            if ($scope.nodesGraph.hasProjectZones && $scope.drawZones.drawZones) {
                updateAllNodeStatusFlowWithZones()
            } else {
                updateAllNodeStatusFlowWithoutZones();
            }

            if (refreshGraph) {
                drawBuildInProgressIndicators(FlowGraph.getSvg(), $scope.nodesGraph);
            }
        });

        let continuousActivityStateChangeListener = Notification.registerEvent("continuous-activity-state-change", function(evt, message) {

            let refreshGraph = false;


            function isRunning(message) {
                if (message.state=='STOPPED') return false;
                if (message.state=='STARTED') return true;
                return undefined;
            }

            function updateNodeStatus(isRunIcon, node) {
                if (!node) return;

                if (isRunIcon && !node.beingBuilt) {
                    refreshGraph = true;
                    node.beingBuilt = true;
                    node.continuousActivityDone = false;
                    node.aboutToBeBuilt = false;
                }
                else if (node.beingBuilt || node.aboutToBeBuilt) {
                    refreshGraph = true;
                    node.beingBuilt = false;
                    node.continuousActivityDone = false;
                    node.aboutToBeBuilt = false;
                }

            }

            let isRun = isRunning(message);
            if (isRun!==undefined) {
                let nodeName = getNodeNameForItem('recipe', message.continuousActivityId);
                updateNodeStatus(isRun, $scope.nodesGraph.nodes[nodeName]);
            }

            if (refreshGraph) {
                drawBuildInProgressIndicators(FlowGraph.getSvg(), $scope.nodesGraph);
            }
        });

        $scope.$on("$destroy", function() {
            if ($scope.svg) {
                $scope.svg.empty();
                $scope.svg.remove();
                $scope.svg = null;
                jobActivityStartedListener();
                jobActivityDoneListener();
                jobStatusUpdatedListener();
            }
            interestsListener();
            continuousActivityStateChangeListener();
        });

        $scope.unfoldAll = function() {
            FlowGraphFolding.unfoldAll();
        }

        $scope.zoomOnSelection = function(paddingFactor) {
            const selectedNodes = $('.selected', $scope.svg)
            // Only calculate padding factor if not provided
            if (paddingFactor == null) {
                paddingFactor = 1.2
                if (selectedNodes && selectedNodes.length === 1){
                    if (selectedNodes[0].classList.contains('zone_cluster')) {
                        //only one zone, padding is 1.5
                        paddingFactor = 1.5;
                    } else {
                        //only one element, show some context
                        paddingFactor = 3
                    }
                }
            }
            $scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector($scope.svg, '.selected'), paddingFactor);
        };

        $scope.exportFlow = function() {
            const graphBBox = $scope.svg.find('g.graph')[0].getBBox();
            CreateModalFromTemplate("/templates/flow-editor/export-flow-modal.html", $scope, "ExportFlowModalController", function(newScope) {
                newScope.init($stateParams.projectKey, graphBBox);
            });
        };

        $scope.generateFlowDocument = function() {
            CreateModalFromTemplate("templates/flow-editor/generate-flow-document-modal.html", $scope, "GenerateFlowDocumentModalController", function(newScope) {
                newScope.init($stateParams.projectKey);
            }, false, 'static');
        };

        // Toolbox used by export-flow.js to prepare the flow to be exported
        $scope.exportToolbox = {
            checkLoading: function() {
                return $scope.httpRequests.length !== 0 || !$scope.isFlowLoaded || ToolBridgeService.isQueryLoading() || ToolBridgeService.isViewLoading();
            },
            removeDecorations: function(drawZones) {
                executeWithInstantDigest(function() {
                    $scope.hideForExport = true;
                    $scope.fullScreen = true;
                    $scope.inFlowExport = true; // Prevent the flow from automatically refreshing when zones are shown/hidden
                }, $scope);
            },
            getGraphBoundaries: function () {
                const graphBBox = $scope.svg.find('g.graph')[0].getBBox();
                return {
                    x: graphBBox.x,
                    y: graphBBox.y,
                    width: graphBBox.width,
                    height: graphBBox.height
                };
            },
            adjustViewBox: function(x, y, width, height) {
                $scope.svg[0].setAttribute('viewBox', [x, y, width, height].join(', '));
            },
            configureZones: function(drawZones, collpasedZones) {
                $scope.drawZones.drawZones = drawZones;
                $scope.collapsedZones = collpasedZones;
                // Reload the flow graph
                $scope.isFlowLoaded = false;
                $scope.updateGraph();
            }
        };

        $scope.explainProject = function() {
            CreateModalFromTemplate(
                "/static/dataiku/ai-explanations/explanation-modal/explanation-modal.html",
                $scope,
                "AIExplanationModalController",
                function(newScope) {
                    newScope.objectType = "PROJECT";
                    newScope.object = $scope.projectSummary;
                    newScope.mode = AI_EXPLANATION_MODAL_MODES.EXPLAIN;
                }
            );
        };

        $scope.zoomOnZone = ZoneService.zoomOnZone;

        $scope.zoomOutOfZone = ZoneService.zoomOutOfZone;

        const hasPreviewKey = `dku.flow.hasPreview`;
        $scope.hasPreview = JSON.parse(localStorageService.get(hasPreviewKey)) || false; // init
        $scope.togglePreview = () => {
            $scope.hasPreview = !$scope.hasPreview;
            localStorageService.set(hasPreviewKey, $scope.hasPreview)
        }

        $scope.isMoveFlowZonesToggleLoading = false;
        $scope.toggleZonesManualPositioning = () => {
            const projectSummary = $rootScope.projectSummary;
            $scope.isMoveFlowZonesToggleLoading = true;
            DataikuAPI.flow.zones.setManualPositioningSetting(projectSummary.projectKey, !projectSummary.zonesManualPositioning)
                .then(() => {
                    projectSummary.zonesManualPositioning = !projectSummary.zonesManualPositioning;
                    WT1.tryEvent("flow-toggle-zones-manual-positioning", () => ({
                        enabled: projectSummary.zonesManualPositioning,
                    }));
                    $scope.$emit('zonesManualPositioningChanged');
                })
                .catch(function(data, status, headers) {
                    $scope.isMoveFlowZonesToggleLoading = false;
                    setErrorInScope.bind($scope)(data, status, headers);
                });
        }

        $scope.autoArrangeFlowZones = () => {
            Dialogs.confirmAlert(
                $scope,
                translate(
                    "PROJECT.FLOW.AUTO_ARRANGE_FLOW_ZONES.TITLE",
                    "Auto-arrange flow zones"
                ),
                translate(
                    "PROJECT.FLOW.AUTO_ARRANGE_FLOW_ZONES.TEXT",
                    "Are you sure you want to continue?",
                ),
                translate(
                    "PROJECT.FLOW.AUTO_ARRANGE_FLOW_ZONES.ALERT",
                    "This will reset all flow zones positions for this project",
                )
                ,
                "WARNING"
            ).then(
                function () {
                    // Confirm
                    $scope.isSavingZonePosition = true;
                    WT1.tryEvent("flow-auto-arrange-zones", () => {});
                    const projectSummary = $rootScope.projectSummary;
                    DataikuAPI.flow.zones.resetPositions(projectSummary.projectKey)
                        .then(() => $scope.$emit('zonesManualPositioningChanged'))
                        .catch(function(data, status, headers) {
                            $scope.isSavingZonePosition = false;
                            setErrorInScope.bind($scope)(data, status, headers);
                        });
                },
                function () {
                    // Cancel: do nothing
                }
            );
        }

        const checkIsFlowActionTool = function () {
            const tool = FlowTool.getCurrent();
            return tool && tool.action;
        };

        $scope.onFlowColor = flowColoring => {
            $scope.flowColoring = flowColoring; // save coloring for later
            if (checkIsFlowActionTool()) {
                ToolBridgeService.queryLoaded();
                return;
            }
            if (flowColoring === undefined) {
                resetDefaultGraphColor();
                ToolBridgeService.queryLoaded();
                return;
            }
            // however sometimes we just want to apply an highlight on a flow that did not get updated
            // yes it can be called two times in a row
            colorFlowNodes(flowColoring.coloring, flowColoring.matchedObjectsId);
            ToolBridgeService.queryLoaded();
        };
        $scope.onSuggestionFocus = flowItemColoring => {
            if (checkIsFlowActionTool()) {
                return;
            }
            if (flowItemColoring === undefined) {
                if ($scope.flowColoring === undefined) {
                    return;
                }
                colorFlowNodes($scope.flowColoring.coloring, $scope.flowColoring.matchedObjectsId);
                return;
            }
            colorFlowNodes(flowItemColoring.coloring, flowItemColoring.matchedObjectsId);
        };

        $scope.onObjectSelect = function(objects) {
            if (objects.length === 0) {
                FlowGraphSelection.clearSelection();
                return;
            }

            FlowGraphSelection.clearSelection();
            FlowGraphSelection.selectMulti(objects);
            if (objects.length === 1) {
                // sc-196337 call zoomGraph to behave like legacy for one object.
                // 65px of offCenterShift to put zoom bottom screen
                const objectsByType = $scope.nodesGraph.includedObjectsByType;
                const hasOnlyFlowZone =
                  objectsByType && Object.keys(objectsByType).length === 1 && objectsByType.hasOwnProperty("FLOW_ZONE");
                if (hasOnlyFlowZone) {
                    $scope.zoomGraph(objects[0].id,3,null, 300);
                } else {
                    $scope.zoomGraph(objects[0].id, 3, null, -65);
                }

            } else {
                $scope.zoomOnSelection();
            }
        };

        let flowViewsSubscriptions = [];
        flowViewsSubscriptions.push(
            SummaryService.selectedObjects$.subscribe(obj => {
                if (obj && obj.length === 0) {
                    FlowGraphSelection.clearSelection();
                    return;
                }
                FlowGraphSelection.clearSelection();
                FlowGraphSelection.selectMulti(obj);
            })
        );

        flowViewsSubscriptions.push(
            SummaryService.zoomOnSelection$.subscribe(paddingFactor => {
                $scope.zoomOnSelection(paddingFactor);
            })
        );

        // when changing some configuration on the flow, for example adding a tag to an element
        // highlighting the nodes will happen before the flow is being rendered
        // so we add a listener to react after the flow display has been updated
        const flowDisplayUpdatedUnsubscribe = $rootScope.$on('flowDisplayUpdated', function(_, shouldResetDefault) {
            if (checkIsFlowActionTool()){
                return;
            }
            if ($scope.flowColoring) {
                colorFlowNodes($scope.flowColoring.coloring, $scope.flowColoring.matchedObjectsId);
            } else if (shouldResetDefault) {
                resetDefaultGraphColor();
            }
        });

        // Cleanup when the scope is destroyed
        $scope.$on('$destroy', function(){
            if (flowViewsSubscriptions) {
                flowViewsSubscriptions.forEach(subs => subs.unsubscribe());
            }
            flowDisplayUpdatedUnsubscribe();
        });

        /**
         * Highlights nodes in a flow based on provided criteria.
         *
         * @param {?Map<string, string | undefined>} coloring - A map where each key corresponds to a node ID and its value corresponds to the color of that node. If the value is `undefined`, the node will not be colored (or greyed out).
         * @param {?Set<string>} matchedObjectsId - A set of string identifiers for nodes that have matched certain criteria and are to be highlighted.

         */
        function colorFlowNodes(coloring, matchedObjectsId) {
            const graphSvg = FlowGraph.getSvg();
            if (!graphSvg) return;
            // Initialize cache if needed
            for (const nodeId of Object.keys($scope.nodesGraph.nodes)) {
                const d3NodeElement = getD3Node(nodeId);
                if (d3NodeElement == null) {
                    continue;
                }
                resetNodeStyle(nodeId, d3NodeElement);
                // Mute all nodes
                if (!matchedObjectsId.has(nodeId)) {
                    d3NodeElement.classed('filter-remove', true);
                }
                // If node is not present in the mapping, it should be colored using its default color
                if (coloring) {
                    let color = coloring.get(nodeId);
                    if (color != null) {
                        colorNode($scope.nodesGraph.nodes[nodeId], d3NodeElement, color);
                        d3NodeElement.classed('focus', true);
                    } else {
                        d3NodeElement.classed('filter-remove', true);
                    }
                }
            }
        }

        function resetNodeStyle(nodeId, d3node) {
            colorNode($scope.nodesGraph.nodes[nodeId], d3node, undefined);
            d3node.classed('focus', false).classed('filter-remove', false);
        }

        /**
         * Color all nodes in the graph with their default co
         */
        function resetDefaultGraphColor() {
            for (const nodeId of Object.keys($scope.nodesGraph.nodes)) {
                const d3NodeElement = getD3Node(nodeId);
                if (d3NodeElement == null) {
                    continue;
                }
                resetNodeStyle(nodeId, d3NodeElement);
            }
        }

        /**
         * Color node by its ID with a given color.
         * @param node the node to be colored
         * @param {D3Node} d3NodeElement D3 node element (a single node in the flow graph, not the whole graph)
         * @param {?string} color color of node. `''` and `undefined` means default node color inferred by its type
         */
        function colorNode(node, d3NodeElement, color) {
            if (node.nodeType === 'ZONE') return;
            FlowToolsUtils.colorNode(node, d3NodeElement, color);
        }

        /**
         * Get the D3 Node for a given node ID
         * @param {string} nodeId
         * @returns {?D3Node}
         */
        function getD3Node(nodeId) {
            const node = FlowGraph.node(nodeId);
            const d3Node = FlowGraph.d3NodeWithIdFromType(
                nodeId,
                node.nodeType
            );
            return d3Node;
        }

        const unregisterFlowTourListener = $rootScope.$on('startFlowTour', function() {
            PageSpecificTourService.startFlowTour({ scope: $scope, fromContext: 'opals' });
        });
        $scope.$on("$destroy", function() {
            unregisterFlowTourListener();
        });

    },
    link: function(scope, element) {
        // Try to find the more recent item for the project and zone
        function getLastItemInHistory(projectKey, zoneId) {
            const items = HistoryService.getRecentlyViewedItems();
            const validItems = items.filter(it => it.type !== 'PROJECT' && it.projectKey === projectKey);
            if (items && items.length) {
                if (!zoneId) {
                    return validItems[0];
                }
                const zoneName = graphVizEscape(`zone_${zoneId}`);
                const zoneContent = Object.keys(scope.nodesGraph.nodes).filter(it => it.startsWith(zoneName)).map(it => scope.nodesGraph.nodes[it]);
                return validItems.find(item => zoneContent.find(it => it.name === item.id));
            }
            return null;
        }

        function getName(item) {
            if (item.type === 'RECIPE' || item.type === 'LABELING_TASK') {
                return item.type.toLowerCase() + graphVizEscape(`_${item.id}`)
            }
            return item.type.toLowerCase().replace('_', '') + graphVizEscape(`_${item.projectKey}.${item.id}`);
        }

        function zoomOnLast() {
            if (!scope.nodesGraph || !scope.nodesGraph.nodes) {
                return; // not ready
            }
            const itemFound = getLastItemInHistory($stateParams.projectKey, $stateParams.zoneId);
            if (itemFound) {
                const id = GraphZoomTrackerService.getZoomedName(FlowGraph, getName(itemFound));
                Logger.info("zooming on " + id + "--> ", scope.nodesGraph.nodes[id]);
                scope.zoomGraph(id);
                FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[id]);
                scope.$apply();
            }
        }

        const lastUsedZoneKey = `dku.flow.lastUsedZone.${$stateParams.projectKey}`;

        scope.moveToFlowZone = (movingItems, forceCreation = false, computedImpact = []) => {
            scope.movingItems = movingItems;
            scope.computedImpact = computedImpact;

            CreateModalFromTemplate("/templates/flow-editor/move-to-zone.html", scope, null, newScope => {
                newScope.uiState = {
                    creationMode: forceCreation ? 'CREATE' : 'SELECT',
                    forceCreation,
                };
                newScope.onClick = () => {
                    let movingTo = newScope.uiState.selectedZone;
                    let promise = null
                    movingItems = movingItems.concat(scope.computedImpact);
                    if (newScope.uiState.creationMode === 'CREATE') {
                        promise = DataikuAPI.flow.zones.create($stateParams.projectKey, newScope.uiState.name, newScope.uiState.color).success(zoneCreated => {
                            movingTo = zoneCreated.id;
                            $rootScope.$emit('zonesListChanged');
                        }).error($q.reject);
                    } else {
                        promise = $q.resolve();
                    }

                    if (movingItems.length > 0) {
                        promise = promise.then(() => DataikuAPI.flow.zones.moveItems($stateParams.projectKey, movingTo, movingItems).error($q.reject));
                    }
                    promise.then(() => {
                        localStorageService.set(lastUsedZoneKey, movingTo);
                        GraphZoomTrackerService.setFocusItemCtx({id: `zone_${movingTo}`}, true);
                        newScope.$emit('reloadGraph');
                        newScope.dismiss();
                    }, setErrorInScope.bind(newScope))
                }
            });
        };

        scope.shareToFlowZone = (sharingItems, forceCreation = false) => {
            CreateModalFromTemplate("/templates/flow-editor/share-to-zone.html", scope, null, newScope => {
                newScope.uiState = {
                    creationMode: forceCreation ? 'CREATE' : 'SELECT',
                    forceCreation,
                };
                newScope.onClick = () => {
                    let sharedTo = newScope.uiState.selectedZone;
                    let promise = null
                    if (newScope.uiState.creationMode === 'CREATE') {
                        promise = DataikuAPI.flow.zones.create($stateParams.projectKey, newScope.uiState.name, newScope.uiState.color).success(zoneCreated => {
                            sharedTo = zoneCreated.id;
                            $rootScope.$emit('zonesListChanged');
                        }).error($q.reject);
                    } else {
                        promise = $q.resolve();
                    }

                    if (sharingItems.length > 0) {
                        promise = promise.then(() => DataikuAPI.flow.zones.shareItems($stateParams.projectKey, sharedTo, sharingItems).error($q.reject));
                    }

                    promise.then(() => {
                        localStorageService.set(lastUsedZoneKey, sharedTo);
                        newScope.$emit('reloadGraph');
                        newScope.dismiss()
                    }, setErrorInScope.bind(newScope));
                }
            });
        };

        scope.unshareToFlowZone = (sharingItems, zoneIds) => {
            if (sharingItems.length > 0) {
                DataikuAPI.flow.zones.unshareItems($stateParams.projectKey, zoneIds, sharingItems).success(scope.$emit('reloadGraph'));
            }
        };

        scope.onItemDblClick = function(item, evt) {
            let destUrl = StateUtils.href.node(item);
            fakeClickOnLink(destUrl, evt);
        };

        scope.onContextualMenu = function(item, evt) {
            let $itemEl = $(evt.target).parents("g[data-type]").first();
            if ($itemEl.length > 0) {
                let x = evt.pageX;
                let y = evt.pageY;
                let ctxMenuScope = scope.$new();
                const selectedNodes = FlowGraphSelection.getSelectedNodes();
                let type = selectedNodes.length > 1 ? 'MULTI' : item.nodeType;

                let controller = {
                    "LOCAL_DATASET": "DatasetContextualMenuController",
                    "FOREIGN_DATASET": "ForeignDatasetContextualMenuController",
                    "LOCAL_STREAMING_ENDPOINT": "StreamingEndpointContextualMenuController",
                    "RECIPE": "RecipeContextualMenuController",
                    "LABELING_TASK": "LabelingTaskContextualMenuController",
                    "LOCAL_SAVEDMODEL": "SavedModelContextualMenuController",
                    "FOREIGN_SAVEDMODEL": "SavedModelContextualMenuController",
                    "LOCAL_MODELEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "FOREIGN_MODELEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "LOCAL_GENAIEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "FOREIGN_GENAIEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "LOCAL_MANAGED_FOLDER": "ManagedFolderContextualMenuController",
                    "FOREIGN_MANAGED_FOLDER": "ManagedFolderContextualMenuController",
                    "LOCAL_RETRIEVABLE_KNOWLEDGE": "KnowledgeBankContextualMenuController",
                    "FOREIGN_RETRIEVABLE_KNOWLEDGE": "KnowledgeBankContextualMenuController",
                    "ZONE": "ZoneContextualMenuController",
                    "MULTI": "MultiContextualMenuController",
                }[type];

                let template = "/templates/flow-editor/" + {
                    "LOCAL_DATASET": "dataset-contextual-menu.html",
                    "FOREIGN_DATASET": "foreign-dataset-contextual-menu.html",
                    "LOCAL_STREAMING_ENDPOINT": "streaming-endpoint-contextual-menu.html",
                    "RECIPE": "recipe-contextual-menu.html",
                    "LABELING_TASK": "labeling-task-contextual-menu.html",
                    "LOCAL_SAVEDMODEL": "savedmodel-contextual-menu.html",
                    "FOREIGN_SAVEDMODEL": "savedmodel-contextual-menu.html",
                    "LOCAL_MODELEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "FOREIGN_MODELEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "LOCAL_GENAIEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "FOREIGN_GENAIEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "LOCAL_MANAGED_FOLDER": "managed-folder-contextual-menu.html",
                    "FOREIGN_MANAGED_FOLDER": "managed-folder-contextual-menu.html",
                    "LOCAL_RETRIEVABLE_KNOWLEDGE": "knowledge-bank-contextual-menu.html",
                    "FOREIGN_RETRIEVABLE_KNOWLEDGE": "knowledge-bank-contextual-menu.html",
                    "ZONE": "zone-contextual-menu.html",
                    "MULTI": "multi-contextual-menu.html",
                }[type];

                ctxMenuScope.object = item;
                ctxMenuScope.hasZone = [...selectedNodes, item].find(it => it.nodeType === "ZONE") !== undefined;

                let menu = new ContextualMenu({
                    template: template,
                    scope: ctxMenuScope,
                    contextual: true,
                    controller: controller,
                });
                menu.openAtXY(x, y);
                return false;
            } else {
                ContextualMenu.prototype.closeAny();
                return true;
            }
        };

        FlowGraphSelection.clearSelection();

        scope.flowViews = FlowToolsRegistry.getFlowViews();
        const togglePreviewShortcut = 'shift+p';
        Mousetrap.bind("z", zoomOnLast);

        Mousetrap.bind("left", scope.moveLeft);
        Mousetrap.bind("right", scope.moveRight);
        Mousetrap.bind("up", scope.moveUp);
        Mousetrap.bind("down", scope.moveDown);

        Mousetrap.bind("-", scope.zoomOut);
        Mousetrap.bind("+", scope.zoomIn);
        Mousetrap.bind("=", scope.zoomIn); // For more practicity on qwerty keyboard without numpad
        // Wrap handler in $timeout to trigger a $digest cycle that triggers the watcher on hasPreview
        // Otherwise, the watcher won't be triggered until the next key event (particularly on shift and not on p)
        Mousetrap.bind("shift+p", () => { $timeout(scope.togglePreview); });
        Mousetrap.bind("mod+f", () => { ToolBridgeService.emitOmniboxFocus(); return false; });
        Mousetrap.bind("mod+z", () => { ToolBridgeService.emitOmniboxUndo(); return false; });
        Mousetrap.bind("mod+shift+z", () => { ToolBridgeService.emitOmniboxRedo(); return false; });
        const updateGraphDebounced = Debounce().withDelay(200,200).wrap(scope.updateGraph);
        const updateGraphDebouncedShort = Debounce().withDelay(0,0).withSpinner(false).wrap(scope.updateGraph);

        const deregister1 = $rootScope.$on('datasetsListChangedFromModal', updateGraphDebounced);
        const deregister2 = $rootScope.$on('taggableObjectTagsChanged', updateGraphDebounced);
        const deregister3 = $rootScope.$on('flowItemAddedOrRemoved', updateGraphDebounced);
        const deregister4 = $rootScope.$on('reloadGraph', (event, { zoomTo } = {}) => updateGraphDebounced(zoomTo));
        const deregister5 = $rootScope.$on('objectMetaDataChanged', updateGraphDebounced);
        const deregister6 = $rootScope.$on('discussionCountChanged', updateGraphDebounced);
        //const deregister7 = $rootScope.$on('unreadDiscussionsChanged', updateGraphDebounced); TODO: find a better solution to live-refresh the unread discussions
        const deregister8 = $rootScope.$on('featureGroupStatusChanged', updateGraphDebounced);
        const deregister9 = $rootScope.$on('zonesManualPositioningChanged', (event, { zoomTo } = {}) => updateGraphDebouncedShort(zoomTo, true, false));

        scope.$on("$destroy", function() {
            Mousetrap.unbind("z");
            Mousetrap.unbind("left");
            Mousetrap.unbind("right");
            Mousetrap.unbind("up");
            Mousetrap.unbind("down");
            Mousetrap.unbind("-");
            Mousetrap.unbind("+");
            Mousetrap.unbind("=");
            Mousetrap.unbind(togglePreviewShortcut);
            Mousetrap.unbind("mod+f");
            Mousetrap.unbind("mod+z");
            Mousetrap.unbind("mod+shift+z");
            deregister1();
            deregister2();
            deregister3();
            deregister4();
            deregister5();
            deregister6();
            //deregister7(); TODO: find a better solution to live-refresh the unread discussions
            deregister8();
            deregister9();

        });

        scope.$on('graphRendered', function graphRendered() {
            drawBuildInProgressIndicators(scope.svg, scope.nodesGraph);
            drawExposedIndicators(scope.svg, scope.nodesGraph);
            drawForbiddenIndicators(scope.svg, scope.nodesGraph);
            if (scope.nodesGraph) {

                WT1.tryEvent('project-flow-rendered', () => {
                    let payload = {
                        hasZones: scope.nodesGraph.hasZones,
                        isFlowEmpty: scope.isFlowEmpty,
                        zonesManualPositioning: scope.zonesManualPositioning,
                        canMoveZones: scope.canMoveZones,
                        count_nodes: scope.nodesGraph.nodesOnGraphCount
                    }
                    if (scope.nodesGraph.includedObjectsByType) {
                        for (const [typeName, countForType] of Object.entries(scope.nodesGraph.includedObjectsByType)) {
                            payload[typeName.toLowerCase()] = countForType;
                        }
                    }
                    return payload;
                });
                if (PageSpecificTourService.canStartFlowTour()) {
                    PageSpecificTourService.startFlowTour({ scope: scope, fromContext: 'flow' });
                    OpalsService.sendPageSpecificTourRecommendation(OpalsMessageService.PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.FLOW);
                } else {
                    OpalsService.sendPageSpecificTourRecommendation(null);
                }
            }
        });

        scope.$on('indexNodesDone', function indexNodesDone () {
            scope.cleanupCollapsedZones();
        });
    }
};
});

app.directive('lineageFlowExport', function() {
    return {
        restrict: 'EA',
        scope: true,
        controller: function($scope) {
            // Toolbox used by export-flow.js to prepare the data lineage flow to be exported
            $scope.exportToolbox = {
                checkLoading: function() {
                    return !$scope.svg;
                },
                removeDecorations: function() {}, // Not needed for data lineage as there is a dedicated page for the export
                getGraphBoundaries: function () {
                    const graphBBox = $scope.svg.querySelector("g.graph").getBBox();
                    return {
                        x: graphBBox.x,
                        y: graphBBox.y,
                        width: graphBBox.width,
                        height: graphBBox.height
                    };
                },
                adjustViewBox: function(x, y, width, height) {
                    $scope.svg.setAttribute('viewBox', [x, y, width, height].join(', '));
                },
                configureZones: function() {} // No zone on data lineage graph but it's needed as the generic flow export script use it
            }
        },
        link: function() {}
    }
});

app.directive('flowExportForm', function(GRAPHIC_EXPORT_OPTIONS, WT1, GraphicImportService) {
    return {
        replace: false,
        require: '^form',
        restrict: 'EA',
        scope: {
            params: '=',
            graphBoundaries: '='
        },
        templateUrl: '/templates/flow-editor/export-flow-form.html',
        link: function($scope, element, attrs, formCtrl) {
            WT1.event("flow-export-form-displayed", {});

            $scope.exportFormController = formCtrl;
            // Utilities that give us all the choices possible
            $scope.paperSizeMap = GRAPHIC_EXPORT_OPTIONS.paperSizeMap;
            $scope.orientationMap = GRAPHIC_EXPORT_OPTIONS.orientationMap;
            $scope.ratioMap = GRAPHIC_EXPORT_OPTIONS.ratioMap;
            $scope.paperInchesMap = GRAPHIC_EXPORT_OPTIONS.paperInchesMap;
            $scope.fileTypes = GRAPHIC_EXPORT_OPTIONS.fileTypes;
            $scope.tileScaleModes = GRAPHIC_EXPORT_OPTIONS.tileScaleModes;

            $scope.minResW = 500;
            $scope.minResH = 500;
            $scope.maxResW = 10000;
            $scope.maxResH = 10000;
            $scope.maxDpi = 300;

            let computeTileScale = function (tileScaleProps) {
                if (!tileScaleProps.enabled || tileScaleProps.percentage === undefined) {
                    return 1;
                } else {
                    return Math.max(1, tileScaleProps.percentage / 100)
                }
            };

            let computeBestTileScale = function(width, height) {
                const targetFactor = 1.0; // 1-to-1 between size of graph and exported image
                const xFactor = $scope.graphBoundaries.width / width;
                const yFactor = $scope.graphBoundaries.height / height;
                return Math.max(1, Math.ceil(Math.max(xFactor, yFactor) / targetFactor));
            };

            let capWidth = function(width) {
                return Math.min($scope.maxResW, Math.max($scope.minResW, width));
            };
            let capHeight = function(height) {
                return Math.min($scope.maxResH, Math.max($scope.minResH, height));
            };

            // Given an image width, height and tile scale, compute how many pages
            // will be required to render the whole graph
            let computeTileScaleSheets = function(width, height, tileScale) {
                if (width === undefined || height === undefined || tileScale == undefined) {
                    return {x: 0, y: 0, count: 0};
                }
                const sheetRatio = width / height;
                const graphRatio = $scope.graphBoundaries.width / $scope.graphBoundaries.height;
                let graphSheetWidth;
                let graphSheetHeight;
                if (sheetRatio < graphRatio) {
                    // Dominant width
                    graphSheetWidth = $scope.graphBoundaries.width / tileScale;
                    graphSheetHeight = graphSheetWidth / sheetRatio;
                } else {
                    // Dominant height
                    graphSheetHeight = $scope.graphBoundaries.height / tileScale;
                    graphSheetWidth = graphSheetHeight * sheetRatio;
                }
                const x = Math.max(1, Math.ceil($scope.graphBoundaries.width / graphSheetWidth));
                const y = Math.max(1, Math.ceil($scope.graphBoundaries.height / graphSheetHeight));
                const count = x * y;
                return {x: x, y: y, count: count};
            };

            // Compute the best width, height and tile scale for the exported image
            // for the supplied paper size and orientation.
            let setBestDimensions = function(authorizeTileScaling = true) {
                let exportFormat = $scope.params.exportFormat;

                let width, height;
                const sheetRatio = (exportFormat.orientation == "LANDSCAPE") ?
                    $scope.ratioMap[exportFormat.paperSize] :
                    1 / $scope.ratioMap[exportFormat.paperSize];
                const graphRatio = $scope.graphBoundaries.width / $scope.graphBoundaries.height;
                if (sheetRatio < graphRatio) {
                    // Dominant width
                    width = $scope.graphBoundaries.width;
                    height = width / sheetRatio;
                } else {
                    // Dominant height
                    height = $scope.graphBoundaries.height;
                    width = height * sheetRatio;
                }

                let tileScale = 1;
                let dpi = Math.max(width, height) / $scope.paperInchesMap[exportFormat.paperSize];
                if (authorizeTileScaling && dpi > $scope.maxDpi) {
                    width = (width * $scope.maxDpi) / dpi;
                    height = (height * $scope.maxDpi) / dpi;
                    tileScale = computeBestTileScale(width, height);
                }

                exportFormat.width = capWidth(Math.round(width));
                exportFormat.height = capHeight(Math.round(height));
                exportFormat.tileScale = tileScale;
            };

            // Parameters of the export
            $scope.params.exportFormat = {
                paperSize: "A4",
                orientation: "LANDSCAPE",
                fileType: "PDF",
                width: 1920,
                height: 1358,
                tileScale: 1,
            };
            let exportFormat = $scope.params.exportFormat;

            // Restore values from LocalStorage if they have been saved
            let savedFileType = localStorage.getItem("dku.flow.export.fileType");
            if (savedFileType && $scope.fileTypes.indexOf(savedFileType) >= 0) {
                exportFormat.fileType = savedFileType;
            }
            let savedPaperSize = localStorage.getItem("dku.flow.export.paperSize");
            if (savedPaperSize && $scope.paperSizeMap[savedPaperSize]) {
                exportFormat.paperSize = savedPaperSize;
            }
            if (savedPaperSize == "CUSTOM") {
                let savedWidth = localStorage.getItem("dku.flow.export.width");
                if (savedWidth && !isNaN(Number(savedWidth))) {
                    exportFormat.width = capWidth(Number(savedWidth));
                }
                let savedHeight = localStorage.getItem("dku.flow.export.height");
                if (savedHeight && !isNaN(Number(savedHeight))) {
                    exportFormat.height = capHeight(Number(savedHeight));
                }
            } else {
                let savedOrientation = localStorage.getItem("dku.flow.export.orientation");
                if (savedOrientation && $scope.orientationMap[savedOrientation]) {
                    exportFormat.orientation = savedOrientation;
                }
            }
            if (exportFormat.paperSize != "CUSTOM") {
                // Choose the best width & height and compute the tile scale
                setBestDimensions();
            }
            $scope.tileScale = {};
            $scope.tileScale.enabled = exportFormat.tileScale > 1;
            $scope.tileScale.percentage = exportFormat.tileScale * 100;
            $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);

            let onUpdatePaperSizeOrOrientation = function() {
                setBestDimensions();
                let exportFormat = $scope.params.exportFormat;
                $scope.tileScale.enabled = exportFormat.tileScale > 1;
                $scope.tileScale.percentage = exportFormat.tileScale * 100;
                $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
            };

            $scope.$watch('params.exportFormat.paperSize', function (newVal, oldVal) {
                if (newVal !== oldVal && newVal != 'CUSTOM') {
                    onUpdatePaperSizeOrOrientation();
                }
            });

            $scope.$watch('params.exportFormat.orientation', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    onUpdatePaperSizeOrOrientation();
                }
            });

            $scope.$watch('params.exportFormat.width', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
                }
            });

            $scope.$watch('params.exportFormat.height', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
                }
            });

            $scope.$watch('tileScale.enabled', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    if (newVal == true) {
                        // Try to keep the DPI of exported images around 300 dpi
                        if (exportFormat.paperSize != "CUSTOM") {
                            let dpi = Math.max(exportFormat.width, exportFormat.height) / GRAPHIC_EXPORT_OPTIONS.paperInchesMap[exportFormat.paperSize];
                            if (dpi > $scope.maxDpi) {
                                exportFormat.width = capWidth(Math.round(exportFormat.width * $scope.maxDpi / dpi));
                                exportFormat.height = capHeight(Math.round(exportFormat.height * $scope.maxDpi / dpi));
                            }
                        }
                        $scope.tileScale.percentage = computeBestTileScale(exportFormat.width, exportFormat.height) * 100;
                        exportFormat.tileScale = computeTileScale($scope.tileScale);
                    } else {
                        if (exportFormat.paperSize != "CUSTOM") {
                            setBestDimensions(false);
                        } else {
                            exportFormat.tileScale = 1;
                        }
                    }

                }
            });
            $scope.$watch('tileScale.percentage', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    exportFormat.tileScale = computeTileScale($scope.tileScale);
                    $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
                }
            });
        }
    }
});

app.controller("ExportFlowModalController", function($scope, $stateParams, $rootScope, DataikuAPI, ActivityIndicator, FutureProgressModal, WT1, ViewsLocalStorage) {
    $scope.init = function (projectKey, graphBoundaries) {
        $scope.params = {};
        $scope.modalTitle = "Export Flow graph";
        $scope.projectKey = projectKey;
        $scope.graphBoundaries = graphBoundaries;
    };

    $scope.doExportFlow = function() {
        WT1.event("flow-exported", {});

        // Duplicate export format and add the zones export information
        let exportFormat = JSON.parse(JSON.stringify($scope.params.exportFormat));
        exportFormat.drawZones = $scope.drawZones.drawZones;
        exportFormat.collapsedZones = $scope.collapsedZones;
        const currentSearchQuery = ViewsLocalStorage.getQueryForProject(
            $rootScope.appConfig.login,
            $stateParams.projectKey
        );
        if (currentSearchQuery && currentSearchQuery.trim().length !== 0) {
            exportFormat.searchQuery = currentSearchQuery;
        }

        // Save options into LocalStorage to use them again for next export
        localStorage.setItem("dku.flow.export.fileType", exportFormat.fileType);
        localStorage.setItem("dku.flow.export.paperSize", exportFormat.paperSize);
        if (exportFormat.paperSize === "CUSTOM") {
            localStorage.setItem("dku.flow.export.width", exportFormat.width);
            localStorage.setItem("dku.flow.export.height", exportFormat.height);
        } else {
            localStorage.setItem("dku.flow.export.orientation", exportFormat.orientation);
        }
        // Starting with Puppeteer 13.7.0, it's no longer possible to use tiling with PDF output
        if (exportFormat.fileType === 'PDF') {
            exportFormat.tileScale = 1;
        }

        // Export the flow
        DataikuAPI.flow.export($scope.projectKey, exportFormat)
            .error(setErrorInScope.bind($scope))
            .success(function (resp) {
                FutureProgressModal.show($scope, resp, "Export Flow graph").then(function (result) {
                    if (result) { // undefined in case of abort
                        downloadURL(DataikuAPI.flow.getExportURL(result.projectKey, result.exportId));
                        ActivityIndicator.success("Flow graph export downloaded!", 5000);
                    } else {
                        ActivityIndicator.error("Export Flow failed", 5000);
                    }
                    $scope.resolveModal();
                });
            });
    }
});

app.controller("ExportDataLineageModalController", function($scope, DataikuAPI, ActivityIndicator, FutureProgressModal, WT1) {
    $scope.init = function (projectKey, graphBoundaries, datasetName, columnName) {
        $scope.params = {};
        $scope.modalTitle = "Export Lineage graph";
        $scope.exportButtonText = "Export";
        $scope.graphBoundaries = graphBoundaries;
        $scope.datasetName = datasetName;
        $scope.columnName = columnName;
        $scope.projectKey = projectKey;
    };

    $scope.doExportFlow = function() {
        WT1.tryEvent("data-lineage-flow-exported", () => ({}));

        // Duplicate export format and add export information
        let exportFormat = JSON.parse(JSON.stringify($scope.params.exportFormat));

        // Save options into LocalStorage to use them again for next export
        localStorage.setItem("dku.datalineage.export.fileType", exportFormat.fileType);
        localStorage.setItem("dku.datalineage.export.paperSize", exportFormat.paperSize);
        if (exportFormat.paperSize === "CUSTOM") {
            localStorage.setItem("dku.datalineage.export.width", exportFormat.width);
            localStorage.setItem("dku.datalineage.export.height", exportFormat.height);
        } else {
            localStorage.setItem("dku.datalineage.export.orientation", exportFormat.orientation);
        }
        // Starting with Puppeteer 13.7.0, it's no longer possible to use tiling with PDF output
        if (exportFormat.fileType === 'PDF') {
            exportFormat.tileScale = 1;
        }

        // Export the data lineage flow
        DataikuAPI.datalineage.export($scope.projectKey, $scope.datasetName, $scope.columnName, exportFormat)
            .error(setErrorInScope.bind($scope))
            .success(function (resp) {
                FutureProgressModal.show($scope, resp, "Export Data Lineage graph").then(function (result) {
                    if (result) { // undefined in case of abort
                        downloadURL(DataikuAPI.datalineage.getExportURL(result.projectKey, result.exportId));
                        ActivityIndicator.success("Data Lineage graph export downloaded!", 5000);
                    } else {
                        ActivityIndicator.error("Export Data Lineage failed", 5000);
                    }
                    $scope.resolveModal();
                });
            });
    }
});

app.controller('GenerateFlowDocumentModalController', function($scope, DataikuAPI, WT1, FutureWatcher, ProgressStackMessageBuilder) {
    $scope.init = function (projectKey) {
        $scope.projectKey = projectKey;
        $scope.template = { type: "DEFAULT" };
        $scope.newTemplate = {};
        $scope.state = "WAITING";
    };

    $scope.export = function() {
        if ($scope.template.type == "DEFAULT") {
            WT1.event("render-flow-documentation", {type: "default"});
            DataikuAPI.flow.docGenDefault($scope.projectKey)
            .success(watchFuture)
            .error(setErrorInScope.bind($scope));
        } else {
            WT1.event("render-flow-documentation", {type: "custom"});
            DataikuAPI.flow.docGenCustom($scope.newTemplate.file, $scope.projectKey, (e) => {
                // Unlikely to upload big files so no need to track progress
            })
            .then(watchFuture)
            .catch((error) => { setErrorInScope2.call($scope, error); });
        }

        function watchFuture(initialResponse) {
            $scope.initialResponse = angular.fromJson(initialResponse);
            $scope.data = undefined;
            $scope.state = "LOADING";

            FutureWatcher.watchJobId($scope.initialResponse.jobId)
            .success(function(response) {
                let exportId = response.result.exportId;
                $scope.data = response.result.data;
                $scope.state = "READY";

                $scope.text = "The flow documentation is ready.";
                $scope.errorOccurred = false;
                if ($scope.data.maxSeverity === 'WARNING') {
                    $scope.text += " Be aware that the placeholders which couldn't be resolved are not shown in the flow documentation.";
                } else if ($scope.data.maxSeverity === 'ERROR') {
                    $scope.text = "";
                    $scope.errorOccurred = true;
                }

                if (!$scope.errorOccurred) {
                    $scope.flowDocumentationURL = DataikuAPI.flow.getFlowDocumentationExportURL(exportId);
                }

            }).update(function(response) {
                $scope.futureResponse = response;
                $scope.percentage =  ProgressStackMessageBuilder.getPercentage(response.progress);
                $scope.stateLabels = ProgressStackMessageBuilder.build(response.progress, true);
            }).error(function(response, status, headers) {
               setErrorInScope.bind($scope)(response, status, headers);
            });
        }
    }

    $scope.download = function() {
        downloadURL($scope.flowDocumentationURL);
        WT1.event("download-flow-documentation");
        $scope.state = "DOWNLOADED";
    };

    $scope.abort = function() {
        DataikuAPI.futures.abort($scope.initialResponse.jobId).error(setErrorInScope.bind($scope));
        $scope.dismiss();
        WT1.event("abort-flow-documentation-rendering");
    }
});

app.directive('facetFilterableList', function ($filter) {
    return {
        scope: {items: '=', model: '=facetFilterableList', showAllItems: '=?', orderBy:'@'},
        transclude: true,
        link: function (scope, element, attr) {
            if (attr.filterFunction) {
                scope.filterFunction = scope.$parent.$eval(attr.filterFunction);
            } else {
                scope.filterFunction = $filter('filter');
            }
            scope.model = scope.model || [];
            scope.onFacetSearchKeyDown = function (e) {
                if (e.keyCode === 27) { // ESC key
                    e.target.blur();
                    angular.element(e.target).scope().$parent.showInput = false;
                    angular.element(e.target).scope().$parent.facetValueSearch = '';
                }
            };
        },
        templateUrl: '/templates/flow-editor/facet-filterable-list.html'
    }
});


app.directive('multiItemsRightColumnSummary', function($controller, $rootScope, $stateParams,
    DataikuAPI, Fn, TaggableObjectsUtils, RecipeDescService, CodeEnvsService, SavedModelsService,
    FlowGraphSelection, SelectablePluginsService, WatchInterestState, FlowGraph, SubFlowCopyService, WT1, translate, PluginCategoryService) {

    return {
        templateUrl:'/templates/flow-editor/multi-items-right-column-summary.html',

        link: function(scope, element, attrs) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            const getType = attrs.singleType ? () => attrs.singleType : item => item.nodeType;
            const getSelectedItems = attrs.selectedItems ? () => scope.$eval(attrs.selectedItems) : FlowGraphSelection.getSelectedNodes;
            const newItemsWatch = attrs.selectedItems ? () => scope.$eval(attrs.selectedItems) : 'rightColumnItem';

            scope.projectsWithManageExposedElementsPrivilege = null;
            if (!scope.canWriteProject()) {
                // No need to load this information unless the user doesn't have write privilege
                DataikuAPI.projects.listHeads("MANAGE_EXPOSED_ELEMENTS").success(function(projects) {
                    scope.projectsWithManageExposedElementsPrivilege = projects;
                }).error(setErrorInScope.bind(scope));
            }

            function getCountByNodeType(selectedNodes) {
                let ret = {};
                selectedNodes.forEach(function(item) {
                    const type = getType(item);
                    ret[type] = (ret[type] || 0) + 1;
                });
                return ret;
            }

            function getCountByIndexableType(selectedNodes) {
                return selectedNodes.reduce((acc, curr) => {
                    const indexableType = TaggableObjectsUtils.getIndexableTypeFromNode(curr, getType(curr));
                    acc[indexableType] = (acc[indexableType] || 0) + 1;
                    return acc;
                }, {});
            }

            scope.getTaggableTypeMap = function () {
                let ret = {};
                scope.getSelectedNodes().forEach(function (item) {
                    let type = TaggableObjectsUtils.fromNodeType(getType(item));
                    if (ret.hasOwnProperty(type)) {
                        ret[type].push(getSmartName(item));
                    } else {
                        ret[type] = [getSmartName(item)];
                    }
                })
                return ret;
            }

            function count(nodeType) {
                return scope.selection.countByNodeType[nodeType] || 0;
            }

            function selectedNodes() {
                return scope.selection.selectedObjects;
            }
            scope.getSelectedNodes = selectedNodes;

            function isAll(nodeTypes) {
                return function() {
                    const total = scope.selection.selectedObjects.length;
                    return total > 0 && nodeTypes.map(count).reduce(Fn.SUM) == total;
                };
            }
            function containsNodeType(nodeTypes) {
                return () => scope.selection.selectedObjects.length > 0 && nodeTypes.map(count).reduce(Fn.SUM) > 0;
            }
            function allHaveFlag(propName) {
                return function() {
                    const total = scope.selection.selectedObjects.length;
                    return total > 0 && scope.selection.selectedObjects.filter(Fn.prop(propName)).length == total
                };
            }
            // TODO @labeling right panel : multi items
            scope.isAllRecipes = isAll(['RECIPE']);
            scope.containsRecipes = containsNodeType(['RECIPE']);
            scope.isAllContinuousRecipes = allHaveFlag("continuous");
            scope.isAllDatasets = isAll(['LOCAL_DATASET', 'FOREIGN_DATASET']);
            scope.containsDatasets = containsNodeType(['LOCAL_DATASET', 'FOREIGN_DATASET']);
            scope.isAllFolders = isAll(['LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER']);
            scope.isAllStreamingEndpoints = isAll(['LOCAL_STREAMING_ENDPOINT']);
            scope.isAllModels = isAll(['LOCAL_SAVEDMODEL', 'FOREIGN_SAVEDMODEL']);
            scope.isAllEvaluationStores = isAll(['LOCAL_MODELEVALUATIONSTORE', 'FOREIGN_MODELEVALUATIONSTORE', 'LOCAL_GENAIEVALUATIONSTORE', 'FOREIGN_GENAIEVALUATIONSTORE']);
            scope.isAllCodeStudios = isAll(['CODE_STUDIO']);
            scope.isAllZones = isAll(['ZONE']);
            scope.isAllProjects = isAll(['PROJECT']);
            scope.isAllLocal = isAll(['RECIPE', 'LOCAL_DATASET', 'LOCAL_MANAGED_FOLDER', 'LOCAL_SAVEDMODEL', 'LOCAL_MODELEVALUATIONSTORE', 'LOCAL_GENAIEVALUATIONSTORE', 'LOCAL_RETRIEVABLE_KNOWLEDGE']);
            scope.isAllForeign = isAll(['FOREIGN_DATASET', 'FOREIGN_MANAGED_FOLDER', 'FOREIGN_SAVEDMODEL', 'FOREIGN_MODELEVALUATIONSTORE', 'FOREIGN_GENAIEVALUATIONSTORE', 'FOREIGN_RETRIEVABLE_KNOWLEDGE']);
            scope.isAllComputables = isAll(['LOCAL_DATASET', 'FOREIGN_DATASET', 'LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER', 'LOCAL_SAVEDMODEL', 'FOREIGN_SAVEDMODEL', 'LOCAL_MODELEVALUATIONSTORE', 'FOREIGN_MODELEVALUATIONSTORE', 'LOCAL_GENAIEVALUATIONSTORE', 'FOREIGN_GENAIEVALUATIONSTORE', 'LOCAL_RETRIEVABLE_KNOWLEDGE', 'FOREIGN_RETRIEVABLE_KNOWLEDGE']);
            scope.isAllDatasetsAndFolders = isAll(['LOCAL_DATASET', 'FOREIGN_DATASET', 'LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER']);
            scope.isAllSharable = isAll(['LOCAL_DATASET', 'DASHBOARD', 'WEB_APP']);
            scope.hasVisualSection = scope.isAllDatasets || scope.isAllFolders || scope.isDatasetAndFolder;

            scope.getSingleSelectedDataset = function() {
                const candidates = selectedNodes().filter(({nodeType}) => ['LOCAL_DATASET', 'FOREIGN_DATASET'].includes(nodeType));
                return candidates.length == 1 ? candidates[0] : null;
            }

            scope.getSingleSelectedModel = function() {
                const candidates = selectedNodes().filter(({nodeType}) => ['LOCAL_SAVEDMODEL', 'FOREIGN_SAVEDMODEL'].includes(nodeType));
                return candidates.length == 1 ? candidates[0] : null;
            }

            scope.getSingleSelectedFolder = function() {
                const candidates = selectedNodes().filter(({nodeType}) => ['LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER'].includes(nodeType));
                return candidates.length == 1 ? candidates[0] : null;
            }

            scope.isPredictionModel = function() {
                return scope.singleSelectedModelInfos
                    && scope.singleSelectedModelInfos.model.miniTask
                    && scope.singleSelectedModelInfos.model.miniTask.taskType == 'PREDICTION'
                    && scope.singleSelectedModelInfos.model.miniTask.backendType !== 'VERTICA';
            }

            scope.isClusteringModel = function() {
                return scope.singleSelectedModelInfos
                    && scope.singleSelectedModelInfos.model.miniTask
                    && scope.singleSelectedModelInfos.model.miniTask.taskType == 'CLUSTERING'
                    && scope.singleSelectedModelInfos.model.miniTask.backendType !== 'VERTICA';
            }

            scope.isLLM = function() {
                return scope.singleSelectedModelInfos
                    && (SavedModelsService.isAgent(scope.singleSelectedModelInfos.model)
                        || SavedModelsService.isRetrievalAugmentedLLM(scope.singleSelectedModelInfos.model)
                        || SavedModelsService.isLLMGeneric(scope.singleSelectedModelInfos.model));
            };

            scope.isDatasetAndModel = function() {
                return selectedNodes().length == 2 && scope.getSingleSelectedDataset() && scope.getSingleSelectedModel();
            }

            scope.isDatasetAndFolder = function() {
                return selectedNodes().length == 2 && scope.getSingleSelectedDataset() && scope.getSingleSelectedFolder();
            }

            scope.isTwoDatasets = function() {
                return selectedNodes().length == 2 && scope.isAllDatasets();
            }

            scope.isAllMetastoreAware = function() {
                const total = selectedNodes().length;
                const hiveRecipes = selectedNodes().filter(n => TaggableObjectsUtils.isHDFSAbleType(n.datasetType)).length;
                return total > 0 && hiveRecipes == total;
            };
            scope.isAllImpalaRecipes = function() {
                const total = selectedNodes().length;
                const impalaRecipes = selectedNodes().filter(n => (n.recipeType||n.type) == 'impala').length;
                return total > 0 && impalaRecipes == total;
            };
            scope.isAllPythonCodeEnvSelectableRecipes = function() {
                const total = selectedNodes().length;
                const codeEnvSelectableRecipes = selectedNodes().filter(n => (n.recipeType||n.type) && CodeEnvsService.canPythonCodeEnv(n)).length;
                return total > 0 && codeEnvSelectableRecipes == total;
            };
            scope.isAllRCodeEnvSelectableRecipes = function() {
                const total = selectedNodes().length;
                const codeEnvSelectableRecipes = selectedNodes().filter(n => (n.recipeType||n.type) && CodeEnvsService.canRCodeEnv(n)).length;
                return total > 0 && codeEnvSelectableRecipes == total;
            };
            scope.isAllHiveRecipes = function() {
                const total = selectedNodes().length;
                const hiveRecipes = selectedNodes().filter(n => (n.recipeType||n.type) == 'hive').length;
                return total > 0 && hiveRecipes == total;
            };

            scope.isAllManaged = function() {
                const total = selectedNodes().length;
                const managed = selectedNodes().filter(n => n.managed).length;
                return total > 0 && managed == total;
            };
            scope.isAllWatched = function() {
                const total = selectedNodes().length;
                const watched = selectedNodes().filter(n => n.interest && WatchInterestState.isWatching(n.interest.watching)).length;
                return total > 0 && watched == total;
            };
            scope.isAllStarred = function() {
                const total = selectedNodes().length;
                const starred = selectedNodes().filter(n => n.interest && n.interest.starred).length;
                return total > 0 && starred == total;
            };
            scope.isAllVirtualizable = function() {
                return selectedNodes().map(x => !!x.virtualizable).reduce((a,b) => a && b, true);
            };
            scope.canManageExposedElementsOnAllOriginalProjects = function() {
                if (scope.projectsWithManageExposedElementsPrivilege == null) {
                    return false;
                }

                const projectKeysWithManageExposedElementsPrivilege = scope.projectsWithManageExposedElementsPrivilege.map(p => p.projectKey);
                const originalProjectKeys = new Set(selectedNodes().map(n => n.projectKey));
                return [...originalProjectKeys].every(key => projectKeysWithManageExposedElementsPrivilege.includes(key));
            };

            scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary.sparkPipelinesEnabled || $rootScope.projectSummary.sqlPipelinesEnabled;
            };

            function showVirtualizationAction(showDeactivate) {
                return function() {
                    return scope.isProjectAnalystRW()
                        && scope.isAllDatasets()
                        && scope.isAllLocal()
                        && showDeactivate === scope.isAllVirtualizable();
                }
            }
            scope.showAllowVirtualizationAction = showVirtualizationAction(false);
            scope.showStopVirtualizationAction = showVirtualizationAction(true);


            scope.anyMultiEngineRecipe = function() {
                function isMultiEngine(recipeType) {
                    const desc = RecipeDescService.getDescriptor(recipeType);
                    return !!desc && desc.isMultiEngine;
                }
                return !!selectedNodes().filter(node => isMultiEngine(node.recipeType||node.type)).length;
            };

            scope.anyImpala = function() {
                return !!selectedNodes().filter(n => (n.recipeType||n.type) == 'impala').length;
            };

            scope.anyHive = function() {
                return !!selectedNodes().filter(n => (n.recipeType||n.type) == 'hive').length;
            };

            scope.anyCanSpark = function() {
                return !!selectedNodes().filter(node => scope.canSpark(node)).length;
            };

            scope.allAreSparkNotSQLRecipes = function() {
                return selectedNodes().every(node => ['spark_scala','pyspark','sparkr'].indexOf(node.recipeType||node.type) >= 0);
            };

            scope.anyCanSparkPipeline = function() {
                return selectedNodes().some(node => scope.canSparkPipeline(node));
            };

            scope.anyCanSqlPipeline = function() {
                return selectedNodes().some(node => scope.canSqlPipeline(node));
            };

            scope.allAutoTriggersDisabled = function() {
                return scope.getAutoTriggerDisablingReason($rootScope.appConfig, $rootScope.projectSummary);
            };

            scope.autoTriggersObjects = function(autoTriggerStatus, objects) {
                objects.forEach(function(object){
                    object.active = autoTriggerStatus;
                    scope.toggleActive(object);
                })
            };

            scope.isAllUnshareable = function() {
                const total = selectedNodes().length;
                const unshareables = selectedNodes().filter(n => n.usedByZones && n.usedByZones.length && !n.successors.length).length;
                return total > 0 && unshareables == total;
            }

            scope.canPublishAllToDataCollection = function() {
                // we already know isAllDataset is true, nodes are either 'LOCAL_DATASET' or 'FOREIGN_DATASET'
                // we only check global auth & local auth, as foreign would require to much info (it will trigger a warning on modal open if there is an issue)
                return $rootScope.appConfig.globalPermissions.mayPublishToDataCollections &&
                    (count('LOCAL_DATASET') === 0 || $rootScope.projectSummary.canPublishToDataCollections);
            }

            scope.getSelectedObjectsZones = function() {
                return getSelectedItems().map(n => n.usedByZones[0]);
            }

            scope.getCommonZone = function () {
                const nodesSelected = selectedNodes();
                return nodesSelected.length
                    ? (nodesSelected[0].usedByZones || [])[0] || nodesSelected[0].ownerZone
                    : null;
            };

            function getSmartName(it) {
                return it.projectKey == $stateParams.projectKey ? it.name : it.projectKey+'.'+it.name;
            }
            scope.getSmartNames = function () {
                return selectedNodes().map(getSmartName);
            };

            scope.clearSelection = function() {
                FlowGraphSelection.clearSelection();
            };

            scope.refreshData = function() {
                let selectedNodes = getSelectedItems();
                scope.selection = {
                    selectedObjects: selectedNodes,
                    taggableType: TaggableObjectsUtils.getCommonType(selectedNodes, node => TaggableObjectsUtils.fromNodeType(getType(node))),
                    countByNodeType: getCountByNodeType(selectedNodes),
                    indexableType: TaggableObjectsUtils.getCommonIndexableType(selectedNodes, node => TaggableObjectsUtils.getIndexableTypeFromNode(node, getType(node))),
                    countByIndexableType: getCountByIndexableType(selectedNodes),
                };
                scope.usability = scope.computeActionsUsability();
                scope.selectablePlugins =  scope.isAllComputables(selectedNodes) ? SelectablePluginsService.listSelectablePlugins(scope.selection.countByIndexableType) : [];
                let availableCategories = []
                if (scope.hasVisualSection) {
                    availableCategories.push('visual')
                }
                if (scope.isAllComputables) {
                    availableCategories.push('code')
                }
                scope.noRecipesCategoryPlugin = PluginCategoryService.standardCategoryPlugins(scope.selectablePlugins, availableCategories);

                scope.singleSelectedDataset = scope.getSingleSelectedDataset();
                scope.singleSelectedFolder = scope.getSingleSelectedFolder();
                scope.selectedFolderIsSharepoint = scope.singleSelectedFolder?.folderType === "SharePointOnline";
                scope.refreshSingleSelectedModelInfos();
            };

            scope.refreshSingleSelectedModelInfos = function() {
                if(!scope.getSingleSelectedModel()) {
                    scope.singleSelectedModelInfos = null;
                    scope.singleSelectedModel = null;
                }
                if(scope.getSingleSelectedModel() == scope.singleSelectedModel) {
                    return;
                }
                scope.singleSelectedModel = scope.getSingleSelectedModel();
                scope.singleSelectedModelInfos = null;
                let projectKey = scope.singleSelectedModel.projectKey;
                let name = scope.singleSelectedModel.name;
                DataikuAPI.savedmodels.getFullInfo($stateParams.projectKey, getSmartName(scope.singleSelectedModel)).success(data => {
                    if (!scope.singleSelectedModel || scope.singleSelectedModel.projectKey != projectKey || scope.singleSelectedModel.name != name) {
                        return; // too late, the selected model has changed in the meantime
                    }
                    scope.singleSelectedModelInfos = data;
                    scope.singleSelectedModelInfos.zone = (scope.singleSelectedModel.usedByZones || [])[0] || scope.singleSelectedModel.ownerZone;
                }).error(setErrorInScope.bind(scope));
            };

            scope.filterSelection = function(indexableType) {
                FlowGraphSelection.filterByIndexableType(indexableType);
                scope.refreshData();
            }

            scope.refreshData();
            scope.$watch(newItemsWatch, scope.refreshData);

            scope.collapseSelectedZones = () => {
                scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'collapseAll');
            }

            scope.expandSelectedZones = () => {
                scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'expandAll');
            }

            scope.isAllZonesExpanded = function() {
                const allZones = scope.selection.selectedObjects.filter(so => getType(so) === 'ZONE');
                return allZones.filter(z => z.customData.isCollapsed === false).length === allZones.length;
            };

            scope.isAllZonesCollapsed = function() {
                const allZones = scope.selection.selectedObjects.filter(so => getType(so) === 'ZONE');
                return allZones.filter(z => z.customData.isCollapsed === true).length === allZones.length;
            };

            scope.copyZone = () => {
                const startCopyTool = scope.startTool('COPY',  {preselectedNodes: FlowGraphSelection.getSelectedNodes().map(n => n.id)});
                startCopyTool.then(() => {
                    const selectedTaggableObjectRefs = FlowGraphSelection.getSelectedTaggableObjectRefs();
                    const itemsByZones = FlowGraph.nodesByZones((node) => TaggableObjectsUtils.fromNode(node));
                    SubFlowCopyService.start(selectedTaggableObjectRefs, itemsByZones, scope.stopAction);
                });
            };

            scope.insertRecipeBetween = function() {
                scope.showInsertRecipeModal(scope.selectedDataset, scope.selectedRecipes)
                    .then(selectedRecipe => WT1.event('rightpanelrecipe_actions_insertrecipe', {position: 'between', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
            };

            scope.canInsertRecipeBetween = function() {
                const selectedNodes = getSelectedItems();
                const selectedDatasets = [];
                const selectedRecipes =  [];
                for (const node of selectedNodes) {
                    if (node.nodeType === "LOCAL_DATASET" || node.nodeType === "FOREIGN_DATASET") {
                        selectedDatasets.push(node);
                    } else if (node.nodeType === "RECIPE") {
                        selectedRecipes.push(node);
                    }
                }
                scope.selectedDataset = selectedDatasets[0];    // Because we know for sure only one dataset is selected
                scope.selectedRecipes = selectedRecipes;
                return FlowGraphSelection.canInsertRecipeBetweenSelection(selectedNodes, selectedDatasets, selectedRecipes);
            }

            scope.getInsertRecipeBetweenTooltip = function() {
                if (!scope.canWriteProject()) return translate('PROJECT.PERMISSIONS.WRITE_ERROR', 'You don\'t have the permissions to write to this project');
                if (!scope.canInsertRecipeBetween()) return translate('PROJECT.FLOW.RIGHT_PANEL.INSERT_RECIPE.ERROR', 'The current selection does not allow to insert a new recipe. Select a dataset and a specific downstream recipe to insert a recipe in between.');
                return translate('PROJECT.FLOW.RIGHT_PANEL.INSERT_RECIPE.TOOLTIP', 'Insert recipe (and corresponding output dataset) between selected items');
            }
        }
    }
});


app.controller('_FlowContextMenus', function($scope, $state, $stateParams, $controller, WT1, GlobalProjectActions, FlowGraph, FlowGraphSelection, FlowGraphFolding, TaggableObjectsUtils, Logger) {

    WT1.event("flow-context-menu-open");

    $controller('_TaggableObjectsCapabilities', {$scope: $scope});
    $controller('_TaggableObjectsMassActions', {$scope: $scope});

    $scope.toggleTab = tabName => {
        FlowGraphSelection.clearSelection();
        FlowGraphSelection.onItemClick($scope.object);

       $scope.standardizedSidePanel.toggleTab(tabName);
    };

    $scope.getSelectedTaggableObjectRefs = function() {
        return [TaggableObjectsUtils.fromNode($scope.object)];
    };

    $scope.computeMovingImpact = function() {
        let realNode = $scope.object.usedByZones.length ? FlowGraph.node(`zone__${$scope.object.ownerZone}__${$scope.object.realId}`) : $scope.object;
        var computedImpact = [];
        function addSuccessors(node) {
            if (node.nodeType != "RECIPE") return;
            let successors = node.successors;
            successors.forEach(function(successor) {
                if (successor == realNode.id) return;
                computedImpact.push(TaggableObjectsUtils.fromNode(FlowGraph.node(successor)));
            });
        }

        let predecessor = realNode.predecessors[0];
        if (predecessor && realNode.nodeType != "RECIPE" && !realNode.isHiddenLinkTarget) {
            computedImpact.push(TaggableObjectsUtils.fromNode(FlowGraph.node(predecessor)));
            addSuccessors(FlowGraph.node(predecessor));
        }

        addSuccessors(realNode);
        return computedImpact;
    }

    $scope.zoomToOtherZoneNode = function(zoneId) {
        const otherNodeId = $scope.object.id.replace(/zone__.+?__/, "zone__" + zoneId + "__");
        if ($stateParams.zoneId) {
            $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId, id: graphVizUnescape(otherNodeId) }));
        }
        else {
            $scope.zoomGraph(otherNodeId);
            FlowGraphSelection.clearSelection();
            FlowGraphSelection.onItemClick($scope.nodesGraph.nodes[otherNodeId]);
        }
    }

    $scope.$state = $state;
    $scope.$stateParams = $stateParams;
    $scope.othersZones = FlowGraph.nodeSharedBetweenZones($scope.object) ? Array.from(FlowGraph.nodeSharedBetweenZones($scope.object)) : null;

    $scope.startPropagateToolFromRecipe = function(node) {
        const predecessorNodeId = node.predecessors[0];
        const predecessorNode = FlowGraph.get().nodes[predecessorNodeId];
        $scope.startTool('PROPAGATE_SCHEMA', {projectKey: predecessorNode.projectKey, datasetName: predecessorNode.name});
    }

    $scope.selectSuccessors = function() {
        WT1.event("flow-context-menu-select-successors");
        FlowGraphSelection.selectSuccessors($scope.object);
    };

    $scope.selectPredecessors = function() {
        WT1.event("flow-context-menu-select-predecessors");
        FlowGraphSelection.selectPredecessors($scope.object);
    };

    $scope.hasPredecessorsInOtherZone = function(object) {
        return !$stateParams.zoneId && FlowGraphSelection.hasPredecessorsInOtherZone(object);
    }

    $scope.hasSuccessorsInOtherZone = function(object) {
        return !$stateParams.zoneId && FlowGraphSelection.hasSuccessorsInOtherZone(object);
    }

    $scope.foldSuccessors = function() {
        WT1.event("flow-context-menu-fold", {direction:'successors'});
        FlowGraphFolding.foldSuccessors($scope.object);
    };

    $scope.foldPredecessors = function() {
        WT1.event("flow-context-menu-fold", {direction: 'predecessors'});
        FlowGraphFolding.foldPredecessors($scope.object);
    };

    $scope.previewSelectSuccessors = function(object) {
        FlowGraphFolding.previewSelect($scope.object, "successors");
    };

    $scope.previewSelectPredecessors = function(object) {
        FlowGraphFolding.previewSelect($scope.object, "predecessors");
    };

    $scope.previewFoldSuccessors = function(object) {
        FlowGraphFolding.previewFold($scope.object, "successors");
    };

    $scope.previewFoldPredecessors = function(object) {
        FlowGraphFolding.previewFold($scope.object, "predecessors");
    };

    $scope.endPreviewBranch = function() {
        FlowGraphFolding.endPreviewBranch();
    };

    $scope.deleteFlowItem = function() {
        WT1.event('flow-context-menu-delete');

        const type = TaggableObjectsUtils.fromNodeType($scope.object.nodeType);
        const id = $scope.object.name;
        const displayName = $scope.object.description;
        GlobalProjectActions.deleteTaggableObject($scope, type, id, displayName)
            .then(FlowGraphSelection.clearSelection);
    };

    $scope.canInsertRecipeAfter = function() {
        return $scope.object.successors && $scope.object.successors.length && $scope.object.successors.every(successor => FlowGraph.node(successor).nodeType === 'RECIPE');
    }

    $scope.insertRecipeAfter = function() {
        $scope.showInsertRecipeModal($scope.object)
            .then(selectedRecipe => WT1.event('flow-context-menu-insertrecipe', {position: 'after', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
    }

    $scope.isDeleteAndReconnectAllowed = false;

    // Call async function to set isDeleteAndReconnectAllowed appropriately (do not use directly from html)
    $scope.canDeleteAndReconnectObject($scope.object).then(function(result) {
        $scope.isDeleteAndReconnectAllowed = result;
    }).catch(function(error) {
        Logger.warn(`Problem determining whether to show delete & reconnect option for ${$scope.object.id}:`, error);
    });

    $scope.deleteAndReconnectRecipe = function(wt1_event_name) {
        $scope.doDeleteAndReconnectRecipe($scope.object, wt1_event_name);
    }
});


app.controller("SavedModelContextualMenuController", function($scope, $state, $controller, WT1, SavedModelRenameService) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.renameSavedModel = function(savedModel) {
        SavedModelRenameService.renameSavedModel({
            scope: $scope,
            state: $state,
            projectKey: savedModel.projectKey,
            savedModelId: savedModel.name,
            savedModelName: savedModel.description
        });
    }

    $scope.trainThisModel = function() {
        WT1.event('flow-context-menu-train');
        $scope.trainModel($scope.object.projectKey, $scope.object.name);
    };
});

app.controller("ModelEvaluationStoreContextualMenuController", function ($scope, $controller, $state, ModelEvaluationStoreRenameService) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.renameMES = function(modelEvaluationStore) {
        ModelEvaluationStoreRenameService.renameModelEvaluationStore({
            scope: $scope,
            state: $state,
            projectKey: modelEvaluationStore.projectKey,
            modelEvaluationStoreId: modelEvaluationStore.name,
            modelEvaluationStoreName: modelEvaluationStore.description
        });
    };
});

app.controller("ManagedFolderContextualMenuController", function($scope, $controller, $state, WT1, ManagedFolderRenameService) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.renameManagedFolder = function (managedFolder) {
        const projectKey = managedFolder.projectKey;
        const managedFolderCurrentName = managedFolder.description;
        const scopeWithObjectType = $scope.$new();
        scopeWithObjectType.objectType = "MANAGED_FOLDER";

        ManagedFolderRenameService.renameManagedFolder({
            scope: scopeWithObjectType,
            state: $state,
            projectKey,
            managedFolderId: managedFolder.name,
            managedFolderCurrentName
        });
    };

    $scope.buildThis = function() {
        WT1.event('flow-context-menu-build');
        $scope.buildManagedFolder($scope.object.projectKey, $scope.object.id);
    };
});

app.controller("KnowledgeBankContextualMenuController", function($scope, $controller, WT1, DatasetsService) {
    $controller('_FlowContextMenus', {$scope: $scope});
});

app.controller("ZoneContextualMenuController", function($scope, $rootScope, $controller, DataikuAPI, TaggableObjectsUtils, $stateParams, CreateModalFromTemplate, FlowGraph, FlowGraphSelection) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.deleteZone = () => {
        DataikuAPI.flow.zones.delete($stateParams.projectKey, $scope.object.name).success(() => {
            if ($stateParams.zoneId) {
                $scope.zoomOutOfZone();
            } else {
                $scope.$emit('reloadGraph');
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.openZone = items => {
        const zoneToOpen = items.map(ref => ref.id)[0];
        $scope.zoomOnZone(zoneToOpen);
    }

    $scope.collapseAllZones = () => {
        const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
        $scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'collapseAll');
    }

    $scope.expandAllZones = () => {
        const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
        $scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'expandAll');
    }

    $scope.collapseSelectedZones = () => {
        $scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'collapseAll');
    }

    $scope.expandSelectedZones = () => {
        $scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'expandAll');
    }

    $scope.editZone = () => {
        CreateModalFromTemplate("/templates/zones/edit-zone-modal.html", $scope, null, function(newScope){
            newScope.uiState = {
                color: $scope.object.customData.color,
                name: $scope.object.description
            };

            newScope.go = function(){
                DataikuAPI.flow.zones.edit($stateParams.projectKey, $scope.object.name, newScope.uiState.name, newScope.uiState.color).success(function () {
                    $scope.$emit('reloadGraph');
                    if ($stateParams.zoneId) {
                        $rootScope.$emit("zonesListChanged", newScope.uiState.name);
                    }
                    newScope.dismiss()
                }).error(setErrorInScope.bind(newScope));
            }
        });
    }
});


app.controller("DatasetContextualMenuController", function($scope, $rootScope, $controller, WT1, DataikuAPI, TaggableObjectsUtils) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.buildThisDataset = function() {
        WT1.event('flow-context-menu-build');
        $scope.buildDataset($scope.object.projectKey, $scope.object.name, $scope.object.predecessors.length, $scope.object.successors.length);
    };

    $scope.reloadSchema = function() {
        WT1.event('flow-context-menu-dataset-reload-schema');
        DataikuAPI.datasets.reloadSchema($scope.object.projectKey, $scope.object.name).error(setErrorInScope.bind($scope));
    };

    $scope.markAsBuilt = function() {
        WT1.event('flow-context-menu-mark-as-built');
        DataikuAPI.datasets.markAsBuilt([TaggableObjectsUtils.fromNode($scope.object)]).then(function() {
            $rootScope.$emit('reloadGraph');
        }, setErrorInScope.bind($scope));
    };

});


app.controller("ForeignDatasetContextualMenuController", function($scope, $controller, $state, WT1, DataikuAPI) {
    $controller('_FlowContextMenus', {$scope: $scope});
});


app.controller("StreamingEndpointContextualMenuController", function($scope, $rootScope, $controller, WT1, DataikuAPI, TaggableObjectsUtils) {
    $controller('_FlowContextMenus', {$scope: $scope});
});

app.controller("RecipeContextualMenuController", function($scope, $controller, $stateParams, WT1, ComputableSchemaRecipeSave) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.propagateSchema = function() {
        WT1.event('flow-context-menu-propagate-schema');
        ComputableSchemaRecipeSave.handleSchemaUpdateFromAnywhere($scope, $stateParams.projectKey, $scope.object.name)
    }
});

app.controller("LabelingTaskContextualMenuController", function($scope, $controller, $stateParams, WT1, ComputableSchemaRecipeSave) {
    $controller('_FlowContextMenus', {$scope: $scope});
    // TODO @labeling implement propagateSchema
});


app.controller("MultiContextualMenuController", function($scope, $controller, WT1, FlowGraphSelection, FlowGraphFolding, TaggableObjectsService, TaggableObjectsUtils, FlowGraph) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $controller('_TaggableObjectsMassActions', {$scope: $scope});
    $controller('_TaggableObjectsCapabilities', {$scope: $scope});

    $scope.getSelectedTaggableObjectRefs = FlowGraphSelection.getSelectedTaggableObjectRefs;

    $scope.computeMovingImpact = function() {
        const computedImpact = [];
        const movingItems = FlowGraphSelection.getSelectedTaggableObjectRefs();

        function addSuccessors(node, original) {
            if (!['RECIPE', 'LABELING_TASK'].includes(node.nodeType)) return;
            node.successors.forEach(function(successor) {
                let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(successor));
                if (original && successor === original.id
                    || movingItems.some(it => it.id === newTaggableObjectRef.id)
                    || computedImpact.some(it => it.id === newTaggableObjectRef.id)
                ) return;
                computedImpact.push(newTaggableObjectRef);
            });
        }
        function computeImpact(node) {
            let predecessor = node.predecessors[0];
            if (predecessor && !['RECIPE', 'LOCAL_SAVEDMODEL', 'LABELING_TASK'].includes(node.nodeType)) {
                let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(predecessor));
                if (computedImpact.some(it => it.id === newTaggableObjectRef.id)) return;
                if (!movingItems.some(it => it.id === newTaggableObjectRef.id)) {
                    computedImpact.push(newTaggableObjectRef);
                }
                addSuccessors(FlowGraph.node(predecessor), node);
            }

            addSuccessors(node);
        }

        FlowGraphSelection.getSelectedNodes().forEach(function(node) {
            let realNode = node.usedByZones.length ? FlowGraph.node(`zone__${node.ownerZone}__${node.realId}`) : node;
            computeImpact(realNode);
        });
        return computedImpact;
    }

    $scope.selectedObjectsZones = FlowGraphSelection.getSelectedNodes().map(n => n.usedByZones[0]);

    $scope.deleteFlowItems = function() {
        WT1.event('flow-context-menu-delete-multi');

        TaggableObjectsService.delete(FlowGraphSelection.getSelectedTaggableObjectRefs())
            .then(FlowGraphSelection.clearSelection);
    };

    $scope.selectSuccessors = function() {
        WT1.event("flow-context-menu-select-successors-multi");
        FlowGraphSelection.getSelectedNodes().forEach(FlowGraphSelection.selectSuccessors);
    };

    $scope.selectPredecessors = function() {
        WT1.event("flow-context-menu-select-predecessors-multi");
        FlowGraphSelection.getSelectedNodes().forEach(FlowGraphSelection.selectPredecessors);
    };

    $scope.startPropagate = function() {
        const items = FlowGraphSelection.getSelectedTaggableObjectRefs().map(r => {
            return {
                projectKey: r.projectKey,
                id: r.id
            }
        });
        $scope.startTool("PROPAGATE_SCHEMA", {"sources": items})
    }

    $scope.havePredecessors = false;
    $scope.haveSuccessors = false;
    $scope.anyLocalDataset = false;
    $scope.anyLocalFolder = false;
    $scope.anyLocalComputable = false;
    $scope.anyRecipe = false;
    $scope.anyNonVirtualizable = false;
    $scope.anyCanSpark = false;
    $scope.anyCanChangeConnection = false;
    $scope.allShareable = true;
    $scope.allUnshareable = true;
    $scope.isAllZonesCollapsed = true;
    $scope.isAllZonesExpanded = true;
    $scope.selectedDatasets = [];
    $scope.selectedRecipes = [];
    $scope.selectedNodes = [];

    $scope.selectedNodes = FlowGraphSelection.getSelectedNodes();

    $scope.selectedNodes.forEach(function(node) {
        if (node.nodeType.startsWith('LOCAL')) {
            $scope.anyLocalComputable = true;
        }
        if (node.nodeType == 'LOCAL_DATASET') {
            $scope.anyLocalDataset = true;
            if (!node.virtualizable) {
                $scope.anyNonVirtualizable = true;
            }
        }
        if (node.nodeType === 'LOCAL_DATASET' || node.nodeType === 'FOREIGN_DATASET') {
            $scope.selectedDatasets.push(node);
        }
        if (node.nodeType == 'LOCAL_MANAGED_FOLDER') {
            $scope.anyLocalFolder = true;
        }
        if (node.nodeType == 'RECIPE') {
            $scope.anyRecipe = true;
            if ($scope.canSpark(node)) {
                $scope.anyCanSpark = true;
            }
            $scope.selectedRecipes.push(node);
        }
        if (node.predecessors.length) {
            $scope.havePredecessors = true;
        }
        if (node.successors.length) {
            $scope.haveSuccessors = true;
        }
        if (["ZONE","RECIPE"].includes(node.nodeType)) {
            $scope.allShareable = false;
        }
        if (!node.usedByZones.length || node.successors.length) {
            $scope.allUnshareable = false;
        }
        if (node.nodeType == "ZONE" && !node.customData.isCollapsed) {
            $scope.isAllZonesCollapsed = false;
        }
        if (node.nodeType == "ZONE" && node.customData.isCollapsed) {
            $scope.isAllZonesExpanded = false;
        }

        $scope.anyCanChangeConnection = $scope.anyCanChangeConnection || $scope.canChangeConnection(node);
    })

    $scope.canInsertRecipeBetween = function() {
        return FlowGraphSelection.canInsertRecipeBetweenSelection($scope.selectedNodes, $scope.selectedDatasets, $scope.selectedRecipes);
    }

    $scope.insertRecipeBetween = function() {
        const selectedDataset = $scope.selectedDatasets[0];
        const selectedRecipes = $scope.selectedRecipes;

        $scope.showInsertRecipeModal(selectedDataset, selectedRecipes)
            .then(selectedRecipe => WT1.event('flow-context-menu-insertrecipe', {position: 'between', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
    }
});

app.service('FlowFilterQueryService', function() {
    const svc = this;

    this.escapeStr = function (string) {
        if (string.includes(' ') || string.includes('"') || string.includes(':')) {
            return `"${string.replace(/"/g, '\\"')}"`
        }
        return string;
    };

    function uiFilterArrayToQueryClause(elements, key) {
        if (!elements) return;
        const resultString = elements.map(el => key + svc.escapeStr(el)).join(' OR ');
        return elements.length > 1 ? `(${resultString})` : resultString;
    }

    this.pickerFormat = "YYYY-MM-DD HH:mm";

    const queryClauseOrNull = (types, type) => types && types.includes(type) ? uiFilterArrayToQueryClause([type], "type:"): null;

    this.uiFilterToQuery = function(structuredFlowObjectFilter) {

        function formatDate(date) {
            return moment(date).format(svc.pickerFormat);
        }

        const creationDate = structuredFlowObjectFilter.customCreationDateRange;
        const modificationDate = structuredFlowObjectFilter.customModificationDateRange;

        let createdRangeClause;
        let modifiedRangeClause;
        if (structuredFlowObjectFilter.creationDateRange) {
            if (structuredFlowObjectFilter.creationDateRange === 'CUSTOM') {
                createdRangeClause = creationDate && creationDate.from && creationDate.to ? `createdBetween:${formatDate(creationDate.from)} / ${formatDate(creationDate.to)}` : null;
            } else {
                createdRangeClause = `created:${structuredFlowObjectFilter.creationDateRange}`;
            }
        }
        if (structuredFlowObjectFilter.modificationDateRange) {
            if (structuredFlowObjectFilter.modificationDateRange === 'CUSTOM') {
                modifiedRangeClause = modificationDate && modificationDate.from && modificationDate.to ? `modifiedBetween:${formatDate(modificationDate.from)} / ${formatDate(modificationDate.to)}` : null;
            } else {
                modifiedRangeClause = `modified:${structuredFlowObjectFilter.modificationDateRange}`;
            }
        }
        const datasetTypeClause = uiFilterArrayToQueryClause(structuredFlowObjectFilter.datasetTypes, "datasetType:");
        const recipeTypeClause = uiFilterArrayToQueryClause(structuredFlowObjectFilter.recipeTypes, "recipeType:");

        const recipeClauseArr = [queryClauseOrNull(structuredFlowObjectFilter.types, 'RECIPE'), recipeTypeClause].filter(e=>e);
        const recipeClause = recipeClauseArr.length > 1 ? `(${recipeClauseArr.join(' AND ')})` : recipeClauseArr.join(' AND ');
        const datasetClauseArr = [queryClauseOrNull(structuredFlowObjectFilter.types, 'DATASET'), datasetTypeClause].filter(e=>e);
        const datasetClause = datasetClauseArr.length > 1 ? `(${datasetClauseArr.join(' AND ')})` : datasetClauseArr.join(' AND ');

        const typeClauses = structuredFlowObjectFilter.types.filter(e => (e !== 'RECIPE' && e !== 'DATASET')).map(e => uiFilterArrayToQueryClause([e], "type:"));
        const typeWithRefinements = [...typeClauses,recipeClause, datasetClause].filter(e=>e);

        let typeWithRefinementClause = typeWithRefinements.join(' OR ');
        if (typeWithRefinements.length > 1){
            typeWithRefinementClause = `(${typeWithRefinementClause})`
        }

        return [
            uiFilterArrayToQueryClause(structuredFlowObjectFilter.tags, "tag:"),
            uiFilterArrayToQueryClause(structuredFlowObjectFilter.creator, "user:"),
            typeWithRefinementClause,
            createdRangeClause,
            modifiedRangeClause
        ].filter(e => e).join(' AND ');
    }
});

})();

;
(function() {
'use strict';

/**
 * This file groups functionalities for selecting and highlighting items in the flow graphs
 * (flow, inter project graph and job preview subgraph)
*/


const app = angular.module('dataiku.flow.project');


app.service('FlowGraphSelection', function($rootScope, FlowGraphFolding, TaggableObjectsUtils, FlowGraph, FlowGraphHighlighting) {
    /*
    * In all this service, "items" are flow items corresponding to graph nodes
    *
    * Several selection strategies (single items selectec at a time, etc)
    * Selection strategies are also responsible for adding highlighting
    * (not for removing it as for now as we always do before calling then)
    *
    * A selection strategy implements:
    * - onItemClick(nodeId, event)
    * - clearSelection()
    */
    const svc = this;

    // contains all selected nodes, including all duplicated objects node if they are in multiple flow-zones
    // however only one node of it has the field 'selected' set to true (later referred as flag-selected)
    // => calling FlowGraphSelection.getSelectedNodes will only get flag-selected nodes, so only one instance of each real item
    // => when applicable, the flag-selected node is the last selected by the user, so that reading its zone gives the zone from which it was selected (not always its source flow-zone)
    let selectedItems = [];

    function _isSelected(item) {
        return selectedItems.includes(item);
    }

    function _hasOnlyZonesOrNone(item) {
        return (_isZone(item) && selectedItems.find(elem => !_isZone(elem)) === undefined) || (!_isZone(item) && selectedItems.find(elem => _isZone(elem)) === undefined);
    }

    /*
    * Return true if all the currently selected objects are zones
    */
    function _hasOnlyZones() {
        return selectedItems.every(_isZone);
    }

    /*
    * Return true if all the currently selected objects are not zones
    */
    function _hasOnlyNonZones() {
        return !selectedItems.some(_isZone);
    }

    function _clearSelection() {
        const element = FlowGraph.getDOMElement();
        if (!element) return; // too early
        selectedItems.forEach(it => it.selected = false);
        FlowGraphHighlighting.removeAllZoneClusterHighlights();
        d3.select('#flow-graph').classed('has-selection', false);
        d3.selectAll('#flow-graph .zone_cluster.selected').attr('style', null);
        d3.selectAll('#flow-graph .selected').classed('selected', false);
        selectedItems = [];
    }

    function _addToSelection(items) {

        items = items.filter(it => !_isSelected(it));
        // For objects used in multiple zones, we mark as flag-selected the nodes just clicked
        // but in case of lasso-selection, it's possible multiple nodes for the same real object have been captured.
        // In this case, we flag-select one node. which one is semi-random but stable (depends on DOM order)
        _.uniqBy(items, it => it.realId).forEach(it => it.selected = true);

        var realIdsSelected = items.map(it => it.realId);
        selectedItems = selectedItems.filter(it => !it.realId || !realIdsSelected.includes(it.realId));
        if (!items.length) {
            return;
        }

        const selector = items.map(it => _isZoned(it) ? `svg [data-node-id="${it.realId}"]` : _isZone(it) ? `svg [id="cluster_${it.id}"]` : `svg [data-id="${it.id}"]`).join(', ');
        d3.selectAll(selector).each(function() {
            const type = this.getAttribute('data-type');
            let key = this.getAttribute('data-id');
            if (type === "ZONE") {
                key = key.replace("cluster_", "");
                FlowGraphHighlighting.highlightZoneCluster(this)
            }
            selectedItems.push(FlowGraph.node(key));
        }).classed("selected", true);

        if (selectedItems.length) {
            $('#flow-graph').addClass('has-selection');
        }
        FlowGraphHighlighting.removeHighlights();
    }

    function _isZoned(item) {
        return item.id && item.realId && item.id !== item.realId;
    }

    function _isZone(item) {
        return item && item.nodeType === 'ZONE';
    }

    function _removeFromSelection(item) {
        const items = _isZoned(item) ? selectedItems.filter(it => it.realId === item.realId) : [item];

        items.forEach(item => {
            item.selected = false;
            const index = selectedItems.indexOf(item);
            if (index > -1) {
                if (_isZone(item)) {
                    d3.select(`g[data-id="${item.id}"]`)[0][0].style = null;
                    FlowGraph.d3ZoneNodeWithId(item.id).classed("selected", false);
                } else {
                    FlowGraph.d3NodeWithId(item.id).classed("selected", false);
                }
                selectedItems.splice(index, 1);
            }
        });

        if (!selectedItems.length) {
            $('#flow-graph').removeClass('has-selection');
        }
    }

    /*
    * Return true if the newly selected object is of a type (zone or non-zoned) different that all currently selected objects
    */
    function _selectedObjectMixTypes(item) {
        if (selectedItems == 0) {
            return false;
        }
        const itemIsZone = _isZone(item);
        return (!itemIsZone && _hasOnlyZones()) || (itemIsZone && _hasOnlyNonZones())
    }

    /**
     * Selects one item.
     * Highlights predecessors and successors.
     */
    const defaultSingleSelectionStrategy = {
        onItemClick: function (item) {
            _clearSelection();
            _addToSelection([item]);
            if (item.filterRemove) {
                return;
            }
            FlowGraphHighlighting.highlightPredecessors(item);
            FlowGraphHighlighting.highlightSuccessors(item);
            if (!_isZone(item)) {
                FlowGraphHighlighting.highlightUsedZonesForSingleItem(item);
            }
            if (_hasOnlyZonesOrNone(item)) {
                FlowGraphHighlighting.highlightZoneElements(item.name);
            }
        }
    }

    /**
     * Selects multiple items at once.
     * Highlights predecessors and successors.
     */
    const defaultMultiSelectionStrategy = {
        onItemClick: function (item, evt) {
            if (_selectedObjectMixTypes(item)) {
                return;
            }
            if (!_isSelected(item)) {
                _addToSelection([item]);
            } else {
                _removeFromSelection(item);
            }
            if (new Set(selectedItems.map(item => item.realId)).size == 1) {
                // only one real element is selected, redirect to single click selection strategy
                return selectionStrategies.SINGLE.onItemClick(selectedItems[0], evt);
            } else {
                svc.recomputeHighlight();
            }
        }
    }

    function recomputeHighlight() {
        FlowGraphHighlighting.removeAllZoneClusterHighlights();
        FlowGraphHighlighting.removeAllEdgesBetweenZonesHighlights();

        const zoneList = selectedItems.filter(_isZone).map(it => it.name);
        zoneList.forEach(it => {
            FlowGraphHighlighting.highlightZoneById(it);
            FlowGraphHighlighting.highlightZoneElements(it);
        });

        const links = d3.selectAll('#flow-graph g');
        links.filter(function() {
            const from = d3.select(this).attr('data-from');
            const to = d3.select(this).attr('data-to');
            if(from === to) return false;  // Early return if they are the same
          
            // Remove the "zone_" prefix
            const fromId = from.replace('zone_', '');
            const toId = to.replace('zone_', '');
          
            // Check that both zone IDs are in the zoneList
            return zoneList.includes(fromId) && zoneList.includes(toId);
        }).classed('highlight', true); // highlight links between selected zones (the links between selected zone gets bolder)

        // Highlight the zone of all selected elements (even when it exists in multiple zones)
        const zoneIdList = selectedItems.filter(_isZoned)
            .map(it => it.id.toString().split('__')[1]);
        const zoneIdSet = new Set(zoneIdList);

        zoneIdSet.forEach(it => {
            FlowGraphHighlighting.highlightZoneById(it);
        });
    }

    svc.recomputeHighlight = recomputeHighlight;

    function withAllPredecessorsOrSuccessorsMultiItemSelectionStrategy(mode) {
        if (mode != 'predecessors' && mode != 'successors') {
            throw new Error("mode should be either 'predecessors' or 'successors'")
        }
        return {
            onItemClick: function(item, evt) {
                // For performance we list and then select them all at once:
                const itemsToSelect = this._listPredecessorsOrSuccessors(item).map(FlowGraph.node);
                _addToSelection(itemsToSelect);
            },

            _listPredecessorsOrSuccessors: function(item, list) {
                list = list || [];

                if (list.includes(item.id)) {
                    // Avoid loops
                    return list;
                }
                list.push(item.id);

                const that = this;

                if (_isZoned(item) && !(mode == "predecessors" && item.usedByZones.length == 0)) {
                    let zoneNode = FlowGraph.node("zone_" + (item.usedByZones[0] || item.ownerZone));
                    if (zoneNode) {
                        $.each(zoneNode[mode], function (index, otherZoneNodeId) {
                            const otherNodeId = graphVizEscape((mode == "predecessors" ? `zone_${item.ownerZone}` : otherZoneNodeId));
                            const otherNode = FlowGraph.node(otherNodeId + "__" + item.realId);
                            if (otherNode) {
                                list = that._listPredecessorsOrSuccessors(otherNode, list);
                            }
                        });
                    }
                }

                $.each(item[mode], function (index, otherNodeId) {
                    const otherNode = FlowGraph.node(otherNodeId);
                    list = that._listPredecessorsOrSuccessors(otherNode, list);
                });
                return list;
            }
        }
    }

    const selectionStrategies = {
        SINGLE: defaultSingleSelectionStrategy,
        MULTI: defaultMultiSelectionStrategy,
        MULTI_WITH_SUCCESSORS: withAllPredecessorsOrSuccessorsMultiItemSelectionStrategy('successors'),
        MULTI_WITH_PREDECESSORS: withAllPredecessorsOrSuccessorsMultiItemSelectionStrategy('predecessors')
    };

    let activeSingleSelectionStrategy = selectionStrategies.SINGLE;
    let activeMultiSelectionStrategy = selectionStrategies.MULTI;

    this.setSingleSelectionStrategy = strategy => {
        activeSingleSelectionStrategy = strategy;
    };

    this.setMultiSelectionStrategy = strategy => {
        activeMultiSelectionStrategy = strategy;
    };

    this.resetSelectionStrategies = () => {
        activeSingleSelectionStrategy = selectionStrategies.SINGLE;
        activeMultiSelectionStrategy = selectionStrategies.MULTI;
    };

    this.onRectangularSelection = function (nodeIds) {
        const nodes = FlowGraph.nodes(nodeIds).filter(node => !node.filterRemove);
        if (new Set(nodes.map(item => item.realId)).size === 1) {
            // only one real element is selected, redirect to single click selection strategy
            activeSingleSelectionStrategy.onItemClick(nodes[0]);
        } else {
            _clearSelection();
            _addToSelection(nodes);
            svc.recomputeHighlight();
        }
        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
    }

    this.onItemClick = function (item, evt) {
        if (evt && (evt.shiftKey || evt.metaKey || evt.ctrlKey)) {
            activeMultiSelectionStrategy.onItemClick(item, evt);
        } else {
            activeSingleSelectionStrategy.onItemClick(item, evt);
        }
        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
        $rootScope.$emit('flowItemClicked', evt, item);
    };

    this.selectNodesFromIds = function (nodeIds) {
        if(!nodeIds || nodeIds.length === 0) {
            return
        }
        _clearSelection();
        const nodes = FlowGraph.nodes(nodeIds).filter(node => angular.isDefined(node));
        if(nodes.length === 1) {
            activeSingleSelectionStrategy.onItemClick(nodes[0]);
        } else {
            nodes.forEach(item => {
                activeMultiSelectionStrategy.onItemClick(item);
            })
        }
        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
        nodes.forEach(item => {
            $rootScope.$emit('flowItemClicked', null, item);
        })
    }

    this.clearSelection = function() {
        if (selectedItems.length) {
            FlowGraphHighlighting.removeHighlights();
            _clearSelection();

            $rootScope.$emit('flowSelectionUpdated');
            $rootScope.$emit('flowDisplayUpdated');
        }
    };

    // We need to call this after a new serialized graph has been fetched and rendered
    this.refreshStyle = function(redoSelection = false) {
        const element = FlowGraph.getDOMElement();
        if (!element) {
            return; //Too early
        }
        d3.selectAll(element.find('svg .selected')).classed("selected", false);
        if (selectedItems.length) {
            // Selected items are nodes, if they are part of the old graph they should be replaced by their new version
            selectedItems = selectedItems.map(it => Object.assign({}, FlowGraph.node(it.id), {selected: it.selected})).filter(x => !!x && x.id);
            const selector = selectedItems.map(it => _isZoned(it) ? `svg [data-node-id="${it.realId}"]` : _isZone(it) ? `svg [id="cluster_${it.id}"]` : `svg [data-id="${it.id}"]`).join(', ');
            if (selector.length) {
                d3.selectAll(selector).each(function() {
                    const type = this.getAttribute('data-type');
                    if (type === "ZONE") {
                        FlowGraphHighlighting.highlightZoneCluster(this)
                    }
                }).classed("selected", true);
            }
            if (redoSelection) {
                const oldSelection = [...selectedItems.filter(it => it.selected)];
                const strategy = oldSelection.length > 1 ? selectionStrategies.MULTI : selectionStrategies.SINGLE;
                oldSelection.forEach(item => {
                    strategy.onItemClick(item); // Deselect in multi
                    if (oldSelection.length > 1) {
                        strategy.onItemClick(item); // Select in multi
                    }
                });
            }
        }
        $rootScope.$emit('flowSelectionUpdated');
    };

    function select(predicate1) {
        return function (predicate) {
            let toSelect = Object.values(FlowGraph.get().nodes);
            if (predicate1) {
                toSelect = toSelect.filter(predicate1);
            }
            if (predicate) {
                toSelect = toSelect.filter(predicate);
            }
            toSelect = toSelect.filter((node) => !FlowGraphFolding.isNodeFolded(node.id));
            _clearSelection();
            _addToSelection(toSelect);
            selectedItems.filter(_isZone).forEach(it => FlowGraphHighlighting.highlightZoneElements(it.name));
            $rootScope.$emit('flowSelectionUpdated');
        };
    }

    /* These functions allow to select items based on a predicate
    * (User code does NOT provide items, they are read from FlowGraph)
     */
    this.select = select();
    this.selectAllByType = indexableType => {
        select(it => !it.filterRemove && TaggableObjectsUtils.getIndexableTypeFromNode(it, it.nodeType) == indexableType)();
    };

    this.filterByIndexableType = function(indexableType) {
        const selectedBefore = selectedItems.length;

        const toRemove = selectedItems.filter(it => TaggableObjectsUtils.getIndexableTypeFromNode(it, it.nodeType) != indexableType);
        toRemove.forEach(_removeFromSelection);

        const selectedAfter = selectedItems.length;
        if (selectedAfter != selectedBefore) {
            $rootScope.$emit('flowSelectionUpdated');
        }
    };

    this.getSelectedNodes = function() {
        return selectedItems.filter(it => it.selected);
    };

    this.getSelectedTaggableObjectRefs = function() {
        return svc.getSelectedNodes().map(TaggableObjectsUtils.fromNode);
    };

    this.selectSuccessors = function (item, evt) {
        $('.select-preview').removeClass('select-preview'); // remove preview styling
        selectionStrategies.MULTI_WITH_SUCCESSORS.onItemClick(item, evt);
        $rootScope.$emit('flowSelectionUpdated');
    };

    this.selectPredecessors = function (item, evt) {
        $('.select-preview').removeClass('select-preview'); // remove preview styling
        selectionStrategies.MULTI_WITH_PREDECESSORS.onItemClick(item, evt);
        $rootScope.$emit('flowSelectionUpdated');
    };

    this.hasPredecessorsInOtherZone = function(item) {
        return item.usedByZones.length > 0;
    };
    this.hasSuccessorsInOtherZone = function(item) {
        let hasSuccessors = false;
        if (_isZoned(item)) {
            let zoneNode = FlowGraph.node("zone_" + item.ownerZone);
            if (zoneNode) {
                $.each(zoneNode["successors"], function (index, otherZoneNodeId) {
                    const otherNode = FlowGraph.node(graphVizEscape(otherZoneNodeId) + "__" + item.realId);
                    if (otherNode) {
                        hasSuccessors = true;
                        return false;
                    }
                });
            }
        }
        return hasSuccessors;
    }

    this.canInsertRecipeBetweenSelection = (selectedNodes, selectedDatasets, selectedRecipes) => {
        if (selectedDatasets.length !== 1   // Only accept one dataset as origin
        || !selectedRecipes.length   // At least one recipe must be selected
        || selectedNodes.length !== selectedDatasets.length + selectedRecipes.length    // Disable insert if any type other than recipe and dataset is selected
        ) return false;        
        const selectedDataset = selectedDatasets[0];
        return selectedRecipes.every(recipe => recipe.predecessors.includes(selectedDataset.id));
    };

    this.selectMulti = function(nodes) {
        _clearSelection();
        _addToSelection(nodes);
        svc.recomputeHighlight();

        d3.selectAll('#flow-graph .node:not(.highlight), #flow-graph .edge:not(.highlight)').classed('fade-out', true);
        $rootScope.$emit('flowSelectionUpdated');
        $rootScope.$emit('flowDisplayUpdated');
    }
});


app.service('FlowGraphHighlighting', function(FlowGraph) {

    function removeHighlights() {
        d3.selectAll('#flow-graph .highlight, #flow-graph .fade-out').classed('highlight', false).classed('fade-out', false);
    }

    function highlightPredecessors(item) {
        let element;
        let nodeElt;
        function _highlightPredecessorsRecursive(nodeType, nodeId) {
            nodeElt = FlowGraph.d3NodeWithIdFromType(nodeId, nodeType);
            if (!nodeElt || !nodeElt.node()) {
                // eslint-disable-next-line no-console
                console.debug('Graph node not found', nodeId)
                return;
            }
            if (nodeElt.classed('filter-remove')) {
                return;
            }

            if (!nodeElt.classed('highlight')) {
                // prevents cycles and dreadful infinite loops
                nodeElt.classed('highlight', true).classed('fade-out', false);
                // highlight nodes
                FlowGraph.rawEdgesWithToId(nodeId).forEach(function (elt) {
                    d3.select(elt).classed('highlight', true).classed('fade-out', false);
                });
                const node = FlowGraph.node(nodeId);
                // highlight original node in other zone and its predecessors
                if (node.usedByZones && node.usedByZones.length) {
                    const originalNodeId = `zone__${node.ownerZone}__${node.realId}`
                    _highlightPredecessorsRecursive(nodeType, originalNodeId);
                }
                // highlight former nodes
                $.each(node.predecessors, function (index, id) {
                    _highlightPredecessorsRecursive(nodeType, id);
                });
            }
        }
        try {
            element = FlowGraph.getDOMElement();
            _highlightPredecessorsRecursive(item.nodeType, item.id);
            d3.selectAll('svg .node:not(.highlight), svg .edge:not(.highlight)').classed('fade-out', true);
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error("Failed to highlight items", nodeElt, e); // NOSONAR: OK to use console.
        }
    }

    function highlightSuccessors(item) {
        let element;
        let nodeElt;
        function _highlightSuccessorsRecursive(nodeType, nodeId, force) {
            nodeElt = FlowGraph.d3NodeWithIdFromType(nodeId, nodeType);
            if (!nodeElt || !nodeElt.node()) {
                // eslint-disable-next-line no-console
                console.debug('Graph node not found', nodeId)
                return;
            }
            if (nodeElt.classed('filter-remove')) {
                return;
            }

            if (force || !nodeElt.classed('highlight')) {
                // prevents cycles and dreadful infinite loops
                nodeElt.classed('highlight', true).classed('fade-out', false);
                // highlight nodes
                FlowGraph.rawEdgesWithFromId(nodeId).forEach(function (elt) {
                    d3.select(elt).classed('highlight', true).classed('fade-out', false);
                });
                
                const node = FlowGraph.node(nodeId);
                // highlight usages in other zones and their successors
                const zonesUsingDataset = FlowGraph.nodeSharedBetweenZones(node);
                if (zonesUsingDataset != null) {
                    zonesUsingDataset.forEach((index, zoneId) => {
                        const nodeIdForZone = `zone__${zoneId}__${node.realId}`
                        _highlightSuccessorsRecursive(nodeType, nodeIdForZone, false);
                    });
                }
                // highlight following nodes
                $.each(node.successors, function (index, successorNodeId) {
                    _highlightSuccessorsRecursive(nodeType, successorNodeId, false);
                });
            }
        }

        try {
            element = FlowGraph.getDOMElement();
            _highlightSuccessorsRecursive(item.nodeType, item.id, true);
            d3.selectAll('svg .node:not(.highlight), svg .edge:not(.highlight)').classed('fade-out', true);
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error("Failed to highlight items", nodeElt, e); // NOSONAR: OK to use console.
        }
    }

    function highlightZoneElements(zoneId) {
        d3.selectAll(`#flow-graph .node:not(.filter-remove)[data-zone-id="${zoneId}"]`)
            .classed('highlight', true).classed('fade-out', false)
            .each(function () { d3.selectAll(`#flow-graph .edge:not(.filter-remove)[data-to="${this.id}"]`)
                    .classed('highlight', true).classed('fade-out', false); });
    }

    function addDropFeedbackToZoneElement(zoneId) {
        d3.selectAll(".zone_cluster.clusterDropFeedback").classed("clusterDropFeedback", false);

        let cluster = d3.select(`g[id="cluster_${zoneId}"]`)[0][0];
        $(cluster).toggleClass('clusterDropFeedback',true);
    }
    function removeDropFeedbackFromZoneElements() {
        d3.selectAll(".zone_cluster.clusterDropFeedback").classed("clusterDropFeedback", false);
    }

    function highlightUsedZonesForSingleItem(item) {
        let zoneIdList = [...FlowGraph.nodeSharedBetweenZones(item) || [], item.ownerZone];
        zoneIdList.forEach(highlightZoneById);
        zoneIdList.filter(id => id !== item.ownerZone).forEach(id => d3.select(`#flow-graph g[data-to="zone_${id}"][data-from="zone_${item.ownerZone}"]`).classed('highlight', true));
    }

    function highlightZoneById(zoneId) {
        const zone = 'zone_' + zoneId;
        highlightZoneCluster(d3.select(`#flow-graph g[id="cluster_${zone}"]`).node());
    }

    function removeAllZoneClusterHighlights() {
        d3.selectAll('#flow-graph .zone_cluster.clusterHighlight').classed("clusterHighlight", false).attr('style', null);
    }

    function removeAllEdgesBetweenZonesHighlights() {
        d3.selectAll(`#flow-graph g.edge[data-from^="zone_"]:not([data-from^="zone__"])`).classed('highlight', false);
    }

    function highlightZoneCluster(cluster, forcedColor) {
        if (!cluster) return;
        let node = FlowGraph.node(cluster.getAttribute("data-id"));
        const color = d3.rgb(forcedColor || node.customData.color);
        let zoneTitleColor = (color.r*0.299 + color.g*0.587 + color.b*0.114) >= 128 ? "#000" : "#FFF"; //black or white depending on the zone color
        $(cluster).toggleClass('clusterHighlight',true);
        cluster.style = `color:${zoneTitleColor};background-color:${color.toString()};stroke:${color.toString()}`
    }

    return {
        removeHighlights,
        highlightPredecessors,
        highlightSuccessors,
        highlightZoneElements,
        highlightUsedZonesForSingleItem,
        highlightZoneById,
        highlightZoneCluster,
        removeAllZoneClusterHighlights,
        removeAllEdgesBetweenZonesHighlights,
        addDropFeedbackToZoneElement,
        removeDropFeedbackFromZoneElements
    };
});


app.directive('highlightDependenciesOnHover', function($rootScope, $timeout, FlowGraphSelection, FlowGraphHighlighting, FlowGraph) {
    return {
        link: function(scope, element) {
            let cur = null;

            const shouldSkipHighlight = (event) => {
                // when the user is moving (zoom, panning) OR some items are selected
                // then we don't highlight dependencies to avoid slowdowns
                const movingFlow = event.currentTarget.closest('#flow-graph > svg.moving');
                return movingFlow || FlowGraphSelection.getSelectedNodes().length;
            }

            element.on('mouseenter', 'svg [class~=node]', function (e) {
                if (shouldSkipHighlight(e)) {
                    return;
                }
                let node = $(this);
                if (cur) $timeout.cancel(cur);
                cur = $timeout(function() {
                    const nodeId = node.attr('data-id');
                    const item = FlowGraph.node(nodeId);
                    if (!item || item.filterRemove) {
                        return;
                    }
                    FlowGraphHighlighting.highlightPredecessors(item);
                    FlowGraphHighlighting.highlightSuccessors(item);
                    cur = null;
                }, 100);
                e.stopPropagation();
            });

            element.on('mouseleave', 'svg [class~=node]', function (e) {
                if (shouldSkipHighlight(e)) {
                    return;
                }
                if (cur) $timeout.cancel(cur);
                FlowGraphHighlighting.removeHighlights();
            });
        }
    };
});

})();

;
(function() {
    'use strict';

    /**
     * This file contains the functionality for folding and unfolding branches of the flow
     * The UI allows you to select any node in the flow (referred to as the rootItem) and fold the branch
     * upstream (FoldDirection.predecessors) or downstream (FoldDirection.successors)
     *
     * The underlying SVG flowgraph produced by GraphViz on the server is not changed when folding.
     * Instead, the folding is achieved by manipulating the browser DOM directly.  Mostly this involves tagging nodes
     * and edges as folded using the CSS classes: folded-node and folded-edge
     *
     * To unfold a node you click on a + symbol alongside folded node.  This symbol is created by inserting
     * SVG elements into the most appropriate edge from the folded node.  The most appropriate edge is currently
     * the line that leaves the node at the most horizontal angle - which is calculated by decomposing the SVG path
     * path that GraphViz created for the line, and doing some dodgy trigonometry to determine the angle of the last
     * section of the path.
     *
     * The elements inserted to form a 'boundary marker' look something like:
     *  <g class="folded-icon" data-folded-node-id="..." data-folded-direction="successors">
     *    <path d="...." class="folded-boundary-marker-line"></path>
     *    <g>
     *        <circle cx="0" cy="0" r="16" class="folded-boundary-marker-circle"></circle>
     *        <path d="M0,-11 V11 M-11,0 H11" stroke="#000000" fill="none" class="folded-boundary-marker-plus"></path>
     *     </g>
     *  </g>
     *
     *  data-folded-node-id indicates the node to be unfolded when the boundary marker clicked.  The direction
     *  indicated by data-folded-direction.
     *
     * Boundary markers are also needed where ever the folded branches have an edge linked to a node outside the branch.
     *
     * We need to flag boundary markers that are hidden when the branch they are in has been folded -
     * these are called 'nested boundary edges' and marked with CSS class folded-nested.
     *
     * To achieve a consistent behaviour when folding and unfolding nodes in different orders, and when refreshing
     * the page, we remember the state as a sequence of fold requests.  An unfold command simply removes a previous
     * fold command from the state.  This fold state is held in the GraphZoomTrackerService.
     *
     * The processing of a fold/unfold command is made as follows:
     * 1.  Update the fold state held in the GraphZoomTrackerService.
     * 2.  Retrieve the full fold state from the service. This consists entirely of 'fold' commands - there are no
     *     'unfold' commands.
     * 3.  Build a complete logical view of the fold state of the flow by looping through each fold command (FoldCommand)
     *     in the fold state and calling buildFoldReqst to calculate the impact of the command.  This logical view is
     *     held in the FoldReqst object.
     * 4.  The FoldReqst object is passed to applyFoldReqstToDom, which make the necessary
     *     DOM changes based on the contents of the object via D3.
     * 5.  Since the FoldReqst object tracks the active elements, we need to also be able to remove previous DOM
     *     updates that are unnecessary (i.e. we 'reshow items').  This appears to be slightly beyond what D3 can achieve
     *     (happy to hear to the contrary!), so we keep track of which DOM elements we have updated in the FoldDomStatus
     *     object. Each new FoldReqst object is applied to the FoldDomStatus object, and a list of elements to 'reshow'
     *     is generated.
     * 6.  The list of items to reshow is processed via reShowFoldedDomItems.
     *
     * The previewing of folding, unfolding and branch selection uses much the same sequences of steps, but is
     * lighter-weight since it only changes the CSS classes on nodes and edges.  It does not change boundary markers.
     *
     * * NB: Anytime the doc mentions an 'activated' node, it means that the node has been manipulated by the flow_folding system
     * (to hide it or to change its appearance in the case of boundaryEdges)
     *
     */

    const app = angular.module('dataiku.flow.project');

    const FoldDirections = {
        predecessors: 'predecessors',
        successors: 'successors',
        getOpposite(val) {
            return (val == this.predecessors ? this.successors : this.predecessors);
        }
    };

    const FoldActions = {
        fold: 'fold',
        unfold: 'unfold'
    };

    app.service('FlowGraphFolding', function ($rootScope, WT1, TaggableObjectsUtils, FlowGraph, FlowGraphHighlighting, GraphZoomTrackerService, $timeout, LoggerProvider) {

        const Logger = LoggerProvider.getLogger('FlowGraphFolding');

        const svc = this;

        let foldedNodeCount = 0; // tracks the fold status to drive a 'hidden items' count on the flow view
        let activeFoldDomStatus; // maps of every DOM manipulation we've map so far

        const FoldDomStatusMapIdx = {
            node: 0,
            edge: 1,
            nestedBoundaryEdge: 2,
            boundaryEdge: 3
        }

        function getSelectorByIdFromList (list) {
            const sel = list.map(
                id => `svg [id="${id}"]`).join(', ');
            return sel == "" ? [] : sel;
        }

        let FoldDomStatus = function(){
            return {
                // Four maps for: nodes, edges, nestedBoundaryEdges, boundaryEdges
                maps:[{},{},{},{}],

                /**
                 * Takes the new list of ids which are part of the complete fold and
                 * compares these with the 'FoldDomStatus' map of currently 'activated' DOM ids.
                 * The function updates the 'FoldDomStatus' map of DOM ids which are 'activated',
                 * and returns a list of ids to be deactivated.
                 *
                 * @param newIdsList - array of Ids now be 'activated'
                 * @param mapIdx - 0, 1, 2, or 3 depending on which type of item we want to update (0 to update the nodes map, 1 for the edges one, etc.)
                 * @returns a list of existing activated DOM items to be de-
                 */
                updateStatusList: function (newIdsList, mapIdx) {

                    // flag all the items which need to be kept, so we can see which don't
                    const existingItemsMap = this.maps[mapIdx];
                    newIdsList.forEach(id => {
                        existingItemsMap[id] = true; // we need to keep this entry, or we add new entry
                    });

                    let newItemsMap = {};
                    let deactivateList = [];

                    Object.keys(existingItemsMap).forEach(id => {
                        if (existingItemsMap[id]) {
                            newItemsMap[id] = false;
                        }
                        else  {
                            deactivateList.push(id);
                        }
                    });

                    this.maps[mapIdx] = newItemsMap; // update map with new items
                    return deactivateList; // return list of items to deactivate
                },

                /**
                 * Update all the FoldDomStatus' maps of acticate DOM elements.
                 * @param listofListOfIds - array of four lists of IDs, corresponding to the four maps held in FoldDomStatus.
                 *                          This param represents the new foldState to be applied
                 * @returns - array of fours lists containing Ids of DOM items to be 'deactivated' - loosely speaking 'unfolded'
                 */
                update: function (listofListOfIds) {
                    const unFoldInfo = [];
                    listofListOfIds.forEach((list, i) => unFoldInfo.push(this.updateStatusList(list, i)));
                    return unFoldInfo;
                },

                previewStatusList: function (newIdsList, mapIdx) {

                    // flag all the items which need to be kept, so we can see which don't
                    const existingItemsMap = this.maps[mapIdx];
                    const differencesMap = angular.copy(existingItemsMap);

                    // flag all matching items as true, add missing items as false.
                    // result is all differences are marked false
                    newIdsList.forEach(id => {
                        if (differencesMap.hasOwnProperty(id)){
                            differencesMap[id] = true; // we need to keep this entry,
                        }
                         else {
                            differencesMap[id] = false; // new entry
                        }
                    });

                    let differencesList = [];

                    Object.keys(differencesMap).forEach(id => {
                        if (!differencesMap[id]) {
                            differencesList.push(id);
                        }
                    });

                    return differencesList;
                },

                getPreviewItems : function (listofListOfIds) {
                    const itemsToActivate = [];
                    listofListOfIds.forEach((list, i) => itemsToActivate.push(this.previewStatusList(list, i)));
                    return itemsToActivate;
                }
            };
        }

        /**
         * FoldReqst - returns the structure that defines the DOM operations required to perform a sequence of
         * fold commands
         * @param rootItem: the item which is being folded to unfolded
         * @param direction: a value of FoldDirections i.e. upstream (predecessors) or downstream (successors)
         * @returns a structure describing who are the DOM elements that need to be manipulated
         *
         */
        let FoldReqst = function (rootItem, direction, isUseCssTransitions) {
            return {
                direction: direction,
                rootItem: rootItem,
                idsToRemainShownList: [],       // list of ids of items that we do not want to hide with the folding
                                                // e.g. an item we are zooming to, or on the hidden end of a +  sign
                                                // we are trying to unfold
                nodeIdMap: {},                  // id=>element map of all the node elements being folded/unfolded. Used to detect loops
                nodeEls: [],                    // array of all the node elements being folded/unfolded

                edgeEls: [],                    // array of edge elements to fold/unfold
                edgeIdMap: {},
                branchEdgeIdMap: {},            // id=>element map of all the edges in the core branch being manipulated. Used to validate potential boundary markers

                boundaryEdges: [],              // array of edges (Flow item structures not SCG elements) which are boundary makers for the fold
                boundaryEdgeMap: {},
                boundaryEdgeMapByNode: {},

                nestedBoundaryEdgeIds: [],      // boundary edges from earlier folds that must now be hidden
                nestedBoundaryEdgeIdMap: {},
 
                isUseCssTransitions: isUseCssTransitions,

                copyFoldStateData: function(from, to) {
                    ['nodeIdMap', 'edgeIdMap', 'branchEdgeIdMap', 'boundaryEdgeMap', 'boundaryEdgeMapByNode', 'nestedBoundaryEdgeIdMap'].forEach(key => {
                        to[key] = Object.assign({}, from[key]);
                    });

                    ['nodeEls', 'edgeEls', 'boundaryEdges', 'nestedBoundaryEdgeIds'].forEach(key => {
                        to[key] = from[key].slice();
                    });

                    return to;
                },

                backupFoldState: function() {
                    this.foldStateBackup  = this.copyFoldStateData(this, {});
                },

                restoreFoldState: function() {
                    this.copyFoldStateData(this.foldStateBackup, this);
                },

                getRootNode: function () {
                    return FlowGraph.rawNodeWithId(this.rootItem.Id);
                },

                getBoundaryEdgeDescr: function (edgeId) {
                    return this.boundaryEdgeMap.hasOwnProperty(edgeId) ? this.boundaryEdgeMap[edgeId] : undefined;
                },

                getBoundaryEdges: function () {
                    return this.boundaryEdges.map(it => it.el);
                },

                getNestedBoundaryEdgesSelector: function () {
                    return getSelectorByIdFromList(this.nestedBoundaryEdgeIds);
                },

                isNodeFolded: function (nodeId) {
                    return this.nodeIdMap.hasOwnProperty(nodeId);
                },

                isEdgeFolded: function (edgeId) {
                    return this.edgeIdMap.hasOwnProperty(edgeId);
                },

                isBoundaryEdge: function (edgeId) {
                    return this.boundaryEdgeMap.hasOwnProperty(edgeId);
                },

                addNode: function(el, item) {
                    if (this.nodeIdMap[el.id]) return;

                    this.idsToRemainShownList.forEach(idToRemainShown => {
                        if (idToRemainShown == item.id) {
                            this.isItemToRemainShownWillBeHidden = true;
                        }
                    });

                    this.nodeEls.push(el);
                    this.nodeIdMap[item.id] = item;
                },

                addEdge: function(el, isMainBranch) {
                    if (this.edgeIdMap[el.id]) return;

                    this.edgeEls.push(el);
                    this.edgeIdMap[el.id] = el;
                    if (isMainBranch) this.branchEdgeIdMap[el.id] = el;
                },

                addBoundaryEdge: function(newEdgeDescr) {
                    const mapKey = newEdgeDescr.boundaryNodeId + this.rootItem.id + this.direction;

                    if (this.boundaryEdgeMapByNode.hasOwnProperty(mapKey)) return;

                    newEdgeDescr.rootItemId = this.rootItem.id;
                    newEdgeDescr.direction = this.direction;
                    this.boundaryEdgeMap[newEdgeDescr.el.id] = newEdgeDescr;
                    this.boundaryEdgeMapByNode[mapKey] = newEdgeDescr;
                    this.boundaryEdges.push(newEdgeDescr);

                    // these should always be hidden edges
                    this.addEdge(newEdgeDescr.el);
                },

                addNestedBoundaryEdge: function(edge) {
                    if (this.nestedBoundaryEdgeIds.hasOwnProperty(edge.id)) return;

                    this.nestedBoundaryEdgeIds.push(edge.id);
                    this.nestedBoundaryEdgeIdMap[edge.id] = edge;
                },

                isEdgeNeedsProcessing: function(edge) {
                    return !this.edgeIdMap[edge.id];
                }
            };
        }

        /**
         * Fold command - a structure to describe a fold or unfold operation.
         * An array of these are saved in the GraphZoomTrackerService to enable the fold statue to be restored
         */
        const FoldCommand = function (nodeId, direction, action) {
            return {
                nodeId: nodeId,
                direction: direction,
                action: action
            }
        }

        /**
         * lineAnalyser - a set of functions for processing edge (SVG path) information and help us determine the new
         * path we will add from the node to the boundary marker circle.
         */
        const lineAnalyser = {

            /**
             * addXyDeltasToLineInfo: enrich a line description which is in global co-ordinates to
             * give the relative change in X and Y co-ordinates (dX, dY)
             *
             * @param lineInfo: the line description to be enriched
             * @returns enriched lineInfo with dX and dY
             */
            addXyDeltasToLineInfo: function (lineInfo) {
                lineInfo.dX = lineInfo.endPoint.x - lineInfo.ctlPoint.x;
                lineInfo.dY = lineInfo.endPoint.y - lineInfo.ctlPoint.y;
                return lineInfo;
            },

            /**
             * addTrigToLineInfo: enrich a line description (dX, dY calculated already) to contain the length
             * if the line (hypot) and the angle of the line (angle) in radians
             *
             * @param lineInfo: the line description to be enriched
             * @returns enriched lineInfo with hypot and angle
             */
            addTrigToLineInfo: function (lineInfo) {
                this.addXyDeltasToLineInfo(lineInfo);
                lineInfo.hypot = Math.sqrt(Math.pow(lineInfo.dX, 2) + Math.pow(lineInfo.dY, 2));
                lineInfo.angle = Math.asin(lineInfo.dY / lineInfo.hypot);
                return lineInfo;
            },

            /**
             * findIconCentre: calculate the centre of the + sign in boundary marker.
             *
             *
             * @param lineInfo: a fully enriched line description
             * @param distXFromEnd: how far the centre should be from the end of the line along the X axis.
             * @returns an {x,y) co-ord structure
             */
            findIconCentre: function (lineInfo, distXFromEnd) {
                // some shockingly sloppy geometry.  This is not proper trig but near enough for a short line
                const scalingFactor = distXFromEnd / lineInfo.hypot;
                return {
                    x: lineInfo.endPoint.x - lineInfo.dX * scalingFactor,
                    y: lineInfo.endPoint.y - lineInfo.dY * scalingFactor
                };
            },

            /**
             * extendEndPoint: extend the length of a line that will be a boundary marker
             *
             * We need to do this because the line path we base our marker on doesn't actually go all the way to the
             * node at the end with the 'arrow'.  It stops shorts to give room for said arrow (actually a circle).
             *
             * @param lineInfo: a fully enriched line description
             * @param extendBy: how far you extend the line by
             * @returns lineInfo enriched with endPoint co-ordinate record
             */
            extendEndPoint: function (lineInfo, extendBy) {
                lineInfo.endPoint.x = lineInfo.endPoint.x + extendBy * Math.cos(lineInfo.angle);
                lineInfo.endPoint.y = lineInfo.endPoint.y + extendBy * Math.sin(lineInfo.angle);
                return lineInfo;
            },

            /**
             * extractFoldingInfoFromEdgePath: extract a description of the last section of the SVG path that
             * represents a curved edge between nodes.  We use this line as the basis for the edge from the node to the
             * boundary marker circle.  Ultimately a new path will be added from the node to the boundary circle which
             * uses the same start point and angle as the line we extract here.
             *
             * @param el: the edge SVG element
             * @param direction: which end of the curve we want to look at
             * @returns lineInfo fully enriched with trig info
             */
            extractFoldingInfoFromEdgePath: function (el, direction) {
                let info = {};

                const dAttr = el.firstElementChild.getAttribute("d");

                //extract end point and direct from the end of a Bezier cubic curve definition
                // PATH d attribute has format:
                //  d="M<start-x>,<start-y>C<ctl-point-x1>,<ctl-point-y1> <ctl-point-x2>,<ctl-point-y2> ... <end-x><end-y>"

                let tokens = dAttr.split(/[ MC]/);
                if (tokens.length < 2) return undefined;

                function buildCoord(s) { //expect string of format 123456,456789,
                    const tokens = s.split(",");
                    return (tokens.length > 1) ? {x: parseInt(tokens[0], 10), y: parseInt(tokens[1], 10)} : undefined;
                }

                if (direction == FoldDirections.successors)
                    info = {endPoint: buildCoord(tokens[1]), ctlPoint: buildCoord(tokens[2])};
                else
                    info = {endPoint: buildCoord(tokens.pop()), ctlPoint: buildCoord(tokens.pop())};

                return this.addTrigToLineInfo(info);
            },

            /**
             * initBoundaryEdgeData = the externally-called function to build the description of the boundary marker
             * @param el: the SVG element which is the edge we are aligning our boundary marker with.
             * @param direction: which end of the edge lement we want to put the boundar marker
             * @returns a structure which defines the new path we will create for the boundary marker
             */
            initBoundaryEdgeData: function (el, direction) {
                const distPlusIconFromEnd = 60; // how far the + icon is from the end of the line.
                const arrowTipWidth = 5; // the arrow tip is actually a circle.

                //get the first PATH statement, and then extract end point and last ctl-point.
                let info = this.extractFoldingInfoFromEdgePath(el, direction);

                if (direction != FoldDirections.successors) info = this.extendEndPoint(info, arrowTipWidth); //extend lines when folding upstream to account for 'arrow' circle on end of line

                info.iconCentre = this.findIconCentre(info, distPlusIconFromEnd);
                return info;
            },

            /**
             * getEdgeLineAngleForEl:calculate an indicator of the steepness of the line, ignoring direction
             * @param el: the edge SVG element to be analyzed
             * @param direction: the end of the edge we are interested in
             * @returns the modulus of the angle ie. ignoring its sign
             */
             getEdgeLineAngleForEl: function (el, direction) {
                let info = this.extractFoldingInfoFromEdgePath(el, direction);
                return info.angle < 0 ? -info.angle : info.angle;
            }
        };


        /**
         * applyPreviewCssChangesToDom
         * Apply CSS class to all highlight all nodes and edges affected by the preview
         * @param previewItemsInfo - array of lists of DSS ids of items beinb previewed.
         * @param previewClass - class to apply to DOM objects being preview
         */
        function applyPreviewCssChangesToDom(previewItemsInfo, previewClass) {
            d3.selectAll(previewItemsInfo[FoldDomStatusMapIdx.node].map(id => FlowGraph.rawNodeWithId(id)))  // id values for nodes are actually data-id= values
                .classed(previewClass, true);

            [FoldDomStatusMapIdx.edge, FoldDomStatusMapIdx.nestedBoundaryEdge].forEach( i =>
                d3.selectAll(getSelectorByIdFromList(previewItemsInfo[i]))
                    .classed(previewClass, true)
            )
        }

        function removeAllPreviewStyling() {
            $('.fold-preview').removeClass('fold-preview');
            $('.unfold-preview').removeClass('unfold-preview');
            $('.select-preview').removeClass('select-preview');
        }

        /**
         * applyPreviewFoldReqstToDom
         * Action all the DOM changes for a preview
         * @param previewFoldReqst - FoldReqst structure for preview
         * @param previewClass - CSS class to be applied
         */
        function applyPreviewFoldReqstToDom (previewFoldReqst, previewClass) {

            const itemsToPreview = activeFoldDomStatus.getPreviewItems(
                [previewFoldReqst.nodeEls.map(el => el.getAttribute('data-id')),
                 previewFoldReqst.edgeEls.map(el => el.id),
                 previewFoldReqst.nestedBoundaryEdgeIds]
            );
            applyPreviewCssChangesToDom(itemsToPreview, previewClass);
        }

        /**
         * reShowFoldedDomItems
         * Remove CSS and other DOM changes previously applied but no longer needed
         * @param itemsToReShow - array of id lists for items affected
         */
        function reShowFoldedDomItems(itemsToReShow) {

            d3.selectAll(itemsToReShow[FoldDomStatusMapIdx.node].map(id => FlowGraph.rawNodeWithId(id))).classed('folded-node', false); //node Ids are actually data-ids, which are slow to select.  use element lookup instead
            d3.selectAll(getSelectorByIdFromList(itemsToReShow[FoldDomStatusMapIdx.edge])).classed('folded-edge', false);
            d3.selectAll(getSelectorByIdFromList(itemsToReShow[FoldDomStatusMapIdx.nestedBoundaryEdge])).classed('folded-nested', false);

            d3.selectAll(getSelectorByIdFromList(itemsToReShow[FoldDomStatusMapIdx.boundaryEdge]))
                .classed('folded-boundary-edge', false)
                .attr('data-folded-boundary-node-id', null)
                .select('g.folded-icon')
                    .remove();  // the folded-icon group containing + sign
        }

        /**
         * applyFoldReqstToDom: use a fully built FoldReqst structure to make
         * the changes to the DOM necessary to execute a fold command.
         * @param foldReqst: a built FoldReqst structure
         */
        function applyFoldReqstToDom(foldReqst) {
            const radiusPlusIcon = 16; // radius of + icon
            const lenPlusArm = 11; // length of each arm of plus path

            if (!activeFoldDomStatus) activeFoldDomStatus = new FoldDomStatus();  // this remembers our DOM updates

            removeAllPreviewStyling();

            // hide nodes in branch
            d3.selectAll(foldReqst.nodeEls)
                .classed('folded-node', true)
                .classed('fold-transition', foldReqst.isUseCssTransitions);

            // hide edges in branch
            d3.selectAll(foldReqst.edgeEls)
                .classed('folded-edge', true)
                .classed('fold-transition', foldReqst.isUseCssTransitions);

            // hide nested boundary markers
            d3.selectAll(foldReqst.getNestedBoundaryEdgesSelector())
                .classed('folded-nested', true)
                .classed('fold-transition', foldReqst.isUseCssTransitions);

            // sort out hidden edges going to other nodes
            let boundaryEdges = foldReqst.boundaryEdges.map(item => {
                item.data = lineAnalyser.initBoundaryEdgeData(item.el, item.direction);
                return item
            });

            // find boundary edges that haven't been treated yet and add '+' indicator
            let boundaryEdgesSelectionNew =
                d3.selectAll(foldReqst.getBoundaryEdges())
                .filter(":not(.folded-boundary-edge)")
                    .data(boundaryEdges, function (d) {
                        return d ? d.el.id : this.id;
                    });

            // existing boundary makers need to bew processed - their rootItem data sometimes changes
            let boundaryEdgesSelectionExisting =
                d3.selectAll(foldReqst.getBoundaryEdges())
                    .filter(".folded-boundary-edge")
                    .data(boundaryEdges, function (d) {
                        return d ? d.el.id : this.id;
                    });

            boundaryEdgesSelectionNew
                .classed('folded-boundary-edge', true)
                .attr('data-folded-boundary-node-id', d => d.boundaryNodeId);

            //make sure existing boundary markers are updated
            boundaryEdgesSelectionExisting.select("g.folded-icon")  // .select forces the propagation of the updated data binding to the children
                .attr('data-folded-node-id', d => d.rootItemId)
                .attr('data-folded-direction', d => d.direction)
                .attr('data-folded-hidden-node-id', d => d.hiddenNodeId); // the hidden node on the end of the edge

            //create marker DOM elements for new ones
            //bundle it all in a <g> for easy removal
            const boundaryEdgeMarker = boundaryEdgesSelectionNew
                .append('g')
                .classed('folded-icon', true)
                .attr('data-folded-node-id', d => d.rootItemId)
                .attr('data-folded-direction', d => d.direction)
                .attr('data-folded-hidden-node-id', d => d.hiddenNodeId);

            // add the short line to the boundary marker
            const drawLine = d3.svg.line()
                .x(function (d) {
                    return d.x;
                })
                .y(function (d) {
                    return d.y;
                })
                .interpolate('linear');

            boundaryEdgeMarker
                .append('path')
                .attr('d', d => drawLine([d.data.iconCentre, d.data.endPoint]))
                .attr('stroke', '#000000')
                .attr('fill', 'none')
                .classed('folded-boundary-marker-line', true);

            // add a <g> to hold the plus-in-a-circle - mainly so we can
            // transform the co-ordinates to make centering the contents trivial
            const boundaryEdgeMarkerIconG = boundaryEdgeMarker
                .append('g')
                .attr('transform', d => {
                    return 'translate(' + d.data.iconCentre.x + ',' + d.data.iconCentre.y + ')'
                })

            // add a boundary marker circle
            boundaryEdgeMarkerIconG
                .append('circle')
                .attr('cx', 0)
                .attr('cy', 0)
                .attr('r', radiusPlusIcon)
                .attr('class', 'folded-boundary-marker-circle')
                .on('mouseover', d => {
                    const elG = $(d.el).find("g.folded-icon");
                    return svc.previewUnfold(FlowGraph.node(elG.attr('data-folded-node-id')), elG.attr('data-folded-direction'), elG.attr('data-folded-hidden-node-id'));
                    // by pulling the attributes dynamically, it is easier to handle boundarymarkers that change their rootItemId.
                })
                .on('mouseleave', d => svc.endPreviewBranch());

            // add the + sign as a <path>
            boundaryEdgeMarkerIconG
                .append('path')
                .attr("d", `M0,-${lenPlusArm} V${lenPlusArm} M-${lenPlusArm},0 H${lenPlusArm}`)
                .attr('stroke', '#000000')
                .attr('fill', 'none')
                .classed('folded-boundary-marker-plus', true);

            // re-show all the DOM stuff we no longer want hidden
            const itemsToReShow = activeFoldDomStatus.update(
                [foldReqst.nodeEls.map(el => el.getAttribute('data-id')),
                foldReqst.edgeEls.map(el => el.id),
                foldReqst.nestedBoundaryEdgeIds,
                foldReqst.boundaryEdges.map(item => item.el.id)]
            );

            reShowFoldedDomItems(itemsToReShow);

            // zap the CSS transition classes once we are done with them
            $timeout(_ => {
                $('.fold-transition').removeClass('fold-transition');
            }, 2000);

        }

        function getFuncEdgesAwayFromRoot (direction) {
            // edges on the side of the node further from the root item
            return direction == FoldDirections.successors ? FlowGraph.rawEdgesWithFromId : FlowGraph.rawEdgesWithToId;
        }

        function getFuncEdgesTowardRoot (direction) {
            // edges on the side of the node nearer the root item
            return direction == FoldDirections.successors ? FlowGraph.rawEdgesWithToId : FlowGraph.rawEdgesWithFromId;
        }

        function isEdgeSuitableAsBoundaryMarker(foldReqst, edge) {
            const existingDescr = foldReqst.getBoundaryEdgeDescr(edge.id);
            return !existingDescr || existingDescr.rootItemId == foldReqst.rootItem.id;
        }

        function getDirectionForEdgeEl(foldReqst, edge) {
            const existingDescr = foldReqst.getBoundaryEdgeDescr(edge.id);
            return existingDescr ? existingDescr.direction : undefined;
        }

        function updateMostLevelEdge(edge, mostLevelEdge, direction) {
            edge.angle = lineAnalyser.getEdgeLineAngleForEl(edge, direction);
            if (!mostLevelEdge || mostLevelEdge.angle > edge.angle) {
                mostLevelEdge = edge;
            }
            return mostLevelEdge;
        }

        /**
         * buildFoldReqst - build a FoldReqst structure for a fold command
         * This is the real engine of the folding logic.  It loops through elements starting form the rootItem in a
         * recursive pattern following the same pattern as the FlowSelection logic. However, it is made much more
         * complicated because of:
         * a) Nested folding
         * b) The need to define boundary markers for every element in the flow that loses an input or output edge
         *    due to the folding.
         * c) The ability to unfold in different order from the folding.
         * d) Loops in the graph
         *
         * The processing of a fold and unfold are broadly similarly, with some specific logic where necesssary.
         *
         * @param foldReqst: a single FoldReqst structure being built up throughout the recursive calls
         * @param item: item to process.  If undefined, then assume use the rootItem. Is set for the 'next' item in a recursive call
         * @param idsToRemainShownList: list of ids of items that we do not want hidden.  We flag if this foldReqst would hide an item
         * @returns the completed FoldReqst structure
         */

        function buildFoldReqst(foldReqst, item, idsToRemainShownList) {

            let isRootItem = false;
            if (typeof item === 'undefined') {
                item = foldReqst.rootItem;
                isRootItem = true;
            }

            foldReqst.idsToRemainShownList = idsToRemainShownList || [];

            let inValidContext = (!item || !item.id);
            inValidContext = inValidContext || (foldReqst.nodeIdMap && foldReqst.nodeIdMap.hasOwnProperty(item.id)); //reprocessing non-rootItem
            inValidContext = inValidContext || (!isRootItem && foldReqst.rootItem.id == item.id); //reprocessing rootItem - usually a looping flow

            if (inValidContext) {
                return foldReqst;
            }

            const fEdgesAwayFromRoot = getFuncEdgesAwayFromRoot(foldReqst.direction); // edges on the side of the node further from the root item
            const fEdgesTowardRoot = getFuncEdgesTowardRoot(foldReqst.direction); // edges on the side of the node nearer the root item
            const boundaryEdgeNodeAttr = 'data-' + (foldReqst.direction == FoldDirections.successors ? 'from' : 'to');
            const boundaryEdgeHiddenNodeAttr = 'data-' + (foldReqst.direction == FoldDirections.successors ? 'to' : 'from');
            const oppositeDirection = FoldDirections.getOpposite(foldReqst.direction);

            if (isRootItem) {
                const rootEdgesArray = fEdgesAwayFromRoot(foldReqst.rootItem.id);
                let mostLevelEdge;
                foldReqst.isChangeMadeForCommand = false;

                rootEdgesArray.forEach(
                    edge => {
                        if (foldReqst.isEdgeNeedsProcessing(edge)) {
                            foldReqst.isChangeMadeForCommand = true;
                            foldReqst.addEdge(edge);

                            //fold - we only want a single boundary marker added per node
                            // The most horizontal line looks visually tidiest
                            // but with unfold-strategy=rootitem we need to find an edge that is not already a boundary marker.
                            if (isEdgeSuitableAsBoundaryMarker(foldReqst, edge)) {
                                mostLevelEdge = updateMostLevelEdge(edge, mostLevelEdge, foldReqst.direction)
                            }
                        }
                    }
                );

                if (mostLevelEdge) {
                    foldReqst.addBoundaryEdge({el: mostLevelEdge, data: {},
                        boundaryNodeId: foldReqst.rootItem.id,
                        hiddenNodeId: mostLevelEdge.getAttribute(boundaryEdgeHiddenNodeAttr),

                    });
                }
            }
            else {
                //non-root item on the branch
                foldReqst.addNode(FlowGraph.rawNodeWithId(item.id), item);

                // we want boundary markers on all nodes that are left hanging: side branches of this branch
                fEdgesTowardRoot(item.id).forEach(
                    edge => {

                        const remoteNodeId = edge.getAttribute(boundaryEdgeNodeAttr);
                        const isBoundaryMarkerForNodeAlreadyBeingAdded =
                                    foldReqst.boundaryEdges.find(boundaryEdge =>
                                            boundaryEdge.el.id != edge.id &&
                                            boundaryEdge.boundaryNodeId == remoteNodeId &&
                                            boundaryEdge.direction == foldReqst.direction);

                        if (!isBoundaryMarkerForNodeAlreadyBeingAdded) {
                            //case of boundary marker pointing back upstream that need to be nested/hidden
                            if (getDirectionForEdgeEl(foldReqst, edge) == oppositeDirection){
                                foldReqst.addNestedBoundaryEdge(edge);
                            }

                            if (!foldReqst.isEdgeFolded(edge.id)) { //don't show new boundaries for edges already hidden
                                foldReqst.addBoundaryEdge({el: edge, data: {},
                                    boundaryNodeId: remoteNodeId,
                                    hiddenNodeId: item.id
                                });
                            }
                        }
                        else {
                            // we have a second edge on a boundary node.  We only want one edge to show the + sign,
                            // so we don't add to boundary edges list, but we still need to hide the edge so we
                            // add to edges list.  This is not an edge on the main branch though, so we don't add to
                            // branchEdgeIdMap
                            foldReqst.addEdge(edge);
                        }
                    });

                let isItemFolded = foldReqst.isNodeFolded(item.id);
                fEdgesAwayFromRoot(item.id).forEach( // edges on the branch being folded
                    edge => {
                        // we need to hide any boundary markers now nested in this fold.
                        if (foldReqst.isBoundaryEdge(edge.id) || isItemFolded) { //<<<<< extra conditional test only
                            foldReqst.addNestedBoundaryEdge(edge);
                        }

                        foldReqst.addEdge(edge, true);
                    });
            }

            $.each(item[foldReqst.direction], function (index, otherNodeId) {
                if (!foldReqst.isNodeFolded(otherNodeId)) {
                    const otherNode = FlowGraph.node(otherNodeId);
                    if (otherNode) foldReqst = buildFoldReqst(foldReqst, otherNode, idsToRemainShownList);
                }
            });

            // boundary edges cannot be edges inside the folded branch
            foldReqst.boundaryEdges = foldReqst.boundaryEdges.filter(item => !foldReqst.branchEdgeIdMap.hasOwnProperty(item.el.id))

            return foldReqst;
        }

        /**
         * tidyFoldState - ensure we clear down the restore state if things do awry.
         */
        function tidyFoldState(foldReqst) {
            const allFoldedNodes = $('.folded-node');
            foldedNodeCount = allFoldedNodes.length;

            if (foldedNodeCount==0) {
                GraphZoomTrackerService.resetFoldState();
            } else if (foldReqst && !foldReqst.isChangeMadeForCommand) {
                // check if the last command did nothing.  If so, we remove from state list
                // This can happened when build close alread-yclosed nodes and would lead to
                // unfolds that appear to do nothing.
                GraphZoomTrackerService.removeLastFoldCommand();
            }
        }

        /**
         * cleanRestoreState
         * Takes the restore state and removes any fold commands that reference flow items that don't exist anymore
         * @param commands - list of fold commands
         * @returns {*}
         */
        function cleanRestoreState(commands) {
            let origLen = commands.length;

            commands = commands.filter(cmd => !!FlowGraph.node(cmd.nodeId))
            if (origLen!=commands.length) {
                GraphZoomTrackerService.resetFoldState(commands);
            }
            return commands;
        }

        function applyFoldCommandToReqst (foldReqst, foldCommand, idsToRemainShownList) {
            // build total action into single foldReqst to apply at once
            const rootItem = FlowGraph.node(foldCommand.nodeId);

            foldReqst.rootItem = rootItem;
            foldReqst.direction = foldCommand.direction;
            foldReqst.action = foldCommand.action;
            foldReqst.isItemToRemainShownWillBeHidden = false;

            return buildFoldReqst(foldReqst, undefined, idsToRemainShownList);
        }

        /**
         * buildFoldReqstForCompleteState
         * Create a FoldReqst object that represents the complete sequence of fold commands.
         * This is called for previews, page refreshes, and user-driven folding / unfolding
         * @param commands - list of fold commands
         * @param isInteractiveReqst - if is a user-driven fold change, rather than a page refresh.  This determines if
         *        a CSS transition is used
         * @returns {FoldReqst}
         */
        function buildFoldReqstForCompleteState(commands, isInteractiveReqst, idsToRemainShownList) {
            let foldReqst = new FoldReqst(null, FoldDirections.successors, isInteractiveReqst);
            let foldCmdsApplied = [];
            let isCmdSkipped = false;

            cleanRestoreState(commands).forEach(cmd => {
                foldReqst.backupFoldState();
                foldReqst = applyFoldCommandToReqst (foldReqst, cmd, idsToRemainShownList);

                if (foldReqst.isItemToRemainShownWillBeHidden) {
                    foldReqst.restoreFoldState()
                    isCmdSkipped = true;
                }
                else {
                    foldCmdsApplied.push(cmd)
                }
            });

            if (isCmdSkipped) foldReqst.revisedFoldCmds = foldCmdsApplied;
            return foldReqst;
        }

        /**
         * applyFoldState
         * Apply the foldstate to the DOM.
         * This is called for pages refreshes and and user-driven folding / unfolding, but not previews
         * @param commands - the fold state
         * @param isInteractiveReqst - if is a user-driven fold change, rather than a page refresh.  This determines if
         *        a CSS transition is used
         */
        function applyFoldState(commands, isInteractiveReqst, idsToRemainShownList) {
            const foldReqst = buildFoldReqstForCompleteState(commands, isInteractiveReqst, idsToRemainShownList);

            applyFoldReqstToDom(foldReqst);
            $rootScope.$emit('flowSelectionUpdated');

            if (foldReqst.revisedFoldCmds) GraphZoomTrackerService.resetFoldState(foldReqst.revisedFoldCmds)
            tidyFoldState(foldReqst);
        }

        /**
         * foldMultiItems - action a fold command
         * @param rootItem: the item being folded/unfolded
         * @param direction: a value of FoldDirections, successor or predecessor
         * @param action: a value of FoldActions, fold or unfold
         * @param idsToRemainShownList: list of ids node that must be visible after unfolding.  For example, when you press
         *          a + you want something to shown on the end of that edge or it seems like nothing happened!
         */
        function foldMultiItems(rootItem, direction, action, idsToRemainShownList) {
            trackFoldCommand(rootItem.id, direction, action);
            applyFoldState(GraphZoomTrackerService.getFoldState(), true, idsToRemainShownList);
        }

        /**
         * trackFoldCommand - Update the GraphZoomTrackerService's list of active fold commands
         * @param nodeId - id of rootItem
         * @param direction: a value of FoldDirections, successor or predecessor
         * @param action: a value of FoldActions, fold or unfold
         */
        function trackFoldCommand(nodeId, direction, action) {
            Logger.debug(action.toString().toUpperCase() + " - " + direction.toUpperCase() + " " + nodeId);

            GraphZoomTrackerService.setFoldCommand(new FoldCommand(nodeId, direction, action));
        }

        // We only want to restore the fold status when the SVG graph has been reloaded.
        let foldStateRestored = false;
        $rootScope.$on('graphRendered', function() {
            foldStateRestored = false;
        });

        /* public methods */

        /**
         * unfoldNode: Unfold a folded node
         * @param unfoldEl: the boundary marker element (circle with a +) to unfold
         * */
        this.unfoldNode = function (unFoldEl) {
            const nodeId = unFoldEl.getAttribute('data-folded-node-id');
            const direction = unFoldEl.getAttribute('data-folded-direction');
            const idMustBeShown = unFoldEl.getAttribute('data-folded-hidden-node-id');
            foldMultiItems(FlowGraph.node(nodeId), direction, FoldActions.unfold, [idMustBeShown, nodeId]);
        };

        /**
         * foldSuccessors: fold a node downstream
         * @param item: a DSS flow data object (not a DOM element)
         */
        this.foldSuccessors = function (item) {
            foldMultiItems(item, FoldDirections.successors, FoldActions.fold);
        };

        /**
         * foldPredecessors: fold a node upstream
         * @param item: a DSS flow data object (not a DOM element)
         */
        this.foldPredecessors = function (item) {
            foldMultiItems(item, FoldDirections.predecessors, FoldActions.fold);
        };

        /**
         * previewSelect
         * Highlights the nodes / edges that will be affected by a 'Select all upstream/downstream' operation
         */
        this.previewSelect = function (item, direction) {
            const previewSelectReqst = buildFoldReqst(new FoldReqst(item, direction, true));
            if (previewSelectReqst) applyPreviewFoldReqstToDom(previewSelectReqst, 'select-preview');
        };

        /**
         * previewFoldOrUnfoldAction
         * Highlights the nodes / edges that will be affected by a 'Hide all upstream/downstream' operation
         * or display the nodes that will reappear when clicking on a boundaryEdgeMarkerIcon
         */
        function previewFoldOrUnfoldAction(item, direction, action, idsMustRemainShownList) {
            const previewFoldState = GraphZoomTrackerService.getPreviewFoldState(new FoldCommand(item.id, direction, action));
            const previewFoldReqst = buildFoldReqstForCompleteState(previewFoldState, true, idsMustRemainShownList);
            if (previewFoldReqst) applyPreviewFoldReqstToDom(previewFoldReqst, action + '-preview');
        }

        /**
         * previewFold
         * Highlights the nodes / edges that will be affected by a 'Hide all upstream/downstream' operation
         */
        this.previewFold = function (item, direction) {
            previewFoldOrUnfoldAction(item, direction, FoldActions.fold);
        };

        /**
         * previewUnfold
         * Display the nodes that will reappear when clicking on a boundaryEdgeMarkerIcon
         * For intuitive results, we need to ensure both the node we are unfolding appears incases it is nested in a
         * fold, and that the node on the end of the edge being unfolded appears, else it can seem like nothing
         * happened.
         */
        this.previewUnfold = function (item, direction, idsMustRemainShownList) {
            previewFoldOrUnfoldAction(item, direction, FoldActions.unfold, [idsMustRemainShownList, item.id]);
        };

        this.endPreviewBranch = function () {
            removeAllPreviewStyling();
        };

        /**
         * restoreState: restore the active sequence of fold/unfold commands
         * We only need to do this when the SVG graph is reloaded, not on
         * all resizes, but we need the node maps created by the graph resize
         *  to have been prepared, hence we trigger form the resize, but
         *  only action if we have not restored the state since the last
         *  draw_graph call.
         * @param commands: array of FoldCommands
         */
        this.restoreState = function (commands) {
            if (!foldStateRestored && commands) {
                applyFoldState(commands, false);
                foldStateRestored = true;
            }
        };

        this.clearFoldState = function() {
            GraphZoomTrackerService.resetFoldState();
            foldStateRestored = false;
        };

        this.unfoldAll = function() {
            GraphZoomTrackerService.resetFoldState();
            $rootScope.$emit('drawGraph', true);
            tidyFoldState();
        }

        /**
         * ensureNodeNotFolded
         * called when the flow view tries to focus on an item. We need the item to be visible.
         * We re-apply the fold state, but specify the id is to remain shown, resulting in and
         * fold commands that contradict this are removed from the fold state.
         * @param idsToRemainShownList - list of ids of items that needs to be visible
         */
        //
        this.ensureNodesNotFolded = function (idsToRemainShownList) {
            applyFoldState(GraphZoomTrackerService.getFoldState(), true, idsToRemainShownList);
        }

        this.isNodeFolded = function(nodeId) {
            return activeFoldDomStatus
                && activeFoldDomStatus.maps[FoldDomStatusMapIdx.node].hasOwnProperty(nodeId);
        }

        this.getFoldedNodeCount = function () {
            return foldedNodeCount;
        }

    })

})();

;
(function() {
'use strict';

/**
* Search function in main flow
*/

var app = angular.module('dataiku.flow.project');


app.directive('flowSearchPopover', function($stateParams, $rootScope, ContextualMenu, ListFilter, DataikuAPI, StateUtils, FlowGraphSelection) {
    return {
        restrict : 'A',
        scope : true,
        templateUrl : '/templates/flow-editor/search-popover.html',

        link : function($scope, element, attrs) {
            /************************** SHOW / Hide logic ******************* */

            function hide() {
                //$scope.removeHighlights();
                element.hide();
                $("html").unbind("click", hide);
                shown=false;
                $scope.shown = false;
            };

            function show() {
                shown = true;
                $scope.shown = true;
                $(".flow-search-popover", element).css("left", $("#flow-search-input input").offset().left);
                $(element).show();
                element.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    // Let "a" flow
                    if ($(e.target).parents(".directlink").length) return;
                    e.stopPropagation();
                });
                $("#flow-search-input").off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    e.stopPropagation();
                });
                window.setTimeout(function() { $("html").on("click.dku-pop-over", hide); }, 0);
            }

            $scope.pommef = function() {
                $("#flow-search-input").focus();
            };

            $scope.hidePopover = function() {
                if (shown) {
                    hide();
                }
            };

            var shown = false;
            $(element).hide();

            $scope.$on("$destroy", function() {
                $("html").off("click.dku-pop-over", hide);
            })
            $scope.$watch("flowSearch.pattern", function(nv, ov) {
                if (shown && (!nv || nv.length === 0)) {
                    hide();
                }
                if (!shown && nv && nv.length > 0) {
                    show();
                }
            });
            $("#flow-search-input input").on("focus", function() {
                if (!shown && $scope.flowSearch && $scope.flowSearch.pattern.length) {
                    show();
                }
            });

            /************************** Execution ******************* */

            $scope.$watch("flowSearch.pattern", function() {
                $scope.onFlowSearchQueryChange();
            });

            var extraneous = [];
            var formerPattern;

            function isInZone(node, zoneId) {
                if (!zoneId) {
                    return true;
                }
                return node.id.startsWith(`zone__${zoneId}`)
            }
            const isInOwnerZone = node => node && node.ownerZone && node.ownerZone !== "" && node.id.startsWith('zone') && node.id.startsWith(`zone__${node.ownerZone}__`);
            const getNode = (realId, nodes, zoneId) => {
                let node = nodes[realId];
                if (node) {
                    return node;
                }
                for (let key in nodes) {
                    node = nodes[key];
                    if (node.realId === realId && (!zoneId ? isInOwnerZone(node) : isInZone(node, zoneId))) {
                        return node;
                    }
                }
                return undefined;
            }
            $scope.onFlowSearchQueryChange = function() {
                if (!$scope.flowSearch) return;
                function getDatasets() {
                    if (!filteredDatasets) return [];
                    //First get local datasets
                    var results = ListFilter.filter(filteredDatasets.items, $scope.flowSearch.pattern);
                    $.map(results, function(item) {
                        item.nodeType = "LOCAL_DATASET";
                        const potentialId = graphVizEscape("dataset_" + item.projectKey + "." + item.name);
                        const foundNode = getNode(potentialId, $scope.nodesGraph.nodes, $stateParams.zoneId);
                        item.id = foundNode ? foundNode.id : potentialId;
                    });
                    results = results.filter(node => isInZone(node, $stateParams.zoneId));
                    // Add foreign datasets
                    results = results.concat(getItemsFromGraph("FOREIGN_DATASET"));
                    return results;
                }

                function getItemsFromGraph(type) {
                    if (!$scope.nodesGraph) return [];
                    var result = [];
                    for (var key in $scope.nodesGraph.nodes) {
                        const node = $scope.nodesGraph.nodes[key];
                        if (node.nodeType.endsWith(type) && isInZone(node, $stateParams.zoneId)) {
                            if (isInOwnerZone(node, $stateParams.zoneId) || node.ownerZone === undefined) {
                                // No ownerZone when the graph does not have zones
                                result.push(node);
                            }
                        }
                    }
                    return ListFilter.filter(result, $scope.flowSearch.pattern);
                }

                if(formerPattern != $scope.flowSearch.pattern) {
                    formerPattern = $scope.flowSearch.pattern;
                    const datasets = getDatasets(),
                        labelingTasks = getItemsFromGraph("LABELING_TASK"),
                        recipes = getItemsFromGraph("RECIPE"),
                        folders = getItemsFromGraph("FOLDER"),
                        models = getItemsFromGraph("MODEL"),
                        mes = getItemsFromGraph("MODELEVALUATIONSTORE"),
                        gaes = getItemsFromGraph("GENAIEVALUATIONSTORE"),
                        zones = getItemsFromGraph("ZONE").map(item => Object.assign(item, {zoneId: item.id.split('_').splice(1).join('')}));
                    $scope.flowSearch.nbDatasets = datasets.length;
                    $scope.flowSearch.nbRecipes = recipes.length;
                    $scope.flowSearch.nbFolders = folders.length;
                    $scope.flowSearch.nbModels = models.length;
                    $scope.flowSearch.nbLabelingTasks = labelingTasks.length;
                    $scope.flowSearch.nbMes = mes.length;
                    $scope.flowSearch.nbGaes = gaes.length;
                    $scope.flowSearch.items = datasets.concat(recipes).concat(folders).concat(models).concat(mes).concat(zones).concat(labelingTasks).sort(function(a,b) {
                        var aIsGood = a.name.startsWith($scope.flowSearch.pattern.toLowerCase())?'0':'1';
                        var bIsGood = b.name.startsWith($scope.flowSearch.pattern.toLowerCase())?'0':'1';
                        return (aIsGood + a.name).localeCompare(bIsGood + b.name);
                    });
                }
                $scope.flowSearch.index = -1;
                $scope.currentlyDisplayedItems = 20;
            };

            $scope.currentlyDisplayedItems = 20;
            $scope.loadMoreItems = function() {
                $scope.currentlyDisplayedItems += 20;
            };

            let filteredDatasets;
            DataikuAPI.datasets.listHeads($stateParams.projectKey, $rootScope.tagFilter || {}, false).success(function(data){
                filteredDatasets = data;
            }).error(setErrorInScope.bind($scope));

            /*************************** Navigation **************** */

            $scope.flowSearchSelectPrevious = function($event) {
                if (!shown || !$scope.flowSearch.items.length) {
                    return;
                }
                $scope.flowSearchSelectIndex(Math.max(0, $scope.flowSearch.index-1));
                if ($event) $event.stopPropagation();
                const el = $("li", element)[$scope.flowSearch.index];
                if (el) {
                    const parent = $("ul", element).parent();
                    ensureVisible(el, parent);
                }
            };

            $scope.flowSearchSelectNext = function($event) {
                if (!shown || !$scope.flowSearch.items.length) {
                    return;
                }
                $scope.flowSearchSelectIndex(Math.min($scope.flowSearch.items.length - 1, $scope.flowSearch.index+1));
                if ($event) $event.stopPropagation();
                const el = $("li", element)[$scope.flowSearch.index];
                if (el) {
                    const parent = $("ul", element).parent();
                    ensureVisible(el, parent);
                }
            };

            $scope.flowSearchSelectIndex = function(index) {
                if (!shown) return;
                if (!$scope.nodesGraph || !$scope.nodesGraph.nodes) return;
                $scope.flowSearch.index = index;
                /* Highlight on selection */
                //$scope.removeHighlights();
                const item = $scope.flowSearch.items[$scope.flowSearch.index]
                const id = item.id;
                FlowGraphSelection.clearSelection($scope.nodesGraph.nodes[id]);
                FlowGraphSelection.onItemClick($scope.nodesGraph.nodes[id], null);
                $scope.zoomGraph(id, item.nodeType=="RECIPE" ? 5 : 3, item); //recipe nodes don't have names, so bbox ends up smaller
            };

            $scope.flowSearchGo = function() {
                if (!shown) return;
                if ($scope.flowSearch.index < 0) {
                    //No match
                    return;
                }
                if ($scope.flowSearch.items.length) {
                    var item = $scope.flowSearch.items[$scope.flowSearch.index];
                    StateUtils.go.node(item);
                    $scope.flowSearch.pattern = "";
                    hide();
                    $("#flow-search-input").blur();
                }
            };

            $scope.contextMenu = function(idx, $event) {
                var x = $event.pageX;
                var y = $event.pageY;
                var newScope = $scope.$new();
                var item = $scope.flowSearch.items[idx];
                newScope.object = angular.copy(item);
                var menuParams = {
                    scope: newScope,
                    template: "/templates/flow-editor/dataset-contextual-menu.html"
                };
                var menu = new ContextualMenu(menuParams);
                menu.openAtXY(x, y);
            };

            $scope.onFlowSearchQueryChange();
        }
    };
});


})();


;
(function() {
    'use strict';

    var app = angular.module('dataiku.flow.project');

    /**
     * Preview function in main flow
     */
    app.directive('flowPreview', function($rootScope, $timeout, $stateParams, translate, FlowGraphSelection, Logger) {
        return {
            restrict: 'AE',
            scope: true,
            templateUrl: '/templates/flow-editor/flow-preview.html',
            controller: function($scope, WT1) {
                $scope.previewContextProjectKey = $stateParams.projectKey;
                updatePreview(); // initialize the preview when it got added to the DOM
                const deregisterFlowSelectionUpdated = $rootScope.$on('flowSelectionUpdated', () => {
                    const selectedNodes = FlowGraphSelection.getSelectedNodes();
                    if (selectedNodes && selectedNodes.length === 1 && selectedNodes[0]
                        && $scope.previewProjectKey && $scope.previewDatasetName
                        && selectedNodes[0].projectKey === $scope.previewProjectKey && selectedNodes[0].name === $scope.previewDatasetName) {
                        return; // Selection has not actually changed. Do nothing.
                    }
                    // Disable the preview before triggering the preview update to clear the view
                    // before the loading spinner
                    $scope.showPreviewTable = false;
                    // async execution allowing the view to refresh before updating the preview
                    // Use $timeout because setTimeout happens outside of Angular
                    // Using $timeout it will handle $apply for you to kick off a $digest cycle to refresh the view based on dirtiness check
                    $timeout(updatePreview);
                }); // subsequent changes
                $scope.$on('$destroy', deregisterFlowSelectionUpdated);

                // When the toggle is pressed then the flag gets updated. Watching `hasPreview` is the same as watching
                // the toggle button click event.
                // We need to update the preview whenever the toggle button is clicked and it opens up the preview pane.
                $scope.$watch('hasPreview', updatePreview);
                $scope.reloadPreview = () => {
                    $scope.$broadcast('refresh-preview-table-without-cache');
                }

                function updatePreview() {
                    // Don't update the preview when the preview pane is not active. It will affect the performance.
                    if (!$scope.hasPreview) return;
                    const selectedNodes = FlowGraphSelection.getSelectedNodes();
                    let problem, solution;
                    if (selectedNodes.length === 0) {
                        problem = translate("PROJECT.FLOW.PREVIEW.NO_DATESET_SELECTED", "No dataset selected");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_DATASET_TO_PREVIEW", "Click a dataset to preview its content");
                    } else if (selectedNodes.length > 1) {
                        problem = translate("PROJECT.FLOW.PREVIEW.MULTIPLE_ITEMS_SELECTED", "Multiple items selected");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_SINGLE_DATASET_TO_PREVIEW", "Click a single dataset to preview its content");
                    } else if (selectedNodes[0].nodeType && !selectedNodes[0].nodeType.includes('DATASET')) {
                        problem = translate("PROJECT.FLOW.PREVIEW.NOT_DATASET_SELECTED", "Selected item is not a dataset");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_DATASET_TO_PREVIEW", "Click a dataset to preview its content");
                    } else if (selectedNodes[0].datasetType === 'hiveserver2') {
                        problem = translate("PROJECT.FLOW.PREVIEW.HIVE_NOT_SUPPORTED", "Preview is not supported for Hive tables");
                        solution = "";
                    } else if (selectedNodes[0].neverBuilt) {
                        problem = translate("PROJECT.FLOW.PREVIEW.DATASET_NOT_BUILT", "Selected dataset is not built");
                        solution = translate("PROJECT.FLOW.PREVIEW.CLICK_BUILT_DATASET_TO_PREVIEW", "Click a built dataset to preview its content");
                    }

                    if (!problem && !solution) {
                        // All good, trigger the actual load of the preview data
                        // This will trigger ctrl.$onChanges of datasetPreviewTable since it updates the props of the component
                        $scope.previewProjectKey = selectedNodes[0].projectKey;
                        $scope.previewDatasetName = selectedNodes[0].name;
                        // Show the preview table now that everything is OK.
                        $scope.showPreviewTable = true;
                        $scope.previewProblem = null;
                        $scope.previewSolution = null;
                    
                        try {
                            // A quick check to make sure that the WT1 event won't break the app
                            if (selectedNodes[0]) {
                                WT1.event('flow-preview-update',
                                    {
                                        nodeType: selectedNodes[0].nodeType,
                                        datasetType: selectedNodes[0].datasetType,
                                    }
                                );
                            }
                        } catch (e) {
                            Logger.error('Failed to report flow dataset preview event', e);
                        }
                    } else {
                        $scope.previewProjectKey = null;
                        $scope.previewDatasetName = null;
                        $scope.previewProblem = problem;
                        $scope.previewSolution = solution;
                    }
                }
            }
        };
    });
})();
;
(function() {
    'use strict';

    angular.module('dataiku.flow.graph').factory('PageSpecificTourService', pageSpecificTourService);

    function pageSpecificTourService($rootScope, $timeout, translate, $filter, FlowGraphSelection, Dialogs, ContextualMenu, TopbarDrawersService, TOPBAR_DRAWER_IDS, WT1, QuestionnaireService, OpalsService, DataikuAPI, ProfileService) {

        const BUILT_DATASET_SELECTOR = 'g[data-type="LOCAL_DATASET"] > g:not(.never-built-computable)';

        /** Tours **/

        const TOUR_NAMES = {
            FLOW: 'flow',
            TUTORIAL: 'tutorial', // Flow Tour variation when coming from the questionnaire
            EXPLORE: 'explore',
            PREPARE: 'prepare'
        };

        function getTourCompletionSetting(tourName) {
            switch(tourName) {
                case TOUR_NAMES.FLOW:
                case TOUR_NAMES.TUTORIAL:
                    return $rootScope.appConfig?.pageSpecificTourSettings?.flowTourCompleted;
                case TOUR_NAMES.EXPLORE:
                    return $rootScope.appConfig?.pageSpecificTourSettings?.exploreTourCompleted;
                case TOUR_NAMES.PREPARE:
                    return $rootScope.appConfig?.pageSpecificTourSettings?.prepareTourCompleted;
            }
        }

        function setTourCompletionSetting(tourName, value) {
            switch(tourName) {
                case TOUR_NAMES.FLOW:
                case TOUR_NAMES.TUTORIAL:
                    $rootScope.appConfig.pageSpecificTourSettings.flowTourCompleted = value;
                    break;
                case TOUR_NAMES.EXPLORE:
                    $rootScope.appConfig.pageSpecificTourSettings.exploreTourCompleted = value;
                    break;
                case TOUR_NAMES.PREPARE:
                    $rootScope.appConfig.pageSpecificTourSettings.prepareTourCompleted = value;
                    break;
            }
        }

        /** Conditions to display the Flow Tour - exposed so we can check whether to open
         *  the Help Center on the Tutorial page when coming from the questionnaire
         */
        function checkFlowTourConditions(fromContext) {
            if (!canStartFlowTour()) {
                return Promise.resolve(false); 
            }

            return checkContextAllowTour(TOUR_NAMES.FLOW, fromContext);
        }

        function startFlowTour({
            scope,
            fromContext
        }) {
            return checkFlowTourConditions(fromContext)
            .then((tourNeedToStart) => {
                if (!tourNeedToStart) {
                    return;
                }

                TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).hide(true);

                const isFromQuestionnaire = QuestionnaireService.isFromQuestionnaire();
                const tourName = isFromQuestionnaire ? TOUR_NAMES.TUTORIAL : TOUR_NAMES.FLOW;
                const introJSInstance = initialTourSetup(scope, tourName, fromContext);

                const builtDatasets = $(BUILT_DATASET_SELECTOR);
                const builtDataset = builtDatasets[0].parentNode;
                let builtDatasetNodeId = builtDataset.id;

                // Common steps for the tour when coming both from the Questionnaire or not
                const commonSteps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.DATASET.TITLE', 'Everything in Dataiku begins with data'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.DATASET.DESCRIPTION', 'Datasets are always represented as blue squares on the Flow.'),
                        beforeChange: function() {
                            hideHighlightDuringTransition(introJSInstance, 400);
                            this.element = document.querySelector(BUILT_DATASET_SELECTOR);
                            this.position = 'right';
                            scope.zoomGraph(builtDatasetNodeId, 5);
                            FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[builtDatasetNodeId]); // select the item for the right panel and preview
                            if (scope.standardizedSidePanel.opened) {
                                // if the right panel is already opened, make sure the actions tab is opened
                                scope.standardizedSidePanel.toggleTab('actions');
                            }
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.RIGHT_PANEL.TITLE', 'Start building your data pipeline from the right panel'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.RIGHT_PANEL.DESCRIPTION', 'The right panel is the main command center of the Flow.<br/><br/>It enables you to view information about your data, use pre-built transformation tools (known as Recipes), create ML models and more.'),
                        element: '[data-page-tour="right-panel"]',
                        position: 'left',
                        beforeChange: function() {
                            if (!scope.standardizedSidePanel.opened) {
                                // open right panel on actions tab
                                scope.standardizedSidePanel.toggleTab('actions');
                                hideHighlightDuringTransition(introJSInstance, 300);
                            }
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.PREVIEW.TITLE', 'View a preview of your data'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.PREVIEW.DESCRIPTION', 'Quickly get a snapshot of your dataset in the preview panel.<br/><br/>You can minimize and reopen the preview tab at any time by clicking the "Hide Preview" button.'),
                        element: '[data-page-tour="preview-button"]',
                        position: 'top',
                        beforeChange: function() {
                            if (!scope.hasPreview) {
                                scope.togglePreview();
                                scope.$apply();
                                hideHighlightDuringTransition(introJSInstance, 400);
                            }
                                setTimeout(() => {
                                    // highlight the preview pane too
                                    $('[data-page-tour="preview-pane"]').addClass('introjs-showElement');
                                }, 0);
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.HOME_BUTTON.TITLE', 'Click the Home button to go to the home page'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.HOME_BUTTON.DESCRIPTION', 'View and create new projects, workspaces and more from the homepage.'),
                        element: '[data-page-tour="navbar-home-button"]',
                        position: 'bottom',
                        beforeChange: function() {
                            // close the preview panel
                            scope.togglePreview();
                            scope.$apply();
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.TOP_NAV.TITLE', 'All your work, accessible in one place'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.TOP_NAV.DESCRIPTION', 'The navigation bar is essential to moving around the different areas of your Dataiku project.<br/><br/>Simply hover your cursor on a menu item to see a full list of the available functionalities.'),
                        element: '[data-page-tour="project-menus"]',
                        position: 'bottom'
                    }
                ];

                // Extra steps the user will see when doing only the Flow Tour (not coming from the questionnaire)
                const tourOnlySteps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.INTRO.TITLE', 'Take a quick tour of the Flow?'),
                        intro: `<img src="static/dataiku/images/flow/flow-animation.gif"/>` + translate('PAGE_SPECIFIC_TOUR.FLOW.INTRO.DESCRIPTION', 'Learn how to build your data pipeline and navigate through the items in your project.'),
                        hideStepNumber: true,
                    },
                    ...commonSteps,
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.OUTRO.TITLE', '✅ You\'ve finished the Flow tour'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.OUTRO.DESCRIPTION', 'That\'s all for now. You can re-activate the Flow tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]',
                        position: 'bottom'
                    },
                ];

                if (fromContext === 'opals') {
                    tourOnlySteps.shift(); // no need to display the first step to the user since they launched the tour
                }
                if (isFromQuestionnaire) {
                    introJSInstance.setOptions({
                        nextLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next"),
                        doneLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next")
                    });
                    introJSInstance.afterexit = () => {
                        // if the user finished the tour or exited before the end, show the user the Tutorial
                        displayTutorialPopup({ scope });
                    }
                }

                introJSInstance.setOptions({
                    steps: isFromQuestionnaire ? commonSteps : tourOnlySteps,
                });

                $timeout(() => {
                    introJSInstance.start();
                }, 100); // give some time for the DOM to fully load
            });
        }

        function displayTutorialPopup({
            scope,
        }) {
            const tutorialPage = QuestionnaireService.getOpalsPage();
            const tutorialStartingStep = QuestionnaireService.getTutorialStartingStep();
            if (!tutorialPage) {
                return; // don't know which tutorial to show, return
            }
            if (scope.standardizedSidePanel.opened) {
                scope.standardizedSidePanel.slidePanel(); // close right panel
            }
            OpalsService.navigateToAndShowDrawer(tutorialPage, { number : tutorialStartingStep, numCompletedTasks: tutorialStartingStep - 1 });

            const introJSInstance = initialPopupSetup();

            introJSInstance.setOptions({
                steps: [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.FLOW.TUTORIAL.TITLE', 'Continue your tutorial here'),
                        intro: translate('PAGE_SPECIFIC_TOUR.FLOW.TUTORIAL.DESCRIPTION', 'Follow the detailed step-by-step guideline to learn how to use Dataiku'),
                        hideStepNumber: true,
                        element: 'opals-help-center-container',
                        position: 'left'
                    }
                ]
            });

            $timeout(() => {
                introJSInstance.start();
            }, 0);
        }

        function displayReopenTutorialPopup() {
            if (!$rootScope.appConfig.opalsEnabled) {
                return; // the Help Center is deactivated, do not start the Help Center Tour
            }
            if (!$rootScope.appConfig.onboardingExperience) {
                return; // Onboarding Experience is disabled, do not start the Help Center Tour
            }
            if (ProfileService.isTechnicalAccount()) {
                return; // do not show the Tour to technical accounts
            }

            const introJSInstance = initialPopupSetup();

            introJSInstance.setOptions({
                steps: [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.TUTORIAL.INTRO.TITLE', 'You can re-open the tutorial from here'),
                        intro: translate('PAGE_SPECIFIC_TOUR.TUTORIAL.INTRO.DESCRIPTION', 'Open the Help Center with this button and go to \'Onboarding\' to find all the quick tutorials for your use case'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]'
                    },
                ]
            });

            $timeout(() => {
                introJSInstance.start();
            }, 0); // give some time for the DOM to fully load
        }

        function startExploreTour({
            scope,
            fromContext
        }) {
            if (!canStartExploreTour(scope)) {
                return Promise.resolve();  
            }

            return checkContextAllowTour(TOUR_NAMES.EXPLORE, fromContext)
            .then((tourNeedToStart) => {
                if (!tourNeedToStart) {
                    return;
                }

                const introJSInstance = initialTourSetup(scope, TOUR_NAMES.EXPLORE, fromContext);

                const steps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.INTRO.TITLE', 'Take a quick tour of the Explore page ?'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.INTRO.DESCRIPTION', 'Learn how to analyze your data, adjust the sample, create charts and more..'),
                        hideStepNumber: true
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.SAMPLE.TITLE', 'Configure your dataset sample'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.SAMPLE.DESCRIPTION', 'For datasets larger than 10,000 rows, Dataiku shows only a sample of a dataset to enable efficient data exploration.<br/><br/>Use the sample settings panel to configure the sampling method, number of records and more.'),
                        element: '[data-page-tour="sampling-badge"]',
                        position: 'bottom'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.COLUMN.TITLE', 'Instantly check your column characteristics'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.COLUMN.DESCRIPTION', 'Beneath each column name, you will see that Dataiku automatically detects the storage type (shown in grey), the meaning (shown in blue) and data validity (red/green bar).'),
                        beforeChange: function() {
                            // set the element now that it's rendered
                            this.element = document.querySelector('[data-page-tour="column-header"]');
                            this.position = "right";
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.ANALYZE.TITLE', 'Analyze your columns at a glance'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.ANALYZE.DESCRIPTION', 'Click on the <i>Analyze</i> option in the drop-down menu to get some quick descriptive statistics on the contents of your column.'),
                        beforeChange: function() {
                            document.querySelector('[data-page-tour="column-name"]').click(); // open the column dropdown
                            // set the element now that it's rendered
                            this.element = document.querySelector('[data-page-tour="analyze-column"]');
                            this.position = "right";
                            closeContextualMenuOnlyOnNextButtonClick(scope);
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.CHARTS.TITLE', 'Visualize your data in a couple of clicks'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.CHARTS.DESCRIPTION', 'Visit the Charts tab to start visualizing your data with our drag-and-drop chart building interface.'),
                        element: '[data-page-tour="tab-charts"]',
                        position: 'bottom'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.EXPLORE.OUTRO.TITLE', '✅ You\'ve finished the Explore tour'),
                        intro: translate('PAGE_SPECIFIC_TOUR.EXPLORE.OUTRO.DESCRIPTION', 'That\'s all for now. You can re-activate the Explore tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]',
                        position: 'bottom'
                    }
                ];

                if (fromContext === 'opals') {
                    steps.shift(); // no need to display the first step to the user since they launched the tour
                }

                introJSInstance.setOptions({
                    helperElementPadding: 5,
                    steps: steps
                });

                $timeout(() => {
                    introJSInstance.start();
                }, 0); // give some time for the DOM to fully load
            });
        }

        function startPrepareTour({
            scope,
            fromContext
        }) {
            if (!canStartPrepareTour(scope)) {
                return Promise.resolve();  
            }
            return checkContextAllowTour(TOUR_NAMES.PREPARE, fromContext)
            .then((tourNeedToStart) => {
               if (!tourNeedToStart) {
                return;
            }

                const introJSInstance = initialTourSetup(scope, TOUR_NAMES.PREPARE, fromContext);

                const steps = [
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.INTRO.TITLE', 'Take a quick tour of the Prepare recipe?'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.INTRO.DESCRIPTION', 'Learn how to cleanse, normalize and enrich your data in visual and interactive way.'),
                        hideStepNumber: true,
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.NEW_STEP.TITLE', 'Add steps to your prepare script'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.NEW_STEP.DESCRIPTION', 'Access hundreds of pre-made processors such as filtering rows, rounding numbers, splitting columns and more.'),
                        element: '[data-page-tour="shaker-new-step-btn"]',
                        position: 'bottom'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.PROCESSORS.TITLE', 'Choose from a range of pre-made processors'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.PROCESSORS.DESCRIPTION', 'Processors have been designed to handle one specific task, such as filtering rows, rounding numbers and more.'),
                        element: '[data-page-tour="processors-library"]',
                        position: 'top',
                        beforeChange: function() {
                            scope.toggleLibrary();
                            hideHighlightDuringTransition(introJSInstance, 400);
                        },
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.COLUMN_SUGGESTIONS.TITLE', 'Get suggested preparation steps for each column'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.COLUMN_SUGGESTIONS.DESCRIPTION', 'Get suggestions for processors that can be applied to each column of your dataset.'),
                        beforeChange: function() {
                            scope.toggleLibrary(false);
                            document.querySelector('[data-page-tour="column-name"]').click();
                            this.element = document.querySelector('[data-page-tour="suggested-actions"]');
                            this.position = "bottom";
                            closeContextualMenuOnlyOnNextButtonClick(scope);
                            hideHighlightDuringTransition(introJSInstance, 100);
                        }
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.PREVIEW.TITLE', 'Preview your Prepare recipe script'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.PREVIEW.DESCRIPTION', 'When new steps are added to the script, the step output is immediately visible thanks to the preview.<br/><br/>This preview is only computed on your sample dataset. The full script will be executed when you run the recipe.'),
                        element: '[data-page-tour="step-preview-button"]',
                        position: 'right'
                    },
                    {
                        title: translate('PAGE_SPECIFIC_TOUR.PREPARE.OUTRO.TITLE', '✅ You\'ve finished the Prepare recipe tour'),
                        intro: translate('PAGE_SPECIFIC_TOUR.PREPARE.OUTRO.DESCRIPTION', 'That\'s all for now. You can re-activate the Prepare recipe tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>'),
                        hideStepNumber: true,
                        element: '[data-page-tour="help-center-trigger"]',
                        position: 'bottom'
                    }
                ]
                const bypassPreviewStep =  document.querySelectorAll('[data-page-tour="step-preview-button"]').length === 0;
                if (bypassPreviewStep) {
                    steps.splice(4, 1);
                }
                if (fromContext === 'opals') {
                    steps.shift(); // no need to display the first step to the user since they launched the tour
                }

                introJSInstance.setOptions({
                    helperElementPadding: 5,
                    steps: steps
                });

                $timeout(() => {
                    introJSInstance.start();
                }, 0); // give some time for the DOM to fully load
            });
        }

        /****** WT1 Methods ******/

        function onTourStarted(introJSInstance) {
            if (introJSInstance.$tourName) {
                WT1.tryEvent('page-tour-started', () => ({ from: introJSInstance.$fromContext, scope: introJSInstance.$tourName }));
            }
        }

        function onTourDismissed(introJSInstance) {
            introJSInstance.$skipped = true;
            if (introJSInstance.$tourName) {
                WT1.tryEvent('page-tour-dismissed', () => ({ from:  introJSInstance.$fromContext, scope: introJSInstance.$tourName }));
            }
        }

        function onTourClosed(introJSInstance) {
            if (introJSInstance.$tourName && !introJSInstance.$skipped && introJSInstance.currentStep() >= 0) {
                WT1.tryEvent('page-tour-closed', () => ({
                    from: introJSInstance.$fromContext,
                    scope: introJSInstance.$tourName,
                    numberOfSteps: introJSInstance._options.steps.length,
                    currentStep: introJSInstance.currentStep() + 1
                }));
            }
        }


        /****** Utils methods *******/


        /** Conditions that prevent the Flow Tour from starting **/
        function canStartFlowTour() {
            const builtDatasets = $(BUILT_DATASET_SELECTOR);
            if (!builtDatasets.length) {
                return false; // don't start the tour if there are no built datasets
            }
            return checkGlobalSettingsAllowTour(TOUR_NAMES.FLOW);
        }

        /** Conditions that prevent the Explore Tour from starting **/
        function canStartExploreTour(scope) {
            if (!scope.table || scope.table.headers.length === 0) {
                return; // no table loaded -- dismiss temporarily
            }
            return checkGlobalSettingsAllowTour(TOUR_NAMES.EXPLORE);
        }

        /** Conditions that prevent the Prepare Tour from starting **/
        function canStartPrepareTour(scope) {
            if (!scope.projectSummary || !scope.projectSummary.canWriteProjectContent) {
                return; // user doesn't have write content on this project, do not show the Tour
            }
            if (!scope.table || scope.table.headers.length === 0) {
                return; // no table loaded -- dismiss temporarily
            }
            if (!scope.shaker || scope.shaker.origin !== 'PREPARE_RECIPE') {
                return; // not on a prepare recipe
            }
            return checkGlobalSettingsAllowTour(TOUR_NAMES.PREPARE);
        }


        /** Check if the tour can be started based on global settings and user profile - dismiss it permanently if appropriate  **/
        function checkGlobalSettingsAllowTour(tourName) {
            // the Help Center is deactivated, dismiss permanently
            if (!$rootScope.appConfig.opalsEnabled) {
                recordTourCompleted(tourName);
                return false;
            }

            // Onboarding Experience is disabled, dismiss permanently
            if (!$rootScope.appConfig.onboardingExperience) {
                recordTourCompleted(tourName);
                return false;
            }

            // do not show the Tour to technical accounts - dismiss permanently
            if (ProfileService.isTechnicalAccount()) {
                recordTourCompleted(tourName);
                return false;
            }

            // do not show the users which profiles don't allow write project content - dismiss permanently
            if (!$rootScope.appConfig.userProfile.mayWriteProjectContent) {
                recordTourCompleted(tourName);
                return false;
            }
            return true;
        }

        /** Check if the tour can be started based on contextual settings (already completed, Help Center opened, ...) - dismiss it temporarily if appropriate  **/
        function checkContextAllowTour(tourName, fromContext) {
            // if the tour is started from the Help Center, launch it even if the tour is completed
            if (fromContext === 'opals') {
                return Promise.resolve(true);
            }

            //  if was already dismissed or completed, do not show them again
            if (getTourCompletionSetting(tourName)) {
                return Promise.resolve(false);
            }

            // the Help Center is shown - dismiss temporarily
            if (TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).isToggledOn() && tourName !== TOUR_NAMES.FLOW) {
                return Promise.resolve(false);
            }

            // check if the tour has been completed on another instance (using opals local storage)
            return OpalsService.getLocalStorage(tourName+"TourCompleted")
            .then(tourCompleted => {
                if (tourCompleted === "true") {
                    recordTourCompleted(tourName);
                    return false;
                }
                return  true;
            })
        }

        /** Common setup for all tours (callbacks, WT1, styling, state change handling, ...) **/
        function initialTourSetup(scope, tourName, fromContext) {
            const introJSInstance = introJs();
            introJSInstance.$tourName = tourName;
            introJSInstance.$fromContext = fromContext;

            // End the tour when leaving the page (e.g. with the back arrow)
            const unbindStateChangeListener = scope.$on('$stateChangeStart', () => {
                introJSInstance.$forceExit = true;
                introJSInstance.exit();
            });

            /** send the WT1 when leaving the page with the tour on **/
            const handlePageLeave = () => {
                onTourClosed(introJSInstance);
            }
            window.addEventListener('beforeunload', handlePageLeave)
            scope.$on("$destroy", () => {
                window.removeEventListener("beforeunload", handlePageLeave);
                unbindStateChangeListener();
            });

            /** Defaults common options, can be overridden in the specific tour if necessary **/
            introJSInstance.setOptions({
                disableInteraction: true,
                keyboardNavigation: false,
                buttonClass: 'btn btn--primary',
                doneLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.FINISH_TOUR", "Finish Tour"),
                nextLabel: fromContext === 'opals' ? translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next") : translate("PAGE_SPECIFIC_TOUR.BUTTONS.LETS_GO", "Let's go"),
                showBullets: false,
                helperElementPadding: 0
            });

            /** common callbacks **/
            introJSInstance.onstart(function() {
                onTourStarted(introJSInstance);
                Mousetrap.pause();
            });

            let skipped = false;
            introJSInstance.onskip(function() {
                skipped = true;
            });

            introJSInstance.onbeforechange(function() {
                const currentStep = this._introItems[this._currentStep];
                if (currentStep.beforeChange) {
                    currentStep.beforeChange(); // callback defined at step level
                }
            });
            introJSInstance.onafterchange(function() {
                restyleTour(introJSInstance);
                if (introJSInstance.currentStep() === introJSInstance._options.steps.length - 1) {
                    recordTourCompleted(introJSInstance.$tourName);
                    OpalsService.setLocalStorage(tourName+"TourCompleted","true");
                }
            });
            introJSInstance.onbeforeexit(function () {
                
                if (introJSInstance.currentStep() === introJSInstance._options.steps.length - 1) {
                    return true;  
                }

                if (skipped) {
                    // custom behavior when closing the tour by clicking the cross, directly go the bye step without confirmation modal
                    onTourClosed(introJSInstance);
                    goToByeStep(introJSInstance);
                    return false; 
                }

                if (!introJSInstance.$forceExit) {
                    pauseTour(introJSInstance);
                    const tourNameTranslated = translate('PAGE_SPECIFIC_TOUR.TOUR_NAME.' + $filter('uppercase')(tourName),$filter('capitalize')(tourName));
                    return Dialogs.confirm(scope,
                        translate('PAGE_SPECIFIC_TOUR.CONFIRM_MODAL.TITLE',"👋 Are you sure you want to leave the tour?"),
                        translate('PAGE_SPECIFIC_TOUR.REACTIVATE.DESCRIPTION', `You can re-activate the {{tourName}} tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>`, {tourName: tourNameTranslated}),
                        { confirmOnExit : true }
                    ).then(() => {
                        recordTourCompleted(introJSInstance.$tourName);
                        OpalsService.setLocalStorage(tourName+"TourCompleted","true");
                    }, () => {
                        unpauseTour(introJSInstance);
                        return false;
                    });
                }
                return true;
            });
            introJSInstance.onexit(function() {
                if (!skipped) {
                    onTourClosed(introJSInstance);
                }
                window.removeEventListener("beforeunload", handlePageLeave);
                unbindStateChangeListener();
                ContextualMenu.prototype.closeAny();
                Mousetrap.unpause();
                if (introJSInstance.afterexit) {
                    introJSInstance.afterexit();
                }
            });

            return introJSInstance;
        }

        /** Minimal setup for single step tours (no WT1, no step change management, ...) **/
        function initialPopupSetup() {
            const introJSInstance = introJs();

            introJSInstance.setOptions({
                disableInteraction: true,
                keyboardNavigation: false,
                buttonClass: 'btn btn--primary',
                doneLabel: "OK",
                nextLabel: translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next"),
                showBullets: false,
                helperElementPadding: 0
            });
            introJSInstance.onstart(function() {
                Mousetrap.pause();
            });
            introJSInstance.onafterchange(function() {
                restyleTour(introJSInstance);
            });
            introJSInstance.onexit(function() {
                Mousetrap.unpause();
            });
            return introJSInstance;
        }

        /**
            Save the page specific tour settings with the current tour marked as completed
        **/
        const recordTourCompleted = (tourName) => {
            if (tourName) {
                if (getTourCompletionSetting(tourName)) {
                    return;
                }
                setTourCompletionSetting(tourName, true);
                DataikuAPI.profile.updatePageSpecificTourSettings($rootScope.appConfig.pageSpecificTourSettings);
            }
        }

        /**
        * When opening a dropdown menu during a Tour, we most of the time don't want to close it on any click
        * This method helps with closing the dropdown only when going to the next step
        * If the tour is closed at that step, we also ensure that all dropdowns get closed
        **/
        const closeContextualMenuOnlyOnNextButtonClick = () => {
            window.setTimeout(() => {
                // override click handler to close contextual menu only when clicking next
                $(document).off('click.closeMenu');
                $(document).on('click.closeMenu', function(evt) {
                    const isClickOnNextButton = $(evt.target).hasClass('introjs-nextbutton');
                    if (isClickOnNextButton) {
                        ContextualMenu.prototype.closeAny();
                    }
                });
            }, 0);
        }

        /**
        *   introJS doesn't handle transitions very well so in that case we hide the highlight and tooltip
        *   during the transition and display them again when it's finished
        *   (looks like it will be fixed in v8.0.0 when it's released)
        **/
        const hideHighlightDuringTransition = (introJSInstance, delay) => {
            $timeout(() => $('.introjs-helperLayer, .introjs-tooltipReferenceLayer').addClass("display-none")); // hide the highlight and tooltip during the transition
            $timeout(() => {
                introJSInstance.refresh();  // refresh to position the highlight correctly
                $('.introjs-helperLayer, .introjs-tooltipReferenceLayer').removeClass("display-none"); // show the highlight and tooltip once the transition is complete
            }, delay);
        }

        let highlightedElements = [];
        /**
            Pause the tour to display the confirm modal. We hide all elements related to the Tour
        **/
        const pauseTour = (introJSInstance) => {
            $('.introjs-tooltipReferenceLayer, .introjs-helperLayer, .introjs-overlay, .introjs-disableInteraction').addClass("display-none");
            highlightedElements = $('.introjs-showElement');
            /** Elements that are highlighted before the pause */
            highlightedElements.removeClass('introjs-showElement');
            introJSInstance.refresh();
        }

        /**
            Resume the tour if user doesn't exit. We display again all elements of the Tour
        **/
        const unpauseTour = (introJSInstance) => {
            $('.introjs-tooltipReferenceLayer, .introjs-helperLayer, .introjs-overlay, .introjs-disableInteraction').removeClass('display-none');
            /** Elements that need to be highlighted again when un-pausing */
            highlightedElements.addClass('introjs-showElement');
            introJSInstance.refresh();
        }


        // ----- Tour styling ----- //

        const restyleTour = (introJSInstance) => {
            const currentStep = introJSInstance.currentStep();
            if (currentStep === 0) {
                // we only need to do the following on the first steps since it's persisted for later steps
                restyleDismissButton();
                introJSInstance.setOption('nextLabel', translate("PAGE_SPECIFIC_TOUR.BUTTONS.NEXT", "Next"));
                if (!QuestionnaireService.isFromQuestionnaire() && introJSInstance.$fromContext !== 'opals') {
                    // when coming from the questionnaire, do not show the 'No thanks' button
                    // as the user already agreed to follow the tutorial (they can still close)
                    addNoThanksButton(introJSInstance);
                }
            } else if (currentStep === 1) {
                removeNoThanksButton();
            }
            restyleStepNumber(introJSInstance);
        }

        const restyleDismissButton = () => {
            $('.introjs-skipbutton').addClass('dku-icon-dismiss-20');
            $('.introjs-skipbutton').text('');
        }

        /** Display the step numbers except for steps with the "hideStepNumber" property **/
        const restyleStepNumber = (introJSInstance) => {
            var stepNumberElements = document.getElementsByClassName("introjs-stepNumber");
            var helperNumberLayer;
            if (!stepNumberElements.length) {
                helperNumberLayer = document.createElement("div");
                helperNumberLayer.className = 'introjs-stepNumber';
                const tooltipButtons = $('.introjs-tooltipbuttons');
                const parentNode = tooltipButtons[0];
                parentNode.insertBefore(helperNumberLayer, parentNode.firstChild);
            } else {
                helperNumberLayer = stepNumberElements[0];
            }
            // give this step a number but only counting numbered steps
            const currentStep = introJSInstance._introItems[introJSInstance.currentStep()];
            const numberedSteps = introJSInstance._introItems.filter(item => !item.hideStepNumber);
            const stepNumber = numberedSteps.indexOf(currentStep);
            helperNumberLayer.innerText = !currentStep.hideStepNumber ? `${stepNumber + 1} ${translate("PAGE_SPECIFIC_TOUR.STEP_NUMBER_LABEL", "of")} ${numberedSteps.length}` : '';
        }

        const goToByeStep = (introJSInstance) => {
            removeNoThanksButton();
            const tourName = introJSInstance.$tourName;
            const tourNameTranslated = translate('PAGE_SPECIFIC_TOUR.TOUR_NAME.' + $filter('uppercase')(tourName),$filter('capitalize')(tourName));
            const steps = introJSInstance._introItems;
            // Update the last step and go to it to reflect that the user has skipped the tour
            steps[steps.length - 1] = {
                title: translate("PAGE_SPECIFIC_TOUR.SKIPPED_OUTRO.TITLE", "👋 Bye for now!"),
                intro: translate('PAGE_SPECIFIC_TOUR.REACTIVATE.DESCRIPTION', `You can re-activate the {{tourName}} tour or browse resources at any time through the Help Center <i class="dku-icon-question-circle-fill-16 vab"></i>`, {tourName: tourNameTranslated}),
                element: document.querySelector('[data-page-tour="help-center-trigger"]'),
                position: 'bottom',
                hideStepNumber: true
            };
            introJSInstance.goToStep(steps.length);
        }

        const addNoThanksButton = (introJSInstance) => {
            if ($('.introjs-nothanksbutton').length) {
                return;
            }
            const tooltipButtons = $('.introjs-tooltipbuttons');
            var btn = document.createElement("button");
            btn.className = 'introjs-nothanksbutton btn btn--text btn--secondary';
            btn.innerText = translate("PAGE_SPECIFIC_TOUR.BUTTONS.NO_THANKS", 'No Thanks');
            btn.onclick = () => {
                onTourDismissed(introJSInstance);
                goToByeStep(introJSInstance);
            }
            const parentNode = tooltipButtons[0];
            parentNode.insertBefore(btn, parentNode.firstChild);
        }

        const removeNoThanksButton = () => {
            $('.introjs-nothanksbutton').remove();
        }

        return {
            TOUR_NAMES,
            checkFlowTourConditions,
            canStartFlowTour,
            startFlowTour,
            canStartExploreTour,
            startExploreTour,
            canStartPrepareTour,
            startPrepareTour,
            displayReopenTutorialPopup
        };
    }
})();

;
(function() {
'use strict';

const app = angular.module('dataiku.flow.graph');

/**
 * @name zoneEditor
 * @description
 *      Component to display the zone edition tool. User can edit the name and color of the zone.
 *
 * @param { String }    name                    - name of the zone
 * @param { String }    color                   - color of the zone (if empty, picks a random color in the stock)
 * @param { Boolean }   disableCompactStyling   - if True, controls and control labels don't have the compact styling and take a larger width
 *
 * @example
 *      <zone-editor name="uiState.name" color="uiState.color"></zone-editor>
 */
app.component('zoneEditor', {
    bindings: {
        name: '=',
        color: '=',
        disableCompactStyling: '<?'
    },
    templateUrl: '/templates/zones/edit-zone-box.html',
    controller: function zoneEditorCtrl($timeout) {
        this.$onInit = function () {
            this.stockColors = ["#C82423","#8C2DA7","#31439C","#087ABF","#0F786B","#4B8021","#F9BE40","#C54F00","#D03713","#465A64"];
            this.color = this.color === undefined ? this.stockColors[Math.floor(Math.random() * this.stockColors.length)] : this.color;
            this.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
        }

        this.pickStockColor = color => {
            $timeout(() => {
                this.color = color;
            })
        };
    }
});

/**
 * @name zoneSelector
 * @description
 *      Component to display the zone selector. Provide the option to either create a new zone, or select an already existing one.
 *
 * @param { String }    name                    - name of the zone
 * @param { String }    color                   - color of the zone (if empty, picks a random color in the stock)
 * @param { String }    selectedZone            - id of the selected existing zone when in 'select' mode (preselects the selected or last used zone)
 * @param { Boolean }   creationMode            - 'CREATE' to create a new zone or 'SELECT' to pick an already existing one (by default: 'SELECT' if forceCreation is False)
 * @param { Boolean }   forceCreation           - if True, only displays the creation option
 * @param { Boolean }   disableCompactStyling   - if True, controls and control labels don't have the compact styling and take a larger width
 *
 * @example
 * <zone-selector name="uiState.name"
 *                color="uiState.color"
 *                selected-zone="uiState.selectedZone"
 *                creation-mode="uiState.creationMode"
 *                force-creation="uiState.forceCreation">
 * </zone-selector>
 */
app.component('zoneSelector', {
    bindings: {
        name: '=',
        color: '=',
        creationMode: '=',
        selectedZone: '=',
        forceCreation: '<?',
        projectKey: '<?',
        disableCompactStyling: '<?'
     },
    templateUrl: '/templates/flow-editor/zone-selection-box.html',
    controller:  ['$scope', '$rootScope', '$stateParams', 'DataikuAPI', 'localStorageService', 'FlowGraphSelection', function zoneSelectionCtrl(scope, $rootScope, $stateParams, DataikuAPI, localStorageService, FlowGraphSelection) {
        const ctrl = this;
        
        ctrl.$onInit = function() {
            ctrl.projectKey = ctrl.projectKey === undefined ? $stateParams.projectKey : ctrl.projectKey;
            if (!ctrl.forceCreation) {
                scope.$watch("$ctrl.projectKey", function () {
                    listZones();
                }, true);
                ctrl.selectedZone = preselectZone();
            }  
        };

        ctrl.zoneComparator = (v1, v2) => {
            // If we don't get strings, just compare by index
            if (v1.type !== 'string' || v2.type !== 'string') {
                return (v1.index < v2.index) ? -1 : 1;
            }
            if (ctrl.zones[v1.index].id === 'default') {
                return 1;
            }
            if (ctrl.zones[v2.index].id === 'default') {
                return -1;
            }
            // Compare strings alphabetically, taking locale into account
            return v1.value.localeCompare(v2.value);
        };

        const preselectZone = () => {
            const lastUsedZoneKey = `dku.flow.lastUsedZone.${ctrl.projectKey}`;
            return localStorageService.get(lastUsedZoneKey);
        };

        function listZones() {
            DataikuAPI.flow.zones.list(ctrl.projectKey).success(zones => {
                ctrl.zones = zones;
                ctrl.forceCreation = zones.length === 0;
                ctrl.creationMode = !ctrl.forceCreation ? 'SELECT' : 'CREATE';
            }).error(setErrorInScope.bind(scope));
        };

    }]
});

app.directive('zoneRightColumnSummary', function(
    $controller,
    $rootScope,
    $state,
    $stateParams,
    DataikuAPI,
    CreateModalFromTemplate,
    TaggableObjectsUtils,
    FlowGraph,
    ActivityIndicator,
    Dialogs,
    SubFlowCopyService,
    FlowGraphSelection,
    ZoneService
) {
    return {
        templateUrl: '/templates/zones/right-column-summary.html',

        link: function(scope, element, attrs) {

            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            scope.$stateParams = $stateParams;

            scope.zoomOnZone = ZoneService.zoomOnZone;

            scope.zoomOutOfZone = ZoneService.zoomOutOfZone;

            scope.$on("objectSummaryEdited", function() {
                const zone = scope.zoneFullInfo.zone;
                const tor = {type: 'FLOW_ZONE', projectKey: $stateParams.projectKey, id: zone.id};
                DataikuAPI.taggableObjects.getMetadata(tor).success(function(metadata) {
                    metadata.tags = zone.tags;
                    DataikuAPI.taggableObjects.setMetaData(tor, metadata).success(function() {
                        ActivityIndicator.success("Saved");
                    });
                }).error(setErrorInScope.bind(scope));
            });

            scope.editZone = () => {
                CreateModalFromTemplate("/templates/zones/edit-zone-modal.html", scope, null, function(newScope){
                    newScope.uiState = {
                        color: scope.selection.selectedObject.color,
                        name: scope.selection.selectedObject.name
                    }
                    newScope.go = function(){
                        DataikuAPI.flow.zones.edit($stateParams.projectKey, scope.selection.selectedObject.id, newScope.uiState.name, newScope.uiState.color).success(function () {
                            scope.$emit('reloadGraph');
                            if ($stateParams.zoneId) {
                                $rootScope.$emit("zonesListChanged", newScope.uiState.name);
                            }
                            newScope.dismiss()
                        }).error(setErrorInScope.bind(newScope));
                    }
                });
            }

            // TODO: code duplication
            function openMultiBuildModal(errorMessage, defaultLimitToZone) {
                return function(data) {
                    if (!data.length) {
                        Dialogs.error(scope, "Nothing to build", errorMessage);
                    } else {
                        CreateModalFromTemplate(
                            "/templates/flow-editor/tools/build-multiple-flow-computables-modal.html",
                            scope,
                            "BuildMultipleComputablesController",
                            function(modalScope) {
                                modalScope.initModal(data, undefined, { origin: "FLOW_ZONE" });
                                modalScope.jobOptions.stopAtFlowZoneBoundary = defaultLimitToZone;
                            }
                        );
                    }
                }
            }

            scope.buildZone = function(zoneId) {
                DataikuAPI.flow.listDownstreamComputables($stateParams.projectKey, {zoneId: scope.selection.selectedObject.id})
                    .success(openMultiBuildModal("This zone has no buildable dataset.", true))
                    .error(setErrorInScope.bind(scope));
            }

            scope.refreshData = function() {
                DataikuAPI.zones.getFullInfo(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.cleanId).success(function(data) {
                    data.zone.cleanId = data.zone.id
                    scope.zoneFullInfo = data;
                    // check that the selection didn't change while getFullInfo was called
                    if (scope.selection.selectedObject && scope.selection.selectedObject.cleanId === data.zone.cleanId) {
                        scope.selection.selectedObject = data.zone;
                        scope.selection.selectedObject.isCollapsed = scope.collapsedZones.find(it => it === data.zone.id) !== undefined;
                    }
                }).error(setErrorInScope.bind(scope));
            };

            scope.deleteZone = () => {
                let items = scope.getSelectedTaggableObjectRefs();
                let success = undefined;
                if ($stateParams.zoneId) {
                    items = [TaggableObjectsUtils.fromNode(scope.nodesGraph.nodes[`zone_${$stateParams.zoneId}`])];
                    success = () => scope.zoomOutOfZone();
                }
                scope.deleteSelected(items, success);
            };

            scope.collapseAllZones = () => {
                const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
                scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'collapseAll');
            }

            scope.expandAllZones = () => {
                const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
                scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'expandAll');
            }

            scope.copyZone = () => {
                const startCopyTool = scope.startTool('COPY',  {preselectedNodes: FlowGraphSelection.getSelectedNodes().map(n => n.id)});
                startCopyTool.then(() => {
                    const selectedTaggableObjectRefs = FlowGraphSelection.getSelectedTaggableObjectRefs();
                    const itemsByZones = FlowGraph.nodesByZones((node) => TaggableObjectsUtils.fromNode(node));
                    SubFlowCopyService.start(selectedTaggableObjectRefs, itemsByZones, scope.stopAction);
                });
            };

            scope.$watch("selection.selectedObject",function(nv) {
                if (!scope.selection) scope.selection = {};
                scope.zoneFullInfo = {zone: scope.selection.selectedObject, timeline: {}}; // display temporary (incomplete) data
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) return;
                scope.selection.selectedObject.cleanId = scope.selection.selectedObject.id.split('_')[1];
                scope.refreshData();
            });

            const zonesListChangedListener = $rootScope.$on("zonesListChanged", scope.refreshData);
            scope.$on('$destroy',zonesListChangedListener);
        }
    }
});

app.service('ZoneService', function($stateParams, $state) {
    this.zoomOnZone = zoneId => {
        $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId }));
    };

    this.zoomOutOfZone = (id = null) => {
        $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: null, id}))
    }
})
})();

;
(function() {
'use strict';

const app = angular.module('dataiku.flow.tools', []);

// Magic trick to work around dependency injection order
// because ng2 ToolBridgeService is not initialized yet when bootstrapping ng1 services
app.factory('Ng1ToolBridgeService', function($injector) {
    return () => $injector.get('ToolBridgeService');
});

app.service('FlowToolsRegistry', function() {
    const flowViews = {};
    const flowTools = {};

    this.registerView = function(service) {
        let def = service.getDefinition();
        def.isTool = false;
        flowViews[def.getName()] = def;
    };
    this.registerFlowTool = function(service) {
        let def = service.getDefinition();
        def.isTool = true;
        flowTools[def.getName()] = def;
    };

    this.getDef = function(name) {
        return flowViews[name] || flowTools[name];
    };
    this.getFlowViews = function() {
        return Object.values(flowViews);
    };
});


app.service('FlowToolsLoader', function(FlowToolsRegistry,
    FlowZonesView, TagsView, CustomFieldsView, ConnectionsView, FileformatsView, RecipesEnginesView, RecipesCodeEnvsView,
    PipelinesView, ImpalaWriteModeView, HiveModeView, SparkConfigView,
    PartitioningView, PartitionsView, ScenariosView, WatchView, CountOfRecordsView, FilesizeView,
    CreationView, LastModifiedView, LastBuildView, LastBuildDurationView,
    RecentActivityView, DataQualityView,
    CopyFlowTool, PropagateSchemaFlowTools, CheckConsistencyFlowTool, ColumnUsageView) {

    FlowToolsRegistry.registerView(FlowZonesView);
    FlowToolsRegistry.registerView(TagsView);
    FlowToolsRegistry.registerView(CustomFieldsView);

    FlowToolsRegistry.registerView(ConnectionsView);
    FlowToolsRegistry.registerView(RecipesEnginesView);
    FlowToolsRegistry.registerView(RecipesCodeEnvsView);

    FlowToolsRegistry.registerView(ImpalaWriteModeView);
    FlowToolsRegistry.registerView(HiveModeView);
    FlowToolsRegistry.registerView(SparkConfigView);
    FlowToolsRegistry.registerView(PipelinesView.getService("SPARK_PIPELINES"));
    FlowToolsRegistry.registerView(PipelinesView.getService("SQL_PIPELINES"));

    FlowToolsRegistry.registerView(CreationView);
    FlowToolsRegistry.registerView(LastModifiedView);
    FlowToolsRegistry.registerView(LastBuildView);
    FlowToolsRegistry.registerView(LastBuildDurationView);
    FlowToolsRegistry.registerView(RecentActivityView);

    FlowToolsRegistry.registerView(PartitioningView);
    FlowToolsRegistry.registerView(PartitionsView);
    FlowToolsRegistry.registerView(ScenariosView);

    FlowToolsRegistry.registerView(DataQualityView);
    FlowToolsRegistry.registerView(CountOfRecordsView);
    FlowToolsRegistry.registerView(FilesizeView);
    FlowToolsRegistry.registerView(FileformatsView);

    FlowToolsRegistry.registerView(WatchView);

    FlowToolsRegistry.registerView(ColumnUsageView);

    FlowToolsRegistry.registerFlowTool(CopyFlowTool);
    FlowToolsRegistry.registerFlowTool(CheckConsistencyFlowTool);
    FlowToolsRegistry.registerFlowTool(PropagateSchemaFlowTools);
});


/*
* Note that for now, flow views are simply flow tools
*/
app.service('FlowTool', function($rootScope, Assert, Logger, FlowToolsRegistry, $injector) {
    const svc = this;
    let currentTool = {};

    this.setCurrent = function(tool)  {
        currentTool = tool;
    };

    this.getCurrent = function()  {
        return currentTool;
    };

    this.unactivateCurrentTool = function(redraw = true) {
        if (!currentTool.def) return; // None active
        if (currentTool.def.destroyFlowTool) {
            currentTool.def.destroyFlowTool();
        }
        svc.setCurrent({});
        
        if (redraw) {
            $rootScope.$emit('drawGraph', false, false);
        }
        return currentTool;
    };

    this.activateTool = function(currentToolSession, viewData) {
        Assert.trueish(currentToolSession, 'no currentToolSession');

        Logger.info("Activating tool", currentToolSession, FlowToolsRegistry.registry);

        svc.unactivateCurrentTool(false);

        let def = FlowToolsRegistry.getDef(currentToolSession.type);
        Assert.trueish(def, 'no tool def');
        svc.setCurrent({
            drawHooks: {},
            actionHooks: {},
            type: currentToolSession.type,
            currentSession: currentToolSession,
            def: def
        });

        return def.initFlowTool(currentTool, viewData);
    };


    /*
    * Refresh the flow state only for some views
    * (or all is not specified)
    */
    this.refreshFlowStateWhenViewIsActive = function(viewNames) {
        const currentTool = svc.getCurrent();
        if (currentTool.def && (!viewNames || !viewNames.length || viewNames.includes(currentTool.def.getName()))) {
            currentTool.refreshState();
        }
    };

    $rootScope.$on('flowDisplayUpdated', function() {
        if (currentTool.drawHooks && currentTool.drawHooks.updateFlowToolDisplay) {
            currentTool.drawHooks.updateFlowToolDisplay();
        }
    });

    $rootScope.$on('flowItemClicked', function(evt, evt2, item) {
        if (currentTool.actionHooks && currentTool.actionHooks.onItemClick) {
            currentTool.actionHooks.onItemClick(item, evt2);
        }
    });
});


app.service('FlowViewsUtils', function($stateParams, WT1, DataikuAPI, Logger, MonoFuture, ProgressStackMessageBuilder,
    FlowGraphSelection, FlowGraphFiltering, FlowGraphFolding) {

    this.addFocusBehavior = function(tool) {


        function isItemSelectedbyId(itemId) {

            let val = tool.user.state.valueByNode[itemId];
            if (val === undefined) {
                return false;
            }
            // Multi-valued view (tags, scenarios)
            if (angular.isArray(tool.getRepr(val))) {
                for (let v of tool.getRepr(val)) {
                    if (tool.user.state.focusMap[v]) {
                        return true;
                    }
                }
                return false;
            }
            // Single-valued view
            return tool.user.state.focusMap[tool.getRepr(val)];
        }

        function isItemSelected(item) {
            return isItemSelectedbyId(item.realId);
        }

        function getSelectedIdsList() {
            const selectedIdslist = [];
            Object.keys(tool.user.state.valueByNode).forEach (itemId => {
                if (isItemSelectedbyId(itemId))  selectedIdslist.push(itemId);
            });
            return selectedIdslist;
        }

        tool.user.isFocused = function(val) {
            // Disable focus when in continuous mode
            if (typeof tool.colorScale == 'function' && tool.colorScale().continuous) {
                return true;
            }

            const repr = tool.getRepr(val);
            if (angular.isArray(repr)) {
                let any = false;
                repr.forEach(function(it) {
                    if (tool.user.state.focusMap[it]) {
                        any = true;
                    }
                });
                return any;
            } else {
                return tool.user.state.focusMap[repr];
            }
        };

        tool.user.getFocusedAsList = function() {
            return [];
        };

        //TODO @flow deprecated
        tool.user.getSingleFocused = function() {
            for (let val in tool.user.state.focusMap) {
                if(tool.user.state.focusMap[val]) {
                    return val;
                }
            }
        };

        tool.user.zoomToFocused = function() {
            FlowGraphFolding.ensureNodesNotFolded(getSelectedIdsList());
            let scope = $('#flow-graph').scope();
            if($('#flow-graph svg .focus').length) {
                scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector(scope.svg, '.focus'), 1.2);
            } else {
                scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector(scope.svg, '.node'), 1.2);
            }
        };

        tool.user.selectFocused = function() {
            WT1.event("flow-view-select-focused", {tool: tool.def.name});
            FlowGraphFolding.ensureNodesNotFolded(getSelectedIdsList());
            FlowGraphSelection.select(isItemSelected);
        };
    };

    //TODO @flow move or rename service
    this.addAsynchronousStateComputationBehavior = function(tool) {
        tool.user.update = function(scope) {
            tool.user.updateStatus.updating = true;
            tool.user.firstUpdateDone = true;
            return MonoFuture(scope).wrap(DataikuAPI.flow.tools.startUpdate)($stateParams.projectKey, tool.def.getName(), tool.user.updateOptions)
            .success(function(data) {
                tool.user.state = data.result;
                tool.drawHooks.updateFlowToolDisplay();
                tool.user.updateStatus.updating = false;
            }).error(function(a,b,c) {
                tool.user.updateStatus.updating = false;
                setErrorInScope.bind(scope)(a,b,c);
            }).update(function(data) {
                tool.user.updateStatus.progress = data.progress;
                tool.user.updateStatus.totalPercent = ProgressStackMessageBuilder.getPercentage(data.progress);
            });
        };
    };

});


app.service('FlowToolsUtils', function(Logger, FlowGraph) {
    const svc = this;

    this.notSoGrey = function(node, elt) {
        svc.colorNode(node, elt, '#ACACAC');
    },

    this.greyOutNode = function(node, elt) {
        svc.colorNode(node, elt, '#DADADA');
    },

    /**
     * 
     * @param {FlowGraphNode} node node object from DSS `FlowGraph.node()`
     * @param {import('./project_flow').D3Node} elt D3 node object
     * @param {?string} color `''` or `undefined` means default color
     * @returns 
     */
    this.colorNode = function(node, elt, color) {
        try {
            if(elt === undefined) {
                return;
            }
            if (node.nodeType == 'LOCAL_DATASET' || node.nodeType == 'FOREIGN_DATASET') {
                elt.style('fill', color);
                if (node.neverBuilt) {
                    elt.select('.never-built-computable .main-dataset-rectangle').style('stroke', color);
                    elt.select('.never-built-computable .nodeicon').style('color', color);
                    elt.select('.never-built-computable .nodelabel-wrapper').style('color', color);
                }
            } else if (node.nodeType == 'LOCAL_MANAGED_FOLDER'  || node.nodeType == 'FOREIGN_MANAGED_FOLDER') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_STREAMING_ENDPOINT') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_SAVEDMODEL' || node.nodeType == 'FOREIGN_SAVEDMODEL') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_MODELEVALUATIONSTORE' || node.nodeType == 'FOREIGN_MODELEVALUATIONSTORE') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_GENAIEVALUATIONSTORE' || node.nodeType == 'FOREIGN_GENAIEVALUATIONSTORE') {
                elt.style('fill', color);
            } else if (node.nodeType == 'LOCAL_RETRIEVABLE_KNOWLEDGE' || node.nodeType == 'FOREIGN_RETRIEVABLE_KNOWLEDGE') {
                elt.style('fill', color);
                if (node.neverBuilt) {
                    elt.select('.never-built-computable .main-retrievable-knowledge-rectangle').style('stroke', color);
                    elt.select('.never-built-computable .nodeicon').style('color', color);
                    elt.select('.never-built-computable .nodelabel-wrapper').style('color', color);
                }
            } else if (node.nodeType == 'RECIPE') {
                elt.select('.bzicon').style('fill', color);
            } else if (node.nodeType == 'LABELING_TASK') {
                elt.select('.bzicon').style('fill', color);
            } else if (node.nodeType == 'ZONE') {
                elt.style('background-color', color);
                elt.style('stroke', color);
                const rgbColor = d3.rgb(color);
                const titleColor = (rgbColor.r*0.299 + rgbColor.g*0.587 + rgbColor.b*0.114) >= 128 ? "#000" : "#FFF";
                elt.style('color', titleColor);
            } else {
                Logger.warn("Cannot color node", node);
            }
            elt.select("g, rect").attr("color", color); //text color
        } catch (e) {
            Logger.error("Failed to color node", e);
        }
    }

    // Bottom right colored indicator
    // There might be several (Ex: tags, so there is an index)
    const RADIUS = 6;
    this.addViewValueIndicator = function(elt, color='rgba(0,0,0,0)', idx = 0, onClick) {
        let tsz = elt.select(".tool-simple-zone");

        if (!tsz.empty()) {
            if (idx == 0) {
                tsz.selectAll("*").remove();
            }
            let tszHeight = tsz.attr("data-height");
            tsz.append("circle")
                .attr("cx", RADIUS + 2)
                .attr("cy", tszHeight - RADIUS - idx * (RADIUS*2 + 2))
                .attr("r", RADIUS)
                .attr("fill", color)
                .on("click", onClick)
                ;
        }
    }
});

app.directive('flowToolSupport', function($rootScope, $stateParams, Assert, WT1, Logger, DataikuAPI, FlowToolsRegistry, FlowTool, FlowGraph, ToolBridgeService, $injector) {
    return {
        restrict: 'A',
        link : function(scope, element, attrs) {
            function activateFromStateIfNeeded() {
                Assert.trueish(scope.toolsState, 'no tool state');
                scope.toolsState.otherActive = {}
                $.each(scope.toolsState.active, function(k, v) {
                    if (k != scope.toolsState.currentId) {
                        scope.toolsState.hasOtherActive = true;
                        scope.toolsState.otherActive[k] = v;
                    }
                });

                if (scope.toolsState.currentId) {
                    scope.tool = FlowTool.activateTool(scope.toolsState.active[scope.toolsState.currentId]);
                }
            }

            scope.refreshToolsState = function() {
                DataikuAPI.flow.tools.getSessions($stateParams.projectKey).success(function(data) {
                    scope.toolsState = data;
                    activateFromStateIfNeeded();
                }).error(setErrorInScope.bind(scope));
            };

            /**
             * Register a flow tool start record associated to the project.
             * @param {string} type tool name.
             * @param {Object} data configuration and properties of the tool.
             * @returns {Promise<void>} a promise that will be resolved when the tool has been started successfully and activated.
             */
            scope.startTool = function(type, data) {
                WT1.event("flow-tool-start", {tool: type});
                if (!scope.drawZones.drawZones) {
                    scope.showZones();
                }
                return DataikuAPI.flow.tools.start($stateParams.projectKey, type, data).success(function(data) {
                    scope.toolsState = data;
                    activateFromStateIfNeeded();
                }).error(setErrorInScope.bind(scope));
            };

            scope.$on('projectTagsUpdated', function (e, args) {
                if (scope.tool && scope.tool.type=="TAGS") {
                    scope.tool.refreshState(false, args.updateGraphTags);
                }
            });

            scope.stopAction = function() {
                if (!scope.toolsState || !scope.toolsState.currentId) {
                    Logger.warn('no active tool, cannot stop');
                    return;
                }

                $.each(FlowGraph.get().nodes, function (nodeId) {
                    const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                    nodeElt.classed('focus', false).classed('out-of-focus', false);
                });
                if (scope.tool.type == "PROPAGATE_SCHEMA") { //reset paths colors applied on
                    FlowGraph.getSvg().find('.grey-out-path').each(function () {
                        d3.select(this).classed('grey-out-path', false);
                    });
                }
                scope.tool = FlowTool.unactivateCurrentTool();
                /* if last view was a flow action, we want to refresh the search to get the coloration that matches the query */
                $rootScope.$emit('flowDisplayUpdated', true);
                DataikuAPI.flow.tools.stop($stateParams.projectKey, scope.toolsState.currentId)
                    .success(function (data) {
                        scope.toolsState = data;
                        if (!scope.drawZones.drawZones) {
                            scope.showZones();
                        }
                        activateFromStateIfNeeded();
                    })
                    .error(setErrorInScope.bind(scope));
            };

            scope.activateTool = function(toolId) {
                 DataikuAPI.flow.tools.setActive($stateParams.projectKey, toolId).success(function(data) {
                    scope.toolsState = data;
                    activateFromStateIfNeeded();
                }).error(setErrorInScope.bind(scope));
            };

            const h = $rootScope.$on('stopAction', scope.stopAction);
            scope.$on('$destroy', h);

            scope.refreshToolsState();

            // show zones
            scope.showZones = function () {
                ToolBridgeService.emitShouldDrawZones(true);
            };
            
            scope.startView = function (view) {
                const tool = FlowTool.getCurrent();
                if (tool && tool.def && tool.action) {
                    scope.stopAction();
                } 
                const params = view.getViewParams();
                const options = params ? params.toOptions() : {};
                FlowTool.activateTool(
                    {
                        type: view.getName(),
                        options: options
                    },
                    view.latestData
                );
                // Reset draw zone except for zone view
                if (
                    !scope.drawZones.drawZones &&
                    view.getName() !== 'FLOW_ZONES'
                ) {
                    scope.showZones();
                }
                scope.tool = FlowTool.getCurrent();
                ToolBridgeService.emitViewTool(scope.tool);
            };

            scope.stopView = function () {
                const tool = FlowTool.getCurrent();
                if (!tool || !tool.def || tool.action) {
                    // don't stop an empty tool
                    return;
                }
                scope.tool = FlowTool.unactivateCurrentTool();
                if (!scope.drawZones.drawZones) {
                    scope.showZones();
                }
                ToolBridgeService.emitViewTool(scope.tool);
            };

            var toolBridgeSubscriptions = [];

            /* SUBSCRIPTION AND SIGNALS */
            toolBridgeSubscriptions.push(
                ToolBridgeService.viewActivation$.subscribe(view => {
                    if (view) {
                        scope.startView(view);
                    } else {
                        scope.stopView();
                    }
                })
            );

            toolBridgeSubscriptions.push(
                ToolBridgeService.shouldDrawZones$.subscribe(shouldDrawZones => {
                    if (scope.drawZones.drawZones === shouldDrawZones) {
                        return;
                    }
                    scope.drawZones.drawZones = shouldDrawZones;
                    scope.redrawZone();
                })
            );

            // Cleanup when the scope is destroyed
            scope.$on('$destroy', function() {
                if (toolBridgeSubscriptions) {
                    toolBridgeSubscriptions.forEach(subs => subs.unsubscribe());
                }
                FlowTool.unactivateCurrentTool();
            });
        }
    }
});

app.directive("flowToolFacetElt", function() {
    return {
        template:
            `<label class="horizontal-flex" ng-class="{'single-focused': states[key]}">
                <input type="checkbox" ng-if="!singleFocused" ng-model="states[key]" ng-click="$event.stopPropagation()"/>
                <span class="dib flex horizontal-flex" ng-click="click(key, $event)">
                    <span class="bullet noflex" style="background-color: {{color}};" />
                    <span class="text flex">
                        <span ng-if="!displayGlobalTags">{{displayName ? displayName : (isNumber(key) ? (key | number) : key) }}</span>
                        <span ng-if="displayGlobalTags" ui-global-tag="displayName ? displayName : (isNumber(key) ? (key | number) : key)" object-type="'TAGGABLE_OBJECT'"/>
                    </span>
                    <span class="number noflex">{{number}}</span>
                </span>
            </label>`,
        scope: {
            color: '=',
            key: '=',
            displayName: '=',
            number: '=',
            singleFocused: '=',
            states: '=',
            displayGlobalTags: '='
        },
        link: function(scope, element, attr) {
            scope.click = function(key, evt) {
                if (!scope.states) return;
                $.each(scope.states, function(k) {
                    scope.states[k] = false;
                });
                scope.states[key] = true;
                evt.preventDefault();
                evt.stopPropagation();
            };
            scope.isNumber = n  => angular.isNumber(n);
        }
    }
});
})();

;
(function() {
'use strict';

/*
* This file defines a set of "Standard" flow views
*
* They implement a common API, in particular they are essentially a mapping: node -> single value
* (the single value can be structured but will be displayed as a single value as opposed to multi-valued views like tags)
*
*/
const app = angular.module('dataiku.flow.tools');

app.service('WatchView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('WATCH', 'Watched and starred items', {
            getRepr: val => val.w ? 'Watching' : undefined,
            totem: function(val) {
                return {
                    class: val.s ? 'icon-star' : '',
                    style: val.s ? 'color: gold; font-size: 32px;' : ''
                }
            }
        });
    }
});


app.service('SparkConfigView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('SPARK_CONFIG', 'Spark configurations', {
            getRepr: val => val.inheritConf,
            totem: function(val) {
                return {
                    class: val.conf.length ? 'icon-plus flow-totem-ok' : '',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/spark-config-view-tooltip.html'
        });
    }
});


app.service('ConnectionsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('CONNECTIONS', 'Connections', {
            getRepr: val => val.connection,
            tooltipTemplate: '/templates/flow-editor/tools/connections-view-tooltip.html'
        });
    }
});

app.service('FlowZonesView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('FLOW_ZONES', 'Flow Zones', {
            getRepr: function(val) {
                return val.id;
            },
            tooltipTemplate: '/templates/flow-editor/tools/flow-zones-view-tooltip.html',
            settingsTemplate: '/templates/flow-editor/tools/flow-zones-settings.html'
        });
    }
});

/*
* Note that the tag view is a multi-valued one (each node has several labels)
*/
app.service('TagsView', function($rootScope, $filter, $injector, $stateParams, translate,
    DataikuAPI, CreateModalFromTemplate, TaggableObjectsUtils, TaggingService,
    FlowTool, FlowGraph, FlowToolsUtils, StandardFlowViews) {

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('TAGS', 'Tags', {
            postInit: function(tool) {
                tool.manageTags = function() {
                    CreateModalFromTemplate("/templates/widgets/edit-tags-modal.html", $rootScope, null, function(modalScope) {
                        modalScope.translate = translate;
                        modalScope.tagsDirty = angular.copy(TaggingService.getProjectTags());

                        modalScope.save = function() {
                            TaggingService.saveToBackend(modalScope.tagsDirty)
                                .then(modalScope.resolveModal)
                                .catch(setErrorInScope.bind(modalScope));
                        };
                        modalScope.cancel = function() {modalScope.dismiss();};
                    });
                };
                tool.displayGlobalTags = true;
            },
            getRepr: function(val) {
                return val;
            },
            postProcessNode: function(tags, nodeElt, tool) {
                if (!tags) return;
                tags.forEach(function(tag, idx) {
                    function onClick() {
                        tool.user.focus(tag);
                        $rootScope.$digest();
                        d3.event.stopPropagation();
                        d3.event.preventDefault();
                    }   
                    FlowToolsUtils.addViewValueIndicator(nodeElt, $filter('tagToColor')(tag), idx, onClick);
                });
            },
            actions: {
                setTags: function(tags, nodes, mode) { // mode = TOGGLE, ADD or REMOVE
                    const request = {
                        elements: nodes.map(TaggableObjectsUtils.fromNode),
                        operations: [{mode: mode, tags: tags}]
                    };

                    DataikuAPI.taggableObjects.applyTagging($stateParams.projectKey, request).success(function(data) {
                        TaggingService.bcastTagUpdate(false, true);
                    }).error(FlowGraph.setError());
                }
            },
            autoSelectFirstOnly: true,
            tooltipTemplate: '/templates/flow-editor/tools/tags-view-tooltip.html'
        });
    }
});


app.service('CustomFieldsView', function($rootScope, FlowTool, StandardFlowViews, objectTypeFromNodeFlowType) {
    function getSelectedOption(value, fromLabel) {
        if (!FlowTool.getCurrent().currentSession || !FlowTool.getCurrent().currentSession.options || FlowTool.getCurrent().currentSession.options.selectedCustomField) {
            return null;
        }
        // TODO: selectedCustomField  come from Angular 2 now adays
        let selectedCustomField = FlowTool.getCurrent().currentSession.options.selectedCustomField;
        for (let taggableType in $rootScope.appConfig.customFieldsMap) {
            if ($rootScope.appConfig.customFieldsMap.hasOwnProperty(taggableType)) {
                let componentList = $rootScope.appConfig.customFieldsMap[taggableType];
                for (let i = 0; i < componentList.length; i++) {
                    let paramDesc = (componentList[i].customFields.filter(cf => cf.type == 'SELECT' && cf.selectChoices) || []).find(cf => cf.name == selectedCustomField);
                    if (paramDesc) {
                        let selOpt = (paramDesc.selectChoices || []).find(function(choice) {
                            if (fromLabel) {
                                return value && choice.label == value;
                            } else {
                                return value ? choice.value == value : (paramDesc.defaultValue && choice.value == paramDesc.defaultValue);
                            }
                        });
                        if (selOpt) {
                            return selOpt;
                        }
                    }
                }
            }
        }
        return null;
    }

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('CUSTOM_FIELDS', 'Metadata fields', {
            getRepr: function(val) {
                let selOpt = getSelectedOption(val);
                return (selOpt && (selOpt.label || selOpt.value)) || val;
            },
            postInit: function(tool) {
                tool.objectTypeFromNodeFlowType = objectTypeFromNodeFlowType;
            },
            tooltipTemplate: '/templates/flow-editor/tools/custom-fields-view-tooltip.html'
        });
    };
});


/*
* Note that the scenarios view is a multi-valued one (each node has several labels)
*/
app.service('ScenariosView', function($rootScope, FlowToolsUtils, StandardFlowViews) {
    const ACTIONS = {
        'build_flowitem': 'Build',
        'clear_items': 'Clear',
        'check_dataset': 'Verify rules or run checks',
        'compute_metrics': 'Compute metrics',
        'sync_hive': 'Synchronize Hive',
        'update_from_hive': 'Update from Hive'
    };

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('SCENARIOS', 'Scenarios', {
            getRepr: function(uses) {
                return uses.map(use => use.scenarioName+' ('+use.scenarioId+')');
            },
            postProcessNode: function(uses, nodeElt, tool) {
                if (!uses) return;
                uses.forEach(function(use, idx) {
                    function onClick() {
                        tool.user.focus(use);
                        $rootScope.$digest();
                        d3.event.stopPropagation();
                        d3.event.preventDefault();
                    }
                    const fullId = use.scenarioName+' ('+use.scenarioId+')';
                    FlowToolsUtils.addViewValueIndicator(
                        nodeElt,
                        tool.user.getColor(fullId),
                        idx,
                        onClick
                    );
                });
            },
            actions: {
                getActionsNames(actions) {
                    if (!actions) return;
                    return actions.map(a => ACTIONS[a]);
                }
            },
            autoSelectFirstOnly: true,
            tooltipTemplate: '/templates/flow-editor/tools/scenarios-view-tooltip.html'
        });
    }
});


app.service('FileformatsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('FILEFORMATS', 'File format', {
            getRepr: val => val.formatType,
            tooltipTemplate: '/templates/flow-editor/tools/fileformats-view-tooltip.html'
        });
    }
});


app.service('PipelinesView', function(StandardFlowViews) {
    this.getService = function(toolName) {
        let displayName;
        if (toolName === "SPARK_PIPELINES") {
            displayName = "Spark pipelines";
        } else if (toolName === "SQL_PIPELINES") {
            displayName = "SQL pipelines";
        }
        return {
            getDefinition: function() {
                return StandardFlowViews.getDefinition(toolName, displayName, {
                    getRepr: function(val) {
                        if (val.pipelineId) {
                            return val.pipelineId
                        }
                        if (val.virtualizable) {
                            return null;// Dataset
                        }
                        return;
                    },
                    totem: function(val) {
                        return {
                            class: val.virtualizable ? 'icon-forward flow-totem-ok' : '',
                            style: ''
                        }
                    },
                    tooltipTemplate: '/templates/flow-editor/tools/pipeline-view-tooltip.html',
                });
            }
        }
    };
});


app.service('ImpalaWriteModeView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('IMPALA_WRITE_MODE', 'Impala write mode', {
            getRepr: val => val
        });
    }
});


app.service('HiveModeView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('HIVE_MODE', 'Hive mode', {
            getRepr: function(val) {
                if (val == "HIVECLI_LOCAL") return "Hive CLI (isolated metastore)";
                if (val == "HIVECLI_GLOBAL") return "Hive CLI (global metastore)";
                if (val == "HIVESERVER2") return "HiveServer2";
                return val;
            }
        });
    }
});


app.service('PartitioningView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('PARTITIONING',  'Partitioning schemes', {
            getRepr: function(val) {
                if (val.dimensions.length) {
                    return val.dimensions.map(x => x.name).sort().join(', ');
                } else {
                    return 'Not partitioned';
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/partitioning-view-tooltip.html'
        });
    }
});


app.service('PartitionsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('PARTITIONS', 'Partitions count', {
            getRepr: function(val) {
                return val;
            },
            tooltipTemplate: '/templates/flow-editor/tools/partitions-view-tooltip.html'
        });
    }
});


app.service('ColumnUsageView', function(StandardFlowViews, ColorPalettesService, FlowToolsUtils, $rootScope) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('COLUMN_USAGE', 'Column usage', {
            getRepr: function(val) {
                return val;
            },
            postProcessNode: function(values, nodeElt, tool) {
                if (!values) return;
                values.forEach(function(value, idx) {
                    function onClick() {
                        tool.user.focus(value);
                        $rootScope.$digest();
                        d3.event.stopPropagation();
                        d3.event.preventDefault();
                    }
                    FlowToolsUtils.addViewValueIndicator(nodeElt, tool.user.getColor(value), idx, onClick);
                });
            },
            tooltipTemplate: '/templates/flow-editor/tools/column-usage-tooltip.html',
        });
    }
});

app.service('DataQualityView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('DATA_QUALITY', 'Data quality', {
            postInit: function(tool) {
                tool.currentSession.options.includeForeignObjects = tool.currentSession.options.includeForeignObjects || false;
            },
            getRepr: function(val) {
                return {
                    ERROR: 'Error',
                    WARNING: 'Warning',
                    OK: 'OK',
                    EMPTY: 'Empty',
                    _: 'Not Computed'
                } [val.lastOutcome || '_'];
            },
            tooltipTemplate: '/templates/flow-editor/tools/data-quality-tooltip.html',
        });
    }
});


app.service('RecentActivityView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('RECENT_ACTIVITY', 'Recent modifications', {
            getRepr: function(val) {
                return val.numberOfModifications;
            },
            tooltipTemplate: '/templates/flow-editor/tools/recent-activity-view-tooltip.html',
        });
    }
});


app.service('FilesizeView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('FILESIZE', 'File size', {
            getRepr: function(val) {
                let totalValue = parseFloat(val.size.totalValue);
                if (isNaN(totalValue) || totalValue <= 0) {
                    return 'Unknown';
                }
                return totalValue;
            },
            tooltipTemplate: '/templates/flow-editor/tools/filesize-view-tooltip.html',
        });
    }
});


app.service('CountOfRecordsView', function($filter, StandardFlowViews, DataikuAPI, $stateParams, FlowGraph, $rootScope,DatasetsService, Dialogs) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('COUNT_OF_RECORDS', 'Records count', {
            getRepr: function(val) {
                let totalValue = parseFloat(val.countOfRecords.totalValue);
                if (totalValue == -1) {
                    return 'Unknown';
                }
                return totalValue;
            },
            postInit: function(tool) {
                const listDatasetsToCompute = function (datasets, computeOnlyMissingRecordsCount) {
                    if (!computeOnlyMissingRecordsCount) {
                        return datasets
                    }
                    const datasetsToCount = [];
                    for (const dataset of datasets) {
                        if (dataset.id) {
                            const datasetGraphId = graphVizEscape(`dataset_${dataset.id}`);
                            const isRecordCountComputed = (tool.user.state.valueByNode && tool.user.state.valueByNode[datasetGraphId] && tool.user.state.valueByNode[datasetGraphId].countOfRecords && tool.user.state.valueByNode[datasetGraphId].countOfRecords.hasData);
                            if (!isRecordCountComputed) {
                                datasetsToCount.push(dataset);
                            }
                        }
                    }
                    return datasetsToCount
                }
                // Action used in Angular 2
                tool.compute = function (computeOnlyMissingRecordsCount) {
                    DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {datasetsOnly: true})
                        .success((datasets) => {
                            const datasetsToCount = listDatasetsToCompute(datasets, computeOnlyMissingRecordsCount);
                            DatasetsService.refreshSummaries($rootScope, datasetsToCount, true, false, true)
                                .then(function (result) {
                                    tool.refreshState();
                                    return result;
                                })
                                .then(function (result) {
                                    if (result && result.anyMessage) {
                                        Dialogs.infoMessagesDisplayOnly($rootScope, "Datasets statuses update results", result);
                                    }
                                }).catch(FlowGraph.setError())
                        }).error(FlowGraph.setError())
                }
            },
            totem: function(val) {
                const countOfRecordsText = sanitize(val.countOfRecords.totalValue < 0 ? '-' : $filter('longSmartNumber')(val.countOfRecords.totalValue));
                return {
                    class: 'icon-refresh '+(val.autoCompute ? 'flow-totem-ok' : 'flow-totem-disabled'),
                    style: '',
                    text: countOfRecordsText
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/count-of-records-view-tooltip.html'
        });
    }
});



const DATE_REPR = [
    'Just now',
    'Past hour',
    'Past 24h',
    'Past week',
    'Past month',
    'Past year',
    'More than a year ago',
    'Unknown'
];
function simpleTimeDelta(timestamp) {
    if (typeof timestamp == 'string' && !isNaN(parseFloat(timestamp))) { //TODO @flow dirty
        timestamp = parseFloat(timestamp);
    }
    if (!timestamp || typeof timestamp != 'number') {
        return DATE_REPR[7];
    }
    const seconds = (new Date().getTime() - timestamp)/1000;
    if (seconds < 60) {
        return DATE_REPR[0];
    }
    if (seconds < 3600) {
        return DATE_REPR[1];
    }
    if (seconds < 3600*24) {
        return DATE_REPR[2];
    }
    if (seconds < 3600*24*7) {
        return DATE_REPR[3];
    }
    if (seconds < 3600*24*30) {
        return DATE_REPR[4];
    }
    if (seconds < 3600*24*365) {
        return DATE_REPR[5];
    }
    return DATE_REPR[6];
}



app.service('CreationView', function(StandardFlowViews, UserImageUrl, FlowTool) {

    const viewByUser = {
        getRepr: function(val) {
            return val.userLogin;
        }
    };
    const viewByDate = {
        getRepr: function(val) {
            const time = parseFloat(val.time);
            return simpleTimeDelta(time);
        }
    };

    function getSubview() {
        const mode = FlowTool.getCurrent().currentSession.options.mode;
        if (mode == 'BY_USER') {
            return viewByUser;
        }
        return viewByDate;
    }

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('CREATION', 'Creation', {
            postInit: function(tool) {
                tool.currentSession.options.mode = tool.currentSession.options.mode || 'BY_DATE';
            },
            getRepr: function(val) {
                if (!val) return;
                return getSubview().getRepr(val);
            },
            totem: function(val) {
                return {
                    class: 'avatar32',
                    style: "background-image: url('" + UserImageUrl(val.userLogin, 128) + "'); box-sizing: border-box;"
                };
            },
            tooltipTemplate: '/templates/flow-editor/tools/creation-view-tooltip.html',
            settingsTemplate: '/templates/flow-editor/tools/creation-view-settings.html'
        });
    };
});


app.service('LastModifiedView', function(StandardFlowViews, UserImageUrl, FlowTool) {

    const viewByUser = {
        getRepr: function(val) {
            return val.userLogin;
        }
    };
    const viewByDate = {
        getRepr: function(val) {
            const time = parseFloat(val.time);
            return simpleTimeDelta(time);
        }
    };

    function getSubview() {
        const mode = FlowTool.getCurrent().currentSession.options.mode;
        if (mode == 'BY_USER') {
            return viewByUser;
        }
        return viewByDate;
    }

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('LAST_MODIFICATION', 'Last modification', {
            postInit: function(tool) {
                tool.currentSession.options.mode = tool.currentSession.options.mode || 'BY_DATE';
            },
            getRepr: function(val) {
                if (!val) return;
                return getSubview().getRepr(val);
            },
            totem: function(val) {
                return {
                    class: 'avatar32',
                    style: "background-image: url('" + UserImageUrl(val.userLogin, 128) + "'); box-sizing: border-box;"
                };
            },
            tooltipTemplate: '/templates/flow-editor/tools/last-modification-view-tooltip.html',
        });
    };
});


app.service('LastBuildView', function(StandardFlowViews) {

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('LAST_BUILD', 'Last build', {
            getRepr: function(val) {
                if (!val) return;
                const time = parseFloat(val.buildEndTime);
                return simpleTimeDelta(time);
            },
            totem: function(val) {
                return {
                    class: val.buildSuccess ? 'icon-ok flow-totem-ok' : 'icon-remove flow-totem-error',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/last-build-view-tooltip.html',
        });
    };
});


app.service('LastBuildDurationView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('LAST_BUILD_DURATION', 'Last build duration', {
            getRepr: function(val) {
                if (val < 0) return;
                return val;
            },
            tooltipTemplate: '/templates/flow-editor/tools/last-build-duration-view-tooltip.html'
        });
    };
});

app.service('RecipesEnginesView', function($filter, StandardFlowViews) {
    const RECIPE_ENGINE_NAMES = {
        'DSS': 'DSS',
        'USER_CODE': 'User code',
        'PLUGIN_CODE': 'Plugin code',
        'HADOOP_MAPREDUCE': 'Hadoop MapReduce',
        'SPARK': 'Spark',
        'PIG': 'Pig',
        'SQL': 'SQL',
        'HIVE': 'Hive',
        'IMPALA': 'Impala',
        'S3_TO_REDSHIFT': 'S3 to Redshift',
        'REDSHIFT_TO_S3': 'Redshift to S3',
        'AZURE_TO_SQLSERVER': 'Azure to SQLServer',
        'TDCH': 'Teradata Hadoop Connector',
        'GCS_TO_BIGQUERY': 'GCS to BigQuery',
        'BIGQUERY_TO_GCS': 'BigQuery to GCS',
        'S3_TO_SNOWFLAKE': 'S3 to Snowflake',
        'SNOWFLAKE_TO_S3': 'Snowflake to S3',
        'WASB_TO_SNOWFLAKE': 'WASB to Snowflake',
        'SNOWFLAKE_TO_WASB': 'Snowflake to WASB',
        'GCS_TO_SNOWFLAKE': 'GCS to Snowflake',
        'SNOWFLAKE_TO_GCS': 'Snowflake to GCS',
        'AZURE_TO_DATABRICKS': 'Azure to Databricks',
        'DATABRICKS_TO_AZURE': 'Databricks to Azure',
        'S3_TO_DATABRICKS': 'S3 to Databricks',
        'DATABRICKS_TO_S3': 'Databricks to S3',
        'DOCKER/KUBERNETES': 'Docker/Kubernetes' // Fake engine
    };
    const RUN_IN_CONTAINER_SUFFIX = " in container";

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('RECIPES_ENGINES', 'Recipe engines', {
            getRepr: function(engineStatus) {
                let engineType = engineStatus.type;
                let runInContainer = engineType.endsWith(RUN_IN_CONTAINER_SUFFIX);
                if (runInContainer) {
                    engineType = engineType.slice(0, -RUN_IN_CONTAINER_SUFFIX.length);
                }
                let result = RECIPE_ENGINE_NAMES[engineType];
                if (!result) {
                    return $filter('capitalize')(engineStatus.type.toLowerCase().replaceAll('_', ' '));
                }
                return result + (runInContainer ? RUN_IN_CONTAINER_SUFFIX : '');
            },
            totem: function(val) {
                return {
                    class: val.statusWarnLevel === 'ERROR' ? 'icon-remove flow-totem-error' : '',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/recipes-engines-view-tooltip.html'
        });
    }
});

app.service('RecipesCodeEnvsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('RECIPES_CODE_ENVS', 'Recipe code environments', {
            getRepr: function(codeEnvState) {
                return codeEnvState.selectedEnvName || codeEnvState.envName || 'DSS builtin env';
            },
            totem: function(val) {
                return {
                    class: val.preventedByProjectSettings ? 'icon-remove flow-totem-error' : '',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/recipes-code-envs-view-tooltip.html',
        });
    }
});


app.service('StandardFlowViews', function ($stateParams, Ng1ToolBridgeService, Debounce, FlowViewsUtils, FlowGraph, FlowGraphHighlighting) {
    this.getDefinition = function (name, displayName, {
        // TODO every view is holding a definition of this method, but it should come from Angular 2 so we have a single source of truth
        getRepr,
        totem,
        tooltipTemplate,
        settingsTemplate,
        postInit,
        postProcessNode,
        actions
    }) {

    return {
        getName: () => name,
        getToolDisplayName: () => displayName,

        initFlowTool: function (tool, viewData) {
            tool.user = {};
            tool.projectKey = $stateParams.projectKey;
            tool.update = (viewData) => {
                tool.user.state = viewData;
                tool.currentSession.options.mode = tool.user.state.mode;

                const countByValue = {};
                $.each(tool.user.state.valueByNode, function (nodeId, val) {
                    const repr = getRepr(val);
                    if (angular.isArray(repr)) {
                        repr.forEach(function (it) {
                            countByValue[it] = (countByValue[it] || 0) + 1;
                        });
                    } else if (repr !== null && repr !== undefined) {
                        countByValue[repr] = (countByValue[repr] || 0) + 1;
                    }
                });
                tool.user.state.countByValue = countByValue;
                tool.user.state.values = Object.keys(countByValue);

                if (postInit) {
                    postInit(tool);
                }

                tool.drawHooks.updateFlowToolDisplay();
                Ng1ToolBridgeService().viewLoaded();
            }

            tool.refreshState = function () {
                Ng1ToolBridgeService().emitRefreshView();
            };

            tool.refreshStateLater = Debounce().withDelay(400, 400).wrap(tool.refreshState);

            FlowViewsUtils.addFocusBehavior(tool, true);
            tool.user.getColor = function (repr) {
                if (!repr || repr === 'Unknown' || !Ng1ToolBridgeService().activeView) {
                    return '#333';
                }
                return Ng1ToolBridgeService().activeView.getColor(repr);
            };

            tool.drawHooks.updateFlowToolDisplay = function () {
                if (!tool.user.state) return; // protect against slow state fetching
                if (!FlowGraph.ready()) return; // protect against slow graph fetching

                // TODO @flow too slow?
                $.each(FlowGraph.get().nodes, function (nodeId, node) {
                    let realNodeId = node.realId || nodeId;
                    const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                    if (tool.user.state.valueByNode[realNodeId] === undefined) {
                        $('.node-totem span', nodeElt[0]).removeAttr('style').removeClass();
                        $('.nodecounter__text div span', nodeElt[0]).text('');
                        $('.never-built-computable *', nodeElt[0]).removeAttr('style');
                    }
                    $('.nodecounter__wrapper', nodeElt[0]).removeClass('nodecounter__wrapper--shown');
                });

                $('.tool-simple-zone', FlowGraph.getSvg()).empty();

                // We first iterate over all non-recipes then on recipes,
                // This is because in some cases, recipes color their outputs
                function styleNodes(recipesOnly) {
                    $.each(FlowGraph.get().nodes, function (nodeId, node) {
                        let val = tool.user.state.valueByNode[node.realId];
                        if (!node) { // If some nodes are broken, they might not be rendered in the flow
                            return;
                        }
                        const isRecipe = node.nodeType == 'RECIPE';
                        if (recipesOnly != isRecipe) {
                            return;
                        }
                        const isZone = node.nodeType == "ZONE";
                        if (isZone && Ng1ToolBridgeService().activeView && tool.user.state.focusMap && tool.user.state.focusMap[node.name] && !$stateParams.zoneId) {
                            FlowGraphHighlighting.highlightZoneCluster(d3.select(`g[id=cluster_zone_${node.name}]`)[0][0], Ng1ToolBridgeService().activeView.getColor(node.name));
                        }
                        if (val !== undefined) {
                            const nodeElt =  FlowGraph.d3NodeWithIdFromType(nodeId, node.nodeType);
                            if (!isZone && FlowGraph.rawNodeWithId(nodeId) === undefined) {
                                return;
                            }
                            const nodeTotem = $('.node-totem span', nodeElt[0]);
                            nodeTotem.removeAttr('style').removeClass();
                            if (totem && totem(val)) {
                                nodeTotem.attr('style', totem(val).style).addClass(totem(val).class);
                            }

                            const nodeCounter = $('.nodecounter__text div span', nodeElt[0]);
                            const nodeCounterWrapper = $('.nodecounter__wrapper', nodeElt[0]);
                            nodeCounter.text('');
                            if (totem && totem(val)) {
                                if (totem(val).text !== undefined) {
                                    const nodeCounterText = totem(val).text;
                                    nodeCounter.text(nodeCounterText);
                                    nodeCounterWrapper.addClass("nodecounter__wrapper--shown");
                                }
                            }

                            if (postProcessNode) {
                                postProcessNode(val, nodeElt, tool);
                            }
                        }
                    });
                }
                styleNodes(false);
                styleNodes(true);
            };

            tool.drawHooks.setupTooltip = function (node) {
                const activeView = Ng1ToolBridgeService().activeView;
                if (!tool.user || !tool.user.state || !activeView) return;
                const tooltip = {};
                tooltip.val = tool.user.state.valueByNode[node.realId];
                tooltip.template = tooltipTemplate || '/templates/flow-editor/tools/default-view-tooltip.html';
                const nodeViewCategories = activeView.getNodeCategories(node);
                if (!tooltip.val || nodeViewCategories.length === 0) {
                    return tooltip;
                }
                if (nodeViewCategories.length === 1) {
                    const category = nodeViewCategories[0];
                    let bulletText = tool.type == "FLOW_ZONES" ? tooltip.val.name : category;
                    tooltip.bullets = [{ text: bulletText, color: activeView.getColor(category) }];
                } else if (nodeViewCategories.length > 1) {
                    const focused = tool.user.getFocusedAsList();
                    const matchedValues = nodeViewCategories.filter(_ => focused.indexOf(_) !== -1);
                    if (matchedValues.length === 1) {
                        tooltip.bullets = [{text: matchedValues[0], color: activeView.getColor(matchedValues[0])}];
                    } else {
                        tooltip.bullets = []
                        matchedValues.forEach((value) => {
                            tooltip.bullets.push({ text: value, color: activeView.getColor(value) });
                        });
                    }
                }
                return tooltip;
            };

            tool.getRepr = getRepr;
            tool.def.settingsTemplate = settingsTemplate;
            tool.actions = actions;

            tool.update(viewData);
        }
    };
};
});


app.controller("StandardFlowViewsMainController", function($scope, FlowTool) {
    $scope.tool = FlowTool.getCurrent();
});
})();

;
(function() {
'use strict';

const app = angular.module('dataiku.flow.tools');


app.service('CopyFlowTool', function($rootScope, $stateParams,
    DataikuAPI, FlowToolsUtils, FlowGraph, Ng1ToolBridgeService) {

    const NAME = 'COPY';
    const DISPLAY_NAME = 'Copy';

    this.getDefinition = function() {
        return {
            getName: () => NAME,
            getToolDisplayName: () => DISPLAY_NAME,

            initFlowTool: function(tool) {
                tool.user = {
                    updateOptions: {
                        recheckAll: false,
                        datasets: {
                            consistencyWithData: true
                        },
                        recipes: {
                            schemaConsistency: true,
                            otherExpensiveChecks: true
                        }
                    },
                    updateStatus: {
                        updating: false
                    }
                };
                tool.action = true;
                Ng1ToolBridgeService().emitActionTool(true);

                /*
                * Since the items to copy is not he user selection (we force recipes outputs, etc)
                * we maintain a user selection and recompute the list to copy when updated
                * to be sure that the graph updates don't change the nodes objects, and break the lists lookups
                * we only keep ids there
                */

                function updateNodeStates() {
                    tool.user.state.stateByNode = {};
                    tool.user.state.countByState = {REQUESTED: 0, REQUIRED: 0, REUSED: 0}
                    const stateByNode = tool.user.state.stateByNode;

                    // select the requested items
                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        if (tool.user.state.requested[nodeId]) {
                            if (node.projectKey === $stateParams.projectKey) {
                                stateByNode[nodeId] = 'REQUESTED';
                            } else {
                                stateByNode[nodeId] = 'REUSED'; //Can't deep copy a foreign dataset
                            }
                        }
                    });
                    // By default, don't copy the sources of the subflow, reuse them, copy only if forced
                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        if (tool.user.state.requested[nodeId] === 'REQUESTED' && node.nodeType !== 'RECIPE' && node.nodeType !== 'LABELING_TASK') {
                            for (let p of node.predecessors) {
                                if (stateByNode[p]) {
                                    return; //A predecessor is requested => not a source for the subflow
                                }
                            }
                            let anyCopiedSuccessor = false;
                            for (let p of node.successors) {
                                if (stateByNode[p]) {
                                    anyCopiedSuccessor = true;
                                    break;
                                }
                            }
                            if (!anyCopiedSuccessor) {
                                return; //Isolated node, the user probably actually want to copy it
                            }
                            stateByNode[nodeId] = 'REUSED';
                        }
                    });
                    // select the non requested but required items
                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        if (stateByNode[nodeId]) {
                            if (['RECIPE', 'LOCAL_SAVEDMODEL', 'LABELING_TASK'].includes(node.nodeType)) {
                                $.each(node.successors, function(index, nodeId2) {
                                    if (!stateByNode[nodeId2]) {
                                        stateByNode[nodeId2] = 'REQUIRED';
                                    }
                                });
                            }
                        }
                    });
                    // select the non requested, non required but reused items
                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        if (stateByNode[nodeId]) {
                            if (['RECIPE', 'LOCAL_SAVEDMODEL', 'LABELING_TASK'].includes(node.nodeType)) {
                                $.each(node.predecessors, function(index, nodeId2) {
                                    if (!stateByNode[nodeId2]) {
                                        stateByNode[nodeId2] = 'REUSED';
                                    }
                                });
                            } 
                        }
                    });

                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        const nodeState = stateByNode[nodeId];
                        if (nodeState) {
                            tool.user.state.countByState[nodeState]++;
                        }
                    });
                }

                const COLORS = {
                    'REQUESTED': 'green',
                    'REQUIRED': '#41f544',
                    'REUSED': '#ffc500'
                };

                function colorNodes() {
                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        const nodeElt = FlowGraph.d3NodeWithIdFromType(nodeId, node.nodeType);
                        const nodeState = tool.user.state.stateByNode[nodeId];

                        //TODO @flow factorize cleanNode
                        nodeElt.classed('focus', false).classed('out-of-focus', false);
                        $('.tool-simple-zone', FlowGraph.getSvg()).empty();
                        $('.node-totem span', nodeElt[0]).removeAttr('style').removeClass();
                        $('.never-built-computable *', nodeElt[0]).removeAttr('style');

                        const color = COLORS[nodeState] || '#e2e2e2';
                        FlowToolsUtils.colorNode(node, nodeElt, color);

                    });
                }

                tool.drawHooks.updateFlowToolDisplay = function() {
                    if (!tool.user.state) return; // protect against slow state fetching
                    if (!FlowGraph.ready()) return; // protect against slow graph fetching

                    updateNodeStates();
                    colorNodes();
                }

                DataikuAPI.flow.tools.getState($stateParams.projectKey, NAME, {}).success(function(data) {
                    tool.user.state = data;
                    tool.user.state.requested = tool.user.state.requested || [];

                    if (tool.user.state.preselectedNodes) {
                        tool.user.state.preselectedNodes.forEach(function(nodeId) {
                            tool.user.state.requested[nodeId] = 'REQUESTED';
                        });
                    }

                    tool.drawHooks.updateFlowToolDisplay();
                }).error(FlowGraph.setError());
                return tool;
            },

            template: "/templates/flow-editor/tools/tool-copy.html"
        };
    };
});

app.controller("CopyToolController", function($scope, $stateParams, Assert, DataikuAPI, TaggableObjectsUtils, FlowGraphSelection, FlowGraph, FlowToolsUtils, SubFlowCopyService) {
    Assert.inScope($scope, 'tool');

    $scope.addSelected = function(forceAdd) {
        const statesBefore = angular.copy($scope.tool.user.state.stateByNode);
        const requested = $scope.tool.user.state.requested;
        FlowGraphSelection.getSelectedNodes().forEach(function(it) {
            if(requested[it.id] != 'FORCED') {
                requested[it.id] = forceAdd ? 'FORCED' : 'REQUESTED';
            }
        });
        $scope.tool.drawHooks.updateFlowToolDisplay();

        // This had no effect, try with force
        if (!forceAdd && angular.equals($scope.tool.user.state.stateByNode, statesBefore)) {
            $scope.addSelected(true);
        }
    };

    $scope.removeSelected = function() {
        const requested = $scope.tool.user.state.requested;
        FlowGraphSelection.getSelectedNodes().forEach(function(it) {
            delete requested[it.id];
        });
        $scope.tool.drawHooks.updateFlowToolDisplay();
    };

    $scope.reset = function() {
        $scope.tool.user.state.requested = [];
        $scope.tool.drawHooks.updateFlowToolDisplay();
    };

    function getSelectedTaggableObjectRefs() {
        const items = [];
        const itemsByZones = {};
        $.each(FlowGraph.get().nodes, function(nodeId, node) {
            if (nodeId.startsWith("zone__") && node.nodeType !== 'ZONE' && (!node.isSource || node.isSink || node.nodeType === 'RECIPE')) {
                // example data
                // `nodeId`: zone__default__recipe__compute__customers__copy__27
                // `realId`: recipe__compute__customers__copy__27
                // `node.id`:  zone__default__recipe__compute__customers__copy__27
                const zoneId = nodeId.substring("zone__".length, node.id.length - node.realId.length - 2);
                if (node.ownerZone === zoneId) {
                    if (!itemsByZones[zoneId]) {
                        itemsByZones[zoneId] = [];
                    }
                    const zoneContent = itemsByZones[zoneId];
                    zoneContent.push(TaggableObjectsUtils.fromNode(node));
                }
            }
            if (['REQUESTED', 'REQUIRED'].includes($scope.tool.user.state.stateByNode[nodeId])) {
                items.push(TaggableObjectsUtils.fromNode(node));
            }
        });
        return {selectedTaggableObjectRefs: items, itemsByZones};
    }

    $scope.go = function() {
        const { selectedTaggableObjectRefs, itemsByZones } = getSelectedTaggableObjectRefs();
        SubFlowCopyService.start(selectedTaggableObjectRefs, itemsByZones);
    };
});

})();
;
(function() {
'use strict';

const app = angular.module('dataiku.flow.tools');


app.service('PropagateSchemaFlowTools', function($rootScope, $stateParams,
    DataikuAPI, ContextualMenu, Logger,
    FlowGraph, FlowViewsUtils, FlowToolsUtils, $q, ComputableSchemaRecipeSave, Ng1ToolBridgeService) {

    const NAME = 'PROPAGATE_SCHEMA';

    this.getDefinition = function() {
        return {
            getName: () => NAME,
            getToolDisplayName: function(toolDef) {
                return "Check schema from " + toolDef.toolInitialData.datasetName;
            },

            initFlowTool: function(tool) {

                tool.user = {
                    updateStatus: {
                        updating: false
                    },
                    preconfiguredProfile: "AUTO_FAST",
                    updateOptions: {
                        performExpensive: true,
                        doAnyRebuildingReqd: false,
                        doBuildAll: false,
                        recheckAll: false,
                        markUncheckableAsOK: false,
                        alwaysRebuildInputOfRecipesUsuallyComputingOutputSchemaBasedOnData: true,
                        alwaysRebuildOutputOfRecipesUsuallyComputingOutputSchemaAtRuntime: true
                    }
                };
                tool.action = true;
                Ng1ToolBridgeService().emitActionTool(true);

                tool.user.canMarkRecipeAsOK = function(recipeName) {
                    let nodeId = graphVizEscape("recipe_" + recipeName);
                    if (tool.user.state &&  tool.user.state.stateByNode[nodeId]) {
                        return tool.user.state.stateByNode[nodeId].state != "OK";
                    } else {
                        return false;
                    }
                }

                tool.user.markRecipeAsOK = function(recipeName) {
                    DataikuAPI.flow.tools.propagateSchema.markRecipeAsOKForced($stateParams.projectKey, recipeName).success(function(data) {
                        tool.user.state = data;
                        tool.drawHooks.updateFlowToolDisplay();
                    })
                }

                tool.drawHooks.updateFlowToolDisplay = function() {
                    if (!tool.user.state) return; // protect against slow state fetching
                    if (!FlowGraph.ready()) return; // protect against slow graph fetching

                    let svg = FlowGraph.getSvg();
                    tool.user.state.itemsToRebuild = [];

                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                        const nodeState = tool.user.state.stateByNode[node.realId];

                        //TODO @flow factorize cleanNode
                        nodeElt.classed('focus', false).classed('out-of-focus', false);
                        $('.tool-simple-zone', FlowGraph.getSvg()).empty();
                        $('.node-totem span', nodeElt[0]).removeAttr('style').removeClass();
                        $('.never-built-computable *', nodeElt[0]).removeAttr('style');

                        // Logger.info("NodeState: ", nodeId, nodeState);
                        if (!nodeState) {
                            const color = "#E9E9E9";
                            // Node is not involved in this
                            FlowToolsUtils.colorNode(node, nodeElt, color);

                            svg.find('[data-to="' + nodeId + '"]').each(function () {
                                d3.select(this).classed('grey-out-path',true).selectAll("path, ellipse").classed('grey-out-path',true);
                            });
                            svg.find('[data-from="' + nodeId + '"]').each(function () {
                                d3.select(this).classed('grey-out-path',true).selectAll("path, ellipse").classed('grey-out-path',true);
                            });
                        } else if (nodeState.state == "DATASET_NEEDS_REBUILD") {
                            node.partitioning = nodeState.partitioning;
                            node.buildPartitions = nodeState.buildPartitions;
                            tool.user.state.itemsToRebuild.push({type: 'DATASET', node: node});
                            FlowToolsUtils.colorNode(node, nodeElt, "orange");
                        } else if (nodeState.state == "UNCHECKABLE") {
                            FlowToolsUtils.colorNode(node, nodeElt, "orange");
                        } else if (nodeState.state == "EXCLUDED") {
                            FlowToolsUtils.colorNode(node, nodeElt, "lightgrey");
                        } else if (nodeState.state == "UNCHECKED") {
                            FlowToolsUtils.colorNode(node, nodeElt, "grey");
                        } else if (nodeState.state == "OK") {
                            FlowToolsUtils.colorNode(node, nodeElt, "green");
                        } else if (nodeState.state == "NOK") {
                            FlowToolsUtils.colorNode(node, nodeElt, "red");
                         } else if (nodeState.state == "FAILED_CHECK") {
                            FlowToolsUtils.colorNode(node, nodeElt, "purple");
                        }
                    });
                }

                tool.user.ignoreAllSuggestionsWithState = function(parentScope, state) {
                    var promises = [];

                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                        const nodeState = tool.user.state.stateByNode[node.realId];

                        if (nodeState && nodeState.state == state) {
                            Logger.info("Ignoring suggestion on node", nodeId);
                            const recipeName = node.name;

                            var deferred = $q.defer();
                            DataikuAPI.flow.tools.propagateSchema.markRecipeAsOKForced($stateParams.projectKey, recipeName).success(function(data) {
                                deferred.resolve();
                            }).error(FlowGraph.setError());
                            promises.push(deferred.promise);
                        }
                    });
                    Logger.info("Waiting on ", promises.length, "promises")
                    $q.all(promises).then(function() {
                        Logger.info("Done Waiting on ", promises.length, "promises");
                        parentScope.update();
                    });
                }

                tool.user.acceptAllRecipeSuggestions = function(parentScope){
                    var promises = [];

                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                        const nodeState = tool.user.state.stateByNode[node.realId];

                        if (nodeState && nodeState.state == "NOK") {
                            Logger.info("Accepting suggestion on node", nodeId);
                            const recipeName = node.name;

                            var deferred = $q.defer();

                            ComputableSchemaRecipeSave.handleSchemaUpdateWithPrecomputedUnattended(parentScope,
                                nodeState.updateSolution).then(function() {
                                Logger.info("Acceptance done on ", recipeName)
                                DataikuAPI.flow.tools.propagateSchema.markRecipeAsOKAfterUpdate($stateParams.projectKey, recipeName).success(function(data) {
                                    Logger.info("recipe marked as done, resolving deferred")
                                    deferred.resolve();
                                }).error(FlowGraph.setError());
                            });
                            promises.push(deferred.promise);
                        }
                     });
                    Logger.info("Waiting on ", promises.length, "promises")
                    $q.all(promises).then(function() {
                        Logger.info("Done Waiting on ", promises.length, "promises")
                        parentScope.update();
                    });
                }

                tool.actionHooks.onItemClick = function(node, evt) {
                    if (!tool.user.state) return; // protect against slow state fetching
                    let nodeState = tool.user.state.stateByNode[node.realId];

                    Logger.info("onItemClick nodeState ", nodeState);

                    ContextualMenu.prototype.closeAny();

                    if (nodeState && ["NOK", "FAILED_CHECK", "UNCHECKABLE"].indexOf(nodeState.state) >=0) {
                        let menuScope = $rootScope.$new();
                        menuScope.nodeState = nodeState;
                        menuScope.node = node;

                        new ContextualMenu({
                            template: "/templates/flow-editor/tools/propagate-schema-item-popup.html",
                            scope: menuScope,
                            contextual: false,
                            controller: "FlowToolPropagateItemPopupController"
                        }).openAtEventLoc(evt);
                    } else if (nodeState && nodeState.state == "DATASET_NEEDS_REBUILD") {
                        let menuScope = $rootScope.$new();
                        menuScope.nodeState = nodeState;
                        menuScope.node = node;

                        new ContextualMenu({
                            template: "/templates/flow-editor/tools/propagate-dataset-needs-rebuild-popup.html",
                            scope: menuScope,
                            contextual: false,
                            controller: "FlowToolPropagateDatasetNeedsRebuildPopupController"
                        }).openAtEventLoc(evt);
                    }
                };

                FlowViewsUtils.addAsynchronousStateComputationBehavior(tool);

                DataikuAPI.flow.tools.getState($stateParams.projectKey, NAME, {}).success(function(data) {
                    tool.user.state = data;
                    tool.drawHooks.updateFlowToolDisplay();
                }).error(FlowGraph.setError());
                return tool;
            },

            template: "/templates/flow-editor/tools/tool-propagate-schema.html"
        };
    };
});


app.controller("PropagateSchemaFlowToolMainController", function($scope, $q, Logger, DataikuAPI, $timeout, $stateParams, ActivityIndicator, Assert,
                                                                 JobDefinitionComputer, ComputableSchemaRecipeSave, PartitionSelection, CreateModalFromTemplate, $filter, AnyLoc) {

    checkChangesBeforeLeaving($scope, function () {return $scope.tool.user.updateStatus.multiPhaseUpdating;}, "Schema propagation in progress.  Leaving will abort the operation");

    function getCountOfProblemItems() {
        let sum = $scope.tool.user.state.summary;
        return sum.NOK + sum.UNCHECKABLE + sum.UNCHECKED + sum.DATASET_NEEDS_REBUILD + sum.FAILED_CHECK;
    }

    function isAnyRebuildingReqd() {
        let ret = $scope.tool.user.updateOptions.doAnyRebuildingReqd || $scope.tool.user.updateOptions.doBuildAll;
        Logger.info("isAnyRebuildingReqd:", ret);
        return ret;
    }

    function getItemToRebuild (toDoList, doneList) {
        let item =  toDoList.find( (item) => {
            return !doneList.includes(item.node.id);
        });
        if (item) {doneList.push(item.node.id)}
        return item;
    }

    function showIndicator(txt, isSuccess) {
        Logger.info("Showing indicator text:", txt, "success:", isSuccess);
        $scope.updateText = txt;
        if (isSuccess) {$timeout(() => $scope.updateText = "", 4000);}
    }

    function markDatasetAsOK(datasetName) {
        return DataikuAPI.flow.tools.propagateSchema.markDatasetAsBeingRebuilt($stateParams.projectKey, datasetName);
    }

    $scope.$watch("tool.user.preconfiguredProfile", function(nv, ov) {
        Logger.info("Update profile", nv);
        if (!nv) return;
        switch (nv) {
            case "AUTO_FAST":
                $scope.tool.user.updateOptions.performExpensive = true;
                $scope.tool.user.updateOptions.doAnyRebuildingReqd = true;
                $scope.tool.user.updateOptions.markUncheckableAsOK = true;
                $scope.tool.user.updateOptions.doBuildAll = false;
                $scope.tool.user.updateOptions.alwaysRebuildInputOfRecipesUsuallyComputingOutputSchemaBasedOnData = false;
                $scope.tool.user.updateOptions.alwaysRebuildOutputOfRecipesUsuallyComputingOutputSchemaAtRuntime = false;
                break;
            case "AUTO_THOROUGH":
                $scope.tool.user.updateOptions.performExpensive = true;
                $scope.tool.user.updateOptions.doAnyRebuildingReqd = true;
                $scope.tool.user.updateOptions.markUncheckableAsOK = false;
                $scope.tool.user.updateOptions.doBuildAll = false;
                $scope.tool.user.updateOptions.alwaysRebuildInputOfRecipesUsuallyComputingOutputSchemaBasedOnData = true;
                $scope.tool.user.updateOptions.alwaysRebuildOutputOfRecipesUsuallyComputingOutputSchemaAtRuntime = true;
                break;
            case "INTERACTIVE_FAST":
                $scope.tool.user.updateOptions.performExpensive = true;
                $scope.tool.user.updateOptions.doAnyRebuildingReqd = false;
                $scope.tool.user.updateOptions.doBuildAll = false;
                $scope.tool.user.updateOptions.markUncheckableAsOK = true;
                $scope.tool.user.updateOptions.alwaysRebuildInputOfRecipesUsuallyComputingOutputSchemaBasedOnData = false;
                $scope.tool.user.updateOptions.alwaysRebuildOutputOfRecipesUsuallyComputingOutputSchemaAtRuntime = false;
                break;
            case "INTERACTIVE_THOROUGH":
                $scope.tool.user.updateOptions.performExpensive = true;
                $scope.tool.user.updateOptions.doAnyRebuildingReqd = false;
                $scope.tool.user.updateOptions.doBuildAll = false;
                $scope.tool.user.updateOptions.markUncheckableAsOK = false;
                $scope.tool.user.updateOptions.alwaysRebuildInputOfRecipesUsuallyComputingOutputSchemaBasedOnData = true;
                $scope.tool.user.updateOptions.alwaysRebuildOutputOfRecipesUsuallyComputingOutputSchemaAtRuntime = true;
                break;
         }
         Logger.info("Set options to", $scope.tool.user.updateOptions);
    });

    /**
     * Rebuild a dataset, marking as OK when done.
     * Returns promise to wait on for completion
     */
    function buildDataset(node) {
        const deferred = $q.defer();

        let jd = JobDefinitionComputer.computeJobDefForSingleDataset($stateParams.projectKey, "RECURSIVE_BUILD", node, node.buildPartitions ? node.buildPartitions : {});
        if (!$scope.isAborting) {
            $scope.startedJob = {nodeType:'DATASET', nodeName: node.name, nodeId: node.id};
            DataikuAPI.flow.jobs.start(jd).then((data) => {
                $scope.startedJob.jobId = data.data.id;
                waitForEndOfStartedJob().then (() => {
                    showIndicator("Marking dataset as OK: " + $scope.startedJob.nodeName);
                    let stateInfo = $scope.tool.user.state.stateByNode[node.realId];
                    if (stateInfo && stateInfo.state==="OK") {
                        deferred.resolve();
                    }else {
                        markDatasetAsOK($scope.startedJob.nodeName).finally(() => {
                            deferred.resolve();
                        });
                    }
                }, deferred.reject);
            }, (data) => {
                setErrorInScope.bind($scope)(data.data, data.status, data.headers);
                deferred.reject();
            });
        }
        return deferred.promise;
    }

    function rebuildNextAsReqd() {
        Logger.info("Making next step of auto-progress");
        if ($scope.isAborting) {return false;}
        if ($scope.tool.user.state.summary.UNCHECKABLE > 0) {
            Logger.info("I have some uncheckables, marking them as OK");
            showIndicator("Marking all uncheckable as OK");
            $scope.tool.user.ignoreAllSuggestionsWithState($scope, "UNCHECKABLE");
            return true;
        }
        if ($scope.tool.user.state.summary.NOK > 0) {
            Logger.info("I have some NOK, accepting suggestions");
            showIndicator("Accepting all suggestings");
            $scope.tool.user.acceptAllRecipeSuggestions($scope);
            return true;
        }
        else {
            Logger.info("No NOK nor UNCHECKABLE, building items that need to be built");
            return startRebuildNextItem();
        }
    }

    function startRebuildNextItem() {
        if (!$scope.itemsRebuildAttempted) {$scope.itemsRebuildAttempted = [];}

        const item = getItemToRebuild($scope.tool.user.state.itemsToRebuild, $scope.itemsRebuildAttempted);
        if (!item) {
            return false;
        } else {
            Assert.trueish(item.type === 'DATASET'); // We don't rebuild other stuff at the moment
            showIndicator("Rebuilding dataset " + item.node.name);
            buildDataset(item.node).finally($scope.update);
            return true;
        }
    }

    function markRecipeAsOK(recipeName, nodeId) {
        const nodeState = $scope.tool.user.state.stateByNode[nodeId];

        if (nodeState) {
            Logger.info("Accepting node after update", nodeId);

            let deferred = $q.defer();
            if (nodeState.state == "OK") {
                deferred.resolve();
            } else {
                ComputableSchemaRecipeSave.handleSchemaUpdateWithPrecomputedUnattended($scope,
                    nodeState.updateSolution).then(function () {
                    Logger.info("Acceptance done on ", recipeName)
                    DataikuAPI.flow.tools.propagateSchema.markRecipeAsOKAfterUpdate($stateParams.projectKey, recipeName).success(function (data) {
                        Logger.info("recipe marked as done, resolving deferred")
                        deferred.resolve();
                    }).error(setErrorInScope.bind($scope));
                });
            }
            return deferred.promise;
        }
    }

    function waitForEndOfStartedJob() {
        const deferred = $q.defer();

        function poll() {
            DataikuAPI.flow.jobs.getJobStatus($stateParams.projectKey, $scope.startedJob.jobId).then(function (data) {
                $scope.startedJob.jobStatus = data.data;
                let status = $scope.startedJob.jobStatus.baseStatus.state;

                if (status != "DONE" && status != "ABORTED" && status != "FAILED" && !$scope.isAborting) {
                    $scope.jobCheckTimer = $timeout(function () {
                        poll();
                    }, 2000);
                } else if (status == "DONE") {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            }, setErrorInScope.bind($scope));
        }

        poll();
        return deferred.promise;
    }

    function getBuildAllJobDef() {
        var outputs = $scope.computables.filter(d => !d.removed).map(function(d) {
            // eslint-disable-next-line no-undef
            const fullId = graphIdFor(d.type, AnyLoc.makeLoc(d.projectKey, d.id).fullId);
            const nodeFound = $scope.tool.user.state.stateByNode[fullId];
            if (d.type === 'DATASET') {
                return JobDefinitionComputer.computeOutputForDataset(d.serializedDataset, nodeFound ? nodeFound.buildPartitions : PartitionSelection.getBuildPartitions(d.serializedDataset.partitioning));
            } else if (d.type === 'MANAGED_FOLDER') {
                return JobDefinitionComputer.computeOutputForBox(d.box, nodeFound ? nodeFound.buildPartitions : PartitionSelection.getBuildPartitions(d.box.partitioning));
            } else {
                return { "targetDataset": d.id, "targetDatasetProjectKey": d.projectKey, "type": d.type };
            }
        });

        return {
            "type": "RECURSIVE_BUILD",
            "refreshHiveMetastore":true,
            "projectKey": $stateParams.projectKey,
            "outputs": outputs
        };
    }

    function startBuildAllJob(computables) {
        if ($scope.isAborting) {return;}

        $scope.computables = computables;
        showIndicator("Starting to build all...");
        DataikuAPI.flow.jobs.start(getBuildAllJobDef()).then((data) => {
            $scope.startedJob = {"jobId": data.data.id};
            showIndicator("Build all started...");
            waitForEndOfStartedJob().then (() => {
                    showIndicator("Build all completed", true);
                    $scope.tool.user.updateStatus.multiPhaseUpdating = false;
                }, () => {
                    showIndicator("Build all failed", true);
                    $scope.tool.user.updateStatus.multiPhaseUpdating = false;})
        }, setErrorInScope.bind($scope));
    }

    $scope.update = function(isStart) {
        const deferred = $q.defer();
        if (isStart) {
            $scope.isAborting = false;
            $scope.itemsRebuildAttempted = [];
            $scope.tool.user.updateOptions.recheckAll = (getCountOfProblemItems() == 0);
            $scope.tool.user.partitionedObjects = Object.values($scope.tool.user.state.stateByNode).filter(e => e.partitioning).map(e => Object.assign(e, {buildPartitions: [], name: AnyLoc.getLocFromSmart($stateParams.projectKey, e.fullId).localId }));
            if ($scope.tool.user.partitionedObjects && $scope.tool.user.partitionedObjects.length > 0 && isAnyRebuildingReqd()) {
                CreateModalFromTemplate("templates/flow-editor/tools/tool-propagate-schema-partitioning.html", $scope, null, function(newScope) {
                    newScope.computables = $scope.tool.user.partitionedObjects;
                    newScope.getIcon = computable => {
                        switch(computable.type) {
                            case 'DATASET':            return 'dataset ' + $filter('datasetTypeToIcon')(computable.subType);
                            case 'MANAGED_FOLDER':     return 'icon-folder-open';
                            case 'SAVED_MODEL':        return 'icon-machine_learning_regression';
                        }
                    };

                    newScope.continue = () => {
                        newScope.dismiss();
                        deferred.resolve();
                    }
                });
            } else {
                deferred.resolve();
            }
        } else {
            $scope.tool.user.updateOptions.recheckAll = false;
            deferred.resolve();
        }
        deferred.promise.finally(() => {
            if (!$scope.tool.user.updateStatus.multiPhaseUpdating) {
                $scope.tool.user.updateStatus.multiPhaseUpdating = isAnyRebuildingReqd();
                $scope.totalItemsToProcess = getCountOfProblemItems();
            }
            $scope.tool.user.updateStatus.totalPercent = 0;
            showIndicator("Propagating schema");
            $scope.tool.user.update($scope).then(() => {
                let multiPhase = false;
                for (const partitionedObject of $scope.tool.user.partitionedObjects) {
                    // eslint-disable-next-line no-undef
                    const escapedId = graphIdFor(partitionedObject.type, partitionedObject.fullId);
                    // Try with the index and fallback on slow search in case we missed a specific case for graphIds
                    let node = $scope.tool.user.state.stateByNode[escapedId];
                    if (node === undefined) {
                        node = Object.values($scope.tool.user.state.stateByNode).find(n => n.type === partitionedObject.type && n.fullId === partitionedObject.fullId)
                    }
                    if (node) {
                        node.buildPartitions = partitionedObject.buildPartitions;
                        node.partitioning = partitionedObject.partitioning;
                    }
                }
                $scope.tool.drawHooks.updateFlowToolDisplay()

                if ($scope.tool.user.updateOptions.recheckAll) {
                    $scope.totalItemsToProcess = getCountOfProblemItems();
                }

                if (isAnyRebuildingReqd()) {
                    multiPhase = rebuildNextAsReqd();
                }

                if (!multiPhase) {
                    if ($scope.tool.user.updateOptions.doBuildAll && !$scope.isAborting ) {
                        $scope.buildAll();
                        multiPhase = true;
                    }
                    else {
                        showIndicator($scope.isAborting ? "Schema propagation aborted ": "Schema propagation complete", true);
                    }
                }
                $scope.tool.user.updateStatus.multiPhaseUpdating = multiPhase;
            });

            $scope.tool.drawHooks.updateFlowToolDisplay();
        });
    };

    /**
     * Handles the implicitly checked boxes for doAnyRebuildingReqd and markUncheckableAsOK when profile === 'CUSTOM'
     */
    let lastDoAnyRebuildingReqdValue;
    let lastMarkUncheckableAsOKValue;

    // initial & consistent state when switching from an other profile
    $scope.$watch(() => $scope.tool.user.preconfiguredProfile === 'CUSTOM', (isCustom) => {
        if(!isCustom) return; // we apply auto-checks only in custom mode

        if($scope.tool.user.updateOptions.doBuildAll) {
            lastDoAnyRebuildingReqdValue = true;
            $scope.tool.user.updateOptions.doAnyRebuildingReqd = true;
        }
        if($scope.tool.user.updateOptions.doAnyRebuildingReqd) {
            lastMarkUncheckableAsOKValue = true;
            $scope.tool.user.updateOptions.markUncheckableAsOK = true;
        }
    })

    // save & restore doAnyRebuildingReqd when doBuildAll changes
    $scope.$watch('tool.user.updateOptions.doBuildAll', function (newVal) {
        if($scope.tool.user.preconfiguredProfile === 'CUSTOM') {
            if(newVal) {
                // save previously selected value
                lastDoAnyRebuildingReqdValue = $scope.tool.user.updateOptions.doAnyRebuildingReqd;
                $scope.tool.user.updateOptions.doAnyRebuildingReqd = true;
            } else {
                // restore previously selected value
                $scope.tool.user.updateOptions.doAnyRebuildingReqd = lastDoAnyRebuildingReqdValue;
            }
        }
    });

    // save & restore markUncheckableAsOK when doAnyRebuildingReqd changes
    $scope.$watch('tool.user.updateOptions.doAnyRebuildingReqd', function (newVal) {
        if($scope.tool.user.preconfiguredProfile === 'CUSTOM') {
            if(newVal) {
                // save previously selected value
                lastMarkUncheckableAsOKValue = $scope.tool.user.updateOptions.markUncheckableAsOK;
                $scope.tool.user.updateOptions.markUncheckableAsOK = true;
            } else {
                // restore previously selected value
                $scope.tool.user.updateOptions.markUncheckableAsOK = lastMarkUncheckableAsOKValue;
            }
        }
    });

    $scope.acceptAllRecipeSuggestions = function(){
         $scope.tool.user.acceptAllRecipeSuggestions($scope);
    };

    $scope.ignoreAllRecipeSuggestions = function(){
         $scope.tool.user.ignoreAllSuggestionsWithState($scope, "NOK");
    };

    $scope.markAllUncheckableAsOK = function(){
        $scope.tool.user.ignoreAllSuggestionsWithState($scope, "UNCHECKABLE");
    };

    $scope.getTotalPercent = function() {
        if ($scope.tool.user.updateStatus.multiPhaseUpdating) {
            let fixedItemsToDo = 1 + ($scope.tool.user.updateOptions.doBuildAll || 0); // '1 +' is allow us to show some progress from the start - gives positive feedback to user
            return 100 * (1+ $scope.totalItemsToProcess - getCountOfProblemItems()) / (fixedItemsToDo + Math.max($scope.totalItemsToProcess,1));
        }
        else {
            return $scope.tool.user.updateStatus.totalPercent;
        }
    };

    $scope.abortUpdate = function () {
        $scope.isAborting = true;
        if ($scope.startedJob && $scope.startedJob.jobId) {
            DataikuAPI.flow.jobs.abort($stateParams.projectKey, $scope.startedJob.jobId).error(setErrorInScope.bind($scope));
        }
        $scope.tool.user.updateStatus.multiPhaseUpdating = false;
        $scope.tool.user.updateStatus.updating = false;
        ActivityIndicator.hide();
    };

    $scope.buildAll = function () {
        DataikuAPI.flow.listDownstreamComputables($stateParams.projectKey, {computable: $scope.tool.currentSession.toolInitialData.datasetName})
            .success((computables) => {
                startBuildAllJob(computables);
            })
            .error(setErrorInScope.bind($scope));
    };
});

app.controller("FlowToolPropagateItemPopupController", function($scope, $controller, Assert, DataikuAPI, $stateParams, ComputableSchemaRecipeSave) {
    $controller('StandardFlowViewsMainController', {$scope: $scope});

    Assert.inScope($scope, 'tool');
    let recipeName = $scope.node.name;

    $scope.reviewSuggestion = function() {
        ComputableSchemaRecipeSave.handleSchemaUpdateWithPrecomputed($scope,
            $scope.nodeState.updateSolution).then(function() {
                DataikuAPI.flow.tools.propagateSchema.markRecipeAsOKAfterUpdate($stateParams.projectKey, recipeName).success(function(data) {
                    $scope.tool.user.state = data;
                    $scope.tool.drawHooks.updateFlowToolDisplay();
                })
            });
    }

    $scope.ignoreSuggestion = function() {
        let recipeName = $scope.node.name;

        DataikuAPI.flow.tools.propagateSchema.markRecipeAsOKForced($stateParams.projectKey, recipeName).success(function(data) {
            $scope.tool.user.state = data;
            $scope.tool.drawHooks.updateFlowToolDisplay();
        })
    }
});


app.controller("FlowToolPropagateDatasetNeedsRebuildPopupController", function($scope, $controller, Assert, DataikuAPI, $stateParams, ActivityIndicator, CreateModalFromTemplate, FlowBuildService, AnyLoc) {
    $controller('StandardFlowViewsMainController', {$scope: $scope});
    Assert.inScope($scope, 'tool');
    let datasetName = $scope.node.name;

    $scope.build = function() {
        const loc = AnyLoc.makeLoc($stateParams.projectKey, datasetName);
        FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "DATASET", loc);

        $scope.$on("datasetBuildStarted", function() {
            ActivityIndicator.success("Dataset build started ... Please wait for end before continuing propagation");
            DataikuAPI.flow.tools.propagateSchema.markDatasetAsBeingRebuilt($stateParams.projectKey, datasetName).success(function(data) {
                $scope.tool.user.state = data;
                $scope.tool.drawHooks.updateFlowToolDisplay();
            });
        });
    };

    $scope.ignoreSuggestion = function() {
        DataikuAPI.flow.tools.propagateSchema.markDatasetAsBeingRebuilt($stateParams.projectKey, datasetName).success(function(data) {
            $scope.tool.user.state = data;
            $scope.tool.drawHooks.updateFlowToolDisplay();
        })
    };
});


})();
;
(function() {
'use strict';

const app = angular.module('dataiku.flow.tools');


app.service('CheckConsistencyFlowTool', function($rootScope, $stateParams,
    DataikuAPI, ContextualMenu, LoggerProvider,
    FlowToolsUtils, FlowViewsUtils, FlowGraph, Ng1ToolBridgeService) {

    const NAME = 'CHECK_CONSISTENCY';
    const DISPLAY_NAME = 'Check consistency';

    this.getDefinition = function() {
        return {
            getName: () => NAME,
            getToolDisplayName: () => DISPLAY_NAME,

            initFlowTool: function(tool) {
                tool.user = {
                    updateOptions: {
                        recheckAll: false,
                        datasets: {
                            consistencyWithData: true
                        },
                        recipes: {
                            schemaConsistency: true,
                            otherExpensiveChecks: true
                        }
                    },
                    updateStatus: {
                        updating: false
                    }
                };
                tool.action = true;
                Ng1ToolBridgeService().emitActionTool(true);

                tool.user.markAsOK = function(nodes) {
                    const nodeIds = nodes.map(n => n.realId);
                    DataikuAPI.flow.tools.checkConsistency.markAsOK($stateParams.projectKey, nodeIds).success(function(data) {
                        tool.user.state = data;
                        tool.drawHooks.updateFlowToolDisplay();
                    })
                };

                tool.user.recheck = function(nodes) {
                    const nodeIds = nodes.map(n => n.realId);
                    DataikuAPI.flow.tools.checkConsistency.recheck($stateParams.projectKey, nodeIds).success(function(data) {
                        tool.user.state = data;
                        tool.drawHooks.updateFlowToolDisplay();
                    })
                };

                tool.user.canRecheck = function(nodes) {
                    if (tool.user.state == null) return false; // protect against slow state fetching
                    return !!nodes.filter(n => tool.user.state.stateByNode[n.realId] != 'UNCHECKED').length;
                };


                function colorFromMessageHolder(holder, node, sel) {
                    if (holder.maxSeverity == "ERROR") {
                        FlowToolsUtils.colorNode(node, sel, "red");
                    } else if (holder.maxSeverity == "WARNING") {
                        FlowToolsUtils.colorNode(node, sel, "orange");
                    } else if (holder.maxSeverity == "INFO") {
                        FlowToolsUtils.colorNode(node, sel, "lightblue");
                    } else {
                        FlowToolsUtils.colorNode(node, sel, "green");
                    }
                }

                function needsPopup(nodeState) {
                    if (nodeState.state == "FAILED_CHECK") return true;
                    if (nodeState.state == "CHECKED") {
                        if (nodeState.recipeCheckResult && nodeState.recipeCheckResult.maxSeverity) return true;
                        if (nodeState.datasetCheckResult && nodeState.datasetCheckResult.maxSeverity) return true;
                    }
                    return false;
                }

                tool.drawHooks.updateFlowToolDisplay = function() {
                    if (!tool.user.state) return; // protect against slow state fetching
                    if (!FlowGraph.ready()) return; // protect against slow graph fetching

                    $.each(FlowGraph.get().nodes, function(nodeId, node) {
                        const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                        const nodeState = tool.user.state.stateByNode[node.realId];

                        //TODO @flow factorize cleanNode
                        nodeElt.classed('focus', false).classed('out-of-focus', false);
                        $('.tool-simple-zone', FlowGraph.getSvg()).empty();
                        $('.node-totem span', nodeElt[0]).removeAttr('style').removeClass();
                        $('.never-built-computable *', nodeElt[0]).removeAttr('style');

                        if (!nodeState) {
                            // Node is not involved in this
                            FlowToolsUtils.colorNode(node, nodeElt, "#ccc");
                        } else if (nodeState.state == "UNCHECKED") {
                            FlowToolsUtils.colorNode(node, nodeElt, "#808080");
                        } else if (nodeState.state == "CHECKED") {
                            if (nodeState.recipeCheckResult) {
                                nodeState.errorHolder = nodeState.recipeCheckResult;

                                colorFromMessageHolder(nodeState.recipeCheckResult, node, nodeElt);
                            } else if (nodeState.labelingTaskCheckResult) {
                                nodeState.errorHolder = nodeState.labelingTaskCheckResult;
                                colorFromMessageHolder(nodeState.labelingTaskCheckResult, node, nodeElt);
                            } else if (nodeState.datasetCheckResult) {
                                nodeState.errorHolder = nodeState.datasetCheckResult;
                                colorFromMessageHolder(nodeState.datasetCheckResult, node, nodeElt);
                            }
                        } else if (nodeState.state == "FAILED_CHECK") {
                            FlowToolsUtils.colorNode(node, nodeElt, "purple");
                        }
                    });
                }

                tool.actionHooks.onItemClick = function(node, evt) {
                    if (!tool.user.state) return; // protect against slow state fetching
                    let nodeState = tool.user.state.stateByNode[node.realId];

                    ContextualMenu.prototype.closeAny();

                    if (nodeState && needsPopup(nodeState)) {
                        let menuScope = $rootScope.$new();

                        menuScope.nodeState = nodeState;
                        menuScope.node = node;
                        menuScope.tool = tool;

                        let menuParams = {
                            template: "/templates/flow-editor/tools/consistency-item-popup.html",
                            scope: menuScope,
                            contextual: false
                        };
                        let menu = new ContextualMenu(menuParams);
                        menu.openAtEventLoc(evt);
                    }
                }

                FlowViewsUtils.addAsynchronousStateComputationBehavior(tool);

                DataikuAPI.flow.tools.getState($stateParams.projectKey, NAME, {}).success(function(data) {
                    tool.user.state = data;
                    tool.drawHooks.updateFlowToolDisplay();
                }).error(FlowGraph.setError());
                return tool;
            },

            template: "/templates/flow-editor/tools/tool-check-consistency.html"
        };
    };
});


app.controller("ConsistencyFlowToolMainController", function($scope, Assert, DataikuAPI, $stateParams) {
    Assert.inScope($scope, 'tool');
    $scope.update = $scope.tool.user.update;
});


})();
;
(function() {
'use strict';

const app = angular.module('dataiku.recipes', ['dataiku.common.lists', 'dataiku.aiSqlGeneration']);


app.directive('checkRecipeNameUnique', function(DataikuAPI, $stateParams) {
    return {
        require: 'ngModel',
        link: function(scope, elem, attrs, ngModel) {
            DataikuAPI.flow.recipes.list($stateParams.projectKey).success(function(data) {
                scope.unique_recipes_names = $.map(data, function(recipe) {
                    return recipe.name;
                });
                /* Re-apply validation as soon as we get the list */
                apply_validation(ngModel.$modelValue);
            });
            var initialValue = null, initialValueInitialized = false;
            function apply_validation(value) {
                // Implicitely trust the first value (== our own name)
                if (initialValueInitialized == false && value != undefined && value != null && value.length > 0) {
                    initialValue = value;
                    initialValueInitialized = true;
                }
                // It is fake, but other check will get it.
                if (value == null || value.length === 0) return true;
                // We are back to our name, accept.
                if (initialValueInitialized && value == initialValue) return value;
                var valid = scope.unique_recipes_names ? scope.unique_recipes_names.indexOf(value) === -1 : true;
                ngModel.$setValidity('recipeNameUnique', valid);
                return valid ? value : undefined;
            }
             //For DOM -> model validation
            ngModel.$parsers.unshift(apply_validation);

            //For model -> DOM validation
            ngModel.$formatters.unshift(function(value) {
                apply_validation(value);
                return value;
            });
        }
    };
});

app.filter("buildModeDescription", function(){
    var dict = {
        "NON_RECURSIVE_FORCED_BUILD": "Build only this dataset",
        "RECURSIVE_BUILD": "Build required datasets",
        "RECURSIVE_FORCED_BUILD": "Force-rebuild dataset and dependencies",
        "RECURSIVE_MISSING_ONLY_BUILD": "Build missing dependencies then this one"
    };
    return function(input) {
        return dict[input] || input;
    }
});

app.directive('recipePipelineConfig', function() {
    return {
        restrict: 'E',
        templateUrl: '/templates/recipes/fragments/recipe-pipeline-config.html',
        scope: {
          config: "=",
          anyPipelineTypeEnabled: "&"
        }
    };
});

app.directive('otherActionListItem', function() {
    return {
        restrict: 'E',
        templateUrl: '/templates/recipes/fragments/other-action-list-item.html',
        scope: {
            icon: "@",
            label: "@",
            onClick: '&',
            showCondition: "<?",
            enableCondition: "<?",
            disabledTooltip: "@?"
        },
        link: function(scope) {
            if (scope.showCondition === undefined) {
                scope.showCondition = true;
            }
            if (scope.enableCondition === undefined) {
                scope.enableCondition = true;
            }
        }
    };
});

app.directive("sparkDatasetsReadParamsBehavior", function(Assert, $stateParams, RecipesUtils, Logger, DatasetUtils) {
    return {
        scope: true,
        link: function($scope, element, attrs) {
            Logger.info("Loading spark behavior");
            Assert.inScope($scope, 'recipe');
            let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$scope.recipe.projectKey;

            $scope.readParams = $scope.$eval(attrs.readParams);
            Assert.inScope($scope, 'readParams');

            function autocomplete() {
                RecipesUtils.getFlatInputsList($scope.recipe).forEach(function(input) {
                    Assert.inScope($scope, 'computablesMap');
                    const computable = $scope.computablesMap[input.ref];
                    if (!computable) {
                        throw Error('dataset is not in computablesMap, try reloading the page');
                    }
                    const dataset = computable.dataset;
                    if (dataset && !$scope.readParams.map[input.ref]) {
                        $scope.readParams.map[input.ref] = {
                            repartition: ['HDFS', 'hiveserver2'].includes(dataset.type) ? 1 : 10,
                            cache: false
                        };
                    }
                });
                Logger.info("Updated map", $scope.readParams.map);
            }

            $scope.$watch("recipe.inputs", function(nv, ov) {
                if (nv && $scope.computablesMap) {
                    DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                        .then(_ => autocomplete());
                }
            }, true);
            $scope.$watch("computablesMap", function(nv, ov) {
                if (nv) {
                    DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                        .then(_ => autocomplete());
                }
            }, true);
        }
    }
});


app.directive("sparkDatasetsReadParams", function(Assert, RecipesUtils) {
    return {
        scope: true,
        templateUrl: "/templates/recipes/fragments/spark-datasets-read-params.html",
        link: function($scope, element, attrs) {
            Assert.inScope($scope, 'recipe');
            $scope.readParams = $scope.$eval(attrs.readParams);
            Assert.inScope($scope, 'readParams');
        }
    };
});


app.service('RecipesCapabilities', function(RecipeDescService, CodeEnvsService, AppConfig, $rootScope) {

    function getRecipeType(recipe) {
        if (recipe) {
            // A bit dirty, we don't know what recipe is (taggableObjectRef, graphNode, listItem...)
            if (recipe.recipeType) {
                return recipe.recipeType;
            } else if (recipe.subType) {
                return recipe.subType;
            } else if (recipe.type) {
                return recipe.type;
            }
        }
        return undefined;
    }

    this.isMultiEngine = function(recipe) {
        const desc = RecipeDescService.getDescriptor(getRecipeType(recipe));
        return !!desc && desc.isMultiEngine;
    };

    this.canEngine = function(recipe, engine) {
        if (!recipe) {
            return false;
        }
        const recipeType = getRecipeType(recipe);
        if (!recipeType) {
            return false;
        }
        if (recipeType.toLowerCase().includes(engine)) {
            return true;
        }
        const desc = RecipeDescService.getDescriptor(recipeType);
        // we can't be sure though...
        return !!(desc && desc.isMultiEngine);
    };

    this.isSparkEnabled = function() {
        return !AppConfig.get() || AppConfig.get().sparkEnabled;
    };

    this.canSpark = function(recipe) {
        return this.isSparkEnabled() && this.canEngine(recipe, 'spark');
    };

    this.canSparkPipeline = function (recipe) {
        return $rootScope.projectSummary.sparkPipelinesEnabled &&
            this.canSpark(recipe) &&
            !(['pyspark', 'sparkr'].includes(getRecipeType(recipe)));
    };

    this.canSqlPipeline = function(recipe) {
        const canEngine = this.canEngine(recipe, 'sql');
        const b = !(['spark_sql_query', 'sql_script'].includes(getRecipeType(recipe)));
        return $rootScope.projectSummary.sqlPipelinesEnabled &&
            canEngine &&
            b;
    };

    this.canChangeSparkPipelineability= function(recipe) {
        if (recipe) {
            // Prediction scoring is supported but there is a bug that prevent the backend to compute the pipelineabilty (Clubhouse #36393)
            if (getRecipeType(recipe) === 'prediction_scoring') {
                return false;
            }
            return this.canSpark(recipe);
        }
        return false;
    };

    this.canChangeSqlPipelineability= function(recipe) {
        const recipeType = getRecipeType(recipe);
        if (recipeType) {
            // The following recipes are the only ones that can run on SQL and be part of a SQL pipeline.
            if (['sync', 'shaker', 'sampling', 'grouping', 'upsert', 'distinct', 'window', 'join', 'split', 'topn', 'sort',
                    'pivot', 'vstack', 'sql_query', 'prediction_scoring'].includes(recipeType)) {
                return true;
            }
        }
        return false;
    };

    this.canImpala = function(recipe) {
        if (recipe) {
            const recipeType = getRecipeType(recipe);
            if (recipeType === 'impala') {
                return true;
            }
            if (AppConfig.get() && !AppConfig.get().sparkEnabled) {
                return false;
            }
            const desc = RecipeDescService.getDescriptor(recipeType);
            if (desc && desc.isMultiEngine) {
                return true; // we can't be sure...
            }
        }
        return false;
    };

    this.canHive = function(recipe) {
        if (recipe) {
            const recipeType = getRecipeType(recipe);
            if (recipeType === 'hive') {
                return true;
            }
            if (AppConfig.get() && !AppConfig.get().sparkEnabled) {
                return false;
            }
            const desc = RecipeDescService.getDescriptor(recipeType);
            if (desc && desc.isMultiEngine) {
                return true; // we can't be sure...
            }
        }
        return false;
    };

    this.canPythonCodeEnv = function(recipe) {
        return CodeEnvsService.canPythonCodeEnv(recipe);
    };

    this.canRCodeEnv = function(recipe) {
        return CodeEnvsService.canRCodeEnv(recipe);
    };
});

app.controller("RecipePageRightColumnActions", async function($controller, $scope, $stateParams, ActiveProjectKey, DataikuAPI) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});
    $controller('_RecipeWithEngineBehavior', {$scope: $scope});

    $scope.recipeData = (await DataikuAPI.flow.recipes.getFullInfo(ActiveProjectKey.get(), $stateParams.recipeName)).data;

    $scope.recipe = $scope.recipeData.recipe;
    $scope.recipe.recipeType = $scope.recipe.type;
    $scope.recipe.nodeType = 'RECIPE';
    $scope.recipe.id = $stateParams.recipeName;
    $scope.recipe.interest = $scope.recipeData.interest;

    $scope.selection = {
        selectedObject : $scope.recipe,
        confirmedItem : $scope.recipe
    };
});


app.directive('recipeRightColumnSummary', function($controller, $stateParams, $state, $rootScope, RecipeRenameService,
    DataikuAPI, TopNav, CreateModalFromTemplate, ActiveProjectKey, ActivityIndicator, WT1, FlowBuildService, LambdaServicesService,
    Logger, translate) {
    return {
        templateUrl: '/templates/recipes/right-column-summary.html',

        link: function(scope, element, attrs) {

            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            var enrichSelectedObject = function (selObj, recipe) {
                selObj.tags = recipe.tags; // for apply-tagging modal
            }

            scope.isDeleteAndReconnectAllowed = false;

            scope.$watch("selection", () => {
                if (scope.selection && scope.selection.selectedObject) {
                    scope.canDeleteAndReconnectObject(scope.selection.selectedObject).then(function(result) {
                        scope.isDeleteAndReconnectAllowed = result;
                    }).catch(function(error) {
                        Logger.warn(`Problem determining whether to show delete & reconnect option for ${scope.selection.selectedObject.id}:`, error);
                    });
                }
            });

            scope.deleteAndReconnectRecipe = function(wt1_event_name) {
                scope.doDeleteAndReconnectRecipe(scope.selection.selectedObject, wt1_event_name);
            }

            scope.getDeleteAndReconnectTooltip = function() {
                if (!scope.canWriteProject()) return translate("FLOW.DELETE_AND_RECONNECT.NO_PERMISSIONS", "You don't have write permissions for this project");
                if (!scope.isDeleteAndReconnectAllowed) return translate("FLOW.DELETE_AND_RECONNECT.ONLY_AVAILABLE", "This feature is only available for single-input recipes before the flow's end where reconnection is possible");
                return translate("FLOW.DELETE_AND_RECONNECT.DELETE_THIS_RECIPE", "Delete this recipe and its output datasets, relinking the upstream dataset as an input to the downstream recipes");
            }

            scope.getRunTooltip = function() {
                if (scope.isOutputEmpty)
                    return scope.translate('PROJECT.RECIPE.RIGHT_PANEL.RUN.NO_OUTPUT_ERROR', 'Running a recipe that has no output is not possible');
                if (scope.payload?.backendType !== undefined && scope.payload.backendType === 'VERTICA')
                    return  scope.translate('PROJECT.PERMISSIONS.VERTICA_NOT_SUPPORTED', 'Vertica ML backend is no longer supported');
                if (!scope.canWriteProject())
                    return scope.translate('PROJECT.PERMISSIONS.WRITE_ERROR', 'You don\'t have write permissions for this project');
                return null;
            }

            scope.refreshData = function() {
                DataikuAPI.flow.recipes.getFullInfo(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name).success(function(data){
                    scope.recipeData = data;
                    scope.recipe = data.recipe;
                    scope.recipeBeingDeselected = null; // We no longer need to keep a reference to recipe that was selected for "event that will arrive late".
                    if (/^\s*\{/.test(data.script || '')) {
                        try { // payload may not be JSON; if it is we only need backendType
                            scope.payload = { backendType: JSON.parse(data.script).backendType };
                        } catch (ignored) { /* Nothing for now */ }
                    }

                    enrichSelectedObject(scope.selection.selectedObject, scope.recipe);

                    if (scope.selection.selectedObject.continuous) {
                        // update the build indicator on the flow
                        let selObj = scope.selection.selectedObject;
                        let ps = data.continuousState;
                        // the only change that could not be on the flow is when the activity fails
                        if (!selObj.continuousActivityDone) {
                            if (ps && ps.mainLoopState != null && ps.mainLoopState.futureInfo != null && ps.desiredState == "STARTED" && ps.mainLoopState.futureInfo.hasResult) {
                                selObj.continuousActivityDone = true;
                                $rootScope.$broadcast("graphRendered");
                            }
                        }
                    }
                    scope.recipe.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;

                    scope.outputs = data.outputs;
                    scope.isOutputEmpty = scope.outputs === undefined || _.isEmpty(scope.outputs);

                }).error(setErrorInScope.bind(scope));
            };

            scope.$on('taggableObjectTagsChanged', () => scope.refreshData());

            /* Auto save when summary is modified */
            scope.$on("objectSummaryEdited", () => {
                const editedRecipe = scope.recipe !== null ? scope.recipe : scope.recipeBeingDeselected;
                if (editedRecipe) {
                    DataikuAPI.flow.recipes.save(ActiveProjectKey.get(), editedRecipe, { summaryOnly: true })
                        .success(() => ActivityIndicator.success("Saved"))
                        .error(setErrorInScope.bind(scope));
                }
            });

            scope.$watch("selection.selectedObject", function(nv, ov) {
                if (!nv) return;
                scope.recipeData = {recipe: nv, timeline: {}}; // display temporary (incomplete) data
                if(scope.selection.confirmedItem != scope.selection.selectedObject) {
                    // We need to keep a reference on the recipe that was selected in case an event "objectSummaryEdited" arrives just after this event
                    // See https://app.shortcut.com/dataiku/story/152840 for more details.
                    scope.recipeBeingDeselected = scope.recipe;
                    scope.recipe = null;
                }
                scope.recipeType = nv.recipeType || nv.type;
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                scope.refreshData();
            });

            scope.saveCustomFields = function(newCustomFields) {
                WT1.event('custom-fields-save', {objectType: 'RECIPE'});
                const oldCustomFields = angular.copy(scope.recipe.customFields);
                scope.recipe.customFields = newCustomFields;
                return DataikuAPI.flow.recipes.save(ActiveProjectKey.get(), scope.recipe, { summaryOnly: true })
                    .success(() => $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), scope.recipe.customFields))
                    .error((data, status, headers, config, statusText, xhrStatus) => {
                        scope.recipe.customFields = oldCustomFields;
                        setErrorInScope.bind(scope)(data, status, headers, config, statusText, xhrStatus);
                    });
            };

            scope.editCustomFields = function() {
                if (!scope.recipe) {
                    return;
                }
                let modalScope = angular.extend(scope, {objectType: 'RECIPE', objectName: scope.recipe.name, objectCustomFields: scope.recipe.customFields});
                CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                    scope.saveCustomFields(customFields);
                });
            };

            scope.renameRecipe = function() {
                RecipeRenameService.startRenamingRecipe(scope, $stateParams.projectKey, scope.recipe.name);
            }

            scope.runRecipe = function(){
                FlowBuildService.openRecipeRunModal(scope, scope.recipe.projectKey, scope.recipe);
            }

            scope.startContinuous = function() {
                WT1.event("start-continuous", {from:'recipe'})
                CreateModalFromTemplate("/templates/continuous-activities/start-continuous-activity-modal.html", scope, "StartContinuousActivityController", function(newScope) {
                    newScope.recipeId = scope.recipe.name;
                }).then(function(loopParams) {
                    DataikuAPI.continuousActivities.start($stateParams.projectKey, scope.recipe.name, loopParams).success(function(data){
                        scope.refreshData();
                    }).error(setErrorInScope.bind(scope));
                });
            }
            scope.stopContinuous = function(){
                WT1.event("stop-continuous", {from:'recipe'})
                DataikuAPI.continuousActivities.stop($stateParams.projectKey, scope.recipe.name).success(function(data){
                    scope.refreshData();
                }).error(setErrorInScope.bind(scope));
            }

            scope.goToCurrentRun = function() {
                let recipeState = scope.recipeData.continuousState || {};
                let mainLoopState = recipeState.mainLoopState || {};
                $state.go("projects.project.continuous-activities.continuous-activity.runs", {continuousActivityId: recipeState.recipeId, runId: mainLoopState.runId, attemptId: mainLoopState.attemptId});
            };

            scope.updateUserInterests = function() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "RECIPE", ActiveProjectKey.get(), scope.selection.selectedObject.name)
                    .success(function(data){
                        scope.selection.selectedObject.interest = data;
                        scope.recipeData.interest = data;
                    })
                    .error(setErrorInScope.bind(scope));
            }
            const interestsListener = $rootScope.$on('userInterestsUpdated', scope.updateUserInterests);
            scope.$on("$destroy", interestsListener);
        }
    }
});


app.controller("RecipeDetailsController", function ($scope, $state, $filter, CachedAPICalls, ShakerProcessorsInfo, ShakerProcessorsUtils, Notification, RecipesUtils) {
    if (!$scope.processors) {
        // Get the processor, to display their description when they are failing
        CachedAPICalls.processorsLibrary.success(function(processors){
            $scope.processors = processors;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getObjectType = function(object) {
        switch(object.type) {
            case 'SAVED_MODEL':     return 'SAVED_MODEL';
            case 'MANAGED_FOLDER':  return 'MANAGED_FOLDER';
            default:                return 'DATASET_CONTENT';
        }
    };

    $scope.isOnRecipeObjectPage = function() {
        return $state.includes('projects.project.recipes.recipe');
    }

    $scope.showCodeRecipeSummary = function() {
        if (!$scope.data) {
            return false;
        }

        const recipeHasLanguageAndScript = $filter('recipeTypeToLanguage')($scope.data.recipe.type) && $scope.data.script;
        if (!recipeHasLanguageAndScript) {
            return false
        }

        const recipeHasAssociatedNotebook =  $scope.data.notebook && $scope.data.notebook.projectKey && $scope.data.notebook.name;
        if ($scope.isOnRecipeObjectPage() && !recipeHasAssociatedNotebook) {
            return false;
        }

        return true;
    }

    $scope.getFlatAggregates = function(values) {
        if (!values) {
            return [];
        }
        var aggregates = [];
        values.forEach(function(value) {
            if (value.customExpr) {
                aggregates.push(value);
            } else {
                angular.forEach(value, function(x, agg) {
                    if (agg.startsWith("__")) return; // temp field
                    if (x === true) {
                        aggregates.push({agg:agg, column:value.column, type:value.type});
                    }
                });
            }
        });
        return aggregates;
    }

    $scope.validateStep = function(step) {
        return ShakerProcessorsUtils.validateStep(step, $scope.processors);
    };

    /*
     * Displaying info to user
     */
    var tmpFindGroupIndex = [];
    $scope.findGroupIndex = function(step) {
        const groupIndex = tmpFindGroupIndex(step);
        return groupIndex < 0 ? '' : groupIndex + 1;
    }

    $scope.$watch("data.script.steps", function(nv) {
        if (!nv) return;

        tmpFindGroupIndex =Array.prototype.indexOf.bind($scope.data.script.steps.filter(function(s) { return s.metaType === 'GROUP'; }));

        if ($scope.data.recipe.type === "shaker") {
            const { flattenedEnabledSteps } = ShakerProcessorsUtils.getStepsWithSubSteps($scope.data.script.steps, $scope.processors);
            $scope.data.stepsWithSubSteps = flattenedEnabledSteps;
        }
    });

    const recipeSaveListener = Notification.registerEvent("recipe-save", (type, data) => {
        if(!$scope.data) $scope.data = {};
        $scope.data.script = data.payloadData;
        $scope.data.recipe = data.recipe;
        RecipesUtils.parseScriptIfNeeded($scope.data);
    });

    $scope.$on('$destroy', recipeSaveListener);
});

app.controller("RecipeEditorController",
    function ($scope, $rootScope, $timeout, $stateParams, $filter, $location, $state, $q,
    Assert, BigDataService, DataikuAPI, Dialogs, WT1, FutureProgressModal,
    TopNav, PartitionDeps, DKUtils, Logger, HistoryService,
    CreateModalFromTemplate, AnyLoc, JobDefinitionComputer, RecipeComputablesService, RecipesUtils,
    RecipeRunJobService, PartitionSelection, RecipeDescService, InfoMessagesUtils, StateUtils, GraphZoomTrackerService,
    DatasetUtils, CodeStudiosService, Notification, RecipeContextService,
    ActivityIndicator, ManualLineageModalService, RatingFeedbackParams) {

    $scope.ratingFeedbackParams = RatingFeedbackParams;
    $scope.hasJobBanner = false;

    $scope.$on('$stateChangeSuccess', (event, toState, toParams) => {
        if (toState.name === "projects.project.recipes.recipe" && toParams.isAIGenerated === true && !$rootScope.appConfig.isUsingLocalAiAssitant?.prepareAICompletion) {
            // We are entering a freshly IA generated recipe, so we can show the rating feedback banner
            $scope.ratingFeedbackParams.showRatingFeedback = true;
        }
    });

    //This is necessary to make sure  the banner disappears when we route to another page (otherwise, it appears again after we open a recipe). In case we're routing to explore the dataset we instead want to display back the banner
    $scope.$on('$stateChangeStart', (event, toState, toParams, fromState, fromParams) => {
        const outputDatasets = $scope.recipe?.outputs?.['main']?.items.map(item => item.ref) ?? [];
        if (!(toState.name === 'projects.project.datasets.dataset.explore' && outputDatasets.includes(toParams.datasetName) && fromParams.isAIGenerated === true)) {
            // We are NOT leaving a freshly IA generated recipe to go to one of its output dataset, so we can drop the rating feedback banner
            $scope.ratingFeedbackParams.showRatingFeedback = false;
        }
    });

    $scope.onCloseRatingFeedbackBanner = function() {
        $scope.ratingFeedbackParams.showRatingFeedback = false;
    }

    $scope.InfoMessagesUtils = InfoMessagesUtils;

    let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

    $scope.recipeUpdateData = function(initialCall = false) {
        return DataikuAPI.flow.recipes.getWithInlineScript($stateParams.projectKey, $scope.recipeName.name).success(function(data) {
            RecipeContextService.setCurrentRecipe(data);
            $scope.recipe = data.recipe;
            $scope.script.data = data.script;
            $scope.canEditInCodeStudio = data.canEditInCodeStudio;
            $scope.origRecipe = angular.copy($scope.recipe);
            $scope.origScript = angular.copy($scope.script);

            $scope.recipeDesc = RecipeDescService.getDescriptor($scope.recipe.type);

            TopNav.setItem(TopNav.ITEM_RECIPE, data.recipe.name, {
                recipeType :data.recipe.type,
                name : data.recipe.name,
                inputs: data.recipe.inputs,
                outputs: data.recipe.outputs
            });

            RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
                $scope.setComputablesMap(map);
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => {
                        if(initialCall){
                            $scope.onload();
                        }
                        $scope.$broadcast('computablesMapChanged'); // because the schema are there now (they weren't when setComputablesMap() was called)
                    });
            });
            DataikuAPI.flow.zones.getZoneId($stateParams.projectKey, {id: data.recipe.name, type: "RECIPE", projectKey: data.recipe.projectKey}).success(zone => {
                if (zone) {
                    // Put it in zone so the io:getDatasetCreationSettings can find it
                    // and we can target more recipes
                    $scope.zone = zone.id;
                }
            });

        }).error(function(){
            HistoryService.notifyRemoved({
                type: "RECIPE",
                id: $scope.recipeName.name,
                projectKey: $stateParams.projectKey
            });
            setErrorInScope.apply($scope, arguments);
        });
    }

	function main(){
        /* Init scope */
        $scope.uiState = {editSummary:false};
        $scope.startedJob = {};
        $scope.recipe = null;
        $scope.recipeStatus = null;
        $scope.payloadRequired = false; // override for recipe specific recipe types. Avoids to get-status before the payload is ready
        $scope.script = {};
        $scope.creation = false;
        $scope.recipeName = { "name" : $scope.$state.params.recipeName };
        $scope.projectKey = $stateParams.projectKey;
        $scope.hooks = $scope.hooks || {};
        GraphZoomTrackerService.setFocusItemByName("recipe", $scope.recipeName.name);
        $scope.RecipesUtils = RecipesUtils

        // Validation context
        $scope.valCtx = {};

        const tabToSelect = StateUtils.defaultTab("settings");
        TopNav.setLocation(TopNav.TOP_FLOW, "recipes", TopNav.TABS_RECIPE, tabToSelect);
        TopNav.setItem(TopNav.ITEM_RECIPE, $stateParams.recipeName);

        $scope.validations = [
            function(){
                return $scope.renaming.recipe_name.$valid && $scope.recipeName.name.length;
            }
        ];

        $scope.PartitionDeps = PartitionDeps;
        addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

        // DO NOT INITIALIZE IT, IT HELPS CATCH ERRORS
        $scope.computablesMap = null;
        $scope.$broadcast('computablesMapChanged');

        Assert.trueish($scope.recipeName.name, 'no recipe name');

        $scope.recipeUpdateData(true);

        TopNav.setTab(tabToSelect);

        $scope.$watchGroup(['topNav.tab', 'valCtx.preRunValidationError', 'startedJob.jobId'], function([tab, preRunValidationError, jobId]) {
            // The job banner only shows up in the "settings" or "code" tab for recipes, except for the Sync recipe where it shows up in the "io" tab
            $scope.hasJobBanner = (['code', 'settings'].includes(tab) || ($scope.recipe?.type === 'sync' && tab === 'io')) && (preRunValidationError || jobId)
        })

    }
    main();

    function extractWT1EventParams(recipe, payload) {
        try {
            if (recipe.type === "prediction_scoring") {
                const recipeParams = JSON.parse(payload.data);
                let eventParams = {
                    filterInputColumns: recipeParams.filterInputColumns,
                    forceOriginalEngine: recipeParams.forceOriginalEngine,
                    outputExplanations: recipeParams.outputExplanations,
                    outputProbaPercentiles: recipeParams.outputProbaPercentiles,
                    outputProbabilities: recipeParams.outputProbabilities,
                    taskType: "PREDICTION",
                    predictionType: recipeParams.predictionType,
                    savedModelType: recipeParams.savedModelType,
                    proxyModelProtocol: recipeParams.proxyModelProtocol,
                };
                if (eventParams.outputExplanations) {
                    eventParams = {
                        individualExplMethod: recipeParams.individualExplanationParams.method,
                        individualExplCount: recipeParams.individualExplanationParams.nbExplanations,
                        ... eventParams
                    };
                }
                return eventParams;
            }
            if (["clustering_scoring", "clustering_training"].includes(recipe.type)) {
                return {
                    taskType: "CLUSTERING",
                    savedModelType: "DSS_MANAGED"
                    // No Prediction type, saved relates to clustering
                }
            }
            if (recipe.type === "eda_stats") {
                const recipeParams = JSON.parse(payload.data);
                return { recipeSubType: recipeParams.type };
            }
            if (recipe.type === "nlp_llm_finetuning") {
                const recipeParams = JSON.parse(payload.data);
                if (recipeParams.llmId !== undefined) {
                    const llmIdParts = recipeParams.llmId.split(":");
                    return { inputModelType: llmIdParts[0] === "savedmodel" ? llmIdParts[1] : llmIdParts[0] }
                }
            }
            if (recipe.type === "nlp_llm_evaluation") {
                const recipeParams = JSON.parse(payload.data);
                return { inputFormat: recipeParams.inputFormat, llmTaskType: recipeParams.llmTaskType}
            }
            if (recipe.type === "nlp_agent_evaluation") {
                const recipeParams = JSON.parse(payload.data);
                return { inputFormat: recipeParams.inputFormat, llmTaskType: recipeParams.llmTaskType}
            }
            if (recipe.type === "nlp_llm_rag_embedding") {
                const recipeParams = JSON.parse(payload.data);
                return { vectorStoreUpdateMethod: recipeParams.vectorStoreUpdateMethod };
            }
            if (recipe.type === "embed_documents") {
                const payloadData = JSON.parse(payload.data);
                const allOtherRuleAction = recipe.params.allOtherRule.actionToPerform;

                // parsing the vlm full ids to extract only the connectionType (the rest is user-private, shouldn't be sent to wt1)
                let vlms = recipe.params.rules.filter(r => r.actionToPerform === 'VLM').map(r => r.vlmSettings.llmId);
                if (allOtherRuleAction === 'VLM'){
                    vlms.push(recipe.params.allOtherRule.vlmSettings.llmId);
                }
                let vlmConnectionsType = vlms.filter(fullId => fullId !== undefined).map(fullId => fullId.split(':')[0]);

                return {
                    vectorStoreUpdateMethod: payloadData.vectorStoreUpdateMethod,
                    rulesCount: recipe.params.rules.length + 1, // includes allOtherRule
                    vlmRulesCount: recipe.params.rules.filter(r => r.actionToPerform === 'VLM').length + (allOtherRuleAction === 'VLM'? 1: 0),
                    structuredRulesCount: recipe.params.rules.filter(r => r.actionToPerform === 'STRUCTURED').length + (allOtherRuleAction === 'STRUCTURED'? 1: 0),
                    donotextractRulesCount: recipe.params.rules.filter(r => r.actionToPerform === 'DONOTEXTRACT').length + (allOtherRuleAction === 'DONOTEXTRACT'? 1: 0),
                    vlmConnectionTypes:  Array.from(new Set(vlmConnectionsType)).join(",") // only keep unique values
                };
            }
            if (recipe.type === "evaluation") {
                const recipeParams = JSON.parse(payload.data);
                return {
                    taskType: recipeParams.taskType,
                    predictionType: recipeParams.predictionType,
                    savedModelType: recipeParams.savedModelType,
                    proxyModelProtocol: recipeParams.proxyModelProtocol
                }
            }
            if (recipe.type === "prediction_training") {
                const recipeParams = JSON.parse(payload.data);
                return {
                    taskType: recipeParams.core.taskType,
                    predictionType: recipeParams.core.prediction_type,
                    savedModelType: "DSS_MANAGED"
                }
            }
            return {};
        } catch (e) {
            Logger.error("Failed to get wt1 loggable recipe event payload params", e);
            return {};
        }
    }

    const DEFAULT_WT1_EVENT_OPTS = {
        withRecipeEventParamsEnrichment: false,
        withRecipeOutputAppendModeTracking: false,
    };

    function enrichRecipeWT1Event(params, opts = {}) {
        if (params == null) params = {};
        params.recipeId = ($scope.recipeName && $scope.recipeName.name) ? $scope.recipeName.name.dkuHashCode() : "unknown";
        params.recipeType = ($scope.recipe ? $scope.recipe.type : "unknown");
        params.creation = $scope.creation;
        if ($scope.recipe && $scope.recipe.type) {
            const extractParams = extractWT1EventParams($scope.recipe, $scope.script);
            params = { ...params, ...extractParams };
        }
        const _opts = { ...DEFAULT_WT1_EVENT_OPTS, ...opts }
        if (_opts.withRecipeEventParamsEnrichment) {
            params = {
                ...params,
                ...RecipesUtils.getWT1LoggableRecipeEventParams($scope.recipe, $scope.recipeDesc),
            };
        }
        if (_opts.withRecipeOutputAppendModeTracking) {
            params = {
                ...params,
                ...RecipesUtils.getWT1OutputAppendModeTracking($scope.recipe),
            };
        }
        return params;
    }

    // prefer using tryRecipeWT1Event for new events
    $scope.recipeWT1Event = function(type, params, opts) {
        WT1.event(type, enrichRecipeWT1Event(params, opts));
    };

    $scope.tryRecipeWT1Event = function(type, paramsGetter, message) {
        WT1.tryEvent(type, () => enrichRecipeWT1Event(paramsGetter()), message);
    };

    $scope.canEditRecipeInNotebook = function() {
        return ['python', 'pyspark', 'r', 'julia', 'sparkr', 'spark_scala', 'sql_query'].includes($scope.recipe.type);
    };
    $scope.editThisRecipeInNotebook = function() {
        $scope.recipeWT1Event('recipe-edit-in-notebook');
        if($scope.recipe.type === 'sql_query') {
            $scope.saveRecipeIfPossible()
                .then(() => DataikuAPI.flow.recipes.editInSQLNotebook($stateParams.projectKey, $stateParams.recipeName))
                .then(({ data }) => StateUtils.go.sqlNotebook(data.notebookId, $stateParams.projectKey, {cellId: data.cellId}))
                .catch(setErrorInScope.bind($scope));
        } else {
            var editInNotebook = function() {
                DataikuAPI.flow.recipes.editInNotebook($stateParams.projectKey, $stateParams.recipeName, $scope.recipe.params.envSelection, $scope.recipe.params.containerSelection).success(function(data) {
                    StateUtils.go.jupyterNotebook(data.id, $stateParams.projectKey);
                }).error(setErrorInScope.bind($scope));
            };
            $scope.saveRecipeIfPossible().then(function() {
                DataikuAPI.flow.recipes.checkNotebookEdition($stateParams.projectKey, $stateParams.recipeName).success(function(data) {
                    if (!data || data.conflict !== true || !data.notebook) {
                        editInNotebook();
                    } else {
                        Dialogs.openEditInNotebookConflictDialog($scope).then(
                            function(resolutionMethod) {
                                if(resolutionMethod == 'erase') {
                                    editInNotebook();
                                } else if(resolutionMethod == 'ignore') {
                                    StateUtils.go.jupyterNotebook(data.notebook, $stateParams.projectKey);
                                }
                            }
                        );
                    }
                }).error(setErrorInScope.bind($scope));
            }).catch(setErrorInScope.bind($scope));
        }
    };

    $scope.canEditRecipeInCodeStudio = function() {
        // not all the python recipe or r recipes are listed here : it's because you won't realistically
        // be able to debug or test python code that relies on spark or kafka in a CodeStudio
        if (!$scope.canEditInCodeStudio) return false;
        return ['python', 'r', 'sql_query', 'sql_script', 'spark_sql_query'].includes($scope.recipe.type);
    };

    const getExtension = function() {
        switch($scope.recipe.type) {
            case 'sql_query':
            case 'sql_script':
            case 'spark_sql_query':
                return '.sql';
            case 'python':
                return '.py';
            case 'r':
                return '.r';
            default:
                throw Error('Edition of recipes of type ' + $scope.recipe.type + 'is not supported by Code Studio!');
        }
    }

    $scope.editThisRecipeInCodeStudio = function() {
        $scope.saveRecipeIfPossible().then(function() {
            CodeStudiosService.editFileInCodeStudio($scope, "recipes", $stateParams.recipeName + getExtension());
        }).catch(setErrorInScope.bind($scope));
    }

    $scope.saveCustomFields = function(newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'RECIPE'});
        let oldCustomFields = angular.copy($scope.recipe.customFields);
        $scope.recipe.customFields = newCustomFields;
        return $scope.hooks.save().then(function() {
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.recipe.customFields);
            }, function() {
                $scope.recipe.customFields = oldCustomFields;
            });
    };

    $scope.editCustomFields = function() {
        if (!$scope.recipe) {
            return;
        }
        let modalScope = angular.extend($scope, {objectType: 'RECIPE', objectName: $scope.recipe.name, objectCustomFields: $scope.recipe.customFields});
        CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
            $scope.saveCustomFields(customFields);
        });
    };

    $scope.gotoLine = function(cm, line) {
        if(cm && line>0) {
            var pos = {ch:0,line:line-1};
            cm.scrollIntoView(pos);
            cm.setCursor(pos);
            cm.focus();
        }
    };

    let lastVeLoopConfig = null;
    $scope.loadRecipeVariables = function() {
        const veLoopConfig = $scope.recipe.params && $scope.recipe.params.variablesExpansionLoopConfig;
        if (!angular.equals(lastVeLoopConfig, veLoopConfig) || lastVeLoopConfig === null) {
            lastVeLoopConfig = typeof veLoopConfig === "object"
                ? JSON.parse(JSON.stringify(veLoopConfig))
                : veLoopConfig;
            DataikuAPI.flow.recipes.generic.getVariables($scope.recipe).success(function(data) {
                $scope.recipeVariables = data;
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.specificControllerLoadedDeferred = $q.defer();

    /* Method called once recipe is loaded */
    var onloadcalled = false;
    $scope.onload = function() {
        Assert.inScope($scope, 'recipe');
        Assert.trueish(!onloadcalled, 'already loaded');
        onloadcalled = true;

        $scope.loadRecipeVariables();

        $scope.fixupPartitionDeps();

        $scope.recipeWT1Event("recipe-open");

        // TODO: Check if still needed
        $scope.ioFilter = {};

        $scope.testRun = {
            build_partitions: {},
            runMode: "NON_RECURSIVE_FORCED_BUILD"
        };

        /* Synchronize the definition of build_partitions for the test run
         * with the partitioning schema of the first partitioned output */
        $scope.$watch("recipe.outputs", function(nv, ov) {
            if (nv != null) {
                clear($scope.testRun.build_partitions);
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => {
                        const definingOutputPartitioning = RecipeRunJobService.getOutputAndPartitioning($scope.recipe, $scope.computablesMap).partitioning;
                        if (definingOutputPartitioning && definingOutputPartitioning.dimensions.length) {
                            $scope.outputPartitioning = definingOutputPartitioning;
                            $scope.testRun.build_partitions = PartitionSelection.getBuildPartitions($scope.outputPartitioning);
                        } else {
                            $scope.outputPartitioning = { dimensions: [] };
                        }
                    });
            } else {
                $scope.testRun.build_partitions = null;
            }
        }, true);

        $scope.fixupPartitionDeps();

        /* When the specific recipe controller has finished loading AND we have
         * the computables map, then we call its own onload hook */
        $scope.specificControllerLoadedDeferred.promise.then(function() {
            if ($scope.hooks && $scope.hooks.onRecipeLoaded){
                $scope.hooks.onRecipeLoaded();
            }
        });
    };

    $scope.hooks.getRecipeSerialized = function(){
        var recipeSerialized = angular.copy($scope.recipe);
        PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
        return recipeSerialized;
    };

    $scope.hooks.resetScope = function() {
        clear($scope.startedJob);
        clear($scope.valCtx);
    };

    //Override it to return a string representing the payload
    $scope.hooks.getPayloadData = function() {};

    /* ***************************** Inputs/Outputs *************************** */

    $scope.hasAllRequiredOutputs = function() {
        if (!$scope.recipe || !$scope.recipe.outputs) {
            return false;
        }
        var out = $scope.recipe.outputs;
        //TODO implement for any role
        if(out.main) {
            return !!(out.main.items && out.main.items.length);
        }
        return true;//Other roles: don't know
    };

    $scope.hasPartitionedInput = function() {
        return $scope.getInputDimensions().length > 0;
    };

    $scope.hasPartitionedOutput = function() {
        return $scope.getOutputDimensions().length > 0;
    };

    $scope.hasDangerousAllAvailablePartitioning = function() {
        const hasAllAvailableDependency = () => {
            const recipeInputs = $scope.recipe.inputs || {};
            const inputValues = Object.values(recipeInputs);

            return inputValues.some(input =>
                input.items.some(item =>
                    item.deps.some(dep => dep.func === 'all_available')
                )
            );
        };
        return $scope.hasPartitionedInput() && $scope.hasPartitionedOutput() && hasAllAvailableDependency();
    }

    $scope.hasInvalidPartitionSelection = function() {
        if ($scope.recipe && $scope.recipe.redispatchPartitioning) {
            return false; // Redispatch partitioning does not require target partitions
        }
        return $scope.getOutputDimensions().some((dimension) => {
            return !$scope.testRun || $scope.testRun.build_partitions[dimension.name] === void 0 || $scope.testRun.build_partitions[dimension.name] === "";
        });
    };

    // This method should be called each time inputs or outputs are modified.
    $scope.fixupPartitionDeps = function(){
        if (!$scope.recipe || !$scope.computablesMap) return;
        var ret = PartitionDeps.fixup($scope.recipe, $scope.computablesMap);
        $scope.outputDimensions = ret[0];
        $scope.outputDimensionsWithNow = ret[1];
    };

    $scope.testPDep = function(inputRef, pdep) {
        PartitionDeps.test($scope.recipe, inputRef, pdep, $scope);
    };

    $scope.refreshDatasetInComputablesMap = function(dataset) {
        var found = null;
        $.each($scope.computablesMap, function(smartName, computable) {
            if (computable.projectKey == dataset.projectKey && computable.name == dataset.name)
                found = computable;
        });
        // the dataset has to be in the computablesMap, otherwise that means it's not even shown in the dataset left pane
        Assert.trueish(found);
        found.dataset = dataset;
    };

    // Keep veloop input in sync with VELoop config
    $scope.$watch("recipe.params.variablesExpansionLoopConfig", function() {
        if (!$scope.recipe) {
            return;
        }
        // Replicate what happens in the backend when a recipe is saved
        const role = "veloop";
        RecipesUtils.removeInputsForRole($scope.recipe, role);
        if (!$scope.recipe.params) {
            return;
        }
        const veLoopConfig = $scope.recipe.params.variablesExpansionLoopConfig;
        if (veLoopConfig && veLoopConfig.enabled && veLoopConfig.datasetRef) {
            RecipesUtils.addInput($scope.recipe, role, veLoopConfig.datasetRef);
        }
    }, true);

    /* Simple recipes that don't want to manage themselves inputs and outputs
     * should enable auto fixup */
    $scope.enableAutoFixup = function() {
        $scope.$watch("recipe.inputs", function() {
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => $scope.fixupPartitionDeps());
        }, true);
        $scope.$watch("recipe.outputs", function() {
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => $scope.fixupPartitionDeps());
        }, true);
    };

    $scope.$watch("recipe.inputs", function(nv, ov) {
        if (!nv) return;
        if (!$scope.outputDimensions) return;
        DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => {
            RecipesUtils.getFlatInputsList($scope.recipe).forEach(function(input) {
                if (!input.deps) return;
                input.deps.forEach(function(pdep){
                    PartitionDeps.autocomplete(pdep, $scope.outputDimensions, $scope.outputDimensionsWithNow);
                });
            });
        });
    }, true);

    $scope.$watch("recipe.outputs", function() {
        DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey);
    }, true);

    $scope.setComputablesMap = function(map) {
        $scope.computablesMap = map;
        $scope.$broadcast('computablesMapChanged');
    };

    $scope.getInputDimensions = function(){
        if (!$scope.recipe || !$scope.computablesMap) return [];
        return RecipeRunJobService.getInputDimensions($scope.recipe, $scope.computablesMap);
    };

    $scope.getOutputDimensions = function(){
        if (!$scope.recipe || !$scope.computablesMap) return [];
        return RecipeRunJobService.getOutputDimensions($scope.recipe, $scope.computablesMap);
    };

    $scope.hasAnyPartitioning = function(){
        return RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
    };

    /* ***************************** MANUAL LINEAGE  *************************** */

    $scope.showManualLineageModal = function() {
        const recipeInfo = {
            projectKey: $scope.recipe.projectKey,
            name: $scope.recipe.name,
            type: $scope.recipe.type,
        }
        const datasetInfos = (recipeIO) => {
             return Object.values(recipeIO).flatMap(inputs => {
                return inputs.items.map(input => {
                    const datasetLoc = AnyLoc.getLocFromSmart(recipeInfo.projectKey, input.ref);
                    const datasetInfo = {
                        projectKey: datasetLoc.projectKey,
                        name: datasetLoc.localId,
                    };
                    return datasetInfo;
                });
            });
        };
        const inputs = datasetInfos($scope.recipe.inputs);
        const outputs = datasetInfos($scope.recipe.outputs);
        const showInCurrentLineageOnlyFilter = false;
        const wt1EventFrom = "recipe-advanced-settings";
        ManualLineageModalService.openManualLineageModal(recipeInfo, inputs, outputs, showInCurrentLineageOnlyFilter, wt1EventFrom)
            .then(function(saveInformation) {
                if (saveInformation) {
                    DataikuAPI.dataLineage.saveManualLineage(saveInformation.projectKey, saveInformation.recipeName, saveInformation.manualDataLineages, saveInformation.ignoreAutoComputedLineage)
                        .then(({data}) => {
                            ActivityIndicator.success("Updated lineage");
                        })
                        .catch(setErrorInScope.bind($scope));
                }
            });
    }

    /* ***************************** Save *************************** */

    $scope.hooks.save = function() {
        return $scope.baseSave($scope.hooks.getRecipeSerialized(), $scope.script ? $scope.script.data : null);
    };
    $scope.hooks.origSaveHook = $scope.hooks.save;

    $scope.baseSave = function(recipeSerialized, payloadData){
        $scope.recipeWT1Event("recipe-save", null, {withRecipeEventParamsEnrichment: true, withRecipeOutputAppendModeTracking: true});
        Notification.publishToFrontend("recipe-save", {recipe: recipeSerialized, payloadData: payloadData});
        return DataikuAPI.flow.recipes.save($stateParams.projectKey, recipeSerialized,
            payloadData, $scope.currentSaveCommitMessage).success(function(savedRecipe){
            var newVersionTag = savedRecipe.versionTag;
            $scope.origRecipe = angular.copy($scope.recipe);
            $scope.origScript = angular.copy($scope.script);
            $scope.recipe.versionTag = newVersionTag;
            $scope.origRecipe.versionTag = newVersionTag;
            $scope.creation = false;
            $scope.currentSaveCommitMessage = null;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.canSave = function(){
        if (!$scope.creation) return true;
        return $scope.recipeName.name && $scope.recipeName.name.length;
    };

    $scope.hooks.recipeIsDirty = function() {
        if (!$scope.recipe) return false;
        if ($scope.creation) {
            return true;
        } else {
            // compare after fixing up the partition deps, otherwise their change is missed by the dirtyness tracking
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            var origRecipeSerialized = angular.copy($scope.origRecipe);
            PartitionDeps.prepareRecipeForSerialize(origRecipeSerialized, true);

            var dirty = !angular.equals(recipeSerialized, origRecipeSerialized);
            if ($scope.script) {
                dirty = dirty || !angular.equals($scope.origScript, $scope.script);
            }
            return dirty;
        }
    };

    $scope.hooks.recipeContainsUnsavedFormulaChanges = function() {
        return false;
    };

    //Don't link to the default recipeIsDirty is function, get the actual one that may be defined later
    checkChangesBeforeLeaving($scope, (function(_scope) { return function() {
        return _scope.hooks.recipeIsDirty() || _scope.hooks.recipeContainsUnsavedFormulaChanges();
     }
     })($scope));

    $scope.saveRecipe = function(commitMessage){
        var deferred = $q.defer();

        var saveAfterConflictCheck = function() {
            $scope.currentSaveCommitMessage = commitMessage;
            $scope.hooks.save().then(function() {
                $scope.$broadcast('recipeSaved');
                deferred.resolve('recipe saved');
            },function() {
                deferred.reject();
            });
        };

        DataikuAPI.flow.recipes.checkSaveConflict($stateParams.projectKey, $stateParams.recipeName,$scope.recipe).success(function(conflictResult) {
            if(!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope,conflictResult).then(
                        function(resolutionMethod) {
                            if(resolutionMethod == 'erase') {
                                saveAfterConflictCheck();
                            } else if(resolutionMethod == 'ignore') {
                                deferred.reject();
                                DKUtils.reloadState();
                            }
                        }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));
        return deferred.promise;
    };

    $scope.saveRecipeIfPossible = function(){
        if ($scope.canSave()) {
            return $scope.saveRecipe();
        }
        return $q.defer().promise;
    };

    $scope.displayAllMessagesInModal = function(){
        Dialogs.infoMessagesDisplayOnly($scope, "Recipe validation",
            $scope.valCtx.validationResult.allMessagesForFrontend);
    };

    /* ***************************** Execution *************************** */

    $scope.buildModes = [
        ["NON_RECURSIVE_FORCED_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.NON_RECURSIVE_FORCED_BUILD", "Run only this recipe")],
        ["RECURSIVE_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.RECURSIVE_BUILD", "Build required dependent datasets")],
        ["RECURSIVE_FORCED_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.RECURSIVE_FORCED_BUILD", "Force-rebuild all dependent datasets")],
        ["RECURSIVE_MISSING_ONLY_BUILD", $scope.translate("RECIPE.RUN_OPTIONS.BUILD_MODE.RECURSIVE_MISSING_ONLY_BUILD", "Build missing dependencies and run this recipe")]
    ];

    $scope.jobCheckTimer = null;

    $scope.hooks.preRunValidate = function() {
        var deferred = $q.defer();
        DataikuAPI.flow.recipes.generic.validate($stateParams.projectKey,
            $scope.hooks.getRecipeSerialized()).success(function(data) {
            deferred.resolve(data);
        }).error(function(a,b,c) {
            setErrorInScope.bind($scope)(a,b,c);
            deferred.reject("Validation failed");
        });
        return deferred.promise;
    };

    $scope.editRunOptions = function(){
        CreateModalFromTemplate("/templates/recipes/recipe-run-options-modal.html", $scope);
    };

    $scope.multipleLogsToDisplay = function() {
        //We're restricting this down to specific engines for now
        let logsWantedForEngine = "python" == $scope.recipe.type || "r" == $scope.recipe.type || "shell" == $scope.recipe.type;
        return $scope.startedJob.jobId && $scope.startedJob.jobStatus && logsWantedForEngine && $scope.startedJob.jobStatus.manyLogs;
    }

    $scope.selectLogType = function(logTypeKey) {
        if (!$scope.startedJob || !$scope.startedJob.jobStatus || !$scope.startedJob.jobStatus.logsByType) {
            return;
        }

        let logType = $scope.startedJob.jobStatus.logsByType[logTypeKey];

        let updateChosenLogState = function() {
            $scope.startedJob.jobStatus.logTailHTML = logType.logHTML;
            $scope.startedJob.jobStatus.chosenLog = logTypeKey;
        }

        if (logType) {
            if (logType.logHTML) {
                updateChosenLogState();
            } else {
                 DataikuAPI.flow.jobs.smartTailActivityAdditionalLog($scope.projectKey, $scope.startedJob.jobId, logType.activityId, logType.logPath, 500)
                    .success(function (data) {
                        logType.logHTML = smartLogTailToHTML(data, false);
                        updateChosenLogState();
                    })
                    .error(setErrorInScope.bind($scope));
            }
        }
    }

    $scope.waitForEndOfStartedJob = function() {
        Logger.info("Wait for end of job:", $scope.startedJob.jobId);
        DataikuAPI.flow.jobs.getJobStatus($stateParams.projectKey, $scope.startedJob.jobId).success(function(data) {
            $scope.startedJob.jobStatus = data;
            data.totalWarningsCount = 0;
            if (data.logTail != null) {
                data.logTailHTML = smartLogTailToHTML(data.logTail, false);
            }
            for (var actId in data.baseStatus.activities) {
                var activity = data.baseStatus.activities[actId];
                if (activity.warnings) {
                    data.totalWarningsCount += activity.warnings.totalCount;
                }
            }
            if (data.baseStatus.state != "DONE" && data.baseStatus.state != "ABORTED" &&
                data.baseStatus.state != "FAILED") {
                $scope.jobCheckTimer = $timeout($scope.waitForEndOfStartedJob, 2000);
            } else {
                // The run has finished

                //We'll have a log if there was a failure
                if (data.logTailHTML) {
                    //If there are additional logs from the first failed activity (e.g. python ones), we offer these to the user
                    // but we don't download them yet, just collect their info
                    let logsByType = {};
                    $scope.startedJob.jobStatus.logsByType = logsByType;

                    // the main log is displayed by default but we add it to the logsByType map so the user can switch back to it
                    const MAIN_LOG_KEY_LABEL = "Main activity log";
                    logsByType[MAIN_LOG_KEY_LABEL] =  { logHTML : data.logTailHTML};
                    $scope.startedJob.jobStatus.chosenLog = MAIN_LOG_KEY_LABEL;
                    $scope.startedJob.jobStatus.manyLogs = false;

                    if (data.logTailActivityId) {
                        let failedActivity = data.baseStatus.activities[data.logTailActivityId];
                        if (failedActivity && failedActivity.statusOutputs && failedActivity.statusOutputs.length > 0) {
                            $scope.startedJob.jobStatus.manyLogs = true;
                            for (let statusOutput of failedActivity.statusOutputs) {
                                //Add an entry with empty logHTML - we will lazy load it
                                logsByType[statusOutput.label] = { activityId : failedActivity.activityId, logPath : statusOutput.path, logHTML : null};
                            }
                        }
                    }

                }

                $scope.recipeWT1Event("recipe-run-finished", {
                    state : data.baseStatus.state
                });
            }
            $timeout(function() {$rootScope.$broadcast("reflow");},50);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.waitForEndOfStartedContinuousActivity = function() {
        Logger.info("Wait for end of continuous activity:", $scope.startedJob.jobId);
        DataikuAPI.continuousActivities.getState($stateParams.projectKey, $stateParams.recipeName).success(function(data) {
            $scope.startedJob.persistent = data;
            $scope.startedJob.current = data.mainLoopState;
            if ($scope.startedJob.current && $scope.startedJob.current.futureInfo && ($scope.startedJob.current.futureInfo.alive || !$scope.startedJob.current.futureInfo.hasResult)) {
                $scope.jobCheckTimer = $timeout($scope.waitForEndOfStartedContinuousActivity, 2000);
            } else {
                // not running anymore
            }
            $timeout(function() {$rootScope.$broadcast("reflow");},50);
        }).error(setErrorInScope.bind($scope));

    };

    $scope.discardStartedJob = function(){
        clear($scope.startedJob);
        if($scope.jobCheckTimer) {
           $timeout.cancel($scope.jobCheckTimer);
           $scope.jobCheckTimer = null;
           $timeout(function() {
               $rootScope.$broadcast('redrawFatTable');
           });
        }
    };

    $scope.abortSingleRecipeExecution = function() {
        Dialogs.confirm($scope, 'Aborting a job','Are you sure you want to abort this job?').then(function() {
            DataikuAPI.flow.jobs.abort($stateParams.projectKey,$scope.startedJob.jobId).success(function(data) {
                $scope.discardStartedJob();
            }).error(function(e) {
                // swallow this error
                Logger.error(e);
            });
            $scope.recipeWT1Event("recipe-running-abort");
        });
    };

    $scope.isJobRunning = function() { return RecipeRunJobService.isRunning($scope.startedJob); };

    $scope.isJobPending = function() { return RecipeRunJobService.isPending($scope.startedJob); };

    $scope.isContinuousActivityRunning = function() { return $scope.startedJob && $scope.startedJob.jobId && $scope.startedJob.current && $scope.startedJob.current.futureInfo && ($scope.startedJob.current.futureInfo.alive || !$scope.startedJob.current.futureInfo.hasResult); };

    //TODO @recipes32 this is a little flawed, there is a short moment between starting and running...
    $scope.isJobRunningOrStarting = function() {
        return $scope.isJobRunning() || !!$scope.startedJob.starting || $scope.isJobPending();
    };
    $scope.isContinuousActivityRunningOrStarting = function() {
        return $scope.isContinuousActivityRunning() || !!$scope.startedJob.starting;
    };

    $scope.startSingleRecipeExecution = function(forced) {
        $scope.hooks.resetScope();
        $scope.startedJob.starting = true;

        $scope.currentSaveIsForAnImmediateRun = true;

        if ($scope.recipe.redispatchPartitioning) {
            $scope.testRun.build_partitions = {}; // Build partitions make no sense when in redispatch partitioning mode
        }

        function doIt() {
            const runRecipeOnlyRecipeTypes = ["nlp_llm_finetuning"]
            RecipeRunJobService.run($scope.recipe, $scope.computablesMap, $scope.testRun, $scope.startedJob, $scope.waitForEndOfStartedJob, runRecipeOnlyRecipeTypes.includes($scope.recipe.type), $scope)
        }

        $scope.saveRecipe().then(function() {
            $scope.currentSaveIsForAnImmediateRun = false;
            if (forced) {
                $scope.recipeWT1Event("recipe-run-start-forced");
                doIt();
            } else if ($scope.recipe.params && $scope.recipe.params.skipPrerunValidate) {
                $scope.recipeWT1Event("recipe-run-start-no-validation");
                doIt();
            } else {
                $scope.recipeWT1Event("recipe-run-start");
                $scope.hooks.preRunValidate().then(function(validationResult) {
                    if (validationResult.ok == true || validationResult.error == false || validationResult.allMessagesForFrontend && !validationResult.allMessagesForFrontend.error) {
                        $scope.recipeWT1Event("recipe-run-start-validated");
                        doIt();
                    } else {
                        $scope.startedJob.starting = false;
                        $scope.valCtx.preRunValidationError = validationResult;
                        $scope.recipeWT1Event("recipe-run-start-blocked", {
                            firstError : validationResult.allMessagesForFrontend && validationResult.allMessagesForFrontend.messages.length ? validationResult.allMessagesForFrontend.messages[0].message : "unknown"
                        });
                    }
                }, function(error) {
                    $scope.startedJob.starting = false;
                });
            }
        }, function(error) {
            $scope.currentSaveIsForAnImmediateRun = false;
            $scope.startedJob.starting = false;
        });
    };


    $scope.startContinuousActivity = function(forced) {
        $scope.hooks.resetScope();
        $scope.startedJob.starting = true;

        function doIt() {
            const onceLoopParams = { abortAfterCrashes: 0 };
            DataikuAPI.continuousActivities.start($stateParams.projectKey, $stateParams.recipeName, onceLoopParams).success(function(data){
                FutureProgressModal.show($scope, data, "Starting continuous recipe...").then(function(data) {
                    $scope.startedJob.jobId = data.futureId;
                    $scope.waitForEndOfStartedContinuousActivity();
                });
            }).error(setErrorInScope.bind($scope));
        }

        $scope.saveRecipe().then(function() {
            if (forced) {
                $scope.recipeWT1Event("recipe-run-start-forced");
                doIt();
            } else if ($scope.recipe.params && $scope.recipe.params.skipPrerunValidate) {
                $scope.recipeWT1Event("recipe-run-start-no-validation");
                doIt();
            } else {
                $scope.recipeWT1Event("recipe-run-start");
                $scope.hooks.preRunValidate().then(function(validationResult) {
                    if (validationResult.ok == true || validationResult.error == false || !validationResult.allMessagesForFrontend.error) {
                        $scope.recipeWT1Event("recipe-run-start-validated");
                        doIt();
                    } else {
                        $scope.startedJob.starting = false;
                        $scope.valCtx.preRunValidationError = validationResult;
                        $scope.recipeWT1Event("recipe-run-start-blocked", {
                            firstError : validationResult.allMessagesForFrontend && validationResult.allMessagesForFrontend.messages.length ? validationResult.allMessagesForFrontend.messages[0].message : "unknown"
                        });
                    }
                }, function(error) {
                    $scope.startedJob.starting = false;
                });
            }
        }, function(error) {
            $scope.startedJob.starting = false;
        });
    };

    $scope.stopContinuousActivity = function(){
        $scope.continuousActivityState = null;
        DataikuAPI.continuousActivities.stop($stateParams.projectKey, $stateParams.recipeName).success(function(data){
            // TODO - start displaying some useful stuff...
        }).error(setErrorInScope.bind($scope));
    }

    $scope.openContinuousActivity = function() {
        $state.go("projects.project.continuous-activities.continuous-activity.runs", {continuousActivityId: $scope.recipe.name});
    };

    // Stop the timer at exit
    $scope.$on("$destroy",function() {
       if($scope.jobCheckTimer) {
           $timeout.cancel($scope.jobCheckTimer);
           $scope.jobCheckTimer = null;
       }
       Mousetrap.unbind("@ r u n");
       Mousetrap.unbind("shift+enter");
       $scope.hooks = null;
    });

    Mousetrap.bind(['@ r u n','shift+enter'], function(){
        $scope.startSingleRecipeExecution();
    });

});


app.controller("_RecipeWithEngineBehavior", function($rootScope, $scope, $q, $stateParams, DataikuAPI, Dialogs, PartitionDeps, DKUtils, Logger, CreateModalFromTemplate, AnyLoc, ActivityIndicator, translate) {
    $scope.setRecipeStatus = function(data) {
        $scope.recipeStatus = data;

        const engineType = $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.type;
        if (engineType === "SPARK") {
            $scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary.sparkPipelinesEnabled;
            };
        } else if (engineType === "SQL") {
            $scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary.sqlPipelinesEnabled;
            };
        }
    };

    $scope.hooks.updateRecipeStatus = function() {};

    var requestsInProgress = 0;
    var sendTime = 0;
    var lastSequenceId = 0;
    var lastPromise;
    // to avoid updating multiple times with same data:
    var lastPayload;
    var lastRequestData;
    var lastRecipeSerialized; //(json string)
    $scope.updateRecipeStatusBase = function(forceUpdate, payload, requestSettings) {
        var recipeCopy = angular.copy($scope.recipe);
        /* Complete the partition deps from the "fixedup" version */
        PartitionDeps.prepareRecipeForSerialize(recipeCopy);
        var recipeSerialized = angular.toJson(recipeCopy);
        var requestData = angular.toJson(requestSettings || {});

        if (!forceUpdate
            && lastPayload == payload
            && lastRequestData == requestData
            && lastRecipeSerialized == recipeSerialized) {
            Logger.info("Update recipe: cache hit, not requesting");
            // We already made this request
            return lastPromise;
        }

        lastPayload = payload;
        lastRequestData = requestData;
        lastRecipeSerialized = recipeSerialized;
        lastSequenceId++;

        requestsInProgress++;
        sendTime = new Date().getTime();
        $scope.recipeStateUpdateInProgress = true;
        lastPromise = DataikuAPI.flow.recipes.generic.getStatus(recipeCopy, payload, lastSequenceId, requestSettings)
            .catch(function(response) {
                setErrorInScope.bind($scope)(response.data, response.status, response.headers);
                //we can't get the sequenceId so wait for all answers to mark as idle
                if (requestsInProgress == 1) {
                    $scope.recipeStateUpdateInProgress = false;
                }
                return response;
            })
            .finally(function(){
                requestsInProgress--;
            })
            .then(function(response) {
                if (parseInt(response.data.sequenceId) < lastSequenceId) {
                    return; //Too late!
                }
                if (new Date().getTime() - sendTime > 1500) {
                    aPreviousCallWasLong = true;
                }
                $scope.recipeStateUpdateInProgress = false;
                $scope.setRecipeStatus(response.data);
                return response.data;
            });
        return lastPromise;
    };

    var timeout;
    $scope.updateRecipeStatusLater = function() {
        clearTimeout(timeout);
        timeout = setTimeout(function() {
            $('.CodeMirror').each(function(idx, el){Logger.debug(el.CodeMirror.refresh())});//Make sure codemirror is always refreshed (#6664 in particular)
            if (!$scope.hooks) return;
            $scope.hooks.updateRecipeStatus();
        }, 400);
    };

    /* this function helps the UI have a more appropriate look when status computation is long (small spinner, etc) */
    var aPreviousCallWasLong = false;
    $scope.expectLongRecipeStatusComputation = function() {
        return !$scope.recipeStatus || !$scope.recipeStatus.selectedEngine || aPreviousCallWasLong;
    };

    $scope.canChangeEngine = function() {
        if(!$scope.recipeStatus || !$scope.recipeStatus.engines) {
            return false;
        }
        if ($scope.isJobRunningOrStarting() || $scope.recipeStateUpdateInProgress) {
            return false;
        }
        return true;
    };

    $scope.convertToQueryRecipe = function(type, label) {
        Dialogs.confirm($scope, translate("RECIPES.CONVERT_TO", "Convert to " + label + " recipe", {recipeType: label}),
                        translate("RECIPES.CONVERTING_TO", "Converting the recipe to " + label + " will enable you to edit the query, but you will not be able to use the visual editor anymore.", {recipeType: label})
                        + "<br/><strong>" + translate("RECIPES.THIS_OPERATION_IS_IRREVERSIBLE", "This operation is irreversible.") + "</strong>")
        .then(function() {
            var payloadData = $scope.hooks.getPayloadData();
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            $scope.hooks.save().then(function() {
                DataikuAPI.flow.recipes.visual.convert($stateParams.projectKey, recipeSerialized, payloadData, type)
                .success(function(data) {
                    DKUtils.reloadState();
                }).error(setErrorInScope.bind($scope));
            });
        });
    };

    $scope.showSQLModal = function(){
        var newScope = $scope.$new();
        newScope.convert = $scope.convertToQueryRecipe;
        newScope.uiState = {currentTab: 'query'};
        $scope.hooks.updateRecipeStatus(false, true).then(function(){
            // get the latest values, not the ones of before the updatestatus call
        	newScope.query = $scope.recipeStatus.sql;
        	newScope.engine = $scope.recipeStatus.selectedEngine.type;
        	newScope.executionPlan = $scope.recipeStatus.executionPlan;
        	CreateModalFromTemplate("/templates/recipes/fragments/sql-modal.html", newScope);
        });
    };

    var save = $scope.baseSave;
    $scope.baseSave = function() {
        var p = save.apply(this, arguments);
        p.then($scope.updateRecipeStatusLater);
        return p;
    };

    $scope.$watchCollection("recipe.inputs.main.items", () => {
        //call updateRecipeStatus without args!
        Promise.resolve($scope.hooks.updateRecipeStatus()).catch(() => {
            Logger.info("Failed to updateRecipeStatus. Likely due to result of backend call discarded due to multiple parallel calls.");
        });
    });
    $scope.$watchCollection("recipe.outputs.main.items", () => {
        //call updateRecipeStatus without args!
        Promise.resolve($scope.hooks.updateRecipeStatus()).catch(() => {
            Logger.info("Failed to updateRecipeStatus. Likely due to result of backend call discarded due to multiple parallel calls.");
        });
    });

    $scope.$watch("params.engineParams", $scope.updateRecipeStatusLater, true);
});


app.controller("SqlModalController", function($scope, CodeMirrorSettingService) {

    $scope.editorOptions = CodeMirrorSettingService.get('text/x-sql2');

    // if ($scope.engine == 'HIVE' || $scope.engine == 'IMPALA' || $scope.engine == 'SPARK') {
    //     $scope.editorOptions.mode = 'text/x-hive';
    // }
});


app.directive("recipeEnginesPreferenceConfig", function(){
    return {
        restrict: 'A',
        templateUrl : '/templates/recipes/widgets/recipe-engines-preference-config.html',
        scope: {
            model: '='
        }
    }
});


app.service('RecipesEnginesService', function($rootScope, $q, Assert, CreateModalFromTemplate, DataikuAPI) {
    this.startChangeEngine = function(selectedItems) {
        return CreateModalFromTemplate("/templates/recipes/fragments/change-recipes-engines-modal.html", $rootScope, null, function(modalScope) {
            modalScope.selectedRecipes = selectedItems.filter(it => it.type == 'RECIPE');
            modalScope.options = {};
            modalScope.AUTO = '__AUTO__';

            modalScope.getEngineShortStatus = function(engine) {
                for(let msg of engine.messages.messages) {
                    if (msg.severity == "ERROR") {
                        return msg.details;
                    }
                }
                for(let msg of engine.messages.messages) {
                    if (msg.severity == "WARNING") {
                        return msg.details;
                    }
                }
            };

            DataikuAPI.flow.recipes.massActions.startChangeEngines(modalScope.selectedRecipes).success(function(data) {
                Assert.trueish(data.engines, 'no engines');
                modalScope.availableEngines = data.engines;
                modalScope.options.engine = data.currentEngine;
                modalScope.nUnselectableEngines = data.engines.filter(e => !e.isSelectable).length;
                modalScope.messages = data.messages;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });


            modalScope.test = function() {
                const deferred = $q.defer();
                delete modalScope.messages;
                delete modalScope.maxSeverity;
                resetErrorInScope(modalScope);
                DataikuAPI.flow.recipes.massActions.testChangeEngines(modalScope.selectedRecipes, modalScope.options.engine).success(function(data) {
                    modalScope.messages = data.messages;
                    modalScope.maxSeverity = data.maxSeverity || 'OK';
                    if (modalScope.maxSeverity != 'OK') {
                        deferred.reject();
                    } else {
                        deferred.resolve(data)
                    }
                }).error(setErrorInScope.bind(modalScope));
                return deferred.promise;
            };

            modalScope.ok = function(force) {
                if (force || modalScope.options.engine == modalScope.AUTO) { //No need to test AUTO
                    performChange();
                } else {
                    modalScope.test().then(performChange);
                }
            };

            function performChange() {
                DataikuAPI.flow.recipes.massActions.changeEngines(modalScope.selectedRecipes, modalScope.options.engine).success(function(data) {
                    modalScope.resolveModal();
                }).error(setErrorInScope.bind(modalScope));
            }
        });
    };
});


    app.directive("codeEnvSelectionForm", function(DataikuAPI, $stateParams, translate) {
    return {
        restrict: 'A',
        templateUrl : '/templates/recipes/fragments/code-env-selection-form.html',
        scope: {
            envSelection: '=codeEnvSelectionForm',
            inPlugin: '=',
            isStep: '=',
            envLang: '=',
            selectionLabel: '=',
            callback: '=?',
            helpLabel: '=',
            checkCodeEnv: '=?',
            readOnly: '<',
        },
        link: function($scope, element, attrs) {
            $scope.customWarningMessage = null;
            if ($scope.inPlugin == true) {
                $scope.envModes = [
                    ['USE_BUILTIN_MODE', translate('SCENARIO.SETTINGS.CODE_ENV.IN_PLUGIN.USE_BUILTIN_MODE', 'Use plugin environment')],
                    ['EXPLICIT_ENV', translate('SCENARIO.SETTINGS.CODE_ENV.IN_PLUGIN.EXPLICIT_ENV', 'Select an environment')],
                ];
            } else {
                $scope.envModes = [
                    ['USE_BUILTIN_MODE', translate('SCENARIO.SETTINGS.CODE_ENV.USE_BUILTIN_MODE', 'Use DSS builtin env')],
                    ['INHERIT', translate('SCENARIO.SETTINGS.CODE_ENV.INHERIT', 'Inherit project default')],
                    ['EXPLICIT_ENV', translate('SCENARIO.SETTINGS.CODE_ENV.EXPLICIT_ENV', 'Select an environment')]
                ];
            }

            function setDefaultValue() {
                if (!$scope.envSelection) { // not ready
                    return;
                }
                if ($scope.envSelection.envMode == "EXPLICIT_ENV" && $scope.envSelection.envName == null && $scope.envNamesWithDescs && $scope.envNamesWithDescs.envs && $scope.envNamesWithDescs.envs.length > 0) {
                    $scope.envSelection.envName = $scope.envNamesWithDescs.envs[0].envName;
                }
            }
            $scope.$watch("envSelection.envMode", setDefaultValue);

            $scope.envNamesWithDescs = [];
            DataikuAPI.codeenvs.listNamesWithDefault($scope.envLang, $stateParams.projectKey).success(function(data) {
                $scope.envNamesWithDescs = data;
                data.envs.forEach(function(x) {
                    if (x.owner) {
                        x.envDesc = x.envName + " (" + x.owner + ")";
                    } else {
                        x.envDesc = x.envName;
                    }
                });
                if (!$scope.inPlugin) {
                    if (data.resolvedInheritDefault == null) {
                        $scope.envModes[1][1] = translate('SCENARIO.SETTINGS.CODE_ENV.INHERIT_DSS_BUILTIN', "Inherit project default (DSS builtin env)")
                    } else {
                        $scope.envModes[1][1] = translate('SCENARIO.SETTINGS.CODE_ENV.INHERIT_CUSTOM', "Inherit project default ({{resolvedInheritDefault}})", { resolvedInheritDefault: data.resolvedInheritDefault });
                        $scope.inheritedEnv = data.envs.filter(e=>e.envName === data.resolvedInheritDefault)[0];
                    }
                }
                setDefaultValue();
            }).error(setErrorInScope.bind($scope));
            function setSelectedEnv() {
                if ($scope.envNamesWithDescs.envs && $scope.envSelection) {
                    $scope.selectedEnv = $scope.envNamesWithDescs.envs.filter(e=>e.envName === $scope.envSelection.envName)[0];
                }
            }
            $scope.$watchGroup(["envSelection.envName", "envNamesWithDescs.envs"], setSelectedEnv);
            if (typeof $scope.callback === 'function') {
                $scope.callback($scope);
            }
            if (typeof $scope.checkCodeEnv === 'function') {
                $scope.$watch("envSelection", function(envSelection) {
                    $scope.customWarningMessage = $scope.checkCodeEnv(envSelection);
                }, true);
            }
        }
    }
});

app.directive("containerSelectionForm", function(DataikuAPI, $stateParams){
        return {
            restrict: 'A',
            templateUrl : '/templates/recipes/fragments/container-selection-form.html',
            scope: {
                containerSelection: '=containerSelectionForm',
                selectionLabel: '=',
                inPlugin: '=',
                workloadType: '=',
                hideInherit: '=',
                inheritedFrom: '<',
                readOnly: '<',
                inlineHelp: '<',
                containerSpecificContext: '=?',
            },
            link: {
                post: function($scope, element, attrs) {
                    if (!$scope.inheritedFrom) {
                        $scope.inheritedFrom = 'project'
                    }

                    if ($scope.hideInherit) {
                        $scope.containerModes = [
                            ['NONE', 'None - Use backend to execute'],
                            ['EXPLICIT_CONTAINER', 'Select a container configuration'],
                        ];
                    } else {
                        $scope.containerModes = [
                            ['NONE', 'None - Use backend to execute'],
                            ['INHERIT', 'Inherit ' + $scope.inheritedFrom +' default'],
                            ['EXPLICIT_CONTAINER', 'Select a container configuration'],
                        ];
                    }

                    $scope.containerNames = [];
                    let workloadType = $scope.workloadType || "USER_CODE";
                    if ($stateParams.projectKey) {
                        DataikuAPI.containers.listNamesWithDefault($stateParams.projectKey, null, workloadType, $scope.containerSpecificContext).success(function(data) {
                            $scope.containerNames = data.containerNames;
                            if (data.resolvedInheritValue) {
                                $scope.containerModes[1][1] += ' (' + data.resolvedInheritValue + ')';
                            } else {
                                $scope.containerModes[1][1] += ' (local execution)';
                            }
                        }).error(setErrorInScope.bind($scope));
                    } else {
                        DataikuAPI.containers.listNames(null, workloadType).success(function(data) {
                            $scope.containerNames = data;
                        }).error(setErrorInScope.bind($scope));
                    }
                }
            }



        }
    });

/**
 * Inputs for specifying the container configuration to apply in the context of the hyperparameter search.
 * @param {object} searchParams: the parameters of the hyperparameter search (either from an analysis or a recipe)
 * @param {function} hasSelectedK8sContainer: tells whether the user has selected a k8s container to run the search
 * @param {boolean} isTimeseriesForecasting: used to determine whether we show a warning for non reproductible search
 */
app.component('mlHpDistribution', {
     templateUrl : '/templates/recipes/fragments/ml-hp-distribution.html',
     bindings: {
         searchParams: '=',
         hasSelectedK8sContainer: '<',
         k8sRuntimeEnvTooltip: '@?',
         isTimeseriesForecasting: '<'
     },
     controller: function() {
         const $ctrl = this;

         $ctrl.getK8sRuntimeEnvTooltip = () => {
             if ($ctrl.k8sRuntimeEnvTooltip) {
                 return $ctrl.k8sRuntimeEnvTooltip;
             }

             return  "Distributed search requires a Kubernetes container configuration to be selected";
         };

         $ctrl.showHpSearchNotReproducibleWarning = () => {
            return $ctrl.isTimeseriesForecasting && $ctrl.searchParams.nJobs !== 1;
        };

     },
});

app.controller("_ContinuousRecipeInitStartedJobBehavior", function ($scope, $stateParams, DataikuAPI, Logger) {
    // get the current state
    DataikuAPI.continuousActivities.getState($stateParams.projectKey, $stateParams.recipeName).success(function(data) {
        $scope.startedJob = $scope.startedJob || {};
        $scope.startedJob.persistent = data;
        $scope.startedJob.current = data.mainLoopState;
        if (data.mainLoopState) {
            if (!data.mainLoopState.futureInfo || !data.mainLoopState.futureInfo.hasResult) {
                $scope.startedJob.jobId = data.mainLoopState.futureId;
                $scope.waitForEndOfStartedContinuousActivity();
            }
        }
    }).error(function() {
        Logger.warn("Recipe " + $stateParams.recipeName + " doesn't have a continuous activity yet")
    });

});

/**
 * Show the interface to chose options for some aggregations in Group, Pivot and Window visual recipes
 * Different options can be shown of not depending on whether they make sense for that recipe.
 * @param {boolean} showOrderBy controls whether the 'Order First/Last By' option is shown in the popover
 * @param {boolean} showFirstLastNotNull controls whether the 'First/Last not null' option is shown in the popover
 * @param {function} orderByColumnsGetter a function returning the list of valid columns that can be chosen in the order by option. Required if and only if showOrderBy is true
 * @param {OnBeforeUnloadEventHandlerNonNull} engineCanAggrFirstNotNull disables the 'First/Last not null' option with a popup explaining that the engine is limiting. Required if and only if showFirstLastNotNull is true
 * @param {OnBeforeUnloadEventHandlerNonNull} engineCanAggrConcatDistinct same with the 'Concat distinct' option
 * @param {object} column the colums on which the aggregations are being defined
 */
app.component('aggregationOptionsPopover', {
    templateUrl: '/templates/recipes/fragments/aggregation-options-popover.html',
    bindings: {
        showOrderBy: '<',
        orderByColumnsGetter: '<?', // required if and only if showOrderBy is true
        showFirstLastNotNull: '<',
        engineCanAggrFirstNotNull: '<?', // required if and only if showFirstLastNotNull is true
        engineCanAggrConcatDistinct: '<',
        column: '=',
    },
    controller: function(translate) {
        const $ctrl = this;

        $ctrl.translate = translate;
    },
});

})();

;
(function(){
'use strict';

const app = angular.module('dataiku.recipes');


app.controller("_RecipeCreationControllerBase", function($scope, WT1, Dialogs, PartitionDeps, DataikuAPI, RecipeDescService, RecipeComputablesService, SqlConnectionNamespaceService) {
    $scope.setComputablesMap = function(map) {
        $scope.computablesMap = map;
        $scope.$broadcast('computablesMapChanged');
    };

    $scope.recipeWT1Event = function(type, params) {
        WT1.tryEvent(type, () => {
            if (params == null) params = {};
            params.recipeId = ($scope.recipeName && $scope.recipeName.name) ? $scope.recipeName.name.dkuHashCode() : "unknown";
            params.recipeType = ($scope.recipe ? $scope.recipe.type : "unknown");
            params.creation = $scope.creation;
            return params;
        });
    };

    // Creates the recipe object and sends it to the backend
    // Default generic implementation, override it in the recipe controller for type specific handling
    $scope.doCreateRecipe = function() {
        var recipe = angular.copy($scope.recipe);
        if ($scope.recipeName) {
            recipe.name = $scope.recipeName.name; // TODO @recipes move to backend
        }
        PartitionDeps.prepareRecipeForSerialize(recipe); //TODO @recipes move to backend

        const settings = {
            script: $scope.script
        };
        if ($scope.zone) {
            settings.zone = $scope.zone;
        }
        if ($scope.recipeAdditionalParams) {
            settings.originRecipes = $scope.recipeAdditionalParams.originRecipes;
            settings.originDataset = $scope.recipeAdditionalParams.originDataset;
        }

        return DataikuAPI.flow.recipes.generic.create(recipe, settings);
    };

    // launches the recipe creation and triggers associated events
    // for tracking, disabling creation button while the recipe is in creation, etc
    // transitions to the recipe page when it is creations
    $scope.createRecipe = function() {
        const isInsert = $scope.recipeAdditionalParams != null && $scope.recipeAdditionalParams.originDataset != null && $scope.recipeAdditionalParams.originRecipes != null;
        const wt1EventParams = { insert: isInsert, ...($scope.wt1EventAdditionalParams || {}) };
        if ($scope.recipeSubType) {
            wt1EventParams.recipeSubType = $scope.recipeSubType;
        }
        $scope.recipeWT1Event("recipe-create-" + $scope.recipeType, wt1EventParams);
        var p = $scope.doCreateRecipe();
        if (p) {
            $scope.creatingRecipe = true;
            p.success(function(data) {
                if ($scope.io && $scope.io.newOutputTypeRadio == 'create') {
                    const fromElasticSearchExportQuery = $scope.recipeAdditionalParams && !!$scope.recipeAdditionalParams.elasticSearchQuery;  // Was created from dataset "search" tab
                    WT1.tryEvent("create-dataset", () => ({
                        connectionType: ($scope.newOutputDataset && $scope.newOutputDataset.connectionOption) ? $scope.newOutputDataset.connectionOption.connectionType : "unknown",
                        partitioningFrom: $scope.newOutputDataset ? $scope.newOutputDataset.partitioningOption : "unknown",
                        recipeType: $scope.recipeType,
                        fromElasticSearchExportQuery: fromElasticSearchExportQuery,
                        insert: isInsert
                    }));
                }
                $scope.creatingRecipe = false;
                Dialogs.confirmInfoMessages($scope.$parent.$parent, "Recipe creation", data.messages, null, true).then(function(){
                    $scope.$state.go('projects.project.recipes.recipe', {recipeName: data.id, recipeAdditionalParams: $scope.recipeAdditionalParams});
                })
                $scope.dismiss();


            }).error(function(a, b, c) {
                $scope.creatingRecipe = false;
                setErrorInScope.bind($scope)(a,b,c);
            });
        }
    };

    $scope.cannotModifyInputDataset = function() {
        return $scope.recipeAdditionalParams && $scope.recipeAdditionalParams.originDataset && $scope.recipeAdditionalParams.originRecipes && $scope.recipeAdditionalParams.originRecipes.length;
    }

    $scope.getDatasetInputSelectorTooltip = function () {
        if ($scope.cannotModifyInputDataset()) {
            return $scope.translate("FLOW.CREATE_RECIPE.INSERT_NO_CHANGE_INPUT", "You cannot change the input dataset when inserting a recipe into the flow");
        }
        return null;
    }

    function updateRecipeDesc() {
        $scope.recipeDesc = RecipeDescService.getDescriptor($scope.recipe.type);
        const editableFilterLocation = "modal";
        $scope.isSingleInputRecipe = RecipeDescService.isSingleInputRecipe($scope.recipe.type, editableFilterLocation);
    }

    if ($scope.recipe && $scope.recipe.type) {
        updateRecipeDesc();
    } else {
        $scope.$watch("recipe.type", function(nv) {
            nv && updateRecipeDesc();
        });
    }

    $scope.$watch("newOutputDataset.connectionOption.connectionName", function() {
        if ($scope.newOutputDataset && $scope.newOutputDataset.connectionOption) {
            SqlConnectionNamespaceService.setTooltips($scope, $scope.newOutputDataset.connectionOption.connectionType);
        }
        SqlConnectionNamespaceService.resetState($scope, $scope.newOutputDataset);
    });
});


app.controller("RecipeCopyController", function($scope, $controller, $stateParams, WT1,
               Assert, DataikuAPI, DatasetUtils, RecipeComputablesService, RecipeDescService, SavedModelsService, Logger) {
    Assert.inScope($scope, 'recipe');
    Assert.inScope($scope, 'newInputs');
    Assert.inScope($scope, 'newOutputs');

    $controller("_RecipeCreationControllerBase", {$scope: $scope});
    $controller("_RecipeOutputNewManagedBehavior", {$scope:$scope});
    addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

    // to use the _RecipeOutputNewManagedBehavior fully (when adding a new output)
    $scope.setErrorInTopScope = function(scope) {
        return setErrorInScope.bind($scope);
    };

    $scope.role = null;

    function endEditOutput() {
        $scope.uiState.editingOutput = null;
        $scope.role = null;
    }


    $scope.isOutputRoleAvailableHook = function(role) {
        return true; // an available output in the copied recipe will always be available in the resulting recipe no matter the payload.
    };

    $scope.setupUsableOutputs = function (role, acceptedType) {
        if (!$scope.computablesMap) {
            return;
        }
        Assert.trueish(role, 'no role');

        $scope.role = role;
        var roleName = role.name;
        $scope.noDataset = acceptedType != 'DATASET';
        $scope.noFolder = acceptedType != 'MANAGED_FOLDER';
        $scope.noEvaluationStore = acceptedType != 'MODEL_EVALUATION_STORE';
        $scope.noStreamingEndpoint = acceptedType != 'STREAMING_ENDPOINT';
        if (!$scope.noDataset) {
            $scope.io.newOutputTypeRadio = 'create';
        } else if (!$scope.noFolder) {
            $scope.io.newOutputTypeRadio = 'new-odb';
        } else if (!$scope.noEvaluationStore) {
            RecipeComputablesService.setMesFlavor($scope, $scope.role);
            $scope.io.newOutputTypeRadio = 'new-mes';
        } else if (!$scope.noStreamingEndpoint) {
            $scope.io.newOutputTypeRadio = 'new-se';
        }

        var outputList = RecipeComputablesService.buildPossibleOutputList($scope.recipe, $scope.computablesMap, $scope.role, $scope.editOutput.filter)
            .filter(function(item) {
                return usableAsOutput(item);
            });

        // Sort possible outputs
        function usableAsOutput(computable) {
            return computable.usableAsOutput[roleName] && computable.usableAsOutput[roleName].usable && !computable.alreadyUsedAsOutputOf;
        }
        outputList.sort(function(a,b) {
            // put usable first
            if (usableAsOutput(a) && !usableAsOutput(b)) {
                return -1;
            }
            if (usableAsOutput(b) && !usableAsOutput(a)) {
                return 1;
            }
            //Otherwise sort by "label" (display name)
            return (a.label || '').localeCompare((b.label || ''));
        });

        $scope.editOutput.usable = outputList;
    };


    $scope.startEditOutput = function(roleName, index) {
        $scope.uiState.backendWarnings = null;
        $scope.uiState.editingOutput = {role: roleName, index: index};
        var outputName = $scope.recipe.outputs[roleName].items[index].ref;
        var computableType = $scope.computablesMap[outputName].type;

        if ($scope.outputRolesIndex == null || !$scope.outputRolesIndex[roleName]) {
            // This should not happen, maybe a new custom type was added and descriptors are not up-to-date
            throw new Error("Role not found in recipe descriptor, try reloading the page");
        }
        $scope.setupUsableOutputs($scope.outputRolesIndex[roleName], computableType);

        // the select element seems to be caching something, and after hiding and showing the
        // create new dataset form a few times (2 times on firefox, 3 on chrome) the option
        // shown to be selected is incorrect ('nothing selected' but the option is not null).
        // it's probably a race condition somewhere, so we solve it the hard way: make the
        // select reinitialize its sate each  time
        $scope.newOutputDataset.connectionOption = null;
        $scope.getManagedDatasetOptions($scope.role.name).then(function(data){
            $scope.setupManagedDatasetOptions(data);
        });
        $scope.getManagedFolderOptions($scope.role.name).then(function(data){
            $scope.setupManagedFolderOptions(data);
        });
        $scope.getModelEvaluationStoreOptions($scope.role.name).then(function(data){
            $scope.setupModelEvaluationStoreOptions(data);
        });
        $scope.getStreamingEndpointOptions($scope.role.name).then(function(data){
            $scope.setupStreamingEndpointOptions(data);
        });
    };

    //Called by a click on close new output
    $scope.cancelAddOutput = function() {
        endEditOutput();
    };

    $scope.acceptEdit = function(computable) {
        var editingOutput = $scope.uiState.editingOutput;
        $scope.newOutputs[editingOutput.role].items[editingOutput.index] = {ref: computable.smartName};
        endEditOutput();
    };

    // We should have a replacement for each input/output
    $scope.formIsValid = function() {
        if ($scope.isSingleOutputRecipe) {
            if ($scope.io.newOutputTypeRadio == 'create') {
                return $scope.newOutputDataset &&
                    $scope.newOutputDataset.name &&
                    $scope.newOutputDataset.connectionOption &&
                    $scope.isDatasetNameUnique($scope.newOutputDataset.name);
            } else if ($scope.io.newOutputTypeRadio == 'new-se') {
                return $scope.newOutputSE &&
                    $scope.newOutputSE.name &&
                    $scope.newOutputSE.connectionOption &&
                    $scope.isStreamingEndpointNameUnique($scope.newOutputSE.name);
            } else if ($scope.io.newOutputTypeRadio == 'select') {
                return $scope.io.existingOutputDataset && $scope.io.existingOutputDataset.length;
            } else {
                return false;
            }
        } else {
            var valid = true;
            $.each($scope.newInputs, function(roleName, role) {
                role.items.forEach(function(input) {
                    if (!input || !input.ref) {
                        valid = false;
                    }
                });
            });
            $.each($scope.newOutputs, function(roleName, role) {
                role.items.forEach(function(output) {
                    if (!output || !output.ref) {
                        valid = false;
                    }
                });
            });
            return valid;
        }
    };

    // Clicked on "create recipe", force=true to ignore warnings
    $scope.copy = function(force) {
        var doIt = function() {
            $scope.recipeWT1Event("recipe-copy-" + $scope.recipe.type);

            $scope.creatingRecipe = true; // for ui, avoids clicking twice on create recipe

            // Single ouput recipes have a simpler UI with different models to store the data
            var copySettings;
            if ($scope.isSingleOutputRecipe) {
                var createOutput = $scope.io.newOutputTypeRadio == 'create';
                var outputName = $scope.io.newOutputTypeRadio == 'create' ? $scope.newOutputDataset.name : ($scope.io.newOutputTypeRadio == 'new-se' ? $scope.newOutputSE.name : $scope.io.existingOutputDataset);
                var singleOutputRoleName = Object.keys($scope.newOutputs)[0];
                var outputs = {}
                outputs[singleOutputRoleName] = {items: [{ref: outputName}]}
                copySettings = {
                    zone: $scope.zone,
                    inputs : $scope.newInputs,
                    outputs : outputs,
                    createOutputDataset : $scope.io.newOutputTypeRadio == 'create',
                    createOutputStreamingEndpoint : $scope.io.newOutputTypeRadio == 'new-se',
                    outputDatasetSettings : $scope.getDatasetCreationSettings(),
                    outputStreamingEndpointSettings : $scope.getStreamingEndpointCreationSettings()
                };
            } else {
                copySettings = {
                    zone: $scope.zone,
                    inputs : $scope.newInputs,
                    outputs : $scope.newOutputs
                };
            }

            DataikuAPI.flow.recipes.generic.copy($stateParams.projectKey,
                                                 $scope.recipe.projectKey,
                                                 $scope.recipe.name,
                                                 copySettings)
            .success(function(data){
                if (copySettings.createOutputDataset) {
                    WT1.event("create-dataset", {
                        connectionType: ($scope.newOutputDataset && $scope.newOutputDataset.connectionOption) ? $scope.newOutputDataset.connectionOption.connectionType : "unknown",
                        partitioningFrom: $scope.newOutputDataset ? $scope.newOutputDataset.partitioningOption : "unknown",
                        recipeType: $scope.recipe ? $scope.recipe.type : "unknown"
                    });
                }
                $scope.creatingRecipe = false;
                $scope.dismiss();
                $scope.$state.go('projects.project.recipes.recipe', {
                    recipeName: data.id
                });
            }).error(function(a, b, c) {
                $scope.creatingRecipe = false;
                setErrorInScope.bind($scope)(a,b,c);
            });
        };

        if (!$scope.isSingleOutputRecipe || ['select', 'new-odb', 'new-se'].indexOf($scope.io.newOutputTypeRadio) >= 0 || force) {
            doIt();
        } else {
            DataikuAPI.datasets.checkNameSafety($stateParams.projectKey, $scope.newOutputDataset.name, $scope.getDatasetCreationSettings())
                .success(function(data) {
                    $scope.uiState.backendWarnings = data.messages;
                    if (!data.messages || !data.messages.length) {
                        doIt();
                    }
                })
                .error(function(){
                    Logger.error("Check name failed.", arguments);
                    doIt(); // don't block the creation
                });
        }
    };

    $scope.editOutput = {filter: ''};
    $scope.uiState = $scope.uiState || {};
    $scope.uiState.editingOutput = null;

    var index = RecipeDescService.getRolesIndex($scope.recipe.type);
    $scope.inputRolesIndex = index.inputs;
    $scope.outputRolesIndex = index.outputs;

    // Init new outputs with the same roles as original recipe outputs
    // Also count inputs/outputs to adapt UI
    var nOutputs = 0;
    var nInputs = 0;
    $.each($scope.recipe.outputs, function(roleName, role) {
        $scope.newOutputs[roleName] = {items: $scope.recipe.outputs[roleName].items.map(function(){return null;})};
        nOutputs += role.items.length;
    });
    $.each($scope.recipe.inputs, function(roleName, role) {
        nInputs += role.items.length;
    });

    $scope.hasSingleOutput = nOutputs == 1;
    $scope.hasSingleInput = nInputs == 1;
    $scope.isSingleOutputRecipe = RecipeDescService.isSingleOutputRecipe($scope.recipe.type);

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map) {
        $scope.setComputablesMap(map);

        DatasetUtils.listUsabilityInAndOut($scope.recipe.projectKey, $scope.recipe.type).then(function(data) {
            // Compute usable inputs for each role
            $scope.availableInputDatasets = {}
            $scope.availableInputFolders = {};
            $scope.availableInputModels = {};
            $scope.availableInputEndpoints = {};
            $scope.availableInputMES = {};
            // TODO @rag: add mapping for RKs

            $scope.recipeDesc.inputRoles.forEach(function(role) {
                $scope.availableInputDatasets[role.name] = data[0].filter(function(computable) {
                    return computable.type == 'DATASET' && computable.usableAsInput[role.name] && computable.usableAsInput[role.name].usable;
                });
                $scope.availableInputFolders[role.name] = data[0].filter(function(computable) {
                    return computable.type == 'MANAGED_FOLDER' && computable.usableAsInput[role.name] && computable.usableAsInput[role.name].usable;
                });
                $scope.availableInputMES[role.name] = data[0].filter(function(computable) {
                    return computable.type == 'MODEL_EVALUATION_STORE' && computable.usableAsInput[role.name] && computable.usableAsInput[role.name].usable;
                });

                const savedModelInput = $scope.recipe.inputs.model && $scope.recipe.inputs.model.items[0];
                const savedModelDetails = ($scope.computablesMap[savedModelInput && savedModelInput.ref] || {}).model;
                const originalInputModelPredType = savedModelDetails && savedModelDetails.miniTask && savedModelDetails.miniTask.predictionType;

                $scope.availableInputModels[role.name] = data[0].filter(function(computable) {
                    if (computable.type != 'SAVED_MODEL') {
                        return false;
                    }
                    if ($scope.recipe.type == 'prediction_scoring' || $scope.recipe.type == 'evaluation') {
                        if (computable.model.miniTask && computable.model.miniTask.taskType != 'PREDICTION') {
                            return false;
                        }
                        if (computable.model.miniTask && computable.model.miniTask.predictionType !== originalInputModelPredType) {
                            return false;
                        }
                    } else if ($scope.recipe.type == 'clustering_scoring' && computable.model.miniTask && computable.model.miniTask.taskType != 'CLUSTERING') {
                        return false;
                    }
                    return computable.usableAsInput[role.name] && computable.usableAsInput[role.name].usable;
                });
                $scope.availableInputEndpoints[role.name] = data[0].filter(function(computable) {
                    return computable.type == 'STREAMING_ENDPOINT' && computable.usableAsInput[role.name] && computable.usableAsInput[role.name].usable;
                });
            });
        });
    });

    $scope.$watch("editOutput.filter", function() {
        try {
            if (!$scope.uiState.editingOutput) return;
            var roleName = $scope.uiState.editingOutput.role;
            var index = $scope.uiState.editingOutput.index; //index within role
            var outputName = $scope.recipe.outputs[roleName].items[index].ref;
            var computableType = $scope.computablesMap[outputName].type;
            $scope.setupUsableOutputs($scope.role, computableType);
        } catch (e) {
            Logger.error("Filter output failed", e)
        }
    });
});


app.controller("SingleOutputRecipeCopyController", function($scope, $controller, $stateParams, Assert, DataikuAPI, DatasetUtils, RecipeComputablesService) {
    Assert.inScope($scope, 'recipeDesc');
    Assert.trueish($scope.recipeDesc.outputRoles, 'no output roles');

    $scope.role = $scope.recipeDesc.outputRoles[0];

    var updateUsableOutputs = function() {
        var outputName = $scope.recipe.outputs[$scope.role.name].items[0].ref;
        var computableType = $scope.computablesMap[outputName].type;
        $scope.setupUsableOutputs($scope.role, computableType);
    };

    var updateManagedDatasetOptions = function(forceUpdate) {
        var fakeRecipe = angular.copy($scope.recipe);
        fakeRecipe.projectKey = $stateParams.projectKey;
        fakeRecipe.inputs = $scope.newInputs;
        DataikuAPI.datasets.getManagedDatasetOptions(fakeRecipe, $scope.role.name).success(function(data) {
            $scope.setupManagedDatasetOptions(data, forceUpdate);
        });
        DataikuAPI.datasets.getManagedFolderOptions(fakeRecipe, $scope.role.name).success(function(data) {
            $scope.setupManagedFolderOptions(data, forceUpdate);
        });
        DataikuAPI.datasets.getModelEvaluationStoreOptions(fakeRecipe, $scope.role.name).success(function(data) {
            $scope.setupModelEvaluationStoreOptions(data, forceUpdate);
        });
        DataikuAPI.datasets.getStreamingEndpointOptions(fakeRecipe, $scope.role.name).success(function(data) {
            $scope.setupStreamingEndpointOptions(data, forceUpdate);
        });
    };

    updateUsableOutputs();
    updateManagedDatasetOptions();

    $scope.$watch("sourceRecipe", function(nv) {
        if (!nv) return;
        updateManagedDatasetOptions(true);
    });

    $scope.$watch("computablesMap", function(nv) {
        if (!nv) return;
        updateUsableOutputs();
    });
});


app.controller("SingleOutputDatasetRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $q,Dialogs, DatasetsService, WT1, DatasetUtils, $controller, RecipeComputablesService, Logger, SmartId) {
    $controller("_RecipeCreationControllerBase", {$scope:$scope});
    $controller("_RecipeOutputNewManagedBehavior", {$scope:$scope});

    addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

    // for safety, to use the _RecipeOutputNewManagedBehavior fully (maybe one day)
    $scope.setErrorInTopScope = function(scope) {
        return setErrorInScope.bind($scope);
    };

    $scope.singleOutputRole = {name:"main", arity:"UNARY", acceptsDataset:true};

    var updateManagedDatasetOptions = function(recipeType, inputRef, forceUpdate) {
        var fakeRecipe = {
            type : recipeType,
            projectKey : $stateParams.projectKey,
        }
        if (inputRef) {
            fakeRecipe.inputs = {main : {items : [{ref : inputRef}]}};
        }
        DataikuAPI.datasets.getManagedDatasetOptions(fakeRecipe, "main").success(function(data) {
            $scope.setupManagedDatasetOptions(data, forceUpdate);
        });
        DataikuAPI.datasets.getManagedFolderOptions(fakeRecipe, "main").success(function(data) {
            $scope.setupManagedFolderOptions(data, forceUpdate);
        });
        DataikuAPI.datasets.getModelEvaluationStoreOptions(fakeRecipe, "main").success(function(data) {
            $scope.setupModelEvaluationStoreOptions(data, forceUpdate);
        });
        DataikuAPI.datasets.getStreamingEndpointOptions(fakeRecipe, "main").success(function(data) {
            $scope.setupStreamingEndpointOptions(data, forceUpdate);
        });
    };

    $scope.maybeSetNewDatasetName = function(newName) {
        if ($scope.newOutputDataset && !$scope.newOutputDataset.name && newName) {
            $scope.newOutputDataset.name = newName;
        }
    };

    var makeMainRole = function (refs) {
        return {
            main: {
                items: refs.filter(function(ref) {return !!ref;}).map(function(ref) {return {ref: ref}; })
            }
        }
    };

    // Override to gather recipe type specific settings
    $scope.getCreationSettings = function () {
        return {};
    }

    // Creates the recipe object and sends it to the backend
    $scope.doCreateRecipe = function() {
        var createOutput = $scope.io.newOutputTypeRadio == 'create' || $scope.io.newOutputTypeRadio == 'new-se';
        var outputName = $scope.io.newOutputTypeRadio == 'create' ? $scope.newOutputDataset.name : ($scope.io.newOutputTypeRadio == 'new-se' ? $scope.newOutputSE.name : $scope.io.existingOutputDataset);
        var inputs = $scope.recipe && $scope.recipe.inputs ? $scope.recipe.inputs : ($scope.inputFolderOnly ? makeMainRole([$scope.io.inputFolder.smartName]) : makeMainRole([$scope.io.inputDataset]));
        var recipe = {
            type: $scope.recipeType,
            projectKey: $stateParams.projectKey,
            name: "compute_" + outputName, //TODO @recipes remove,

            inputs: inputs,
            outputs: makeMainRole([outputName]),
        };

        const settings = $scope.getCreationSettings();
        if ($scope.zone) {
            settings.zone = $scope.zone;
        }
        settings.createOutputDataset = $scope.io.newOutputTypeRadio == 'create';
        settings.createOutputStreamingEndpoint = $scope.io.newOutputTypeRadio == 'new-se';
        settings.outputDatasetSettings = $scope.getDatasetCreationSettings();
        settings.outputStreamingEndpointSettings = $scope.getStreamingEndpointCreationSettings();
        if ($scope.recipeAdditionalParams) {
            settings.originRecipes = $scope.recipeAdditionalParams.originRecipes;
            settings.originDataset = $scope.recipeAdditionalParams.originDataset;
        }
        return DataikuAPI.flow.recipes.generic.create(recipe, settings);
    };

    var createRecipeAndDoStuff = $scope.createRecipe;
    // Called from UI, force means that no check-name-safety call is done
    $scope.createRecipe = function(force) {
        if (['select', 'new-odb', 'new-se'].indexOf($scope.io.newOutputTypeRadio) >= 0 || force) {
            createRecipeAndDoStuff();
        } else {
            DataikuAPI.datasets.checkNameSafety($stateParams.projectKey, $scope.newOutputDataset.name, $scope.getDatasetCreationSettings())
                .success(function(data) {
                    $scope.uiState.backendWarnings = data.messages;
                    if (!data.messages || !data.messages.length) {
                        createRecipeAndDoStuff();
                    }
                })
                .error(function(){
                    Logger.error("Check name failed.", arguments);
                    createRecipeAndDoStuff(); // don't block the creation
                });
        }
    };

    $scope.showOutputPane = function() {
        return !!$scope.io.inputDataset || !!$scope.io.inputFolder;
    };

    $scope.subFormIsValid = function() { return true; }; // overridable by sub-controllers for additional checks
    $scope.formIsValid = function() {
        if (!$scope.subFormIsValid()) return false;
        if (!$scope.inputFolderOnly && !($scope.io.inputDataset && $scope.activeSchema && $scope.activeSchema.columns && $scope.activeSchema.columns.length)) return false;
        if ($scope.io.newOutputTypeRadio == 'create') {
            return $scope.newOutputDataset && $scope.newOutputDataset.name && $scope.newOutputDataset.connectionOption && $scope.isDatasetNameUnique($scope.newOutputDataset.name);
        } else if ($scope.io.newOutputTypeRadio == 'new-se') {
            return $scope.newOutputSE && $scope.newOutputSE.name && $scope.newOutputSE.connectionOption && $scope.isStreamingEndpointNameUnique($scope.newOutputSE.name);
        } else if ($scope.io.newOutputTypeRadio == 'select') {
            return $scope.io.existingOutputDataset && $scope.io.existingOutputDataset.length;
        } else {
            return false;
        }
    };

    let updateInputDatasetSchema = function() {
        if ($scope.availableInputDatasets == null) return;
        if (!$scope.io.inputDataset) return;
        let resolvedSmartId = SmartId.resolve($scope.io.inputDataset, contextProjectKey);
        // get the object to first assert that we need to grab the schema
        let availableInput = $scope.availableInputDatasets.filter(o => o.name == resolvedSmartId.id && o.projectKey == resolvedSmartId.projectKey)[0];
        if (availableInput == null || availableInput.type == 'DATASET') {
            DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                $scope.activeSchema = data.schema;
            }).error(setErrorInScope.bind($scope));
        } else if (availableInput.type == 'STREAMING_ENDPOINT') {
            DataikuAPI.streamingEndpoints.get(resolvedSmartId.projectKey, resolvedSmartId.id).success(function(data){
                $scope.activeSchema = data.schema;
            }).error(setErrorInScope.bind($scope));
        } else {
            // other objects don't have a schema
            $scope.activeSchema = {columns:[]};
        }
    };

    var inputsIndex = {};
    DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, $scope.recipeType, $scope.datasetsOnly).then(function(data){
        $scope.availableInputDatasets = data[0];
        if ($scope.filterUsableInputsOn) {
            $scope.availableInputDatasets.forEach(function(c) {
                let usability = c.usableAsInput[$scope.filterUsableInputsOn] || {};
                c.usable = usability.usable;
                c.usableReason = usability.reason;
            });
        } else if ($scope.inputDatasetsOnly) {
            $scope.availableInputDatasets = data[0].filter(function(computable){
                return computable.usableAsInput['main'] && computable.usableAsInput['main'].usable;
            });
        }
        $scope.availableOutputDatasets = data[1].filter(function(computable){
            return computable.usableAsOutput['main'] && computable.usableAsOutput['main'].usable && !computable.alreadyUsedAsOutputOf;
        });
        $scope.availableInputDatasets.forEach(function(it) {
            inputsIndex[it.id] = it;
        });
        updateInputDatasetSchema(); // if the inputDataset arrived before the availableInputDatasets
    });

    let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.preselectInputDataset(preselectedInputDataset);
    });

    $scope.preselectInputDataset = (preselectedInputDataset) => {
        $scope.io.inputDataset = preselectedInputDataset;
        $scope.preselectedInputDataset = preselectedInputDataset;
    };

    $scope.$watch("io.inputDataset", function(nv) {
        updateManagedDatasetOptions($scope.recipeType, $scope.io.inputDataset, true);
        if (!nv) return;

        if ($scope.preselectedInputDataset && $scope.io.inputDataset != $scope.preselectedInputDataset){
            $scope.zone = null;
        }
        $scope.autosetName();

        updateInputDatasetSchema();
    });
    $scope.$watch("io.inputDataset2", Fn.doIfNv(function() {
        let resolvedSmartId = SmartId.resolve($scope.io.inputDataset2, contextProjectKey);
        DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
            $scope.activeSchema2 = data.schema;
        }).error(setErrorInScope.bind($scope));
    }));

    $scope.$on("preselectInputFolder", function(scope, preselectedInputFolder) {
        $scope.io.inputFolder = preselectedInputFolder;
        $scope.preselectedInputFolder = preselectedInputFolder;
    });
    $scope.$watch("io.inputFolder", Fn.doIfNv(function() {
        if ($scope.preselectedInputFolder && $scope.io.inputFolder.smartName != $scope.preselectedInputFolder.smartName){
            $scope.zone = null;
        }
        $scope.autosetName();

        updateManagedDatasetOptions($scope.recipeType, $scope.io.inputFolder.smartName, true);
    }));

});
})();

;
(function() {
'use strict';

const app = angular.module('dataiku.recipes');


/*
    updateStatus : a function, will be called when the status is updated
    canChangeEngine : a function, the modal can only be opened if this returns true
    recipeParams : the recipe parameters in which the engineType will be selected
    recipeStatus : the status object, containing the available engines
*/
app.directive('engineSelectorButton', function(Assert, DataikuAPI, Dialogs, $stateParams, CreateModalFromTemplate){
	return {
		scope: {
            recipeType : '=',
			recipeStatus : '=',
			recipeParams : '=',
			updateStatus : '=',
			canChangeEngine : '=',
            hideStatus : '=',
            iconSize: '<'
		},
		templateUrl: '/templates/recipes/fragments/recipe-engine-selection-button.html',
        link: function($scope, element, attrs) {
            Assert.inScope($scope, 'recipeParams');

            var modalDisplayed = false;
        	$scope.showEngineSelectionModal = function() {
                if (!modalDisplayed) {
                    modalDisplayed = true;
                    var newScope = $scope.$new();
                    CreateModalFromTemplate("/templates/recipes/fragments/recipe-engines-modal.html", newScope, null, function (modalScope) {
                        modalScope.nUnselectableEngines = modalScope.recipeStatus.engines.filter(e => !e.isSelectable).length;
                        modalScope.options = {};
                        modalScope.resetEngineType = function () {
                            delete $scope.recipeParams.engineType;
                            $scope.updateStatus();
                            modalScope.dismiss();
                        };
                        modalScope.selectEngine = function (engineType) {
                            $scope.recipeParams.engineType = engineType;
                            $scope.updateStatus();
                            modalScope.dismiss();
                        };
                        modalScope.$on("$destroy", _ => modalDisplayed = false);
                    });
                }
	        };
	    }
	}
});

})();
;
(function() {
    'use strict';
	var app = angular.module('dataiku.recipes');

	app.factory("ComputableSchemaRecipeSave", function(DataikuAPI, CreateModalFromTemplate, $q, $stateParams, $rootScope, ActivityIndicator, Logger, Dialogs, RecipesUtils) {
		var setFlags = function(datasets, stricter) {
            $.each(datasets, function(idx, val) {
                val.dropAndRecreate = val.incompatibilities.length > 0;
                val.synchronizeMetastore = val.incompatibilities.length > 0 && val.isHDFS;
            });
		};

		var getUpdatePromises = function(computables){
			var promises = [];
            $.each(computables, function(idx, val) {
                let extraOptions = {};
                if (val.type == 'STREAMING_ENDPOINT') {
                    extraOptions.ksqlParams = val.ksqlParams;
                }
                promises.push(DataikuAPI.flow.recipes.saveOutputSchema($stateParams.projectKey,
                    val.type, val.type == "DATASET" ? val.datasetName : val.id, val.newSchema,
                    val.dropAndRecreate, val.synchronizeMetastore, extraOptions));
            });
            return promises;
        }
        var getRecipePromises = function(data){
            var promises = [];
            if (data.updatedRecipe && data.updatedPayload) {
                promises.push(DataikuAPI.flow.recipes.save($stateParams.projectKey,
                    data.updatedRecipe, data.updatedPayload,
                    "Accept recipe update suggestion"));
            }
            return promises;
        }

        var displayPromisesError = function(errRet) {
            var scope = this;
			let errDetails;
            if (errRet.data) {
                errDetails = getErrorDetails(errRet.data, errRet.status, errRet.headers, errRet.statusText);
            } else {
                errDetails = getErrorDetails(errRet[0].data, errRet[0].status, errRet[0].headers, errRet[0].statusText);
            }
            Dialogs.displaySerializedError(scope, errDetails)
        }

		return {
            decorateChangedDatasets: setFlags,
            getUpdatePromises: getUpdatePromises,
            getRecipePromises: getRecipePromises,

			// The scope must be a recipe scope and contain the "doSave" function
			handleSave : function($scope, recipeSerialized, serializedData, deferred) {
	            var doSave = function(){
	                $scope.baseSave(recipeSerialized, serializedData).then(function(){
	                    deferred.resolve("Save done");
	                }, function(error) {
	                	Logger.error("Could not save recipe");
	                	deferred.reject("Could not save recipe");
	                })
	            }

	            DataikuAPI.flow.recipes.getComputableSaveImpact($stateParams.projectKey,
	             recipeSerialized, serializedData).success(function(data) {
	             	const allPreviousSchemasWereEmpty = data.computables.every(x => x.previousSchemaWasEmpty)

	             	if (data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty) {
						Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                        setFlags(data.computables, true);
                        $q.all(getUpdatePromises(data.computables)).then(function() {
                            doSave();
                        }).catch(displayPromisesError.bind($scope));
	             	} else if (data.totalIncompatibilities > 0) {
                        $scope.schemaChanges = data;

                        let outputContainsAppendMode = false;
                        $.each($scope.recipe.outputs, function(x, outputRole){
                           outputContainsAppendMode = outputContainsAppendMode || outputRole.items.some(output => output.appendMode);
                        });
                        const partitioned = RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
                        if ($scope.currentSaveIsForAnImmediateRun && !outputContainsAppendMode && !partitioned) {
                            Logger.info("Schema incompatibilities detected, but we are about to run the recipe anyway, so propagate without asking");

                            setFlags($scope.schemaChanges.computables, false);
                            const promises = getUpdatePromises($scope.schemaChanges.computables);
                            $q.all(promises).then(doSave).catch(setErrorInScope.bind($scope));

                        } else if (!outputContainsAppendMode && !partitioned && $rootScope.appConfig.autoAcceptSchemaChangeAtEndOfFlow
                            && data.computables.every(x => x.isLastInFlow && ! x.isPartitioned)) {
                            Logger.info("Schema incompatibilities detected, recipe at end of flow, auto-propagating schema", data);

                            setFlags($scope.schemaChanges.computables, false);
                            const promises = getUpdatePromises($scope.schemaChanges.computables);
                            $q.all(promises).then(doSave).catch(setErrorInScope.bind($scope));

                        } else {
                            Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);

                            let closedWithButton = false;
                            CreateModalFromTemplate("/templates/recipes/fragments/recipe-incompatible-schema-multi.html", $scope, null,
                                function(newScope) {
                                    setFlags($scope.schemaChanges.computables, false);
                                    newScope.cancelSave = function(){
                                        closedWithButton = true;
                                        newScope.dismiss();
                                        Logger.info("Save cancelled");
                                        deferred.reject("Save cancelled");
                                    };
                                    newScope.updateSchemaFromSuggestion = function() {
                                        closedWithButton = true;
                                        var promises = getUpdatePromises($scope.schemaChanges.computables);
                                        $q.all(promises).then(function() {
                                            newScope.dismiss();
                                            doSave();
                                        }).catch(function(data){
                                            setErrorInScope.bind($scope)(data.data, data.status, data.headers)
                                        });
                                    };
                                    newScope.ignoreSchemaChangeSuggestion = function() {
                                        closedWithButton = true;
                                        newScope.dismiss();
                                        doSave();
                                    };
                                }
                                ).then(function(){}, function(){if (!closedWithButton) {deferred.reject("Modal closed impolitely");}});
                            }
	                } else {
	                    Logger.info("No incompatible change, saving");
	                    doSave();
	                }
	            }).error(function(data, status, header){
                    Logger.error("Failed to compute recipe save impact");
                    // Failed to compute impact, don't block recipe save but ask user
                    var closedWithButton = false;
                    CreateModalFromTemplate("/templates/recipes/fragments/compute-save-impact-failed.html", $scope, null,
                            function(newScope) {
                                setErrorInScope.bind(newScope)(data, status, header);

                                newScope.cancelSave = function(){
                                    closedWithButton = true;
                                    newScope.dismiss();
                                    Logger.info("Save cancelled");
                                    deferred.reject("Save cancelled");
                                };
                                newScope.saveAnyway = function() {
                                    closedWithButton = true;
                                    newScope.dismiss();
                                    doSave();
                                };
                            }
                        ).then(function(){}, function(){if (!closedWithButton) {deferred.reject("Modal closed impolitely");}});
	            });
			},

			// Specialized version for the Shaker that needs to call a different API
			handleSaveShaker : function($scope, recipeSerialized, shaker, recipeOutputSchema, deferred) {
                const doSave = function(){
                    $scope.baseSave(recipeSerialized, JSON.stringify(shaker)).then(function() {
                        $scope.origShaker = angular.copy(shaker);
                        $scope.schemaDirtiness.dirty = false;
                        deferred.resolve("Save done");
                    },  function(error) {
                        Logger.error("Could not save recipe");
                        deferred.reject("Could not save recipe");
	                })
                }

                $scope.waitAllRefreshesDone().then(function() {
                    DataikuAPI.flow.recipes.getShakerSaveImpact($stateParams.projectKey,
                        $scope.recipe, shaker, $scope.recipeOutputSchema).success(function (data) {

                            var allPreviousSchemasWereEmpty = data.computables.every(x => x.previousSchemaWasEmpty);

                            if (data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty) {
                                Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                                setFlags(data.computables, true);
                                $q.all(getUpdatePromises(data.computables)).then(function () {
                                    doSave();
                                }).catch(displayPromisesError.bind($scope));
                            } else if (data.totalIncompatibilities > 0) {
                                $scope.schemaChanges = data;

                                let outputContainsAppendMode = false;
                                $.each($scope.recipe.outputs, function(x, outputRole){
                                   outputContainsAppendMode = outputContainsAppendMode || outputRole.items.some(output => output.appendMode);
                                });
                                const partitioned = RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);

                                if ($scope.currentSaveIsForAnImmediateRun && !outputContainsAppendMode && !partitioned) {
                                    Logger.info("Schema incompatibilities detected, but we are about to run the recipe anyway, so propagate without asking");

                                    setFlags($scope.schemaChanges.computables, false);
                                    const promises = getUpdatePromises($scope.schemaChanges.computables);
                                    $q.all(promises).then(doSave).catch(setErrorInScope.bind($scope));

                                } else if (!outputContainsAppendMode && !partitioned && $rootScope.appConfig.autoAcceptSchemaChangeAtEndOfFlow
                                    && data.computables.every(x => x.isLastInFlow && ! x.isPartitioned)) {
                                    Logger.info("Schema incompatibilities detected, recipe at end of flow, auto-propagating schema", data);

                                    setFlags($scope.schemaChanges.computables, false);
                                    const promises = getUpdatePromises($scope.schemaChanges.computables);
                                    $q.all(promises)
                                        .then(doSave)
                                        .catch(setErrorInScope.bind($scope));
                                } else {
                                    Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);
                                    CreateModalFromTemplate("/templates/recipes/fragments/recipe-incompatible-schema-multi.html", $scope, null,
                                        function (newScope) {
                                            setFlags($scope.schemaChanges.computables, false);
                                            newScope.cancelSave = function () {
                                                newScope.dismiss();
                                                Logger.info("Save cancelled");
                                                deferred.reject("Save cancelled");
                                            }
                                            newScope.updateSchemaFromSuggestion = function () {
                                                var promises = getUpdatePromises($scope.schemaChanges.computables);
                                                $q.all(promises).then(function () {
                                                    newScope.dismiss();
                                                    doSave();
                                                }).catch(function (data) {
                                                    setErrorInScope.bind($scope)(data.data, data.status, data.headers)
                                                });
                                            }
                                            newScope.ignoreSchemaChangeSuggestion = function () {
                                                newScope.dismiss();
                                                doSave();
                                            }
                                        }
                                    );
                                }
                            } else {
                                Logger.info("No incompatible change, saving");
                                doSave();
                            }
                        }).error(function (data, status, header) {
                            setErrorInScope.bind($scope)(data, status, header);
                            deferred.reject("failed to execute getComputableSaveImpact");
                        });
                });
            },

            /* handleSchemaUpdate for recipes which have just been reconnected after a delete and reconnect operation
            */
            handleSchemaUpdateForDeleteAndReconnect : function(parentScope, recipeProjectKey, recipeName) {
                this.handleSchemaUpdateFromAnywhere(parentScope, recipeProjectKey, recipeName, true);
            },

            /**
             * Check for schema mismatch with recipe and it's output datasets/computables and if so show a modal to offer the user a choice to resolve the issue.
             * @param {Object} parentScope scope this was spawned from
             * @param {string} recipeProjectKey project id
             * @param {string} recipeName name/id of recipe
             * @param {boolean} [partOfReconnect] - normally omit this - we set this to true for a delete and reconnect operation in handleSchemaUpdateForDeleteAndReconnect()
             */
            handleSchemaUpdateFromAnywhere : function(parentScope, recipeProjectKey, recipeName, partOfReconnect) {
                var serviceScope = parentScope.$new();
                DataikuAPI.flow.recipes.getSchemaUpdateResult(recipeProjectKey, recipeName).then(function(resp) {
                    const data = resp.data;
                    var allPreviousSchemasWereEmpty = data.computables.every(x => x.previousSchemaWasEmpty)

                    if (data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty) {
                        Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                        setFlags(data.computables, true);
                        $q.all(getUpdatePromises(data.computables)).then(function() {
                            // Nothing to do
                        }).catch(displayPromisesError.bind(parentScope));
                    }  else if (data.totalIncompatibilities > 0) {
                        Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);
                        serviceScope.schemaChanges = data;
                        serviceScope.partOfReconnect = Boolean(partOfReconnect);
                        serviceScope.recipeName = recipeName;


                        CreateModalFromTemplate("/templates/recipes/incompatible-schema-external-modal.html", serviceScope, null,

                            function(newScope) {
                                setFlags(serviceScope.schemaChanges.computables, true);

                                newScope.updateSchemaFromSuggestion = function() {
                                    var promises = getUpdatePromises(serviceScope.schemaChanges.computables);
                                    $q.all(promises).then(function() {
                                        newScope.dismiss();
                                    }).catch(function(data){
                                        setErrorInScope.bind(newScope)(data.data, data.status, data.headers)
                                    });
                                }
                                newScope.ignoreSchemaChangeSuggestion = function() {
                                    newScope.dismiss();
                                }
                            }
                        );
                    } else {
                        if (!partOfReconnect) {
                            ActivityIndicator.success("Schema is already up-to-date");
                        }
                    }
                }).catch(function(data, status, header){
                    //For reconnects, if there is any error here we don't show anything for recipes we can't check the schema for
                    // The user hasn't explicitly chosen to validate the schema so it
                    // would be confusing to show this, and there's nothing they can do anyway
                    if (!partOfReconnect) {
                        CreateModalFromTemplate("/templates/recipes/propagate-schema-changes-failed-modal.html", serviceScope, null,
                        function(newScope) {
                            setErrorInScope.bind(newScope)(data, status, header);
                        });
                    }
                });
            },

            handleSchemaUpdateWithPrecomputedUnattended : function(parentScope, data) {
                var deferred = $q.defer();

                if (data && (data.totalIncompatibilities > 0 || data.recipeChanges.length > 0)) {
                    Logger.info("Schema incompatibilities, unattended mode, updating and saving");
                    setFlags(data.computables, true);
                    $q.all(getUpdatePromises(data.computables).concat(getRecipePromises(data))).then(function() {
                        deferred.resolve({changed:true});
                    }).catch(displayPromisesError.bind(parentScope));
                } else {
                    deferred.resolve({changed:false});
                }
                return deferred.promise;
            },

			handleSchemaUpdateWithPrecomputed : function(parentScope, data) {
				var deferred = $q.defer();
				var serviceScope = parentScope.$new();
                var allPreviousSchemasWereEmpty = data && data.computables.every(x => x.previousSchemaWasEmpty)

	            if (data && data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty && data.recipeChanges.length == 0) {
                    Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                    setFlags(data.computables, true);
                    $q.all(getUpdatePromises(data.computables).concat(getRecipePromises(data))).then(function() {
                        deferred.resolve({changed:true});
                    }).catch(displayPromisesError.bind(parentScope));
                } else if (data && (data.totalIncompatibilities > 0 || data.recipeChanges.length > 0)) {
                    Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);
                    serviceScope.schemaChanges = data;

                    CreateModalFromTemplate("/templates/recipes/incompatible-schema-external-modal.html", serviceScope, null,
                        function(newScope) {
                            setFlags(serviceScope.schemaChanges.computables, true);

                            newScope.updateSchemaFromSuggestion = function() {
                                var promises = getUpdatePromises(serviceScope.schemaChanges.computables).concat(getRecipePromises(serviceScope.schemaChanges));
                                $q.all(promises).then(function() {
                                    newScope.dismiss();
                                    deferred.resolve({changed: true});
                                }).catch(function(data){
                                    setErrorInScope.bind(newScope)(data.data, data.status, data.headers)
                                    deferred.reject("Change failed");
                                });
                            }
                            newScope.ignoreSchemaChangeSuggestion = function() {
                                newScope.dismiss();
                                deferred.resolve({changed:false});
                            }
                        }
                    );
                } else {
                	deferred.resolve({changed:false});
                }
                return deferred.promise;
            },
        }
    });

    app.directive('codeRecipeSchemaList', function(DataikuAPI, Dialogs, $stateParams, CreateModalFromTemplate) {
        return {
            link : function($scope, element, attrs) {
                $scope.beginEditSchema = function(datasetSmartName) {
                    const computable = $scope.computablesMap[datasetSmartName];
                    if (!computable) {
                        throw new Error("Dataset not in computablesMap, try reloading the page");
                    }
                    const dataset = computable.dataset;
                    DataikuAPI.datasets.get(dataset.projectKey, dataset.name, $stateParams.projectKey)
                        .success(function(data){
                            CreateModalFromTemplate("/templates/recipes/code-edit-schema.html", $scope,
                            null, function(newScope) {
                                    newScope.dataset = data;
                                    // do not showGenerateMetadata from the schema edition modal (found in join recipe)
                                    newScope.showGenerateMetadata = false;
                                }).then(function(schema) {
                                    dataset.schema = schema;
                                });
                        }).error(setErrorInScope.bind($scope));
                }
            }
        }
    });

	app.directive("schemaEditorBase", function(DatasetUtils, $timeout, CreateModalFromTemplate, ContextualMenu, ExportUtils, ColumnTypeConstants, ActivityIndicator, DataikuAPI, ActiveProjectKey, translate) {
		return {
			scope : true,
			link : function($scope, element, attrs) {
                $scope.columnTypes = ColumnTypeConstants.types;
                $scope.columnTypesWithDefault = [
                    {name:'',label:'All types'},
				    ...ColumnTypeConstants.types
                ];
                $scope.translate = translate;
                $scope.meaningsWithDefault = Object.fromEntries([
                    ['','All meanings'],
                    ...Object.entries($scope.appConfig.meanings.labelsMap),
                ]);
				$scope.menusState = {meaning:false};
				$scope.startEditName = function(column, $event) {
				    $event.stopPropagation();
					$scope.dataset.schema.columns.forEach(function(x){
						x.$editingName = false;
						x.$editingComment = false;
					});

					var grandpa = $($event.target.parentNode.parentNode.parentNode);
					$timeout(function() { grandpa.find("input").focus(); });
				}
				$scope.handleInputClick = function($event) {
				    $event.stopPropagation();
				}
				$scope.blur = function(event) {
					$timeout(function() { event.currentTarget.blur(); });
				}
				$scope.setSchemaUserModifiedIfDirty = function() {
					if ($scope.datasetIsDirty && $scope.datasetIsDirty()) {
						$scope.setSchemaUserModified()
					}
				}

				function arrayMove(arr, from, to) {
					arr.splice(to, 0, arr.splice(from, 1)[0]);
				}

				$scope.moveColumnUp = function(column){
					var index = $scope.dataset.schema.columns.indexOf(column);
					if (index > 0) {
						arrayMove($scope.dataset.schema.columns, index, index - 1);
						$scope.setSchemaUserModified();
					}
				}
				$scope.moveColumnDown = function(column){
					var index = $scope.dataset.schema.columns.indexOf(column);
					if (index >= 0 && index < $scope.dataset.schema.columns.length - 1) {
						arrayMove($scope.dataset.schema.columns, index, index + 1);
						$scope.setSchemaUserModified();
					}
				}

				$scope.startEditComment = function(column, $event) {
					$scope.dataset.schema.columns.forEach(function(x){
						x.$editingName = false;
						x.$editingComment = false;
					});
					column.$editingComment = true;
					$timeout(function(){
	        			$($event.target).find("input").focus()
	        		}, 50);
				}
				$scope.addNew = function() {
					if ($scope.dataset.schema == null) {
						$scope.dataset.schema = { "columns" : []};
					}
					if ($scope.dataset.schema.columns == null) {
						$scope.dataset.schema.columns = [];
					}
					$scope.setSchemaUserModified();
					$scope.dataset.schema.columns.push({$editingName : true, name: '', type: 'string', comment: '', maxLength: 1000});
                    $scope.clearFilters();
                    $timeout(function(){
	                    $scope.$broadcast('scrollToLine', -1);
	                });
				}
				$scope.selection.orderQuery = "$idx";

				/** meanings **/
				$scope.openMeaningMenu = function($event, column) {
					$scope.meaningMenu.openAtXY($event.pageX, $event.pageY);
					$scope.meaningColumn = column;
				};

                $scope.openMassActionMenu = function() {
                    // rests the state that tracks on which column the meaning should be applied
                    // (without this, changing the meaning on a single column would 'lock' the mass aciton on this column instead of on the selection)
                    $scope.meaningColumn = null;
                }

				// use delete instead of '... = null' because when it comes as json, the property is just not there when null
				$scope.setColumnMeaning = function(meaningId) {
					if ($scope.meaningColumn == null) {
	                	$scope.selection.selectedObjects.forEach(function(c) {
							if (meaningId == null) {
								delete c.meaning;
							} else {
			                	c.meaning = meaningId;
							}
	                	});
					} else {
						if (meaningId == null) {
							delete $scope.meaningColumn.meaning;
						} else {
		                	$scope.meaningColumn.meaning = meaningId;
						}
					}
                    $(".code-edit-schema-box").css("display", "block");
                    if ($scope.setSchemaUserModified) $scope.setSchemaUserModified();
                };

                $scope.editColumnUDM = function(){
                    CreateModalFromTemplate("/templates/meanings/column-edit-udm.html", $scope, null, function(newScope){
                    	var columnName;
                        if ($scope.meaningColumn == null) {
                            columnName = $scope.selection.selectedObjects[0].name;
                        } else {
                            columnName = $scope.meaningColumn.name;
                        }
                        newScope.initModal(columnName, $scope.setColumnMeaning);
                    })
                }

                $scope.generateDatasetDescription = function() {
                    DataikuAPI.datasets.get($scope.dataset.projectKey, $scope.dataset.name, ActiveProjectKey.get()).noSpinner()
                    .success(function(data) {
                        angular.extend($scope.dataset, data);
                        CreateModalFromTemplate(
                            "/static/dataiku/ai-dataset-descriptions/generate-documentation-modal/generate-documentation-modal.html",
                            $scope,
                            "AIDatasetDescriptionsModalController",
                            function(scope) {
                                scope.init($scope.dataset, $scope.canWriteProject)
                            }
                        )
                    })
                    .error(setErrorInScope.bind($scope));
                }

				$scope.exportSchema = function() {
                    if (!$scope.dataset.schema || !$scope.dataset.schema.columns) {
                        ActivityIndicator.error('Empty schema.');
                        return;
                    }
					ExportUtils.exportUIData($scope, {
						name: "Schema of " + $scope.dataset.name,
						columns: [
							{ name: "name", type: "string" },
							{ name: "type", type: "string" },
							{ name: "meaning_id", type: "string" },
							{ name: "description", type: "string" },
							{ name: "max_length", type: "int" }
						],
						data: $scope.dataset.schema.columns.map(function(c){
							return [c.name, c.type, c.meaning, c.comment, c.maxLength >= 0 ? c.maxLength : "" ]
						})
					}, "Export schema");
				}

                $scope.meaningMenu = new ContextualMenu({
                    template: "/templates/shaker/edit-meaning-contextual-menu.html",
                    cssClass : "column-header-meanings-menu pull-right",
                    scope: $scope,
                    contextual: false,
                    onOpen: function() {
                    },
                    onClose: function() {
                    }
                });

                var reNumberColumns = function() {
                	var columns = $scope.$eval(attrs.ngModel);
                	if (!columns) return;
                	// columns.forEach(function(c, i) {c.$idx = i;});
                };

                /** column type **/
                $scope.setColumnsType = function(columnType) {
            		$scope.selection.selectedObjects.forEach(function(c) {
            			c.type = columnType;
            		});
                };
                /** renaming **/
                $scope.doRenameColumns = function(renamings) {
                	renamings.forEach(function(renaming) {
                		$scope.selection.selectedObjects.forEach(function(c) {
                			if (c.name == renaming.from) {
                				c.name = renaming.to;
                			}
                		});
                	});
                };

                $scope.renameColumns = function() {
                	CreateModalFromTemplate('/templates/shaker/modals/shaker-rename-columns.html', $scope, 'MassRenameColumnsController', function(newScope) {
                   		newScope.setColumns($scope.selection.selectedObjects.map(function(c) {return c.name;}));
                   		newScope.doRenameColumns = function(renamings) {
                   			$scope.doRenameColumns(renamings);
                   		};
                    });
                };
                /** data for the right pane **/
                var commonBaseTypeChanged = function() {
                	if (!$scope.selection || !$scope.selection.multiple) return;
                	if (!$scope.multipleSelectionInfo || !$scope.multipleSelectionInfo.commonBaseType) return;
                	var columns = $scope.selection.selectedObjects;
                	columns.forEach(function(column) {column.type = $scope.multipleSelectionInfo.commonBaseType.type;});
                };
                var commonTypeChanged = function() {
                	if (!$scope.selection || !$scope.selection.multiple) return;
                	if (!$scope.multipleSelectionInfo || !$scope.multipleSelectionInfo.commonType) return;
                	var columns = $scope.selection.selectedObjects;
                	var setFullType = function(column, commonType) {
            			column.type = commonType.type;
            			column.maxLength = commonType.maxLength;
            			column.objectFields = commonType.objectFields ? angular.copy(commonType.objectFields) : null;
            			column.arrayContent = commonType.arrayContent ? angular.copy(commonType.arrayContent) : null;
            			column.mapKeys = commonType.mapKeys ? angular.copy(commonType.mapKeys) : null;
            			column.mapValues = commonType.mapValues ? angular.copy(commonType.mapValues) : null;
                	};
                	columns.forEach(function(column) {setFullType(column, $scope.multipleSelectionInfo.commonType);});
                };
                var updateInfoForMultipleTab = function() {
                	if ($scope.commonTypeChangedDeregister) {
                		$scope.commonTypeChangedDeregister();
                		$scope.commonTypeChangedDeregister = null;
                	}
                	if ($scope.commonBaseTypeChangedDeregister) {
                		$scope.commonBaseTypeChangedDeregister();
                		$scope.commonBaseTypeChangedDeregister = null;
                	}
                	if (!$scope.selection || !$scope.selection.multiple) return;
                	var getFullType = function(column) {
                		return {
                			type:column.type ? column.type : null,
                			maxLength:column.maxLength ? column.maxLength : null,
                			objectFields:column.objectFields ? column.objectFields : null,
                			arrayContent:column.arrayContent ? column.arrayContent : null,
                			mapKeys:column.mapKeys ? column.mapKeys : null,
                			mapValues:column.mapValues ? column.mapValues : null
                		};
                	};
                	var columns = $scope.selection.selectedObjects;
                	var names = columns.map(function(column) {return column.name;});
                	var meanings = columns.map(function(column) {return column.meaning;});
                	var types = columns.map(function(column) {return column.type;});
                	var fullTypes = columns.map(function(column) {return getFullType(column);});
                	var firstFullType = fullTypes[0];
                	var sameTypes = fullTypes.map(function(t) {return angular.equals(t, firstFullType);}).reduce(function(a,b) {return a && b;});
                	var commonType = sameTypes ? firstFullType : null;
                	$scope.multipleSelectionInfo = {sameTypes: sameTypes, commonType : commonType, commonBaseType : null};

                    $scope.commonBaseTypeChangedDeregister = $scope.$watch('multipleSelectionInfo.commonBaseType', commonBaseTypeChanged);
                    $scope.commonTypeChangedDeregister = $scope.$watch('multipleSelectionInfo.commonType', commonTypeChanged, true);
                };
                $scope.$watch('selection.multiple', updateInfoForMultipleTab);
                $scope.$watch('selection.selectedObjects', updateInfoForMultipleTab, true);


                $scope.$watch(attrs.ngModel, reNumberColumns); // for when the schema is inferred again (but nothing changes)
                $scope.$watch(attrs.ngModel, reNumberColumns, true);
			}
		}
	});

	app.directive('codeRecipeSchemaEditing', function(DataikuAPI, DatasetUtils, DatasetsService,
		Dialogs, $stateParams, $timeout, Logger){
    	return {
	        link : function($scope, element, attrs) {
	        	$scope.overwriteSchema = function(newSchema) {
	        		$scope.dataset.schema = angular.copy(newSchema);
	        		$scope.schemaJustModified = false;
	        		$scope.consistency = null;
	        	};

	        	$scope.saveSchema = function() {
	        		DataikuAPI.datasets.save($scope.dataset.projectKey, $scope.dataset).success(function(data){
                        $scope.resolveModal($scope.dataset.schema);
	        		}).error(setErrorInScope.bind($scope));
	        	};

	        	$scope.discardConsistencyError= function(){
	        		$scope.consistency = null;
	        	};

	        	$scope.setSchemaUserModified = function() {
            		$scope.schemaJustModified = true;
              		$scope.dataset.schema.userModified = true;
              		$scope.consistency = null;
          		};

                $scope.addColumn = function(){
                    if ($scope.dataset.schema == null) {
                        $scope.dataset.schema = { "columns" : []};
                    }
                    $scope.setSchemaUserModified();
                    $scope.dataset.schema.columns.push({$editingName : true, name: '', type: 'string', comment: '', maxLength: 1000});
                };

	        	$scope.checkConsistency = function () {
			        Logger.info('Checking consistency');
			        $scope.schemaJustModified = false;

			        DataikuAPI.datasets.testSchemaConsistency($scope.dataset).success(function (data) {
            			Logger.info("Got consistency result", data);
            			$scope.consistency = data;
            			$scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
            		});
	        	};
	        }
        }
    });
})();

;
(function() {
'use strict';

const app = angular.module('dataiku.recipes');


app.directive("recipeIoInputs", function(RecipesUtils, RecipeComputablesService, $stateParams) {
    return {
        scope: true,
        templateUrl: function(element, attrs) {
            return '/templates/recipes/io/' + attrs.location + '-inputs.html';
        },
        link: {
            // pre, because otherwise link is post by default, and executed after its children's link
            pre : function($scope, element, attrs) {
                // propagate
                $scope.roles = $scope.$eval(attrs.roles);
                $scope.uiState = $scope.uiState || {};
                $scope.uiState.warningMessages = $scope.uiState.warningMessages || {};
                $scope.location = attrs.location;
                $scope.longRoleList = $scope.roles.filter(role => $scope.recipe.inputs[role.name] && $scope.recipe.inputs[role.name].items.length > 0).length > 2; // only consider roles with actual inputs
                $scope.editInputs = [];
                
                if ($scope.roles) {
                    $scope.roles.forEach(function(role) {role.editing = false;});
                }
                $scope.setErrorInTopScope = function(scope) {
                    return setErrorInScope.bind($scope.$parent);
                };
                $scope.isInputRoleAvailableHook = function(role) {
                    if (!role.availabilityDependsOnPayload) { 
                        return true ; // by default the roles are available
                    }
                    if (!$scope.isInputRoleAvailableForPayload) {  
                        throw new Error(`No rule defined for availability of input role "${role.name}"`); // the callback has not been defined, but availability depends on it
                    } else {
                        return $scope.isInputRoleAvailableForPayload(role);
                    }
                };

                function getEditableRoles() {
                    if (!$scope.roles) {
                        return [];
                    }

                    const isRoleEditableFn = attrs.location === "modal" ? (role) => role.editableInModal : (role) => role.editableInEditor;
                    return $scope.roles.filter(isRoleEditableFn);
                }

                // Determines if we show explanation for disabled recipe creation button
                $scope.shouldDisplayDisabledCreateExplanation = function () {
                    let roles = getEditableRoles();

                    if (
                        roles.length === 0 || // No input roles
                        (roles.some((role) => role.editing) || $scope.editInputs.length > 0) // Roles are being edited
                    ) {
                        return false;
                    }

                    // Has a specific condition defined in shouldDisplayInputExplanation
                    if ("shouldDisplayInputExplanation" in $scope) { return $scope.shouldDisplayInputExplanation(); }

                    // Has some unset required input
                    return roles.some(
                        (role) =>
                            $scope.isInputRoleAvailableHook(role) && role.required &&
                            (!$scope.recipe.inputs[role.name] || !$scope.recipe.inputs[role.name].items.length)
                    );
                };

                $scope.generateDisabledCreateExplanation = function () {
                    // Has a specific explanation message defined in generateInputExplanation
                    if ("generateInputExplanation" in $scope) { return $scope.generateInputExplanation(); }

                    let roles = getEditableRoles();

                    if (roles.length === 0) {
                        return "";
                    }

                    if (roles.length === 1) {
                        return "This recipe requires at least one input.";
                    } else {
                        const requiredRoles = roles
                            .filter((role) => $scope.isInputRoleAvailableHook(role) && role.required)
                            .map((role, inputRoleIdx) => {
                                if (role.name === "main" && !role.label) {
                                    return "main input";
                                } else if (!role.name && !role.label) {
                                    return "input " + (inputRoleIdx + 1); // No label at all => print role index
                                } else {
                                    return '"' + (role.label || role.name) + '"'; // Otherwise print displayed label
                                }
                            });

                        return "This recipe requires at least one input in: "
                            + requiredRoles.slice(0, -1).join(', ')
                            + (requiredRoles.length === 2 ? ' and ' : ', and ')
                            + requiredRoles.slice(-1) + ".";
                    }
                }
            }
        }
    }
});


app.directive("recipeIoOutputs", function(RecipesUtils, RecipeComputablesService, $stateParams){
    return {
        scope: true,
        templateUrl: function(element, attrs) {
            return '/templates/recipes/io/' + attrs.location + '-outputs.html';
        },
        link: {
            // pre, because otherwise link is post by default, and executed after its children's link
            pre : function($scope, element, attrs) {
                // propagate
                $scope.roles = $scope.$eval(attrs.roles);
                $scope.location = attrs.location;
                $scope.longRoleList = $scope.roles.filter(role => $scope.recipe.outputs[role.name] && $scope.recipe.outputs[role.name].items.length > 0).length > 2; // only consider roles with actual outputs
                $scope.editOutputs = [];
                if ($scope.roles) {
                    $scope.roles.forEach(function(role) {role.editing = false;});
                }
                $scope.canAppend = function(computable) {
                    if (computable.noAppend) return false; // no ambiguity here
                    if (['cpython', 'ksql', 'csync', 'streaming_spark_scala'].indexOf($scope.recipe.type) >= 0) return false; // can't overwrite with continuous activities
                    if (['upsert', 'extract_content'].indexOf($scope.recipe.type) >= 0) return false; // append or overwrite is meaningless for those recipes
                    const selectedEngine = $scope.recipeStatus ? $scope.recipeStatus.selectedEngine : null;
                    if (computable.onlyAppendOnStreamOrSQLEngine && selectedEngine) {
                        //`canEngineAppend` is defined by the backend, but for now,
                        //only `SPARK` and `HADOOP_MR` (deprecated) can't append.
                        return !!selectedEngine.canEngineAppend; //return a true boolean
                    } else {
                        return true; // maybe.
                    }
                };
                $scope.setErrorInTopScope = function(scope) {
                    return setErrorInScope.bind($scope.$parent);
                };

                $scope.isOutputRoleAvailableHook = function(role) {
                    if (!role.availabilityDependsOnPayload) { return true ; } // by default the role is available.
                    if (!$scope.isOutputRoleAvailableForPayload) {  
                        throw new Error(`No rule defined for availability of output role "${role.name}"`); // the callback has not been defined, but availability depends on it
                    } else {
                        return $scope.isOutputRoleAvailableForPayload(role);
                    }
                };

                // Determines if we show explanation for disabled recipe creation button
                $scope.shouldDisplayDisabledCreateExplanation = function () {
                    if (
                        !$scope.roles // No output roles
                        || ($scope.roles.some((role) => role.editing) || $scope.editOutputs.length > 0) // Roles are being edited
                    ) { return false; }

                    // Has a specific condition defined in shouldDisplayOutputExplanation
                    if ("shouldDisplayOutputExplanation" in $scope) { return $scope.shouldDisplayOutputExplanation(); }

                    // Has some unset required output
                    return $scope.roles.some(
                        (role) =>
                            $scope.isOutputRoleAvailableHook(role) && role.required &&
                            (!$scope.recipe.outputs[role.name] || !$scope.recipe.outputs[role.name].items.length)
                    );
                };

                $scope.generateDisabledCreateExplanation = function () {
                    // Has a specific explanation message defined in generateOutputExplanation
                    if ("generateOutputExplanation" in $scope) { return $scope.generateOutputExplanation(); }

                    if ($scope.roles.length === 0) { return ""; }

                    if ($scope.roles.length === 1) {
                        return "This recipe requires at least one output.";
                    } else {
                        const requiredRoles = $scope.roles
                            .filter((role) => $scope.isOutputRoleAvailableHook(role) && role.required)
                            .map((role, outputRoleIdx) => {
                                if (role.name === "main" && !role.label) {
                                    return "main output";
                                } else if (!role.name && !role.label) {
                                    return "output " + (outputRoleIdx + 1); // No label at all => print role index
                                } else {
                                    return '"' + (role.label || role.name) + '"'; // Otherwise print displayed label
                                }
                            });

                        return "This recipe requires at least one output in: "
                            + requiredRoles.slice(0, -1).join(', ')
                            + (requiredRoles.length === 2 ? ' and ' : ', and ')
                            + requiredRoles.slice(-1) + ".";
                    }
                }

                $scope.shouldDisplayConfigurationAlert = function() {
                    return $scope.recipeAdditionalParams && $scope.recipeAdditionalParams.originRecipes && $scope.recipeAdditionalParams.originRecipes.length &&
                        !($scope.roles && $scope.roles.some((role) => role.editing));
                }
            }
        }
    }
});


// this is more or less a custom ng-repeat, because ngRepeat AND another directive on the same element makes some things
// impossible, like using interpolated attributes for the other directive
app.directive("recipeIoInputList", function(RecipesUtils, RecipeComputablesService, $stateParams, $compile) {
    return {
        scope: true,
        restrict: 'E',
        link : function($scope, element, attrs) {

            var roleElements = [];
            $scope.roles.forEach(function(role, index){
                roleElements.push('<div recipe-io-input-display-list role-index="' + index + '" location="' + $scope.location + '"/>');
                roleElements.push('<div recipe-io-input-add-list role-index="' + index + '"location="' + $scope.location + '"/>');
            });

            element.replaceWith($compile(roleElements.join('\n'))($scope));
        }
    }
});


app.directive("recipeIoOutputList", function(RecipesUtils, RecipeComputablesService, $stateParams, $compile, $rootScope) {
    return {
        scope: true,
        restrict: 'E',
        link : function($scope, element, attrs) {

            var roleElements = [];
            $scope.roles.forEach(function(role, index) {
                roleElements.push('<div recipe-io-output-display-list role-index="' + index + '" location="' + $scope.location + '"/>');
                roleElements.push('<div recipe-io-output-add-list role-index="' + index + '"location="' + $scope.location + '"/>');
            });
            element.replaceWith($compile(roleElements.join('\n'))($scope));
        }
    }
});


app.directive("recipeIoInputDisplayList", function(RecipesUtils){
    return {
        scope: true,
        replace: true,
        templateUrl: function(element, attrs) {
            return '/templates/recipes/io/' + attrs.location + '-input-display-list.html';
        },
        link : function($scope, element, attrs){
            $scope.role = $scope.roles[parseInt(attrs.roleIndex)];
            $scope.hasAnyPartitioning = function(){
                return RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
            }

            $scope.canDeleteInput = function(role) {
                return role.editableInEditor && (role.arity != 'UNARY' || !role.required);
            }

            $scope.cannotEditInputForInsert = function(dataset){
                return $scope.recipeAdditionalParams && $scope.recipeAdditionalParams.originDataset
                    && $scope.recipeAdditionalParams.originDataset === dataset.ref
                    && $scope.recipeAdditionalParams.originRecipes && $scope.recipeAdditionalParams.originRecipes.length;
            }

            $scope.getDatasetInputTooltip = function(dataset){
                if ($scope.cannotEditInputForInsert(dataset)) {
                    return "You cannot change this input dataset when inserting a recipe into the flow";
                }
                return null;
            }
        }
    }
});

app.directive("recipeIoInputAddList", function(Assert, RecipesUtils, RecipeDescService, RecipeComputablesService,
                                               $stateParams, DKUtils, DataikuAPI, $q) {
    return {
        scope: true,
        replace: true,
        templateUrl: function(element, attrs) {
            return '/templates/recipes/io/' + attrs.location + '-input-add-list.html';
        },
        link : function($scope, element, attrs) {
            $scope.role = $scope.roles[parseInt(attrs.roleIndex)];
            $scope.hasAnyPartitioning = function(){
                return RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
            }

            $scope.addInput = {
                adding : false,
                role:null,
                filter : null
            }
            var beginEdition = function() {
                $scope.addInput.adding = true;
                $scope.role.editing = true;
                $scope.editInputs.push($scope.addInput);
            };

            var endEdition = function() {
                $scope.addInput.adding = false;
                $scope.role.editing = false;
                var idx = $scope.editInputs.indexOf($scope.addInput);
                if (idx >= 0) $scope.editInputs.splice(idx, 1);
            };

            var setUsable = function(list) {
                // put usable datasets at the beginning
                var roleName = $scope.role.name;
                list.sort(function(a,b) {
                    var aIsUsable = a.usableAsInput[roleName] && a.usableAsInput[roleName].usable;
                    var bIsUsable = b.usableAsInput[roleName] && b.usableAsInput[roleName].usable;
                    if (aIsUsable && !bIsUsable)
                        return -1;
                    if (!aIsUsable && bIsUsable)
                        return 1;
                    return (a.label || '').localeCompare((b.label || ''));
                });
                $scope.addInput.usable = list;
            };
            $scope.$watch("addInput.filter", function(nv){
                if ($scope.recipe && $scope.computablesMap) {
                    setUsable(RecipeComputablesService.buildPossibleInputList(
                        $scope.recipe, $scope.computablesMap, $scope.addInput.role, $scope.addInput.filter));
                }
            });

            $scope.itemsWatchHooked = false;
            var hookItemsWatch = function() {
                $scope.itemsArray = $scope.recipe.inputs[$scope.role.name].items;
                $scope.$watchCollection("itemsArray", function(nv){
                    if ($scope.roleChanged) {
                        $scope.roleChanged($scope.role.name);
                    }
                });
                $scope.itemsWatchHooked = true;
            };
            if ( $scope.recipe.inputs[$scope.role.name] != null ) {
                // items can be null in a recipe newly created
                hookItemsWatch();
            }

            $scope.enterAddInput = function(role) {
                beginEdition();
                $scope.addInput.role = role;
                setUsable(RecipeComputablesService.buildPossibleInputList(
                            $scope.recipe, $scope.computablesMap, role, $scope.addInput.filter));
            }
            $scope.cancelAddInput = function(){
                endEdition();
            }

            $scope.acceptAddInput = function(computable){
                Assert.trueish($scope.addInput.adding, 'not adding inputs');
                var promise = $q.when(null);

                if (attrs.location == "modal") {
                    if ($scope.recipe.inputs[$scope.addInput.role] == null || $scope.role.arity == 'UNARY') {
                        $scope.recipe.inputs[$scope.addInput.role] = { items : []}
                    }
                    $scope.recipe.inputs[$scope.addInput.role].items.push({
                        ref : computable.smartName,
                        deps : []
                    });
                } else {
                    var currentRecipeAndPayload = {
                        recipe : angular.copy($scope.recipe),
                        payload: angular.copy($scope.script.data)
                    }
                    var newRecipeAndPayload = {
                        recipe : angular.copy($scope.recipe),
                        payload: angular.copy($scope.script.data)
                    }
                    if (newRecipeAndPayload.recipe.inputs[$scope.addInput.role] == null || $scope.role.arity == 'UNARY') {
                        newRecipeAndPayload.recipe.inputs[$scope.addInput.role] = { items : []}
                    }
                    newRecipeAndPayload.recipe.inputs[$scope.addInput.role].items.push({
                        ref : computable.smartName,
                        deps : []
                    });
                    promise = DataikuAPI.flow.recipes.getIOChangeResult($stateParams.projectKey, currentRecipeAndPayload, newRecipeAndPayload)
                        .error($scope.setErrorInTopScope($scope))
                        .then(function(resp) {
                            var roleDesc = RecipeDescService.getInputRoleDesc($scope.recipe.type, $scope.addInput.role);
                            $scope.recipe.inputs = resp.data.updated.recipe.inputs;
                            $scope.recipe.outputs = resp.data.updated.recipe.outputs;

                            const messages = resp.data.messages.messages || [];
                            $scope.uiState.warningMessages[$scope.role.name] = messages.filter(x => x.severity == 'WARNING');

                            if (roleDesc.saveAndReloadAfterEditInEditor) {
                                $scope.baseSave($scope.hooks.getRecipeSerialized(), $scope.script ? $scope.script.data : null).then(function(){
                                    DKUtils.reloadState();
                                });
                            }

                            return;
                        });
                }

                promise.then(function(){
                    endEdition();
                    if (!$scope.itemsWatchHooked) {
                       hookItemsWatch();
                    }
                });
            }
        }
    }
});


app.controller("_RecipeOutputNewManagedBehavior", function($scope, WT1, Logger, DataikuAPI, $stateParams, RecipeComputablesService, $rootScope, SqlConnectionNamespaceService) {
    $scope.newOutputDataset = {};
    $scope.newOutputODB = {};
    $scope.newOutputMES = {};
    $scope.newOutputSE = {};
    $scope.io = $scope.io || {};
    $scope.io.newOutputTypeRadio = "create";
    $scope.forms = {};
    $scope.uiState = $scope.uiState || {};
    delete $scope.uiState.backendWarnings;

    $scope.getManagedDatasetOptions = function(role){
        return DataikuAPI.datasets.getManagedDatasetOptions($scope.recipe, role)
            .then(function(data){return data.data})
            .catch($scope.setErrorInTopScope($scope));
    };

    $scope.setupManagedDatasetOptions = function(data, forceUpdate){
        $scope.managedDatasetOptions = data;
        if (data.connections.length && (!$scope.newOutputDataset.connectionOption || forceUpdate) ){
            $scope.newOutputDataset.connectionOption = data.connections[0];
        }

         // in a competition with the other "Not partitioned" in this file to set the option - maybe fix 
        $scope.partitioningOptions = [
            {"id" : "NP", "label" : $scope.translate("FLOW.CREATE_RECIPE.PARTITIONING.NOT_PARTITIONED", "Not partitioned")}
        ];

        $scope.partitioningOptions = $scope.partitioningOptions
                                        .concat(data.inputPartitionings)
                                        .concat(data.projectPartitionings)

        if (data.inputPartitionings.length) {
            $scope.newOutputDataset.partitioningOption = data.inputPartitionings[0].id
        } else {
            $scope.newOutputDataset.partitioningOption = "NP";
        }
    };

    $scope.getManagedFolderOptions = function(role){
        return DataikuAPI.datasets.getManagedFolderOptions($scope.recipe, role)
            .then(function(data){return data.data})
            .catch($scope.setErrorInTopScope($scope));
    };

    $scope.getModelEvaluationStoreOptions = function(role){
        return DataikuAPI.datasets.getModelEvaluationStoreOptions($scope.recipe, role)
            .then(function(data){return data.data})
            .catch($scope.setErrorInTopScope($scope));
    };

    $scope.getStreamingEndpointOptions = function(role){
        return DataikuAPI.datasets.getStreamingEndpointOptions($scope.recipe, role)
            .then(function(data){return data.data;})
            .catch($scope.setErrorInTopScope($scope));
    };

    var updateFolderConnection = function() {
        if ($scope.newOutputODB.$connection == null) return;
        $scope.newOutputODB.connectionOption = $scope.newOutputODB.$connection.connectionName;
        $scope.newOutputODB.typeOption = $scope.newOutputODB.$connection.fsProviderTypes[0];
    };

    var updateStreamingEndpointConnection = function() {
        if ($scope.newOutputSE.$connection == null) return;
        $scope.newOutputSE.connectionOption = $scope.newOutputSE.$connection.connectionName;
        if ($scope.newOutputSE.$connection.formats && $scope.newOutputSE.$connection.formats.length) {
            $scope.newOutputSE.formatOptionId = $scope.newOutputSE.$connection.formats[0].id;
        }
    };

    $scope.setupManagedFolderOptions = function(data, forceUpdate){
        $scope.managedFolderOptions = data;
        $scope.managedFolderOptions.connections = $scope.managedFolderOptions.connections.filter(function(c) {return c.fsProviderTypes != null;});
        if (data.connections.length && (!$scope.newOutputODB.connectionOption || forceUpdate) ){
            $scope.newOutputODB.$connection = data.connections[0];
            updateFolderConnection();
        }

        $scope.partitioningOptions = [
            {"id" : "NP", "label" : $scope.translate("FLOW.CREATE_RECIPE.PARTITIONING.NOT_PARTITIONED", "Not partitioned")}
        ];

        $scope.partitioningOptions = $scope.partitioningOptions
                                        .concat(data.inputPartitionings)
                                        .concat(data.projectPartitionings)

        if (data.inputPartitionings.length) {
            $scope.newOutputODB.partitioningOption = data.inputPartitionings[0].id
        } else {
            $scope.newOutputODB.partitioningOption = "NP";
        }
    };

    $scope.setupModelEvaluationStoreOptions = function(data, forceUpdate){
        $scope.modelEvaluationStoreOptions = data;

        $scope.partitioningOptions = [
            {"id" : "NP", "label" : $scope.translate("FLOW.CREATE_RECIPE.PARTITIONING.NOT_PARTITIONED", "Not partitioned")}
        ];

        $scope.partitioningOptions = $scope.partitioningOptions
                                        .concat(data.inputPartitionings)
                                        .concat(data.projectPartitionings)

        //if (data.inputPartitionings.length) {
        //    $scope.newOutputMES.partitioningOption = data.inputPartitionings[0].id
        //} else {
            $scope.newOutputMES.partitioningOption = "NP";
        //}
    };

    $scope.setupStreamingEndpointOptions = function(data, forceUpdate){
        $scope.streamingEndpointOptions = data;
        if (data.connections.length && (!$scope.newOutputSE.connectionOption || forceUpdate) ){
            $scope.newOutputSE.$connection = data.connections[0];
            updateStreamingEndpointConnection();
        }
    };

    $scope.$watch("newOutputDataset.connectionOption", function(nv, ov){
        if (nv && nv.formats && nv.formats.length) {
            $scope.newOutputDataset.formatOptionId = nv.formats[0].id;
        }
        if (nv && nv.fsProviderTypes && nv.fsProviderTypes.length > 1) {
            $scope.newOutputDataset.typeOption = nv.fsProviderTypes[0];
        }

        if ($scope.newOutputDataset.connectionOption) {
            SqlConnectionNamespaceService.setTooltips($scope, $scope.newOutputDataset.connectionOption.connectionType);
        }
        SqlConnectionNamespaceService.resetState($scope, $scope.newOutputDataset);
    }, true);

    function doCreateAndUseNewOutputDataset(projectKey, datasetName, settings) {
        Logger.info("Create and use ", $scope);
        DataikuAPI.datasets.newManagedDataset(projectKey, datasetName, settings).success(function(dataset) {
                RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
                    $scope.setComputablesMap(map);

                    $scope.acceptEdit($scope.computablesMap[dataset.name]);

                    // Clear form
                    $scope.newOutputDataset.name = '';
                    $scope.forms.newOutputDatasetForm.$setPristine(true);

                    $rootScope.$emit('datasetsListChangedFromModal'); // communicate with the flow editor (note: don't broadcast)
                });
                WT1.event("create-dataset", {
                    connectionType: ($scope.newOutputDataset && $scope.newOutputDataset.connectionOption) ? $scope.newOutputDataset.connectionOption.connectionType : "unknown",
                    partitioningFrom: $scope.newOutputDataset ? $scope.newOutputDataset.partitioningOption : "unknown",
                    recipeType: $scope.recipe ? $scope.recipe.type : "unknown"
                });

        }).error($scope.setErrorInTopScope($scope));
    }

    $scope.getDatasetCreationSettings = function() {
        let datasetCreationSetting = {
            connectionId : ($scope.newOutputDataset.connectionOption || {}).id,
            specificSettings : {
                overrideSQLCatalog: $scope.newOutputDataset.overrideSQLCatalog,
                overrideSQLSchema: $scope.newOutputDataset.overrideSQLSchema,
                formatOptionId : $scope.newOutputDataset.formatOptionId,
            },
            partitioningOptionId : $scope.newOutputDataset.partitioningOption,
            inlineDataset : $scope.inlineDataset,
            zone : $scope.zone
        };
        if ($scope.newOutputDataset &&
            $scope.newOutputDataset.connectionOption &&
            $scope.newOutputDataset.connectionOption.fsProviderTypes &&
            $scope.newOutputDataset.connectionOption.fsProviderTypes.length > 1) {
            datasetCreationSetting['typeOptionId'] = $scope.newOutputDataset.typeOption;
        }
        return datasetCreationSetting;
    }

    $scope.getFolderCreationSettings = function() {
        return {
            partitioningOptionId : $scope.newOutputODB.partitioningOption,
            connectionId : $scope.newOutputODB.connectionOption,
            typeOptionId : $scope.newOutputODB.typeOption,
            zone: $scope.zone
        };
    }

    $scope.getEvaluationStoreCreationSettings = function() {
        return {
            mesFlavor: $scope.mesFlavor,
            partitioningOptionId : $scope.newOutputMES.partitioningOption,
            zone: $scope.zone
        };
    }

    $scope.getStreamingEndpointCreationSettings = function() {
        return {
            connectionId : $scope.newOutputSE.connectionOption,
            formatOptionId : $scope.newOutputSE.formatOptionId,
            typeOptionId : $scope.newOutputSE.typeOption,
            zone: $scope.zone
        };
    }

    $scope.$watch("newOutputODB.$connection", updateFolderConnection);
    $scope.$watch("newOutputSE.$connection", updateStreamingEndpointConnection);


    $scope.createAndUseNewOutputDataset = function(force) {
        var projectKey = $stateParams.projectKey,
            datasetName = $scope.newOutputDataset.name,
            settings = $scope.getDatasetCreationSettings();

        if (force) {
            doCreateAndUseNewOutputDataset(projectKey, datasetName, settings);
        } else {
            DataikuAPI.datasets.checkNameSafety(projectKey, datasetName, settings).success(function(data) {
                $scope.uiState.backendWarnings = data.messages;
                if (!data.messages || !data.messages.length) {
                    doCreateAndUseNewOutputDataset(projectKey, datasetName, settings);
                }
            }).error($scope.setErrorInTopScope($scope));
        }
    };

    $scope.createAndUseManagedFolder = function() {
        Logger.info("Create and use managed folder", $scope);
        var settings = $scope.getFolderCreationSettings();
        DataikuAPI.datasets.newManagedFolder($stateParams.projectKey, $scope.newOutputODB.name, settings).success(function(odb) {

            RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
                $scope.setComputablesMap(map);

                $scope.acceptEdit($scope.computablesMap[odb.id]);

                // Clear form
                $scope.newOutputODB.name = '';
                $scope.forms.newOutputODBForm.$setPristine(true);

                $rootScope.$emit('datasetsListChangedFromModal'); // communicate with the flow editor (note: don't broadcast)
            });

        }).error($scope.setErrorInTopScope($scope));

        $scope.recipeWT1Event("recipe-create-managed-folder", {});
    };
    
    $scope.createAndUseModelEvaluationStore = function() {
        Logger.info("Create and use model evaluation store", $scope);
        var settings = $scope.getEvaluationStoreCreationSettings();
        DataikuAPI.datasets.newModelEvaluationStore($stateParams.projectKey, $scope.newOutputMES.name, settings).success(function(mes) {

            RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
                $scope.setComputablesMap(map);

                $scope.acceptEdit($scope.computablesMap[mes.id]);

                // Clear form
                $scope.newOutputMES.name = '';
                $scope.forms.newOutputMESForm.$setPristine(true);

                $rootScope.$emit('datasetsListChangedFromModal'); // communicate with the flow editor (note: don't broadcast)
            });

        }).error($scope.setErrorInTopScope($scope));

        $scope.recipeWT1Event("recipe-create-model-evaluation-store", {});
    };
    
    $scope.createAndUseStreamingEndpoint = function() {
        Logger.info("Create and use streaming endpoint", $scope);
        var settings = $scope.getStreamingEndpointCreationSettings();
        DataikuAPI.datasets.newStreamingEndpoint($stateParams.projectKey, $scope.newOutputSE.name, settings).success(function(se) {

            RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
                $scope.setComputablesMap(map);

                $scope.acceptEdit($scope.computablesMap[se.id]);

                // Clear form
                $scope.newOutputSE.name = '';
                $scope.forms.newOutputSEForm.$setPristine(true);

                $rootScope.$emit('datasetsListChangedFromModal'); // communicate with the flow editor (note: don't broadcast)
            });

        }).error($scope.setErrorInTopScope($scope));

        $scope.recipeWT1Event("recipe-create-streaming-endpoint", {});
    };

    $scope.fetchCatalogs = function(origin, connectionType) {
        SqlConnectionNamespaceService.listSqlCatalogs($scope.newOutputDataset.connectionOption.connectionName, $scope, origin, connectionType);
    };

    $scope.fetchSchemas = function(origin, connectionType) {
        const catalog = $scope.newOutputDataset.overrideSQLCatalog || 
            ($scope.newOutputDataset.connectionOption ? $scope.newOutputDataset.connectionOption.unoverridenSQLCatalog : '');
        SqlConnectionNamespaceService.listSqlSchemas($scope.newOutputDataset.connectionOption.connectionName, $scope, catalog, origin, connectionType);
    };
});


app.directive("recipeIoOutputDisplayList", function(RecipesUtils, RecipeComputablesService, $stateParams){
    return {
        scope: true,
        replace: true,
        templateUrl: function(element, attrs) {
            return '/templates/recipes/io/' + attrs.location + '-output-display-list.html';
        },
        link : function($scope, element, attrs){
            $scope.role = $scope.roles[parseInt(attrs.roleIndex)];
            $scope.hasAnyPartitioning = function(){
                return RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
            }

            $scope.canDeleteOutput = function(role, recipeType, recipeDesc) {
                return role.editableInEditor && (role.arity != 'UNARY' || !role.required);
            }
        }
    }
});


app.directive("recipeIoOutputAddList", function($controller, Assert, RecipesUtils, RecipeComputablesService, Logger,
                                                DataikuAPI, $state, $stateParams, $q) {
    return {
        scope: true,
        replace: true,
        // Not using isolate scope because we need to $eval
        templateUrl: function(element, attrs) {
            return '/templates/recipes/io/' + attrs.location + '-output-add-list.html';
        },
        link : function($scope, elemnt, attrs){
            $controller("_RecipeOutputNewManagedBehavior", {$scope:$scope});
            $scope.role = $scope.roles[parseInt(attrs.roleIndex)];

            $scope.hasAnyPartitioning = function(){
                return RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
            }

            $scope.editOutput = {
                adding : false,
                role:null
            }

            var beginEdition = function() {
                $scope.editOutput.adding = true;
                $scope.role.editing = true;
                $scope.editOutputs.push($scope.editOutput);

                var selectOption = function(role) {
                    if (role.acceptsDataset) {
                        return "create";
                    } else if (role.acceptsManagedFolder) {
                        return "new-odb";
                    } else if (RecipeComputablesService.acceptsEvaluationStore(role)) {
                        RecipeComputablesService.setMesFlavor($scope, role);
                        return "new-mes";
                    } else {
                        return "select";
                    }
                };
                if ( $scope.io.newOutputTypeRadio == null ) {
                    $scope.io.newOutputTypeRadio = selectOption($scope.role);
                } else if ( $scope.io.newOutputTypeRadio == "create" && !$scope.role.acceptsDataset ) {
                    $scope.io.newOutputTypeRadio = selectOption($scope.role);
                } else if ( $scope.io.newOutputTypeRadio == "new-odb" && !$scope.role.acceptsManagedFolder ) {
                    $scope.io.newOutputTypeRadio = selectOption($scope.role);
                } else if ($scope.io.newOutputTypeRadio == "new-mes" && RecipeComputablesService.acceptsEvaluationStore($scope.role)) {
                    RecipeComputablesService.setMesFlavor($scope, $scope.role);
                    $scope.io.newOutputTypeRadio = selectOption($scope.role);
                }
            };
            var endEdition = function() {
                $scope.editOutput.adding = false;
                $scope.role.editing = false;
                var idx = $scope.editOutputs.indexOf($scope.editOutput);
                if (idx >= 0) $scope.editOutputs.splice(idx, 1);
            };

            var setUsable = function(list){
                   // put usable datasets at the beginning
                   list.sort(function(a,b) {
                       if (a.usableAsOutput[$scope.role.name] && a.usableAsOutput[$scope.role.name].usable && !a.alreadyUsedAsOutputOf &&
                        (!b.usableAsOutput[$scope.role.name] || !b.usableAsOutput[$scope.role.name].usable || b.alreadyUsedAsOutputOf))
                           return -1;
                       if ((!a.usableAsOutput[$scope.role.name] || !a.usableAsOutput[$scope.role.name].usable || a.alreadyUsedAsOutputOf) &&
                        b.usableAsOutput[$scope.role.name] && b.usableAsOutput[$scope.role.name].usable && !b.alreadyUsedAsOutputOf)
                           return 1;
                    return (a.label || '').localeCompare((b.label || ''));
                   });
                   $scope.editOutput.usable = list;
            };

            $scope.itemsWatchHooked = false;
            var hookItemsWatch = function() {
                $scope.itemsArray = $scope.recipe.outputs[$scope.role.name].items;
                $scope.$watchCollection("itemsArray", function(nv){
                    if ($scope.roleChanged) {
                        $scope.roleChanged($scope.role.name);
                    }
                });
                $scope.itemsWatchHooked = true;
            };
            if ( $scope.recipe.outputs[$scope.role.name] != null ) {
                // items can be null in a recipe newly created
                hookItemsWatch();
            }

            $scope.$watch("editOutput.filter", function(){
                setUsable(RecipeComputablesService.buildPossibleOutputList(
                        $scope.recipe, $scope.computablesMap, $scope.editOutput.role, $scope.editOutput.filter));
            });

            $scope.cleanInputs = function() {
                // Quick and dirty fix because of 110420 & 169778
                // In case of split/pca recipe, the $scope.recipe can have an input with an empty name
                let cleaningNeeded = $scope.recipe && ($scope.recipe.type==="split" || $scope.recipe.type==="eda_pca") &&
                                     $scope.role.name && $scope.recipe.inputs[$scope.role.name] && $scope.recipe.inputs[$scope.role.name].items.length===1 &&
                                     'ref' in $scope.recipe.inputs[$scope.role.name].items[0] && $scope.recipe.inputs[$scope.role.name].items[0].ref==="";
                if (cleaningNeeded) {
                    $scope.recipe.inputs[$scope.role.name].items = [];
                }
            }

            $scope.enterAddOutput = function(role) {
                $scope.cleanInputs();
                $scope.uiState.backendWarnings = null;
                beginEdition();
                $scope.editOutput.role = role;
                setUsable(RecipeComputablesService.buildPossibleOutputList(
                            $scope.recipe, $scope.computablesMap, role, $scope.editOutput.filter));

                // the select element seems to be caching something, and after hiding and showing the
                // create new dataset form a few times (2 times on firefox, 3 on chrome) the option
                // shown to be selected is incorrect ('nothing selected' but the option is not null).
                // it's probably a race condition somewhere, so we solve it the hard way: make the
                // select reinitialize its sate each  time
                $scope.newOutputDataset.connectionOption = null;
                $scope.getManagedDatasetOptions(role).then(function(data){
                    $scope.setupManagedDatasetOptions(data);
                })
                $scope.getManagedFolderOptions(role).then(function(data){
                    $scope.setupManagedFolderOptions(data);
                })
                $scope.getModelEvaluationStoreOptions(role).then(function(data){
                    $scope.setupModelEvaluationStoreOptions(data);
                })
                $scope.getStreamingEndpointOptions(role).then(function(data){
                    $scope.setupStreamingEndpointOptions(data);
                })
            };

            $scope.cancelAddOutput = function(){
                endEdition();
            };

            $scope.acceptEdit = function(computable){
                Assert.trueish($scope.editOutput.adding, 'not adding inputs');
                var promise = $q.when(null);

                if (attrs.location == "modal") {
                    if ($scope.role.arity == "UNARY") {
                        $scope.recipe.outputs[$scope.role.name] = { items : []};
                    }
                    RecipesUtils.addOutput($scope.recipe, $scope.role.name, computable.smartName);
                } else {
                    var currentRecipeAndPayload = {
                        recipe : angular.copy($scope.recipe),
                        payload: angular.copy($scope.script.data)
                    }
                    var newRecipeAndPayload = {
                        recipe : angular.copy($scope.recipe),
                        payload: angular.copy($scope.script.data)
                    }
                    if ($scope.role.arity == "UNARY") {
                        newRecipeAndPayload.recipe.outputs[$scope.role.name] = { items : []};
                    }
                    RecipesUtils.addOutput(newRecipeAndPayload.recipe, $scope.role.name, computable.smartName);
                    promise = DataikuAPI.flow.recipes.getIOChangeResult($stateParams.projectKey, currentRecipeAndPayload, newRecipeAndPayload)
                        .error($scope.setErrorInTopScope($scope))
                        .then(function(resp){
                            $scope.recipe.inputs = resp.data.updated.recipe.inputs;
                            $scope.recipe.outputs = resp.data.updated.recipe.outputs;
                            return;
                        });
                }

                promise.then(function(){
                    endEdition();
                    if (!$scope.itemsWatchHooked) {
                          hookItemsWatch();
                    }
                });
            };

            $scope.showOnlyNewDataset = function(){
                return $scope.recipeAdditionalParams && $scope.recipeAdditionalParams.originRecipes;
            }

            if ($scope.role.arity == 'UNARY' && attrs.location != 'modal') {
                /* Auto enter edit mode if none selected */
                if ($scope.isOutputRoleAvailableHook($scope.role) && $scope.role.required && (!$scope.recipe.outputs[$scope.role.name] || $scope.recipe.outputs[$scope.role.name].items.length == 0)) {
                    beginEdition();
                }
            }
        }
    }
});

})();

;
(function() {
'use strict';

var app = angular.module('dataiku.recipes');


var PYTHON_SAMPLE_DEPENDENCY = 'def get_dependencies(target_partition_id):\n'
                            + '    return [target_partition_id]';

app.filter('retrievePartitioning', function(){
    return function(computable){
        if (!computable) { return null; }

        switch (computable.type) {
        case 'DATASET':            return computable.dataset.partitioning;
        case 'MANAGED_FOLDER':     return computable.box.partitioning;
        case 'SAVED_MODEL':        return computable.model.partitioning;
        }
        return null;
    };
});

app.directive("partitionedByInfo", function() {
    return {
        templateUrl : '/templates/recipes/io/partitioned-by-info.html',
        scope:true,
        link : function($scope, element, attrs) {
            $scope.lookupRef = $scope.$eval(attrs.ref);
        }
    }
});

app.directive("customPythonDependencyEditor", function(CreateModalFromTemplate,PartitionDeps,DataikuAPI, CodeMirrorSettingService) {
    return {
        restrict:'E',
        template : '<button class="btn" ng-click="openDialog()" translate="GLOBAL.MENU.EDIT">Edit</button>',
        scope : {
            pdepIndex : '=',
            recipe : '=',
            input : '='
        },
        link : function(isolatedScope, element, attrs) {
            isolatedScope.openDialog = function() {
                CreateModalFromTemplate("/templates/recipes/fragments/python-dep-editor.html", isolatedScope,null, function(scope) {

                    scope.localRecipe = angular.copy(isolatedScope.recipe);
                    scope.localInput = angular.copy(isolatedScope.input);
                    // replace input in local recipe, to be able to test
                    $.each(isolatedScope.recipe.inputs, function(roleName, inputRole) {
                        var inputIndex = inputRole.items.indexOf(isolatedScope.input);
                        if (inputIndex >= 0) {
                        	scope.localRecipe.inputs[roleName].items[inputIndex] = scope.localInput;
                        }
                    });
                    scope.localPdep = scope.localInput.deps[isolatedScope.pdepIndex];

                    if(!scope.localPdep.params) {
                        scope.localPdep.params = {};
                    }
                    if(!scope.localPdep.params.code) {
                        scope.localPdep.params.code = PYTHON_SAMPLE_DEPENDENCY;
                    }

                    scope.editorOptions = CodeMirrorSettingService.get('text/x-python');

                    scope.saveAndClose = function() {
                        isolatedScope.input.deps[isolatedScope.pdepIndex].params = angular.copy(scope.localPdep.params);
                        scope.dismiss();
                    };

                    scope.test = function() {
                        scope.testResult = undefined;
                        DataikuAPI.flow.recipes.generic.pdepTest(scope.localRecipe, isolatedScope.input.ref, PartitionDeps.prepareForSerialize(scope.localPdep)).success(function(data) {
                            scope.testResults = data;
                        }).error(setErrorInScope.bind(scope));
                    };
                });
            };
        }
    };
});

app.factory("PartitionDeps", function(Assert, DataikuAPI, Logger, RecipesUtils, $filter, translate) {
    function neverNeedsOutput(pdep) {
        return ["values", "all_available", "latest_available"].indexOf(pdep.func) >= 0;
    }
    function mayWorkWithoutOutput(pdep) {
        return ["time_range"].indexOf(pdep.func) >= 0;
    }

    var svc = {
        // Auto fills proper parameters for a single partition dependency
        // This should be called each time the pdep is changed
        autocomplete : function(pdep, outputDimensions, outputDimensionsWithNow) {
            Logger.info("Autocompleting pdep:"  + JSON.stringify(pdep));
            if (pdep.func == "time_range") {
                if (!pdep.params) pdep.params = {};

                if (!pdep.params.fromMode) pdep.params.fromMode = "RELATIVE_OFFSET";
                if (!pdep.params.fromGranularity) pdep.params.fromGranularity = "DAY";
                if (angular.isUndefined(pdep.params.fromOffset)) pdep.params.fromOffset = 0;
                if (!pdep.params.toGranularity) pdep.params.toGranularity = pdep.params.fromGranularity;
                if (angular.isUndefined(pdep.params.toOffset)) pdep.params.toOffset = 0;

                if (pdep.params.fromMode == "FIXED_DATE" && !pdep.params.fromDate) {
                    pdep.params.fromDate = "2024-01-01";
                }

                if (!pdep.$$output) {
                    if (outputDimensions.length) {
                        pdep.$$output = outputDimensions[0];
                    } else {
                        pdep.$$output = outputDimensionsWithNow[0];
                    }
                }
            } else if(pdep.func == 'custom_python') {
                if (!pdep.params) pdep.params = {};
                if (!pdep.params.code) pdep.params.code = PYTHON_SAMPLE_DEPENDENCY;
            }
            Logger.info("Autocompleted pdep:"  + JSON.stringify(pdep));
        },

        // Fixup pdep definitions. This should be called each time inputs or outputs
        // of the recipe are modified.
        // We don't do it with $watch because it actually makes handling corner cases more difficult
        // Returns :
        //  [outputDimensions, outputDimensiosnWithNow]
        fixup : function(recipe, computablesMap) {
            Logger.info("fixup pdep, recipe is ", recipe);
            Assert.trueish(recipe, 'no recipe');
            Assert.trueish(computablesMap, 'no computablesMap');
            RecipesUtils.getFlatInputsList(recipe).forEach(function(input){
                // console.info("CHECKING IF I HAVE " , input.ref, "in", computablesMap);
                Assert.trueish(computablesMap[input.ref], 'input not in computablesMap');
            });
            RecipesUtils.getFlatOutputsList(recipe).forEach(function(output){
                // console.info("CHECKING IF I HAVE " , output.ref, "in", computablesMap);
                Assert.trueish(computablesMap[output.ref], 'output not in computablesMap');
            });
            /* End sanity checks */

            /* Remove pdeps that were only here temporarily.
             * At the moment, it only means empty "values" deps.
             * If they were needed, we'll recreate them later on) */

            RecipesUtils.getFlatInputsList(recipe).forEach(function(input){
                Logger.info("Cleaning up input deps", input.deps);
                input.deps = input.deps.filter(function(dep) {
                    var isTemp =  dep.out == null && dep.func == "values" && !dep.values;
                    return !isTemp;
                });
                Logger.info("cleaned:" , input.deps);
            });

            // Each partition dep points to an (output, odim) couple or to nothing ...
            // So we keep up to date a list of (output, odim) couples
            var outputDimensions = [];
            RecipesUtils.getFlatOutputsList(recipe).forEach(function(output){
                const computable = computablesMap[output.ref];
                const partitioning = $filter('retrievePartitioning')(computable);

                if (partitioning == null) {
                    return;
                }

                partitioning.dimensions.forEach(function(dim) {
                    outputDimensions.push({
                        out: output.ref,
                        odim: dim.name,
                        label: dim.name + translate("RECIPE.IO_SELECT.PARTITIONING.OF", " of ") + output.ref
                    });
                });
            });

            Logger.info("Valid possible outputs", outputDimensions);

            // Very important: make a shallow copy of the array because the matching
            // of $$output to outputDimensions is not deep
            var outputDimensionsWithNow = outputDimensions.slice();
            var currentTimeDep = {
                "label" : translate("RECIPE.IO_SELECT.PARTITIONING.CURRENT_TIME", "Current time" )
            };
            outputDimensionsWithNow.push(currentTimeDep)

            // Assign in $$output the correct out/odim couple for existing valid dependencies
            RecipesUtils.getFlatInputsList(recipe).forEach(function(input){
                input.deps.forEach(function(pdep) {
                    if (neverNeedsOutput(pdep)) {
                        Logger.info("Pdep does not need an output", pdep);
                        return;
                    }
                	if (pdep.$$output != null) {
                		// try to find it in the 'new' outputDimensions, in case it's
                		// currently being edited
                		var matchingOd;
                		if (pdep.$$output.label == currentTimeDep.label) {
                		    matchingOd = [currentTimeDep];
                		} else {
                            matchingOd = outputDimensions.filter(function(nod) {return nod.out == pdep.$$output.out && nod.odim == pdep.$$output.odim;});
                		}
                		if ( matchingOd.length == 1) {
                			pdep.$$output = matchingOd[0];
                		} else {
                			pdep.$$output = null;
                		}
                	}
                	if (pdep.$$output == null) {
                        for (var i in outputDimensions) {
                            var od = outputDimensions[i];
                            Logger.info("Compare ", od, pdep);
                            if (od.out == pdep.out && od.odim == pdep.odim) {
                                pdep.$$output = od;
                                Logger.info("YES, matches");
                                break;
                            }
                        }
                	}
                    if (!pdep.$$output) {
                        if (mayWorkWithoutOutput(pdep)) {
                            Logger.info("Failed to find a matching output dimension for pdep ", pdep);
                            pdep.$$output = currentTimeDep;
                        } else {
                            Logger.error("Failed to find a matching output dimension for pdep ", pdep);
                        }
                        // This can happen when removing an output ...
                    }
                });
            });

            // Add entries for missing dependencies
            RecipesUtils.getFlatInputsList(recipe).forEach(function(input){
                const computable = computablesMap[input.ref];
                const partitioning = $filter('retrievePartitioning')(computable);

                if (partitioning) {
                    for (var dimIdx in partitioning.dimensions) {
                        var dim = partitioning.dimensions[dimIdx];
                        Logger.info("Searching for pdep setting ", dim.name, "in", angular.copy(input.deps));

                        if ($.grep(input.deps, dep => dep.idim == dim.name).length === 0) {
                            Logger.info("Will add new pdep ...");
                            var recipeFirstOut = RecipesUtils.getFlatOutputsList(recipe)[0];
                            var newPdep = {
                                out: recipeFirstOut ? recipeFirstOut.ref : null,
                                func : 'equals',
                                idim : dim.name
                            };
                            if (newPdep.out && partitioning.dimensions.length) {
                                var outputDimensions = partitioning.dimensions;
                                // try to match with same dimension on the output
                                newPdep.odim = outputDimensions[0].name;
                                outputDimensions.forEach(function(dim) {
                                    if (dim.name == newPdep.idim) {
                                        newPdep.odim = dim.name;
                                        return;
                                    }
                                });
                                // then fetch the $$output
                                for (var i in outputDimensions) {
                                    var od = outputDimensions[i];
                                    if (od.out == newPdep.out && od.odim == newPdep.odim) {
                                        newPdep.$$output = od;
                                        break;
                                    }
                                }
                            } else {
                                newPdep.func = 'values';
                            }

                            Logger.info("Creating missing pdep for " , input, dim.name, angular.copy(newPdep));
                            input.deps.push(newPdep);
                        }
                    }
                }
            });

            /* If we still have some incomplete dependencies, try to fix them.
             * This happens in the following case:
             *  - I0 partitioned by D0, O0 partitioned by D0, equals dep
             *  - Remove O0 as output
             *  - Add a new dep with the same partitioning. Since the pdep
             *    was already here, we didn't create it.
             *
             * The logic is:
             *  - If the pdep needs absolutely an output
             *  - And it does not have a $$output reference
             *  - then: if there is partitioned output dataset, we set the output to it
             *    else: we fallback to a VALUES
             */
            RecipesUtils.getFlatInputsList(recipe).forEach(function(input){
                input.deps.forEach(function(pdep) {
                    if (neverNeedsOutput(pdep)) {
                        Logger.info("Pdep does not need an output", pdep);
                        return;
                    }
                    if (pdep.$$output == null) {
                        Logger.info("Pdep has no valid output", pdep);

                        if (mayWorkWithoutOutput(pdep)) {
                            Logger.info("but it might work without ....");
                            return;
                        }

                        const recipeFirstOut = RecipesUtils.getFlatOutputsList(recipe)[0];
                        const partitioning = $filter('retrievePartitioning')(computablesMap[recipeFirstOut.ref]);

                        if (recipeFirstOut && partitioning.dimensions.length) {
                            pdep.out = recipeFirstOut.ref;
                            pdep.odim = partitioning.dimensions[0].name;
                            Logger.info("Assign as output", pdep);
                            // Assign the $$output
                            for (var i in outputDimensions) {
                                var od = outputDimensions[i];
                                if (od.out == pdep.out && od.odim == pdep.odim) {
                                    pdep.$$output = od;
                                    break;
                                }
                            }
                        } else {
                            pdep.func = 'values';
                        }
                    }
                });
            });

            Logger.info("After fixup, inputs", recipe.inputs, outputDimensions, outputDimensionsWithNow);
            return [outputDimensions, outputDimensionsWithNow];
        },

        /* Rewrite the pdep in serializable form */
        prepareForSerialize : function(pdep, notEdited) {
            var ret = angular.copy(pdep);
            if (ret.$$output) {
                if (ret.$$output.odim) {
                    ret.odim = ret.$$output.odim;
                } else {
                    delete ret.odim;
                }
                if (ret.$$output.out) {
                    ret.out = ret.$$output.out;
                } else {
                    delete ret.out;
                }
                //ret.$$output = null;
            } else if (neverNeedsOutput(ret) || mayWorkWithoutOutput(ret)) {
                /* Nothing for now */
            } else {
                if (!notEdited) { // under edition => should have the $$output
                    Logger.warn("Saving incomplete pdep", ret);
                }
            }
            if (ret.func != 'values') delete ret.values;
            if (ret.func == 'values' || ret.func == "latest_available" || ret.func == "all_available") {
                delete ret.odim; // Meaningless
            }
            // convert params to string
            if (ret.params) {
                angular.forEach(ret.params, function(value, key) {ret.params[key] = value == null ? null : value.toString();});
            }
            //console.info("Prepare for serialize", pdep, "gives", ret);
            return ret;
        },

        prepareRecipeForSerialize : function(recipe, notEdited) {
            RecipesUtils.getFlatInputsList(recipe).forEach(function(input){
                if (input.deps != null) {
                    input.deps = input.deps.map(function(p) {return svc.prepareForSerialize(p, notEdited);});
                }
            });
            return recipe;
        },

        test : function(recipe, pdepInputRef, pdep, errorScope) {
            var recipeSerialized = angular.copy(recipe);
            svc.prepareRecipeForSerialize(recipeSerialized);
            /* The result of the test is directly written in pdep.$$testResult */
            DataikuAPI.flow.recipes.generic.pdepTest(recipeSerialized,  pdepInputRef, svc.prepareForSerialize(pdep)).success(function(data) {
                pdep.$$testResult = data;
            }).error(setErrorInScope.bind(errorScope));
        },

        timeRangeFromModes : [
            ["RELATIVE_OFFSET", translate("RECIPE.IO_SELECT.PARTITIONING.OFFSET", "Offset to reference time")],
            ["FIXED_DATE", translate("RECIPE.IO_SELECT.PARTITIONING.FIXED_DATE", "Fixed date")]
        ],
        timeRangeGranularities : [
            ["YEAR", translate("RECIPE.IO_SELECT.PARTITIONING.YEAR", "Year(s)"), 4],
            ["MONTH", translate("RECIPE.IO_SELECT.PARTITIONING.MONTH", "Month(s)"), 3],
            ["DAY", translate("RECIPE.IO_SELECT.PARTITIONING.DAY", "Day(s)"), 2],
            ["HOUR", translate("RECIPE.IO_SELECT.PARTITIONING.HOUR", "Hour(s)"), 1]
        ],
        getGranularities : function(input, partition, computables) {
            const dim = computables[input].dataset.partitioning.dimensions.find((element) => element.name === partition.idim)
            const weight = this.timeRangeGranularities.find((element) => element[0] === dim.params.period)[2]
            /* ensure the current values are correct */
            const currentFromWeight = this.timeRangeGranularities.find((element) => element[0] === partition.params.fromGranularity)[2]
            const currentToWeight = this.timeRangeGranularities.find((element) => element[0] === partition.params.toGranularity)[2]
            if (currentFromWeight<weight) {
                partition.params.fromGranularity = dim.params.period;
            }
            if (currentToWeight<weight) {
                partition.params.toGranularity = dim.params.period;
            }
            return this.timeRangeGranularities.filter((element) => element[2] >= weight);
        },
        depFunctions : [
            ['equals', translate("RECIPE.IO_SELECT.PARTITIONING.EQUALS", "Equals")],
            ['time_range', translate("RECIPE.IO_SELECT.PARTITIONING.TIME_RANGE", "Time Range")],
            ['current_week', translate("RECIPE.IO_SELECT.PARTITIONING.CURRENT_WEEK", "Since beginning of week")],
            ['current_month', translate("RECIPE.IO_SELECT.PARTITIONING.CURRENT_MONTH", "Since beginning of month")],
            ['whole_month', translate("RECIPE.IO_SELECT.PARTITIONING.WHOLE_MONTH", "Whole month")],
            ['values', translate("RECIPE.IO_SELECT.PARTITIONING.VALUES", "Explicit values")],
            ['latest_available', translate("RECIPE.IO_SELECT.PARTITIONING.LATEST_AVAILABLE", 'Latest available')],
            ['all_available', translate("RECIPE.IO_SELECT.PARTITIONING.ALL_AVAILABLE", 'All available')],
            ['custom_python', translate("RECIPE.IO_SELECT.PARTITIONING.CUSTOM_PYTHON",'Python dependency function')],
            ['sliding_days', translate("RECIPE.IO_SELECT.PARTITIONING.SLIDING_DAYS", "Sliding days (deprecated, use Time Range)")],

        ],
    }
    return svc;
});

})();

;
(function(){
'use strict';

var app = angular.module('dataiku.recipes');


app.controller("VisualRecipeEditorController", function ($scope, $stateParams, $q, $controller, DataikuAPI, PartitionDeps, CreateModalFromTemplate,
               ComputableSchemaRecipeSave, Dialogs, DKUtils, DatasetUtils, Logger, translate) {
    $controller("_RecipeWithEngineBehavior", {$scope:$scope});
    var visualCtrl = this;

    $scope.hooks.preRunValidate = function() {
        var deferred = $q.defer();
        $scope.hooks.updateRecipeStatus().then(function(data) {
            if (data && data.invalid) {
                Logger.info("preRunValidate failed",data)
                Dialogs.confirm($scope, "Recipe contains errors", "The recipe contains errors. Are you sure you want to run it?").then(function() {
                    deferred.resolve({ok: true});
                }, function(){
                    deferred.reject("Validation failed");
                });
            } else {
                deferred.resolve({ok: true});
            }
        },
        function(data){
            Logger.error("Error when getting status", data);
            setErrorInScope.bind($scope);
            deferred.reject("Validation failed");
        });
        return deferred.promise;
    };

    var paramsSavedOnServer = undefined;
    visualCtrl.saveServerParams = function() {
        paramsSavedOnServer = angular.copy($scope.hooks.getPayloadData());
    }

    var superRecipeIsDirty = $scope.hooks.recipeIsDirty;
    $scope.hooks.recipeIsDirty = function() {
        var currentPayload = $scope.hooks.getPayloadData();
        if (currentPayload) {
            currentPayload = angular.fromJson(currentPayload);
        }
        var savedPayload = paramsSavedOnServer;
        if (savedPayload) {
            savedPayload = angular.fromJson(savedPayload);
        }
        return superRecipeIsDirty() || !angular.equals(currentPayload, savedPayload);
    };

    $scope.hooks.save = function() {
        var deferred = $q.defer();
        var recipeSerialized = angular.copy($scope.recipe);
        PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
        var payloadData = $scope.hooks.getPayloadData();
        ComputableSchemaRecipeSave.handleSave($scope, recipeSerialized, payloadData, deferred);
        return deferred.promise.then(visualCtrl.saveServerParams);
    };

    $scope.showChangeInputModal = function(virtualInputIndex) {
        var newScope = $scope.$new();
        newScope.virtualInputIndex = virtualInputIndex;
        CreateModalFromTemplate("/templates/recipes/visual-recipes-fragments/visual-recipe-change-input-modal.html", newScope);
    };

    $scope.convert = function(type, label) {
        Dialogs.confirm($scope, translate("RECIPES.SQL_CONVERSION.CONVERT_TO", "CONVERT To " + label + " recipe", {type: label}),
                        translate("RECIPES.SQL_CONVERSION.CONVERTING_THE_RECIPE_TO", "Converting the recipe to "+label+" will enable you to edit the query, but you will not be able to use the visual editor anymore.", {type: label})+
                        "<br/><strong>" + translate("RECIPES.SQL_CONVERSION.THIS_OPERATION_IS_IRREVERSIBLE", "This operation is irreversible.") + "</strong>")
        .then(function() {
            var payloadData = $scope.hooks.getPayloadData();
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            $scope.hooks.save().then(function() {
                DataikuAPI.flow.recipes.visual.convert($stateParams.projectKey, recipeSerialized, payloadData, type)
                .success(function(data) {
                    DKUtils.reloadState();
                }).error(setErrorInScope.bind($scope));
            });
        });
    };

    $scope.showSQLModal = function(){
        var newScope = $scope.$new();
        newScope.convert = $scope.convert;
        newScope.uiState = {currentTab: 'query'};
        $scope.hooks.updateRecipeStatus(false, true).then(function(){
        	// get the latest values, not the ones of before the updatestatus call
        	newScope.query = $scope.recipeStatus.sql;
        	newScope.engine = $scope.recipeStatus.selectedEngine.type;
        	newScope.executionPlan = $scope.recipeStatus.executionPlan;
            newScope.cannotConvert = $scope.hasMultipleOutputs();
            CreateModalFromTemplate("/templates/recipes/fragments/sql-modal.html", newScope);
        });
    };

    $scope.hasMultipleOutputs = function() {
        return $scope.recipeStatus.sqlWithExecutionPlanList && $scope.recipeStatus.sqlWithExecutionPlanList.length > 1;
    };

    $scope.selectOutputForSql = outputName => {
        let sqlWithExecutionPlan = $scope.recipeStatus.sqlWithExecutionPlanList.find(s => s.outputName === outputName);
        if (sqlWithExecutionPlan === undefined) {
            sqlWithExecutionPlan = $scope.recipeStatus.sqlWithExecutionPlanList[0];
        }
        $scope.selectedOutputName = sqlWithExecutionPlan.outputName;
        $scope.recipeStatus.sql = sqlWithExecutionPlan.sql;
        $scope.recipeStatus.executionPlan = sqlWithExecutionPlan.executionPlan;
    };

    $scope.hasSchemaToDisplay = () => {
        if(!$scope.recipeStatus) return false;
        const schema = $scope.selectedOutputSchema();
        return schema && schema.columns.length > 0;
    }

    $scope.selectedOutputSchema = () => {
        if(!$scope.recipeStatus) return; // ignore while status is not available, the output will not render anyway.

        // If the recipe has multiple outputs & declares a schema for the currently selected output, use it
        if($scope.hasMultipleOutputs()) {
            const sqlWithExecutionPlan = $scope.recipeStatus.sqlWithExecutionPlanList.find(s => s.outputName === $scope.selectedOutputName);
            if(sqlWithExecutionPlan && sqlWithExecutionPlan.schema) {
                return sqlWithExecutionPlan.schema;
            }
        }
        // otherwise, use main output schema
        return $scope.recipeStatus.outputSchema;
    }

    $scope.getSingleInputName = function() {
        if ($scope.recipe && $scope.recipe.inputs && $scope.recipe.inputs.main && $scope.recipe.inputs.main.items.length) {
            return $scope.recipe.inputs.main.items[0].ref;
        }
    };

    $scope.getColumns = function(datasetName) {
        var schema = DatasetUtils.getSchema($scope, datasetName || $scope.getSingleInputName());
        return schema ? schema.columns : [];
    };

    $scope.getColumnNames = function(datasetName) {
        return $scope.getColumns(datasetName).map(function(col) {return col.name});
    };

    $scope.getColumn = function(datasetName, name) {
        return $scope.getColumns(datasetName).filter(function(col) {return col.name==name})[0];
    };

    $scope.datasetHasColumn = function(datasetName, columnName) {
        return !!$scope.getColumn(datasetName, columnName);
    };

    $scope.columnTypes = [
        {name:'TINYINT',label:'tinyint (8 bit)'},
        {name:'SMALLINT',label:'smallint (16 bit)'},
        {name:'INT',label:'int'},
        {name:'BIGINT',label:'bigint (64 bit)'},
        {name:'FLOAT',label:'float'},
        {name:'DOUBLE',label:'double'},
        {name:'BOOLEAN',label:'boolean'},
        {name:'STRING',label:'string'},
        {name:'DATE',label:'datetime with tz'},
        {name:'DATEONLY',label:'date only'},
        {name:'DATETIMENOTZ',label:'Datetime no tz'},
        {name:'ARRAY',label:'array<...>'},
        {name:'MAP',label:'map<...>'},
        {name:'OBJECT',label:'object<...>'}
    ];

});


app.controller("ChangeRecipeVirtualInputController", function ($scope, DataikuAPI, $stateParams, DatasetUtils, RecipesUtils, translate) {
    if ($scope.recipe.type === "generate_features") {
        if ($scope.virtualInputIndex === 0) {
            $scope.title = "Replace primary dataset";
        } else {
            $scope.title = "Replace enrichment dataset";
        }
    } else {
        $scope.title = translate("VISUAL_RECIPE.CHANGE_INPUT_MODAL.REPLACE_RECIPE_INPUT", "Replace recipe input");
    }


    DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, $scope.recipe.type).then(function (data) {
        $scope.availableInputDatasets = $scope.getAvailableReplacementDatasets(data[0]);
    });

    $scope.replacement = {};

    $scope.$watch("replacement.name", function(nv) {
        if (nv) {
            let payload = $scope.hooks.getPayloadData();
            let newInputName = nv;
            delete $scope.replacementImapact;
            resetErrorInScope($scope);
            DataikuAPI.flow.recipes.visual.testInputReplacement($stateParams.projectKey, $scope.recipe, payload, $scope.virtualInputIndex, newInputName)
                .then(response => $scope.replacementImapact = response.data)
                .catch(setErrorInScope.bind($scope));
        }
    })

    $scope.ok = function(dismiss) {
        // Add dataset to recipes
        if (RecipesUtils.getInput($scope.recipe, "main", $scope.replacement.name) == null) {
            RecipesUtils.addInput($scope.recipe, "main", $scope.replacement.name);
        }

        $scope.onInputReplaced($scope.replacement, $scope.virtualInputIndex);

        dismiss();
        $scope.hooks.updateRecipeStatus();
    }
});

app.directive('fieldsForFilterDesc', function() {
    return {
        restrict: 'A',
        scope: false,
        link : function($scope, element, attrs) {
            $scope.distinctOptionDisabled=true
            var updateFields = function() {
                if ($scope.params == null || $scope.recipe == null || $scope.computablesMap == null) {
                    return;
                }
                $scope.filterDesc = $scope.params.preFilter;
                $scope.dataset = $scope.recipe.inputs['main'].items[0].ref;
                $scope.schema = $scope.computablesMap[$scope.recipe.inputs['main'].items[0].ref].dataset.schema;
            };
            $scope.$watch('params', updateFields);
            $scope.$watch('recipe', updateFields, true);
            $scope.$watch('computablesMap', updateFields); // not deep => won't react to changes, but hopefully watching the recipe is enough
        }
    };
});

app.directive('baseTypeSelector', function($timeout, ListFilter) {
    return {
        restrict: 'A',
        scope: {
            schemaColumn: '='
        },
        templateUrl: '/templates/recipes/visual-recipes-fragments/base-type-selector.html',
        link : function($scope, element, attrs) {
            $scope.columnTypes = [
                                  {name:'tinyint',label:'tinyint (8 bit)'},
                                  {name:'smallint',label:'smallint (16 bit)'},
                                  {name:'int',label:'int'},
                                  {name:'bigint',label:'bigint (64 bit)'},
                                  {name:'float',label:'float'},
                                  {name:'double',label:'double'},
                                  {name:'boolean',label:'boolean'},
                                  {name:'string',label:'string'},
                                  {name:'date',label:'datetime with tz'},
                                  {name:'dateonly',label:'date only'},
                                  {name:'datetimenotz',label:'datetime no tz'},
                                  {name:'array',label:'array<...>'},
                                  {name:'map',label:'map<...>'},
                                  {name:'object',label:'object<...>'}
                              ];
            $scope.select = function(columnType) {
                $scope.schemaColumn.type = columnType.name;
            };
        }
    };
});

app.directive('aggregateTypeEditor', function() {
    return {
        restrict: 'A',
        scope: {
            schemaColumn: '='
        },
        replace: true,
        templateUrl: '/templates/recipes/visual-recipes-fragments/aggregate-type-editor.html',
        link : function($scope, element, attrs) {
            $scope.addObjectField = function() {
                $scope.schemaColumn.objectFields = $scope.schemaColumn.objectFields || [];
                $scope.schemaColumn.objectFields.push({name:'', type:'string'});
            };
            var ensureSubFields = function() {
                if ($scope.schemaColumn.type == 'array') {
                    $scope.schemaColumn.arrayContent = $scope.schemaColumn.arrayContent || {name:'', type:'string'};
                }
                if ($scope.schemaColumn.type == 'map') {
                    $scope.schemaColumn.mapKeys = $scope.schemaColumn.mapKeys || {name:'', type:'string'};
                    $scope.schemaColumn.mapValues = $scope.schemaColumn.mapValues || {name:'', type:'string'};
                }
                if ($scope.schemaColumn.type == 'object') {
                    $scope.schemaColumn.objectFields = $scope.schemaColumn.objectFields || [];
                }
            };
            $scope.$watch('schemaColumn.type', ensureSubFields);
        }
    };
});

})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("GroupingRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller) {
        $scope.recipeType = "grouping";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset && $scope.io.targetVariable) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                var niceTargetVariable = $scope.io.targetVariable.replace(/[^\w ]+/g,"").replace(/ +/g,"_");
                $scope.maybeSetNewDatasetName(niceInputName + "_by_" + niceTargetVariable);
            }
        };

        $scope.getCreationSettings = function () {
            return {groupKey: $scope.io.targetVariable};
        };

        var superFormIsValid = $scope.formIsValid;
        $scope.formIsValid = function() {
            return !!(superFormIsValid() && $scope.io.targetVariable !== undefined);
        };
        $scope.showOutputPane = function() {
            return !!($scope.io.inputDataset && $scope.io.targetVariable !== undefined);
        };

        $scope.$watch("io.targetVariable", Fn.doIfNv($scope.autosetName));
    });


    app.controller("GroupingRecipeController", function($scope, $stateParams, DataikuAPI, $q, Dialogs, ContextualMenu, PartitionDeps, $rootScope,
     $timeout, DKUtils, Expressions, Logger, $controller,  RecipesUtils, Fn, DatasetUtils) {
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope}); //Controller inheritance
        this.visualCtrl = visualCtrl;
        $scope.aggregateUsabilityFlag = "usableInGroup";
        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

        $scope.hooks.getPayloadData = function () {
            return angular.toJson($scope.params);
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) return $q.reject("payload not ready");
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, {reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan}).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                var outputSchema = $scope.recipeStatus.outputSchema;
                var outputSchemaBO = $scope.recipeStatus.outputSchemaBeforeOverride;
                if (outputSchema) {
                    $scope.params.postFilter.$status = $scope.params.postFilter.$status || {};
                    $scope.params.postFilter.$status.schema = outputSchemaBO;
                    // override handling:

                    $scope.params.outputColumnNameOverrides = $scope.params.outputColumnNameOverrides || {};
                    var columnsAO = outputSchema.columns; // after override
                    var columnsBO = (outputSchemaBO && outputSchemaBO.columns) ? outputSchemaBO.columns : outputSchema.columns; // before override

                    for (var i in columnsAO) {
                        if (columnsAO[i].name != columnsBO[i].name) {$scope.params.outputColumnNameOverrides[columnsBO[i].name] = columnsAO[i].name;}
                        columnsAO[i].$beforeOverride = columnsBO[i].name;
                        columnsAO[i].name = $scope.params.outputColumnNameOverrides[columnsBO[i].name] || columnsBO[i].name;
                    }
                    resyncWithEngine();
                }
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        /******  overrides *****/
        $scope.updateColumnNameOverride = function(column) {
            if (column.$beforeOverride != column.name) {
                $scope.params.outputColumnNameOverrides[column.$beforeOverride] = column.name;
            } else {
                delete $scope.params.outputColumnNameOverrides[column.$beforeOverride];
            }
        };


        /******  filters  ******/

        function validateFilters() {
            if (!$scope.params) return;//not ready
            var inputRef = RecipesUtils.getSingleInput($scope.recipe, "main").ref
            var inputSchema = $scope.computablesMap[inputRef].dataset.schema
            validateFilter($scope.params.preFilter, inputSchema);
            validateFilter($scope.params.postFilter);
        }

        function validateFilter(filterDesc, schema) {
            var deferred = $q.defer();
            if (!filterDesc || !filterDesc.enabled) return;
            if (angular.isUndefined(filterDesc.expression)) return;
            Expressions.validateExpression(filterDesc.expression, schema)
                .success(function(data) {
                    if (data.ok && $scope.mustRunInDatabase && !data.fullyTranslated) {
                        data.ok = false;
                    }
                    filterDesc.$status = data;
                    filterDesc.$status.validated = true;
                    deferred.resolve(data);
                })
                .error(function(data) {
                    setErrorInScope.bind($scope);
                    deferred.reject('Error while validating filter');
                });
            return deferred.promise;
        };

        /* callback given to the filter module */
        $scope.onFilterUpdate = $scope.updateRecipeStatusLater;

        /****** computed columns ********/
        function computedColumnListUpdated(computedColumns) {
            $scope.params.computedColumns = angular.copy(computedColumns);
            resyncWithInputSchema();
            $scope.updateRecipeStatusLater();
        }

        /* callback given to the computed columns module */
        $scope.onComputedColumnListUpdate = computedColumnListUpdated;

        $scope.getColumnsWithComputed = function() {
            if (!$scope.uiState.columnsWithComputed) {
                var columns = [].concat($scope.getColumns());
                for (var i = 0; i < (($scope.params || {}).computedColumns || []).length; i++) {
                    var computedCol = $scope.params.computedColumns[i];
                    // do not add computed columns if they are blank
                    if (computedCol.name && columns.map(function(col){return col.name}).indexOf(computedCol.name) == -1) {
                        columns.push({
                            name: computedCol.name,
                            type: computedCol.type
                        });
                    }
                }
                $scope.uiState.columnsWithComputed = columns;
            }
            return $scope.uiState.columnsWithComputed;
        };

        /******  grouping key/values  ********/

        $scope.removeGroupKey = function(removedGroupKey) {
            if (removedGroupKey.column) {
                const presentColumns = $scope.params.values.map(Fn.prop('column'));
                presentColumns.push(removedGroupKey.column);
                const insertIndex = $scope.listColumnsForCumstomColumnsEditor()
                    .filter(Fn.inArray(presentColumns)).indexOf(removedGroupKey.column);
                $scope.params.values.splice(Math.max(insertIndex,0), 0, removedGroupKey);
            }
            $scope.hooks.updateRecipeStatus();
        }

        $scope.addGroupKey = function(col) {
            if (!col) {
                return;
            }
            const idx = $scope.params.values.map(Fn.prop('column')).indexOf(col.column);
            if (idx === -1) {
                return;
            }
            const key = angular.copy(col);
            key.$selected = false;
            $scope.params.keys.push(key);
            $scope.params.values.splice(idx, 1);
            $scope.hooks.updateRecipeStatus();
            resyncWithInputSchema(); // force update to show display changes on the editable list
        };
        $scope.groupKeyFilter = {};

        $scope.addCustomValue = function(){
            var newVal = {
                // userFriendlyTransmogrify is in utils.js
                /* eslint-disable-next-line no-undef */
                customName : userFriendlyTransmogrify('custom_aggr', $scope.params.values.filter(function(v) {return v.column == null;}), 'customName', '_', true),
                customExpr : ''
            }
            $scope.params.values.push(newVal);

            //activate edit mode and focus name field
            $timeout(function(){
                var el = angular.element('[computed-column-editor]').last();
                $timeout(function(){
                    $('.name-editor', el).trigger("focus");
                });
                $scope.$apply();
            });
        };
        // /!\ Watch out /!\
        // params comes from a json.parse : if multiple calls are made
        // a $watch(true) won't trigger the copy to realValues, thus makeing object refs
        // used in frontend disconnected from actual data
        $scope.$watchCollection("params.values", function(nv) {
            $scope.realValues = getRealValues();
        });
        var getRealValues = function() {
            var realValues = (!$scope.params||!$scope.params.values) ? [] : $scope.params.values.filter(function(val) { return !!val.column });
            var realValuesNames = realValues.map(function(rv){return rv.column});
            var newRealValues = [];
            var i;
            var oldRVIdx;
            var columns = $scope.getColumns();
            for (i = 0; i < columns.length; i++) {
                var col = columns[i];
                if (col.name && col.name.length > 0) {
                    oldRVIdx = realValuesNames.indexOf(col.name);
                    if (oldRVIdx >= 0) {
                        newRealValues.push(realValues[oldRVIdx]);
                    }
                }
            }
            var newRealValuesNames = newRealValues.map(function(rv){return rv.column});
            var computedColumns = (($scope.params || {}).computedColumns || []);
            for (i = 0; i < computedColumns.length; i++) {
                var compCol = computedColumns[i];
                // make sure that there is no previous value named the same
                if (compCol.name && compCol.name.length > 0 && newRealValuesNames.indexOf(compCol.name) == -1) {
                    oldRVIdx = realValuesNames.indexOf(compCol.name);
                    if (oldRVIdx >= 0) {
                        newRealValues.push(realValues[oldRVIdx]);
                    }
                }
            }
            return newRealValues;
        }
        $scope.getCustomValues = function() {
            if (!$scope.params||!$scope.params.values) { return [] }
            return $scope.params.values.filter(function(val) { return !val.column });
        }

        //This lists the input dataset column names
        $scope.listColumnsForCumstomColumnsEditor = function(){
            return $scope.getColumns().map(function (col) {
                return col.name;
            });
        };

        /********  aggregations selector *******/

        $scope.aggregationTypes =  [
            {name: "countDistinct", opType:"DISTINCT", label: $scope.translate("GROUP_RECIPE.GROUP.DISTINCT", "Distinct"), tooltip: $scope.translate("GROUP_RECIPE.GROUP.COUNT_DISTINCT", "Count distinct values")},
            {name: "min", label: $scope.translate("GROUP_RECIPE.GROUP.MIN", "Min")},
            {name: "max", label : $scope.translate("GROUP_RECIPE.GROUP.MAX", "Max")},

            {name: "avg", label: $scope.translate("GROUP_RECIPE.GROUP.AVG", "Avg")},
            {name: "median", label: $scope.translate("GROUP_RECIPE.GROUP.MEDIAN", "Median")},
            {name: "sum", label: $scope.translate("GROUP_RECIPE.GROUP.SUM", "Sum")},
            {name: "stddev", label: $scope.translate("GROUP_RECIPE.GROUP.STD_DEV", "Std. dev.")},
            {name: "count", label: $scope.translate("GROUP_RECIPE.GROUP.COUNT", "Count"), tooltip: $scope.translate("GROUP_RECIPE.GROUP.COUNT_NON_NULL", "Count non-null"), separatorAfter: true},

            {name: "first", label: $scope.translate("GROUP_RECIPE.GROUP.FIRST", "First")},
            {name: "last", label: $scope.translate("GROUP_RECIPE.GROUP.LAST", "Last")},
            {name: "concat", label: $scope.translate("GROUP_RECIPE.GROUP.CONCAT", "Concat"), tooltip: $scope.translate("GROUP_RECIPE.GROUP.CONCAT_VALUES", "Concatenate values in one string")},
        ];

        $scope.columnHasSomeComputation = function (col) {
            var ret = false;
            $scope.aggregationTypes.forEach(function(agg) {
                ret = ret || col[agg.name];
            });
            return ret;
        };

        // Checks if we can perform the specified aggregation on column col
        $scope.colCanAggr = function(col, agg) {
            if (!$scope.engineCanAggr(agg)) return false;
            var opType = agg.opType || agg.name.toUpperCase();
            var aggregability = $scope.recipeStatus.selectedEngine.aggregabilities[opType];
            var typeCategory = {"string":"strings",
                                "date":"dates",
                                "dateonly":"dates",
                                "datetimenotz":"dates",
                                "boolean":"booleans",
                                "tinyint":"numerics",
                                "smallint":"numerics",
                                "int":"numerics",
                                "bigint":"numerics",
                                "float":"numerics",
                                "double":"numerics"
                            }[col.type];
            return aggregability && typeCategory && aggregability[typeCategory];
        };

        $scope.engineCanAggrType = function(opType) {
            if (!$scope.recipeStatus || !$scope.recipeStatus.selectedEngine) return false;
            var aggregability = $scope.recipeStatus.selectedEngine.aggregabilities[opType];
            return aggregability && aggregability[$scope.aggregateUsabilityFlag];
        };
        $scope.engineCanAggr = function(agg) {
            if (!$scope.recipeStatus || !$scope.recipeStatus.selectedEngine) return false;
            var opType = agg.opType || agg.name.toUpperCase();
            var aggregability = $scope.recipeStatus.selectedEngine.aggregabilities[opType];
            return aggregability && aggregability[$scope.aggregateUsabilityFlag];
        };

        // some aggregations require additional fields to not be invalid - we init them when the aggregation is activated, and before status update
        $scope.onAggregationChange = function(column, agg) {
            if (column[agg.name]) {
                if(['first', 'last'].includes(agg.name)) {
                    var cols = $scope.getColumns();
                    column.orderColumn = column.orderColumn || (cols && cols.length ? cols[0].name : undefined);
                } else if('concat' === agg.name) {
                    column.concatSeparator = column.concatSeparator != null ? column.concatSeparator : ',';
                    column.concatDistinct = column.concatDistinct || false;
                }
            }
        }

        $scope.aggregation = {'all':{},'some':{},'none':{},'disabled':{}};

        function filterUnused(vals) {
            return vals.filter(function(val){
                return !$scope.uiState.hideUseless || $scope.columnHasSomeComputation(val);
            });
        }

        $scope.shouldDisplayOptions = function(column) {
            return column !== undefined && (column.first || column.last || column.concat);
        }

        $scope.selection = {
            'customFilter': filterUnused,
            'customFilterWatch': 'uiState.hideUseless'
        }

        $scope.recomputeAggregationStates = function(cols) {
            for (var k in $scope.aggregation) {$scope.aggregation[k]={};}

            cols.forEach(function(column){
                $scope.aggregationTypes.forEach(function(agg) {
                    var colEnabled = $scope.colCanAggr(column, agg);
                    $scope.aggregation.all[agg.name] =
                        ($scope.aggregation.all[agg.name] == undefined ? true : $scope.aggregation.all[agg.name])
                        && (colEnabled ? column[agg.name] : false);
                    $scope.aggregation.some[agg.name] =
                        ($scope.aggregation.some[agg.name] || false)
                        || (colEnabled ? column[agg.name] : false);
                    $scope.aggregation.disabled[agg.name] =
                        ($scope.aggregation.disabled[agg.name] || false)
                        || colEnabled;
                });
            });
            angular.forEach($scope.aggregationTypes,function(agg){
                $scope.aggregation.disabled[agg.name] = !$scope.aggregation.disabled[agg.name];
                $scope.aggregation.some[agg.name] = $scope.aggregation.some[agg.name] && !$scope.aggregation.all[agg.name];
                $scope.aggregation.none[agg.name] = !$scope.aggregation.some[agg.name] && !$scope.aggregation.all[agg.name];
            });
        }

        // Apply/disapply aggregation to all selected columns
        $scope.massAction = function(agg, selectedObjects){
            selectedObjects.forEach(function(val) {
                if ($scope.colCanAggr(val, agg)) {
                    val[agg.name] = $scope.aggregation.all[agg.name];
                    $scope.onAggregationChange(val, agg);
                }
            });
            $scope.aggregation.some[agg.name] = false;
            $scope.aggregation.none[agg.name] = !$scope.aggregation.all[agg.name];
            $scope.hooks.updateRecipeStatus();
        }

        $scope.massUseAsKeys = function() {
            // add selected values to keys
            $scope.selection.selectedObjects.forEach(function(val) {
                var key = angular.copy(val);
                key.$selected = false;
                $scope.params.keys.push(key);
            });
            //remove from values list
            $scope.params.values = $scope.selection.allObjects.filter(function(val) {
                return !val.$selected;
            });
            $scope.hooks.updateRecipeStatus();
            resyncWithInputSchema(); // force update to show display changes on the editable list
        };

        /********  general init  ********/

        function loadParamsFromScript(scriptData) {
            if (!scriptData) return;
            $scope.params = JSON.parse(scriptData);
            $scope.params.preFilter = $scope.params.preFilter || {};
            $scope.params.computedColumns = $scope.params.computedColumns || [];
            $scope.params.postFilter = $scope.params.postFilter || {};
            $scope.params.outputColumnNameOverrides = $scope.params.outputColumnNameOverrides || {};

            $scope.uiState.computedColumns = angular.copy($scope.params.computedColumns);

            //keep params for dirtyness detection
            visualCtrl.saveServerParams();

            // update recipe according to current schema
            resyncWithInputSchema();

            // update aggragation according to engine capabilities
            resyncWithEngine();

            //Add column types in the grouping values & keys, it will make things easier
            var colsByName = {};
            $scope.getColumns().forEach(function(col){
                colsByName[col.name] = col;
            });
            $scope.params.values.forEach(function(gv){
                if (colsByName[gv.column]) {
                    gv.type = colsByName[gv.column].type;
                }
            });
            $scope.params.keys.forEach(function(gv){
                if (colsByName[gv.column]) {
                    gv.type = colsByName[gv.column].type;
                }
            });
        }

        function resyncWithInputSchema() {
            // in case the dataset schema changed since the recipe creation/last edition
            // reset the calculated columns with computed to force refresh
            $scope.uiState.columnsWithComputed = undefined;
            var inputColumnsWithComputed = $scope.getColumnsWithComputed();
            var inputColumnsWithComputedNames = inputColumnsWithComputed.map(function(col){return col.name});

            var keys = {};
            (($scope.params || {}).keys || []).forEach(function(col, i) {
                col.$$originalIndex = i;
                if (col.column && inputColumnsWithComputedNames.indexOf(col.column) >= 0) {
                    keys[col.column] = col;
                }
            });
            var values = {}, customValues = [];
            (($scope.params || {}).values || []).forEach(function(col) {
                if (col.column) {
                    if (inputColumnsWithComputedNames.indexOf(col.column) >= 0) {
                        values[col.column] = col;
                    }
                } else {
                    customValues.push(col);
                }
            });

            var newKeys = [];
            var newValues = [];
            inputColumnsWithComputed.forEach(function(col){
                var newCol;
                if (keys[col.name]) {
                    newCol = keys[col.name];
                } else if (values[col.name]) {
                    newCol = values[col.name];
                } else {
                    //this is apparently a new column in the dataset. Add an empty value
                    //put everything to false to avoid dirtyness on check/uncheck
                    newCol = {
                        column: col.name,
                        type: col.type
                    };
                }
                angular.extend(newCol, {
                    column: col.name,
                    type: col.type
                });
                $scope.aggregationTypes.forEach(function(agg){
                    newCol[agg.name] = newCol[agg.name] || false;
                });
                if (keys[col.name]) {
                    newKeys.push(newCol);
                } else {
                    newValues.push(newCol);
                }
            });

            customValues.forEach(function(val){
                newValues.push(val);
            });

            // Sorting the keys by origin index to preserve pre-existing order
            newKeys.sort((a, b) => a.$$originalIndex - b.$$originalIndex);

            // remove outdated columns (keep computed columns or column that is in the schema)
            $scope.params = $scope.params || {};
            $scope.params.keys = newKeys;
            $scope.params.values = newValues;

            // call the callback if it exists
            if ($scope.onResyncWithInputSchema) {
                $scope.onResyncWithInputSchema();
            }
        }

        function resyncWithEngine() {
            if (!$scope.recipeStatus || !$scope.recipeStatus.selectedEngine) return; // no aggregability available yet, let's not jump to conclusions and deactivate everything
            //prevent inconsistent aggregations (ex: avg selected for number then type changed to string)
            $scope.params.values.forEach(function(val){
                $scope.aggregationTypes.forEach(function(agg){
                    if (val[agg.name] && !$scope.colCanAggr(val, agg)) {
                        val[agg.name] = false;
                    }
                });
            });
            if ($scope.engineCanAggrType('CONCAT_DISTINCT') != true) {
                $scope.params.values.forEach(function(val){
                    val.concatDistinct = false; // otherwise you never get to click on the checkbox
                });
            }
            if ($scope.engineCanAggrType('FIRST_NOTNULL') != true) {
                $scope.params.values.forEach(function(val){
                    val.firstLastNotNull = false; // otherwise you never get to click on the checkbox
                });
            }
        }

        function onScriptChanged(nv, ov) {
             if (nv) {
                loadParamsFromScript($scope.script.data);
                DKUtils.reflowNext();
                DKUtils.reflowLater();
                $scope.hooks.updateRecipeStatus();
            }
        }

        $scope.uiState = {
            currentStep: 'group',
            outputColumnNamesOverridable: true,
            computedColumns: []
        };

        $scope.hooks.onRecipeLoaded = function(){
            Logger.info("On Recipe Loaded");
            validateFilters();
            $scope.$watch("script.data", onScriptChanged, true); // this will call $scope.hooks.updateRecipeStatus when ready
            $scope.$watchCollection("recipe.inputs.main.items", function() {
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => resyncWithInputSchema());
            });
        };

        $scope.$watch('topNav.tab',function(nv){
            if (nv == 'settings') {
                $timeout(function() {
                    $scope.$broadcast('redrawFatTable');
                });
            }
        });

        $scope.enableAutoFixup();
        $scope.specificControllerLoadedDeferred.resolve();
    });
})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("UpsertRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller) {
        $scope.recipeType = "upsert";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset && $scope.io.targetVariable) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                var niceTargetVariable = $scope.io.targetVariable.replace(/[^\w ]+/g,"").replace(/ +/g,"_");
                $scope.maybeSetNewDatasetName(niceInputName + "_by_" + niceTargetVariable);
            }
        };

        $scope.getCreationSettings = function () {
            return {upsertKey: $scope.io.targetVariable};
        };

        var superFormIsValid = $scope.formIsValid;
        $scope.formIsValid = function() {
            return !!(superFormIsValid() && $scope.io.targetVariable !== undefined);
        };
        $scope.showOutputPane = function() {
            return !!($scope.io.inputDataset && $scope.io.targetVariable !== undefined);
        };

        $scope.$watch("io.targetVariable", Fn.doIfNv($scope.autosetName));
    });


    app.controller("UpsertRecipeController", function($scope, $stateParams, DataikuAPI, $q, Dialogs, ContextualMenu, PartitionDeps, $rootScope,
     $timeout, DKUtils, Expressions, Logger, $controller,  RecipesUtils, Fn, DatasetUtils) {
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope}); //Controller inheritance
        this.visualCtrl = visualCtrl;
        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

        $scope.hooks.getPayloadData = function () {
            return angular.toJson($scope.params);
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) return $q.reject("payload not ready");
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, {reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan}).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        /******  filters  ******/
        function validateFilters() {
            if (!$scope.params) return;//not ready
            var inputRef = RecipesUtils.getSingleInput($scope.recipe, "main").ref
            var inputSchema = $scope.computablesMap[inputRef].dataset.schema
            validateFilter($scope.params.preFilter, inputSchema);
        }

        function validateFilter(filterDesc, schema) {
            var deferred = $q.defer();
            if (!filterDesc || !filterDesc.enabled) return;
            if (angular.isUndefined(filterDesc.expression)) return;
            Expressions.validateExpression(filterDesc.expression, schema)
                .success(function(data) {
                    if (data.ok && $scope.mustRunInDatabase && !data.fullyTranslated) {
                        data.ok = false;
                    }
                    filterDesc.$status = data;
                    filterDesc.$status.validated = true;
                    deferred.resolve(data);
                })
                .error(function(data) {
                    setErrorInScope.bind($scope);
                    deferred.reject('Error while validating filter');
                });
            return deferred.promise;
        };

        /* callback given to the filter module */
        $scope.onFilterUpdate = $scope.updateRecipeStatusLater;

        /****** computed columns ********/
        function computedColumnListUpdated(computedColumns) {
            $scope.params.computedColumns = angular.copy(computedColumns);
            resyncWithInputSchema();
            $scope.updateRecipeStatusLater();
        }

        /* callback given to the computed columns module */
        $scope.onComputedColumnListUpdate = computedColumnListUpdated;

        $scope.getColumnsWithComputed = function() {
            if (!$scope.uiState.columnsWithComputed) {
                var columns = [].concat($scope.getColumns());
                for (var i = 0; i < (($scope.params || {}).computedColumns || []).length; i++) {
                    var computedCol = $scope.params.computedColumns[i];
                    // do not add computed columns if they are blank
                    if (computedCol.name && columns.map(function(col){return col.name}).indexOf(computedCol.name) == -1) {
                        columns.push({
                            name: computedCol.name,
                            type: computedCol.type
                        });
                    }
                }
                $scope.uiState.columnsWithComputed = columns;
            }
            return $scope.uiState.columnsWithComputed;
        };

        /****** upsert keys (note: similar to distinct recipe) *****/
        $scope.selectLine = function(event, col, ignore) {
            event.preventDefault();
            var hasLastSelection = $scope.uiState.columnStatus.filter(function(c) {
                return c.ignore == ignore && !!c.$lastSelection;
            }).length > 0 ;
            if (event.shiftKey && hasLastSelection) {
                var selecting = false;
                for (var i = 0; i < $scope.uiState.columnStatus.length; i++) {
                    var c = $scope.uiState.columnStatus[i];
                    if (c.ignore != ignore) {
                        continue;
                    }
                    var bound = !!c.$lastSelection || c.name === col.name;
                    var firstBound = !selecting && bound;
                    var lastBound = !!selecting && bound;
                    if (firstBound) {
                        selecting = true;
                        c.$selected = true;
                    }
                    c.$selected = selecting;
                    if (lastBound) {
                        selecting = false;
                    }
                }
            } else {
                // refresh the last clicked item
                $scope.uiState.columnStatus
                        .filter(function(c) {
                            return c.ignore == ignore;
                        }).forEach(function(c) {
                            c.$lastSelection = c.name === col.name;
                        });
                // handle meta/ctrl click or normal click
                if (event.metaKey || event.ctrlKey) {
                    col.$selected = !col.$selected;
                } else {
                    $scope.uiState.columnStatus
                        .filter(function(c) {
                            return c.ignore == ignore;
                        }).forEach(function(c) {
                            c.$selected = c.name === col.name;
                        });
                }
            }
        };

        function assignIgnoreSelected(ignore, selected) {
            return function(col) {
                col.ignore = ignore;
                col.$selected = selected;
                col.$lastSelection = false;
            };
        }

        $scope.removeAllUpsertColumns = function() {
            if (!$scope.uiState.columnStatus) {
                return;
            }
            $scope.uiState.columnStatus.forEach(assignIgnoreSelected(true, false));
        };
        $scope.addAllUpsertColumns = function() {
            if (!$scope.uiState.columnStatus) {
                return;
            }
            $scope.uiState.columnStatus.forEach(assignIgnoreSelected(false, false));
        };
        $scope.removeUpsertColumns = function(col) {
            if (col) {
                assignIgnoreSelected(true, false)(col);
            } else if ($scope.uiState.columnStatus) {
                $scope.uiState.columnStatus
                    .filter(function(col) {
                        return !col.ignore && col.$selected;
                    }).forEach(assignIgnoreSelected(true, false));
            }
        };
        $scope.addUpsertColumns = function(col) {
            if (col) {
                assignIgnoreSelected(false, false)(col);
            } else if ($scope.uiState.columnStatus) {
                $scope.uiState.columnStatus
                    .filter(function(col) {
                        return col.ignore && col.$selected;
                    }).forEach(assignIgnoreSelected(false, false));
            }
        };

        /********  general init  ********/
        function loadParamsFromScript(scriptData) {
            if (!scriptData) return;
            $scope.params = JSON.parse(scriptData);
            $scope.params.preFilter = $scope.params.preFilter || {};
            $scope.params.computedColumns = $scope.params.computedColumns || [];

            $scope.uiState.computedColumns = angular.copy($scope.params.computedColumns);

            $scope.uiState.columnStatus = angular.copy($scope.getColumnsWithComputed());
            var keyColNames = $scope.params.keys.map(function(k){return k.column});
            $scope.uiState.columnStatus.forEach(function(col) {
                col.ignore = keyColNames.indexOf(col.name) == -1;
            });

            //keep params for dirtyness detection
            visualCtrl.saveServerParams();

            // update recipe according to current schema
            resyncWithInputSchema();
            onColumnStatusChanged();
        }

        function resyncWithInputSchema() {
            // in case the dataset schema changed since the recipe creation/last edition
            // reset the calculated columns with computed to force refresh
            $scope.uiState.columnsWithComputed = undefined;
            var inputColumnsWithComputed = $scope.getColumnsWithComputed();

            var newColumnStatus = [];
            var oldColumnStatusNames = ($scope.uiState.columnStatus || []).map(function(col){return col.name});
            inputColumnsWithComputed.forEach(function(col) {
                var oldCSNIdx = oldColumnStatusNames.indexOf(col.name);
                if (oldCSNIdx >= 0) {
                    newColumnStatus.push(angular.extend($scope.uiState.columnStatus[oldCSNIdx], col));
                } else {
                    var ncs = angular.copy(col);
                    ncs.ignore = true;
                    newColumnStatus.push(ncs);
                }
            });
            $scope.uiState.columnStatus = newColumnStatus;

            // call the callback if it exists
            if ($scope.onResyncWithInputSchema) {
                $scope.onResyncWithInputSchema();
            }
        }

        function onScriptChanged(nv, ov) {
             if (nv) {
                loadParamsFromScript($scope.script.data);
                DKUtils.reflowNext();
                DKUtils.reflowLater();
                $scope.hooks.updateRecipeStatus();
            }
        }

        $scope.uiState = {
            currentStep: 'upsert',
            computedColumns: []
        };

        $scope.hooks.onRecipeLoaded = function(){
            Logger.info("On Recipe Loaded");
            validateFilters();
            $scope.$watch("script.data", onScriptChanged, true); // this will call $scope.hooks.updateRecipeStatus when ready
            $scope.$watchCollection("recipe.inputs.main.items", function() {
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => resyncWithInputSchema());
            });
        };

        function onColumnStatusChanged() {
            if (!$scope.params) {
                return;
            }
            $scope.params.keys = ($scope.uiState.columnStatus || [])
                                    .filter(function(col){return !col.ignore})
                                    .map(function(col){return {column:col.name}});
            $scope.updateRecipeStatusLater();
        }

        $scope.$watch('topNav.tab',function(nv){
            if (nv == 'settings') {
                $timeout(function() {
                    $scope.$broadcast('redrawFatTable');
                });
            }
        });

        $scope.enableAutoFixup();
        $scope.specificControllerLoadedDeferred.resolve();
        $scope.$watch("uiState.columnStatus", onColumnStatusChanged, true);
    });
})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("DistinctRecipeCreationController", function($scope, $controller) {
        $scope.recipeType = "distinct";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                $scope.maybeSetNewDatasetName(niceInputName + "_distinct");
            }
        };
    });


    app.controller("DistinctRecipeController", function($scope, $stateParams, $q, DKUtils, Expressions, Logger, $controller, DatasetUtils, RecipesUtils) {
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope}); //Controller inheritance
        this.visualCtrl = visualCtrl;

        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;
        /****** distinct column selection *****/
        $scope.selectLine = function(event, col, ignore) {
            event.preventDefault();
            var hasLastSelection = $scope.uiState.columnStatus.filter(function(c) {
                return c.ignore == ignore && !!c.$lastSelection;
            }).length > 0 ;
            if (event.shiftKey && hasLastSelection) {
                var selecting = false;
                for (var i = 0; i < $scope.uiState.columnStatus.length; i++) {
                    var c = $scope.uiState.columnStatus[i];
                    if (c.ignore != ignore) {
                        continue;
                    }
                    var bound = !!c.$lastSelection || c.name === col.name;
                    var firstBound = !selecting && bound;
                    var lastBound = !!selecting && bound;
                    if (firstBound) {
                        selecting = true;
                        c.$selected = true;
                    }
                    c.$selected = selecting;
                    if (lastBound) {
                        selecting = false;
                    }
                }
            } else {
                // refresh the last clicked item
                $scope.uiState.columnStatus
                        .filter(function(c) {
                            return c.ignore == ignore;
                        }).forEach(function(c) {
                            c.$lastSelection = c.name === col.name;
                        });
                // handle meta/ctrl click or normal click
                if (event.metaKey || event.ctrlKey) {
                    col.$selected = !col.$selected;
                } else {
                    $scope.uiState.columnStatus
                        .filter(function(c) {
                            return c.ignore == ignore;
                        }).forEach(function(c) {
                            c.$selected = c.name === col.name;
                        });
                }
            }
        };

        function assignIgnoreSelected(ignore, selected) {
            return function(col) {
                col.ignore = ignore;
                col.$selected = selected;
                col.$lastSelection = false;
            };
        }

        $scope.removeAllDistinctColumns = function() {
            if (!$scope.uiState.columnStatus) {
                return;
            }
            $scope.uiState.columnStatus.forEach(assignIgnoreSelected(true, false));
        };
        $scope.addAllDistinctColumns = function() {
            if (!$scope.uiState.columnStatus) {
                return;
            }
            $scope.uiState.columnStatus.forEach(assignIgnoreSelected(false, false));
        };
        $scope.removeDistinctColumns = function(col) {
            if (col) {
                assignIgnoreSelected(true, false)(col);
            } else if ($scope.uiState.columnStatus) {
                $scope.uiState.columnStatus
                    .filter(function(col) {
                        return !col.ignore && col.$selected;
                    }).forEach(assignIgnoreSelected(true, false));
            }
        };
        $scope.addDistinctColumns = function(col) {
            if (col) {
                assignIgnoreSelected(false, false)(col);
            } else if ($scope.uiState.columnStatus) {
                $scope.uiState.columnStatus
                    .filter(function(col) {
                        return col.ignore && col.$selected;
                    }).forEach(assignIgnoreSelected(false, false));
            }
        };

        /****** recipe related ******/
        $scope.hooks.getPayloadData = function() {
            return angular.toJson($scope.params);
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) {
                return $q.reject("payload not ready");
            }
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, {reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan}).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) {
                    return deferred.reject();
                }
                var outputSchema = $scope.recipeStatus.outputSchema;
                var outputSchemaBO = $scope.recipeStatus.outputSchemaBeforeOverride;
                if (outputSchema) {
                    $scope.params.postFilter.$status = $scope.params.postFilter.$status || {};
                    $scope.params.postFilter.$status.schema = outputSchemaBO;
                    // override handling:

                    $scope.params.outputColumnNameOverrides = $scope.params.outputColumnNameOverrides || {};
                    var columnsAO = outputSchema.columns; // after override
                    var columnsBO = (outputSchemaBO && outputSchemaBO.columns) ? outputSchemaBO.columns : outputSchema.columns; // before override

                    for (var i in columnsAO) {
                        if (columnsAO[i].name != columnsBO[i].name) {
                            $scope.params.outputColumnNameOverrides[columnsBO[i].name] = columnsAO[i].name;
                        }
                        columnsAO[i].$beforeOverride = columnsBO[i].name;
                        columnsAO[i].name = $scope.params.outputColumnNameOverrides[columnsBO[i].name] || columnsBO[i].name;
                    }
                }
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        /******  overrides *****/
        $scope.updateColumnNameOverride = function(column) {
            if (column.$beforeOverride != column.name) {
                $scope.params.outputColumnNameOverrides[column.$beforeOverride] = column.name;
            } else {
                delete $scope.params.outputColumnNameOverrides[column.$beforeOverride];
            }
        };

        /******  filters  ******/

        function validateFilters() {
            if (!$scope.params) {
                return;//not ready
            }
            var inputRef = RecipesUtils.getSingleInput($scope.recipe, "main").ref;
            var inputSchema = $scope.computablesMap[inputRef].dataset.schema;
            validateFilter($scope.params.preFilter, inputSchema);
            validateFilter($scope.params.postFilter);
        }

        function validateFilter(filterDesc, schema) {
            var deferred = $q.defer();
            if (!filterDesc.enabled) {
                return;
            }
            if (angular.isUndefined(filterDesc.expression)) {
                return;
            }
            Expressions.validateExpression(filterDesc.expression, schema)
                .success(function(data) {
                    if (data.ok && $scope.mustRunInDatabase && !data.fullyTranslated) {
                        data.ok = false;
                    }
                    filterDesc.$status = data;
                    filterDesc.$status.validated = true;
                    deferred.resolve(data);
                })
                .error(function(data) {
                    setErrorInScope.bind($scope);
                    deferred.reject('Error while validating filter');
                });
            return deferred.promise;
        }

        /* callback given to the filter module */
        $scope.onFilterUpdate = $scope.updateRecipeStatusLater;

        /********  general init  ********/

        function loadParamsFromScript(scriptData) {
            if (!scriptData) {
                return;
            }
            $scope.params = JSON.parse(scriptData);
            $scope.params.keys = $scope.params.keys || [];
            $scope.params.preFilter = $scope.params.preFilter || {};
            $scope.params.postFilter = $scope.params.postFilter || {};
            $scope.params.outputColumnNameOverrides = $scope.params.outputColumnNameOverrides || {};

            $scope.uiState.columnStatus = angular.copy($scope.getColumns());
            var keyColNames = $scope.params.keys.map(k => k.column);
            $scope.uiState.columnStatus.forEach(function(col) {
                col.ignore = !$scope.params.selectAllColumns && keyColNames.indexOf(col.name) == -1;
            });

            //keep params for dirtyness detection
            visualCtrl.saveServerParams();

            // update recipe according to current schema
            resyncWithInputSchema();
            onColumnStatusChanged();
        }

        function resyncWithInputSchema() {
            // in case the dataset schema changed since the recipe creation/last edition
            var inputColumns = $scope.getColumns();

            var newColumnStatus = [];
            var oldColumnStatusNames = ($scope.uiState.columnStatus || []).map(function(col){return col.name});
            inputColumns.forEach(function(col) {
                var oldCSNIdx = oldColumnStatusNames.indexOf(col.name);
                if (oldCSNIdx >= 0) {
                    newColumnStatus.push(angular.extend($scope.uiState.columnStatus[oldCSNIdx], col));
                } else {
                    newColumnStatus.push(angular.copy(col));
                }
            });
            $scope.uiState.columnStatus = newColumnStatus;
        }

        function onColumnStatusChanged() {
            if (!$scope.params) {
                return;
            }
            if(!$scope.params.selectAllColumns) {
                // we only update keys if !selectAllColumns in order to avoid dirtying the recipe on input schema changes since it doesn't matter when selectAllColumns
                $scope.params.keys = ($scope.uiState.columnStatus || [])
                        .filter(function(col){return !col.ignore})
                        .map(function(col){return {column:col.name}});
            }
            $scope.updateRecipeStatusLater();
        }

        function onScriptChanged(nv) {
             if (nv) {
                loadParamsFromScript($scope.script.data);
                DKUtils.reflowNext();
                DKUtils.reflowLater();
                $scope.hooks.updateRecipeStatus();
            }
        }

        // UI:
        $scope.uiState = {
            currentStep: 'group',
            outputColumnNamesOverridable: true,
        };

        $scope.hooks.onRecipeLoaded = function(){
            Logger.info("On Recipe Loaded");
            validateFilters();
            $scope.$watch("script.data", onScriptChanged, true); // this will call $scope.hooks.updateRecipeStatus when ready
        };

        $scope.enableAutoFixup();
        $scope.specificControllerLoadedDeferred.resolve();
        $scope.$watchCollection("recipe.inputs.main.items", function() {
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                .then(_ => resyncWithInputSchema());
        });
        $scope.$watch("uiState.columnStatus", onColumnStatusChanged, true);
        $scope.$watch("params.globalCount", $scope.updateRecipeStatusLater);
        $scope.$watch("params.selectAllColumns", onColumnStatusChanged);
    });
})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    const SELECTION_MODE = {
        ALL: "ALL",
        EXPLICIT: "EXPLICIT"
    };

    app.controller("WindowRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller) {
        $scope.recipeType = "window";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                $scope.maybeSetNewDatasetName(niceInputName + "_windows");
            }
        };
    });

    app.controller("WindowRecipeController", function($scope, $stateParams, DataikuAPI, $q,Dialogs, TopNav, ContextualMenu, PartitionDeps, $rootScope,
     $timeout, DKUtils, Expressions, Logger, $controller,  RecipesUtils, CreateModalFromTemplate, Fn, DatasetTypesService, translate) {
        var groupingCtrl = $controller('GroupingRecipeController', {$scope: $scope}); //Controller inheritance
        var visualCtrl = groupingCtrl.visualCtrl; //FIXME ugly: inheritance cannot be expressed this way
        $scope.aggregateUsabilityFlag = "usableInWindow";
        $scope.uiState = $scope.uiState || {};
        const scriptData = JSON.parse($scope.script.data)
        $scope.uiState.retrieveAll = scriptData.retrievedColumnsSelectionMode === SELECTION_MODE.ALL;

        $scope.$watch('selection.filteredObjects', function () {
            if ($scope.uiState.retrieveAll && $scope.selection.filteredObjects) {
                $scope.selection.filteredObjects.forEach((obj) => obj.value = true);
            }
        });

        function filterUnused(vals) {
            return vals.filter(function(val){
                return $scope.uiState.retrieveAll || !$scope.uiState.hideUseless || $scope.columnHasSomeComputation(val);
            });
        }
        
        $scope.selection = {
            'customFilter': filterUnused,
            'customFilterWatch': ['uiState.hideUseless', 'uiState.retrieveAll']
        }

        $scope.simpleAggregationTypes =  [
            {name: "value", opType: "RETRIEVE", label: translate("WINDOW_RECIPE.AGGREGATIONS.RETRIEVE", "Retrieve"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.RETRIEVE_DESCRIPTION", "Retrieve original value")},
            {name: "min", label: translate("WINDOW_RECIPE.AGGREGATIONS.MIN", "Min")},
            {name: "max", label : translate("WINDOW_RECIPE.AGGREGATIONS.MAX", "Max")},

            {name: "avg", label: translate("WINDOW_RECIPE.AGGREGATIONS.AVG", "Avg")},
            {name: "sum", label: translate("WINDOW_RECIPE.AGGREGATIONS.SUM", "Sum")},
            {name: "stddev", label: translate("WINDOW_RECIPE.AGGREGATIONS.STD_DEV", "Std. dev.")},
            {name: "count", label: translate("WINDOW_RECIPE.AGGREGATIONS.COUNT", "Count"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.COUNT_DESCRIPTION", "Count non-null"), separatorAfter: true},
            
            {name: "first", label: translate("WINDOW_RECIPE.AGGREGATIONS.FIRST", "First")},
            {name: "last", label: translate("WINDOW_RECIPE.AGGREGATIONS.LAST", "Last")},
            {name: "concat", label: translate("WINDOW_RECIPE.AGGREGATIONS.CONCAT", "Concat"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.CONCAT_DESCRIPTION", "Concatenate values in one string")},
        ];
        $scope.lagAggregationTypes = [
            {name: "lag", label: translate("WINDOW_RECIPE.AGGREGATIONS.LAG", "Lag"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.LAG_DESCRIPTION", "Value in a previous row")},
            {name: "lagDiff", opType: "LAG_DIFF", label: translate("WINDOW_RECIPE.AGGREGATIONS.LAD_DIFF", "LagDiff"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.LAG_DIFF_DESCRIPTION", "Difference with a previous row")},
        ];
        $scope.leadAggregationTypes = [
            {name: "lead", label: translate("WINDOW_RECIPE.AGGREGATIONS.LEAD", "Lead"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.LEAD_DESCRIPTION", "Value in a following row")},
            {name: "leadDiff", opType: "LEAD_DIFF", label: translate("WINDOW_RECIPE.AGGREGATIONS.LEAD_DIFF", "LeadDiff"), tooltip: translate("WINDOW_RECIPE.AGGREGATIONS.LEAD_DIFF_DESCRIPTION", "Difference with a following row")},
        ];
        $scope.filteredSimpleAggTypes = $scope.simpleAggregationTypes.filter((agg) => $scope.uiState.retrieveAll ? agg.name !== "value" : true);
        $scope.aggregationTypes = $scope.simpleAggregationTypes.concat($scope.lagAggregationTypes,$scope.leadAggregationTypes);

        function makeSelectionTest(f) {
            return function() {
                if (!$scope.selection||!$scope.selection.allObjects) { return false }
                return $scope.selection.allObjects.map(f).reduce(Fn.OR,false);
            }
        }
        $scope.shouldDisplayDateUnit = makeSelectionTest.call(null,function(o) {return (o.leadDiff || o.lagDiff) && DatasetTypesService.isTemporalType(o.type)});

        $scope.displayRetrieveAll = function() {
            $scope.filteredSimpleAggTypes = $scope.simpleAggregationTypes.filter((agg) => $scope.uiState.retrieveAll ? agg.name !== "value" : true);
            $scope.selection.filteredObjects.forEach((obj) => obj.value = true);
            $scope.params.retrievedColumnsSelectionMode = $scope.uiState.retrieveAll ? SELECTION_MODE.ALL : SELECTION_MODE.EXPLICIT;            
        }

        $scope.addWindow = function() {
            $scope.params.windows = $scope.params.windows || [];
            $scope.params.windows.push({
                prefix: $scope.params.windows.length ? "w"+($scope.params.windows.length+1) : ""
            });
        };

        $scope.removeWindow = function(index) {
            $scope.params.windows.splice(index,1);
        };

        $scope.allWindowsOrdered = function() {
            var ret = true;
            (($scope.params || {}).windows || []).forEach(function(w){
                ret = w.enableOrdering && w.orders && w.orders.length && ret;
            });
            return ret;
        }

        $scope.addPartitioningColumn = function(win) {
            win.partitioningColumns = win.partitioningColumns || [];
            var columns = $scope.getColumnsWithComputed();
            var colName;
            if (columns) {
                var columnNames = columns.map(function(col){return col.name});
                //TODO smarter autoselect ?
                for (var i = 0; i < columns.length; ++i) {
                    if (win.partitioningColumns.indexOf(columnNames[i]) < 0) {
                        colName = columnNames[i];
                        break;
                    }
                }
                colName = colName || columnNames[0]; //TODO smarter autoselect
            }
            win.partitioningColumns.push(colName);
        };

        $scope.removePartitioningColumn = function(win, index) {
            win.partitioningColumns.splice(index, 1);
        };

        $scope.addOrderColumn = function(win) {
            win.orders = win.orders || [];
            var columns = $scope.getColumnsWithComputed();
            var colName;
            if (columns) {
                var columnNames = columns.map(function(col){return col.name});
                var orderColumns = win.orders.map(function(order){return order.column});
                //TODO smarter autoselect => prefer dates
                for (var i = 0; i < columns.length; ++i) {
                    if (orderColumns.indexOf(columnNames[i]) < 0) {
                        colName = columnNames[i];
                        break;
                    }
                }
                colName = colName || columnNames[0];
            }
            win.orders.push({column: colName});
        };

        $scope.getOrderColumnType = function(win) {
            if (!win.orders || !win.orders.length) {
                return;
            }
            const colName = win.orders[0].column;
            //TODO build index
            const col = $scope.getColumnsWithComputed().find(c => c.name == colName);
            return col && col.type;
        };

        $scope.removeOrderColumn = function(win, index) {
            win.orders.splice(index, 1);
        };

        $scope.onResyncWithInputSchema = function() {
            var inputColumnsWithComputed = $scope.getColumnsWithComputed(true);
            var inputColumnsWithComputedNames = inputColumnsWithComputed.map(function(col){return col.name});

            (($scope.params || {}).windows || []).forEach(function(win) {
                var i = (win.partitioningColumns || []).length;
                while (i--) {
                    if (!win.partitioningColumns[i] || inputColumnsWithComputedNames.indexOf(win.partitioningColumns[i]) == -1) {
                        win.partitioningColumns.splice(i, 1);
                    }
                }
                i = (win.orders || []).length;
                while (i--) {
                    if (!win.orders[i] || !win.orders[i].column || inputColumnsWithComputedNames.indexOf(win.orders[i].column) == -1) {
                        win.orders.splice(i, 1);
                    }
                }
            });
        };

        $scope.isWindowFrameRowsLimitationInvalid = (window) =>  window.limitFollowing && window.limitPreceding && (window.followingRows + window.precedingRows < 0)

        $scope.uiState.currentStep = 'windows';

        $scope.$watch('topNav.tab',function(nv){
            if (nv == 'settings') {
                $timeout(function() {
                    $scope.$broadcast('redrawFatTable');
                });
            }
        });

        $scope.$watch("params.windows", $scope.updateRecipeStatusLater, true);

    });
})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("SamplingRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller) {
        $scope.recipeType = "sampling";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                $scope.maybeSetNewDatasetName(niceInputName + "_filtered");
            }
        };
    });


    // Recipe edition page controller
    app.controller("SamplingRecipeController", function ($scope, $stateParams, WT1, $q, DataikuAPI, TopNav, Dialogs, PartitionDeps,
                                                         RecipesUtils, $controller, Logger, SamplingData,
                                                         CreateModalFromTemplate, DatasetUtils, RecipeComputablesService) {
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope}); //Controller inheritance
        this.visualCtrl = visualCtrl;

        $scope.SamplingData = SamplingData;

        var defaultSampling = {
            "samplingMethod": "FULL",
            "maxRecords": 30000,
            "targetRatio": 0.1
        };

        $scope.hooks.getPayloadData = function() {
            return angular.toJson($scope.filter);
        };

        $scope.hooks.preRunValidate = function() {
            var deferred = $q.defer();
            $scope.hooks.updateRecipeStatus().then(function(data) {
                if (data) {
                    Logger.info("preRunValidate failed",data);
                    var validationData = {error: false, messages: []};
                    if (data.filter.invalid) {
                        validationData.error=true;
                        data.filter.errorMessages.forEach(function(m) {validationData.messages.push({"message":m});});
                    }
                    if (data.output.invalid) {
                        validationData.error=true;
                        data.output.errorMessages.forEach(function(m) {validationData.messages.push({"message":m});});
                    }
                    deferred.resolve(validationData);
                } else {
                    deferred.resolve({error: false});
                }
            },
            function(data){
                Logger.error("Error when getting status", data);
                setErrorInScope.bind($scope);
                deferred.reject("Validation failed");
            });
            return deferred.promise;
        };

        var superSave = $scope.hooks.save;
        $scope.hooks.save = function() {
            return superSave().then(function(){
                origSelection = angular.copy($scope.selection);
            });
        };

        var superRecipeIsDirty = $scope.hooks.recipeIsDirty;
        var origSelection;
        $scope.hooks.recipeIsDirty = function() {
            if (superRecipeIsDirty()) return true;
            // no need to compare the contents of the filter object if it is disabled
            var selectionEquals = angular.equals($scope.selection, origSelection);
            return !selectionEquals;
        };

        $scope.hooks.onRecipeLoaded = function(){
            Logger.info("On Recipe Loaded");
            origSelection = angular.copy($scope.selection);
            //keep params for dirtyness detection
            visualCtrl.saveServerParams();
            $scope.hooks.updateRecipeStatus();
            $scope.$watch("recipe.params",  $scope.updateRecipeStatusLater, true);
            $scope.$watch("filter", $scope.updateRecipeStatusLater, true);
        };

        $scope.hooks.updateRecipeStatus = function() {
            var deferred = $q.defer();
            var payload = $scope.hooks.getPayloadData();
            $scope.updateRecipeStatusBase(false, payload).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.resyncSchema = function() {
            Dialogs.confirmPositive($scope,
                'Resynchronize schema',
                'The schema of "'+$scope.recipe.inputs[0]+'" will be copied to "'+$scope.recipe.outputs[0]+'". Are you sure you want to continue ?'
            )
            .then(function() {
                DataikuAPI.flow.recipes.basicResyncSchema($stateParams.projectKey,
                        $scope.hooks.getRecipeSerialized()).error(setErrorInScope.bind($scope));
            });
        };

        $scope.availableOutputDatasets = [];
        $scope.convertToSplitRecipe = function () {
            function doConvertToSplitRecipe(secondOutputDataset) {
                DataikuAPI.flow.recipes.visual.convertSamplingRecipeToSplitRecipe($stateParams.projectKey, $scope.recipe, secondOutputDataset)
                    .then(function () {
                        location.reload();
                    }, setErrorInScope.bind($scope));
            }
            if ($scope.hooks.recipeIsDirty()) {
                $scope.hooks.save();
            }
            CreateModalFromTemplate("/templates/recipes/io/output-selection-modal.html", $scope, null, function(modalScope) {
                $controller("_RecipeOutputNewManagedBehavior", {$scope: modalScope});
                modalScope.singleOutputRole = {name:"main", arity:"UNARY", acceptsDataset:true};

                DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, $scope.recipe.type).then(function(data) {
                    const alreadyInOutput = function(computable) {
                        if ($scope.recipe && $scope.recipe.outputs && $scope.recipe.outputs.main && $scope.recipe.outputs.main.items) {
                            return $scope.recipe.outputs.main.items.filter(item => item.ref == computable.smartName).length > 0;
                        } else {
                            return false;
                        }
                    };
                    $scope.availableOutputDatasets = data[1].filter(function(computable) {
                        return computable.usableAsOutput['main'].usable && !computable.alreadyUsedAsOutputOf && !alreadyInOutput(computable);
                    });
                });

                DataikuAPI.datasets.getManagedDatasetOptions($scope.recipe, 'main').success(function(data) {
                    modalScope.setupManagedDatasetOptions(data);
                });

                modalScope.ok = function(_dismissModal, force = false) {
                    if (modalScope.io.newOutputTypeRadio == 'select') {
                        if (!modalScope.io.existingOutputDataset) return;
                        doConvertToSplitRecipe(modalScope.io.existingOutputDataset);
                    } else {
                        const creationSettings = {
                            connectionId : modalScope.newOutputDataset.connectionOption.id,
                            specificSettings : {
                                formatOptionId : modalScope.newOutputDataset.formatOptionId,
                                overrideSQLCatalog: modalScope.newOutputDataset.overrideSQLCatalog,
                                overrideSQLSchema: modalScope.newOutputDataset.overrideSQLSchema
                            },
                            partitioningOptionId : modalScope.newOutputDataset.partitioningOption
                        };

                        const checkPromise = force
                            ? $q.resolve({data: {messages: []}})
                            : DataikuAPI.datasets.checkNameSafety($stateParams.projectKey, modalScope.newOutputDataset.name, creationSettings)

                        checkPromise.then(({data}) => {
                            modalScope.uiState.backendWarnings = data.messages;
                            return !data.messages || !data.messages.length;
                        }).then((proceed) => {
                            if(proceed) {
                                return DataikuAPI.datasets.newManagedDataset($stateParams.projectKey, modalScope.newOutputDataset.name, creationSettings).then(({data: dataset}) => {
                                    RecipeComputablesService.getComputablesMap(modalScope.recipe, modalScope).then(function(map){
                                        modalScope.setComputablesMap(map);
                                        doConvertToSplitRecipe(dataset.name);
                                    }, setErrorInScope.bind(modalScope));
                                    WT1.event("create-dataset", {
                                        connectionType: (modalScope.newOutputDataset && modalScope.newOutputDataset.connectionOption) ? modalScope.newOutputDataset.connectionOption.connectionType : "unknown",
                                        partitioningFrom: modalScope.newOutputDataset ? modalScope.newOutputDataset.partitioningOption : "unknown",
                                        recipeType: modalScope.recipe ? modalScope.recipe.type : "unknown"
                                    });
                                })
                            }
                        }).catch(setErrorInScope.bind($scope));
                    }
                };
            });
        };

        $scope.filter = {};
        if ($scope.script && $scope.script.data) {
            $scope.filter = JSON.parse($scope.script.data);
        }

        $scope.fromElasticSearchExportQuery = () => $scope.recipeAdditionalParams && !!$scope.recipeAdditionalParams.elasticSearchQuery;
        if ($scope.fromElasticSearchExportQuery()) {
            Logger.info("Enabling filter because it was created via 'export to filter recipe' from an ElasticSearch dataset");
            $scope.filter.enabled = true;
        }

        $scope.params = $scope.recipe.params;

        //TODO @sampling, why is this necessary?
        $scope.selection = $scope.recipe.params.selection;
        $scope.$watch("selection", function(nv, ov) {
            Logger.info("Selection changed", nv);
            if (nv) {
                $scope.recipe.params.selection = nv;
            }
        }, true);

        $scope.$watch("filter.uiData.mode", function(nv, ov) {
            Logger.info("Filter mode changed", nv);
            // For Elastic Search query string filter, we want to default to FULL sampling method
            if (nv && nv === "ES_QUERY_STRING") {
                $scope.selection.samplingMethod = "FULL";
            }
        });

        $scope.enableAutoFixup();
        $scope.specificControllerLoadedDeferred.resolve();
    });
})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("VStackRecipeCreationController", function($scope, $controller, $stateParams, DataikuAPI, Fn, RecipeComputablesService) {
        $scope.recipeType = "vstack";
        $scope.recipe = {
            type: 'vstack',
            projectKey: $stateParams.projectKey,
            inputs: {
                main: {
                    items: []
                }
            },
            outputs: {
                main: {
                    items: []
                }
            }
        };

        RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map) {
            $scope.setComputablesMap(map);
        });

        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                $scope.maybeSetNewDatasetName(niceInputName + "_stacked");
            }
        };

        $scope.getCreationSettings = function () {
            return {virtualInputs: $scope.recipe.inputs.main.items.map(input => input.ref)};
        };


        $scope.formIsValid = function() {
            return $scope.recipe.inputs.main.items.length &&
            (
                $scope.io.newOutputTypeRadio == 'create' && $scope.newOutputDataset && $scope.newOutputDataset.name && $scope.newOutputDataset.connectionOption && $scope.isDatasetNameUnique($scope.newOutputDataset.name)
                || $scope.io.newOutputTypeRadio == 'select' && $scope.io.existingOutputDataset
            );
        };

        $scope.showOutputPane = function() {
            return $scope.recipe.inputs.main.items.length > 0;
        };

        $scope.$watchCollection('recipe.inputs.main.items', function(nv) {
            if (nv && nv.length) {
                $scope.io.inputDataset = nv[0].ref;
            }
        })
    });


    app.controller("VStackRecipeController", function ($scope, $controller, $q, $stateParams, DataikuAPI, Dialogs, PartitionDeps,
        CreateModalFromTemplate, RecipesUtils, Logger, DatasetUtils) {
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope});

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) return $q.reject("payload not ready");
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, {reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan}).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                if ($scope.recipeStatus.outputSchema) {
                    $scope.params.postFilter = $scope.params.postFilter || {};
                    $scope.params.postFilter.$status = $scope.params.postFilter.$status || {};
                    $scope.params.postFilter.$status.schema = $scope.recipeStatus.outputSchema;
                }
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.hooks.getPayloadData = function () {
            if (!$scope.params) {
                return null;
            }
            var us = $scope.unionSchema || [];
        	if ($scope.params.mode != 'FROM_INDEX' && $scope.params.mode != 'REMAP') {
                $scope.params.selectedColumns = us.filter(function(column){
                    return column.selected;
                })
                .map(function(column) {
                    return column.name
                });
            }
            return angular.toJson($scope.params);
        };

        $scope.showNewInputModal = function() {
            CreateModalFromTemplate("/templates/recipes/fragments/virtual-input-modal.html", $scope);
        };

        $scope.selected = {
        };

        $scope.selectedFromIndex = {
        };

        $scope.addDataset = function(datasetName) {
            if (RecipesUtils.getInput($scope.recipe, "main", datasetName) == null) {
                RecipesUtils.addInput($scope.recipe, "main", datasetName);
            }
            var inputNames = RecipesUtils.getInputsForRole($scope.recipe, "main").map(function(input){return input.ref});
            var inputDesc = {
                index: inputNames.indexOf(datasetName),
                originLabel: datasetName,
                preFilter: {},
                columnsMatch: []
            };

            DatasetUtils.updateDatasetInComputablesMap($scope, datasetName, $stateParams.projectKey, $stateParams.projectKey) // necessary when manually remapping otherwise the dataset' s schema is not up to date
            .then(() => {
                buildInitialColumnsMatchForInput(inputDesc);
                $scope.params.virtualInputs.push(inputDesc);
                $scope.updateColumnsSelection();
                $scope.updateRecipeStatusLater();
            });
        }

        $scope.removeDataset = function(index) {
            removeDatasets([index]);
        };

        $scope.updateSelectAllColumns = function() {
            for (let i = 0; i < $scope.unionSchema.length; i++) {
                if ($scope.unionSchema[i]) {
                    $scope.unionSchema[i].selected = $scope.selected.all;
                }
            }
            $scope.selected.any = $scope.selected.all;
        };

        $scope.updateSelectAllColumnsFromIndex = function() {
            for (let i = 0; i < $scope.getSelectableColumns.length; i++) {
                if (!$scope.selectedColumns[i]) {
                    $scope.selectedColumns[i] = {name: $scope.getSelectableColumns[i][0]};
                }
                $scope.selectedColumns[i].selected = $scope.selectedFromIndex.all;
            }
            $scope.selectedFromIndex.any = $scope.selectedFromIndex.all;
        };

        $scope.updateGlobalSelectionStatus = function() {
            var all = true, any = false;
            for (let i = 0; i < $scope.unionSchema.length; i++) {
                if ($scope.unionSchema[i] && $scope.unionSchema[i].selected) {
                    any = true;
                } else {
                    all = false;
                }
            }
            $scope.selected = {
                all: all, any: any
            };
            all = true;any=false;
            if ($scope.params.mode == 'FROM_INDEX') {
                for (let i = 0 ; i < $scope.getSelectableColumns.length; i++)  {
                    const selectableCol = $scope.getSelectableColumns[i];
                    const selCol = $scope.selectedColumns.find(col => col && selectableCol.indexOf(col.name) >= 0);
                    if (selCol && selCol.selected) {
                        any = true;
                    } else {
                        all = false;
                    }
                }
            } else {
                for (let i = 0 ; i < $scope.selectedColumns.length; i++) {
                    if ($scope.selectedColumns[i] && $scope.selectedColumns[i].selected) {
                        any = true;
                    } else {
                        all = false;
                    }
                }
            }
            $scope.selectedFromIndex = {
                all: all, any: any
            };
        };

        // gets the dataset name from the index within the virtual inputs
        $scope.getDatasetName = function(virtualIndex) {
            var dataset = $scope.params.virtualInputs[virtualIndex];
            return $scope.getDatasetNameFromRecipeInputIndex(dataset.index);
        };

        $scope.getColumnList = function(datasetIndex) {
            var selectedColumns;
            if (datasetIndex != null) {
                selectedColumns = $scope.getColumns($scope.getDatasetNameFromRecipeInputIndex(datasetIndex));
            } else {
                selectedColumns = [];
            }
            return selectedColumns;
        };

        $scope.selectedColumns = [];// Used to decouple from $scope.params.selectedColumns

        $scope.addColumn = function () {
            let newColumnName = null;
            // find a name that has not been used yet
            $scope.params.virtualInputs.forEach(function(virtualInput){
                if (newColumnName) return;
                let inputColumns = $scope.getColumns(virtualInput.originLabel);
                for (const inputColumn of inputColumns) {
                    if (virtualInput.columnsMatch.indexOf(inputColumn.name) == -1) {
                        if (!newColumnName) newColumnName = inputColumn.name;
                        break;
                    }
                }
            });
            if (!newColumnName) { // No valid name found, Col-X instead
                let idx = 1;
                let possibleColName = "Col-" + idx;
                while ($scope.params.selectedColumns.indexOf(possibleColName) != -1) {
                    idx++;
                    possibleColName = "Col-" + idx;
                }
                newColumnName = possibleColName;
            }
            $scope.params.selectedColumns = $scope.params.selectedColumns.slice(); // Will enforce ng2-values-list two-way data binding
            $scope.params.selectedColumns.push(newColumnName);
            $scope.unionSchema.push({'name':newColumnName, 'selected':true });
            $scope.params.virtualInputs.forEach(function(virtualInput) { // building an index based columnsMatch
                const inputColumns = $scope.getColumns(virtualInput.originLabel);
                if (inputColumns.some(col => col.name === newColumnName)) {
                    virtualInput.columnsMatch.push(newColumnName);
                } else {
                    virtualInput.columnsMatch.push('');
                }
            });
        };

        $scope.removeColumn = function (columnIndex) {
            // We're not removing the item from params.selectedColumns as editable-list already does it.
            if (columnIndex > -1) {
                const colName = $scope.params.selectedColumns[columnIndex];
                ($scope.unionSchema.find(col => col.name == colName) || {}).selected = false;
                $scope.params.virtualInputs.forEach(virtualInput => {virtualInput.columnsMatch.splice(columnIndex, 1);})
            }
        };

        $scope.reorderColumns = function (event) {
            $scope.params.virtualInputs.forEach(virtualInput => {
                moveItemInArray(virtualInput.columnsMatch, event.previousIndex, event.currentIndex);
            })
        }

        $scope.removeAllColumns = function() {
            $scope.params.selectedColumns = [];
            $scope.params.selectedColumnsIndexes = [];
            $scope.params.virtualInputs.forEach(virtualInput => {virtualInput.columnsMatch = []});
            $scope.unionSchema.forEach(column => { column.selected = false });
        };

        $scope.sortableOptions = {
            stop: function(e, ui) {
                if (ui.item.sortable.dropindex != null) {
                    moveSelectableColumn(ui.item.sortable.index, ui.item.sortable.dropindex);
                }
            },
            axis:'y', cursor: 'move', cancel:'', handle: '.handle-row'
        };

        function moveSelectableColumn(initialIndex, targetIndex) {
            function shiftArray(inputArray, initialIndex, targetIndex) {
                if (inputArray) inputArray.splice(targetIndex, 0, inputArray.splice(initialIndex, 1)[0]);
            }
            shiftArray($scope.params.selectedColumns, initialIndex, targetIndex);
            shiftArray($scope.params.selectedColumnsIndexes, initialIndex, targetIndex);
            shiftArray($scope.selectedColumns, initialIndex, targetIndex);
            for (const virtualInput of $scope.params.virtualInputs) {
                shiftArray(virtualInput.columnsMatch, initialIndex, targetIndex);
            }
        }

        $scope.useAsReference = function(referenceIndex) {
            let selectedColumnsNames = [];
            $scope.params.copySchemaFromDatasetWithName = referenceIndex;
            let selectedColumns = $scope.getColumns($scope.params.copySchemaFromDatasetWithName);
            selectedColumnsNames = selectedColumns.map(col => col.name);
            updateSelectableColumns();
            $scope.params.selectedColumns = selectedColumnsNames;
            $scope.selectedColumns = [];
            $scope.params.selectedColumnsIndexes = [];
            for (let index in selectedColumnsNames) {
                $scope.selectedColumns.push({
                    name:selectedColumnsNames[index],
                    selected:true
                });
                $scope.params.selectedColumnsIndexes.push(index);
            }
            $scope.updateGlobalSelectionStatus();
        }

        $scope.syncSelectedColumns = function() {
            $scope.params.selectedColumns = [];
            $scope.params.selectedColumnsIndexes = [];
            for (let columnIndex = 0 ; columnIndex < $scope.selectedColumns.length; columnIndex++) {
                if (($scope.selectedColumns[columnIndex] || {}).selected) {
                    $scope.selectedColumns[columnIndex].name = $scope.selectedColumns[columnIndex].name || $scope.getSelectableColumns[columnIndex][0];
                    $scope.params.selectedColumns.push($scope.selectedColumns[columnIndex].name);
                    $scope.params.selectedColumnsIndexes.push(columnIndex);
                    $scope.unionSchema[columnIndex].selected = true;
                } else if ($scope.unionSchema[columnIndex]) {
                    $scope.unionSchema[columnIndex].selected = false;
                }
            }
        }

        function syncSelectors() {
            $scope.selectedColumns = [];
            for (let columnIndex in $scope.params.selectedColumns) {
                let targetColumn = $scope.params.selectedColumnsIndexes[columnIndex];
                $scope.selectedColumns[targetColumn] = {};
                $scope.selectedColumns[targetColumn].selected = true;
                $scope.selectedColumns[targetColumn].name = $scope.params.selectedColumns[columnIndex];
            }
        }

        $scope.updateNewSchema = function(obj){
            $scope.params.selectedColumns = [];
        }

        $scope.columnsSelection = {};

        // gets the dataset name from the index within the recipe's inputs
        $scope.getDatasetNameFromRecipeInputIndex = function(index) {
            var input = $scope.recipe.inputs.main.items[index];
            return input ? input.ref : "";
        };

        $scope.getDatasetColorClass = function(datasetIndex) {
            return 'dataset-color-'+(datasetIndex%6);
        };

        $scope.updateColumnsSelection = function() {
            updateUnionSchema();
            if(!$scope.params) {
                return;
            }

            // clear the columnsMatch value in each input if the mode is not REMAP
            if ($scope.params.mode != 'REMAP') {
                $scope.params.virtualInputs.forEach(vi => { delete vi.columnsMatch; });
            }

            let selectedColumnsNames = [];

            if ($scope.params.mode == 'CUSTOM') {
                selectedColumnsNames = $scope.params.selectedColumns;
            } else if ($scope.params.mode == 'UNION') {
                selectedColumnsNames = $scope.unionSchema.map(function(col) {
                    return col.name;
                });
            } else if ($scope.params.mode == 'FROM_DATASET') {
                $scope.params.copySchemaFromDatasetWithName = $scope.params.copySchemaFromDatasetWithName || $scope.recipe.inputs.main.items[0].ref;
                let selectedColumns = $scope.getColumns($scope.params.copySchemaFromDatasetWithName);
                selectedColumnsNames = selectedColumns.map(col => col.name);
            } else if ($scope.params.mode == 'FROM_INDEX') {
                if ($scope.params.selectedColumnsIndexes && $scope.params.selectedColumns
                    && $scope.params.selectedColumnsIndexes.length == $scope.params.selectedColumns.length) {
                    selectedColumnsNames = $scope.params.selectedColumns;
                } else {
                    $scope.params.copySchemaFromDatasetWithName = $scope.params.copySchemaFromDatasetWithName || $scope.recipe.inputs.main.items[0].ref;
                    let selectedColumns = $scope.getColumns($scope.params.copySchemaFromDatasetWithName);
                    selectedColumnsNames = selectedColumns.map(col => col.name);
                    $scope.useAsReference($scope.params.copySchemaFromDatasetWithName);
                }
                updateSelectableColumns();
                syncSelectors();
                let maxColNb = 0;
                for (let i = 0; i < $scope.recipe.inputs.main.items.length; i++) {
                    maxColNb = Math.max(maxColNb, $scope.getColumns($scope.recipe.inputs.main.items[i].ref).length);
                }
                $scope.selectedColumns.forEach((col, idx) => { if (idx >= maxColNb) { col.selected = false; } });
            } else if ($scope.params.mode == 'REMAP') {
                $scope.params.copySchemaFromDatasetWithName = $scope.params.copySchemaFromDatasetWithName || $scope.recipe.inputs.main.items[0].ref;
                $scope.columnsSelection.possibleColumnNames = $scope.columnsSelection.possibleColumnNames || []
                let selectedColumns = $scope.getColumns($scope.params.copySchemaFromDatasetWithName);
                if ($scope.params.selectedColumns) {
                    selectedColumnsNames = $scope.params.selectedColumns;
                    $scope.unionSchema.forEach(col => { col.selected = false })
                } else {
                    selectedColumnsNames = selectedColumns.map(col => col.name);
                }
                buildInitialColumnsMatch(selectedColumnsNames);
            } else if ($scope.params.mode == 'INTERSECT') {
                let allInputs = RecipesUtils.getFlatInputsList($scope.recipe);
                let selectedColumns = $scope.getColumns(allInputs[0].ref);
                selectedColumnsNames = selectedColumns.map(col => col.name);
                for (var i = 1; i < allInputs.length; i++) {
                    var columnNames = $scope.getColumns(allInputs[i].ref).map(function(col){return col.name;});
                    for (var c = selectedColumnsNames.length - 1; c >= 0; c--) {
                        if (columnNames.indexOf(selectedColumnsNames[c]) < 0) {
                            selectedColumnsNames.splice(c, 1);
                        }
                    }
                }
            }

            $scope.params.selectedColumns = selectedColumnsNames;
            updateSelectedColumns();
            $scope.updateGlobalSelectionStatus(); // keep "select all" checkbox synchronized
        };

        $scope.isColumnsMatch = function(referenceColumns, selectedColumns) {
            if (referenceColumns.length != selectedColumns.length) {
                return false;
            } else {
                for (let columnIndex in referenceColumns) {
                    if (!referenceColumns[columnIndex] || !selectedColumns[columnIndex] || !(selectedColumns[columnIndex] === referenceColumns[columnIndex].name)) {
                        return false;
                    }
                }
                return true;
            }
        }

        function updateSelectableColumns(){
            //returns a 2D table containing possible header names for each index
            let selectableColumns = [];
            for (const input of $scope.params.virtualInputs) {
                var inputColumns = $scope.getColumns(input.originLabel);
                for (let columnIndex in inputColumns) {
                    let thisInput = {};
                    if(selectableColumns[columnIndex]) thisInput = selectableColumns[columnIndex];
                    thisInput[inputColumns[columnIndex].name] = 1;
                    selectableColumns[columnIndex] = thisInput;
                }
            }
            $scope.getSelectableColumns = selectableColumns.map(record => {
                let column = [];
                for (var key in record){
                    column.push(key);
                }
                return column;
            });
        }

        function buildInitialColumnsMatch(selectedColumnsNames) {
            for (const input of $scope.params.virtualInputs) {
                const inputColumns = $scope.getColumns(input.originLabel);
                let columnsMatch = [];
                for (let indexColumn in selectedColumnsNames) {
                    if (input.columnsMatch && indexColumn < input.columnsMatch.length) {
                        columnsMatch.push(input.columnsMatch[indexColumn]);
                    } else if (indexColumn < inputColumns.length) {
                        columnsMatch.push(inputColumns[indexColumn].name);
                    } else {
                        columnsMatch.push('');
                    }
                }
                input.columnsMatch = columnsMatch;
            }
        }

        function buildInitialColumnsMatchForInput(input) {
            const inputColumns = $scope.getColumns(input.originLabel);
            let columnsMatch = [];
            for (let indexColumn in $scope.params.selectedColumns) {
                if (indexColumn < inputColumns.length) {
                    columnsMatch.push(inputColumns[indexColumn].name);
                } else {
                    columnsMatch.push('');
                }
            }
            input.columnsMatch = columnsMatch;
        }

        function removeDatasets(virtualIndices) {
            /* removes a dataset from recipe inputs if it not used anymore */
            var updateRecipeInputs = function (index) {
                var used = false;
                $scope.params.virtualInputs.forEach(function(vi) {
                    if (vi.index == index) {
                        used = true;
                    }
                });
                if (!used) {
                    RecipesUtils.removeInput($scope.recipe, "main", $scope.getDatasetNameFromRecipeInputIndex(index));
                    $scope.params.virtualInputs.forEach(function(vi) {
                        if (vi.index > index) {
                            vi.index--;
                        }
                    });
                }
            }

            virtualIndices.sort().reverse();
            virtualIndices.forEach(function(virtualIndex) {
                var recipeInputsIndex = $scope.params.virtualInputs[virtualIndex].index;
                $scope.params.virtualInputs.splice(virtualIndex, 1);
                updateRecipeInputs(recipeInputsIndex);
            });


            $scope.updateColumnsSelection();
            if ($scope.params.mode == 'FROM_INDEX') {
                $scope.syncSelectedColumns();
            }
            updateSelectableColumns();
            updateSelectedColumns();
            $scope.hooks.updateRecipeStatus();
        }

        function updateSelectedColumns() {
            if ($scope.params.selectedColumns) {
                $scope.unionSchema.forEach(col => {
                    col.selected = $scope.params.selectedColumns.indexOf(col.name) >= 0;
                });
            }
        }

        function updateUnionSchema () {
            $scope.flatInputRefs = [];

            var columns = [];
            var columnNames = [];
            $scope.unionSchema = $scope.unionSchema || [];
            RecipesUtils.getFlatInputsList($scope.recipe).forEach(function(input) {
                $scope.flatInputRefs.push(input.ref);
                $scope.getColumns(input.ref).forEach(function(column) {
                    if (columnNames.indexOf(column.name) < 0) {
                        columns.push(column);
                        columnNames.push(column.name);
                    }
                });
            });
            $scope.unionSchema = columns;
        }

        $scope.uiState = {
            currentStep: 'selectedColumns'
        };

        $scope.enableAutoFixup();

        function onScriptChanged(nv, ov) {
            if (nv) {
                if ($scope.script.data) {
                    $scope.params = JSON.parse($scope.script.data);

                    $scope.updateColumnsSelection();

                    visualCtrl.saveServerParams(); //keep for dirtyness detection

                    $scope.updateGlobalSelectionStatus();
                    $scope.hooks.updateRecipeStatus();
                }
            }
        }

        $scope.hooks.onRecipeLoaded = function(){
            Logger.info("On Recipe Loaded");

            $scope.$watch("script.data", onScriptChanged);
            $scope.$watch("recipe.inputs", function() {
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, $stateParams.projectKey)
                    .then(_ => () => $scope.updateColumnsSelection());
            }, true);
            $scope.$watch("params.postFilter", $scope.updateRecipeStatusLater, true);
            $scope.$watch("params.virtualInputs", $scope.updateRecipeStatusLater, true);
            $scope.$watch("params.selectedColumns", $scope.updateRecipeStatusLater, true);
            $scope.$watch("params.mode", () => $scope.updateColumnsSelection(), true);
            // don't pass $scope.updateRecipeStatus as the callback, because it will get parameters which are absolutely not what is expected:
            $scope.$watch("unionSchema", function(){$scope.updateRecipeStatusLater()}, true); //call updateRecipeStatus without args!
            $scope.$watch("recipe.outputs", function(){
                var outputs = RecipesUtils.getOutputsForRole($scope.recipe, "main");
                if (outputs.length == 1) {
                    $scope.outputDatasetName = outputs[0].ref;
                }
                $scope.updateRecipeStatusLater();
            }, true);

            $scope.updateColumnsSelection();
            onScriptChanged($scope.script.data);
        };

        $scope.specificControllerLoadedDeferred.resolve();
    });

    app.controller("NewVirtualInputController", function ($scope, $stateParams, DatasetUtils) {
        $scope.newInput = {};

        $scope.isValid = function() {
            return !!$scope.newInput.dataset;
        };

        $scope.addInput = function() {
            $scope.addDataset($scope.newInput.dataset);
        };

        DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, "vstack").then(function(data){
            $scope.availableInputDatasets = data[0];
        });
    });

})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("AfgRecipeCreationController", function($scope, $controller, $stateParams, RecipeComputablesService) {
        $scope.recipeType = "generate_features";

        $scope.recipe = {
            type: 'generate_features',
            projectKey: $stateParams.projectKey,
            inputs: {
                main: {
                    items: []
                }
            },
            outputs: {
                main: {
                    items: []
                }
            }
        };

        RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map) {
            // Retrieve all the datasets of the project and store them in $scope.computablesMap.
            // The object computablesMap maps the name of a dataset with its configuration.
            $scope.setComputablesMap(map);
        });

        $scope.inputDatasetsOnly = true;
        $controller("SingleOutputDatasetRecipeCreationController", { $scope: $scope });

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                const niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./, "");
                $scope.maybeSetNewDatasetName(niceInputName + "_feats");
            }
        };

        $scope.getCreationSettings = function() {
            let inputs = [];
            if ($scope.io.inputDataset !== undefined) {
                inputs.push($scope.io.inputDataset);
            }
            return { virtualInputs: inputs };
        };

        $scope.showOutputPane = function() {
            return !!($scope.io.inputDataset);
        };

        $scope.$watchCollection('recipe.inputs.main.items', function(nv) {
            if (nv && nv.length) {
                $scope.io.inputDataset = nv[0].ref;
            }
        })
    });

    app.controller("AfgRecipeController", function($scope, $controller, $q, $stateParams, Logger, $timeout, CreateModalFromTemplate, RecipesUtils, AutoFeatureGenerationRecipeService, Dialogs, DatasetUtils, DataikuAPI) {
        $scope.hooks.onRecipeLoaded = function() {
            Logger.info("On Recipe Loaded");
            $scope.$watch("script.data", onScriptChanged);
            // the onScriptChanged will be called because adding a $watch on the scope triggers an 'initialization' run
        };
        var visualCtrl = $controller('VisualRecipeEditorController', { $scope: $scope }); //Controller inheritance
        $scope.specificControllerLoadedDeferred.resolve();

        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

        $scope.hooks.getPayloadData = function() {
            if (!$scope.params || Object.keys($scope.params).length === 0) return;
            return angular.toJson($scope.params);
        };

        $scope.fixUpParams = function(params) {
            /**
             * Fixes recipe params in case input columns have been deleted or renamed by Removing selected columns that dont exist
             */
            const allColumns = {};
            for (let i = 0; i < params.virtualInputs.length; i++) {
                allColumns[i] = $scope.getAfgColumns(AutoFeatureGenerationRecipeService.getDatasetName(i, params.virtualInputs, $scope.recipe.inputs));
            }
            AutoFeatureGenerationRecipeService.filterSelectedColumns(params, allColumns);
        }

        function onScriptChanged(nv, ov) {
            if (nv) {
                $scope.params = JSON.parse($scope.script.data);
                visualCtrl.saveServerParams(); //keep for dirtyness detection
                $scope.fixUpParams($scope.params);
                $scope.hooks.updateRecipeStatus();
            }
        }

        $scope.uiState = {
            currentStep: 'dataRelationships'
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) {
                return $q.reject("payload not ready");
            }
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, { reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan }).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) {
                    return deferred.reject();
                }
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.$watchCollection("recipe.outputs.main.items", function() {
            const outputs = RecipesUtils.getOutputsForRole($scope.recipe, "main");
            if (outputs.length === 1) {
                $scope.outputDatasetName = outputs[0].ref;
            }
            $scope.updateRecipeStatusLater();
        });

        $scope.getDateColumnsFromDataset = function(datasetName) {
            const allColumns = $scope.getColumns(datasetName);
            return AutoFeatureGenerationRecipeService.getDateColumns(allColumns);
        }

        $scope.showNewDatasetModal = function(virtualIndex) {
            $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
            $scope.editTimeSettings = false;
            $scope.tableOneIndex = virtualIndex;
            $scope.tableTwoIndex = $scope.params.virtualInputs.length;
            CreateModalFromTemplate("/static/dataiku/auto-feature-generation/dataset-modal/dataset-modal.component.html", $scope);
        };

        $scope.showEditCutoffTimeModal = function() {
            $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
            CreateModalFromTemplate("/static/dataiku/auto-feature-generation/cutoff-time-modal/cutoff-time-modal.component.html", $scope);
        }

        $scope.showEditTimeSettingsModal = function(index) {
            $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
            $scope.editTimeSettings = true;
            $scope.tableOneIndex = null;
            $scope.tableTwoIndex = index;
            $scope.timeIndexMode = $scope.timeIndexColumn ? AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DATE_COLUMN.name : AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DEFAULT.name;
            CreateModalFromTemplate("/static/dataiku/auto-feature-generation/dataset-modal/dataset-modal.component.html", $scope);
        }

        $scope.disableEditTimeSettingsModal = function(virtualInputIndex) {
            const datasetName = $scope.getDatasetName(virtualInputIndex);
            const hasDateColumns = AutoFeatureGenerationRecipeService.hasDateColumns(datasetName, $scope.computablesMap);
            return AutoFeatureGenerationRecipeService.disableTimeIndexEdition($scope.params.cutoffTime.mode, hasDateColumns, $scope.params.virtualInputs[virtualInputIndex].timeIndexColumn);
        }

        $scope.disableEditCutoffTimeModal = function(){
            const primaryDatasetName = $scope.getDatasetName(0);
            return !AutoFeatureGenerationRecipeService.hasDateColumns(primaryDatasetName, $scope.computablesMap) && $scope.params.cutoffTime.mode !== AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name;
        }

        $scope.setCutoffTimeTooltip = function(){
            return $scope.disableEditCutoffTimeModal() ? "No date columns found." : null;
        }

        $scope.setTimeIndexTooltip = function(virtualInputIndex) {
            const datasetName = $scope.getDatasetName(virtualInputIndex);
            const hasDateColumns = AutoFeatureGenerationRecipeService.hasDateColumns(datasetName, $scope.computablesMap);
            if(!hasDateColumns) {
                return "No date columns found.";
            } else if (!AutoFeatureGenerationRecipeService.isCutoffTimeDate($scope.params.cutoffTime.mode)) {
                return "Cutoff time is required to configure enrichment dataset time settings.";
            } return null;
        }

        $scope.isCutoffTimeShown = function() {
            return AutoFeatureGenerationRecipeService.isCutoffTimeShown($scope.params.virtualInputs.length, $scope.params.cutoffTime.mode);
        }

        $scope.isTimeIndexDefined = function(index) {
            if (index < $scope.params.virtualInputs.length && $scope.params.virtualInputs[index].timeIndexColumn) {
                return true;
            }
            return false;
        }

        $scope.getTimeIndexValue = function(index) {
            if (index < $scope.params.virtualInputs.length && $scope.params.virtualInputs[index].timeIndexColumn) {
                return $scope.params.virtualInputs[index].timeIndexColumn;
            }
            return AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DEFAULT.label;
        }

        $scope.isTimeWindowDefined = function(index){
            return $scope.params.virtualInputs[index].timeWindows.length;
        }

        $scope.getTimeWindowValue = function(index){
            const timeWindow = $scope.params.virtualInputs[index].timeWindows[0];
            return timeWindow.from + " to " + timeWindow.to + " " + AutoFeatureGenerationRecipeService.TIME_UNITS[timeWindow.windowUnit].label + " before cutoff time";
        }

        $scope.isCutoffTimeDefined = function() {
            if ($scope.params.virtualInputs.length && $scope.params.virtualInputs[0].timeIndexColumn) {
                return true;
            }
            return false;
        }

        $scope.getCutoffTimeValue = function() {
            if ($scope.params.virtualInputs.length && $scope.params.virtualInputs[0].timeIndexColumn) {
                return $scope.params.virtualInputs[0].timeIndexColumn;
            }
            return AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DEFAULT.label;
        }

        $scope.relationshipDesc = AutoFeatureGenerationRecipeService.relationshipDesc;

        $scope.getAfgColumns = function(datasetName) {
            return angular.copy($scope.getColumns(datasetName));
        }

        $scope.getDatasetsList = function() {
            return AutoFeatureGenerationRecipeService.getUsedDatasets($scope.params.virtualInputs, $scope.recipe.inputs);
        }

        $scope.addEmptyCondition = function(relationship, current) {
            const newCondition = {
                column1: {
                    table: relationship.table1,
                    name: $scope.getAfgColumns($scope.getDatasetName(relationship.table1))[0].name
                },
                column2: {
                    table: relationship.table2,
                    name: $scope.getAfgColumns($scope.getDatasetName(relationship.table2))[0].name
                },
                type: 'EQ'
            };
            relationship.on = relationship.on || [];
            relationship.on.push(newCondition);
            if (current) {
                current.condition = newCondition;
            }
            //  Update recipe status to validate the conditions of the relationship
            if (relationship.on.length === 1) {
                $scope.hooks.updateRecipeStatus();
            }
        };

        $scope.removeCondition = function(scope, relationship, condition) {
            if (scope.current && scope.current.condition === condition) {
                scope.current.condition = null;
            }
            const index = relationship.on.indexOf(condition);
            relationship.on.splice(index, 1);
            //  Update recipe status to validate the new conditions
            $scope.hooks.updateRecipeStatus();
        };


        $scope.removeAllConditions = function(scope, relationship) {
            if ( scope.current != null ) {
                scope.current.condition = null;
            }
            relationship.on = [];
            $scope.hooks.updateRecipeStatus();
        };

        const removeDatasets = function (virtualIndexes, deletedDatasetVirtualIndex) {
            AutoFeatureGenerationRecipeService.removeDatasets(virtualIndexes, deletedDatasetVirtualIndex, $scope.params, $scope.recipe, $scope.columnsTabData);
        }

        $scope.removeDataset = function(virtualIndex) {
            let datasetsToBeRemoved = [];
            if ($scope.params.virtualInputs.length > 2) {
                datasetsToBeRemoved = AutoFeatureGenerationRecipeService.getDependantDatasets(virtualIndex, $scope.params.relationships);
            }
            datasetsToBeRemoved.push(virtualIndex);
            if (datasetsToBeRemoved.length === 1) {
                removeDatasets(datasetsToBeRemoved, virtualIndex);
                if ($scope.params.virtualInputs.length === 0) {
                    $scope.showNewDatasetModal(0);
                }
            } else {
                const datasetList = datasetsToBeRemoved.map(function(index) {
                    return $scope.getDatasetName(index);
                })
                Dialogs.confirm($scope,
                    'Remove datasets',
                    'The following datasets will be removed from the recipe:' +
                    '<ul><li>' + datasetList.join('</li><li>') + '</li></ul>'
                )
                    .then(function() {
                        removeDatasets(datasetsToBeRemoved, virtualIndex);
                        if ($scope.params.virtualInputs.length === 0) {
                            $scope.showNewDatasetModal(0);
                        }
                    });
            }
            // Update recipe status to retrieve any schema change
            $scope.hooks.updateRecipeStatus();
        };

        $scope.range = AutoFeatureGenerationRecipeService.range;

        $scope.getDatasetName = function(virtualIndex) {
            return AutoFeatureGenerationRecipeService.getDatasetName(virtualIndex, $scope.params.virtualInputs, $scope.recipe.inputs);
        }

        $scope.showRelationshipEditModal = function(relationship, tab) {
            //check if the modal is already shown
            const newScope = $scope.$new();
            newScope.relationship = relationship;
            newScope.current = {};
            newScope.current.tab = tab || 'conditions'
            newScope.current.condition = null; //no selected condition when the modal is created


            CreateModalFromTemplate("/templates/recipes/visual-recipes-fragments/relationship-edit-modal.html", newScope, "RelationshipEditController", (scope, el) => {
                $timeout(() => {
                        scope.AfgBlockBodyEl = el[0].getElementsByClassName('modal-body')[0];
                    }
                );
            });
        };

        $scope.getDatasetColorClass = AutoFeatureGenerationRecipeService.getDatasetColorClass;

        $scope.getLeftSymbol = AutoFeatureGenerationRecipeService.getLeftSymbolFromRelationship;

        $scope.getRightSymbol = AutoFeatureGenerationRecipeService.getRightSymbolFromRelationship;

        $scope.getRelationshipStyle = AutoFeatureGenerationRecipeService.getRelationshipClass;

        $scope.params = $scope.params || {};

        $scope.columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData($scope.recipe);

        $scope.toggleColumnsForComputation = function(datasetId) {
            $scope.columnsTabData[datasetId].isSectionOpen = !$scope.columnsTabData[datasetId].isSectionOpen;
        }

        $scope.updateSelectedColumns = function(datasetId, selectedColumns) {
            $scope.columnsTabData[datasetId].selectedColumns = angular.copy(selectedColumns);
            // Update recipe status to get new schema and any column limit warnings
            $scope.updateRecipeStatusLater(700);
        }

        $scope.updateSelectedFeatures = function() {
            // Update recipe status to get new schema and any column limit warnings
            $scope.updateRecipeStatusLater(700);
        }

        $scope.getAvailableReplacementDatasets = function(datasets) {
            const primaryDatasetName = AutoFeatureGenerationRecipeService.getDatasetName(0, $scope.params.virtualInputs, $scope.recipe.inputs);
            const primaryDataset = $scope.computablesMap[primaryDatasetName];
            return AutoFeatureGenerationRecipeService.checkDatasetsUsability(datasets, primaryDataset, $scope.outputDatasetName, $scope.recipe, ($scope.appConfig && $scope.appConfig.sparkEnabled) ? $scope.appConfig.sparkEnabled : false);
        }

        $scope.onInputReplaced = function (replacement, virtualIndex) {
            const recipeInputs = $scope.recipe.inputs;
            const oldInput = angular.copy($scope.params.virtualInputs[virtualIndex]);

            const recipeSerialized = angular.copy($scope.recipe);
            const payload = $scope.hooks.getPayloadData();
            let columnsForComputation = [];
            //Get default columns before replacing the input, otherwise the backend returns an empty list of columns
            DataikuAPI.flow.recipes.autofeaturegeneration.getDefaultColumns($stateParams.projectKey, replacement.name, recipeSerialized, payload)
                .success(function (defaultColumns) {
                    columnsForComputation = defaultColumns;
                })
                .error(setErrorInScope.bind($scope))
                .finally(function () {
                    AutoFeatureGenerationRecipeService.replaceVirtualInput($scope.params, virtualIndex, replacement.name, $scope.computablesMap, recipeInputs);
                    $scope.params.virtualInputs[virtualIndex].selectedColumns = columnsForComputation;
                    //Open all the sections of the columns for computation tab
                    $scope.columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData($scope.recipe);
                    DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                        .then(function () {
                            AutoFeatureGenerationRecipeService.resyncSchemas($scope.params, virtualIndex, $scope.computablesMap, recipeInputs, oldInput);
                            $scope.hooks.updateRecipeStatus();
                        })
                });
        }

        $scope.$watch("topNav.tab", function(nv, ov) {
            // Necessary to fix UI bugs on the fatRepeat directive that occur when changing recipe tabs.
            if (nv==="settings") {
                $scope.$broadcast("repaintFatTable");
            }
        });

    });

    /*
    Controller for relationship edit modal
    */
    app.controller("RelationshipEditController", function($scope) {
        $scope.uiState = $scope.uiState || {};
        if ($scope.relationship.on.length === 0) {
            $scope.addEmptyCondition($scope.relationship);
            $scope.current.condition = $scope.relationship.on[0];
        }

        $scope.getColumn = function(condition, columnIdx) {
            const col = !columnIdx || columnIdx === 1 ? 'column1' : 'column2';
            return $scope.getColumnFromName($scope.getDatasetName(condition[col].table), condition[col].name);
        };

        $scope.getColumnFromName = function(datasetName, name) {
            return $scope.getAfgColumns(datasetName).filter(function(col) {
                return col.name === name
            })[0];
        };

        $scope.ok = function(){
            $scope.dismiss();
            //  Update recipe status to validate the conditions of the relationship when a user clicks on the OK button
            $scope.hooks.updateRecipeStatus();
        }

        $scope.$on('$destroy', function() {
            //  Update recipe status to validate the conditions of the relationship when a user clicks outside of the modal
            $scope.hooks.updateRecipeStatus();
        });
    });

    app.directive('afgBlockEmpty', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-block-empty.html',
        };
    });

    app.directive('afgBlockDropdownRelationship', function(AutoFeatureGenerationRecipeService) {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-block-dropdown-relationship.html',
            link: function(scope, element, attrs) {
                scope.relationshipTypes = AutoFeatureGenerationRecipeService.SUPPORTED_RELATIONSHIP_TYPES;
                scope.relationshipIndex = attrs.relationshipIndex;
                scope.setRelationshipType = function(relationship, type) {
                    relationship.type = type;
                    // Update recipe status to get any new schema changes and validate the new relationship type
                    scope.hooks.updateRecipeStatus();
                };

                scope.getRelationshipTypeLabel = function(relationship) {
                    const relationshipDescription = AutoFeatureGenerationRecipeService.getRelationshipDescription(relationship);
                    return relationshipDescription.type;
                }

                scope.getRelationshipTypeDescription = function(relationship) {
                    const relationshipDescription = AutoFeatureGenerationRecipeService.getRelationshipDescription(relationship);
                    return relationshipDescription.description;
                }

                scope.getClass = function(type) {
                    return `{selected: relationship.type === '${type}'}`;
                };
            }
        }
    });

    app.directive('afgConditionsEditor', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-conditions-editor.html'
        };
    });

    app.directive('afgCondition', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-condition.html',
            link: function(scope, element, attrs) {
                scope.relationshipIndex = attrs.relationshipIndex;
                scope.displayRightActions = attrs.displayRightActions;
                scope.actOnConditionClicked = attrs.actOnConditionClicked;
                scope.toggleConditionFocus = function(condition) {
                    if (scope.current.condition === condition) {
                        scope.current.condition = null;
                    } else {
                        scope.current.condition = condition;
                    }
                };
            }
        };
    });

    app.directive('afgBlockWithRelationship', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-block-with-relationship.html',
            link: function(scope, element, attrs) {
                scope.relationshipIndex = attrs.relationshipIndex;
            }
        };
    });
})();
;
(function(){
    'use strict';

    var services = angular.module('dataiku.services');

    // Service to handle Expressions for Range Splits, which represent one or two bands intervals for types: num & dates
    services.factory('RangeExpressions', function(Expressions) {

        // Available date formats for Ranges with Date columns
        var dateFormats = {
            dateWithTimeFormat: "YYYY-MM-DD HH:mm",
            dateFormat: "YYYY-MM-DD"
        };

        // support for initializing/switching between open (> or <) and close (>= or <=) operators (for range mode in split recipe)
        function switchOpenCloseComparisonOperator(operator) {
            if (!operator) return null;
            if (operator.search('<=') > -1) return operator.replace('<=', '< ');
            if (operator.search('>=') > -1) return operator.replace('>=', '> ');
            if (operator.search('<') > -1) return operator.replace('< ', '<=');
            if (operator.search('>') > -1) return operator.replace('> ', '>=');
            return null;
        }

        function isOpenComparisonOperator(operator) {
            if (!operator) return false;
            return (operator.search("=") == -1);
        }

        function getTypeComparisonOperator(operator) {
            if (!operator) return;
            if (operator.search(">") > -1) return 'min';
            if (operator.search("<") > -1) return 'max';
        }

        function modifyColTypeInComparisonOperator(operator, colType) {
            if (!operator) return;
            return initializeComparisonOperator(colType, getTypeComparisonOperator(operator), isOpenComparisonOperator(operator));
        }

        function initializeComparisonOperator(colType, comparisonType, open) {
            var colGenericType = Expressions.genericType(colType);
            var initOperatorDic = {"min": ">  ","max": "<  "}
            var genericTypeDic = {"num": "number","date": "date"}
            var operator = initOperatorDic[comparisonType] + "["+genericTypeDic[colGenericType]+"]";

            if(!open) {
                operator = switchOpenCloseComparisonOperator(operator);
            }
            return operator;
        }

        function setValuesFromOtherCondition(condition, otherCondition, colType) {
            var colGenericType = Expressions.genericType(colType);
            var valuesToSet = {
                "num": ["num"],
                "date": ["date", "time"]
            }
            var fieldsToSet = valuesToSet[colGenericType];
            if (!fieldsToSet) return;
            fieldsToSet.forEach(function(field) {
                condition[field] = otherCondition[field];
            });
        }

        function indexOfComparisonType(conditions, comparisonType) {
            if(!conditions) return -1;
            for (var i=0; i<conditions.length; i++) {
                var condition = conditions[i];
                if (!condition || !condition.operator) return;
                if (getTypeComparisonOperator(condition.operator) == comparisonType) {
                    return i;
                }
            }
            return -1;
        }


        // Methods particular to organisation of splits ( with filter.uiData.conditions )

        // Retrieve conditions from a split list with an index
        function getRangeConditions(splits, splitIndex) {
            if (!(splits && splits[splitIndex])) return;
            var split = splits[splitIndex];
            if (!(split.filter && split.filter.uiData && split.filter.uiData.conditions)) return;
            return split.filter.uiData.conditions;
        }


        function indexOfMinCond(splits, splitIndex) {
            var conditions = getRangeConditions(splits, splitIndex);
            if (!conditions) return -1;
            return indexOfComparisonType(conditions, "min");
        }

        function getMinCond(splits, splitIndex) {
            var conditions = getRangeConditions(splits, splitIndex);
            var index = indexOfComparisonType(conditions, "min");
            if (index === -1 || !conditions[index]) return;
            return conditions[index];
        }

        function hasMinCond(splits, splitIndex) {
            return (indexOfMinCond(splits, splitIndex) > -1);
        }

        function indexOfMaxCond(splits, splitIndex) {
            var conditions = getRangeConditions(splits, splitIndex);
            if (!conditions) return -1;
            return indexOfComparisonType(conditions, "max");
        }

        function getMaxCond(splits, splitIndex) {
            var conditions = getRangeConditions(splits, splitIndex);
            var index = indexOfComparisonType(conditions, "max");
            if ( index === -1 || !conditions[index]) return;
            return conditions[index];
        }

        function hasMaxCond(splits, splitIndex) {
            return (indexOfMaxCond(splits, splitIndex) > -1);
        }

        function createCondition(isOpen, comparisonType, colType, inputCol) {
            var operator = initializeComparisonOperator(colType, comparisonType, isOpen);
            return {
                "input": inputCol,
                "operator": operator
            };
        }

        function getValuesFromCond(condition, colType) {
            if (!condition) return;
            var colGenericType = Expressions.genericType(colType);
            if (colGenericType == 'num') {
                return condition.num;
            } else if (colGenericType == "date") {
                if (!condition.date || !condition.time) return;
                return moment(condition.date + " " + condition.time, "YYYY-MM-DD HH:mm").toDate()
            }
        }

        return {
            dateFormats,
            switchOpenCloseComparisonOperator,
            isOpenComparisonOperator,
            initializeComparisonOperator,
            modifyColTypeInComparisonOperator,
            setValuesFromOtherCondition,
            getTypeComparisonOperator,
            getRangeConditions,
            indexOfMaxCond,
            indexOfMinCond,
            getMinCond,
            getMaxCond,
            hasMinCond,
            hasMaxCond,
            createCondition,
            getValuesFromCond
        }
    });

    // Service to help build Gauges for shares (RANDOM, RANDOM_COLUMNS, CENTILES) and RANGE modes
    // In particular to compute extent of values and adapt values to scales
    services.factory('GaugeHelper', function(Fn) {

                var spaceToInfinity = 100;
                var spaceMinEqualsMax = 10;
                function getValuesModified(data, index, getValuesFn, min, max) {
                    var values = getValuesFn(data, index);
                    if(!values) return;
                    var minMaxValue = getMinMaxValue(data, getValuesFn, min, max);
                    if (!minMaxValue) return;

                    // Check whether values is array or array of array
                    var isArrayOfArrays = angular.isArray(values[0]);

                    if (!isArrayOfArrays) {
                        return computeValues(values, minMaxValue);
                    } else {
                        return values.map(function(v) { return computeValues(v, minMaxValue);});
                    }
                }

                function computeValues(values, minMaxValue) {
                        var convertedValues = []
                        // Convert Infinity values to work with scale
                        convertedValues[0] = (values[0] == - Infinity) ? minMaxValue.min - spaceToInfinity : values[0];
                        convertedValues[1] =  (values[1] == + Infinity) ? minMaxValue.max + spaceToInfinity : values[1];
                        return convertedValues;
                }

                // utils
                function getMinMaxValue(data, getValuesFn, min, max) {
                    var mustFindMin = true;
                    var mustFindMax = true;
                    var hasInfinityMin = false;
                    var hasInfinityMax = false;
                    var maxValue = null;
                    var minValue = null;
                    if (min != null) {
                        mustFindMin = false;
                        minValue = min;
                    }
                    if (max != null) {
                        mustFindMax = false;
                        maxValue = max;
                    }
                    if (!data) return;

                    if (mustFindMin || mustFindMax) {
                        for (var i=0; i < data.length; i++) {
                            var values = getValuesFn(data, i);
                            if (!values) continue;
                            // Flattening if values is array of arrays
                            values = [].concat.apply([], values);

                            // Checking if has +/- Infinity
                            if (Fn.inArray(values)(-Infinity)){
                                hasInfinityMin = true;
                            }
                            if (Fn.inArray(values)(Infinity)){
                                hasInfinityMax = true;
                            }
                            // Removing +/- Infinity from values
                            values = values.filter(function(v) { return Math.abs(v) != Infinity;});

                            var tmpMax = Math.max(...values);
                            var tmpMin = Math.min(...values);

                            // Testing if must define new maxValues, leave aside Infinite values
                            if (mustFindMax && (maxValue == null || maxValue < tmpMax)) {
                                maxValue = tmpMax;
                            }
                            if (mustFindMin && (minValue == null || minValue > tmpMin)) {
                                minValue = tmpMin;
                            }
                        }
                    }

                    return {
                        min: minValue,
                        max: maxValue,
                        hasInfinityMin: hasInfinityMin,
                        hasInfinityMax: hasInfinityMax
                    };
                }

                function buildScale(extremities, width) {
                    var minValue = extremities.min;
                    var maxValue = extremities.max;

                    // When extremities are equal, artificially spread them
                    if(extremities.min != null && extremities.min == extremities.max) {
                        minValue -= spaceMinEqualsMax;
                        maxValue += spaceMinEqualsMax;
                    }

                    var axisRange = [minValue, maxValue];
                    var range = [0, width];

                    if (extremities.hasInfinityMin) {
                        range.splice(1, 0, width / 4);
                        axisRange.unshift(extremities.min - spaceToInfinity);
                    }
                    if (extremities.hasInfinityMax) {
                        range.splice(-1, 0, 3 * width / 4);
                        axisRange.push(extremities.max + spaceToInfinity);
                    }
                    var scale = d3.scale.linear().domain(axisRange).range(range);
                    return scale;
                }

                return {
                    getValuesModified,
                    getMinMaxValue,
                    buildScale
                };

    });

})();


(function(){
    'use strict';

    var widgets = angular.module('dataiku.directives.widgets');

    widgets.directive("splitSharesSelector", function() {
        return {
            scope: true,
            templateUrl: "/templates/recipes/fragments/split-shares-selector.html",
            link: function($scope, element, attrs) {

                $scope.getSelectedSplits = function() {
                    return $scope.getSplits(attrs.selectedMode);
                }

                $scope.getMaxShare = function(currentIndex) {
                    var splits = $scope.getSelectedSplits();
                    var cumulatedShare = 0;
                    for (var i = 0; i < splits.length ; i++) {
                        if (i != currentIndex) {
                            cumulatedShare += splits[i].share;
                        }
                    }
                    return (cumulatedShare > 100) ? 0 : 100 - cumulatedShare;
                };

                $scope.addShare = function(splitIndex) {
                    var splits = $scope.getSelectedSplits();
                    var share = 100;
                    for (var i=0; i < splitIndex; i++) {
                        share -= splits[i].share;
                    }
                    // ensuring to have a positive share (if user does not respect sum share < 100)
                    share = Math.max(share, 0);
                    splits.splice(splitIndex, 0, {share: share, outputIndex: 0});
                };

                $scope.removeShare = function(splitIndex) {
                    var splits = $scope.getSelectedSplits();
                    splits.splice(splitIndex, 1);
                };

                $scope.getShareFromIndex = function(splits, splitIndex) {
                    if (!splits) return;

                    // Regular Share
                    if (splitIndex > -1 || splits[splitIndex]) {
                        if (splits[splitIndex].share == null) return null;
                        var previousCumulatedShare = 0;
                        for (var i=0; i < splitIndex; i++) {
                            previousCumulatedShare += splits[i].share;
                        }
                        return [previousCumulatedShare, previousCumulatedShare + splits[splitIndex].share];
                    }

                    // splitIndex == -1 => return remaining share
                    if (splitIndex == -1) {
                        var totalShare = splits.reduce(function(memo, split) { return memo + ((split.share != null) ? split.share : 0);}, 0);
                        return (totalShare < 100) ? [[totalShare, 100]] : [[100,100]];
                    }
                };

                $scope.getRemainingShare = function() {
                    var splits = $scope.getSelectedSplits();
                    if (!splits) return;
                    var totalShare = splits.reduce(function(memo, split) { return memo + ((split.share != null) ? split.share : 0);}, 0);
                    return Math.max(100 - totalShare, 0);
                };

                function getShareInInterval(newValue, splits , index) {
                    var totalShareWithoutIndex = splits.reduce(function(mem, split, i){
                        if (i != index) {
                            return mem + split.share;
                        } else {
                            return mem;
                        }
                    }, 0);
                    return Math.max(Math.min(newValue, 100 - totalShareWithoutIndex), 0);
                }

                $scope.updateShareFromIndex = function(splits, splitIndex, min, max) {
                    if (!splits || !splits[splitIndex]) return;

                    var newShare = Math.max(Math.floor(max - min), 0);
                    var previousTotalShare = splits.reduce(function(mem, split){ return mem + split.share;}, 0);
                    var shareToDispatch  = newShare - splits[splitIndex].share;

                    // Update current value
                    splits[splitIndex].share = newShare;

                    // Update following values if new total > 100 or share to dispatch < 0
                    var numImpactedSplits = splits.length - splitIndex - 1;
                    if (numImpactedSplits >= 1 && (previousTotalShare + shareToDispatch > 100 || (shareToDispatch < 0))) {
                        var impactShare = null;
                        if (shareToDispatch > 0) {
                            impactShare = Math.ceil(shareToDispatch / numImpactedSplits);
                        } else {
                            impactShare =  - Math.floor( - shareToDispatch / numImpactedSplits);
                        }
                        var previousShare = null;
                        var cumulatedImpactShare = 0;
                        // update all following values except last to round numbers
                        for (var i = splitIndex + 1; i < splits.length - 1 ; i++) {
                            previousShare = splits[i].share;
                            splits[i].share = getShareInInterval(splits[i].share - impactShare, splits, i);
                            cumulatedImpactShare += (splits[i].share - previousShare);
                        }
                        // update last value
                        splits[splits.length - 1].share = getShareInInterval(splits[splits.length - 1].share - (shareToDispatch - cumulatedImpactShare), splits, splits.length - 1);
                    }
                };
            }
        }
    });

    widgets.directive("rangeBracket", function(RangeExpressions) {
        return {
            scope: {
                type: "=",
                rangeIndex: "=",
                ranges: "=",
                isValidCol: "="
            },
            template: '<button class="btn btn--icon btn--secondary bracket" ng-click="switchExtremity()" ng-if="isOpeningBracket()" title="{{isOpenExtremity() ? \'Exclude\' : \'Include\'}}" data-toggle="tooltip" data-placement="top">[</button>' +
                      '<button class="btn btn--icon btn--secondary bracket" ng-click="switchExtremity()" ng-if="isClosingBracket()" title="{{isOpenExtremity() ? \'Exclude\' : \'Include\'}}" data-toggle="tooltip" data-placement="top">]</button>',
            link: function($scope) {

                function getExtremity() {
                    return getExtremityFromIndexAndType($scope.rangeIndex, $scope.type);
                }

                function getExtremityFromIndexAndType(index, extremityType) {
                    if (extremityType == "min") {
                        return RangeExpressions.getMinCond($scope.ranges, index);
                    }
                    if (extremityType == "max") {
                        return RangeExpressions.getMaxCond($scope.ranges, index);
                    }
                    return null;
                }

                function getConnectedExtremity() {
                    if ($scope.type == "min") {
                        return getExtremityFromIndexAndType($scope.rangeIndex - 1, 'max');
                    }
                    if ($scope.type == "max") {
                        return getExtremityFromIndexAndType($scope.rangeIndex + 1, 'min');
                    }
                    return null;
                }

                function isDefined() {
                    if (!$scope.isValidCol) return false;
                    if ($scope.type == "min") {
                        return RangeExpressions.hasMinCond($scope.ranges, $scope.rangeIndex);
                    }
                    if ($scope.type == "max") {
                        return RangeExpressions.hasMaxCond($scope.ranges, $scope.rangeIndex);
                    }
                }

                $scope.switchExtremity = function() {
                    var extremity = getExtremity();
                    var connectedExtremity = getConnectedExtremity();
                    if(extremity) {
                        extremity.operator = RangeExpressions.switchOpenCloseComparisonOperator(extremity.operator);
                    }
                    if(connectedExtremity) {
                        connectedExtremity.operator = RangeExpressions.switchOpenCloseComparisonOperator(connectedExtremity.operator);
                    }
                };

                $scope.isOpenExtremity = function() {
                    var extremity = getExtremity();
                    return RangeExpressions.isOpenComparisonOperator(extremity.operator);
                };

                $scope.isOpeningBracket = function() {
                    if (!isDefined()) return false;
                    var extremity = getExtremity();
                    var hasExtremity = (extremity && extremity.operator);
                    var isMinClosedExtremity = ($scope.type === "min" && !$scope.isOpenExtremity());
                    var isMaxOpenExtremity = ($scope.type === "max" && $scope.isOpenExtremity());
                    return (hasExtremity && (isMinClosedExtremity || isMaxOpenExtremity));
                };

                $scope.isClosingBracket = function() {
                    if (!isDefined()) return false;
                    var extremity = getExtremity();
                    var hasExtremity = (extremity && extremity.operator);
                    var isMinOpenExtremity = ($scope.type === "min" && $scope.isOpenExtremity());
                    var isMaxClosedExtremity = ($scope.type === "max" && !$scope.isOpenExtremity());
                    return (hasExtremity && (isMinOpenExtremity || isMaxClosedExtremity));
                };

            }
        }
    });

    widgets.directive('splitRangesSelector', function(Assert, Expressions, RangeExpressions) {
        return {
            scope: true,
            templateUrl : "/templates/recipes/fragments/split-ranges-selector.html",
            link : function($scope, element, attrs) {

                $scope.getSelectedSplits = function() {
                    return $scope.getSplits(attrs.selectedMode);
                }

                $scope.getRangeColType = function() {
                    return $scope.getColType($scope.params.column);
                };

                $scope.getRangeColGenericType = function() {
                    return Expressions.genericType($scope.getRangeColType());
                };

                $scope.rangeInputsClass = function() {
                    return "input-"+$scope.getRangeColGenericType();
                };

                $scope.createMinCondition = function(isOpen) {
                    return RangeExpressions.createCondition(isOpen, "min", $scope.getRangeColType(), $scope.params.column);
                };

                $scope.createMaxCondition = function(isOpen) {
                    return RangeExpressions.createCondition(isOpen, "max", $scope.getRangeColType(), $scope.params.column);
                };

                $scope.addRangeSplit = function(splitIndex) {
                    let splits = $scope.getSelectedSplits();
                    let conditions = [];
                    let genericType = $scope.getRangeColGenericType();
                    let maxPreviousCond = null;
                    let isOpenPreviousCond = false;
                    let minCondition;
                    if (genericType === "date") {
                        if (RangeExpressions.hasMaxCond(splits, splitIndex - 1)) {
                            maxPreviousCond = RangeExpressions.getMaxCond(splits, splitIndex - 1);
                            isOpenPreviousCond = (maxPreviousCond && RangeExpressions.isOpenComparisonOperator(maxPreviousCond.operator));
                        }
                        minCondition = $scope.createMinCondition(!isOpenPreviousCond);
                        var maxCondition = $scope.createMaxCondition(false);
                        conditions = [minCondition, maxCondition];

                    } else if (genericType === "num" && splitIndex > 0) {
                        if (!RangeExpressions.hasMaxCond(splits, splitIndex - 1)) {
                            // Create max conditions for previous if do not exist yet
                            maxPreviousCond = $scope.createMaxCondition(false);
                            var previousConditions = RangeExpressions.getRangeConditions(splits, splitIndex - 1);
                            previousConditions.push(maxPreviousCond);
                        } else {
                            maxPreviousCond = RangeExpressions.getMaxCond(splits, splitIndex - 1);
                        }
                        isOpenPreviousCond = (maxPreviousCond && RangeExpressions.isOpenComparisonOperator(maxPreviousCond.operator));
                        minCondition = $scope.createMinCondition(!isOpenPreviousCond);
                        conditions = [minCondition];
                    }

                    let newRange = $scope.getNewRangeSplit(conditions);

                    // Autofill new value if needed
                    if (maxPreviousCond) {
                        RangeExpressions.setValuesFromOtherCondition(minCondition, maxPreviousCond, $scope.getRangeColType());

                        // For dates, set also end of interval to previous max date
                        if(genericType === "date") {
                            RangeExpressions.setValuesFromOtherCondition(maxCondition, maxPreviousCond, $scope.getRangeColType());
                        }
                    } else if (genericType === "date") {
                        Assert.trueish(minCondition, 'minCondition');
                        Assert.trueish(maxCondition, 'maxCondition');
                        // Autofill value to today for first date split
                        let today = moment().format(RangeExpressions.dateFormats["dateFormat"]);
                        let time = "00:00";
                        minCondition.date = today; //NOSONAR not undefined
                        minCondition.time = time; //NOSONAR not undefined
                        maxCondition.date = today; //NOSONAR not undefined
                        maxCondition.date = time; //NOSONAR not undefined
                    }
                    splits.splice(splitIndex, 0, newRange);
                };

                $scope.removeRangeSplit = function(splitIndex) {
                    var splits = $scope.getSelectedSplits();
                    splits.splice(splitIndex, 1);
                };

                $scope.getRangeFromIndex = function(splits, splitIndex) {

                    // Returning normal values for particular index with Infinity if not set
                    if (splitIndex > -1) {
                        var minValue = - Infinity;
                        if (RangeExpressions.hasMinCond(splits,splitIndex)) {
                            var minCond = RangeExpressions.getMinCond(splits, splitIndex);
                            minValue = RangeExpressions.getValuesFromCond(minCond, $scope.getRangeColType());
                            if (minValue == null) {
                                return null;
                            }
                        }
                        var maxValue = Infinity;
                        if (RangeExpressions.hasMaxCond(splits,splitIndex)) {
                            var maxCond = RangeExpressions.getMaxCond(splits, splitIndex);
                            maxValue = RangeExpressions.getValuesFromCond(maxCond, $scope.getRangeColType());
                            if (maxValue == null) {
                                return null;
                            }
                        }
                        if (minValue > maxValue) return null;
                        return [[minValue, maxValue]];
                    }


                    // Returning missing pieces in [-Infinity, Infinity] interval to build Remaining gauge
                    else if (splitIndex == -1) {

                        // First retrieving all values from the ranges
                        var ranges = [];
                        for(var i=0; i < splits.length; i++) {
                            ranges.push($scope.getRangeFromIndex(splits, i));
                        }

                        // Then filter null values && sort the ranges according to their min value
                        ranges = ranges.filter(function(values) { return values != null;})
                                       .map(function(values) { return [].concat.apply([], values);})
                                       .sort(function(v1, v2) { return (v1[0] < v2[0]) ? -1 : 1;});

                        if(ranges.length == 0) return;
                        // Then build disjoint intervals made of unions of intervals that intersect
                        var disjointAggRanges = [];
                        var currentAggRange = ranges[0];
                        for (var j=1; j<ranges.length; j++) {

                            if( currentAggRange[1] >= ranges[j][0] ) {
                                // if must stay in same Adgg Range => Set new upper bound to max of two intervals
                                currentAggRange[1] = Math.max(currentAggRange[1], ranges[j][1]);
                            } else {
                                // Archive AggRange and set new one
                                disjointAggRanges.push(currentAggRange);
                                currentAggRange = ranges[j];
                            }
                        }
                        // Archive last AggRange
                        disjointAggRanges.push(currentAggRange);

                        // Finally, build complement set of intervals
                        var remainingValues = [];

                        // Add [endValue(i), endValue(i+1)]
                        for (var k=0; k < disjointAggRanges.length - 1; k++) {
                            remainingValues.push([disjointAggRanges[k][1], disjointAggRanges[k+1][0]]);
                        }

                        return (remainingValues.length > 0) ? remainingValues : null;
                    }
                };

                // When switching for the first time to RANGE mode with a Date variable
                // must initialize conditions for first split
                var unregisterDateRangeWatch = $scope.$watch(function() {
                    var isDateColumn = Expressions.genericType($scope.getColType($scope.params.column)) === "date";
                    return ($scope.params.mode === "RANGE" && isDateColumn);
                }, function(shouldInitialize) {
                    if (!shouldInitialize) return;

                    var hasMoreThanOneSplit = ($scope.params.rangeSplits && $scope.params.rangeSplits.length > 1);
                    var conditionsOfFirstSplit = RangeExpressions.getRangeConditions($scope.params.rangeSplits, 0);
                    var firstSplitHasConditions = (conditionsOfFirstSplit && conditionsOfFirstSplit.length > 0);
                    if (hasMoreThanOneSplit || firstSplitHasConditions) {
                        unregisterDateRangeWatch();
                        return;
                    }

                    var conditions = $scope.getConditionsForFirstDateSplit($scope.params.column);
                    $scope.params.rangeSplits = [$scope.getNewRangeSplit(conditions)];
                    unregisterDateRangeWatch();
                });

            }
        }
    });

    widgets.directive('rangeInputs', function(Expressions, RangeExpressions, $timeout) {
        return {
            scope: {
                ranges: "=",
                rangeIndex: "=",
                column: "=",
                colType: "=",
                setTime: "=",
                isValidCol: "="
            },
            templateUrl: "/templates/recipes/fragments/range-inputs.html",
            link: function($scope, element) {

                $scope.getDateFormat = function() {
                    if ($scope.setTime) {
                        return RangeExpressions.dateFormats['dateWithTimeFormat'];
                    } else {
                        return RangeExpressions.dateFormats['dateFormat'];
                    }
                };

                $scope.hasMinCond = function() {
                    return RangeExpressions.hasMinCond($scope.ranges, $scope.rangeIndex);
                };

                $scope.getMinCond = function() {
                    return RangeExpressions.getMinCond($scope.ranges, $scope.rangeIndex);
                };

                $scope.getMaxCond = function() {
                    return RangeExpressions.getMaxCond($scope.ranges, $scope.rangeIndex);
                };

                $scope.hasMaxCond = function() {
                    return RangeExpressions.hasMaxCond($scope.ranges, $scope.rangeIndex);
                };

                $scope.getGenericColType = function() {
                    return Expressions.genericType($scope.colType);
                };

                $scope.deleteMinIfNecessary = function(currentValue) {
                    var shouldDeleteMin = ($scope.rangeIndex == 0);
                    if(shouldDeleteMin && (currentValue == null)) {
                        var conditions = RangeExpressions.getRangeConditions($scope.ranges, $scope.rangeIndex);
                        var index = RangeExpressions.indexOfMinCond($scope.ranges, $scope.rangeIndex);
                        if (index > -1) {
                            conditions.splice(index, 1);
                        }
                    }
                };

                $scope.deleteMaxIfNecessary = function(currentValue) {
                    var shouldDeleteMax = ($scope.ranges && ($scope.rangeIndex == $scope.ranges.length - 1));
                    if(shouldDeleteMax && (currentValue == null)) {
                        var conditions = RangeExpressions.getRangeConditions($scope.ranges, $scope.rangeIndex);
                        var index = RangeExpressions.indexOfMaxCond($scope.ranges, $scope.rangeIndex);
                        if (index > -1) {
                            conditions.splice(index, 1);
                        }
                    }
                };

                $scope.createMinCondition = function() {
                    var conditions = RangeExpressions.getRangeConditions($scope.ranges, $scope.rangeIndex);
                    if (!$scope.hasMinCond()) {
                        var minCond = RangeExpressions.createCondition(false, "min", $scope.colType, $scope.column);
                        conditions.unshift(minCond);

                        // Focus on new input Min (with little timeout to let dom create input)
                        $timeout(function() {
                            $("#inputMin"+$scope.rangeIndex).focus();
                        }, 0);
                    }
                };

                $scope.createMaxCondition = function() {
                    var conditions = RangeExpressions.getRangeConditions($scope.ranges, $scope.rangeIndex);
                    if (!$scope.hasMaxCond()) {
                        var maxCond = RangeExpressions.createCondition(true, "max", $scope.colType, $scope.column);
                        conditions.push(maxCond);

                        // Focus on new input Max (with little timeout to let dom create input)
                        $timeout(function() {
                            $("#inputMax"+$scope.rangeIndex).focus();
                        }, 0);
                    }
                };

                function syncFrontDatesWithConditions() {
                    var minCond = $scope.getMinCond();
                    if (minCond && minCond.frontDate) {
                        minCond.date = moment(minCond.frontDate, $scope.getDateFormat()).format("YYYY-MM-DD");
                        minCond.time = $scope.setTime ? moment(minCond.frontDate, $scope.getDateFormat()).format("HH:mm") : "00:00";
                    }
                    var maxCond = $scope.getMaxCond();
                    if (maxCond && maxCond.frontDate) {
                        maxCond.date = moment(maxCond.frontDate, $scope.getDateFormat()).format("YYYY-MM-DD");
                        maxCond.time = $scope.setTime ? moment(maxCond.frontDate, $scope.getDateFormat()).format("HH:mm") : "00:00";
                    }
                }

                function syncConditionsWithFrontDates() {
                    var minCond = $scope.getMinCond();
                    if (minCond && minCond.date && minCond.time) {
                        minCond.frontDate = moment(minCond.date + " " + minCond.time, RangeExpressions.dateFormats["dateWithTimeFormat"]).format($scope.getDateFormat());
                    }
                    var maxCond = $scope.getMaxCond();
                    if (maxCond && maxCond.date && maxCond.time) {
                        maxCond.frontDate = moment(maxCond.date + " " + maxCond.time, RangeExpressions.dateFormats["dateWithTimeFormat"]).format($scope.getDateFormat());
                    }
                }

                // WATCHERS

                $scope.$watch(function() {
                    var toWatch = {};
                    var minCond = $scope.getMinCond();
                    var maxCond = $scope.getMaxCond();
                    if (minCond) {
                        toWatch.min = minCond.frontDate;
                    }
                    if (maxCond) {
                        toWatch.max = maxCond.frontDate;
                    }
                    return toWatch;
                }, syncFrontDatesWithConditions, true);

                // INIT
                syncConditionsWithFrontDates();
            }
        }
    });

    widgets.directive('splitGauge', function(GaugeHelper) {
        return {
            restrict : 'A',
            template : "<svg class='split-gauge'></svg>",
            scope : {
                data : '=',
                getValuesFn : '=',
                index : '=',
                max: "=",
                min: "="
            },
            replace : true,
            link : function($scope, element, attrs) {

                // get the gauge: root of template
                var gaugeSvg = d3.select(element[0]);

                // filling the gauge
                var gaugeG = gaugeSvg.append("g").attr("class", "x brush");

                // Retrieving width and height
                var width = $(element).innerWidth();
                var height = $(element).innerHeight();

                // Building background rect
                gaugeG.append('rect')
                      .attr('class', 'gauge-background')
                      .attr('width', width)
                      .attr('height', height)
                      .attr('fill', "#f0f1f1")

                var xScale = null;

                function buildGauge() {
                    if($scope.data == null || $scope.index == null) return;

                    var rectData = GaugeHelper.getValuesModified($scope.data, $scope.index, $scope.getValuesFn, $scope.min, $scope.max);

                    // Invalid input => delete all rects
                    if (!rectData) {
                        gaugeG.selectAll('.extent').remove();
                        return;
                    }

                    // Update value of width
                    width = $(element).innerWidth();

                    var extremities = GaugeHelper.getMinMaxValue($scope.data, $scope.getValuesFn, $scope.min, $scope.max);

                    // build scale
                    xScale = GaugeHelper.buildScale(extremities, width);

                    // Adding data rects
                    // Enter
                    gaugeG.selectAll('.extent')
                          .data(rectData)
                          .enter()
                          .append('rect')
                          .attr('class', 'extent')
                          .attr("x", function(d) { return xScale(d[0]);})
                          .attr("height", height)
                          .attr('width', function(d) { return xScale(d[1]) - xScale(d[0]); });

                    // Update
                    gaugeG.selectAll('.extent')
                          .data(rectData)
                          .attr("x", function(d) { return xScale(d[0]);})
                          .attr('width', function(d) { return xScale(d[1]) - xScale(d[0]); });

                    // Remove
                    gaugeG.selectAll('.extent')
                          .data(rectData)
                          .exit()
                          .remove();

                    // Adding gradient rects if needed (to render feeling of infinity)
                    // Remove previous gradient rects
                    gaugeSvg.selectAll('.gauge-gradient').remove();

                    // Retrieving values
                    var flattenRowValues = [].concat.apply([], $scope.getValuesFn($scope.data, $scope.index));
                    // Rebuilding new gauge gradients
                    if (Math.min(...flattenRowValues) == - Infinity) {
                        var leftGradient = gaugeSvg.append('defs')
                                                   .append('linearGradient')
                                                   .attr('id', 'leftGradient');

                        leftGradient.append('stop')
                                    .attr('offset', '0%')
                                    .attr('stop-color', "#f0f1f1")
                                    .attr('stop-opacity', 1);
                        leftGradient.append('stop')
                                    .attr('offset', '100%')
                                    .attr('stop-color', "rgba(255, 255, 255, 0)")
                                    .attr('stop-opacity', 1);


                        gaugeSvg.append('rect')
                                .attr('class', 'gauge-gradient')
                                .attr('width', width / 4)
                                .attr('height', height + 2)
                                .attr('fill', "url(#leftGradient)");
                    }
                    if (Math.max(...flattenRowValues) == Infinity) {
                        var rightGradient = gaugeSvg.append('defs')
                                                   .append('linearGradient')
                                                   .attr('id', 'rightGradient');

                        rightGradient.append('stop')
                                    .attr('offset', '0%')
                                    .attr('stop-color', "rgba(255, 255, 255, 0)")
                                    .attr('stop-opacity', 1);
                        rightGradient.append('stop')
                                    .attr('offset', '100%')
                                    .attr('stop-color', "#f0f1f1")
                                    .attr('stop-opacity', 1);


                        gaugeSvg.append('rect')
                                .attr('x', 3 * width / 4)
                                .attr('class', 'gauge-gradient')
                                .attr('width', width / 4)
                                .attr('height', height + 2)
                                .attr('fill', "url(#rightGradient)");
                    }

                };


                // Add watchers
                $scope.$watch('data', function(nv, ov) {
                    if (nv == null) return;
                    buildGauge();
                }, true);
            }
        };
    });

    widgets.directive('movingSplitGauge', function(GaugeHelper) {
        return {
            restrict : 'A',
            template : "<svg class='split-gauge'></svg>",
            scope : {
                data : '=',
                getValuesFn : '=',
                updateValuesFn : '=',
                index : '=',
                max: "=",
                min: "="
            },
            replace : true,
            link : function($scope, element, attrs) {
                var handleWidth = 5;
                var handleHeight = 9;

                // get the gauge: root of template
                var gaugeSvg = d3.select(element[0]);

                // filling the gauge
                var gaugeG = gaugeSvg.append("g").attr("class", "x brush");
                var gaugeHandleG = gaugeSvg.append("g").attr("class", "x gauge-handles"); // the gauge handles (click-through)

                var xScale = null;

                // update the total range, and then the graph
                $scope.refreshRange = function() {
                    if($scope.data == null || $scope.index == null) return;

                    var extentRange = GaugeHelper.getValuesModified($scope.data, $scope.index, $scope.getValuesFn, $scope.min, $scope.max);

                    // Invalid value => remove gauge
                    if (!extentRange) {
                        gaugeG.selectAll('.extent').remove();
                        gaugeHandleG.selectAll('.resize').remove();
                    }

                    // Retrieving width and height
                    var width = $(element).innerWidth();
                    var height = $(element).innerHeight();


                    var extremities = GaugeHelper.getMinMaxValue($scope.data, $scope.getValuesFn, $scope.min, $scope.max);

                    // build scale
                    xScale = GaugeHelper.buildScale(extremities, width);

                    // prepare callbacks
                    // when dragged
                    function brushed() {
                        var extent = gauge.extent();

                        //Verify that east part of brush stays at the east
                        var currentValues = GaugeHelper.getValuesModified($scope.data, $scope.index, $scope.getValuesFn, $scope.min, $scope.max);
                        if (extent[0] < currentValues[0]) { // i.e. when the brush is inverted, what we want to prevent
                            extent = currentValues;
                        }

                        // Modify gauge extent
                        d3.select(this).call(gauge.extent(extent));

                        // Move Handle to new position of gauge
                        var xE = xScale(extent[1]);
                        gaugeHandleG.selectAll(".e").attr("transform", "translate(" + xE + ", 0)");

                        // Update converned values
                        $scope.updateValuesFn($scope.data, $scope.index, extent[0], extent[1]);
                        $scope.$apply();
                    }

                    // add display in resize e if gauge of size 0
                    function addResizeE() {
                        actualGauge.selectAll('.resize.e')
                                   .style('display', 'initial');
                    }


                    if (extentRange && (extentRange[0] != null) && (extentRange[1] != null)) {

                        var gauge = d3.svg.brush()
                                          .x(xScale)
                                          .on('brush', brushed)
                                          .on('brushend', addResizeE)
                                          .extent(extentRange);


                        //Create objects
                        var actualGauge = gaugeG.call(gauge);


                        var xE = xScale(extentRange[1]);

                        //Style the brush
                        actualGauge.selectAll("rect")
                                   .attr("y", 0)
                                   .attr("height", height);

                        actualGauge.selectAll('.extent, .background, .resize')
                                   .attr('pointer-events', 'none');

                        // Create the right handle
                        gaugeHandleG.selectAll('.resize').remove();

                        // Add pointer events
                        actualGauge.selectAll('.resize.e')
                                   .attr('pointer-events', 'all');

                        addResizeE();

                        // Add g for handle
                        gaugeHandleG.append('g').classed('resize', true).classed('e', true).attr("transform", "translate(" + xE + ", 0)");

                        var gh = gaugeHandleG.selectAll(".resize");

                        gh.append("rect")
                          .classed("separator", true)
                          .attr("y", 0)
                          .attr("height", height)
                          .attr("x", -1.5)
                          .attr("width", 3);

                        gh.append("rect")
                          .classed("handle", true)
                          .attr("y", (height - handleHeight) / 2)
                          .attr("height", handleHeight)
                          .attr("x", -(handleWidth/2))
                          .attr("width", handleWidth);
                    }

                };

                // Add watchers
                $scope.$watch('data', function(nv, ov) {
                    if (nv == null) return;
                    $scope.refreshRange();
                }, true);
            }
        };
    });

    widgets.directive('columnsSelect', function(Fn, $filter) {
        return {
            scope: {
                title: "=",
                selectedColumns: "=",
                columns: "=",
                getColType: "=",
                hasOrder: "=",
                needsInfo: "=",
                isInfoOpen: "=",
                infoText: "@"
            },
            templateUrl: "/templates/recipes/fragments/columns-select.html",
            link: function($scope) {

                // Manipulate columns : getColumns, add/remove cols from available columns

                $scope.addColumn = function(col) {
                    // Formating column
                    var formatedCol;
                    if ($scope.hasOrder) {
                        formatedCol = {column: col.name, desc: false};
                    } else {
                        formatedCol = col.name;
                    }
                    $scope.selectedColumns.push(formatedCol);
                };

                $scope.getColumn = function(col) {
                    if ($scope.hasOrder) {
                        return col.column;
                    } else {
                        return col;
                    }
                };

                $scope.removeColumn = function(idx) {
                    if (idx > -1) {
                        $scope.selectedColumns.splice(idx, 1);
                    }
                };

                $scope.isInColumns = function(col) {
                    var allColumnsNames = $scope.columns.map(Fn.prop('name'));
                    return Fn.inArray(allColumnsNames)($scope.getColumn(col));
                };

                $scope.filterSelectedColumns = function(col) {
                    var selectedColumnsNames;
                    if ($scope.hasOrder && $scope.selectedColumns) {
                        selectedColumnsNames = $scope.selectedColumns.map(Fn.prop("column"));
                    } else {
                        selectedColumnsNames = $scope.selectedColumns;
                    }
                    return (!selectedColumnsNames || !Fn.inArray(selectedColumnsNames)(col.name));
                };

                function computeTypes() {
                    let typeToName = $filter('columnTypeToName');
                    $scope.filteredTypeNames =  $scope.columns.filter($scope.filterSelectedColumns).map(function(column){ return typeToName(column.type)});
                }
                computeTypes();
                $scope.$watch(function() {
                    return $scope.columns.filter($scope.filterSelectedColumns);
               