(function(){
    'use strict';

    const app = angular.module("dataiku.ml.explainability");

    app.component('explorationResults', {
        bindings: {
            isClassification: '<',
            isRegression: '<',
            whatIfRouter: '<',
            fmi: '<',
            explorationLocalStore: '<',
            modelData: '<',
            storageTypes: '<',
            featuresSortingService: '<'
        },
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-results.html',
        controller: function($scope, DataikuAPI, OutcomeOptimizationSpecialTarget, ExplorationWT1Service, FeaturesDistributionAdapter, ExplorationChartsConfig,
                             FutureProgressModal, EmuType, ConstraintHelper, NO_TARGET_CLASS, LoggerProvider, InteractiveModelCommand, ExportUtils) {
            const logger = LoggerProvider.getLogger('ml.exploration.results');
            this.$onInit = function () {
                $scope.sortOptions = this.featuresSortingService.getSortOptions();
                $scope.selection = this.featuresSortingService.cleanSelection();
                $scope.$watch("selection.orderQuery", this.featuresSortingService.updateSortOrderAfterOptionChange.bind(this.featuresSortingService));
                if (!this.explorationLocalStore.canUseResultsPage()) {
                    logger.warn("Cannot open results: some variables are missing in the local storage");
                    this.whatIfRouter.openConstraints();
                    return;
                }
    
                $scope.NO_TARGET_CLASS = NO_TARGET_CLASS;
    
                FeaturesDistributionAdapter.init(this.fmi, this.storageTypes).then(distributions => {
                    // Unlocked features
                    angular.copy(this.explorationLocalStore.getFeatureDomains().filter(feature => feature.type !== EmuType.FROZEN), $scope.unlockedFeatures);
                    $scope.unlockedFeatures.forEach(feature => {
                        const distributionAdapter = distributions.getSingle(feature.name, ExplorationChartsConfig.OTHER_CATEGORY_NAME);
                        const scale = distributionAdapter.scale;
                        const distribution = distributionAdapter.distribution;
                        feature.constraintHelper = ConstraintHelper.init(distributionAdapter);
                        feature.distribution = distribution.map((d, i) => {
                            return { tick: scale[i], width: d }
                        });
                        if (feature.type === EmuType.NUMERICAL) {
                            feature.shouldBeRepresentedWithHistograms = distributionAdapter.shouldBeRepresentedWithHistograms;
                        }
                    });
    
                    angular.copy($scope.unlockedFeatures.map(feature => feature.name), $scope.shownFeatures);
                    $scope.unlockedFeaturesNames = $scope.unlockedFeatures.map(feature => feature.name);
    
                    // Locked features
                    const isLocked = featureName => $scope.unlockedFeatures.find(feature => feature.name === featureName) === undefined;
                    $scope.lockedFeatureNames = Object.keys($scope.reference).filter(isLocked);
    
                    $scope.startExploration();
                });
    
                const computationCallback = (results, duration) => {
                    const result = results[0];  // Computation run on a single record, so results is a 1-sized array
                    wt1Service.emitComputationSucceeded(result.syntheticData.length, duration);
                    if (!result.syntheticData || result.syntheticData.length === 0) {
                        $scope.searchFailed = true;
                        return;
                    }
                    const getProbas = metadataRow => {
                        if (this.isClassification) {
                            return Object.fromEntries(this.modelData.classes.map(cat => [cat, metadataRow[`proba_${cat}`]]));
                        } else {
                            return undefined;
                        }
                    }
    
                    const initLoss = point => {
                        if (this.isClassification) {
                            point.loss = point.plausibility === undefined ? Number.NEGATIVE_INFINITY : -point.plausibility;
                        } else if ($scope.target === OutcomeOptimizationSpecialTarget.MIN) {
                            point.loss = point.prediction;
                        } else if ($scope.target === OutcomeOptimizationSpecialTarget.MAX) {
                            point.loss = -point.prediction;
                        } else {
                            point.loss = Math.abs($scope.target - point.prediction);
                        }
                        return point;
                    }
    
                    // Update the records
                    angular.copy(result['syntheticData'].map((e, i) => ({
                        values: e,
                        plausibility: result['syntheticMetadata'][i].plausibility,
                        prediction: result['syntheticMetadata'][i].prediction,
                        probas: getProbas(result['syntheticMetadata'][i]),
                        isReference: false,
                        isHovered: false,
                        isShown: false,
                        loss: undefined,
                        overrideInfoString: result['syntheticMetadata'][i].override
                    })), $scope.records);
                    $scope.records.forEach(initLoss);
    
                    // If we have more than 10 samples, show only the 5 best samples
                    const sortByLoss = (a, b) => (a.loss > b.loss) - (a.loss < b.loss);
                    $scope.records.sort(sortByLoss);
                    const nbRecords = $scope.records.length;
                    const nbShownByDefault = (nbRecords > 10) ? 5 : nbRecords;
                    for (let i = 0; i < nbShownByDefault; i++) {
                        $scope.records[i].isShown = true;
                    }
    
                    // Add the user input
                    const referenceRecord = initLoss({
                        values: $scope.reference,
                        plausibility: undefined,
                        prediction: this.isClassification ? score.prediction : Number(score.prediction),
                        probas: getProbas(score),
                        isReference: true,
                        isHovered: false,
                        isShown: true,
                        loss: undefined,
                        overrideInfoString: JSON.stringify(score.override)
                    });
                    $scope.records.push(referenceRecord);
    
                    const enrichRecordWithWarning = point => {
                        if (this.isClassification) {
                            return;
                        }
                        if (point.loss > referenceRecord.loss) {
                            if ($scope.target === OutcomeOptimizationSpecialTarget.MIN) {
                                point.warning = 'Greater than reference prediction';
                            } else if ($scope.target === OutcomeOptimizationSpecialTarget.MAX) {
                                point.warning = 'Less than reference prediction';
                            } else {
                                point.warning = 'Farther from target than reference prediction';
                            }
                        }
                    }
                    $scope.records.forEach(enrichRecordWithWarning);
    
                    if (this.isClassification) {
                        // Find the right colors for the predictions
                        const predictionColorMap = this.explorationLocalStore.getPredictionColorMap();
                        const OTHERS_COLOR = '#dddddd';
                        const getColor = prediction => Object.keys(predictionColorMap).includes(prediction) ? predictionColorMap[prediction] : OTHERS_COLOR;
                        $scope.records.forEach(record => {
                            record.color = getColor(record.prediction);
                        });
                    }
                    $scope.searchFailed = false;
    
                    this.featuresSortingService.enrichWithSortingAttributes($scope.unlockedFeatures);
                }
    
                $scope.startExploration = (computeEvenIfCached=false) => {
                    const {reference, target, unlockedFeatures} = $scope;
                    const explorationParams = {
                        featureDomains: this.explorationLocalStore.getFeatureDomains()
                    };
                    if (this.isClassification) {
                        explorationParams["target"] = (target === NO_TARGET_CLASS) ? undefined : target;
                        explorationParams["type"] = InteractiveModelCommand.COUNTERFACTUALS;
                    } else {
                        explorationParams["target"] = target;
                        explorationParams["type"] = InteractiveModelCommand.OUTCOME_OPTIMIZATION;
                    }
                    wt1Service.emitComputationStarted(explorationParams, unlockedFeatures);
                    const computationStartTime = performance.now();
                    DataikuAPI.interactiveModel.compute(this.fmi, explorationParams, [reference], computeEvenIfCached)
                        .success(initialResponse => {
                            const modalTitle = this.isClassification ? "Computing counterfactual explanations" : "Looking for optima";
                            FutureProgressModal.show($scope, initialResponse, modalTitle, undefined, 'static', false, true)
                                .then(futureResult => {
                                    if (futureResult === undefined) {
                                        // The computation has been aborted, going back to constraint page
                                        this.whatIfRouter.openConstraints();
                                        return;
                                    }
                                    const computationEndTime = performance.now();
                                    computationCallback(futureResult, Math.floor(computationEndTime - computationStartTime));
                                })
                                .catch(setErrorInScope.bind($scope));
                        })
                        .error(setErrorInScope.bind($scope));
                }
    
                $scope.exportResults = async () => {
                    wt1Service.emitExportClicked($scope.records.length);
                    if (!$scope.records.length) {
                        // The button should be hidden when there are no records, but to be safe...
                        logger.warn("Cannot export empty counterfactual explanations");
                        return;
                    }
                    const splitDescResp = await DataikuAPI.ml.prediction.getSplitDesc(this.fmi);
                    const inputColumnsNames = Object.keys($scope.records[0].values);
                    const columns = splitDescResp.data.schema.columns.filter(column => inputColumnsNames.includes(column.name));
                    const data = $scope.records.map(record => columns.map(column => record.values[column.name]));
                    let name;
                    // prediction
                    if (this.isClassification) {
                        // probas
                        for (const category of Object.keys($scope.records[0].probas)) {
                            columns.push({
                                name: `proba_${category}`,
                                type: "double"
                            });
                            data.forEach((record, i) => record.push($scope.records[i].probas[category]));
                        }
                        columns.push({
                            name: "prediction",
                            type: "string"
                        });
                        name = "Counterfactual explanations";
                    } else {
                        columns.push({
                            name: "prediction",
                            type: "double"
                        });
                        name = "Outcome optimization results";
                    }
                    data.forEach((record, i) => record.push($scope.records[i].prediction));
                    // plausibility
                    if ($scope.records.some(e => e.plausibility !== undefined)) {
                        columns.push({
                            name: "plausibility",
                            type: "float"
                        });
                        data.forEach((record, i) => {
                            const plausibility = $scope.records[i].plausibility;
                            record.push(plausibility !== undefined ? Number(plausibility.toFixed(2)) : undefined)
                        });
                    }
                    // is_reference
                    columns.push({
                        name: "record_type",
                        type: "string"
                    });
                    data.forEach((record, i) => {
                        let recordType;
                        if ($scope.records[i].isReference) {
                            recordType = "REFERENCE";
                        } else if (this.isClassification) {
                            recordType = "COUNTERFACTUAL_EXPLANATION";
                        } else {
                            recordType = "OPTIMUM";
                        }
                        record.push(recordType);
                    });
                    // override info
                    const withOverrides = $scope.records.some(record => !!record.overrideInfoString);
                    if (withOverrides) {
                        columns.push({
                            name: "override",
                            type: "string"
                        });
                        data.forEach((record, i) => {
                            record.push($scope.records[i].overrideInfoString);
                        });
                    }
                    ExportUtils.exportUIData($scope, { name, data, columns }, "Export records");
                };
    
                // Init to empty arrays until we receive the constraints
                $scope.unlockedFeatures = [];
                $scope.shownFeatures = [];
                $scope.unlockedFeaturesNames = [];
    
                // Init to empty array until we receive the results
                $scope.records = [];
    
                const score = this.explorationLocalStore.getScore();
                $scope.reference = this.explorationLocalStore.getReference();
                $scope.prediction = score.prediction;
                $scope.target = this.explorationLocalStore.getTarget();
    
                const wt1Service = ExplorationWT1Service.init(this.isClassification, Object.keys($scope.reference).length);
    
                // Friendly warning message when the engine did not find anything
                $scope.searchFailed = false;
    
                // Top right buttons
                $scope.close = () => {
                    this.whatIfRouter.openMainView();
                };
                $scope.editConstraints = () => {
                    this.whatIfRouter.openConstraints();
                };
            }
        }
    });

    app.component('explorationResultsTable', {
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-results-table.html',
        bindings: { records: '<', features: '<', shownFeatures: '<', selection: '<' },
        controller: function($scope, $filter) {
            this.$onInit = function () {
                $scope.reference = this.records.find(x => x.isReference).values;
                $scope.selection = this.selection; // `filtered-multi-select-rows` interacts with `$scope.selection`
                $scope.shouldDisplayProbas = this.records.map(x => x.probas).every(p => p !== undefined);
                $scope.formatPrediction = prediction => ($scope.shouldDisplayProbas ? prediction : $filter('smartNumber')(Number(prediction)));
                $scope.plausibilityScale = getColorScale(this.records.map(x => x.plausibility));
                if ($scope.shouldDisplayProbas) {
                    $scope.probaScale = getColorScale(this.records.map(x => x.probas[x.prediction]));
                }
                $scope.isShown = feature => this.shownFeatures.includes(feature.name);
                // Mass action button
                $scope.atLeastOneRowIsShown = () => this.records.some(record => record.isShown);
                $scope.setVisibilityOfAllRows = isShown => this.records.forEach(record => { record.isShown = isShown; });
            }
            

            // Gradient color for plausibility and proba cells
            const getColorScale = values => {
                const min = d3.min(values);
                const max = d3.max(values);
                if (min === max) {
                    return () => "green"; // default scale
                }
                return d3.scale.linear().range(['red', 'orange', 'green']).domain([min, (max + min) / 2, max]);
            };

            // Highlight corresponding CFs when rows of the table are hovered
            $scope.setHover = (record, value) => {
                record.isHovered = value;
            };

            // Helpers
            $scope.getConstraintDescription = feature => feature.constraintHelper.getConstraintDescription(feature);
            $scope.formatProba = proba => d3.format(",.2f")(proba * 100);
            $scope.formatProbas = probas => {
                const MAX_NB_PROBAS_TO_DISPLAY = 4;
                let sortedProps = Object.keys(probas).map(key => ({ value: probas[key], key })).sort((a, b) => (b.value - a.value));
                if (sortedProps.length > MAX_NB_PROBAS_TO_DISPLAY) {
                    sortedProps = sortedProps.slice(0, MAX_NB_PROBAS_TO_DISPLAY - 1);
                    const sumTopProbas = sortedProps.reduce((acc, e) => acc + e.value, 0);
                    sortedProps.push({ key: 'Others', value: 1 - sumTopProbas });
                }
                return sortedProps.map(e => `${e.key}: ${d3.format(",.0f")(100 * e.value)}%`).join(' - ');
            }

            // Sorting default params
            $scope.sortColumn = 'loss';
            $scope.sortDescending = false;

            // Watch for changes because shownFeatures is a ng-model in the parent controller, so its reference is subject to change
            this.$onChanges = changes => {
                if (changes.shownFeatures) {
                    this.shownFeatures = changes.shownFeatures.currentValue;
                }
            }
        }
    });

    app.component('explorationResultsPlotRow', {
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-results-plot-row.html',
        bindings: { records: '<', shownFeatures: '=', unlockedFeatures: '<' },
        controller: function($scope) {
            $scope.getShownFeatures = () => this.shownFeatures; // shownFeatures is a ng-model in the parent controller, so its reference is subject to change
        }
    });

    app.component('frozenFeaturesPopin', {
        bindings: { lockedFeatureNames: '<', reference: '<'},
        template: `
            <a class="exploration-results__popover-button">
                <span class="icon-info-sign"></span>
                {{ $ctrl.lockedFeatureNames.length }} Frozen {{'feature' | plurify: $ctrl.lockedFeatureNames.length}}
            </a>`,
        controller: function($scope, $element, openDkuPopin) {
            const popinTemplate = `
                    <div class="popover dropdown-menu exploration-frozen-features" listen-keydown="{ 'esc': 'dismiss()' }">
                        <div class="arrow"></div>
                        <div class="popover-title">
                            Frozen {{'feature' | plurify: $ctrl.lockedFeatureNames.length}}
                        </div>
                        <ul class="exploration-frozen-features__list">
                            <li ng-repeat="lockedFeature in $ctrl.lockedFeatureNames" class="exploration-frozen-features__item">
                                <div class="exploration-frozen-features__item-name">
                                    {{ lockedFeature }}
                                </div>
                                <div class="exploration-frozen-features__item-value">
                                    {{ $ctrl.reference[lockedFeature] }}
                                </div>
                            </li>
                        </ul>
                    </div>`;

            let popinOpen = false;
            let dismissPopin = () => void 0;
            function togglePopin($event) {
                if (popinOpen === false) {
                    const dkuPopinOptions = {
                        template: popinTemplate,
                        isElsewhere: (elt, e) => $(e.target).parents(".dropdown-menu").length === 0,
                        onDismiss: () => { popinOpen = false; }
                    };
                    dismissPopin = openDkuPopin($scope, $event, dkuPopinOptions);
                    popinOpen = true;
                } else if (dismissPopin) {
                    dismissPopin();
                }
            }
            $element.on("click", togglePopin);
        }
    });

})();
