(function() {
'use strict';

const app = angular.module('dataiku.notebooks.sql', ['dataiku.services', 'dataiku.filters', 'dataiku.aiSqlGeneration']);

/**
 * Directive for defining implementations of shaker hooks relevant for SQL notebook result explore view.
 *
 * It uses the `$scope.queryIdentifier` from the containing component, along with state params (project key and
 * notebook id) to populate the required hook data.
 *
 * Note: Must be a separate directive with `scope: true` and not part of the parent component. This is required due to how
 *       hooks are declared in the directive `shakerExploreBase`.
 * */
app.directive("sqlNotebookResultExploreHooksDefinition", function() {
    return {
        scope: true,
        controller: function($scope, $q, $rootScope, $stateParams, ActiveProjectKey, DataikuAPI, MonoFuture) {
            $scope.shakerWithSteps = false;
            $scope.shakerState.filtersExplicitlyAllowed = true;
            /* ********************* Callbacks for shakerExploreBase ******************* */

            $scope.shakerHooks.saveForAuto = function () {
                const deferred = $q.defer();
                DataikuAPI.sqlNotebooks.saveExploreScript(ActiveProjectKey.get(), $stateParams.notebookId, $scope.queryIdentifier.cellId, $scope.shaker).success(function(ignored) {
                    $scope.originalShaker = angular.copy($scope.shaker);
                    deferred.resolve();
                }).error(setErrorInScope.bind($scope));
                return deferred.promise;
            };

            // We somehow miss that part of DataikuController, that is needed for shaker-in-dashboard
            $scope.setSpinnerPosition = $scope.setSpinnerPosition || function (position) {
                $rootScope.spinnerPosition = position;
            };

            let monoFuturizedRefresh = MonoFuture($scope).wrap(DataikuAPI.sqlNotebooks.sampleRefreshTable);

            $scope.shakerHooks.getRefreshTablePromise = function (filtersOnly, filterRequest) {
                return monoFuturizedRefresh(ActiveProjectKey.get(), $stateParams.notebookId, $scope.queryIdentifier.queryId, $scope.shaker, filterRequest);
            };

            $scope.shakerHooks.afterTableRefresh = function () {
                // NO-OP for now
            };

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                $scope.shaker.columnWidthsByName[name] = width;
                $scope.autoSaveAutoRefresh();
            };

            $scope.shakerHooks.getTableChunk = function (firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.sqlNotebooks.sampleGetTableChunk(ActiveProjectKey.get(), $stateParams.notebookId, $scope.queryIdentifier.queryId, $scope.shaker,
                    firstRow, nbRows, firstCol, nbCols, filterRequest)
            };

            $scope.shakerReadOnlyActions = true;
            $scope.shakerWritable = false;
            $scope.isCompareCellAvailable = false;

            $scope.shakerHooks.shakerForQuery = function () {
                return angular.copy($scope.shaker);
            }

            $scope.shakerHooks.fetchDetailedAnalysis = function (setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
                // withFullSampleStatistics, fullSamplePartitionId are not relevant in this context
                DataikuAPI.sqlNotebooks.sampleDetailedColumnAnalysis(ActiveProjectKey.get(), $stateParams.notebookId, $scope.queryIdentifier.queryId, $scope.shakerHooks.shakerForQuery(), columnName, alphanumMaxResults).success(function (data) {
                    setAnalysis(data);
                }).error(function (a, b, c) {
                    if (handleError) {
                        handleError(a, b, c);
                    }
                    setErrorInScope.bind($scope)(a, b, c);
                });
            };

            $scope.canWriteProject = () => $rootScope.topNav.isProjectAnalystRW;

            $scope.$watch("queryIdentifier", function(ov, nv) {
                if (nv) {
                    loadExploreParams();
                }
            }, true);

            function loadExploreParams() {
                DataikuAPI.sqlNotebooks.getExploreScript(ActiveProjectKey.get(), $stateParams.notebookId, $scope.queryIdentifier.cellId).success(function(data) {
                    $scope.shaker = data;
                    $scope.originalShaker = angular.copy($scope.shaker);
                    $scope.shaker.origin = "SQL_NOTEBOOK";
                    $scope.shaker.flagNumericValues = false;
                    $scope.shaker.$headerOptions = {
                        showName: true,
                        showMeaning: false,
                        showDescription: false,
                        showCustomFields: false,
                        showProgressBar: false,
                        showOriginalSQLTypes: true,
                        showHeaderSeparator: true
                    };
                    $scope.fixupShaker();
                    $scope.refreshTable(
                        false,
                        "SkipDebounceAndRunImmediately"
                    );
                }).error(setErrorInScope.bind($scope));
            }
        }
    }
});

app.component("sqlNotebookResultExplore", {
    templateUrl: "/templates/notebooks/sql-notebook-result-explore.html",
    bindings: {
        isRunning: '<',
        cellId: '<',
        queryId: '<'
    },
    controller: function($scope, translate) {
        const $ctrl = this;
        $scope.translate = translate;
        // Required for detailed analysis of alphanumeric columns
        $scope.sanitize = sanitize;

        $ctrl.$onChanges = () => {
            if (!$ctrl.isRunning) { // Only apply changes when query is done running
                $scope.queryIdentifier = {queryId: $ctrl.queryId, cellId: $ctrl.cellId};
            }
        };
    }
});

app.constant("SQL_NOTEBOOK_CHART_MODE", {
    CREATE: "create",
    EDIT: "edit"
});

app.controller('SqlNotebookChartEditorController', function ($scope, $rootScope, $controller, SqlNotebookResultDataFetcher, DashboardPageUtils, DSSVisualizationThemeUtils, ChartTypeChangeHandler, SQL_NOTEBOOK_CHART_MODE) {
    $controller('ShakerChartsCommonController', {$scope: $scope});
    $scope.isSqlNotebook = true;
    $scope.SQL_NOTEBOOK_CHART_MODE = SQL_NOTEBOOK_CHART_MODE;

    $scope.getDefaultNewChart = function() {
        const defaultTheme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
        return {
            theme: defaultTheme,
            def: ChartTypeChangeHandler.defaultNewChart(defaultTheme)
        };
    };

    $scope.overrideFormattingWithTheme = function(theme) {
        const currentChartCopy = angular.copy($scope.chart);
        $scope.chart.theme = theme;
        DSSVisualizationThemeUtils.applyToChart({ chart: $scope.chart.def, theme, formerTheme: $scope.chart.theme });
        DSSVisualizationThemeUtils.showThemeAppliedSnackbar($scope.chart, currentChartCopy);
    };

    $scope.$on('$destroy', () => DSSVisualizationThemeUtils.hideThemeAppliedSnackbar());

    $scope.validity = {}; // Needed by chart redraw
    $scope.getDataSpec = SqlNotebookResultDataFetcher.getDataSpec;

    $scope.saveChart = function() {
        // This editor is only to define the chart, saving is delegated to after modal resolution.
        // Therefore, there is no need to save here.
    };

    $scope.fixExpression = (expression, fixName = "plus") => {
        return SqlNotebookResultDataFetcher.fixExpression(expression, fixName, $scope.queryId).catch(setErrorInScope.bind($scope));
    };

    $scope.getExecutePromise = function(request, saveShaker = false, noSpinner = true, requiredSampleId = undefined, dataSpec = $scope.getDataSpec()) {
        return SqlNotebookResultDataFetcher.getPivotResponse(request, $scope.queryId);
    };

    $scope.fetchColumnsSummary = function () {
        return SqlNotebookResultDataFetcher.getColumnSummary($scope.queryId).then(function (columns) {
            $scope.chart.summary = columns.data;
            $scope.makeUsableColumns($scope.chart.summary);
        }).catch(setErrorInScope.bind($scope));
    };

    /** We are a bit stricter than just valid for creating/editing charts, as we only ever want to allow creation
     * of charts that can be rendered. Otherwise, user needs to make adjustments in the modal.
     *  */
    $scope.canConfirm = function() {
        return $scope.validity && $scope.validity.valid && !$scope.allFilteredOut && !$scope.display0Warning;
    };

    // There will be at most one chart in this editor
    $scope.charts = [];
    if ($scope.mode === SQL_NOTEBOOK_CHART_MODE.CREATE) {
        // This weird code will create a default chart and add it to charts
        $scope.addChart();
        $scope.chart = $scope.charts[$scope.currentChart.index];
        $scope.chart.customMeasures = [];
        $scope.chart.reusableDimensions = [];
        $scope.chart.hierarchies = [];
    } else {
        $scope.chart = $scope.chartToEdit;
        $scope.addChart({chart: $scope.chart});
    }

    $scope.validateChart = () => {
        $scope.resolveModal($scope.chart);
    };
    $scope.cancel = () => {
        $scope.dismiss();
    };

    // chartHandler options
    $scope.noThumbnail = true;
    $scope.addCustomMeasuresToScopeAndCache($scope.chart.customMeasures);
    $scope.addBinnedDimensionToScopeAndCache($scope.chart.reusableDimensions);
    $scope.addHierarchiesToScopeAndCache($scope.chart.hierarchies);
});

app.factory("SqlNotebookChartEditor", function($rootScope, Logger, CreateModalFromTemplate, SQL_NOTEBOOK_CHART_MODE) {
    function createOrEditChart(queryId, mode, chartToEdit, postConfirmCallBack) {
        const newScope = $rootScope.$new();
        newScope.mode = mode;
        newScope.queryId = queryId;
        if (chartToEdit) {
            newScope.chartToEdit = angular.copy(chartToEdit);
        }
        CreateModalFromTemplate("/templates/notebooks/sql-notebooks-new-chart-modal.html", newScope, 'SqlNotebookChartEditorController', undefined, true).then(function(chart) {
            Logger.info("Chart edition confirmed");
            postConfirmCallBack(chart);
        }).catch(() => {
            Logger.info("Chart edition dismissed");
        });
    }

    return {
        createChart: function(queryId, postCreationCallback) {
            createOrEditChart(queryId, SQL_NOTEBOOK_CHART_MODE.CREATE, null, postCreationCallback);
        },
        editChart: function(queryId, chartToEdit, postEditionCallback) {
            createOrEditChart(queryId, SQL_NOTEBOOK_CHART_MODE.EDIT, chartToEdit, postEditionCallback)
        }
    };
});

app.factory("SqlNotebookResultDataFetcher", function(ActiveProjectKey, DataikuAPI, $stateParams) {
    return {
        getColumnSummary: function(queryId) {
            return DataikuAPI.sqlNotebooks.getColumnSummary(ActiveProjectKey.get(), $stateParams.notebookId, queryId);
        },
        getPivotResponse: function(request, queryId) {
            return DataikuAPI.sqlNotebooks.getPivotResponse(request, ActiveProjectKey.get(), $stateParams.notebookId, queryId);
        },
        fixExpression: function(expression, fixName, queryId) {
            return DataikuAPI.sqlNotebooks.fixExpression(ActiveProjectKey.get(), $stateParams.notebookId, queryId, expression, fixName);
        },
        // This is not per se a data callback, but used in multiple places, so it is upgraded as a first class citizen
        getDataSpec: function() {
            return {
                datasetProjectKey: $stateParams.projectKey,
                datasetName: "result_from_sql_query"
            };
        }
    }
});

app.component("sqlNotebookResultChart", {
    templateUrl: "/templates/notebooks/sql-notebook-result-chart.html",
    bindings: {
        chart: '<',
        queryId: "<",
        isRunning: '<'
    },
    controller: function($scope, $element, $controller, ChartSetErrorInScope, ChartDefinitionChangeHandler, SqlNotebookResultDataFetcher, Logger, MonoFuture, ChartRequestComputer, translate, Debounce) {
        const $ctrl = this;
        $controller('ShakerChartsCommonController', { $scope: $scope });
        $scope.initChartCommonScopeConfig($scope);
        ChartSetErrorInScope.defineInScope($scope);

        $scope.translate = translate;

        $scope.getDataSpec = SqlNotebookResultDataFetcher.getDataSpec;

        function fetchColumnsSummary() {
            return SqlNotebookResultDataFetcher.getColumnSummary($ctrl.queryId).then(function (columns) {
                $scope.chart.summary = columns.data;
                $scope.makeUsableColumns($scope.chart.summary);
            });
        }

        $scope.getExecutePromise = function(request) {
            return SqlNotebookResultDataFetcher.getPivotResponse(request, $ctrl.queryId);
        };

        const executePivotRequest = MonoFuture($scope).wrap($scope.getExecutePromise);
        $scope.chartSpecific = {};
        $scope.noThumbnail = true;

        function computeChartResults() {
            Logger.info("Force recompute chart");
            fetchColumnsSummary().then(function() {
                const request = ChartRequestComputer.compute($scope.chart.def, $element.width(), $element.height(), $scope.chartSpecific);
                executePivotRequest(request).success(data => {
                    Logger.info('Got pivot response for chart');
                    resetErrorInScope($scope);
                    $scope.response = data;
                    $scope.setValidity({valid: true});
                }).error(function(data, headers, status) {
                    $scope.setValidity({valid: false});
                    $scope.chartSetErrorInScope(data, headers, status);
                });
            }).catch(setErrorInScope.bind($scope));
        }

        function redrawChart() {
            $scope.$broadcast("redraw");
        }

        // listener to handle chart redraw when window is horizontally resized
        const debouncedResizeListener = Debounce().withDelay(10,20).wrap(() => $scope.$broadcast('resize'));
        $(window).on('resize', debouncedResizeListener);
        $scope.$on('$destroy', () =>  $(window).off('resize', debouncedResizeListener));

        // Minimal handling of resizing, triggered when resizer is done moving
        $scope.$on("reflow", function() {
            redrawChart();
        });

        function chartPropertyChanged(newChart, oldChart, propertyName) {
            if (!newChart || !newChart[propertyName]) {
                return false;
            }
            if (!oldChart || !oldChart[propertyName]) {
                return true;
            }
            return !angular.equals(newChart[propertyName], oldChart[propertyName]);
        }

        function shouldChartDefinitionChangeTriggerRedraw(newChart, oldChart) {
            return ChartDefinitionChangeHandler.getFrontImportantChange(newChart, oldChart) ||
                ChartDefinitionChangeHandler.getDelayedFrontImportantChange(newChart, oldChart);
        }

        function shouldChartDefinitionChangeTriggerRecompute(newChart, oldChart) {
            return ChartDefinitionChangeHandler.getRecomputeChange(newChart, oldChart) || chartPropertyChanged(newChart, oldChart, 'referenceLines');
        }
        // Change logic to trigger change in chart or redraw
        let oldChart, oldQueryId;
        $ctrl.$doCheck = () => {
            if ($ctrl.isRunning) {
                return; // we are waiting for query result, we can't redraw anything
            }
            // First hit or when changing query, we know we want to redraw
            if (!oldChart || oldQueryId !== $ctrl.queryId) {
                oldChart = angular.copy($ctrl.chart);
                oldQueryId = $ctrl.queryId;
                $scope.chart = $ctrl.chart;
                computeChartResults();
                return;
            }
            // Otherwise, check if the chart definition has changed
            // If so, infer if we need complete chart computation or if it's pure UI change (like changing labels, ...)
            // This part is kind of duplicated with the recompute logic from chart-configuration directive
            // that is called whenever there is an internal change in its scope, because in our case
            // the change comes from the outside (the edit modal).
            if (!angular.equals(oldChart, $ctrl.chart)) {
                $scope.chart = $ctrl.chart;
                if (shouldChartDefinitionChangeTriggerRecompute($ctrl.chart.def, oldChart.def)) {
                    computeChartResults();
                } else if (shouldChartDefinitionChangeTriggerRedraw($ctrl.chart.def, oldChart.def)) {
                    redrawChart();
                }
                oldChart = angular.copy($ctrl.chart);
            }
        };
    }
});

app.controller('SQLNotebookControllerCM6', function (
        $scope, $state, $timeout, $q, $modal, $stateParams, $rootScope,
        Assert, WT1, Logger, DataikuAPI, Dialogs, TopNav, BigDataService,
        LocalStorage, SQLExplorationService, CreateExportModal, Debounce,
        ExportUtils, CreateModalFromTemplate, NotebooksUtils, MonoFuture, AISqlGenerationService, RatingFeedbackParams, ClipboardUtils,
        Ng2SQLNotebookStore
) {
	/* *********************** Basic CRUD ********************* */
    $scope.initSQLNotebookStore = function() {
        Ng2SQLNotebookStore.initNotebook($scope.notebookParams, $scope.connectionDetails);
    }

    $scope.clearSQLNotebookStore = function() {
        Ng2SQLNotebookStore.clearNotebook();
    }

    $scope.loadNotebook = function() {
    	return DataikuAPI.sqlNotebooks.getSummary($stateParams.projectKey, $stateParams.notebookId).then(({data}) => {
            $scope.notebookParams = data.object;
            $scope.objectInterest = data.interest;
            $scope.objectTimeline = data.timeline;

            TopNav.setItem(TopNav.ITEM_SQL_NOTEBOOK, $scope.notebookParams.id, {
                name : $scope.notebookParams.name
                , isHive: $scope.notebookParams.language == 'HIVE'
                , isImpala: $scope.notebookParams.language == 'IMPALA'
                , isSpark: $scope.notebookParams.language == 'SPARKSQL'
            });
            TopNav.setPageTitle($scope.notebookParams.name + " - SQL");

            WT1.event("sql-notebook-load", {language: $scope.notebookParams.language});
        }).then(() => DataikuAPI.sqlNotebooks.listConnections($stateParams.projectKey)).then(({data}) => {
            const connections = data;
            $scope.connectionDetails = undefined;
            $scope.connectionFailed = false;
            for (var i = 0; i < connections.nconns.length; i++) {
                if (connections.nconns[i].name == $scope.notebookParams.connection) {
                    $scope.connectionDetails = connections.nconns[i];
                    $scope.initSQLNotebookStore();
                    break;
                }
            }
            if(!$scope.connectionDetails) {
                $scope.connectionFailed = true;
            } else {
                $scope.notebookMode = $scope.connectionDetails.type == "HiveServer2" ? "HIVE" : "SQL";
            }

            $scope.notebookLocalState = {};
            $scope.notebookTmpState = {};
            $scope.notebookLocalState.tableListingMode = 'PROJECT';
            $scope.notebookLocalState.tableOrdering = 'TABLE';
            $scope.notebookLocalState.leftPaneTab = 'Cells';

            $scope.$watch("notebookLocalState", $scope.saveLocalStates, true);

            $scope.updateNotebookHistory().then(function() {
                $scope.$broadcast("history-first-time-loaded");
            });

            if($scope.connectionDetails) {
                $scope.$broadcast('notebookLoaded');
            }
    	}).catch(setErrorInScope.bind($scope));
    };

    $scope.refreshTimeline = function(projectKey) {
        DataikuAPI.timelines.getForObject(projectKey || $stateParams.projectKey, "SQL_NOTEBOOK", $stateParams.notebookId).success(function(data) {
            $scope.objectTimeline = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.copyNotebook = () => {
        NotebooksUtils.copySqlOrSearchNotebook($stateParams.projectKey, "SQL", $stateParams.notebookId, $scope.notebookParams.name, $scope);
    };

	function saveParamsPromise() {
        let deferred = $q.defer();
        Logger.info("Saving notebook params");
		DataikuAPI.sqlNotebooks.save($scope.notebookParams).success(function(data) {
            Logger.info("Notebook params saved");
			if (data && data.versionTag) {
				$scope.notebookParams.versionTag = data.versionTag;
			}
			doSaveLocalStates();
			$scope.refreshTimeline();
            deferred.resolve("ok");
	   	}).error(setErrorInScope.bind($scope));
        return deferred.promise;
	}

    const debouncer = Debounce().withScope($scope).withDelay(5000, 5000)
    const debouncedSave = debouncer.wrap(saveParamsPromise);
    /**
     * Synchronously flushes any pending save to ensure data consistency. This solves a "race condition" between a
     * debounced save and subsequent action, like running a query.
     *
     * @returns a promise that resolves immediately if no save is pending, otherwise returns save promise.
     * */
    $scope.flushPendingSave = function() {
        if (!debouncer.active()) {
            return $q.when("ok");
        }
        debouncer.abort(); // Cancel timer to execute save immediately
        return saveParamsPromise();
    };

    /* Auto save */
    $scope.$watch("notebookParams", function(nv, ov) {
        if (nv && ov) {
            Logger.info("Notebook params updated");
            const beforeCopy = angular.copy(ov);
            const afterCopy = angular.copy(nv);
            beforeCopy.versionTag = null;
            afterCopy.versionTag = null;
            if (angular.equals(beforeCopy, afterCopy)) {
                Logger.debug("Only the version tag was modifed, ignoring")
                return;
            }
            debouncedSave();
        }
    }, true);

	/* *********************** Initialization code **************** */

    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, 'notebooks', TopNav.TABS_SQL_NOTEBOOK, "query");
    TopNav.setItem(TopNav.ITEM_SQL_NOTEBOOK, $stateParams.notebookId);

    $scope.uiState = { codeSamplesSelectorVisible: false };

	$scope.loadNotebook().then(function() {
        $scope.cells = $scope.notebookParams.cells;//TODO is that useful?
        let cellToSelectIndex;
        $scope.cells.forEach(function(cell, idx){
            cell.$localState = cell.$localState || {}; // localStorage, not sent to server
            cell.$tmpState = cell.$tmpState || {}; // non persistent state
            cell.type = cell.type || 'QUERY';
            if (cell.type == 'QUERY') {
                cell.$localState.query = cell.$localState.query || {};
                var q = cell.$localState.query;
                if ((!q.sql || !q.sql.length) && cell.code) {
                    q.sql = cell.code;
                }
            }
            if($stateParams.cellId && cell.id === $stateParams.cellId) {
                cellToSelectIndex = idx;
            }
        })
        loadLocalStates();
        if(cellToSelectIndex) {
            $scope.selectCellAndScroll(cellToSelectIndex);
        }
    });

    $scope.$watch("notebookLocalState.leftPaneTab", function() {
        $scope.$broadcast("reflow");
    });

    /* ************  Cells management *************** */

    var getSpecifiedCellOrSelectedOrLast = function(index) {
        if (index !== undefined) return index;
        var selected = $scope.getSelectedCellIndex();
        if (selected !== undefined) return selected;
        var last = $scope.cells.length - 1;
        return last;
    };

    $scope.selectCell = function(index) {
        $scope.cells.forEach(function(cell, idx) {
            var wasSelected = cell.$localState.selected;
            cell.$localState.selected = idx == index;
            if (!wasSelected && cell.$localState.selected && cell.focusQuery) {
                $timeout(function(){
                    cell.focusQuery();
                }, 500);
            }
        });
        loadChartParamsIfNeeded();
        loadResultsIfNeeded();
    };

    //returns the index of the selected cell or undefined if none selected
    $scope.getSelectedCellIndex = function() {
        if (!$scope.cells) return;
        for (var i = 0; i < $scope.cells.length; i++) {
            var cell = $scope.cells[i];
            if (cell.$localState && cell.$localState.selected) return i;
        }
    };

    $scope.selectedCell = function() {
        if (!$scope.cells) return;
        return $scope.cells[$scope.getSelectedCellIndex()];
    };

    $scope.addCell = function(type, index) {
        if (index === undefined) index = $scope.cells.length;
        var cell = {
            type: type,
            id: generateUniqueId(),
            $localState: {unfolded: true},
            $tmpState: {},
            querySettings: {
                addLimitToStatement: true,
                statementsParseMode : "SPLIT",
                statementsExecutionMode : "PREPARED"
            }
        };
        $scope.cells.splice(index, 0, cell);
        $scope.selectCell(index);
        $scope.scrollToCell(index);

        WT1.event("sql-notebook-add-cell", {number_of_cells: $scope.cells.length});
        return cell;
    };

    $scope.removeCell = function(index) {
        if (index === undefined) return;
        Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to remove this cell ?").then(function() {
            $scope.cells.splice(index, 1);
            if ($scope.cells.length) {
                $scope.selectCell(index < $scope.cells.length ? index : $scope.cells.length - 1);
            }
        });
    };

    $scope.duplicateCell = function(index) {
        if (index === undefined) return;
        WT1.event("sql-notebook-duplicate-cell", {});
        var originalCell = $scope.cells[index];
        //avoid copying some things. Not very pretty...
        var l = originalCell.$localState;
        var t = originalCell.$tmpState;
        originalCell.$localState = {};
        originalCell.$tmpState = {};
        var newCell = angular.copy(originalCell);
        newCell.id = Math.random();
        originalCell.$localState = l;
        originalCell.$tmpState = t;
        if (originalCell.type == 'QUERY') {
            newCell.$localState.query = {sql: l.query.sql};
        }
        $scope.cells.splice(index + 1, 0, newCell);
    };

    $scope.moveCell = function(index, shift) {
        if (index === undefined) return;
        WT1.event("sql-notebook-move-cell", {});
        if (index + shift < 0) {
            shift = -index;
        } else if (index + shift >= $scope.cells.length) {
            shift = $scope.cells.length - 1 - index;
        }
        var cell = $scope.cells[index + shift];
        $scope.cells[index + shift] = $scope.cells[index];
        $scope.cells[index] = cell;
        $scope.scrollToCell(index + shift);
    };

    $scope.hasUnfoldedCell = function() {
        if (!$scope.cells) return;
        for (var i = 0; i < $scope.cells.length; i++) {
            var cell = $scope.cells[i];
            if (cell.$localState && cell.$localState.unfolded) return true;
        }
        return false;
    };

    $scope.unfoldAllCells = function(unfold) {
        WT1.event("sql-notebook-unfold-all-cells", {});
        if (!$scope.cells) return;
        for (var i = 0; i < $scope.cells.length; i++) {
            var cell = $scope.cells[i];
            cell.$localState.unfolded = unfold;
        }
    };

    $scope.selectCellAndScroll = function(index) {
        $scope.selectCell(index);
        $timeout(function(){
            $scope.scrollToCell(index);
        }, 200)
    };

    $scope.scrollToCell = function(index) {
        $timeout(function() {
            $('.multi-query-editor').scrollTop($('.sql-notebook-cell')[index].offsetTop);
        }, 100);
    };

    $scope.ratingFeedbackParams = RatingFeedbackParams;
    //This is necessary to make sure  the banner disappears when we route to another page (otherwise, it appears again after we open a recipe)
    $scope.$on('$stateChangeStart', () => {
        $scope.ratingFeedbackParams.showRatingFeedback = false;
    });       

    $scope.aiGenerateSQLEnabled = $rootScope.appConfig.aiGenerateSQLEnabled;
    $scope.aiQuery = {};
    $scope.isAiGenerationFutureRunning = false;
    const requestOrigin = "SQL_NOTEBOOK";
    let requestId = null;
    $scope.aiGenerationFuture = MonoFuture($scope);
    $scope.generate = function() {
        WT1.tryEvent('ai-sql-generation', () => {
            return {
                ...AISqlGenerationService.buildWT1SqlGenerationParams(requestOrigin, null, $scope.connectionDetails.type),
                aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiGenerateSQL ? "webapp" : "default"
            };
        });
        $scope.ratingFeedbackParams.showRatingFeedback = false;
        $scope.ratingFeedbackParams.requestIdForFeedback = null;
        $scope.ratingFeedbackParams.featureRated = "text2sql";
        $scope.aiQuery = {};
        $scope.isAiGenerationFutureRunning = true;
        $scope.aiGenerationFuture.wrap(DataikuAPI.sqlNotebooks.startSQLQueryGeneration)(
            $stateParams.projectKey,
            $scope.notebookLocalState.generationQuery,
            $scope.connectionDetails.name
        ).success(function(data) {
            $scope.aiQuery = AISqlGenerationService.processSQLGenerationResponse(data.result.queryName, data.result.sqlQuery, data.result.reasoning, data.result.messages);
            requestId = data.result.requestId;
            WT1.tryEvent('ai-sql-generation-response', () => {
                return {
                    ...AISqlGenerationService.buildWT1SqlGenerationResponseParams(requestOrigin, $scope.connectionDetails.type, requestId, $scope.aiQuery.warnings, $scope.aiQuery.errors),
                    aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiGenerateSQL ? "webapp" : "default"
                };
            });
            $scope.installingFuture = null;
            if (!$rootScope.appConfig.isUsingLocalAiAssitant?.aiGenerateSQL) {
                $scope.ratingFeedbackParams.showRatingFeedback = true;
                $scope.ratingFeedbackParams.requestIdForFeedback = data.result.requestId;
                $scope.ratingFeedbackParams.featureRated = "text2sql";
            }
        }).error(function (data, status, headers) {
            setErrorInScope.bind($scope)(data, status, headers);
            WT1.tryEvent('ai-sql-generation-error', () => {
                return {
                    ...AISqlGenerationService.buildWT1SqlErrorResponseParams(requestOrigin, $scope.connectionDetails.type),
                    aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiGenerateSQL ? "webapp" : "default"
                };
            });
        }).finally(function() {
            $scope.isAiGenerationFutureRunning = false;
        });
    };

    $scope.abortAiGeneration = function() {
        $scope.aiGenerationFuture.abort();
        $scope.isAiGenerationFutureRunning = false;
    };

    $scope.createAiQuery = function() {
        const toInsert = AISqlGenerationService.getAIQueryToInsert($scope.aiQuery);
        var cell = $scope.addCell('QUERY');
        cell.name = $scope.aiQuery.name;
        cell.$localState.query = {sql: toInsert};
        $scope.notebookLocalState.leftPaneTab = 'Cells';
        WT1.tryEvent('ai-sql-insert-query', () => {
            return {
                ...AISqlGenerationService.buildWT1SqlInsertQueryParams(requestOrigin, $scope.connectionDetails.type, requestId),
                aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiGenerateSQL ? "webapp" : "default"
            };
        });
    };

    $scope.disableGenerateButton = () => AISqlGenerationService.isGenerateButtonDisabled($rootScope.topNav.isProjectAnalystRW, $scope.notebookLocalState.generationQuery)
    $scope.disableGenerateButtonMessage = () => AISqlGenerationService.getGenerateButtonDisabledMessage($rootScope.topNav.isProjectAnalystRW);
    $scope.copyAiQueryToClipboard = () => ClipboardUtils.copyToClipboard($scope.aiQuery.query, "SQL query copied to clipboard!");

    $scope.filterCells = function() {
        var filteredCells = $scope.cells;
        if (!$scope.cells || !$scope.notebookLocalState) return;
        var query = $scope.notebookLocalState.cellsQuery;
        if (query && query.trim().length) {
            angular.forEach(query.split(/\s+/), function(token){
                token = token.toLowerCase();
                if (token.length) {
                    filteredCells = $.grep(filteredCells, function(cell){
                        return cell.name && cell.name.toLowerCase().indexOf(token) >= 0 ||
                        (cell.$localState && cell.$localState.query && cell.$localState.query.sql && cell.$localState.query.sql.toLowerCase().indexOf(token) >= 0);
                    });
                }
            });
            $scope.cells.forEach(function(cell){
                cell.$tmpState.filteredOut = true;
            });
        }
        filteredCells.forEach(function(cell){
            cell.$tmpState.filteredOut = false;
        });
        $scope.filteredCells = filteredCells.length;
    };
    $scope.$watchCollection("cells", $scope.filterCells);
    $scope.filterCells();

    /* ************  Locally persistent state *************** */

    function localStorateId() {
        return "SQL_NOTEBOOK_"+$stateParams.projectKey + "_" + $scope.notebookParams.id;
    }

    var onLocalStatesLoaded = function() {
        // Make sure a cell is selected
        if ($scope.cells && $scope.cells.length) {
            var selectedCell = $scope.selectedCell();
            if (!selectedCell) {
                $scope.selectCell(0);
            } else { // This is required for first cell selection coming from local state
                loadChartParamsIfNeeded();
            }
        }
    };

    var loadLocalStates = function() {
        if (!$scope.cells) return;
        var localState = LocalStorage.get(localStorateId()) || {cellsStates:{}};
        Logger.info("Loading SQL notebook local state", localState);
        if (localState.notebookLocalState && localState.notebookLocalState.versionTag && localState.notebookLocalState.versionTag.versionNumber < $scope.notebookParams.versionTag.versionNumber) {
            Logger.info("Local state is outdated, discarding saved queries");
            // the localState is outdated, discard all saved queries
            $.each(localState.cellsStates, function(id, cell) {
                delete cell.query;
            });
        }
        $scope.notebookLocalState = $.extend($scope.notebookLocalState || {}, localState.notebookLocalState);
        cleanLocalState();
        if (!$scope.aiGenerateSQLEnabled && $scope.notebookLocalState.leftPaneTab === 'Generate') {
            $scope.notebookLocalState.leftPaneTab = 'Tables';
        }
        $scope.cells.forEach(function(cell){
            cell.$localState = $.extend(cell.$localState || {}, localState.cellsStates[cell.id]);
        });
        onLocalStatesLoaded();
    };
    
    function cleanLocalState() {
        // Cell mode is deprecated and no longer used. We remove it to avoid confusion.
        delete $scope.notebookLocalState.cellMode;
    }

    var doSaveLocalStates = function() {
        if (!$scope.notebookParams) return;
        var localState = {cellsStates: {}};
        localState.notebookLocalState = $scope.notebookLocalState;
        localState.notebookLocalState.versionTag = $scope.notebookParams.versionTag
        if ($scope.cells) {
            $scope.cells.forEach(function(cell){
                localState.cellsStates[cell.id] = cell.$localState;
            });
        }
        const now = new Date().getTime();
        LocalStorage.set(localStorateId(), localState);
        Logger.info("Saved SQL notebook local state time=" + (new Date().getTime() - now));
    };

    var saveLocalStateTimer;
    $scope.saveLocalStates = function() {
        $timeout.cancel(saveLocalStateTimer);
        saveLocalStateTimer = $timeout(doSaveLocalStates, 250);
    };

    /* ************  Text insertion *************** */

    $scope.insertText = function (text, addSelectIfEmptyQuery) {
        Ng2SQLNotebookStore.insertText(text, addSelectIfEmptyQuery);
    }

    $scope.onTableClicked = function(table) {
        $scope.insertText(table.quoted, true);
    };

    $scope.onFieldClicked = function(field) {
        $scope.insertText(field.quotedName);
    };

    $scope.onTableExplorerError = function(args) {
        return setErrorInScope.apply($scope, args);
    }

    $scope.insertCodeSnippet = function(snippet) {
        $scope.insertText(snippet.code);
    }

    /* ************ History ************ */

    $scope.updateNotebookHistory = function() {
        var deferred = $q.defer();
        DataikuAPI.sqlNotebooks.getHistory($stateParams.projectKey, $stateParams.notebookId).success(function(data) {
            $scope.notebookTmpState.history = data;
            deferred.resolve("ok");
        }).error(setErrorInScope.bind($scope));
        return deferred.promise;
    };

    $scope.clearHistory = function(cellId) {
        var deferred = $q.defer();
        Dialogs.confirm($scope, 'Are you sure you want to clear query history?').then(function() {
            WT1.event("sql-notebook-clear-history", {});
            DataikuAPI.sqlNotebooks.clearHistory($stateParams.projectKey, $stateParams.notebookId, cellId).success(function(data) {
                delete $scope.notebookTmpState.history[cellId];
                deferred.resolve("ok");
            }).error(setErrorInScope.bind($scope));
        });
        return deferred.promise;
    };

    var loadResultsIfNeeded = function() {
        var cell = $scope.selectedCell();
        if (cell.$localState.selected) {
            if (!$scope.fetchingResults[cell.id] && cell.$tmpState.lastQuery  && !cell.$tmpState.runningQuery && !cell.$tmpState.results) { //&& cell.$tmpState.lastQuery.state == 'DONE'
                $scope.fetchLastResults(cell, cell.$tmpState.lastQuery.id)
            }
        }
    };

    function loadChartParamsIfNeeded() {
        const cell = $scope.selectedCell();
        if (!cell.$localExploreChart) {
            DataikuAPI.sqlNotebooks.getExploreChart($stateParams.projectKey, $stateParams.notebookId, cell.id).success(function(data) {
                cell.$localExploreChart = data;
            }).error(setErrorInScope.bind($scope));
        }
    }

    $scope.loadQuery = function(cell, query, fetchResults) {
        Assert.trueish(cell, 'no cell');
        delete cell.$tmpState.results;
        cell.$localState.query = angular.copy(query);
        cell.$localState.unfolded = true;
        if(query.state == "RUNNING" || query.state == "NOT_STARTED") {
            cell.$tmpState.runningQuery = angular.copy(query);
            cell.$tmpState.waitFuture();
        } else {
            cell.$tmpState.lastQuery = angular.copy(query);
            loadResultsIfNeeded();
        }
    };

    $scope.createCellWithQuery = function(hQuery, index) {
        var initialCell = $scope.selectedCell();
        var cell = $scope.addCell('QUERY', index);
        cell.name = initialCell.name ? initialCell.name + '_copy' : '';
        cell.$localState.query = cell.$localState.query || {};
        cell.$localState.query.sql = hQuery.sql;
        //TODO add the query to the cell history
        $scope.notebookLocalState.leftPaneTab = 'Cells';
    };

    $scope.removeQuery = function(q) {
        var cell = $scope.selectedCell();
        if (cell && cell.$tmpState && cell.$tmpState.removeQuery) {
            cell.$tmpState.removeQuery(q);
        }
    };

    $scope.fetchingResults = {};
    $scope.fetchLastResults = function(cell, queryId) {
        if ($scope.fetchingResults[cell.id]) {
            Logger.warn("Cell Already fetching results", cell.id);
            return;
        }
        $scope.fetchingResults[cell.id] = true;
        return DataikuAPI.sqlNotebooks.getHistoryResult($stateParams.projectKey, $stateParams.notebookId, queryId).then(function(resp) {
            cell.$tmpState.results = resp.data;
            cell.$tmpState.clearError();
            return resp;
        }, function(resp) {
            cell.$tmpState.error(resp.data, resp.status, resp.headers);
        })
        .finally(function(){
            delete $scope.fetchingResults[cell.id];
        });
    };

    $scope.ratingFeedbackParams.showRatingFeedback = false;
    $scope.ratingFeedbackParams.requestId = null;
    $scope.ratingFeedbackParams.featureRated = "text2sql";

    $scope.$on("$destroy", () => {
        $scope.clearSQLNotebookStore();
    });

});

/**
 * Controller to be removed once we remove feature flag dku.feature.newSQLNotebookEditor
 * isolates all overrides to new sql editor controller that are necessary for legacy editor to work
 */
app.controller('SQLNotebookControllerCM5', function ($scope, $controller, Logger) {
    $controller('SQLNotebookControllerCM6', {$scope: $scope});

    $scope.insertText = function(text, addSelectIfEmptyQuery) {
        var cell = $scope.selectedCell();
        if (cell && cell.$tmpState.insertText) {
            cell.$tmpState.insertText(text, addSelectIfEmptyQuery);
        } else {
            Logger.warn("Cannot insert text: no current cell or no insertText function");
        }
    }

    $scope.initSQLNotebookStore = function () {
        // NO-OP
    }

    $scope.clearSQLNotebookStore = function() {
        // NO-OP
    }

});


app.controller('SqlNotebookQueryCellController', function ($scope, $element, $stateParams, $timeout, $q, DataikuAPI, WT1, Logger, $rootScope,
               StateUtils, Dialogs, CreateModalFromTemplate, CreateExportModal, BigDataService, SQLExplorationService, ExportUtils, CodeMirrorSettingService, DKUSQLFormatter, Ng2SQLNotebookStore, SpinnerService) {

    /* ************  Execution ************ */

    function resetResults() {
        delete $scope.cell.$tmpState.logs;
        if ($scope.cell.$tmpState.results) {
            delete $scope.cell.$tmpState.results.hasResultset;
        }
    }

    $scope.run = function(selectedStatement) {
        $scope.flushPendingSave().then(function() {
            runStatement(selectedStatement);
        });
    };

    function runStatement(selectedStatement) {
        if($scope.isQueryEmpty() || $scope.cell.$tmpState.runningQuery) {
            return;
        }
        resetResults();
        WT1.event("sql-notebook-run", {});

        var query = angular.copy($scope.cell.$localState.query);
        query.id = Math.random();
        query.connection = $scope.connectionDetails.name;
        query.mode = $scope.notebookMode;
        query.querySettings = $scope.cell.querySettings;
        if (selectedStatement) {
            $scope.fullQueryBeforeSelection = query.sql;
            query.sql = selectedStatement;
        }

        var full = false;
        
        $scope.cell.$tmpState.initializingQuery = true;
        DataikuAPI.sqlNotebooks.run($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id, query, full).success(function(startedQuery) {
            $scope.cell.$tmpState.runningQuery = startedQuery.toAddtoHistory;
            $scope.waitFuture();
            $scope.updateCellHistory();
            if ($scope.notebookMode == 'HIVE') {
                $scope.cell.$tmpState.resultsTab = 'LOGS';
            }
            $scope.cell.$tmpState.clearError();
        }).error($scope.cell.$tmpState.error)
        .finally(function(){$scope.cell.$tmpState.initializingQuery = false;});
    }

    $scope.abort = function() {
        if(!$scope.cell.$tmpState.runningQuery) return;
        WT1.event("sql-notebook-abort", {});
        DataikuAPI.sqlNotebooks.abort($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id, $scope.cell.$tmpState.runningQuery.id).success(function(data) {
            // No need to do more, the next future refresh it will handle it
        }).error($scope.cell.$tmpState.error);
    };

    $scope.computeFullCount = function(full) {
        resetResults();
        WT1.event("sql-notebook-full-count", {});
        DataikuAPI.sqlNotebooks.computeFullCount($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id, $scope.cell.$localState.query.id).success(function(startedQuery) {
            $scope.cell.$tmpState.runningQuery = startedQuery.toAddtoHistory;
            $scope.cell.$localState.query = angular.copy(startedQuery.toAddtoHistory);
            $scope.waitFuture();
            $scope.updateCellHistory();

            if ($scope.notebookMode == 'HIVE') {
                $scope.cell.$tmpState.resultsTab = 'LOGS';
            }
            $scope.cell.$tmpState.clearError();
        }).error($scope.cell.$tmpState.error);
    };

    $scope.waitFuture = function() {
        $scope.stopfutureTimer();
        DataikuAPI.sqlNotebooks.getProgress($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id, $scope.cell.$tmpState.runningQuery.id).success(function(data) {
            $scope.runningStatus = data;
            $scope.cell.$tmpState.runningQuery = data.query;
            if (data.logTail) {
                $scope.cell.$tmpState.logs = data.logTail;
            }
            if (data.running) {
                $scope.stopfutureTimer();
                $scope.futureTimer = $timeout($scope.waitFuture, 1000);
            } else {
                $scope.onFutureDone(data);
            }
        }).error(function(a,b,c) {
            $scope.onFutureFailed(a,b,c);
        });
    };
    $scope.cell.$tmpState.waitFuture = $scope.waitFuture;

    $scope.onFutureDone = function() {
        var flr = $scope.fetchLastResults($scope.cell, $scope.cell.$tmpState.runningQuery.id)
        if (flr) {
            flr.then(function(){
                $scope.stopfutureTimer();

                $scope.cell.$localState.query = angular.copy($scope.cell.$tmpState.runningQuery);//TODO move query to tmpState and remove running query
                if ($scope.fullQueryBeforeSelection) {
                    $scope.cell.$localState.query.sql = $scope.fullQueryBeforeSelection;
                    delete $scope.fullQueryBeforeSelection;
                }
                $scope.cell.$tmpState.lastQuery = angular.copy($scope.cell.$tmpState.runningQuery);

                $scope.cell.$tmpState.resultsTab = 'RESULTS';

                delete $scope.cell.$tmpState.runningQuery;
                $scope.updateCellHistory();
            });
        }
    };

    $scope.onFutureFailed = function(a,b,c) {
        $scope.cell.$tmpState.error(a,b,c);
        $scope.runningStatus = null;
        $scope.updateCellHistory(); // TODO per cell
    };

    $scope.stopfutureTimer = function() {
        if($scope.futureTimer) {
            $timeout.cancel($scope.futureTimer);
            $scope.futureTimer = null;
        }
    };

    $scope.showExecutionPlan = function(selectedStatement) {
        if($scope.isQueryEmpty()) {
            return;
        }
        var query = angular.copy($scope.cell.$localState.query);
        query.id = Math.random();
        query.connection = $scope.connectionDetails.name;
        query.mode = $scope.notebookMode;
        query.querySettings = $scope.cell.querySettings;
        if (selectedStatement) {
            query.sql = selectedStatement;
        }

        DataikuAPI.sqlNotebooks.getExecutionPlan($stateParams.projectKey, query).success(function(data) {
            CreateModalFromTemplate("/templates/recipes/fragments/sql-modal.html", $scope, null, function(newScope) {
                newScope.executionPlan = data.executionPlan;
                newScope.failedToComputeExecutionPlan = data.failedToComputeExecutionPlan;
                if (!data.failedToComputeExecutionPlan){
                    newScope.query = data.executionPlan.query;
                }
                newScope.uiState = {currentTab: 'plan'};
                newScope.engine = query.mode;
                newScope.isNotebook = true;
            });
        }).error($scope.cell.$tmpState.error);
    };

    /* ************ Code edition ************ */
    // Provide CM5-like API while we are in CM6 world.
    // We are doing this to support both editors with the same code base.
    // TODO remove once we remove feature flag dku.feature.newSQLNotebookEditor.enabled
    $scope.cm = {
        getSelection() {
            return $scope.codeSelectionContent;
        },
        focus() {
            Ng2SQLNotebookStore.focusEditor();
        },
        // NO-OPs
        setOption() {},
        on() {}
    }

    $scope.autocompleteSQL = function(cm,type) {
        SpinnerService.lockOnPromise(
            SQLExplorationService.listTables($scope.notebookParams.connection, $stateParams.projectKey).then(function(tables) {
            var fieldsToAutocomplete = CodeMirror.sqlFieldsAutocomplete(cm, tables);
            if (fieldsToAutocomplete && fieldsToAutocomplete.length) {
                SQLExplorationService.listFields($scope.notebookParams.connection, fieldsToAutocomplete).then(function(data) {
                    CodeMirror.showHint(cm, function(editor) {
                        return CodeMirror.sqlNotebookHint(editor, type+"-notebook", tables.map(function(t) {return t.table;}),data);
                    }, {completeSingle:false});
                });
            } else {
                CodeMirror.showHint(cm, function(editor){
                    return CodeMirror.sqlNotebookHint(editor, type+"-notebook", tables.map(function(t) {return t.table;}), null);
                }, {completeSingle:false});
            }
        }));
    };

    // TODO: remove when we end support for CM5
    // Triggered within CM5 insertText()
    $scope.cell.$tmpState.insertText = function(text, addSelectIfEmptyQuery) {
        if (addSelectIfEmptyQuery && $scope.isQueryEmpty()) {
            text = 'SELECT * FROM '+text;
        }
        $scope.cm.replaceSelection(text);
        var endPos = $scope.cm.getCursor(false);
        $scope.cm.setCursor(endPos);
        $scope.cm.focus();
    };

    $scope.cell.$tmpState.removeQuery = function(hQuery) {
        // if(q.id == $scope.query.id) {
        //     $scope.query.id = undefined;
        //     $scope.query.cachedResult = undefined;
        // }

        DataikuAPI.sqlNotebooks.removeQuery($stateParams.projectKey, $stateParams.notebookId, $scope.selectedCell().id, hQuery.id)
            .success($scope.updateCellHistory)
            .error($scope.cell.$tmpState.error);
    };

    /* ************ History ************ */

    $scope.showHistoryModal = function() {
        CreateModalFromTemplate("/templates/notebooks/sql-notebook-history-modal.html", $scope);
    };

    $scope.updateCellHistory = function() {
        var deferred = $q.defer();
        DataikuAPI.sqlNotebooks.getCellHistory($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id).success(function(data) {
            $scope.notebookTmpState.history = $scope.notebookTmpState.history || {};
            $scope.notebookTmpState.history[$scope.cell.id] = data;
            deferred.resolve("ok");
            $scope.cell.$tmpState.clearError();
        }).error($scope.cell.$tmpState.error);
        return deferred.promise;
    };

    /* ************ Export to recipe ************ */

    // https://developer.mozilla.org/en/docs/Web/JavaScript/Guide/Regular_Expressions
    function escapeRegExp(string){
        return string.replace(/([.*+?^=!:${}()|[\]/\\])/g, "\\$1");
    };

    $scope.isRecipeEditor = function() {
        return $scope.notebookParams.recipeId;
    }

    $scope.createRecipe = function() {
        WT1.event("sql-notebook-create-recipe", {});
        var script = _.get($scope, 'cell.$localState.query.sql', '');
        var fromPosition = script.search(/\sfrom\s/i);
        var candidateTables = [];

        if(fromPosition != -1) {
            // Extract table names
            var afterFrom = script.substring(fromPosition+5).toLowerCase();
            candidateTables = afterFrom.split(/['"`.\s]+/);
        }

        // Load table mapping
        DataikuAPI.connections.getSQLTableMapping($scope.notebookParams.connection).success(function(mapping){
            var candidateInputs = [];
            // Assign inputs
            for(var i in candidateTables) {
                for(var j in mapping) {
                    if(mapping[j].projectKey == $stateParams.projectKey
                    // table is not necessarily defined, for datasets based on query for instance
                    && (mapping[j].table?.toLowerCase() == candidateTables[i].toLowerCase())) {
                        candidateInputs.push(mapping[j].dataset);
                    }
                }
            }

            // Dedup
            candidateInputs = candidateInputs.filter(function(e,i) { return candidateInputs.indexOf(e) == i;});
            var recipeType = 'sql_query';
            if ($scope.notebookParams.connection.startsWith('@virtual(hive-hproxy)')) recipeType = 'hive';
            if ($scope.notebookParams.connection.startsWith('@virtual(hive-jdbc)')) recipeType = 'hive';
            if ($scope.notebookParams.connection.startsWith('@virtual(impala-jdbc)')) recipeType = 'impala';
            if ($scope.notebookParams.connection.startsWith('@virtual(spark-livy)')) recipeType = 'spark_sql_query';
            var prefillKey = BigDataService.store({
                script : script,
                input : candidateInputs,
                output :[],
                notebookId: $stateParams.notebookId
            });
            $scope.showCreateCodeBasedModal(recipeType, null, null, prefillKey);
        }).error($scope.cell.$tmpState.error);
    };

    $scope.saveBackToRecipe = function() {
        WT1.event("sql-notebook-save-back-to-recipe", {});
        const script = _.get($scope, 'cell.$localState.query.sql', '');
        const recipeId = $scope.notebookParams.recipeId;
        return DataikuAPI.sqlNotebooks.save($scope.notebookParams)
            .then(() => DataikuAPI.sqlNotebooks.saveBackToRecipe($stateParams.projectKey, $stateParams.notebookId, script))
            // an empty id means the recipe does not exist anymore so we need to create a new one
            .then(({ data }) => data.id ? StateUtils.go.recipe(data.id, data.projectId) : $scope.createRecipe())
            .catch($scope.cell.$tmpState.error);
    }

    /* ************ Export results ************ */

    $scope.exportCurrent = function() {
        WT1.event("sql-notebook-export", {});
        DataikuAPI.sqlNotebooks.testStreamedExport($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id, $scope.cell.$tmpState.lastQuery.id).success(function(data) {
            var features = {
                advancedSampling : false,
                partitionListLoader : null,
                isDownloadable : data.streamedExportAvailable
            };
            var dialog = {
                title : 'SQL Query',
                warn : data.streamedExportAvailable ? null : 'Warning! The query will be re-run'
            };
            CreateExportModal($scope,dialog,features).then(function(params) {
                DataikuAPI.sqlNotebooks.exportResults($stateParams.projectKey, $stateParams.notebookId, $scope.cell.id, $scope.cell.$tmpState.lastQuery.id, params).success(function(data) {
                    ExportUtils.defaultHandleExportResult($scope, params, data);
                }).error($scope.cell.$tmpState.error);
            });

        }).error($scope.cell.$tmpState.error);
    };

    /* ************ UI ************ */

    var setupAutoComplete = function(tables, suggestions) { //TODO tables should be in suggestions?
        if (!$scope.cm) {
            Logger.warn("Failed to setup autocomplete");
            return;
        }
        CodeMirror.showHint($scope.cm, function(editor) {
            // eslint-disable-next-line no-undef
            return CodeMirror.sqlNotebookHint(editor, $scope.notebookMode+"-notebook", tables.map(function(t) {return t.table;}), data);
        }, {completeSingle:false});
    };

    var scrollToResults = function(){
        $('.multi-query-editor').scrollTop($('.sql-results-header', $element)[0].offsetTop);
    };

    $scope.focusQuery = function() {
        if (!$scope.cm) return;
        // TODO: remove when we end support for CM5
        $scope.cm.focus();
    };

    $scope.focusQueryAfter = function() {
        $scope.cell.$tpmState = $scope.cell.$tpmState || {};
    };

    $scope.isQueryEmpty = function() {
        return !$scope.cell.$localState.query || !$scope.cell.$localState.query.sql || !$scope.cell.$localState.query.sql.trim();
    };

    $scope.foldQuery = function(fold) {
        WT1.event("sql-notebook-fold-cell", {});
        $scope.cell.$localState.foldQuery = fold;
    };

    $scope.toggleUnfoldTable = function() {
        $scope.cell.$localState.unfoldTable = !$scope.cell.$localState.unfoldTable;
        $scope.$broadcast("reflow"); //For fattable to resize
        if ($scope.cell.$localState.unfoldTable) {
            $timeout(scrollToResults);
        };
    };

    /* ************ init ************ */
    $scope.cell.focusQuery = $scope.focusQuery;

    function initQuery() {
        $scope.cell.$localState.query = {
            sql : ''
        };
        if ($scope.cm && $scope.notebookMode == 'HIVE' || $scope.notebookMode == 'IMPALA' || $scope.notebookMode == 'SPARKSQL') {
            // TODO: remove when we end support for CM5
            $scope.cm.setOption('mode', 'text/x-hivesql');
        }
    }

    $scope.$on('notebookLoaded', initQuery);
    $scope.$on('autocompleteSuggestionsLoaded', setupAutoComplete);

    function selectionChange() {
        // TODO: remove when we end support for CM5
        $scope.somethingSelected = $scope.cm && $scope.cm.somethingSelected() && $scope.cm.getSelection().trim();
        $timeout(() => $scope.$apply());
    }
    $scope.codeChange = function(code) {
        // Because the angular model() is not properly triggering AngularJS digest cycle
        $scope.$applyAsync(() => {
            $scope.cell.$localState.query.sql = code;
        });
    }
    $scope.selectionChange = function ({ from, to }) {
        const newContent = $scope.cell?.$localState?.query?.sql?.slice(from, to);
        if ($scope.codeSelectionContent === newContent) {
            return;
        }
        $scope.$applyAsync(() => {
            $scope.somethingSelected = from != null && to != null && from < to;
            $scope.codeSelectionContent = newContent;
        });
    };

    $scope.editorOptions = function() {
        var mode = ($scope.notebookMode == 'HIVE' || $scope.notebookMode == 'IMPALA' || $scope.notebookMode == 'SPARKSQL') ? 'text/x-hivesql' : 'text/x-sql';
        var opt = {
            noFullScreen: true,
            onLoad: function(cm) {
                $scope.cm = cm;
                // TODO: remove when we end support for CM5
                $scope.cm.on("cursorActivity", selectionChange);
                if ($scope.notebookMode) {
                    setTimeout(function () {
                        if ($scope.cell.$localState.selected) {
                            cm.focus();
                        }
                    }, 200);
                    $(".CodeMirror", $element).append($('<div class="running-query-overlay"><i class="dku-loader icon-spin" /></div>'));
                }
            }
        };
        var editorOptions = CodeMirrorSettingService.get(mode, opt);
        editorOptions.extraKeys[CodeMirrorSettingService.getShortcuts()['AUTOCOMPLETE_SHORTCUT']] = function(cm) {
            return $scope.autocompleteSQL(cm, $scope.notebookMode);
        };
        return editorOptions;
    };
    $scope.columnWidths = [];

    $scope.futureTimer = null;
    $scope.$on("$destroy", $scope.stopfutureTimer);

    var saveLocalStatesTimer;
    function saveLocalStateLater() {
        $scope.cell.code = _.get($scope, 'cell.$localState.query.sql', null);
        $timeout.cancel(saveLocalStatesTimer);
        saveLocalStatesTimer = $timeout($scope.saveLocalStates, 400);
    }
    $scope.$watch('cell.$localState.query.sql', saveLocalStateLater);

    $scope.$watch('cell.$tmpState.resultsTab', function(nv) {
        if (nv != 'logs') {
            $scope.reflow();
        }
    });

    $scope.reflow = function() {
        $scope.$broadcast("reflow"); // update fat repeat layout
    }

    $scope.formatQuery = () => {
        const query = $scope.cell?.$localState?.query?.sql;
        if (query) {
            const editorSettings = $rootScope.appConfig.userSettings.codeEditor;
            const cellLocalStateCopy = angular.copy($scope.cell.$localState);
            cellLocalStateCopy.query.sql = DKUSQLFormatter.format(query, editorSettings.indentUnit, editorSettings.indentWithTabs);
            $scope.cell.$localState = cellLocalStateCopy;
        }
    };

    var initLastQuery = function() {
        // Check if the current code is the same as last query, in this case mark the last query as active and load results
        const cellHistory = $scope.notebookTmpState.history[$scope.cell.id] || [];
        const cell = $scope.cell.$localState.query
        
        if(!cell || !cell.sql || !cell.sql.trim()) {
            // either no local state or it's empty, we load the most recent state from history
            if(cellHistory.length > 0) {
                $scope.loadQuery($scope.cell, cellHistory[0]);
            }
        } else if(!cell.id) {
            // we have locally non-empty code, but no id => we never saved to server
            // do nothing & keep current state
        } else {
            // else we should have a local saved that matches a state in the history
            const matchingHistoryState = cellHistory.find(hist => hist.id === cell.id);
            if(matchingHistoryState && cell.sql === matchingHistoryState.sql) {
                // we only load the result if there is no local change of the query
                $scope.loadQuery($scope.cell, matchingHistoryState);
            }
        }
    };

    $scope.loadLastVersion = function loadLastVersion() {
        const cellHistory = $scope.notebookTmpState.history[$scope.cell.id];
        $scope.loadQuery($scope.cell, cellHistory[0]);
    }

    $scope.$on("history-first-time-loaded", initLastQuery);
    if ($scope.notebookTmpState.history) {
        initLastQuery();
    }
});


app.directive('sqlNotebookCell', function(DataikuAPI, $stateParams){
    return {
        link : function(scope, element, attrs) {
            if (scope.cells.length == 1) {
                scope.cell.$localState.unfolded = true;
            }
            scope.$watch("cell.$localState", scope.saveLocalStates, true);

            scope.toggleCell = function(event) {
                const $target = $(event.target);
                // if clicked element is a button or contained in a button, do not trigger folding
                // if clicked element is inside a dropdown menu, do not trigger folding
                if($target.is('button') || $target.parents('button, .dropdown-menu').length > 0) {
                    return;
                }
                scope.cell.$localState.unfolded = !scope.cell.$localState.unfolded;
            };

            $(element).focus(function(){
                scope.selectCell(scope.$index);
                scope.$apply();
            });
        }
    };
});


app.directive('sqlNotebookQueryCell', function($stateParams, DataikuAPI){
    return {
        templateUrl:'/templates/notebooks/sql-notebook-query-cell.html',
        controller: 'SqlNotebookQueryCellController',
        link : function(scope, element, attrs) {
            scope.cell.$localState.query = scope.cell.$localState.query || {};

            scope.cell.$tmpState.error = function(a,b,c){
                if ($('.local-api-error', element).length > 0) {
                    setErrorInScope.bind($('.local-api-error', element).scope())(a,b,c);
                }
            };

            scope.cell.$tmpState.clearError = function(){
                if ($('.local-api-error', element).length > 0) {
                    resetErrorInScope($('.local-api-error', element).scope());
                }
            };

            $(element).on('click', '.CodeMirror-gutters', function() {
                if (scope.cell.$localState.selected) {
                    scope.foldQuery(true);
                    scope.$apply();
                }
            });
        }
    };
});


app.directive('sqlNotebookQuerySingleCell', function (ActiveProjectKey, CreateModalFromTemplate, SqlNotebookChartEditor, Logger, Dialogs, DataikuAPI, $stateParams) {
    return {
        templateUrl: '/templates/notebooks/sql-notebook-query-single-cell.html',
        controller: 'SqlNotebookQueryCellController',
        link: function (scope, element, attrs) {
            scope.cell.$localState.query = scope.cell.$localState.query || {};

            scope.cell.$tmpState.error = function (a, b, c) {
                if ($('.local-api-error', element).length > 0) {
                    setErrorInScope.bind($('.local-api-error', element).scope())(a, b, c);
                }
            };
            scope.cell.$tmpState.clearError = function () {
                if ($('.local-api-error', element).length > 0) {
                    resetErrorInScope($('.local-api-error', element).scope());
                }
            };

            scope.saveChartData = function() {
                DataikuAPI.sqlNotebooks.saveExploreChart(ActiveProjectKey.get(), $stateParams.notebookId, scope.cell.id, scope.cell.$localExploreChart).success(function(ignored) {
                    // Nothing to do
                }).error(setErrorInScope.bind(scope));
            }

            scope.onSelectChart = (chartId) => {
                scope.cell.$localState.activeTab = {id: chartId, type: 'chart'};
                scope.cell.$selectedChart = getChart(chartId);
            };

            scope.selectTableTab = () => {
                scope.cell.$localState.activeTab = {name: 'Table', type: 'table'}
            };

            scope.onCreateChart = () => {
                SqlNotebookChartEditor.createChart(scope.cell.$localState.query.id, (createdChart) => {
                    Logger.info("Chart created");
                    scope.cell.$localExploreChart.charts.push(createdChart);
                    scope.saveChartData();
                    scope.onSelectChart(createdChart.def.id);
                });
            };

            function getChart(chartId) {
                return scope.cell.$localExploreChart.charts.find(chart => chart.def.id === chartId);
            }

            scope.onEditChart = (chartIdToEdit) => {
                const chartToEdit = getChart(chartIdToEdit);
                SqlNotebookChartEditor.editChart(scope.cell.$localState.query.id, chartToEdit, (editedChart) => {
                    Logger.info("Chart edited");
                    updateChart(chartToEdit, editedChart);
                    scope.saveChartData();
                    scope.onSelectChart(editedChart.def.id);
                });
            };

            scope.onDeleteChart = (deletedChartId) => {
                if (!scope.cell.$localExploreChart) {
                    return;
                }
                Dialogs.confirm(scope, "Confirm deletion", "Are you sure you want to remove this chart ?").then(function() {
                    scope.cell.$localExploreChart.charts = scope.cell.$localExploreChart.charts.filter(chart => chart.def.id !== deletedChartId);
                    scope.saveChartData();
                    if (scope.cell.$localState.activeTab.id === deletedChartId) {
                        scope.selectTableTab();
                    }
                }, () => {});
            };

            function updateChart(baseChart, updatedChart) {
                scope.cell.$localExploreChart.charts = scope.cell.$localExploreChart.charts.map(chart => {
                    if (chart.def.id === baseChart.def.id) {
                        return updatedChart;
                    }
                    return chart;
                })
            }

            // Counter for duplicated chart names
            scope.$watch('cell.$localExploreChart', function (nv) {
                if (!nv) {
                    return;
                }
                const counts = {};
                for (const item of scope.cell.$localExploreChart.charts) {
                    counts[item.def.name] = (counts[item.def.name] || 0) + 1;
                }
                // An object to keep track of the current suffix index for each item.
                const trackers = {};

                scope.cell.$localExploreChart.chartNames = scope.cell.$localExploreChart.charts.map(chart => {
                    if (counts[chart.def.name] > 1) {
                        trackers[chart.def.name] = (trackers[chart.def.name] || 0) + 1;
                        return `${chart.def.name} (${trackers[chart.def.name]})`;
                    }
                    return chart.def.name;
                })
            }, true);
        }
    }
});

app.directive('localApiError', function(){
    return {
        scope: {}
    }
});


app.directive('sqlNotebookMdCell', function(DataikuAPI, $stateParams){
    return {
        templateUrl :'/templates/notebooks/sql-notebook-md-cell.html',
        link : function(scope, element, attrs) {
            scope.cell.$localState.tmpCode = scope.cell.code;
            scope.ok = function() {
                scope.cell.code = scope.cell.$localState.tmpCode;
                scope.cell.$tmpState.mdCellEditModeOn = false;
            }
            if (scope.cell.$localState.unfolded === undefined) {
                scope.cell.$localState.unfolded = true;
            }
        }
    };
});


app.directive('sqlNotebookMdSingleCell', function(DataikuAPI, $stateParams){
    return {
        templateUrl :'/templates/notebooks/sql-notebook-md-single-cell.html',
        link : function(scope, element, attrs) {
            scope.cell.$localState.tmpCode = scope.cell.$localState.tmpCode || scope.cell.code;
            scope.ok = function() {
                scope.cell.code = scope.cell.$localState.tmpCode;
                scope.cell.$tmpState.mdCellEditModeOn = false;
            };
        }
    };
});

app.directive('sqlTableExplorer', function($stateParams, SQLExplorationService, Debounce, Logger, ConnectionExplorationService, translate, SpinnerService) {
    function generateOnlyLastCall() {
        var ref = null;
        return function() {
            ref = {};
            var curr = ref;
            return function() {
                return curr === ref;
            };
        };
    }

    return {
        templateUrl :'/templates/notebooks/sql-explorer.html',
        restrict: 'E',
        scope : {
            connection: '=',
            connectionDetails: '=',
            notebook: '=',
            onTableClicked: '&?',
            onFieldClicked: '&?',
            reportErrorFn: '<'
        },

        link : function(scope) {
            const ANY = '_any_';
            const anyLabel = translate("CONNECTION_EXPLORER.GLOBAL.FILTER.NO_RESTRICTIONS", "No restrictions");
            scope.uiState = {
                confirmListTablesFromDBAll: false, // listing from DB requires user explicit action
                confirmListTablesFromDBProject: false, // listing from DB requires user explicit action
                fetchingTables: true
            };
            scope.sortBy = [
                { value: 'DATASET', label: 'Dataset name' },
                { value: 'TABLE', label: 'Table name' }
            ];

            scope.connectionOptions = {
                catalogs: [{ label: anyLabel, catalog: ANY }],
                schemas: [{ label: anyLabel, schema: ANY }],
                catalog: ANY,
                schema: ANY
            }

            scope.$watch("connectionDetails", function(nv, ov) {
                if (!nv) return;

                /* Replace ANY by null for free text */
                if (scope.connectionDetails.defaultCatalogAndSchemaInputMode == "FREE_TEXT") {
                    scope.connectionOptions.catalog = null;
                    scope.connectionOptions.schema = null;
                }
                if (scope.connectionDetails.defaultCatalog) {
                    scope.connectionOptions.catalog = scope.connectionDetails.defaultCatalog;
                }
                if (scope.connectionDetails.defaultSchema) {
                    scope.connectionOptions.schema = scope.connectionDetails.defaultSchema;
                }
            });

            scope.fetchSchemas = function() {
                ConnectionExplorationService.fetchSchemas(scope.isCatalogAware, scope.connection, $stateParams.projectKey, scope.connectionOptions, scope.reportErrorFn);
            }

            scope.isCatalogAware = ['Databricks', 'BigQuery', 'Snowflake', 'SQLServer', 'JDBC', 'Trino', 'FabricWarehouse'].includes(scope.connectionDetails.type);
            scope.isSchemaAware = scope.connectionDetails.type && !['MySQL', 'hiveserver2', 'Impala'].includes(scope.connectionDetails.type);

            var lastFetchSpecs = null;
            var ignorePreviousListSQLTables = generateOnlyLastCall();
            
            function load() {
                Logger.info('Init SQL explorer');
                if(!scope.connection) {
                    scope.tables = [];
                    rebuildFatList();
                } else {
                    if (scope.notebook.tableListingMode === 'ALL' && !scope.uiState.confirmListTablesFromDBAll) {
                        return;
                    }
                    scope.listTables();
                }
            }

            scope.listTables = function() {
                if (scope.notebook.tableListingMode === 'ALL') {
                    scope.uiState.confirmListTablesFromDBAll = true;
                } else {
                    scope.uiState.confirmListTablesFromDBProject = true;
                }

                var fetchSpecs = {mode:scope.notebook.tableListingMode, connection:scope.connection, projectKey:$stateParams.projectKey};
                if (scope.uiState.fetchingTables && angular.equals(lastFetchSpecs, fetchSpecs)) {
                    Logger.info('Same table list fetch is already ongoing');
                    return;
                }
                lastFetchSpecs = fetchSpecs;
                scope.uiState.fetchingTables = true;

                const chosenCatalog = (scope.connectionOptions.catalog === ANY) ? null : scope.connectionOptions.catalog;
                const chosenSchema = (scope.connectionOptions.schema === ANY) ? null : scope.connectionOptions.schema;

                scope.fatList = []; // purge previous tables

                return SpinnerService.lockOnPromise((scope.notebook.tableListingMode === 'PROJECT'
                    ? SQLExplorationService.listTablesFromProject(scope.connection, $stateParams.projectKey)
                    : SQLExplorationService.listTables(scope.connection, $stateParams.projectKey, chosenCatalog, chosenSchema))
                    .then((tables) => {
                        handleTables(tables);
                    })
                )
            }

            function handleTables(tables) {
                const isLast = ignorePreviousListSQLTables();
                if(!isLast()) {
                    return;
                }
                scope.uiState.fetchingTables = false;
                let schemas;
                if (!scope.isCatalogAware || (scope.notebook.tableListingMode === 'PROJECT' && scope.notebook.tableOrdering === 'DATASET')) {
                    // Build classic "Schema > Table" hierarchy
                    let hasNull = false;
                    let tablesBySchema = {};
                    $.each(tables, function(index, table) {
                        let schema = table.schema;
                        if (!schema) {
                            hasNull = true;
                            schema = '(default)';
                        }
                        if (!tablesBySchema[schema]) {
                            tablesBySchema[schema] = [];
                        }
                        tablesBySchema[schema].push(table);
                    });
                    scope.singleCatalog = true;
                    scope.singleSchema = Object.keys(tablesBySchema).length === 1;
                    scope.schemalessDatabase = scope.singleSchema && hasNull;
                    schemas = $.map(tablesBySchema, function(v, k) {
                        return {
                            name: k,
                            tables: v.sort(function(a, b) {
                                return a.table.localeCompare(b.table);
                            }),
                            state: { shown: scope.singleSchema }
                        };
                    });
                } else {
                    // Build "Catalog > Schema > Table" hierarchy
                    let tablesByCatalogAndSchema = {};
                    $.each(tables, function(index, table) {
                        let schema = table.schema;
                        if (!schema) {
                            schema = '(default)';
                        }
                        let catalog = table.catalog;
                        if (!catalog) {
                            catalog = '(default)';
                        }
                        if (!tablesByCatalogAndSchema[catalog]) {
                            tablesByCatalogAndSchema[catalog] = {};
                        }
                        if (!tablesByCatalogAndSchema[catalog][schema]) {
                            tablesByCatalogAndSchema[catalog][schema] = [];
                        }
                        tablesByCatalogAndSchema[catalog][schema].push(table);
                    });
                    scope.singleCatalog = Object.keys(tablesByCatalogAndSchema).length === 1;
                    scope.singleSchema = false;
                    scope.schemalessDatabase = false;
                    schemas = $.map(tablesByCatalogAndSchema, function(v, k) {
                        return {
                            name: k,
                            schemas: $.map(v, function(sv, sk) {
                                return {
                                    name: sk,
                                    tables: sv.sort(function(a, b) {
                                        return a.table.localeCompare(b.table);
                                    }),
                                    state: { shown: false }
                                }
                            }),
                            state: { shown: scope.singleCatalog }
                        };
                    });
                }
                if (scope.notebook.tableListingMode === 'PROJECT') {
                    scope.schemasRestrictedToProject = schemas;
                } else {
                    scope.schemasAll = schemas;
                }
                rebuildFatList(true);
            }

            function loadAfterChangingListingMode() {
                if (scope.notebook.tableListingMode === 'ALL') {
                    const chosenCatalog = (scope.connectionOptions.catalog === ANY) ? null : scope.connectionOptions.catalog;
                    const chosenSchema = (scope.connectionOptions.schema === ANY) ? null : scope.connectionOptions.schema;
                    if (!SQLExplorationService.isTableInCache(SQLExplorationService.getCacheId(scope.connection, $stateParams.projectKey, chosenCatalog, chosenSchema))) {
                        scope.uiState.confirmListTablesFromDBAll = false;
                    } else {
                        loadAndRebuildFatList(true);
                    }
                } else {
                    loadAndRebuildFatList(true);
                }
            }

            function loadAndRebuildFatList(canAutomaticallyExpand = false) {
                load();
                rebuildFatList(canAutomaticallyExpand);
            }

            scope.$watch('connection', load);

            scope.refreshTableList = function() {
                SQLExplorationService.clearCache();
                load();
            };

            scope.openTable = function(table) {
                SQLExplorationService.listFields(scope.connection,[table]).then(function(data) {
                     table.fields = data;
                     rebuildFatList();
                });
            };

            scope.closeTable = function(table) {
                table.fields = undefined;
                rebuildFatList();
            };

            scope.toggleTable = function(table) {
                if(table.fields) {
                    scope.closeTable(table);
                } else {
                    scope.openTable(table);
                }
            };

            scope.toggleCatalog = function(catalog) {
                catalog.state.shown = !catalog.state.shown;
                rebuildFatList();
            };

            scope.toggleSchema = function(schema) {
                schema.state.shown = !schema.state.shown;
                rebuildFatList();
            };

            scope.filterSort = {};

            /**
             * Builds a fat list from a "Schema > Table" hierarchy
             */
            var makeSchemaTableFieldFatList = function(schemas, query, canAutomaticallyExpand) {
                let displayedSchemas = [];
                for (const schema of Object.values(schemas)) {
                    let tables = filterTables(schema.tables, query);
                    if (tables.length > 0) {
                        displayedSchemas.push({
                            name: schema.name,
                            tables: tables,
                            state: schema.state
                        });
                    }
                }

                if (canAutomaticallyExpand && displayedSchemas.length === 1) {
                    displayedSchemas[0].state.shown = true;
                }

                let fatList = [];
                for (const schema of Object.values(displayedSchemas)) {
                    if (!scope.schemalessDatabase) {
                        fatList.push({ type: 's', schema: schema, class: "schema-item" });
                    }
                    if (schema.state.shown) {
                        schema.tables.forEach(function(table, k) {
                            let clazz = k % 2 ? 'even' : 'odd';
                            fatList.push({ type: 't', table: table, 'class': (scope.schemalessDatabase ? 'flat-table-item' : 'table-item') + ' ' + clazz });
                            if (table.fields) {
                                table.fields.forEach(function(f) {
                                    fatList.push({ type: 'f', field: f, 'class': (scope.schemalessDatabase ? 'field-item' : 'flat-field-item') + ' ' + clazz });
                                });
                                if (table.fields.length === 0) {
                                    fatList.push({ type: 'nf', 'class': (scope.schemalessDatabase ? 'flat-nofield-item' : 'nofield-item') + ' ' + clazz });
                                }
                            }
                        });
                    }
                }
                return fatList;
            };

            /**
             * Builds a fat list from a "Catalog > Schema > Table" hierarchy
             */
            var makeCatalogSchemaTableFieldFatList = function(catalogs, query, canAutomaticallyExpand) {
                let displayedCatalogs = [];
                for (const catalog of Object.values(catalogs)) {
                    let displayedSchemas = [];
                    for (const schema of Object.values(catalog.schemas)) {
                        let tables = filterTables(schema.tables, query);
                        if (tables.length > 0) {
                            displayedSchemas.push({
                                name: schema.name,
                                tables: tables,
                                state: schema.state
                            });
                        }
                    }
                    if (displayedSchemas.length > 0) {
                        displayedCatalogs.push({
                            name: catalog.name,
                            schemas: displayedSchemas,
                            state: catalog.state
                        });
                    }
                }

                if (canAutomaticallyExpand && displayedCatalogs.length === 1) {
                    displayedCatalogs[0].state.shown = true;
                    if (displayedCatalogs[0].schemas.length === 1) {
                        displayedCatalogs[0].schemas[0].state.shown = true;
                    }
                }
                
                let fatList = [];
                for (const catalog of Object.values(displayedCatalogs)) {
                    fatList.push({ type: 'c', catalog: catalog });
                    if (catalog.state.shown) {
                        for (const schema of Object.values(catalog.schemas)) {
                            fatList.push({ type: 's', schema: schema, class: "catalog-schema-item" });
                            if (schema.state.shown) {
                                schema.tables.forEach(function(table, k) {
                                    let clazz = k % 2 ? 'even' : 'odd';
                                    fatList.push({ type: 't', table: table, 'class': 'catalog-table-item ' + clazz });
                                    if (table.fields) {
                                        table.fields.forEach(function(f) {
                                            fatList.push({ type: 'f', field: f, 'class': 'catalog-field-item ' + clazz });
                                        });
                                        if (table.fields.length === 0) {
                                            fatList.push({ type: 'nf', 'class': 'catalog-nofield-item ' + clazz });
                                        }
                                    }
                                });
                            }
                        }
                    }
                }
                return fatList;
            };

            var makeDatasetFieldFatList = function(schemas, query) {
                let tables = [];
                for (const schema of Object.values(schemas)) {
                    tables.push.apply(tables, filterTables(schema.tables, query)); //in place concat
                }
                tables.sort(function(a, b) {
                    return a.dataset.localeCompare(b.dataset);
                });

                let fatList = [];
                tables.forEach(function(table, k) {
                    var clazz = k % 2 ? 'even' : 'odd';
                    fatList.push({ type: 't', table: table, 'class': clazz });
                    if (table.fields) {
                        table.fields.forEach(function(f) {
                            fatList.push({ type: 'f', field: f, 'class': clazz });
                        });
                        if (table.fields.length === 0) {
                            fatList.push({ type: 'nf', 'class': clazz });
                        }
                    }
                });
                return fatList;
            };

            var filterTables = function(tables, query) {
                angular.forEach(query.split(/\s+/), function(token){
                    token = token.toLowerCase();
                    if (token.length) {
                        tables = $.grep(tables, function(item){
                            return item.table.toLowerCase().indexOf(token) >= 0 ||
                            (item.schema && item.schema.toLowerCase().indexOf(token) >= 0);
                        });
                    }
                });
                return tables;
            };

            var rebuildFatList = Debounce().withScope(scope).withDelay(10,200).wrap(function(canAutomaticallyExpand = false) {
                let query = scope.filterSort.tableFilter || '';
                let schemas = (scope.notebook.tableListingMode == 'PROJECT' ? scope.schemasRestrictedToProject : scope.schemasAll) || [];
                if (scope.notebook.tableListingMode == 'PROJECT' && scope.notebook.tableOrdering == 'DATASET') {
                    scope.fatList = makeDatasetFieldFatList(schemas, query);
                } else {
                    if (scope.isCatalogAware) {
                        scope.fatList = makeCatalogSchemaTableFieldFatList(schemas, query, canAutomaticallyExpand);
                    } else {
                        scope.fatList = makeSchemaTableFieldFatList(schemas, query, canAutomaticallyExpand);
                    }
                }
                scope.$broadcast("reflow"); // update fat repeat layout
            });

            scope.$watch('filterSort', rebuildFatList, true);
            scope.$watch('notebook.tableListingMode', loadAfterChangingListingMode);
            scope.$watch('notebook.tableOrdering', loadAndRebuildFatList);
            scope.$watch('connectionOptions.catalog', (newVal, oldVal) => {
                if (newVal != oldVal && scope.connectionOptions.fetchedSchemas) {
                    let availableSchemas;
                    if (newVal && newVal !== ANY) {
                        availableSchemas = scope.connectionOptions.fetchedSchemas.filter(schema => schema.catalog === newVal)
                    } else {
                        availableSchemas = scope.connectionOptions.fetchedSchemas;
                    }

                    const uniqueSchemas = [...new Set(availableSchemas.map(schema => schema.schema))];
                    scope.connectionOptions.schemas = [{label: anyLabel, schema: ANY}].concat(uniqueSchemas.map(s => ({
                        label: s,
                        schema: s
                    })));

                    if (scope.connectionOptions.schema !== ANY && !uniqueSchemas.includes(scope.connectionOptions.schema)) {
                        scope.connectionOptions.schema = ANY;
                    }
                }
            });
        }
    };
});

})();
