#! /bin/bash

# This is script is based on resources/container-exec/kubernetes/azure-acr-prepush.sh
# with one main modif:
# On dataiku online, we can't use `az acr login -n $REPOSITORY_ROOT`, because it relies on docker internally
# instead, we get the token, and pass it to crane auth

# make commands non-fatal, it's ok if they fail as long as we log on stdout what failed.
# basically the script attempts to log docker into acr, but it may still fail to do so
# and you get a kaniko push error afterwards
set +e

REPOSITORY=$1
IMAGE=$2

# if repository is like foobarregistry.azurecr.io/stuff/here/
# then repository name for the az acr command is just the foobarregistry.azurecr.io
# part (azure will gracefully strip the azurecr.io)
REPOSITORY_ROOT=$(echo "$REPOSITORY"/"$IMAGE" | sed 's_^\([^/]*\)/.*$_\1_')
echo "INFO: making sure repository '$REPOSITORY_ROOT' exists on ACR" >&2

if [ -z `command -v az` ]; then
    echo 'ERROR: required `az` binary is not in PATH' >&2
    exit 1
fi

AZ_VERSION=$(az version --query '"azure-cli"' 2>&1 | tr -d '"')
echo "az cli version = $AZ_VERSION"

# Flag to indicate that service principal login is done
if [ -n "$DKU_AZURE_CLI_LOGIN_PERFORMED" ]; then
    echo 'INFO: Already logged in.'
# login the az cli, in case the creds have expired
elif [ -n "$DKU_AZURE_CLIENT_ID" ]; then
    # if DKU_AZURE_CLIENT_ID starts with /, it may be a resource ID
    if [[ "$DKU_AZURE_CLIENT_ID" == /* ]]; then
        az login --identity --resource-id "$DKU_AZURE_CLIENT_ID"
    # otherwise, we consider it a client ID
    else
        az login --identity --client-id "$DKU_AZURE_CLIENT_ID"
    fi
    if [ $? -ne 0 ]; then
        echo 'WARNING: could not login UAMI' >&2
    fi
else
    # pray that there is only one identity assigned to the VM, or this
    # login will fail
    az login --identity
    if [ $? -ne 0 ]; then
        echo 'WARNING: could not login default UAMI' >&2
    fi
fi

REGISTRY_TOKEN=$(az acr login -n $REPOSITORY_ROOT --expose-token | jq -r .accessToken)
if [ $? -ne 0 ]; then
    echo 'WARNING: could not get password' >&2
fi
crane auth login -u 00000000-0000-0000-0000-000000000000 -p $REGISTRY_TOKEN $REPOSITORY_ROOT
if [ $? -ne 0 ]; then
    echo 'WARNING: could not write credentials to .docker/config.json' >&2
else
    echo 'INFO: Logged in to Azure ACR.' >&2
fi

