

(function() {
    'use strict';

    angular.module('dataiku.charts')
        .service('ChartColorScales', ChartColorScales);

    /**
     * Colors scales creation logic
     * (!) This service previously was in static/dataiku/js/simple_report/common/colors.js
     */
    function ChartColorScales(ChartUADimension, ChartDataUtils, StringNormalizer, ColorUtils, ChartColorUtils, ChartsStaticData, DKU_PALETTE_NAMES, CHART_AXIS_TYPES) {

        const svc = {

            /**
             * Create a color scale
             * @param {ChartColorContext} colorContext (typed in chart-color-context.interface.ts)
             * @return {*}
             */
            createColorScale: function(colorContext) {
                if (!colorContext.colorSpec) {
                    return null;
                }

                let colorScale;
                switch (colorContext.colorSpec.type) {
                    case CHART_AXIS_TYPES.DIMENSION:
                        colorScale = svc.discreteColorScale(colorContext.colorOptions, colorContext.defaultLegendDimension, colorContext.chartData, colorContext.colorSpec.withRgba,
                            ChartColorUtils.getColorMeaningInfo(colorContext.colorSpec.dimension, colorContext.chartHandler), colorContext.chartData.getAxisLabels(colorContext.colorSpec.name), colorContext.theme, colorContext.uaColorIndex, colorContext.ignoreLabels);
                        if (colorScale.domain) {
                            if (colorContext.chartData.getAxisIdx(colorContext.colorSpec.name) != undefined) {
                                colorScale.domain(colorContext.chartData.getAxisLabels(colorContext.colorSpec.name).map(function(d, i) {
                                    return i;
                                }));
                            } else {
                                colorScale.domain(colorContext.defaultLegendDimension.map(function(d, i) {
                                    return i;
                                }));
                            }
                        }
                        break;
                    case CHART_AXIS_TYPES.MEASURE:
                        if (!colorContext.colorSpec.extent) {
                            if (colorContext.colorSpec.measureIdx === undefined || colorContext.colorSpec.measureIdx < 0) {
                                return null;
                            }
                            colorContext.colorSpec.extent = ChartDataUtils.getMeasureExtent(colorContext.chartData, colorContext.colorSpec.measureIdx, true, colorContext.colorSpec.binsToInclude, colorContext.colorSpec.colorAggrFn);
                        }

                        if (!colorContext.colorSpec.values) {
                            if (colorContext.colorSpec.measureIdx === undefined || colorContext.colorSpec.measureIdx < 0) {
                                return null;
                            }
                            colorContext.colorSpec.values = ChartDataUtils.getMeasureValues(colorContext.chartData.data, colorContext.colorSpec.measureIdx, colorContext.colorSpec.binsToInclude);
                        }

                        colorScale = svc.continuousColorScale(colorContext.colorOptions, colorContext.colorSpec.extent[0], colorContext.colorSpec.extent[1], colorContext.colorSpec.values, !colorContext.colorSpec.withRgba, colorContext.theme, colorContext.colorSpec.binsToInclude);
                        break;
                    case CHART_AXIS_TYPES.UNAGGREGATED:
                        if (!colorContext.colorSpec.dimension) {
                            return null;
                        }
                        var extent = ChartDataUtils.getUnaggregatedAxisExtent(colorContext.colorSpec.dimension, colorContext.colorSpec.data, colorContext.chartData.data.afterFilterRecords);
                        if (ChartUADimension.isTrueNumerical(colorContext.colorSpec.dimension) || ChartUADimension.isDateRange(colorContext.colorSpec.dimension)) {
                            colorScale = svc.continuousColorScale(colorContext.colorOptions, extent.min, extent.max, extent.values, !colorContext.colorSpec.withRgba, colorContext.theme);
                            colorScale.isContinuous = true;
                        } else {
                            colorScale = svc.discreteColorScale(colorContext.colorOptions, colorContext.defaultLegendDimension, colorContext.chartData, colorContext.colorSpec.withRgba,
                                ChartColorUtils.getColorMeaningInfo(colorContext.colorSpec.dimension, colorContext.chartHandler), colorContext.colorSpec.data.str.sortedMapping, colorContext.theme, colorContext.uaColorIndex);
                            if (colorScale.domain) {
                                colorScale.domain(extent.values.map((v, i) => i));
                            }
                        }
                        break;
                    case 'CUSTOM': {
                        const measures = colorContext.colorSpec.buildGenericMeasures(colorContext.defaultLegendDimension, (colorContext.chartData.getAxisLabels(colorContext.colorSpec.name) || []));
                        colorScale = svc.discreteColorScale(colorContext.colorOptions, measures, colorContext.chartData, colorContext.colorSpec.withRgba,
                            ChartColorUtils.getColorMeaningInfo(colorContext.colorSpec.dimension, colorContext.chartHandler), colorContext.chartData.getAxisLabels(colorContext.colorSpec.name), colorContext.theme, colorContext.uaColorIndex, colorContext.ignoreLabels, colorContext.colorSpec.ignoreColorDim);
                        if (colorScale.domain) {
                            colorScale.domain(measures.map(function(d, i) {
                                return i;
                            }));
                            colorScale.customHandle = colorContext.colorSpec.handle;
                        }
                        break;
                    }
                    default:
                        throw new Error('Unknown scale type: ' + colorContext.colorSpec.type);
                }

                if (colorScale) {
                    colorScale.type = colorContext.colorSpec.type;
                }

                return colorScale;
            },


            /**
             * Create a continuous color scale
             * @param {ChartDef.java} colorOptions
             * @param {number} domainMin
             * @param {number} domainMax
             * @param {array} domainValues: values in the domain (not uniques, this is used to compute quantiles)
             * @param {boolean} noRgba: do not include the opacity setting in the color scale
             * @param {DSSVisualizationTheme} theme
             * @return {*}
             */
            continuousColorScale: function(colorOptions, domainMin, domainMax, domainValues, noRgba, theme, binsToInclude) {

                const isDiverging = colorOptions.paletteType === 'DIVERGING';

                let p = ChartColorUtils.getContinuousPalette(colorOptions.colorPalette, colorOptions.customPalette, isDiverging, theme);

                if (!p) {
                    const defaultDivergingPalette = theme?.palettes.diverging ?? ChartsStaticData.DEFAULT_DIVERGING_PALETTE;
                    const defaultContinuousPalette = theme?.palettes.continuous ?? ChartsStaticData.DEFAULT_CONTINUOUS_PALETTE;
                    const defaultPaletteId = isDiverging ? defaultDivergingPalette : defaultContinuousPalette;
                    colorOptions.colorPalette = defaultPaletteId;
                    p = ChartColorUtils.getContinuousPalette(colorOptions.colorPalette, colorOptions.customPalette, isDiverging, theme);
                }

                // Custom interpolation function to take care of transparency
                function d3_interpolateRgbRound(a, b) {
                    const transparency = !isNaN(colorOptions.transparency) ? colorOptions.transparency : 1;
                    a = d3.rgb(a);
                    b = d3.rgb(b);
                    const ar = a.r,
                        ag = a.g,
                        ab = a.b,
                        br = b.r - ar,
                        bg = b.g - ag,
                        bb = b.b - ab;
                    return function(t) {
                        const tr = Math.round(ar + br * t);
                        const tg = Math.round(ag + bg * t);
                        const tb = Math.round(ab + bb * t);
                        if (!noRgba) {
                            return ['rgba(', tr, ',', tg, ',', tb, ',', transparency, ')'].join('');
                        } else {
                            return ['rgb(', tr, ',', tg, ',', tb, ')'].join('');
                        }
                    };
                }

                if (p.d3Scale) {
                    return p.d3Scale;
                }

                let innerScale;

                if (colorOptions.quantizationMode !== 'QUANTILES') {
                    /*
                     * We use an innerScale to implement the scale computation mode (linear, log, square, sqrt),
                     * that maps the values to a [0, 1] range that will be the input of the actual color scale
                     */

                    if (colorOptions.ccScaleMode == 'LOG') {
                        innerScale = d3.scale.log();
                        domainMin++;
                        domainMax++;
                        innerScale.mode = 'LOG';
                    } else if (colorOptions.ccScaleMode == 'SQRT') {
                        innerScale = d3.scale.sqrt();
                        innerScale.mode = 'SQRT';
                    } else if (colorOptions.ccScaleMode == 'SQUARE') {
                        innerScale = d3.scale.pow().exponent(2);
                        innerScale.mode = 'SQUARE';
                    } else {
                        innerScale = d3.scale.linear();
                        innerScale.mode = 'LINEAR';
                    }
                } else {
                    // No compute mode for quantiles quantization
                    innerScale = d3.scale.linear();
                    innerScale.mode = 'LINEAR';
                }

                switch (colorOptions.paletteType) {
                    case 'DIVERGING':
                        var mid = colorOptions.paletteMiddleValue || 0;
                        if (Math.abs(domainMax - mid) > Math.abs(domainMin - mid)) {
                            innerScale.domain([mid, domainMax]).range([0.5, 1]);
                        } else {
                            innerScale.domain([domainMin, mid]).range([0, 0.5]);
                        }
                        break;
                    case 'CONTINUOUS':
                    default:
                        if (p.fixedValues) {
                            const domain = [],
                                range = [];
                            p.values.forEach(function(value, i) {
                                if (i > p.colors.length - 1) {
                                    return;
                                }
                                if (value == null) {
                                    if (i == 0) {
                                        domain.push(domainMin);
                                        range.push(0);
                                    } else if (i == p.colors.length - 1) {
                                        domain.push(domainMax);
                                        range.push(1);
                                    }
                                } else {
                                    domain.push(value);
                                    range.push(i / (p.colors.length - 1));
                                }
                            });
                            innerScale.domain(domain).range(range);
                        } else {
                            innerScale.domain([domainMin, domainMax]).range([0, 1]);
                        }
                        break;
                }

                let outerScale;

                switch (colorOptions.quantizationMode) {
                    case 'LINEAR':
                    case 'QUANTILES':
                        // Find step colors
                        var numSteps = colorOptions.numQuantizeSteps;
                        var colors = p[numSteps] || p.colors; // Palettes can define special colors for a given number of steps (i.e. colorbrewer palettes)
                        var numColors = colors.length;

                        var linearScale = d3.scale.linear()
                            .domain(Array(numColors).fill().map(function(d, i) {
                                return i / (numColors - 1);
                            }))
                            .range(colors)
                            .interpolate(d3_interpolateRgbRound);
                        var steps = Array(numSteps).fill().map(function(d, i) {
                            return linearScale(i / (numSteps - 1));
                        });

                        if (colorOptions.quantizationMode === 'LINEAR') {
                            outerScale = d3.scale.quantize().domain([0, 1]).range(steps);
                        } else {
                            outerScale = d3.scale.quantile().domain(domainValues.map(innerScale)).range(steps);
                        }
                        break;

                    case 'NONE':
                    default:
                        outerScale = d3.scale.linear()
                            .domain(Array(p.colors.length).fill().map(function(d, i) {
                                return i / (p.colors.length - 1);
                            }))
                            .range(p.colors)
                            .interpolate(d3_interpolateRgbRound);
                        break;

                }

                const ret = function(d, bin) {
                    if (binsToInclude && !binsToInclude.has(bin)) {
                        return undefined;
                    }
                    return outerScale(innerScale(d));
                };

                ret.outerScale = outerScale;
                ret.innerScale = innerScale;
                ret.quantizationMode = colorOptions.quantizationMode;
                ret.diverging = colorOptions.paletteType === 'DIVERGING';

                return ret;
            },

            /**
             * Create a discrete color scale
             * @param colorOptions
             * @param defaultLegendDimension uaDimensionPair for scatters MP, genericMeasures for other charts
             * @param {ChartTensorDataWrapper} chartData
             * @param {boolean} withRgba
             * @param meaningInfo
             * @param colorLabels
             * @param {DSSVisualizationTheme} theme, theme that is applied to the chart
             * @param {number} uaColorIndex (Optional), index of the details column to color a chart
             * @param {Set} ignoreLabels (Optional), labels that should not be included in the colorScale
             * @return {*}
             */
            discreteColorScale: function(colorOptions, defaultLegendDimension, chartData, withRgba, meaningInfo, colorLabels, theme, uaColorIndex, ignoreLabels, ignoreColorDim) {
                if (!colorOptions.colorPalette) {
                    colorOptions.colorPalette = theme?.palettes?.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                }
                if (colorOptions.colorPalette === DKU_PALETTE_NAMES.MEANING) {
                    return svc.meaningColorScale(colorOptions, withRgba, meaningInfo, colorLabels);
                }

                let p = ChartColorUtils.getDiscretePalette(colorOptions.colorPalette, colorOptions.customPalette, theme);

                if (!p) {
                    colorOptions.colorPalette = theme?.palettes?.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                    p = ChartColorUtils.getDiscretePalette(colorOptions.colorPalette, colorOptions.customPalette, theme);
                }
                if (p.d3Scale) {
                    return p.d3Scale;
                } else {
                    let colorScale = d3.scale.ordinal().range(p.colors);
                    if (colorOptions.customColors && Object.keys(colorOptions.customColors).length) {
                        colorScale = svc.customColorScale(colorOptions, defaultLegendDimension, chartData, colorScale, uaColorIndex, ignoreLabels, ignoreColorDim);
                    }
                    return withRgba ? convertToRgbaColorScale(colorScale, colorOptions.transparency) : colorScale;
                }
            },

            meaningColorScale: function(colorOptions, withRgba, meaningInfo, colorLabels) {
                const normalizer = StringNormalizer.get(meaningInfo.normalizationMode);
                const ret = function(idx) {
                    // TODO fixed fallback color? defined in the chart? in the meaning?
                    if (withRgba) {
                        return ColorUtils.toRgba(meaningInfo.colorMap[normalizer(colorLabels[idx].label)] || 'grey', colorOptions.transparency);
                    } else {
                        return meaningInfo.colorMap[normalizer(colorLabels[idx].label)] || 'grey';
                    }
                };
                ret.domain = function() {
                    return Array.from(Array(colorLabels.length).keys());
                };
                return ret;
            },

            customColorScale: function(colorOptions, dimension, chartData, colorScale, uaColorIndex, ignoreLabels, ignoreColorDim) {
                const colorScaleProxy = index => {
                    const customColors = colorOptions.customColors;
                    const id = ChartColorUtils.getColorId(dimension, chartData, index, uaColorIndex, ignoreLabels, ignoreColorDim && ignoreColorDim(index, dimension));
                    if (id in customColors) {
                        return customColors[id];
                    }
                    return colorScale(index);
                };
                colorScaleProxy.range = colorScale.range;
                colorScaleProxy.domain = colorScale.domain;
                return colorScaleProxy;
            },
            getColor: function(uaColor, color, i, colorScale, colorCache) {
                let cacheKey, rgb;
                if (ChartUADimension.isTrueNumerical(uaColor[0])) {
                    cacheKey = color.num.data[i];
                } else if (ChartUADimension.isDateRange(uaColor[0])) {
                    cacheKey = color.ts.data[i];
                } else {
                    cacheKey = color.str.data[i];
                }

                if (colorCache[cacheKey]) {
                    rgb = colorCache[cacheKey];
                } else {
                    rgb = colorScale(cacheKey);
                    colorCache[cacheKey] = rgb;
                }
                return rgb;
            }
        };

        function convertToRgbaColorScale(colorScale, transparency) {
            const rgbaColorScale = index => {
                const color = colorScale(index);
                return ColorUtils.toRgba(color, transparency);
            };

            rgbaColorScale.range = colorScale.range;
            rgbaColorScale.domain = colorScale.domain;

            return rgbaColorScale;
        }

        /**
         * Create samples for colorpalettes
         */
        function createSamples() {
            $.each(window.dkuColorPalettes.continuous, function(idx, p) {
                if (!p.sample) {
                    const scale = svc.continuousColorScale({ colorPalette: p.id, transparency: 1 }, 0, 100);
                    p.sample = $.map([0, 20, 40, 60, 80, 100], scale);
                    p.sample2 = $.map([0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100], scale);
                }
            });
            $.each(window.dkuColorPalettes.discrete, function(idx, p) {
                if (!p.sample) {
                    const scale = svc.discreteColorScale({ colorPalette: p.id, transparency: 1 });
                    p.sample = $.map([0, 1, 2, 3, 4], scale);
                }
            });
        }

        createSamples();
        return svc;
    }
})();
