(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesChart', LinesChart);

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesChart(ChartManager, ChartDataWrapperFactory, LinesDrawer, LinesUtils, ChartDataUtils, MonoFuture, ChartDimension, ReferenceLines, ColumnAvailability, ChartYAxisPosition, ChartAxesUtils, ChartCustomMeasures, ChartUsableColumns, ChartZoomControlAdapter, CHART_ZOOM_CONTROL_TYPES, LinesBrushDrawer, ChartStoreFactory) {
        return function($container, chartDef, chartHandler, axesDef, data, pivotRequest, uiDisplayState, chartActivityIndicator) {
            recomputeAndRedraw($container, chartDef, chartHandler, axesDef, data, pivotRequest, null, uiDisplayState, chartActivityIndicator);
        };

        function recomputeAndRedraw($container, chartDef, chartHandler, axesDef, data, pivotRequest, displayInterval, uiDisplayState, chartActivityIndicator) {

            const initialChartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const pivotRequestCallback = MonoFuture().wrap(pivotRequest);

            const dataSpec = chartHandler.getDataSpec();

            const xDimension = ChartDimension.getGenericDimension(chartDef);
            const includeEmptyBins = LinesUtils.hasEmptyBinsToIncludeAsZero(xDimension, initialChartData);
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);
            const xSpec = { type: 'DIMENSION', mode: 'POINTS', dimension: xDimension, name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };

            const facetLabels = initialChartData.getAxisLabels('facet') || [null], // We'll through the next loop only once if the chart is not facetted
                yExtents = ChartDataUtils.getMeasureExtents(chartDef, initialChartData, 'x', undefined, includeEmptyBins),
                leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT),
                rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT),
                isLeftYPercentScale = yExtents[leftYAxisID].onlyPercent,
                isRightYPercentScale = yExtents[rightYAxisID].onlyPercent;

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, initialChartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: isLeftYPercentScale }, [rightYAxisID]: { isPercentScale: isRightYPercentScale } });

            const leftYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]),
                rightYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(initialChartData, 'x', xSpec.dimension), referenceLinesExtents.x, xSpec);
            const linesData = LinesUtils.prepareData(chartDef, initialChartData);
            const isInteractive = ChartDimension.isInteractiveChart(chartDef);
            let zoomContext, initLinesZoom;

            const drawFrame = function(frameIdx, chartBase, redraw, chartData = initialChartData) {
                ReferenceLines.removeReferenceLines($container[0]);

                chartData.fixAxis('animation', frameIdx);

                if (isInteractive && !_.isNil(chartDef.$zoomControlInstanceId)) {
                    const zoomControlInstance = ChartZoomControlAdapter.get(chartDef.$zoomControlInstanceId);
                    zoomControlInstance.setZoomUtils({ ...zoomControlInstance.getZoomUtils(), frameIndex: frameIdx });
                }

                facetLabels.forEach(function(facetLabel, facetIndex) {
                    const g = d3.select(chartBase.$svgs.eq(facetIndex).find('g.chart').get(0));
                    LinesDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', facetIndex), chartBase, linesData, facetIndex, redraw, isInteractive);
                });
            };

            const cleanFrame = function(chartBase) {
                facetLabels.forEach(function(facetLabel, facetIndex) {
                    const g = d3.select(chartBase.$svgs.eq(facetIndex).find('g.chart').get(0));
                    LinesUtils.cleanChart(g, chartBase);
                });
            };

            const drawBrush = function(chartBase, g, brushAxes) {
                const isAnimated = chartBase.chartData.axesDef.animation !== undefined;
                const hasSubcharts = facetLabels && facetLabels.length > 1;

                if (isAnimated || hasSubcharts) {
                    return;
                }

                LinesBrushDrawer(g, chartDef, initialChartData, chartBase, linesData, 0, brushAxes, referenceLinesValues);
            };

            if (isInteractive) {
                zoomContext = {
                    d3,
                    chartContainer: $container,
                    chartDef,
                    chartHandler,
                    drawFrame,
                    drawBrush,
                    axesDef,
                    chartActivityIndicator,
                    cleanFrame,
                    pivotRequestCallback,
                    linesChartCallback: recomputeAndRedraw,
                    uiDisplayState
                };

                initLinesZoom = (zoomContext) => {
                    let previousZoomUtils = {};
                    if (!_.isNil(chartDef.$zoomControlInstanceId)) {
                        const zoomControlInstance = ChartZoomControlAdapter.get(chartDef.$zoomControlInstanceId);
                        previousZoomUtils = zoomControlInstance.getZoomUtils();
                    }

                    zoomContext = { ...zoomContext, zoomUtils: previousZoomUtils };

                    const id = ChartZoomControlAdapter.create(CHART_ZOOM_CONTROL_TYPES.LINES, zoomContext);
                    ChartZoomControlAdapter.init(CHART_ZOOM_CONTROL_TYPES.LINES, id);

                    return id;
                };
                chartHandler.forceRotation = 0.5;
            } else {
                chartHandler.forceRotation = undefined;
            }

            const leftYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, leftYAxisID);
            const rightYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, rightYAxisID);

            const ySpecs = {
                [leftYAxisID]: { id: leftYAxisID, type: 'MEASURE', extent: leftYExtent, isPercentScale: isLeftYPercentScale, customExtent: leftYCustomExtent, position: ChartYAxisPosition.LEFT },
                [rightYAxisID]: { id: rightYAxisID, type: 'MEASURE', extent: rightYExtent, isPercentScale: isRightYPercentScale, customExtent: rightYCustomExtent, position: ChartYAxisPosition.RIGHT }
            };

            if (displayInterval && _.isFinite(displayInterval[0]) && _.isFinite(displayInterval[1])) {
                xSpec.initialInterval = { min: displayInterval[0], max: displayInterval[1] };
            }

            if (!isInteractive) {
                // Use x custom user extent only when there is no zoom, otherwise let zoom manage x axis range
                xSpec.customExtent = chartDef.xAxisFormatting.customExtent;
            }

            const availableAxes = ReferenceLines.getAvailableAxesForReferenceLines(chartDef);
            const axisSpecs = { x: xSpec, ...ySpecs };

            ChartAxesUtils.setNumberOfBinsToDimensions(initialChartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: availableAxes['LEFT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[leftYAxisID].isPercentScale },
                { axis: 'RIGHT_Y_AXIS', isDisplayed: availableAxes['RIGHT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[rightYAxisID].isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, initialChartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] },
                zoomContext, initLinesZoom);
        };
    }
})();
