(function() {
    'use strict';

    angular.module('dataiku.charts')
        .service('ChartTooltipsUtils', ChartTooltipsUtils);

    /**
     * Older tooltip service, now only used in boxplots.js and graph.js
     * (!) This service previously was in static/dataiku/js/simple_report/common/tooltips.js
     */
    function ChartTooltipsUtils(LoggerProvider, $q, $templateCache, $http, $timeout, $compile) {
        let globalTooltipId = 0;
        const Logger = LoggerProvider.getLogger('charts');

        const tooltipDisappear = function(tooltip) {
            tooltip.interrupt('fade').transition('fade')
                .duration(100)
                .style('opacity', 0);
            tooltip.style('pointer-events', 'none');
            return tooltip;
        };

        const svc = {
            /**
             * Returns a promise to [tooltip (as D3 sel), tooltipScope]
             */

            create: function(parentScope, type, chart) {
                const deferred = $q.defer();
                let tooltipScope, tooltip;
                const location = '/templates/simple_report/tooltips/' + type + '.html';
                $q.when($templateCache.get(location) || $http.get(location, { cache: true })).then(function(template) {

                    if (parentScope.noTooltips) {
                        // return fake tooltip scope with marker
                        deferred.resolve([null, { '$apply': function() { }, 'noTooltips': true }]);
                        return;
                    }

                    if (angular.isArray(template)) {
                        template = template[1];
                    } else if (angular.isObject(template)) {
                        template = template.data;
                    }
                    globalTooltipId++;

                    const newDOMElt = $(template);
                    newDOMElt.addClass('ng-cloak');

                    newDOMElt.attr('g-tooltip-id', globalTooltipId);

                    $('body').append(newDOMElt);

                    Logger.info('Create tooltip: ' + globalTooltipId + ', now have in DOM: ' + $('[g-tooltip-id]').length);

                    tooltip = d3.selectAll(newDOMElt.toArray());
                    tooltip.style('top', 0).style('left', '-50%');

                    $timeout(function() {
                        $compile(newDOMElt)(parentScope);
                        tooltipScope = angular.element(newDOMElt).scope();

                        tooltipScope.$on('$destroy', function() {
                            Logger.info('Destroying tooltip: ' + tooltip.attr('g-tooltip-id'));
                            tooltip.remove();
                        });
                        deferred.resolve([tooltip, tooltipScope]);
                    });

                    tooltip.on('mouseenter', function() {
                        tooltipScope.mouseOnTooltip = true;
                        tooltipScope.$apply();
                    }).on('mouseleave', function() {
                        tooltipScope.mouseOnTooltip = false;
                        if (!tooltipScope.mouseOnElement) {
                            tooltipDisappear(tooltip);
                        }
                        tooltipScope.$apply();
                    });

                    svc.flagTooltipAndRemoveOrphans(chart, tooltip);

                });
                return deferred.promise;
            },

            createWithStdAggr1DBehaviour: function(parentScope, type, chart) {
                return svc.create(parentScope, type, chart).then(function(x) {
                    return x;
                });
            },

            appear: function(tooltip, color, event, element, xOffset) {
                if (!tooltip) {
                    return;
                }

                //initializing tooltip position values to top left of cursor
                let tooltipX = event.pageX + 3 + (xOffset ? xOffset : 0);
                let tooltipY = event.pageY - 28;
                //checking wether there is some better positionning
                const eventSvgX = event.pageX - $(element).offset().left;
                const eventSvgY = event.pageY - $(element).offset().top;
                if (eventSvgX > $(element).outerWidth() / 2) {
                    tooltipX = event.pageX - tooltip.node().getBoundingClientRect().width - 3 - (xOffset ? xOffset : 0);
                }
                if (eventSvgY > $(element).outerHeight() / 2) {
                    tooltipY = event.pageY - tooltip.node().getBoundingClientRect().height + 28;
                }
                // Border is not transitionable
                tooltip.interrupt('fade').transition('fade').duration(300)
                    .style('opacity', 1)
                    .style('left', (tooltipX) + 'px')
                    .style('top', (tooltipY) + 'px');
                tooltip.style('pointer-events', 'none');
                return tooltip;
            },

            handleMouseOverElement: function(tooltipScope) {
                if (!tooltipScope || tooltipScope.noTooltips) {
                    return;
                }
                tooltipScope.mouseOnElement = true;
                tooltipScope.tooltipIsPersistent = false;
            },

            handleMouseOutElement: function(tooltip, tooltipScope, digestInProgress) {
                if (tooltipScope.noTooltips) {
                    return;
                }
                tooltipScope.mouseOnElement = false;
                if (!tooltipScope.tooltipIsPersistent) {
                    tooltipDisappear(tooltip);
                } else {
                    $timeout(function() {
                        if (!tooltipScope.mouseOnTooltip) {
                            tooltipDisappear(tooltip);
                            tooltipScope.tooltipIsPersistent = false;
                            if (!digestInProgress) {
                                tooltipScope.$apply();
                            }
                        }
                    }, 150);
                }
                if (!digestInProgress) {
                    tooltipScope.$apply();
                }
            },

            handleClickElement: function(tooltip, tooltipScope) {
                if (tooltipScope.noTooltips) {
                    return;
                }
                if (tooltipScope.tooltipIsPersistent) {
                    tooltip.style('pointer-events', 'none');
                } else {
                    tooltip.style('pointer-events', 'auto');
                }
                tooltipScope.tooltipIsPersistent = !tooltipScope.tooltipIsPersistent;
                tooltipScope.$apply();
            },

            flagTooltipAndRemoveOrphans: function(chart, tooltip) {
                const flagChartAndTooltipTogether = function(chart, tooltip) {
                    const id = Date.now();
                    $(chart).parents('.pivot-chart').attr('data-tooltip-id', id);
                    tooltip.attr('data-tooltip-id', id);
                };

                const removeOrphanTooltips = function() {
                    $('.chart-tooltip').each(function(index, element) {
                        const id = $(element).data('tooltipId');
                        if ($('.pivot-chart[data-tooltip-id="' + id + '"]').length <= 0) {
                            $(element).remove();
                        }
                    });
                };

                flagChartAndTooltipTogether(chart, tooltip);
                removeOrphanTooltips();
            }
        };
        return svc;
    }
})();
