(function() {
    'use strict';
    /** @typedef {import('../../types').ChartDef} ChartDef */

    angular.module('dataiku.charts')
        .service('ChartTooltips', ChartTooltips);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/tooltips.js
     */
    function ChartTooltips($q, $templateCache, $http, $compile, $rootScope, ChartFilters, ChartFormatting, ChartColorScales, ChartLabels, ChartDimension, ChartColorUtils, AnimatedChartsUtils, CHART_TYPES, ChartStoreFactory, ChartFeatures, Logger, ChartUADimension, $stateParams, DKU_OTHERS_VALUE, DashboardFilters, DashboardUtils, HierarchicalChartsUtils, ConditionalFormattingOptions, ChartDrilldown, ChartHierarchyDimension, DKU_TOTAL_VALUE, DRILL_UP_SOURCE, DRILL_DOWN_SOURCE) {
        const DASHBOARD_TOOLTIPS_CONTAINER_SELECTOR = '.dashboard-tile__chart-tooltips-container';
        const DASHBOARD_TOOLTIP_CONTAINER_CLASS = 'dashboard-tile__chart-tooltip-container';
        const DASHBOARD_TOOLTIP_CONTAINER_SELECTOR = `.${DASHBOARD_TOOLTIP_CONTAINER_CLASS}`;
        const DASHBOARD_TILE_SELECTOR = '.dashboard-tile';
        const ECHARTS_CHART_CONTAINER_SELECTOR = '.main-echarts-zone > .chart';
        const D3_CHART_CONTAINER_SELECTOR = '.mainzone > .charts';
        const CHART_WRAPPER_SELECTOR = '.chart-wrapper';
        const DEFAULT_COLOR_TRANSITION_DURATION = 300;

        let globalTooltipScopes = 0;
        let resetColorsTimeout = null;

        /**
         * Retrieves the tooltip container element.
         * @param {JQLite} element the element to search from.
         * @param {'container' | 'tooltip'} from whether the element is the tooltip itself or a upper level container.
         * @param {boolean} isEChart whether the chart is an ECharts chart or not.
         * @returns {JQLite}
         */
        function getTooltipContainers(element, from, isEChart) {
            if (from !== 'container' && from !== 'tooltip') {
                Logger.error(`Incorrect parameter encountered while trying to get tooltip container: from = ${from}`);
                return $();
            }
            if (DashboardUtils.isInDashboard()) {
                const chartsContainerSelector = isEChart ? ECHARTS_CHART_CONTAINER_SELECTOR : D3_CHART_CONTAINER_SELECTOR;
                const chartsContainer = (from === 'container' ? element.find(chartsContainerSelector) : element.closest(chartsContainerSelector)).get(0);
                if (chartsContainer) {
                    const isChartsContainerScrollable = chartsContainer.scrollHeight > chartsContainer.clientHeight;
                    if (!isChartsContainerScrollable) {
                        return element.closest(DASHBOARD_TILE_SELECTOR).find(DASHBOARD_TOOLTIP_CONTAINER_SELECTOR);
                    }
                }
            }
            if (from === 'container') {
                return element.find(CHART_WRAPPER_SELECTOR);
            } else if (from === 'tooltip') {
                return element.closest(CHART_WRAPPER_SELECTOR);
            }
        }

        function createDashboardTooltipContainers(element, facets, forceSingleContainer) {
            const dashboardContainers = element.closest(DASHBOARD_TILE_SELECTOR).find(DASHBOARD_TOOLTIPS_CONTAINER_SELECTOR);
            const numbersOfContainersToCreate = forceSingleContainer ? 1 : facets.length;

            dashboardContainers.empty();

            let cumulatedHeight = 0;
            for (let i = 0; i < numbersOfContainersToCreate; i++) {
                const chartContainerHeight = $(element.find(CHART_WRAPPER_SELECTOR).get(i)).outerHeight(true);
                const newTooltipContainer = $(`<div class="${DASHBOARD_TOOLTIP_CONTAINER_CLASS}"></div>`);
                newTooltipContainer.css('top', cumulatedHeight);
                newTooltipContainer.height(chartContainerHeight);
                cumulatedHeight += chartContainerHeight;
                dashboardContainers.append(newTooltipContainer);
            }
        }

        function exclude(chartDef, filterableElements) {
            const filter = ChartFilters.getExcludeNDFilter(filterableElements);

            if (DashboardUtils.isInDashboard()) {
                filter.isAGlobalFilter = true;
                $rootScope.$emit('crossFiltersAdded', {
                    filters: [filter],
                    wt1Args: {
                        from: 'chart',
                        action: 'tooltip',
                        chartType: chartDef.type,
                        filterType: 'exclude'
                    }
                });
            } else if (!chartDef.filters.some(f => ChartFilters.areFiltersEqual(f, filter))) {
                chartDef.filters = ChartFilters.updateOrAddFilter(chartDef.filters, filter);
            }
        }

        function includeOnly(chartDef, filterableElements) {
            if (DashboardUtils.isInDashboard()) {
                const filters = filterableElements.map((filterableElement) => {
                    const { dimension, axisElt, value } = filterableElement;
                    return ChartFilters.createFilter({
                        column: dimension.column,
                        columnType: dimension.type,
                        isAGlobalFilter: true,
                        excludeOtherValues: true,
                        label: `ONLY ${dimension.column}: ${value}`,
                        useMinimalUi: true,
                        includeEmptyValues: false
                    }, {
                        dimension,
                        axisElt
                    });
                });
                $rootScope.$emit('crossFiltersAdded', {
                    filters,
                    wt1Args: {
                        from: 'chart',
                        action: 'tooltip',
                        chartType: chartDef.type,
                        filterType: 'include'
                    }
                });
            } else {
                let filters = chartDef.filters;
                filterableElements.forEach((filterableElement) => {
                    const { dimension, axisElt } = filterableElement;
                    const { oldFilter, newFilter } = ChartFilters.getIncludeOnly1DFilter(dimension, axisElt, filters);
                    if (!ChartFilters.areFiltersEqual(oldFilter, newFilter)) {
                        filters = ChartFilters.updateOrAddFilter(filters, newFilter, oldFilter);
                    }
                });
                chartDef.filters = filters;
            }
        }

        const createTooltip = ($container, template, chartHandler, chartData, chartDef, defaultTooltipState, ignoreLabels) => {

            let $tooltip;
            const tooltipScopes = {};

            const facets = chartData.getAxisLabels('facet', ignoreLabels) || [{ $tensorIndex: 0 }];
            const isInDashboard = DashboardUtils.isInDashboard();
            const isEChart = ChartFeatures.isEChart(chartDef);
            if (isInDashboard) {
                createDashboardTooltipContainers($container, facets, isEChart);
            }
            const containers = getTooltipContainers($container, 'container', isEChart);

            facets.forEach((facet, index) => {
                const facetIndex = facet && facet.$tensorIndex || index;
                const tooltipScope = chartHandler.$new();
                tooltipScope.ChartFeatures = ChartFeatures;
                tooltipScope.chartData = chartData;
                tooltipScope.chartDef = chartDef;
                tooltipScope.facetIndex = index;
                tooltipScope.tooltipState = { ...defaultTooltipState };
                tooltipScope.dimensionEntries = [];
                tooltipScope.getDimensionLabel = ChartLabels.getDimensionLabel;
                tooltipScope.getDimensionNumberFormattingOptions = ChartDimension.getNumberFormattingOptions.bind(ChartDimension);
                tooltipScope.isInDashboard = isInDashboard;

                // get only groups with applied columns and based on another column for tooltips
                if (ChartFeatures.canHaveConditionalFormatting(chartDef.type)) {

                    const getValueFormatter = (measure, measureIdx) => (coords) => {
                        // get formatter for tooltip that will display the value of the cell
                        if (!chartData.isBinMeaningful(measure.function, coords, measureIdx)) {
                            return ChartFormatting.getForIsolatedNumber(measure)(ChartLabels.NO_VALUE);
                        }

                        const cellValue = chartData.aggr(measureIdx).get(coords);

                        return ChartFormatting.getForIsolatedNumber(measure)(cellValue);
                    };

                    // tooltips for values with no color
                    const defaultValueTooltips = chartDef.genericMeasures.map((measure, i) => ({
                        cellMeasure: measure,
                        genericMeasureIndex: i,
                        cellValueFormatter: getValueFormatter(measure, i)
                    }));

                    const chartStore = ChartStoreFactory.get(chartDef.$chartStoreId),
                        xDimensionIds = chartDef.xDimension.map(xDim => chartStore.getDimensionId(xDim)).filter(dim => dim !== undefined),
                        yDimensionIds = chartDef.yDimension.map(yDim => chartStore.getDimensionId(yDim)).filter(dim => dim !== undefined);

                    const currentHierarchyXDim = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'x');
                    const currentHierarchyYDim = ChartHierarchyDimension.getCurrentHierarchyDimension(chartDef, 'y');

                    if (currentHierarchyXDim) {
                        xDimensionIds.push(chartStore.getDimensionId(currentHierarchyXDim));
                    }
                    if (currentHierarchyYDim) {
                        yDimensionIds.push(chartStore.getDimensionId(currentHierarchyYDim));
                    }

                    const getColorContext = (colorOptions, colorMeasureIndex) => ({
                        chartData: chartData,
                        colorOptions: colorOptions,
                        defaultLegendDimension: chartDef.genericMeasures,
                        colorSpec: {
                            type: 'MEASURE',
                            measureIdx: colorMeasureIndex,
                            withRgba: true,
                            colorAggrFn: HierarchicalChartsUtils.getColorMeasureAggFunction(chartDef, colorMeasureIndex),
                            binsToInclude: xDimensionIds.length > 0 && yDimensionIds.length > 0 ?
                                HierarchicalChartsUtils.getBinsToIncludeInColorScale(
                                    chartData,
                                    xDimensionIds,
                                    yDimensionIds,
                                    colorMeasureIndex,
                                    chartDef,
                                    chartDef.pivotTableOptions.displayTotals.subTotals.rows,
                                    chartDef.pivotTableOptions.displayTotals.subTotals.columns
                                ) : null // avoid getting errors when everything is not set yet
                        }
                    });

                    // index of measure group colors are based of in data.aggregations
                    let basedOnMeasureIndexCounter = chartDef.genericMeasures.length - 1;
                    tooltipScope.tooltipColorGroupValues = chartDef.colorGroups
                        .filter(({ appliedColumns }) => appliedColumns)
                        .reduce((acc, { appliedColumns, rules, colorOptions, colorGroupMode, colorMeasure }) => {
                            let groupColorMeasure;

                            if (colorMeasure && colorMeasure.length > 0) { // base on another column
                                groupColorMeasure = colorMeasure[0];
                                basedOnMeasureIndexCounter += 1;
                            }

                            appliedColumns.forEach(column => {
                                const columnAggIndex = ConditionalFormattingOptions.getColorColumnIndex(
                                    column,
                                    chartDef.genericMeasures,
                                    chartDef.colorMode
                                );

                                if (columnAggIndex < 0) { // don't create a tooltip for removed values
                                    return;
                                }

                                let currColorColumnIndex;
                                let cellColorMeasure;

                                if (groupColorMeasure) { // base on another column
                                    currColorColumnIndex = basedOnMeasureIndexCounter;
                                    cellColorMeasure = groupColorMeasure;
                                } else { // base on the column (value) itself
                                    currColorColumnIndex = columnAggIndex;
                                    cellColorMeasure = chartDef.genericMeasures[columnAggIndex];
                                }

                                const hasColorProperties = colorOptions &&
                                    currColorColumnIndex >= 0 &&
                                    currColorColumnIndex < chartData.data.aggregations.length; // check if aggregation is present on the request response

                                const colorContext = hasColorProperties ? getColorContext(colorOptions, currColorColumnIndex) : {};
                                colorContext.theme = chartHandler.getChartTheme();

                                // remove default tooltip
                                acc = acc.filter(tooltip => tooltip.genericMeasureIndex !== columnAggIndex);

                                acc.push({
                                    rules,
                                    colorGroupMode,
                                    colorProperties: hasColorProperties ? {
                                        getColorScale: () => ChartColorScales.createColorScale(colorContext),
                                        getBin: (cellCoords) => chartData.getAggrLoc(currColorColumnIndex, cellCoords)
                                    } : undefined,
                                    basedOnMeasure: cellColorMeasure,
                                    basedOnMeasureIndex: currColorColumnIndex,
                                    colorValueFormatter: getValueFormatter(cellColorMeasure, currColorColumnIndex),
                                    cellValueFormatter: getValueFormatter(chartDef.genericMeasures[columnAggIndex], columnAggIndex),
                                    cellMeasure: chartDef.genericMeasures[columnAggIndex], // applied column
                                    genericMeasureIndex: columnAggIndex // index of applied column
                                });
                            });

                            return acc;
                        }, defaultValueTooltips);

                    tooltipScope.tooltipColorGroupValues.sort((a, b) => a.genericMeasureIndex - b.genericMeasureIndex);
                }

                // Initialize flags
                tooltipScope.canMultiDimensionalFilter = false;
                tooltipScope.canFilter = false;

                tooltipScope.coords = function() {
                    return angular.extend({}, tooltipScope.tooltipState.coords, { facet: facetIndex });
                };

                tooltipScope.exclude = function() {
                    const { filterableElements } = ChartFilters.getFilterableOptions(ChartStoreFactory.get(chartDef.$chartStoreId), chartData, tooltipScope.coords(), chartDef);
                    exclude(chartDef, filterableElements);
                };

                tooltipScope.includeOnly = function() {
                    const { filterableElements } = ChartFilters.getFilterableOptions(ChartStoreFactory.get(chartDef.$chartStoreId), chartData, tooltipScope.coords(), chartDef);
                    includeOnly(chartDef, filterableElements);
                };

                let $div = $(containers.get(index));

                if (isEChart) {
                    $div = $(containers.get(0));
                }

                //  We append the tooltip template to the chart wrapper
                $tooltip = $(template).appendTo($div);
                tooltipScopes[facetIndex] = tooltipScope;

                globalTooltipScopes++;

                tooltipScope.$on('$destroy', function() {
                    globalTooltipScopes--;
                    Logger.debug('Destroying tooltipScope: ' + tooltipScopes.$id + ' - new total ' + globalTooltipScopes);
                    tooltipScopes[facetIndex] = null;
                    delete tooltipScopes[facetIndex];
                    if (tooltipScope.unregisterUnfixTooltipEvent) {
                        tooltipScope.unregisterUnfixTooltipEvent();
                    }
                });

                $compile($tooltip)(tooltipScope);
            });

            return {
                tooltipScopes
            };
        };

        const hasUaDimensions = (chartDef) => {
            return chartDef.type === CHART_TYPES.SCATTER;
        };

        const getScatterMPDimensions = (chartDef, coords) => {
            const pairIndex = coords.extras.pair;
            const displayablePairs = ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
            const uaXDimension = ChartUADimension.getPairUaXDimension(displayablePairs, displayablePairs[pairIndex]);
            return [
                { isDisplayed: !!displayablePairs[pairIndex] && !!uaXDimension[0], dimension: uaXDimension[0], axisName: 'x' },
                { isDisplayed: !!displayablePairs[pairIndex] && !!displayablePairs[pairIndex].uaYDimension[0], dimension: displayablePairs[pairIndex].uaYDimension[0], axisName: 'y' }
            ];
        };

        const unfocusElement = function(desaturatedColor) {
            const d3Elem = d3.select(this);
            if (d3Elem.datum().transparent) {
                d3Elem.interrupt('updateFocus').transition('updateFocus').duration(DEFAULT_COLOR_TRANSITION_DURATION).attr('opacity', 0);
            } else {
                d3Elem.interrupt('updateFocus').transition('updateFocus').duration(DEFAULT_COLOR_TRANSITION_DURATION).attr(this._colorAttr, desaturatedColor);
            }
        };

        const focusElement = function(color, chartTransparency) {
            const d3Elem = d3.select(this);
            if (d3Elem.datum().transparent) {
                d3Elem.interrupt('updateFocus').transition('updateFocus').duration(DEFAULT_COLOR_TRANSITION_DURATION)
                    .attr('opacity', chartTransparency)
                    .attr(this._colorAttr, color);
            } else {
                d3Elem.transition(DEFAULT_COLOR_TRANSITION_DURATION)
                    .attr(this._colorAttr, color);
            }
        };

        const resetElementColor = function(color) {
            const d3Elem = d3.select(this);
            if (d3Elem.datum().transparent) {
                d3Elem.interrupt('updateFocus').transition('updateFocus').duration(DEFAULT_COLOR_TRANSITION_DURATION)
                    .attr('opacity', 0)
                    .attr(this._colorAttr, color);
            } else {
                d3Elem.interrupt('updateFocus').transition('updateFocus').duration(DEFAULT_COLOR_TRANSITION_DURATION)
                    .attr(this._colorAttr, color);
            }
        };

        return {
            /**
             * Initialize the tooltip(s) for the given chart and return an object with methods to control them, and enable tooltips on new elements.
             * This also controls the saturation/desaturation of svg elements on hover by keeping a list of registered elements for every color coordinate.
             * @param {jQuery} $container
             * @param {$scope} chartHandler
             * @param {ChartTensorDataWrapper} chartData
             * @param {ChartDef} chartDef
             * @param {string} colorSpecId
             * @param {array} measureFormatters
             * @param {Function} colorScale
             * @return {{showForCoords: showForCoords, hide: hide, fix: fix, unfix: unfix, setAnimationFrame: setAnimationFrame, getAnimationContext: getAnimationContext, registerEl: registerEl, addTooltipHandlers: addTooltipHandlers, removeTooltipsHandlers: removeTooltipsHandlers, focusColor: focusColor, resetColors: resetColors}}
             */
            create: function($container, chartHandler, chartData, chartDef, measureFormatters, colorScale, ignoreLabels = new Set()) {
                /**
                 * Clears previous tooltips if any
                 * (especially useful for Echarts, because the chart-wrapper is not recreated when we modify chartHeight with subcharts)
                 */
                const tooltipContainer = getTooltipContainers($container, 'container', ChartFeatures.isEChart(chartDef));
                const $oldTooltips = tooltipContainer.find('.svg-tooltip.pivot-chart-tooltip');
                $oldTooltips.remove();

                function isColorDimensionDisplayed(chartDef, chartData, options = {}) {
                    if (chartData.axesDef.color !== undefined) {
                        if (chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES) {
                            return options.displayColorTooltip !== false; // To handle undefined case properly
                        }
                        return ![CHART_TYPES.TREEMAP, CHART_TYPES.RADAR].includes(chartDef.type);
                    }
                    return false;
                }

                function getDimensionTooltipEntry(chartDef, chartData, coords, dimension, axisName, colored) {
                    if (!coords || !axisName) {
                        return {};
                    }
                    let label = ChartLabels.getDimensionLabel(dimension);
                    let axisElt = {};
                    // Charts that cannot have binning options also cannot display their values as binned. To prevent that, we set the min and max to null.
                    if (!ChartFeatures.canHaveBinningOptions(chartDef)) {
                        axisElt.min = null;
                        axisElt.max = null;
                    }

                    const dimensionTooltipId = `qa_charts_tooltip-dimension-${(dimension.column || '').replace(/\s/g, '_')}`;

                    let value, filterValue;

                    if (chartDef.type === CHART_TYPES.SCATTER || chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                        label = ChartLabels.getUaLabel(dimension);
                        value = chartData.uaFormattedVal(dimension, coords, axisName);
                        filterValue = value;
                    } else if (chartDef.type === CHART_TYPES.RADAR) {
                        const radarDimension = ChartDimension.getGenericDimension(chartDef);
                        // Format labels other than first row
                        label = dimension.isA === 'measure' ?
                            ChartLabels.getLongMeasureLabel(dimension) : // formatting for text
                            ChartLabels.getFormattedLabel( // formatting for numbers/bins
                                dimension,
                                ChartDimension.getNumberFormattingOptions(radarDimension),
                                dimension.min,
                                dimension.max
                            );

                        const formattingOptions = ChartDimension.getNumberFormattingOptions({
                            ...chartDef.genericMeasures[
                                chartData.shouldDrawPolygonsFromMeasures() ?
                                    coords.measure : // For measures as polygons, get formatting by polygon
                                    dimension.index // For measures as axes, get formatting by axis
                            ],
                            type: 'NUMERICAL' // force alphanumerical data (countD & nonNull) to numerical to get formatting
                        });

                        value = ChartLabels.getFormattedLabel(
                            dimension.value,
                            formattingOptions
                        );
                        filterValue = ChartLabels.getFormattedLabel(
                            { label: dimension.label },
                            dimension.isA === 'measure' ? formattingOptions : {}
                        );
                        axisElt = dimension;
                        dimension = radarDimension;
                    } else {
                        axisElt = chartData.getAxisLabels(axisName)[coords[axisName]];
                        value = ChartLabels.getAxisEltFormattedLabel({
                            chartDef,
                            chartData,
                            dimension,
                            axisName,
                            axisElt
                        });
                        filterValue = value;
                    }

                    const canChartTypeBeFiltered = ChartFeatures.canFilterInTooltip(chartDef);
                    const isInDashboard = DashboardUtils.isInDashboard();
                    const isRadarChartMeasure = chartDef.type === CHART_TYPES.RADAR && axisElt.isA === 'measure';
                    const isInReusableDimensionPreview = chartDef.$isInReusableDimensionPreview;
                    const isOthersBin = axisElt.label === DKU_OTHERS_VALUE;
                    const isTotalBar = axisElt.label === DKU_TOTAL_VALUE;
                    const canAddEmptyFilter = canChartTypeBeFiltered && !isRadarChartMeasure && !isInReusableDimensionPreview;
                    const canFilter = canChartTypeBeFiltered && !isRadarChartMeasure && !isOthersBin && !isTotalBar && (!isInDashboard || DashboardFilters.canCrossFilter($stateParams.pageId)) && !isInReusableDimensionPreview;
                    const isDrillable = canFilter && !isInDashboard && ChartDimension.isDrillableForElement(chartDef, dimension, axisElt);
                    const nativeDrilldownType = isDrillable ? ChartDimension.isGroupedNumerical(dimension) ? 'NUMERICAL' : 'DATE' : null;
                    const isHierarchyDrillable = dimension && !isTotalBar && ChartFeatures.canDrillHierarchy(chartDef.type) && dimension.hierarchyId && !isOthersBin && !isRadarChartMeasure;
                    const isHierarchyDrillableDown = isHierarchyDrillable && ChartHierarchyDimension.canDimensionDrillDown(chartDef, dimension);
                    const isHierarchyDrillableUp = isHierarchyDrillable && ChartHierarchyDimension.canDimensionDrillUp(chartDef, dimension);
                    const hierarchyName = !isRadarChartMeasure && ChartHierarchyDimension.getDimensionHierarchyName(chartDef, dimension);
                    const previousHierarchyDimension = ChartHierarchyDimension.getDimensionHierarchyPreviousDimensionName(chartDef, dimension);
                    const nextHierarchyDimension = ChartHierarchyDimension.getDimensionHierarchyNextDimensionName(chartDef, dimension);
                    const tooltipLabel = `${ChartLabels.getDimensionLabel(dimension)} - ${filterValue}`;
                    const chartTheme = chartHandler.getChartTheme();
                    return {
                        label,
                        value,
                        labelId: `${dimensionTooltipId}-name`,
                        valueId: `${dimensionTooltipId}-value`,
                        drillDownActionId: `${dimensionTooltipId}-drill-down`,
                        drillUpActionId: `${dimensionTooltipId}-drill-up`,
                        colored,
                        tooltipLabel,
                        filterValue,
                        isDrillable,
                        nativeDrilldownType,
                        isHierarchyDrillableDown,
                        isHierarchyDrillableUp,
                        canFilter,
                        hierarchyName,
                        previousHierarchyDimension,
                        nextHierarchyDimension,
                        hasActions: canFilter || (canAddEmptyFilter && !isInDashboard) || isHierarchyDrillableDown || isHierarchyDrillableUp,
                        canAddEmptyFilter,
                        isOthersBin,
                        chartTheme,
                        isTotalBar,
                        addEmptyFilter: () => {
                            const newFilter = ChartFilters.createFilter({ column: dimension.column, columnType: dimension.type, excludeOtherValues: false });
                            chartDef.filters = ChartFilters.updateOrAddFilter(chartDef.filters, newFilter);
                        },
                        drill: () => {
                            const { updatedFilters, newDateRange } = ChartDrilldown.getUpdateForNativeDrillDown(chartDef, axisElt, dimension);
                            if (newDateRange) {
                                dimension.dateParams.mode = newDateRange;
                            }
                            chartDef.filters = updatedFilters;
                        },
                        hierarchyDrillDown: () => {
                            ChartDrilldown.handleDrillDown(chartDef, dimension, DRILL_DOWN_SOURCE.TOOLTIP, axisElt);
                        },
                        hierarchyDrillUp: () => {
                            ChartDrilldown.handleDrillUp(chartDef, dimension, DRILL_UP_SOURCE.TOOLTIP);
                        },
                        includeOnly: () => {
                            includeOnly(chartDef, [{ axisElt, dimension, value: filterValue }]);
                        },
                        exclude: () => {
                            exclude(chartDef, [{ axisElt, dimension, value: filterValue }]);
                        }
                    };
                }

                function isColored(chartDef) {
                    return chartDef.type === CHART_TYPES.SCATTER || chartDef.type === CHART_TYPES.BOXPLOTS;
                }

                function getDimensionEntries(coords, options) {
                    /** @type {ChartStore} */
                    const chartStore = ChartStoreFactory.get(chartDef.$chartStoreId);
                    const dimensionDefs = [
                        { isDisplayed: isColorDimensionDisplayed(chartDef, chartData, options) && !!coords, dimension: ChartColorUtils.getColorDimensionOrMeasure(chartDef), axisName: 'color', colored: isColored(chartDef) },
                        { isDisplayed: !!chartDef.uaXDimension[0] && hasUaDimensions(chartDef), dimension: chartDef.uaXDimension[0], axisName: 'x' },
                        { isDisplayed: !!chartDef.uaYDimension[0] && hasUaDimensions(chartDef), dimension: chartDef.uaYDimension[0], axisName: 'y' },
                        { isDisplayed: !!chartDef.uaShape[0] && hasUaDimensions(chartDef), dimension: chartDef.uaShape[0], axisName: 'shape' },
                        { isDisplayed: !!chartDef.uaSize[0] && hasUaDimensions(chartDef), dimension: chartDef.uaSize[0], axisName: 'size' },
                        {
                            isDisplayed: chartDef.genericDimension0.length && ![CHART_TYPES.PIE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY].includes(chartDef.type),
                            dimension: chartDef.genericDimension0[0],
                            axisName: chartDef.type === CHART_TYPES.STACKED_BARS ? 'y' : 'x'
                        },
                        { isDisplayed: chartDef.groupDimension.length, dimension: chartDef.groupDimension[0], axisName: 'group' },
                        { isDisplayed: chartData.axesDef && chartData.axesDef.facet, dimension: chartDef.facetDimension[0], axisName: 'facet' },
                        { isDisplayed: chartData.axesDef && chartData.axesDef.animation, dimension: chartDef.animationDimension[0], axisName: 'animation' },
                        { isDisplayed: chartDef.xDimension.length && !ChartDimension.hasCustomDimId(chartDef.type), dimension: chartDef.xDimension[0], axisName: 'x' },
                        { isDisplayed: chartDef.yDimension.length && !ChartDimension.hasCustomDimId(chartDef.type), dimension: chartDef.yDimension[0], axisName: 'y' },
                        { isDisplayed: chartDef.boxplotBreakdownDim.length && coords, dimension: chartDef.boxplotBreakdownDim[0], axisName: 'x' },

                        ...(chartDef.genericHierarchyDimension.length && ![CHART_TYPES.PIE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY].includes(chartDef.type) ?
                            chartDef.genericHierarchyDimension[0].dimensions.map(dim => getHierarchyDimensionDef(dim, chartDef.type === CHART_TYPES.STACKED_BARS ? 'y' : 'x', coords))
                            : []),

                        ...(chartDef.xHierarchyDimension.length && !ChartDimension.hasCustomDimId(chartDef.type) ?
                            chartDef.xHierarchyDimension[0].dimensions.map(dim => getHierarchyDimensionDef(dim, 'x', coords))
                            : []),

                        ...(chartDef.yHierarchyDimension.length && !ChartDimension.hasCustomDimId(chartDef.type) ?
                            chartDef.yHierarchyDimension[0].dimensions.map(dim => getHierarchyDimensionDef(dim, 'y', coords))
                            : []),

                        ...(chartDef.groupHierarchyDimension.length ?
                            chartDef.groupHierarchyDimension[0].dimensions.map(dim => getHierarchyDimensionDef(dim, 'group', coords))
                            : []),

                        ...(chartDef.boxplotBreakdownHierarchyDimension.length ?
                            chartDef.boxplotBreakdownHierarchyDimension[0].dimensions.map(dim => getHierarchyDimensionDef(dim, 'x', coords))
                            : []),

                        ...(chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) ? getScatterMPDimensions(chartDef, coords) : [],

                        chartDef.type === CHART_TYPES.RADAR && { // first row for radar chart
                            isDisplayed: true,
                            dimension: {
                                ...chartData.getAxisLabels('polygon')[coords.measure],
                                label: ChartLabels.getLongMeasureLabel(chartData.getAxisLabels('polygon')[coords.measure])
                            },
                            axisName: 'radarAxis',
                            colored: true
                        },

                        ...(chartDef.type === CHART_TYPES.RADAR ?
                            // Remove last element from tensor
                            chartData.aggr(coords.measure).get().slice(0, chartData.getAxisLabels('axis').length).map((value, index) => ({
                                isDisplayed: true,
                                dimension: {
                                    ...chartData.getAxisLabels('axis')[index], // left column (label)
                                    value: { sortValue: value }, // right column (value)
                                    index: index
                                },
                                axisName: 'radarSeries'
                            })) : []),

                        ...(ChartDimension.hasCustomDimId(chartDef.type) ? chartDef.xDimension.map(xDimension => {
                            return getCustomDimensionDef(chartStore, coords, xDimension);
                        }) : []),
                        ...(ChartDimension.hasCustomDimId(chartDef.type) ? chartDef.yDimension.map(yDimension => {
                            const yDimensionId = chartStore.getDimensionId(yDimension);

                            return {
                                isDisplayed: coords[yDimensionId] !== undefined && coords[yDimensionId] !== chartData.getSubtotalLabelIndex(yDimensionId),
                                dimension: yDimension,
                                axisName: yDimensionId
                            };
                        }) : []),
                        ...(ChartDimension.hasCustomDimId(chartDef.type) && chartDef.xHierarchyDimension && chartDef.xHierarchyDimension.length ? chartDef.xHierarchyDimension[0].dimensions.map(dimension => {
                            return getHierarchyDimensionDef(dimension, chartStore.getDimensionId(dimension), coords);
                        }) : []),
                        ...(ChartDimension.hasCustomDimId(chartDef.type) && chartDef.yHierarchyDimension && chartDef.yHierarchyDimension.length ? chartDef.yHierarchyDimension[0].dimensions.map(dimension => {
                            return getHierarchyDimensionDef(dimension, chartStore.getDimensionId(dimension), coords);
                        }) : [])
                    ];

                    return dimensionDefs
                        .filter(({ isDisplayed }) => isDisplayed)
                        .map(({ dimension, axisName, colored }) => getDimensionTooltipEntry(chartDef, chartData, coords, dimension, axisName, colored));
                }

                const getCustomDimensionDef = (chartStore, coords, dimension) => {
                    const dimensionId = chartStore.getDimensionId(dimension);

                    return {
                        isDisplayed: coords[dimensionId] !== undefined,
                        dimension: dimension,
                        axisName: dimensionId
                    };
                };

                const getHierarchyDimensionDef = (dimension, dimensionId, coords) => {
                    const dimensionLevel = ChartHierarchyDimension.getDimensionLevel(chartDef, dimension);
                    const hierarchyLevel = ChartHierarchyDimension.getDimensionHierarchyLevel(chartDef, dimension);
                    return {
                        isDisplayed: coords && dimensionLevel === hierarchyLevel && !_.isNil(coords[dimensionId]),
                        dimension: dimension,
                        axisName: dimensionId
                    };
                };

                /**
                 * renderTooltipScopes is used to update tooltipScopes and then render them (with $rootScope.$apply)
                 * @param {Object} tooltipScopes a set of tooltip scopes
                 * @param {Function} callback a function which mutates a tooltipScope individually
                 */
                const renderTooltipScopes = (tooltipScopes, callback) => {
                    angular.forEach(tooltipScopes, (tooltipScope, tooltipScopeIndex) => {
                        //  tooltipScopes is an object, and we're iterating on its keys (which are stringified numbers)
                        const facetIndex = parseInt(tooltipScopeIndex);
                        callback(tooltipScope, facetIndex, tooltipScopes);
                    });

                    //  Triggers digest to display modifications in the DOM
                    if (!$rootScope.$$phase) {
                        $rootScope.$apply();
                    }
                };

                let tooltipScopes = {};
                const defaultTooltipState = { shown: false, formatters: measureFormatters, canPinTooltip: ChartFeatures.canPinTooltip(chartDef.type) };
                const templateUrl = '/templates/simple_report/tooltips/std-aggr-nd.html';

                $q.when($templateCache.get(templateUrl) || $http.get(templateUrl, { cache: true })).then(function(template) {
                    if (angular.isArray(template)) {
                        template = template[1];
                    } else if (angular.isObject(template)) {
                        template = template.data;
                    }

                    const tooltipOptions = createTooltip($container, template, chartHandler, chartData, chartDef, defaultTooltipState, ignoreLabels);
                    tooltipScopes = tooltipOptions.tooltipScopes;
                });

                const ret = {
                    colorScale,

                    destroy: function() {
                        angular.forEach(tooltipScopes, function(tooltipScope) {
                            tooltipScope.$destroy();
                        });
                    },

                    /**
                     * Show the tooltip for the given measure/coords
                     * @param {number} measure : measure idx to show data for
                     * @param {array} coords : coords to show data for
                     * @param event : mousemove event
                     * @param color : tooltip color
                     */
                    showForCoords: function(measure, coords, event, color, options) {

                        if (!ChartFeatures.shouldDisplayTooltips(chartHandler.noTooltips, chartDef.tooltipOptions, ret.fixed)) {
                            return;
                        }

                        const currentFacetIndex = coords && coords.facet || 0;
                        if (coords) {
                            coords.animation = AnimatedChartsUtils.getAnimationCoord(chartHandler.animation);
                        }

                        // color can be an object { value: rbg(...) } or string 'rgb(...)'
                        color = (typeof color === 'string' ? color : color && color.value) || 'transparent';

                        //  Computes tooltip content (called first to retrieve tooltip height) and assign it to given tooltipScope
                        const updateTooltipContent = (tooltipScope, facetIndex) => {
                            tooltipScope.shown = true;
                            const canPinTooltip = ChartFeatures.canPinTooltip(chartDef.type);
                            tooltipScope.dimensionEntries = getDimensionEntries(coords ? { ...coords, facet: facetIndex } : undefined, options);
                            const canDisplayTooltipMeasures = tooltipScope.dimensionEntries.every(entry => !entry.isTotalBar);
                            tooltipScope.additionalMeasures = options && options.additionalMeasures !== null ? options.additionalMeasures : [];
                            const hasFilterableAxisElts = tooltipScope.dimensionEntries.some(entry => entry.canFilter);
                            tooltipScope.canMultiDimensionalFilter = tooltipScope.dimensionEntries.every(entry => entry.canFilter);
                            tooltipScope.canFilter = ChartFeatures.canFilterInTooltip(chartDef) && hasFilterableAxisElts;

                            tooltipScope.tooltipState = {
                                ...tooltipScope.tooltipState,
                                shown: true,
                                measure,
                                coords,
                                color,
                                canPinTooltip,
                                canDisplayTooltipMeasures
                            };
                        };

                        //  Retrieves tooltip position
                        const getTooltipPosition = (tooltipScopes, facetIndex) => {
                            const tooltipScopeFacetIndex = tooltipScopes[facetIndex].facetIndex;
                            const $wrapper = getTooltipContainers($(event.target), 'tooltip', ChartFeatures.isEChart(chartDef));
                            const $tooltips = $wrapper.find('.svg-tooltip.pivot-chart-tooltip');
                            const tooltip = $tooltips.length > 1 ? $tooltips.get(tooltipScopeFacetIndex) : $tooltips.get(0);
                            const tooltipHeight = tooltip.clientHeight;

                            let left = 'auto',
                                right = 'auto',
                                top = 0;

                            const wrapperOffset = $wrapper.offset();
                            if (!wrapperOffset) {
                                // can happen if event.target has been detached right after hover and is not a child of .chart-wrapper anymore (in pivot fattable)
                                return;
                            }

                            const wrapperWidth = $wrapper.width();
                            const wrapperHeight = $wrapper.height();
                            const offsetX = event.pageX - wrapperOffset.left;
                            const offsetY = event.pageY - wrapperOffset.top;
                            const tooltipMargin = 20;

                            if (offsetX < wrapperWidth / 2) {
                                left = (offsetX + 10) + 'px';
                            } else {
                                right = (wrapperWidth - offsetX + 10) + 'px';
                            }

                            //  maxTooltipTop is the max top a tooltip can be positioned at without causing an overflow
                            const maxTooltipTop = wrapperHeight - tooltipHeight - tooltipMargin;

                            //  centeredTooltipTop is the actual top value when trying to center the current tooltip next to the mouse
                            const centeredTooltipTop = offsetY - tooltipHeight / 2;

                            top = Math.max(0, Math.min(maxTooltipTop, centeredTooltipTop)) + 'px';

                            /**
                             * Echarts tooltips have a different implementation:
                             * all tooltip scopes belongs to the same chart div wrapper, which means that we've got to compute each "top"
                             * in relation with the one we hover.
                             */
                            if (chartDef.displayWithECharts || chartDef.displayWithEChartsByDefault) {
                                const tooltipScopeTop = chartDef.chartHeight * tooltipScopeFacetIndex;
                                const hoveredTooltipScopeTop = chartDef.chartHeight * tooltipScopes[currentFacetIndex].facetIndex;
                                top = Math.max(0, Math.min(maxTooltipTop, tooltipScopeTop - hoveredTooltipScopeTop + centeredTooltipTop)) + 'px';

                                /*
                                 * multiTooltips can be true even if there is no facets, one facet = one tooltipScope,
                                 * so we check if we have more than one.
                                 */
                                const numberOfFacets = Object.keys(tooltipScopes || []).length;
                                if (numberOfFacets > 1 && chartDef.multiTooltips) {
                                    const lastFacetIndex = numberOfFacets - 1;
                                    const lastTooltipScopeTop = chartDef.chartHeight * lastFacetIndex;
                                    const lastCenteredTooltipTop = lastTooltipScopeTop - hoveredTooltipScopeTop + centeredTooltipTop;
                                    const lastTooltipTopGap = lastCenteredTooltipTop - maxTooltipTop;
                                    top = Math.max(0, tooltipScopeTop - hoveredTooltipScopeTop + centeredTooltipTop - lastTooltipTopGap) + 'px';
                                }
                            }

                            return { left, right, top };
                        };

                        //  Computes tooltip positioning and assign it to given tooltipScope
                        const updateTooltipPosition = (tooltipScope, facetIndex, tooltipScopes) => {
                            const { left, right, top } = getTooltipPosition(tooltipScopes, facetIndex);
                            tooltipScope.tooltipState = {
                                ...tooltipScope.tooltipState,
                                left,
                                right,
                                top
                            };
                        };

                        /**
                         * We have to update the tooltip content before updating its position
                         * because we need to know its height beforehand.
                         */
                        if (!chartDef.multiTooltips) {
                            updateTooltipContent(tooltipScopes[currentFacetIndex], currentFacetIndex, tooltipScopes);
                            tooltipScopes[currentFacetIndex].$apply();

                            updateTooltipPosition(tooltipScopes[currentFacetIndex], currentFacetIndex, tooltipScopes);
                            tooltipScopes[currentFacetIndex].$apply();

                            renderTooltipScopes(tooltipScopes, (tooltipScope, facetIndex) => {
                                if (facetIndex !== currentFacetIndex) {
                                    tooltipScope.shown = false;
                                    tooltipScope.tooltipState.shown = false;
                                }
                            });
                        } else {
                            renderTooltipScopes(tooltipScopes, updateTooltipContent);
                            renderTooltipScopes(tooltipScopes, updateTooltipPosition);
                        }
                    },


                    /**
                     * Hide the tooltip
                     */
                    hide: function() {
                        renderTooltipScopes(tooltipScopes, function(tooltipScope) {
                            if (tooltipScope.tooltipState.persistent) {
                                return;
                            }

                            tooltipScope.shown = false;
                            tooltipScope.tooltipState.shown = false;
                            tooltipScope.dimensionEntries = [];
                        });
                    },

                    /**
                     * Fix the tooltip (won't follow mouse anymore and won't auto-hide)
                     */
                    fix: function() {
                        if (!ChartFeatures.shouldDisplayTooltips(chartHandler.noTooltips, chartDef.tooltipOptions, false)) {
                            return;
                        }
                        if (ret.fixed) {
                            ret.unfix();
                            ret.resetColors();
                        } else {
                            renderTooltipScopes(tooltipScopes, tooltipScope => {
                                tooltipScope.unregisterUnfixTooltipEvent = $rootScope.$on('unfixTooltip', ret.unfix);
                                if (tooltipScope.tooltipState.shown) {
                                    ret.fixed = true;
                                    tooltipScope.tooltipState.persistent = true;
                                }
                            });
                        }
                    },

                    /**
                     * Unfix the tooltip
                     */
                    unfix: function() {
                        if (!ret.fixed) {
                            return ret.hide();
                        }
                        ret.fixed = false;

                        renderTooltipScopes(tooltipScopes, tooltipScope => {
                            if (tooltipScope.unregisterUnfixTooltipEvent) {
                                tooltipScope.unregisterUnfixTooltipEvent();
                            }
                            tooltipScope.shown = false;
                            tooltipScope.tooltipState.shown = false;
                            tooltipScope.tooltipState.persistent = false;
                        });
                    },

                    handleChartMouseOver: function(getTargetElementProperties, colorFocusHandler) {
                        const target = d3.event.target;
                        const properties = getTargetElementProperties(target);
                        if (!_.isNil(properties) && !_.isNil(colorFocusHandler)) {
                            const colorIndex = ChartColorUtils.getColorIndex(properties.coords, colorFocusHandler.hasColorDim);
                            if (colorFocusHandler.focusColor && !ret.fixed && colorIndex != null) {
                                ret.focusColor(colorIndex);
                            }
                        }
                    },

                    handleChartMouseOut: function(getTargetElementProperties, colorFocusHandler) {
                        if (!ret.fixed) {
                            ret.hide();
                        }
                        const properties = d3.event.relatedTarget && getTargetElementProperties(d3.event.relatedTarget);
                        if (!_.isNil(colorFocusHandler)) {
                            if (!_.isNil(properties)) {
                                const colorIndex = ChartColorUtils.getColorIndex(properties.coords, colorFocusHandler.hasColorDim);
                                if (colorFocusHandler.focusColor && !ret.fixed && colorIndex != null) {
                                    ret.focusColor(colorIndex);
                                }
                            } else {
                                ret.resetColors();
                            }
                        }
                    },

                    handleChartMouseMove: function(getTargetElementProperties, colorFocusHandler, tooltipOptions) {
                        const target = d3.event.target;
                        const properties = getTargetElementProperties(target);
                        if (!_.isNil(properties)) {
                            const measure = properties.measure;
                            const coords = properties.coords;
                            const colorIndex = ChartColorUtils.getColorIndex(coords, colorFocusHandler.hasColorDim);
                            if (properties.showTooltip) {
                                const color = colorIndex == null ? undefined : chartHandler.legendsWrapper.getLegend(0).items[colorIndex].rgbaColor || chartHandler.legendsWrapper.getLegend(0).items[colorIndex].color;
                                ret.showForCoords(
                                    measure,
                                    coords,
                                    d3.event,
                                    color,
                                    tooltipOptions
                                );
                            }
                        }
                    },

                    handleChartClick: function(getTargetElementProperties) {
                        const properties = getTargetElementProperties(d3.event.target);
                        if (properties != null) {
                            d3.event.stopPropagation();
                            if (ret.fixed) {
                                ret.unfix();
                            } else {
                                ret.fix();
                                const unfixTooltip = () => {
                                    document.removeEventListener('click', unfixTooltip);
                                    ret.unfix();
                                    ret.resetColors();
                                };
                                document.addEventListener('click', unfixTooltip);
                            }
                        }
                    },

                    handleChartContextMenu: function(getTargetElementProperties) {
                        const target = d3.event.target;
                        const properties = getTargetElementProperties(target);
                        if (properties != null) {
                            d3.event.preventDefault();
                        }
                        ret.unfix();
                        ret.resetColors();
                    },

                    addChartTooltipAndHighlightHandlers: function(chart, getTargetElementProperties, colorFocusHandler, tooltipOptions) {
                        const element = d3.select(chart);
                        element
                            .on('mouseover', () => ret.handleChartMouseOver(getTargetElementProperties, colorFocusHandler))
                            .on('mouseout', () => ret.handleChartMouseOut(getTargetElementProperties, colorFocusHandler));
                        if (!chartHandler.noTooltips) {
                            element
                                .on('mousemove', () => ret.handleChartMouseMove(getTargetElementProperties, colorFocusHandler, tooltipOptions))
                                .on('click', () => ret.handleChartClick(getTargetElementProperties))
                                .on('contextmenu.tooltip', () => ret.handleChartContextMenu(getTargetElementProperties));
                        }
                    },

                    removeChartTooltipHandlers: function(chart) {
                        d3.select(chart)
                            .on('mouseover', null)
                            .on('mouseout', null)
                            .on('mousemove', null)
                            .on('click', null)
                            .on('contextmenu.tooltip', null);
                    },

                    /**
                     * Update the tooltipState's animation frame
                     * @param {number} frameIdx: animation coordinate
                     */
                    setAnimationFrame: function(frameIdx) {
                        angular.forEach(tooltipScopes, tooltipScope => {
                            if (tooltipScope.tooltipState.coords) {
                                tooltipScope.tooltipState.coords.animation = frameIdx;
                            }
                        });
                    },


                    /**
                     * Register an element for his color coord and add handlers to show tooltip on mousemove
                     * @param {DOMElement} el
                     * @param {array} coords: coords dict of this element
                     * @param {string} colorAttr: the color attribute to control the element (usually 'fill' or 'stroke')
                     * @param {boolean} noTooltip: only register the element for color change but don't add tooltip handlers
                     */
                    registerEl: function(el, coords, colorAttr, noTooltip, options, hasColorDim = false) {
                        if (chartHandler.noTooltips) {
                            return;
                        }

                        el._colorAttr = colorAttr;
                        const c = coords ? ChartColorUtils.getColorIndex(coords, hasColorDim) : null;

                        if (colorAttr) {
                            chartHandler.legendsWrapper.getLegend(0).items[c].elements[0].push(el);
                        }

                        d3.select(el)
                            .attr('tooltip-el', true)
                            .on('mousemove', function() {
                                if (!noTooltip) {
                                    ret.showForCoords(coords ? coords.measure : 0, coords, d3.event, c == null ? undefined : chartHandler.legendsWrapper.getLegend(0).items[c].rgbaColor || chartHandler.legendsWrapper.getLegend(0).items[c].color, options);
                                }
                            })
                            .on('mouseleave', function() {
                                if (!ret.fixed) {
                                    if (!noTooltip) {
                                        ret.hide();
                                    }
                                    if (colorAttr) {
                                        ret.resetColors();
                                    }
                                }
                            })
                            .on('click', function() {
                                if (!noTooltip) {
                                    ret.fix();
                                }
                            })
                            .on('contextmenu.tooltip', function() {
                                d3.event.preventDefault();
                                ret.unfix();
                                if (colorAttr) {
                                    ret.resetColors();
                                }
                            })
                            .on('mouseenter', function(elemData) {
                                if (colorAttr && !ret.fixed) {
                                    ret.focusColor(c, elemData);
                                }
                            });
                    },


                    unregisterEl: function(el) {
                        d3.select(el)
                            .on('mousemove', null)
                            .on('mouseleave', null)
                            .on('click', null)
                            .on('contextmenu.tooltip', null)
                            .on('mouseenter', null);
                    },

                    /**
                     * Add tooltip handlers to an element
                     * @param el
                     * @param coords
                     * @param color
                     */
                    addTooltipHandlers: function(el, coords, color) {
                        if (chartHandler.noTooltips) {
                            return;
                        }

                        d3.select(el)
                            .attr('tooltip-el', true)
                            .on('mousemove.tooltip', function() {
                                ret.showForCoords(-1, coords, d3.event, color);
                            })
                            .on('mouseleave.tooltip', ret.hide)
                            .on('click', ret.fix)
                            .on('contextmenu.tooltip', function() {
                                d3.event.preventDefault();
                                ret.unfix();
                            });
                    },


                    /**
                     * Remove tooltip handlers from an element
                     * @param el
                     */
                    removeTooltipsHandlers: function(el) {
                        d3.select(el)
                            .on('mousemove.tooltip', null)
                            .on('mouseleave.tooltip', null)
                            .on('click', null)
                            .on('contextmenu.tooltip', null);
                    },


                    /**
                     * Focus on the given color coordinates (ie desaturate all other colors)
                     * @param {number} colorIndex: the color coordinate
                     */
                    focusColor: function(colorIndex, focusedElementData) {
                        if (resetColorsTimeout) {
                            // we dont need to reset colors if we immediately focus on another color
                            clearTimeout(resetColorsTimeout);
                            resetColorsTimeout = null;
                        }
                        setTimeout(() => {
                            // we need a timeout here to fix the race condition when we use a transition to draw lines/bars/labels/circles...
                            const legend = chartHandler.legendsWrapper.getLegend(0);
                            if (!_.isNil(legend)) {
                                if (legend.focusFnList) {
                                    legend.focusFnList.forEach(currentFocusColorFn => currentFocusColorFn(colorIndex));
                                }
                                if (!_.isNil(legend.items)) {
                                    legend.items.forEach(function(item, i) {
                                        if (i != colorIndex) {
                                            if (item.elements) {
                                                item.elements.each(function() {
                                                    unfocusElement.call(this, item.desaturatedColor);
                                                });
                                            }
                                        } else if (item.elements) {
                                            item.elements.each(function(elementData) {
                                                if (focusedElementData) {
                                                    // hover from the chart
                                                    if (focusedElementData.isUnaggregated && elementData !== focusedElementData) {
                                                        // particular case for unaggregated data point, we unfocus also all other elements of the same color
                                                        return unfocusElement.call(this, item.desaturatedColor);
                                                    }
                                                    return focusElement.call(this, item.color, chartDef.colorOptions.transparency);
                                                }
                                                // hover from the legend
                                                return resetElementColor.call(this, item.color);
                                            });
                                        }
                                    });
                                }
                            }
                        });
                    },


                    /**
                     * Unfocus everything
                     */
                    resetColors: function() {
                        if (resetColorsTimeout) {
                            clearTimeout(resetColorsTimeout);
                        }
                        resetColorsTimeout = setTimeout(() => {
                            const legend = chartHandler.legendsWrapper.getLegend(0);
                            if (!_.isNil(legend)) {
                                if (legend.unfocusFnList) {
                                    legend.unfocusFnList.forEach(currentUnfocusFn => currentUnfocusFn());
                                }
                                if (!_.isNil(legend.items)) {
                                    legend.items.forEach(function(item) {
                                        if (item.elements) {
                                            item.elements.each(function() {
                                                resetElementColor.call(this, item.color);
                                            });
                                        }
                                    });
                                }
                            }
                        }, 5);
                    },

                    getAnimationContext: function() {
                        return AnimatedChartsUtils.getAnimationContext(chartDef.animationDimension[0], chartHandler.animation);
                    }
                };

                return ret;
            }
        };
    }
})();
