import os, json, re, logging
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid_gid

from block_utils import LibLocationPathReplacer, generate_python_codenv, build_generate_python_codenv_script, python_version

class JupyterLabCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    def _get_adjustment_script_path(self):
        return '/opt/dataiku/jupyterlab-adjustments.py' # fixed for the time being
    
    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, "jupyterlab-entrypoint.sh")
        return entrypoint_path
        
    def _get_port(self):
        return self.config.get("port", 8989)

    def build_spec(self, spec, env, template):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        adjustments_path = self._get_adjustment_script_path()
        settings_path = self.config.get("settingsPath", "/home/dataiku/.jupyter/lab/user-settings/")
        workspaces_path = self.config.get("workspacesPath", "/home/dataiku/.jupyter/lab/workspaces")
        open_in_path = self.config.get("openInPath", "/home/dataiku/workspace")
        enable_xsrf = self.config.get("enableXSRF", False)

        # replace the lib locations in settings_path and open_in_path
        replacer = LibLocationPathReplacer(spec)
        settings_path = replacer.replace_variable_by_path(settings_path)
        workspaces_path = replacer.replace_variable_by_path(workspaces_path)
        open_in_path = replacer.replace_variable_by_path(open_in_path)

        # get code env stuff
        older_than_py39 = python_version(self.config.get("pythonVersion", "python3.9")) < python_version("python3.9")
        if older_than_py39:
            default_packages = "Babel==2.10.3 Jinja2==3.1.2 MarkupSafe==2.1.1 Pygments==2.12.0 Send2Trash==1.8.0 anyio==3.6.1 argon2_cffi==21.3.0 argon2_cffi_bindings==21.2.0 attrs==21.4.0 backcall==0.2.0 beautifulsoup4==4.11.1 bleach==5.0.0 certifi==2022.6.15 cffi==1.15.0 charset_normalizer==2.0.12 debugpy==1.6.0 decorator==5.1.1 defusedxml==0.7.1 entrypoints==0.4 fastjsonschema==2.15.3 idna==3.3 importlib_metadata==4.11.4 importlib_resources==5.8.0 ipykernel==6.15.0 ipython==7.34.0 ipython_genutils==0.2.0 ipywidgets==7.6.5 jedi==0.18.1 jsonschema==4.6.0 jupyter_client==7.3.4 jupyter_core==4.10.0 jupyter_server==1.18.0 jupyterlab==3.2.9 jupyterlab_pygments==0.2.2 jupyterlab_server==2.14.0 jupyterlab_widgets==1.1.1 matplotlib_inline==0.1.3 mistune==0.8.4 nbclassic==0.3.7 nbclient==0.6.4 nbconvert==6.5.0 nbformat==5.4.0 nest_asyncio==1.5.5 notebook==6.4.12 notebook_shim==0.1.0 packaging==21.3 pandocfilters==1.5.0 parso==0.8.3 pexpect==4.8.0 pickleshare==0.7.5 prometheus_client==0.14.1 prompt_toolkit==3.0.29 psutil==5.9.1 ptyprocess==0.7.0 pycparser==2.21 pyparsing==3.0.9 pyrsistent==0.18.1 python_dateutil>=2.8.2,<3 pyzmq==23.2.0 requests==2.28.0 six==1.16.0 sniffio==1.2.0 soupsieve==2.3.2-post1 terminado==0.15.0 tinycss2==1.1.1 tornado==6.1 traitlets==5.3.0 typing_extensions==4.2.0 urllib3==1.26.9 wcwidth==0.2.5 webencodings==0.5.1 websocket_client==1.3.3 widgetsnbextension==3.5.2 zipp==3.8.0 ipywidgets==7.6.5 jupyterlab==3.2.9"
            jupyter_commands_raw = self.config.get("jupyterCommands", """
                nbextension enable --py widgetsnbextension
                serverextension enable --py jupyterlab
                labextension install @jupyter-widgets/jupyterlab-manager --minimize=False
            """)
        else:
            default_packages = "jupyterlab==4.2.2 widgetsnbextension==4.0.11 ipywidgets==8.1.3 urllib3<2"
            jupyter_commands_raw = self.config.get("jupyterCommands", "")
        generate_codenv, pyenv_path = generate_python_codenv("JUPYTERLAB", self.config, template, default_packages, "/opt/dataiku/pyenv-jupyter", "python3.9", env.get("globalCodeEnvsExtraSettings"))

        python_bin = os.path.join(pyenv_path, "bin/python")
        jupyter_bin = os.path.join(pyenv_path, "bin/jupyter")

        # setup jupyter installation commands
        jupyter_commands_list = []
        for cmd in jupyter_commands_raw.split("\n"):
            cmd = cmd.strip()
            if cmd:
                jupyter_commands_list.append(jupyter_bin + " " + replacer.replace_variable_by_path(cmd))
        jupyter_commands = "RUN " + " \\\n   && ".join(jupyter_commands_list) if jupyter_commands_list else ""

        # prepare the code env for the "default" kernel
        py_dss_env_name = 'py39-dss'
        py_dss_env_label = 'Pandas 2.2 (Python 3.9)'
        kernel_packages = '"pandas>=2.2.2,<2.3" "numpy>=2,<2.1" "python-dateutil>=2.8.2,<3" "urllib3<2" "requests<3" "decorator==5.1.1" "ipykernel==6.23.3" "ipython>=8.12,<8.13" "ipython_genutils==0.2.0" "jupyter_client==6.1.12" "jupyter_core==4.12.0" "pexpect==4.8.0" "pickleshare==0.7.5" "ptyprocess==0.7.0" "pyzmq==23.2.1" "simplegeneric==0.8.1" "tornado>=6.3,<6.4" "traitlets==5.9.0"'
        py_dss_pyenv_path = "/opt/dataiku/" + py_dss_env_name
        generate_py_dss_codenv = build_generate_python_codenv_script("KERNEL", kernel_packages, py_dss_pyenv_path, "python3.9", env.get("globalCodeEnvsExtraSettings"))

        # add the entrypoint script in the buildir
        additional_exec_params = replacer.replace_variable_by_path(self.config.get("addExecParams", ""))
        entrypoint_script = """#! /bin/bash

USER=dataiku
HOME=/home/dataiku

unset JUPYTER_DATA_DIR
unset IPYTHONDIR
unset JUPYTER_RUNTIME_DIR
unset JUPYTER_CONFIG_DIR
export SHELL=/bin/bash

mkdir -p {settings_path}/lab
export JUPYTERLAB_SETTINGS_DIR={settings_path}
export JUPYTER_CONFIG_FILE={settings_path}/jupyter_notebook_config.py
export JUPYTERLAB_WORKSPACES_DIR={workspaces_path}

mkdir -p {settings_path}
mkdir -p {workspaces_path}

# Register user kernels if only the two default ones exist
if [ $(ls {pyenv_path}/share/jupyter/kernels/ | wc -l) -le 2 ]; then
    # TODO : if python >= 3.7, should we upgrade ipykernel ?
    if [ -d /opt/dataiku/python-code-envs ]; then
        for i in $(ls /opt/dataiku/python-code-envs); do
            source /opt/dataiku/python-code-envs/$i/bin/activate
            python3 -m ipykernel install --prefix {pyenv_path}/ --name py-dku-venv-$i --display-name "DSS Codeenv - $i"
            deactivate
        done
    fi
fi

export BASE_URL=$(eval echo "$DKU_CODE_STUDIO_BROWSER_PATH_{port}")

if [ $DKU_CODE_STUDIO_IS_PUBLIC_PORT_{port} = "1" ]; then
    BIND_ADDR=0.0.0.0
else
    BIND_ADDR=127.0.0.1
fi

{jupyter_bin} lab --ip=$BIND_ADDR --port={port} --no-browser --allow-root --ServerApp.password='' --ServerApp.token='' --ServerApp.allow_origin='*' --ServerApp.allow_remote_access=True --ServerApp.base_url="$BASE_URL" --ServerApp.root_dir={open_in_path} --ServerApp.disable_check_xsrf={disable_xsrf} {additional_exec_params}
""".format(settings_path=settings_path, 
           workspaces_path=workspaces_path, 
           open_in_path=open_in_path, 
           port=port, 
           disable_xsrf=not enable_xsrf,
           pyenv_path=pyenv_path,
           jupyter_bin=jupyter_bin,
           additional_exec_params=additional_exec_params)
        with open(os.path.join(env["buildDir"], "jupyterlab-entrypoint.sh"), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))
            
        # add the adjustment script in the buildir (to fiddle with kernels in ipynb files coming from DSS)
        adjustments_script = """#! /opt/dataiku/pyenv/bin/python
import os, sys, glob, json, re
def adjust(path):
    with open(path, 'r') as f:
        nbk = json.load(f)
    changed = False
    metadata = nbk.get("metadata", {{}})
    kernel_spec = metadata.get('kernelspec', {{}})
    kernel_name = kernel_spec.get('name')
    kernel_display_name = kernel_spec.get('display_name')
    containerized_info = re.match('Python in .* \(((builtin env)|(env (.*)))\)', kernel_display_name)
    containerized_code_env_part = containerized_info.group(1) if containerized_info is not None else None
    containerized_code_env_name = containerized_info.group(4) if containerized_info is not None else None
    if kernel_name == 'python3':
        print("Switch default kernel to matching DSS kernel")
        kernel_spec["name"] = "{py_dss_env_name}"
        kernel_spec["display_name"] = "{py_dss_env_label}"
        changed = True
    elif kernel_name.startswith('py-dku-containerized-venv--') and containerized_code_env_part == 'builtin env':
        print("Switch default containerized kernel to matching DSS kernel")
        kernel_spec["name"] = "{py_dss_env_name}"
        kernel_spec["display_name"] = "{py_dss_env_label}"
        changed = True
    elif kernel_name.startswith('py-dku-containerized-venv-') and containerized_code_env_name is not None:
        print("Switch containerized kernel to non-containerized version")
        kernel_spec["name"] = "py-dku-venv-%s" % containerized_code_env_name
        kernel_spec["display_name"] = "DSS Codeenv - %s" % containerized_code_env_name
        changed = True
    if changed:
        print("write updated notebook %s" % path)
        with open(path, 'w') as f:
            json.dump(nbk, f)
if __name__ == '__main__':
    zone_to_adjust = sys.argv[1]
    if zone_to_adjust.startswith("notebooks"): # only notebooks, not stuff in history, for example (they're in settings)
        folder_to_adjust = sys.argv[3]
        print("Perform kernel ajustments on %s" % folder_to_adjust)
        for path in glob.glob(os.path.join(folder_to_adjust, "**/*.ipynb"), recursive=True):
            adjust(path)
""".format(py_dss_env_name=py_dss_env_name,
           py_dss_env_label=py_dss_env_label)
        with open(os.path.join(env["buildDir"], "jupyterlab-adjustments.py"), "wb") as f:
            f.write(adjustments_script.encode("utf8"))

        # the dockerfile addition
        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### JUPYTER LAB BLOCK #####

USER root
WORKDIR /opt/dataiku

RUN yum -y install epel-release && yum -y install python3 python3-devel nginx git \
    && yum -y groupinstall "Development tools" \
    && yum install -y gcc openssl-devel bzip2-devel libffi-devel zlib-devel xz-devel \
    && yum -y autoremove && yum clean all 
    
# Install node & npm
RUN case $(bash -c 'ID=; source /etc/os-release && echo "$ID"' 2>/dev/null) in \
      centos) \
        yum install -y nodejs npm \
      ;; \
      almalinux) \
        dnf -qy module switch-to nodejs:22 && dnf install -y nodejs \
      ;; \
      *) \
        echo "Unsupported distrib, install nodejs manually" >&2 \
      ;; \
    esac \
    && yum -y autoremove && yum clean all 

{generate_codenv}

# Post install commands
{jupyter_commands}

# Clean up jupyterlab's staging/yarn.lock
RUN find {pyenv_path} -type f -path "*/jupyterlab/staging/yarn.lock" -delete

# Default kernel
{generate_py_dss_codenv}
RUN source {py_dss_pyenv_path}/bin/activate && python3 -m ipykernel install --prefix {pyenv_path}/ --name '{py_dss_env_name}' --display-name "{py_dss_env_label}"

COPY jupyterlab-entrypoint.sh {entrypoint_path}
COPY jupyterlab-adjustments.py {adjustments_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path} && chown dataiku:root {adjustments_path} && chmod +x {adjustments_path}

# To enable the registering of kernels
RUN chown dataiku:root {pyenv_path}/share/jupyter/kernels

# USER dataiku
USER {uid_gid}
WORKDIR /home/dataiku
""".format(entrypoint_path=entrypoint_path, 
           adjustments_path=adjustments_path,
           generate_codenv=generate_codenv,
           jupyter_commands=jupyter_commands,
           pyenv_path=pyenv_path,
           generate_py_dss_codenv=generate_py_dss_codenv,
           py_dss_pyenv_path=py_dss_pyenv_path,
           py_dss_env_name=py_dss_env_name,
           py_dss_env_label=py_dss_env_label,
           uid_gid=get_dataiku_user_uid_gid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) + "${baseUrlPort" + str(port) + "}/static/favicons/favicon.ico" # baseUrlPort should be replaced by actual url in BlockBasedCodeStudioMeta/buildYaml
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {
            "label": "Jupyter Lab",
            "proxiedUrlSuffix": "$request_uri",
            "exposeHtml": True,
            "port": port,
            "readinessProbeUrl": readiness_probe_url
        }
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        spec['fileAdjustmentScript'] = self._get_adjustment_script_path()
        return spec

    def build_creation(self, spec, env):
        return spec
