import os, json, re, logging
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid_gid

from block_utils import LibLocationPathReplacer, generate_python_codenv

from distutils.version import LooseVersion

class StreamlitCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    _ENTRYPOINT_FILE = "streamlit-entrypoint.sh"
        
    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, self._ENTRYPOINT_FILE)
        return entrypoint_path

    def _get_port(self):
        return self.config.get("port", 8181)
        
    def build_spec(self, spec, env, template):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        start_file = self.config.get("startFile", "__CODE_STUDIO_VERSIONED__/streamlit/app.py")
        settings_path = self.config.get("settingsPath", "__CODE_STUDIO_VERSIONED__/streamlit")
        enable_xsrf = self.config.get("enableXSRF", False)

        # replace the lib locations in settings_path and open_in_path
        replacer = LibLocationPathReplacer(spec)
        start_file = replacer.replace_variable_by_path(start_file)
        settings_path = replacer.replace_variable_by_path(settings_path)

        # get code env stuff

        default_packages = "streamlit==1.9.2 altair==4.2.2 urllib3<2"
        generate_codenv, pyenv_path = generate_python_codenv("STREAMLIT", self.config, template, default_packages, "/opt/dataiku/python-code-envs/pyenv-streamlit", "python3.9", env.get("globalCodeEnvsExtraSettings"))

        # add the entrypoint script in the buildir
        entrypoint_script = """
#! /bin/bash

USER=dataiku
HOME=/home/dataiku

# CREATE THE START APP.PY IF NEEDED

if [ ! -f {start_file} ]; then
    mkdir -p $(dirname {start_file})
    cat << 'EOF' > {start_file}
import streamlit as st
import pandas as pd
import numpy as np
import altair as alt
import dataiku

st.title('Hello Streamlit!')

# This loads dummy data into a dataframe
df = pd.DataFrame({{
    'Trial A': np.random.normal(0, 0.8, 1000),
    'Trial B': np.random.normal(-2, 1, 1000),
    'Trial C': np.random.normal(3, 2, 1000)
}})

# Uncomment the following to read your own dataset
#dataset = dataiku.Dataset("YOUR_DATASET_NAME_HERE")
#df = dataset.get_dataframe()

c = alt.Chart(df).transform_fold(
    ['Trial A', 'Trial B', 'Trial C'], 
    as_=['Experiment', 'Measurement']
).mark_bar(
    opacity=0.3, 
    binSpacing=0
).encode(
    alt.X('Measurement:Q', bin=alt.Bin(maxbins=100)), 
    alt.Y('count()', stack=None), 
    alt.Color('Experiment:N')
)

st.altair_chart(c, use_container_width=True)
EOF
fi

# CREATE THE STREAMLIT CONFIG.TOML

if [ $DKU_CODE_STUDIO_IS_PUBLIC_PORT_{port} = "1" ]; then
    export BIND_ADDR=0.0.0.0
else
    export BIND_ADDR=127.0.0.1
fi

mkdir -p /home/dataiku/.streamlit
export BASE_URL=$(eval echo "$DKU_CODE_STUDIO_BROWSER_PATH_{port}")

cat << 'EOF' | envsubst > /home/dataiku/.streamlit/config.toml
# Autogenerated by Dataiku for Code Studio - DO NOT EDIT
# If you need to change streamlit config, either do it from the Code Studio template , or use the "Settings folder" setting, 
# which store a per-app (persisted) config, by default in /home/dataiku/workspace/code_studio-versioned/streamlit/.streamlit/config.toml

[logger]
level = "info"

[server]
address = "$BIND_ADDR"
port = {port}
enableCORS = false
enableXsrfProtection = {enable_xsrf}
runOnSave = true
baseUrlPath = "$BASE_URL"

[browser]
gatherUsageStats = false
EOF

# CREATE THE STREAMLIT CREDENTIALS IF NEEDED

if [ ! -f /home/dataiku/.streamlit/credentials.toml ]; then
    cat << 'EOF' > /home/dataiku/.streamlit/credentials.toml
[general]
email = ""
EOF
fi

# CREATE THE USER_CUSTOMIZABLE CONFIG.TOML
if [ ! -f {settings_path}/.streamlit/config.toml ]; then
    mkdir -p {settings_path}/.streamlit
    cat << 'EOF' > {settings_path}/.streamlit/config.toml
# Set any custom config here. Will take precedence over default config from /home/dataiku/.streamlit/config.toml
# Changes will need a restart of the Code Studio to take effect.

[global]
# By default, Streamlit checks if the Python watchdog module is available and, if not, prints a warning asking for you to install it. The watchdog module is not required, but highly recommended. It improves Streamlit's ability to detect changes to files in your filesystem.
# If you'd like to turn off this warning, set this to True.
# Default: false
# disableWatchdogWarning = false

# DataFrame serialization.
# Acceptable values: - 'legacy': Serialize DataFrames using Streamlit's custom format. Slow but battle-tested. - 'arrow': Serialize DataFrames using Apache Arrow. Much faster and versatile.
# Default: "arrow"
# dataFrameSerialization = "arrow"


[runner]

# Allows you to type a variable or string by itself in a single line of Python code to write it to the app.
# Default: true
#magicEnabled = true

# Install a Python tracer to allow you to stop or pause your script at any point and introspect it. As a side-effect, this slows down your script's execution.
# Default: false
#installTracer = false

# Sets the MPLBACKEND environment variable to Agg inside Streamlit to prevent Python crashing.
# Default: true
#fixMatplotlib = true

# Run the Python Garbage Collector after each script execution. This can help avoid excess memory use in Streamlit apps, but could introduce delay in rerunning the app script for high-memory-use applications.
# Default: true
#postScriptGC = true

# Handle script rerun requests immediately, rather than waiting for script execution to reach a yield point. This makes Streamlit much more responsive to user interaction, but it can lead to race conditions in apps that mutate session_state data outside of explicit session_state assignment statements.
# Default: true
#fastReruns = true

# Raise an exception after adding unserializable data to Session State. Some execution environments may require serializing all data in Session State, so it may be useful to detect incompatibility during development, or when the execution environment will stop supporting it in the future.
# Default: false
#enforceSerializableSessionState = false


[server]
# Server settings are configured by Dataiku for Code Studio.
# Do not override them.


[browser]
# Browser settings are configured by Dataiku for Code Studio.
# Do not override them.


[logger]

# Level of logging: 'error', 'warning', 'info', or 'debug'.
# Default: 'info'
# level = "info"

# String format for logging messages. If logger.datetimeFormat is set, logger messages will default to `%(asctime)s.%(msecs)03d %(message)s`. See [Python's documentation](https://docs.python.org/2.6/library/logging.html#formatter-objects) for available attributes.
# Default: "%(asctime)s %(message)s"
# messageFormat = "%(asctime)s %(message)s"


[client]

# Whether to enable st.cache.
# Default: true
# caching = true

# Controls whether uncaught app exceptions and deprecation warnings are displayed in the browser. By default, this is set to True and Streamlit displays app exceptions and associated tracebacks, and deprecation warnings, in the browser.
# If set to False, an exception or deprecation warning will result in a generic message being shown in the browser, and exceptions, tracebacks, and deprecation warnings will be printed to the console only.
# Default: true
# showErrorDetails = true


[mapbox]

# Configure Streamlit to use a custom Mapbox token for elements like st.pydeck_chart and st.map. To get a token for yourself, create an account at https://mapbox.com. It's free (for moderate usage levels)!
# Default: ""
# token = ""


[deprecation]

# Set to false to disable the deprecation warning for the file uploader encoding.
# Default: true
# showfileUploaderEncoding = true

# Set to false to disable the deprecation warning for using the global pyplot instance.
# Default: true
# showPyplotGlobalUse = true


[theme]

# The preset Streamlit theme that your custom theme inherits from. One of "light" or "dark".
# base =

# Primary accent color for interactive elements.
# primaryColor =

# Background color for the main content area.
# backgroundColor =

# Background color used for the sidebar and most interactive widgets.
# secondaryBackgroundColor =

# Color used for almost all text.
# textColor =

# Font family for all text in the app, except code blocks. One of "sans serif", "serif", or "monospace".
# font =
EOF
fi

# START STREAMLIT

# cd'ing in order to pick the custom config
cd {settings_path}
LC_ALL=en_US.utf8 {pyenv_path}/bin/streamlit run {start_file}
""".format(pyenv_path=pyenv_path, start_file=start_file, settings_path=settings_path, port=port, enable_xsrf="true" if enable_xsrf else "false")
        with open(os.path.join(env["buildDir"], self._ENTRYPOINT_FILE), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))
            
        # the dockerfile addition
        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### STREAMLIT BLOCK #####

USER root
WORKDIR /opt/dataiku

RUN yum install -y gettext

{generate_codenv}

# entrypoint.sh
COPY {entrypoint_file} {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}

# USER dataiku
USER {uid_gid}
WORKDIR /home/dataiku
""".format(pyenv_path=pyenv_path, start_file=start_file, port=port,
           entrypoint_path=entrypoint_path, entrypoint_file=self._ENTRYPOINT_FILE,
           generate_codenv=generate_codenv,
           uid_gid=get_dataiku_user_uid_gid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) + "${baseUrlPort" + str(port) + "}/index.html" # baseUrlPort should be replaced by actual url in BlockBasedCodeStudioMeta/buildYaml
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {
            "label": "Streamlit", 
            "proxiedUrlSuffix": "$uri$is_args$args",
            "exposeHtml": True, 
            "port": port,
            "readinessProbeUrl": readiness_probe_url
        }
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        return spec
        
    def build_creation(self, spec, env):
        return spec
        
