import atexit
import json
import logging
import os
import re
import shutil
import sys
import tarfile
import tempfile
import threading

from dataiku.base import remoterun
from dataiku.base.utils import tar_extractall
from dataiku.core import intercom

logger = logging.getLogger(__name__)

TAR_BUFFER_SIZE = 10 * 16 * 1024

plugins_tmp_path = ""


def _get_plugins_tmp_path():
    global plugins_tmp_path
    if not plugins_tmp_path:
        plugins_tmp_path = os.path.join(tempfile.mkdtemp(), "downloaded_plugin_files")
        if not os.path.exists(plugins_tmp_path):
            os.makedirs(plugins_tmp_path)
        logger.info(u"Created plugins temporary path: {}".format(plugins_tmp_path))
    return plugins_tmp_path


def _cleanup_plugins_tmp_path():
    global plugins_tmp_path
    if plugins_tmp_path and os.path.exists(plugins_tmp_path):
        try:
            shutil.rmtree(plugins_tmp_path)
            logger.info(u"Cleaned up plugins temporary path: {}".format(plugins_tmp_path))
        except:
            logger.exception(u"Error cleaning up plugins temporary path: {}".format(plugins_tmp_path))


# Register the cleanup function to be called at exit
atexit.register(_cleanup_plugins_tmp_path)


def _extract_stream(raw_response, plugin_id):
    """
    Takes a response from the backend with the plugin data and extracts it into the local plugins folder

    Since we are calling this once per model, we might be downloading the same plugin more than once.
    We are merging the extracted files into the plugin id folder if it already existed.
    :param raw_response:
    :param str plugin_id:
    :return: the path to the extracted plugin folder
    """

    plugin_path = os.path.join(_get_plugins_tmp_path(), plugin_id)
    with tarfile.open(fileobj=raw_response, mode='r|', bufsize=TAR_BUFFER_SIZE) as tar:
        tar_extractall(tar, plugin_path)
    return plugin_path


def _prepare_env_ml_libs(plugin_path, plugin_id):
    if not os.path.isdir(plugin_path):
        return

    if plugin_path not in sys.path:
        logger.info(u"Adding '{}' to sys path".format(plugin_path))
        sys.path.append(plugin_path)

    # Add python-lib directly to path to be consistent with other usages of plugin
    lib_python_dir = os.path.join(plugin_path, "dku-ml-plugins", plugin_id, "python-lib")
    if os.path.isdir(lib_python_dir) and lib_python_dir not in sys.path:
        logger.info(u"Adding '{}' to sys path".format(lib_python_dir))
        sys.path.append(lib_python_dir)
    # Add resource dir to env if it exists
    resource_dir = os.path.join(plugin_path, "dku-ml-plugins", plugin_id, "resource")
    if os.path.isdir(resource_dir):
        remoterun.set_dku_env_var_and_sys_env_var("DKU_CUSTOM_ML_RESOURCE_FOLDER_{}".format(plugin_id),
                                                  resource_dir)


plugin_pattern = re.compile(r"dku-ml-plugins.([^.]+).python-prediction-algos.([^.]+).algo")


def _get_ml_plugin_files(plugin_id, element_id):
    return intercom.jek_or_backend_post_call(
        "plugins/get-ml-plugin-files/",
        stream=True,
        data={"pluginId": plugin_id, "elementId": element_id})


def _get_ml_plugin_version(plugin_id):
    return intercom.jek_or_backend_post_call(
        "plugins/get-plugin-version/",
        data={"pluginId": plugin_id})


def _get_ml_used_plugins(modeling_params):
    return intercom.jek_or_backend_post_call(
        "plugins/get-used-plugins/",
        data={"modeling_params": json.dumps(modeling_params)})

def prepare_for_plugin_from_params(modeling_params):
    used_plugins = _get_ml_used_plugins(modeling_params).json()
    for plugin_id, plugin_info in used_plugins.items():
        for element_id in plugin_info.get("customAlgos", []):
            prepare_for_plugin(plugin_id, element_id)


_preparation_lock = threading.Lock()


def prepare_for_plugin(plugin_id, element_id):
    """
    :param str plugin_id: the plugin name
    :param str element_id: the model name
    """
    with _preparation_lock:
        files_stream = _get_ml_plugin_files(plugin_id, element_id)
        if files_stream.status_code == 200:
            plugin_path = _extract_stream(files_stream.raw, plugin_id)
            _prepare_env_ml_libs(plugin_path, plugin_id)
        else:
            raise RuntimeError(u"Error fetching ml pluginID: {}, elementId: {}, HTTP status {}: {}".format(
                plugin_id, element_id,
                files_stream.status_code,
                files_stream.text))
