/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.dao;

import com.dataiku.dip.DKUApp;
import com.dataiku.dip.aigenerations.SqlAssistantConversation;
import com.dataiku.dip.utils.DKUFileUtils;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.JSON;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class SqlAssistantConversationsDAO {
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.sqlassistant.dao");
    private static final String AI_ASSISTANTS_ROOT = "ai-assistants";
    private static final String PROJECTS_FOLDER = "projects";
    private static final String NOTEBOOKS_FOLDER = "notebooks";
    private static final String SQL_FOLDER = "sql";
    private static final String CELLS_FOLDER = "cells";
    private static final String USERS_FOLDER = "users";
    private static final int DEFAULT_CACHE_MAX_SIZE = 1000;
    private static final int DEFAULT_CACHE_EXPIRE_AFTER_ACCESS_SECONDS = 86400;
    private static final int DEFAULT_FLUSH_INTERVAL_SECONDS = 30;
    private static final Duration SHUTDOWN_TIMEOUT = Duration.ofSeconds(15L);
    private final File baseDir;
    private final ScheduledExecutorService scheduler;
    private final LoadingCache<ConversationCacheKey, SqlAssistantConversation.QueryConversation> cache;
    private final Set<ConversationCacheKey> dirtyKeys = ConcurrentHashMap.newKeySet();

    @Autowired
    public SqlAssistantConversationsDAO() {
        this(DKUApp.getFile((String)AI_ASSISTANTS_ROOT), DKUApp.getParams().getIntParam("dku.sql.assistant.cache.maxSize", Integer.valueOf(1000)), Duration.ofSeconds(DKUApp.getParams().getIntParam("dku.sql.assistant.cache.expireAfterAccessSeconds", Integer.valueOf(86400))), Duration.ofSeconds(DKUApp.getParams().getIntParam("dku.sql.assistant.cache.flushIntervalSeconds", Integer.valueOf(30))), Ticker.systemTicker());
    }

    @VisibleForTesting
    SqlAssistantConversationsDAO(File baseDir, int maxSize, Duration expireAfterAccess, Duration flushInterval, Ticker ticker) {
        logger.info((Object)"Initializing SQL Assistant conversations DAO");
        this.baseDir = baseDir;
        try {
            DKUFileUtils.mkdirs((File)baseDir);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to create base directory: " + String.valueOf(baseDir), e);
        }
        RemovalListener removalListener = notification -> {
            if (notification.wasEvicted()) {
                ConversationCacheKey key = (ConversationCacheKey)notification.getKey();
                SqlAssistantConversation.QueryConversation conversation = (SqlAssistantConversation.QueryConversation)notification.getValue();
                if (this.dirtyKeys.remove(key) && conversation != null) {
                    this.writeToDisk(key, conversation);
                }
            }
        };
        this.cache = CacheBuilder.newBuilder().maximumSize((long)maxSize).expireAfterAccess(expireAfterAccess).ticker(ticker).removalListener(removalListener).build(CacheLoader.from(this::loadFromDisk));
        this.scheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("sql-assistant-cache-flusher").setDaemon(true).build());
        this.scheduler.scheduleAtFixedRate(this::flushDirtyEntries, flushInterval.toSeconds(), flushInterval.toSeconds(), TimeUnit.SECONDS);
        logger.debugV("SQL Assistant conversations DAO initialized (cacheMaxSize=%d, expireAfterAccess=%s, flushInterval=%s)", new Object[]{maxSize, expireAfterAccess, flushInterval});
    }

    @PreDestroy
    public void shutdown() {
        logger.info((Object)"Shutting down SQL Assistant conversations DAO");
        if (this.scheduler != null) {
            this.scheduler.shutdown();
            try {
                if (!this.scheduler.awaitTermination(SHUTDOWN_TIMEOUT.toSeconds(), TimeUnit.SECONDS)) {
                    this.logPendingEntriesWarning();
                    this.scheduler.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.logPendingEntriesWarning();
                this.scheduler.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        this.flushDirtyEntries();
        logger.info((Object)"SQL Assistant conversations DAO shutdown complete");
    }

    private void logPendingEntriesWarning() {
        if (!this.dirtyKeys.isEmpty()) {
            logger.warn((Object)("Forcing shutdown with " + this.dirtyKeys.size() + " pending dirty entries"));
        }
    }

    public SqlAssistantConversation.QueryConversation getConversation(String projectKey, String notebookId, String cellId, String userId) {
        this.validateParams(projectKey, notebookId, cellId, userId);
        ConversationCacheKey key = new ConversationCacheKey(projectKey, notebookId, cellId, userId);
        try {
            return (SqlAssistantConversation.QueryConversation)this.cache.get((Object)key);
        }
        catch (ExecutionException e) {
            logger.warn((Object)("Failed to load conversation from cache: " + String.valueOf(key)), (Throwable)e);
            return new SqlAssistantConversation.QueryConversation();
        }
    }

    public void saveConversation(String projectKey, String notebookId, String cellId, String userId, SqlAssistantConversation.QueryConversation conversation) {
        this.validateParams(projectKey, notebookId, cellId, userId);
        conversation.lastUpdated = System.currentTimeMillis();
        ConversationCacheKey key = new ConversationCacheKey(projectKey, notebookId, cellId, userId);
        this.cache.put((Object)key, (Object)conversation);
        this.dirtyKeys.add(key);
    }

    public void deleteUserConversation(String projectKey, String notebookId, String cellId, String userId) {
        this.validateParams(projectKey, notebookId, cellId, userId);
        logger.info((Object)("Deleting SQL Assistant conversation for cell " + cellId + " and user " + userId));
        ConversationCacheKey key = new ConversationCacheKey(projectKey, notebookId, cellId, userId);
        this.cache.invalidate((Object)key);
        this.dirtyKeys.remove(key);
        File file = this.getUserConversationFile(key);
        if (file.exists()) {
            try {
                DKUFileUtils.delete((File)file);
            }
            catch (IOException e) {
                logger.warn((Object)("Failed to delete conversation from disk: " + String.valueOf(key)), (Throwable)e);
            }
        }
    }

    public void deleteCellConversations(String projectKey, String notebookId, String cellId) {
        this.validateParams(projectKey, notebookId, cellId);
        logger.info((Object)("Deleting SQL Assistant conversations for cell " + cellId));
        this.invalidateCellEntries(projectKey, notebookId, cellId);
        File cellDir = this.getCellDir(projectKey, notebookId, cellId);
        if (cellDir.exists()) {
            try {
                DKUFileUtils.deleteDirectory((File)cellDir);
            }
            catch (IOException e) {
                logger.warn((Object)("Failed to delete cell directory: " + String.valueOf(cellDir)), (Throwable)e);
            }
        }
    }

    public void deleteNotebookConversations(String projectKey, String notebookId) {
        this.validateParams(projectKey, notebookId);
        logger.info((Object)("Deleting all SQL Assistant conversations for notebook " + notebookId));
        this.invalidateNotebookEntries(projectKey, notebookId);
        File notebookDir = this.getNotebookDir(projectKey, notebookId);
        if (notebookDir.exists()) {
            try {
                DKUFileUtils.deleteDirectory((File)notebookDir);
            }
            catch (IOException e) {
                logger.warn((Object)("Failed to delete notebook directory: " + String.valueOf(notebookDir)), (Throwable)e);
            }
        }
    }

    @VisibleForTesting
    void cleanUp() {
        this.cache.cleanUp();
    }

    @VisibleForTesting
    synchronized void flushDirtyEntries() {
        for (ConversationCacheKey key : List.copyOf(this.dirtyKeys)) {
            this.flushEntry(key);
        }
    }

    private void flushEntry(ConversationCacheKey key) {
        if (this.dirtyKeys.remove(key)) {
            this.cache.asMap().computeIfPresent(key, (k, conversation) -> {
                this.writeToDisk((ConversationCacheKey)k, (SqlAssistantConversation.QueryConversation)conversation);
                return conversation;
            });
        }
    }

    private void invalidateCellEntries(String projectKey, String notebookId, String cellId) {
        this.cache.asMap().keySet().removeIf(key -> key.projectKey.equals(projectKey) && key.notebookId.equals(notebookId) && key.cellId.equals(cellId));
        this.dirtyKeys.removeIf(key -> key.projectKey.equals(projectKey) && key.notebookId.equals(notebookId) && key.cellId.equals(cellId));
    }

    private void invalidateNotebookEntries(String projectKey, String notebookId) {
        this.cache.asMap().keySet().removeIf(key -> key.projectKey.equals(projectKey) && key.notebookId.equals(notebookId));
        this.dirtyKeys.removeIf(key -> key.projectKey.equals(projectKey) && key.notebookId.equals(notebookId));
    }

    private void writeToDisk(ConversationCacheKey key, SqlAssistantConversation.QueryConversation conversation) {
        File targetFile = this.getUserConversationFile(key);
        try {
            DKUFileUtils.mkdirsParent((File)targetFile);
            JSON.prettyToFile((Object)conversation, (File)targetFile);
        }
        catch (IOException e) {
            logger.warn((Object)("Failed to persist conversation: " + String.valueOf(key)), (Throwable)e);
        }
    }

    private SqlAssistantConversation.QueryConversation loadFromDisk(ConversationCacheKey key) {
        File file = this.getUserConversationFile(key);
        if (!file.exists()) {
            return new SqlAssistantConversation.QueryConversation();
        }
        try {
            return (SqlAssistantConversation.QueryConversation)JSON.parseFile((File)file, SqlAssistantConversation.QueryConversation.class);
        }
        catch (IOException e) {
            logger.warn((Object)("Failed to load conversation from disk: " + String.valueOf(key)), (Throwable)e);
            return new SqlAssistantConversation.QueryConversation();
        }
    }

    private File getUserConversationFile(ConversationCacheKey key) {
        return new File(this.getCellDir(key.projectKey, key.notebookId, key.cellId), USERS_FOLDER + File.separator + key.userId + ".json");
    }

    private File getCellDir(String projectKey, String notebookId, String cellId) {
        return new File(this.getNotebookDir(projectKey, notebookId), CELLS_FOLDER + File.separator + cellId);
    }

    private File getNotebookDir(String projectKey, String notebookId) {
        return new File(this.baseDir, PROJECTS_FOLDER + File.separator + projectKey + File.separator + NOTEBOOKS_FOLDER + File.separator + SQL_FOLDER + File.separator + notebookId);
    }

    private void validateParams(String projectKey, String notebookId, String cellId, String userId) {
        this.validateParams(projectKey, notebookId, cellId);
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)userId), (Object)"User id is not specified");
    }

    private void validateParams(String projectKey, String notebookId, String cellId) {
        this.validateParams(projectKey, notebookId);
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)cellId), (Object)"Cell id is not specified");
    }

    private void validateParams(String projectKey, String notebookId) {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)projectKey), (Object)"Project key is not specified");
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)notebookId), (Object)"Notebook id is not specified");
    }

    private record ConversationCacheKey(String projectKey, String notebookId, String cellId, String userId) {
    }
}

