/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.sanitycheck.detectors;

import com.dataiku.dip.DKUApp;
import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.sanitycheck.SanityCheckDetectorBase;
import com.dataiku.dip.sanitycheck.SanityCheckInfoMessages;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.DKUtils;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.SystemUtils;
import org.springframework.stereotype.Component;

@Component
public class DiskDetector
implements SanityCheckDetectorBase {
    public static final DKULogger logger = DKULogger.getLogger((String)"dip.sanitycheck.diskdetector");

    @Override
    public List<InfoMessage.MessageCode> getCodes() {
        return Arrays.asList(Codes.values());
    }

    @Override
    public SanityCheckInfoMessages runAnalysis(Set<String> exclusionList) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (exclusionList.containsAll(Arrays.stream(Codes.values()).map(InfoMessage.MessageCode::getCode).collect(Collectors.toList()))) {
            return messages;
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            return messages;
        }
        try {
            DfOutputParser dssTmpDfOutputParser = this.getDfOutput(DKUApp.getFile("tmp"));
            DfOutputParser dssHomeDfOutputParser = this.getDfOutput(DKUApp.getBaseFolderF());
            String tmpDirPath = System.getenv().getOrDefault("TMPDIR", "/tmp");
            DfOutputParser tmpDfOutputParser = this.getDfOutput(new File(tmpDirPath));
            MountOutputParser mountOutput = this.getMountOutput();
            if (!exclusionList.contains(Codes.ERR_MISC_DISK_FULL.getCode())) {
                messages.mergeFrom(this.checkDiskUsage(dssTmpDfOutputParser));
                messages.mergeFrom(this.checkDiskUsage(dssHomeDfOutputParser));
            }
            if (!exclusionList.contains(Codes.WARN_MISC_DISK_MOUNT_TYPE.getCode())) {
                messages.mergeFrom(this.checkFsType(dssHomeDfOutputParser, mountOutput));
            }
            if (!exclusionList.contains(Codes.WARN_MISC_DISK_NOEXEC_FLAG.getCode())) {
                messages.mergeFrom(this.checkNoexecFlag(tmpDfOutputParser, mountOutput));
            }
            if (SystemUtils.IS_OS_LINUX && !exclusionList.contains(Codes.WARN_MISC_DISK_ROTATIONAL.getCode())) {
                LsblkOutputParser lsblkOutput = this.getLsblkOutput();
                messages.mergeFrom(this.checkDSSHomeDiskType(dssHomeDfOutputParser.fsDisk, lsblkOutput));
            }
        }
        catch (Exception e) {
            logger.error((Object)"Disk sanity check failed", (Throwable)e);
            messages.addMessage(this.createFatalMessage(e));
        }
        return messages;
    }

    private SanityCheckInfoMessages checkNoexecFlag(DfOutputParser dfOutput, MountOutputParser mountOutput) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!mountOutput.mounts.containsKey(dfOutput.mountPoint)) {
            throw new IllegalStateException(String.format("Could not find %s in mount output", dfOutput.mountPoint));
        }
        MountOutputParser.Mount mount = mountOutput.mounts.get(dfOutput.mountPoint);
        if (mount.flags.contains("noexec")) {
            messages.withWarningV(Codes.WARN_MISC_DISK_NOEXEC_FLAG, "The 'noexec' flag has been set on the '%s' partition. It may cause issues, especially with code environments.", new Object[]{mount.mount});
        }
        return messages;
    }

    private InfoMessage.InfoMessages checkFsType(DfOutputParser dfOutput, MountOutputParser mountOutput) {
        InfoMessage.InfoMessages infoMessages = new InfoMessage.InfoMessages();
        if (SystemUtils.IS_OS_LINUX) {
            if (!mountOutput.mounts.containsKey(dfOutput.mountPoint)) {
                throw new IllegalStateException(String.format("Could not find %s in mount output", dfOutput.mountPoint));
            }
            MountOutputParser.Mount mount = mountOutput.mounts.get(dfOutput.mountPoint);
            if (!Arrays.asList("ext4", "xfs").contains(mount.fsType.toLowerCase())) {
                infoMessages.withWarningV((InfoMessage.MessageCode)Codes.WARN_MISC_DISK_MOUNT_TYPE, "The type of the filesystem '%s' mounted on '%s' is '%s'. The recommended filesystems are ext4 and XFS.", new Object[]{dfOutput.fsDisk, dfOutput.mountPoint, mount.fsType});
            }
        }
        return infoMessages;
    }

    private InfoMessage.InfoMessages checkDiskUsage(DfOutputParser dfOutputParser) {
        InfoMessage.InfoMessages infoMessages = new InfoMessage.InfoMessages();
        if (dfOutputParser.capacity >= DKUApp.getParams().getIntParam("dku.sanitycheck.diskusage.threshold", Integer.valueOf(90))) {
            infoMessages.withErrorV((InfoMessage.MessageCode)Codes.ERR_MISC_DISK_FULL, "Filesystem '%s' mounted on '%s' is filled at %s%%, %s left.", new Object[]{dfOutputParser.fsDisk, dfOutputParser.mountPoint, dfOutputParser.capacity, dfOutputParser.availableSpace});
        }
        return infoMessages;
    }

    private InfoMessage.InfoMessages checkDSSHomeDiskType(String fsPath, LsblkOutputParser lsblkOutputParser) {
        SanityCheckInfoMessages infoMessages = new SanityCheckInfoMessages();
        LsblkOutputParser.Drive drive = lsblkOutputParser.drives.get(fsPath);
        if (drive != null && drive.isRotational) {
            infoMessages.withTrustedHTMLV(InfoMessage.Severity.WARNING, Codes.WARN_MISC_DISK_ROTATIONAL, "<p>The hard drive '%s'%s on which your DATA_DIR is located seems to be a rotational hard drive. Performance will be impacted. It is highly recommended to run DSS on SSD drives.</p><p><i class=\"icon-info-sign\"></i> Note that virtualization (such as RAID) may show a rotational hard drive while the underlying drive is a SSD.</p>", drive.name, StringUtils.isNotEmpty((String)drive.vendor) || StringUtils.isNotEmpty((String)drive.model) ? String.format(" (vendor: %s, model: %s)", StringUtils.defaultIfEmpty((String)drive.vendor, (String)"unknown"), StringUtils.defaultIfEmpty((String)drive.model, (String)"unknown")) : "");
        }
        return infoMessages;
    }

    private DfOutputParser getDfOutput(File path) throws Exception {
        DfOutputParser dfOutputParser = new DfOutputParser();
        CommandErrorConsumer commandErrorConsumer = new CommandErrorConsumer();
        int dfReturnCode = new DKUtils.ExecBuilder().withArgs(Arrays.asList("df", "-P", "-h", path.getAbsolutePath())).withOutputConsumer((DKUtils.ExecSubscription)dfOutputParser).withErrorConsumer((DKUtils.ExecSubscription)commandErrorConsumer).exec();
        if (StringUtils.isNotBlank((String)commandErrorConsumer.failureMessage)) {
            throw new IllegalStateException(commandErrorConsumer.failureMessage);
        }
        if (StringUtils.isNotBlank((String)dfOutputParser.failureMessage)) {
            throw new IllegalStateException(dfOutputParser.failureMessage);
        }
        if (dfReturnCode != 0) {
            throw new IllegalStateException("df returned a non zero status code");
        }
        return dfOutputParser;
    }

    private MountOutputParser getMountOutput() throws InterruptedException, IOException {
        MountOutputParser mountOutputParser = new MountOutputParser();
        CommandErrorConsumer commandErrorConsumer = new CommandErrorConsumer();
        int mountReturnCode = new DKUtils.ExecBuilder().withArgs(Collections.singletonList("mount")).withOutputConsumer((DKUtils.ExecSubscription)mountOutputParser).withErrorConsumer((DKUtils.ExecSubscription)commandErrorConsumer).exec();
        if (StringUtils.isNotBlank((String)commandErrorConsumer.failureMessage)) {
            throw new IllegalStateException(commandErrorConsumer.failureMessage);
        }
        if (StringUtils.isNotBlank((String)mountOutputParser.failureMessage)) {
            throw new IllegalStateException(mountOutputParser.failureMessage);
        }
        if (mountReturnCode != 0) {
            throw new IllegalStateException("mount returned a non zero status code");
        }
        return mountOutputParser;
    }

    private LsblkOutputParser getLsblkOutput() throws Exception {
        LsblkOutputParser lsblkOutputParser = new LsblkOutputParser();
        CommandErrorConsumer commandErrorConsumer = new CommandErrorConsumer();
        int dfReturnCode = new DKUtils.ExecBuilder().withArgs(Arrays.asList("lsblk", "-p", "-P", "-o", "NAME,ROTA,MODEL,VENDOR")).withOutputConsumer((DKUtils.ExecSubscription)lsblkOutputParser).withErrorConsumer((DKUtils.ExecSubscription)commandErrorConsumer).exec();
        if (StringUtils.isNotBlank((String)commandErrorConsumer.failureMessage)) {
            throw new IllegalStateException(commandErrorConsumer.failureMessage);
        }
        if (StringUtils.isNotBlank((String)lsblkOutputParser.failureMessage)) {
            throw new IllegalStateException(lsblkOutputParser.failureMessage);
        }
        if (dfReturnCode != 0) {
            throw new IllegalStateException("df returned a non zero status code");
        }
        return lsblkOutputParser;
    }

    public static enum Codes implements InfoMessage.MessageCode
    {
        ERR_MISC_DISK_FULL("Disk - Almost full", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_MISC_DISK_NOEXEC_FLAG("Disk - noexec flag", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_MISC_DISK_MOUNT_TYPE("Disk - non recommended filesystem type", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING),
        WARN_MISC_DISK_ROTATIONAL("Disk - legacy rotational hard drives", InfoMessage.FixabilityCategory.ADMIN_TROUBLESHOOTING);

        private final String title;
        private final InfoMessage.FixabilityCategory fixability;

        private Codes(String title, InfoMessage.FixabilityCategory fixability) {
            this.title = title;
            this.fixability = fixability;
        }

        public String getCode() {
            return this.name();
        }

        public String getCodeTitle() {
            return this.title;
        }

        public InfoMessage.FixabilityCategory getFixability() {
            return this.fixability;
        }
    }

    static class DfOutputParser
    implements DKUtils.LineSubscription {
        private static final Pattern dfOutputPattern = Pattern.compile("^(?<fsdisk>[^ ]+) +([^ ]+ +){2}(?<available>[^ ]+) +(?<capacity>\\d+)% +(?<mount>[^ ]+).*$");
        int lineNumber = 0;
        public String fsDisk;
        public int capacity;
        public String availableSpace;
        public String mountPoint;
        public String failureMessage;

        DfOutputParser() {
        }

        public void handle(String line, boolean replace) throws IOException {
            if (this.lineNumber == 0) {
                ++this.lineNumber;
            } else if (this.lineNumber == 1) {
                Matcher matcher = dfOutputPattern.matcher(line);
                if (matcher.matches()) {
                    this.fsDisk = matcher.group("fsdisk");
                    this.capacity = Integer.parseInt(matcher.group("capacity"));
                    this.availableSpace = matcher.group("available");
                    this.mountPoint = matcher.group("mount");
                } else {
                    this.failureMessage = "Could not parse df output: " + line;
                }
            }
        }

        public void close() throws IOException {
        }
    }

    static class MountOutputParser
    implements DKUtils.LineSubscription {
        private static final Pattern mountOutputPattern = Pattern.compile("^(?<fsDisk>.+) on (?<mount>.+?) (type (?<fsType>[^ ]+) )?\\((?<flags>.+)\\)$");
        public Map<String, Mount> mounts = new HashMap<String, Mount>();
        public String failureMessage;

        MountOutputParser() {
        }

        public void handle(String line, boolean replace) throws IOException {
            Matcher matcher = mountOutputPattern.matcher(line);
            if (matcher.matches()) {
                Mount mount = new Mount();
                mount.fsDisk = matcher.group("fsDisk");
                mount.mount = matcher.group("mount");
                mount.fsType = null;
                try {
                    mount.fsType = matcher.group("fsType");
                }
                catch (IllegalArgumentException illegalArgumentException) {
                    // empty catch block
                }
                mount.flags = Arrays.stream(matcher.group("flags").split(",")).map(String::trim).collect(Collectors.toList());
                this.mounts.put(mount.mount, mount);
            } else {
                this.failureMessage = "Could not parse mount output: " + line;
            }
        }

        public void close() throws IOException {
        }

        static class Mount {
            public String fsDisk;
            public String mount;
            public String fsType;
            public List<String> flags;

            Mount() {
            }
        }
    }

    static class LsblkOutputParser
    implements DKUtils.LineSubscription {
        private static final Pattern lsblkOutputPattern = Pattern.compile("^NAME=\"(?<name>.*)\" ROTA=\"(?<rota>\\d)\" MODEL=\"(?<model>.*)\" VENDOR=\"(?<vendor>.*)\"$");
        public Map<String, Drive> drives = new HashMap<String, Drive>();
        public String failureMessage;

        LsblkOutputParser() {
        }

        public void handle(String line, boolean replace) throws IOException {
            Matcher matcher = lsblkOutputPattern.matcher(line);
            if (matcher.matches()) {
                Drive drive = new Drive();
                drive.name = matcher.group("name");
                drive.isRotational = "1".equals(matcher.group("rota"));
                drive.model = matcher.group("model").trim();
                drive.vendor = matcher.group("vendor").trim();
                this.drives.put(drive.name, drive);
            } else {
                this.failureMessage = "Could not parse lsblk output: " + line;
            }
        }

        public void close() throws IOException {
        }

        static class Drive {
            public String name;
            public boolean isRotational;
            public String model;
            public String vendor;

            Drive() {
            }
        }
    }

    static class CommandErrorConsumer
    implements DKUtils.LineSubscription {
        public String failureMessage;

        CommandErrorConsumer() {
        }

        public void handle(String line, boolean replace) throws IOException {
            this.failureMessage = line;
        }

        public void close() throws IOException {
        }
    }
}

