// @ts-check
(function () {
    'use strict';
    angular.module('dataiku.common.datastructures', [])
        .factory('LinkedList', function () {
            /**
             * Linked list node.
             * @template T
             * @class LinkedListNode
             */
            class LinkedListNode {
                /**
                 * Creates an instance of LinkedListNode.
                 * @param {T} element
                 * @memberof LinkedListNode
                 */
                constructor(element) {
                    this.element = element;
                    /** @type {LinkedListNode | undefined} */
                    this.next = undefined;
                    /** @type {LinkedListNode | undefined} */
                    this.prev = undefined;
                }
            }

            /**
             * Linked list data structure to be used to as a queue or a stack.
             * @template T
             * @class LinkedList
             */
            class LinkedList {
                /**
                 * Creates an instance of LinkedList.
                 * @param {T [] | undefined} elements
                 * @memberof LinkedList
                 */
                constructor(elements) {
                    /** @type {LinkedListNode | undefined} */
                    this.head = undefined;
                    /** @type {LinkedListNode | undefined} */
                    this.tail = undefined;
                    /** @type {number} */
                    this.length = 0;

                    if (elements !== undefined) {
                        elements.forEach(element => this.push(element));
                    }
                }

                /**
                 * Removes the last element from the linked list and returns it, with O(1) complexity. 
                 * If the linked list is empty, undefined is returned and the linked list is not modified.
                 * @returns {T} the last element from the linked list
                 * @memberof LinkedList<T>
                 */
                pop() {
                    if (this.tail) {
                        const node = this.tail;
                        if (this.length === 1) {
                            this.head = undefined;
                            this.tail = undefined;
                        } else {
                            this.tail = this.tail.prev;
                            this.tail.next = undefined;
                        }
                        this.length--;
                        return node.element;
                    }
                    return undefined;
                }

                /**
                 * Appends a new element to the end of the linked list, and returns the new length of the linked list, with O(1) complexity.
                 * @param {T} element to append at the end of the linked list.
                 * @returns {number} the new length of the linked list
                 * @memberof LinkedList
                 */
                push(element) {
                    const newTail = new LinkedListNode(element);
                    if (this.head === undefined && this.tail === undefined) {
                        this.head = this.tail = newTail;
                    } else {
                        this.connect(this.tail, newTail);
                        this.tail = newTail;
                    }
                    this.length++;
                    return this.length;
                }

                /**
                 * Removes the first element from the linked list and returns it, with O(1) complexity. 
                 * If the linked list is empty, undefined is returned and the linked list is not modified.. 
                 * @returns {T} the last element from the linked list
                 * @memberof LinkedList
                 */
                shift() {
                    if (this.head) {
                        const node = this.head;
                        if (this.length === 1) {
                            this.head = undefined;
                            this.tail = undefined;
                        } else {
                            this.head = this.head.next;
                            this.head.prev = undefined;
                        }
                        this.length--;
                        return node.element;
                    }
                    return undefined;
                }

                /**
                 * Inserts new elements at the start of the linked list, and returns the new length of the linked list, with O(1) complexity.
                 * @param {T} element to append at the start of the linked list.
                 * @returns {number} the new length of the linked list
                 * @memberof LinkedList
                 */
                unshift(element) {
                    const newHead = new LinkedListNode(element);
                    if (this.head === undefined && this.tail === undefined) {
                        this.head = this.tail = newHead;
                    } else {
                        this.connect(newHead, this.head);
                        this.head = newHead;
                    }
                    this.length++;
                    return this.length;
                }

                /**
                 * Connect two linked list nodes
                 * @param {LinkedListNode} prevEl
                 * @param {LinkedListNode} nextEl
                 */
                connect(prevEl, nextEl) {
                    prevEl.next = nextEl;
                    nextEl.prev = prevEl;
                }
            }
            return LinkedList;

        });
})();
