(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('DataQualityInsightHandler', {
        name: 'Data Quality',
        desc: 'Dataset or project Data Quality status information',
        i18nNameKey: 'INSIGHTS.DATA_QUALITY.NAME',
        i18nDescKey: 'INSIGHTS.DATA_QUALITY.DESC',
        icon: 'dku-icon-shield-check',
        color: 'data-quality-insight',

        getSourceId: function(insight) {
            return insight.params.objectSmartId;
        },
        getSourceType: function(insight) {
            return insight.params.objectType;
        },
        hasEditTab: false,
        defaultTileParams: {
        },
        defaultTileDimensions: [9, 6]
    });

    app.service('DataQualityInsightUtils', function(translate) {
        return {
            computeName(objectType, objectSmartId, statusType) {
                const statusTypeLabel = statusType === 'BREAKDOWN' ? 'breakdown' : 'status';
                if (objectType === 'PROJECT') {
                    return translate(
                        `INSIGHTS.DATA_QUALITY.DATA_QUALITY_${statusType}_ON_PROJECT`,
                        `Data quality ${statusTypeLabel} on project`
                    );
                } else if (objectSmartId) {
                    return translate(
                        `INSIGHTS.DATA_QUALITY.DATA_QUALITY_${statusType}_ON_OBJECT_WITH_NAME`,
                        `Data quality ${statusTypeLabel} on ${objectSmartId}`,
                        { objectName: objectSmartId }
                    );
                } else {
                    return translate(
                        `INSIGHTS.DATA_QUALITY.DATA_QUALITY_${statusType}_ON_OBJECT`,
                        `Data quality ${statusTypeLabel} on object`
                    );
                }
            }
        };
    });

    app.directive('dqInsightCreateForm', function($stateParams, DataQualityInsightUtils) { // dq-insight-create-form
        return {
            templateUrl: '/templates/dashboards/insights/data-quality/data-quality_create_form.html',
            scope: true,
            link: function($scope) {
                $scope.hook.sourceTypes = ['DATASET', 'PROJECT'];
                // disabled & set the forced value for the project in the 'pick existing' tab.
                $scope.$watchGroup(['filter.sourceType', 'filter.sourceId'], ([selectedSourceType]) => {
                    // we need to also watch sourceId because the default value cannot be applied before the object-picker has selected the list of projects.
                    if(selectedSourceType === 'PROJECT') {
                        $scope.hook.disableObjectSelector = true;
                        $scope.filter.sourceId = $stateParams.projectKey;
                    } else {
                        $scope.hook.disableObjectSelector = false;
                    }
                });

                // default initial values
                $scope.insight.params.objectType = 'DATASET';
                $scope.insight.params.statusType = 'CURRENT_STATUS';

                // clear selection on object type change, or re-select last selected of this type if any
                const lastSelectedByType = {};
                $scope.$watch('insight.params.objectType', (nv, ov) => {
                    if (ov) {
                        lastSelectedByType[ov] = $scope.insight.params.objectSmartId;
                    }
                    if (nv) {
                        const defaultValue = nv === 'PROJECT' ? $stateParams.projectKey : null;
                        $scope.insight.params.objectSmartId = lastSelectedByType[nv] || defaultValue;
                    }
                });

                // auto-update default name
                $scope.$watchGroup([
                    'insight.params.objectType',
                    'insight.params.objectSmartId',
                    'insight.params.statusType'
                ], ([objectType, objectSmartId, statusType]) => {
                    $scope.hook.defaultName = DataQualityInsightUtils.computeName(objectType, objectSmartId, statusType);
                });
            }
        };
    });


    function loadDataQualityInsightData(DataikuAPI, contextProjectKey, insightParams) {
        switch(insightParams.objectType) {
            case 'PROJECT': {
                switch(insightParams.statusType) {
                    case 'BREAKDOWN': return DataikuAPI.dataQuality.getProjectCurrentStatusBreakdown(contextProjectKey);
                    case 'CURRENT_STATUS': return DataikuAPI.dataQuality.getProjectCurrentDailyStatus(contextProjectKey);
                }
                break;
            }
            case 'DATASET': {
                const { projectKey, datasetName } = resolveDatasetFullName(insightParams.objectSmartId, contextProjectKey);
                switch(insightParams.statusType) {
                    case 'BREAKDOWN': return DataikuAPI.dataQuality.getDatasetCurrentStatusBreakdown(contextProjectKey, projectKey, datasetName);
                    case 'CURRENT_STATUS': return DataikuAPI.dataQuality.getDatasetCurrentDailyStatus(contextProjectKey, projectKey, datasetName);
                }
                break;
            }
        }
    }

    // converts the backend object repr to the frontend monitoring scope one.
    function fixupBreakdownScope(data) {
        if(data.monitoringScope) {
            data.fixedMonitoringScope = {
                'OBJECT': 'dataset',
                'PARTITION': 'partition',
                'RULE': 'rule'
            }[data.monitoringScope];
        }
    }

    // dq-insight-tile
    app.directive('dqInsightTile', function(DataikuAPI, DashboardUtils, TileLoadingState, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/data-quality/data-quality_tile.html',
            scope: {
                insight: '<',
                tile: '<',
                hook: '='
            },
            link: function($scope){
                $scope.loading = false;
                $scope.loaded = false;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    loadDataQualityInsightData(DataikuAPI, $stateParams.projectKey, $scope.insight.params)
                        .then(({ data }) => {
                            fixupBreakdownScope(data);
                            $scope.insightData = data;
                        }).then(
                            DashboardUtils.setLoaded.bind([$scope, resolve]),
                            DashboardUtils.setError.bind([$scope, reject])
                        );
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    // dq-insight-view
    app.directive('dqInsightView', function(DataikuAPI, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/data-quality/data-quality_view.html',
            scope: {
                insight: '<',
                preLoadedInsightData: '<?'
            },
            link: function($scope){
                if($scope.preLoadedInsightData) {
                    $scope.insightData = $scope.preLoadedInsightData;
                } else {
                    loadDataQualityInsightData(DataikuAPI, $stateParams.projectKey, $scope.insight.params)
                        .success((data) => {
                            fixupBreakdownScope(data);
                            $scope.insightData = data;
                        })
                        .error(setErrorInScope.bind($scope));
                }
            }
        };
    });

})();
