(function() {
'use strict';

const app = angular.module('dataiku.recipes');

    app.constant('METRICS', {
            BINARY_CLASSIFICATION: [
                "accuracy",
                "precision",
                "recall",
                "f1",
                "costMatrixGain",
                "hammingLoss",
                "mcc",
                "auc",
                "lift",
                "averagePrecision",
                "logLoss",
                "calibrationLoss",
                "customScore"
            ],
            MULTICLASS: [
                "accuracy",
                "precision",
                "recall",
                "hammingLoss",
                "mrocAUC",
                "averagePrecision",
                "logLoss",
                "mcalibrationLoss",
                "customScore"
            ],
            REGRESSION: [
                "evs",
                "mape",
                "mae",
                "mse",
                "rmse",
                "rmsle",
                "r2",
                "pearson",
                "customScore"
            ],
            TIMESERIES: [
                "mase",
                "mape",
                "smape",
                "mae",
                "meanAbsoluteQuantileLoss",
                "meanWeightedQuantileLoss",
                "mse",
                "rmse",
                "msis",
                "nd"
            ],
            TIMESERIES_SKIP_SCORING : [
                "mape",
                "smape",
                "mae",
                "meanAbsoluteQuantileLoss",
                "meanWeightedQuantileLoss",
                "mse",
                "rmse",
                "nd"
            ],
            MULTIPLE_TIMESERIES : ["mase", "mape", "smape", "mae", "mse", "msis"],
            MULTIPLE_TIMESERIES_SKIP_SCORING : ["mape", "smape", "mae", "mse"],
            CAUSAL: ["auuc", "qini", "netUplift", "propensityAuc", "propensityLogLoss", "propensityCalibrationLoss"],
            CAUSAL_WITHOUT_PROPENSITY: ["auuc", "qini", "netUplift"]
        });

    app.controller("_BaseMLRecipeEditor", function($scope, $q, $state, Assert, GraphZoomTrackerService, MLContainerInfoService,
        $stateParams, FullModelLikeIdUtils, GPU_SUPPORTING_CAPABILITY, GpuUsageService) {
        Assert.inScope($scope, 'script');
        Assert.inScope($scope, 'recipe');
        GraphZoomTrackerService.setFocusItemById("recipe", $state.params.recipeName);
        $scope.GPU_SUPPORTING_CAPABILITY = GPU_SUPPORTING_CAPABILITY;
        $scope.GpuUsageService = GpuUsageService;

        $scope.desc = JSON.parse($scope.script.data);

        $scope.hooks.preRunValidate = function() {
            return $q.when({ ok : true});
        };

        $scope.hooks.recipeIsDirty = function() {
            if (!$scope.recipe) return false;
            if ($scope.creation) {
                return true;
            } else {
                var dirty = !angular.equals($scope.recipe, $scope.origRecipe);
                var origDesc = JSON.parse($scope.origScript.data);
                dirty = dirty || !angular.equals(origDesc, $scope.desc);
               return dirty;
            }
        };

        $scope.shouldShowGpuControls = function(inUseCapabilities) {
            if (inUseCapabilities.length == 0) {
                return false;
            }
            const allowChangingMode = GpuUsageService.allowChangingMode(true, inUseCapabilities);
            return allowChangingMode || $scope.desc.gpuConfig.params.useGpu;
        }

        $scope.getInUseGpuCapabilities = function() {
            const capabilities = []

            if (!$scope.modelDetails) {
                return capabilities;
            }

            if ($scope.usesCodeEnvSentenceEmbedding()) {
                capabilities.push(GPU_SUPPORTING_CAPABILITY.SENTENCE_EMBEDDING);
            }

            if ($scope.isTimeseriesForecasting()) {
                capabilities.push(GPU_SUPPORTING_CAPABILITY.GLUONTS);
                return capabilities;
            }

            if (["DEEP_HUB_IMAGE_OBJECT_DETECTION", "DEEP_HUB_IMAGE_CLASSIFICATION"].includes($scope.modelDetails.modeling.type)) {
                capabilities.push(GPU_SUPPORTING_CAPABILITY.DEEP_HUB);
                return capabilities;
            }

            const alg = $scope.modelDetails.modeling.algorithm;

            if (["XGBOOST_REGRESSION", "XGBOOST_CLASSIFICATION"].includes(alg)) {
               capabilities.push(GPU_SUPPORTING_CAPABILITY.XGBOOST);
            }

            if (["TABICL_CLASSIFICATION"].includes(alg)){
                capabilities.push(GPU_SUPPORTING_CAPABILITY.TABICL);
            }

            if (["DEEP_NEURAL_NETWORK_REGRESSION", "DEEP_NEURAL_NETWORK_CLASSIFICATION"].includes(alg)) {
                capabilities.push(GPU_SUPPORTING_CAPABILITY.DEEP_NN);
            }

            if ("KERAS_CODE" === alg) {
                capabilities.push(GPU_SUPPORTING_CAPABILITY.KERAS);
            }

            return capabilities
        }

        $scope.usesCodeEnvSentenceEmbedding = function() {
            if (!$scope.modelDetails
                || !$scope.modelDetails.preprocessing
                || !$scope.modelDetails.preprocessing.per_feature) {
                return false;
            }
            return Object.values($scope.modelDetails.preprocessing.per_feature).some(featPreproc =>
                featPreproc.role === "INPUT" && featPreproc.text_handling === "SENTENCE_EMBEDDING" && featPreproc.sentenceEmbeddingModel != null && !featPreproc.isStructuredRef);
        };

        $scope.isMLBackendType = function(mlBackendType){
            return $scope.desc.backendType === mlBackendType;
        };

        $scope.goToAnalysisModel = function(){
            Assert.trueish($scope.desc.generatingModelId, 'no generatingModelId');
            // Enforcing projectKey to be current Project and not the one hard coded in fullModelId
            // to prevent from breaking when changing projectKey of analysis (e.g. importing project
            // and changing projectKey)
            const { elements, fullModelId } = FullModelLikeIdUtils.parseWithEnforcedProjectKey($scope.desc.generatingModelId, $stateParams.projectKey);

            const params = {
                projectKey: elements.projectKey,
                analysisId: elements.analysisId,
                mlTaskId: elements.mlTaskId,
                fullModelId: fullModelId
            };

            let state = "projects.project.analyses.analysis.ml.";
            if ($scope.recipe.type == "prediction_training") {
                state += "predmltask.model.report";
            } else {
                state += "clustmltask.model.report";
            }
            $state.go(state, params);
        };

        $scope.inContainer = MLContainerInfoService.inContainer($scope, $stateParams.projectKey);

        $scope.getModelUsedCodeEnvName = function() {
            if ($scope.modelDetails
                && $scope.modelDetails.coreParams
                && $scope.modelDetails.coreParams.executionParams) {
                    return $scope.modelDetails.coreParams.executionParams.envName;
            } else {
                return undefined;
            }
        };

        $scope.isPartitionedModel = function () {
            return $scope.modelDetails
                && $scope.modelDetails.coreParams
                && $scope.modelDetails.coreParams.partitionedModel
                && $scope.modelDetails.coreParams.partitionedModel.enabled;
        }

        $scope.isClassicalPrediction = function() {
            return $scope.modelDetails && ['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS'].includes($scope.modelDetails.coreParams.prediction_type);
        };

    });


    app.controller("_MLRecipeWithOutputSchemaController", function($scope, $q, ComputableSchemaRecipeSave, PartitionDeps, DataikuAPI) {
        $scope.hooks.save = function(){
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            var payload = angular.toJson($scope.desc);

            if ($scope.forceDisableOutputSchemaHandling && $scope.forceDisableOutputSchemaHandling()) {
                // Even in ML recipes *WITH* output schema handling (scoring), there
                // are cases where we actually can't handle output schema, but we did not know
                // statically. The main case for this is MLflow.
                $scope.script.data = payload;
                return $scope.baseSave(recipeSerialized, payload);
            } else {
                var deferred = $q.defer();
                ComputableSchemaRecipeSave.handleSave($scope, recipeSerialized, payload, deferred);
                $scope.script.data = payload;
                return deferred.promise;
            }
        };

        $scope.updateUsabilityOfSMInputs = function() {
            DataikuAPI.flow.recipes.updateUsabilityOfSMInputs(
                $scope.recipe.projectKey, $scope.model.needsInputDataFolder, $scope.model.miniTask.predictionType, Object.values($scope.computablesMap)
            ).then(function({data}) {
                // [Hack, dirty] Not the proper way of changing the computables map, should use `setComputablesMap` instead
                const newComputablesMap = Object.fromEntries(data.map(d => [d.smartName, d]));
                Object.assign($scope.computablesMap, newComputablesMap);
           }).catch(setErrorInScope.bind($scope));
        };

        function recipeSaved() {
            if ($scope.uiState && $scope.uiState.warningMessages && $scope.uiState.warningMessages.model && $scope.uiState.warningMessages.model.length > 0) {
                const customMetricsChanged = $scope.uiState.warningMessages.model.some(x => x && x.code === "WARN_RECIPE_INPUT_MODEL_CUSTOM_METRICS_CHANGED");
                if(customMetricsChanged) {
                    $scope.recipeUpdateData().then(_ => {
                        $scope.uiState.warningMessages.model = [];
                        $scope.desc = JSON.parse($scope.script.data); // prevent the change of the custom metrics following recipeUpdateData from marking recipe as dirty
                        $scope.$broadcast('customMetricsUpdated');
                    });
                }
            }
        }
        $scope.$on('recipeSaved', recipeSaved);
    });


    app.controller("_MLRecipeWithoutOutputSchemaController", function($scope, PartitionDeps, Assert){
        Assert.inScope($scope, 'recipe');
        Assert.inScope($scope, 'desc');

        $scope.hooks.save = function(){
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            var payload = angular.toJson($scope.desc);
            $scope.script.data = payload;
            return $scope.baseSave(recipeSerialized, payload);
        };
    });


    app.controller("_TimeseriesScoringOrEvaluationController", function ($scope, TimeseriesForecastingUtils) {
        $scope.prettyTimeSteps = TimeseriesForecastingUtils.prettyTimeSteps;
        $scope.prettySelectedDate = TimeseriesForecastingUtils.prettySelectedDate;
        $scope.getWeekDayName = TimeseriesForecastingUtils.getWeekDayName;

        $scope.isTimeseriesForecasting = function() {
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "TIMESERIES_FORECAST";
        };

        $scope.externalFeatures = function() {
            if (!$scope.modelDetails || !$scope.modelDetails.modeling
                    || TimeseriesForecastingUtils.ALGOS_WITHOUT_EXTERNAL_FEATURES.names.includes($scope.modelDetails.modeling.algorithm)) return [];
            return Object.keys($scope.modelDetails.preprocessing.per_feature).filter(function(name) {
                const preprocessing = $scope.modelDetails.preprocessing.per_feature[name];
                return preprocessing.role === 'INPUT';
            });

        };

        $scope.canRefitModel = function () {
            return $scope.modelDetails
                   && $scope.modelDetails.modeling
                   && ["AUTO_ARIMA", "ARIMA", "CROSTON", "ETS", "PROPHET", "SEASONAL_LOESS"].includes($scope.modelDetails.modeling.algorithm);
        };

        $scope.mustRefitModel = function () {
            return $scope.modelDetails
                   && $scope.modelDetails.modeling
                   && ["SEASONAL_LOESS", "CROSTON"].includes($scope.modelDetails.modeling.algorithm);
        };

        $scope.isMultipleTimeseries = function() {
            return $scope.model
                   && $scope.model.miniTask
                   && $scope.model.miniTask.timeseriesIdentifiers
                   && $scope.model.miniTask.timeseriesIdentifiers.length > 0;
        };

        $scope.isGluonTSBased = function() {
            return $scope.modelDetails.modeling.algorithm.startsWith("GLUONTS");
        }

        $scope.showMaxUsedTimestepsForScoring = function () {
            return $scope.modelDetails.iperf.maxUsedTimestepsForScoring > $scope.modelDetails.iperf.minTimeseriesSizeForScoring
        };

        let timeUnitInMilliSeconds = function(timeunit) {
            switch(timeunit) {
                case "MILLISECOND":
                    return 1;
                case "SECOND":
                    return 1000;
                case "MINUTE":
                    return 60000;
                case "HOUR":
                    return 3600000;
                case "DAY":
                    return 86400000;
                case "BUSINESS_DAY":
                    return 86400000;
                case "WEEK":
                    return 604800000;
                case "MONTH":
                    return 2629800000;
                case "QUARTER":
                    return 7889400000;
                case "HALF_YEAR":
                    return 15778800000;
                case "YEAR":
                    return 31557600000;
            }
        }

        let getPredictionLengthInMilliSeconds = function() {
            return $scope.desc.predictionLength * $scope.modelDetails.coreParams.timestepParams.numberOfTimeunits * timeUnitInMilliSeconds($scope.modelDetails.coreParams.timestepParams.timeunit);
        }

        $scope.isLastTimestampLikelyToOverflow = function () {
            return getPredictionLengthInMilliSeconds() >  7.0057872e+12; // Timestamp later than 2262-04-11 will make Pandas crash, taking 2020 as a common reference
        }

        $scope.isPredictionLengthTooLong = function() {
            return getPredictionLengthInMilliSeconds() > 1.844674407370955e+13; // WILL make Pandas crash
        }

        $scope.isAlgoSupportingQuantiles = function() {
            return !TimeseriesForecastingUtils.ALGOS_WITHOUT_QUANTILES.includes($scope.modelDetails?.modeling?.algorithm);
        }

        $scope.isAlgoScoringLessThanPredictionLength = function() {
            return $scope.modelDetails
                && $scope.modelDetails.modeling
                && TimeseriesForecastingUtils.ALGOS_SCORING_UP_TO_PREDICTION_LENGTH.includes($scope.modelDetails.modeling.algorithm.toLowerCase());
        };

        // Initializing for recipes created before 13.5
        // See TabularPredictionScoringRecipePayloadParams time-series forecasting specific params
        $scope.initPredictionLength = function() {
            if (!$scope.desc.predictionLength) $scope.desc.predictionLength = $scope.modelDetails.coreParams.predictionLength;
        };

        function initImputeMethods(modelDetails) {
            if (modelDetails && modelDetails.preprocessing && modelDetails.preprocessing.timeseriesSampling )  {
                const timeseriesSampling = modelDetails.preprocessing.timeseriesSampling;
                $scope.numericalInterpolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
                  obj => obj.value === timeseriesSampling.numericalInterpolateMethod
                );
                $scope.numericalExtrapolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
                  obj => obj.value === timeseriesSampling.numericalExtrapolateMethod
                );
                $scope.categoricalImputeMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
                  obj => obj.value === timeseriesSampling.categoricalImputeMethod
                );
                $scope.duplicateTimestampsHandlingMethod = TimeseriesForecastingUtils.DUPLICATE_TIMESTAMPS_HANDLING_METHODS.find(
                  obj => obj.value === timeseriesSampling.duplicateTimestampsHandlingMethod
                );
            }
        }
        $scope.hasChangedNbEvalTimesteps = function () {
            if (!$scope?.origRecipe.versionTag?.versionNumber) {
                return false
            }
            if ($scope.origRecipe.versionTag.versionNumber == 0){
                return false
            }
            let origDesc = JSON.parse($scope.origScript.data);
            return $scope.desc.maxNbForecastTimeSteps !== origDesc.maxNbForecastTimeSteps;
        };

        function setMaxNbTimesteps(modelDetails) {
            // in 14.1.0, we moved from max horizons to max timesteps
            if (modelDetails && $scope.desc) {

                // Only set maxNbForecastTimeSteps when predictionLength has a valid value to avoid NaN or null issues.
                if ($scope.isEvaluationRecipe 
                    && $scope.isTimeseriesForecasting()
                    && $scope.isMLBackendType("PY_MEMORY")
                    && undefined === $scope.desc.maxNbForecastTimeSteps) {

                    $scope.desc.maxNbForecastTimeSteps = $scope.desc.maxNbForecastHorizons *
                                                         (modelDetails.coreParams.predictionLength ?? 10);
                    $scope.hooks.save();
                }
                deregisterWatch();
            }
        }

        $scope.$watch('modelDetails', initImputeMethods, true);
        const deregisterWatch = $scope.$watch('modelDetails', setMaxNbTimesteps, true);
    });

    app.controller("_CausalPredictionScoringOrEvaluationController", function($scope) {

        $scope.isCausalPrediction = function() {
            return $scope.modelDetails
                && ["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes(
                    $scope.modelDetails.coreParams.prediction_type
                );
        };

        $scope.isCausalBinaryClassification = function() {
            return ($scope.modelDetails
                    && "CAUSAL_BINARY_CLASSIFICATION" === $scope.modelDetails.coreParams.prediction_type);
        };

        $scope.isMultiValuedTreatment = function() {
            return ($scope.modelDetails &&
                    $scope.modelDetails.coreParams.enable_multi_treatment &&
                    $scope.modelDetails.coreParams.treatment_values.length > 2);
        };

        $scope.getFilteredTreatmentValues = function() {
            const controlValue = $scope.modelDetails.coreParams.control_value;
            const treatmentValues = $scope.modelDetails.coreParams.treatment_values.filter(x => x !== controlValue);
            if ($scope.modelDetails.preprocessing.drop_missing_treatment_values) {
                return treatmentValues.filter(x => x !== "");
            } else {
                return treatmentValues;
            }
        };

    });


    app.controller("PredictionTrainingRecipeEditor", function($scope, $controller, ModelLabelUtils) {
        $controller("_BaseMLRecipeEditor", { $scope });
        $controller("_MLRecipeWithoutOutputSchemaController", { $scope });
        $controller("_K8sConfigurationCheckerController", { $scope });

        $scope.operationModeChanged = function(nv) {
            $scope.desc.splitParams.kfold = nv === 'TRAIN_KFOLD'
        }

        $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
            if (!$scope.desc) { return false; }

            switch(role.name){
                case "data":
                    // Data role is only available for deephub and classical models with image preprocessing on training recipes
                    return $scope.desc && $scope.desc.needsInputDataFolder;
                case "test":
                    return $scope.desc.splitParams && $scope.desc.splitParams.ttPolicy == "EXPLICIT_FILTERING_TWO_DATASETS";
                default:
                    throw new Error(`Rules for availability of input role "${role.name}" not implemented`);
            }
        }

        $scope.enableAutoFixup();
        $scope.isMLLib = function() { return $scope.desc.backendType === 'MLLIB' };

        // Overriding `$scope.isPartitionedModel` defined in _BaseMLRecipeEditor
        // because train recipe does not have `modelDetails`. Instead, coreParams
        // are stored in `desc.core`
        $scope.isPartitionedModel = function() {
            return $scope.desc
                   && $scope.desc.core
                   && $scope.desc.core.partitionedModel
                   && $scope.desc.core.partitionedModel.enabled;
        };

        function isClassification() {
            return $scope.desc
                   && $scope.desc.core
                   && $scope.desc.core.prediction_type
                   && ["BINARY_CLASSIFICATION", "MULTICLASS", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.desc.core.prediction_type);
        };

        function isKFoldSplit() {
            return $scope.desc
                   && $scope.desc.splitParams
                   && $scope.desc.splitParams.kfold;
        };

        $scope.isCompatibleWithStratifiedSplitting = function() {
            return $scope.isMLBackendType('PY_MEMORY') && isKFoldSplit() && isClassification();
        };

        $scope.isCompatibleWithGroupKFold = function() {
            return $scope.isMLBackendType('PY_MEMORY') && isKFoldSplit() && !$scope.isTimeseriesForecasting();
        }

        // Cache the return value of this function after it is initialized, so it is only computed once in digests
        let groupKFoldColumnNames = null;


        function withPredictionIntervals () {
            if (isClassification()){
                return false
            }
            return $scope.desc.core.uncertainty != null &&
                $scope.desc.core.uncertainty.predictionIntervalsEnabled;
        };

        $scope.canTrainOnSplit = function () {
            return !$scope.isTimeseriesForecasting();
        };

        $scope.canTrainOnSplitAndFull = function () {
            return !withPredictionIntervals();
        };

        $scope.canTrainOnKFold = function () {
            return $scope.isMLBackendType('PY_MEMORY')
                && (!($scope.desc.core.time != null && $scope.desc.core.time.enabled) || $scope.isTimeseriesForecasting())
                && !$scope.isCausalPrediction()
                && !withPredictionIntervals();
        };

        $scope.getGroupKFoldColumnNames = function() {
            if (groupKFoldColumnNames === null && $scope.desc && $scope.desc.preprocessing && $scope.desc.preprocessing.per_feature) {
                groupKFoldColumnNames = Object.keys($scope.desc.preprocessing.per_feature).filter(function(name) {
                    if ($scope.isCausalPrediction() && name === $scope.desc.core.treatment_variable) {
                        return false;
                    }
                    return $scope.desc.core.target_variable !== name;
                });
            }

            return groupKFoldColumnNames;
        };

        $scope.isTimeseriesForecasting = function() {
            return $scope.desc
                   && $scope.desc.core
                   && $scope.desc.core.prediction_type
                   && $scope.desc.core.prediction_type === "TIMESERIES_FORECAST";
        }

        $scope.isCausalPrediction = function() {
            return $scope.desc
                   && $scope.desc.core
                   && $scope.desc.core.prediction_type
                   && ["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.desc.core.prediction_type);
        }

        $scope.mayUseContainer = function() {
            if (!$scope.desc) return false; // not ready

            const backendType = $scope.desc && $scope.desc.core && $scope.desc.core.backendType;
            return ["PY_MEMORY", "KERAS", "DEEP_HUB"].includes(backendType);
        };

        $scope.partitionedSourceOptions = [
            ["ACTIVE_VERSION", "Active"],
            ["LATEST_VERSION", "Latest"],
            ["EXPLICIT_VERSION", "Explicit"],
            ["NONE", "None"]
        ];

        $scope.partitionedSourceDescs = [
            "Train upon the currently active saved model version",
            "Train upon the most recently trained version",
            "Choose which version to train upon",
            "Build a new partitioned models from scratch"
        ];

        $scope.hasSelectedK8sContainer = () => {
            const { backendType } = $scope.desc;
            const { containerSelection } = $scope.recipe.params;
            return $scope.isK8sContainer(backendType, containerSelection);
        };

        const updateHpSearchDistribution = (newSelection, oldSelection) => {
            if (angular.equals(newSelection, oldSelection)) {
                return;
            }

            const searchParams = $scope.desc.modeling.grid_search_params;
            if (!searchParams){
                return; // deephub doesn't support HP search
            }
            searchParams.distributed = searchParams.distributed && $scope.hasSelectedK8sContainer();
        };

        const baseCanSave = $scope.canSave;

        $scope.canSave = function() {
            return ModelLabelUtils.validateLabels($scope.recipe)
                && baseCanSave();
        }

        $scope.$watch('recipe.params.containerSelection', updateHpSearchDistribution, true);
    });


    app.controller("ClusteringTrainingRecipeEditor", function($scope, $controller, ModelLabelUtils) {
        $controller("_BaseMLRecipeEditor", {$scope:$scope});
        $controller("_MLRecipeWithoutOutputSchemaController", {$scope:$scope})
        $scope.enableAutoFixup();
        $scope.isMLLib = function() { return $scope.desc.backendType === 'MLLIB' };

        const baseCanSave = $scope.canSave;

        $scope.canSave = function() {
            return ModelLabelUtils.validateLabels($scope.recipe)
                && baseCanSave();
        }

    });


    app.controller("PredictionScoringRecipeEditor", function($scope, $controller, $q, DataikuAPI, Assert,
        MLExportService, FullModelLikeIdUtils, SavedModelsService) {
        $controller("_BaseMLRecipeEditor", {$scope:$scope});
        $controller("_MLRecipeWithOutputSchemaController", {$scope:$scope});
        $controller("_RecipeWithEngineBehavior", {$scope:$scope});
        $controller("_TimeseriesScoringOrEvaluationController", { $scope: $scope });
        $controller("_CausalPredictionScoringOrEvaluationController", { $scope: $scope });

        const backendSupportsExplanations = () => $scope.desc.backendType !== "DEEP_HUB";

        $scope.treatmentAssignmentModes = [["SAMPLE_RATIO_EXACT", "Ratio of the data (exact)"],
                                           ["SAMPLE_RATIO_APPROX", "Ratio of the data (approximate)"],
                                           ["THRESHOLD", "Threshold on predicted effect"]];

        // Payload is not expanded by backend, need defaults in the frontend
        // See PredictionScoringRecipePayloadParams.IndividualExplanationParams
        if (backendSupportsExplanations()) {  // Only add explanations if backend supports it (also dirties settings)
            $scope.desc.individualExplanationParams = {
                method: "ICE",
                nbExplanations: 3,
                shapleyBackgroundSize: 100,
                subChunkSize: 5000,
                ... ($scope.desc.individualExplanationParams || {})
            };

            $scope.$watch("recipeStatus.selectedEngine.type", (nv) => {
                if (nv && $scope.canComputeExplanations() === false) {
                    $scope.desc.outputExplanations = false;
                }
                if (nv && $scope.willUseSpark()) {
                    $scope.desc.outputModelMetadata = false;
                }
            })
        }

        $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
            if (!$scope.model) return false;

            switch(role.name){
                case "data":
                    // Data role is only available for deephub and classical models with image preprocessing on scoring recipes
                    return $scope.model && $scope.model.needsInputDataFolder;
                default:
                    throw new Error(`Rules for availability of input role "${role.name}" not implemented`);
            }
        }

        $scope.enableAutoFixup();
        $scope.canChangeEngine = function(){
            return true;
        };

        $scope.selectedEngine = function(){
            return $scope.recipeStatus && $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.type;
        };

        $scope.selectedEngineVariant = function(){
            return $scope.recipeStatus && $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.variant;
        };

        $scope.hooks.onRecipeLoaded = function(){
             $scope.hooks.updateRecipeStatus();
        };

        $scope.hooks.getPayloadData = function(){
            return angular.toJson($scope.desc);
        };

        $scope.hooks.updateRecipeStatus = function() {
            var deferred = $q.defer();
            var payload = $scope.hooks.getPayloadData();
            $scope.updateRecipeStatusBase(false, payload).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        var noProbAlgos = ['DECISION_TREE_CLASSIFICATION', 'MLLIB_DECISION_TREE'];
        $scope.noSQLProbas = function(){
            return $scope.isMulticlass() && $scope.selectedEngine() == "SQL"
                                  && noProbAlgos.indexOf($scope.modelDetails.modeling.algorithm) >= 0;
        }

        var safeSQLAlgorithms = ['LASSO_REGRESSION', 'RIDGE_REGRESSION', 'LEASTSQUARE_REGRESSION',
        'LOGISTIC_REGRESSION', 'DECISION_TREE_CLASSIFICATION', 'DECISION_TREE_REGRESSION', 'MLLIB_LOGISTIC_REGRESSION',
        'MLLIB_DECISION_TREE', 'MLLIB_LINEAR_REGRESSION'];
        $scope.isRiskySQL = function(){
            if ($scope.recipeStatus && $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.variant == "IN_SNOWFLAKE") {
                return false;
            }
            if(!$scope.modelDetails){
                return false;
            }
            return safeSQLAlgorithms.indexOf($scope.modelDetails.modeling.algorithm) < 0;
        };

        $scope.hasConditionalOutputs = function(){
            return $scope.model.conditionalOutputs && $scope.model.conditionalOutputs.length > 0;
        };

        $scope.isSQL = function(){
            return $scope.selectedEngine() == 'SQL'
                        && $scope.selectedEngineVariant() != "IN_SNOWFLAKE"
                        && $scope.model.miniTask.backendType != 'VERTICA';
        };


        $scope.hasCalibration = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.calibration.calibrationMethod != 'NO_CALIBRATION';
        };

        $scope.canForceOriginalEngine = function(){
            if (!$scope.model || !$scope.model.miniTask) return; // not ready
            if ($scope.isExternalMLflowModel()) return false;
            if ($scope.isCausalPrediction()) return false;
            var pyCase = $scope.model.miniTask.backendType == 'PY_MEMORY' && $scope.selectedEngine() == 'DSS';
            var kerasCase = $scope.model.miniTask.backendType == 'KERAS' && $scope.selectedEngine() == 'DSS';
            var mllibCase = $scope.model.miniTask.backendType == 'MLLIB' && $scope.selectedEngine() == 'SPARK';
            return pyCase || mllibCase || kerasCase;
        };

        $scope.willUseSpark = function(){
            return $scope.selectedEngine() == 'SPARK';
        };

        $scope.canComputeExplanations = function() {
            if (!$scope.model || !$scope.model.miniTask || !$scope.modelDetails) {  // not ready
                return;
            }

            if ($scope.isMLflowModel() && $scope.desc.mlFlowOutputStyle == 'RAW') {
                return false;
            }

            if ($scope.isTimeseriesForecasting() || $scope.isCausalPrediction()) {
                return false;
            }

            if (!$scope.isExternalMLflowModel()) {
                return $scope.model.miniTask.backendType == 'PY_MEMORY' && $scope.selectedEngine() == 'DSS';
            }

            const canHavePerf = ($scope.isBinaryClassification() || $scope.isMulticlass()) && $scope.isProbaAware() || $scope.isRegression();
            if (!canHavePerf) {
                return false;
            }

            if (!$scope.modelDetails.perf) {  // not evaluated
                return false;
            }
            return true;
        }

        $scope.onMlFlowOutputStyleChange = function() {
            if ($scope.desc.mlFlowOutputStyle === "RAW") {
                $scope.desc.outputExplanations = false;
            }
        }

        $scope.onOutputExplanationsChange = function() {
            $scope.desc.forceOriginalEngine = $scope.desc.outputExplanations;
            $scope.desc.individualExplanationParams.method = "ICE";
        }

        $scope.mayUseContainer = function() {
            if (!$scope.model || !$scope.model.miniTask) return false; // not ready
            return ["PY_MEMORY", "KERAS", "DEEP_HUB"].includes($scope.model.miniTask.backendType) && $scope.selectedEngine() == 'DSS';
        };

        $scope.hasSQLWarnings = function(){
            return $scope.hasConditionalOutputs() || $scope.isRiskySQL() || $scope.noSQLProbas();
        };

        $scope.showDownloadSQL = function(){
            return $scope.appConfig.licensedFeatures && $scope.appConfig.licensedFeatures.modelsRawSQLExport;
        };

        $scope.downloadSQL = function(){
            MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.getSql($scope.recipe),
                (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL("sql", exportId));
        };

        $scope.zeTrue = true;
        $scope.zeFalse = false;

        $scope.isBinaryClassification = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "BINARY_CLASSIFICATION";
        };

        $scope.isMulticlass = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "MULTICLASS";
        };

        $scope.isRegression = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "REGRESSION";
        };

        $scope.hasAPredictionType = function(){
            return ($scope.model && $scope.model.miniTask && $scope.model.miniTask.predictionType);
        };

        $scope.isDeepHubObjectDetection = function() {
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "DEEP_HUB_IMAGE_OBJECT_DETECTION";
        };

        $scope.isDeepHubBackendType = function() {
            return $scope.model && $scope.model.miniTask.backendType === "DEEP_HUB";
        };

        $scope.isProbaAware = function(){
            return $scope.modelDetails && $scope.modelDetails.iperf && $scope.modelDetails.iperf.probaAware && !$scope.noSQLProbas()
                && (!$scope.isMLflowModel() || $scope.desc.mlFlowOutputStyle == 'PARSED');
        };

        $scope.canManageThreshold = function(){
            return $scope.isBinaryClassification() && $scope.isProbaAware() && $scope.model && $scope.model.miniTask.backendType !== 'VERTICA';
        };

        $scope.isExternalMLflowModel = function() {
            return SavedModelsService.isExternalMLflowModel($scope.model);
        }

        $scope.isMLflowModel = function() {
            return SavedModelsService.isMLflowModel($scope.model);
        }

        $scope.isProxyModel = function() {
            return SavedModelsService.isProxyModel($scope.model);
        }

        $scope.forceDisableOutputSchemaHandling = function() {
            return $scope.isExternalMLflowModel();
        }

        function updateSavedModel() {
            if (!$scope.computablesMap) return;

            $scope.recipe.inputs['model'].items.forEach(function(inp){
                const computable = $scope.computablesMap[inp.ref];
                if (computable.type === "SAVED_MODEL") {
                    $scope.model = computable.model;
                }
            });

            Assert.inScope($scope, 'model');
            Assert.trueish($scope.model.miniTask.taskType === "PREDICTION", 'not a prediction task');

            $scope.updateUsabilityOfSMInputs();

            if ($scope.model.savedModelType == "DSS_MANAGED") {
                DataikuAPI.ml.prediction.getPreparedInputSchema($scope.recipe).success(function(data) {
                    $scope.preparedInputSchema = data;
                }).error(setErrorInScope.bind($scope));
            }

            const fmiComponents = {
                projectKey: $scope.model.projectKey,
                savedModelId: $scope.model.id,
                versionId: $scope.model.activeVersion
            };

            DataikuAPI.ml.prediction.getModelDetails(FullModelLikeIdUtils.buildSavedModelFmi(fmiComponents))
                .success(function(data){
                    $scope.modelDetails = data;

                    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
                    $scope.puppeteerHook_elementContentLoaded = true;
                });
        }
        $scope.$on('computablesMapChanged', updateSavedModel);
        updateSavedModel(); // May have loaded before this controller (itself ~ajax loaded by template)
        $scope.$watch('recipe.inputs.model', function (nv, ov) {
            if (!nv || !nv.items[0] || nv === ov) return;

            updateSavedModel();
        });
    });


    app.controller("ClusteringClusterRecipeEditor", function($scope, $controller, DataikuAPI) {
        $controller("_BaseMLRecipeEditor", {$scope:$scope});
        $controller("_MLRecipeWithOutputSchemaController", {$scope:$scope})
        $scope.enableAutoFixup();

        DataikuAPI.ml.clustering.getPreparedInputSchema($scope.recipe, $scope.desc).success(function(data) {
            $scope.preparedInputSchema = data;
        }).error(setErrorInScope.bind($scope));
    });


    app.controller("ClusteringScoringRecipeEditor", function($scope, $controller, DataikuAPI, Assert) {
        $controller("_BaseMLRecipeEditor", {$scope:$scope});
        $controller("_MLRecipeWithOutputSchemaController", {$scope:$scope})
        $scope.enableAutoFixup();

        DataikuAPI.ml.clustering.getPreparedInputSchema($scope.recipe, $scope.desc).success(function(data) {
            $scope.preparedInputSchema = data;

            $scope.$watch("computablesMap", (nv) => {
                if (nv) {
                    $scope.recipe.inputs['model'].items.forEach(function(inp){
                        var computable = $scope.computablesMap[inp.ref];
                        if (computable.type == "SAVED_MODEL") {
                            $scope.model = computable.model;
                        }
                    });

                    Assert.inScope($scope, 'model');
                    Assert.trueish($scope.model.miniTask.taskType == "CLUSTERING", 'not a clustering task');
                }
            })
        }).error(setErrorInScope.bind($scope));
    });

    /**
     * @ngdoc component
     * @name embeddingDriftConfig
     *
     * @description
     * Component to enable embedding drift for either text or image data.
     *
     * @property {string} embeddingType. Specifies the type of embedding ('TEXT' or 'IMAGE').
     * @property {Object} driftParams. The configuration object for drift parameters.
     * @property {boolean|undefined} hasDrift. A boolean that controls whether drift analysis is enabled.
     * @property {boolean} showDriftWarning. Controls the visibility of a warning message.
     * @property {string} driftWarningMessage. The text content of the warning message.
     * @property {string} projectKey. The key of the current project, used for API calls.
     * @property {boolean} isInBlock. A flag to indicate where the component is used.
     */
    app.component('embeddingDriftConfig', {
        templateUrl: "/templates/recipes/fragments/evaluation-recipe-embedding-drift.html",
        bindings: {
            embeddingType: '<',
            driftParams: '=',
            hasDrift: '=',
            showDriftWarning: '<',
            driftWarningMessage: '<',
            projectKey: "<",
            isInBlock: '<'
        },
        controller: ["DataikuAPI", function(DataikuAPI) {
            var $ctrl = this;

            $ctrl.$onInit = function () {
                // initial params
                $ctrl.hasInitialEmbeddingDrift = $ctrl.hasDrift !== undefined; // const
                $ctrl.initialDriftParams = angular.copy($ctrl.driftParams);    // const

                // ui.state
                $ctrl.uiState = {};
                $ctrl.uiState.driftParams = angular.copy($ctrl.driftParams);
                $ctrl.uiState.hasDrift = $ctrl.hasInitialEmbeddingDrift ? $ctrl.hasDrift : false;

                // embedding type property
                $ctrl.embeddingType = ($ctrl.embeddingType || '').toLowerCase();
                const embeddingTypeNormalized = ($ctrl.embeddingType || '').toUpperCase();
                const purposeMap = {
                    'TEXT': 'TEXT_EMBEDDING_EXTRACTION',
                    'IMAGE': 'IMAGE_EMBEDDING_EXTRACTION'
                };
                const settingsKeyMap = {
                    'TEXT': 'defaultEvalTextEmbeddingModelId',
                    'IMAGE': 'defaultEvalImageEmbeddingModelId'
                };

                // API calls
                DataikuAPI.pretrainedModels.listAvailableLLMs($ctrl.projectKey, purposeMap[embeddingTypeNormalized])
                    .success(function(data) { $ctrl.availableEmbeddingLLMs = data.identifiers || []; })
                    .error(setErrorInScope.bind($ctrl));
                
                DataikuAPI.admin.getGeneralSettings()
                    .success(function(data) {
                        const defaultEmbeddingModel = data.generativeAISettings[settingsKeyMap[embeddingTypeNormalized]];
                        if (!$ctrl.uiState.driftParams?.embeddingModelId && defaultEmbeddingModel) {
                            $ctrl.uiState.driftParams = {};
                            $ctrl.uiState.driftParams.embeddingModelId = defaultEmbeddingModel;
                            $ctrl.updateDriftParams();
                        }})
                    .error(setErrorInScope.bind($ctrl));
            };

            $ctrl.updateDriftParams = function() {
                $ctrl.driftParams = $ctrl.hasDrift? $ctrl.uiState.driftParams : $ctrl.initialDriftParams;
            };

            $ctrl.updateHasDrift = function() {
                $ctrl.hasDrift = ($ctrl.hasInitialEmbeddingDrift || $ctrl.uiState.hasDrift)?
                                  $ctrl.uiState.hasDrift : undefined;
                $ctrl.updateDriftParams();
            }
        }]
    });


    app.controller("EvaluationRecipeEditor", function($scope, $controller, $q, DataikuAPI, Assert, FullModelLikeIdUtils, ModelLabelUtils, SavedModelsService, $stateParams, ModelEvaluationUtils, SamplingData, DatasetUtils, StringUtils, WT1) {
        $controller("_BaseMLRecipeEditor", {$scope:$scope});
        $controller("_MLRecipeWithOutputSchemaController", {$scope:$scope});
        $controller("_RecipeWithEngineBehavior", {$scope:$scope});
        $controller("EvaluationLabelUtils", {$scope:$scope});
        $controller("_TimeseriesScoringOrEvaluationController", { $scope: $scope });
        $controller("_CausalPredictionScoringOrEvaluationController", { $scope: $scope });

        $scope.uiState = {};
        $scope.canSkipScoring = false;

        $scope.isEvaluationRecipe = true;

        $scope.enableAutoFixup();

        $scope.selectedEngine = function(){
            return $scope.recipeStatus ? $scope.recipeStatus.selectedEngine.type : undefined;
        };

        $scope.hooks.onRecipeLoaded = function(){
             $scope.hooks.updateRecipeStatus();
        };

        $scope.hooks.getPayloadData = function(){
            return angular.toJson($scope.desc);
        };

        $scope.hooks.updateRecipeStatus = function() {
            var deferred = $q.defer();
            var payload = $scope.hooks.getPayloadData();
            $scope.updateRecipeStatusBase(false, payload).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.hasAllRequiredOutputs = function() {
            if (!$scope.recipe || !$scope.recipe.outputs) {
                return false;
            }
            var out = $scope.recipe.outputs;
            // at least one of the outputs is needed
            if(out.main && out.main.items && out.main.items.length) {
                return true;
            }
            if(out.evaluationStore && out.evaluationStore.items && out.evaluationStore.items.length) {
                return true;
            }
            if(out.metrics && out.metrics.items && out.metrics.items.length) {
                return true;
            }
            return false;
        };

        $scope.displayMetricsWeightingInformations = function () {
            if (!$scope.modelDetails?.coreParams || $scope.modelDetails.backendType !== "PY_MEMORY") {
                return false;
            }
            else if (['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS'].includes($scope.modelDetails.coreParams.prediction_type) &&
                     ['SAMPLE_WEIGHT','CLASS_AND_SAMPLE_WEIGHT'].includes($scope.modelDetails.coreParams.weight.weightMethod)) {
                return true;
            }
            else if ($scope.modelDetails.coreParams.prediction_type === 'MULTICLASS') {
                return true;
            }
            else if ($scope.isCausalPrediction() && $scope.modelDetails.modeling.metrics.causalWeighting==='INVERSE_PROPENSITY') {
                return true;
            }
            return false;
        }

        function getFirstSnippet(modelVersions) {
            // $scope.modelVersions[0] is always {versionId:'', label:"Active version"}, see updateSavedModel
            if (modelVersions?.length > 1 && modelVersions[1].snippet) {
                return $scope.modelVersions[1].snippet;
            } else {
                return null;
            }
        }
        $scope.differentSampleWeights = function () {
            const firstSnippet = getFirstSnippet($scope.modelVersions);
            if (firstSnippet && ['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS'].includes($scope.modelDetails.coreParams.prediction_type)) {
                const firstSampleWeight = firstSnippet.sampleWeightsVariable;
                return !$scope.modelVersions.slice(1).every(x => x.snippet.sampleWeightsVariable === firstSampleWeight)
            }
            return false;
        }
        $scope.differentClassAveraging = function () {
            const firstSnippet = getFirstSnippet($scope.modelVersions);
            if (firstSnippet && $scope.modelDetails.coreParams.prediction_type === 'MULTICLASS') {
                const firstClassAveraging = firstSnippet.classAveragingMethod;
                return !$scope.modelVersions.slice(1).every(x => x.snippet.classAveragingMethod === firstClassAveraging)
            }
            return false;
        }
        $scope.differentCausalWeighting = function () {
            const firstSnippet = getFirstSnippet($scope.modelVersions);
            if (firstSnippet && ['CAUSAL_BINARY_CLASSIFICATION', 'CAUSAL_REGRESSION'].includes($scope.modelDetails.coreParams.prediction_type)) {
                const firstCausalWeighting = firstSnippet.causalWeighting;
                return !$scope.modelVersions.slice(1).every(x => x.snippet.causalWeighting === firstCausalWeighting)
            }
            return false;
        }

        // END differences with scoring

        var safeSQLAlgorithms = ['LASSO_REGRESSION', 'RIDGE_REGRESSION', 'LEASTSQUARE_REGRESSION',
        'LOGISTIC_REGRESSION', 'DECISION_TREE_CLASSIFICATION', 'DECISION_TREE_REGRESSION', 'MLLIB_LOGISTIC_REGRESSION',
        'MLLIB_DECISION_TREE', 'MLLIB_LINEAR_REGRESSION'];
        $scope.isRiskySQL = function(){
            if(!$scope.modelDetails){
                return false;
            }
            return safeSQLAlgorithms.indexOf($scope.modelDetails.modeling.algorithm) < 0;
        };

        $scope.mayUseContainer = function() {
            if (!$scope.model || !$scope.model.miniTask) return false; // not ready
            return ["PY_MEMORY", "KERAS", "DEEP_HUB"].includes($scope.model.miniTask.backendType)
                && $scope.selectedEngine() == 'DSS';
        };

        $scope.willUseSpark = function(){
            return $scope.selectedEngine() == 'SPARK';
        };

        $scope.versionDisplayFn = function(version) {
            return version.label;
        }

        $scope.versionValueFn = function(version) {
            return version.versionId;
        }

        const baseCanSave = $scope.canSave;

        $scope.targetUnavailableMessage = function() {
            if (!$scope.modelDetails || !$scope.modelDetails.coreParams
                || !$scope.preparedInputSchema || $scope.desc.dontComputePerformance) {
                    return "";
            }
            const targetInEvalDataset = $scope.preparedInputSchema.columns.find(c => {
                if (!$scope.desc.evaluationDatasetType || $scope.desc.evaluationDatasetType === ModelEvaluationUtils.CLASSIC_EVALUATION_DATASET_TYPE) {
                    return c.name === $scope.modelDetails.coreParams.target_variable;
                } else if ($scope.desc.evaluationDatasetType === ModelEvaluationUtils.API_NODE_LOGS_EVALUATION_DATASET_TYPE) {
                    return c.name === ModelEvaluationUtils.API_NODE_FEATURE_PREFIX + $scope.modelDetails.coreParams.target_variable;
                } else if ($scope.desc.evaluationDatasetType === ModelEvaluationUtils.CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE) {
                    return c.name === ModelEvaluationUtils.CLOUD_API_NODE_FEATURE_PREFIX + $scope.modelDetails.coreParams.target_variable;
                }
            });
            if (!targetInEvalDataset) {
                return `The evaluation dataset must have a column called '${$scope.modelDetails.coreParams.target_variable}'
                containing the ground truth. Running this recipe will fail unless you disable computing the model performance using the checkbox.`;
            }
        }

        $scope.canSave = function() {
            return ModelLabelUtils.validateLabels($scope.desc)
                && baseCanSave();
        };

        $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
            if (role.name == "data"){
                // Data role is only available for classical models with image preprocessing on eval recipes
                return $scope.model && $scope.model.needsInputDataFolder;
            }
            return true; // by default an input role is available to fill/select
        }

        $scope.isOutputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
            if (!$scope.modelDetails) return false;
            const smCoreParams = $scope.modelDetails.coreParams;

            switch(role.name){
                case 'evaluationStore':
                    // Only non-partitioned classical models (binary classification, multiclass, regression) can have an evaluation recipe with a MES as output
                    // TODO @causal @deephub handle MES as evaluation output
                    return ["BINARY_CLASSIFICATION", "MULTICLASS", "REGRESSION", "TIMESERIES_FORECAST"].includes(smCoreParams.prediction_type)
                            && (!smCoreParams.partitionedModel || !smCoreParams.partitionedModel.enabled);
                default:
                    throw new Error(`Rules for availability of output role "${role.name}" not implemented`);
            }
        };

        $scope.zeTrue = true;
        $scope.zeFalse = false;

        $scope.isBinaryClassification = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "BINARY_CLASSIFICATION";
        };

        $scope.isMulticlass = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "MULTICLASS";
        };

        $scope.isClassification = () => $scope.isBinaryClassification() || $scope.isMulticlass();

        $scope.isRegression = function(){
            return $scope.modelDetails && $scope.modelDetails.coreParams.prediction_type === "REGRESSION";
        };

        $scope.isProbaAware = function(){
            return $scope.modelDetails && $scope.modelDetails.iperf && $scope.modelDetails.iperf.probaAware;
        };

        $scope.isExternalMLflowModel = function() {
            return SavedModelsService.isExternalMLflowModel($scope.model);
        }

        $scope.forceDisableOutputSchemaHandling = function() {
            return $scope.isExternalMLflowModel();
        }

        function updateSavedModel() {
            if (!$scope.computablesMap) return;

            $scope.recipe.inputs['model'].items.forEach(function(inp){
                const computable = $scope.computablesMap[inp.ref];
                if (computable.type === "SAVED_MODEL") {
                    $scope.model = computable.model;
                }
            });

            Assert.inScope($scope, 'model');
            Assert.trueish($scope.model.miniTask.taskType === "PREDICTION", 'not a prediction task');

            $scope.updateUsabilityOfSMInputs();

            DataikuAPI.savedmodels.prediction.getStatus( $scope.model.projectKey, $scope.model.id).success(function(data){
                $scope.modelVersions = [];
                if (!$scope.isExternalMLflowModel()) {
                    data.versions.forEach(function(v) {
                        $scope.modelVersions.push({
                            versionId: sanitize(v.versionId),
                            label: '<i>' + moment(v.snippet.trainInfo.startTime).format('YYYY/MM/DD HH:mm') + " " + sanitize(v.snippet.userMeta.name) + (v.active ? ' (active)' : '') + '</i>&nbsp;-&nbsp;<b>' + sanitize(v.versionId) + '</b>',
                            snippet: v.snippet
                        })
                    });
                } else {
                    data.versions.forEach(function(v) {
                        $scope.modelVersions.push({versionId:sanitize(v.versionId), label:'<i>' + moment(v.snippet.importedOn).format('YYYY/MM/DD HH:mm')
                        + (v.active ? ' (active)' : '') + '</i>&nbsp;-&nbsp;<b>' + sanitize(v.versionId) + '</b>'})
                    });
                }
                $scope.modelVersions.sort((a, b) => a.label.localeCompare(b.label));
                $scope.modelVersions.unshift({versionId:'', label:"Active version"});

                // this copy must be made for angular state detection to detect the change.
                $scope.modelVersions = $scope.modelVersions.slice();
            }).error(setErrorInScope.bind($scope));

            const fmiComponents = {
                projectKey: $scope.model.projectKey,
                savedModelId: $scope.model.id,
                versionId: $scope.desc.modelVersionId || $scope.model.activeVersion
            };
            DataikuAPI.ml.prediction.getModelDetails(FullModelLikeIdUtils.buildSavedModelFmi(fmiComponents))
                .success(function(data){
                    $scope.modelDetails = data;

                    setInitialDriftColumns();
                    updateShowTextDriftWarning();
                    updateShowImageDriftWarningAndMessage();

                    switch($scope.modelDetails.coreParams.executionParams.envSelection.envMode) {
                        case "USE_BUILTIN_MODE":
                            $scope.modelEnvName = "Built-in";
                            break;
                        case "EXPLICIT_ENV":
                        case "INHERIT":
                            $scope.modelEnvName = $scope.modelDetails.coreParams.executionParams.envSelection.envName
                            break;
                    }

                    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
                    $scope.puppeteerHook_elementContentLoaded = true;
                });
        }

        function setInitialDriftColumns() {
            if ($scope.model.savedModelType === "DSS_MANAGED") {
                setInitialDriftColumnsFromPreparedInputSchema();
            } else if ($scope.isExternalMLflowModel()) {
                if ($scope.desc.evaluationDatasetTypeDetected === ModelEvaluationUtils.SAGEMAKER_EVALUATION_DATASET_TYPE) {
                    setInitialDriftColumnsForSagemakerLogsDataset();
                } else {
                    setInitialDriftColumnsFromDataset();
                }
            }
        }

        function setInitialDriftColumnsForSagemakerLogsDataset() {
            if ($scope.desc.evaluationDatasetType === ModelEvaluationUtils.SAGEMAKER_EVALUATION_DATASET_TYPE) {
                setInitialDriftColumnsFromModelSchema();
            } else {
                setInitialDriftColumnsFromDataset();
            }
        }

        function setInitialDriftColumnsFromDataset() {
            const contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$scope.recipe.projectKey;
            const datasetLoc = DatasetUtils.getLocFromSmart(contextProjectKey, $scope.recipe.inputs.main.items[0].ref);
            DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.name, contextProjectKey).success(function(data){
                $scope.initialDataDriftColumns = data.schema.columns;
            }).error(setErrorInScope.bind($scope));
        }

        function setInitialDriftColumnsFromPreparedInputSchema() {
            DataikuAPI.ml.prediction.getPreparedInputSchema($scope.recipe).success(function(data) {
                $scope.preparedInputSchema = data;
            }).error(setErrorInScope.bind($scope));
        }

        function setInitialDriftColumnsFromModelSchema() {
            if ($scope.modelDetails) {
                $scope.initialDataDriftColumns = Object.values($scope.modelDetails.preprocessing.per_feature);
            }
        }

        const hasInitialSkipScoring = $scope.desc.skipScoring !== undefined;
        $scope.uiState.skipScoring = hasInitialSkipScoring ? $scope.desc.skipScoring : false;

        $scope.$watch('uiState.skipScoring', function(skipScoring) {
            if (hasInitialSkipScoring) {
                $scope.desc.skipScoring = skipScoring;
            } else {
                $scope.desc.skipScoring = skipScoring === false ? undefined : true;
            }
        })
        const hasInitialTreatPerfMetricsFailureAsError = $scope.desc.treatPerfMetricsFailureAsError !== undefined;
        $scope.uiState.treatPerfMetricsFailureAsError = hasInitialTreatPerfMetricsFailureAsError ? $scope.desc.treatPerfMetricsFailureAsError : true;

        $scope.$watch('uiState.treatPerfMetricsFailureAsError', function(treatPerfMetricsFailureAsError) {
            if (hasInitialTreatPerfMetricsFailureAsError) {
                $scope.desc.treatPerfMetricsFailureAsError = treatPerfMetricsFailureAsError;
            } else {
                $scope.desc.treatPerfMetricsFailureAsError = treatPerfMetricsFailureAsError === true ? undefined : false;
            }
        })

        function updateDataset() {
            const contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$scope.recipe.projectKey;
            if (!$scope.computablesMap) return;
            const datasetLoc = DatasetUtils.getLocFromSmart(contextProjectKey, $scope.recipe.inputs.main.items[0].ref);
            DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.name, contextProjectKey).success(function(data){
                $scope.canSkipScoring = data.schema.columns.some(column => ['prediction', ModelEvaluationUtils.TIMESERIES_FORECAST_COLUMN, ModelEvaluationUtils.API_NODE_PREDICTION_COLUMN, ModelEvaluationUtils.CLOUD_API_NODE_PREDICTION_COLUMN, ModelEvaluationUtils.SAGEMAKER_PREDICTION_COLUMN].includes(column['name']));
                $scope.uiState.skipScoring =  $scope.canSkipScoring && $scope.uiState.skipScoring;
            }).error(setErrorInScope.bind($scope));
        }

        $scope.$on('computablesMapChanged', function() {
            updateSavedModel();
            updateDataset();
        });

        $scope.$watch('recipe.inputs.model', function (nv, ov) {
            if (!nv || !nv.items[0] || nv === ov) return;

            updateSavedModel();
        });
        $scope.$watch('recipe.inputs.main', function (nv, ov) {
            if (!nv || !nv.items[0] || nv === ov) return;

            updateDataset();
        });

        updateSavedModel(); // May have loaded before this controller (itself ~ajax loaded by template)
        updateDataset();

        $scope.isApiLogs = $scope.desc.evaluationDatasetType && $scope.desc.evaluationDatasetType !== ModelEvaluationUtils.CLASSIC_EVALUATION_DATASET_TYPE;

        $scope.handleEvaluationDatasetTypeChange = () => {
            $scope.isApiLogs = !$scope.isApiLogs;
            $scope.desc.evaluationDatasetType = $scope.desc.evaluationDatasetType === ModelEvaluationUtils.CLASSIC_EVALUATION_DATASET_TYPE ? $scope.desc.evaluationDatasetTypeDetected : ModelEvaluationUtils.CLASSIC_EVALUATION_DATASET_TYPE;

            setInitialDriftColumns();
        }

        $scope.$watch('preparedInputSchema', function(nv){
            if (!nv) return;
            $scope.initialDataDriftColumns = nv.columns;
        })

        $scope.skipScoringInfoMessage = () => {
            if ($scope.isPartitionedModel()) return 'This option is not available for partitioned models.';

            if (!$scope.canSkipScoring) {
                let predictionColumnName;
                if ($scope.desc.evaluationDatasetType === ModelEvaluationUtils.API_NODE_LOGS_EVALUATION_DATASET_TYPE) {
                    predictionColumnName = ModelEvaluationUtils.API_NODE_PREDICTION_COLUMN;
                } else if ($scope.desc.evaluationDatasetType === ModelEvaluationUtils.CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE) {
                    predictionColumnName = ModelEvaluationUtils.CLOUD_API_NODE_PREDICTION_COLUMN;
                } else if ($scope.desc.evaluationDatasetType === ModelEvaluationUtils.SAGEMAKER_EVALUATION_DATASET_TYPE) {
                    predictionColumnName = ModelEvaluationUtils.SAGEMAKER_PREDICTION_COLUMN;
                } else if ($scope.isTimeseriesForecasting()){
                    predictionColumnName = ModelEvaluationUtils.TIMESERIES_FORECAST_COLUMN;
                }else {
                    predictionColumnName = ModelEvaluationUtils.CLASSIC_PREDICTION_COLUMN;
                }
                return `A '${predictionColumnName}' column in the input dataset is mandatory for an evaluation without scoring.`;
            }

            else return (
            'The metrics will be computed on the '
            + ($scope.isTimeseriesForecasting() ? 'forecast' : 'prediction')
            +  ($scope.isClassification() ? ' and probability columns' : ' column')
            + ' of your dataset.')
        }

        $scope.userModifiedThreshold = false;
        $scope.$watch("desc.overrideModelSpecifiedThreshold", function(nv, ov) {
            if (ov === true && nv === true) {  // Loading a saved recipe with override=true => do not reset the threshold
                $scope.userModifiedThreshold = true;
            }
            if (nv === true && !$scope.userModifiedThreshold) {
                $scope.desc.forcedClassifierThreshold = $scope.modelDetails.userMeta.activeClassifierThreshold;
                $scope.userModifiedThreshold = true;
            }
        })

        ModelEvaluationUtils.synchronizeLimitSamplingUiStateAndDesc($scope);

        $scope.hasReferenceDataset = function() {
            return true;
        }

        $scope.onModelVersionUpdate = function(newModelVersionId) {
            if ($scope.model) {
                DataikuAPI.savedmodels.getFullInfo($stateParams.projectKey, $scope.model.id)
                    .success(function(data) {
                        let modelVersionId = newModelVersionId || data.status.activeVersionId;
                        const modelVersion = data.status.versions.find(v => v.versionId === modelVersionId);
                        if (!modelVersion) return;
                        const modelVersionCustomMetrics = (modelVersion.snippet.customMetricsResults || [])
                            .map(customMetricResult => customMetricResult.metric.name);
                        $scope.desc.possibleCustomMetrics = modelVersionCustomMetrics;
                        $scope.desc.customMetrics = modelVersionCustomMetrics;
                        $scope.$broadcast('customMetricsUpdated');
                        $scope.modelVersionUpdated = newModelVersionId !== JSON.parse($scope.origScript.data).modelVersionId;

                        updateSavedModel();
                    });
            }
        }

        $scope.isApiNodeLogsDatasetDetected = () => [ModelEvaluationUtils.API_NODE_LOGS_EVALUATION_DATASET_TYPE, ModelEvaluationUtils.CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE].includes($scope.desc.evaluationDatasetTypeDetected);

        $scope.isLogsDatasetDetected = () => ModelEvaluationUtils.isLogsDatasetType($scope.desc.evaluationDatasetTypeDetected);

        $scope.getLogsDatasetTypeFormatted = (evaluationDatasetType) => ModelEvaluationUtils.evaluationDatasetTypesFormatted[evaluationDatasetType];

        const recipeCreatedBeforeCustomEvaluationMetrics = $scope.desc.customEvaluationMetrics === undefined; // it was created before custom eval metrics existed
        $scope.uiState.customEvaluationMetrics = $scope.desc.customEvaluationMetrics || [];
        $scope.$watch('uiState.customEvaluationMetrics', function(customEvaluationMetrics) {
            const hasCustomMetrics = !!(customEvaluationMetrics && customEvaluationMetrics.length);
            if (recipeCreatedBeforeCustomEvaluationMetrics && !hasCustomMetrics) {  // we don't dirty legacy recipes
                $scope.desc.customEvaluationMetrics = undefined;
            } else {
                $scope.desc.customEvaluationMetrics = customEvaluationMetrics;
            }
        }, true);

        let customMetricDefaultCode = "def score(y_valid, y_pred, eval_df, output_df, ref_sample_df=None, ";
        if ($scope.desc.backendType !== "KERAS") {
            customMetricDefaultCode += "sample_weight=None, ";
        }
        customMetricDefaultCode += `**kwargs):
    """
    Custom scoring function.
    Must return a float quantifying the estimator prediction quality.
    - y_valid: pandas Series. Only the column with the ground truth.
      Shape is (nb_records).
    - y_pred: Predictions or probabilities given by the evaluated model.
      If "Skip scoring" is active, this is the 'prediction' column from the input dataset.
      Can be
        - pandas Series of shape (nb_records) for regression problems and classification problems
            where 'needs probas' (see below) is false
            (for classification, the values are the numeric class indexes)
        - numpy ndarray of shape (nb_records, nb_classes) for classification problems where
            'needs probas' is true;
    - eval_df: pandas Dataframe. The evaluation dataset, which contains both the columns
      that were used as input, as well as the column with the ground truth.
      Shape is (nb_records, nb_input_columns).
    - output_df: pandas Dataframe. Prediction output, ie, the eval_df processed and
      enriched with Predictions or probability computed by the evaluated model:
        - 'prediction': prediction made
        plus, for classification problems:
          - 'prediction_correct': whether the prediction was correct or not
          - 'proba_X': one column per class, with probabilities.
        and for regression problems:
          - 'error_decile'
          - 'abs_error_decile'
          - 'relative_error'
      Shape is:
        - (nb_records, 2 + nb_classes) for classification problems
            (for classification, the values are the numeric class indexes)
        - (nb_records, 4) for regression problems
    - [optional] ref_sample_df: pandas Dataframe. A sample from the dataset used at train-time 
      of the evaluated model.
      NB: this option requires a model evaluation store output.`;

        if ($scope.desc.backendType !== "KERAS") {
            customMetricDefaultCode += `
    - [optional] sample_weight is a numpy ndarray with shape (nb_records)
      NB: this option requires a variable set as "Sample weights" in the original evaluated model training`;
        }
        customMetricDefaultCode += `
    """
    return 0.5`;

        $scope.getNewMetricTemplate = function() {
            const name = StringUtils.transmogrify("Custom Evaluation Metric #" + ($scope.uiState.customEvaluationMetrics.length + 1).toString(),
                $scope.uiState.customEvaluationMetrics.map(a => a.name),
                function(i){return "Custom Evaluation Metric #" + (i+1).toString() }
            );

            const template = {
                name,
                metricCode: customMetricDefaultCode,
                description: "",
                greaterIsBetter: true,
                needsProbability: false,
                type: 'EVALUATION_RECIPE',
                $foldableOpen: true
            };
            return template;
        }

        $scope.snippetCategory = 'py-er-custom-metric';

        $scope.addNewCustomMetric = function() {
            if (!$scope.uiState.customEvaluationMetrics) {$scope.uiState.customEvaluationMetrics = [];};
            $scope.uiState.customEvaluationMetrics.push($scope.getNewMetricTemplate());
        }

        $scope.fireCustomMetricAddedWT1Event = function() {
            WT1.event("clicked-item", {"item-id": 'evaluation-add-custom-metric'});
        };

        $scope.fireCustomMetricRemovedWT1Event = function() {
            WT1.event("clicked-item", {"item-id": 'evaluation-remove-custom-metric'});
        };

        $scope.toggleFoldable = function(index) {
            if($scope.uiState.customEvaluationMetrics[index]){
                $scope.uiState.customEvaluationMetrics[index].$foldableOpen = !$scope.uiState.customEvaluationMetrics[index].$foldableOpen
            }
        }

        $scope.mayWriteCustomMetrics = $scope.mayWriteSafeCode();

        // Define show text drift warning
        function updateShowTextDriftWarning() {
            const handling = $scope.desc.dataDriftColumnHandling;
            const featuresInInput = Object.entries($scope.modelDetails?.preprocessing?.per_feature || {})
                .filter(([, column]) => column.role === "INPUT");

            $scope.showTextDriftWarning = ! featuresInInput
                .some(([columnName, column]) =>
                    (!(columnName in handling) && column.type === "TEXT") ||
                    (columnName in handling && handling[columnName].enabled && handling[columnName].handling === "TEXT")
                );
        };
        updateShowTextDriftWarning()

        // Define show image drift warning and message
        function updateShowImageDriftWarningAndMessage() {
            const isNotDssManaged = $scope.desc.savedModelType !== "DSS_MANAGED";
            const hasNoImageFolder = !$scope.recipe.inputs.data?.items.length;

            const handling = $scope.desc.dataDriftColumnHandling;
            const handleImageFeatures = Object.entries($scope.modelDetails?.preprocessing?.per_feature || {})
                .filter(([, column]) => column.role === "INPUT")
                .some(([columnName, column]) => (!(columnName in handling) && column.type === "IMAGE") ||
                                                (columnName in handling && handling[columnName].enabled
                                                && handling[columnName].handling === "IMAGE"))

            const hasImageFeaturesInInput = Object
                .entries($scope.modelDetails?.preprocessing?.per_feature || {})
                .filter(([columnName, column]) =>
                    column.role === "INPUT" && column.type === "IMAGE"
                    && ( // Check if overridden by handling
                        !(columnName in handling)
                        || (handling[columnName].enabled && ["IMAGE","AUTO"].includes(handling[columnName].handling))
                    )
                ).length > 0;

            if (isNotDssManaged) {
                $scope.showImageDriftWarning = true;
                $scope.imageDriftWarningMessage = "Image drift will not be computed: " +
                                                  "Only model type DSS_MANAGED is supported (current: " +
                                                  $scope.desc.savedModelType + ").";
            } else if (! hasImageFeaturesInInput && ! handleImageFeatures) {
                $scope.showImageDriftWarning = true;
                $scope.imageDriftWarningMessage = "Image drift will not be computed: No image columns are handled " +
                                                  "as image features or detected in the model input.";
            } else if (! handleImageFeatures){
                $scope.showImageDriftWarning = true;
                $scope.imageDriftWarningMessage = "Image drift will not be computed: " +
                                                  "No columns are handled as image features.";
            } else {
                $scope.showImageDriftWarning = hasNoImageFolder;
                $scope.imageDriftWarningMessage = "Image drift will not be computed: " +
                                                  "No Evaluation Data folder is found.";
            }
        }
        updateShowImageDriftWarningAndMessage();

        $scope.onFeatureHandlingChange = function(newHandling){
             $scope.desc.dataDriftColumnHandling = newHandling;
             updateShowTextDriftWarning();
             updateShowImageDriftWarningAndMessage();
        }

        // Specify if a column can drift
        $scope.isColumnEligibleForDrift = function(feature) {
            const perFeature = $scope.modelDetails?.preprocessing?.per_feature;
            if (!perFeature) return true;
            return perFeature[feature]?.role === "INPUT" || perFeature[feature]?.role === "TARGET";
        };

    });

    app.controller("StandaloneEvaluationRecipeEditor", function($scope, $controller, $q, Assert, SamplingData,
                                                                CodeMirrorSettingService, PMLSettings, ModelLabelUtils, $stateParams, DatasetUtils,
                                                                CreateModalFromTemplate, ModelEvaluationUtils, ClipboardReadWriteService,
                                                                StringUtils, WT1) {
        $controller("_BaseMLRecipeEditor", {$scope:$scope});
        $controller("_MLRecipeWithoutOutputSchemaController", {$scope:$scope});
        $controller("_RecipeWithEngineBehavior", {$scope:$scope});
        $controller("EvaluationLabelUtils", {$scope:$scope});

        $scope.enableAutoFixup();

        $scope.uiState = {
            thresholdMode: undefined
        };

        // Update 'autoOptimizeThreshold' & 'thresholdOptimizationMetric' on recipe desc based 'uiState.thresholdMode' change
        $scope.$watch('uiState.thresholdMode', ()=> {
            if(!$scope.uiState.thresholdMode) {
                return;
            }
            if($scope.uiState.thresholdMode != 'MANUAL' && $scope.desc.metricParams) {
                $scope.desc.autoOptimizeThreshold = true;
                $scope.desc.metricParams.thresholdOptimizationMetric = $scope.uiState.thresholdMode;
            } else {
                $scope.desc.autoOptimizeThreshold = false;
            }
        });

        // Set 'uiState.thresholdMode' based on 'autoOptimizeThreshold' & 'thresholdOptimizationMetric' in recipe desc
        $scope.$watch('desc', ()=> {
            if(!$scope.desc) {
                return;
            }
            if($scope.desc.autoOptimizeThreshold && $scope.desc.metricParams) {
                $scope.uiState.thresholdMode = $scope.desc.metricParams.thresholdOptimizationMetric || 'F1';
            } else {
                $scope.uiState.thresholdMode = 'MANUAL';
            }
        });

        $scope.codeMirrorSettingService = CodeMirrorSettingService;

        $scope.selectedEngine = function(){
            return $scope.recipeStatus && $scope.recipeStatus.selectedEngine ? $scope.recipeStatus.selectedEngine.type : undefined;
        };

        $scope.hooks.onRecipeLoaded = function(){
             $scope.hooks.updateRecipeStatus();
        };

        $scope.hooks.getPayloadData = function(){
            return angular.toJson($scope.desc);
        };

        $scope.hooks.updateRecipeStatus = function() {
            var deferred = $q.defer();
            var payload = $scope.hooks.getPayloadData();
            $scope.updateRecipeStatusBase(false, payload).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.mayUseContainer = function() {
            return $scope.selectedEngine() == 'DSS';
        };

        $scope.willUseSpark = function(){
            return $scope.selectedEngine() == 'SPARK';
        };

        $scope.hasReferenceDataset = function() {
            return "reference" in $scope.recipe.inputs && $scope.recipe.inputs['reference'].items && $scope.recipe.inputs['reference'].items.length > 0;
        }

        function computablesMapChanged() {
            if (!$scope.computablesMap) return;
            $scope.recipe.inputs['main'].items.forEach(function(inp){
                var computable = $scope.computablesMap[inp.ref];
                if (computable.type == "DATASET") {
                    if (computable.dataset.schema) {
                        $scope.inputColumns = computable.dataset.schema.columns.map(_ => _.name);
                        $scope.initialDataDriftColumns = computable.dataset.schema.columns;
                    }
                }
            });
        };
        $scope.$on('computablesMapChanged', computablesMapChanged);
        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$scope.recipe.projectKey;
        DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey).then(
            _ => computablesMapChanged()
        );

        $scope.isBinaryClassification = function(){
            return $scope.desc.predictionType == "BINARY_CLASSIFICATION";
        };

        $scope.isMulticlass = function(){
            return $scope.desc.predictionType == "MULTICLASS";
        };

        $scope.isRegression = function(){
            return $scope.desc.predictionType == "REGRESSION";
        };

        $scope.isProbaAware = function(){
            return $scope.desc.isProbaAware;
        };

        $scope.buildNewFeature = function() {
            return {name:'', type:'NUMERIC', role:'INPUT'};
        };

        $scope.thresholdOptimizationMetrics = PMLSettings.task.thresholdOptimizationMetrics;

        const baseCanSave = $scope.canSave;
        $scope.canSave = function() {
            return ModelLabelUtils.validateLabels($scope.desc) && baseCanSave();
        };

        $scope.isInputRoleAvailableForPayload = function(role) {
            // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
            return true; // by default an input role is available to fill/select
        }

        $scope.copyClasses = function() {
            ClipboardReadWriteService.writeItemsToClipboard($scope.desc.classes);
            $scope.$applyAsync()
        }

        $scope.pasteClasses = async function() {
            try {
                const classes = await ClipboardReadWriteService.readItemsFromClipboard(
                    $scope,
                    "The following classes will be used:"
                );
                $scope.desc.classes = classes
                $scope.$applyAsync()
            } catch (error) {
                // do nothing
            }
        }

        $scope.copyProbas = function() {
            let items = []
            $scope.desc.probas.forEach(proba => {
                let [key, value] = [proba.key, proba.value]
                key = key ? key.trim() : ""
                value = value ? value.trim() : ""
                if (key || value) {
                    items.push(key + "," + value)
                }
            })
            ClipboardReadWriteService.writeItemsToClipboard(items);
            $scope.$applyAsync()
        }

        $scope.pasteProbas = async function () {
            try {
                const pastedProbas = await ClipboardReadWriteService.readItemsFromClipboard(
                    $scope,
                    "The following class values & probability columns will be used:"
                );
                if (pastedProbas && Array.isArray(pastedProbas) && pastedProbas.length > 0) {
                    $scope.desc.probas = processPastedProbas(pastedProbas);
                    $scope.$applyAsync();
                }
            }
            catch (error) {
                // do nothing
            }
        }

        function processPastedProbas(pastedProbas, separator = ",") {
            let keys = []
            let values = []
            pastedProbas.forEach(proba => {
                let [key, value] = proba.split(separator)
                keys.push(key ? key.trim() : null)
                values.push(value ? value.trim() : null)
            })

            if (values.every(v => v === null)) {
                return keys.map(key => ({key: key, value: "proba_" + key}))
            }
            return keys.map((key, index) => ({key: key, value: values[index]}))
        }

        const baseSaveHook = $scope.hooks.save;
        $scope.hooks.save = function() {
            const saveWarning = function(data, willFail) {
                const deferred = $q.defer();
                CreateModalFromTemplate('/templates/recipes/standalone_evaluation-recipe-warning-modal.html', $scope, null, function(modalScope) {
                    modalScope.willFail = willFail;
                    modalScope.acceptDeferred = deferred;
                    modalScope.messages = data;
                    modalScope.saveAnyway = function() {
                        baseSaveHook().then((baseSaveHookResult) => {
                            modalScope.acceptDeferred.resolve(baseSaveHookResult);
                            modalScope.acceptDeferred = null;
                            modalScope.dismiss();
                        }, function(error) {
                            if(modalScope.acceptDeferred) {
                                modalScope.acceptDeferred.reject(error);
                            }
                            modalScope.acceptDeferred = null;
                            modalScope.dismiss();
                        })
                    }
                    modalScope.$on("$destroy",function() {
                        if(modalScope.acceptDeferred) {
                            modalScope.acceptDeferred.reject();
                        }
                        modalScope.acceptDeferred = null;
                    });
                });
                return deferred.promise;
            };

            const desc = $scope.desc;
            const validLabels = $scope.inputColumns || [];
            const isClassification = PMLSettings.task.isClassification(desc.predictionType);
            const isTargetValid = validLabels.includes(desc.targetVariable);
            const useProba = desc.isProbaAware;
            let warningMessages = [];
            let willFail = false;
            if (desc.hasModel){
                if (!isClassification && (!$scope.inputColumns || !validLabels.includes(desc.predictionVariable))) {
                    warningMessages.push('The prediction column does not exist.');
                    willFail = true;
                }
                if (isClassification && !useProba && (!$scope.inputColumns || !validLabels.includes(desc.predictionVariable))) {
                    warningMessages.push('You did not specify where the prediction information is given. This can be done by specifying a column containing the prediction or a mapping of columns and class probabilities.');
                    willFail = true;
                }
                if (desc.predictionType === "BINARY_CLASSIFICATION" && useProba && (!desc.probas || desc.probas.length != 2)) {
                    warningMessages.push('Because the prediction type is "Two-class classification", you have to specify the mapping between exactly two classes and two probability columns for this recipe to run successfully.');
                    willFail = true;
                }
                if (desc.predictionType === "BINARY_CLASSIFICATION" && useProba && !validLabels.includes(desc.predictionVariable) && desc.dontComputePerformance && desc.autoOptimizeThreshold) {
                    warningMessages.push('The computation of performance metrics is necessary to optimize the threshold. If you run this recipe, a default threshold of 0.5 will be used.');
                }
                if (!isTargetValid && !desc.dontComputePerformance) {
                    warningMessages.push('The labels column does not exist. Either specify a valid column, or if there is no suitable column in the dataset, you can still run the recipe by checking "Skip performance metrics computation".');
                    willFail = true;
                }
                if (isClassification && validLabels.includes(desc.predictionVariable) && useProba && desc.probas && desc.probas.length) {
                    warningMessages.push('Since the model is proba-aware, the prediction column will be ignored and predictions will instead be computed from the probabilities.');
                }
                if (desc.weightsVariable && !validLabels.includes(desc.weightsVariable)) {
                    warningMessages.push('The weights column does not exist.');
                    willFail = true;
                }
                if (isClassification && !desc.dontComputePerformance && useProba && (!desc.probas || !desc.probas.length)) {
                    warningMessages.push('You have to specify the mapping between classes and proba columns for this recipe to run successfully, or uncheck "Proba aware".');
                    willFail = true;
                }
            } else if (!$scope.hasReferenceDataset()) {
                warningMessages.push('Without a reference dataset nor a model, no metrics will be computed.');
            }

            if (desc.features.filter(feat => !["NUMERIC", "CATEGORY", "TEXT", "IMAGE"].includes(feat.type)).length){
                warningMessages.push('In your advanced settings, one of your feature type is non Numeric/Categorical/Text/Image. Only those types are supported : any other type will be considered as Categorical.');
            }

            if (warningMessages.length !== 0) {
                return saveWarning(warningMessages, willFail)
            }
            return baseSaveHook();
        };

        ModelEvaluationUtils.synchronizeLimitSamplingUiStateAndDesc($scope);

        // Only Numeric and Categorical feature types are supported but Vector, Text and Image were available too.
        // We want to 'forbid' their usage so we remove them from the UI as much as possible
        $scope.initialFeatureTypes = new Set($scope.desc.features.map(feature => feature.type));

        const hasInitialHasModel = $scope.desc.hasModel != undefined;
        $scope.uiState.hasModel = hasInitialHasModel ? $scope.desc.hasModel : true;
        $scope.$watch('uiState.hasModel', function(hasModel) {
            if (!hasInitialHasModel) {
                $scope.desc.hasModel = hasModel ? undefined : false;
            } else {
                $scope.desc.hasModel = hasModel;
            }
        });

        const hasInitialTreatPerfMetricsFailureAsError = $scope.desc.treatPerfMetricsFailureAsError !== undefined;
        $scope.uiState.treatPerfMetricsFailureAsError = hasInitialTreatPerfMetricsFailureAsError ? $scope.desc.treatPerfMetricsFailureAsError : true;

        $scope.$watch('uiState.treatPerfMetricsFailureAsError', function(treatPerfMetricsFailureAsError) {
            if (hasInitialTreatPerfMetricsFailureAsError) {
                $scope.desc.treatPerfMetricsFailureAsError = treatPerfMetricsFailureAsError;
            } else {
                $scope.desc.treatPerfMetricsFailureAsError = treatPerfMetricsFailureAsError === true ? undefined : false;
            }
        })

        const recipeCreatedBeforeCustomEvaluationMetrics = $scope.desc.customEvaluationMetrics === undefined; // it was created before custom eval metrics existed
        $scope.uiState.customEvaluationMetrics = $scope.desc.customEvaluationMetrics || [];
        $scope.$watch('uiState.customEvaluationMetrics', function(customEvaluationMetrics) {
            const hasCustomMetrics = !!(customEvaluationMetrics && customEvaluationMetrics.length);
            if (recipeCreatedBeforeCustomEvaluationMetrics && !hasCustomMetrics) {  // we don't dirty legacy recipes
                $scope.desc.customEvaluationMetrics = undefined;
            } else {
                $scope.desc.customEvaluationMetrics = customEvaluationMetrics;
            }
        }, true);

        let customMetricDefaultCode = `def score(eval_df, ref_df, model_parameters, **kwargs):
    """
    Custom scoring function.
    Must return a float quantifying the estimator prediction quality.
    - eval_df: pandas DataFrame. Sample from the evaluation dataset, using Sampling method (evaluation), normalized.
    - ref_df: pandas DataFrame. Sample from the reference dataset, using Sampling method (reference), normalized.
    - model_parameters: object. Parameters from the 'Model' section. Has the following fields:
        - has_model: bool
        - dont_compute_performance: bool
        - prediction_column_name: str
        - prediction_type: str
        - is_proba_aware: bool
        - proba_definition: list
        - target_column_name: str
        - weight_column_name: str
        - user_defined_classes:  List[str]
        and function:
        - get_preds(df: pd.DataFrame) -> pd.Series:
            get predictions from a reference or evaluation dataset.
            if the Model is 'Proba aware', the predictions are re-computed from the probabilities
    """
    return 0.5`;

        $scope.getNewMetricTemplate = function() {
            const name = StringUtils.transmogrify("Custom Evaluation Metric #" + ($scope.uiState.customEvaluationMetrics.length + 1).toString(),
                $scope.uiState.customEvaluationMetrics.map(a => a.name),
                function(i){return "Custom Evaluation Metric #" + (i+1).toString() }
            );

            const template = {
                name,
                metricCode: customMetricDefaultCode,
                description: "",
                greaterIsBetter: true,
                needsProbability: false,
                type: 'STANDALONE_EVALUATION_RECIPE',
                $foldableOpen: true
            };
            return template;
        }

        $scope.snippetCategory = 'py-ser-custom-metric';

        $scope.addNewCustomMetric = function() {
            if (!$scope.uiState.customEvaluationMetrics) {$scope.uiState.customEvaluationMetrics = [];};
            $scope.uiState.customEvaluationMetrics.push($scope.getNewMetricTemplate());
        }

        $scope.fireCustomMetricAddedWT1Event = function() {
            WT1.event("clicked-item", {"item-id": 'standalone-evaluation-add-custom-metric'});
        };

        $scope.fireCustomMetricRemovedWT1Event = function() {
            WT1.event("clicked-item", {"item-id": 'standalone-evaluation-remove-custom-metric'});
        };

        $scope.toggleFoldable = function(index) {
            if($scope.uiState.customEvaluationMetrics[index]){
                $scope.uiState.customEvaluationMetrics[index].$foldableOpen = !$scope.uiState.customEvaluationMetrics[index].$foldableOpen
            }
        }

        $scope.mayWriteCustomMetrics = $scope.mayWriteSafeCode();

        // Define show text drift warning
        function updateShowTextDriftWarningAndMessage() {
            // Extract column handled under the Feature handling section
            const featuresHandling = $scope.desc.dataDriftColumnHandling || {};
            // Extract Text column declared under the Feature Type section
            const inputTextFeatures = ($scope.desc.features || [])
                .filter(column => column.type === "TEXT" && column.role === "INPUT");
            // Extract Text column in the dataset schema
            const textDriftColumns = ($scope.initialDataDriftColumns || [])
                .filter(column => column.meaning == "FreeText")

            const overrideAllInputTextFeatures = inputTextFeatures
                .every( column =>
                    column.name in featuresHandling
                    && ( ! featuresHandling[column.name].enabled
                        || ! ["TEXT","AUTO"].includes(featuresHandling[column.name].handling)
                       )
                );
            const overrideAllTextDriftColumns = textDriftColumns
                .every(column =>
                    column.name in featuresHandling
                    && ( ! featuresHandling[column.name].enabled
                        || ! ["TEXT","AUTO"].includes(featuresHandling[column.name].handling)
                        )
                );

            $scope.showTextDriftWarning = overrideAllTextDriftColumns && overrideAllInputTextFeatures
            $scope.textDriftWarningMessage = "Text drift will not be computed: No column was found of " +
                                              "type \"input : Text\"."
        };
        $scope.$watch('initialDataDriftColumns', updateShowTextDriftWarningAndMessage, true)
        updateShowTextDriftWarningAndMessage()

        // Define show image drift warning
        function updateShowImageDriftWarningAndMessage() {
            const hasNoImageFolder = !$scope.recipe.inputs.data?.items.length;
            const hasNoReferenceImageFolder = !$scope.recipe.inputs.referenceData?.items.length;

            const featuresHandling = $scope.desc.dataDriftColumnHandling || {};
            const inputImageFeatures = ($scope.desc.features || [])
                .filter(column => column.type === "IMAGE" && column.role === "INPUT");

            const overrideAllInputImageFeatures = inputImageFeatures.every(column => column.name in featuresHandling
                && ( ! featuresHandling[column.name].enabled
                || ! ["IMAGE","AUTO"].includes(featuresHandling[column.name].handling)));
            const handleNoImageFeatures = ! Object.values(featuresHandling)
                .some(handling => handling.enabled && handling.handling === "IMAGE");

            if (handleNoImageFeatures && overrideAllInputImageFeatures){
                $scope.showImageDriftWarning = true;
                $scope.imageDriftWarningMessage = "Image drift will not be computed: " +
                                                  "No columns are handled as image features.";
            } else if (hasNoImageFolder){
                $scope.showImageDriftWarning = true;
                $scope.imageDriftWarningMessage = "Image drift will not be computed: " +
                                                  "No Evaluation Data folder is found.";
            }else {
                 $scope.showImageDriftWarning = hasNoReferenceImageFolder;
                 $scope.imageDriftWarningMessage = "Image drift will not be computed: " +
                                                   "No Reference Data folder is found.";
            }
        }
        $scope.$watch('recipe.inputs.data', updateShowImageDriftWarningAndMessage, true);
        $scope.$watch('recipe.inputs.referenceData', updateShowImageDriftWarningAndMessage, true);
        updateShowImageDriftWarningAndMessage();

        $scope.onFeatureTypeChange = function(){
            updateShowTextDriftWarningAndMessage();
            updateShowImageDriftWarningAndMessage();
        }

        $scope.onFeatureHandlingChange= function(newHandling){
            $scope.desc.dataDriftColumnHandling = newHandling;
            updateShowTextDriftWarningAndMessage();
            updateShowImageDriftWarningAndMessage();
         }

        $scope.isColumnEligibleForDrift = function(feature) { return true; };
    });


    app.directive('scoringColumnsFilter', function(Assert) {
        return {
            restrict: 'AE',
            replace: false,
            templateUrl: "/templates/recipes/scoring-column-filter.html",
            link: function(scope) {
                Assert.inScope(scope, 'preparedInputSchema');

                scope.uiState = scope.uiState || {};
                scope.selectionState = {};

                scope.desc.keptInputColumns = scope.desc.keptInputColumns || [];

                scope.columns = scope.preparedInputSchema.columns
                scope.filteredColumns = scope.columns;

                scope.updateFilteredColumnsSelection = function() {
                    scope.desc.keptInputColumns = scope.columns.filter(function(col){return col.$selected}).map(function(col){return col.name});
                    updateSelectionUiState();
                };

                scope.updateColumnsFilter = function(query) {
                    if (!query || !query.trim().length) {
                        scope.filteredColumns = scope.columns;
                    } else {
                        var lowercaseQuery = query.toLowerCase();
                        scope.filteredColumns = scope.columns.filter(function(col) {col.$filtered = !(col.name.toLowerCase().indexOf(lowercaseQuery) >= 0 || col.type.toLowerCase() == lowercaseQuery); return !col.$filtered});
                    }
                    updateSelectionUiState();
                };

                scope.updateSelectAllColumns = function(selectAll) {
                    scope.filteredColumns.forEach(function(col){col.$selected = col.$filtered ? col.$selected : selectAll});
                    scope.updateFilteredColumnsSelection();
                };

                var updateSelectionUiState = function() {
                    scope.selectionState.all = true;
                    scope.selectionState.any = false;
                    scope.filteredColumns.forEach(function(col) {
                        scope.selectionState.any = scope.selectionState.any || col.$selected;
                        scope.selectionState.all = scope.selectionState.all && col.$selected;
                    });
                };

                scope.columns.forEach(function(col) {
                    col.$selected = scope.desc.keptInputColumns.indexOf(col.name) >= 0;
                });
                updateSelectionUiState();
            }
        };
    });

app.component('inputDataDriftColumnHandling', {
    templateUrl: '/templates/recipes/data-drift-column-handling-list.html',
    bindings: {
        treatDataDriftColumnHandling: '=',
        columns: '<',
        driftColumnHandling: '=',
        evaluationDatasetType: '<',
        isColumnEligibleForDrift: '&',
        onChange: '&'
    },
    controller: function(ModelEvaluationUtils){
        const $ctrl = this;

        $ctrl.selectionState = {};
        $ctrl.displayedColumns = [];
        $ctrl.uiState = {};

        $ctrl.$onChanges = function(changes) {
            if (changes.columns && !Object.keys(changes.columns.previousValue).length && Object.keys(changes.columns.currentValue).length) {
                 const newColumns = $ctrl.columns.map(col => (
                    {...col,
                    $handling: $ctrl.driftColumnHandling && $ctrl.driftColumnHandling[col.name] ? $ctrl.driftColumnHandling[col.name].handling : "AUTO",
                    $enabled: $ctrl.driftColumnHandling && $ctrl.driftColumnHandling[col.name] ? $ctrl.driftColumnHandling[col.name].enabled : true
                    }));
                 if ($ctrl.evaluationDatasetType === ModelEvaluationUtils.API_NODE_LOGS_EVALUATION_DATASET_TYPE) {
                     $ctrl.displayedColumns = newColumns
                         .filter(col => col.name.startsWith(ModelEvaluationUtils.API_NODE_FEATURE_PREFIX))
                         .map(col => ({...col, $formattedName: col.name.split(ModelEvaluationUtils.API_NODE_FEATURE_PREFIX)[1]}))
                 } else if ($ctrl.evaluationDatasetType === ModelEvaluationUtils.CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE) {
                     $ctrl.displayedColumns = newColumns
                         .filter(col => col.name.startsWith(ModelEvaluationUtils.CLOUD_API_NODE_FEATURE_PREFIX))
                         .map(col => ({...col, $formattedName: col.name.split(ModelEvaluationUtils.CLOUD_API_NODE_FEATURE_PREFIX)[1]}))
                 } else {
                     $ctrl.displayedColumns = newColumns;
                 }

                $ctrl.handlingOptions = [['AUTO', 'Auto'], ['NUMERICAL', 'Numerical'], ['CATEGORICAL', 'Categorical'], ['TEXT', 'Text'], ['IMAGE', 'Image']];
                updateSelectionUiState();
            }
        }

        $ctrl.updateColumnsFilter = function(query) {
            if (!query || !query.trim().length) {
                $ctrl.displayedColumns = $ctrl.displayedColumns.map(col => ({...col, $filtered: false}));
            } else {
                const lowercaseQuery = query.toLowerCase();
                $ctrl.displayedColumns = $ctrl.displayedColumns.map(col => ({...col, $filtered: !(col.name.toLowerCase().indexOf(lowercaseQuery) >= 0)}));
            }
            updateSelectionUiState();
        };

        $ctrl.updateSelectAllColumns = function(selectAll) {
            $ctrl.displayedColumns = $ctrl.displayedColumns.map(col => ({...col, $enabled: !col.$filtered ? selectAll : col.$enabled}));
            $ctrl.displayedColumns.forEach($ctrl.updateDataDriftColumnHandling);
        };

        $ctrl.getKeptColumnsLength = function() {
            return $ctrl.displayedColumns.filter(col => col.$enabled).length;
        }

        $ctrl.getNbColumns = function() {
            return $ctrl.displayedColumns.length;
        }

        $ctrl.anyMatchingColumns = function() {
            return $ctrl.displayedColumns.some(col => !col.$filtered);
        }

        $ctrl.updateDataDriftColumnHandling = function(column) {
            if (!column.$enabled || column.$handling !== "AUTO"){
                $ctrl.driftColumnHandling = {...($ctrl.driftColumnHandling || {}), [column.name]: {enabled : column.$enabled, handling: column.$handling}}
            } else {
                delete $ctrl.driftColumnHandling[column.name];
            }
            if ($ctrl.onChange){
                $ctrl.onChange({ newHandling: $ctrl.driftColumnHandling });
            }
        }

        function updateSelectionUiState() {
            $ctrl.selectionState.any = $ctrl.displayedColumns.some(col => col.$enabled);
            $ctrl.selectionState.all = $ctrl.displayedColumns.every(col => col.$enabled);
        }

        $ctrl.canColumnDrift = function(feature) {
            return $ctrl.isColumnEligibleForDrift({ feature: feature });
        };
    }
});

    app.directive('metricsFilter', function(METRICS){
        return {
            templateUrl: '/templates/recipes/metrics-filter.html',
            link: function(scope, element){

                scope.uiState = scope.uiState || {};
                scope.metricsSelectionState = {};
                scope.desc.metrics = scope.desc.metrics || [];

                var metrics;
                if (scope.isBinaryClassification()) {
                    metrics = METRICS.BINARY_CLASSIFICATION;
                } else if (scope.isMulticlass()) {
                    metrics = METRICS.MULTICLASS;
                } else if (scope.isTimeseriesForecasting() && !scope.desc.skipScoring) {
                    metrics = METRICS.TIMESERIES;
                } else if (scope.isTimeseriesForecasting() && scope.desc.skipScoring) {
                    metrics = METRICS.TIMESERIES_SKIP_SCORING;
                }
                else if(scope.isRegression()){
                    metrics = METRICS.REGRESSION;
                } else if(scope.isCausalPrediction()) {
                    metrics = METRICS.CAUSAL;
                } else {
                    metrics = [];
                }
                scope.metrics = metrics.map(function(m){return {name: m}; });

                scope.updateFilteredMetricsSelection = function() {
                    scope.desc.metrics = scope.metrics.filter(function(m){return m.$selected}).map(function(m){return m.name});
                    updateSelectionUiState();
                };

                var updateSelectionUiState = function() {
                    scope.metricsSelectionState.all = true;
                    scope.metricsSelectionState.any = false;
                    scope.metrics.forEach(function(m) {
                        scope.metricsSelectionState.any = scope.metricsSelectionState.any || m.$selected;
                        scope.metricsSelectionState.all = scope.metricsSelectionState.all && m.$selected;
                    });
                };

                scope.metrics.forEach(function(m) {
                    m.$selected = scope.desc.metrics.indexOf(m.name) >= 0;
                });


                scope.updateSelectedCustomMetrics = function() {
                    scope.desc.customMetrics = scope.possibleCustomMetrics.filter(function(m){return m.$selected}).map(function(m){return m.name});
                }

                function setCustomMetrics() {
                    scope.desc.customMetrics = scope.desc.customMetrics || [];
                    scope.desc.possibleCustomMetrics = scope.desc.possibleCustomMetrics || [];
                    scope.possibleCustomMetrics = scope.desc.possibleCustomMetrics.map(function(m){return {name: m}; });
                    scope.possibleCustomMetrics.forEach(function(m) {
                        m.$selected = scope.desc.customMetrics.indexOf(m.name) >= 0;
                    });
                    scope.updateSelectedCustomMetrics();
                }


                setCustomMetrics();
                updateSelectionUiState();
                scope.$on('customMetricsUpdated', function() {
                    setCustomMetrics();
                    updateSelectionUiState();
                });

                if (scope.isTimeseriesForecasting() && scope.isMultipleTimeseries()) {
                    scope.$watchGroup(["desc.computePerTimeseriesMetrics", "desc.skipScoring"], function(nv, ov){
                        const availableMetrics = nv[0] ?
                            nv[1] ? METRICS.MULTIPLE_TIMESERIES_SKIP_SCORING : METRICS.MULTIPLE_TIMESERIES
                            : nv[1] ? METRICS.TIMESERIES_SKIP_SCORING : METRICS.TIMESERIES;
                        const allMetricsExisting = scope.metrics.every(x => availableMetrics.includes(x.name)) &&
                                                                                                scope.metrics.length === availableMetrics.length;
                        if (!allMetricsExisting) { // Only if there are additional or less metrics in available metrics
                            const prevSelectedMetrics = scope.metrics.filter(m => m.$selected).map(m => m.name);
                            scope.metrics = availableMetrics.map(m => ({ name: m, $selected: prevSelectedMetrics.includes(m) }));
                            scope.updateFilteredMetricsSelection();
                        }
                    })
                }

                if (scope.isCausalPrediction()) {
                    scope.$watch("desc.computePropensity", function(nv, ov){
                        const availableMetrics = nv ? METRICS.CAUSAL
                            : METRICS.CAUSAL_WITHOUT_PROPENSITY;
                        const allMetricsExisting = scope.metrics.every(x => availableMetrics.includes(x.name)) &&
                                                                        scope.metrics.length === availableMetrics.length;
                        if (!allMetricsExisting) { // Only if there are additional or less metrics in available metrics
                            const prevSelectedMetrics = scope.metrics.filter(m => m.$selected).map(m => m.name);
                            scope.metrics = availableMetrics.map(m => ({ name: m, $selected: prevSelectedMetrics.includes(m) }));
                            scope.updateFilteredMetricsSelection();
                        }
                    })
                }
            }
        };
    });

    app.directive('outputsFilter', function(){
        return {
            templateUrl: '/templates/recipes/outputs-filter.html',
            link: function(scope, element){

                scope.uiState = scope.uiState || {};
                scope.outputSelectionState = {};

                var outputs;
                if (scope.isRegression()) {
                    outputs = ["error", "error_decile", "abs_error_decile", "relative_error"];
                } else {
                    outputs = ["prediction_correct"];
                }
                scope.outputs = outputs.map(function(o){return {name: o}; });

                scope.updateFilteredOutputsSelection = function() {
                    scope.desc.outputs = scope.outputs.filter(function(o){return o.$selected}).map(function(o){return o.name});
                    updateSelectionUiState();
                };

                var updateSelectionUiState = function() {
                    scope.outputSelectionState.all = true;
                    scope.outputSelectionState.any = false;
                    scope.outputs.forEach(function(o) {
                        scope.outputSelectionState.any = scope.outputSelectionState.any || o.$selected;
                        scope.outputSelectionState.all = scope.outputSelectionState.all && o.$selected;
                    });
                };

                scope.outputs.forEach(function(o) {
                    o.$selected = scope.desc.outputs.indexOf(o.name) >= 0;
                });


                updateSelectionUiState();
            }
        };
    });

    app.service('ModelLabelUtils', function() {
        function validateLabels(labeledItem) {
            if (!labeledItem || !labeledItem.labels) {
                return true;
            }
            const existingLabels = new Set();
            for(const label of labeledItem.labels) {
                if (!label.key || existingLabels.has(label.key)) {
                    return false;
                }
                existingLabels.add(label.key);
            }
            return true;
        }

        return { validateLabels };
    });
})();
