(function(){
'use strict';
const app = angular.module('dataiku.retrieval-augmented-llm', ['dataiku.ml.report', 'dataiku.lambda']);

app.controller("RetrievalAugmentedLLMSavedModelController", function($scope, $rootScope, $state, $controller, $stateParams, $q, DataikuAPI, Dialogs, DKUtils, ActiveProjectKey, FullModelLikeIdUtils, SmartId, Debounce) {
    $controller("_SavedModelReportController", {$scope});
    $scope.uiState = $scope.uiState || {};

    DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.fillVersionSelectorStuff(data);
    }).catch(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").then(function({data}) {
        $scope.availableAugmentableLLMs = data.identifiers.filter(id => id.type !== "RETRIEVAL_AUGMENTED" && id.type !== "SAVED_MODEL_AGENT");
    }).catch(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").then(function({data}) {
        $scope.availableEmbeddingModels = data.identifiers;
    }).catch(setErrorInScope.bind($scope));

    const currentVersionId = FullModelLikeIdUtils.parse($stateParams.fullModelId).versionId;
    $scope.currentVersionId = currentVersionId;

    $scope.$watch("savedModel", function(nv) {
        if (!nv) return;

        $scope.currentVersionIdx = nv.inlineVersions.findIndex(v => v.versionId === currentVersionId);
        if ($scope.currentVersionIdx < 0) {
            return;
        }
        $scope.currentlySavedInlineModel = angular.copy(nv.inlineVersions[$scope.currentVersionIdx]);
        $scope.currentVersionData = nv.inlineVersions[$scope.currentVersionIdx];
        $scope.uiState.ragllmSettings = $scope.currentVersionData.ragllmSettings;

        $scope.getMetadataColumns = function() {
            const schema = $scope.retrievableKnowledge?.metadataColumnsSchema ?? [];
            const multimodalColumn = $scope.retrievableKnowledge?.multimodalColumn;

            return schema.map(sc => sc.name)
                .filter(col => col !== multimodalColumn);
        };

        DataikuAPI.retrievableknowledge.getFullInfo(ActiveProjectKey.get(), SmartId.create($scope.uiState.ragllmSettings.kbRef, $stateParams.projectKey)).then(function({ data }) {
            $scope.rkFullInfo = data;
            $scope.retrievableKnowledge = data.retrievableKnowledge;
            if (data.retrievableKnowledge.vectorStoreType === 'VERTEX_AI_GCS_BASED') {
                $scope.uiState.ragllmSettings.enforceDocumentLevelSecurity = false;
            }
            $scope.metadataColumns = $scope.getMetadataColumns();
        }).catch(setErrorInScope.bind($scope));

        DataikuAPI.retrievableknowledge.getCurrentVersionInfo($stateParams.projectKey, $scope.uiState.ragllmSettings.kbRef)
            .then(function({ data }) {
                $scope.embeddingRecipeDesc = data.embeddingRecipeParams;
                $scope.isRkBuilt = data.isBuilt;
            }).catch(setErrorInScope.bind($scope));
    });

    // to trigger change detection in child components
    $scope.currentVersionUpdateCounter = 0;
    $scope.$watch('currentVersionData', Debounce().withDelay(200, 200).wrap(() => $scope.currentVersionUpdateCounter++), true);

    $scope.save = function() {
        if (!$scope.isDirty()) {
            return $q.when('retrieval-augmented LLM not dirty');
        }

        const deferred = $q.defer();
        const saveAfterConflictCheck = function() {
            DataikuAPI.savedmodels.retrievalAugmentedLLMs.save($scope.savedModel, $scope.currentVersionId).success(function(data) {
                $scope.savedModel = data;
                $scope.currentlySavedInlineModel = angular.copy($scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
                deferred.resolve('retrieval-augmented LLM saved');
            }, () => deferred.reject());
        };

        DataikuAPI.savedmodels.retrievalAugmentedLLMs.checkSaveConflict($scope.savedModel).success(function(conflictResult) {
            if (!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope, conflictResult).then(
                    function(resolutionMethod) {
                        if (resolutionMethod === 'erase') {
                            saveAfterConflictCheck();
                        } else if (resolutionMethod === 'ignore') {
                            deferred.reject();
                            DKUtils.reloadState();
                        }
                    }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));

        return deferred.promise;
    };
    $rootScope.saveRetrievalAugmentedModel = $scope.save;

    $scope.saveIgnoringQuickTestQuery = function() {
        if (!$scope.isDirty()) {
            return $q.when('retrieval-augmented LLM not dirty');
        }

        if (!$scope.isConfigDirty()) {
            return $q.when('only difference is the quicktest query, not saving');
        }

        return $scope.save();
    };

    $scope.isConfigDirty = function() {
        let frankenVersion = angular.copy($scope.currentlySavedInlineModel);
        frankenVersion["quickTestQueryStr"] = $scope.savedModel.inlineVersions[$scope.currentVersionIdx]["quickTestQueryStr"];

        return !angular.equals(frankenVersion, $scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
    };

    $scope.isDirty = function() {
        if (!$scope.savedModel) return false;
        return !angular.equals($scope.savedModel.inlineVersions[$scope.currentVersionIdx], $scope.currentlySavedInlineModel);
    };
    $rootScope.retrievalAugmentedModelIsDirty = $scope.isDirty;

    function allowedTransitionsFn(data) {
        return data.toState?.name?.startsWith('projects.project.savedmodels.savedmodel.retrievalaugmentedllm') 
            && data.fromState?.name?.startsWith('projects.project.savedmodels.savedmodel.retrievalaugmentedllm') 
            && data.toParams?.fullModelId === data.fromParams?.fullModelId;
    }

    checkChangesBeforeLeaving($scope, $scope.isDirty, null, allowedTransitionsFn);
});

app.controller("RetrievalAugmentedLLMSavedModelDesignController", function($scope, $rootScope, $state, $stateParams, $controller, ActivityIndicator, DataikuAPI, PluginConfigUtils, PluginsService, TopNav) {
    // Keep in sync with RAGLLMSettings.DEFAULT_CONTEXT_MESSAGE
    $scope.defaultPrompt = "Answer strictly  based on the provided context.\n"
                           + "If the answer to a factual question is not in the context, say you don't know and briefly state what information is missing.\n"
                           + "Do not use external knowledge, prior trained data, or provide information not explicitly found in the text.";

    // Keep in sync with RAGLLMSettings.SearchInputStrategySettings.DEFAULT_REWRITE_PROMPT
    $scope.defaultRewritePrompt = "When a message requires document retrieval, rewrite it to optimize retrieval from the vector store.\n"
                                  + "Focus on clarity, disambiguation, and extracting the core intent.\n"
                                  + "Output only the rewritten query. Do not include comments or tips.";

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        if (!$scope.noSetLoc) {
            TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "RETRIEVAL-AUGMENTED_LLM-SAVED_MODEL-VERSION", "design");
        }
    });

    $scope.hasMetadataColumns = function() {
        const schema = $scope.retrievableKnowledge?.metadataColumnsSchema ?? [];
        return schema.length > 0;
    };

    $scope.hasLoadedContext = function() {
        return $scope.uiState.ragllmSettings && $scope.metadataColumns && $scope.availableAugmentableLLMs;
    }

    $scope.disableTextGuardrails = function (ragllmSettings) {
        ragllmSettings.ragSpecificGuardrails.faithfulnessSettings.enabled = false;
        ragllmSettings.ragSpecificGuardrails.relevancySettings.enabled = false;
    }

    $scope.disableMultimodalGuardrails = function (ragllmSettings) {
        ragllmSettings.ragSpecificGuardrails.multimodalFaithfulnessSettings.enabled = false;
        ragllmSettings.ragSpecificGuardrails.multimodalRelevancySettings.enabled = false;
    }

    $scope.getParentRecipeHref = function () {
        if (!$scope.rkFullInfo) return '';
        return $state.href('projects.project.recipes.recipe', {projectKey: $scope.rkFullInfo.creatingRecipe.projectKey, recipeName: $scope.rkFullInfo.creatingRecipe.name});
    }

    $scope.hasGuardrailsEnabled = function(ragllmSettings) {
        return ragllmSettings.ragSpecificGuardrails.faithfulnessSettings.enabled
            || ragllmSettings.ragSpecificGuardrails.relevancySettings.enabled
            || ragllmSettings.ragSpecificGuardrails.multimodalFaithfulnessSettings.enabled
            || ragllmSettings.ragSpecificGuardrails.multimodalRelevancySettings.enabled;
    }
});

app.controller("RetrievalAugmentedLLMSavedModelSettingsController", function($scope, $state, $controller, DataikuAPI, TopNav) {

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        if (!$scope.noSetLoc) {
            TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "RETRIEVAL-AUGMENTED_LLM-SAVED_MODEL-VERSION", "settings");
        }
        const currentVersion = nv.inlineVersions.find(v => v.versionId===$scope.currentVersionId);
        // needed for getting the temperature ranges of the LLM
        DataikuAPI.pretrainedModels.listAvailableLLMs($state.projectKey, "GENERIC_COMPLETION").success(function(data) {
            if (currentVersion.ragllmSettings.llmId && data.identifiers) {
                $scope.activeLLM = data.identifiers.find(l => l.id === currentVersion.ragllmSettings.llmId);
            }
        }).error(setErrorInScope.bind($scope));

    });

    $scope.getKnowledgeBankSettingsHref = function() {
        if (!$scope.retrievableKnowledge) {
            return "";
        }
        const { projectKey, id } = $scope.retrievableKnowledge;
        return $state.href("projects.project.retrievableknowledges.retrievableknowledge.settings", {
            projectKey,
            retrievableKnowledgeId: id,
            '#': 'core-settings',
        });
    }
});

app.controller("RetrievalAugmentedLLMSavedModelLogsController", function($scope, $stateParams, $controller, DataikuAPI, Dialogs, TopNav) {

    $scope.listLogs = function() {
        DataikuAPI.savedmodels.retrievalAugmentedLLMs.listLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId).success(function(data) {
            $scope.rallmLogs = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.clearLogs = function() {
        const title = 'Confirm logs deletion';
        const message = 'Are you sure you want to clear the logs for this retrieval-augmented LLM?';

        Dialogs.confirm($scope, title, message).then(() => {
            DataikuAPI.savedmodels.retrievalAugmentedLLMs.clearLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId)
                .error(setErrorInScope.bind($scope))
                .finally($scope.listLogs);
        });
    }

    $scope.deleteLog = function(projectKey, savedModelId, versionId, logName) {
        return DataikuAPI.savedmodels.retrievalAugmentedLLMs.deleteLog(projectKey, savedModelId, versionId, logName)
            .error(setErrorInScope.bind($scope))
            .finally($scope.listLogs);
    }

    $scope.getLog = DataikuAPI.savedmodels.retrievalAugmentedLLMs.getLog;

    $scope.logDownloadURL = (projectKey, savedModelId, versionId, logName) => {
        const params = new URLSearchParams({
            projectKey,
            smId: savedModelId,
            versionId,
            logName
        });
        return `/dip/api/savedmodels/retrieval-augmented-llm/stream-log?${params}`;
    };

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        $scope.listLogs();

        if (!$scope.noSetLoc) {
            TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "RETRIEVAL-AUGMENTED_LLM-SAVED_MODEL-VERSION", "logs");
        }
    });
});

app.controller("RetrievalAugmentedLLMSavedModelHistoryController", function($scope, $controller, TopNav) {
    TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "RETRIEVAL-AUGMENTED_LLM-SAVED_MODEL-VERSION", "history");
});

})();
