(function() {
    'use strict';

    /**
     * List of available chart types used in the chart picker ordered by usage data from analytics
     */

    const app = angular.module('dataiku.charts');

    app.factory('ChartsAvailableTypes', function(CHART_TYPES, CHART_VARIANTS, PluginsService, translate, $rootScope) {

        const svc = {};

        const ENGLISH_KEYWORDS = {
            ADMINISTRATIVE: 'administrative',
            AGGREGATION: 'aggregation',
            ASSOCIATIONS: 'associations',
            BARS: 'bars',
            BINNED: 'binned',
            CHANGE: 'change',
            COMPARISON: 'comparison',
            CURVES: 'curves',
            DISTRIBUTION: 'distribution',
            DONUTS: 'donuts',
            DOUGHNUT: 'doughnut',
            FLOW: 'flow',
            GEOGRAPHICAL: 'geographical',
            GROUP: 'group',
            HIERARCHY: 'hierarchy',
            HISTOGRAM: 'histogram',
            LEARNING: 'learning',
            LINES: 'lines',
            LOCATION: 'location',
            MACHINE: 'machine',
            MAGNITUDE: 'magnitude',
            MAPS: 'maps',
            METRIC: 'metric',
            MODELS: 'models',
            PART: 'part',
            PATTERNS: 'patterns',
            PERCENTAGE: 'percentage',
            PIES: 'pies',
            PROPORTION: 'proportion',
            RANGE: 'range',
            RELATIONSHIPS: 'relationships',
            SCATTERS: 'scatters',
            SPEEDOMETER: 'speedometer',
            SPIDER: 'spider',
            SUMMARY: 'summary',
            TABLES: 'tables',
            TIME: 'time',
            WEB: 'web'
        };

        /** @type {Record<keyof typeof ENGLISH_KEYWORDS, string[]>} */
        const KEYWORDS = Object.entries(ENGLISH_KEYWORDS).reduce((acc, [key, value]) => {
            const keywords = [value];
            const translated = translate(`CHARTS.TYPE_PICKER.KEYWORD.${key}`, value);
            // Avoid duplicating the keywords if DSS is in English.
            if (translated !== value) {
                keywords.push(translated);
            }
            acc[key] = keywords;
            return acc;
        }, {});

        const ENGLISH_DISPLAY_NAMES = {
            VERTICAL_BARS: 'Vertical bars',
            WATERFALL: 'Waterfall',
            VERTICAL_STACKED_BARS: 'Vertical stacked bars',
            VERTICAL_STACKED_BARS_100: 'Vertical stacked_bars 100%',
            HORIZONTAL_STACKED_BARS: 'Horizontal stacked bars',
            HORIZONTAL_STACKED_BARS_100: 'Horizontal stacked bars 100%',
            LINES: 'Lines',
            STACKED_AREAS: 'Stacked_areas',
            STACKED_AREAS_100: 'Stacked areas 100%',
            PIE: 'Pie',
            DONUT: 'Donut',
            MIX: 'Mix',
            PIVOT_TABLE: 'Pivot table',
            KPIS: 'KPIs',
            RADAR: 'Radar',
            GAUGE: 'Gauge',
            SANKEY: 'Sankey',
            TREEMAP: 'Treemap',
            SCATTER_PLOT: 'Scatter plot',
            SCATTER_MULTI_PAIR: 'Scatter multi-pair',
            GEOMETRY_MAP: 'Geometry map',
            GRID_MAP: 'Grid map',
            SCATTER_MAP: 'Scatter map',
            ADMINISTRATIVE_MAP_FILLED: 'Administrative map (filled)',
            ADMINISTRATIVE_MAP_BUBBLES: 'Administrative map (bubbles)',
            DENSITY_MAP: 'Density map',
            GROUPED_BUBBLES: 'Grouped bubbles',
            BINNED_BUBBLES: 'Binned bubbles',
            BINNED_HEXAGONS: 'Binned hexagons',
            BINNED_RECTANGLES: 'Binned rectangles',
            BOXPLOT: 'Boxplot',
            '2D_DISTRIBUTION': '2D distribution',
            LIFT_CHART: 'Lift chart'
        };

        /** @type {Record<keyof typeof ENGLISH_DISPLAY_NAMES, string[]>} */
        const ENGLISH_DISPLAY_NAME_KEYWORDS = Object.entries(ENGLISH_DISPLAY_NAMES).reduce((acc, [key, value]) => {
            acc[key] = splitToKeywords(value);
            return acc;
        }, {});

        function splitToKeywords(phrase) {
            return phrase.trim().toLowerCase().split(/[, ]+/);
        }

        svc.getAvailableChartTypes = function() {
            // Keep in sync with dip/pivot/frontend/model/ChartDef.java::getChartDisplayName()
            return [
                {
                    id: 'qa_charts_histogram-chart-type',
                    type: CHART_TYPES.GROUPED_COLUMNS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.VERTICAL_BARS', ENGLISH_DISPLAY_NAMES.VERTICAL_BARS),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.HISTOGRAM, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.VERTICAL_BARS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_waterfall-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.VERTICAL_BARS', 'Compares values across grouped items or within a subgroup vertically')
                },
                {
                    id: 'qa_charts_waterfall-chart-type',
                    type: CHART_TYPES.GROUPED_COLUMNS,
                    variant: CHART_VARIANTS.waterfall,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.WATERFALL', 'Waterfall chart'),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.HISTOGRAM, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.WATERFALL, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.WATERFALL', 'Shows how negative and positive changes affect a starting value, leading to the final value')
                },
                {
                    id: 'qa_charts_stacked-chart-type',
                    type: CHART_TYPES.STACKED_COLUMNS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.VERTICAL_STACKED_BARS', ENGLISH_DISPLAY_NAMES.VERTICAL_STACKED_BARS),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.VERTICAL_STACKED_BARS];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.VERTICAL_STACKED_BARS', 'Shows part-to-whole relationship within a group vertically')
                },
                {
                    id: 'qa_charts_stacked-100-chart-type',
                    type: CHART_TYPES.STACKED_COLUMNS,
                    variant: CHART_VARIANTS.stacked100,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.VERTICAL_STACKED_BARS_100', ENGLISH_DISPLAY_NAMES.VERTICAL_STACKED_BARS_100),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.VERTICAL_STACKED_BARS_100, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.VERTICAL_STACKED_BARS_100', 'Highlights the relative difference among values in each group - each bar is always 100%')
                },
                {
                    id: 'qa_charts_bars-chart-type',
                    type: CHART_TYPES.STACKED_BARS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.HORIZONTAL_STACKED_BARS', ENGLISH_DISPLAY_NAMES.HORIZONTAL_STACKED_BARS),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.HORIZONTAL_STACKED_BARS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.HORIZONTAL_STACKED_BARS', 'Compares values across grouped items or within a subgroup horizontally')
                },
                {
                    id: 'qa_charts_bars-100-chart-type',
                    type: CHART_TYPES.STACKED_BARS,
                    variant: CHART_VARIANTS.stacked100,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.HORIZONTAL_STACKED_BARS_100', ENGLISH_DISPLAY_NAMES.HORIZONTAL_STACKED_BARS_100),
                    get keywords() {
                        return [...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.BARS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.HORIZONTAL_STACKED_BARS_100, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_pivot-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.HORIZONTAL_STACKED_BARS_100', 'Highlights the relative difference among values in each group - each bar is always 100%')
                },
                {
                    id: 'qa_charts_lines-chart-type',
                    type: CHART_TYPES.LINES,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.LINES', ENGLISH_DISPLAY_NAMES.LINES),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.PATTERNS, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.LINES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_mix-chart-type', 'qa_charts_stacked-area-chart-type', 'qa_charts_stacked-area-100-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.LINES', 'Display quantitative values over a continuous interval or time period')
                },
                {
                    id: 'qa_charts_stacked-area-chart-type',
                    type: CHART_TYPES.STACKED_AREA,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.STACKED_AREAS', ENGLISH_DISPLAY_NAMES.STACKED_AREAS),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.PATTERNS, ...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.STACKED_AREAS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_stacked-area-100-chart-type', 'qa_charts_lines-chart-type', 'qa_charts_mix-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.STACKED_AREAS', 'Shows how values evolve over a continuous interval or time period')
                },
                {
                    id: 'qa_charts_stacked-area-100-chart-type',
                    type: CHART_TYPES.STACKED_AREA,
                    variant: CHART_VARIANTS.stacked100,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.STACKED_AREAS_100', ENGLISH_DISPLAY_NAMES.STACKED_AREAS_100),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.PATTERNS, ...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.STACKED_AREAS_100, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_stacked-area-chart-type', 'qa_charts_lines-chart-type', 'qa_charts_mix-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.STACKED_AREAS_100', 'Highlights the relative difference among values evolving over a continuous time period')
                },
                {
                    id: 'qa_charts_pie-chart-type',
                    type: CHART_TYPES.PIE,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.PIE', ENGLISH_DISPLAY_NAMES.PIE),
                    get keywords() {
                        return [...KEYWORDS.PART, ...KEYWORDS.PROPORTION, ...KEYWORDS.PIES, ...KEYWORDS.DONUTS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.PIE, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_donut-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.PIE', 'Shows a quick part-to-whole comparison')
                },
                {
                    id: 'qa_charts_donut-chart-type',
                    type: CHART_TYPES.PIE,
                    variant: CHART_VARIANTS.donut,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.DONUT', ENGLISH_DISPLAY_NAMES.DONUT),
                    get keywords() {
                        return [...KEYWORDS.PART, ...KEYWORDS.PROPORTION, ...KEYWORDS.PIES, ...KEYWORDS.DONUTS, ...KEYWORDS.DOUGHNUT, ...ENGLISH_DISPLAY_NAME_KEYWORDS.DONUT, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_pie-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.DONUT', 'Shows a quick part-to-whole comparison with a hole inside')
                },
                {
                    id: 'qa_charts_mix-chart-type',
                    type: CHART_TYPES.MULTI_COLUMNS_LINES,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.MIX', ENGLISH_DISPLAY_NAMES.MIX),
                    get keywords() {
                        return [...KEYWORDS.CHANGE, ...KEYWORDS.TIME, ...KEYWORDS.MAGNITUDE, ...KEYWORDS.COMPARISON, ...KEYWORDS.PATTERNS, ...KEYWORDS.PROPORTION, ...KEYWORDS.PERCENTAGE, ...KEYWORDS.BARS, ...KEYWORDS.LINES, ...KEYWORDS.CURVES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.MIX, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_lines-chart-type', 'qa_charts_stacked-area-chart-type', 'qa_charts_stacked-area-100-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.MIX', 'Shows a direct comparison between two sets of values with separate axes')
                },
                {
                    id: 'qa_charts_pivot-chart-type',
                    type: CHART_TYPES.PIVOT_TABLE,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.PIVOT_TABLE', ENGLISH_DISPLAY_NAMES.PIVOT_TABLE),
                    get keywords() {
                        return [...KEYWORDS.SUMMARY, ...KEYWORDS.GROUP, ...KEYWORDS.AGGREGATION, ...KEYWORDS.HIERARCHY, ...KEYWORDS.TABLES, ...ENGLISH_DISPLAY_NAME_KEYWORDS.PIVOT_TABLE, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type', 'qa_charts_treemap-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.PIVOT_TABLE', 'Displays aggregated values in a table grouped across several dimensions')
                },
                {
                    id: 'qa_charts_kpi-chart-type',
                    type: CHART_TYPES.KPI,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.KPI', ENGLISH_DISPLAY_NAMES.KPIS),
                    get keywords() {
                        return [...KEYWORDS.METRIC, ...ENGLISH_DISPLAY_NAME_KEYWORDS.KPIS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_gauge-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.KPI', 'Displays a simple value visualization for single or multiple aggregations')
                },
                {
                    id: 'qa_charts_radar-chart-type',
                    type: CHART_TYPES.RADAR,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.RADAR', ENGLISH_DISPLAY_NAMES.RADAR),
                    get keywords() {
                        return [...KEYWORDS.WEB, ...KEYWORDS.SPIDER, ...ENGLISH_DISPLAY_NAME_KEYWORDS.RADAR, ...splitToKeywords(this.displayName)];
                    },
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.RADAR', 'Displays data in a radial axis, where the radii represents the performance of values')
                },
                {
                    id: 'qa_charts_gauge-chart-type',
                    type: CHART_TYPES.GAUGE,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GAUGE', ENGLISH_DISPLAY_NAMES.GAUGE),
                    get keywords() {
                        return [...KEYWORDS.SPEEDOMETER, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GAUGE, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_kpi-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GAUGE', 'Displays how a value performs in a semi-circular scale')
                },
                {
                    id: 'qa_charts_sankey-chart-type',
                    type: CHART_TYPES.SANKEY,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SANKEY', ENGLISH_DISPLAY_NAMES.SANKEY),
                    get keywords() {
                        return [...KEYWORDS.FLOW, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SANKEY, ...splitToKeywords(this.displayName)];
                    },
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SANKEY', 'Displays a weighted flow of resources')
                },
                {
                    id: 'qa_charts_treemap-chart-type',
                    type: CHART_TYPES.TREEMAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.TREEMAP', ENGLISH_DISPLAY_NAMES.TREEMAP),
                    get keywords() {
                        return [...KEYWORDS.HIERARCHY, ...KEYWORDS.COMPARISON, ...KEYWORDS.PROPORTION, ...KEYWORDS.PART, ...ENGLISH_DISPLAY_NAME_KEYWORDS.TREEMAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_pivot-chart-type', 'qa_charts_histogram-chart-type', 'qa_charts_stacked-chart-type', 'qa_charts_stacked-100-chart-type', 'qa_charts_bars-chart-type', 'qa_charts_bars-100-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.TREEMAP', 'Displays hierarchical data as nested rectangles with proportional sizes')
                },
                {
                    id: 'qa_charts_scatter-plot-chart-type',
                    type: CHART_TYPES.SCATTER,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SCATTER_PLOT', ENGLISH_DISPLAY_NAMES.SCATTER_PLOT),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SCATTER_PLOT, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SCATTER_PLOT', 'Shows the relationships between numerical variables drawn on the axes')
                },
                {
                    id: 'qa_charts_scatter-plot-multiple-pairs-chart-type',
                    type: CHART_TYPES.SCATTER_MULTIPLE_PAIRS,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SCATTER_MULTI_PAIR', ENGLISH_DISPLAY_NAMES.SCATTER_MULTI_PAIR),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SCATTER_MULTI_PAIR, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SCATTER_MULTI_PAIR', 'Shows the relationships between numerical variables drawn on the axes (with multiple pairs)')
                },
                {
                    id: 'qa_charts_geo-map-chart-type',
                    type: CHART_TYPES.GEOMETRY_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GEOMETRY_MAP', ENGLISH_DISPLAY_NAMES.GEOMETRY_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.SCATTERS, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GEOMETRY_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GEOMETRY_MAP', 'Shows geometries as areas on a map')
                },
                {
                    id: 'qa_charts_grid-map-chart-type',
                    type: CHART_TYPES.GRID_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GRID_MAP', ENGLISH_DISPLAY_NAMES.GRID_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.BINNED, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GRID_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GRID_MAP', 'Shows geographical data at administrative level on a map as a grid')
                },
                {
                    id: 'qa_charts_scatter-map-chart-type',
                    type: CHART_TYPES.SCATTER_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.SCATTER_MAP', ENGLISH_DISPLAY_NAMES.SCATTER_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.SCATTERS, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.SCATTER_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.SCATTER_MAP', 'Shows geographical data as points on a map')
                },
                {
                    id: 'qa_charts_filled-map-chart-type',
                    type: CHART_TYPES.ADMINISTRATIVE_MAP,
                    variant: CHART_VARIANTS.filledMap,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.ADMINISTRATIVE_MAP_FILLED', ENGLISH_DISPLAY_NAMES.ADMINISTRATIVE_MAP_FILLED),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.ADMINISTRATIVE, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.ADMINISTRATIVE_MAP_FILLED, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.ADMINISTRATIVE_MAP_FILLED', 'Draws the boundary of geographic data at administrative level as a filled shaped)'),
                    requiredPluginId: 'geoadmin',
                    isRequiredPluginInstalled: PluginsService.isPluginLoaded('geoadmin')
                },
                {
                    id: 'qa_charts_bubble-map-chart-type',
                    type: CHART_TYPES.ADMINISTRATIVE_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.ADMINISTRATIVE_MAP_BUBBLES', ENGLISH_DISPLAY_NAMES.ADMINISTRATIVE_MAP_BUBBLES),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.ADMINISTRATIVE, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.ADMINISTRATIVE_MAP_BUBBLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type', 'qa_charts_density-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.ADMINISTRATIVE_MAP_BUBBLES', 'Shows geographical data at administrative level on the map as a circle'),
                    requiredPluginId: 'geoadmin',
                    isRequiredPluginInstalled: PluginsService.isPluginLoaded('geoadmin')
                },
                {
                    id: 'qa_charts_density-map-chart-type',
                    type: CHART_TYPES.DENSITY_HEAT_MAP,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.DENSITY_MAP', ENGLISH_DISPLAY_NAMES.DENSITY_MAP),
                    get keywords() {
                        return [...KEYWORDS.GEOGRAPHICAL, ...KEYWORDS.LOCATION, ...KEYWORDS.SCATTERS, ...KEYWORDS.MAPS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.DENSITY_MAP, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-map-chart-type', 'qa_charts_bubble-map-chart-type', 'qa_charts_filled-map-chart-type', 'qa_charts_geo-map-chart-type', 'qa_charts_grid-map-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.DENSITY_MAP', 'Shows where points are concentrated on a map by their spatial proximity')
                },
                {
                    id: 'qa_charts_grouped-bubbles-chart-type',
                    type: CHART_TYPES.GROUPED_XY,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.GROUPED_BUBBLES', ENGLISH_DISPLAY_NAMES.GROUPED_BUBBLES),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.GROUPED_BUBBLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.GROUPED_BUBBLES', 'Shows the relationships among continuous numerical variables across a shared dimension')
                },
                {
                    id: 'qa_charts_bubble-chart-type',
                    type: CHART_TYPES.BINNED_XY,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BINNED_BUBBLES', ENGLISH_DISPLAY_NAMES.BINNED_BUBBLES),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...KEYWORDS.BINNED, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BINNED_BUBBLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BINNED_BUBBLES', 'Shows the relationships between categorical or binned numerical variables on the axes')
                },
                {
                    id: 'qa_charts_hexagon-chart-type',
                    type: CHART_TYPES.BINNED_XY,
                    variant: CHART_VARIANTS.binnedXYHexagon,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BINNED_HEXAGONS', ENGLISH_DISPLAY_NAMES.BINNED_HEXAGONS),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...KEYWORDS.BINNED, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BINNED_HEXAGONS, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_rectangle-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BINNED_HEXAGONS', 'Shows a regular array of hexagons to show density for large datasets')
                },
                {
                    id: 'qa_charts_rectangle-chart-type',
                    type: CHART_TYPES.BINNED_XY,
                    variant: CHART_VARIANTS.binnedXYRectangle,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BINNED_RECTANGLES', ENGLISH_DISPLAY_NAMES.BINNED_RECTANGLES),
                    get keywords() {
                        return [...KEYWORDS.ASSOCIATIONS, ...KEYWORDS.PATTERNS, ...KEYWORDS.RELATIONSHIPS, ...KEYWORDS.SCATTERS, ...KEYWORDS.BINNED, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BINNED_RECTANGLES, ...splitToKeywords(this.displayName)];
                    },
                    similar: ['qa_charts_scatter-plot-chart-type', 'qa_charts_scatter-plot-multiple-pairs-chart-type', 'qa_charts_bubble-chart-type', 'qa_charts_hexagon-chart-type', 'qa_charts_grouped-bubbles-chart-type'],
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BINNED_RECTANGLES', 'Shows the relationships between categorical or binned numerical variables')
                },
                {
                    id: 'qa_charts_boxplot-chart-type',
                    type: CHART_TYPES.BOXPLOTS,
                    variant: CHART_VARIANTS.normal,
                    get keywords() {
                        return [...KEYWORDS.DISTRIBUTION, ...KEYWORDS.RANGE, ...ENGLISH_DISPLAY_NAME_KEYWORDS.BOXPLOT, ...splitToKeywords(this.displayName)];
                    },
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.BOXPLOT', ENGLISH_DISPLAY_NAMES.BOXPLOT),
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.BOXPLOT', 'Displays the distribution of data showing minimum, Q1, median, Q3 and maximum')
                },
                {
                    id: 'qa_charts_density-2d-chart-type',
                    type: CHART_TYPES.DENSITY_2D,
                    variant: CHART_VARIANTS.normal,
                    get keywords() {
                        return [...KEYWORDS.DISTRIBUTION, ...ENGLISH_DISPLAY_NAME_KEYWORDS['2D_DISTRIBUTION'], ...splitToKeywords(this.displayName)];
                    },
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.2D_DISTRIBUTION', ENGLISH_DISPLAY_NAMES['2D_DISTRIBUTION']),
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.2D_DISTRIBUTION', 'Shows the bivariate distribution of two fields')
                },
                {
                    id: 'qa_charts_lift-chart-type',
                    type: CHART_TYPES.LIFT,
                    variant: CHART_VARIANTS.normal,
                    displayName: translate('CHARTS.TYPE_PICKER.NAME.LIFT', ENGLISH_DISPLAY_NAMES.LIFT_CHART),
                    get keywords() {
                        return [...KEYWORDS.MACHINE, ...KEYWORDS.LEARNING, ...KEYWORDS.MODELS, ...ENGLISH_DISPLAY_NAME_KEYWORDS.LIFT_CHART, ...splitToKeywords(this.displayName)];
                    },
                    tooltip: translate('CHARTS.TYPE_PICKER.TOOLTIP.LIFT', 'Measures the effectiveness of models between results with and without a model')
                }
            ].filter(chartType => !!chartType);
        };

        return svc;
    });
})();
