CREATE VIEW artifacts_with_external_fields_and_attachments AS

WITH relationships_references_by_artifact AS (
    SELECT
        reference_relationship_artifact_id_from AS artifact_id,
        reference_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
        min(reference_relationship_json #>> '{relationshipData, details, type}') AS field_plain_or_list,
        jsonb_agg(reference_relationship_artifact_id_to ORDER BY COALESCE(reference_relationship_json -> 'relationshipData' -> 'details' -> 'index', reference_relationship_json -> 'relationshipData' -> 'index')) AS ref_ids
    FROM reference_relationships
    GROUP BY reference_relationship_artifact_id_from, field_id
),

relationships_uploaded_files_by_artifact AS (
    SELECT
        uploaded_file_relationship_artifact_id AS artifact_id,
        uploaded_file_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
        min(uploaded_file_relationship_json #>> '{relationshipData, details, type}') AS field_plain_or_list,
        jsonb_agg(uploaded_file_relationship_uploaded_file_id ORDER BY COALESCE(uploaded_file_relationship_json -> 'relationshipData' -> 'details' -> 'index', uploaded_file_relationship_json -> 'relationshipData' -> 'index')) AS ref_ids
    FROM uploaded_file_relationships
    GROUP BY uploaded_file_relationship_artifact_id, field_id
),

relationships_timeseries_by_artifact AS (
    SELECT
        time_series_relationship_artifact_id AS artifact_id,
        time_series_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
        min(time_series_relationship_json #>> '{relationshipData, details, type}') AS field_plain_or_list,
        jsonb_agg(time_series_relationship_time_series_id ORDER BY time_series_relationship_json -> 'relationshipData' -> 'details' -> 'index') AS ref_ids
    FROM time_series_relationships
    GROUP BY time_series_relationship_artifact_id, field_id
),

blueprint_backreference_field_definitions AS (
    SELECT
        blueprint_version_blueprint_id,
        blueprint_version_version_id,
        fields_definitions.key AS field_id,
        backref_reference_entry.value AS backref_reference
    FROM blueprint_versions,
    jsonb_each(blueprint_version_json -> 'fieldDefinitions') AS fields_definitions,
    jsonb_array_elements(fields_definitions.value -> 'references') AS backref_reference_entry
    WHERE fields_definitions.value ->> 'fieldType' = 'REFERENCE' AND fields_definitions.value ->> 'sourceType' = 'COMPUTE'
),

relationships_backreferences_by_artifact AS (
    SELECT
        reference_relationships.reference_relationship_artifact_id_to AS artifact_id,
        blueprint_backreference_field_definitions.field_id AS field_id,
        'list' AS field_plain_or_list,
        jsonb_agg(reference_relationship_artifact_id_from) AS ref_ids
    FROM artifacts AS artifacts_to
    INNER JOIN blueprint_backreference_field_definitions
        ON  (blueprint_backreference_field_definitions.blueprint_version_blueprint_id = artifacts_to.artifact_blueprint_id
        AND blueprint_backreference_field_definitions.blueprint_version_version_id = artifacts_to.artifact_version_id)
    INNER JOIN reference_relationships
        ON (blueprint_backreference_field_definitions.backref_reference ->> 'fieldId' = reference_relationships.reference_relationship_json #>> '{relationshipData, fieldId}'
        AND reference_relationships.reference_relationship_artifact_id_to = artifacts_to.artifact_id)
    INNER JOIN artifacts AS artifacts_from
        ON (blueprint_backreference_field_definitions.backref_reference ->> 'blueprintId' = artifacts_from.artifact_blueprint_id
        AND reference_relationships.reference_relationship_artifact_id_from = artifacts_from.artifact_id)
    GROUP BY reference_relationship_artifact_id_to, field_id
),

relationships_all AS (
    SELECT 'references' AS rel_type, artifact_id, field_id, field_plain_or_list, ref_ids FROM relationships_references_by_artifact
    UNION ALL
    SELECT 'uploaded_files' AS rel_type, artifact_id, field_id, field_plain_or_list, ref_ids FROM relationships_uploaded_files_by_artifact
    UNION ALL
    SELECT 'timeseries' AS rel_type, artifact_id, field_id, field_plain_or_list, ref_ids FROM relationships_timeseries_by_artifact
    UNION ALL
    SELECT 'backreferences' AS rel_type, artifact_id, field_id, field_plain_or_list, ref_ids FROM relationships_backreferences_by_artifact
),

grouped_fields_and_references AS (
    SELECT
        relationships_all.artifact_id,
        jsonb_object_agg(field_id, CASE field_plain_or_list WHEN 'plain' THEN ref_ids -> 0 ELSE ref_ids END) FILTER (WHERE field_id IS NOT NULL) AS fields,
        jsonb_agg(ref_ids) FILTER (WHERE field_id IS NULL AND rel_type = 'references') -> 0 AS attachments_references,
        jsonb_agg(ref_ids) FILTER (WHERE field_id IS NULL AND rel_type = 'uploaded_files') -> 0 AS attachments_uploaded_files
    FROM relationships_all
    GROUP BY relationships_all.artifact_id
)

SELECT
    artifacts.artifact_id,
    artifacts.artifact_blueprint_id,
    artifacts.artifact_version_id,
    artifacts.artifact_json
        || jsonb_build_object('fields', (artifacts.artifact_json -> 'fields' || COALESCE(grouped_fields_and_references.fields, jsonb_build_object())))
        || jsonb_build_object('attachments', jsonb_build_object('references', COALESCE(grouped_fields_and_references.attachments_references, jsonb_build_array()), 'files', COALESCE(grouped_fields_and_references.attachments_uploaded_files, jsonb_build_array())))
        AS artifact_json
FROM artifacts
LEFT JOIN grouped_fields_and_references ON grouped_fields_and_references.artifact_id = artifacts.artifact_id;
