(function() {
    'use strict';

    const app = angular.module('dataiku.businessapps', ['dataiku.filters']);

    app.controller('BusinessAppCommonController', function($scope, DataikuAPI, CreateModalFromTemplate, TopNav, FutureProgressModal, WT1) {
        TopNav.setLocation(TopNav.DSS_HOME, 'businessapp');

        /**
         * Uninstalls a Business Application after showing a confirmation modal.
         * @param {BusinessApp} businessApp - The installed Business Application to uninstall
         * @returns {Promise} A promise that resolves with {businessAppId, businessAppVersion} on success,
         *                    or rejects with {data, status, headers} on failure
         */
        $scope.uninstallBusinessApp = function(businessApp) {
            const desc = businessApp.desc;

            function onSuccess(resolve) {
                resolve(desc.id, desc.version);
                WT1.event("business-app-delete", { businessAppId: desc.id, businessAppVersion: desc.version, success: true });
            }
            function onError(reject, data, status, headers) {
                reject(data, status, headers);
                WT1.event("business-app-delete", { businessAppId: desc.id, businessAppVersion: desc.version, success: false });
                setErrorInScope.call($scope, data, status, headers);
            }

            return new Promise((resolve, reject) => {
                CreateModalFromTemplate("/templates/business-apps/modals/business-app-uninstall-confirm.html", $scope, null, function(newScope) {
                    newScope.businessAppId = desc.id;
                    newScope.businessAppName = desc.name;
                    newScope.confirmBusinessAppUninstall = function() {
                        DataikuAPI.businessApps.delete(desc.id, true).success(function(initialResponse) {
                            if (initialResponse && initialResponse.jobId && !initialResponse.hasResult) {
                                FutureProgressModal.show($scope, initialResponse, "Uninstalling " + (desc.name || desc.id)).then(function() {
                                    onSuccess(resolve);
                                }).catch(function(data, status, headers) {
                                    onError(reject, data, status, headers);
                                });
                            } else {
                                onSuccess(resolve);
                            }
                        }).error(function(data, status, headers) {
                            onError(reject, data, status, headers);
                        });
                    };
                });
            });
        };
    });

    app.controller('BusinessAppInstancesCommonController', function($scope, DataikuAPI, $stateParams, CreateModalFromTemplate, WT1, ActivityIndicator) {

        $scope.deleteBusinessAppInstances = function(projectKeys) {
            return new Promise((resolve, reject) => {
                DataikuAPI.businessApps.checkInstancesDeletability($stateParams.businessAppId, projectKeys).success(function(data) {
                    if (data.anyMessage) {
                        // Some error happened!
                        CreateModalFromTemplate("/templates/business-apps/modals/business-app-delete-instances-results.html", $scope, null, function(newScope) {
                            newScope.beforeDeletion = true;
                            newScope.results = data.messages;
                            newScope.$on('$destroy', function() {
                                reject();
                            });
                        });
                    } else {
                        CreateModalFromTemplate("/templates/business-apps/modals/business-app-delete-instances-confirm.html", $scope, null, function(newScope) {
                            newScope.projectKeys = projectKeys;
                            newScope.clearManagedDatasets = true;
                            newScope.clearOutputManagedFolders = true;
                            newScope.clearJobAndScenarioLogs = true;
                            newScope.clearManagedKnowledgeBanks = true;
                            newScope.confirmProjectDeletion = function(clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs, clearManagedKnowledgeBanks) {
                                DataikuAPI.businessApps.deleteInstances($stateParams.businessAppId, projectKeys, clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs, clearManagedKnowledgeBanks).success(function(deletionResult) {
                                    if (deletionResult.anyMessage) {
                                        CreateModalFromTemplate("/templates/business-apps/modals/business-app-delete-instances-results.html", $scope, null, function(newScope) {
                                            newScope.beforeDeletion = false;
                                            newScope.results = deletionResult.messages;
                                            newScope.$on('$destroy', function() {
                                                ActivityIndicator.success("Instances deleted");
                                                resolve();
                                            });
                                        });
                                    } else {
                                        ActivityIndicator.success("Instances deleted");
                                        resolve();
                                    }
                                }).error((data, status, headers) => {
                                    setErrorInScope.call($scope, data, status, headers);
                                    ActivityIndicator.error("Failed to delete instances");
                                    reject();
                                });
                                WT1.event("business-app-delete-instances", {
                                    clearManagedDatasets: clearManagedDatasets,
                                    clearOutputManagedFolders: clearOutputManagedFolders,
                                    clearJobAndScenarioLogs: clearJobAndScenarioLogs,
                                    clearManagedKnowledgeBanks: clearManagedKnowledgeBanks
                                });
                            }
                        });
                    }
                }).error((data, status, headers) => {
                    setErrorInScope.call($scope, data, status, headers)
                    reject();
                });
            });
        };
    });

    app.controller("BusinessAppsExploreController", function($rootScope, $scope, $controller, $state, DataikuAPI, Assert, CreateModalFromTemplate, TopNav, RequestCenterService, ActivityIndicator, SemanticVersionService) {
        $controller("BusinessAppCommonController", { $scope });

        TopNav.setLocation(TopNav.DSS_HOME, 'businessapps');

        $scope.businessAppsUIState = {
            filteredStoreBusinessApps: {},
            filteredInstalledBusinessApps: {},
            searchQuery: ''
        };
        $scope.uploadedBusinessApp = { file: null }; // Need a nested object for binding to work propertly

        /*
         *  Accessible either from the preview installation modal or directly from a Business Application card in the store
         */
        $scope.installBusinessApp = function(businessAppToInstall, isUpdate) {
            if ($rootScope.appConfig.admin && businessAppToInstall) {
                if (isUpdate === true) {
                    $state.transitionTo('businessapp.update', {
                        businessAppId: businessAppToInstall.id,
                        businessAppVersion: businessAppToInstall.storeVersion
                    });
                } else {
                    $state.transitionTo('businessapp.installation', {
                        businessAppId: businessAppToInstall.id,
                        businessAppVersion: businessAppToInstall.storeVersion
                    });
                }
                $scope.dismiss && $scope.dismiss();
            }
        };

        $scope.resetStoreQuery = function() {
            $scope.businessAppsUIState.searchQuery = '';
        };

        $scope.resetInstalledQuery = function() {
            $scope.businessAppsUIState.searchQuery = '';
        };

        $scope.hasNoResultsForStoreQuery = function() {
            const isListEmpty = $scope.businessAppsUIState.filteredStoreBusinessApps && $scope.businessAppsUIState.filteredStoreBusinessApps.length === 0;
            const hasSearched = $scope.businessAppsUIState.searchQuery && $scope.businessAppsUIState.searchQuery.length;
            return isListEmpty && hasSearched;
        }

        $scope.hasNoResultsForInstalledQuery = function() {
            const isListEmpty = $scope.businessAppsUIState.filteredInstalledBusinessApps && $scope.businessAppsUIState.filteredInstalledBusinessApps.length === 0;
            const hasSearched = $scope.businessAppsUIState.searchQuery && $scope.businessAppsUIState.searchQuery.length;
            return isListEmpty && hasSearched;
        }

        $scope.refreshList = function(forceFetch) {
            if ($scope.businessAppsList && forceFetch === false) {
                return;
            }

            DataikuAPI.businessApps.list(forceFetch).success(function(data) {
                $scope.businessAppsList = data;
                $scope.businessAppsUIState.storeBusinessAppsCount = 0;
                $scope.businessAppsUIState.installedBusinessAppsCount = 0;
                $scope.businessAppsList.businessApps.forEach(app => {
                    if (app.storeDesc) {
                        $scope.businessAppsUIState.storeBusinessAppsCount++;
                        app.imageURL = app.storeDesc.imageURL;
                    }
                    if (app.installedDesc) {
                        $scope.businessAppsUIState.installedBusinessAppsCount++;
                        app.imageURL = '/dip/api/image/get-image?type=BUSINESS_APP&id=' + encodeURIComponent(app.installedDesc.desc.id);
                    }
                    if (app.storeDesc?.storeVersion && app?.installedDesc?.desc.version && SemanticVersionService.compareVersions(app.storeDesc.storeVersion, app.installedDesc.desc.version) > 0) {
                        app.updateAvailable = true;
                    }

                });
                if (forceFetch) {
                    if (data.couldFetch) {
                        ActivityIndicator.success("Store content refreshed");
                    } else {
                        ActivityIndicator.error("Failed to reach the store");
                    }
                }
            }).error((data, status, headers) => {
                setErrorInScope.call($scope, data, status, headers);
                if (!$scope.businessAppsList) {
                    $scope.businessAppsList = {
                        businessAppsNumberInStore: 0,
                        couldFetch: false
                    };
                }
            });
        };

        $scope.previewInstallStoreBusinessApp = function(businessApp) {
            Assert.trueish(businessApp.inStore, "Business Application not in store");
            CreateModalFromTemplate("/templates/business-apps/modals/business-app-install-preview.html", $scope, null, function(newScope) {
                newScope.attachDownloadTo = $scope;
                newScope.isUpdate = businessApp.installed;
                if (businessApp.installed) {
                    newScope.installedVersion = businessApp.installedDesc.desc.version;
                }
                newScope.uiState = { activeTab: 'details' };
                newScope.storeBusinessApp = businessApp.storeDesc;
            });
        };

        $scope.previewUninstallBusinessApp = function(businessApp) {
            $scope.uninstallBusinessApp(businessApp.installedDesc).then(() => $scope.refreshList());
        };

        /** On the scope passed in, set hasPreviousRequest as boolean and latestRequest if there was one, empty object if not */
        function checkForPreviousBusinessAppRequest(modalScope, errorScope, businessAppId) {
            //defaults
            modalScope.hasPreviousRequest = false;
            modalScope.latestRequest = {};

            if (modalScope.appConfig.admin) {
                //Don't need the info, defaults are fine
                return;
            }

            DataikuAPI.requests.getLatestRequestForCurrentUser(businessAppId, "BUSINESS_APP", "").then(response => {
                if (response.data.status === "PENDING" || (modalScope.queryRequestType === "INSTALL_BUSINESS_APP" && response.data.status === "APPROVED")) {
                    modalScope.hasPreviousRequest = true;
                    modalScope.latestRequest = response.data;
                }
            }, error => {
                if (error.status !== 404) {
                    setErrorInScope.bind(errorScope)(error);
                }
            });
        }

        $scope.requestInstallOrUpdateBusinessApp = function(businessApp) {
            Assert.trueish(businessApp.inStore, "Business Application not in store");
            CreateModalFromTemplate("/templates/business-apps/modals/business-app-install-request.html", $scope, null, function(newScope) {
                newScope.businessApp = businessApp;
                newScope.ui = { message: "" };
                newScope.requestType = newScope.businessApp.installed ? "update" : "install";
                newScope.storeBusinessApp = newScope.businessApp.storeDesc;
                newScope.queryRequestType = newScope.businessApp.installed ? "UPDATE_BUSINESS_APP" : "INSTALL_BUSINESS_APP";

                //sets newScope.hasPreviousRequest and newScope.latestRequest
                checkForPreviousBusinessAppRequest(newScope, $scope, businessApp.id);

                // Refresh Business Application info in case has just been installed
                DataikuAPI.businessApps.list(false).success(function(data) {
                    // Updating Business Application list ensures recent installations are captured on the store page
                    $scope.businessAppsList = data;
                    // this will capture if the Business Application is now installed
                    // (but we keep the request type the same as it reflects the original user intention)
                    newScope.businessApp = data.businessApps.find(aBusinessApp => aBusinessApp.id === businessApp.id);
                }).error(setErrorInScope.bind($scope));

                newScope.sendRequest = (requestMessage) => {
                    DataikuAPI.requests.createBusinessAppRequest(newScope.queryRequestType, newScope.businessApp.id, newScope.ui.message).success((data) => {
                        RequestCenterService.WT1Events.onRequestSent("BUSINESS_APP", null, newScope.businessApp.id, requestMessage, data.id);
                    }).error(setErrorInScope.bind($scope));
                    newScope.dismiss();
                    $rootScope.$broadcast("dismissModals");
                };
            });
        }

        $scope.newZippedBusinessApp = function() {
            CreateModalFromTemplate('/templates/business-apps/modals/business-app-upload.html', $scope);
        };

        $scope.uploadBusinessApp = function() {
            if ($scope.uploadedBusinessApp.file && $scope.uploadedBusinessApp.file !== '') {
                let fileName = $scope.uploadedBusinessApp.file.name;
                $state.transitionTo('businessapp.upload', {
                    businessAppId: fileName,
                    uploadedBusinessAppFile: $scope.uploadedBusinessApp.file
                });
            }
        };

        $scope.reload = function() {
            location.reload();
        };

        $scope.goToBusinessAppPage = function(businessAppId) {
            $state.transitionTo('businessapp.settings', { businessAppId: businessAppId });
        }

        // shortcut to force the list refresh
        Mousetrap.bind("r l", function() {
            $scope.refreshList(true);
        });

        $scope.$on("$destroy", function() {
            Mousetrap.unbind("r l");
        });

        $scope.refreshList(false);

        if ($state.is('businessapps')) {
            $state.transitionTo('businessapps.store');
        }
    });

    app.controller("BusinessAppController", function($scope, $controller, $state, $stateParams, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, 'businessapp');

        $scope.businessAppsUIState = $scope.businessAppsUIState || {};

        $controller("BusinessAppCommonController", { $scope });

        $scope.isInstallingOrUpdatingBusinessApp = function() {
            return $state.includes('businessapp.installation')
                || $state.includes('businessapp.update')
                || $state.includes('businessapp.upload')
        };

        $scope.previewUninstallBusinessApp = function() {
            $scope.uninstallBusinessApp($scope.businessAppStateInfo.installedDesc).then(() => {
                $state.transitionTo('businessapps.store');
            }).catch((data, status, headers) => {
                $scope.state = "FAILED";
                $scope.failure = {
                    message: getErrorDetails(data, status, headers).detailedMessage
                };
            });
        };

        $scope.getBusinessApp = function() {
            return DataikuAPI.businessApps.get($stateParams.businessAppId).success(function(data) {
                $scope.businessAppStateInfo = data;
            }).error(setErrorInScope.bind($scope));
        };

        // Initialization
        if ($scope.isInstallingOrUpdatingBusinessApp()) {
            $scope.businessAppLabel = $stateParams.businessAppId;
        } else {
            $scope.getBusinessApp();
        }
    });

    app.controller("BusinessAppInstallationController", function($scope, DataikuAPI, MonoFuture, Fn, WT1, $state, $stateParams, FutureWatcher, Logger) {
        $scope.state = "NOT_STARTED";
        $scope.environmentState = 'NOT_STARTED';
        $scope.businessAppId = $stateParams.businessAppId;
        $scope.businessAppVersion = $stateParams.businessAppVersion;
        $scope.isUpdate = $state.includes('businessapp.update');
        $scope.isUpload = $state.includes('businessapp.upload');
        $scope.isCodeEnvDefined = false;
        const uploadedBusinessAppFile = $stateParams.uploadedBusinessAppFile;

        function go() {
            MonoFuture($scope).wrap(DataikuAPI.businessApps.install)($scope.businessAppId).success(function(data) {
                if (data && data.log) {
                    $scope.installationLog = data.log;
                }
                if (!data.result.success) {
                    $scope.state = "FAILED";
                    WT1.event("business-app-download", {
                        businessAppId: $scope.businessAppId,
                        businessAppVersion: $scope.businessAppVersion,
                        success: "FAILED"
                    });
                    $scope.failure = {
                        message: data.result.installationError.detailedMessage,
                        isAlreadyInstalled: data.result.installationError.detailedMessage.includes('already installed')
                    }
                    $scope.installingFuture = null;
                    return;
                }
                WT1.event("business-app-download", {
                    businessAppId: $scope.businessAppId,
                    businessAppVersion: $scope.businessAppVersion,
                    success: "DONE"
                });
                const confirmInstallation = () => {
                    $scope.state = "DONE";
                    $scope.businessApp = data.result.businessApp;
                    $scope.businessAppSettings = data.result.businessApp.settings;
                    $scope.isCodeEnvDefined = !!($scope.businessApp.desc.codeEnvName);
                    if ($scope.isCodeEnvDefined && $scope.businessApp.codeEnvExist) {
                        $scope.environmentState = ($scope.businessApp.codeEnvUpToDate) ? "DONE" : "WAITING_CONFIRMATION";
                    }
                    $scope.installingFuture = null;
                    $scope.$parent.businessAppLabel = $scope.businessApp.desc.name;
                }
                if (data.result.success && data.result.messages && data.result.messages.warning) {
                    // We display confirmation only if main operation is success but some warnings were found
                    $scope.state = "WAITING_CONFIRMATION";
                    $scope.installationMessages = data.result.messages;
                    $scope.confirmInstallation = confirmInstallation;
                    return;
                }
                confirmInstallation();
            }).update(function(data) {
                $scope.state = "RUNNING";
                $scope.installingFuture = data;
                $scope.installationLog = data.log;
            }).error(function(data) {
                $scope.state = "FAILED";
                if (data.aborted) {
                    $scope.failure = { message: "Aborted" }
                } else if (data.hasResult) {
                    $scope.failure = { message: data.result.errorMessage }
                } else if (data.errorType && data.message) {
                    $scope.failure = { message: data.message }
                } else {
                    $scope.failure = { message: "Unexpected error" }
                }
                $scope.installingFuture = null;
            });
        }

        const handleError = (message) => {
            $scope.state = "FAILED";
            $scope.failure = {
                message: message
            }
        }

        const handleAPIError = (data, status, headers) => {
            handleError(getErrorDetails(data, status, headers).detailedMessage);
        };

        const handleFutureError = (data) => {
            let errorMessage;
            if (data && data.result && data.result.installationError && data.result.installationError.detailedMessage) {
                errorMessage = data.result.installationError.detailedMessage;
            } else {
                errorMessage = 'An error occurred';
            }
            handleError(errorMessage);
        }

        const handleResult = (eventId, businessAppId) => (data) => {
            if (data.aborted) return;
            if (data && data.log) {
                $scope.installationLog = data.log;
            }
            if (!data || !data.result || !data.result.success) {
                WT1.event(eventId, { businessAppId: businessAppId, success: "FAILED" });
                handleFutureError(data);
                return;
            }
            WT1.event(eventId, { businessAppId: businessAppId, success: "DONE" });
            const confirmInstallation = () => {
                $scope.state = "DONE";
                $scope.businessApp = data.result.businessApp;
                $scope.businessAppSettings = data.result.settings;
                $scope.isCodeEnvDefined = !!($scope.businessApp.desc.codeEnvName);
                if ($scope.isCodeEnvDefined && $scope.businessApp.codeEnvExist) {
                    $scope.environmentState = ($scope.businessApp.codeEnvUpToDate) ? "DONE" : "WAITING_CONFIRMATION";
                }
                unregisterBusinessAppIdWatcher();
                $scope.businessAppId = data.result.businessApp.desc.id;
                $scope.$parent.businessAppLabel = data.result.businessApp.desc.name;
            }
            if (data.result.success && data.result.messages && data.result.messages.warning) {
                // We display confirmation only if main operation is success but some warnings were found
                $scope.state = "WAITING_CONFIRMATION";
                $scope.installationMessages = data.result.messages;
                $scope.confirmInstallation = confirmInstallation;
                return;
            }
            confirmInstallation();
        }

        function upload() {
            $scope.state = 'RUNNING';
            DataikuAPI.businessApps.uploadBusinessApp(uploadedBusinessAppFile).then(function(payload) {
                const data = JSON.parse(payload);
                FutureWatcher.watchJobId(data.jobId)
                    .success(handleResult("business-app-upload"))
                    .update(function(data) {
                        $scope.installingFuture = data;
                        $scope.installationLog = data.log;
                    }).error(handleAPIError);
            }, function(payload) {
                let errorMessage;
                try {
                    const parsedResponse = JSON.parse(payload.response);
                    errorMessage = parsedResponse.detailedMessage;
                } catch (exception) {
                    Logger.error(exception);
                    errorMessage = 'An unknown error occurred.'
                }
                handleError(errorMessage);
            });
        }

        $scope.abort = function() {
            $scope.state = "FAILED";
            $scope.failure = {
                message: "Aborted"
            }
            DataikuAPI.futures.abort($scope.installingFuture.jobId);
        };

        $scope.skipEnvironmentCreation = function() {
            $scope.state = 'DONE';
            $scope.environmentState = 'SKIPPED';
        }

        $scope.approveEnvironmentCreation = function() {
            $scope.environmentState = 'WAITING_CONFIRMATION';
        }

        $scope.disapproveEnvironmentCreation = function() {
            $scope.environmentState = 'NOT_STARTED';
        }

        $scope.confirmEnvironmentCreation = function() {
            $scope.environmentState = 'DONE';
        }

        $scope.goToBusinessAppPage = function(businessAppId) {
            window.location = $state.href('businessapp.settings', { businessAppId });
        };

        $scope.$on("$destroy", function() {
            if ($scope.state === "RUNNING") {
                $scope.abort();
            }
        });

        let unregisterBusinessAppIdWatcher = $scope.$watch("businessAppId", Fn.doIfNv($scope.isUpload ? upload : go));
    });

    app.controller("BusinessAppSettingsController", function($scope, $controller, $state, $stateParams, $filter, DataikuAPI, MonoFuture, Fn, WT1, Dialogs, CreateModalFromTemplate, FutureProgressModal, ActivityIndicator, PermissionsDataFetcher) {
        const INSTANCE_OWNER = "%%INSTANCE_OWNER%%";

        // Run As
        function updateRunAsUserOptions() {
            if (!$scope.allUsers || !$scope.businessAppSettings) {
                return;
            }
            const options = [{ login: INSTANCE_OWNER, displayName: "< New instance owner >" }, ...$scope.allUsers];
            const defaultRunAsUser = $scope.businessAppSettings.defaultRunAsUser;
            if (defaultRunAsUser && !options.some(user => user.login === defaultRunAsUser)) {
                options.push({ login: defaultRunAsUser, displayName: `${defaultRunAsUser} (not found)` });
            }
            $scope.runAsUserOptions = options;
            $scope.runAsUserOptionsLogin = options.map(user => user.login === INSTANCE_OWNER ? "User creating the instance" : `@${user.login}`);
        }

        // Container configuration
        function ensureContainerSelection(selection) {
            const containerSelection = selection || {};
            if (!containerSelection.containerMode) {
                containerSelection.containerMode = "INHERIT";
            }
            return containerSelection;
        }

        // Connection remapping
        function checkConnectionRemapping(nv) {
            $scope.needsRemapping = nv && nv.some(r => !r.target);
        }
        $scope.$watch("businessAppSettings.remapping.connections", checkConnectionRemapping, true);
        $scope.getConnectionNameWithDescription = (connection) => $filter('connectionNameFormatter')(connection.name) + (connection.description ? (' | ' + connection.description) : '');

        // Form validation / save
        $scope.isBusinessAppSettingsFormsInvalid = function() {
            return $scope.businessAppSettingsForms.$invalid;
        }

        $scope.dirtyBusinessAppSettings = function() {
            return $scope.originalBusinessAppSettings !== null && !angular.equals($scope.originalBusinessAppSettings, $scope.businessAppSettings);
        };

        checkChangesBeforeLeaving($scope, $scope.dirtyBusinessAppSettings);

        $scope.saveBusinessAppSettings = function() {
            DataikuAPI.businessApps.saveSettings($scope.businessAppStateInfo.installedDesc.desc.id, $scope.businessAppSettings)
                .success(() => {
                    // make sure dirtyBusinessAppSettings says it's ok to avoid checkChangesBeforeLeaving complaining
                    $scope.originalBusinessAppSettings = angular.copy($scope.businessAppSettings);
                    WT1.event("business-app-settings-changed", { businessAppId: $scope.businessAppStateInfo.installedDesc.desc.id });
                })
                .error(setErrorInScope.bind($scope));
        };

        //---------------------------------
        // Initialization
        //---------------------------------
        $scope.permissionsDataFetcher = PermissionsDataFetcher($scope);
        if (!$stateParams.businessAppId || $scope.installed) {
            return;
        }
        // Load Business Application
        DataikuAPI.businessApps.get($stateParams.businessAppId).success(function(data) {
            $scope.businessAppStateInfo = data;
            $scope.installed = data.installedDesc;
            $scope.businessAppSettings = data?.installedDesc?.settings || {};
            $scope.businessAppSettings.container = ensureContainerSelection($scope.businessAppSettings.container);
            $scope.businessAppSettings.containerForVisualRecipesWorkloads = ensureContainerSelection($scope.businessAppSettings.containerForVisualRecipesWorkloads);
            $scope.originalBusinessAppSettings = angular.copy($scope.businessAppSettings);
            $scope.requiredConnections = data.installedDesc?.desc?.connections || {};
            checkConnectionRemapping($scope.businessAppSettings?.remapping?.connections);
            updateRunAsUserOptions();
        }).error(setErrorInScope.bind($scope));

        // Load connections
        DataikuAPI.admin.connections.list()
            .success((data) => {
                $scope.connectionsByName = data;
                $scope.connections = Object.values(data);
            }).error(setErrorInScope.bind($scope));

        // Load containers
        DataikuAPI.containers.listNames(null, "USER_CODE").success(function(data) {
            $scope.containerNames = data;
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.containers.listNames(null, "VISUAL_RECIPES").success(function(data) {
            $scope.containerNamesForVisualRecipesWorkloads = data;
        }).error(setErrorInScope.bind($scope));

        // Load users
        DataikuAPI.security.listUsers().success(function(data) {
            $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
            updateRunAsUserOptions();
        }).error(setErrorInScope.bind($scope));
    });

    app.controller("BusinessAppInstancesController", function($scope, $controller, $state, $stateParams, $filter, DataikuAPI, MonoFuture, Fn, WT1, Dialogs, CreateModalFromTemplate, FutureProgressModal, ActivityIndicator) {
        $controller("BusinessAppInstancesCommonController", { $scope });

        function loadInstances() {
            DataikuAPI.businessApps.listInstances($stateParams.businessAppId)
                .success(data => {
                    $scope.businessAppInstanceListLoaded = true;
                    $scope.businessAppInstanceList = data;
                })
                .error(setErrorInScope.bind($scope));
        }

        $scope.upgradeBusinessAppInstance = function(instance) {
            CreateModalFromTemplate("/templates/business-apps/modals/business-app-upgrade-instance-confirm.html", $scope, null, newScope => {
                newScope.projectKey = instance.projectKey;
                newScope.projectName = instance.name;
                newScope.version = $scope.installed.desc.version;
                newScope.confirmBusinessAppInstanceUpgrade = function() {
                    WT1.event("business-app-instance-upgrade", {
                        businessAppId: instance.businessAppId,
                        oldVersion: instance.businessAppVersion,
                        newVersion: $scope.installed.desc.version,
                        projectHash: md5(instance.projectKey),
                    });
                    DataikuAPI.businessApps.upgradeInstance(instance.businessAppId, instance.projectKey).success(function(data) {
                        FutureProgressModal.show($scope, data, "Upgrade " + instance.projectKey, undefined, 'static', false, true).then(function(result) {
                            loadInstances();
                            if (result && result.anyMessage) {
                                Dialogs.infoMessagesDisplayOnly($scope, "Upgrade " + instance.projectKey, result);
                            }
                        }).catch(setErrorInScope.bind($scope));
                    }).error(setErrorInScope.bind($scope));
                };
            });
        }

        $scope.deleteBusinessAppInstance = function(instance) {
            $scope.deleteBusinessAppInstances([instance.projectKey]).then(() => {
                $scope.businessAppInstanceList = $scope.businessAppInstanceList.filter(item => item !== instance);
                $scope.$apply();
            }).catch(setErrorInScope.bind($scope));
        };

        $scope.createBusinessAppInstance = function() {
            CreateModalFromTemplate("/templates/business-apps/modals/business-app-new-instance.html", $scope, "BusinessAppNewInstanceModalController", newScope => {
                newScope.businessAppId = $scope.installed.desc.id;
                newScope.businessAppVersion = $scope.installed.desc.version;
                newScope.businessAppName = $scope.installed.desc.name;
                newScope.defaultRunAsUser = $scope.businessAppSettings?.defaultRunAsUser;
                newScope.allowRunAsUserOverride = $scope.businessAppSettings?.allowRunAsUserOverride;
            }).then(function({ projectKey, webAppId }) {
                $state.go("projects.project.webapps.webapp.view", { projectKey, webAppId });
            }).catch(loadInstances); // Reload all instances, a new one might have been created (in case creation of instance succeeded but webapp failed to start)
        };

        $scope.restartWebAppBackend = function(instance) {
            DataikuAPI.webapps.restartBackend({ projectKey: instance.projectKey, id: instance.webAppId })
                .error(setErrorInScope.bind($scope))
                .success(data => {
                    FutureProgressModal.show($scope, data, "Starting backend").then(function(result) {
                        if (result) { // undefined in case of abort
                            instance.isWebAppBackendRunning = true;
                        }
                    }).catch(setErrorInScope.bind($scope));
                });
        };

        $scope.stopWebAppBackend = function(instance) {
            DataikuAPI.webapps.stopBackend({ projectKey: instance.projectKey, id: instance.webAppId })
                .error(setErrorInScope.bind($scope))
                .success(() => {
                    ActivityIndicator.success("Backend stopped");
                    instance.isWebAppBackendRunning = false;
                });
        };

        // Initialization
        if (!$stateParams.businessAppId || $scope.installed) {
            return;
        }
        // Load Business Application
        DataikuAPI.businessApps.get($stateParams.businessAppId).success(function(data) {
            $scope.businessAppStateInfo = data;
            $scope.installed = data.installedDesc;
            $scope.businessAppSettings = data?.installedDesc?.settings || {};
        }).error(setErrorInScope.bind($scope));

        // Load instances
        $scope.businessAppInstanceListLoaded = false;
        loadInstances();
    });

    /* Directives */
    app.directive('businessAppSecurityPermissions', function() {
        return {
            restrict: 'A',
            templateUrl: '/templates/business-apps/business-app-security-permissions.html',
            scope: {
                businessAppSettings: '=',
                permissionsDataFetcher: '<'
            },
            link: function($scope) {
                $scope.securityPermissionsHooks = {};
                $scope.securityPermissionsHooks.makeNewPerm = function() {
                    return {
                        use: true,
                        create: true
                    };
                }
                $scope.securityPermissionsHooks.fixupPermissionItem = function(p) {
                    p.$useDisabled = p.create;
                    p.$createDisabled = false;
                };
                $scope.securityPermissionsHooks.fixupWithDefaultPermissionItem = function(p, d) {
                    if (d.use || d.$useDisabled) {
                        p.$useDisabled = true;
                    }
                    if (d.create || d.$createDisabled) {
                        p.$createDisabled = true;
                    }
                };
            }
        };
    });

    app.directive('businessAppRequirements', function(DataikuAPI, $rootScope) {
        return {
            restrict: 'A',
            templateUrl: '/templates/business-apps/business-app-requirements.html',
            scope: {
                businessApp: '=',
                onValid: '=',
                onInvalid: '=',
                showTitle: '<',
            },
            link: function($scope) {
                function markAsValid(valid) {
                    if (valid) {
                        $scope.onValid && $scope.onValid();
                    } else {
                        $scope.onInvalid && $scope.onInvalid();
                    }
                }

                function checkCodeEnv() {
                    if (!$scope.businessApp) {
                        return;
                    }
                    if ($scope.businessApp.desc.codeEnvName) {
                        DataikuAPI.codeenvs.getForBusinessApp($scope.businessApp.desc.id, $scope.businessApp.desc.codeEnvName).success(function(data) {
                            $scope.codeEnv = data;
                            markAsValid($scope.codeEnv !== null && $scope.codeEnv !== undefined);
                        }).error(function() {
                            setErrorInScope.apply($scope, arguments);
                            markAsValid(false);
                        });
                    } else {
                        markAsValid(true);
                    }
                }

                $scope.checkValidity = function() {
                    const codeEnvOk = !$scope.businessApp.desc.codeEnvName || $scope.codeEnv !== undefined;
                    markAsValid(codeEnvOk);
                }

                $scope.onCodeEnvValid = function() {
                    $scope.businessApp.codeEnvExist = true;
                    $scope.businessApp.codeEnvUpToDate = true;
                    markAsValid(true);
                }
                $scope.onCodeInvalid = function() {
                    markAsValid(false);
                }

                if ($scope.showTitle === undefined) {
                    $scope.showTitle = true;
                }
                $scope.$watch("businessApp", checkCodeEnv);
                $scope.appConfig = $rootScope.appConfig;
                $scope.codeEnv = null;
            }
        };
    });

    app.directive('businessAppCodeEnv', function(DataikuAPI, Dialogs, WT1, FutureProgressModal, $rootScope, MonoFuture, CodeEnvService) {
        return {
            restrict: 'A',
            templateUrl: '/templates/business-apps/modals/business-app-code-env.html',
            scope: {
                businessApp: '=',
                onValid: '=',
                onInvalid: '='
            },
            link: function($scope) {
                $scope.codeEnv = undefined;
                $scope.uiState = {
                    state: "DISPLAY"
                }

                $scope.newEnv = {
                    deploymentMode: 'BUSINESS_APP_MANAGED',
                    pythonInterpreter: 'PYTHON39', // will get overridden by getBusinessAppDefaultAvailableInterpreter later on
                    allContainerConfs: false, containerConfs: [],
                    allSparkKubernetesConfs: false, sparkKubernetesConfs: [],
                    rebuildDependentCodeStudioTemplates: "NONE" // we don't expect Business Applications envs to be used in Code Studios
                };

                $scope.defaultInterpreterFound = false;

                $scope.getCodeEnv = function() {
                    return DataikuAPI.codeenvs.getForBusinessApp($scope.businessApp.desc.id, $scope.businessApp.desc.codeEnvName).success(function(data) {
                        $scope.codeEnv = data;
                    }).error(function() {
                        $scope.onInvalid && $scope.onInvalid();
                        setErrorInScope.apply($scope, arguments);
                    });
                };

                $scope.buildNewCodeEnv = function(newEnv) {
                    DataikuAPI.codeenvs.createForBusinessApp($scope.businessApp.desc.id, newEnv).success(function(data) {
                        FutureProgressModal.show($scope, data, "Environment creation", undefined, 'static', false, true).then(function(result) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Creation result", result.messages, result.futureLog, undefined, 'static', false);
                            // what matters is that the env was properly created, others errors do not really matter
                            const isCodeEnvImported = result.messages && result.messages.messages && result.messages.messages.findIndex(message => message.code === "INFO_CODEENV_IMPORT_OK") > -1;
                            if (result.messages.error && !isCodeEnvImported) {
                                $scope.onInvalid && $scope.onInvalid();
                            } else {
                                DataikuAPI.codeenvs.getForBusinessApp($scope.businessApp.desc.id, $scope.businessApp.desc.codeEnvName).success(function(data) {
                                    $scope.codeEnv = data;
                                    $scope.onValid && $scope.onValid();
                                }).error(function() {
                                    $scope.onInvalid && $scope.onInvalid();
                                    setErrorInScope.apply($scope, arguments);
                                });
                            }
                        });
                    }).error(function() {
                        $scope.onInvalid && $scope.onInvalid();
                        setErrorInScope.apply($scope, arguments);
                    });
                };

                $scope.updateCodeEnv = function() {
                    DataikuAPI.codeenvs.updateForBusinessApp($scope.businessApp.desc.id).success(function(data) {
                        FutureProgressModal.show($scope, data, "Environment update").then(function(result) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                            $scope.getCodeEnv().then(() => {
                                if ($scope.codeEnv && $scope.codeEnv.isUptodate) {
                                    $scope.onValid && $scope.onValid();
                                }
                            });
                        });
                    }).error(function() {
                        $scope.onInvalid && $scope.onInvalid();
                        setErrorInScope.apply($scope, arguments);
                    });
                };

                $scope.isCurrentSelectedEnvUpToDate = function() {
                    return $scope.codeEnv && $scope.codeEnv.isUptodate;
                };

                const getBusinessAppDefaultAvailableInterpreter = (codeEnvLang) => {
                    if (codeEnvLang !== 'PYTHON') {
                        return;
                    }
                    return DataikuAPI.codeenvs.getBusinessAppDefaultAvailableInterpreter($scope.businessApp.desc.id).success(data => {
                        $scope.newEnv.pythonInterpreter = data;
                        $scope.defaultInterpreterFound = true;
                    }).error(function() {
                        // Do not break UI if call failed
                        $scope.defaultInterpreterFound = true;
                        setErrorInScope.apply($scope, arguments);
                    });
                }
                $scope.$watch("businessApp.codeEnvLang", getBusinessAppDefaultAvailableInterpreter);

                $scope.containerNames = [];
                DataikuAPI.containers.listNames(null, "USER_CODE").success(function(data) {
                    $scope.containerNames = data;
                }).error(function() {
                    $scope.onInvalid && $scope.onInvalid();
                    setErrorInScope.apply($scope, arguments);
                });
                $scope.sparkKubernetesNames = [];
                DataikuAPI.containers.listSparkNames().success(function(data) {
                    $scope.sparkKubernetesNames = data;
                }).error(function() {
                    $scope.onInvalid && $scope.onInvalid();
                    setErrorInScope.apply($scope, arguments);
                });

                $scope.codeEnvDeploymentModes = [
                    ['BUSINESS_APP_MANAGED', "Managed by DSS (recommended)"],
                    ['BUSINESS_APP_NON_MANAGED', "Managed manually"]
                ];
                $scope.possiblePythonInterpreters = [];
                if ($scope.businessApp.codeEnvSpec) {
                    const codeEnvSpec = $scope.businessApp.codeEnvSpec;
                    if (!codeEnvSpec.forceConda) {
                        $scope.possiblePythonInterpreters = $scope.possiblePythonInterpreters.concat(['CUSTOM']);
                    }
                    if (codeEnvSpec.acceptedPythonInterpreters && codeEnvSpec.acceptedPythonInterpreters.length > 0) {
                        $scope.possiblePythonInterpreters = $scope.possiblePythonInterpreters.concat(codeEnvSpec.acceptedPythonInterpreters);
                        $scope.newEnv.pythonInterpreter = codeEnvSpec.acceptedPythonInterpreters[0];
                    }
                }

                CodeEnvService.getPythonInterpreters().then(function(enrichedInterpreters) {
                    const allowed = new Set($scope.possiblePythonInterpreters);
                    $scope.enrichedPossiblePythonInterpreters = enrichedInterpreters.filter(i => allowed.has(i[0]));
                });

                $scope.getCodeEnv();
            }
        };
    });

    /*----- Business Application instance -----*/
    app.controller("BusinessAppInstanceController", function($scope, $controller, ProjectStatusService, $state, $stateParams, $rootScope, $filter, $element, CreateModalFromTemplate, DataikuAPI, TopNav, Debounce, ListFilter, Assert, DKUConstants, WT1, TaggingService, StateUtils, openDkuPopin, DetectUtils) {
        $controller("BusinessAppInstancesCommonController", { $scope });
        $controller("_ProjectsListBaseBehavior", { $scope });

        if (!$scope.preventSettingLocation) {
            TopNav.setLocation(TopNav.DSS_HOME);
        }
        TaggingService.fetchGlobalTags();
        $scope.getDefaultTagColor = TaggingService.getTagColor;

        $scope.uiState = {};
        $scope.displayMode = { mode: 'mosaic' };
        $scope.query = { tags: [], projectStatus: [], contributors: [], q: '' };
        $scope.sortBy = { mode: 'commit' };
        $scope.tagsMap = {};
        $scope.tagsList = [];
        $scope.contributorsMap = {};
        $scope.contributorsList = [];

        $scope.os = DetectUtils.getOS();

        const orderBy = $filter('orderBy');

        $scope.noAccessPermission = false;

        //Items rows for mosaic view
        const getMosaicRows = function(itemsList) {
            /* Compute display characteristics for mosaic mode */
            const tileW = 310;
            const margins = 40;
            const leftPaneWidth = 400 + 40; // don't forget padding

            let itemsPerRow = 1;
            let elementWidth = $element.width();
            elementWidth -= margins;
            elementWidth -= leftPaneWidth;

            if (elementWidth > tileW) {
                itemsPerRow = Math.floor(elementWidth / tileW);
            }

            const mosaicItemsPerRow = [];
            let i, j;
            for (i = 0, j = itemsList.length; i < j; i += itemsPerRow) {
                mosaicItemsPerRow.push(itemsList.slice(i, i + itemsPerRow));
            }
            return mosaicItemsPerRow;
        };

        const updateDisplayedItems = function() {
            $scope.filteredAppInstancesList = filterAppInstancesList($scope.appInstancesList, $scope.query);
            $scope.filteredAppInstancesList = sortAppInstancesList($scope.filteredAppInstancesList);
            $scope.filteredAppInstancesRows = getMosaicRows($scope.filteredAppInstancesList);
        };

        const debouncedResizeCB = Debounce().withDelay(200, 200).wrap(updateDisplayedItems);
        $(window).on("resize.appPageResize", debouncedResizeCB);
        $scope.$on("$destroy", function() {
            $(window).off("resize.appPageResize", debouncedResizeCB);
        });

        $scope.$on("topbarDrawerStateChanged", (_, args) => {
            if (!args.autoHide) {
                updateDisplayedItems();
            }
        });

        $scope.appInstancesList = [];
        $scope.fetchAppSummary = function() {
            $scope.appInstancesList = [];
            return DataikuAPI.businessApps.getSummary($stateParams.businessAppId).success(function(data) {
                $scope.appSummary = data;
                $scope.appInstancesList = $scope.appSummary.instances;

                // don't forget to re-generate the selectedProjects list (and trim it of now-deleted projects at the same time)
                $scope.selectedProjects = [...$scope.appInstancesList.filter(p => $scope.selectedProjects.findIndex(p2 => p.projectKey === p2.projectKey) >= 0)]
                updateDisplayedItems();
            }).error((data, status, headers) => {
                if (data && data.errorType) { // to check that the error is from the api call and not for example a proxy error
                    $scope.noAccessPermission = status === 403;
                }
                setErrorInScope.bind($scope)(data, status, headers);
            });
        }

        $scope.goToBusinessAppPage = function(businessAppId) {
            $state.transitionTo('businessapp.settings', { businessAppId: businessAppId });
        }

        $scope.createBusinessAppInstance = function() {
            CreateModalFromTemplate("/templates/business-apps/modals/business-app-new-instance.html", $scope, "BusinessAppNewInstanceModalController", function(newScope) {
                newScope.businessAppId = $scope?.appSummary?.appId;
                newScope.businessAppVersion = $scope?.appSummary?.appVersion;
                newScope.businessAppName = $scope?.appSummary?.label;
                newScope.defaultRunAsUser = $scope?.appSummary?.defaultRunAsUser;
                newScope.allowRunAsUserOverride = $scope?.appSummary?.allowRunAsUserOverride;
            }).then(function({ projectKey, webAppId }) {
                $state.go("projects.project.webapps.webapp.view", { projectKey, webAppId });
            }).catch($scope.fetchAppSummary); // Reload all instances, a new one might have been created (in case creation of instance succeeded but webapp failed to start)
        }

        $scope.clickOnAppInstance = function(appInstance, $event) {
            if ($scope.isPopupActive) {
                return;
            }
            $event.preventDefault();
            if ($event.ctrlKey || $event.metaKey) {
                toggleSelectProject(appInstance);
                window.getSelection().removeAllRanges(); //FF fix for messy text selection
            } else {
                $state.go('projects.project.webapps.webapp.view', {
                    projectKey: appInstance.projectKey,
                    webAppId: appInstance.webAppId
                });
            }
        };

        $scope.selectedProjects = [];

        function toggleSelectProject(project, expectSelected) {
            let index = $scope.selectedProjects.findIndex(p => p.projectKey === project.projectKey);
            if (index === -1) {
                if (expectSelected === true || expectSelected === undefined) {
                    $scope.selectedProjects.push(project);
                }
            } else {
                if (expectSelected === false || expectSelected === undefined) {
                    $scope.selectedProjects.splice(index, 1);
                }
            }
        }

        $scope.isProjectSelected = project => $scope.selectedProjects.findIndex(sp => sp.projectKey === project.projectKey) !== -1;

        $scope.selectAllInstances = function() {
            $scope.filteredAppInstancesList.forEach(p => toggleSelectProject(p, true))
        };
        $scope.unselectAllInstances = function() {
            $scope.filteredAppInstancesList.forEach(p => toggleSelectProject(p, false))
        };

        function toggleAllInstancesStartingFrom(project, selectAfter, expectSelected) {
            let idx = $scope.filteredAppInstancesList.findIndex(p => p.projectKey === project.projectKey);
            if (idx >= 0) {
                let affected = selectAfter ? $scope.filteredAppInstancesList.filter((p, i) => i >= idx) : $scope.filteredAppInstancesList.filter((p, i) => i <= idx);
                affected.forEach(p => toggleSelectProject(p, expectSelected))
            } else {
                // project not found, suspicious
            }
        }

        $scope.getProjectContributorDisplayList = function(contributors, maxDisplayedContributors) {
            if (contributors.length > maxDisplayedContributors) {
                return contributors.slice(0, maxDisplayedContributors - 1);
            }
            return contributors
        };

        // refactor status color handling out into a service for easier usage in separated scopes
        $scope.getProjectStatusColor = function(status) {
            return ProjectStatusService.getProjectStatusColor(status);
        }

        $scope.isFiltering = function() {
            return $scope.query.tags.length > 0 || $scope.query.contributors.length > 0 || $scope.query.projectStatus.length > 0 || $scope.isFullStringQuerying();
        };

        $scope.isFullStringQuerying = function() {
            return typeof ($scope.query.q) !== "undefined" && $scope.query.q.length > 0;
        };

        $scope.clearFilters = function() {
            $scope.query.tags = [];
            $scope.query.projectStatus = [];
            $scope.query.contributors = [];
            $scope.query.q = "";
        };

        $scope.toggleTag = function(tagTitle) {
            if (tagTitle) {
                var index = $scope.query.tags.indexOf(tagTitle);
                index > -1 ? $scope.query.tags.splice(index, 1) : $scope.query.tags.push(tagTitle);
            }
        };

        /**
         * Returns a list of app instances filtered by full text query, tags, users, status, path.
         * Keep apps that match at least one condition for each non-empty filtering category (text query, tags, contributors, status, path)
         * @param appInstancesList input list to filter
         * @param query object wrapping query attributes:
         *      - q: textQuery on which projects list will be filtered (looking through all project's attribute)
         *      - tags: list of tags to filter projects list (inclusive filtering - keep items that match at least one tag)
         *      - contributors: list of contributors to filter projects list (inclusive filtering - keep items that match at least one contributor)
         *      - projectStatus: list of projectStatus to filter projects list (inclusive filtering - keep items that match at least one project status)
         *      - path: path used to filter projects list (project's path needs to be equal to it, or an extension of it in case of full text filtering)
         * @returns {*}
         */
        function filterAppInstancesList(appInstancesList, query) {
            if ($scope.isFiltering()) {
                WT1.event('app-instances-filtering', {
                    tags: (query.tags && query.tags.length),
                    contributors: (query.contributors && query.contributors.length),
                    projectStatus: (query.projectStatus && query.projectStatus.length),
                    fullString: $scope.isFullStringQuerying()
                })
            }
            // Filtering on full text query
            return ListFilter.filter(appInstancesList || [], query.q).filter(app => {

                // Keep apps that have at least one of the tags selected in the 'Tags' filter (if there are any)
                if (query.tags && query.tags.length) {
                    if (!app.tags || !query.tags.some(tag => app.tags.includes(tag))) {
                        return;
                    }
                }

                // Keep apps that have at least one of the contributors selected in the 'Users' filter (if there are any)
                if (query.contributors && query.contributors.length) {
                    if (!app.contributors || !app.contributors.some(contributor => query.contributors.includes(contributor.login))) {
                        return;
                    }
                }

                // Keep apps that have at least one of the project status selected in the 'Status' filter (if there are any)
                if (query.projectStatus && query.projectStatus.length) {
                    if (query.projectStatus.indexOf(app.projectStatus) < 0) {
                        return;
                    }
                } else if (app.projectStatus === DKUConstants.ARCHIVED_PROJECT_STATUS) {
                    return;
                }

                return true;
            });
        }

        $scope.$watch("query", function(nv, ov) {
            if (!angular.equals(nv, ov)) {
                updateDisplayedItems();
            }
        }, true);
        $scope.$watch("sortBy", function(nv, ov) {
            if (!angular.equals(nv, ov)) {
                updateDisplayedItems();
            }
        }, true);

        /*
         * Sorting projects list
         */
        $scope.sortByModeTitles = Object.freeze({
            name: "Project Name",
            commit: "Last Modified",
            commit_for_user: "Last Modified By Me",
            status: "Status"
        });

        function sortAppInstancesList(appInstancesList) {
            if (!$scope.sortBy) {
                return;
            }
            switch ($scope.sortBy.mode) {
                case "name":
                    sortByName(appInstancesList);
                    break;
                case "status":
                    sortByStatus(appInstancesList);
                    break;
                case "commit":
                    appInstancesList = orderBy(appInstancesList, '-lastCommitTime');
                    break;
                case "commit_for_user":
                    appInstancesList = orderBy(appInstancesList, '-lastCommitTimeForUser');
                    break;
            }
            if ($scope.sortBy.isReversedSort) {
                appInstancesList.reverse();
            }
            return appInstancesList;
        }

        function sortByName(appInstancesList) {
            appInstancesList.sort(function(p1, p2) {
                return alphabeticalSort(p1.name, p2.name);
            });
        }

        function sortByStatus(appInstancesList) {
            Assert.inScope($rootScope, 'appConfig');
            const projectStatusNames = [];
            $rootScope.appConfig.projectStatusList.forEach(function(s) {
                projectStatusNames.push(s.name);
            })
            appInstancesList.sort(function(p1, p2) {
                if (p1.projectStatus && p2.projectStatus) {
                    var indexOfStatus1 = projectStatusNames.indexOf(p1.projectStatus);
                    var indexOfStatus2 = projectStatusNames.indexOf(p2.projectStatus);
                    return indexOfStatus1 < indexOfStatus2 ? -1 : indexOfStatus1 === indexOfStatus2 ? alphabeticalSort(p1.name, p2.name) : 1;
                } else if (p1.projectStatus) {
                    return -1;
                } else if (p2.projectStatus) {
                    return 1;
                } else {
                    return alphabeticalSort(p1.name, p2.name);
                }
            });
        }

        $scope.deleteSelectedInstances = function() {
            const projectKeys = $scope.selectedProjects.map(p => p.projectKey);
            $scope.deleteBusinessAppInstances(projectKeys)
                .then($scope.fetchAppSummary)
                .catch(setErrorInScope.bind($scope));
        };

        $scope.openInstanceMenu = function(project, $event) {
            let template = `<ul class="dropdown-menu projects-dropdown-menu" >
        <li><a ng-click="toggleSelection()">{{selectedProjects.indexOf(project) < 0 ? 'Select' : 'Unselect'}}</a></li>
        <li><a ng-click="selectBefore()">Select all before</a></li>
        <li><a ng-click="selectAfter()">Select all after</a></li>
        <li><a ng-click="unselectBefore()">Unselect all before</a></li>
        <li><a ng-click="unselectAfter()">Unselect all after</a></li>
        </ul>`;
            let callback = newScope => {
                newScope.projects = newScope.selectedProjects.length > 0 ? newScope.selectedProjects : [project];
                newScope.project = project;
                newScope.appConfig = $scope.appConfig;
                newScope.toggleSelection = () => toggleSelectProject(project);
                newScope.selectAfter = () => toggleAllInstancesStartingFrom(project, true, true);
                newScope.selectBefore = () => toggleAllInstancesStartingFrom(project, false, true);
                newScope.unselectAfter = () => toggleAllInstancesStartingFrom(project, true, false);
                newScope.unselectBefore = () => toggleAllInstancesStartingFrom(project, false, false);
            };
            let isElsewhere = (_, e) => $(e.target).parents('.dropdown-menu').length === 0;
            $scope.lockForPopup();
            let dkuPopinOptions = {
                template: template,
                isElsewhere: isElsewhere,
                popinPosition: 'CLICK',
                callback: callback,
                onDismiss: $scope.unlockAfterPopup
            };
            $scope.popupDismiss = openDkuPopin($scope, $event, dkuPopinOptions);
        };

        $scope.fetchAppSummary();
    });

    app.controller("BusinessAppNewInstanceModalController", function($scope, $rootScope, DataikuAPI, Dialogs, FutureProgressModal, WT1) {
        function isProjectKeyUnique(value) {
            return !$scope.allProjectKeys || $scope.allProjectKeys.indexOf(value) < 0;
        }

        function generateValidProjectKey(projectName) {
            const slug = projectName.toUpperCase().replace(/\W+/g, "");
            let cur = slug;
            let i = 2;
            while (!isProjectKeyUnique(cur)) {
                cur = slug + "_" + (i++);
            }
            return cur;
        }

        $scope.$watch("context.projectKey", function(nv) {
            $scope.uniq = !nv || isProjectKeyUnique(nv);
        });

        $scope.$watch("context.projectName", function(nv) {
            if (nv) {
                $scope.context.projectKey = generateValidProjectKey(nv);
            }
        });

        $scope.create = function() {
            WT1.event('business-app-create-instance', { businessAppId: $scope.businessAppId, businessAppVersion: $scope.businessAppVersion })
            const parentScope = $scope.$parent;
            const runAsUser = $scope.allowRunAsUserOverride ? $scope.context.runAsUser : null;
            DataikuAPI.businessApps.createInstance($scope.businessAppId, $scope.context.projectKey, $scope.context.projectName, $scope.context.shortDesc, runAsUser)
                .success(data => {
                    FutureProgressModal.show(parentScope, data, "Create instance").then(function(result) {
                        if (result && result.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly(parentScope, "Create instance", result).then(function() {
                                if (result.error) {
                                    $scope.dismiss();
                                } else {
                                    $scope.resolveModal({ projectKey: result.projectKey, webAppId: result.businessAppWebAppId });
                                }
                            });
                        } else {
                            $scope.resolveModal({ projectKey: result.projectKey, webAppId: result.businessAppWebAppId });
                        }
                    }).catch(setErrorInScope.bind($scope));
                }).error(setErrorInScope.bind($scope));
        }

        // Initialization
        $scope.context = {
            projectKey: "",
            projectName: "",
            shortDesc: "",
            runAsUser: ""
        }
        $scope.usersLoaded = false;
        DataikuAPI.projects.listAllKeys().success(function(data) {
            $scope.allProjectKeys = data;
            // Initialize the project name with the name of the Business Application by default
            if ($scope.context.projectName === "" && $scope.businessAppName) {
                $scope.context.projectName = $scope.businessAppName;
                $scope.context.projectKey = generateValidProjectKey($scope.businessAppName)
            }
        }).error(setErrorInScope.bind($scope));

        // Load users list for runAs selector if allowRunAsUserOverride=true
        $scope.$watch("allowRunAsUserOverride", function() {
            if ($scope.allowRunAsUserOverride && !$scope.usersLoaded) {
                $scope.usersLoaded = true;
                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = data.map(user => '@' + user.login);
                    if ($scope.defaultRunAsUser && $scope.defaultRunAsUser !== "%%INSTANCE_OWNER%%") {
                        $scope.context.runAsUser = $scope.defaultRunAsUser;
                    } else {
                        // Default to instance owner (current user)
                        $scope.context.runAsUser = $rootScope.appConfig.login;
                    }
                }).error(setErrorInScope.bind($scope));
            }
        })
    });
})();
