(function() {
'use strict';

const app = angular.module('dataiku.lambda', []);


app.controller("LambdaServicesListController", function($scope, $controller, $stateParams, DataikuAPI, Dialogs, $state,
    TopNav, WT1) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_MORE, "lambda", TopNav.TABS_NONE, null);

    $scope.sortBy = [
        {
            value: 'id',
            label: 'Id'
        },
        {
            value: 'name',
            label: 'Name'
        },
        {
            value: 'endpoints.length',
            label: 'Number of endpoints'
        }
    ];
    $scope.sortCookieKey = 'lambdaservices';
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
        },
        filterParams: {
            userQueryTargets: ["id",'tags'],
            propertyRules: {tag:"tags"},
        },
        orderQuery: "name",
        orderReversed: false,
    }, $scope.selection || {});

    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.lambda.services.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = data.items;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };
    $scope.list() ;

    $scope.goToItem = function(data) {
        $state.go("projects.project.lambdaservices.service.endpoints", {projectKey: $stateParams.projectKey, serviceId: data.id});
    }

    $scope.newService = function() {
        Dialogs.prompt($scope, "Create API service", "Unique service ID", "", { pattern: "[\\w-]+" })
            .then(function(id) {
                WT1.event('create-api-service');
                DataikuAPI.lambda.services.create($stateParams.projectKey, id)
                    .success(function() {
                        $state.go("projects.project.lambdaservices.service.endpoints", {projectKey: $stateParams.projectKey, serviceId: id});
                    }).error(setErrorInScope.bind($scope));
            });
    };

    $scope.customMassDeleteSelected = DataikuAPI.lambda.services.deleteMulti;
});

app.directive('lambdaServiceRightColumnSummary', function(DataikuAPI, $stateParams, ActivityIndicator) {
    return {
        templateUrl :'/templates/lambda/right-column-summary.html',
        link: function(scope) {
            scope.refreshData = function() {
                DataikuAPI.lambda.services.getSummary(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.id)
                .success(function(summary) {
                    scope.serviceSummary = summary;
                }).error(setErrorInScope.bind(scope));
            };

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) return;
                scope.refreshData();
            });

            scope.$on("objectSummaryEdited", function() {
                return DataikuAPI.lambda.services.save($stateParams.projectKey, scope.serviceSummary.object).success(
                    function(data) {
                        ActivityIndicator.success("Saved");
                    }).error(setErrorInScope.bind(scope));
            });
        }
    }
});

app.controller("LambdaServiceSummaryController", function($scope, $rootScope, $stateParams, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.TOP_HOME, 'lambda', null, 'summary');

    $scope.$on("objectSummaryEdited", $scope.saveService);

    $scope.$on('customFieldsSummaryEdited', function(event, customFields) {
        $scope.saveCustomFields(customFields);
    });
});


app.controller("LambdaServiceBaseController", function($scope, $state, $stateParams, $q, DataikuAPI, TopNav, Logger, Dialogs, Fn, FutureProgressModal, CreateModalFromTemplate, $rootScope, WT1, SavedModelsService) {
    TopNav.setItem(TopNav.ITEM_LAMBDA_SERVICE, $stateParams.serviceId);

    $scope.savedService = null; //for dirtyness detection
    function getSummary() {
        return DataikuAPI.lambda.services.getSummary($stateParams.projectKey, $stateParams.serviceId).success(function(data) {
            $scope.service = data.object;
            $scope.timeline = data.timeline;
            $scope.interest = data.interest;
            $scope.savedService = angular.copy($scope.service);

            TopNav.setItem(TopNav.ITEM_LAMBDA_SERVICE, $stateParams.serviceId, {name: $scope.service.name});
            TopNav.setPageTitle($scope.service.name + " - API Service");
        }).error(setErrorInScope.bind($scope));
    }

    getSummary().then(function() {
        DataikuAPI.lambda.devServer.getStatus($stateParams.projectKey).success(function(data) {
            $scope.lambdaDevServerStatus = data;
        });
    })

    $scope.serviceIsDirty = function() {
        return $scope.service && $scope.savedService && !angular.equals($scope.service, $scope.savedService);
    };

    $scope.saveServiceIfNeeded = function() {
        if ($scope.serviceIsDirty()) {
            return $scope.saveService();
        }
        return $q.when(null);
    };

    $scope.saveService = function() {
        return DataikuAPI.lambda.services.save($stateParams.projectKey, $scope.service).success(function(data) {
            // Update API keys, some may have been generated on save
            $scope.service.authRealm.queryKeys = data.authRealm.queryKeys;
            // Update OpenAPI fields since in case of automatic generation it can change (i.e. changing the endpoint name)
            $scope.service.endpoints.forEach((ep) => {
                if (ep.openAPI) {
                    ep.openAPI = data.endpoints.find(e => e.id === ep.id).openAPI;
                }
            });
            $scope.savedService = angular.copy($scope.service);
        }).error(setErrorInScope.bind($scope));
    };

    const allowedTransitions = [
        'projects.project.lambdaservices.service.summary',
        'projects.project.lambdaservices.service.endpoints',
        'projects.project.lambdaservices.service.packages',
        'projects.project.lambdaservices.service.config',
    ];
    checkChangesBeforeLeaving($scope, $scope.serviceIsDirty, null, allowedTransitions);

    // Suggest next non-existing name (preferably vX+1, else based on last package)
    $scope.suggestNextName = function(names) {
        if (!names || !names.length) return "v1";
        let matches = names.filter(Fn.regexp(/^v\d+$/i));
        let v;
        let last;
        if (matches.length) {
            v = parseInt(matches[0].substr(1)) + 1;
            while (matches.indexOf("v" + v) >= 0) v++;
            return "v" + v;
        }
        last = names.pop();
        matches = last.match(/(.*)(\d+)(\D*)$/); // rightmost digits
        if (matches) {
            v = parseInt(matches[2]) + 1
            while (names.indexOf(matches[1] + v + matches[3]) >= 0) v++;
            return matches[1] + v + matches[3];
        }
        return last + "2";
    };

    $scope.createNewVersion = function () {
        $scope.saveServiceIfNeeded().then(() => CreateModalFromTemplate(
            "/templates/lambda/new-service-version-modal.html",
            $scope,
            "NewServiceVersionModalController"
        ));
    };

    $scope.publishVersionOnDeployer = function(versionId, canChangeVersion) {
        if ($rootScope.appConfig.remoteDeployerMisconfigured) {
            Dialogs.ack($scope, 'Remote deployer not properly configured',
                'To push to a remote deployer, you must previously register it in the DSS instance settings (admin rights required)');
        } else {
            if (!versionId) {
                DataikuAPI.lambda.packages.list($stateParams.projectKey, $stateParams.serviceId)
                    .success(function(packages) {
                        $scope.publishVersionOnDeployer($scope.suggestNextName(packages.map(p => p.id)), true);
                    }).error(setErrorInScope.bind($scope));
            } else {
                $scope.saveService().then(function() {
                    CreateModalFromTemplate("/templates/lambda/publish-version-on-deployer-modal.html", $scope, "PublishVersionOnDeployerModalController", function(modalScope) {
                        modalScope.uploadParams = {canChangeVersion, versionId};
                        modalScope.dssExternalURL = $rootScope.appConfig.dssExternalURL;
                        modalScope.isDSSAdmin = $scope.isDSSAdmin();
                    });
                });
            }
        }
    };

    $scope.deployToDevServer = function() {
        Logger.info("Deploying to dev server");
        let firstPromise = $q.when(null);
        if ($scope.serviceIsDirty()) {
            firstPromise = $scope.saveService();
        }
        return firstPromise.then(function() {
            const promise = DataikuAPI.lambda.services.deployDev($stateParams.projectKey, $stateParams.serviceId);
            promise.error(setErrorInScope.bind($scope));
            $scope.$broadcast("devServerDeploymentStarted", promise);

            promise.then(function(data) {
                $scope.lambdaDevServerStatus = data.data.devServerState;
            });
            return promise;
        });
    };

    $scope.restartAndDeployToDevServer = function() {
        Logger.info("Restarting dev server");
        DataikuAPI.lambda.devServer.stopDevServer($stateParams.projectKey)
            .success(function() {
                $scope.deployToDevServer();
            })
            .error(setErrorInScope.bind($scope))
    };

    $scope.stopDevServer = function() {
        Logger.info("Stopping dev server");
        DataikuAPI.lambda.devServer.stopDevServer($stateParams.projectKey)
            .success(function(data) {
                $scope.lambdaDevServerStatus = {
                    status: "STOPPED"
                };
            })
            .error(setErrorInScope.bind($scope))
    }

    $scope.saveCustomFields = function(newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'LAMBDA_SERVICE'});
        let oldCustomFields = angular.copy($scope.service.customFields);
        $scope.service.customFields = newCustomFields;
        return $scope.saveService().then(function() {
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.service.customFields);
            }, function() {
                $scope.service.customFields = oldCustomFields;
            });
    };

    $scope.editCustomFields = function(editingTabIndex = 0) {
        if (!$scope.service) {
            return;
        }
        let modalScope = angular.extend($scope, {objectType: 'LAMBDA_SERVICE', objectName: $scope.service.name, objectCustomFields: $scope.service.customFields, editingTabIndex});
        CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
            $scope.saveCustomFields(customFields);
        });
    };
});


app.controller("LambdaServiceConfigController", function($stateParams, $scope, $state, DataikuAPI, TopNav, Dialogs, CreateModalFromTemplate, Fn) {
    TopNav.setLocation(TopNav.TOP_HOME, "lambda", TopNav.TABS_LAMBDA, "conf");

    $scope.uiState = {
        settingsPane: 'authorization'
    };
});


app.controller("LambdaServicePackagesController", function($stateParams, $scope, $state, DataikuAPI, TopNav, Dialogs, CreateModalFromTemplate) {
    TopNav.setLocation(TopNav.TOP_HOME, "lambda", TopNav.TABS_LAMBDA, "packages");

    $scope.noTags = true;
    $scope.noWatch = true;
    $scope.noStar = true;

    $scope.sortBy = [
        {
            value: 'id',
            label: 'Id'
        },
        {
            value: 'mtime',
            label: 'Built on'
        },
        {
            value: 'created By',
            label: 'Created By'
        }
    ];
    $scope.sortCookieKey = 'packages';
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: ''
        },
        filterParams: {
            userQueryTargets: ["id"]
        },
        orderQuery: "id",
        orderReversed: false,
    }, $scope.selection || {});

    function listPackages() {
        DataikuAPI.lambda.packages.list($stateParams.projectKey, $stateParams.serviceId)
            .success(function(packages) {
                $scope.packages = packages;
                $scope.listItems = packages;
            }).error(setErrorInScope.bind($scope));
    }
    listPackages();
    $scope.$on("packageReady", listPackages);



    $scope.deletePackage = function(packageId) {
        Dialogs.confirmSimple($scope, "Are you sure you want to delete package <code>" + packageId + "</code>?")
            .then(function() {
                DataikuAPI.lambda.packages.delete($stateParams.projectKey, $stateParams.serviceId, packageId)
                    .success(listPackages)
                    .error(setErrorInScope.bind($scope));
            });
    };

    $scope.downloadPackage = function(packageId) {
        $('body').append(['<iframe src="/dip/api/lambda-services/package/download?projectKey=',
            encodeURIComponent($stateParams.projectKey),
            "&serviceId=", encodeURIComponent($stateParams.serviceId),
            "&packageId=", encodeURIComponent(packageId),
            '"></iframe>'
        ].join(''));
    };

    $scope.deletePackageSelected = function() {
        if (_.isEmpty($scope.selection.selectedObjects)) {
            return;
        } else if ($scope.selection.selectedObjects.length == 1) {
            $scope.deletePackage($scope.selection.selectedObject.id);
        } else {
            Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the selected API packages?").then(function() {
                $scope.selection.selectedObjects.forEach(o => DataikuAPI.lambda.packages.delete($stateParams.projectKey, $stateParams.serviceId, o.id)
                    .success(listPackages)
                    .error(setErrorInScope.bind($scope)));
            });
        }
    };

    $scope.downloadPackageSelected = function() {
        if (!_.isEmpty($scope.selection.selectedObjects)){
            $scope.selection.selectedObjects.forEach(o => $scope.downloadPackage(o.id));
        }
    };

});

app.directive('lambdaPackageRightColumnSummary', function(DataikuAPI, $stateParams, ActivityIndicator) {
    return {
        templateUrl :'/templates/lambda/packages-right-column-summary.html',
        link: function(scope) {
            scope.refreshData = function(packageId) {
                DataikuAPI.lambda.packages.getSummary($stateParams.projectKey, $stateParams.serviceId, packageId)
                .success(function(summary) {
                    scope.packageSummary = summary;
                }).error(setErrorInScope.bind(scope));

            };

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) return;
                scope.refreshData(nv.id);
            });
        }
    }
});

app.controller('NewEndpointFromSavedModelModalController', function($scope, $stateParams, DataikuAPI, StateUtils, WT1) {
    DataikuAPI.lambda.services.list($stateParams.projectKey).success(function(lambdaServices) {
        $scope.lambdaServices = lambdaServices;
    }).error(setErrorInScope.bind($scope));

    $scope.createEndPoint = function() {
        WT1.tryEvent('add-model-api-endpoint', () => {
            return $scope.event;
        });
        DataikuAPI.lambda.services.addEndpoint($stateParams.projectKey, $scope.service.id, $scope.service.create, $scope.ep)
            .success(function(service) {
                $scope.resolveModal();
                StateUtils.go.lambdaService(service.id, service.projectKey);
            })
            .error(setErrorInScope.bind($scope));
    };
});


app.controller('NewServiceVersionModalController', function(
    $scope, $rootScope, $stateParams, DataikuAPI, FutureProgressModal, Dialogs, Fn
) {
    let warningMessageContent = "";
    if (!$rootScope.appConfig.dssExternalURL) {
        warningMessageContent = "The link to this API designer from the API deployer will not work because <strong>DSS URL</strong> is not set. ";
        if ($scope.isDSSAdmin()) {
            warningMessageContent += 'Configure it in <a class="tab" href="/admin/general/notifications/">DSS global settings</a>.';
        } else {
            warningMessageContent += 'Please contact your administrator to configure it.';
        }
    }
    $scope.warningMessageContent = warningMessageContent;
    DataikuAPI.lambda.packages.list($stateParams.projectKey, $stateParams.serviceId).success(
        (listedPackageData) => {
            const names = listedPackageData.map(Fn.prop("id"));
            $scope.serviceVersionRegex = "^(?!\\.)[\\w\\.-]+$";
            $scope.newServiceVersionId = $scope.suggestNextName(names);
        }
    );

    $scope.ok = () => {
        DataikuAPI.lambda.services.startPreparePackage(
            $stateParams.projectKey,
            $stateParams.serviceId,
            $scope.newServiceVersionId,
            $scope.releaseNotes
        ).success((data) => {
            FutureProgressModal.show(
                $rootScope, data, "Build package"
            ).then((result) => {
                if (result && result.anyMessage) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Build result", result, result.futureLog);
                }
                $rootScope.$broadcast("packageReady");
            });

            $scope.resolveModal();
        }).error(setErrorInScope.bind($scope));
    }
});


app.controller('PublishVersionOnDeployerModalController', function($scope, $rootScope, $stateParams, $q, DataikuAPI, WT1, FutureProgressModal, ActivityIndicator, Dialogs, StringUtils) {
    $scope.publishedServiceIds = [];
    let suggestedServiceId;
    $scope.$watch('uploadParams.targetService.createServiceMessage', function(nv, ov) {
        if (nv) {
            $scope.uploadParams.targetService.serviceBasicInfo.id = suggestedServiceId;
        }
    });

    DataikuAPI.apideployer.client.listPublishedServices()
        .success(function(response) {
            $scope.publishedServices = response.filter(serviceStatus => serviceStatus.canWrite).sort((a, b) => a.serviceBasicInfo.name.localeCompare(b.serviceBasicInfo.name));
            suggestedServiceId = StringUtils.transmogrify($stateParams.serviceId,
                                                          $scope.publishedServices.map(_ => _.serviceBasicInfo.id),
                                                          (count, name) => `${name}-${count}`);
            $scope.publishedServices.unshift({createServiceMessage: "Create a new service...", packages: [], serviceBasicInfo: {id: suggestedServiceId}});
            $scope.publishedServiceIds = $scope.publishedServices.map(function(serviceStatus) {
                if (serviceStatus.createServiceMessage || (serviceStatus.serviceBasicInfo.id === serviceStatus.serviceBasicInfo.name)) return "";
                return serviceStatus.serviceBasicInfo.id;
            });
            $scope.uploadParams.targetService = $scope.publishedServices.find(service => service.serviceBasicInfo.id === $stateParams.serviceId);
            if (!$scope.uploadParams.targetService || $scope.uploadParams.targetService.packages.find(version => version.id === $scope.uploadParams.versionId)) {
                $scope.uploadParams.targetService = $scope.publishedServices[0];
            }
        })
        .error(setErrorInScope.bind($scope));

    $scope.ok = () => {
        function createAVersionIfNecessary() {
            if ($scope.uploadParams.canChangeVersion) {
                const deferred = $q.defer();
                DataikuAPI.lambda.services.startPreparePackage(
                    $stateParams.projectKey,
                    $stateParams.serviceId,
                    $scope.uploadParams.versionId,
                    $scope.uploadParams.releaseNotes
                ).success(data => {
                    FutureProgressModal.show(
                        $scope, data, "Build package"
                    ).then((result) => {
                        if (result && result.anyMessage) { // undefined in case of abort
                            Dialogs.infoMessagesDisplayOnly(
                                $scope, "Build result", result, result.futureLog
                            ).then($scope.resolveModal)
                        } else {
                            $rootScope.$broadcast("packageReady");
                            deferred.resolve();
                        }
                    });
                }).error(setErrorInScope.bind($scope));
                return deferred.promise;
            }
            return $q.when(null);
        }

        createAVersionIfNecessary().then(function() {
            DataikuAPI.lambda.packages.publishToAPIDeployer($scope.service.projectKey, $scope.service.id, $scope.uploadParams.versionId, $scope.uploadParams.targetService.serviceBasicInfo.id)
                .success(function() {
                    if ($rootScope.appConfig.deployerMode == 'LOCAL') {
                        ActivityIndicator.success(`Service published on API deployer! <a href="/api-deployer/services/${$scope.uploadParams.targetService.serviceBasicInfo.id}/?versions=${$scope.uploadParams.versionId}"  target="_blank">Open API deployer.</a>`, 5000);
                    } else if ($rootScope.appConfig.deployerURL) {
                        const deployerURL = $rootScope.appConfig.deployerURL + '/api-deployer/services/' + $scope.uploadParams.targetService.serviceBasicInfo.id + "/?versions=" + $scope.uploadParams.versionId;
                        ActivityIndicator.success(`Service published on API deployer! <a href="${deployerURL}" target="_blank">Open API deployer.</a>`, 5000);
                    }
                    $scope.resolveModal();

                    // The controller and the related modal are used for two different use cases,
                    // and canChangeVersion is useful to distinguish between them
                    if ($scope.uploadParams.canChangeVersion) {
                        WT1.event("create-and-publish-to-api-deployer");
                    } else {
                        WT1.event("publish-to-api-deployer");
                    }
                })
                .error(setErrorInScope.bind($scope));
        });
    };
});

app.controller('ConfigureMonitoringModalController', function($scope, $stateParams, DataikuAPI, ActivityIndicator, $state, WT1){
    $scope.ok = function() {
        DataikuAPI.lambda.services.monitoring.create($stateParams.projectKey, $scope.deploymentId, $scope.endpoint.id)
            .success(function(data) {
                ActivityIndicator.success(`Monitoring created ! <a href="${$state.href('projects.project.flow', {id : 'dataset_' + $stateParams.projectKey + "." + data.datasetName })}">
                    View in flow 
                </a>`, 5000);
                $scope.resolveModal();
                $scope.reloadDeployments();
                WT1.event('deployment-monitoring-created');
            })
            .catch((err) => {
                WT1.event('deployment-monitoring-creation-failure');
                setErrorInScope.bind($scope)(err);
            });
    }
})

app.controller('DeleteMonitoringModalController', function($scope, $stateParams, DataikuAPI, ActivityIndicator, WT1){
    $scope.ok = function() {
        DataikuAPI.lambda.services.monitoring.delete($stateParams.projectKey, $scope.deploymentId, $scope.endpointId, $scope.deleteObjects)
            .success(function() {
                ActivityIndicator.success("Monitoring deleted");
                $scope.resolveModal();
                $scope.reloadDeployments();
                WT1.event('deployment-monitoring-unlinked')
            })
            .catch((err) => {
                WT1.event('deployment-monitoring-unlinked-failure')
                setErrorInScope.bind($scope)(err);
            });
    }
})

app.component('simpleLlmTestQueriesResponse', {
    bindings: {
        response: '<'
    },
    template: `
        <p ng-if="$ctrl.response.result">
            <span ng-if="$ctrl.response.result.formattedOutput">{{$ctrl.response.result.formattedOutput}}</span>
            <span ng-if="!$ctrl.response.result.formattedOutput && $ctrl.response.result.validationStatus ==  'INVALID'">Validation failed: {{$ctrl.response.result.validationMessage}}</span>
        </p>`
});

app.service("LambdaServicesService", function($rootScope, CreateModalFromTemplate, DataikuAPI, $stateParams, MetricsUtils, PMLFilteringService, $filter) {
    this.newEndpointFromSavedModel = function(savedModel) {
        const miniTask = savedModel.miniTask;
        CreateModalFromTemplate("/templates/lambda/new-endpoint-from-saved-model-modal.html", $rootScope, 'NewEndpointFromSavedModelModalController', function(modalScope) {
            modalScope.event = {
                savedModelType: savedModel.savedModelType,
                predictionType: miniTask.predictionType,
                taskType: miniTask.taskType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            };
            modalScope.service = {
                create: true,
                id: null
            };
            if (miniTask.taskType == "CLUSTERING") {
                modalScope.ep = {
                    type: 'STD_CLUSTERING',
                    modelRef: savedModel.id,
                    useJava: false
                };
            } else if (miniTask.predictionType === "TIMESERIES_FORECAST") {
                modalScope.ep = {
                    type: 'STD_FORECAST',
                    modelRef: savedModel.id,
                    useJava: false
                };
            }
             else if (["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes(miniTask.predictionType)) {
                modalScope.ep = {
                    type: 'STD_CAUSAL_PREDICTION',
                    modelRef: savedModel.id,
                    computePropensity: miniTask.modeling.propensityModeling.enabled,
                    useJava: false
                };
            } else {
                modalScope.ep = {
                    type: 'STD_PREDICTION',
                    modelRef: savedModel.id,
                    useJava: true
                };
            }
             modalScope.savedModelIcon = $filter("savedModelSubtypeToIcon")(
                 miniTask.taskType,
                 miniTask.backendType,
                 miniTask.predictionType,
                 savedModel.savedModelType,
                 undefined
             );
             modalScope.savedModelColor = $filter("savedModelTypeToClassColor")(savedModel.savedModelType, true);

            modalScope.savedModelName = savedModel.name;
        });
    };

    this.openConfigureMonitoringModal = function(serviceId, endpoint, deploymentId, reloadDeployments) {
        return CreateModalFromTemplate("/templates/lambda/monitoring-configuration-modal.html", $rootScope, 'ConfigureMonitoringModalController', function(modalScope) {
            modalScope.endpoint = endpoint;
            modalScope.deploymentId = deploymentId;
            modalScope.reloadDeployments = reloadDeployments;
        });
    }

    this.openMonitoringDetailsModal = function(deploymentMonitoring, projectKey) {
        return CreateModalFromTemplate("/templates/lambda/monitoring-details-modal.html", $rootScope, null, function(modalScope) {
            modalScope.infoMessages = {
                "apiLogs": "This dataset contains the API prediction logs. It is configured by your administrator on the API infrastructure and uses Dataiku Event Server to store these logs in a file storage connection.",
                "er": "This Evaluate recipe computes a Model Evaluation based on the prediction logs dataset (usually on a single timeframe defined using partitions).",
                "mes": "This Model Evaluation Store is the central repository of all Model Evaluations for the model, giving in-depth understanding of the model behavior. It is also used to store Metrics and compute checks usable to automate actions on model montoring through scenarios."
            };
            modalScope.deploymentMonitoring = deploymentMonitoring;
            modalScope.projectKey = projectKey;

            const getMetricFormatted = metric => ({[PMLFilteringService.getEvaluationMetricName(metric)]: MetricsUtils.getMetricValue(modalScope.latestModelEvaluation.metrics, metric)});

            DataikuAPI.modelevaluationstores.getFullInfo($stateParams.projectKey, deploymentMonitoring.mesId).success(function(mesData){
                modalScope.modelEvaluationStore = mesData.evaluationStore;
                DataikuAPI.modelevaluationstores.getLatestEvaluationDetails($stateParams.projectKey, deploymentMonitoring.mesId)
                    .success(function(meData) {
                        if (meData === ""){
                            modalScope.latestModelEvaluation = null;
                            return;
                        }
                        modalScope.latestModelEvaluation = meData;
                        if (!modalScope.modelEvaluationStore.displayParams.displayedMetrics){
                            modalScope.displayedMetrics = MetricsUtils.getDefaultMesMetrics(meData.evaluation.predictionType, meData)
                                .map(x => x[0])
                                .filter(x => x)
                                .reduce((pre, metric) => ({...pre, ...getMetricFormatted(metric)}), {});
                            // Remove empty ones on default display
                            Object.entries(modalScope.displayedMetrics).forEach(([metric, value]) => {if (value === '-') delete modalScope.displayedMetrics[metric]});
                        } else {
                            modalScope.displayedMetrics = modalScope.modelEvaluationStore.displayParams.displayedMetrics.reduce((pre, metric) => ({...pre, ...getMetricFormatted(metric)}), {});
                        }
                    })
                    .error(setErrorInScope.bind(modalScope));
            }).error(setErrorInScope.bind(modalScope));
        });
    }

    this.deleteMonitoringModal = function(deploymentId, endpointId, reloadDeployments) {
        return CreateModalFromTemplate("/templates/lambda/monitoring-delete-modal.html", $rootScope, 'DeleteMonitoringModalController', function(modalScope) {
            modalScope.deploymentId = deploymentId;
            modalScope.endpointId = endpointId;
            modalScope.reloadDeployments = reloadDeployments;
            modalScope.deleteObjects = true;
        })
    }
});


})();
