(function() {
    'use strict';
	var app = angular.module('dataiku.recipes');

	app.factory("ComputableSchemaRecipeSave", function(DataikuAPI, CreateModalFromTemplate, $q, $stateParams, $rootScope, ActivityIndicator, Logger, Dialogs, RecipesUtils) {

        /** If the only incompatibilities are column description changes, do not check "Drop and recreate" by default */
		const checkDropAndRecreateByDefault = (incompatibilities) => {
		    // !! SYNC "Column description changed for" - Keep the message in sync with the one to match in SchemaComparator.java#findDifferences
		    return incompatibilities && incompatibilities.some(incompatibility => !incompatibility.includes("Column description changed for"))
		}

		var setFlags = function(datasets, stricter) {
            $.each(datasets, function(idx, val) {
                if (!val) {
                    return;
                }
                val.dropAndRecreate = checkDropAndRecreateByDefault(val.incompatibilities);
                val.synchronizeMetastore = val.incompatibilities.length > 0 && val.isHDFS;
            });
		};

		var getUpdatePromises = function(computables){
			var promises = [];
            $.each(computables, function(idx, val) {
                let extraOptions = {};
                if (val.type == 'STREAMING_ENDPOINT') {
                    extraOptions.ksqlParams = val.ksqlParams;
                }
                promises.push(DataikuAPI.flow.recipes.saveOutputSchema($stateParams.projectKey,
                    val.type, val.type == "DATASET" ? val.datasetName : val.id, val.newSchema,
                    val.dropAndRecreate, val.synchronizeMetastore, extraOptions));
            });
            return promises;
        }
        var getRecipePromises = function(data){
            var promises = [];
            if (data.updatedRecipe && data.updatedPayload) {
                promises.push(DataikuAPI.flow.recipes.save($stateParams.projectKey,
                    data.updatedRecipe, data.updatedPayload,
                    "Accept recipe update suggestion"));
            }
            return promises;
        }

        var displayPromisesError = function(errRet) {
            var scope = this;
			let errDetails;
            if (errRet.data) {
                errDetails = getErrorDetails(errRet.data, errRet.status, errRet.headers, errRet.statusText);
            } else {
                errDetails = getErrorDetails(errRet[0].data, errRet[0].status, errRet[0].headers, errRet[0].statusText);
            }
            Dialogs.displaySerializedError(scope, errDetails)
        }

		return {
            decorateChangedDatasets: setFlags,
            getUpdatePromises: getUpdatePromises,
            getRecipePromises: getRecipePromises,

			// The scope must be a recipe scope and contain the "doSave" function
			handleSave : function($scope, recipeSerialized, serializedData, deferred) {
	            var doSave = function(){
	                $scope.baseSave(recipeSerialized, serializedData).then(function(){
	                    deferred.resolve("Save done");
	                }, function(error) {
	                	Logger.error("Could not save recipe");
	                	deferred.reject("Could not save recipe");
	                })
	            }

	            DataikuAPI.flow.recipes.getComputableSaveImpact($stateParams.projectKey,
	             recipeSerialized, serializedData).success(function(data) {
	             	const allPreviousSchemasWereEmpty = data.computables.every(x => x.previousSchemaWasEmpty)

	             	if (data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty) {
						Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                        setFlags(data.computables, true);
                        $q.all(getUpdatePromises(data.computables)).then(function() {
                            doSave();
                        }).catch(displayPromisesError.bind($scope));
	             	} else if (data.totalIncompatibilities > 0) {
                        $scope.schemaChanges = data;

                        let outputContainsAppendMode = false;
                        $.each($scope.recipe.outputs, function(x, outputRole){
                           outputContainsAppendMode = outputContainsAppendMode || outputRole.items.some(output => output.appendMode);
                        });
                        const partitioned = RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);
                        if ($scope.currentSaveIsForAnImmediateRun && !outputContainsAppendMode && !partitioned) {
                            Logger.info("Schema incompatibilities detected, but we are about to run the recipe anyway, so propagate without asking");

                            setFlags($scope.schemaChanges.computables, false);
                            const promises = getUpdatePromises($scope.schemaChanges.computables);
                            $q.all(promises).then(doSave).catch(setErrorInScope.bind($scope));

                        } else if (!outputContainsAppendMode && !partitioned && $rootScope.appConfig.autoAcceptSchemaChangeAtEndOfFlow
                            && data.computables.every(x => x.isLastInFlow && ! x.isPartitioned)) {
                            Logger.info("Schema incompatibilities detected, recipe at end of flow, auto-propagating schema", data);

                            setFlags($scope.schemaChanges.computables, false);
                            const promises = getUpdatePromises($scope.schemaChanges.computables);
                            $q.all(promises).then(doSave).catch(setErrorInScope.bind($scope));

                        } else {
                            Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);

                            let closedWithButton = false;
                            CreateModalFromTemplate("/templates/recipes/fragments/recipe-incompatible-schema-multi.html", $scope, null,
                                function(newScope) {
                                    setFlags($scope.schemaChanges.computables, false);
                                    newScope.cancelSave = function(){
                                        closedWithButton = true;
                                        newScope.dismiss();
                                        Logger.info("Save cancelled");
                                        deferred.reject("Save cancelled");
                                    };
                                    newScope.updateSchemaFromSuggestion = function() {
                                        closedWithButton = true;
                                        var promises = getUpdatePromises($scope.schemaChanges.computables);
                                        $q.all(promises).then(function() {
                                            newScope.dismiss();
                                            doSave();
                                        }).catch(function(data){
                                            setErrorInScope.bind($scope)(data.data, data.status, data.headers)
                                        });
                                    };
                                    newScope.ignoreSchemaChangeSuggestion = function() {
                                        closedWithButton = true;
                                        newScope.dismiss();
                                        doSave();
                                    };
                                }
                                ).then(function(){}, function(){if (!closedWithButton) {deferred.reject("Modal closed impolitely");}});
                            }
	                } else {
	                    Logger.info("No incompatible change, saving");
	                    doSave();
	                }
	            }).error(function(data, status, header){
                    Logger.error("Failed to compute recipe save impact");
                    // Failed to compute impact, don't block recipe save but ask user
                    var closedWithButton = false;
                    CreateModalFromTemplate("/templates/recipes/fragments/compute-save-impact-failed.html", $scope, null,
                            function(newScope) {
                                setErrorInScope.bind(newScope)(data, status, header);

                                newScope.cancelSave = function(){
                                    closedWithButton = true;
                                    newScope.dismiss();
                                    Logger.info("Save cancelled");
                                    deferred.reject("Save cancelled");
                                };
                                newScope.saveAnyway = function() {
                                    closedWithButton = true;
                                    newScope.dismiss();
                                    doSave();
                                };
                            }
                        ).then(function(){}, function(){if (!closedWithButton) {deferred.reject("Modal closed impolitely");}});
	            });
			},

			// Specialized version for the Shaker that needs to call a different API
			handleSaveShaker : function($scope, recipeSerialized, shaker, recipeOutputSchema, deferred) {
                const doSave = function(){
                    $scope.baseSave(recipeSerialized, JSON.stringify(shaker)).then(function() {
                        $scope.origShaker = angular.copy(shaker);
                        $scope.schemaDirtiness.dirty = false;
                        deferred.resolve("Save done");
                    },  function(error) {
                        Logger.error("Could not save recipe");
                        deferred.reject("Could not save recipe");
	                })
                }

                $scope.waitAllRefreshesDone().then(function() {
                    DataikuAPI.flow.recipes.getShakerSaveImpact($stateParams.projectKey,
                        $scope.recipe, shaker, $scope.recipeOutputSchema).success(function (data) {

                            var allPreviousSchemasWereEmpty = data.computables.every(x => x.previousSchemaWasEmpty);

                            if (data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty) {
                                Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                                setFlags(data.computables, true);
                                $q.all(getUpdatePromises(data.computables)).then(function () {
                                    doSave();
                                }).catch(displayPromisesError.bind($scope));
                            } else if (data.totalIncompatibilities > 0) {
                                $scope.schemaChanges = data;

                                let outputContainsAppendMode = false;
                                $.each($scope.recipe.outputs, function(x, outputRole){
                                   outputContainsAppendMode = outputContainsAppendMode || outputRole.items.some(output => output.appendMode);
                                });
                                const partitioned = RecipesUtils.hasAnyPartitioning($scope.recipe, $scope.computablesMap);

                                if ($scope.currentSaveIsForAnImmediateRun && !outputContainsAppendMode && !partitioned) {
                                    Logger.info("Schema incompatibilities detected, but we are about to run the recipe anyway, so propagate without asking");

                                    setFlags($scope.schemaChanges.computables, false);
                                    const promises = getUpdatePromises($scope.schemaChanges.computables);
                                    $q.all(promises).then(doSave).catch(setErrorInScope.bind($scope));

                                } else if (!outputContainsAppendMode && !partitioned && $rootScope.appConfig.autoAcceptSchemaChangeAtEndOfFlow
                                    && data.computables.every(x => x.isLastInFlow && ! x.isPartitioned)) {
                                    Logger.info("Schema incompatibilities detected, recipe at end of flow, auto-propagating schema", data);

                                    setFlags($scope.schemaChanges.computables, false);
                                    const promises = getUpdatePromises($scope.schemaChanges.computables);
                                    $q.all(promises)
                                        .then(doSave)
                                        .catch(setErrorInScope.bind($scope));
                                } else {
                                    Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);
                                    CreateModalFromTemplate("/templates/recipes/fragments/recipe-incompatible-schema-multi.html", $scope, null,
                                        function (newScope) {
                                            setFlags($scope.schemaChanges.computables, false);
                                            newScope.cancelSave = function () {
                                                newScope.dismiss();
                                                Logger.info("Save cancelled");
                                                deferred.reject("Save cancelled");
                                            }
                                            newScope.updateSchemaFromSuggestion = function () {
                                                var promises = getUpdatePromises($scope.schemaChanges.computables);
                                                $q.all(promises).then(function () {
                                                    newScope.dismiss();
                                                    doSave();
                                                }).catch(function (data) {
                                                    setErrorInScope.bind($scope)(data.data, data.status, data.headers)
                                                });
                                            }
                                            newScope.ignoreSchemaChangeSuggestion = function () {
                                                newScope.dismiss();
                                                doSave();
                                            }
                                        }
                                    );
                                }
                            } else {
                                Logger.info("No incompatible change, saving");
                                doSave();
                            }
                        }).error(function (data, status, header) {
                            setErrorInScope.bind($scope)(data, status, header);
                            deferred.reject("failed to execute getComputableSaveImpact");
                        });
                });
            },

            /* handleSchemaUpdate for recipes which have just been reconnected after a delete and reconnect operation
            */
            handleSchemaUpdateForDeleteAndReconnect : function(parentScope, recipeProjectKey, recipeName) {
                this.handleSchemaUpdateFromAnywhere(parentScope, recipeProjectKey, recipeName, true);
            },

            /**
             * Check for schema mismatch with recipe and it's output datasets/computables and if so show a modal to offer the user a choice to resolve the issue.
             * @param {Object} parentScope scope this was spawned from
             * @param {string} recipeProjectKey project id
             * @param {string} recipeName name/id of recipe
             * @param {boolean} [partOfReconnect] - normally omit this - we set this to true for a delete and reconnect operation in handleSchemaUpdateForDeleteAndReconnect()
             */
            handleSchemaUpdateFromAnywhere : function(parentScope, recipeProjectKey, recipeName, partOfReconnect) {
                var serviceScope = parentScope.$new();
                DataikuAPI.flow.recipes.getSchemaUpdateResult(recipeProjectKey, recipeName).then(function(resp) {
                    const data = resp.data;
                    var allPreviousSchemasWereEmpty = data.computables.every(x => x.previousSchemaWasEmpty)

                    if (data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty) {
                        Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                        setFlags(data.computables, true);
                        $q.all(getUpdatePromises(data.computables)).then(function() {
                            // Nothing to do
                        }).catch(displayPromisesError.bind(parentScope));
                    }  else if (data.totalIncompatibilities > 0) {
                        Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);
                        serviceScope.schemaChanges = data;
                        serviceScope.partOfReconnect = Boolean(partOfReconnect);
                        serviceScope.recipeName = recipeName;


                        CreateModalFromTemplate("/templates/recipes/incompatible-schema-external-modal.html", serviceScope, null,

                            function(newScope) {
                                setFlags(serviceScope.schemaChanges.computables, true);

                                newScope.updateSchemaFromSuggestion = function() {
                                    var promises = getUpdatePromises(serviceScope.schemaChanges.computables);
                                    $q.all(promises).then(function() {
                                        newScope.dismiss();
                                    }).catch(function(data){
                                        setErrorInScope.bind(newScope)(data.data, data.status, data.headers)
                                    });
                                }
                                newScope.ignoreSchemaChangeSuggestion = function() {
                                    newScope.dismiss();
                                }
                            }
                        );
                    } else {
                        if (!partOfReconnect) {
                            ActivityIndicator.success("Schema is already up-to-date");
                        }
                    }
                }).catch(function(data, status, header){
                    //For reconnects, if there is any error here we don't show anything for recipes we can't check the schema for
                    // The user hasn't explicitly chosen to validate the schema so it
                    // would be confusing to show this, and there's nothing they can do anyway
                    if (!partOfReconnect) {
                        CreateModalFromTemplate("/templates/recipes/propagate-schema-changes-failed-modal.html", serviceScope, null,
                        function(newScope) {
                            setErrorInScope.bind(newScope)(data, status, header);
                        });
                    }
                });
            },

            handleSchemaUpdateWithPrecomputedUnattended : function(parentScope, data) {
                var deferred = $q.defer();

                if (data && (data.totalIncompatibilities > 0 || data.recipeChanges.length > 0)) {
                    Logger.info("Schema incompatibilities, unattended mode, updating and saving");
                    setFlags(data.computables, true);
                    $q.all(getUpdatePromises(data.computables).concat(getRecipePromises(data))).then(function() {
                        deferred.resolve({changed:true});
                    }).catch(displayPromisesError.bind(parentScope));
                } else {
                    deferred.resolve({changed:false});
                }
                return deferred.promise;
            },

			handleSchemaUpdateWithPrecomputed : function(parentScope, data) {
				var deferred = $q.defer();
				var serviceScope = parentScope.$new();
                var allPreviousSchemasWereEmpty = data && data.computables.every(x => x.previousSchemaWasEmpty)

	            if (data && data.totalIncompatibilities > 0 && allPreviousSchemasWereEmpty && data.recipeChanges.length == 0) {
                    Logger.info("Schema incompatibilities, but all previous schemas were empty, updating and saving");
                    setFlags(data.computables, true);
                    $q.all(getUpdatePromises(data.computables).concat(getRecipePromises(data))).then(function() {
                        deferred.resolve({changed:true});
                    }).catch(displayPromisesError.bind(parentScope));
                } else if (data && (data.totalIncompatibilities > 0 || data.recipeChanges.length > 0)) {
                    Logger.info("Schema incompatibilities detected, and some schemas were not empty, displaying modal", data);
                    serviceScope.schemaChanges = data;

                    CreateModalFromTemplate("/templates/recipes/incompatible-schema-external-modal.html", serviceScope, null,
                        function(newScope) {
                            setFlags(serviceScope.schemaChanges.computables, true);

                            newScope.updateSchemaFromSuggestion = function() {
                                var promises = getUpdatePromises(serviceScope.schemaChanges.computables).concat(getRecipePromises(serviceScope.schemaChanges));
                                $q.all(promises).then(function() {
                                    newScope.dismiss();
                                    deferred.resolve({changed: true});
                                }).catch(function(data){
                                    setErrorInScope.bind(newScope)(data.data, data.status, data.headers)
                                    deferred.reject("Change failed");
                                });
                            }
                            newScope.ignoreSchemaChangeSuggestion = function() {
                                newScope.dismiss();
                                deferred.resolve({changed:false});
                            }
                        }
                    );
                } else {
                	deferred.resolve({changed:false});
                }
                return deferred.promise;
            },
        }
    });

    app.directive('codeRecipeSchemaList', function(DataikuAPI, Dialogs, $stateParams, CreateModalFromTemplate) {
        return {
            link : function($scope, element, attrs) {
                $scope.beginEditSchema = function(datasetSmartName) {
                    const computable = $scope.computablesMap[datasetSmartName];
                    if (!computable) {
                        throw new Error("Dataset not in computablesMap, try reloading the page");
                    }
                    const dataset = computable.dataset;
                    DataikuAPI.datasets.get(dataset.projectKey, dataset.name, $stateParams.projectKey)
                        .success(function(data){
                            CreateModalFromTemplate("/templates/recipes/code-edit-schema.html", $scope,
                            null, function(newScope) {
                                    newScope.dataset = data;
                                    // do not showGenerateMetadata from the schema edition modal (found in join recipe)
                                    newScope.showGenerateMetadata = false;
                                }).then(function(schema) {
                                    dataset.schema = schema;
                                });
                        }).error(setErrorInScope.bind($scope));
                }
            }
        }
    });

	app.directive("schemaEditorBase", function(DatasetUtils, $timeout, CreateModalFromTemplate, ContextualMenu, ExportUtils, ColumnTypeConstants, ActivityIndicator, DataikuAPI, ActiveProjectKey, translate) {
		return {
			scope : true,
			link : function($scope, element, attrs) {
                $scope.columnTypes = ColumnTypeConstants.types;
                $scope.columnTypesWithDefault = [
                    {name:'',label:'All types'},
				    ...ColumnTypeConstants.types
                ];
                $scope.translate = translate;
                $scope.meaningsWithDefault = Object.fromEntries([
                    ['','All meanings'],
                    ...Object.entries($scope.appConfig.meanings.labelsMap),
                ]);
				$scope.menusState = {meaning:false};
				$scope.startEditName = function(column, $event) {
				    $event.stopPropagation();
					$scope.dataset.schema.columns.forEach(function(x){
						x.$editingName = false;
						x.$editingComment = false;
					});

					var grandpa = $($event.target.parentNode.parentNode.parentNode);
					$timeout(function() { grandpa.find("input").focus(); });
				}
				$scope.handleInputClick = function($event) {
				    $event.stopPropagation();
				}
				$scope.blur = function(event) {
					$timeout(function() { event.currentTarget.blur(); });
				}
				$scope.setSchemaUserModifiedIfDirty = function() {
					if ($scope.datasetIsDirty && $scope.datasetIsDirty()) {
						$scope.setSchemaUserModified()
					}
				}

				function arrayMove(arr, from, to) {
					arr.splice(to, 0, arr.splice(from, 1)[0]);
				}

				$scope.moveColumnUp = function(column){
					var index = $scope.dataset.schema.columns.indexOf(column);
					if (index > 0) {
						arrayMove($scope.dataset.schema.columns, index, index - 1);
						$scope.setSchemaUserModified();
					}
				}
				$scope.moveColumnDown = function(column){
					var index = $scope.dataset.schema.columns.indexOf(column);
					if (index >= 0 && index < $scope.dataset.schema.columns.length - 1) {
						arrayMove($scope.dataset.schema.columns, index, index + 1);
						$scope.setSchemaUserModified();
					}
				}

				$scope.startEditComment = function(column, $event) {
					$scope.dataset.schema.columns.forEach(function(x){
						x.$editingName = false;
						x.$editingComment = false;
					});
					column.$editingComment = true;
					$timeout(function(){
	        			$($event.target).find("input").focus()
	        		}, 50);
				}
				$scope.addNew = function() {
					if ($scope.dataset.schema == null) {
						$scope.dataset.schema = { "columns" : []};
					}
					if ($scope.dataset.schema.columns == null) {
						$scope.dataset.schema.columns = [];
					}
					$scope.setSchemaUserModified();
					$scope.dataset.schema.columns.push({$editingName : true, name: '', type: 'string', comment: '', maxLength: 1000});
                    $scope.clearFilters();
                    $timeout(function(){
	                    $scope.$broadcast('scrollToLine', -1);
	                });
				}
				$scope.selection.orderQuery = "$idx";

				/** meanings **/
				$scope.openMeaningMenu = function($event, column) {
					$scope.meaningMenu.openAtXY($event.pageX, $event.pageY);
					$scope.meaningColumn = column;
				};

                $scope.openMassActionMenu = function() {
                    // rests the state that tracks on which column the meaning should be applied
                    // (without this, changing the meaning on a single column would 'lock' the mass aciton on this column instead of on the selection)
                    $scope.meaningColumn = null;
                }

				// use delete instead of '... = null' because when it comes as json, the property is just not there when null
				$scope.setColumnMeaning = function(meaningId) {
					if ($scope.meaningColumn == null) {
	                	$scope.selection.selectedObjects.forEach(function(c) {
							if (meaningId == null) {
								delete c.meaning;
							} else {
			                	c.meaning = meaningId;
							}
	                	});
					} else {
						if (meaningId == null) {
							delete $scope.meaningColumn.meaning;
						} else {
		                	$scope.meaningColumn.meaning = meaningId;
						}
					}
                    $(".code-edit-schema-box").css("display", "block");
                    if ($scope.setSchemaUserModified) $scope.setSchemaUserModified();
                };

                $scope.editColumnUDM = function(){
                    CreateModalFromTemplate("/templates/meanings/column-edit-udm.html", $scope, null, function(newScope){
                    	var columnName;
                        if ($scope.meaningColumn == null) {
                            columnName = $scope.selection.selectedObjects[0].name;
                        } else {
                            columnName = $scope.meaningColumn.name;
                        }
                        newScope.initModal(columnName, $scope.setColumnMeaning);
                    })
                }

                $scope.generateDatasetDescription = function() {
                    DataikuAPI.datasets.get($scope.dataset.projectKey, $scope.dataset.name, ActiveProjectKey.get()).noSpinner()
                    .success(function(data) {
                        angular.extend($scope.dataset, data);
                        CreateModalFromTemplate(
                            "/static/dataiku/ai-dataset-descriptions/generate-documentation-modal/generate-documentation-modal.html",
                            $scope,
                            "AIDatasetDescriptionsModalController",
                            function(scope) {
                                scope.init($scope.dataset, $scope.canWriteProject)
                            }
                        )
                    })
                    .error(setErrorInScope.bind($scope));
                }

				$scope.exportSchema = function() {
                    if (!$scope.dataset.schema || !$scope.dataset.schema.columns) {
                        ActivityIndicator.error('Empty schema.');
                        return;
                    }
					ExportUtils.exportUIData($scope, {
						name: "Schema of " + $scope.dataset.name,
						columns: [
							{ name: "name", type: "string" },
							{ name: "type", type: "string" },
							{ name: "meaning_id", type: "string" },
							{ name: "description", type: "string" },
							{ name: "max_length", type: "int" }
						],
						data: $scope.dataset.schema.columns.map(function(c){
							return [c.name, c.type, c.meaning, c.comment, c.maxLength >= 0 ? c.maxLength : "" ]
						})
					}, "Export schema");
				}

                $scope.meaningMenu = new ContextualMenu({
                    template: "/templates/shaker/edit-meaning-contextual-menu.html",
                    cssClass : "column-header-meanings-menu pull-right",
                    scope: $scope,
                    contextual: false,
                    onOpen: function() {
                    },
                    onClose: function() {
                    }
                });

                var reNumberColumns = function() {
                	var columns = $scope.$eval(attrs.ngModel);
                	if (!columns) return;
                	// columns.forEach(function(c, i) {c.$idx = i;});
                };

                /** column type **/
                $scope.setColumnsType = function(columnType) {
            		$scope.selection.selectedObjects.forEach(function(c) {
            			c.type = columnType;
            		});
                };
                /** renaming **/
                $scope.doRenameColumns = function(renamings) {
                	renamings.forEach(function(renaming) {
                		$scope.selection.selectedObjects.forEach(function(c) {
                			if (c.name == renaming.from) {
                				c.name = renaming.to;
                			}
                		});
                	});
                };

                $scope.renameColumns = function() {
                	CreateModalFromTemplate('/templates/shaker/modals/shaker-rename-columns.html', $scope, 'MassRenameColumnsController', function(newScope) {
                   		newScope.setColumns($scope.selection.selectedObjects.map(function(c) {return c.name;}));
                   		newScope.doRenameColumns = function(renamings) {
                   			$scope.doRenameColumns(renamings);
                   		};
                    });
                };
                /** data for the right pane **/
                var commonBaseTypeChanged = function() {
                	if (!$scope.selection || !$scope.selection.multiple) return;
                	if (!$scope.multipleSelectionInfo || !$scope.multipleSelectionInfo.commonBaseType) return;
                	var columns = $scope.selection.selectedObjects;
                	columns.forEach(function(column) {column.type = $scope.multipleSelectionInfo.commonBaseType.type;});
                };
                var commonTypeChanged = function() {
                	if (!$scope.selection || !$scope.selection.multiple) return;
                	if (!$scope.multipleSelectionInfo || !$scope.multipleSelectionInfo.commonType) return;
                	var columns = $scope.selection.selectedObjects;
                	var setFullType = function(column, commonType) {
            			column.type = commonType.type;
            			column.maxLength = commonType.maxLength;
            			column.objectFields = commonType.objectFields ? angular.copy(commonType.objectFields) : null;
            			column.arrayContent = commonType.arrayContent ? angular.copy(commonType.arrayContent) : null;
            			column.mapKeys = commonType.mapKeys ? angular.copy(commonType.mapKeys) : null;
            			column.mapValues = commonType.mapValues ? angular.copy(commonType.mapValues) : null;
                	};
                	columns.forEach(function(column) {setFullType(column, $scope.multipleSelectionInfo.commonType);});
                };
                var updateInfoForMultipleTab = function() {
                	if ($scope.commonTypeChangedDeregister) {
                		$scope.commonTypeChangedDeregister();
                		$scope.commonTypeChangedDeregister = null;
                	}
                	if ($scope.commonBaseTypeChangedDeregister) {
                		$scope.commonBaseTypeChangedDeregister();
                		$scope.commonBaseTypeChangedDeregister = null;
                	}
                	if (!$scope.selection || !$scope.selection.multiple) return;
                	var getFullType = function(column) {
                		return {
                			type:column.type ? column.type : null,
                			maxLength:column.maxLength ? column.maxLength : null,
                			objectFields:column.objectFields ? column.objectFields : null,
                			arrayContent:column.arrayContent ? column.arrayContent : null,
                			mapKeys:column.mapKeys ? column.mapKeys : null,
                			mapValues:column.mapValues ? column.mapValues : null
                		};
                	};
                	var columns = $scope.selection.selectedObjects;
                	var names = columns.map(function(column) {return column.name;});
                	var meanings = columns.map(function(column) {return column.meaning;});
                	var types = columns.map(function(column) {return column.type;});
                	var fullTypes = columns.map(function(column) {return getFullType(column);});
                	var firstFullType = fullTypes[0];
                	var sameTypes = fullTypes.map(function(t) {return angular.equals(t, firstFullType);}).reduce(function(a,b) {return a && b;});
                	var commonType = sameTypes ? firstFullType : null;
                	$scope.multipleSelectionInfo = {sameTypes: sameTypes, commonType : commonType, commonBaseType : null};

                    $scope.commonBaseTypeChangedDeregister = $scope.$watch('multipleSelectionInfo.commonBaseType', commonBaseTypeChanged);
                    $scope.commonTypeChangedDeregister = $scope.$watch('multipleSelectionInfo.commonType', commonTypeChanged, true);
                };
                $scope.$watch('selection.multiple', updateInfoForMultipleTab);
                $scope.$watch('selection.selectedObjects', updateInfoForMultipleTab, true);


                $scope.$watch(attrs.ngModel, reNumberColumns); // for when the schema is inferred again (but nothing changes)
                $scope.$watch(attrs.ngModel, reNumberColumns, true);
			}
		}
	});

	app.directive('codeRecipeSchemaEditing', function(DataikuAPI, DatasetUtils, DatasetsService,
		Dialogs, $stateParams, $timeout, Logger){
    	return {
	        link : function($scope, element, attrs) {
	        	$scope.overwriteSchema = function(newSchema) {
	        		$scope.dataset.schema = angular.copy(newSchema);
	        		$scope.schemaJustModified = false;
	        		$scope.consistency = null;
	        	};

	        	$scope.saveSchema = function() {
	        		DataikuAPI.datasets.save($scope.dataset.projectKey, $scope.dataset).success(function(data){
                        $scope.resolveModal($scope.dataset.schema);
	        		}).error(setErrorInScope.bind($scope));
	        	};

	        	$scope.discardConsistencyError= function(){
	        		$scope.consistency = null;
	        	};

	        	$scope.setSchemaUserModified = function() {
            		$scope.schemaJustModified = true;
              		$scope.dataset.schema.userModified = true;
              		$scope.consistency = null;
          		};

                $scope.addColumn = function(){
                    if ($scope.dataset.schema == null) {
                        $scope.dataset.schema = { "columns" : []};
                    }
                    $scope.setSchemaUserModified();
                    $scope.dataset.schema.columns.push({$editingName : true, name: '', type: 'string', comment: '', maxLength: 1000});
                };

	        	$scope.checkConsistency = function () {
			        Logger.info('Checking consistency');
			        $scope.schemaJustModified = false;

			        DataikuAPI.datasets.testSchemaConsistency($scope.dataset).success(function (data) {
            			Logger.info("Got consistency result", data);
            			$scope.consistency = data;
            			$scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
            		});
	        	};
	        }
        }
    });
})();
