(function () {
  "use strict";

  angular.module("dataiku.opals").factory("OpalsService", OpalsService);

  function OpalsService(
    $sce,
    $rootScope,
    $http,
    Logger,
    TopbarDrawersService,
    TOPBAR_DRAWER_IDS,
    OpalsMessageService
  ) {
    const PAGES = {
      EMBEDDED_BROWSER: "/embedded-browser",
      HELP: "/help",
      QUICKSTART_DEVELOPER: "/onboarding/quick-start-developer-guide/task",
      QUICKSTART_ML : "/onboarding/quick-start-machine-learning/task",
      QUICKSTART_DATA_PREPARATION : "/onboarding/quick-start-data-preparation/task",
      QUICKSTART_MANUFACTURING_DATA_PREPARATION : "/onboarding/quick-start-manufacturing-data-preparation/task",
      QUICKSTART_CODE_NOTEBOOKS : "/onboarding/quick-start-code-notebooks-and-recipes/task",
      QUICKSTART_AGENTS: "/onboarding/quick-start-agents/task",
      
    };

    let helpCenterFrame = null;

    function isEnabled() {
      return OpalsMessageService.getAppConfig().then((appConfig) => {
        return appConfig.opalsEnabled;
      });
    }

    function getUrl(path) {
      return OpalsMessageService.getBaseUrl().then((url) => {
        if (!url) {
          return null;
        }
        if (path) {
          url.pathname = path;
        }
        if (!!window.devInstance) {
          url.searchParams.set("isDevInstance", 1);
        }
        return $sce.trustAsResourceUrl(url.toString());
      });
    }

    function getHelpCenterUrl() {
      return getUrl();
    }

    function getHomeBannerUrl() {
      return getUrl("/home-banner");
    }

    function getHeadlessUrl() {
      return getUrl("/headless.html");
    }

    function isUrlAvailable(url) {
      return $http.head(url, { headers: { "Accept": "*/*" } }).then(
        () => {
          Logger.info(`OPALS URL ${url} available`);
          return true;
        },
        () => {
          Logger.error(`OPALS URL ${url} unavailable`);
          return false;
        }
      );
    }

    function isHelpCenterAvailable() {
      return getHelpCenterUrl().then((url) => {
        return isUrlAvailable(url);
      });
    }

    function isHomeBannerAvailable() {
      return getHomeBannerUrl().then((url) => {
        return isUrlAvailable(url);
      });
    }

    function isHeadlessAvailable() {
      return getHeadlessUrl().then((url) => {
        return isUrlAvailable(url);
      });
    }

    function setHelpCenterFrame(frame) {
      OpalsMessageService.setFrame(OpalsMessageService.FRAME_IDS.HELP_CENTER, frame);
      helpCenterFrame = frame;
    }

    function setHomeBannerFrame(frame) {
      OpalsMessageService.setFrame(OpalsMessageService.FRAME_IDS.HOME_BANNER, frame);
    }

    function setHeadlessFrame(frame) {
      OpalsMessageService.setFrame(OpalsMessageService.FRAME_IDS.HEADLESS, frame);
    }

    function unregisterHomeBannerFrame(frame) {
        OpalsMessageService.unregisterFrame(OpalsMessageService.FRAME_IDS.HOME_BANNER, frame);
    }

    function focusHelpCenterFrame() {
      if (helpCenterFrame) {
        helpCenterFrame.contentWindow.focus();
      }
    }

    function navigateToAndShowDrawer(page, searchParams) {
        OpalsMessageService.navigateToAndShowDrawer(page, searchParams);
    }

    function notifyDrawerDocked(state) {
      OpalsMessageService.sendMessage(
        OpalsMessageService.FRAME_IDS.HELP_CENTER,
        OpalsMessageService.DSS_MESSAGE_TYPES.OPALS_HELP_DRAWER_DOCKED,
        state
      );
    }

    function notifyDrawerMinimized(state) {
      OpalsMessageService.sendMessage(
        OpalsMessageService.FRAME_IDS.HELP_CENTER,
        OpalsMessageService.DSS_MESSAGE_TYPES.OPALS_HELP_DRAWER_MINIMIZED,
        state
      );
    }

    function sendPageSpecificTourRecommendation(tourName) {
      OpalsMessageService.sendMessage(
        OpalsMessageService.FRAME_IDS.HELP_CENTER,
        OpalsMessageService.DSS_MESSAGE_TYPES.OPALS_RECOMMEND_PAGE_SPECIFIC_TOUR,
        tourName
      );
      const deregisterStateChangeListener = $rootScope.$on("$stateChangeSuccess", function () {
        OpalsMessageService.sendMessage(
          OpalsMessageService.FRAME_IDS.HELP_CENTER,
          OpalsMessageService.DSS_MESSAGE_TYPES.OPALS_RECOMMEND_PAGE_SPECIFIC_TOUR,
          null
        );
        deregisterStateChangeListener();
      });
    }

    // To be used for DSS route state change
    function sendDssStateChange(data) {
      OpalsMessageService.setDssState(data);
      OpalsMessageService.sendMessage(
        OpalsMessageService.FRAME_IDS.HELP_CENTER,
        OpalsMessageService.DSS_MESSAGE_TYPES.OPALS_DSS_STATE_CHANGE,
        data
      );
    }

    // To be used for any DSS context change
    function sendDssContextChange(context) {
      OpalsMessageService.sendMessage(
        OpalsMessageService.FRAME_IDS.HELP_CENTER,
        OpalsMessageService.DSS_MESSAGE_TYPES.OPALS_DSS_CONTEXT_CHANGE,
        context
      );
    }

    function sendWT1Event(type, params) {
      OpalsMessageService.sendWT1Event(type, params);
    }

    function getLocalStorage(key) {
      return OpalsMessageService.getLocalStorage(key);
    }

    function setLocalStorage(key, value) {
      return OpalsMessageService.setLocalStorage(key, value);
    }

    return {
      isEnabled,
      getHelpCenterUrl,
      getHomeBannerUrl,
      getHeadlessUrl,
      isHelpCenterAvailable,
      isHomeBannerAvailable,
      isHeadlessAvailable,
      setHelpCenterFrame,
      setHomeBannerFrame,
      setHeadlessFrame,
      focusHelpCenterFrame,
      navigateToAndShowDrawer,
      notifyDrawerDocked,
      notifyDrawerMinimized,
      sendDssStateChange,
      sendDssContextChange,
      sendPageSpecificTourRecommendation,
      sendWT1Event,
      unregisterHomeBannerFrame,
      getUrl,
      getLocalStorage,
      setLocalStorage,
      PAGES,
    };
  }
})();
