(function() {
    'use strict';

    angular.module('dataiku.shared').component('lottie', {
        bindings: {
            animationData: '@',
            autoplay: '<?', // should the animation autoplay, default = true
            loop: '<?', // should the animation loop when finished, default = true
            animationCb: '<?' // callback that gives the caller the lottie animation object when advanced controls are required
        },
        templateUrl : '/static/dataiku/shared/components/lottie/lottie.component.html',
        controller: function($scope, $element, AssetsUtils) {
            const ctrl = this;

            ctrl.$onInit = () => {
                ctrl.initialized = true;
                if (ctrl.animationData) {
                    loadAnimation();
                }
            };

            ctrl.$onChanges = changesObj => {
                if (!ctrl.initialized) {
                    return;
                }

                if (changesObj.animationData && changesObj.animationData.previousValue !== changesObj.animationData.currentValue) {
                    // To avoid flickering, we're trying to be smart here and temporarily force the dimension of the inner div to the dimension of the
                    // animation that was displayed, until the new animation is ready. For that, we need to capture the width & height of the currently
                    // loaded animation.
                    const innerDiv = $element.find("div");
                    const svgDimensions = getDimensions(innerDiv.find("svg"));

                    // Destroy the animation (i.e. remove the svg element from the DOM)
                    destroyAnimation();

                    // Load the new animation
                    if (ctrl.animationData) {
                        // Force the width & height of the inner div to avoid the flickering
                        if (svgDimensions.width && svgDimensions.height) {
                            innerDiv.width(svgDimensions.width).height(svgDimensions.height);
                        }
                        loadAnimation();

                        // As soon as the svg node is added back to the DOM, we can clear the width & height of the inner div
                        if (svgDimensions.width && svgDimensions.height) {
                            ctrl.mutationObserver = createMutationObserver(innerDiv, function() {
                                if (innerDiv.find("svg").length) {
                                    innerDiv.width("").height("");
                                    ctrl.mutationObserver.disconnect();
                                    ctrl.mutationObserver = null;
                                }
                            });
                        }
                    }
                }
            };

            $scope.$on("$destroy", () => {
                destroyAnimation();
            });

            function loadAnimation() {
                if (ctrl.animationData) {
                    const innerDiv = $element.find("div");
                    ctrl.animation = bodymovin.loadAnimation({
                        container: innerDiv.get(0),
                        path: AssetsUtils.appendAssetsHash(ctrl.animationData),
                        renderer: 'svg',
                        loop: ctrl.loop !== false,
                        autoplay: ctrl.autoplay !== false,
                    });

                    if (typeof ctrl.animationCb === 'function') {
                        ctrl.animation.addEventListener('data_ready', () => {
                            ctrl.animationCb(ctrl.animation)
                        });
                    }
                }
            }

            function destroyAnimation() {
                if (ctrl.animation) {
                    ctrl.animation.destroy();
                    ctrl.animation = null;
                }
                if (ctrl.mutationObserver) {
                    ctrl.mutationObserver.disconnect();
                    ctrl.mutationObserver = null;
                }
            }

            function getDimensions(element) {
                return { width: element.attr("width"), height: element.attr("height") };
            }

            function createMutationObserver(innerDiv, callback) {
                const MutationObserver = window.MutationObserver || window.WebKitMutationObserver;
                if (MutationObserver) {
                    let observer = new MutationObserver(callback);
                    observer.observe(innerDiv.get(0), { childList: true });
                    return observer;
                }
            }
        }
    });
})();
